/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.database;

import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.SequencedSet;
import java.util.Set;
import java.util.stream.Collectors;
import org.jabref.logic.util.OS;
import org.jabref.logic.util.strings.StringSimilarity;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.AuthorList;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibEntryType;
import org.jabref.model.entry.BibEntryTypesManager;
import org.jabref.model.entry.field.BibField;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.FieldProperty;
import org.jabref.model.entry.field.OrFields;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.identifier.ISBN;
import org.jabref.model.entry.types.StandardEntryType;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DuplicateCheck {
    private static final double DUPLICATE_THRESHOLD = 0.75;
    private static final Logger LOGGER = LoggerFactory.getLogger(DuplicateCheck.class);
    private static final int NOT_EQUAL = 0;
    private static final int EQUAL = 1;
    private static final int EMPTY_IN_ONE = 2;
    private static final int EMPTY_IN_TWO = 3;
    private static final int EMPTY_IN_BOTH = 4;
    private static final double DOUBT_RANGE = 0.05;
    private static final double REQUIRED_WEIGHT = 3.0;
    private static final Map<Field, Double> FIELD_WEIGHTS = new HashMap<Field, Double>();
    private final BibEntryTypesManager entryTypesManager;

    public DuplicateCheck(BibEntryTypesManager entryTypesManager) {
        this.entryTypesManager = entryTypesManager;
    }

    private static boolean haveSameIdentifier(BibEntry one, BibEntry two) {
        return one.getFields().stream().filter(field -> field.getProperties().contains((Object)FieldProperty.IDENTIFIER)).anyMatch(field -> two.getField((Field)field).map(content -> one.getField((Field)field).orElseThrow().equals(content)).orElse(false));
    }

    private static boolean haveDifferentEntryType(BibEntry one, BibEntry two) {
        return !one.getType().equals(two.getType());
    }

    private static boolean haveDifferentEditions(BibEntry one, BibEntry two) {
        Optional<String> editionOne = one.getField(StandardField.EDITION);
        Optional<String> editionTwo = two.getField(StandardField.EDITION);
        return editionOne.isPresent() && editionTwo.isPresent() && !editionOne.get().equals(editionTwo.get());
    }

    private static boolean haveDifferentChaptersOrPagesOfTheSameBook(BibEntry one, BibEntry two) {
        return DuplicateCheck.compareSingleField(StandardField.AUTHOR, one, two) == 1 && DuplicateCheck.compareSingleField(StandardField.TITLE, one, two) == 1 && (DuplicateCheck.compareSingleField(StandardField.CHAPTER, one, two) == 0 || DuplicateCheck.compareSingleField(StandardField.PAGES, one, two) == 0);
    }

    private static double[] compareRequiredFields(BibEntryType type, BibEntry one, BibEntry two) {
        double[] dArray;
        SequencedSet<OrFields> requiredFields = type.getRequiredFields();
        if (requiredFields.isEmpty()) {
            double[] dArray2 = new double[2];
            dArray2[0] = 0.0;
            dArray = dArray2;
            dArray2[1] = 0.0;
        } else {
            dArray = DuplicateCheck.compareFieldSet(requiredFields.stream().map(OrFields::getPrimary).collect(Collectors.toSet()), one, two);
        }
        return dArray;
    }

    private static boolean isFarFromThreshold(double value) {
        if (value < 0.0) {
            LOGGER.debug("Value {} is below zero. Should not happen", (Object)value);
        }
        return value - 0.75 > 0.05;
    }

    private static boolean compareOptionalFields(BibEntryType type, BibEntry one, BibEntry two, double[] req) {
        double denominator;
        SequencedSet<BibField> optionalFields = type.getOptionalFields();
        if (optionalFields.isEmpty()) {
            return req[0] >= 0.75;
        }
        double[] opt = DuplicateCheck.compareFieldSet(optionalFields.stream().map(BibField::field).collect(Collectors.toSet()), one, two);
        double numerator = 3.0 * req[0] * req[1] + opt[0] * opt[1];
        double totValue = numerator / (denominator = req[1] * 3.0 + opt[1]);
        return totValue >= 0.75;
    }

    private static double[] compareFieldSet(Collection<Field> fields, BibEntry one, BibEntry two) {
        if (fields.isEmpty()) {
            return new double[]{0.0, 0.0};
        }
        double equalWeights = 0.0;
        double totalWeights = 0.0;
        for (Field field : fields) {
            double currentWeight = FIELD_WEIGHTS.getOrDefault(field, 1.0);
            totalWeights += currentWeight;
            int result = DuplicateCheck.compareSingleField(field, one, two);
            if (result == 1) {
                equalWeights += currentWeight;
                continue;
            }
            if (result != 4) continue;
            totalWeights -= currentWeight;
        }
        if (totalWeights > 0.0) {
            return new double[]{equalWeights / totalWeights, totalWeights};
        }
        return new double[]{0.0, 0.0};
    }

    private static int compareSingleField(Field field, BibEntry one, BibEntry two) {
        Optional<String> optionalStringOne = one.getFieldLatexFree(field);
        Optional<String> optionalStringTwo = two.getFieldLatexFree(field);
        if (optionalStringOne.isEmpty()) {
            if (optionalStringTwo.isEmpty()) {
                return 4;
            }
            return 2;
        }
        if (optionalStringTwo.isEmpty()) {
            return 3;
        }
        String stringOne = optionalStringOne.get();
        String stringTwo = optionalStringTwo.get();
        if (field.getProperties().contains((Object)FieldProperty.PERSON_NAMES)) {
            return DuplicateCheck.compareAuthorField(stringOne, stringTwo);
        }
        if (StandardField.PAGES == field) {
            return DuplicateCheck.comparePagesField(stringOne, stringTwo);
        }
        if (StandardField.JOURNAL == field) {
            return DuplicateCheck.compareJournalField(stringOne, stringTwo);
        }
        if (StandardField.CHAPTER == field) {
            return DuplicateCheck.compareChapterField(stringOne, stringTwo);
        }
        return DuplicateCheck.compareField(stringOne, stringTwo);
    }

    private static int compareAuthorField(String stringOne, String stringTwo) {
        String authorTwo;
        String authorOne = AuthorList.fixAuthorLastNameOnlyCommas(stringOne, false).replace(" and ", " ").toLowerCase(Locale.ROOT);
        double similarity = DuplicateCheck.correlateByWords(authorOne, authorTwo = AuthorList.fixAuthorLastNameOnlyCommas(stringTwo, false).replace(" and ", " ").toLowerCase(Locale.ROOT));
        if (similarity > 0.8) {
            return 1;
        }
        return 0;
    }

    private static int comparePagesField(String stringOne, String stringTwo) {
        String processedStringTwo;
        String processedStringOne = stringOne.replaceAll("[- ]+", "-");
        if (processedStringOne.equals(processedStringTwo = stringTwo.replaceAll("[- ]+", "-"))) {
            return 1;
        }
        return 0;
    }

    private static int compareJournalField(String stringOne, String stringTwo) {
        String processedStringTwo;
        String processedStringOne = stringOne.replace(".", "").toLowerCase(Locale.ROOT);
        double similarity = DuplicateCheck.correlateByWords(processedStringOne, processedStringTwo = stringTwo.replace(".", "").toLowerCase(Locale.ROOT));
        if (similarity > 0.8) {
            return 1;
        }
        return 0;
    }

    private static int compareChapterField(String stringOne, String stringTwo) {
        String processedStringOne = stringOne.replaceAll("(?i)chapter", "").trim();
        String processedStringTwo = stringTwo.replaceAll("(?i)chapter", "").trim();
        return DuplicateCheck.compareField(processedStringOne, processedStringTwo);
    }

    private static int compareField(String stringOne, String stringTwo) {
        String processedStringTwo;
        String processedStringOne = StringUtil.unifyLineBreaks(stringOne.toLowerCase(Locale.ROOT).trim(), OS.NEWLINE);
        double similarity = DuplicateCheck.correlateByWords(processedStringOne, processedStringTwo = StringUtil.unifyLineBreaks(stringTwo.toLowerCase(Locale.ROOT).trim(), OS.NEWLINE));
        if (similarity > 0.8) {
            return 1;
        }
        return 0;
    }

    public static double compareEntriesStrictly(BibEntry one, BibEntry two) {
        HashSet<Field> allFields = new HashSet<Field>();
        allFields.addAll(one.getFields());
        allFields.addAll(two.getFields());
        int score = 0;
        for (Field field : allFields) {
            if (!DuplicateCheck.isSingleFieldEqual(one, two, field)) continue;
            ++score;
        }
        if (score == allFields.size()) {
            return 1.01;
        }
        return (double)score / (double)allFields.size();
    }

    private static boolean isSingleFieldEqual(BibEntry one, BibEntry two, Field field) {
        Optional<String> stringOne = one.getField(field);
        Optional<String> stringTwo = two.getField(field);
        if (stringOne.isEmpty() && stringTwo.isEmpty()) {
            return true;
        }
        if (stringOne.isEmpty() || stringTwo.isEmpty()) {
            return false;
        }
        return StringUtil.unifyLineBreaks(stringOne.get(), OS.NEWLINE).equals(StringUtil.unifyLineBreaks(stringTwo.get(), OS.NEWLINE));
    }

    public static double correlateByWords(String s1, String s2) {
        String[] w1 = s1.split("\\s");
        String[] w2 = s2.split("\\s");
        int n = Math.min(w1.length, w2.length);
        int misses = 0;
        for (int i = 0; i < n; ++i) {
            double corr = DuplicateCheck.similarity(w1[i], w2[i]);
            if (!(corr < 0.75)) continue;
            ++misses;
        }
        double missRate = (double)misses / (double)n;
        return 1.0 - missRate;
    }

    private static double similarity(String first, String second) {
        String shorter;
        String longer;
        if (first.length() < second.length()) {
            longer = second;
            shorter = first;
        } else {
            longer = first;
            shorter = second;
        }
        int longerLength = longer.length();
        if (longerLength == 0) {
            return 1.0;
        }
        double distanceIgnoredCase = new StringSimilarity().editDistanceIgnoreCase(longer, shorter);
        double similarity = ((double)longerLength - distanceIgnoredCase) / (double)longerLength;
        LOGGER.debug("Longer string: {} Shorter string: {} Similarity: {}", new Object[]{longer, shorter, similarity});
        return similarity;
    }

    public boolean isDuplicate(BibEntry one, BibEntry two, BibDatabaseMode bibDatabaseMode) {
        if (DuplicateCheck.haveSameIdentifier(one, two)) {
            return true;
        }
        if (DuplicateCheck.haveDifferentEntryType(one, two) || DuplicateCheck.haveDifferentEditions(one, two) || DuplicateCheck.haveDifferentChaptersOrPagesOfTheSameBook(one, two)) {
            return false;
        }
        Optional<ISBN> oneISBN = one.getISBN();
        Optional<ISBN> twoISBN = two.getISBN();
        if (oneISBN.isPresent() && twoISBN.isPresent() && Objects.equals(oneISBN, twoISBN) && !Set.of(StandardEntryType.Article, StandardEntryType.InBook, StandardEntryType.InCollection).contains(one.getType())) {
            return true;
        }
        Optional<BibEntryType> type = this.entryTypesManager.enrich(one.getType(), bibDatabaseMode);
        if (type.isPresent()) {
            BibEntryType entryType = type.get();
            double[] reqCmpResult = DuplicateCheck.compareRequiredFields(entryType, one, two);
            if (DuplicateCheck.isFarFromThreshold(reqCmpResult[0])) {
                return reqCmpResult[0] >= 0.75;
            }
            if (DuplicateCheck.compareOptionalFields(type.get(), one, two, reqCmpResult)) {
                return true;
            }
        }
        return DuplicateCheck.compareFieldSet((Collection<Field>)Sets.union(one.getFields(), two.getFields()), one, two)[0] >= 0.75;
    }

    public Optional<BibEntry> containsDuplicate(BibDatabase database, BibEntry entry, BibDatabaseMode bibDatabaseMode) {
        return database.getEntries().stream().filter(other -> this.isDuplicate(entry, (BibEntry)other, bibDatabaseMode)).findFirst();
    }

    static {
        FIELD_WEIGHTS.put(StandardField.AUTHOR, 2.5);
        FIELD_WEIGHTS.put(StandardField.EDITOR, 2.5);
        FIELD_WEIGHTS.put(StandardField.TITLE, 3.0);
        FIELD_WEIGHTS.put(StandardField.JOURNAL, 2.0);
        FIELD_WEIGHTS.put(StandardField.NOTE, 0.1);
        FIELD_WEIGHTS.put(StandardField.COMMENT, 0.1);
        FIELD_WEIGHTS.put(StandardField.DOI, 3.0);
    }
}

