/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.exporter;

import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.PosixFilePermission;
import java.util.EnumSet;
import java.util.Set;
import org.jabref.logic.util.BackupFileType;
import org.jabref.logic.util.io.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AtomicFileOutputStream
extends FilterOutputStream {
    private static final Logger LOGGER = LoggerFactory.getLogger(AtomicFileOutputStream.class);
    private static final String TEMPORARY_EXTENSION = ".tmp";
    private static final String SAVE_EXTENSION = "." + BackupFileType.SAVE.getExtensions().getFirst();
    private final Path targetFile;
    private final Path temporaryFile;
    private FileLock temporaryFileLock;
    private final Path backupFile;
    private final boolean keepBackup;
    private boolean errorDuringWrite = false;

    public AtomicFileOutputStream(Path path, boolean keepBackup) throws IOException {
        this(path, AtomicFileOutputStream.getPathOfTemporaryFile(path), new FileOutputStream(AtomicFileOutputStream.getPathOfTemporaryFile(path).toFile()), keepBackup);
    }

    public AtomicFileOutputStream(Path path) throws IOException {
        this(path, false);
    }

    AtomicFileOutputStream(Path path, Path pathOfTemporaryFile, OutputStream temporaryFileOutputStream, boolean keepBackup) throws IOException {
        super(temporaryFileOutputStream);
        this.targetFile = path;
        this.temporaryFile = pathOfTemporaryFile;
        this.backupFile = AtomicFileOutputStream.getPathOfSaveBackupFile(path);
        this.keepBackup = keepBackup;
        try {
            OutputStream outputStream = this.out;
            if (outputStream instanceof FileOutputStream) {
                FileOutputStream stream = (FileOutputStream)outputStream;
                try {
                    this.temporaryFileLock = stream.getChannel().tryLock();
                }
                catch (IOException ex) {
                    LOGGER.warn("Could not acquire file lock. Maybe we are on a network drive?", (Throwable)ex);
                    this.temporaryFileLock = null;
                }
            } else {
                this.temporaryFileLock = null;
            }
        }
        catch (OverlappingFileLockException exception) {
            throw new IOException("Could not obtain write access to " + String.valueOf(this.temporaryFile) + ". Maybe another instance of JabRef is currently writing to the same file?", exception);
        }
    }

    private static Path getPathOfTemporaryFile(Path targetFile) {
        return FileUtil.addExtension(targetFile, TEMPORARY_EXTENSION);
    }

    private static Path getPathOfSaveBackupFile(Path targetFile) {
        return FileUtil.addExtension(targetFile, SAVE_EXTENSION);
    }

    public Path getBackup() {
        return this.backupFile;
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        try {
            this.out.write(b, off, len);
        }
        catch (IOException exception) {
            this.cleanup();
            this.errorDuringWrite = true;
            throw exception;
        }
    }

    public void abort() {
        this.errorDuringWrite = true;
        try {
            super.close();
            Files.deleteIfExists(this.temporaryFile);
            Files.deleteIfExists(this.backupFile);
        }
        catch (IOException exception) {
            LOGGER.debug("Unable to abort writing to file {}", (Object)this.temporaryFile, (Object)exception);
        }
    }

    private void cleanup() {
        try {
            if (this.temporaryFileLock != null && this.temporaryFileLock.isValid()) {
                this.temporaryFileLock.release();
            }
        }
        catch (IOException exception) {
            LOGGER.debug("Unable to release lock on file {}", (Object)this.temporaryFile, (Object)exception);
        }
        try {
            Files.deleteIfExists(this.temporaryFile);
        }
        catch (IOException exception) {
            LOGGER.debug("Unable to delete file {}", (Object)this.temporaryFile, (Object)exception);
        }
    }

    @Override
    public void close() throws IOException {
        try {
            try {
                this.flush();
                OutputStream outputStream = this.out;
                if (outputStream instanceof FileOutputStream) {
                    FileOutputStream stream = (FileOutputStream)outputStream;
                    stream.getFD().sync();
                }
            }
            catch (IOException exception) {
                super.close();
                throw exception;
            }
            super.close();
            if (this.errorDuringWrite) {
                return;
            }
            Set<PosixFilePermission> oldFilePermissions = EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.OTHERS_READ);
            if (Files.exists(this.targetFile, new LinkOption[0])) {
                try {
                    Files.copy(this.targetFile, this.backupFile, StandardCopyOption.REPLACE_EXISTING);
                }
                catch (Exception e) {
                    LOGGER.warn("Could not create backup file {}", (Object)this.backupFile);
                }
                if (FileUtil.IS_POSIX_COMPLIANT) {
                    try {
                        oldFilePermissions = Files.getPosixFilePermissions(this.targetFile, new LinkOption[0]);
                    }
                    catch (IOException exception) {
                        LOGGER.warn("Error getting file permissions for file {}.", (Object)this.targetFile, (Object)exception);
                    }
                }
            }
            try {
                Files.move(this.temporaryFile, this.targetFile, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (Exception e) {
                LOGGER.warn("Could not move temporary file", (Throwable)e);
                throw e;
            }
            if (FileUtil.IS_POSIX_COMPLIANT) {
                try {
                    Files.setPosixFilePermissions(this.targetFile, oldFilePermissions);
                }
                catch (IOException exception) {
                    LOGGER.warn("Error writing file permissions to file {}.", (Object)this.targetFile, (Object)exception);
                }
            }
            if (!this.keepBackup) {
                Files.deleteIfExists(this.backupFile);
            }
        }
        finally {
            this.cleanup();
        }
    }

    @Override
    public void flush() throws IOException {
        try {
            super.flush();
        }
        catch (IOException exception) {
            this.cleanup();
            throw exception;
        }
    }

    @Override
    public void write(int b) throws IOException {
        try {
            super.write(b);
        }
        catch (IOException exception) {
            this.cleanup();
            throw exception;
        }
    }
}

