/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.net.ssl;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import org.jabref.logic.net.ssl.SSLCertificate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TrustStoreManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(TrustStoreManager.class);
    private static final String STORE_PASSWORD = "changeit";
    private final Path storePath;
    private KeyStore store;

    public TrustStoreManager(Path storePath) {
        this.storePath = storePath;
        TrustStoreManager.createTruststoreFileIfNotExist(storePath);
        try {
            this.store = KeyStore.getInstance(KeyStore.getDefaultType());
            this.store.load(new FileInputStream(storePath.toFile()), STORE_PASSWORD.toCharArray());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOGGER.warn("Error while loading trust store from: {}", (Object)storePath.toAbsolutePath(), (Object)e);
        }
    }

    public void addCertificate(String alias, Path certPath) {
        Objects.requireNonNull(alias);
        Objects.requireNonNull(certPath);
        try {
            CertificateFactory certificateFactory = CertificateFactory.getInstance("X509");
            this.store.setCertificateEntry(alias, certificateFactory.generateCertificate(new FileInputStream(certPath.toFile())));
        }
        catch (IOException | KeyStoreException | CertificateException e) {
            LOGGER.warn("Error while adding a new certificate to the truststore: {}", (Object)alias, (Object)e);
        }
    }

    public void deleteCertificate(String alias) {
        Objects.requireNonNull(alias);
        try {
            this.store.deleteEntry(alias);
        }
        catch (KeyStoreException e) {
            LOGGER.warn("Error while deleting certificate entry with alias: {}", (Object)alias, (Object)e);
        }
    }

    public boolean certificateExists(String alias) {
        Objects.requireNonNull(alias);
        try {
            return this.store.isCertificateEntry(alias);
        }
        catch (KeyStoreException e) {
            LOGGER.warn("Error while checking certificate existence: {}", (Object)alias, (Object)e);
            return false;
        }
    }

    public List<String> aliases() {
        try {
            return Collections.list(this.store.aliases());
        }
        catch (KeyStoreException e) {
            LOGGER.warn("Error while reading aliases", (Throwable)e);
            return Collections.emptyList();
        }
    }

    public int certsCount() {
        try {
            return this.store.size();
        }
        catch (KeyStoreException e) {
            LOGGER.warn("Can't count certificates", (Throwable)e);
            return 0;
        }
    }

    public void flush() {
        try {
            this.store.store(Files.newOutputStream(this.storePath, new OpenOption[0]), STORE_PASSWORD.toCharArray());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOGGER.warn("Error while flushing trust store", (Throwable)e);
        }
    }

    private Boolean isCustomCertificate(String alias) {
        return alias.endsWith("[custom]");
    }

    public void clearCustomCertificates() {
        this.aliases().stream().filter(this::isCustomCertificate).forEach(this::deleteCertificate);
        this.flush();
    }

    public List<SSLCertificate> getCustomCertificates() {
        return this.aliases().stream().filter(this::isCustomCertificate).map(this::getCertificate).map(SSLCertificate::fromX509).flatMap(Optional::stream).collect(Collectors.toList());
    }

    public X509Certificate getCertificate(String alias) {
        try {
            return (X509Certificate)this.store.getCertificate(alias);
        }
        catch (KeyStoreException e) {
            LOGGER.warn("Error while getting certificate of alias: {}", (Object)alias, (Object)e);
            return null;
        }
    }

    public static void createTruststoreFileIfNotExist(Path storePath) {
        try {
            LOGGER.debug("Trust store path: {}", (Object)storePath.toAbsolutePath());
            if (Files.notExists(storePath, new LinkOption[0])) {
                Path storeResourcePath = Path.of(TrustStoreManager.class.getResource("/ssl/truststore.jks").toURI());
                Files.createDirectories(storePath.getParent(), new FileAttribute[0]);
                Files.copy(storeResourcePath, storePath, new CopyOption[0]);
            }
            try {
                TrustStoreManager.configureTrustStore(storePath);
            }
            catch (KeyManagementException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                LOGGER.error("Error configuring trust store {}", (Object)storePath, (Object)e);
            }
        }
        catch (IOException e) {
            LOGGER.warn("Bad truststore path", (Throwable)e);
        }
        catch (URISyntaxException e) {
            LOGGER.warn("Bad resource path", (Throwable)e);
        }
    }

    private static void configureTrustStore(Path myStorePath) throws NoSuchAlgorithmException, KeyManagementException, KeyStoreException, CertificateException, IOException {
        X509TrustManager jreTrustManager = TrustStoreManager.getJreTrustManager();
        X509TrustManager myTrustManager = TrustStoreManager.getJabRefTrustManager(myStorePath);
        X509TrustManager mergedTrustManager = TrustStoreManager.createMergedTrustManager(jreTrustManager, myTrustManager);
        TrustStoreManager.setSystemTrustManager(mergedTrustManager);
    }

    private static X509TrustManager getJreTrustManager() throws NoSuchAlgorithmException, KeyStoreException {
        return TrustStoreManager.findDefaultTrustManager(null);
    }

    private static X509TrustManager getJabRefTrustManager(Path myStorePath) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        try (InputStream myKeys = Files.newInputStream(myStorePath, new OpenOption[0]);){
            KeyStore myTrustStore = KeyStore.getInstance("jks");
            myTrustStore.load(myKeys, STORE_PASSWORD.toCharArray());
            X509TrustManager x509TrustManager = TrustStoreManager.findDefaultTrustManager(myTrustStore);
            return x509TrustManager;
        }
    }

    private static X509TrustManager findDefaultTrustManager(KeyStore keyStore) throws NoSuchAlgorithmException, KeyStoreException {
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        tmf.init(keyStore);
        for (TrustManager tm : tmf.getTrustManagers()) {
            if (!(tm instanceof X509TrustManager)) continue;
            X509TrustManager manager = (X509TrustManager)tm;
            return manager;
        }
        return null;
    }

    private static X509TrustManager createMergedTrustManager(final X509TrustManager jreTrustManager, final X509TrustManager customTrustManager) {
        return new X509TrustManager(){

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return jreTrustManager.getAcceptedIssuers();
            }

            @Override
            public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                try {
                    customTrustManager.checkServerTrusted(chain, authType);
                }
                catch (CertificateException e) {
                    jreTrustManager.checkServerTrusted(chain, authType);
                }
            }

            @Override
            public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                jreTrustManager.checkClientTrusted(chain, authType);
            }
        };
    }

    private static void setSystemTrustManager(X509TrustManager mergedTrustManager) throws NoSuchAlgorithmException, KeyManagementException {
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, new TrustManager[]{mergedTrustManager}, null);
        SSLContext.setDefault(sslContext);
        HttpsURLConnection.setDefaultSSLSocketFactory(sslContext.getSocketFactory());
    }
}

