/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.pdf.search;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.en.EnglishAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexNotFoundException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.NIOFSDirectory;
import org.jabref.logic.pdf.search.DocumentReader;
import org.jabref.logic.util.StandardFileType;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.LinkedFile;
import org.jabref.preferences.FilePreferences;
import org.jooq.lambda.Unchecked;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PdfIndexer {
    private static final Logger LOGGER = LoggerFactory.getLogger(PdfIndexer.class);
    @VisibleForTesting
    @Nullable IndexWriter indexWriter;
    private final BibDatabaseContext databaseContext;
    private final FilePreferences filePreferences;
    private final @Nullable Directory indexDirectory;
    private IndexReader reader;

    private PdfIndexer(BibDatabaseContext databaseContext, Directory indexDirectory, FilePreferences filePreferences) {
        this.databaseContext = databaseContext;
        if (indexDirectory == null) {
            String tmpDir = System.getProperty("java.io.tmpdir");
            LOGGER.info("Index directory must not be null. Falling back to {}", (Object)tmpDir);
            NIOFSDirectory tmpIndexDirectory = null;
            try {
                tmpIndexDirectory = new NIOFSDirectory(Path.of(tmpDir, new String[0]));
            }
            catch (IOException e) {
                LOGGER.info("Could not use {}. Indexing unavailable.", (Object)tmpDir, (Object)e);
            }
            this.indexDirectory = tmpIndexDirectory;
        } else {
            this.indexDirectory = indexDirectory;
        }
        this.filePreferences = filePreferences;
    }

    @VisibleForTesting
    public static PdfIndexer of(BibDatabaseContext databaseContext, Path indexDirectory, FilePreferences filePreferences) throws IOException {
        return new PdfIndexer(databaseContext, (Directory)new NIOFSDirectory(indexDirectory), filePreferences);
    }

    public static PdfIndexer of(BibDatabaseContext databaseContext, FilePreferences filePreferences) throws IOException {
        return new PdfIndexer(databaseContext, (Directory)new NIOFSDirectory(databaseContext.getFulltextIndexPath()), filePreferences);
    }

    public void createIndex() {
        if (this.indexDirectory == null) {
            LOGGER.info("Index directory must not be null. Returning.");
            return;
        }
        LOGGER.debug("Creating new index for directory {}.", (Object)this.indexDirectory);
        this.initializeIndexWriterAndReader(IndexWriterConfig.OpenMode.CREATE);
    }

    Optional<IndexWriter> getIndexWriter() {
        LOGGER.trace("Getting the index writer");
        if (this.indexWriter == null) {
            LOGGER.trace("Initializing the index writer");
            this.initializeIndexWriterAndReader(IndexWriterConfig.OpenMode.CREATE_OR_APPEND);
        } else {
            LOGGER.trace("Using existing index writer");
        }
        return Optional.ofNullable(this.indexWriter);
    }

    private void initializeIndexWriterAndReader(IndexWriterConfig.OpenMode mode) {
        if (this.indexDirectory == null) {
            LOGGER.info("Index directory must not be null. Returning.");
            return;
        }
        try {
            this.indexWriter = new IndexWriter(this.indexDirectory, new IndexWriterConfig((Analyzer)new EnglishAnalyzer()).setOpenMode(mode));
        }
        catch (IOException e) {
            LOGGER.error("Could not initialize the IndexWriter", (Throwable)e);
            return;
        }
        try {
            this.reader = DirectoryReader.open((IndexWriter)this.indexWriter);
        }
        catch (IOException e) {
            LOGGER.error("Could not initialize the IndexReader", (Throwable)e);
        }
    }

    public void rebuildIndex() {
        LOGGER.debug("Rebuilding index.");
        this.createIndex();
        this.addToIndex(this.databaseContext.getEntries());
    }

    public void addToIndex(List<BibEntry> entries) {
        int count = 0;
        for (BibEntry entry : entries) {
            this.addToIndex(entry, false);
            if (++count % 100 != 0) continue;
            this.doCommit();
        }
        this.doCommit();
        LOGGER.debug("Added {} documents to the index.", (Object)count);
    }

    public void addToIndex(BibEntry entry) {
        this.addToIndex(entry, entry.getFiles(), true);
    }

    private void addToIndex(BibEntry entry, boolean shouldCommit) {
        this.addToIndex(entry, entry.getFiles(), false);
        if (shouldCommit) {
            this.doCommit();
        }
    }

    public void addToIndex(BibEntry entry, Collection<LinkedFile> linkedFiles) {
        this.addToIndex(entry, linkedFiles, true);
    }

    public void addToIndex(BibEntry entry, Collection<LinkedFile> linkedFiles, boolean shouldCommit) {
        for (LinkedFile linkedFile : linkedFiles) {
            this.addToIndex(entry, linkedFile, false);
        }
        if (shouldCommit) {
            this.doCommit();
        }
    }

    private void doCommit() {
        try {
            this.getIndexWriter().ifPresent(Unchecked.consumer(IndexWriter::commit));
        }
        catch (UncheckedIOException e) {
            LOGGER.warn("Could not commit changes to the index.", (Throwable)e);
        }
    }

    public void removeFromIndex(String linkedFilePath) {
        try {
            this.getIndexWriter().ifPresent(Unchecked.consumer(writer -> {
                writer.deleteDocuments(new Term[]{new Term("path", linkedFilePath)});
                writer.commit();
            }));
        }
        catch (UncheckedIOException e) {
            LOGGER.debug("Could not remove document {} from the index.", (Object)linkedFilePath, (Object)e);
        }
    }

    public void removeFromIndex(BibEntry entry) {
        this.removeFromIndex(entry.getFiles());
    }

    public void removeFromIndex(Collection<LinkedFile> linkedFiles) {
        for (LinkedFile linkedFile : linkedFiles) {
            this.removeFromIndex(linkedFile.getLink());
        }
    }

    public void removePathsFromIndex(Collection<String> linkedFiles) {
        for (String linkedFile : linkedFiles) {
            this.removeFromIndex(linkedFile);
        }
    }

    public void addToIndex(BibEntry entry, LinkedFile linkedFile) {
        this.addToIndex(entry, linkedFile, true);
    }

    private void addToIndex(BibEntry entry, LinkedFile linkedFile, boolean shouldCommit) {
        if (linkedFile.isOnlineLink() || !StandardFileType.PDF.getName().equals(linkedFile.getFileType()) && !linkedFile.getLink().endsWith(".pdf") && !linkedFile.getLink().endsWith(".PDF")) {
            return;
        }
        Optional<Path> resolvedPath = linkedFile.findIn(this.databaseContext, this.filePreferences);
        if (resolvedPath.isEmpty()) {
            LOGGER.debug("Could not find {}", (Object)linkedFile.getLink());
            return;
        }
        try {
            try {
                IndexSearcher searcher = new IndexSearcher(this.reader);
                TermQuery query = new TermQuery(new Term("path", linkedFile.getLink()));
                TopDocs topDocs = searcher.search((Query)query, 1);
                if (topDocs.scoreDocs.length > 0) {
                    BasicFileAttributes attributes;
                    Document doc = this.reader.storedFields().document(topDocs.scoreDocs[0].doc);
                    long indexModificationTime = Long.parseLong(doc.getField("modified").stringValue());
                    if (indexModificationTime >= (attributes = Files.readAttributes(resolvedPath.get(), BasicFileAttributes.class, new LinkOption[0])).lastModifiedTime().to(TimeUnit.SECONDS)) {
                        LOGGER.debug("File {} is already indexed and up-to-date.", (Object)linkedFile.getLink());
                        return;
                    }
                    LOGGER.debug("File {} is already indexed but outdated. Removing from index.", (Object)linkedFile.getLink());
                    this.removeFromIndex(linkedFile.getLink());
                }
            }
            catch (IndexNotFoundException e) {
                LOGGER.debug("Index not found. Continuing.", (Throwable)e);
            }
            LOGGER.debug("Adding {} to index", (Object)linkedFile.getLink());
            Optional<List<Document>> pages = new DocumentReader(entry, this.filePreferences).readLinkedPdf(this.databaseContext, linkedFile);
            if (pages.isPresent()) {
                this.getIndexWriter().ifPresent(Unchecked.consumer(writer -> {
                    writer.addDocuments((Iterable)pages.get());
                    if (shouldCommit) {
                        writer.commit();
                    }
                }));
            } else {
                LOGGER.debug("No content found in file {}", (Object)linkedFile.getLink());
            }
        }
        catch (IOException | UncheckedIOException e) {
            LOGGER.warn("Could not add document {} to the index.", (Object)linkedFile.getLink(), (Object)e);
        }
    }

    public Set<String> getListOfFilePaths() {
        HashSet<String> paths = new HashSet<String>();
        Optional<IndexWriter> optionalIndexWriter = this.getIndexWriter();
        if (optionalIndexWriter.isEmpty()) {
            LOGGER.debug("IndexWriter is empty. Returning empty list.");
            return paths;
        }
        try (DirectoryReader reader = DirectoryReader.open((IndexWriter)optionalIndexWriter.get());){
            IndexSearcher searcher = new IndexSearcher((IndexReader)reader);
            MatchAllDocsQuery query = new MatchAllDocsQuery();
            TopDocs allDocs = searcher.search((Query)query, Integer.MAX_VALUE);
            for (ScoreDoc scoreDoc : allDocs.scoreDocs) {
                Document doc = reader.storedFields().document(scoreDoc.doc);
                paths.add(doc.getField("path").stringValue());
            }
        }
        catch (IOException e) {
            LOGGER.debug("Could not read from index. Returning intermediate result.", (Throwable)e);
            return paths;
        }
        return paths;
    }

    public void close() throws IOException {
        if (this.indexWriter == null) {
            LOGGER.debug("IndexWriter is null.");
            return;
        }
        this.indexWriter.close();
    }
}

