/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.util.io;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jabref.logic.citationkeypattern.BracketedPattern;
import org.jabref.logic.util.StandardFileType;
import org.jabref.logic.util.io.FileNameCleaner;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.preferences.FilePreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtil {
    public static final boolean IS_POSIX_COMPLIANT = FileSystems.getDefault().supportedFileAttributeViews().contains("posix");
    public static final int MAXIMUM_FILE_NAME_LENGTH = 255;
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtil.class);
    private static final int[] ILLEGAL_CHARS = new int[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 34, 42, 58, 60, 62, 63, 123, 124, 125};

    private FileUtil() {
    }

    public static Optional<String> getFileExtension(String fileName) {
        int dotPosition = fileName.lastIndexOf(46);
        if (dotPosition > 0 && dotPosition < fileName.length() - 1) {
            return Optional.of(fileName.substring(dotPosition + 1).trim().toLowerCase(Locale.ROOT));
        }
        return Optional.empty();
    }

    public static Optional<String> getFileExtension(Path file) {
        return FileUtil.getFileExtension(file.getFileName().toString());
    }

    public static String getBaseName(String fileNameWithExtension) {
        return com.google.common.io.Files.getNameWithoutExtension((String)fileNameWithExtension);
    }

    public static String getBaseName(Path fileNameWithExtension) {
        return FileUtil.getBaseName(fileNameWithExtension.getFileName().toString());
    }

    public static String getValidFileName(String fileName) {
        String nameWithoutExtension = FileUtil.getBaseName(fileName);
        if (nameWithoutExtension.length() > 255) {
            Optional<String> extension = FileUtil.getFileExtension(fileName);
            String shortName = nameWithoutExtension.substring(0, 255 - extension.map(s -> s.length() + 1).orElse(0));
            LOGGER.info("Truncated the too long filename '%s' (%d characters) to '%s'.".formatted(fileName, fileName.length(), shortName));
            return extension.map(s -> shortName + "." + s).orElse(shortName);
        }
        return fileName;
    }

    public static Path addExtension(Path path, String extension) {
        return path.resolveSibling(String.valueOf(path.getFileName()) + extension);
    }

    public static Optional<String> getUniquePathDirectory(List<String> paths, Path comparePath) {
        String fileName = comparePath.getFileName().toString();
        List<String> uniquePathParts = FileUtil.uniquePathSubstrings(paths);
        return uniquePathParts.stream().filter(part -> comparePath.toString().contains((CharSequence)part) && !part.equals(fileName) && part.contains(File.separator)).findFirst().map(part -> part.substring(0, part.lastIndexOf(File.separator)));
    }

    public static Optional<String> getUniquePathFragment(List<String> paths, Path comparePath) {
        return FileUtil.uniquePathSubstrings(paths).stream().filter(part -> comparePath.toString().contains((CharSequence)part)).max(Comparator.comparingInt(String::length));
    }

    public static List<String> uniquePathSubstrings(List<String> paths) {
        Deque stack;
        ArrayList stackList = new ArrayList(paths.size());
        for (String path : paths) {
            List<String> directories = Arrays.asList(path.split(Pattern.quote(File.separator)));
            stack = new ArrayDeque(directories.reversed());
            stackList.add(stack);
        }
        ArrayList<String> pathSubstrings = new ArrayList<String>(Collections.nCopies(paths.size(), ""));
        while (!stackList.stream().allMatch(Collection::isEmpty)) {
            int i;
            for (i = 0; i < stackList.size(); ++i) {
                String tempPathString = (String)pathSubstrings.get(i);
                stack = (Deque)stackList.get(i);
                if (tempPathString.isEmpty() && !stack.isEmpty()) {
                    String stringFromDeque = (String)stack.pop();
                    pathSubstrings.set(i, stringFromDeque);
                    continue;
                }
                if (stack.isEmpty()) continue;
                String stringFromStack = (String)stack.pop();
                pathSubstrings.set(i, stringFromStack + File.separator + tempPathString);
            }
            for (i = 0; i < stackList.size(); ++i) {
                String tempString = (String)pathSubstrings.get(i);
                if (Collections.frequency(pathSubstrings, tempString) != 1) continue;
                ((Deque)stackList.get(i)).clear();
            }
        }
        return pathSubstrings;
    }

    public static boolean copyFile(Path pathToSourceFile, Path pathToDestinationFile, boolean replaceExisting) {
        if (!Files.exists(pathToSourceFile, new LinkOption[0])) {
            LOGGER.error("Path to the source file doesn't exist.");
            return false;
        }
        if (Files.exists(pathToDestinationFile, new LinkOption[0]) && !replaceExisting) {
            LOGGER.error("Path to the destination file exists but the file shouldn't be replaced.");
            return false;
        }
        try {
            Files.write(pathToDestinationFile, Files.readAllBytes(pathToSourceFile), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);
            return true;
        }
        catch (IOException e) {
            LOGGER.error("Copying Files failed.", (Throwable)e);
            return false;
        }
    }

    public static Path relativize(Path file, List<Path> directories) {
        if (!file.isAbsolute()) {
            return file;
        }
        for (Path directory : directories) {
            if (!file.startsWith(directory)) continue;
            return directory.relativize(file);
        }
        return file;
    }

    public static Path relativize(Path path, BibDatabaseContext databaseContext, FilePreferences filePreferences) {
        List<Path> fileDirectories = databaseContext.getFileDirectories(filePreferences);
        return FileUtil.relativize(path, fileDirectories);
    }

    public static List<Path> getListOfLinkedFiles(List<BibEntry> bes, List<Path> fileDirs) {
        Objects.requireNonNull(bes);
        Objects.requireNonNull(fileDirs);
        return bes.stream().flatMap(entry -> entry.getFiles().stream()).flatMap(file -> file.findIn(fileDirs).stream()).collect(Collectors.toList());
    }

    public static String createFileNameFromPattern(BibDatabase database, BibEntry entry, String fileNamePattern) {
        String targetName = BracketedPattern.expandBrackets(fileNamePattern, Character.valueOf(';'), entry, database);
        if (targetName.isEmpty()) {
            targetName = entry.getCitationKey().orElse("default");
        }
        targetName = FileNameCleaner.cleanFileName(targetName);
        return targetName;
    }

    public static String createDirNameFromPattern(BibDatabase database, BibEntry entry, String directoryNamePattern) {
        String targetName = BracketedPattern.expandBrackets(directoryNamePattern, Character.valueOf(';'), entry, database);
        if (targetName.isEmpty()) {
            return targetName;
        }
        targetName = FileNameCleaner.cleanDirectoryName(targetName);
        return targetName;
    }

    public static Optional<Path> findSingleFileRecursively(String filename, Path rootDirectory) {
        Optional<Path> optional;
        block8: {
            Stream<Path> pathStream = Files.walk(rootDirectory, new FileVisitOption[0]);
            try {
                optional = pathStream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(f -> f.getFileName().toString().equals(filename)).findFirst();
                if (pathStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (pathStream != null) {
                        try {
                            pathStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException | UncheckedIOException ex) {
                    LOGGER.error("Error trying to locate the file " + filename + " inside the directory " + String.valueOf(rootDirectory));
                    return Optional.empty();
                }
            }
            pathStream.close();
        }
        return optional;
    }

    public static Optional<Path> find(BibDatabaseContext databaseContext, String fileName, FilePreferences filePreferences) {
        Objects.requireNonNull(fileName, "fileName");
        return FileUtil.find(fileName, databaseContext.getFileDirectories(filePreferences));
    }

    public static Optional<Path> find(String fileName, List<Path> directories) {
        if (directories.isEmpty()) {
            Path path = Path.of(fileName, new String[0]);
            if (path.isAbsolute()) {
                return Optional.of(path);
            }
            return Optional.empty();
        }
        return directories.stream().flatMap(directory -> FileUtil.find(fileName, directory).stream()).findFirst();
    }

    public static Optional<Path> find(String fileName, Path directory) {
        Objects.requireNonNull(fileName);
        Objects.requireNonNull(directory);
        if (FileUtil.detectBadFileName(fileName)) {
            LOGGER.error("Invalid characters in path for file {}", (Object)fileName);
            return Optional.empty();
        }
        if (fileName.isEmpty()) {
            return Optional.of(directory);
        }
        Path resolvedFile = directory.resolve(fileName);
        if (Files.exists(resolvedFile, new LinkOption[0])) {
            return Optional.of(resolvedFile);
        }
        String furthestDirFromRoot = directory.getFileName().toString();
        if (fileName.startsWith(furthestDirFromRoot)) {
            resolvedFile = directory.resolveSibling(fileName);
        }
        if (Files.exists(resolvedFile, new LinkOption[0])) {
            return Optional.of(resolvedFile);
        }
        return Optional.empty();
    }

    public static List<Path> findListOfFiles(String filename, List<Path> directories) {
        ArrayList<Path> files = new ArrayList<Path>();
        for (Path dir : directories) {
            FileUtil.find(filename, dir).ifPresent(files::add);
        }
        return files;
    }

    public static String toPortableString(Path path) {
        return path.toString().replace('\\', '/');
    }

    public static boolean isBibFile(Path file) {
        return FileUtil.getFileExtension(file).filter("bib"::equals).isPresent();
    }

    public static boolean isPDFFile(Path file) {
        Optional<String> extension = FileUtil.getFileExtension(file);
        return extension.isPresent() && StandardFileType.PDF.getExtensions().contains(extension.get());
    }

    public static Path getInitialDirectory(BibDatabaseContext databaseContext, Path workingDirectory) {
        return databaseContext.getDatabasePath().map(Path::getParent).orElse(workingDirectory);
    }

    public static boolean detectBadFileName(String fileName) {
        try {
            fileName = Path.of(fileName, new String[0]).getFileName().toString();
        }
        catch (InvalidPathException e) {
            return true;
        }
        for (int i = 0; i < fileName.length(); ++i) {
            char c = fileName.charAt(i);
            if (FileUtil.isCharLegal(c)) continue;
            return true;
        }
        return false;
    }

    public static boolean isCharLegal(char c) {
        return Arrays.binarySearch(ILLEGAL_CHARS, (int)c) < 0;
    }
}

