/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.xmp;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.TreeSet;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.apache.xmpbox.schema.DublinCoreSchema;
import org.apache.xmpbox.type.BadFieldValueException;
import org.jabref.logic.bibtex.TypedBibEntry;
import org.jabref.logic.formatter.casechanger.UnprotectTermsFormatter;
import org.jabref.logic.xmp.XmpPreferences;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.Author;
import org.jabref.model.entry.AuthorList;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.Date;
import org.jabref.model.entry.Month;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.FieldFactory;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.field.UnknownField;
import org.jabref.model.entry.types.EntryTypeFactory;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DublinCoreExtractor {
    public static final String DC_COVERAGE = "coverage";
    public static final String DC_RIGHTS = "rights";
    public static final String DC_SOURCE = "source";
    private static final Logger LOGGER = LoggerFactory.getLogger(DublinCoreExtractor.class);
    private final DublinCoreSchema dcSchema;
    private final XmpPreferences xmpPreferences;
    private final BibEntry bibEntry;
    private final UnprotectTermsFormatter unprotectTermsFormatter = new UnprotectTermsFormatter();

    public DublinCoreExtractor(DublinCoreSchema dcSchema, XmpPreferences xmpPreferences, BibEntry resolvedEntry) {
        this.dcSchema = dcSchema;
        this.xmpPreferences = xmpPreferences;
        this.bibEntry = resolvedEntry;
    }

    private void extractEditor() {
        List contributors = this.dcSchema.getContributors();
        if (contributors != null && !contributors.isEmpty()) {
            this.bibEntry.setField(StandardField.EDITOR, String.join((CharSequence)" and ", contributors));
        }
    }

    private void extractAuthor() {
        List creators = this.dcSchema.getCreators();
        if (creators != null && !creators.isEmpty()) {
            this.bibEntry.setField(StandardField.AUTHOR, String.join((CharSequence)" and ", creators));
        }
    }

    private void extractDate() {
        List dates = this.dcSchema.getUnqualifiedSequenceValueList("date");
        if (dates != null && !dates.isEmpty()) {
            String date = ((String)dates.getFirst()).trim();
            Date.parse(date).ifPresent(dateValue -> {
                dateValue.getDay().ifPresent(day -> this.bibEntry.setField(StandardField.DAY, Integer.toString(day)));
                dateValue.getMonth().ifPresent(this.bibEntry::setMonth);
                dateValue.getYear().ifPresent(year -> this.bibEntry.setField(StandardField.YEAR, Integer.toString(year)));
            });
        }
    }

    private void extractAbstract() {
        String description = null;
        try {
            description = this.dcSchema.getDescription();
        }
        catch (BadFieldValueException e) {
            LOGGER.warn("Could not get abstract", (Throwable)e);
        }
        if (!StringUtil.isNullOrEmpty(description)) {
            this.bibEntry.setField(StandardField.ABSTRACT, description);
        }
    }

    private void extractDOI() {
        String identifier = this.dcSchema.getIdentifier();
        if (!StringUtil.isNullOrEmpty(identifier)) {
            this.bibEntry.setField(StandardField.DOI, identifier);
        }
    }

    private void extractPublisher() {
        List publishers = this.dcSchema.getPublishers();
        if (publishers != null && !publishers.isEmpty()) {
            this.bibEntry.setField(StandardField.PUBLISHER, String.join((CharSequence)" and ", publishers));
        }
    }

    private void extractBibTexFields() {
        Predicate<String> isBibTeXElement = s -> s.startsWith("bibtex/");
        Consumer<String> splitBibTeXElement = s -> {
            String temp = s.substring("bibtex/".length());
            int i = temp.indexOf(47);
            if (i != -1) {
                Field key = FieldFactory.parseField(temp.substring(0, i));
                String value = temp.substring(i + 1);
                this.bibEntry.setField(key, value);
                if (StandardField.MONTH == key) {
                    Optional<Month> parsedMonth = Month.parse(value);
                    parsedMonth.ifPresent(this.bibEntry::setMonth);
                }
            }
        };
        List relationships = this.dcSchema.getRelations();
        if (relationships != null) {
            relationships.stream().filter(isBibTeXElement).forEach(splitBibTeXElement);
        }
    }

    private void extractRights() {
        String rights = null;
        try {
            rights = this.dcSchema.getRights();
        }
        catch (BadFieldValueException e) {
            LOGGER.warn("Could not extract rights", (Throwable)e);
        }
        if (!StringUtil.isNullOrEmpty(rights)) {
            this.bibEntry.setField(new UnknownField(DC_RIGHTS), rights);
        }
    }

    private void extractSource() {
        String source = this.dcSchema.getSource();
        if (!StringUtil.isNullOrEmpty(source)) {
            this.bibEntry.setField(new UnknownField(DC_SOURCE), source);
        }
    }

    private void extractSubject() {
        List subjects = this.dcSchema.getSubjects();
        if (subjects != null && !subjects.isEmpty()) {
            this.bibEntry.addKeywords(subjects, this.xmpPreferences.getKeywordSeparator());
        }
    }

    private void extractTitle() {
        String title = null;
        try {
            title = this.dcSchema.getTitle();
        }
        catch (BadFieldValueException e) {
            LOGGER.warn("Could not extract title", (Throwable)e);
        }
        if (!StringUtil.isNullOrEmpty(title)) {
            this.bibEntry.setField(StandardField.TITLE, title);
        }
    }

    private void extractType() {
        String type;
        List types = this.dcSchema.getTypes();
        if (types != null && !types.isEmpty() && !StringUtil.isNullOrEmpty(type = (String)types.getFirst())) {
            this.bibEntry.setType(EntryTypeFactory.parse(type));
        }
    }

    private void extractCoverage() {
        String coverage = this.dcSchema.getCoverage();
        if (!StringUtil.isNullOrEmpty(coverage)) {
            this.bibEntry.setField(FieldFactory.parseField(DC_COVERAGE), coverage);
        }
    }

    private void extractLanguages() {
        StringBuilder builder = new StringBuilder();
        List languages = this.dcSchema.getLanguages();
        if (languages != null && !languages.isEmpty()) {
            languages.forEach(language -> builder.append(",").append((String)language));
            this.bibEntry.setField(StandardField.LANGUAGE, builder.substring(1));
        }
    }

    public Optional<BibEntry> extractBibtexEntry() {
        this.extractBibTexFields();
        this.extractType();
        this.extractEditor();
        this.extractAuthor();
        this.extractDate();
        this.extractAbstract();
        this.extractDOI();
        this.extractPublisher();
        this.extractRights();
        this.extractSource();
        this.extractSubject();
        this.extractTitle();
        this.extractCoverage();
        this.extractLanguages();
        if (this.bibEntry.getFieldMap().isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(this.bibEntry);
    }

    private void fillContributor(String authors) {
        AuthorList list = AuthorList.parse(authors);
        for (Author author : list.getAuthors()) {
            this.dcSchema.addContributor(author.getGivenFamily(false));
        }
    }

    private void fillCreator(String creators) {
        AuthorList list = AuthorList.parse(creators);
        for (Author author : list.getAuthors()) {
            this.dcSchema.addCreator(author.getGivenFamily(false));
        }
    }

    private void fillDate() {
        this.bibEntry.getFieldOrAlias(StandardField.DATE).ifPresent(publicationDate -> this.dcSchema.addUnqualifiedSequenceValue("date", publicationDate));
    }

    private void fillDescription(String description) {
        this.dcSchema.setDescription(description);
    }

    private void fillIdentifier(String identifier) {
        this.dcSchema.setIdentifier(identifier);
    }

    private void fillPublisher(String publisher) {
        this.dcSchema.addPublisher(publisher);
    }

    private void fillKeywords(String value) {
        String[] keywords;
        for (String keyword : keywords = value.split(this.xmpPreferences.getKeywordSeparator().toString())) {
            this.dcSchema.addSubject(keyword.trim());
        }
    }

    private void fillTitle(String title) {
        this.dcSchema.setTitle(title);
    }

    private void fillCoverage(String coverage) {
        this.dcSchema.setCoverage(coverage);
    }

    private void fillLanguages(String languages) {
        Arrays.stream(languages.split(",")).forEach(arg_0 -> ((DublinCoreSchema)this.dcSchema).addLanguage(arg_0));
    }

    private void fillRights(String rights) {
        this.dcSchema.addRights(null, rights.split(",")[0]);
    }

    private void fillSource(String source) {
        this.dcSchema.setSource(source);
    }

    private void fillCustomField(Field field) {
        String value = this.bibEntry.getField(field).get();
        this.dcSchema.addRelation("bibtex/" + field.getName() + "/" + value);
    }

    private void fillType() {
        TypedBibEntry typedEntry = new TypedBibEntry(this.bibEntry, BibDatabaseMode.BIBTEX);
        String typeForDisplay = typedEntry.getTypeForDisplay();
        if (!typeForDisplay.isEmpty()) {
            this.dcSchema.addType(typeForDisplay);
        }
    }

    public void fillDublinCoreSchema() {
        boolean useXmpPrivacyFilter = this.xmpPreferences.shouldUseXmpPrivacyFilter();
        TreeSet<Field> fields = new TreeSet<Field>(Comparator.comparing(Field::getName));
        fields.addAll(this.bibEntry.getFields());
        block13: for (Field field : fields) {
            if (useXmpPrivacyFilter && this.xmpPreferences.getXmpPrivacyFilter().contains((Object)field)) continue;
            String value = this.unprotectTermsFormatter.format(this.bibEntry.getField(field).get());
            if (field instanceof StandardField) {
                StandardField standardField = (StandardField)field;
                switch (standardField) {
                    case EDITOR: {
                        this.fillContributor(value);
                        continue block13;
                    }
                    case AUTHOR: {
                        this.fillCreator(value);
                        continue block13;
                    }
                    case YEAR: {
                        this.fillDate();
                        continue block13;
                    }
                    case ABSTRACT: {
                        this.fillDescription(value);
                        continue block13;
                    }
                    case DOI: {
                        this.fillIdentifier(value);
                        continue block13;
                    }
                    case PUBLISHER: {
                        this.fillPublisher(value);
                        continue block13;
                    }
                    case KEYWORDS: {
                        this.fillKeywords(value);
                        continue block13;
                    }
                    case TITLE: {
                        this.fillTitle(value);
                        continue block13;
                    }
                    case LANGUAGE: {
                        this.fillLanguages(value);
                        continue block13;
                    }
                    case FILE: {
                        continue block13;
                    }
                    case DAY: 
                    case MONTH: {
                        if (this.bibEntry.hasField(StandardField.YEAR)) continue block13;
                        this.fillCustomField(field);
                        continue block13;
                    }
                }
                this.fillCustomField(field);
                continue;
            }
            if (DC_COVERAGE.equals(field.getName())) {
                this.fillCoverage(value);
                continue;
            }
            if (DC_RIGHTS.equals(field.getName())) {
                this.fillRights(value);
                continue;
            }
            if (DC_SOURCE.equals(field.getName())) {
                this.fillSource(value);
                continue;
            }
            this.fillCustomField(field);
        }
        this.dcSchema.setFormat("application/pdf");
        this.fillType();
    }
}

