/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.model.database;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.jabref.model.database.KeyChangeListener;
import org.jabref.model.database.KeyCollisionException;
import org.jabref.model.database.event.EntriesAddedEvent;
import org.jabref.model.database.event.EntriesRemovedEvent;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibtexString;
import org.jabref.model.entry.Month;
import org.jabref.model.entry.event.EntriesEventSource;
import org.jabref.model.entry.event.FieldChangedEvent;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.FieldFactory;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BibDatabase {
    private static final Logger LOGGER = LoggerFactory.getLogger(BibDatabase.class);
    private static final Pattern RESOLVE_CONTENT_PATTERN = Pattern.compile(".*#[^#]+#.*");
    private final ObservableList<BibEntry> entries = FXCollections.synchronizedObservableList((ObservableList)FXCollections.observableArrayList(BibEntry::getObservables));
    private Map<String, BibtexString> bibtexStrings = new ConcurrentHashMap<String, BibtexString>();
    private final EventBus eventBus = new EventBus();
    private String preamble;
    private String epilog = "";
    private String sharedDatabaseID;
    private String newLineSeparator = System.lineSeparator();

    public BibDatabase(List<BibEntry> entries, String newLineSeparator) {
        this(entries);
        this.newLineSeparator = newLineSeparator;
    }

    public BibDatabase(List<BibEntry> entries) {
        this();
        this.insertEntries(entries);
    }

    public BibDatabase() {
        this.registerListener(new KeyChangeListener(this));
    }

    @Deprecated
    public static String getText(String toResolve, BibDatabase database) {
        if (toResolve != null && database != null) {
            return database.resolveForStrings(toResolve);
        }
        return toResolve;
    }

    public int getEntryCount() {
        return this.entries.size();
    }

    public boolean hasEntries() {
        return !this.entries.isEmpty();
    }

    public List<BibEntry> getEntriesSorted(Comparator<BibEntry> comparator) {
        ArrayList<BibEntry> entriesSorted = new ArrayList<BibEntry>((Collection<BibEntry>)this.entries);
        entriesSorted.sort(comparator);
        return entriesSorted;
    }

    public boolean containsEntryWithId(String id) {
        return this.entries.stream().anyMatch(entry -> entry.getId().equals(id));
    }

    public ObservableList<BibEntry> getEntries() {
        return FXCollections.unmodifiableObservableList(this.entries);
    }

    public Set<Field> getAllVisibleFields() {
        TreeSet<Field> allFields = new TreeSet<Field>(Comparator.comparing(Field::getName));
        for (BibEntry e : this.getEntries()) {
            allFields.addAll(e.getFields());
        }
        return allFields.stream().filter(field -> !FieldFactory.isInternalField(field)).collect(Collectors.toSet());
    }

    public synchronized Optional<BibEntry> getEntryByCitationKey(String key) {
        return this.entries.stream().filter(entry -> Objects.equals(entry.getCitationKey().orElse(null), key)).findFirst();
    }

    public synchronized List<BibEntry> getEntriesByCitationKey(String key) {
        ArrayList<BibEntry> result = new ArrayList<BibEntry>();
        for (BibEntry entry : this.entries) {
            entry.getCitationKey().ifPresent(entryKey -> {
                if (key.equals(entryKey)) {
                    result.add(entry);
                }
            });
        }
        return result;
    }

    public synchronized void insertEntry(BibEntry entry) {
        this.insertEntry(entry, EntriesEventSource.LOCAL);
    }

    public synchronized void insertEntry(BibEntry entry, EntriesEventSource eventSource) {
        this.insertEntries(Collections.singletonList(entry), eventSource);
    }

    public synchronized void insertEntries(BibEntry ... entries) {
        this.insertEntries(Arrays.asList(entries), EntriesEventSource.LOCAL);
    }

    public synchronized void insertEntries(List<BibEntry> entries) {
        this.insertEntries(entries, EntriesEventSource.LOCAL);
    }

    public synchronized void insertEntries(List<BibEntry> newEntries, EntriesEventSource eventSource) {
        Objects.requireNonNull(newEntries);
        for (BibEntry entry : newEntries) {
            entry.registerListener(this);
        }
        if (newEntries.isEmpty()) {
            this.eventBus.post((Object)new EntriesAddedEvent(newEntries, eventSource));
        } else {
            this.eventBus.post((Object)new EntriesAddedEvent(newEntries, newEntries.getFirst(), eventSource));
        }
        this.entries.addAll(newEntries);
    }

    public synchronized void removeEntry(BibEntry bibEntry) {
        this.removeEntries(Collections.singletonList(bibEntry));
    }

    public synchronized void removeEntry(BibEntry bibEntry, EntriesEventSource eventSource) {
        this.removeEntries(Collections.singletonList(bibEntry), eventSource);
    }

    public synchronized void removeEntries(List<BibEntry> toBeDeleted) {
        this.removeEntries(toBeDeleted, EntriesEventSource.LOCAL);
    }

    public synchronized void removeEntries(List<BibEntry> toBeDeleted, EntriesEventSource eventSource) {
        Objects.requireNonNull(toBeDeleted);
        ArrayList<String> ids = new ArrayList<String>();
        for (BibEntry entry2 : toBeDeleted) {
            ids.add(entry2.getId());
        }
        boolean anyRemoved = this.entries.removeIf(entry -> ids.contains(entry.getId()));
        if (anyRemoved) {
            this.eventBus.post((Object)new EntriesRemovedEvent(toBeDeleted, eventSource));
        }
    }

    public synchronized Optional<String> getPreamble() {
        if (StringUtil.isBlank(this.preamble)) {
            return Optional.empty();
        }
        return Optional.of(this.preamble);
    }

    public synchronized void setPreamble(String preamble) {
        this.preamble = preamble;
    }

    public synchronized void addString(BibtexString string) throws KeyCollisionException {
        String id = string.getId();
        if (this.hasStringByName(string.getName())) {
            throw new KeyCollisionException("A string with that label already exists", id);
        }
        if (this.bibtexStrings.containsKey(id)) {
            throw new KeyCollisionException("Duplicate BibTeX string id.", id);
        }
        this.bibtexStrings.put(id, string);
    }

    public void setStrings(List<BibtexString> stringsToAdd) {
        this.bibtexStrings = new ConcurrentHashMap<String, BibtexString>();
        stringsToAdd.forEach(this::addString);
    }

    public void removeString(String id) {
        this.bibtexStrings.remove(id);
    }

    public Set<String> getStringKeySet() {
        return this.bibtexStrings.keySet();
    }

    public Collection<BibtexString> getStringValues() {
        return this.bibtexStrings.values();
    }

    public BibtexString getString(String id) {
        return this.bibtexStrings.get(id);
    }

    public Optional<BibtexString> getStringByName(String name) {
        return this.getStringValues().stream().filter(string -> string.getName().equals(name)).findFirst();
    }

    public int getStringCount() {
        return this.bibtexStrings.size();
    }

    public boolean hasNoStrings() {
        return this.bibtexStrings.isEmpty();
    }

    public void copyPreamble(BibDatabase database) {
        this.setPreamble(database.getPreamble().orElse(""));
    }

    public synchronized boolean hasStringByName(String label) {
        return this.bibtexStrings.values().stream().anyMatch(value -> value.getName().equals(label));
    }

    public String resolveForStrings(String content) {
        Objects.requireNonNull(content, "Content for resolveForStrings must not be null.");
        return this.resolveContent(content, new HashSet<String>(), new HashSet<String>());
    }

    public List<BibtexString> getUsedStrings(Collection<BibEntry> entries) {
        HashSet<String> allUsedIds = new HashSet<String>();
        if (this.preamble != null) {
            this.resolveContent(this.preamble, new HashSet<String>(), allUsedIds);
        }
        for (BibEntry entry : entries) {
            for (String fieldContent : entry.getFieldValues()) {
                this.resolveContent(fieldContent, new HashSet<String>(), allUsedIds);
            }
        }
        return allUsedIds.stream().map(this.bibtexStrings::get).toList();
    }

    public List<BibEntry> resolveForStrings(Collection<BibEntry> entriesToResolve, boolean inPlace) {
        Objects.requireNonNull(entriesToResolve, "entries must not be null.");
        ArrayList<BibEntry> results = new ArrayList<BibEntry>(entriesToResolve.size());
        for (BibEntry entry : entriesToResolve) {
            results.add(this.resolveForStrings(entry, inPlace));
        }
        return results;
    }

    public BibEntry resolveForStrings(BibEntry entry, boolean inPlace) {
        BibEntry resultingEntry = inPlace ? entry : (BibEntry)entry.clone();
        for (Map.Entry<Field, String> field : resultingEntry.getFieldMap().entrySet()) {
            resultingEntry.setField(field.getKey(), this.resolveForStrings(field.getValue()));
        }
        return resultingEntry;
    }

    private String resolveString(String label, Set<String> usedIds, Set<String> allUsedIds) {
        Objects.requireNonNull(label);
        Objects.requireNonNull(usedIds);
        Objects.requireNonNull(allUsedIds);
        for (BibtexString string : this.bibtexStrings.values()) {
            if (!string.getName().equalsIgnoreCase(label)) continue;
            if (usedIds.contains(string.getId())) {
                LOGGER.info("Stopped due to circular reference in strings: {}", (Object)label);
                return label;
            }
            usedIds.add(string.getId());
            if (allUsedIds != null) {
                allUsedIds.add(string.getId());
            }
            String result = string.getContent();
            result = this.resolveContent(result, usedIds, allUsedIds);
            usedIds.remove(string.getId());
            return result;
        }
        Optional<Month> month = Month.getMonthByShortName(label);
        return month.map(Month::getFullName).orElse(null);
    }

    private String resolveContent(String result, Set<String> usedIds, Set<String> allUsedIds) {
        String res = result;
        if (RESOLVE_CONTENT_PATTERN.matcher(res).matches()) {
            int next;
            StringBuilder newRes = new StringBuilder();
            int piv = 0;
            while ((next = res.indexOf(35, piv)) >= 0) {
                int stringEnd;
                if (next > 0) {
                    newRes.append(res, piv, next);
                }
                if ((stringEnd = res.indexOf(35, next + 1)) >= 0) {
                    String refLabel = res.substring(next + 1, stringEnd);
                    String resolved = this.resolveString(refLabel, usedIds, allUsedIds);
                    if (resolved == null) {
                        newRes.append(res, next, stringEnd + 1);
                    } else {
                        newRes.append(resolved);
                    }
                    piv = stringEnd + 1;
                    continue;
                }
                newRes.append(res.substring(next));
                piv = res.length();
                break;
            }
            if (piv < res.length() - 1) {
                newRes.append(res.substring(piv));
            }
            res = newRes.toString();
        }
        return res;
    }

    public String getEpilog() {
        return this.epilog;
    }

    public void setEpilog(String epilog) {
        this.epilog = epilog;
    }

    public void registerListener(Object listener) {
        this.eventBus.register(listener);
    }

    public void unregisterListener(Object listener) {
        try {
            this.eventBus.unregister(listener);
        }
        catch (IllegalArgumentException e) {
            LOGGER.debug("Problem unregistering", (Throwable)e);
        }
    }

    @Subscribe
    private void relayEntryChangeEvent(FieldChangedEvent event) {
        this.eventBus.post((Object)event);
    }

    public Optional<BibEntry> getReferencedEntry(BibEntry entry) {
        return entry.getField(StandardField.CROSSREF).flatMap(this::getEntryByCitationKey);
    }

    public Optional<String> getSharedDatabaseID() {
        return Optional.ofNullable(this.sharedDatabaseID);
    }

    public void setSharedDatabaseID(String sharedDatabaseID) {
        this.sharedDatabaseID = sharedDatabaseID;
    }

    public boolean isShared() {
        return this.getSharedDatabaseID().isPresent();
    }

    public void clearSharedDatabaseID() {
        this.sharedDatabaseID = null;
    }

    public String generateSharedDatabaseID() {
        this.sharedDatabaseID = new BigInteger(128, new SecureRandom()).toString(32);
        return this.sharedDatabaseID;
    }

    public long getNumberOfCitationKeyOccurrences(String key) {
        return this.entries.stream().flatMap(entry -> entry.getCitationKey().stream()).filter(key::equals).count();
    }

    public boolean isDuplicateCitationKeyExisting(String key) {
        return this.getNumberOfCitationKeyOccurrences(key) > 1L;
    }

    public void setNewLineSeparator(String newLineSeparator) {
        this.newLineSeparator = newLineSeparator;
    }

    public String getNewLineSeparator() {
        return this.newLineSeparator;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof BibDatabase)) {
            return false;
        }
        BibDatabase that = (BibDatabase)o;
        return Objects.equals(this.entries, that.entries) && Objects.equals(this.bibtexStrings, that.bibtexStrings) && Objects.equals(this.preamble, that.preamble) && Objects.equals(this.epilog, that.epilog) && Objects.equals(this.sharedDatabaseID, that.sharedDatabaseID) && Objects.equals(this.newLineSeparator, that.newLineSeparator);
    }

    public int hashCode() {
        return Objects.hash(this.entries, this.bibtexStrings, this.preamble, this.epilog, this.sharedDatabaseID, this.newLineSeparator);
    }
}

