/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.gui.frame;

import java.nio.file.Path;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableBooleanValue;
import javafx.scene.control.ButtonType;
import javax.swing.undo.UndoManager;
import org.jabref.gui.ClipBoardManager;
import org.jabref.gui.DialogService;
import org.jabref.gui.LibraryTab;
import org.jabref.gui.LibraryTabContainer;
import org.jabref.gui.StateManager;
import org.jabref.gui.frame.ProcessingLibraryDialog;
import org.jabref.gui.frame.UiMessageHandler;
import org.jabref.gui.importer.ImportEntriesDialog;
import org.jabref.gui.importer.ParserResultWarningDialog;
import org.jabref.gui.importer.actions.OpenDatabaseAction;
import org.jabref.gui.util.BackgroundTask;
import org.jabref.gui.util.TaskExecutor;
import org.jabref.logic.UiCommand;
import org.jabref.logic.importer.ImportCleanup;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.shared.DatabaseNotSupportedException;
import org.jabref.logic.shared.exception.InvalidDBMSConnectionPropertiesException;
import org.jabref.logic.shared.exception.NotASharedDatabaseException;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibEntryTypesManager;
import org.jabref.model.util.FileUpdateMonitor;
import org.jabref.preferences.PreferencesService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JabRefFrameViewModel
implements UiMessageHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(JabRefFrameViewModel.class);
    private final PreferencesService preferences;
    private final StateManager stateManager;
    private final DialogService dialogService;
    private final LibraryTabContainer tabContainer;
    private final BibEntryTypesManager entryTypesManager;
    private final FileUpdateMonitor fileUpdateMonitor;
    private final UndoManager undoManager;
    private final ClipBoardManager clipBoardManager;
    private final TaskExecutor taskExecutor;

    public JabRefFrameViewModel(PreferencesService preferencesService, StateManager stateManager, DialogService dialogService, LibraryTabContainer tabContainer, BibEntryTypesManager entryTypesManager, FileUpdateMonitor fileUpdateMonitor, UndoManager undoManager, ClipBoardManager clipBoardManager, TaskExecutor taskExecutor) {
        this.preferences = preferencesService;
        this.stateManager = stateManager;
        this.dialogService = dialogService;
        this.tabContainer = tabContainer;
        this.entryTypesManager = entryTypesManager;
        this.fileUpdateMonitor = fileUpdateMonitor;
        this.undoManager = undoManager;
        this.clipBoardManager = clipBoardManager;
        this.taskExecutor = taskExecutor;
    }

    void storeLastOpenedFiles(List<Path> filenames, Path focusedDatabase) {
        if (this.preferences.getWorkspacePreferences().shouldOpenLastEdited()) {
            if (filenames.isEmpty()) {
                this.preferences.getGuiPreferences().getLastFilesOpened().clear();
            } else {
                this.preferences.getGuiPreferences().setLastFilesOpened(filenames);
                this.preferences.getGuiPreferences().setLastFocusedFile(focusedDatabase);
            }
        }
    }

    public boolean close() {
        Optional<ButtonType> shouldClose;
        if (((Boolean)this.stateManager.getAnyTasksThatWillNotBeRecoveredRunning().getValue()).booleanValue() && (!(shouldClose = this.dialogService.showBackgroundProgressDialogAndWait(Localization.lang("Please wait...", new Object[0]), Localization.lang("Waiting for background tasks to finish. Quit anyway?", new Object[0]), this.stateManager)).isPresent() || shouldClose.get() != ButtonType.YES)) {
            return false;
        }
        List<Path> openedLibraries = this.tabContainer.getLibraryTabs().stream().map(LibraryTab::getBibDatabaseContext).map(BibDatabaseContext::getDatabasePath).flatMap(Optional::stream).toList();
        Path focusedLibraries = Optional.ofNullable(this.tabContainer.getCurrentLibraryTab()).map(LibraryTab::getBibDatabaseContext).flatMap(BibDatabaseContext::getDatabasePath).orElse(null);
        if (!this.tabContainer.closeTabs(this.tabContainer.getLibraryTabs())) {
            return false;
        }
        this.storeLastOpenedFiles(openedLibraries, focusedLibraries);
        ProcessingLibraryDialog processingLibraryDialog = new ProcessingLibraryDialog(this.dialogService);
        processingLibraryDialog.showAndWait(this.tabContainer.getLibraryTabs());
        return true;
    }

    @Override
    public void handleUiCommands(List<UiCommand> uiCommands) {
        LOGGER.debug("Handling UI commands {}", uiCommands);
        if (uiCommands.isEmpty()) {
            return;
        }
        boolean blank = uiCommands.stream().anyMatch(UiCommand.BlankWorkspace.class::isInstance);
        if (!blank) {
            uiCommands.stream().filter(UiCommand.OpenDatabases.class::isInstance).map(UiCommand.OpenDatabases.class::cast).forEach(command -> this.openDatabases(command.parserResults()));
        }
        uiCommands.stream().filter(UiCommand.JumpToEntryKey.class::isInstance).map(UiCommand.JumpToEntryKey.class::cast).map(UiCommand.JumpToEntryKey::citationKey).filter(Objects::nonNull).findAny().ifPresent(entryKey -> {
            LOGGER.debug("Jump to entry {} requested", entryKey);
            this.waitForLoadingFinished(() -> this.jumpToEntry((String)entryKey));
        });
    }

    private void openDatabases(List<ParserResult> parserResults) {
        ArrayList<ParserResult> failed = new ArrayList<ParserResult>();
        ArrayList<ParserResult> toOpenTab = new ArrayList<ParserResult>();
        List<ParserResult> invalidDatabases = parserResults.stream().filter(ParserResult::isInvalid).toList();
        failed.addAll(invalidDatabases);
        parserResults.removeAll(invalidDatabases);
        Path focusedFile = parserResults.stream().findFirst().flatMap(ParserResult::getPath).orElse(this.preferences.getGuiPreferences().getLastFocusedFile()).toAbsolutePath();
        boolean first = false;
        for (ParserResult parserResult : parserResults) {
            if (parserResult.getPath().filter(path -> path.toAbsolutePath().equals(focusedFile)).isPresent()) {
                first = true;
            }
            if (parserResult.getDatabase().isShared()) {
                try {
                    OpenDatabaseAction.openSharedDatabase(parserResult, this.tabContainer, this.dialogService, this.preferences, this.stateManager, this.entryTypesManager, this.fileUpdateMonitor, this.undoManager, this.clipBoardManager, this.taskExecutor);
                }
                catch (SQLException | DatabaseNotSupportedException | InvalidDBMSConnectionPropertiesException | NotASharedDatabaseException e) {
                    LOGGER.error("Connection error", (Throwable)e);
                    this.dialogService.showErrorDialogAndWait(Localization.lang("Connection error", new Object[0]), Localization.lang("A local copy will be opened.", new Object[0]), e);
                    toOpenTab.add(parserResult);
                }
                continue;
            }
            if (parserResult.toOpenTab()) {
                toOpenTab.add(parserResult);
                continue;
            }
            this.addParserResult(parserResult, first);
            first = false;
        }
        for (ParserResult parserResult : toOpenTab) {
            this.addParserResult(parserResult, first);
            first = false;
        }
        for (ParserResult parserResult : failed) {
            String message = Localization.lang("Error opening file '%0'", parserResult.getPath().map(Path::toString).orElse("(File name unknown)")) + "\n" + parserResult.getErrorMessage();
            this.dialogService.showErrorDialogAndWait(Localization.lang("Error opening file", new Object[0]), message);
        }
        for (ParserResult parserResult : parserResults) {
            if (!parserResult.hasWarnings()) continue;
            ParserResultWarningDialog.showParserResultWarningDialog(parserResult, this.dialogService);
            this.tabContainer.getLibraryTabs().stream().filter(tab -> parserResult.getDatabase().equals(tab.getDatabase())).findAny().ifPresent(this.tabContainer::showLibraryTab);
        }
        parserResults.forEach(pr -> OpenDatabaseAction.performPostOpenActions(pr, this.dialogService, this.preferences));
        LOGGER.debug("Finished adding panels");
    }

    private void addParserResult(ParserResult parserResult, boolean raisePanel) {
        if (parserResult.toOpenTab()) {
            LOGGER.trace("Adding the entries to the open tab.");
            LibraryTab libraryTab = this.tabContainer.getCurrentLibraryTab();
            if (libraryTab == null) {
                LOGGER.debug("No open tab found to add entries to. Creating a new tab.");
                this.tabContainer.addTab(parserResult.getDatabaseContext(), raisePanel);
            } else {
                this.addImportedEntries(libraryTab, parserResult);
            }
        } else {
            Optional<LibraryTab> libraryTab = this.tabContainer.getLibraryTabs().stream().filter(p -> p.getBibDatabaseContext().getDatabasePath().equals(parserResult.getPath())).findFirst();
            if (libraryTab.isPresent()) {
                this.tabContainer.showLibraryTab(libraryTab.get());
            } else {
                this.tabContainer.addTab(parserResult.getDatabaseContext(), raisePanel);
            }
        }
    }

    private void waitForLoadingFinished(Runnable runnable) {
        LOGGER.trace("Waiting for all tabs being loaded");
        CompletableFuture future = new CompletableFuture();
        List loadings = this.tabContainer.getLibraryTabs().stream().map(LibraryTab::getLoading).collect(Collectors.toList());
        ChangeListener listener = (observable, oldValue, newValue) -> {
            if (observable != null) {
                loadings.remove(observable);
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Count of loading tabs: {}", (Object)loadings.size());
                LOGGER.trace("Count of loading tabs really true: {}", (Object)loadings.stream().filter(ObservableBooleanValue::get).count());
            }
            for (ObservableBooleanValue obs : loadings) {
                if (!obs.get()) continue;
                return;
            }
            LOGGER.trace("Future completed");
            future.complete(null);
        };
        for (ObservableBooleanValue obs : loadings) {
            obs.addListener(listener);
        }
        LOGGER.trace("Fire once");
        listener.changed(null, null, (Object)false);
        LOGGER.trace("Waiting for state changes...");
        future.thenRun(() -> {
            LOGGER.debug("All tabs loaded. Jumping to entry.");
            for (ObservableBooleanValue obs : loadings) {
                obs.removeListener(listener);
            }
            runnable.run();
        });
    }

    private void jumpToEntry(String entryKey) {
        LibraryTab currentLibraryTab = this.tabContainer.getCurrentLibraryTab();
        List<LibraryTab> sortedTabs = this.tabContainer.getLibraryTabs().stream().sorted(Comparator.comparing(tab -> tab != currentLibraryTab)).toList();
        for (LibraryTab libraryTab : sortedTabs) {
            Optional<BibEntry> bibEntry = libraryTab.getDatabase().getEntries().stream().filter(entry -> entry.getCitationKey().orElse("").equals(entryKey)).findAny();
            if (!bibEntry.isPresent()) continue;
            LOGGER.debug("Found entry {} in library tab {}", (Object)entryKey, (Object)libraryTab);
            libraryTab.clearAndSelect(bibEntry.get());
            this.tabContainer.showLibraryTab(libraryTab);
            break;
        }
        LOGGER.trace("End of loop");
        if (this.stateManager.getSelectedEntries().isEmpty()) {
            this.dialogService.notify(Localization.lang("Citation key '%0' to select not found in open libraries.", entryKey));
        }
    }

    void addImportedEntries(LibraryTab tab, ParserResult parserResult) {
        BackgroundTask<ParserResult> task = BackgroundTask.wrap(() -> parserResult);
        ImportCleanup cleanup = ImportCleanup.targeting(tab.getBibDatabaseContext().getMode(), this.preferences.getFieldPreferences());
        cleanup.doPostCleanup((Collection<BibEntry>)parserResult.getDatabase().getEntries());
        ImportEntriesDialog dialog = new ImportEntriesDialog(tab.getBibDatabaseContext(), task);
        dialog.setTitle(Localization.lang("Import", new Object[0]));
        this.dialogService.showCustomDialogAndWait(dialog);
    }
}

