/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fileformat;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Objects;
import java.util.Optional;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.Importer;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.fileformat.BibtexParser;
import org.jabref.logic.util.StandardFileType;
import org.jabref.model.database.BibDatabaseModeDetection;
import org.jabref.model.util.FileUpdateMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BibtexImporter
extends Importer {
    private static final Logger LOGGER = LoggerFactory.getLogger(BibtexImporter.class);
    private static final String SIGNATURE = "This file was created with JabRef";
    private final ImportFormatPreferences importFormatPreferences;
    private final FileUpdateMonitor fileMonitor;

    public BibtexImporter(ImportFormatPreferences importFormatPreferences, FileUpdateMonitor fileMonitor) {
        this.importFormatPreferences = importFormatPreferences;
        this.fileMonitor = fileMonitor;
    }

    @Override
    public boolean isRecognizedFormat(BufferedReader reader) {
        Objects.requireNonNull(reader);
        return true;
    }

    @Override
    public ParserResult importDatabase(Path filePath) throws IOException {
        EncodingResult result = BibtexImporter.getEncodingResult(filePath);
        CharsetDecoder decoder = result.encoding().newDecoder();
        decoder.onMalformedInput(CodingErrorAction.REPLACE);
        try (InputStreamReader inputStreamReader = new InputStreamReader(Files.newInputStream(filePath, new OpenOption[0]), decoder);){
            ParserResult parserResult;
            try (BufferedReader reader = new BufferedReader(inputStreamReader);){
                ParserResult parserResult2 = this.importDatabase(reader);
                parserResult2.getMetaData().setEncoding(result.encoding());
                parserResult2.getMetaData().setEncodingExplicitlySupplied(result.encodingExplicitlySupplied());
                parserResult2.setPath(filePath);
                if (parserResult2.getMetaData().getMode().isEmpty()) {
                    parserResult2.getMetaData().setMode(BibDatabaseModeDetection.inferMode(parserResult2.getDatabase()));
                }
                parserResult = parserResult2;
            }
            return parserResult;
        }
    }

    public static Charset getEncoding(Path filePath) throws IOException {
        return BibtexImporter.getEncodingResult(filePath).encoding();
    }

    private static EncodingResult getEncodingResult(Path filePath) throws IOException {
        Charset encoding;
        boolean encodingExplicitlySupplied;
        Charset detectedCharset;
        try (InputStream inputStream = Files.newInputStream(filePath, new OpenOption[0]);){
            BufferedInputStream bufferedInputStream = new BufferedInputStream(inputStream);
            bufferedInputStream.mark(8192);
            detectedCharset = BibtexImporter.getCharset(bufferedInputStream);
            bufferedInputStream.reset();
            LOGGER.debug("Detected charset: {}", (Object)detectedCharset.name());
        }
        try (BufferedReader reader = Files.newBufferedReader(filePath, detectedCharset);){
            Optional<Charset> suppliedEncoding = BibtexImporter.getSuppliedEncoding(reader);
            LOGGER.debug("Supplied encoding: {}", suppliedEncoding);
            encodingExplicitlySupplied = suppliedEncoding.isPresent();
            encoding = suppliedEncoding.orElse(detectedCharset);
            LOGGER.debug("Encoding used to read the file: {}", (Object)encoding);
        }
        EncodingResult result = new EncodingResult(encoding, encodingExplicitlySupplied);
        return result;
    }

    @Override
    public ParserResult importDatabase(BufferedReader reader) throws IOException {
        return new BibtexParser(this.importFormatPreferences, this.fileMonitor).parse(reader);
    }

    @Override
    public String getName() {
        return "BibTeX";
    }

    @Override
    public StandardFileType getFileType() {
        return StandardFileType.BIBTEX_DB;
    }

    @Override
    public String getDescription() {
        return "This importer enables `--importToOpen someEntry.bib`";
    }

    private static Optional<Charset> getSuppliedEncoding(BufferedReader reader) {
        try {
            String line;
            while ((line = reader.readLine()) != null) {
                int percentPos = (line = line.trim()).indexOf(37, 0);
                if (percentPos == -1) {
                    return Optional.empty();
                }
                if ((line = line.substring(percentPos + 1).trim()).startsWith(SIGNATURE)) continue;
                if (line.startsWith("Encoding: ")) {
                    Integer atSymbolIndex = line.indexOf(64);
                    String encoding = atSymbolIndex > 0 ? line.substring("Encoding: ".length(), atSymbolIndex) : line.substring("Encoding: ".length());
                    return Optional.of(Charset.forName(encoding));
                }
                return Optional.empty();
            }
        }
        catch (IOException e) {
            LOGGER.error("Supplied encoding could not be determined", (Throwable)e);
        }
        return Optional.empty();
    }

    private record EncodingResult(Charset encoding, boolean encodingExplicitlySupplied) {
    }
}

