/*
 * Decompiled with CFR 0.152.
 */
package org.meteoinfo.ndarray;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.function.DoubleUnaryOperator;

public final class Complex
implements Serializable {
    public static final Complex I = new Complex(0.0, 1.0);
    public static final Complex ONE = new Complex(1.0, 0.0);
    public static final Complex ZERO = new Complex(0.0, 0.0);
    private static final Complex NAN = new Complex(Double.NaN, Double.NaN);
    private static final double PI_OVER_2 = 1.5707963267948966;
    private static final double PI_OVER_4 = 0.7853981633974483;
    private static final double LN_2 = Math.log(2.0);
    private static final double LOG_10E_O_2 = Math.log10(Math.E) / 2.0;
    private static final double LOG10_2 = Math.log10(2.0);
    private static final double HALF = 0.5;
    private static final double ROOT2 = 1.4142135623730951;
    private static final double ONE_OVER_ROOT2 = 0.7071067811865476;
    private static final long NEGATIVE_ZERO_LONG_BITS = Double.doubleToLongBits(-0.0);
    private static final int EXPONENT_OFFSET = 1023;
    private static final double EPSILON = Double.longBitsToDouble(4368491638549381120L);
    private static final long UNSIGN_MASK = Long.MAX_VALUE;
    private static final long MANTISSA_MASK = 0xFFFFFFFFFFFFFL;
    private static final double MULTIPLIER = 1.34217729E8;
    private static final double A_CROSSOVER = 10.0;
    private static final double B_CROSSOVER = 0.6471;
    private static final double SAFE_MAX = Math.sqrt(Double.MAX_VALUE) / 8.0;
    private static final double SAFE_MIN = Math.sqrt(Double.MIN_NORMAL) * 4.0;
    private static final double SAFE_UPPER = Math.sqrt(Double.MAX_VALUE) / 2.0;
    private static final double SAFE_LOWER = Math.sqrt(Double.MIN_NORMAL) * 2.0;
    private static final double SQRT_SAFE_UPPER = 2.2471164185778946E307;
    private static final double SAFE_EXP = 708.0;
    private static final double EXP_M = Math.exp(708.0);
    private static final int EXP_54 = 0x3600000;
    private static final int EXP_500 = 1596981248;
    private static final int EXP_1024 = 0x7FF00000;
    private static final int EXP_NEG_500 = 0x20B00000;
    private static final double TWO_POW_600 = 4.149515568880993E180;
    private static final double TWO_POW_NEG_600 = 2.409919865102884E-181;
    private static final long serialVersionUID = 20180201L;
    private static final int TO_STRING_SIZE = 64;
    private static final int FORMAT_MIN_LEN = 5;
    private static final char FORMAT_START = '(';
    private static final char FORMAT_END = ')';
    private static final char FORMAT_SEP = ',';
    private static final int BEFORE_SEP = 2;
    private final double imaginary;
    private final double real;

    public Complex(double real) {
        this(real, 0.0);
    }

    public Complex(double real, double imaginary) {
        this.real = real;
        this.imaginary = imaginary;
    }

    public static Complex ofCartesian(double real, double imaginary) {
        return new Complex(real, imaginary);
    }

    public static Complex ofPolar(double rho, double theta) {
        if (!Double.isFinite(theta) || Complex.negative(rho) || Double.isNaN(rho)) {
            return NAN;
        }
        double x = rho * Math.cos(theta);
        double y = rho * Math.sin(theta);
        return new Complex(x, y);
    }

    public static Complex ofCis(double x) {
        return new Complex(Math.cos(x), Math.sin(x));
    }

    public static Complex parse(String s) {
        double im;
        double re;
        int len = s.length();
        if (len < 5) {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Input too short, expected format", "(x,y)", s));
        }
        if (s.charAt(0) != '(') {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Expected start delimiter", Character.valueOf('('), s));
        }
        if (s.charAt(len - 1) != ')') {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Expected end delimiter", Character.valueOf(')'), s));
        }
        int sep = s.lastIndexOf(44, len - 3);
        if (sep < 2) {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Expected separator between two numbers", Character.valueOf(','), s));
        }
        if (s.indexOf(44, sep + 1) != -1) {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Incorrect number of parts, expected only 2 using separator", Character.valueOf(','), s));
        }
        String rePart = s.substring(1, sep);
        try {
            re = Double.parseDouble(rePart);
        }
        catch (NumberFormatException ex) {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Could not parse real part", rePart, s));
        }
        String imPart = s.substring(sep + 1, len - 1);
        try {
            im = Double.parseDouble(imPart);
        }
        catch (NumberFormatException ex) {
            throw new NumberFormatException(Complex.parsingExceptionMsg("Could not parse imaginary part", imPart, s));
        }
        return Complex.ofCartesian(re, im);
    }

    private static String parsingExceptionMsg(String message, Object error, String s) {
        StringBuilder sb = new StringBuilder(100).append(message).append(" '").append(error).append("' for input \"").append(s).append('\"');
        return sb.toString();
    }

    public double getReal() {
        return this.real;
    }

    public double real() {
        return this.getReal();
    }

    public double getImaginary() {
        return this.imaginary;
    }

    public double imag() {
        return this.getImaginary();
    }

    public double abs() {
        return Complex.abs(this.real, this.imaginary);
    }

    private static double abs(double real, double imaginary) {
        return Complex.hypot(real, imaginary);
    }

    public double arg() {
        return Math.atan2(this.imaginary, this.real);
    }

    public double norm() {
        if (this.isInfinite()) {
            return Double.POSITIVE_INFINITY;
        }
        return this.real * this.real + this.imaginary * this.imaginary;
    }

    public boolean isNaN() {
        if (Double.isNaN(this.real) || Double.isNaN(this.imaginary)) {
            return !this.isInfinite();
        }
        return false;
    }

    public boolean isInfinite() {
        return Double.isInfinite(this.real) || Double.isInfinite(this.imaginary);
    }

    public boolean isFinite() {
        return Double.isFinite(this.real) && Double.isFinite(this.imaginary);
    }

    public Complex conj() {
        return new Complex(this.real, -this.imaginary);
    }

    public Complex negate() {
        return new Complex(-this.real, -this.imaginary);
    }

    public Complex proj() {
        if (this.isInfinite()) {
            return new Complex(Double.POSITIVE_INFINITY, Math.copySign(0.0, this.imaginary));
        }
        return this;
    }

    public Complex add(Complex addend) {
        return new Complex(this.real + addend.real, this.imaginary + addend.imaginary);
    }

    public Complex add(double addend) {
        return new Complex(this.real + addend, this.imaginary);
    }

    public Complex addImaginary(double addend) {
        return new Complex(this.real, this.imaginary + addend);
    }

    public Complex subtract(Complex subtrahend) {
        return new Complex(this.real - subtrahend.real, this.imaginary - subtrahend.imaginary);
    }

    public Complex subtract(double subtrahend) {
        return new Complex(this.real - subtrahend, this.imaginary);
    }

    public Complex subtractImaginary(double subtrahend) {
        return new Complex(this.real, this.imaginary - subtrahend);
    }

    public Complex subtractFrom(double minuend) {
        return new Complex(minuend - this.real, -this.imaginary);
    }

    public Complex subtractFromImaginary(double minuend) {
        return new Complex(-this.real, minuend - this.imaginary);
    }

    public Complex rSubtract(double v) {
        return new Complex(v - this.getReal(), -this.getImaginary());
    }

    public Complex multiply(Complex factor) {
        return Complex.multiply(this.real, this.imaginary, factor.real, factor.imaginary);
    }

    private static Complex multiply(double re1, double im1, double re2, double im2) {
        double a = re1;
        double b = im1;
        double c = re2;
        double d = im2;
        double ac = a * c;
        double bd = b * d;
        double ad = a * d;
        double bc = b * c;
        double x = ac - bd;
        double y = ad + bc;
        if (Double.isNaN(x) && Double.isNaN(y)) {
            boolean recalc = false;
            if ((Double.isInfinite(a) || Double.isInfinite(b)) && Complex.isNotZero(c, d)) {
                a = Complex.boxInfinity(a);
                b = Complex.boxInfinity(b);
                c = Complex.changeNaNtoZero(c);
                d = Complex.changeNaNtoZero(d);
                recalc = true;
            }
            if ((Double.isInfinite(c) || Double.isInfinite(d)) && Complex.isNotZero(a, b)) {
                c = Complex.boxInfinity(c);
                d = Complex.boxInfinity(d);
                a = Complex.changeNaNtoZero(a);
                b = Complex.changeNaNtoZero(b);
                recalc = true;
            }
            if (!recalc && (Double.isInfinite(ac) || Double.isInfinite(bd) || Double.isInfinite(ad) || Double.isInfinite(bc))) {
                a = Complex.changeNaNtoZero(a);
                b = Complex.changeNaNtoZero(b);
                c = Complex.changeNaNtoZero(c);
                d = Complex.changeNaNtoZero(d);
                recalc = true;
            }
            if (recalc) {
                x = Double.POSITIVE_INFINITY * (a * c - b * d);
                y = Double.POSITIVE_INFINITY * (a * d + b * c);
            }
        }
        return new Complex(x, y);
    }

    private static double boxInfinity(double component) {
        return Math.copySign(Double.isInfinite(component) ? 1.0 : 0.0, component);
    }

    private static boolean isNotZero(double real, double imaginary) {
        return real != 0.0 || imaginary != 0.0;
    }

    private static double changeNaNtoZero(double value) {
        return Double.isNaN(value) ? Math.copySign(0.0, value) : value;
    }

    public Complex multiply(double factor) {
        return new Complex(this.real * factor, this.imaginary * factor);
    }

    public Complex multiplyImaginary(double factor) {
        return new Complex(-this.imaginary * factor, this.real * factor);
    }

    public Complex divide(Complex divisor) {
        return Complex.divide(this.real, this.imaginary, divisor.real, divisor.imaginary);
    }

    private static Complex divide(double re1, double im1, double re2, double im2) {
        double a = re1;
        double b = im1;
        double c = re2;
        double d = im2;
        int ilogbw = 0;
        int exponent = Complex.getScale(c, d);
        if (exponent <= 1023) {
            ilogbw = exponent;
            c = Math.scalb(c, -ilogbw);
            d = Math.scalb(d, -ilogbw);
        }
        double denom = c * c + d * d;
        if (Complex.getMaxExponent(a, b) > 1021) {
            ilogbw -= 2;
            a /= 4.0;
            b /= 4.0;
        }
        double x = Math.scalb((a * c + b * d) / denom, -ilogbw);
        double y = Math.scalb((b * c - a * d) / denom, -ilogbw);
        if (Double.isNaN(x) && Double.isNaN(y)) {
            if (!(denom != 0.0 || Double.isNaN(a) && Double.isNaN(b))) {
                x = Math.copySign(Double.POSITIVE_INFINITY, c) * a;
                y = Math.copySign(Double.POSITIVE_INFINITY, c) * b;
            } else if ((Double.isInfinite(a) || Double.isInfinite(b)) && Double.isFinite(c) && Double.isFinite(d)) {
                a = Complex.boxInfinity(a);
                b = Complex.boxInfinity(b);
                x = Double.POSITIVE_INFINITY * (a * c + b * d);
                y = Double.POSITIVE_INFINITY * (b * c - a * d);
            } else if ((Double.isInfinite(c) || Double.isInfinite(d)) && Double.isFinite(a) && Double.isFinite(b)) {
                c = Complex.boxInfinity(c);
                d = Complex.boxInfinity(d);
                x = 0.0 * (a * c + b * d);
                y = 0.0 * (b * c - a * d);
            }
        }
        return new Complex(x, y);
    }

    public Complex divide(double divisor) {
        return new Complex(this.real / divisor, this.imaginary / divisor);
    }

    public Complex divideImaginary(double divisor) {
        return new Complex(this.imaginary / divisor, -this.real / divisor);
    }

    public Complex rDivide(double v) {
        return new Complex(v, 0.0).divide(this);
    }

    public Complex exp() {
        if (Double.isInfinite(this.real)) {
            double zeroOrInf;
            if (this.real < 0.0) {
                if (!Double.isFinite(this.imaginary)) {
                    return new Complex(0.0, Math.copySign(0.0, this.imaginary));
                }
                zeroOrInf = 0.0;
            } else {
                if (this.imaginary == 0.0) {
                    return this;
                }
                if (!Double.isFinite(this.imaginary)) {
                    return new Complex(this.real, Double.NaN);
                }
                zeroOrInf = this.real;
            }
            return new Complex(zeroOrInf * Math.cos(this.imaginary), zeroOrInf * Math.sin(this.imaginary));
        }
        if (Double.isNaN(this.real)) {
            return this.imaginary == 0.0 ? this : NAN;
        }
        if (!Double.isFinite(this.imaginary)) {
            return NAN;
        }
        double exp = Math.exp(this.real);
        if (this.imaginary == 0.0) {
            return new Complex(exp, this.imaginary);
        }
        return new Complex(exp * Math.cos(this.imaginary), exp * Math.sin(this.imaginary));
    }

    public Complex log() {
        return this.log(Math::log, 0.5, LN_2, Complex::ofCartesian);
    }

    public Complex log10() {
        return this.log(Math::log10, LOG_10E_O_2, LOG10_2, Complex::ofCartesian);
    }

    private Complex log(DoubleUnaryOperator log, double logOfeOver2, double logOf2, ComplexConstructor constructor) {
        double re;
        double y;
        if (Double.isNaN(this.real) || Double.isNaN(this.imaginary)) {
            if (this.isInfinite()) {
                return constructor.create(Double.POSITIVE_INFINITY, Double.NaN);
            }
            return NAN;
        }
        double x = Math.abs(this.real);
        if (x < (y = Math.abs(this.imaginary))) {
            double tmp = x;
            x = y;
            y = tmp;
        }
        if (x == 0.0) {
            return constructor.create(Double.NEGATIVE_INFINITY, Complex.negative(this.real) ? Math.copySign(Math.PI, this.imaginary) : this.imaginary);
        }
        if (x > 0.5 && x < 1.4142135623730951) {
            re = Math.log1p(Complex.x2y2m1(x, y)) * logOfeOver2;
        } else {
            re = 0.0;
            if (x > 8.988465674311579E307) {
                if (Complex.isPosInfinite(x)) {
                    return constructor.create(x, this.arg());
                }
                x /= 2.0;
                y /= 2.0;
                re = logOf2;
            } else if (y < Double.MIN_NORMAL) {
                if (y == 0.0) {
                    return constructor.create(log.applyAsDouble(x), this.arg());
                }
                x *= 1.8014398509481984E16;
                y *= 1.8014398509481984E16;
                re = -54.0 * logOf2;
            }
            re += log.applyAsDouble(Complex.abs(x, y));
        }
        return constructor.create(re, this.arg());
    }

    public Complex pow(Complex x) {
        if (this.real == 0.0 && this.imaginary == 0.0) {
            if (x.real > 0.0 && x.imaginary == 0.0) {
                return ZERO;
            }
            return NAN;
        }
        return this.log().multiply(x).exp();
    }

    public Complex pow(double x) {
        if (this.real == 0.0 && this.imaginary == 0.0) {
            if (x > 0.0) {
                return ZERO;
            }
            return NAN;
        }
        return this.log().multiply(x).exp();
    }

    public Complex rPow(double v) {
        return new Complex(v, 0.0).pow(this);
    }

    public Complex sqrt() {
        return Complex.sqrt(this.real, this.imaginary);
    }

    private static Complex sqrt(double real, double imaginary) {
        double t;
        double y;
        if (Double.isNaN(real) || Double.isNaN(imaginary)) {
            if (Double.isInfinite(imaginary)) {
                return new Complex(Double.POSITIVE_INFINITY, imaginary);
            }
            if (Double.isInfinite(real)) {
                if (real == Double.NEGATIVE_INFINITY) {
                    return new Complex(Double.NaN, Math.copySign(Double.POSITIVE_INFINITY, imaginary));
                }
                return new Complex(Double.POSITIVE_INFINITY, Double.NaN);
            }
            return NAN;
        }
        double x = Math.abs(real);
        if (Complex.inRegion(x, y = Math.abs(imaginary), Double.MIN_NORMAL, 2.2471164185778946E307)) {
            t = Math.sqrt(2.0 * (Complex.abs(x, y) + x));
        } else {
            double rescale;
            double sy;
            double sx;
            if (Complex.isPosInfinite(y)) {
                return new Complex(Double.POSITIVE_INFINITY, imaginary);
            }
            if (Complex.isPosInfinite(x)) {
                if (real == Double.NEGATIVE_INFINITY) {
                    return new Complex(0.0, Math.copySign(Double.POSITIVE_INFINITY, imaginary));
                }
                return new Complex(Double.POSITIVE_INFINITY, Math.copySign(0.0, imaginary));
            }
            if (y == 0.0) {
                double sqrtAbs = Math.sqrt(x);
                if (real < 0.0) {
                    return new Complex(0.0, Math.copySign(sqrtAbs, imaginary));
                }
                return new Complex(sqrtAbs, imaginary);
            }
            if (x == 0.0) {
                double sqrtAbs = Math.sqrt(y) * 0.7071067811865476;
                return new Complex(sqrtAbs, Math.copySign(sqrtAbs, imaginary));
            }
            if (Math.max(x, y) > 2.2471164185778946E307) {
                sx = x / 16.0;
                sy = y / 16.0;
                rescale = 4.0;
            } else {
                sx = x * 1.8014398509481984E16;
                sy = y * 1.8014398509481984E16;
                rescale = 7.450580596923828E-9;
            }
            t = rescale * Math.sqrt(2.0 * (Complex.abs(sx, sy) + sx));
        }
        if (real >= 0.0) {
            return new Complex(t / 2.0, imaginary / t);
        }
        return new Complex(y / t, Math.copySign(t / 2.0, imaginary));
    }

    public Complex sin() {
        return Complex.sinh(-this.imaginary, this.real, Complex::multiplyNegativeI);
    }

    public Complex cos() {
        return Complex.cosh(-this.imaginary, this.real, Complex::ofCartesian);
    }

    public Complex tan() {
        return Complex.tanh(-this.imaginary, this.real, Complex::multiplyNegativeI);
    }

    public Complex asin() {
        return Complex.asin(this.real, this.imaginary, Complex::ofCartesian);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static Complex asin(double real, double imaginary, ComplexConstructor constructor) {
        double im;
        double re;
        double x = Math.abs(real);
        double y = Math.abs(imaginary);
        if (Double.isNaN(x)) {
            if (!Complex.isPosInfinite(y)) return NAN;
            re = x;
            im = y;
            return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
        } else if (Double.isNaN(y)) {
            if (x == 0.0) {
                re = 0.0;
                im = y;
                return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
            } else {
                if (!Complex.isPosInfinite(x)) return NAN;
                re = y;
                im = x;
            }
            return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
        } else if (Complex.isPosInfinite(x)) {
            re = Complex.isPosInfinite(y) ? 0.7853981633974483 : 1.5707963267948966;
            im = x;
            return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
        } else if (Complex.isPosInfinite(y)) {
            re = 0.0;
            im = y;
            return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
        } else {
            if (y == 0.0 && x <= 1.0) {
                return constructor.create(Math.asin(real), imaginary);
            }
            double xp1 = x + 1.0;
            double xm1 = x - 1.0;
            if (Complex.inRegion(x, y, SAFE_MIN, SAFE_MAX)) {
                double s;
                double yy = y * y;
                double r = Math.sqrt(xp1 * xp1 + yy);
                double a = 0.5 * (r + (s = Math.sqrt(xm1 * xm1 + yy)));
                double b = x / a;
                if (b <= 0.6471) {
                    re = Math.asin(b);
                } else {
                    double apx = a + x;
                    re = x <= 1.0 ? Math.atan(x / Math.sqrt(0.5 * apx * (yy / (r + xp1) + (s - xm1)))) : Math.atan(x / (y * Math.sqrt(0.5 * (apx / (r + xp1) + apx / (s + xm1)))));
                }
                if (a <= 10.0) {
                    double am1 = x < 1.0 ? 0.5 * (yy / (r + xp1) + yy / (s - xm1)) : 0.5 * (yy / (r + xp1) + (s + xm1));
                    im = Math.log1p(am1 + Math.sqrt(am1 * (a + 1.0)));
                    return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
                } else {
                    im = Math.log(a + Math.sqrt(a * a - 1.0));
                }
                return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
            } else if (y <= EPSILON * Math.abs(xm1)) {
                if (x < 1.0) {
                    re = Math.asin(x);
                    im = y / Math.sqrt(xp1 * (1.0 - x));
                    return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
                } else {
                    re = 1.5707963267948966;
                    im = Double.MAX_VALUE / xp1 > xm1 ? Math.log1p(xm1 + Math.sqrt(xp1 * xm1)) : LN_2 + Math.log(x);
                }
                return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
            } else if (y <= SAFE_MIN) {
                re = 1.5707963267948966 - Math.sqrt(y);
                im = Math.sqrt(y);
                return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
            } else if (EPSILON * y - 1.0 >= x) {
                re = x / y;
                im = LN_2 + Math.log(y);
                return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
            } else if (x > 1.0) {
                re = Math.atan(x / y);
                double xoy = x / y;
                im = LN_2 + Math.log(y) + 0.5 * Math.log1p(xoy * xoy);
                return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
            } else {
                double a = Math.sqrt(1.0 + y * y);
                re = x / a;
                im = 0.5 * Math.log1p(2.0 * y * (y + a));
            }
        }
        return constructor.create(Complex.changeSign(re, real), Complex.changeSign(im, imaginary));
    }

    public Complex acos() {
        return Complex.acos(this.real, this.imaginary, Complex::ofCartesian);
    }

    private static Complex acos(double real, double imaginary, ComplexConstructor constructor) {
        double im;
        double re;
        double x = Math.abs(real);
        double y = Math.abs(imaginary);
        if (Complex.isPosInfinite(x)) {
            if (Complex.isPosInfinite(y)) {
                re = 0.7853981633974483;
                im = y;
            } else {
                if (Double.isNaN(y)) {
                    return constructor.create(imaginary, real);
                }
                re = 0.0;
                im = Double.POSITIVE_INFINITY;
            }
        } else {
            if (Double.isNaN(x)) {
                if (Complex.isPosInfinite(y)) {
                    return constructor.create(x, -imaginary);
                }
                return NAN;
            }
            if (Complex.isPosInfinite(y)) {
                re = 1.5707963267948966;
                im = y;
            } else {
                if (Double.isNaN(y)) {
                    return constructor.create(x == 0.0 ? 1.5707963267948966 : y, y);
                }
                if (y == 0.0 && x <= 1.0) {
                    return constructor.create(x == 0.0 ? 1.5707963267948966 : Math.acos(real), -imaginary);
                }
                double xp1 = x + 1.0;
                double xm1 = x - 1.0;
                if (Complex.inRegion(x, y, SAFE_MIN, SAFE_MAX)) {
                    double s;
                    double yy = y * y;
                    double r = Math.sqrt(xp1 * xp1 + yy);
                    double a = 0.5 * (r + (s = Math.sqrt(xm1 * xm1 + yy)));
                    double b = x / a;
                    if (b <= 0.6471) {
                        re = Math.acos(b);
                    } else {
                        double apx = a + x;
                        re = x <= 1.0 ? Math.atan(Math.sqrt(0.5 * apx * (yy / (r + xp1) + (s - xm1))) / x) : Math.atan(y * Math.sqrt(0.5 * (apx / (r + xp1) + apx / (s + xm1))) / x);
                    }
                    if (a <= 10.0) {
                        double am1 = x < 1.0 ? 0.5 * (yy / (r + xp1) + yy / (s - xm1)) : 0.5 * (yy / (r + xp1) + (s + xm1));
                        im = Math.log1p(am1 + Math.sqrt(am1 * (a + 1.0)));
                    } else {
                        im = Math.log(a + Math.sqrt(a * a - 1.0));
                    }
                } else if (y <= EPSILON * Math.abs(xm1)) {
                    if (x < 1.0) {
                        re = Math.acos(x);
                        im = y / Math.sqrt(xp1 * (1.0 - x));
                    } else if (Double.MAX_VALUE / xp1 > xm1) {
                        re = y / Math.sqrt(xm1 * xp1);
                        im = Math.log1p(xm1 + Math.sqrt(xp1 * xm1));
                    } else {
                        re = y / x;
                        im = LN_2 + Math.log(x);
                    }
                } else if (y <= SAFE_MIN) {
                    re = Math.sqrt(y);
                    im = Math.sqrt(y);
                } else if (EPSILON * y - 1.0 >= x) {
                    re = 1.5707963267948966;
                    im = LN_2 + Math.log(y);
                } else if (x > 1.0) {
                    re = Math.atan(y / x);
                    double xoy = x / y;
                    im = LN_2 + Math.log(y) + 0.5 * Math.log1p(xoy * xoy);
                } else {
                    re = 1.5707963267948966;
                    double a = Math.sqrt(1.0 + y * y);
                    im = 0.5 * Math.log1p(2.0 * y * (y + a));
                }
            }
        }
        return constructor.create(Complex.negative(real) ? Math.PI - re : re, Complex.negative(imaginary) ? im : -im);
    }

    public Complex atan() {
        return Complex.atanh(-this.imaginary, this.real, Complex::multiplyNegativeI);
    }

    public Complex sinh() {
        return Complex.sinh(this.real, this.imaginary, Complex::ofCartesian);
    }

    private static Complex sinh(double real, double imaginary, ComplexConstructor constructor) {
        if (Double.isInfinite(real) && !Double.isFinite(imaginary)) {
            return constructor.create(real, Double.NaN);
        }
        if (real == 0.0) {
            if (Double.isFinite(imaginary)) {
                return constructor.create(Complex.changeSign(real, Math.cos(imaginary)), Math.sin(imaginary));
            }
            return constructor.create(real, Double.NaN);
        }
        if (imaginary == 0.0) {
            return constructor.create(Math.sinh(real), imaginary);
        }
        double x = Math.abs(real);
        if (x > 708.0) {
            return Complex.coshsinh(x, real, imaginary, true, constructor);
        }
        return constructor.create(Math.sinh(real) * Math.cos(imaginary), Math.cosh(real) * Math.sin(imaginary));
    }

    public Complex cosh() {
        return Complex.cosh(this.real, this.imaginary, Complex::ofCartesian);
    }

    private static Complex cosh(double real, double imaginary, ComplexConstructor constructor) {
        if (Double.isInfinite(real) && !Double.isFinite(imaginary)) {
            return constructor.create(Math.abs(real), Double.NaN);
        }
        if (real == 0.0) {
            if (Double.isFinite(imaginary)) {
                return constructor.create(Math.cos(imaginary), Complex.changeSign(real, Math.sin(imaginary)));
            }
            return constructor.create(Double.NaN, Complex.changeSign(real, imaginary));
        }
        if (imaginary == 0.0) {
            return constructor.create(Math.cosh(real), Complex.changeSign(imaginary, real));
        }
        double x = Math.abs(real);
        if (x > 708.0) {
            return Complex.coshsinh(x, real, imaginary, false, constructor);
        }
        return constructor.create(Math.cosh(real) * Math.cos(imaginary), Math.sinh(real) * Math.sin(imaginary));
    }

    private static Complex coshsinh(double x, double real, double imaginary, boolean sinh, ComplexConstructor constructor) {
        double re = Math.cos(imaginary);
        double im = Math.sin(imaginary);
        if (sinh) {
            re = Complex.changeSign(re, real);
        } else {
            im = Complex.changeSign(im, real);
        }
        if (x > 2124.0) {
            re *= Double.POSITIVE_INFINITY;
            im *= Double.POSITIVE_INFINITY;
        } else {
            double xm;
            re *= EXP_M / 2.0;
            im *= EXP_M / 2.0;
            if (x > 1416.0) {
                re *= EXP_M;
                im *= EXP_M;
                xm = x - 1416.0;
            } else {
                xm = x - 708.0;
            }
            double exp = Math.exp(xm);
            re *= exp;
            im *= exp;
        }
        return constructor.create(re, im);
    }

    public Complex tanh() {
        return Complex.tanh(this.real, this.imaginary, Complex::ofCartesian);
    }

    private static Complex tanh(double real, double imaginary, ComplexConstructor constructor) {
        double x = Math.abs(real);
        if (!Complex.isPosFinite(x) || !Double.isFinite(imaginary)) {
            if (Complex.isPosInfinite(x)) {
                if (Double.isFinite(imaginary)) {
                    double sign = Math.abs(imaginary) < 1.5707963267948966 ? imaginary : Math.sin(imaginary) * Math.cos(imaginary);
                    return constructor.create(Math.copySign(1.0, real), Math.copySign(0.0, sign));
                }
                return constructor.create(Math.copySign(1.0, real), Math.copySign(0.0, imaginary));
            }
            return constructor.create(real == 0.0 ? real : Double.NaN, imaginary == 0.0 ? imaginary : Double.NaN);
        }
        if (real == 0.0) {
            return constructor.create(real, Math.tan(imaginary));
        }
        if (imaginary == 0.0) {
            return constructor.create(Math.tanh(real), imaginary);
        }
        if (x > 354.0) {
            double re = Math.copySign(1.0, real);
            double im = Math.sin(imaginary) * Math.cos(imaginary);
            im = x > 708.0 ? Math.copySign(0.0, im) : 4.0 * im / EXP_M / Math.exp(2.0 * x - 708.0);
            return constructor.create(re, im);
        }
        double sinhx = Math.sinh(real);
        double coshx = Math.cosh(real);
        double siny = Math.sin(imaginary);
        double cosy = Math.cos(imaginary);
        double divisor = sinhx * sinhx + cosy * cosy;
        return constructor.create(sinhx * coshx / divisor, siny * cosy / divisor);
    }

    public Complex asinh() {
        return Complex.asin(-this.imaginary, this.real, Complex::multiplyNegativeI);
    }

    public Complex acosh() {
        if (Double.isNaN(this.imaginary) && this.real == 0.0) {
            return new Complex(Double.NaN, 1.5707963267948966);
        }
        return Complex.acos(this.real, this.imaginary, (re, im) -> Complex.negative(im) ? new Complex(-im, re) : new Complex(im, -re));
    }

    public Complex atanh() {
        return Complex.atanh(this.real, this.imaginary, Complex::ofCartesian);
    }

    private static Complex atanh(double real, double imaginary, ComplexConstructor constructor) {
        double im;
        double re;
        double x = Math.abs(real);
        double y = Math.abs(imaginary);
        if (Double.isNaN(x)) {
            if (Complex.isPosInfinite(y)) {
                return constructor.create(0.0, Math.copySign(1.5707963267948966, imaginary));
            }
            return NAN;
        }
        if (Double.isNaN(y)) {
            if (Complex.isPosInfinite(x)) {
                return constructor.create(Math.copySign(0.0, real), Double.NaN);
            }
            if (x == 0.0) {
                return constructor.create(real, Double.NaN);
            }
            return NAN;
        }
        if (Complex.inRegion(x, y, SAFE_LOWER, SAFE_UPPER)) {
            double mxp1 = 1.0 - x;
            double yy = y * y;
            re = Math.log1p(4.0 * x / (mxp1 * mxp1 + yy));
            double numerator = 2.0 * y;
            if (x < y) {
                double tmp = x;
                x = y;
                y = tmp;
            }
            double denominator = x >= 1.0 ? (1.0 - x) * (1.0 + x) - y * y : -Complex.x2y2m1(x, y);
            im = Math.atan2(numerator, denominator);
        } else {
            if (x == 0.0) {
                if (imaginary == 0.0) {
                    return constructor.create(real, imaginary);
                }
                return constructor.create(real, Math.atan(imaginary));
            }
            if (x >= SAFE_UPPER) {
                re = Complex.isPosInfinite(x) || Complex.isPosInfinite(y) ? 0.0 : (y >= SAFE_UPPER ? Math.log1p(4.0 / y / (x / y + y / x)) : (y > 1.0 ? Math.log1p(4.0 / (x + y * y / x)) : Math.log1p(4.0 / x)));
            } else if (y >= SAFE_UPPER) {
                if (x > 1.0) {
                    double mxp1 = 1.0 - x;
                    re = Math.log1p(4.0 * x / y / (mxp1 * mxp1 / y + y));
                } else {
                    re = 4.0 * x / y / y;
                }
            } else if (x == 1.0) {
                re = 2.0 * (LN_2 - Math.log(y));
            } else {
                double mxp1 = 1.0 - x;
                re = Math.log1p(4.0 * x / (mxp1 * mxp1 + y * y));
            }
            im = x >= SAFE_UPPER || y >= SAFE_UPPER ? Math.PI : (x <= SAFE_LOWER ? (y <= SAFE_LOWER ? Math.atan2(2.0 * y, 1.0) : Math.atan2(2.0 * y, 1.0 - y * y)) : Math.atan2(2.0 * y, (1.0 - x) * (1.0 + x)));
        }
        return constructor.create(Complex.changeSign(re /= 4.0, real), Complex.changeSign(im /= 2.0, imaginary));
    }

    private static double x2y2m1(double x, double y) {
        double xx = x * x;
        double yy = y * y;
        if (x < 1.0 && xx + yy > 0.5) {
            double xHigh = Complex.splitHigh(x);
            double xLow = x - xHigh;
            double yHigh = Complex.splitHigh(y);
            double yLow = y - yHigh;
            double x2Low = Complex.squareLow(xLow, xHigh, xx);
            double y2Low = Complex.squareLow(yLow, yHigh, yy);
            return Complex.sumx2y2m1(xx, x2Low, yy, y2Low);
        }
        return (x - 1.0) * (x + 1.0) + yy;
    }

    private static double splitHigh(double a) {
        double c = 1.34217729E8 * a;
        return c - (c - a);
    }

    private static double squareLow(double low, double high, double square) {
        double lh = low * high;
        return low * low - (square - high * high - lh - lh);
    }

    private static double fastSumLow(double a, double b, double x) {
        return b - (x - a);
    }

    private static double sumLow(double a, double b, double x) {
        double bVirtual = x - a;
        return a - (x - bVirtual) + (b - bVirtual);
    }

    private static double sumx2y2m1(double x2High, double x2Low, double y2High, double y2Low) {
        double q = x2Low - 1.0;
        double e1 = Complex.fastSumLow(-1.0, x2Low, q);
        double e3 = q + x2High;
        double e2 = Complex.sumLow(q, x2High, e3);
        double f1 = y2Low;
        double f2 = y2High;
        q = f1 + e1;
        e1 = Complex.sumLow(f1, e1, q);
        double p = q + e2;
        e2 = Complex.sumLow(q, e2, p);
        double e4 = p + e3;
        e3 = Complex.sumLow(p, e3, e4);
        q = f2 + e2;
        e2 = Complex.sumLow(f2, e2, q);
        p = q + e3;
        e3 = Complex.sumLow(q, e3, p);
        double e5 = p + e4;
        e4 = Complex.sumLow(p, e4, e5);
        return e1 + e2 + e3 + e4 + e5;
    }

    public List<Complex> nthRoot(int n) {
        if (n == 0) {
            throw new IllegalArgumentException("cannot compute zeroth root");
        }
        ArrayList<Complex> result = new ArrayList<Complex>();
        double nthRootOfAbs = Math.pow(this.abs(), 1.0 / (double)n);
        double nthPhi = this.arg() / (double)n;
        double slice = Math.PI * 2 / (double)n;
        double innerPart = nthPhi;
        for (int k = 0; k < Math.abs(n); ++k) {
            double realPart = nthRootOfAbs * Math.cos(innerPart);
            double imaginaryPart = nthRootOfAbs * Math.sin(innerPart);
            result.add(Complex.ofCartesian(realPart, imaginaryPart));
            innerPart += slice;
        }
        return result;
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof Complex) {
            Complex c = (Complex)other;
            return Complex.equals(this.real, c.real) && Complex.equals(this.imaginary, c.imaginary);
        }
        return false;
    }

    public int hashCode() {
        return 31 * (31 + Double.hashCode(this.real)) + Double.hashCode(this.imaginary);
    }

    public String toString() {
        String sign = this.imaginary >= 0.0 ? "+" : "";
        return new StringBuilder(64).append(this.real).append(sign).append(this.imaginary).append("j").toString();
    }

    private static boolean equals(double x, double y) {
        return Double.doubleToLongBits(x) == Double.doubleToLongBits(y);
    }

    private static boolean negative(double d) {
        return d < 0.0 || Double.doubleToLongBits(d) == NEGATIVE_ZERO_LONG_BITS;
    }

    private static boolean isPosInfinite(double d) {
        return d == Double.POSITIVE_INFINITY;
    }

    private static boolean isPosFinite(double d) {
        return d <= Double.MAX_VALUE;
    }

    private static Complex multiplyNegativeI(double real, double imaginary) {
        return new Complex(imaginary, -real);
    }

    private static double changeSign(double magnitude, double signedValue) {
        return Complex.negative(signedValue) ? -magnitude : magnitude;
    }

    private static int getScale(double a, double b) {
        long y;
        long x = Double.doubleToRawLongBits(a) & Long.MAX_VALUE;
        long bits = Math.max(x, y = Double.doubleToRawLongBits(b) & Long.MAX_VALUE);
        int exp = (int)(bits >>> 52) - 1023;
        if (exp == -1023) {
            if (bits == 0L) {
                return 1024;
            }
            long mantissa = bits & 0xFFFFFFFFFFFFFL;
            exp -= Long.numberOfLeadingZeros(mantissa << 12);
        }
        return exp;
    }

    private static int getMaxExponent(double a, double b) {
        return Math.max(Math.getExponent(a), Math.getExponent(b));
    }

    private static boolean inRegion(double x, double y, double min, double max) {
        return x < max && x > min && y < max && y > min;
    }

    private static double hypot(double x, double y) {
        int hb;
        int ha;
        double b;
        double a;
        long xbits = Double.doubleToRawLongBits(x) & Long.MAX_VALUE;
        long ybits = Double.doubleToRawLongBits(y) & Long.MAX_VALUE;
        if (ybits > xbits) {
            a = y;
            b = x;
            ha = (int)(ybits >>> 32);
            hb = (int)(xbits >>> 32);
        } else {
            a = x;
            b = y;
            ha = (int)(xbits >>> 32);
            hb = (int)(ybits >>> 32);
        }
        if (ha - hb > 0x3600000) {
            return Math.abs(a);
        }
        double rescale = 1.0;
        if (ha > 1596981248) {
            if (ha >= 0x7FF00000) {
                return Math.abs(b) == Double.POSITIVE_INFINITY ? Double.POSITIVE_INFINITY : Math.abs(a);
            }
            a *= 2.409919865102884E-181;
            b *= 2.409919865102884E-181;
            rescale = 4.149515568880993E180;
        } else if (hb < 0x20B00000) {
            if (b == 0.0) {
                return Math.abs(a);
            }
            a *= 4.149515568880993E180;
            b *= 4.149515568880993E180;
            rescale = 2.409919865102884E-181;
        }
        return Math.sqrt(Complex.x2y2(a, b)) * rescale;
    }

    private static double x2y2(double x, double y) {
        double xx = x * x;
        double yy = y * y;
        double xHigh = Complex.splitHigh(x);
        double xLow = x - xHigh;
        double xxLow = Complex.squareLow(xLow, xHigh, xx);
        double yHigh = Complex.splitHigh(y);
        double yLow = y - yHigh;
        double yyLow = Complex.squareLow(yLow, yHigh, yy);
        double r = xx + yy;
        return xx - r + yy + yyLow + xxLow + r;
    }

    @FunctionalInterface
    private static interface ComplexConstructor {
        public Complex create(double var1, double var3);
    }
}

