/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.cli;

import com.airhacks.afterburner.injection.Injector;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.SortedSet;
import java.util.prefs.BackingStoreException;
import org.jabref.cli.AuxCommandLine;
import org.jabref.cli.JabRefCLI;
import org.jabref.gui.externalfiles.AutoSetFileLinksUtil;
import org.jabref.gui.undo.NamedCompound;
import org.jabref.logic.JabRefException;
import org.jabref.logic.UiCommand;
import org.jabref.logic.bibtex.FieldPreferences;
import org.jabref.logic.citationkeypattern.CitationKeyGenerator;
import org.jabref.logic.exporter.AtomicFileWriter;
import org.jabref.logic.exporter.BibWriter;
import org.jabref.logic.exporter.BibtexDatabaseWriter;
import org.jabref.logic.exporter.EmbeddedBibFilePdfExporter;
import org.jabref.logic.exporter.Exporter;
import org.jabref.logic.exporter.ExporterFactory;
import org.jabref.logic.exporter.SelfContainedSaveConfiguration;
import org.jabref.logic.exporter.XmpPdfExporter;
import org.jabref.logic.importer.FetcherException;
import org.jabref.logic.importer.ImportException;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.ImportFormatReader;
import org.jabref.logic.importer.OpenDatabase;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.SearchBasedFetcher;
import org.jabref.logic.importer.WebFetchers;
import org.jabref.logic.importer.fileformat.BibtexParser;
import org.jabref.logic.journals.JournalAbbreviationRepository;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.net.URLDownload;
import org.jabref.logic.search.DatabaseSearcher;
import org.jabref.logic.search.SearchQuery;
import org.jabref.logic.shared.prefs.SharedDatabasePreferences;
import org.jabref.logic.util.OS;
import org.jabref.logic.util.io.FileUtil;
import org.jabref.logic.xmp.XmpPreferences;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibEntryTypesManager;
import org.jabref.model.strings.StringUtil;
import org.jabref.model.util.DummyFileUpdateMonitor;
import org.jabref.model.util.FileUpdateMonitor;
import org.jabref.preferences.FilePreferences;
import org.jabref.preferences.PreferencesService;
import org.jabref.preferences.SearchPreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ArgumentProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(ArgumentProcessor.class);
    private final JabRefCLI cli;
    private final Mode startupMode;
    private final PreferencesService preferencesService;
    private final FileUpdateMonitor fileUpdateMonitor;
    private final BibEntryTypesManager entryTypesManager;
    private boolean guiNeeded;
    private final List<UiCommand> uiCommands = new ArrayList<UiCommand>();

    public ArgumentProcessor(String[] args, Mode startupMode, PreferencesService preferencesService, FileUpdateMonitor fileUpdateMonitor, BibEntryTypesManager entryTypesManager) throws org.apache.commons.cli.ParseException {
        this.cli = new JabRefCLI(args);
        this.startupMode = startupMode;
        this.preferencesService = preferencesService;
        this.fileUpdateMonitor = fileUpdateMonitor;
        this.entryTypesManager = entryTypesManager;
    }

    private Optional<ParserResult> importToOpenBase(String argument) {
        Optional<ParserResult> result = this.importFile(argument);
        result.ifPresent(ParserResult::setToOpenTab);
        return result;
    }

    private Optional<ParserResult> importBibtexToOpenBase(String argument, ImportFormatPreferences importFormatPreferences) {
        BibtexParser parser = new BibtexParser(importFormatPreferences);
        try {
            List<BibEntry> entries = parser.parseEntries(argument);
            ParserResult result = new ParserResult(entries);
            result.setToOpenTab();
            return Optional.of(result);
        }
        catch (ParseException e) {
            System.err.println(Localization.lang("Error occurred when parsing entry", new Object[0]) + ": " + e.getLocalizedMessage());
            return Optional.empty();
        }
    }

    private Optional<ParserResult> importFile(String argument) {
        Path file;
        LOGGER.debug("Importing file {}", (Object)argument);
        String[] data = argument.split(",");
        String address = data[0];
        if (address.startsWith("http://") || address.startsWith("https://") || address.startsWith("ftp://")) {
            try {
                file = new URLDownload(address).toTemporaryFile();
            }
            catch (IOException e) {
                System.err.println(Localization.lang("Problem downloading from %1", address) + e.getLocalizedMessage());
                return Optional.empty();
            }
        } else {
            file = OS.WINDOWS ? Path.of(address, new String[0]) : Path.of(address.replace("~", System.getProperty("user.home")), new String[0]);
        }
        String importFormat = data.length > 1 ? data[1] : "*";
        Optional<ParserResult> importResult = this.importFile(file, importFormat);
        importResult.ifPresent(result -> {
            if (result.hasWarnings()) {
                System.out.println(result.getErrorMessage());
            }
        });
        return importResult;
    }

    private Optional<ParserResult> importFile(Path file, String importFormat) {
        try {
            ImportFormatReader importFormatReader = new ImportFormatReader(this.preferencesService.getImporterPreferences(), this.preferencesService.getImportFormatPreferences(), this.preferencesService.getCitationKeyPatternPreferences(), this.fileUpdateMonitor);
            if (!"*".equals(importFormat)) {
                System.out.println(Localization.lang("Importing %0", file));
                ParserResult result = importFormatReader.importFromFile(importFormat, file);
                return Optional.of(result);
            }
            System.out.println(Localization.lang("Importing file %0 as unknown format", file));
            ImportFormatReader.UnknownFormatImport importResult = importFormatReader.importUnknownFormat(file, new DummyFileUpdateMonitor());
            System.out.println(Localization.lang("Format used: %0", importResult.format()));
            return Optional.of(importResult.parserResult());
        }
        catch (ImportException ex) {
            System.err.println(Localization.lang("Error opening file '%0'", file) + "\n" + ex.getLocalizedMessage());
            return Optional.empty();
        }
    }

    public void processArguments() {
        this.uiCommands.clear();
        if (this.startupMode == Mode.INITIAL_START && this.cli.isShowVersion()) {
            this.cli.displayVersion();
        }
        if (this.startupMode == Mode.INITIAL_START && this.cli.isHelp()) {
            JabRefCLI.printUsage(this.preferencesService);
            this.guiNeeded = false;
            return;
        }
        this.guiNeeded = true;
        if (this.cli.isPreferencesReset()) {
            this.resetPreferences(this.cli.getPreferencesReset());
        }
        if (this.cli.isPreferencesImport()) {
            this.importPreferences();
        }
        List<ParserResult> loaded = this.importAndOpenFiles();
        if (!this.cli.isBlank() && this.cli.isFetcherEngine()) {
            this.fetch(this.cli.getFetcherEngine()).ifPresent(loaded::add);
        }
        if (this.cli.isExportMatches()) {
            if (!loaded.isEmpty()) {
                if (!this.exportMatches(loaded)) {
                    return;
                }
            } else {
                System.err.println(Localization.lang("The output option depends on a valid input option.", new Object[0]));
            }
        }
        if (this.cli.isGenerateCitationKeys()) {
            this.regenerateCitationKeys(loaded);
        }
        if (this.cli.isAutomaticallySetFileLinks()) {
            this.automaticallySetFileLinks(loaded);
        }
        if (this.cli.isWriteXMPtoPdf() && this.cli.isEmbeddBibfileInPdf() || this.cli.isWriteMetadatatoPdf() && (this.cli.isWriteXMPtoPdf() || this.cli.isEmbeddBibfileInPdf())) {
            System.err.println("Give only one of [writeXMPtoPdf, embeddBibfileInPdf, writeMetadatatoPdf]");
        }
        if ((this.cli.isWriteMetadatatoPdf() || this.cli.isWriteXMPtoPdf() || this.cli.isEmbeddBibfileInPdf()) && !loaded.isEmpty()) {
            this.writeMetadataToPdf(loaded, this.cli.getWriteMetadatatoPdf(), this.preferencesService.getXmpPreferences(), this.preferencesService.getFilePreferences(), this.preferencesService.getLibraryPreferences().getDefaultBibDatabaseMode(), (BibEntryTypesManager)Injector.instantiateModelOrService(BibEntryTypesManager.class), this.preferencesService.getFieldPreferences(), (JournalAbbreviationRepository)Injector.instantiateModelOrService(JournalAbbreviationRepository.class), this.cli.isWriteXMPtoPdf() || this.cli.isWriteMetadatatoPdf(), this.cli.isEmbeddBibfileInPdf() || this.cli.isWriteMetadatatoPdf());
        }
        if (this.cli.isFileExport()) {
            if (!loaded.isEmpty()) {
                this.exportFile(loaded, this.cli.getFileExport().split(","));
                LOGGER.debug("Finished export");
            } else {
                System.err.println(Localization.lang("The output option depends on a valid import option.", new Object[0]));
            }
        }
        if (this.cli.isPreferencesExport()) {
            try {
                this.preferencesService.exportPreferences(Path.of(this.cli.getPreferencesExport(), new String[0]));
            }
            catch (JabRefException ex) {
                LOGGER.error("Cannot export preferences", (Throwable)ex);
            }
        }
        if (!this.cli.isBlank() && this.cli.isAuxImport()) {
            this.doAuxImport(loaded);
        }
        if (this.cli.isBlank()) {
            this.uiCommands.add(new UiCommand.BlankWorkspace());
        }
        if (!this.cli.isBlank() && this.cli.isJumpToKey()) {
            this.uiCommands.add(new UiCommand.JumpToEntryKey(this.cli.getJumpToKey()));
        }
        if (!this.cli.isBlank() && !loaded.isEmpty()) {
            this.uiCommands.add(new UiCommand.OpenDatabases(loaded));
        }
    }

    private void writeMetadataToPdf(List<ParserResult> loaded, String filesAndCitekeys, XmpPreferences xmpPreferences, FilePreferences filePreferences, BibDatabaseMode databaseMode, BibEntryTypesManager entryTypesManager, FieldPreferences fieldPreferences, JournalAbbreviationRepository abbreviationRepository, boolean writeXMP, boolean embeddBibfile) {
        if (loaded.isEmpty()) {
            LOGGER.error("The write xmp option depends on a valid import option.");
            return;
        }
        ParserResult pr = loaded.getLast();
        BibDatabaseContext databaseContext = pr.getDatabaseContext();
        XmpPdfExporter xmpPdfExporter = new XmpPdfExporter(xmpPreferences);
        EmbeddedBibFilePdfExporter embeddedBibFilePdfExporter = new EmbeddedBibFilePdfExporter(databaseMode, entryTypesManager, fieldPreferences);
        if ("all".equals(filesAndCitekeys)) {
            for (BibEntry entry : databaseContext.getEntries()) {
                this.writeMetadataToPDFsOfEntry(databaseContext, entry.getCitationKey().orElse("<no cite key defined>"), entry, filePreferences, xmpPdfExporter, embeddedBibFilePdfExporter, abbreviationRepository, writeXMP, embeddBibfile);
            }
            return;
        }
        ArrayList<String> citeKeys = new ArrayList<String>();
        ArrayList<String> pdfs = new ArrayList<String>();
        for (String fileOrCiteKey : filesAndCitekeys.split(",")) {
            if (fileOrCiteKey.toLowerCase(Locale.ROOT).endsWith(".pdf")) {
                pdfs.add(fileOrCiteKey);
                continue;
            }
            citeKeys.add(fileOrCiteKey);
        }
        this.writeMetadataToPdfByCitekey(databaseContext, citeKeys, filePreferences, xmpPdfExporter, embeddedBibFilePdfExporter, abbreviationRepository, writeXMP, embeddBibfile);
        this.writeMetadataToPdfByFileNames(databaseContext, pdfs, filePreferences, xmpPdfExporter, embeddedBibFilePdfExporter, abbreviationRepository, writeXMP, embeddBibfile);
    }

    private void writeMetadataToPDFsOfEntry(BibDatabaseContext databaseContext, String citeKey, BibEntry entry, FilePreferences filePreferences, XmpPdfExporter xmpPdfExporter, EmbeddedBibFilePdfExporter embeddedBibFilePdfExporter, JournalAbbreviationRepository abbreviationRepository, boolean writeXMP, boolean embeddBibfile) {
        try {
            if (writeXMP) {
                if (xmpPdfExporter.exportToAllFilesOfEntry(databaseContext, filePreferences, entry, List.of(entry), abbreviationRepository)) {
                    System.out.printf("Successfully written XMP metadata on at least one linked file of %s%n", citeKey);
                } else {
                    System.err.printf("Cannot write XMP metadata on any linked files of %s. Make sure there is at least one linked file and the path is correct.%n", citeKey);
                }
            }
            if (embeddBibfile) {
                if (embeddedBibFilePdfExporter.exportToAllFilesOfEntry(databaseContext, filePreferences, entry, List.of(entry), abbreviationRepository)) {
                    System.out.printf("Successfully embedded metadata on at least one linked file of %s%n", citeKey);
                } else {
                    System.out.printf("Cannot embedd metadata on any linked files of %s. Make sure there is at least one linked file and the path is correct.%n", citeKey);
                }
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed writing metadata on a linked file of {}.", (Object)citeKey);
        }
    }

    private void writeMetadataToPdfByCitekey(BibDatabaseContext databaseContext, List<String> citeKeys, FilePreferences filePreferences, XmpPdfExporter xmpPdfExporter, EmbeddedBibFilePdfExporter embeddedBibFilePdfExporter, JournalAbbreviationRepository abbreviationRepository, boolean writeXMP, boolean embeddBibfile) {
        for (String citeKey : citeKeys) {
            List<BibEntry> bibEntryList = databaseContext.getDatabase().getEntriesByCitationKey(citeKey);
            if (bibEntryList.isEmpty()) {
                System.err.printf("Skipped - Cannot find %s in library.%n", citeKey);
                continue;
            }
            for (BibEntry entry : bibEntryList) {
                this.writeMetadataToPDFsOfEntry(databaseContext, citeKey, entry, filePreferences, xmpPdfExporter, embeddedBibFilePdfExporter, abbreviationRepository, writeXMP, embeddBibfile);
            }
        }
    }

    private void writeMetadataToPdfByFileNames(BibDatabaseContext databaseContext, List<String> pdfs, FilePreferences filePreferences, XmpPdfExporter xmpPdfExporter, EmbeddedBibFilePdfExporter embeddedBibFilePdfExporter, JournalAbbreviationRepository abbreviationRepository, boolean writeXMP, boolean embeddBibfile) {
        for (String fileName : pdfs) {
            Path filePath = Path.of(fileName, new String[0]);
            if (!filePath.isAbsolute()) {
                filePath = FileUtil.find(fileName, databaseContext.getFileDirectories(filePreferences)).orElse(FileUtil.find(fileName, List.of(Path.of("", new String[0]).toAbsolutePath())).orElse(filePath));
            }
            if (Files.exists(filePath, new LinkOption[0])) {
                try {
                    if (writeXMP) {
                        if (xmpPdfExporter.exportToFileByPath(databaseContext, filePreferences, filePath, abbreviationRepository)) {
                            System.out.printf("Successfully written XMP metadata of at least one entry to %s%n", fileName);
                        } else {
                            System.out.printf("File %s is not linked to any entry in database.%n", fileName);
                        }
                    }
                    if (!embeddBibfile) continue;
                    if (embeddedBibFilePdfExporter.exportToFileByPath(databaseContext, filePreferences, filePath, abbreviationRepository)) {
                        System.out.printf("Successfully embedded XMP metadata of at least one entry to %s%n", fileName);
                        continue;
                    }
                    System.out.printf("File %s is not linked to any entry in database.%n", fileName);
                }
                catch (IOException e) {
                    LOGGER.error("Error accessing file '{}'.", (Object)fileName);
                }
                catch (Exception e) {
                    LOGGER.error("Error writing entry to {}.", (Object)fileName);
                }
                continue;
            }
            LOGGER.error("Skipped - PDF {} does not exist", (Object)fileName);
        }
    }

    private boolean exportMatches(List<ParserResult> loaded) {
        String[] data = this.cli.getExportMatches().split(",");
        String searchTerm = data[0].replace("\\$", " ");
        ParserResult pr = loaded.getLast();
        BibDatabaseContext databaseContext = pr.getDatabaseContext();
        BibDatabase dataBase = pr.getDatabase();
        SearchPreferences searchPreferences = this.preferencesService.getSearchPreferences();
        SearchQuery query = new SearchQuery(searchTerm, searchPreferences.getSearchFlags());
        List<BibEntry> matches = new DatabaseSearcher(query, dataBase).getMatches();
        if (!matches.isEmpty()) {
            String formatName;
            switch (data.length) {
                case 3: {
                    formatName = data[2];
                    break;
                }
                case 2: {
                    formatName = "bib";
                    break;
                }
                default: {
                    System.err.println(Localization.lang("Output file missing", new Object[0]).concat(". \n \t ").concat(Localization.lang("Usage", new Object[0])).concat(": ") + JabRefCLI.getExportMatchesSyntax());
                    this.guiNeeded = false;
                    return false;
                }
            }
            if ("bib".equals(formatName)) {
                this.saveDatabase(new BibDatabase(matches), data[1]);
                LOGGER.debug("Finished export");
            } else {
                ExporterFactory exporterFactory = ExporterFactory.create(this.preferencesService, (BibEntryTypesManager)Injector.instantiateModelOrService(BibEntryTypesManager.class));
                Optional<Exporter> exporter = exporterFactory.getExporterByName(formatName);
                if (exporter.isEmpty()) {
                    System.err.println(Localization.lang("Unknown export format %0", formatName));
                } else {
                    try {
                        System.out.println(Localization.lang("Exporting %0", data[1]));
                        exporter.get().export(databaseContext, Path.of(data[1], new String[0]), matches, Collections.emptyList(), (JournalAbbreviationRepository)Injector.instantiateModelOrService(JournalAbbreviationRepository.class));
                    }
                    catch (Exception ex) {
                        System.err.println(Localization.lang("Could not export file '%0' (reason: %1)", data[1], Throwables.getStackTraceAsString((Throwable)ex)));
                    }
                }
            }
        } else {
            System.err.println(Localization.lang("No search matches.", new Object[0]));
        }
        return true;
    }

    private void doAuxImport(List<ParserResult> loaded) {
        boolean usageMsg = !loaded.isEmpty() ? this.generateAux(loaded, this.cli.getAuxImport().split(",")) : true;
        if (usageMsg) {
            System.out.println(Localization.lang("no base-BibTeX-file specified!", new Object[0]));
            System.out.println(Localization.lang("usage", new Object[0]) + " :");
            System.out.println("jabref --aux infile[.aux],outfile[.bib] base-BibTeX-file");
        }
    }

    private List<ParserResult> importAndOpenFiles() {
        ArrayList<ParserResult> loaded = new ArrayList<ParserResult>();
        ArrayList<String> toImport = new ArrayList<String>();
        if (!this.cli.isBlank() && !this.cli.getLeftOver().isEmpty()) {
            for (String aLeftOver : this.cli.getLeftOver()) {
                boolean bibExtension = aLeftOver.toLowerCase(Locale.ENGLISH).endsWith("bib");
                ParserResult pr = new ParserResult();
                if (bibExtension) {
                    try {
                        pr = OpenDatabase.loadDatabase(Path.of(aLeftOver, new String[0]), this.preferencesService.getImportFormatPreferences(), this.fileUpdateMonitor);
                    }
                    catch (IOException ex) {
                        pr = ParserResult.fromError(ex);
                        LOGGER.error("Error opening file '{}'", (Object)aLeftOver, (Object)ex);
                    }
                }
                if (!bibExtension || pr.isEmpty()) {
                    if (this.startupMode == Mode.INITIAL_START) {
                        toImport.add(aLeftOver);
                        continue;
                    }
                    loaded.add(this.importToOpenBase(aLeftOver).orElse(new ParserResult()));
                    continue;
                }
                loaded.add(pr);
            }
        }
        if (!this.cli.isBlank() && this.cli.isFileImport()) {
            toImport.add(this.cli.getFileImport());
        }
        for (String filenameString : toImport) {
            this.importFile(filenameString).ifPresent(loaded::add);
        }
        if (!this.cli.isBlank() && this.cli.isImportToOpenBase()) {
            this.importToOpenBase(this.cli.getImportToOpenBase()).ifPresent(loaded::add);
        }
        if (!this.cli.isBlank() && this.cli.isBibtexImport()) {
            this.importBibtexToOpenBase(this.cli.getBibtexImport(), this.preferencesService.getImportFormatPreferences()).ifPresent(loaded::add);
        }
        return loaded;
    }

    private boolean generateAux(List<ParserResult> loaded, String[] data) {
        if (data.length == 2) {
            ParserResult pr = loaded.getFirst();
            AuxCommandLine acl = new AuxCommandLine(data[0], pr.getDatabase());
            BibDatabase newBase = acl.perform();
            boolean notSavedMsg = false;
            if (newBase != null && newBase.hasEntries()) {
                String subName = StringUtil.getCorrectFileName(data[1], "bib");
                this.saveDatabase(newBase, subName);
                notSavedMsg = true;
            }
            if (!notSavedMsg) {
                System.out.println(Localization.lang("no library generated", new Object[0]));
            }
            return false;
        }
        return true;
    }

    private void saveDatabase(BibDatabase newBase, String subName) {
        try {
            System.out.println(Localization.lang("Saving", new Object[0]) + ": " + subName);
            try (AtomicFileWriter fileWriter = new AtomicFileWriter(Path.of(subName, new String[0]), StandardCharsets.UTF_8);){
                BibWriter bibWriter = new BibWriter(fileWriter, OS.NEWLINE);
                SelfContainedSaveConfiguration saveConfiguration = (SelfContainedSaveConfiguration)new SelfContainedSaveConfiguration().withReformatOnSave(this.preferencesService.getLibraryPreferences().shouldAlwaysReformatOnSave());
                BibtexDatabaseWriter databaseWriter = new BibtexDatabaseWriter(bibWriter, saveConfiguration, this.preferencesService.getFieldPreferences(), this.preferencesService.getCitationKeyPatternPreferences(), this.entryTypesManager);
                databaseWriter.saveDatabase(new BibDatabaseContext(newBase));
                if (fileWriter.hasEncodingProblems()) {
                    System.err.println(Localization.lang("Warning", new Object[0]) + ": " + Localization.lang("UTF-8 could not be used to encode the following characters: %0", fileWriter.getEncodingProblems()));
                }
            }
        }
        catch (IOException ex) {
            System.err.println(Localization.lang("Could not save file.", new Object[0]) + "\n" + ex.getLocalizedMessage());
        }
    }

    private void exportFile(List<ParserResult> loaded, String[] data) {
        if (data.length == 1) {
            if (!loaded.isEmpty()) {
                ParserResult pr = loaded.getLast();
                if (!pr.isInvalid()) {
                    this.saveDatabase(pr.getDatabase(), data[0]);
                }
            } else {
                System.err.println(Localization.lang("The output option depends on a valid import option.", new Object[0]));
            }
        } else if (data.length == 2) {
            ParserResult parserResult = loaded.getLast();
            Path path = parserResult.getPath().get().toAbsolutePath();
            BibDatabaseContext databaseContext = parserResult.getDatabaseContext();
            databaseContext.setDatabasePath(path);
            List<Path> fileDirForDatabase = databaseContext.getFileDirectories(this.preferencesService.getFilePreferences());
            System.out.println(Localization.lang("Exporting %0", data[0]));
            ExporterFactory exporterFactory = ExporterFactory.create(this.preferencesService, (BibEntryTypesManager)Injector.instantiateModelOrService(BibEntryTypesManager.class));
            Optional<Exporter> exporter = exporterFactory.getExporterByName(data[1]);
            if (exporter.isEmpty()) {
                System.err.println(Localization.lang("Unknown export format %0", data[1]));
            } else {
                try {
                    exporter.get().export(parserResult.getDatabaseContext(), Path.of(data[0], new String[0]), (List<BibEntry>)parserResult.getDatabaseContext().getDatabase().getEntries(), fileDirForDatabase, (JournalAbbreviationRepository)Injector.instantiateModelOrService(JournalAbbreviationRepository.class));
                }
                catch (Exception ex) {
                    System.err.println(Localization.lang("Could not export file '%0' (reason: %1)", data[0], Throwables.getStackTraceAsString((Throwable)ex)));
                }
            }
        }
    }

    private void importPreferences() {
        try {
            this.preferencesService.importPreferences(Path.of(this.cli.getPreferencesImport(), new String[0]));
            Injector.setModelOrService(BibEntryTypesManager.class, (Object)this.preferencesService.getCustomEntryTypesRepository());
        }
        catch (JabRefException ex) {
            LOGGER.error("Cannot import preferences", (Throwable)ex);
        }
    }

    private void resetPreferences(String value) {
        if ("all".equals(value.trim())) {
            try {
                System.out.println(Localization.lang("Setting all preferences to default values.", new Object[0]));
                this.preferencesService.clear();
                new SharedDatabasePreferences().clear();
            }
            catch (BackingStoreException e) {
                System.err.println(Localization.lang("Unable to clear preferences.", new Object[0]));
                LOGGER.error("Unable to clear preferences", (Throwable)e);
            }
        } else {
            String[] keys;
            for (String key : keys = value.split(",")) {
                try {
                    this.preferencesService.deleteKey(key.trim());
                    System.out.println(Localization.lang("Resetting preference key '%0'", key.trim()));
                }
                catch (IllegalArgumentException e) {
                    System.out.println(e.getMessage());
                }
            }
        }
    }

    private void automaticallySetFileLinks(List<ParserResult> loaded) {
        for (ParserResult parserResult : loaded) {
            BibDatabase database = parserResult.getDatabase();
            LOGGER.info(Localization.lang("Automatically setting file links", new Object[0]));
            AutoSetFileLinksUtil util = new AutoSetFileLinksUtil(parserResult.getDatabaseContext(), this.preferencesService.getFilePreferences(), this.preferencesService.getAutoLinkPreferences());
            util.linkAssociatedFiles((List<BibEntry>)database.getEntries(), new NamedCompound(""));
        }
    }

    private void regenerateCitationKeys(List<ParserResult> loaded) {
        for (ParserResult parserResult : loaded) {
            BibDatabase database = parserResult.getDatabase();
            LOGGER.info(Localization.lang("Regenerating citation keys according to metadata", new Object[0]));
            CitationKeyGenerator keyGenerator = new CitationKeyGenerator(parserResult.getDatabaseContext(), this.preferencesService.getCitationKeyPatternPreferences());
            for (BibEntry entry : database.getEntries()) {
                keyGenerator.generateAndSetKey(entry);
            }
        }
    }

    private Optional<ParserResult> fetch(String fetchCommand) {
        if (fetchCommand == null || !fetchCommand.contains(":")) {
            System.out.println(Localization.lang("Expected syntax for --fetch='<name of fetcher>:<query>'", new Object[0]));
            System.out.println(Localization.lang("The following fetchers are available:", new Object[0]));
            return Optional.empty();
        }
        String[] split = fetchCommand.split(":");
        String engine = split[0];
        String query = split[1];
        SortedSet<SearchBasedFetcher> fetchers = WebFetchers.getSearchBasedFetchers(this.preferencesService.getImportFormatPreferences(), this.preferencesService.getImporterPreferences());
        Optional<SearchBasedFetcher> selectedFetcher = fetchers.stream().filter(fetcher -> fetcher.getName().equalsIgnoreCase(engine)).findFirst();
        if (selectedFetcher.isEmpty()) {
            System.out.println(Localization.lang("Could not find fetcher '%0'", engine));
            System.out.println(Localization.lang("The following fetchers are available:", new Object[0]));
            fetchers.forEach(fetcher -> System.out.println("  " + fetcher.getName()));
            return Optional.empty();
        }
        System.out.println(Localization.lang("Running query '%0' with fetcher '%1'.", query, engine));
        System.out.print(Localization.lang("Please wait...", new Object[0]));
        try {
            List<BibEntry> matches = selectedFetcher.get().performSearch(query);
            if (matches.isEmpty()) {
                System.out.println("\r" + Localization.lang("No results found.", new Object[0]));
                return Optional.empty();
            }
            System.out.println("\r" + Localization.lang("Found %0 results.", String.valueOf(matches.size())));
            return Optional.of(new ParserResult(matches));
        }
        catch (FetcherException e) {
            LOGGER.error("Error while fetching", (Throwable)e);
            return Optional.empty();
        }
    }

    public boolean shouldShutDown() {
        return this.cli.isDisableGui() || this.cli.isShowVersion() || !this.guiNeeded;
    }

    public List<UiCommand> getUiCommands() {
        return this.uiCommands;
    }

    public static enum Mode {
        INITIAL_START,
        REMOTE_START;

    }
}

