/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.bibtex;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.jabref.logic.bibtex.FieldWriter;
import org.jabref.logic.bibtex.InvalidFieldValueException;
import org.jabref.logic.bibtex.TypedBibEntry;
import org.jabref.logic.exporter.BibWriter;
import org.jabref.logic.util.OS;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibEntryType;
import org.jabref.model.entry.BibEntryTypesManager;
import org.jabref.model.entry.field.BibField;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.InternalField;
import org.jabref.model.entry.field.OrFields;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BibEntryWriter {
    private static final Logger LOGGER = LoggerFactory.getLogger(BibEntryWriter.class);
    private final BibEntryTypesManager entryTypesManager;
    private final FieldWriter fieldWriter;

    public BibEntryWriter(FieldWriter fieldWriter, BibEntryTypesManager entryTypesManager) {
        this.fieldWriter = fieldWriter;
        this.entryTypesManager = entryTypesManager;
    }

    public String serializeAll(List<BibEntry> entries, BibDatabaseMode databaseMode) throws IOException {
        StringWriter writer = new StringWriter();
        BibWriter bibWriter = new BibWriter(writer, OS.NEWLINE);
        for (BibEntry entry : entries) {
            this.write(entry, bibWriter, databaseMode);
        }
        return writer.toString();
    }

    public void write(BibEntry entry, BibWriter out, BibDatabaseMode bibDatabaseMode) throws IOException {
        this.write(entry, out, bibDatabaseMode, false);
    }

    public void write(BibEntry entry, BibWriter out, BibDatabaseMode bibDatabaseMode, Boolean reformat) throws IOException {
        if (!reformat.booleanValue() && !entry.hasChanged()) {
            out.write(entry.getParsedSerialization());
            out.finishBlock();
            return;
        }
        this.writeUserComments(entry, out);
        this.writeRequiredFieldsFirstRemainingFieldsSecond(entry, out, bibDatabaseMode);
        out.finishBlock();
    }

    private void writeUserComments(BibEntry entry, BibWriter out) throws IOException {
        String userComments = entry.getUserComments();
        if (!userComments.isEmpty()) {
            out.write(userComments);
            out.finishLine();
        }
    }

    private void writeRequiredFieldsFirstRemainingFieldsSecond(BibEntry entry, BibWriter out, BibDatabaseMode bibDatabaseMode) throws IOException {
        TypedBibEntry typedEntry = new TypedBibEntry(entry, bibDatabaseMode);
        out.write("@" + typedEntry.getTypeForDisplay() + "{");
        this.writeKeyField(entry, out);
        HashSet<Field> written = new HashSet<Field>();
        written.add(InternalField.KEY_FIELD);
        int indent = BibEntryWriter.getLengthOfLongestFieldName(entry);
        Optional<BibEntryType> type = this.entryTypesManager.enrich(entry.getType(), bibDatabaseMode);
        if (type.isPresent()) {
            List<Field> requiredFields = type.get().getRequiredFields().stream().map(OrFields::getFields).flatMap(Collection::stream).sorted(Comparator.comparing(Field::getName)).toList();
            for (Field field : requiredFields) {
                this.writeField(entry, out, field, indent);
            }
            written.addAll(requiredFields);
            List<Field> optionalFields = type.get().getOptionalFields().stream().map(BibField::field).sorted(Comparator.comparing(Field::getName)).toList();
            for (Field field : optionalFields) {
                this.writeField(entry, out, field, indent);
            }
            written.addAll(optionalFields);
        }
        SortedSet remainingFields = entry.getFields().stream().filter(key -> !written.contains(key)).collect(Collectors.toCollection(() -> new TreeSet<Field>(Comparator.comparing(Field::getName))));
        for (Field field : remainingFields) {
            this.writeField(entry, out, field, indent);
        }
        out.writeLine("}");
    }

    private void writeKeyField(BibEntry entry, BibWriter out) throws IOException {
        String keyField = StringUtil.shaveString(entry.getCitationKey().orElse(""));
        out.writeLine(keyField + ",");
    }

    private void writeField(BibEntry entry, BibWriter out, Field field, int indent) throws IOException {
        Optional<String> value = entry.getField(field);
        if (value.isPresent() && !value.get().trim().isEmpty()) {
            out.write("  ");
            out.write(BibEntryWriter.getFormattedFieldName(field, indent));
            try {
                out.write(this.fieldWriter.write(field, value.get()));
            }
            catch (InvalidFieldValueException ex) {
                LOGGER.warn("Invalid field value {} of field {} of entry {]", new Object[]{value.get(), field, entry.getCitationKey().orElse(""), ex});
                throw new IOException("Error in field '" + String.valueOf(field) + " of entry " + entry.getCitationKey().orElse("") + "': " + ex.getMessage(), ex);
            }
            out.writeLine(",");
        }
    }

    static int getLengthOfLongestFieldName(BibEntry entry) {
        Predicate<Field> isNotCitationKey = field -> InternalField.KEY_FIELD != field;
        return entry.getFields().stream().filter(isNotCitationKey).mapToInt(field -> field.getName().length()).max().orElse(0);
    }

    static String getFormattedFieldName(Field field, int indent) {
        String fieldName = field.getName();
        return fieldName.toLowerCase(Locale.ROOT) + StringUtil.repeatSpaces(indent - fieldName.length()) + " = ";
    }
}

