/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fetcher;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.hc.core5.net.URIBuilder;
import org.apache.lucene.queryparser.flexible.core.nodes.QueryNode;
import org.jabref.logic.cleanup.EprintCleanup;
import org.jabref.logic.help.HelpFile;
import org.jabref.logic.importer.FetcherException;
import org.jabref.logic.importer.FulltextFetcher;
import org.jabref.logic.importer.IdBasedFetcher;
import org.jabref.logic.importer.IdFetcher;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.PagedSearchBasedFetcher;
import org.jabref.logic.importer.fetcher.DoiFetcher;
import org.jabref.logic.importer.fetcher.TrustLevel;
import org.jabref.logic.importer.fetcher.transformers.ArXivQueryTransformer;
import org.jabref.logic.util.io.XMLUtil;
import org.jabref.logic.util.strings.StringSimilarity;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.KeywordList;
import org.jabref.model.entry.LinkedFile;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.InternalField;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.identifier.ArXivIdentifier;
import org.jabref.model.entry.identifier.DOI;
import org.jabref.model.entry.types.StandardEntryType;
import org.jabref.model.paging.Page;
import org.jabref.model.strings.StringUtil;
import org.jabref.model.util.OptionalUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

public class ArXivFetcher
implements FulltextFetcher,
PagedSearchBasedFetcher,
IdBasedFetcher,
IdFetcher<ArXivIdentifier> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ArXivFetcher.class);
    private static final String DOI_PREFIX = "10.48550/arXiv.";
    private static final Set<Field> CHOSEN_AUTOMATIC_DOI_FIELDS = Set.of(StandardField.KEYWORDS, StandardField.AUTHOR);
    private static final Set<Field> CHOSEN_MANUAL_DOI_FIELDS = Set.of(StandardField.DOI, StandardField.PUBLISHER, InternalField.KEY_FIELD);
    private static final Map<String, String> ARXIV_KEYWORDS_WITH_COMMA_REPLACEMENTS = Collections.unmodifiableMap(Map.of("Computational Engineering, Finance, and Science", "Computational Engineering / Finance / Science", "Distributed, Parallel, and Cluster Computing", "Distributed / Parallel / Cluster Computing"));
    private final ArXiv arXiv;
    private final DoiFetcher doiFetcher;
    private final ImportFormatPreferences importFormatPreferences;

    public ArXivFetcher(ImportFormatPreferences importFormatPreferences) {
        this(importFormatPreferences, new DoiFetcher(importFormatPreferences));
    }

    public ArXivFetcher(ImportFormatPreferences importFormatPreferences, DoiFetcher doiFetcher) {
        this.arXiv = new ArXiv(this, importFormatPreferences);
        this.doiFetcher = doiFetcher;
        this.importFormatPreferences = importFormatPreferences;
    }

    @Override
    public Optional<URL> findFullText(BibEntry entry) throws IOException {
        return this.arXiv.findFullText(entry);
    }

    @Override
    public TrustLevel getTrustLevel() {
        return this.arXiv.getTrustLevel();
    }

    @Override
    public String getName() {
        return this.arXiv.getName();
    }

    @Override
    public Optional<HelpFile> getHelpPage() {
        return this.arXiv.getHelpPage();
    }

    private void adaptKeywordsFrom(BibEntry bibEntry) {
        Optional<String> allKeywords = bibEntry.getField(StandardField.KEYWORDS);
        if (allKeywords.isPresent()) {
            for (Map.Entry<String, String> entry : ARXIV_KEYWORDS_WITH_COMMA_REPLACEMENTS.entrySet()) {
                allKeywords = Optional.of(allKeywords.get().replaceAll(entry.getKey(), entry.getValue()));
            }
            String filteredKeywords = KeywordList.merge(allKeywords.get(), "", this.importFormatPreferences.bibEntryPreferences().getKeywordSeparator()).toString();
            bibEntry.setField(StandardField.KEYWORDS, filteredKeywords);
        }
    }

    private static String getAutomaticDoi(String arXivId) {
        return DOI_PREFIX + arXivId;
    }

    private static Optional<String> getAutomaticDoi(BibEntry arXivBibEntry) {
        Optional<String> entryEprint = arXivBibEntry.getField(StandardField.EPRINT);
        if (entryEprint.isEmpty()) {
            LOGGER.error("Cannot infer ArXiv-issued DOI from BibEntry: no 'EPRINT' field found");
            return Optional.empty();
        }
        return Optional.of(ArXivFetcher.getAutomaticDoi(entryEprint.get()));
    }

    private static String getAutomaticDoi(ArXivIdentifier arXivId) {
        return ArXivFetcher.getAutomaticDoi(arXivId.getNormalizedWithoutVersion());
    }

    private static boolean isManualDoi(String doi) {
        return !doi.toLowerCase().contains(DOI_PREFIX.toLowerCase());
    }

    private static Optional<String> getManualDoi(BibEntry arXivBibEntry) {
        return arXivBibEntry.getField(StandardField.DOI).filter(ArXivFetcher::isManualDoi);
    }

    private static Optional<BibEntry> waitForBibEntryRetrieval(CompletableFuture<Optional<BibEntry>> bibEntryFuture) throws FetcherException {
        try {
            return bibEntryFuture.join();
        }
        catch (CompletionException e) {
            if (!(e.getCause() instanceof FetcherException)) {
                LOGGER.error("The supplied future should only throw a FetcherException.", (Throwable)e);
                throw e;
            }
            throw (FetcherException)e.getCause();
        }
    }

    private void mergeArXivEntryWithFutureDoiEntry(BibEntry arXivEntry, CompletableFuture<Optional<BibEntry>> bibEntryFuture, Set<Field> priorityFields, String id) {
        Optional<BibEntry> bibEntry;
        try {
            bibEntry = ArXivFetcher.waitForBibEntryRetrieval(bibEntryFuture);
        }
        catch (CompletionException | FetcherException e) {
            LOGGER.error("Failed to fetch future BibEntry with id '{}' (skipping merge).", (Object)id, (Object)e);
            return;
        }
        if (bibEntry.isPresent()) {
            this.adaptKeywordsFrom(bibEntry.get());
            arXivEntry.mergeWith(bibEntry.get(), priorityFields);
        } else {
            LOGGER.error("Future BibEntry for id '{}' was completed, but no entry was found (skipping merge).", (Object)id);
        }
    }

    private void inplaceAsyncInfuseArXivWithDoi(BibEntry arXivBibEntry) {
        CompletableFuture<Optional<BibEntry>> arXivBibEntryCompletedFuture = CompletableFuture.completedFuture(Optional.of(arXivBibEntry));
        Optional<ArXivIdentifier> arXivBibEntryId = arXivBibEntry.getField(StandardField.EPRINT).flatMap(ArXivIdentifier::parse);
        try {
            this.inplaceAsyncInfuseArXivWithDoi(arXivBibEntryCompletedFuture, arXivBibEntryId);
        }
        catch (FetcherException e) {
            LOGGER.error("FetcherException should not be found here, as main Bibtex Entry already exists (and failing additional fetches should be skipped)", (Throwable)e);
        }
    }

    private void inplaceAsyncInfuseArXivWithDoi(CompletableFuture<Optional<BibEntry>> arXivBibEntryFuture, Optional<ArXivIdentifier> arXivId) throws FetcherException {
        Optional<BibEntry> arXivBibEntry;
        Optional<CompletableFuture<Optional<BibEntry>>> automaticDoiBibEntryFuture;
        Optional<String> automaticDoi;
        if (arXivId.isPresent()) {
            automaticDoi = Optional.of(ArXivFetcher.getAutomaticDoi(arXivId.get()));
            automaticDoiBibEntryFuture = Optional.of(this.doiFetcher.asyncPerformSearchById(automaticDoi.get()));
            arXivBibEntry = ArXivFetcher.waitForBibEntryRetrieval(arXivBibEntryFuture);
            if (arXivBibEntry.isEmpty()) {
                return;
            }
        } else {
            arXivBibEntry = ArXivFetcher.waitForBibEntryRetrieval(arXivBibEntryFuture);
            if (arXivBibEntry.isEmpty()) {
                return;
            }
            automaticDoi = ArXivFetcher.getAutomaticDoi(arXivBibEntry.get());
            automaticDoiBibEntryFuture = automaticDoi.map(this.arXiv::asyncPerformSearchById);
        }
        Optional<String> manualDoi = ArXivFetcher.getManualDoi(arXivBibEntry.get());
        Optional<CompletableFuture> manualDoiBibEntryFuture = manualDoi.map(this.doiFetcher::asyncPerformSearchById);
        automaticDoiBibEntryFuture.ifPresent(future -> this.mergeArXivEntryWithFutureDoiEntry((BibEntry)arXivBibEntry.get(), (CompletableFuture<Optional<BibEntry>>)future, CHOSEN_AUTOMATIC_DOI_FIELDS, (String)automaticDoi.get()));
        manualDoiBibEntryFuture.ifPresent(future -> this.mergeArXivEntryWithFutureDoiEntry((BibEntry)arXivBibEntry.get(), (CompletableFuture<Optional<BibEntry>>)future, CHOSEN_MANUAL_DOI_FIELDS, (String)manualDoi.get()));
    }

    @Override
    public Page<BibEntry> performSearchPaged(QueryNode luceneQuery, int pageNumber) throws FetcherException {
        Page<BibEntry> result = this.arXiv.performSearchPaged(luceneQuery, pageNumber);
        if (this.doiFetcher == null) {
            return result;
        }
        ExecutorService executor = Executors.newFixedThreadPool(this.getPageSize() * 2);
        List<CompletableFuture> futureSearchResult = result.getContent().stream().map(bibEntry -> CompletableFuture.supplyAsync(() -> {
            this.inplaceAsyncInfuseArXivWithDoi((BibEntry)bibEntry);
            return bibEntry;
        }, executor)).toList();
        Collection modifiedSearchResult = futureSearchResult.stream().map(CompletableFuture::join).collect(Collectors.toList());
        return new Page<BibEntry>(result.getQuery(), result.getPageNumber(), modifiedSearchResult);
    }

    @Override
    public Optional<BibEntry> performSearchById(String identifier) throws FetcherException {
        CompletableFuture<Optional<BibEntry>> arXivBibEntryPromise = this.arXiv.asyncPerformSearchById(identifier);
        if (this.doiFetcher != null) {
            this.inplaceAsyncInfuseArXivWithDoi(arXivBibEntryPromise, ArXivIdentifier.parse(identifier));
        }
        return arXivBibEntryPromise.join();
    }

    @Override
    public Optional<ArXivIdentifier> findIdentifier(BibEntry entry) throws FetcherException {
        return this.arXiv.findIdentifier(entry);
    }

    @Override
    public String getIdentifierName() {
        return this.arXiv.getIdentifierName();
    }

    protected class ArXiv
    implements FulltextFetcher,
    PagedSearchBasedFetcher,
    IdBasedFetcher,
    IdFetcher<ArXivIdentifier> {
        private static final Logger LOGGER = LoggerFactory.getLogger(ArXiv.class);
        private static final String API_URL = "https://export.arxiv.org/api/query";
        private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = DocumentBuilderFactory.newInstance();
        private final ImportFormatPreferences importFormatPreferences;

        public ArXiv(ArXivFetcher this$0, ImportFormatPreferences importFormatPreferences) {
            this.importFormatPreferences = importFormatPreferences;
        }

        @Override
        public Optional<URL> findFullText(BibEntry entry) throws IOException {
            Objects.requireNonNull(entry);
            try {
                Optional<URL> pdfUrl = this.searchForEntries(entry).stream().map(ArXivEntry::getPdfUrl).filter(Optional::isPresent).map(Optional::get).findFirst();
                pdfUrl.ifPresent(url -> LOGGER.info("Fulltext PDF found @ arXiv."));
                return pdfUrl;
            }
            catch (FetcherException e) {
                LOGGER.warn("arXiv API request failed", (Throwable)e);
                return Optional.empty();
            }
        }

        @Override
        public TrustLevel getTrustLevel() {
            return TrustLevel.PREPRINT;
        }

        private Optional<ArXivEntry> searchForEntry(String searchQuery) throws FetcherException {
            List<ArXivEntry> entries = this.queryApi(searchQuery, Collections.emptyList(), 0, 1);
            if (entries.size() == 1) {
                return Optional.of(entries.getFirst());
            }
            return Optional.empty();
        }

        private Optional<ArXivEntry> searchForEntryById(String id) throws FetcherException {
            Optional<ArXivIdentifier> identifier = ArXivIdentifier.parse(id);
            if (identifier.isEmpty()) {
                return Optional.empty();
            }
            List<ArXivEntry> entries = this.queryApi("", Collections.singletonList(identifier.get()), 0, 1);
            if (!entries.isEmpty()) {
                return Optional.of(entries.getFirst());
            }
            return Optional.empty();
        }

        private List<ArXivEntry> searchForEntries(BibEntry originalEntry) throws FetcherException {
            String entryTitle;
            String arxivTitle;
            StringSimilarity match;
            Object query;
            Optional<String> doiString;
            BibEntry entry = (BibEntry)originalEntry.clone();
            new EprintCleanup().cleanup(entry);
            Optional<String> identifier = entry.getField(StandardField.EPRINT);
            if (StringUtil.isNotBlank(identifier)) {
                try {
                    return OptionalUtil.toList(this.searchForEntryById(identifier.get()));
                }
                catch (FetcherException e) {
                    LOGGER.warn("arXiv eprint API request failed", (Throwable)e);
                }
            }
            if ((doiString = entry.getField(StandardField.DOI).flatMap(DOI::parse).map(DOI::getNormalized)).isPresent() && ArXivFetcher.isManualDoi(doiString.get())) {
                query = "doi:" + doiString.get();
            } else {
                Optional<String> authorQuery = entry.getField(StandardField.AUTHOR).map(author -> "au:" + author);
                Optional<String> titleQuery = entry.getField(StandardField.TITLE).map(title -> "ti:" + StringUtil.ignoreCurlyBracket(title));
                query = String.join((CharSequence)"+AND+", OptionalUtil.toList(authorQuery, titleQuery));
            }
            Optional<ArXivEntry> arxivEntry = this.searchForEntry((String)query);
            if (arxivEntry.isPresent() && (match = new StringSimilarity()).isSimilar(arxivTitle = arxivEntry.get().title.orElse(""), entryTitle = StringUtil.ignoreCurlyBracket(entry.getField(StandardField.TITLE).orElse("")))) {
                return OptionalUtil.toList(arxivEntry);
            }
            return Collections.emptyList();
        }

        private List<ArXivEntry> searchForEntries(String searchQuery, int pageNumber) throws FetcherException {
            return this.queryApi(searchQuery, Collections.emptyList(), this.getPageSize() * pageNumber, this.getPageSize());
        }

        private List<ArXivEntry> queryApi(String searchQuery, List<ArXivIdentifier> ids, int start, int maxResults) throws FetcherException {
            Document result = this.callApi(searchQuery, ids, start, maxResults);
            List<Node> entries = XMLUtil.asList(result.getElementsByTagName("entry"));
            return entries.stream().map(ArXivEntry::new).collect(Collectors.toList());
        }

        private Document callApi(String searchQuery, List<ArXivIdentifier> ids, int start, int maxResults) throws FetcherException {
            if (maxResults > 2000) {
                throw new IllegalArgumentException("The arXiv API limits the number of maximal results to be 2000");
            }
            try {
                URIBuilder uriBuilder = new URIBuilder(API_URL);
                if (StringUtil.isNotBlank(searchQuery)) {
                    uriBuilder.addParameter("search_query", StringUtil.stripAccents(searchQuery));
                }
                if (!ids.isEmpty()) {
                    uriBuilder.addParameter("id_list", ids.stream().map(ArXivIdentifier::getNormalized).collect(Collectors.joining(",")));
                }
                uriBuilder.addParameter("start", String.valueOf(start));
                uriBuilder.addParameter("max_results", String.valueOf(maxResults));
                URL url = uriBuilder.build().toURL();
                DocumentBuilder builder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
                HttpURLConnection connection = (HttpURLConnection)url.openConnection();
                if (connection.getResponseCode() == 400) {
                    throw this.getException(builder.parse(connection.getErrorStream()));
                }
                return builder.parse(connection.getInputStream());
            }
            catch (IOException | URISyntaxException | ParserConfigurationException | SAXException exception) {
                throw new FetcherException("arXiv API request failed", exception);
            }
        }

        private FetcherException getException(Document error) {
            Node node;
            Optional<String> id;
            Boolean isError;
            List<Node> entries = XMLUtil.asList(error.getElementsByTagName("entry"));
            if (entries.size() == 1 && (isError = (id = XMLUtil.getNodeContent(node = entries.getFirst(), "id")).map(idContent -> idContent.startsWith("http://arxiv.org/api/errors")).orElse(false)).booleanValue()) {
                String errorMessage = XMLUtil.getNodeContent(node, "summary").orElse("Unknown error");
                return new FetcherException(errorMessage);
            }
            return new FetcherException("arXiv API request failed");
        }

        @Override
        public String getName() {
            return "ArXiv";
        }

        @Override
        public Optional<HelpFile> getHelpPage() {
            return Optional.of(HelpFile.FETCHER_OAI2_ARXIV);
        }

        @Override
        public Page<BibEntry> performSearchPaged(QueryNode luceneQuery, int pageNumber) throws FetcherException {
            ArXivQueryTransformer transformer = new ArXivQueryTransformer();
            String transformedQuery = transformer.transformLuceneQuery(luceneQuery).orElse("");
            List<BibEntry> searchResult = this.searchForEntries(transformedQuery, pageNumber).stream().map(arXivEntry -> arXivEntry.toBibEntry(this.importFormatPreferences.bibEntryPreferences().getKeywordSeparator())).collect(Collectors.toList());
            return new Page<BibEntry>(transformedQuery, pageNumber, this.filterYears(searchResult, transformer));
        }

        private List<BibEntry> filterYears(List<BibEntry> searchResult, ArXivQueryTransformer transformer) {
            return searchResult.stream().filter(entry -> entry.getField(StandardField.DATE).isPresent()).filter(entry -> transformer.getEndYear().isEmpty() || Integer.parseInt(entry.getField(StandardField.DATE).get().substring(0, 4)) <= transformer.getEndYear().get()).filter(entry -> transformer.getStartYear().isEmpty() || Integer.parseInt(entry.getField(StandardField.DATE).get().substring(0, 4)) >= transformer.getStartYear().get()).collect(Collectors.toList());
        }

        protected CompletableFuture<Optional<BibEntry>> asyncPerformSearchById(String identifier) throws CompletionException {
            return CompletableFuture.supplyAsync(() -> {
                try {
                    return this.performSearchById(identifier);
                }
                catch (FetcherException e) {
                    throw new CompletionException(e);
                }
            });
        }

        @Override
        public Optional<BibEntry> performSearchById(String identifier) throws FetcherException {
            return this.searchForEntryById(identifier).map(arXivEntry -> arXivEntry.toBibEntry(this.importFormatPreferences.bibEntryPreferences().getKeywordSeparator()));
        }

        @Override
        public Optional<ArXivIdentifier> findIdentifier(BibEntry entry) throws FetcherException {
            return this.searchForEntries(entry).stream().map(ArXivEntry::getId).filter(Optional::isPresent).map(Optional::get).findFirst();
        }

        @Override
        public String getIdentifierName() {
            return "ArXiv";
        }

        private static class ArXivEntry {
            private final Optional<String> title;
            private final Optional<String> urlAbstractPage;
            private final Optional<String> publishedDate;
            private final Optional<String> abstractText;
            private final List<String> authorNames;
            private final List<String> categories;
            private final Optional<URL> pdfUrl;
            private final Optional<String> doi;
            private final Optional<String> journalReferenceText;
            private final Optional<String> primaryCategory;

            public ArXivEntry(Node item) {
                this.title = XMLUtil.getNodeContent(item, "title").map(ArXivEntry::correctLineBreaks);
                this.urlAbstractPage = XMLUtil.getNodeContent(item, "id");
                this.publishedDate = XMLUtil.getNodeContent(item, "published");
                this.abstractText = XMLUtil.getNodeContent(item, "summary").map(ArXivEntry::correctLineBreaks).map(String::trim);
                this.authorNames = new ArrayList<String>();
                for (Node authorNode : XMLUtil.getNodesByName(item, "author")) {
                    Optional<String> authorName = XMLUtil.getNodeContent(authorNode, "name").map(String::trim);
                    authorName.ifPresent(this.authorNames::add);
                }
                this.categories = new ArrayList<String>();
                for (Node categoryNode : XMLUtil.getNodesByName(item, "category")) {
                    Optional<String> category = XMLUtil.getAttributeContent(categoryNode, "term");
                    category.ifPresent(this.categories::add);
                }
                Optional<Object> pdfUrlParsed = Optional.empty();
                for (Node linkNode : XMLUtil.getNodesByName(item, "link")) {
                    Optional<String> linkTitle = XMLUtil.getAttributeContent(linkNode, "title");
                    if (!linkTitle.equals(Optional.of("pdf"))) continue;
                    pdfUrlParsed = XMLUtil.getAttributeContent(linkNode, "href").map(url -> {
                        try {
                            return new URL((String)url);
                        }
                        catch (MalformedURLException e) {
                            return null;
                        }
                    });
                }
                this.pdfUrl = pdfUrlParsed;
                this.doi = XMLUtil.getNodeContent(item, "arxiv:doi");
                this.journalReferenceText = XMLUtil.getNodeContent(item, "arxiv:journal_ref");
                this.primaryCategory = XMLUtil.getNode(item, "arxiv:primary_category").flatMap(node -> XMLUtil.getAttributeContent(node, "term"));
            }

            public static String correctLineBreaks(String s) {
                String result = s.replaceAll("\\n(?!\\s*\\n)", " ");
                result = result.replaceAll("\\s*\\n\\s*", "\n");
                return result.replaceAll(" {2,}", " ").replaceAll("(^\\s*|\\s+$)", "");
            }

            public Optional<URL> getPdfUrl() {
                return this.pdfUrl;
            }

            public Optional<String> getIdString() {
                return this.urlAbstractPage.flatMap(ArXivIdentifier::parse).map(ArXivIdentifier::getNormalizedWithoutVersion);
            }

            public Optional<ArXivIdentifier> getId() {
                return this.getIdString().flatMap(ArXivIdentifier::parse);
            }

            public Optional<String> getDate() {
                return this.publishedDate.map(date -> {
                    if (date.length() < 10) {
                        return null;
                    }
                    return date.substring(0, 10);
                });
            }

            public BibEntry toBibEntry(Character keywordDelimiter) {
                BibEntry bibEntry = new BibEntry(StandardEntryType.Article);
                bibEntry.setField(StandardField.EPRINTTYPE, "arXiv");
                bibEntry.setField(StandardField.AUTHOR, String.join((CharSequence)" and ", this.authorNames));
                bibEntry.addKeywords(this.categories, keywordDelimiter);
                this.getIdString().ifPresent(id -> bibEntry.setField(StandardField.EPRINT, (String)id));
                this.title.ifPresent(titleContent -> bibEntry.setField(StandardField.TITLE, (String)titleContent));
                this.doi.ifPresent(doiContent -> bibEntry.setField(StandardField.DOI, (String)doiContent));
                this.abstractText.ifPresent(abstractContent -> bibEntry.setField(StandardField.ABSTRACT, (String)abstractContent));
                this.getDate().ifPresent(date -> bibEntry.setField(StandardField.DATE, (String)date));
                this.primaryCategory.ifPresent(category -> bibEntry.setField(StandardField.EPRINTCLASS, (String)category));
                this.journalReferenceText.ifPresent(journal -> bibEntry.setField(StandardField.JOURNAL, (String)journal));
                this.getPdfUrl().ifPresent(url -> bibEntry.setFiles(Collections.singletonList(new LinkedFile((URL)url, "PDF"))));
                return bibEntry;
            }
        }
    }
}

