/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fileformat;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.DateTimeException;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.jabref.logic.importer.AuthorListParser;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.Parser;
import org.jabref.model.entry.AuthorList;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.Date;
import org.jabref.model.entry.LinkedFile;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.types.StandardEntryType;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class MarcXmlParser
implements Parser {
    private static final Logger LOGGER = LoggerFactory.getLogger(MarcXmlParser.class);
    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = DocumentBuilderFactory.newInstance();

    @Override
    public List<BibEntry> parseEntries(InputStream inputStream) throws ParseException {
        try {
            DocumentBuilder documentBuilder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            Document content = documentBuilder.parse(inputStream);
            return this.parseEntries(content);
        }
        catch (IOException | ParserConfigurationException | SAXException exception) {
            throw new ParseException(exception);
        }
    }

    private List<BibEntry> parseEntries(Document content) {
        LinkedList<BibEntry> result = new LinkedList<BibEntry>();
        Element root = (Element)content.getElementsByTagName("zs:searchRetrieveResponse").item(0);
        Element srwrecords = this.getChild("zs:records", root);
        if (srwrecords == null) {
            return result;
        }
        List<Element> records = this.getChildren("zs:record", srwrecords);
        for (Element element : records) {
            Element e = this.getChild("zs:recordData", element);
            if (e == null || (e = this.getChild("record", e)) == null) continue;
            result.add(this.parseEntry(e));
        }
        return result;
    }

    private BibEntry parseEntry(Element element) {
        BibEntry bibEntry = new BibEntry(BibEntry.DEFAULT_TYPE);
        List<Element> datafields = this.getChildren("datafield", element);
        for (Element datafield : datafields) {
            String tag = datafield.getAttribute("tag");
            LOGGER.debug("tag: " + tag);
            if ("020".equals(tag)) {
                this.putIsbn(bibEntry, datafield);
                continue;
            }
            if ("100".equals(tag) || "700".equals(tag) || "710".equals(tag)) {
                this.putPersonalName(bibEntry, datafield);
                continue;
            }
            if ("111".equals(tag)) {
                this.putConferenceDetail(bibEntry, datafield);
                continue;
            }
            if ("245".equals(tag)) {
                this.putTitle(bibEntry, datafield);
                continue;
            }
            if ("250".equals(tag)) {
                this.putEdition(bibEntry, datafield);
                continue;
            }
            if ("264".equals(tag)) {
                this.putPublication(bibEntry, datafield);
                continue;
            }
            if ("300".equals(tag)) {
                this.putPhysicalDescription(bibEntry, datafield);
                continue;
            }
            if ("490".equals(tag) || "830".equals(tag)) {
                this.putSeries(bibEntry, datafield);
                continue;
            }
            if ("502".equals(tag)) {
                this.putThesisDescription(bibEntry, datafield);
                continue;
            }
            if ("520".equals(tag)) {
                this.putSummary(bibEntry, datafield);
                continue;
            }
            if ("653".equals(tag)) {
                this.putKeywords(bibEntry, datafield);
                continue;
            }
            if ("773".equals(tag)) {
                this.putIssue(bibEntry, datafield);
                continue;
            }
            if ("856".equals(tag)) {
                this.putElectronicLocation(bibEntry, datafield);
                continue;
            }
            if ("966".equals(tag)) {
                this.putDoi(bibEntry, datafield);
                continue;
            }
            if (Integer.parseInt(tag) >= 546 && Integer.parseInt(tag) <= 599) {
                this.putNotes(bibEntry, datafield);
                continue;
            }
            LOGGER.debug("Unparsed tag: {}", (Object)tag);
        }
        return bibEntry;
    }

    private void putIsbn(BibEntry bibEntry, Element datafield) {
        String isbn = this.getSubfield("a", datafield);
        if (StringUtil.isNullOrEmpty(isbn)) {
            LOGGER.debug("Empty ISBN recieved");
            return;
        }
        int length = isbn.length();
        if (length != 10 && length != 13) {
            LOGGER.debug("Malformed ISBN recieved, length: " + length);
            return;
        }
        Optional<String> field = bibEntry.getField(StandardField.ISBN);
        if (field.isPresent()) {
            if (field.get().length() == 13) {
                bibEntry.setField(StandardField.ISBN, isbn);
            }
        } else {
            bibEntry.setField(StandardField.ISBN, isbn);
        }
    }

    private void putPersonalName(BibEntry bibEntry, Element datafield) {
        String author = this.getSubfield("a", datafield);
        String relation = this.getSubfield("4", datafield);
        if (StringUtil.isNotBlank(author) && StringUtil.isNotBlank(relation)) {
            AuthorList name = new AuthorListParser().parse(author);
            Optional<StandardField> field = Optional.ofNullable(switch (relation) {
                case "aut" -> StandardField.AUTHOR;
                case "edt" -> StandardField.EDITOR;
                case "pbl" -> StandardField.PUBLISHER;
                default -> null;
            });
            if (field.isPresent()) {
                String ind1 = datafield.getAttribute("ind1");
                Object brackedName = field.get() == StandardField.PUBLISHER && StringUtil.isNotBlank(ind1) && "2".equals(ind1) ? "{" + name.getAsFirstLastNamesWithAnd() + "}" : name.getAsLastFirstNamesWithAnd(false);
                if (bibEntry.getField(field.get()).isPresent()) {
                    bibEntry.setField(field.get(), bibEntry.getField(field.get()).get().concat(" and " + (String)brackedName));
                } else {
                    bibEntry.setField(field.get(), (String)brackedName);
                }
            }
        }
    }

    private void putConferenceDetail(BibEntry bibEntry, Element datafield) {
        String conference = this.getSubfield("a", datafield);
        bibEntry.setType(StandardEntryType.Proceedings);
        if (StringUtil.isNotBlank(conference)) {
            bibEntry.setField(StandardField.EVENTTITLE, conference);
        }
    }

    private void putTitle(BibEntry bibEntry, Element datafield) {
        String title = this.getSubfield("a", datafield);
        String subtitle = this.getSubfield("b", datafield);
        String responsibility = this.getSubfield("c", datafield);
        String number = this.getSubfield("n", datafield);
        String part = this.getSubfield("p", datafield);
        if (StringUtil.isNotBlank(title)) {
            bibEntry.setField(StandardField.TITLE, title);
        }
        if (StringUtil.isNotBlank(subtitle)) {
            bibEntry.setField(StandardField.SUBTITLE, subtitle);
        }
        if (StringUtil.isNotBlank(responsibility)) {
            bibEntry.setField(StandardField.TITLEADDON, responsibility);
        }
        if (StringUtil.isNotBlank(number)) {
            bibEntry.setField(StandardField.NUMBER, number);
        }
        if (StringUtil.isNotBlank(part)) {
            bibEntry.setField(StandardField.PART, part);
        }
    }

    private void putEdition(BibEntry bibEntry, Element datafield) {
        String edition = this.getSubfield("a", datafield);
        String editionAddendum = this.getSubfield("b", datafield);
        if (StringUtil.isNullOrEmpty(edition)) {
            return;
        }
        if (StringUtil.isNotBlank(editionAddendum)) {
            edition = edition.concat(", " + editionAddendum);
        }
        bibEntry.setField(StandardField.EDITION, edition);
    }

    private void putPublication(BibEntry bibEntry, Element datafield) {
        String ind2 = datafield.getAttribute("ind2");
        if (StringUtil.isNotBlank(ind2) && "1".equals(ind2)) {
            String place = this.getSubfield("a", datafield);
            String name = this.getSubfield("b", datafield);
            String date = this.getSubfield("c", datafield);
            if (StringUtil.isNotBlank(place)) {
                bibEntry.setField(StandardField.ADDRESS, place);
            }
            if (StringUtil.isNotBlank(name)) {
                bibEntry.setField(StandardField.PUBLISHER, "{" + name + "}");
            }
            if (StringUtil.isNotBlank(date)) {
                String strippedDate = StringUtil.stripBrackets(date);
                try {
                    Date.parse(strippedDate).ifPresent(bibEntry::setDate);
                }
                catch (DateTimeException exception) {
                    LOGGER.info("Cannot parse date '{}'", (Object)strippedDate);
                    bibEntry.setField(StandardField.DATE, StringUtil.stripBrackets(strippedDate));
                }
            }
        }
    }

    private void putPhysicalDescription(BibEntry bibEntry, Element datafield) {
        String pagetotal = this.getSubfield("a", datafield);
        if (StringUtil.isNotBlank(pagetotal) && (pagetotal.contains("pages") || pagetotal.contains("p.") || pagetotal.contains("S") || pagetotal.contains("Seiten"))) {
            pagetotal = pagetotal.replaceAll(".*?(\\d+)(?:\\s*Seiten|\\s*S|\\s*pages|\\s*p).*", "$1");
            bibEntry.setField(StandardField.PAGETOTAL, pagetotal);
        }
    }

    private void putSeries(BibEntry bibEntry, Element datafield) {
        String name = this.getSubfield("a", datafield);
        String volume = this.getSubfield("v", datafield);
        String issn = this.getSubfield("x", datafield);
        if (StringUtil.isNotBlank(name)) {
            bibEntry.setField(StandardField.SERIES, name);
        }
        if (StringUtil.isNotBlank(volume)) {
            bibEntry.setField(StandardField.VOLUME, volume);
        }
        if (StringUtil.isNotBlank(issn)) {
            bibEntry.setField(StandardField.ISSN, issn);
        }
    }

    private void putThesisDescription(BibEntry bibEntry, Element datafield) {
        String thesisDegree = this.getSubfield("b", datafield);
        String school = this.getSubfield("c", datafield);
        bibEntry.setType(StandardEntryType.MastersThesis);
        if (StringUtil.isNotBlank(school)) {
            bibEntry.setField(StandardField.SCHOOL, school);
        }
        if ("Dissertation".equals(thesisDegree)) {
            bibEntry.setType(StandardEntryType.PhdThesis);
        }
    }

    private void putSummary(BibEntry bibEntry, Element datafield) {
        String summary = this.getSubfield("a", datafield);
        String ind1 = datafield.getAttribute("ind1");
        if (StringUtil.isNotBlank(summary) && StringUtil.isNotBlank(ind1) && "3".equals(ind1)) {
            if (bibEntry.getField(StandardField.ABSTRACT).isPresent()) {
                bibEntry.setField(StandardField.ABSTRACT, bibEntry.getField(StandardField.ABSTRACT).get().concat(summary));
            } else {
                bibEntry.setField(StandardField.ABSTRACT, summary);
            }
        }
    }

    private void putKeywords(BibEntry bibEntry, Element datafield) {
        String keyword = this.getSubfield("a", datafield);
        if (StringUtil.isNotBlank(keyword)) {
            Optional<String> keywords = bibEntry.getField(StandardField.KEYWORDS);
            if (keywords.isPresent()) {
                bibEntry.setField(StandardField.KEYWORDS, keywords.get() + ", " + keyword);
            } else {
                bibEntry.setField(StandardField.KEYWORDS, keyword);
            }
        }
    }

    private void putIssue(BibEntry bibEntry, Element datafield) {
        bibEntry.setType(StandardEntryType.Article);
        List<String> issues = this.getSubfields("g", datafield);
        for (String issue : issues) {
            String[] parts = issue.split(":");
            if (parts.length != 2) continue;
            String key = parts[0].trim();
            String value = parts[1].trim();
            if (!StringUtil.isNotBlank(value)) continue;
            switch (key) {
                case "number": {
                    bibEntry.setField(StandardField.NUMBER, value);
                    break;
                }
                case "year": {
                    bibEntry.setField(StandardField.YEAR, value);
                    break;
                }
                case "pages": {
                    bibEntry.setField(StandardField.PAGES, value);
                    break;
                }
                case "volume": {
                    bibEntry.setField(StandardField.VOLUME, value);
                    break;
                }
                case "day": {
                    bibEntry.setField(StandardField.DAY, value);
                    break;
                }
                case "month": {
                    bibEntry.setField(StandardField.MONTH, value);
                }
            }
        }
    }

    private void putDoi(BibEntry bibEntry, Element datafield) {
        String ind1 = datafield.getAttribute("ind1");
        String resource = this.getSubfield("u", datafield);
        if ("e".equals(ind1) && StringUtil.isNotBlank("u") && StringUtil.isNotBlank(resource)) {
            String fulltext = this.getSubfield("3", datafield);
            if ("Volltext".equals(fulltext)) {
                try {
                    LinkedFile linkedFile = new LinkedFile(new URL(resource), "PDF");
                    bibEntry.setField(StandardField.FILE, linkedFile.toString());
                }
                catch (MalformedURLException e) {
                    LOGGER.info("Malformed URL: {}", (Object)resource);
                }
            } else {
                bibEntry.setField(StandardField.DOI, resource);
            }
        }
    }

    private void putElectronicLocation(BibEntry bibEntry, Element datafield) {
        String ind1 = datafield.getAttribute("ind1");
        String ind2 = datafield.getAttribute("ind2");
        if ("4".equals(ind1) && "0".equals(ind2)) {
            String fulltext = this.getSubfield("3", datafield);
            String resource = this.getSubfield("u", datafield);
            if ("Volltext".equals(fulltext) && StringUtil.isNotBlank(resource)) {
                try {
                    LinkedFile linkedFile = new LinkedFile(new URL(resource), "PDF");
                    bibEntry.setField(StandardField.FILE, linkedFile.toString());
                }
                catch (MalformedURLException e) {
                    LOGGER.info("Malformed URL: {}", (Object)resource);
                }
            } else {
                bibEntry.setField(StandardField.URL, resource);
            }
        }
    }

    private void putNotes(BibEntry bibEntry, Element datafield) {
        String[] notes = new String[]{this.getSubfield("a", datafield), this.getSubfield("0", datafield), this.getSubfield("h", datafield), this.getSubfield("S", datafield), this.getSubfield("c", datafield), this.getSubfield("f", datafield), this.getSubfield("i", datafield), this.getSubfield("k", datafield), this.getSubfield("l", datafield), this.getSubfield("z", datafield), this.getSubfield("3", datafield), this.getSubfield("5", datafield)};
        String notesJoined = Arrays.stream(notes).filter(StringUtil::isNotBlank).collect(Collectors.joining("\n\n"));
        if (bibEntry.getField(StandardField.NOTE).isPresent()) {
            bibEntry.setField(StandardField.NOTE, bibEntry.getField(StandardField.NOTE).get().concat(notesJoined));
        } else {
            bibEntry.setField(StandardField.NOTE, notesJoined);
        }
    }

    private String getSubfield(String a, Element datafield) {
        List<Element> subfields = this.getChildren("subfield", datafield);
        for (Element subfield : subfields) {
            if (!subfield.getAttribute("code").equals(a)) continue;
            return subfield.getTextContent();
        }
        return null;
    }

    private List<String> getSubfields(String a, Element datafield) {
        List<Element> subfields = this.getChildren("subfield", datafield);
        return subfields.stream().filter(field -> field.getAttribute("code").equals(a)).map(Node::getTextContent).toList();
    }

    private Element getChild(String name, Element e) {
        if (e == null) {
            return null;
        }
        NodeList children = e.getChildNodes();
        int j = children.getLength();
        for (int i = 0; i < j; ++i) {
            Element entry;
            Node test = children.item(i);
            if (test.getNodeType() != 1 || !(entry = (Element)test).getTagName().equals(name)) continue;
            return entry;
        }
        return null;
    }

    private List<Element> getChildren(String name, Element e) {
        LinkedList<Element> result = new LinkedList<Element>();
        NodeList children = e.getChildNodes();
        int j = children.getLength();
        for (int i = 0; i < j; ++i) {
            Element entry;
            Node test = children.item(i);
            if (test.getNodeType() != 1 || !(entry = (Element)test).getTagName().equals(name)) continue;
            result.add(entry);
        }
        return result;
    }
}

