/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fileformat;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import org.jabref.logic.importer.Importer;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.Parser;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.fileformat.medline.ArticleId;
import org.jabref.logic.importer.fileformat.medline.Investigator;
import org.jabref.logic.importer.fileformat.medline.MeshHeading;
import org.jabref.logic.importer.fileformat.medline.OtherId;
import org.jabref.logic.importer.fileformat.medline.PersonalNameSubject;
import org.jabref.logic.importer.util.MathMLParser;
import org.jabref.logic.util.StandardFileType;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.Date;
import org.jabref.model.entry.Month;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.FieldFactory;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.field.UnknownField;
import org.jabref.model.entry.types.StandardEntryType;
import org.jabref.model.strings.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MedlineImporter
extends Importer
implements Parser {
    private static final Logger LOGGER = LoggerFactory.getLogger(MedlineImporter.class);
    private static final String KEYWORD_SEPARATOR = "; ";
    private static final Locale ENGLISH = Locale.ENGLISH;
    private final XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance();

    public MedlineImporter() {
        this.xmlInputFactory.setProperty("javax.xml.stream.isCoalescing", true);
        this.xmlInputFactory.setProperty("javax.xml.stream.isNamespaceAware", false);
        this.xmlInputFactory.setProperty("javax.xml.stream.isSupportingExternalEntities", true);
    }

    @Override
    public String getName() {
        return "Medline/PubMed";
    }

    @Override
    public StandardFileType getFileType() {
        return StandardFileType.MEDLINE;
    }

    @Override
    public String getId() {
        return "medline";
    }

    @Override
    public String getDescription() {
        return "Importer for the Medline format.";
    }

    @Override
    public boolean isRecognizedFormat(BufferedReader reader) throws IOException {
        String str;
        for (int i = 0; (str = reader.readLine()) != null && i < 50; ++i) {
            if (!str.toLowerCase(ENGLISH).contains("<pubmedarticle>") && !str.toLowerCase(ENGLISH).contains("<pubmedbookarticle>")) continue;
            return true;
        }
        return false;
    }

    @Override
    public ParserResult importDatabase(BufferedReader input) throws IOException {
        Objects.requireNonNull(input);
        ArrayList<BibEntry> bibItems = new ArrayList<BibEntry>();
        try {
            XMLStreamReader reader = this.xmlInputFactory.createXMLStreamReader(input);
            while (reader.hasNext()) {
                String elementName;
                reader.next();
                if (!this.isStartXMLEvent(reader)) continue;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "PubmedArticle": {
                        this.parseArticle(reader, bibItems, elementName);
                        break;
                    }
                    case "PubmedBookArticle": {
                        this.parseBookArticle(reader, bibItems, elementName);
                    }
                }
            }
        }
        catch (XMLStreamException e) {
            LOGGER.debug("could not parse document", (Throwable)e);
            return ParserResult.fromError(e);
        }
        return new ParserResult(bibItems);
    }

    private void parseBookArticle(XMLStreamReader reader, List<BibEntry> bibItems, String startElement) throws XMLStreamException {
        HashMap<Field, String> fields = new HashMap<Field, String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "BookDocument": {
                        this.parseBookDocument(reader, fields, elementName);
                        break;
                    }
                    case "PublicationStatus": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.PUBSTATE, reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        BibEntry entry = new BibEntry(StandardEntryType.Article);
        entry.setField(fields);
        bibItems.add(entry);
    }

    private void parseBookDocument(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> sectionTitleList = new ArrayList<String>();
        ArrayList<String> keywordList = new ArrayList<String>();
        ArrayList<String> publicationTypeList = new ArrayList<String>();
        ArrayList<String> articleTitleList = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "PMID": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        fields.put(StandardField.PMID, reader.getText());
                        break;
                    }
                    case "DateRevised": 
                    case "ContributionDate": {
                        this.parseDate(reader, fields, elementName);
                        break;
                    }
                    case "Abstract": {
                        this.addAbstract(reader, fields, elementName);
                        break;
                    }
                    case "Pagination": {
                        this.addPagination(reader, fields, elementName);
                        break;
                    }
                    case "Section": {
                        this.parseSections(reader, sectionTitleList);
                        break;
                    }
                    case "Keyword": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        keywordList.add(reader.getText());
                        break;
                    }
                    case "PublicationType": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        publicationTypeList.add(reader.getText());
                        break;
                    }
                    case "ArticleTitle": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        articleTitleList.add(reader.getText());
                        break;
                    }
                    case "Book": {
                        this.parseBookInformation(reader, fields, elementName);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        if (!sectionTitleList.isEmpty()) {
            fields.put(new UnknownField("sections"), String.join((CharSequence)KEYWORD_SEPARATOR, sectionTitleList));
        }
        this.addKeywords(fields, keywordList);
        if (!publicationTypeList.isEmpty()) {
            fields.put(new UnknownField("pubtype"), String.join((CharSequence)", ", publicationTypeList));
        }
        if (!articleTitleList.isEmpty()) {
            fields.put(new UnknownField("article"), String.join((CharSequence)", ", articleTitleList));
        }
    }

    private void parseBookInformation(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> isbnList = new ArrayList<String>();
        ArrayList<String> titleList = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "PublisherName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.PUBLISHER, reader.getText());
                        break;
                    }
                    case "PublisherLocation": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("publocation"), reader.getText());
                        break;
                    }
                    case "BookTitle": {
                        this.handleTextElement(reader, titleList, elementName);
                        break;
                    }
                    case "PubDate": {
                        this.addPubDate(reader, fields, elementName);
                        break;
                    }
                    case "AuthorList": {
                        this.handleAuthorList(reader, fields, elementName);
                        break;
                    }
                    case "Volume": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.VOLUME, reader.getText());
                        break;
                    }
                    case "Edition": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.EDITION, reader.getText());
                        break;
                    }
                    case "Medium": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("medium"), reader.getText());
                        break;
                    }
                    case "ReportNumber": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("reportnumber"), reader.getText());
                        break;
                    }
                    case "ELocationID": {
                        String eidType = reader.getAttributeValue(null, "EIdType");
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.handleElocationId(fields, reader, eidType);
                        break;
                    }
                    case "Isbn": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        isbnList.add(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        if (!isbnList.isEmpty()) {
            fields.put(StandardField.ISBN, String.join((CharSequence)", ", isbnList));
        }
        if (!titleList.isEmpty()) {
            this.putIfValueNotNull(fields, StandardField.TITLE, String.join((CharSequence)" ", titleList));
        }
    }

    private void handleElocationId(Map<Field, String> fields, XMLStreamReader reader, String eidType) {
        if ("doi".equals(eidType)) {
            fields.put(StandardField.DOI, reader.getText());
        }
        if ("pii".equals(eidType)) {
            fields.put(new UnknownField("pii"), reader.getText());
        }
    }

    private void parseSections(XMLStreamReader reader, List<String> sectionTitleList) throws XMLStreamException {
        int sectionLevel = 0;
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "SectionTitle": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader) || sectionLevel != 0) break;
                        sectionTitleList.add(reader.getText());
                        break;
                    }
                    case "Section": {
                        ++sectionLevel;
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"Section".equals(reader.getName().getLocalPart())) continue;
            if (sectionLevel == 0) break;
            --sectionLevel;
        }
    }

    private void parseArticle(XMLStreamReader reader, List<BibEntry> bibItems, String startElement) throws XMLStreamException {
        HashMap<Field, String> fields = new HashMap<Field, String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "MedlineCitation": {
                        this.parseMedlineCitation(reader, fields, elementName);
                        break;
                    }
                    case "PubmedData": {
                        this.parsePubmedData(reader, fields, elementName);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        BibEntry entry = new BibEntry(StandardEntryType.Article);
        entry.setField(fields);
        bibItems.add(entry);
    }

    private void parsePubmedData(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        String publicationStatus = "";
        ArrayList<ArticleId> articleIdList = new ArrayList<ArticleId>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "PublicationStatus": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        publicationStatus = reader.getText();
                        break;
                    }
                    case "ArticleId": {
                        String idType = reader.getAttributeValue(null, "IdType");
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        articleIdList.add(new ArticleId(idType, reader.getText()));
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        if (fields.get(new UnknownField("revised")) != null) {
            this.putIfValueNotNull(fields, StandardField.PUBSTATE, publicationStatus);
            if (!articleIdList.isEmpty()) {
                this.addArticleIdList(fields, articleIdList);
            }
        }
    }

    private void parseMedlineCitation(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> citationSubsets = new ArrayList<String>();
        ArrayList<MeshHeading> meshHeadingList = new ArrayList<MeshHeading>();
        ArrayList<PersonalNameSubject> personalNameSubjectList = new ArrayList<PersonalNameSubject>();
        ArrayList<OtherId> otherIdList = new ArrayList<OtherId>();
        ArrayList<String> keywordList = new ArrayList<String>();
        ArrayList<String> spaceFlightMissionList = new ArrayList<String>();
        ArrayList<Investigator> investigatorList = new ArrayList<Investigator>();
        ArrayList<String> generalNoteList = new ArrayList<String>();
        String status = reader.getAttributeValue(null, "Status");
        String owner = reader.getAttributeValue(null, "Owner");
        int latestVersion = 0;
        fields.put(new UnknownField("status"), status);
        fields.put(StandardField.OWNER, owner);
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "DateCreated": 
                    case "DateCompleted": 
                    case "DateRevised": {
                        this.parseDate(reader, fields, elementName);
                        break;
                    }
                    case "Article": {
                        this.parseArticleInformation(reader, fields);
                        break;
                    }
                    case "PMID": {
                        String versionStr = reader.getAttributeValue(null, "Version");
                        reader.next();
                        if (versionStr == null) break;
                        int version = Integer.parseInt(versionStr);
                        if (!this.isCharacterXMLEvent(reader) || version <= latestVersion) break;
                        latestVersion = version;
                        fields.put(StandardField.PMID, reader.getText());
                        break;
                    }
                    case "MedlineJournalInfo": {
                        this.parseMedlineJournalInfo(reader, fields, elementName);
                        break;
                    }
                    case "ChemicalList": {
                        this.parseChemicalList(reader, fields, elementName);
                        break;
                    }
                    case "CitationSubset": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        citationSubsets.add(reader.getText());
                        break;
                    }
                    case "GeneSymbolList": {
                        this.parseGeneSymbolList(reader, fields, elementName);
                        break;
                    }
                    case "MeshHeading": {
                        this.parseMeshHeading(reader, meshHeadingList, elementName);
                        break;
                    }
                    case "NumberOfReferences": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("references"), reader.getText());
                        break;
                    }
                    case "PersonalNameSubject": {
                        this.parsePersonalNameSubject(reader, personalNameSubjectList, elementName);
                        break;
                    }
                    case "OtherID": {
                        String otherIdSource = reader.getAttributeValue(null, "Source");
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        String content = reader.getText();
                        otherIdList.add(new OtherId(otherIdSource, content));
                        break;
                    }
                    case "Keyword": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        keywordList.add(reader.getText());
                        break;
                    }
                    case "SpaceFlightMission": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        spaceFlightMissionList.add(reader.getText());
                        break;
                    }
                    case "Investigator": {
                        this.parseInvestigator(reader, investigatorList, elementName);
                        break;
                    }
                    case "GeneralNote": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        generalNoteList.add(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        if (!citationSubsets.isEmpty()) {
            fields.put(new UnknownField("citation-subset"), String.join((CharSequence)", ", citationSubsets));
        }
        this.addMeshHeading(fields, meshHeadingList);
        this.addPersonalNames(fields, personalNameSubjectList);
        this.addOtherId(fields, otherIdList);
        this.addKeywords(fields, keywordList);
        if (!spaceFlightMissionList.isEmpty()) {
            fields.put(new UnknownField("space-flight-mission"), String.join((CharSequence)", ", spaceFlightMissionList));
        }
        this.addInvestigators(fields, investigatorList);
        this.addNotes(fields, generalNoteList);
    }

    private void parseInvestigator(XMLStreamReader reader, List<Investigator> investigatorList, String startElement) throws XMLStreamException {
        String lastName = "";
        String foreName = "";
        ArrayList<String> affiliationList = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "LastName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        lastName = reader.getText();
                        break;
                    }
                    case "ForeName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        foreName = reader.getText();
                        break;
                    }
                    case "Affiliation": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        affiliationList.add(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        investigatorList.add(new Investigator(lastName, foreName, affiliationList));
    }

    private void parsePersonalNameSubject(XMLStreamReader reader, List<PersonalNameSubject> personalNameSubjectList, String startElement) throws XMLStreamException {
        String lastName = "";
        String foreName = "";
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "LastName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        lastName = reader.getText();
                        break;
                    }
                    case "ForeName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        foreName = reader.getText();
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        personalNameSubjectList.add(new PersonalNameSubject(lastName, foreName));
    }

    private void parseMeshHeading(XMLStreamReader reader, List<MeshHeading> meshHeadingList, String startElement) throws XMLStreamException {
        String descriptorName = "";
        ArrayList<String> qualifierNames = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "DescriptorName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        descriptorName = reader.getText();
                        break;
                    }
                    case "QualifierName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        qualifierNames.add(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        meshHeadingList.add(new MeshHeading(descriptorName, qualifierNames));
    }

    private void parseGeneSymbolList(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> geneSymbols = new ArrayList<String>();
        while (reader.hasNext()) {
            String elementName;
            reader.next();
            if (this.isStartXMLEvent(reader) && "GeneSymbol".equals(elementName = reader.getName().getLocalPart())) {
                reader.next();
                if (this.isCharacterXMLEvent(reader)) {
                    geneSymbols.add(reader.getText());
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        if (!geneSymbols.isEmpty()) {
            fields.put(new UnknownField("gene-symbols"), String.join((CharSequence)", ", geneSymbols));
        }
    }

    private void parseChemicalList(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> chemicalNames = new ArrayList<String>();
        while (reader.hasNext()) {
            String elementName;
            reader.next();
            if (this.isStartXMLEvent(reader) && "NameOfSubstance".equals(elementName = reader.getName().getLocalPart())) {
                reader.next();
                if (this.isCharacterXMLEvent(reader)) {
                    chemicalNames.add(reader.getText());
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        fields.put(new UnknownField("chemicals"), String.join((CharSequence)", ", chemicalNames));
    }

    private void parseMedlineJournalInfo(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "Country": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("country"), reader.getText());
                        break;
                    }
                    case "MedlineTA": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("journal-abbreviation"), reader.getText());
                        break;
                    }
                    case "NlmUniqueID": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("nlm-id"), reader.getText());
                        break;
                    }
                    case "ISSNLinking": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("issn-linking"), reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
            break;
        }
    }

    private void parseArticleInformation(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        ArrayList<String> titleList = new ArrayList<String>();
        String pubmodel = reader.getAttributeValue(null, "PubModel");
        fields.put(new UnknownField("pubmodel"), pubmodel);
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "Journal": {
                        this.parseJournal(reader, fields);
                        break;
                    }
                    case "ArticleTitle": {
                        this.handleTextElement(reader, titleList, elementName);
                        break;
                    }
                    case "Pagination": {
                        this.addPagination(reader, fields, elementName);
                        break;
                    }
                    case "ELocationID": {
                        String eidType = reader.getAttributeValue(null, "EIdType");
                        String validYN = reader.getAttributeValue(null, "ValidYN");
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader) || !"Y".equals(validYN)) break;
                        this.handleElocationId(fields, reader, eidType);
                        break;
                    }
                    case "Abstract": {
                        this.addAbstract(reader, fields, elementName);
                        break;
                    }
                    case "AuthorList": {
                        this.handleAuthorList(reader, fields, elementName);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"Article".equals(reader.getName().getLocalPart())) continue;
        }
        if (!titleList.isEmpty()) {
            fields.put(StandardField.TITLE, StringUtil.stripBrackets(String.join((CharSequence)" ", titleList)));
        }
    }

    private void parseJournal(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "Title": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.JOURNAL, reader.getText());
                        break;
                    }
                    case "ISSN": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.ISSN, reader.getText());
                        break;
                    }
                    case "Volume": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.VOLUME, reader.getText());
                        break;
                    }
                    case "Issue": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.ISSUE, reader.getText());
                        break;
                    }
                    case "PubDate": {
                        this.addPubDate(reader, fields, elementName);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"Journal".equals(reader.getName().getLocalPart())) continue;
            break;
        }
    }

    private void parseDate(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        Optional<String> year = Optional.empty();
        Optional<String> month = Optional.empty();
        Optional<String> day = Optional.empty();
        Map<String, String> dateFieldMap = Map.of("DateCreated", "created", "DateCompleted", "completed", "DateRevised", "revised", "ContributionDate", "contribution", "PubDate", "");
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "Year": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        year = Optional.of(reader.getText());
                        break;
                    }
                    case "Month": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        month = Optional.of(reader.getText());
                        break;
                    }
                    case "Day": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        day = Optional.of(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        Optional<Date> date = Date.parse(year, month, day);
        date.ifPresent(dateValue -> fields.put(new UnknownField((String)dateFieldMap.get(startElement)), dateValue.getNormalized()));
    }

    private void addArticleIdList(Map<Field, String> fields, List<ArticleId> articleIdList) {
        for (ArticleId id : articleIdList) {
            if (id.idType().isBlank()) continue;
            if ("pubmed".equals(id.idType())) {
                fields.computeIfAbsent(StandardField.PMID, k -> id.content());
                continue;
            }
            fields.computeIfAbsent(FieldFactory.parseField(StandardEntryType.Article, id.idType()), k -> id.content());
        }
    }

    private void addNotes(Map<Field, String> fields, List<String> generalNoteList) {
        ArrayList<String> notes = new ArrayList<String>();
        for (String note : generalNoteList) {
            if (note.isBlank()) continue;
            notes.add(note);
        }
        if (!notes.isEmpty()) {
            fields.put(StandardField.NOTE, String.join((CharSequence)", ", notes));
        }
    }

    private void addInvestigators(Map<Field, String> fields, List<Investigator> investigatorList) {
        ArrayList<String> investigatorNames = new ArrayList<String>();
        ArrayList<String> affiliationInfos = new ArrayList<String>();
        if (!investigatorList.isEmpty()) {
            for (Investigator investigator : investigatorList) {
                StringBuilder result = new StringBuilder(investigator.lastName());
                if (!investigator.foreName().isBlank()) {
                    result.append(", ").append(investigator.foreName());
                }
                investigatorNames.add(result.toString());
                if (investigator.affiliationList().isEmpty()) continue;
                affiliationInfos.addAll(investigator.affiliationList());
            }
            if (!affiliationInfos.isEmpty()) {
                fields.put(new UnknownField("affiliation"), String.join((CharSequence)", ", affiliationInfos));
            }
            fields.put(new UnknownField("investigator"), String.join((CharSequence)" and ", investigatorNames));
        }
    }

    private void addKeywords(Map<Field, String> fields, List<String> keywordList) {
        if (fields.get(StandardField.KEYWORDS) == null) {
            fields.put(StandardField.KEYWORDS, String.join((CharSequence)KEYWORD_SEPARATOR, keywordList));
        } else if (!keywordList.isEmpty()) {
            Object result = String.join((CharSequence)KEYWORD_SEPARATOR, keywordList);
            result = fields.get(StandardField.KEYWORDS) + KEYWORD_SEPARATOR + (String)result;
            fields.put(StandardField.KEYWORDS, (String)result);
        }
    }

    private void addOtherId(Map<Field, String> fields, List<OtherId> otherIdList) {
        for (OtherId id : otherIdList) {
            if (id.source().isBlank() || id.content().isBlank()) continue;
            fields.put(FieldFactory.parseField(StandardEntryType.Article, id.source()), id.content());
        }
    }

    private void addPersonalNames(Map<Field, String> fields, List<PersonalNameSubject> personalNameSubjectList) {
        if (fields.get(StandardField.AUTHOR) == null) {
            ArrayList<String> personalNames = new ArrayList<String>();
            if (!personalNameSubjectList.isEmpty()) {
                for (PersonalNameSubject personalNameSubject : personalNameSubjectList) {
                    StringBuilder result = new StringBuilder(personalNameSubject.lastName());
                    if (!personalNameSubject.foreName().isBlank()) {
                        result.append(", ").append(personalNameSubject.foreName());
                    }
                    personalNames.add(result.toString());
                }
                fields.put(StandardField.AUTHOR, String.join((CharSequence)" and ", personalNames));
            }
        }
    }

    private void addMeshHeading(Map<Field, String> fields, List<MeshHeading> meshHeadingList) {
        ArrayList<String> keywords = new ArrayList<String>();
        if (!meshHeadingList.isEmpty()) {
            for (MeshHeading meshHeading : meshHeadingList) {
                StringBuilder result = new StringBuilder(meshHeading.descriptorName());
                if (meshHeading.qualifierNames() != null) {
                    for (String qualifierName : meshHeading.qualifierNames()) {
                        result.append(", ").append(qualifierName);
                    }
                }
                keywords.add(result.toString());
            }
            fields.put(StandardField.KEYWORDS, String.join((CharSequence)KEYWORD_SEPARATOR, keywords));
        }
    }

    private void addPubDate(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "MedlineDate": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        fields.put(StandardField.YEAR, this.extractYear(reader.getText()));
                        break;
                    }
                    case "Year": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        fields.put(StandardField.YEAR, reader.getText());
                        break;
                    }
                    case "Month": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        Optional<Month> month = Month.parse(reader.getText());
                        month.ifPresent(monthValue -> fields.put(StandardField.MONTH, monthValue.getJabRefFormat()));
                        break;
                    }
                    case "Season": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        fields.put(new UnknownField("season"), reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
            break;
        }
    }

    private void addAbstract(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> abstractTextList = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "CopyrightInformation": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("copyright"), reader.getText());
                        break;
                    }
                    case "AbstractText": {
                        this.handleTextElement(reader, abstractTextList, elementName);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        if (!abstractTextList.isEmpty()) {
            fields.put(StandardField.ABSTRACT, String.join((CharSequence)" ", abstractTextList));
        }
    }

    private void handleTextElement(XMLStreamReader reader, List<String> textList, String startElement) throws XMLStreamException {
        StringBuilder result = new StringBuilder();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "math": {
                        result.append(MathMLParser.parse(reader));
                        break;
                    }
                    case "sup": 
                    case "sub": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        result.append("(").append(reader.getText()).append(")");
                    }
                }
            } else if (this.isCharacterXMLEvent(reader)) {
                result.append(reader.getText().trim()).append(" ");
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        textList.add(result.toString().trim());
    }

    private void addPagination(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        Object startPage = "";
        String endPage = "";
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "MedlinePgn": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.PAGES, this.fixPageRange(reader.getText()));
                        break;
                    }
                    case "StartPage": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        startPage = reader.getText() + endPage;
                        this.putIfValueNotNull(fields, StandardField.PAGES, (String)startPage);
                        break;
                    }
                    case "EndPage": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        endPage = reader.getText();
                        fields.put(StandardField.PAGES, this.fixPageRange((String)startPage + "-" + endPage));
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
            break;
        }
    }

    private String extractYear(String medlineDate) {
        return medlineDate.substring(0, 4);
    }

    private void handleAuthorList(XMLStreamReader reader, Map<Field, String> fields, String startElement) throws XMLStreamException {
        ArrayList<String> authorNames = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "Author": {
                        this.parseAuthor(reader, authorNames);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !reader.getName().getLocalPart().equals(startElement)) continue;
        }
        fields.put(StandardField.AUTHOR, String.join((CharSequence)" and ", authorNames));
    }

    private void parseAuthor(XMLStreamReader reader, List<String> authorNames) throws XMLStreamException {
        StringBuilder authorName = new StringBuilder();
        ArrayList<String> collectiveNames = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "CollectiveName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        collectiveNames.add(reader.getText());
                        break;
                    }
                    case "LastName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        authorName = new StringBuilder(reader.getText());
                        break;
                    }
                    case "ForeName": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        authorName.append(", ").append(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"Author".equals(reader.getName().getLocalPart())) continue;
        }
        if (!collectiveNames.isEmpty()) {
            authorNames.addAll(collectiveNames);
        }
        if (!authorName.toString().isBlank()) {
            authorNames.add(authorName.toString());
        }
    }

    private void putIfValueNotNull(Map<Field, String> fields, Field field, String value) {
        if (value != null) {
            fields.put(field, value);
        }
    }

    private String fixPageRange(String pageRange) {
        int lengthOfStartPage;
        int minusPos = pageRange.indexOf(45);
        if (minusPos < 0) {
            return pageRange;
        }
        String startPage = pageRange.substring(0, minusPos).trim();
        Object endPage = pageRange.substring(minusPos + 1).trim();
        int lengthOfEndPage = ((String)endPage).length();
        if (lengthOfEndPage < (lengthOfStartPage = startPage.length())) {
            endPage = startPage.substring(0, lengthOfStartPage - lengthOfEndPage) + (String)endPage;
        }
        return startPage + "--" + (String)endPage;
    }

    private boolean isCharacterXMLEvent(XMLStreamReader reader) {
        return reader.getEventType() == 4;
    }

    private boolean isStartXMLEvent(XMLStreamReader reader) {
        return reader.getEventType() == 1;
    }

    private boolean isEndXMLEvent(XMLStreamReader reader) {
        return reader.getEventType() == 2;
    }

    @Override
    public List<BibEntry> parseEntries(InputStream inputStream) throws ParseException {
        try {
            return this.importDatabase(new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))).getDatabase().getEntries();
        }
        catch (IOException e) {
            LOGGER.error(e.getLocalizedMessage(), (Throwable)e);
            return Collections.emptyList();
        }
    }
}

