/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.importer.fileformat;

import com.google.common.base.Joiner;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import org.jabref.logic.importer.ImportFormatPreferences;
import org.jabref.logic.importer.Importer;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.Parser;
import org.jabref.logic.importer.ParserResult;
import org.jabref.logic.importer.fileformat.mods.Identifier;
import org.jabref.logic.importer.fileformat.mods.Name;
import org.jabref.logic.importer.fileformat.mods.RecordInfo;
import org.jabref.logic.util.StandardFileType;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.Date;
import org.jabref.model.entry.field.Field;
import org.jabref.model.entry.field.FieldFactory;
import org.jabref.model.entry.field.StandardField;
import org.jabref.model.entry.field.UnknownField;
import org.jabref.model.entry.types.EntryTypeFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModsImporter
extends Importer
implements Parser {
    private static final Logger LOGGER = LoggerFactory.getLogger(ModsImporter.class);
    private static final Pattern MODS_PATTERN = Pattern.compile("<mods .*>");
    private final String keywordSeparator;
    private final XMLInputFactory xmlInputFactory;

    public ModsImporter(ImportFormatPreferences importFormatPreferences) {
        this.keywordSeparator = importFormatPreferences.bibEntryPreferences().getKeywordSeparator() + " ";
        this.xmlInputFactory = XMLInputFactory.newInstance();
    }

    @Override
    public boolean isRecognizedFormat(BufferedReader input) throws IOException {
        return input.lines().anyMatch(line -> MODS_PATTERN.matcher((CharSequence)line).find());
    }

    @Override
    public ParserResult importDatabase(BufferedReader input) throws IOException {
        Objects.requireNonNull(input);
        ArrayList<BibEntry> bibItems = new ArrayList<BibEntry>();
        try {
            XMLStreamReader reader = this.xmlInputFactory.createXMLStreamReader(input);
            this.parseModsCollection(bibItems, reader);
        }
        catch (XMLStreamException e) {
            LOGGER.debug("could not parse document", (Throwable)e);
            return ParserResult.fromError(e);
        }
        return new ParserResult(bibItems);
    }

    private void parseModsCollection(List<BibEntry> bibItems, XMLStreamReader reader) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (!this.isStartXMLEvent(reader) || !"mods".equals(reader.getName().getLocalPart())) continue;
            BibEntry entry = new BibEntry();
            HashMap<Field, String> fields = new HashMap<Field, String>();
            String id = reader.getAttributeValue(null, "ID");
            if (id != null) {
                entry.setCitationKey(id);
            }
            this.parseModsGroup(fields, reader, entry);
            entry.setField(fields);
            bibItems.add(entry);
        }
    }

    private void parseModsGroup(Map<Field, String> fields, XMLStreamReader reader, BibEntry entry) throws XMLStreamException {
        ArrayList<String> notes = new ArrayList<String>();
        ArrayList<String> keywords = new ArrayList<String>();
        ArrayList<String> authors = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "abstract": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.ABSTRACT, reader.getText());
                        break;
                    }
                    case "genre": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        entry.setType(EntryTypeFactory.parse(this.mapGenre(reader.getText())));
                        break;
                    }
                    case "language": {
                        this.parseLanguage(reader, fields);
                        break;
                    }
                    case "location": {
                        this.parseLocationAndUrl(reader, fields);
                        break;
                    }
                    case "identifier": {
                        String type = reader.getAttributeValue(null, "type");
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.parseIdentifier(fields, new Identifier(type, reader.getText()), entry);
                        break;
                    }
                    case "note": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        notes.add(reader.getText());
                        break;
                    }
                    case "recordInfo": {
                        this.parseRecordInfo(reader, fields);
                        break;
                    }
                    case "titleInfo": {
                        this.parseTitle(reader, fields);
                        break;
                    }
                    case "subject": {
                        this.parseSubject(reader, fields, keywords);
                        break;
                    }
                    case "originInfo": {
                        this.parseOriginInfo(reader, fields);
                        break;
                    }
                    case "name": {
                        this.parseName(reader, fields, authors);
                        break;
                    }
                    case "relatedItem": {
                        this.parseRelatedItem(reader, fields);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"mods".equals(reader.getName().getLocalPart())) continue;
        }
        this.putIfListIsNotEmpty(fields, notes, StandardField.NOTE, ", ");
        this.putIfListIsNotEmpty(fields, keywords, StandardField.KEYWORDS, this.keywordSeparator);
        this.putIfListIsNotEmpty(fields, authors, StandardField.AUTHOR, " and ");
    }

    private void parseRelatedItem(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                switch (reader.getName().getLocalPart()) {
                    case "title": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.JOURNAL, reader.getText());
                        break;
                    }
                    case "detail": {
                        this.handleDetail(reader, fields);
                        break;
                    }
                    case "extent": {
                        this.handleExtent(reader, fields);
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"relatedItem".equals(reader.getName().getLocalPart())) continue;
            break;
        }
    }

    private void handleExtent(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        String total = "";
        String startPage = "";
        String endPage = "";
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName = reader.getName().getLocalPart();
                reader.next();
                switch (elementName) {
                    case "total": {
                        if (!this.isCharacterXMLEvent(reader)) break;
                        total = reader.getText();
                        break;
                    }
                    case "start": {
                        if (!this.isCharacterXMLEvent(reader)) break;
                        startPage = reader.getText();
                        break;
                    }
                    case "end": {
                        if (!this.isCharacterXMLEvent(reader)) break;
                        endPage = reader.getText();
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"extent".equals(reader.getName().getLocalPart())) continue;
        }
        if (!total.isBlank()) {
            this.putIfValueNotNull(fields, StandardField.PAGES, total);
        } else if (!startPage.isBlank()) {
            this.putIfValueNotNull(fields, StandardField.PAGES, startPage);
            if (!endPage.isBlank()) {
                fields.put(StandardField.PAGES, startPage + "-" + endPage);
            }
        }
    }

    private void handleDetail(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        String type = reader.getAttributeValue(null, "type");
        Set<String> detailElementSet = Set.of("number", "caption", "title");
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader) && detailElementSet.contains(reader.getName().getLocalPart())) {
                reader.next();
                if (this.isCharacterXMLEvent(reader)) {
                    this.putIfValueNotNull(fields, FieldFactory.parseField(type), reader.getText());
                }
            }
            if (!this.isEndXMLEvent(reader) || !"detail".equals(reader.getName().getLocalPart())) continue;
            break;
        }
    }

    private void parseName(XMLStreamReader reader, Map<Field, String> fields, List<String> authors) throws XMLStreamException {
        ArrayList<Name> names = new ArrayList<Name>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                if ("affiliation".equals(reader.getName().getLocalPart())) {
                    reader.next();
                    if (this.isCharacterXMLEvent(reader)) {
                        this.putIfValueNotNull(fields, new UnknownField("affiliation"), reader.getText());
                    }
                } else if ("namePart".equals(reader.getName().getLocalPart())) {
                    String type = reader.getAttributeValue(null, "type");
                    reader.next();
                    if (this.isCharacterXMLEvent(reader)) {
                        names.add(new Name(reader.getText(), type));
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"name".equals(reader.getName().getLocalPart())) continue;
        }
        this.handleAuthorsInNamePart(names, authors);
    }

    private void parseOriginInfo(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        ArrayList<String> places = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                String elementName;
                switch (elementName = reader.getName().getLocalPart()) {
                    case "issuance": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("issuance"), reader.getText());
                        break;
                    }
                    case "placeTerm": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.appendIfValueNotNullOrBlank(places, reader.getText());
                        break;
                    }
                    case "publisher": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.PUBLISHER, reader.getText());
                        break;
                    }
                    case "edition": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, StandardField.EDITION, reader.getText());
                        break;
                    }
                    case "dateIssued": 
                    case "dateCreated": 
                    case "dateCaptured": 
                    case "dateModified": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putDate(fields, elementName, reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"originInfo".equals(reader.getName().getLocalPart())) continue;
        }
        this.putIfListIsNotEmpty(fields, places, StandardField.ADDRESS, ", ");
    }

    private void parseSubject(XMLStreamReader reader, Map<Field, String> fields, List<String> keywords) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                switch (reader.getName().getLocalPart()) {
                    case "topic": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        keywords.add(reader.getText().trim());
                        break;
                    }
                    case "city": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("city"), reader.getText());
                        break;
                    }
                    case "country": {
                        reader.next();
                        if (!this.isCharacterXMLEvent(reader)) break;
                        this.putIfValueNotNull(fields, new UnknownField("country"), reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"subject".equals(reader.getName().getLocalPart())) continue;
            break;
        }
    }

    private void parseRecordInfo(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        RecordInfo recordInfoDefinition = new RecordInfo();
        List<String> recordContents = recordInfoDefinition.recordContents();
        List<String> languages = recordInfoDefinition.languages();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                if (RecordInfo.elementNameSet.contains(reader.getName().getLocalPart())) {
                    reader.next();
                    if (this.isCharacterXMLEvent(reader)) {
                        recordContents.addFirst(reader.getText());
                    }
                } else if ("languageTerm".equals(reader.getName().getLocalPart())) {
                    reader.next();
                    if (this.isCharacterXMLEvent(reader)) {
                        languages.add(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"recordInfo".equals(reader.getName().getLocalPart())) continue;
        }
        for (String recordContent : recordContents) {
            this.putIfValueNotNull(fields, new UnknownField("source"), recordContent);
        }
        this.putIfListIsNotEmpty(fields, languages, StandardField.LANGUAGE, ", ");
    }

    private void parseLanguage(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader) && "languageTerm".equals(reader.getName().getLocalPart())) {
                reader.next();
                if (this.isCharacterXMLEvent(reader)) {
                    this.putIfValueNotNull(fields, StandardField.LANGUAGE, reader.getText());
                }
            }
            if (!this.isEndXMLEvent(reader) || !"language".equals(reader.getName().getLocalPart())) continue;
            break;
        }
    }

    private void parseTitle(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader) && "title".equals(reader.getName().getLocalPart())) {
                reader.next();
                if (this.isCharacterXMLEvent(reader)) {
                    this.putIfValueNotNull(fields, StandardField.TITLE, reader.getText());
                }
            }
            if (!this.isEndXMLEvent(reader) || !"titleInfo".equals(reader.getName().getLocalPart())) continue;
            break;
        }
    }

    private void parseLocationAndUrl(XMLStreamReader reader, Map<Field, String> fields) throws XMLStreamException {
        ArrayList<String> locations = new ArrayList<String>();
        ArrayList<String> urls = new ArrayList<String>();
        while (reader.hasNext()) {
            reader.next();
            if (this.isStartXMLEvent(reader)) {
                if ("physicalLocation".equals(reader.getName().getLocalPart())) {
                    reader.next();
                    if (this.isCharacterXMLEvent(reader)) {
                        locations.add(reader.getText());
                    }
                } else if ("url".equals(reader.getName().getLocalPart())) {
                    reader.next();
                    if (this.isCharacterXMLEvent(reader)) {
                        urls.add(reader.getText());
                    }
                }
            }
            if (!this.isEndXMLEvent(reader) || !"location".equals(reader.getName().getLocalPart())) continue;
        }
        this.putIfListIsNotEmpty(fields, locations, StandardField.LOCATION, ", ");
        this.putIfListIsNotEmpty(fields, urls, StandardField.URL, ", ");
    }

    private String mapGenre(String genre) {
        return switch (genre.toLowerCase(Locale.ROOT)) {
            case "conference publication" -> "proceedings";
            case "database" -> "dataset";
            case "yearbook", "handbook" -> "book";
            case "law report or digest", "technical report", "reporting" -> "report";
            default -> genre;
        };
    }

    private void parseIdentifier(Map<Field, String> fields, Identifier identifier, BibEntry entry) {
        String type = identifier.type();
        if ("citekey".equals(type) && entry.getCitationKey().isEmpty()) {
            entry.setCitationKey(identifier.value());
        } else if (!"local".equals(type) && !"citekey".equals(type)) {
            this.putIfValueNotNull(fields, FieldFactory.parseField(identifier.type()), identifier.value());
        }
    }

    private void putDate(Map<Field, String> fields, String elementName, String date) {
        if (date != null) {
            Optional<Date> optionalParsedDate = Date.parse(date);
            switch (elementName) {
                case "dateIssued": {
                    optionalParsedDate.ifPresent(parsedDate -> fields.put(StandardField.DATE, parsedDate.getNormalized()));
                    optionalParsedDate.flatMap(Date::getYear).ifPresent(year -> fields.put(StandardField.YEAR, year.toString()));
                    optionalParsedDate.flatMap(Date::getMonth).ifPresent(month -> fields.put(StandardField.MONTH, month.getJabRefFormat()));
                    break;
                }
                case "dateCreated": {
                    fields.computeIfAbsent(StandardField.YEAR, k -> date.substring(0, 4));
                    fields.put(new UnknownField("created"), date);
                    break;
                }
                case "dateCaptured": {
                    optionalParsedDate.ifPresent(parsedDate -> fields.put(StandardField.CREATIONDATE, parsedDate.getNormalized()));
                    break;
                }
                case "dateModified": {
                    optionalParsedDate.ifPresent(parsedDate -> fields.put(StandardField.MODIFICATIONDATE, parsedDate.getNormalized()));
                }
            }
        }
    }

    private void putIfListIsNotEmpty(Map<Field, String> fields, List<String> list, Field key, String separator) {
        if (!list.isEmpty()) {
            fields.put(key, list.stream().collect(Collectors.joining(separator)));
        }
    }

    private void handleAuthorsInNamePart(List<Name> names, List<String> authors) {
        ArrayList<String> foreName = new ArrayList<String>();
        String familyName = "";
        Object author = "";
        for (Name name : names) {
            String type = name.type();
            if (type == null && name.value() != null) {
                String namePartValue = name.value();
                namePartValue = namePartValue.replaceAll(",$", "");
                authors.add(namePartValue);
                continue;
            }
            if ("family".equals(type) && name.value() != null) {
                if (!foreName.isEmpty() && !familyName.isEmpty()) {
                    author = familyName + ", " + Joiner.on((String)" ").join(foreName);
                    authors.add((String)author);
                    foreName.clear();
                } else if (foreName.isEmpty() && !familyName.isEmpty()) {
                    authors.add(familyName);
                }
                familyName = name.value();
                continue;
            }
            if (!"given".equals(type) || name.value() == null) continue;
            foreName.add(name.value());
        }
        if (!foreName.isEmpty() && !familyName.isEmpty()) {
            author = familyName + ", " + Joiner.on((String)" ").join(foreName);
            authors.add(((String)author).trim());
            foreName.clear();
        } else if (foreName.isEmpty() && !familyName.isEmpty()) {
            authors.add(familyName.trim());
        }
    }

    private void putIfValueNotNull(Map<Field, String> fields, Field field, String value) {
        if (value != null) {
            fields.put(field, value);
        }
    }

    private void appendIfValueNotNullOrBlank(List<String> list, String value) {
        if (value != null && !value.isBlank()) {
            list.add(value);
        }
    }

    private boolean isCharacterXMLEvent(XMLStreamReader reader) {
        return reader.getEventType() == 4;
    }

    private boolean isStartXMLEvent(XMLStreamReader reader) {
        return reader.getEventType() == 1;
    }

    private boolean isEndXMLEvent(XMLStreamReader reader) {
        return reader.getEventType() == 2;
    }

    @Override
    public String getName() {
        return "MODS";
    }

    @Override
    public StandardFileType getFileType() {
        return StandardFileType.XML;
    }

    @Override
    public String getDescription() {
        return "Importer for the MODS format";
    }

    @Override
    public List<BibEntry> parseEntries(InputStream inputStream) throws ParseException {
        try {
            return this.importDatabase(new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))).getDatabase().getEntries();
        }
        catch (IOException e) {
            LOGGER.error(e.getLocalizedMessage(), (Throwable)e);
            return Collections.emptyList();
        }
    }
}

