/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import com.google.common.collect.ImmutableList;
import gov.nasa.giss.data.nc.NcCartesianVarType;
import gov.nasa.giss.data.nc.NcCoordinateVarType;
import gov.nasa.giss.data.nc.NcDataGrouping;
import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcDiscreteVarType;
import gov.nasa.giss.data.nc.NcGeometryType;
import gov.nasa.giss.data.nc.NcLLAltGridding;
import gov.nasa.giss.data.nc.NcLLAltGriddingVarType;
import gov.nasa.giss.data.nc.NcUnitUtils;
import gov.nasa.giss.data.nc.NcUtils;
import gov.nasa.giss.data.nc.NcVarType;
import gov.nasa.giss.data.nc.array.NcArrayFeatureTypeTrajectory;
import gov.nasa.giss.data.nc.array.NcArrayLonLatAuxiliary2D;
import gov.nasa.giss.data.nc.array.NcArrayLonLatAuxiliary3D;
import gov.nasa.giss.data.nc.array.NcArrayLonLatCubedSphereCAMSE;
import gov.nasa.giss.data.nc.array.NcArrayLonLatCubedSphereGMAO;
import gov.nasa.giss.data.nc.array.NcArrayLonLatPolyhedron;
import gov.nasa.giss.data.nc.array.NcArrayLonLatReduced2D;
import gov.nasa.giss.data.nc.array.NcArrayLonLatReducedCF;
import gov.nasa.giss.data.nc.array.NcArrayLonLatReducedISCCP;
import gov.nasa.giss.data.nc.array.NcArrayLonLatUgridEdges;
import gov.nasa.giss.data.nc.array.NcArrayLonLatUgridFaces;
import java.lang.invoke.MethodHandles;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.ma2.DataType;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.Variable;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.CoordinateAxis1D;
import ucar.nc2.dataset.CoordinateSystem;
import ucar.nc2.dataset.VariableDS;

public final class NcVarTypeDetector {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

    public static NcVarType getVarType(NcDataset ncd, NcDataGrouping ncg, VariableDS varDS) {
        NcVarType vtype;
        LOGGER.trace("==========");
        LOGGER.trace("var {}", (Object)varDS.getFullName());
        DataType dt = varDS.getDataType();
        if (!dt.isNumeric()) {
            return NcVarType.OTHER;
        }
        if (varDS.isScalar()) {
            return NcVarType.SCALAR;
        }
        NcGeometryType geometry = ncd.getGeometry();
        LOGGER.trace("- geometry {}", (Object)geometry);
        if (geometry != NcGeometryType.GRIDDED) {
            vtype = NcVarTypeDetector.getDiscreteVarTypeRequiresFT(ncd, varDS);
            LOGGER.trace("- trajectory type? {}", (Object)vtype);
            if (vtype != null) {
                return vtype;
            }
        }
        vtype = NcVarTypeDetector.getGriddedVarType(ncd, ncg, varDS);
        LOGGER.trace("- gridded type? {}", (Object)vtype);
        if (vtype == NcVarType.ONE_D) {
            LOGGER.trace("- 1D var, not geo-reffed; is it untagged discrete featureType?");
            NcVarType dvtype = NcVarTypeDetector.getDiscreteVarTypeNoFT(ncd, varDS);
            LOGGER.trace("- got discrete type {}", (Object)dvtype);
            if (dvtype != null) {
                vtype = dvtype;
            }
        } else if (varDS.getRank() == 2 && !(vtype instanceof NcLLAltGriddingVarType)) {
            LOGGER.trace("- 2D var; is it untagged discrete featureType?");
            NcVarType dvtype = NcVarTypeDetector.getDiscreteVarTypeNoFT(ncd, varDS);
            LOGGER.trace("- got discrete type {}", (Object)dvtype);
            if (dvtype != null) {
                vtype = dvtype;
            }
        }
        if (vtype == null && NcUtils.isGriddable1D(varDS)) {
            return NcVarType.ONE_D;
        }
        LOGGER.trace("- returning type {}", (Object)vtype);
        return vtype;
    }

    private static NcVarType getDiscreteVarTypeRequiresFT(NcDataset ncd, VariableDS varDS) {
        block4: {
            NcGeometryType geometry = ncd.getGeometry();
            LOGGER.trace("ncd geometry {}", (Object)geometry);
            try {
                if (geometry == NcGeometryType.TRAJECTORY || geometry == NcGeometryType.TRAJECTORY_PROFILE) {
                    boolean isTrajectory = NcArrayFeatureTypeTrajectory.canGridVariable(ncd, varDS, true);
                    LOGGER.trace("NcArrayFeatureTypeTrajectory returns {}", (Object)isTrajectory);
                    if (isTrajectory) {
                        return new NcDiscreteVarType(NcGeometryType.TRAJECTORY);
                    }
                }
            }
            catch (Exception exc) {
                LOGGER.debug("Exception querying for discrete var type: {}", (Object)exc.toString());
                if (!LOGGER.isTraceEnabled()) break block4;
                exc.printStackTrace();
            }
        }
        return null;
    }

    private static NcVarType getDiscreteVarTypeNoFT(NcDataset ncd, VariableDS varDS) {
        block3: {
            try {
                boolean isTrajectory = NcArrayFeatureTypeTrajectory.canGridVariable(ncd, varDS, false);
                LOGGER.trace("NcArrayFeatureTypeTrajectory returns {}", (Object)isTrajectory);
                if (isTrajectory) {
                    return new NcDiscreteVarType(NcGeometryType.TRAJECTORY);
                }
            }
            catch (Exception exc) {
                LOGGER.debug("Exception querying for discrete var type: {}", (Object)exc.toString());
                if (!LOGGER.isTraceEnabled()) break block3;
                exc.printStackTrace();
            }
        }
        return null;
    }

    private static NcVarType getGriddedVarType(NcDataset ncd, NcDataGrouping ncg, VariableDS varDS) {
        NcVarType vtype;
        block8: {
            int[] vshape = varDS.getShape();
            long vsize = 1L;
            for (int i = 0; i < vshape.length; ++i) {
                if (vshape[i] == 0) {
                    return NcVarType.OTHER;
                }
                vsize *= (long)vshape[i];
            }
            if (vsize == 1L) {
                return NcVarType.SCALAR;
            }
            LOGGER.trace("is it projected lon lat?");
            vtype = NcVarTypeDetector.getProjectedLonLatType(ncd, varDS);
            if (vtype instanceof NcLLAltGriddingVarType) {
                return vtype;
            }
            LOGGER.trace("is it alt lon lat?");
            vtype = null;
            try {
                vtype = NcVarTypeDetector.getAlternativeLonLatType(ncd, ncg, varDS);
            }
            catch (Exception exc) {
                LOGGER.trace("Ouch! getAlternativeLonLatType threw {}", (Object)exc.toString());
                if (!LOGGER.isTraceEnabled()) break block8;
                exc.printStackTrace();
            }
        }
        LOGGER.trace("alternative type returns {}", (Object)vtype);
        if (vtype != null) {
            return vtype;
        }
        List csList = varDS.getCoordinateSystems();
        int csSize = csList.size();
        LOGGER.trace("...CS list size {}", (Object)csSize);
        if (csSize > 0) {
            CoordinateSystem cs = (CoordinateSystem)csList.get(0);
            LOGGER.trace("...CS 0 = {}", (Object)cs);
            NcVarType gvcs = NcVarTypeDetector.getGriddedVarTypeCS(ncd, varDS, cs);
            LOGGER.trace("...getGriddedVarTypeCS returns {}", (Object)gvcs);
            return gvcs;
        }
        return NcVarTypeDetector.getGriddedVarTypeNoCS(ncd, varDS);
    }

    private static NcVarType getGriddedVarTypeCS(NcDataset ncd, VariableDS varDS, CoordinateSystem cs) {
        int ii;
        boolean hasTime;
        boolean hasVert;
        boolean hasLat;
        boolean hasLon;
        CoordinateAxis zAxis;
        CoordinateAxis tAxis;
        CoordinateAxis latAxis;
        CoordinateAxis lonAxis;
        block36: {
            block38: {
                int[] ashape;
                block40: {
                    block39: {
                        block37: {
                            block31: {
                                block33: {
                                    block35: {
                                        block34: {
                                            block32: {
                                                if (LOGGER.isTraceEnabled()) {
                                                    ImmutableList<CoordinateAxis> axes = cs.getCoordinateAxes();
                                                    for (CoordinateAxis axis : axes) {
                                                        LOGGER.trace("axis {} is {}", (Object)axis.getShortName(), (Object)axis.getAxisType());
                                                    }
                                                }
                                                lonAxis = cs.getLonAxis();
                                                latAxis = cs.getLatAxis();
                                                tAxis = cs.getTaxis();
                                                zAxis = cs.getPressureAxis();
                                                if (zAxis == null) {
                                                    zAxis = cs.getHeightAxis();
                                                }
                                                if (zAxis == null) {
                                                    zAxis = cs.getZaxis();
                                                }
                                                hasLon = false;
                                                hasLat = false;
                                                hasVert = false;
                                                hasTime = false;
                                                if (lonAxis instanceof CoordinateAxis1D && tAxis instanceof CoordinateAxis1D && NcVarTypeDetector.doAxesShareDimension((CoordinateAxis1D)lonAxis, (CoordinateAxis1D)tAxis)) {
                                                    lonAxis = null;
                                                }
                                                if (latAxis instanceof CoordinateAxis1D && tAxis instanceof CoordinateAxis1D && NcVarTypeDetector.doAxesShareDimension((CoordinateAxis1D)latAxis, (CoordinateAxis1D)tAxis)) {
                                                    latAxis = null;
                                                }
                                                if (lonAxis == null) break block31;
                                                LOGGER.trace("...found lon axis {}", (Object)lonAxis.getFullName());
                                                ashape = lonAxis.getShape();
                                                if (ashape != null) break block32;
                                                lonAxis = null;
                                                break block33;
                                            }
                                            if (ashape.length >= 1 && ashape.length <= 3) break block34;
                                            lonAxis = null;
                                            break block33;
                                        }
                                        if (ashape.length != 1) break block35;
                                        if (ashape[0] >= 2) break block33;
                                        lonAxis = null;
                                        break block33;
                                    }
                                    for (ii = ashape.length - 2; ii < ashape.length; ++ii) {
                                        if (ashape[ii] >= 2) continue;
                                        lonAxis = null;
                                        break;
                                    }
                                }
                                hasLon = lonAxis != null;
                                LOGGER.trace("...hasLon", (Object)hasLon);
                            }
                            if (latAxis == null) break block36;
                            LOGGER.trace("...found lat axis {}", (Object)latAxis.getFullName());
                            ashape = latAxis.getShape();
                            if (ashape != null) break block37;
                            latAxis = null;
                            break block38;
                        }
                        if (ashape.length >= 1 && ashape.length <= 3) break block39;
                        latAxis = null;
                        break block38;
                    }
                    if (ashape.length != 1) break block40;
                    if (ashape[0] >= 2) break block38;
                    latAxis = null;
                    break block38;
                }
                for (ii = ashape.length - 2; ii < ashape.length; ++ii) {
                    if (ashape[ii] >= 2) continue;
                    latAxis = null;
                    break;
                }
            }
            if (latAxis instanceof CoordinateAxis1D && !NcVarTypeDetector.isMonotonicAxis(latAxis)) {
                latAxis = null;
            }
            hasLat = latAxis != null;
            LOGGER.trace("...hasLat", (Object)hasLat);
        }
        LOGGER.trace("cs.isLatLon {}, lon {}, lat {}", cs.isLatLon(), hasLon, hasLat);
        if (cs.isLatLon() && hasLon && hasLat) {
            int lonDI = -1;
            int latDI = -1;
            if (lonAxis.getRank() == 1 && latAxis.getRank() == 1) {
                Dimension latDD;
                Dimension lonDD = lonAxis.getDimension(0);
                if (lonDD.equals(latDD = latAxis.getDimension(0))) {
                    LOGGER.trace("lon axis dimension equals lat axis dimension!");
                }
                lonDI = NcUtils.findDimensionIndex(varDS, lonDD);
                latDI = NcUtils.findDimensionIndex(varDS, latDD);
            }
            LOGGER.trace("lonDI {}, latDI {}", (Object)lonDI, (Object)latDI);
            if (lonDI >= 0 && lonDI == latDI) {
                LOGGER.trace("Apparently unstructured or aux1D - setting hasLon, hasLat both false");
                hasLon = false;
                hasLat = false;
            } else if (lonDI >= 0 && latDI >= 0) {
                if (NcVarTypeDetector.isLonLatCartesian(varDS)) {
                    LOGGER.trace("Looks like Cartesian lon-lat.");
                    hasLon = true;
                    hasLat = true;
                } else if (NcVarTypeDetector.isLonLatAuxiliary1D(varDS)) {
                    LOGGER.trace("Looks like auxiliary 1D lon-lat.");
                    hasLon = false;
                    hasLat = false;
                } else {
                    LOGGER.trace("Not Cartesian or auxiliary 1D LL. We don't know what it is.");
                    hasLon = true;
                    hasLat = true;
                }
            } else {
                LOGGER.trace("Not clear what LL this is. Either lon or lat dim or both not found.");
                hasLon = false;
                hasLat = false;
            }
        } else if (cs.isGeoXY() || cs.getXaxis() == null || cs.getYaxis() != null) {
            // empty if block
        }
        if (!hasLon && !hasLat && NcVarTypeDetector.hasMessyLonLatDimensions(ncd, varDS)) {
            LOGGER.trace("...looks like messy lon-lat");
            hasLon = true;
            hasLat = true;
        }
        if (hasVert = cs.hasVerticalAxis()) {
            Dimension dd;
            LOGGER.trace("...Vert axis reported; zAxis {}", (Object)(zAxis != null ? 1 : 0));
            if (zAxis != null && zAxis.getShape().length == 1 && (ii = NcUtils.findDimensionIndex(varDS, dd = zAxis.getDimension(0))) < 0) {
                zAxis = null;
            }
            if (!NcVarTypeDetector.isMonotonicAxis(zAxis)) {
                zAxis = null;
            }
            hasVert = zAxis != null && zAxis.getShape().length == 1 && zAxis.getShape()[0] >= 2;
        }
        hasTime = cs.hasTimeAxis();
        if (tAxis != null) {
            hasTime = hasTime ? tAxis != null && tAxis.getShape().length >= 1 && tAxis.getShape()[0] >= 2 : NcVarTypeDetector.hasTimeDimension(ncd, varDS);
        }
        if (hasVert && hasTime && NcVarTypeDetector.doAxesShareDimension((CoordinateAxis1D)zAxis, (CoordinateAxis1D)tAxis)) {
            hasTime = false;
        }
        LOGGER.trace("... lon {}, lat {}, time {}, vert {}", hasLon, hasLat, hasTime, hasVert);
        if (hasLon && hasLat || hasLon && hasTime || hasLon && hasVert || hasLat && hasTime || hasLat && hasVert || hasTime && hasVert) {
            return new NcCartesianVarType(hasLon, hasLat, hasVert, hasTime);
        }
        if (NcUtils.isGriddable2D(varDS)) {
            return new NcCartesianVarType();
        }
        if (NcUtils.isGriddable1D(varDS)) {
            return NcVarType.ONE_D;
        }
        return NcVarType.OTHER;
    }

    private static NcVarType getGriddedVarTypeNoCS(NcDataset ncd, VariableDS varDS) {
        boolean hasLon = NcVarTypeDetector.hasLongitudeDim(ncd, varDS);
        boolean hasLat = NcVarTypeDetector.hasLatitudeDim(ncd, varDS);
        boolean hasTime = NcVarTypeDetector.hasTimeDimension(ncd, varDS);
        if (hasLon && hasLat || hasLon && hasTime || hasLat && hasTime) {
            return new NcCartesianVarType(hasLon, hasLat, false, hasTime);
        }
        if (NcUtils.isGriddable2D(varDS)) {
            return new NcCartesianVarType();
        }
        if (NcUtils.isGriddable1D(varDS)) {
            return NcVarType.ONE_D;
        }
        return NcVarType.OTHER;
    }

    private static boolean doAxesShareDimension(CoordinateAxis1D ax1, CoordinateAxis1D ax2) {
        Dimension dim1 = ax1.getDimension(0);
        Dimension dim2 = ax2.getDimension(0);
        if (dim1 == null || dim2 == null) {
            return false;
        }
        return dim1.equals(dim2);
    }

    private static boolean hasMessyLonLatDimensions(NcDataset ncd, VariableDS varDS) {
        boolean hasLon = NcVarTypeDetector.hasLongitudeDim(ncd, varDS);
        boolean hasLat = NcVarTypeDetector.hasLatitudeDim(ncd, varDS);
        LOGGER.trace("{}, {}", (Object)hasLon, (Object)hasLat);
        return hasLon && hasLat;
    }

    private static boolean hasLongitudeDim(NcDataset ncd, VariableDS varDS) {
        int rank = varDS.getRank();
        for (int i = 0; i < rank; ++i) {
            Dimension d = varDS.getDimension(i);
            VariableDS cv = ncd.getCoordinateVariable(d);
            if (cv == null || !NcUtils.isLongitudeVar1D(cv)) continue;
            return true;
        }
        return false;
    }

    private static boolean hasLatitudeDim(NcDataset ncd, VariableDS varDS) {
        int rank = varDS.getRank();
        for (int i = 0; i < rank; ++i) {
            Dimension d = varDS.getDimension(i);
            VariableDS cv = ncd.getCoordinateVariable(d);
            if (cv == null || !NcUtils.isLatitudeVar1D(cv)) continue;
            return true;
        }
        return false;
    }

    private static boolean hasTimeDimension(NcDataset ncd, VariableDS varDS) {
        int rank = varDS.getRank();
        for (int i = 0; i < rank; ++i) {
            Dimension d = varDS.getDimension(i);
            VariableDS cv = ncd.getCoordinateVariable(d);
            if (cv == null || !NcUtils.isRelativeTimeVar(ncd, cv) && !NcUtils.isJulianTimeVar(cv)) continue;
            return true;
        }
        return false;
    }

    private static boolean isLonLatCartesian(VariableDS varDS) {
        Dimension d2;
        List l = varDS.getCoordinateSystems();
        if (l == null || l.isEmpty()) {
            return false;
        }
        CoordinateSystem cs = (CoordinateSystem)l.get(0);
        if (cs == null) {
            return false;
        }
        CoordinateAxis lonax = cs.getLonAxis();
        CoordinateAxis latax = cs.getLatAxis();
        if (lonax == null || latax == null) {
            return false;
        }
        if (!(lonax instanceof CoordinateAxis1D)) {
            return false;
        }
        if (!(latax instanceof CoordinateAxis1D)) {
            return false;
        }
        int[] lonshape = lonax.getShape();
        int[] latshape = latax.getShape();
        if (lonshape.length != 1 || lonshape[0] < 2) {
            return false;
        }
        if (latshape.length != 1 || latshape[0] < 2) {
            return false;
        }
        Dimension d1 = lonax.getDimension(0);
        return !d1.equals(d2 = latax.getDimension(0));
    }

    private static boolean isLonLatAuxiliary1D(VariableDS varDS) {
        List l = varDS.getCoordinateSystems();
        if (l == null || l.isEmpty()) {
            return false;
        }
        CoordinateSystem cs = (CoordinateSystem)l.get(0);
        if (cs == null) {
            return false;
        }
        CoordinateAxis lonax = cs.getLonAxis();
        CoordinateAxis latax = cs.getLatAxis();
        if (lonax == null || latax == null) {
            return false;
        }
        if (!(lonax instanceof CoordinateAxis1D)) {
            return false;
        }
        if (!(latax instanceof CoordinateAxis1D)) {
            return false;
        }
        int[] lonshape = lonax.getShape();
        int[] latshape = latax.getShape();
        if (lonshape.length != 1 || lonshape[0] < 2) {
            return false;
        }
        if (latshape.length != 1 || latshape[0] < 2) {
            return false;
        }
        Dimension d1 = lonax.getDimension(0);
        Dimension d2 = latax.getDimension(0);
        return d1.equals(d2);
    }

    private static NcVarType getAlternativeLonLatType(NcDataset ncd, NcDataGrouping ncg, VariableDS varDS) {
        NcVarType varType = null;
        if (NcVarTypeDetector.hasUgridAttributes(ncd, varDS)) {
            LOGGER.trace("-- Might be UGRID");
            if (NcArrayLonLatUgridFaces.canGridVariable(ncd, varDS)) {
                varType = new NcLLAltGriddingVarType(NcLLAltGridding.UGRID_FACES);
            } else if (NcArrayLonLatUgridEdges.canGridVariable(ncd, varDS)) {
                varType = new NcLLAltGriddingVarType(NcLLAltGridding.UGRID_EDGES);
            } else if (NcUtils.isGriddable2D(varDS)) {
                LOGGER.trace("-- But not UGRID that we know how to work with");
                varType = new NcCartesianVarType();
            }
        } else if (NcArrayLonLatPolyhedron.canGridVariable(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.POLYHEDRAL_SPHERE);
        } else if (NcArrayLonLatCubedSphereGMAO.canGridVariable(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.CUBED_SPHERE_SIMPLE);
        } else if (NcArrayLonLatCubedSphereCAMSE.canGridVariable(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.CUBED_SPHERE_CAM_SE);
        } else if (NcArrayLonLatAuxiliary2D.canGridVariable(ncd, ncg, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.AUXILIARY_2D);
        } else if (NcArrayLonLatAuxiliary3D.canGridVariable(ncd, ncg, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.AUXILIARY_3D);
        } else if (NcVarTypeDetector.hasRotLonLatDimensions(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.ROTATED_POLE);
        } else if (NcArrayLonLatReducedCF.canGridVariable(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.REDUCED_1);
        } else if (NcArrayLonLatReduced2D.canGridVariable(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.REDUCED_2);
        } else if (NcArrayLonLatReducedISCCP.canGridVariable(ncd, varDS)) {
            varType = new NcLLAltGriddingVarType(NcLLAltGridding.REDUCED_ISCCP);
        }
        LOGGER.trace("-- {}", (Object)varType);
        return varType;
    }

    static boolean hasUgridAttributes(NcDataset ncd, VariableDS varDS) {
        String conventionStr = ncd.getGlobalAttributeString("Conventions");
        LOGGER.trace("-- conventions, '{}'", (Object)conventionStr);
        if (conventionStr == null) {
            return false;
        }
        if (!conventionStr.contains("UGRID-1.0")) {
            return false;
        }
        String meshStr = NcUtils.getAttributeStringIgnoreCase(varDS, "mesh");
        LOGGER.trace("-- meshVar name, {}", (Object)meshStr);
        if (meshStr == null) {
            return false;
        }
        String locStr = NcUtils.getAttributeStringIgnoreCase(varDS, "location");
        LOGGER.trace("-- location attribute, '{}'", (Object)locStr);
        if (locStr == null) {
            return false;
        }
        return "face".equals(locStr) || "edge".equals(locStr) || "node".equals(locStr);
    }

    private static NcVarType getProjectedLonLatType(NcDataset ncd, VariableDS varDS) {
        int rank = varDS.getRank();
        if (rank < 2) {
            LOGGER.trace("Does not have rank 2+");
            return null;
        }
        VariableDS gmvarDS = NcUtils.findGridMappingVariable(ncd, varDS);
        if (gmvarDS == null) {
            return null;
        }
        String gmName = NcUtils.getGridMappingName(gmvarDS);
        if (gmName == null) {
            LOGGER.debug("Apparent grid mapping var had no grid mapping name");
            return null;
        }
        if (gmName.equals("latitude_longitude")) {
            return new NcCartesianVarType();
        }
        if (gmName.equalsIgnoreCase("albers_equal_area_conic") || gmName.equalsIgnoreCase("albers_conical_equal_area")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.ALBERS_CONIC);
        }
        if (gmName.equalsIgnoreCase("azimuthal_equal_area") || gmName.equalsIgnoreCase("lambert_azimuthal_equal_area")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.AZIM_EQUAL_AREA);
        }
        if (gmName.equalsIgnoreCase("azimuthal_equidistant")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.AZIM_EQUIDISTANT);
        }
        if (gmName.equalsIgnoreCase("cylindrical_equal_area") || gmName.equalsIgnoreCase("lambert_cylindrical_equal_area")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.CYL_EQUAL_AREA);
        }
        if (gmName.equalsIgnoreCase("geostationary")) {
            String mvname = gmvarDS.getShortName();
            if ("goes_imager_projection".equalsIgnoreCase(mvname)) {
                LOGGER.trace("Geostationary GOES Imager projection");
                return new NcLLAltGriddingVarType(NcLLAltGridding.GOES_IMAGER);
            }
            return new NcLLAltGriddingVarType(NcLLAltGridding.GEOSTATIONARY);
        }
        if (gmName.equalsIgnoreCase("lambert_conformal_conic")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.LAMBERT_CONFORMAL);
        }
        if (gmName.equalsIgnoreCase("mercator")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.MERCATOR);
        }
        if (gmName.equalsIgnoreCase("MSGnavigation")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.MSG_NAV);
        }
        if (gmName.equalsIgnoreCase("rotated_pole") || gmName.equalsIgnoreCase("rotated_latitude_longitude") || gmName.equalsIgnoreCase("rotated_latlon_grib")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.ROTATED_POLE);
        }
        if (gmName.equalsIgnoreCase("sinusoidal")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.SINUSOIDAL);
        }
        if (gmName.equalsIgnoreCase("polar_stereographic") || gmName.equalsIgnoreCase("stereographic")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.STEREOGRAPHIC);
        }
        if (gmName.equalsIgnoreCase("transverse_mercator")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.TRANSVERSE_MERCATOR);
        }
        if (gmName.equalsIgnoreCase("universal_transverse_mercator")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.UTM);
        }
        if (gmName.toLowerCase().contains("utm")) {
            return new NcLLAltGriddingVarType(NcLLAltGridding.UTM);
        }
        LOGGER.debug("...grid mapping name {} is not recognized", (Object)gmName);
        return null;
    }

    private static boolean hasRotLonLatDimensions(NcDataset ncd, Variable varDS) {
        int rank = varDS.getRank();
        String lonName = null;
        String latName = null;
        for (int i = 0; i < rank; ++i) {
            Dimension d = varDS.getDimension(i);
            String dname = d.getShortName();
            if (dname == null) continue;
            String lcname = dname.toLowerCase();
            if (lcname.contains("lon")) {
                lonName = dname;
                continue;
            }
            if (!lcname.contains("lat")) continue;
            latName = dname;
        }
        if (lonName != null && latName != null) {
            VariableDS lonVar = ncd.getVariableDS(lonName);
            VariableDS latVar = ncd.getVariableDS(latName);
            if (lonVar == null || latVar == null) {
                return false;
            }
            Attribute lonA = lonVar.findAttribute("long_name");
            Attribute latA = latVar.findAttribute("long_name");
            if (lonA == null || latA == null) {
                return false;
            }
            if (lonA.getStringValue().contains("rotated") && latA.getStringValue().contains("rotated")) {
                return true;
            }
        }
        return false;
    }

    private static boolean isMonotonicAxis(VariableDS njvarDS) {
        if (njvarDS == null) {
            LOGGER.trace("Axis var is null!");
            return false;
        }
        int[] shape = njvarDS.getShape();
        if (shape.length != 1) {
            LOGGER.trace("Shape is not 1D -> not monotonic.");
            return false;
        }
        if (!njvarDS.getDataType().isNumeric()) {
            LOGGER.trace("Var is not numeric.");
            return false;
        }
        int numVals = shape[0];
        if (numVals < 2) {
            LOGGER.trace("Var does not have at least two values.");
            return true;
        }
        Array vArray = null;
        try {
            vArray = njvarDS.read();
        }
        catch (Exception exc) {
            LOGGER.trace("Could not verify monotonic due to variable read exception.");
            return false;
        }
        double[] values = new double[numVals];
        for (int j = 0; j < numVals; ++j) {
            double value = vArray.getDouble(vArray.getIndex().set(j));
            if (Double.isNaN(value)) {
                LOGGER.trace("Could not verify monotonic due to presence of NaN,");
                return false;
            }
            values[j] = value;
        }
        boolean increasing = values[1] > values[0];
        for (int j = 2; j < numVals; ++j) {
            if (!(increasing ? values[j] < values[j - 1] : values[j] > values[j - 1])) continue;
            return false;
        }
        return true;
    }

    public static NcCoordinateVarType getCoordinateVarType(VariableDS varDS) {
        String lcname = varDS.getShortName().toLowerCase();
        String unitsStr = NcUtils.getUnitsStr(varDS);
        if ("longitude".equals(lcname) || "lon".equals(lcname) || NcUnitUtils.isDegreesEast(unitsStr)) {
            return NcCoordinateVarType.LONGITUDE;
        }
        if ("latitude".equals(lcname) || "lat".equals(lcname) || NcUnitUtils.isDegreesNorth(unitsStr)) {
            return NcCoordinateVarType.LATITUDE;
        }
        if (NcUtils.isPressure(varDS) || NcUtils.isHeight(varDS) || NcUtils.isDepth(varDS)) {
            return NcCoordinateVarType.VERTICAL;
        }
        return NcCoordinateVarType.NONE;
    }

    private NcVarTypeDetector() {
    }
}

