#ifndef __CS_CDO_MAIN_H__
#define __CS_CDO_MAIN_H__

/*============================================================================
 * Functions for solving equations with CDO discretizations
 *============================================================================*/

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
 *  Local headers
 *----------------------------------------------------------------------------*/

#include "base/cs_base.h"
#include "cdo/cs_domain.h"
#include "mesh/cs_mesh.h"
#include "mesh/cs_mesh_quantities.h"

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*============================================================================
 * Macro definitions
 *============================================================================*/

/*============================================================================
 * Type definitions
 *============================================================================*/

/*============================================================================
 * Public function prototypes
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*!
 * \brief  Solve only steady-state equations
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_solve_steady_state_domain(void);

/*----------------------------------------------------------------------------*/
/*!
 * \brief  Solve all the equations of a computational domain for one time step
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_solve_unsteady_state_domain(void);

/*----------------------------------------------------------------------------*/
/*!
 * \brief Initialize the computational domain when CDO/HHO schemes are
 *        activated and cs_user_model() has been called
 *        At this stage of the settings, mesh quantities and adjacencies are
 *        not defined. Only the major moddeling options are set. The related
 *        equations and main properties have been added.
 *
 * \param[in, out] domain    pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_initialize_setup(cs_domain_t   *domain);

/*----------------------------------------------------------------------------*/
/*!
 * \brief Build additional connectivities and quantities when CDO/HHO schemes
 *        are activated.
 *        Finalize the setup and from the settings, define the structures
 *        related to equations and modules
 *
 * \param[in, out] domain   pointer to a cs_domain_t structure
 * \param[in, out] m        pointer to a cs_mesh_t struct.
 * \param[in]      mq       pointer to a cs_quantities_t struct.
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_initialize_structures(cs_domain_t           *domain,
                             cs_mesh_t             *m,
                             cs_mesh_quantities_t  *mq);

/*----------------------------------------------------------------------------*/
/*!
 * \brief Free all structures allocated during the resolution of CDO/HHO
 *        schemes
 *
 * \param[in, out] domain   pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_finalize(cs_domain_t    *domain);

/*----------------------------------------------------------------------------*/
/*!
 * \brief Check if CDO has been initialized.
 *
 * \param[in, out] setup       indicator if setup has been initialized,
 *                             or null if not queried
 * \param[in, out] structures  indicator if structures have been initialized,
 *                             or null if not queried
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_is_initialized(bool  *setup,
                      bool  *structures);

/*----------------------------------------------------------------------------*/
/*!
 * \brief Main program for running a simulation with the CDO kernel
 *
 * \param[in, out] domain   pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_cdo_main(cs_domain_t   *domain);

/*----------------------------------------------------------------------------*/

END_C_DECLS

#endif /* __CS_CDO_MAIN_H__ */
