/*============================================================================
 * User definition of boundary conditions.
 *============================================================================*/

/* code_saturne version 9.1 */

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

#include "cs_headers.h"

/*----------------------------------------------------------------------------
 * Standard library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*----------------------------------------------------------------------------*/
/*
 * \file cs_user_boundary_conditions-cooling_towers.cpp
 *
 * \brief User functions for boundary condition definitions.
 */
/*----------------------------------------------------------------------------*/

/*=============================================================================
 * User function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*
 * User definition of boundary conditions.
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 * \param[in, out]  bc_type  boundary face types
 */
/*----------------------------------------------------------------------------*/

void
cs_user_boundary_conditions([[maybe_unused]] cs_domain_t  *domain,
                            [[maybe_unused]] int           bc_type[])
{
  const cs_real_3_t *b_face_cog
    = (const cs_real_3_t *)domain->mesh_quantities->b_face_cog;

  const cs_real_t ro0 = cs_glob_fluid_properties->ro0;
  const cs_real_t *gravity = cs_glob_physical_constants->gravity;
  const cs_real_t *xyzp0 = (const cs_real_t *)cs_glob_fluid_properties->xyzp0;

  const cs_zone_t  *zn = nullptr;

  /* Assign a free outlet for faces of group "OUTLET" */

  /*![example_1]*/
  /* Direct pointers (without multiple indirections) */

  int       *p_icodcl  = CS_F_(p)->bc_coeffs->icodcl;
  cs_real_t *p_rcodcl1 = CS_F_(p)->bc_coeffs->rcodcl1;

  /* Set BC's over zone. */

  zn = cs_boundary_zone_by_name("OUTLET");

  for (cs_lnum_t ilelt = 0; ilelt < zn->n_elts; ilelt++) {

    /* outlet: zero flux for velocity and temperature, prescribed pressure
     *         note that pressure will be set to P0 on the free outlet face
     *         (CS_OUTLET) closest to xyz0 */
    const cs_lnum_t face_id = zn->elt_ids[ilelt];

    bc_type[face_id] = CS_OUTLET;

    /* Prescribe a pressure profile for all faces
     * Warning: the pressure is prescribed as the solved pressure,
     * that is p - rho0 g (z-z0)
     * This is done by setting "-" CS_BC_DIRICHLET
     * */

    p_icodcl[face_id] = -CS_BC_DIRICHLET;
    p_rcodcl1[face_id] = 0.;
  }
  /*![example_1]*/
}

/*----------------------------------------------------------------------------*/

END_C_DECLS
