/*============================================================================
 * General-purpose user-defined functions called before time stepping, at
 * the end of each time step, and after time-stepping.
 *
 * These can be used for operations which do not fit naturally in any other
 * dedicated user function.
 *============================================================================*/

/* code_saturne version 9.1 */

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

#include "cs_headers.h"

/*----------------------------------------------------------------------------
 * Standard library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*============================================================================
 * User function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*
 * User operations called at the end of each time step.
 *
 * This function has a very general purpose, although it is recommended to
 * handle mainly postprocessing or data-extraction type operations.
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_user_extra_operations([[maybe_unused]] cs_domain_t  *domain)
{
  cs_balance_by_zone("all[]", "temperature");

  cs_balance_by_zone("box[-0.5, 1.3, 0.0, 1.0, 1.9, 1.0]",
                     "scalar1");

  cs_real_t normal[3] = {0., 0., 1.,};

  cs_surface_balance("selection_criterion", "scalar1", normal);

  /* More advanced usage for pressure drop */

  {
    const char criteria[] = "zone_group";

    cs_lnum_t   n_selected_cells = 0;
    cs_lnum_t  *selected_cells = nullptr;

    cs_real_t balance[CS_BALANCE_N_TERMS];

    CS_MALLOC(selected_cells, domain->mesh->n_cells, cs_lnum_t);

    cs_selector_get_cell_list(criteria,
                              &n_selected_cells,
                              selected_cells);

    cs_balance_by_zone_compute("scalar1",
                               n_selected_cells,
                               selected_cells,
                               balance);

    CS_FREE(selected_cells);

    cs_balance_term_t  mass_in_idx = CS_BALANCE_MASS_IN;
    cs_balance_term_t  mass_out_idx = CS_BALANCE_MASS_OUT;

    bft_printf("inlet mass flow  (scalar 1): %g\n"
               "outlet mass flow (scalar 1): %g\n",
               balance[mass_in_idx],
               balance[mass_out_idx]);
  }

  cs_pressure_drop_by_zone("zone_group");

  /* More advanced usage for pressure drop */

  {
    const char criteria[] = "zone_group";

    cs_lnum_t   n_selected_cells = 0;
    cs_lnum_t  *selected_cells = nullptr;

    cs_real_t balance[CS_BALANCE_P_N_TERMS];

    CS_MALLOC(selected_cells, domain->mesh->n_cells, cs_lnum_t);

    cs_selector_get_cell_list(criteria,
                              &n_selected_cells,
                              selected_cells);

    cs_pressure_drop_by_zone_compute(n_selected_cells,
                                     selected_cells,
                                     balance);

    CS_FREE(selected_cells);

    cs_balance_p_term_t  rhou_in_idx = CS_BALANCE_P_RHOU_IN;
    cs_balance_p_term_t  rhou_out_idx = CS_BALANCE_P_RHOU_OUT;

    bft_printf("inlet mass flow  (rho.u): %g\n"
               "outlet mass flow (rho.u): %g\n",
               balance[rhou_in_idx],
               balance[rhou_out_idx]);
  }
}

/*----------------------------------------------------------------------------*/

END_C_DECLS
