/*============================================================================
 * General-purpose user-defined functions called before time stepping, at
 * the end of each time step, and after time-stepping.
 *
 * These can be used for operations which do not fit naturally in any other
 * dedicated user function.
 *============================================================================*/

/* code_saturne version 9.1 */

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

#include "cs_headers.h"

/*----------------------------------------------------------------------------
 * Standard library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*============================================================================
 * User function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*
 * User operations called at the end of each time step.
 *
 * This function has a very general purpose, although it is recommended to
 * handle mainly postprocessing or data-extraction type operations.
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_user_extra_operations([[maybe_unused]] cs_domain_t  *domain)
{
  {
    cs_field_t *b_forces = cs_field_by_name_try("boundary_stress");

    if (b_forces != nullptr) {
      cs_real_3_t total_b_forces = {0., 0., 0.};
      const cs_real_t *b_face_surf = domain->mesh_quantities->b_face_surf;
      const cs_real_3_t *bpro_forces = (cs_real_3_t *)b_forces->val;

      /* get zone from its name, here "selected_wall" */
      const cs_zone_t *zn = cs_boundary_zone_by_name("selected_wall");

      for (cs_lnum_t e_id = 0; e_id < zn->n_elts; e_id++) {
        cs_lnum_t face_id = zn->elt_ids[e_id];
        for (cs_lnum_t i = 0; i < 3; i++)
          total_b_forces[i] += bpro_forces[face_id][i] * b_face_surf[face_id];
      }

      /* parallel sum */
      cs_parall_sum(3, CS_REAL_TYPE, total_b_forces);
    }
  }

  {
    const cs_real_t *b_face_surf = domain->mesh_quantities->b_face_surf;
    const cs_nreal_3_t *b_face_u_normal
      = domain->mesh_quantities->b_face_u_normal;

    cs_real_3_t total_b_p_forces = {0., 0., 0.};

    /* get zone from its name, here "selected_wall" */
    const cs_zone_t *zn = cs_boundary_zone_by_name("selected_wall");

    /* compute static pressure on selected boundary faces */
    cs_real_t *p_b_val;
    CS_MALLOC(p_b_val, zn->n_elts, cs_real_t);
    cs_post_b_pressure(zn->n_elts, zn->elt_ids, p_b_val);

    for (cs_lnum_t e_id = 0; e_id < zn->n_elts; e_id++) {
      cs_lnum_t face_id = zn->elt_ids[e_id];
      for (cs_lnum_t i = 0; i < 3; i++)
        total_b_p_forces[i] +=   p_b_val[e_id]
                               * b_face_u_normal[face_id][i]
                               * b_face_surf[face_id];
    }

    CS_FREE(p_b_val);

    /* parallel sum */
    cs_parall_sum(3, CS_REAL_TYPE, total_b_p_forces);
  }
}

/*----------------------------------------------------------------------------*/

END_C_DECLS
