/*============================================================================
 * User definition of physical properties.
 *============================================================================*/

/* code_saturne version 9.1 */

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

#include "cs_headers.h"

/*----------------------------------------------------------------------------
 * Standard library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>
#include <string.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*============================================================================
 * User function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*
 * Function called at each time step to define physical properties.
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_user_physical_properties
(
  [[maybe_unused]] cs_domain_t   *domain
)
{

  /* Check fields exists */
  if (CS_F_(lambda) == nullptr)
    bft_error(__FILE__, __LINE__, 0,_("error lambda not variable\n"));
  if (CS_F_(rho) == nullptr)
    bft_error(__FILE__, __LINE__, 0,_("error rho not variable\n"));
  if (CS_F_(cp) == nullptr)
    bft_error(__FILE__, __LINE__, 0,_("error cp not variable\n"));

  cs_real_t *cpro_lambda = CS_F_(lambda)->val;
  cs_real_t *cpro_rho = CS_F_(rho)->val;
  cs_real_t *cpro_cp = CS_F_(cp)->val;

  /* Impose thermal conductivity, density and specific heat for solid zones */
  {
    /* Volume zone "CM" must be defined in the GUI or in cs_user_zones.c */
    const cs_zone_t *z = cs_volume_zone_by_name("CM");

    for (cs_lnum_t i = 0; i < z->n_elts; i++) {
      cs_lnum_t cell_id = z->elt_ids[i];
      cpro_lambda[cell_id] = 3.3;
      cpro_rho[cell_id] = 7800.;
      cpro_cp[cell_id] = 444.;
    }
  }
}

/*----------------------------------------------------------------------------*/
/*
 * User definition of enthalpy to temperature conversion.
 *
 * This allows overwriting the solver defaults if necessary.
 *
 * This function may be called on a per-zone basis, so as to allow different
 * conversion relations in zones representing solids or different fluids.
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 * \param[in]       z        zone (volume or boundary) applying to current call
 * \param[in]       z_local  if true, h and t arrays are defined in a compact
 *                           (contiguous) manner for this zone only;
 *                           if false, h and t are defined on the zone's parent
 *                           location (usually all cells or boundary faces)
 * \param[in]       h        enthalpy values
 * \param[in, out]  t        temperature values
 */
/*----------------------------------------------------------------------------*/

void
cs_user_physical_properties_h_to_t
(
  [[maybe_unused]] cs_domain_t      *domain,
  [[maybe_unused]] const cs_zone_t  *z,
  [[maybe_unused]] bool              z_local,
  [[maybe_unused]] const cs_real_t   h[],
  [[maybe_unused]] cs_real_t         t[]
)
{
  /* Tabulated values */

  static const int n_tv = 5;
  static const cs_real_t ht[5] = {100000.0, 200000.0, 300000.0,
                                  400000.0, 00000.0};
  static const cs_real_t th[5] = {100.0, 200.0, 300.0, 400.0, 500.0};

  /* Conversion:
     Note that z->name or z->location_id can be used as a filter
     if "per-zone" properties are needed (such as with solid zones) */

  for (cs_lnum_t i_l = 0; i_l < z->n_elts; i_l++) {

    cs_lnum_t i = (z_local) ? i_l : z->elt_ids[i_l];

    cs_real_t temperature = 0;  /* Default initialization */

    /* If H is outside the tabulated value range, use range
       start or end value. */

    if (h[i] < ht[0])
      temperature = th[0];
    else if (h[i] > ht[n_tv - 1])
      temperature = th[n_tv - 1];

    /* Otherwise, use piecewise linear interpolation */

    else
      for (int j = 1; j < n_tv; j++) {
        if (h[j] < ht[j]) {
          temperature = th[j-1] +   (h[i]-ht[j-1])*(th[j]-th[j-1])
                                  / (ht[j]-ht[j-1]);
          break;
        }
      }

    t[i] = temperature;

  }
}

/*----------------------------------------------------------------------------*/
/*
 * User definition of temperature to enthalpy conversion.
 *
 * This allows overwriting the solver defaults if necessary.
 *
 * This function may be called on a per-zone basis, so as to allow different
 * conversion relations in zones representing solids or different fluids.
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 * \param[in]       z        zone (volume or boundary) applying to current call
 * \param[in]       z_local  if true, h and t arrays are defined in a compact
 *                           (contiguous) manner for this zone only;
 *                           if false, h and t are defined on the zone's parent
 *                           location (usually all cells or boundary faces)
 * \param[in]       h        temperature values
 * \param[in, out]  t        enthalpy values
 */
/*----------------------------------------------------------------------------*/

void
cs_user_physical_properties_t_to_h
(
  [[maybe_unused]] cs_domain_t      *domain,
  [[maybe_unused]] const cs_zone_t  *z,
  [[maybe_unused]] bool              z_local,
  [[maybe_unused]] const cs_real_t   t[],
  [[maybe_unused]] cs_real_t         h[]
)
{
  /* Tabulated values */

  static const int n_tv = 5;
  static const cs_real_t ht[5] = {100000.0, 200000.0, 300000.0,
                                  400000.0, 00000.0};
  static const cs_real_t th[5] = {100.0, 200.0, 300.0, 400.0, 500.0};

  /* Conversion:
     Note that z->name or z->location_id can be used as a filter
     if "per-zone" properties are needed (such as with solid zones) */

  for (cs_lnum_t i_l = 0; i_l < z->n_elts; i_l++) {

    cs_lnum_t i = (z_local) ? i_l : z->elt_ids[i_l];

    cs_real_t enthalpy = 0;  /* Default initialization */

    /* If H is outside the tabulated value range, use range
       start or end value. */

    if (t[i] < th[0])
      enthalpy = ht[0];
    else if (t[i] > th[n_tv - 1])
      enthalpy = ht[n_tv - 1];

    /* Otherwise, use piecewise linear interpolation */

    else
      for (int j = 1; j < n_tv; j++) {
        if (t[j] < th[j]) {
          enthalpy = ht[j-1] +   (t[i]-th[j-1])*(ht[j]-ht[j-1])
                               / (th[j]-th[j-1]);
          break;
        }
      }

    h[i] = enthalpy;

  }
}

/*----------------------------------------------------------------------------*/
/*
 * User modification of the turbulence viscosity.
 *
 * Turbulent viscosity \f$ \mu_T \f$ (kg/(m s)) can be modified.
 * You can access the field by its name.
 *
 * \param[in, out]   domain      pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_user_physical_properties_turb_viscosity
(
  [[maybe_unused]] cs_domain_t  *domain
)
{

  cs_lnum_t n_cells_ext = domain->mesh->n_cells_with_ghosts;

  /* Recompute Mij:Mij and Lij:Mij */
  cs_real_t  *mijlij;
  cs_real_t  *mijmij;
  cs_real_t *s_n, *sf_n;
  cs_real_3_t *f_vel;
  CS_MALLOC(mijlij, n_cells_ext, cs_real_t);
  CS_MALLOC(mijmij, n_cells_ext, cs_real_t);
  CS_MALLOC(s_n, n_cells_ext, cs_real_t);
  CS_MALLOC(sf_n, n_cells_ext, cs_real_t);
  CS_MALLOC(f_vel, n_cells_ext, cs_real_3_t);

  /* Compute:
   *   s_n (aka sqrt(2SijSij))
   *   sf_n (aka ||<rho*S>/<rho>||),
   *   filtered vel
   *   Mij:Mij
   *   Lij:Mij
   */
  cs_les_mu_t_smago_dyn_prepare(s_n, sf_n, f_vel, mijmij, mijlij);

  const cs_lnum_t n_cells = domain->mesh->n_cells;
  const cs_real_t tot_vol = domain->mesh_quantities->tot_vol;
  const cs_real_t *cell_vol = domain->mesh_quantities->cell_vol;

  /* Overwrite Smagorinsky constant */
  cs_real_t *cpro_smago
    = cs_field_by_name("smagorinsky_constant^2")->val;

  cs_real_t mijmijmoy = 0, mijlijmoy = 0;

  for (cs_lnum_t c_id = 0; c_id < n_cells; c_id++) {
    mijlijmoy += mijlij[c_id]*cell_vol[c_id];
    mijmijmoy += mijmij[c_id]*cell_vol[c_id];
  }

  /* Free memory */
  CS_FREE(s_n);
  CS_FREE(sf_n);
  CS_FREE(f_vel);
  CS_FREE(mijmij);
  CS_FREE(mijlij);

  cs_parall_sum(1, CS_REAL_TYPE, &mijlijmoy);
  cs_parall_sum(1, CS_REAL_TYPE, &mijmijmoy);

  mijmijmoy /= tot_vol;
  mijlijmoy /= tot_vol;

  for (cs_lnum_t c_id = 0; c_id < n_cells; c_id++)
    cpro_smago[c_id] = mijlijmoy/mijmijmoy;

  /* Calculation of (dynamic) viscosity
   * ================================== */

  cs_real_t *visct = CS_F_(mu_t)->val;
  const cs_real_t *crom  = CS_F_(rho)->val;
  for (cs_lnum_t c_id = 0; c_id < n_cells; c_id++) {
    const cs_real_t coef = cpro_smago[c_id];
    const cs_real_t delta = cs_turb_xlesfl * pow(cs_turb_ales*cell_vol[c_id],
                                                 cs_turb_bles);
    visct[c_id] = crom[c_id]*coef*cs_math_pow2(delta)*s_n[c_id];
  }
}

/*----------------------------------------------------------------------------*/

END_C_DECLS
