/*============================================================================
 * User definition of porous media.
 *============================================================================*/

/* code_saturne version 9.1 */

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

#include "cs_headers.h"

/*----------------------------------------------------------------------------
 * Standard library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*============================================================================
 * User function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*
 * Compute the porosity (volume factor \f$ \epsilon \f$
 * when the porosity model is activated.
 *
 * This function is called at the beginning of the simulation only.
 *
 * \param[in, out]  domain  pointer to a cs_domain_t structure
 */
/*----------------------------------------------------------------------------*/

void
cs_user_porosity([[maybe_unused]] cs_domain_t   *domain)
{
  /*!< [init_poro_mq] */
  cs_mesh_t *m = domain->mesh;
  cs_mesh_quantities_t *mq = domain->mesh_quantities;
  cs_mesh_quantities_t *mq_g = cs_glob_mesh_quantities_g;

  const cs_lnum_2_t *i_face_cells = m->i_face_cells;

  const cs_real_3_t *restrict i_face_cog = mq_g->i_face_cog;
  const cs_real_3_t *restrict b_face_cog = mq_g->b_face_cog;
  const cs_real_3_t *restrict cell_cen = mq_g->cell_cen;
  const cs_real_3_t *restrict i_face_normal
    = (const cs_real_3_t *)mq_g->i_face_normal;
  cs_real_3_t *restrict i_f_face_normal
    = (cs_real_3_t *)mq->i_face_normal;
  const cs_real_3_t *restrict b_face_normal
    = (const cs_real_3_t *)mq_g->b_face_normal;
  cs_real_3_t *restrict b_f_face_normal
    = (cs_real_3_t *)mq->b_face_normal;

  const cs_real_t *i_f_face_surf = mq->i_face_surf;
  const cs_real_t *i_face_surf = mq_g->i_face_surf;
  /*!< [init_poro_mq] */

  /* Get the cell porosity field value */
  /*!< [init_poro_pro] */
  cs_real_t *cpro_porosi = cs_field_by_name("porosity")->val;
  /*!< [init_poro_pro] */

  /* First, set cell porosity value; the fluid cell volume will be
   * automatically deduced */

  /*!< [set_poro_cells_1] */
  for (cs_lnum_t cell_id = 0; cell_id < m->n_cells; cell_id++) {

    cs_real_t x = cell_cen[cell_id][0];

    if (x < 20.)
      cpro_porosi[cell_id] = 1.;
    else
      cpro_porosi[cell_id] = 0.5;
  }

  /* synchronize for use in fluid face factor calculation below */
  cs_halo_type_t halo_type = CS_HALO_STANDARD;
  cs_field_synchronize(cs_field_by_name("porosity"),
                       halo_type);

  /*!< [set_poro_cells_1] */

  /* Set interior face values (for cs_glob_porous_model == 3 only) */

  /*!< [set_poro_i_faces_1] */
  for (cs_lnum_t face_id = 0; face_id < m->n_i_faces; face_id++) {

    cs_real_t x = i_face_cog[face_id][0];

    cs_real_t face_porosity = 1.;
    if (x > 19.9999)
      face_porosity = 0.5;

    for (int i = 0; i < 3; i++)
      i_f_face_normal[face_id][i] = face_porosity * i_face_normal[face_id][i];

    mq->i_face_surf[face_id] = cs_math_3_norm(i_f_face_normal[face_id]);

  }
  /*!< [set_poro_i_faces_1] */

  /* Set boundary face values  (for cs_glob_porous_model == 3 only) */

  /*!< [set_poro_b_faces_1] */
  for (cs_lnum_t face_id = 0; face_id < m->n_b_faces; face_id++) {

    cs_real_t x = b_face_cog[face_id][0];

    cs_real_t face_porosity = 1.;
    if (x > 19.9999)
      face_porosity = 0.5;

    for (int i = 0; i < 3; i++)
      b_f_face_normal[face_id][i] = face_porosity * b_face_normal[face_id][i];

    mq->b_face_surf[face_id] = cs_math_3_norm(b_f_face_normal[face_id]);

  }
  /*!< [set_poro_b_faces_1] */

  /* Four set face factor */
  if (mq->i_f_face_factor != nullptr) {
    for (cs_lnum_t face_id = 0; face_id < m->n_i_faces; face_id++) {
      cs_lnum_t ii = i_face_cells[face_id][0];
      cs_lnum_t jj = i_face_cells[face_id][1];

      cs_real_t face_porosity
        = cs::max(i_f_face_surf[face_id] / i_face_surf[face_id],
                  cs_math_epzero);

      mq->i_f_face_factor[face_id][0] = cpro_porosi[ii] / face_porosity;
      mq->i_f_face_factor[face_id][1] = cpro_porosi[jj] / face_porosity;
    }
  }

  /* Example to automatically compute fluid face surfaces
   * if cs_glob_porous_model = 3
   * (fluid surface is min between cell porosity) */

  cs_porous_model_auto_face_porosity();
}

/*----------------------------------------------------------------------------*/

END_C_DECLS
