/*============================================================================
 * Additional user-defined source terms for variable equations.
 *============================================================================*/

/* code_saturne version 9.1 */

/*
  This file is part of code_saturne, a general-purpose CFD tool.

  Copyright (C) 1998-2025 EDF S.A.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*----------------------------------------------------------------------------*/

#include "cs_headers.h"

/*----------------------------------------------------------------------------
 * Standard library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>

/*----------------------------------------------------------------------------
 * PLE library headers
 *----------------------------------------------------------------------------*/

#include <ple_coupling.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

BEGIN_C_DECLS

/*============================================================================
 * User function definitions
 *============================================================================*/

/*----------------------------------------------------------------------------*/
/*
 * Additional user-defined source terms for variable equations
 * (momentum, scalars, turbulence...).
 *
 * \param[in, out]  domain   pointer to a cs_domain_t structure
 * \param[in]       f_id     field id of the variable
 * \param[out]      st_exp   explicit source term
 * \param[out]      st_imp   implicit part of the source term
 */
/*----------------------------------------------------------------------------*/

void
cs_user_source_terms
(
  [[maybe_unused]] cs_domain_t  *domain,
  [[maybe_unused]] int           f_id,
  [[maybe_unused]] cs_real_t    *st_exp,
  [[maybe_unused]] cs_real_t    *st_imp
)
{
  /* field structure */
  const cs_field_t  *f = cs_field_by_id(f_id);

  /* mesh quantities */
  const cs_lnum_t  n_cells = domain->mesh->n_cells;
  const cs_real_t  *cell_f_vol = domain->mesh_quantities->cell_vol;

  /* density */
  const cs_real_t  *cpro_rom = CS_F_(rho)->val;

  /* Example of arbitrary source term for component u:
   *
   *                       S = A * u + B
   *
   *        appearing in the equation under the form
   *
   *                  rho*du/dt = S (+ standard Navier-Stokes terms)
   *
   * In the following example:
   *   A = - rho * CKP
   *   B =   MMT
   *
   * with:
   *  CKP = 1.0   [1/s]      (return term on velocity)
   *  MMT = 100.0 [Pa/m]     (momentum production by volume and time unit)
   *
   * which yields:
   *  st_imp[i][0][0] = cell_f_vol[i] * A = - cell_f_vol[i] * rho * CKP
   *  st_exp[i][0]    = cell_f_vol[i] * B =   cell_f_vol[i]       * MMT
   */

  if (f == CS_F_(vel)) { /* velocity */

    /* cast to 3D vectors for readability */
    cs_real_3_t    *_st_exp = (cs_real_3_t *)st_exp;
    cs_real_33_t   *_st_imp = (cs_real_33_t *)st_imp;

    /* Density */

    const cs_real_t ckp = 10.0;
    const cs_real_t mmt = 100.0;

    for (cs_lnum_t i = 0; i < n_cells; i++) {
      _st_imp[i][0][0] = - cell_f_vol[i] * cpro_rom[i] * ckp;
      _st_exp[i][0]    =   cell_f_vol[i] *               mmt;
    }

  }

  /* Example of boussinesq source term. */

  if (f == CS_F_(vel) && CS_F_(t) != nullptr) { /* velocity and temperature */

    /* expansion coefficient and reference density */

    const cs_real_t beta = 1.;
    const cs_real_t ro0  = cs_glob_fluid_properties->ro0;
    const cs_real_t t0  = cs_glob_fluid_properties->t0;

    /* get temperature */

    const cs_real_t *cvar_temperature = CS_F_(t)->val;

    /* source term (in z direction here) */

    cs_real_3_t  *_st_exp = (cs_real_3_t *)st_exp;

    for (cs_lnum_t i = 0; i < n_cells; i++) {
      _st_exp[i][2] = cell_f_vol[i] * ro0 * beta * (cvar_temperature[i]-t0);
    }

  }
}

/*----------------------------------------------------------------------------*/

END_C_DECLS
