// -*- mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// vi: set et ts=4 sw=4 sts=4:
//
// SPDX-FileCopyrightText: Copyright © DuMux Project contributors, see AUTHORS.md in root folder
// SPDX-License-Identifier: GPL-3.0-or-later
//
/*!
 * \file
 * \ingroup NavierStokesModel
 *
 * \copydoc Dumux::NavierStokesVolumeVariables
 */
#ifndef DUMUX_NAVIERSTOKES_MASS_1PNC_VOLUME_VARIABLES_HH
#define DUMUX_NAVIERSTOKES_MASS_1PNC_VOLUME_VARIABLES_HH

#include <dumux/freeflow/navierstokes/scalarvolumevariables.hh>
#include <dumux/freeflow/navierstokes/energy/volumevariables.hh>

namespace Dumux {

/*!
 * \ingroup NavierStokesModel
 * \brief Volume variables for the single-phase Navier-Stokes model.
 */
template <class Traits>
class NavierStokesMassOnePNCVolumeVariables
: public NavierStokesScalarConservationModelVolumeVariables<Traits>
, public NavierStokesEnergyVolumeVariables<Traits, NavierStokesMassOnePNCVolumeVariables<Traits>>
{
    using ParentType = NavierStokesScalarConservationModelVolumeVariables<Traits>;
    using EnergyVolumeVariables = NavierStokesEnergyVolumeVariables<Traits, NavierStokesMassOnePNCVolumeVariables<Traits>>;

    using Scalar = typename Traits::PrimaryVariables::value_type;

    static constexpr bool useMoles = Traits::ModelTraits::useMoles();
    using DiffusionCoefficients = typename Traits::DiffusionType::DiffusionCoefficientsContainer;


public:
    //! export the underlying fluid system
    using FluidSystem = typename Traits::FluidSystem;
    //! export the fluid state type
    using FluidState = typename Traits::FluidState;
    //! export the diffusion type
    using MolecularDiffusionType = typename Traits::DiffusionType;
    //! export the indices type
    using Indices = typename Traits::ModelTraits::Indices;

    /*!
     * \brief Update all quantities for a given control volume
     *
     * \param elemSol A vector containing all primary variables connected to the element
     * \param problem The object specifying the problem which ought to
     *                be simulated
     * \param element An element which contains part of the control volume
     * \param scv The sub-control volume
     */
    template<class ElementSolution, class Problem, class Element, class SubControlVolume>
    void update(const ElementSolution &elemSol,
                const Problem &problem,
                const Element &element,
                const SubControlVolume& scv)
    {
        ParentType::update(elemSol, problem, element, scv);

        completeFluidState(elemSol, problem, element, scv, fluidState_);

        typename FluidSystem::ParameterCache paramCache;
        paramCache.updateAll(fluidState_);

        auto getDiffusionCoefficient = [&](int phaseIdx, int compIIdx, int compJIdx)
        {
            return FluidSystem::binaryDiffusionCoefficient(this->fluidState_,
                                                            paramCache,
                                                            0,
                                                            compIIdx,
                                                            compJIdx);
        };

        diffCoefficient_.update(getDiffusionCoefficient);
        EnergyVolumeVariables::updateEffectiveThermalConductivity();
    }

    /*!
     * \brief Update the fluid state
     */
    template<class ElementSolution, class Problem, class Element, class SubControlVolume>
    void completeFluidState(const ElementSolution& elemSol,
                                   const Problem& problem,
                                   const Element& element,
                                   const SubControlVolume& scv,
                                   FluidState& fluidState) const
    {
        fluidState.setTemperature(0, EnergyVolumeVariables::getTemperature(elemSol, problem, element, scv));
        fluidState.setPressure(0, elemSol[0][Indices::pressureIdx]);

        // saturation in a single phase is always 1 and thus redundant
        // to set. But since we use the fluid state shared by the
        // immiscible multi-phase models, so we have to set it here...
        fluidState.setSaturation(0, 1.0);

        Scalar sumFracMinorComp = 0.0;

        for(int compIdx = 1; compIdx < ParentType::numFluidComponents(); ++compIdx)
        {
            // temporary add 1.0 to remove spurious differences in mole fractions
            // which are below the numerical accuracy
            Scalar moleOrMassFraction = elemSol[0][Indices::conti0EqIdx+compIdx] + 1.0;
            moleOrMassFraction = moleOrMassFraction - 1.0;
            if(useMoles)
                fluidState.setMoleFraction(0, compIdx, moleOrMassFraction);
            else
                fluidState.setMassFraction(0, compIdx, moleOrMassFraction);
            sumFracMinorComp += moleOrMassFraction;
        }
        if(useMoles)
            fluidState.setMoleFraction(0, 0, 1.0 - sumFracMinorComp);
        else
            fluidState.setMassFraction(0, 0, 1.0 - sumFracMinorComp);

        typename FluidSystem::ParameterCache paramCache;
        paramCache.updateAll(fluidState);

        Scalar value = FluidSystem::density(fluidState, paramCache, 0);
        fluidState.setDensity(0, value);

        value = FluidSystem::molarDensity(fluidState, paramCache, 0);
        fluidState.setMolarDensity(0, value);

        value = FluidSystem::viscosity(fluidState, paramCache, 0);
        fluidState.setViscosity(0, value);

        // compute and set the enthalpy
        const Scalar h = EnergyVolumeVariables::enthalpy(fluidState, paramCache);
        fluidState.setEnthalpy(0, h);
    }

    /*!
     * \brief Return the effective pressure \f$\mathrm{[Pa]}\f$ of a given phase within
     *        the control volume.
     */
    Scalar pressure(int phaseIdx = 0) const
    { return fluidState_.pressure(0); }

    /*!
     * \brief Returns the saturation.
     */
    Scalar saturation(int phaseIdx = 0) const
    { return 1.0; }

    /*!
     * \brief Return the mass density \f$\mathrm{[kg/m^3]}\f$ of a given phase within the
     *        control volume.
     */
    Scalar density(int phaseIdx = 0) const
    { return fluidState_.density(0); }

    /*!
     * \brief Return temperature \f$\mathrm{[K]}\f$ inside the sub-control volume.
     *
     */
    Scalar temperature() const
    { return fluidState_.temperature(); }

    /*!
     * \brief Return the effective dynamic viscosity \f$\mathrm{[Pa s]}\f$ of the fluid within the
     *        control volume.
     */
    Scalar effectiveViscosity() const
    { return viscosity(); }

    /*!
     * \brief Return the dynamic viscosity \f$\mathrm{[Pa s]}\f$ of the fluid within the
     *        control volume.
     */
    Scalar viscosity(int phaseIdx = 0) const
    { return fluidState_.viscosity(0); }

    /*!
     * \brief Returns the mass fraction of a component in the phase \f$\mathrm{[-]}\f$
     *
     * \param phaseIdx the index of the phase
     * \param compIdx the index of the component
     */
    Scalar massFraction(int phaseIdx, int compIdx) const
    {
        return fluidState_.massFraction(0, compIdx);
    }

    /*!
     * \brief Returns the mass fraction of a component in the phase \f$\mathrm{[-]}\f$
     *
     * \param compIdx the index of the component
     */
    Scalar massFraction(int compIdx) const
    {
        return fluidState_.massFraction(0, compIdx);
    }

    /*!
     * \brief Returns the mole fraction of a component in the phase \f$\mathrm{[-]}\f$
     *
     * \param phaseIdx the index of the phase
     * \param compIdx the index of the component
     */
    Scalar moleFraction(int phaseIdx, int compIdx) const
    {
        return fluidState_.moleFraction(0, compIdx);
    }

    /*!
     * \brief Returns the mole fraction of a component in the phase \f$\mathrm{[-]}\f$
     *
     * \param compIdx the index of the component
     */
    Scalar moleFraction(int compIdx) const
    {
        return fluidState_.moleFraction(0, compIdx);
    }

    /*!
     * \brief Returns the mass density of a given phase \f$\mathrm{[kg/m^3]}\f$
     */
    Scalar molarDensity(int phaseIdx = 0) const
    {
        return fluidState_.molarDensity(0);
    }

    /*!
     * \brief Returns the molar mass of a given component.
     *
     * \param compIdx the index of the component
     */
    Scalar molarMass(int compIdx) const
    {
        return FluidSystem::molarMass(compIdx);
    }

    /*!
     * \brief Returns the average molar mass \f$\mathrm{[kg/mol]}\f$ of the fluid phase.
     *
     * \param phaseIdx The phase index
     */
    Scalar averageMolarMass(const int phaseIdx = 0) const
    { return fluidState_.averageMolarMass(phaseIdx); }

    /*!
     * \brief Returns the binary diffusion coefficients for a phase in \f$[m^2/s]\f$.
     */
    Scalar diffusionCoefficient(int phaseIdx, int compIIdx, int compJIdx) const
    { return diffCoefficient_(0, compIIdx, compJIdx); }

    /*!
     * \brief Returns the effective diffusion coefficients for a phase in \f$[m^2/s]\f$.
     */
    Scalar effectiveDiffusionCoefficient(int phaseIdx, int compIIdx, int compJIdx) const
    { return diffusionCoefficient(0, compIIdx, compJIdx); }

    /*!
     * \brief Return the fluid state of the control volume.
     */
    const FluidState& fluidState() const
    { return fluidState_; }

protected:
    FluidState fluidState_;
    // Binary diffusion coefficient
    DiffusionCoefficients diffCoefficient_;
};

} // end namespace Dumux

#endif // DUMUX_NAVIERSTOKES_MOMENTUM_VOLUME_VARIABLES_HH
