// -*- mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// vi: set et ts=4 sw=4 sts=4:
//
// SPDX-FileCopyrightText: Copyright © DuMux Project contributors, see AUTHORS.md in root folder
// SPDX-License-Identifier: GPL-3.0-or-later
//
/*!
 * \file
 * \ingroup Components
 * \brief The default CO2 table provides the CO2 density and enthalpy values with in
 *        the range of 290 K to 340 K and 100 KPa to 100 MPa.
 *
 * \note If you need a CO2 table with specified pressure and temperature ranges,
 *       use the script bin/utils/make_co2_table.py to generate the new table and
 *       include the new table in the properties.hh.
 *
 */
#ifndef DUMUX_COMPONENTS_DEFAULT_CO2_CO2TABLE_HH
#define DUMUX_COMPONENTS_DEFAULT_CO2_CO2TABLE_HH

#include <dune/common/float_cmp.hh>

 /* Tables for CO2 fluid properties calculated according to Span and
 * Wagner (1996) and using a web service of the National Institute
 * of Standards and Techlology of the U.S. Department of Commerce:
 * https://webbook.nist.gov/
 *
 * THIS AN AUTO-GENERATED FILE! DO NOT EDIT IT!
 *
 ********************************************************************

    In case you are using this the data generated with this script
    please cite the following publications:

    P.J. Linstrom and W.G. Mallard, Eds.,
    NIST Chemistry WebBook, NIST Standard Reference Database Number 69,
    National Institute of Standards and Technology, Gaithersburg MD, 20899,
    https://doi.org/10.18434/T4D303, (retrieved March 22, 2023).

    Span, Roland, and Wolfgang Wagner.
    "A new equation of state for carbon dioxide covering
    the fluid region from the triple‐point temperature
    to 1100 K at pressures up to 800 MPa."
    Journal of physical and chemical reference data 25.6 (1996): 1509-1596.
    https://doi.org/10.1063/1.555991

 ********************************************************************
 *
 * Generated using:
 *
 * ./make_co2_table.py -t1 290.0 -t2 340.0 -nt 50 -p1 100000.0 -p2 100000000.0 -np 495
 */


namespace Dumux::GeneratedCO2Tables {

/*!
 * \ingroup Components
 * \brief A generic template for tabulated material laws that depend
 *        on two parameters.
 *
 * \note If you need a CO2 table with specified pressure and temperature ranges,
 *       use the script bin/utils/make_co2_table.py to generate the new table and
 *       include the new table in the properties.hh.
 */
template <class Traits>
class TabulatedProperties
{
    using Scalar = typename Traits::Scalar;

    static constexpr auto numTempSteps = Traits::numTempSteps;
    static constexpr auto numPressSteps = Traits::numPressSteps;

public:
    TabulatedProperties() = default;

    constexpr Scalar minTemp() const { return Traits::minTemp; }
    constexpr Scalar maxTemp() const { return Traits::maxTemp; }
    constexpr Scalar minPress() const { return Traits::minPress; }
    constexpr Scalar maxPress() const { return Traits::maxPress; }

    constexpr bool applies(Scalar temperature, Scalar pressure) const
    {
        return minTemp() <= temperature && temperature <= maxTemp() &&
               minPress() <= pressure && pressure <= maxPress();
    }

    constexpr Scalar at(Scalar temperature, Scalar pressure) const
    {
        if (!applies(temperature, pressure))
        {
            if (temperature<minTemp()) temperature = minTemp();
            else if (temperature>maxTemp()) temperature = maxTemp();

            if (pressure<minPress()) pressure = minPress();
            else if (pressure>maxPress()) pressure = maxPress();
        }

        const int i = findTempIdx_(temperature);
        const int j = findPressIdx_(pressure);

        const Scalar tempAtI = temperatureAt_(i);
        const Scalar tempAtI1 = temperatureAt_(i + 1);
        const Scalar pressAtI = pressureAt_(j);
        const Scalar pressAtI1 = pressureAt_(j + 1);

        const Scalar alpha = (temperature - tempAtI)/(tempAtI1 - tempAtI);
        const Scalar beta = (pressure - pressAtI)/(pressAtI1 - pressAtI);

        // bi-linear interpolation
        const Scalar lowresValue =
            (1-alpha)*(1-beta)*val(i, j) +
            (1-alpha)*(  beta)*val(i, j + 1) +
            (  alpha)*(1-beta)*val(i + 1, j) +
            (  alpha)*(  beta)*val(i + 1, j + 1);

        // return the weighted sum of the low- and high-resolution values
        return lowresValue;
    }

    constexpr Scalar val(int i, int j) const
    { return Traits::vals[i][j]; }

private:
    constexpr int findTempIdx_(Scalar temperature) const
    {
        if (Dune::FloatCmp::eq<Scalar>(temperature, maxTemp()))
            return numTempSteps - 2;

        const int result = static_cast<int>((temperature - minTemp())/(maxTemp() - minTemp())*(numTempSteps - 1));

        using std::clamp;
        return clamp(result, 0, numTempSteps - 2);
    }

    constexpr int findPressIdx_(Scalar pressure) const
    {
        if (Dune::FloatCmp::eq<Scalar>(pressure, maxPress()))
            return numPressSteps - 2;

        const int result = static_cast<int>((pressure - minPress())/(maxPress() - minPress())*(numPressSteps - 1));

        using std::clamp;
        return clamp(result, 0, numPressSteps - 2);
    }

    constexpr Scalar temperatureAt_(int i) const
    { return i*(maxTemp() - minTemp())/(numTempSteps - 1) + minTemp(); }

    constexpr Scalar pressureAt_(int j) const
    { return j*(maxPress() - minPress())/(numPressSteps - 1) + minPress(); }
};

struct TabulatedDensityTraits
{
    using Scalar = double;
    static constexpr std::string_view name = "density";
    static constexpr int numTempSteps = 50;
    static constexpr Scalar minTemp = 290.0;
    static constexpr Scalar maxTemp = 340.0;
    static constexpr int numPressSteps = 495;
    static constexpr Scalar minPress = 100000.0;
    static constexpr Scalar maxPress = 100000000.0;
    static constexpr Scalar vals[numTempSteps][numPressSteps] = {
        {1.835234672750e+00, 5.609580202420e+00, 9.472493322930e+00, 1.342953108800e+01, 1.748683681860e+01, 2.165122860470e+01, 2.593030563670e+01, 3.033257695440e+01, 3.486761866390e+01, 3.954626770210e+01, 4.438086308160e+01, 4.938554963520e+01, 5.457666523520e+01, 5.997324131010e+01, 6.559765994890e+01, 7.147653187200e+01, 7.764189322990e+01, 8.413287502240e+01, 9.099809498650e+01, 9.829919448460e+01, 1.061162697690e+02, 1.145566054680e+02, 1.237695505470e+02, 1.339738126600e+02, 1.455128257180e+02, 1.589845822980e+02, 8.057312854290e+02, 8.108402448210e+02, 8.155824372050e+02, 8.200167155520e+02, 8.241880065360e+02, 8.281315309910e+02, 8.318754989790e+02, 8.354429089570e+02, 8.388527925240e+02, 8.421211012730e+02, 8.452613544230e+02, 8.482851218090e+02, 8.512023906970e+02, 8.540218488390e+02, 8.567511059840e+02, 8.593968694200e+02, 8.619650846650e+02, 8.644610493680e+02, 8.668895063900e+02, 8.692547205140e+02, 8.715605421560e+02, 8.738104606480e+02, 8.760076491190e+02, 8.781550025050e+02, 8.802551699400e+02, 8.823105825040e+02, 8.843234771020e+02, 8.862959171270e+02, 8.882298103940e+02, 8.901269247990e+02, 8.919889020160e+02, 8.938172695420e+02, 8.956134513180e+02, 8.973787771270e+02, 8.991144909270e+02, 9.008217582820e+02, 9.025016729850e+02, 9.041552629890e+02, 9.057834957320e+02, 9.073872829220e+02, 9.089674848520e+02, 9.105249143060e+02, 9.120603400880e+02, 9.135744902290e+02, 9.150680548970e+02, 9.165416890590e+02, 9.179960148990e+02, 9.194316240320e+02, 9.208490795370e+02, 9.222489178100e+02, 9.236316502820e+02, 9.249977649820e+02, 9.263477279900e+02, 9.276819847740e+02, 9.290009614220e+02, 9.303050657860e+02, 9.315946885390e+02, 9.328702041560e+02, 9.341319718260e+02, 9.353803362970e+02, 9.366156286640e+02, 9.378381671070e+02, 9.390482575700e+02, 9.402461944030e+02, 9.414322609590e+02, 9.426067301530e+02, 9.437698649850e+02, 9.449219190270e+02, 9.460631368880e+02, 9.471937546410e+02, 9.483140002310e+02, 9.494240938540e+02, 9.505242483190e+02, 9.516146693830e+02, 9.526955560680e+02, 9.537671009680e+02, 9.548294905240e+02, 9.558829052970e+02, 9.569275202210e+02, 9.579635048400e+02, 9.589910235350e+02, 9.600102357400e+02, 9.610212961450e+02, 9.620243548860e+02, 9.630195577320e+02, 9.640070462560e+02, 9.649869580010e+02, 9.659594266340e+02, 9.669245821010e+02, 9.678825507630e+02, 9.688334555320e+02, 9.697774160040e+02, 9.707145485740e+02, 9.716449665600e+02, 9.725687803100e+02, 9.734860973120e+02, 9.743970222890e+02, 9.753016573040e+02, 9.762001018460e+02, 9.770924529250e+02, 9.779788051490e+02, 9.788592508120e+02, 9.797338799670e+02, 9.806027805030e+02, 9.814660382160e+02, 9.823237368750e+02, 9.831759582900e+02, 9.840227823730e+02, 9.848642871990e+02, 9.857005490640e+02, 9.865316425400e+02, 9.873576405280e+02, 9.881786143100e+02, 9.889946335980e+02, 9.898057665800e+02, 9.906120799690e+02, 9.914136390420e+02, 9.922105076860e+02, 9.930027484370e+02, 9.937904225190e+02, 9.945735898820e+02, 9.953523092380e+02, 9.961266380960e+02, 9.968966327970e+02, 9.976623485450e+02, 9.984238394380e+02, 9.991811584990e+02, 9.999343577070e+02, 1.000683488020e+03, 1.001428599410e+03, 1.002169740880e+03, 1.002906960510e+03, 1.003640305430e+03, 1.004369821920e+03, 1.005095555380e+03, 1.005817550350e+03, 1.006535850560e+03, 1.007250498940e+03, 1.007961537630e+03, 1.008669008010e+03, 1.009372950720e+03, 1.010073405670e+03, 1.010770412050e+03, 1.011464008380e+03, 1.012154232500e+03, 1.012841121580e+03, 1.013524712170e+03, 1.014205040160e+03, 1.014882140870e+03, 1.015556048970e+03, 1.016226798600e+03, 1.016894423300e+03, 1.017558956040e+03, 1.018220429270e+03, 1.018878874890e+03, 1.019534324300e+03, 1.020186808360e+03, 1.020836357470e+03, 1.021483001500e+03, 1.022126769870e+03, 1.022767691550e+03, 1.023405795010e+03, 1.024041108300e+03, 1.024673659040e+03, 1.025303474420e+03, 1.025930581190e+03, 1.026555005710e+03, 1.027176773940e+03, 1.027795911430e+03, 1.028412443380e+03, 1.029026394580e+03, 1.029637789460e+03, 1.030246652090e+03, 1.030853006190e+03, 1.031456875140e+03, 1.032058281950e+03, 1.032657249340e+03, 1.033253799660e+03, 1.033847954970e+03, 1.034439737010e+03, 1.035029167200e+03, 1.035616266660e+03, 1.036201056240e+03, 1.036783556460e+03, 1.037363787590e+03, 1.037941769590e+03, 1.038517522160e+03, 1.039091064750e+03, 1.039662416510e+03, 1.040231596360e+03, 1.040798622940e+03, 1.041363514660e+03, 1.041926289690e+03, 1.042486965920e+03, 1.043045561060e+03, 1.043602092540e+03, 1.044156577590e+03, 1.044709033200e+03, 1.045259476160e+03, 1.045807923010e+03, 1.046354390120e+03, 1.046898893620e+03, 1.047441449440e+03, 1.047982073320e+03, 1.048520780800e+03, 1.049057587210e+03, 1.049592507710e+03, 1.050125557250e+03, 1.050656750620e+03, 1.051186102410e+03, 1.051713627030e+03, 1.052239338730e+03, 1.052763251580e+03, 1.053285379470e+03, 1.053805736130e+03, 1.054324335140e+03, 1.054841189890e+03, 1.055356313630e+03, 1.055869719460e+03, 1.056381420300e+03, 1.056891428950e+03, 1.057399758030e+03, 1.057906420030e+03, 1.058411427300e+03, 1.058914792040e+03, 1.059416526320e+03, 1.059916642050e+03, 1.060415151020e+03, 1.060912064890e+03, 1.061407395180e+03, 1.061901153290e+03, 1.062393350480e+03, 1.062883997890e+03, 1.063373106550e+03, 1.063860687340e+03, 1.064346751040e+03, 1.064831308320e+03, 1.065314369700e+03, 1.065795945630e+03, 1.066276046400e+03, 1.066754682230e+03, 1.067231863210e+03, 1.067707599330e+03, 1.068181900450e+03, 1.068654776360e+03, 1.069126236720e+03, 1.069596291110e+03, 1.070064948990e+03, 1.070532219730e+03, 1.070998112600e+03, 1.071462636780e+03, 1.071925801350e+03, 1.072387615290e+03, 1.072848087490e+03, 1.073307226770e+03, 1.073765041830e+03, 1.074221541310e+03, 1.074676733730e+03, 1.075130627550e+03, 1.075583231140e+03, 1.076034552790e+03, 1.076484600680e+03, 1.076933382950e+03, 1.077380907630e+03, 1.077827182690e+03, 1.078272215990e+03, 1.078716015360e+03, 1.079158588520e+03, 1.079599943110e+03, 1.080040086730e+03, 1.080479026880e+03, 1.080916770980e+03, 1.081353326410e+03, 1.081788700460e+03, 1.082222900350e+03, 1.082655933230e+03, 1.083087806200e+03, 1.083518526270e+03, 1.083948100390e+03, 1.084376535470e+03, 1.084803838310e+03, 1.085230015690e+03, 1.085655074300e+03, 1.086079020780e+03, 1.086501861710e+03, 1.086923603600e+03, 1.087344252900e+03, 1.087763816030e+03, 1.088182299310e+03, 1.088599709030e+03, 1.089016051410e+03, 1.089431332630e+03, 1.089845558790e+03, 1.090258735960e+03, 1.090670870150e+03, 1.091081967300e+03, 1.091492033310e+03, 1.091901074040e+03, 1.092309095280e+03, 1.092716102780e+03, 1.093122102220e+03, 1.093527099260e+03, 1.093931099500e+03, 1.094334108480e+03, 1.094736131700e+03, 1.095137174610e+03, 1.095537242630e+03, 1.095936341110e+03, 1.096334475370e+03, 1.096731650680e+03, 1.097127872260e+03, 1.097523145290e+03, 1.097917474910e+03, 1.098310866210e+03, 1.098703324250e+03, 1.099094854040e+03, 1.099485460540e+03, 1.099875148670e+03, 1.100263923340e+03, 1.100651789370e+03, 1.101038751580e+03, 1.101424814730e+03, 1.101809983560e+03, 1.102194262740e+03, 1.102577656940e+03, 1.102960170760e+03, 1.103341808770e+03, 1.103722575530e+03, 1.104102475530e+03, 1.104481513230e+03, 1.104859693080e+03, 1.105237019460e+03, 1.105613496740e+03, 1.105989129240e+03, 1.106363921250e+03, 1.106737877040e+03, 1.107111000830e+03, 1.107483296810e+03, 1.107854769130e+03, 1.108225421940e+03, 1.108595259310e+03, 1.108964285330e+03, 1.109332504010e+03, 1.109699919350e+03, 1.110066535340e+03, 1.110432355900e+03, 1.110797384960e+03, 1.111161626380e+03, 1.111525084020e+03, 1.111887761700e+03, 1.112249663210e+03, 1.112610792320e+03, 1.112971152760e+03, 1.113330748250e+03, 1.113689582450e+03, 1.114047659030e+03, 1.114404981610e+03, 1.114761553790e+03, 1.115117379140e+03, 1.115472461200e+03, 1.115826803500e+03, 1.116180409530e+03, 1.116533282760e+03, 1.116885426630e+03, 1.117236844560e+03, 1.117587539950e+03, 1.117937516160e+03, 1.118286776530e+03, 1.118635324390e+03, 1.118983163040e+03, 1.119330295740e+03, 1.119676725750e+03, 1.120022456300e+03, 1.120367490580e+03, 1.120711831770e+03, 1.121055483050e+03, 1.121398447530e+03, 1.121740728350e+03, 1.122082328580e+03, 1.122423251300e+03, 1.122763499560e+03, 1.123103076380e+03, 1.123441984780e+03, 1.123780227730e+03, 1.124117808210e+03, 1.124454729160e+03, 1.124790993510e+03, 1.125126604150e+03, 1.125461563970e+03, 1.125795875850e+03, 1.126129542610e+03, 1.126462567100e+03, 1.126794952120e+03, 1.127126700450e+03, 1.127457814860e+03, 1.127788298110e+03, 1.128118152930e+03, 1.128447382030e+03, 1.128775988100e+03, 1.129103973830e+03, 1.129431341870e+03, 1.129758094870e+03, 1.130084235460e+03, 1.130409766230e+03, 1.130734689780e+03, 1.131059008680e+03, 1.131382725490e+03, 1.131705842760e+03, 1.132028362990e+03, 1.132350288700e+03, 1.132671622380e+03, 1.132992366510e+03, 1.133312523530e+03, 1.133632095910e+03, 1.133951086050e+03, 1.134269496370e+03, 1.134587329270e+03, 1.134904587130e+03, 1.135221272310e+03, 1.135537387170e+03, 1.135852934030e+03, 1.136167915230e+03, 1.136482333060e+03, 1.136796189810e+03, 1.137109487770e+03, 1.137422229190e+03, 1.137734416330e+03, 1.138046051410e+03, 1.138357136670e+03, 1.138667674290e+03, 1.138977666490e+03, 1.139287115430e+03, 1.139596023290e+03, 1.139904392210e+03, 1.140212224340e+03, 1.140519521800e+03, 1.140826286700e+03, 1.141132521140e+03, 1.141438227220e+03, 1.141743407000e+03, 1.142048062550e+03, 1.142352195920e+03, 1.142655809140e+03, 1.142958904240e+03, 1.143261483240e+03, 1.143563548120e+03, 1.143865100890e+03, 1.144166143520e+03, 1.144466677960e+03, 1.144766706190e+03, 1.145066230130e+03, 1.145365251720e+03, 1.145663772870e+03, 1.145961795500e+03, 1.146259321500e+03, 1.146556352750e+03, 1.146852891130e+03, 1.147148938500e+03, 1.147444496720e+03, 1.147739567610e+03, 1.148034153030e+03, 1.148328254770e+03, 1.148621874660e+03, 1.148915014490e+03, 1.149207676060e+03, 1.149499861130e+03, 1.149791571470e+03, 1.150082808850e+03, 1.150373575020e+03},
    {1.828683791030e+00, 5.588806393100e+00, 9.436064913890e+00, 1.337584369460e+01, 1.741408556370e+01, 2.155737458890e+01, 2.581303517100e+01, 3.018925159360e+01, 3.469521342930e+01, 3.934129396800e+01, 4.413927130440e+01, 4.910260522550e+01, 5.424678810620e+01, 5.958979547470e+01, 6.515267312630e+01, 7.096031493580e+01, 7.704251284410e+01, 8.343540506120e+01, 9.018352379070e+01, 9.734277612380e+01, 1.049849357930e+02, 1.132046997000e+02, 1.221313588590e+02, 1.319494028820e+02, 1.429381607920e+02, 1.555577412730e+02, 1.706722551530e+02, 7.982211802590e+02, 8.036438397660e+02, 8.086535905650e+02, 8.133200873420e+02, 8.176957310630e+02, 8.218211239510e+02, 8.257284745310e+02, 8.294438294420e+02, 8.329885942660e+02, 8.363806033470e+02, 8.396348925400e+02, 8.427642700920e+02, 8.457797466230e+02, 8.486908645090e+02, 8.515059539720e+02, 8.542323348470e+02, 8.568764774250e+02, 8.594441320700e+02, 8.619404346710e+02, 8.643699931960e+02, 8.667369593250e+02, 8.690450881560e+02, 8.712977883320e+02, 8.734981643740e+02, 8.756490526600e+02, 8.777530521680e+02, 8.798125508650e+02, 8.818297484960e+02, 8.838066763220e+02, 8.857452143170e+02, 8.876471061810e+02, 8.895139725260e+02, 8.913473224750e+02, 8.931485639090e+02, 8.949190125510e+02, 8.966599000410e+02, 8.983723811270e+02, 9.000575400980e+02, 9.017163965520e+02, 9.033499105720e+02, 9.049589873910e+02, 9.065444816000e+02, 9.081072009580e+02, 9.096479098440e+02, 9.111673323900e+02, 9.126661553360e+02, 9.141450306330e+02, 9.156045778140e+02, 9.170453861720e+02, 9.184680167510e+02, 9.198730041710e+02, 9.212608583130e+02, 9.226320658640e+02, 9.239870917440e+02, 9.253263804210e+02, 9.266503571310e+02, 9.279594290020e+02, 9.292539861010e+02, 9.305344024000e+02, 9.318010366760e+02, 9.330542333510e+02, 9.342943232680e+02, 9.355216244210e+02, 9.367364426260e+02, 9.379390721610e+02, 9.391297963530e+02, 9.403088881350e+02, 9.414766105600e+02, 9.426332172900e+02, 9.437789530530e+02, 9.449140540680e+02, 9.460387484480e+02, 9.471532565820e+02, 9.482577914880e+02, 9.493525591460e+02, 9.504377588230e+02, 9.515135833630e+02, 9.525802194720e+02, 9.536378479850e+02, 9.546866441170e+02, 9.557267777010e+02, 9.567584134140e+02, 9.577817109910e+02, 9.587968254230e+02, 9.598039071550e+02, 9.608031022670e+02, 9.617945526430e+02, 9.627783961380e+02, 9.637547667360e+02, 9.647237946940e+02, 9.656856066890e+02, 9.666403259480e+02, 9.675880723760e+02, 9.685289626840e+02, 9.694631104990e+02, 9.703906264780e+02, 9.713116184130e+02, 9.722261913360e+02, 9.731344476100e+02, 9.740364870250e+02, 9.749324068860e+02, 9.758223020990e+02, 9.767062652470e+02, 9.775843866750e+02, 9.784567545580e+02, 9.793234549720e+02, 9.801845719690e+02, 9.810401876350e+02, 9.818903821550e+02, 9.827352338760e+02, 9.835748193600e+02, 9.844092134440e+02, 9.852384892900e+02, 9.860627184370e+02, 9.868819708520e+02, 9.876963149740e+02, 9.885058177630e+02, 9.893105447420e+02, 9.901105600390e+02, 9.909059264290e+02, 9.916967053700e+02, 9.924829570440e+02, 9.932647403930e+02, 9.940421131520e+02, 9.948151318840e+02, 9.955838520120e+02, 9.963483278520e+02, 9.971086126400e+02, 9.978647585650e+02, 9.986168167960e+02, 9.993648375070e+02, 1.000108869910e+03, 1.000848962260e+03, 1.001585161920e+03, 1.002317515340e+03, 1.003046068120e+03, 1.003770864980e+03, 1.004491949850e+03, 1.005209365830e+03, 1.005923155250e+03, 1.006633359660e+03, 1.007340019880e+03, 1.008043175990e+03, 1.008742867360e+03, 1.009439132670e+03, 1.010132009930e+03, 1.010821536480e+03, 1.011507749010e+03, 1.012190683600e+03, 1.012870375700e+03, 1.013546860160e+03, 1.014220171250e+03, 1.014890342660e+03, 1.015557407530e+03, 1.016221398450e+03, 1.016882347460e+03, 1.017540286100e+03, 1.018195245380e+03, 1.018847255820e+03, 1.019496347450e+03, 1.020142549830e+03, 1.020785892030e+03, 1.021426402670e+03, 1.022064109950e+03, 1.022699041590e+03, 1.023331224900e+03, 1.023960686790e+03, 1.024587453720e+03, 1.025211551770e+03, 1.025833006620e+03, 1.026451843580e+03, 1.027068087560e+03, 1.027681763110e+03, 1.028292894400e+03, 1.028901505280e+03, 1.029507619230e+03, 1.030111259370e+03, 1.030712448530e+03, 1.031311209160e+03, 1.031907563440e+03, 1.032501533200e+03, 1.033093139970e+03, 1.033682404970e+03, 1.034269349140e+03, 1.034853993120e+03, 1.035436357260e+03, 1.036016461620e+03, 1.036594326000e+03, 1.037169969930e+03, 1.037743412680e+03, 1.038314673220e+03, 1.038883770320e+03, 1.039450722470e+03, 1.040015547900e+03, 1.040578264630e+03, 1.041138890430e+03, 1.041697442820e+03, 1.042253939110e+03, 1.042808396380e+03, 1.043360831490e+03, 1.043911261080e+03, 1.044459701590e+03, 1.045006169220e+03, 1.045550680000e+03, 1.046093249720e+03, 1.046633894020e+03, 1.047172628290e+03, 1.047709467770e+03, 1.048244427490e+03, 1.048777522310e+03, 1.049308766870e+03, 1.049838175690e+03, 1.050365763060e+03, 1.050891543120e+03, 1.051415529850e+03, 1.051937737030e+03, 1.052458178310e+03, 1.052976867160e+03, 1.053493816870e+03, 1.054009040620e+03, 1.054522551400e+03, 1.055034362060e+03, 1.055544485290e+03, 1.056052933650e+03, 1.056559719540e+03, 1.057064855210e+03, 1.057568352800e+03, 1.058070224290e+03, 1.058570481520e+03, 1.059069136210e+03, 1.059566199930e+03, 1.060061684140e+03, 1.060555600170e+03, 1.061047959190e+03, 1.061538772300e+03, 1.062028050430e+03, 1.062515804430e+03, 1.063002044990e+03, 1.063486782710e+03, 1.063970028080e+03, 1.064451791450e+03, 1.064932083080e+03, 1.065410913120e+03, 1.065888291590e+03, 1.066364228430e+03, 1.066838733460e+03, 1.067311816400e+03, 1.067783486850e+03, 1.068253754350e+03, 1.068722628300e+03, 1.069190118020e+03, 1.069656232730e+03, 1.070120981550e+03, 1.070584373530e+03, 1.071046417590e+03, 1.071507122590e+03, 1.071966497280e+03, 1.072424550330e+03, 1.072881290320e+03, 1.073336725750e+03, 1.073790865020e+03, 1.074243716460e+03, 1.074695288310e+03, 1.075145588730e+03, 1.075594625810e+03, 1.076042407530e+03, 1.076488941830e+03, 1.076934236540e+03, 1.077378299440e+03, 1.077821138210e+03, 1.078262760480e+03, 1.078703173780e+03, 1.079142385590e+03, 1.079580403300e+03, 1.080017234250e+03, 1.080452885700e+03, 1.080887364830e+03, 1.081320678770e+03, 1.081752834580e+03, 1.082183839230e+03, 1.082613699670e+03, 1.083042422740e+03, 1.083470015240e+03, 1.083896483900e+03, 1.084321835400e+03, 1.084746076340e+03, 1.085169213270e+03, 1.085591252680e+03, 1.086012201000e+03, 1.086432064600e+03, 1.086850849800e+03, 1.087268562840e+03, 1.087685209940e+03, 1.088100797230e+03, 1.088515330810e+03, 1.088928816700e+03, 1.089341260900e+03, 1.089752669340e+03, 1.090163047880e+03, 1.090572402350e+03, 1.090980738540e+03, 1.091388062150e+03, 1.091794378880e+03, 1.092199694330e+03, 1.092604014090e+03, 1.093007343680e+03, 1.093409688580e+03, 1.093811054240e+03, 1.094211446030e+03, 1.094610869290e+03, 1.095009329330e+03, 1.095406831390e+03, 1.095803380670e+03, 1.096198982350e+03, 1.096593641540e+03, 1.096987363320e+03, 1.097380152710e+03, 1.097772014720e+03, 1.098162954300e+03, 1.098552976350e+03, 1.098942085740e+03, 1.099330287310e+03, 1.099717585840e+03, 1.100103986080e+03, 1.100489492750e+03, 1.100874110520e+03, 1.101257844030e+03, 1.101640697870e+03, 1.102022676610e+03, 1.102403784770e+03, 1.102784026840e+03, 1.103163407270e+03, 1.103541930480e+03, 1.103919600850e+03, 1.104296422740e+03, 1.104672400440e+03, 1.105047538250e+03, 1.105421840410e+03, 1.105795311130e+03, 1.106167954600e+03, 1.106539774950e+03, 1.106910776300e+03, 1.107280962740e+03, 1.107650338320e+03, 1.108018907060e+03, 1.108386672950e+03, 1.108753639940e+03, 1.109119811980e+03, 1.109485192950e+03, 1.109849786720e+03, 1.110213597140e+03, 1.110576628020e+03, 1.110938883130e+03, 1.111300366240e+03, 1.111661081070e+03, 1.112021031320e+03, 1.112380220650e+03, 1.112738652720e+03, 1.113096331130e+03, 1.113453259490e+03, 1.113809441340e+03, 1.114164880230e+03, 1.114519579670e+03, 1.114873543140e+03, 1.115226774110e+03, 1.115579276000e+03, 1.115931052240e+03, 1.116282106200e+03, 1.116632441240e+03, 1.116982060710e+03, 1.117330967900e+03, 1.117679166130e+03, 1.118026658640e+03, 1.118373448680e+03, 1.118719539470e+03, 1.119064934200e+03, 1.119409636060e+03, 1.119753648180e+03, 1.120096973710e+03, 1.120439615740e+03, 1.120781577370e+03, 1.121122861650e+03, 1.121463471630e+03, 1.121803410330e+03, 1.122142680750e+03, 1.122481285870e+03, 1.122819228650e+03, 1.123156512030e+03, 1.123493138930e+03, 1.123829112240e+03, 1.124164434850e+03, 1.124499109610e+03, 1.124833139360e+03, 1.125166526930e+03, 1.125499275110e+03, 1.125831386690e+03, 1.126162864430e+03, 1.126493711080e+03, 1.126823929370e+03, 1.127153521990e+03, 1.127482491650e+03, 1.127810841020e+03, 1.128138572750e+03, 1.128465689470e+03, 1.128792193810e+03, 1.129118088380e+03, 1.129443375750e+03, 1.129768058500e+03, 1.130092139170e+03, 1.130415620310e+03, 1.130738504420e+03, 1.131060794020e+03, 1.131382491590e+03, 1.131703599600e+03, 1.132024120510e+03, 1.132344056740e+03, 1.132663410730e+03, 1.132982184880e+03, 1.133300381580e+03, 1.133618003220e+03, 1.133935052140e+03, 1.134251530700e+03, 1.134567441240e+03, 1.134882786050e+03, 1.135197567460e+03, 1.135511787740e+03, 1.135825449160e+03, 1.136138554000e+03, 1.136451104490e+03, 1.136763102860e+03, 1.137074551330e+03, 1.137385452110e+03, 1.137695807380e+03, 1.138005619320e+03, 1.138314890090e+03, 1.138623621840e+03, 1.138931816710e+03, 1.139239476820e+03, 1.139546604280e+03, 1.139853201190e+03, 1.140159269630e+03, 1.140464811680e+03, 1.140769829390e+03, 1.141074324800e+03, 1.141378299970e+03, 1.141681756890e+03, 1.141984697600e+03, 1.142287124080e+03, 1.142589038320e+03, 1.142890442290e+03, 1.143191337960e+03, 1.143491727280e+03, 1.143791612180e+03, 1.144090994600e+03, 1.144389876450e+03, 1.144688259630e+03, 1.144986146050e+03, 1.145283537570e+03, 1.145580436080e+03, 1.145876843430e+03, 1.146172761470e+03, 1.146468192050e+03, 1.146763136980e+03, 1.147057598100e+03, 1.147351577200e+03, 1.147645076080e+03, 1.147938096530e+03, 1.148230640330e+03},
    {1.822180409110e+00, 5.568194748330e+00, 9.399942386330e+00, 1.332264094460e+01, 1.734204017190e+01, 2.146449934080e+01, 2.569707962660e+01, 3.004765516080e+01, 3.452504758190e+01, 3.913919044320e+01, 4.390133197970e+01, 4.882428774340e+01, 5.392275893670e+01, 5.921373855810e+01, 6.471703684240e+01, 7.045597172290e+01, 7.645829228630e+01, 8.275743890620e+01, 8.939430300320e+01, 9.641975145160e+01, 1.038983643230e+02, 1.119141827110e+02, 1.205799652700e+02, 1.300529805810e+02, 1.405640311080e+02, 1.524764212250e+02, 1.664243648620e+02, 1.837227753890e+02, 7.905660183750e+02, 7.963288952000e+02, 8.016259033770e+02, 8.065397559160e+02, 8.111316994240e+02, 8.154485966280e+02, 8.195272424210e+02, 8.233971385930e+02, 8.270823547110e+02, 8.306028196910e+02, 8.339752440960e+02, 8.372137947810e+02, 8.403305986520e+02, 8.433361256350e+02, 8.462394844150e+02, 8.490486540350e+02, 8.517706675310e+02, 8.544117591730e+02, 8.569774837280e+02, 8.594728139640e+02, 8.619022210210e+02, 8.642697411890e+02, 8.665790317880e+02, 8.688334182230e+02, 8.710359338660e+02, 8.731893540420e+02, 8.752962251500e+02, 8.773588897310e+02, 8.793795081550e+02, 8.813600774660e+02, 8.833024478110e+02, 8.852083368380e+02, 8.870793423470e+02, 8.889169534410e+02, 8.907225604050e+02, 8.924974634610e+02, 8.942428805740e+02, 8.959599544100e+02, 8.976497585720e+02, 8.993133031890e+02, 9.009515399470e+02, 9.025653666290e+02, 9.041556312050e+02, 9.057231355510e+02, 9.072686388120e+02, 9.087928604650e+02, 9.102964831070e+02, 9.117801550000e+02, 9.132444923970e+02, 9.146900816710e+02, 9.161174812700e+02, 9.175272235100e+02, 9.189198162220e+02, 9.202957442780e+02, 9.216554709910e+02, 9.229994394090e+02, 9.243280735130e+02, 9.256417793320e+02, 9.269409459640e+02, 9.282259465400e+02, 9.294971391030e+02, 9.307548674410e+02, 9.319994618540e+02, 9.332312398710e+02, 9.344505069190e+02, 9.356575569520e+02, 9.368526730330e+02, 9.380361278820e+02, 9.392081843910e+02, 9.403690961040e+02, 9.415191076690e+02, 9.426584552630e+02, 9.437873669880e+02, 9.449060632500e+02, 9.460147571120e+02, 9.471136546240e+02, 9.482029551390e+02, 9.492828516100e+02, 9.503535308700e+02, 9.514151738930e+02, 9.524679560490e+02, 9.535120473380e+02, 9.545476126130e+02, 9.555748117960e+02, 9.565938000760e+02, 9.576047281000e+02, 9.586077421550e+02, 9.596029843440e+02, 9.605905927410e+02, 9.615707015550e+02, 9.625434412720e+02, 9.635089387980e+02, 9.644673175960e+02, 9.654186978070e+02, 9.663631963790e+02, 9.673009271780e+02, 9.682320010990e+02, 9.691565261780e+02, 9.700746076830e+02, 9.709863482190e+02, 9.718918478170e+02, 9.727912040210e+02, 9.736845119740e+02, 9.745718645000e+02, 9.754533521760e+02, 9.763290634150e+02, 9.771990845280e+02, 9.780634997980e+02, 9.789223915410e+02, 9.797758401740e+02, 9.806239242680e+02, 9.814667206130e+02, 9.823043042680e+02, 9.831367486180e+02, 9.839641254200e+02, 9.847865048580e+02, 9.856039555890e+02, 9.864165447830e+02, 9.872243381760e+02, 9.880274001040e+02, 9.888257935470e+02, 9.896195801690e+02, 9.904088203540e+02, 9.911935732430e+02, 9.919738967680e+02, 9.927498476870e+02, 9.935214816170e+02, 9.942888530640e+02, 9.950520154520e+02, 9.958110211570e+02, 9.965659215310e+02, 9.973167669300e+02, 9.980636067410e+02, 9.988064894100e+02, 9.995454624600e+02, 1.000280572520e+03, 1.001011865360e+03, 1.001739385870e+03, 1.002463178140e+03, 1.003183285440e+03, 1.003899750270e+03, 1.004612614320e+03, 1.005321918590e+03, 1.006027703300e+03, 1.006730007990e+03, 1.007428871480e+03, 1.008124331930e+03, 1.008816426830e+03, 1.009505193020e+03, 1.010190666730e+03, 1.010872883540e+03, 1.011551878450e+03, 1.012227685860e+03, 1.012900339620e+03, 1.013569872980e+03, 1.014236318660e+03, 1.014899708870e+03, 1.015560075240e+03, 1.016217448930e+03, 1.016871860580e+03, 1.017523340360e+03, 1.018171917930e+03, 1.018817622490e+03, 1.019460482800e+03, 1.020100527160e+03, 1.020737783410e+03, 1.021372278990e+03, 1.022004040900e+03, 1.022633095740e+03, 1.023259469680e+03, 1.023883188540e+03, 1.024504277700e+03, 1.025122762210e+03, 1.025738666700e+03, 1.026352015480e+03, 1.026962832480e+03, 1.027571141280e+03, 1.028176965110e+03, 1.028780326900e+03, 1.029381249210e+03, 1.029979754290e+03, 1.030575864080e+03, 1.031169600210e+03, 1.031760984010e+03, 1.032350036480e+03, 1.032936778360e+03, 1.033521230100e+03, 1.034103411860e+03, 1.034683343510e+03, 1.035261044680e+03, 1.035836534700e+03, 1.036409832670e+03, 1.036980957410e+03, 1.037549927490e+03, 1.038116761260e+03, 1.038681476790e+03, 1.039244091930e+03, 1.039804624290e+03, 1.040363091260e+03, 1.040919510000e+03, 1.041473897440e+03, 1.042026270300e+03, 1.042576645080e+03, 1.043125038080e+03, 1.043671465380e+03, 1.044215942860e+03, 1.044758486220e+03, 1.045299110930e+03, 1.045837832300e+03, 1.046374665410e+03, 1.046909625210e+03, 1.047442726410e+03, 1.047973983570e+03, 1.048503411080e+03, 1.049031023120e+03, 1.049556833750e+03, 1.050080856810e+03, 1.050603106010e+03, 1.051123594870e+03, 1.051642336780e+03, 1.052159344940e+03, 1.052674632430e+03, 1.053188212130e+03, 1.053700096820e+03, 1.054210299100e+03, 1.054718831440e+03, 1.055225706140e+03, 1.055730935400e+03, 1.056234531250e+03, 1.056736505580e+03, 1.057236870180e+03, 1.057735636680e+03, 1.058232816570e+03, 1.058728421240e+03, 1.059222461930e+03, 1.059714949770e+03, 1.060205895760e+03, 1.060695310780e+03, 1.061183205590e+03, 1.061669590830e+03, 1.062154477040e+03, 1.062637874630e+03, 1.063119793900e+03, 1.063600245030e+03, 1.064079238120e+03, 1.064556783130e+03, 1.065032889940e+03, 1.065507568310e+03, 1.065980827900e+03, 1.066452678270e+03, 1.066923128890e+03, 1.067392189100e+03, 1.067859868190e+03, 1.068326175300e+03, 1.068791119530e+03, 1.069254709850e+03, 1.069716955150e+03, 1.070177864230e+03, 1.070637445800e+03, 1.071095708480e+03, 1.071552660800e+03, 1.072008311210e+03, 1.072462668090e+03, 1.072915739700e+03, 1.073367534250e+03, 1.073818059860e+03, 1.074267324560e+03, 1.074715336310e+03, 1.075162103000e+03, 1.075607632430e+03, 1.076051932320e+03, 1.076495010330e+03, 1.076936874040e+03, 1.077377530950e+03, 1.077816988510e+03, 1.078255254060e+03, 1.078692334920e+03, 1.079128238300e+03, 1.079562971360e+03, 1.079996541190e+03, 1.080428954800e+03, 1.080860219170e+03, 1.081290341170e+03, 1.081719327640e+03, 1.082147185340e+03, 1.082573920970e+03, 1.082999541170e+03, 1.083424052530e+03, 1.083847461560e+03, 1.084269774720e+03, 1.084690998420e+03, 1.085111138990e+03, 1.085530202730e+03, 1.085948195850e+03, 1.086365124550e+03, 1.086780994930e+03, 1.087195813060e+03, 1.087609584950e+03, 1.088022316560e+03, 1.088434013790e+03, 1.088844682500e+03, 1.089254328480e+03, 1.089662957490e+03, 1.090070575230e+03, 1.090477187350e+03, 1.090882799450e+03, 1.091287417090e+03, 1.091691045770e+03, 1.092093690950e+03, 1.092495358040e+03, 1.092896052420e+03, 1.093295779400e+03, 1.093694544250e+03, 1.094092352210e+03, 1.094489208470e+03, 1.094885118170e+03, 1.095280086400e+03, 1.095674118230e+03, 1.096067218680e+03, 1.096459392710e+03, 1.096850645270e+03, 1.097240981240e+03, 1.097630405470e+03, 1.098018922780e+03, 1.098406537950e+03, 1.098793255700e+03, 1.099179080730e+03, 1.099564017700e+03, 1.099948071230e+03, 1.100331245910e+03, 1.100713546280e+03, 1.101094976840e+03, 1.101475542080e+03, 1.101855246430e+03, 1.102234094300e+03, 1.102612090050e+03, 1.102989238020e+03, 1.103365542500e+03, 1.103741007760e+03, 1.104115638040e+03, 1.104489437540e+03, 1.104862410420e+03, 1.105234560810e+03, 1.105605892820e+03, 1.105976410520e+03, 1.106346117950e+03, 1.106715019120e+03, 1.107083118000e+03, 1.107450418540e+03, 1.107816924660e+03, 1.108182640260e+03, 1.108547569170e+03, 1.108911715250e+03, 1.109275082270e+03, 1.109637674030e+03, 1.109999494250e+03, 1.110360546660e+03, 1.110720834950e+03, 1.111080362770e+03, 1.111439133760e+03, 1.111797151520e+03, 1.112154419630e+03, 1.112510941650e+03, 1.112866721110e+03, 1.113221761510e+03, 1.113576066320e+03, 1.113929638990e+03, 1.114282482950e+03, 1.114634601600e+03, 1.114985998320e+03, 1.115336676460e+03, 1.115686639350e+03, 1.116035890290e+03, 1.116384432560e+03, 1.116732269430e+03, 1.117079404110e+03, 1.117425839840e+03, 1.117771579780e+03, 1.118116627120e+03, 1.118460984990e+03, 1.118804656510e+03, 1.119147644790e+03, 1.119489952910e+03, 1.119831583910e+03, 1.120172540840e+03, 1.120512826710e+03, 1.120852444510e+03, 1.121191397220e+03, 1.121529687790e+03, 1.121867319150e+03, 1.122204294210e+03, 1.122540615870e+03, 1.122876287000e+03, 1.123211310450e+03, 1.123545689050e+03, 1.123879425620e+03, 1.124212522950e+03, 1.124544983830e+03, 1.124876811010e+03, 1.125208007220e+03, 1.125538575200e+03, 1.125868517640e+03, 1.126197837230e+03, 1.126526536630e+03, 1.126854618500e+03, 1.127182085460e+03, 1.127508940140e+03, 1.127835185130e+03, 1.128160823000e+03, 1.128485856330e+03, 1.128810287670e+03, 1.129134119530e+03, 1.129457354440e+03, 1.129779994890e+03, 1.130102043360e+03, 1.130423502330e+03, 1.130744374230e+03, 1.131064661500e+03, 1.131384366570e+03, 1.131703491830e+03, 1.132022039670e+03, 1.132340012470e+03, 1.132657412570e+03, 1.132974242330e+03, 1.133290504070e+03, 1.133606200100e+03, 1.133921332720e+03, 1.134235904210e+03, 1.134549916850e+03, 1.134863372890e+03, 1.135176274570e+03, 1.135488624120e+03, 1.135800423750e+03, 1.136111675660e+03, 1.136422382030e+03, 1.136732545050e+03, 1.137042166860e+03, 1.137351249620e+03, 1.137659795450e+03, 1.137967806480e+03, 1.138275284810e+03, 1.138582232550e+03, 1.138888651760e+03, 1.139194544510e+03, 1.139499912880e+03, 1.139804758880e+03, 1.140109084570e+03, 1.140412891960e+03, 1.140716183040e+03, 1.141018959830e+03, 1.141321224300e+03, 1.141622978430e+03, 1.141924224170e+03, 1.142224963470e+03, 1.142525198270e+03, 1.142824930490e+03, 1.143124162040e+03, 1.143422894840e+03, 1.143721130770e+03, 1.144018871700e+03, 1.144316119520e+03, 1.144612876070e+03, 1.144909143210e+03, 1.145204922760e+03, 1.145500216570e+03, 1.145795026430e+03, 1.146089354170e+03},
    {1.815723997780e+00, 5.547743232010e+00, 9.364121459010e+00, 1.326991533530e+01, 1.727068862480e+01, 2.137258460570e+01, 2.558241221580e+01, 2.990774925550e+01, 3.435706695320e+01, 3.893988154570e+01, 4.366694037560e+01, 4.855045260310e+01, 5.360437830370e+01, 5.884479504620e+01, 6.429036887260e+01, 6.996296838670e+01, 7.588847881720e+01, 8.209790164940e+01, 8.862887230260e+01, 9.552780750880e+01, 1.028530333120e+02, 1.106795014030e+02, 1.191062025060e+02, 1.282684325590e+02, 1.383594510330e+02, 1.496721326910e+02, 1.626891266180e+02, 1.783159935320e+02, 7.759656411610e+02, 7.827654883550e+02, 7.888968297090e+02, 7.945016441620e+02, 7.996783839200e+02, 8.044987241470e+02, 8.090167787520e+02, 8.132745780270e+02, 8.173055202630e+02, 8.211366486420e+02, 8.247902097220e+02, 8.282847532440e+02, 8.316359284060e+02, 8.348570731800e+02, 8.379596587980e+02, 8.409536306400e+02, 8.438476735280e+02, 8.466494209180e+02, 8.493656217910e+02, 8.520022752270e+02, 8.545647399650e+02, 8.570578243860e+02, 8.594858610060e+02, 8.618527686190e+02, 8.641621044730e+02, 8.664171083790e+02, 8.686207401890e+02, 8.707757118460e+02, 8.728845149090e+02, 8.749494443190e+02, 8.769726190000e+02, 8.789559998040e+02, 8.809014051940e+02, 8.828105250050e+02, 8.846849325680e+02, 8.865260954150e+02, 8.883353847690e+02, 8.901140839820e+02, 8.918633960580e+02, 8.935844503740e+02, 8.952783087130e+02, 8.969459706750e+02, 8.985883785550e+02, 9.002064217470e+02, 9.018009407230e+02, 9.033727306520e+02, 9.049225446680e+02, 9.064510968630e+02, 9.079590650050e+02, 9.094470930220e+02, 9.109157932750e+02, 9.123657486460e+02, 9.137975144430e+02, 9.152116201630e+02, 9.166085711090e+02, 9.179888498800e+02, 9.193529177470e+02, 9.207012159290e+02, 9.220341667670e+02, 9.233521748210e+02, 9.246556278760e+02, 9.259448978850e+02, 9.272203418470e+02, 9.284823026180e+02, 9.297311096680e+02, 9.309670797940e+02, 9.321905177760e+02, 9.334017170000e+02, 9.346009600300e+02, 9.357885191530e+02, 9.369646568870e+02, 9.381296264560e+02, 9.392836722350e+02, 9.404270301720e+02, 9.415599281830e+02, 9.426825865240e+02, 9.437952181390e+02, 9.448980289920e+02, 9.459912183780e+02, 9.470749792150e+02, 9.481494983240e+02, 9.492149566910e+02, 9.502715297160e+02, 9.513193874420e+02, 9.523586947870e+02, 9.533896117470e+02, 9.544122936000e+02, 9.554268910950e+02, 9.564335506310e+02, 9.574324144280e+02, 9.584236206910e+02, 9.594073037660e+02, 9.603835942820e+02, 9.613526192970e+02, 9.623145024260e+02, 9.632693639720e+02, 9.642173210460e+02, 9.651584876830e+02, 9.660929749510e+02, 9.670208910560e+02, 9.679423414480e+02, 9.688574289080e+02, 9.697662536470e+02, 9.706689133920e+02, 9.715655034690e+02, 9.724561168850e+02, 9.733408444050e+02, 9.742197746240e+02, 9.750929940390e+02, 9.759605871180e+02, 9.768226363640e+02, 9.776792223760e+02, 9.785304239130e+02, 9.793763179470e+02, 9.802169797200e+02, 9.810524827990e+02, 9.818828991250e+02, 9.827082990620e+02, 9.835287514440e+02, 9.843443236230e+02, 9.851550815100e+02, 9.859610896180e+02, 9.867624111010e+02, 9.875591077980e+02, 9.883512402630e+02, 9.891388678080e+02, 9.899220485360e+02, 9.907008393710e+02, 9.914752960960e+02, 9.922454733820e+02, 9.930114248170e+02, 9.937732029370e+02, 9.945308592550e+02, 9.952844442840e+02, 9.960340075700e+02, 9.967795977110e+02, 9.975212623870e+02, 9.982590483800e+02, 9.989930015970e+02, 9.997231670970e+02, 1.000449589110e+03, 1.001172311040e+03, 1.001891375540e+03, 1.002606824450e+03, 1.003318698900e+03, 1.004027039250e+03, 1.004731885170e+03, 1.005433275640e+03, 1.006131248940e+03, 1.006825842680e+03, 1.007517093860e+03, 1.008205038820e+03, 1.008889713270e+03, 1.009571152340e+03, 1.010249390560e+03, 1.010924461890e+03, 1.011596399710e+03, 1.012265236880e+03, 1.012931005700e+03, 1.013593737940e+03, 1.014253464860e+03, 1.014910217230e+03, 1.015564025320e+03, 1.016214918920e+03, 1.016862927340e+03, 1.017508079430e+03, 1.018150403610e+03, 1.018789927840e+03, 1.019426679650e+03, 1.020060686160e+03, 1.020691974050e+03, 1.021320569630e+03, 1.021946498780e+03, 1.022569787010e+03, 1.023190459450e+03, 1.023808540850e+03, 1.024424055600e+03, 1.025037027720e+03, 1.025647480900e+03, 1.026255438470e+03, 1.026860923430e+03, 1.027463958440e+03, 1.028064565850e+03, 1.028662767680e+03, 1.029258585650e+03, 1.029852041170e+03, 1.030443155340e+03, 1.031031948980e+03, 1.031618442620e+03, 1.032202656500e+03, 1.032784610590e+03, 1.033364324580e+03, 1.033941817900e+03, 1.034517109710e+03, 1.035090218920e+03, 1.035661164190e+03, 1.036229963920e+03, 1.036796636270e+03, 1.037361199160e+03, 1.037923670290e+03, 1.038484067110e+03, 1.039042406840e+03, 1.039598706510e+03, 1.040152982890e+03, 1.040705252560e+03, 1.041255531890e+03, 1.041803837020e+03, 1.042350183920e+03, 1.042894588320e+03, 1.043437065790e+03, 1.043977631680e+03, 1.044516301170e+03, 1.045053089230e+03, 1.045588010660e+03, 1.046121080090e+03, 1.046652311950e+03, 1.047181720500e+03, 1.047709319840e+03, 1.048235123890e+03, 1.048759146420e+03, 1.049281401000e+03, 1.049801901070e+03, 1.050320659910e+03, 1.050837690630e+03, 1.051353006200e+03, 1.051866619430e+03, 1.052378542980e+03, 1.052888789370e+03, 1.053397370970e+03, 1.053904300020e+03, 1.054409588600e+03, 1.054913248680e+03, 1.055415292070e+03, 1.055915730460e+03, 1.056414575400e+03, 1.056911838320e+03, 1.057407530510e+03, 1.057901663160e+03, 1.058394247310e+03, 1.058885293890e+03, 1.059374813710e+03, 1.059862817460e+03, 1.060349315720e+03, 1.060834318940e+03, 1.061317837470e+03, 1.061799881550e+03, 1.062280461310e+03, 1.062759586760e+03, 1.063237267820e+03, 1.063713514300e+03, 1.064188335900e+03, 1.064661742220e+03, 1.065133742770e+03, 1.065604346950e+03, 1.066073564060e+03, 1.066541403310e+03, 1.067007873830e+03, 1.067472984630e+03, 1.067936744630e+03, 1.068399162690e+03, 1.068860247540e+03, 1.069320007850e+03, 1.069778452190e+03, 1.070235589040e+03, 1.070691426820e+03, 1.071145973830e+03, 1.071599238310e+03, 1.072051228420e+03, 1.072501952240e+03, 1.072951417750e+03, 1.073399632870e+03, 1.073846605450e+03, 1.074292343240e+03, 1.074736853930e+03, 1.075180145140e+03, 1.075622224410e+03, 1.076063099200e+03, 1.076502776920e+03, 1.076941264880e+03, 1.077378570360e+03, 1.077814700530e+03, 1.078249662510e+03, 1.078683463370e+03, 1.079116110080e+03, 1.079547609580e+03, 1.079977968710e+03, 1.080407194280e+03, 1.080835293020e+03, 1.081262271600e+03, 1.081688136630e+03, 1.082112894650e+03, 1.082536552150e+03, 1.082959115580e+03, 1.083380591290e+03, 1.083800985600e+03, 1.084220304780e+03, 1.084638555020e+03, 1.085055742470e+03, 1.085471873210e+03, 1.085886953300e+03, 1.086300988710e+03, 1.086713985380e+03, 1.087125949180e+03, 1.087536885940e+03, 1.087946801440e+03, 1.088355701410e+03, 1.088763591520e+03, 1.089170477390e+03, 1.089576364620e+03, 1.089981258730e+03, 1.090385165210e+03, 1.090788089480e+03, 1.091190036950e+03, 1.091591012960e+03, 1.091991022800e+03, 1.092390071740e+03, 1.092788164970e+03, 1.093185307680e+03, 1.093581504980e+03, 1.093976761960e+03, 1.094371083650e+03, 1.094764475040e+03, 1.095156941110e+03, 1.095548486750e+03, 1.095939116850e+03, 1.096328836240e+03, 1.096717649710e+03, 1.097105562020e+03, 1.097492577890e+03, 1.097878701990e+03, 1.098263938980e+03, 1.098648293440e+03, 1.099031769950e+03, 1.099414373040e+03, 1.099796107210e+03, 1.100176976900e+03, 1.100556986550e+03, 1.100936140540e+03, 1.101314443230e+03, 1.101691898940e+03, 1.102068511940e+03, 1.102444286500e+03, 1.102819226820e+03, 1.103193337100e+03, 1.103566621490e+03, 1.103939084120e+03, 1.104310729060e+03, 1.104681560380e+03, 1.105051582100e+03, 1.105420798230e+03, 1.105789212720e+03, 1.106156829500e+03, 1.106523652500e+03, 1.106889685580e+03, 1.107254932580e+03, 1.107619397330e+03, 1.107983083620e+03, 1.108345995200e+03, 1.108708135800e+03, 1.109069509140e+03, 1.109430118890e+03, 1.109789968690e+03, 1.110149062180e+03, 1.110507402940e+03, 1.110864994550e+03, 1.111221840550e+03, 1.111577944460e+03, 1.111933309760e+03, 1.112287939940e+03, 1.112641838420e+03, 1.112995008620e+03, 1.113347453940e+03, 1.113699177750e+03, 1.114050183380e+03, 1.114400474150e+03, 1.114750053370e+03, 1.115098924290e+03, 1.115447090180e+03, 1.115794554260e+03, 1.116141319730e+03, 1.116487389760e+03, 1.116832767530e+03, 1.117177456160e+03, 1.117521458760e+03, 1.117864778440e+03, 1.118207418250e+03, 1.118549381260e+03, 1.118890670480e+03, 1.119231288930e+03, 1.119571239590e+03, 1.119910525420e+03, 1.120249149380e+03, 1.120587114380e+03, 1.120924423330e+03, 1.121261079130e+03, 1.121597084620e+03, 1.121932442670e+03, 1.122267156090e+03, 1.122601227700e+03, 1.122934660280e+03, 1.123267456600e+03, 1.123599619420e+03, 1.123931151470e+03, 1.124262055470e+03, 1.124592334110e+03, 1.124921990060e+03, 1.125251026010e+03, 1.125579444570e+03, 1.125907248390e+03, 1.126234440080e+03, 1.126561022210e+03, 1.126886997380e+03, 1.127212368130e+03, 1.127537137020e+03, 1.127861306560e+03, 1.128184879260e+03, 1.128507857610e+03, 1.128830244100e+03, 1.129152041180e+03, 1.129473251300e+03, 1.129793876880e+03, 1.130113920350e+03, 1.130433384090e+03, 1.130752270490e+03, 1.131070581920e+03, 1.131388320730e+03, 1.131705489260e+03, 1.132022089830e+03, 1.132338124750e+03, 1.132653596310e+03, 1.132968506800e+03, 1.133282858480e+03, 1.133596653590e+03, 1.133909894380e+03, 1.134222583080e+03, 1.134534721880e+03, 1.134846312980e+03, 1.135157358580e+03, 1.135467860820e+03, 1.135777821890e+03, 1.136087243900e+03, 1.136396129000e+03, 1.136704479290e+03, 1.137012296890e+03, 1.137319583880e+03, 1.137626342340e+03, 1.137932574340e+03, 1.138238281920e+03, 1.138543467130e+03, 1.138848132000e+03, 1.139152278540e+03, 1.139455908760e+03, 1.139759024650e+03, 1.140061628180e+03, 1.140363721340e+03, 1.140665306060e+03, 1.140966384300e+03, 1.141266957990e+03, 1.141567029060e+03, 1.141866599410e+03, 1.142165670950e+03, 1.142464245550e+03, 1.142762325110e+03, 1.143059911480e+03, 1.143357006520e+03, 1.143653612080e+03, 1.143949729980e+03},
    {1.809314035970e+00, 5.527449844960e+00, 9.328597939360e+00, 1.321765953810e+01, 1.720001921370e+01, 2.128161264660e+01, 2.546900698360e+01, 2.976949678060e+01, 3.419121937920e+01, 3.874329473860e+01, 4.343599635520e+01, 4.828096212800e+01, 5.329145717530e+01, 5.848270510530e+01, 6.387231080940e+01, 6.948080761240e+01, 7.533237648250e+01, 8.145580819540e+01, 8.788581659530e+01, 9.466487284890e+01, 1.018458369050e+02, 1.094958534780e+02, 1.177023415680e+02, 1.265826345430e+02, 1.363004090450e+02, 1.470958213600e+02, 1.593465287760e+02, 1.737100035260e+02, 1.915390514670e+02, 7.675226521460e+02, 7.748212560890e+02, 7.813507987980e+02, 7.872845657060e+02, 7.927398916560e+02, 7.978006824470e+02, 8.025294035120e+02, 8.069740262950e+02, 8.111723167000e+02, 8.151546166630e+02, 8.189457212850e+02, 8.225661875530e+02, 8.260332720370e+02, 8.293616185840e+02, 8.325637728780e+02, 8.356505743000e+02, 8.386314589750e+02, 8.415146973940e+02, 8.443075830420e+02, 8.470165838010e+02, 8.496474647030e+02, 8.522053883730e+02, 8.546949978950e+02, 8.571204857220e+02, 8.594856513670e+02, 8.617939500360e+02, 8.640485338660e+02, 8.662522870810e+02, 8.684078561500e+02, 8.705176757470e+02, 8.725839912290e+02, 8.746088781760e+02, 8.765942594350e+02, 8.785419200590e+02, 8.804535204440e+02, 8.823306079110e+02, 8.841746269650e+02, 8.859869284010e+02, 8.877687774130e+02, 8.895213608330e+02, 8.912457936140e+02, 8.929431246490e+02, 8.946143420020e+02, 8.962603776320e+02, 8.978821116560e+02, 8.994803762170e+02, 9.010559589840e+02, 9.026096063460e+02, 9.041420263100e+02, 9.056538911580e+02, 9.071458398660e+02, 9.086184803200e+02, 9.100723913560e+02, 9.115081246250e+02, 9.129262063140e+02, 9.143271387300e+02, 9.157114017570e+02, 9.170794542120e+02, 9.184317350870e+02, 9.197686647110e+02, 9.210906458170e+02, 9.223980645400e+02, 9.236912913390e+02, 9.249706818620e+02, 9.262365777420e+02, 9.274893073470e+02, 9.287291864800e+02, 9.299565190240e+02, 9.311715975570e+02, 9.323747039200e+02, 9.335661097520e+02, 9.347460769880e+02, 9.359148583350e+02, 9.370726977070e+02, 9.382198306440e+02, 9.393564847050e+02, 9.404828798270e+02, 9.415992286810e+02, 9.427057369920e+02, 9.438026038510e+02, 9.448900220060e+02, 9.459681781320e+02, 9.470372530980e+02, 9.480974222090e+02, 9.491488554380e+02, 9.501917176520e+02, 9.512261688130e+02, 9.522523641860e+02, 9.532704545160e+02, 9.542805862200e+02, 9.552829015430e+02, 9.562775387320e+02, 9.572646321810e+02, 9.582443125810e+02, 9.592167070580e+02, 9.601819393060e+02, 9.611401297130e+02, 9.620913954830e+02, 9.630358507490e+02, 9.639736066830e+02, 9.649047716020e+02, 9.658294510680e+02, 9.667477479820e+02, 9.676597626770e+02, 9.685655930040e+02, 9.694653344200e+02, 9.703590800610e+02, 9.712469208250e+02, 9.721289454440e+02, 9.730052405530e+02, 9.738758907570e+02, 9.747409786990e+02, 9.756005851220e+02, 9.764547889250e+02, 9.773036672220e+02, 9.781472954020e+02, 9.789857471730e+02, 9.798190946210e+02, 9.806474082540e+02, 9.814707570510e+02, 9.822892085080e+02, 9.831028286800e+02, 9.839116822230e+02, 9.847158324360e+02, 9.855153412990e+02, 9.863102695110e+02, 9.871006765240e+02, 9.878866205810e+02, 9.886681587490e+02, 9.894453469500e+02, 9.902182399930e+02, 9.909868916040e+02, 9.917513544570e+02, 9.925116802000e+02, 9.932679194830e+02, 9.940201219840e+02, 9.947683364360e+02, 9.955126106500e+02, 9.962529915390e+02, 9.969895251420e+02, 9.977222566440e+02, 9.984512304010e+02, 9.991764899590e+02, 9.998980780730e+02, 1.000616036730e+03, 1.001330407160e+03, 1.002041229860e+03, 1.002748544630e+03, 1.003452390550e+03, 1.004152806020e+03, 1.004849828790e+03, 1.005543495960e+03, 1.006233843970e+03, 1.006920908670e+03, 1.007604725300e+03, 1.008285328490e+03, 1.008962752320e+03, 1.009637030280e+03, 1.010308195330e+03, 1.010976279870e+03, 1.011641315800e+03, 1.012303334470e+03, 1.012962366760e+03, 1.013618443040e+03, 1.014271593190e+03, 1.014921846640e+03, 1.015569232330e+03, 1.016213778790e+03, 1.016855514050e+03, 1.017494465770e+03, 1.018130661140e+03, 1.018764126950e+03, 1.019394889590e+03, 1.020022975050e+03, 1.020648408920e+03, 1.021271216420e+03, 1.021891422390e+03, 1.022509051310e+03, 1.023124127290e+03, 1.023736674110e+03, 1.024346715170e+03, 1.024954273560e+03, 1.025559372030e+03, 1.026162033020e+03, 1.026762278620e+03, 1.027360130630e+03, 1.027955610540e+03, 1.028548739540e+03, 1.029139538520e+03, 1.029728028090e+03, 1.030314228570e+03, 1.030898160000e+03, 1.031479842140e+03, 1.032059294510e+03, 1.032636536340e+03, 1.033211586600e+03, 1.033784464030e+03, 1.034355187100e+03, 1.034923774040e+03, 1.035490242850e+03, 1.036054611290e+03, 1.036616896870e+03, 1.037177116900e+03, 1.037735288450e+03, 1.038291428370e+03, 1.038845553310e+03, 1.039397679690e+03, 1.039947823730e+03, 1.040496001450e+03, 1.041042228660e+03, 1.041586520970e+03, 1.042128893800e+03, 1.042669362400e+03, 1.043207941790e+03, 1.043744646830e+03, 1.044279492210e+03, 1.044812492420e+03, 1.045343661780e+03, 1.045873014450e+03, 1.046400564390e+03, 1.046926325430e+03, 1.047450311210e+03, 1.047972535220e+03, 1.048493010780e+03, 1.049011751060e+03, 1.049528769090e+03, 1.050044077730e+03, 1.050557689690e+03, 1.051069617540e+03, 1.051579873710e+03, 1.052088470470e+03, 1.052595419980e+03, 1.053100734230e+03, 1.053604425090e+03, 1.054106504300e+03, 1.054606983460e+03, 1.055105874050e+03, 1.055603187400e+03, 1.056098934740e+03, 1.056593127170e+03, 1.057085775660e+03, 1.057576891060e+03, 1.058066484110e+03, 1.058554565420e+03, 1.059041145510e+03, 1.059526234770e+03, 1.060009843460e+03, 1.060491981770e+03, 1.060972659750e+03, 1.061451887360e+03, 1.061929674450e+03, 1.062406030760e+03, 1.062880965930e+03, 1.063354489510e+03, 1.063826610940e+03, 1.064297339570e+03, 1.064766684640e+03, 1.065234655310e+03, 1.065701260640e+03, 1.066166509590e+03, 1.066630411050e+03, 1.067092973790e+03, 1.067554206510e+03, 1.068014117820e+03, 1.068472716250e+03, 1.068930010240e+03, 1.069386008130e+03, 1.069840718200e+03, 1.070294148630e+03, 1.070746307540e+03, 1.071197202960e+03, 1.071646842830e+03, 1.072095235030e+03, 1.072542387350e+03, 1.072988307510e+03, 1.073433003170e+03, 1.073876481890e+03, 1.074318751180e+03, 1.074759818450e+03, 1.075199691080e+03, 1.075638376350e+03, 1.076075881480e+03, 1.076512213620e+03, 1.076947379850e+03, 1.077381387200e+03, 1.077814242610e+03, 1.078245952980e+03, 1.078676525120e+03, 1.079105965810e+03, 1.079534281730e+03, 1.079961479530e+03, 1.080387565780e+03, 1.080812547000e+03, 1.081236429660e+03, 1.081659220140e+03, 1.082080924790e+03, 1.082501549900e+03, 1.082921101690e+03, 1.083339586350e+03, 1.083757009970e+03, 1.084173378640e+03, 1.084588698350e+03, 1.085002975070e+03, 1.085416214710e+03, 1.085828423100e+03, 1.086239606070e+03, 1.086649769350e+03, 1.087058918650e+03, 1.087467059630e+03, 1.087874197870e+03, 1.088280338950e+03, 1.088685488370e+03, 1.089089651590e+03, 1.089492834010e+03, 1.089895041020e+03, 1.090296277930e+03, 1.090696550030e+03, 1.091095862540e+03, 1.091494220650e+03, 1.091891629510e+03, 1.092288094220e+03, 1.092683619850e+03, 1.093078211410e+03, 1.093471873880e+03, 1.093864612190e+03, 1.094256431240e+03, 1.094647335890e+03, 1.095037330950e+03, 1.095426421200e+03, 1.095814611380e+03, 1.096201906180e+03, 1.096588310260e+03, 1.096973828260e+03, 1.097358464760e+03, 1.097742224300e+03, 1.098125111410e+03, 1.098507130550e+03, 1.098888286170e+03, 1.099268582690e+03, 1.099648024460e+03, 1.100026615840e+03, 1.100404361110e+03, 1.100781264560e+03, 1.101157330420e+03, 1.101532562890e+03, 1.101906966150e+03, 1.102280544330e+03, 1.102653301540e+03, 1.103025241850e+03, 1.103396369320e+03, 1.103766687950e+03, 1.104136201730e+03, 1.104504914610e+03, 1.104872830500e+03, 1.105239953310e+03, 1.105606286900e+03, 1.105971835090e+03, 1.106336601710e+03, 1.106700590510e+03, 1.107063805260e+03, 1.107426249660e+03, 1.107787927420e+03, 1.108148842200e+03, 1.108508997640e+03, 1.108868397350e+03, 1.109227044920e+03, 1.109584943900e+03, 1.109942097830e+03, 1.110298510210e+03, 1.110654184530e+03, 1.111009124240e+03, 1.111363332770e+03, 1.111716813530e+03, 1.112069569910e+03, 1.112421605250e+03, 1.112772922890e+03, 1.113123526150e+03, 1.113473418310e+03, 1.113822602630e+03, 1.114171082350e+03, 1.114518860690e+03, 1.114865940840e+03, 1.115212325970e+03, 1.115558019240e+03, 1.115903023770e+03, 1.116247342660e+03, 1.116590979000e+03, 1.116933935860e+03, 1.117276216270e+03, 1.117617823250e+03, 1.117958759800e+03, 1.118299028900e+03, 1.118638633520e+03, 1.118977576570e+03, 1.119315861000e+03, 1.119653489680e+03, 1.119990465510e+03, 1.120326791340e+03, 1.120662470000e+03, 1.120997504330e+03, 1.121331897110e+03, 1.121665651140e+03, 1.121998769170e+03, 1.122331253950e+03, 1.122663108220e+03, 1.122994334670e+03, 1.123324935990e+03, 1.123654914870e+03, 1.123984273960e+03, 1.124313015890e+03, 1.124641143280e+03, 1.124968658740e+03, 1.125295564850e+03, 1.125621864190e+03, 1.125947559310e+03, 1.126272652730e+03, 1.126597146990e+03, 1.126921044580e+03, 1.127244347990e+03, 1.127567059700e+03, 1.127889182150e+03, 1.128210717800e+03, 1.128531669050e+03, 1.128852038320e+03, 1.129171828010e+03, 1.129491040490e+03, 1.129809678120e+03, 1.130127743250e+03, 1.130445238210e+03, 1.130762165330e+03, 1.131078526900e+03, 1.131394325220e+03, 1.131709562560e+03, 1.132024241180e+03, 1.132338363330e+03, 1.132651931240e+03, 1.132964947130e+03, 1.133277413210e+03, 1.133589331660e+03, 1.133900704660e+03, 1.134211534390e+03, 1.134521822980e+03, 1.134831572590e+03, 1.135140785330e+03, 1.135449463310e+03, 1.135757608640e+03, 1.136065223400e+03, 1.136372309670e+03, 1.136678869510e+03, 1.136984904960e+03, 1.137290418070e+03, 1.137595410850e+03, 1.137899885330e+03, 1.138203843500e+03, 1.138507287340e+03, 1.138810218850e+03, 1.139112639970e+03, 1.139414552670e+03, 1.139715958880e+03, 1.140016860550e+03, 1.140317259570e+03, 1.140617157870e+03, 1.140916557340e+03, 1.141215459860e+03, 1.141513867310e+03, 1.141811781550e+03},
    {1.802950010570e+00, 5.507312624000e+00, 9.293367720940e+00, 1.316586639340e+01, 1.713002052850e+01, 2.119156622430e+01, 2.535683877200e+01, 2.963286188230e+01, 3.402745459870e+01, 3.854936036390e+01, 4.320840410230e+01, 4.801568510380e+01, 5.298381618030e+01, 5.812722338200e+01, 6.346252607200e+01, 6.900902527230e+01, 7.478934039740e+01, 8.083025327830e+01, 8.716384809210e+01, 9.382908422220e+01, 1.008740207430e+02, 1.083590545910e+02, 1.163617982660e+02, 1.249847359970e+02, 1.343678653770e+02, 1.447109500010e+02, 1.563161120780e+02, 1.696793188460e+02, 1.857236583040e+02, 2.065974811070e+02, 7.588925216920e+02, 7.667373491850e+02, 7.736959576030e+02, 7.799800914540e+02, 7.857295946990e+02, 7.910426217270e+02, 7.959911912190e+02, 8.006299700210e+02, 8.050015855910e+02, 8.091400137790e+02, 8.130728326350e+02, 8.168227747100e+02, 8.204088276460e+02, 8.238470339600e+02, 8.271510847040e+02, 8.303327683490e+02, 8.334023157700e+02, 8.363686692240e+02, 8.392396948010e+02, 8.420223521820e+02, 8.447228317010e+02, 8.473466660830e+02, 8.498988223230e+02, 8.523837778410e+02, 8.548055840680e+02, 8.571679199060e+02, 8.594741369390e+02, 8.617272979080e+02, 8.639302096280e+02, 8.660854512790e+02, 8.681953988580e+02, 8.702622463880e+02, 8.722880244020e+02, 8.742746161100e+02, 8.762237715970e+02, 8.781371203260e+02, 8.800161822000e+02, 8.818623773590e+02, 8.836770349060e+02, 8.854614006830e+02, 8.872166442240e+02, 8.889438649940e+02, 8.906440979950e+02, 8.923183188130e+02, 8.939674481820e+02, 8.955923561080e+02, 8.971938656090e+02, 8.987727561170e+02, 9.003297665680e+02, 9.018655982280e+02, 9.033809172670e+02, 9.048763571170e+02, 9.063525206340e+02, 9.078099820830e+02, 9.092492889590e+02, 9.106709636670e+02, 9.120755050710e+02, 9.134633899230e+02, 9.148350741800e+02, 9.161909942320e+02, 9.175315680320e+02, 9.188571961480e+02, 9.201682627370e+02, 9.214651364560e+02, 9.227481713090e+02, 9.240177074270e+02, 9.252740718110e+02, 9.265175790130e+02, 9.277485317810e+02, 9.289672216550e+02, 9.301739295330e+02, 9.313689261920e+02, 9.325524727900e+02, 9.337248213210e+02, 9.348862150580e+02, 9.360368889590e+02, 9.371770700520e+02, 9.383069778030e+02, 9.394268244510e+02, 9.405368153380e+02, 9.416371492100e+02, 9.427280185050e+02, 9.438096096270e+02, 9.448821032010e+02, 9.459456743190e+02, 9.470004927690e+02, 9.480467232540e+02, 9.490845256000e+02, 9.501140549510e+02, 9.511354619570e+02, 9.521488929510e+02, 9.531544901180e+02, 9.541523916540e+02, 9.551427319200e+02, 9.561256415850e+02, 9.571012477670e+02, 9.580696741610e+02, 9.590310411690e+02, 9.599854660150e+02, 9.609330628630e+02, 9.618739429240e+02, 9.628082145600e+02, 9.637359833860e+02, 9.646573523610e+02, 9.655724218840e+02, 9.664812898760e+02, 9.673840518670e+02, 9.682808010740e+02, 9.691716284770e+02, 9.700566228950e+02, 9.709358710510e+02, 9.718094576470e+02, 9.726774654190e+02, 9.735399752090e+02, 9.743970660160e+02, 9.752488150590e+02, 9.760952978270e+02, 9.769365881370e+02, 9.777727581820e+02, 9.786038785770e+02, 9.794300184120e+02, 9.802512452930e+02, 9.810676253860e+02, 9.818792234590e+02, 9.826861029240e+02, 9.834883258730e+02, 9.842859531170e+02, 9.850790442240e+02, 9.858676575500e+02, 9.866518502730e+02, 9.874316784310e+02, 9.882071969450e+02, 9.889784596550e+02, 9.897455193470e+02, 9.905084277830e+02, 9.912672357240e+02, 9.920219929610e+02, 9.927727483380e+02, 9.935195497770e+02, 9.942624443020e+02, 9.950014780610e+02, 9.957366963490e+02, 9.964681436310e+02, 9.971958635600e+02, 9.979198989990e+02, 9.986402920390e+02, 9.993570840220e+02, 1.000070315550e+03, 1.000780026520e+03, 1.001486256120e+03, 1.002189042860e+03, 1.002888424590e+03, 1.003584438490e+03, 1.004277121140e+03, 1.004966508450e+03, 1.005652635780e+03, 1.006335537870e+03, 1.007015248860e+03, 1.007691802380e+03, 1.008365231450e+03, 1.009035568590e+03, 1.009702845780e+03, 1.010367094470e+03, 1.011028345630e+03, 1.011686629710e+03, 1.012341976710e+03, 1.012994416120e+03, 1.013643976990e+03, 1.014290687910e+03, 1.014934577040e+03, 1.015575672090e+03, 1.016214000350e+03, 1.016849588700e+03, 1.017482463600e+03, 1.018112651130e+03, 1.018740176970e+03, 1.019365066410e+03, 1.019987344380e+03, 1.020607035430e+03, 1.021224163770e+03, 1.021838753220e+03, 1.022450827300e+03, 1.023060409160e+03, 1.023667521620e+03, 1.024272187190e+03, 1.024874428060e+03, 1.025474266080e+03, 1.026071722830e+03, 1.026666819560e+03, 1.027259577230e+03, 1.027850016540e+03, 1.028438157860e+03, 1.029024021300e+03, 1.029607626710e+03, 1.030188993660e+03, 1.030768141450e+03, 1.031345089120e+03, 1.031919855470e+03, 1.032492459050e+03, 1.033062918150e+03, 1.033631250820e+03, 1.034197474890e+03, 1.034761607940e+03, 1.035323667340e+03, 1.035883670220e+03, 1.036441633490e+03, 1.036997573860e+03, 1.037551507820e+03, 1.038103451640e+03, 1.038653421390e+03, 1.039201432950e+03, 1.039747501990e+03, 1.040291643980e+03, 1.040833874220e+03, 1.041374207810e+03, 1.041912659640e+03, 1.042449244470e+03, 1.042983976830e+03, 1.043516871110e+03, 1.044047941500e+03, 1.044577202040e+03, 1.045104666600e+03, 1.045630348870e+03, 1.046154262380e+03, 1.046676420530e+03, 1.047196836520e+03, 1.047715523430e+03, 1.048232494160e+03, 1.048747761490e+03, 1.049261338020e+03, 1.049773236240e+03, 1.050283468470e+03, 1.050792046900e+03, 1.051298983580e+03, 1.051804290420e+03, 1.052307979200e+03, 1.052810061580e+03, 1.053310549070e+03, 1.053809453050e+03, 1.054306784800e+03, 1.054802555450e+03, 1.055296776020e+03, 1.055789457410e+03, 1.056280610390e+03, 1.056770245620e+03, 1.057258373660e+03, 1.057745004920e+03, 1.058230149730e+03, 1.058713818310e+03, 1.059196020740e+03, 1.059676767030e+03, 1.060156067070e+03, 1.060633930630e+03, 1.061110367400e+03, 1.061585386950e+03, 1.062058998780e+03, 1.062531212250e+03, 1.063002036670e+03, 1.063471481210e+03, 1.063939554970e+03, 1.064406266960e+03, 1.064871626090e+03, 1.065335641170e+03, 1.065798320940e+03, 1.066259674040e+03, 1.066719709040e+03, 1.067178434400e+03, 1.067635858500e+03, 1.068091989660e+03, 1.068546836090e+03, 1.069000405950e+03, 1.069452707270e+03, 1.069903748060e+03, 1.070353536220e+03, 1.070802079560e+03, 1.071249385850e+03, 1.071695462770e+03, 1.072140317910e+03, 1.072583958800e+03, 1.073026392910e+03, 1.073467627630e+03, 1.073907670260e+03, 1.074346528070e+03, 1.074784208230e+03, 1.075220717850e+03, 1.075656063980e+03, 1.076090253600e+03, 1.076523293630e+03, 1.076955190920e+03, 1.077385952260e+03, 1.077815584380e+03, 1.078244093940e+03, 1.078671487540e+03, 1.079097771730e+03, 1.079522952990e+03, 1.079947037760e+03, 1.080370032390e+03, 1.080791943200e+03, 1.081212776440e+03, 1.081632538310e+03, 1.082051234950e+03, 1.082468872460e+03, 1.082885456860e+03, 1.083300994150e+03, 1.083715490240e+03, 1.084128951020e+03, 1.084541382300e+03, 1.084952789880e+03, 1.085363179470e+03, 1.085772556740e+03, 1.086180927330e+03, 1.086588296800e+03, 1.086994670700e+03, 1.087400054510e+03, 1.087804453650e+03, 1.088207873530e+03, 1.088610319490e+03, 1.089011796820e+03, 1.089412310780e+03, 1.089811866580e+03, 1.090210469400e+03, 1.090608124340e+03, 1.091004836500e+03, 1.091400610910e+03, 1.091795452580e+03, 1.092189366450e+03, 1.092582357450e+03, 1.092974430450e+03, 1.093365590280e+03, 1.093755841750e+03, 1.094145189600e+03, 1.094533638560e+03, 1.094921193310e+03, 1.095307858500e+03, 1.095693638720e+03, 1.096078538550e+03, 1.096462562510e+03, 1.096845715110e+03, 1.097228000810e+03, 1.097609424030e+03, 1.097989989170e+03, 1.098369700570e+03, 1.098748562560e+03, 1.099126579430e+03, 1.099503755430e+03, 1.099880094780e+03, 1.100255601680e+03, 1.100630280280e+03, 1.101004134700e+03, 1.101377169030e+03, 1.101749387340e+03, 1.102120793660e+03, 1.102491391980e+03, 1.102861186280e+03, 1.103230180490e+03, 1.103598378520e+03, 1.103965784250e+03, 1.104332401530e+03, 1.104698234180e+03, 1.105063286000e+03, 1.105427560740e+03, 1.105791062150e+03, 1.106153793940e+03, 1.106515759770e+03, 1.106876963320e+03, 1.107237408190e+03, 1.107597098010e+03, 1.107956036330e+03, 1.108314226700e+03, 1.108671672660e+03, 1.109028377680e+03, 1.109384345250e+03, 1.109739578810e+03, 1.110094081780e+03, 1.110447857560e+03, 1.110800909510e+03, 1.111153240990e+03, 1.111504855320e+03, 1.111855755800e+03, 1.112205945700e+03, 1.112555428280e+03, 1.112904206770e+03, 1.113252284370e+03, 1.113599664280e+03, 1.113946349650e+03, 1.114292343620e+03, 1.114637649310e+03, 1.114982269830e+03, 1.115326208230e+03, 1.115669467590e+03, 1.116012050920e+03, 1.116353961250e+03, 1.116695201550e+03, 1.117035774820e+03, 1.117375683980e+03, 1.117714931980e+03, 1.118053521720e+03, 1.118391456090e+03, 1.118728737970e+03, 1.119065370210e+03, 1.119401355630e+03, 1.119736697050e+03, 1.120071397270e+03, 1.120405459050e+03, 1.120738885160e+03, 1.121071678340e+03, 1.121403841300e+03, 1.121735376740e+03, 1.122066287360e+03, 1.122396575810e+03, 1.122726244750e+03, 1.123055296810e+03, 1.123383734590e+03, 1.123711560710e+03, 1.124038777730e+03, 1.124365388220e+03, 1.124691394740e+03, 1.125016799800e+03, 1.125341605920e+03, 1.125665815600e+03, 1.125989431320e+03, 1.126312455540e+03, 1.126634890720e+03, 1.126956739300e+03, 1.127278003670e+03, 1.127598686260e+03, 1.127918789450e+03, 1.128238315610e+03, 1.128557267100e+03, 1.128875646250e+03, 1.129193455410e+03, 1.129510696880e+03, 1.129827372960e+03, 1.130143485940e+03, 1.130459038080e+03, 1.130774031650e+03, 1.131088468880e+03, 1.131402352000e+03, 1.131715683220e+03, 1.132028464760e+03, 1.132340698790e+03, 1.132652387480e+03, 1.132963533010e+03, 1.133274137510e+03, 1.133584203120e+03, 1.133893731960e+03, 1.134202726130e+03, 1.134511187750e+03, 1.134819118870e+03, 1.135126521590e+03, 1.135433397950e+03, 1.135739749990e+03, 1.136045579760e+03, 1.136350889280e+03, 1.136655680540e+03, 1.136959955550e+03, 1.137263716300e+03, 1.137566964750e+03, 1.137869702870e+03, 1.138171932600e+03, 1.138473655900e+03, 1.138774874670e+03, 1.139075590840e+03, 1.139375806310e+03, 1.139675522980e+03},
    {1.796631416300e+00, 5.487329641060e+00, 9.258426781060e+00, 1.311452890490e+01, 1.706068144730e+01, 2.110242857800e+01, 2.524588318640e+01, 2.949780989250e+01, 3.386572415940e+01, 3.835801148670e+01, 4.298407186910e+01, 4.775449636820e+01, 5.268128494190e+01, 5.777811791630e+01, 6.306069812080e+01, 6.854718746020e+01, 7.425877174290e+01, 8.022040284910e+01, 8.646179111070e+01, 9.301875890750e+01, 9.993512958480e+01, 1.072654341430e+02, 1.150789123270e+02, 1.234656507480e+02, 1.325464337910e+02, 1.424894647070e+02, 1.535408622120e+02, 1.660858790820e+02, 1.807896779180e+02, 1.989976276690e+02, 7.401718672430e+02, 7.500793900290e+02, 7.585205234370e+02, 7.659396802450e+02, 7.725954545490e+02, 7.786542979440e+02, 7.842308276870e+02, 7.894079019330e+02, 7.942476728450e+02, 7.987981367810e+02, 8.030972417460e+02, 8.071755815560e+02, 8.110582290890e+02, 8.147660225510e+02, 8.183164917190e+02, 8.217245399840e+02, 8.250029564550e+02, 8.281628070640e+02, 8.312137378290e+02, 8.341642131770e+02, 8.370217055290e+02, 8.397928477590e+02, 8.424835570250e+02, 8.450991362520e+02, 8.476443579960e+02, 8.501235342790e+02, 8.525405751320e+02, 8.548990380150e+02, 8.572021697680e+02, 8.594529424220e+02, 8.616540839450e+02, 8.638081047380e+02, 8.659173206050e+02, 8.679838727290e+02, 8.700097451170e+02, 8.719967799050e+02, 8.739466908180e+02, 8.758610750590e+02, 8.777414238320e+02, 8.795891317040e+02, 8.814055049290e+02, 8.831917689020e+02, 8.849490748210e+02, 8.866785056840e+02, 8.883810816720e+02, 8.900577650150e+02, 8.917094643940e+02, 8.933370389180e+02, 8.949413017410e+02, 8.965230233570e+02, 8.980829345920e+02, 8.996217293430e+02, 9.011400670780e+02, 9.026385751270e+02, 9.041178507860e+02, 9.055784632500e+02, 9.070209553820e+02, 9.084458453580e+02, 9.098536281750e+02, 9.112447770450e+02, 9.126197446860e+02, 9.139789645200e+02, 9.153228517810e+02, 9.166518045400e+02, 9.179662046690e+02, 9.192664187250e+02, 9.205527987820e+02, 9.218256832050e+02, 9.230853973670e+02, 9.243322543290e+02, 9.255665554670e+02, 9.267885910630e+02, 9.279986408580e+02, 9.291969745680e+02, 9.303838523740e+02, 9.315595253790e+02, 9.327242360330e+02, 9.338782185410e+02, 9.350216992440e+02, 9.361548969730e+02, 9.372780233930e+02, 9.383912833170e+02, 9.394948750070e+02, 9.405889904650e+02, 9.416738156920e+02, 9.427495309510e+02, 9.438163110050e+02, 9.448743253410e+02, 9.459237383950e+02, 9.469647097480e+02, 9.479973943290e+02, 9.490219425940e+02, 9.500385007010e+02, 9.510472106850e+02, 9.520482106060e+02, 9.530416347080e+02, 9.540276135610e+02, 9.550062741950e+02, 9.559777402340e+02, 9.569421320200e+02, 9.578995667300e+02, 9.588501584910e+02, 9.597940184890e+02, 9.607312550700e+02, 9.616619738390e+02, 9.625862777580e+02, 9.635042672330e+02, 9.644160402010e+02, 9.653216922130e+02, 9.662213165140e+02, 9.671150041190e+02, 9.680028438830e+02, 9.688849225760e+02, 9.697613249450e+02, 9.706321337810e+02, 9.714974299800e+02, 9.723572926020e+02, 9.732117989260e+02, 9.740610245100e+02, 9.749050432380e+02, 9.757439273730e+02, 9.765777476050e+02, 9.774065730990e+02, 9.782304715360e+02, 9.790495091630e+02, 9.798637508270e+02, 9.806732600230e+02, 9.814780989260e+02, 9.822783284320e+02, 9.830740081920e+02, 9.838651966500e+02, 9.846519510710e+02, 9.854343275790e+02, 9.862123811840e+02, 9.869861658150e+02, 9.877557343480e+02, 9.885211386320e+02, 9.892824295200e+02, 9.900396568920e+02, 9.907928696820e+02, 9.915421159020e+02, 9.922874426660e+02, 9.930288962120e+02, 9.937665219260e+02, 9.945003643620e+02, 9.952304672630e+02, 9.959568735810e+02, 9.966796254970e+02, 9.973987644410e+02, 9.981143311070e+02, 9.988263654740e+02, 9.995349068230e+02, 1.000239993750e+03, 1.000941664190e+03, 1.001639955410e+03, 1.002334904080e+03, 1.003026546200e+03, 1.003714917200e+03, 1.004400051910e+03, 1.005081984570e+03, 1.005760748870e+03, 1.006436377940e+03, 1.007108904360e+03, 1.007778360190e+03, 1.008444776980e+03, 1.009108185770e+03, 1.009768617100e+03, 1.010426101030e+03, 1.011080667150e+03, 1.011732344570e+03, 1.012381161980e+03, 1.013027147590e+03, 1.013670329210e+03, 1.014310734200e+03, 1.014948389500e+03, 1.015583321670e+03, 1.016215556850e+03, 1.016845120790e+03, 1.017472038870e+03, 1.018096336070e+03, 1.018718037030e+03, 1.019337166010e+03, 1.019953746930e+03, 1.020567803350e+03, 1.021179358510e+03, 1.021788435300e+03, 1.022395056280e+03, 1.022999243710e+03, 1.023601019530e+03, 1.024200405360e+03, 1.024797422530e+03, 1.025392092070e+03, 1.025984434720e+03, 1.026574470940e+03, 1.027162220890e+03, 1.027747704490e+03, 1.028330941360e+03, 1.028911950870e+03, 1.029490752130e+03, 1.030067363980e+03, 1.030641805040e+03, 1.031214093660e+03, 1.031784247960e+03, 1.032352285810e+03, 1.032918224850e+03, 1.033482082520e+03, 1.034043875990e+03, 1.034603622250e+03, 1.035161338040e+03, 1.035717039910e+03, 1.036270744190e+03, 1.036822467010e+03, 1.037372224300e+03, 1.037920031770e+03, 1.038465904970e+03, 1.039009859220e+03, 1.039551909680e+03, 1.040092071320e+03, 1.040630358910e+03, 1.041166787060e+03, 1.041701370190e+03, 1.042234122550e+03, 1.042765058230e+03, 1.043294191130e+03, 1.043821535010e+03, 1.044347103450e+03, 1.044870909870e+03, 1.045392967540e+03, 1.045913289570e+03, 1.046431888930e+03, 1.046948778420e+03, 1.047463970700e+03, 1.047977478300e+03, 1.048489313580e+03, 1.048999488770e+03, 1.049508015970e+03, 1.050014907140e+03, 1.050520174100e+03, 1.051023828530e+03, 1.051525882010e+03, 1.052026345950e+03, 1.052525231670e+03, 1.053022550330e+03, 1.053518313010e+03, 1.054012530640e+03, 1.054505214030e+03, 1.054996373880e+03, 1.055486020780e+03, 1.055974165190e+03, 1.056460817480e+03, 1.056945987890e+03, 1.057429686570e+03, 1.057911923530e+03, 1.058392708700e+03, 1.058872051920e+03, 1.059349962880e+03, 1.059826451220e+03, 1.060301526430e+03, 1.060775197960e+03, 1.061247475100e+03, 1.061718367090e+03, 1.062187883050e+03, 1.062656032030e+03, 1.063122822960e+03, 1.063588264710e+03, 1.064052366030e+03, 1.064515135600e+03, 1.064976582020e+03, 1.065436713780e+03, 1.065895539300e+03, 1.066353066930e+03, 1.066809304910e+03, 1.067264261410e+03, 1.067717944540e+03, 1.068170362290e+03, 1.068621522610e+03, 1.069071433350e+03, 1.069520102290e+03, 1.069967537140e+03, 1.070413745540e+03, 1.070858735040e+03, 1.071302513130e+03, 1.071745087230e+03, 1.072186464690e+03, 1.072626652770e+03, 1.073065658690e+03, 1.073503489580e+03, 1.073940152530e+03, 1.074375654550e+03, 1.074810002560e+03, 1.075243203460e+03, 1.075675264050e+03, 1.076106191090e+03, 1.076535991280e+03, 1.076964671230e+03, 1.077392237520e+03, 1.077818696670e+03, 1.078244055110e+03, 1.078668319250e+03, 1.079091495410e+03, 1.079513589880e+03, 1.079934608880e+03, 1.080354558580e+03, 1.080773445080e+03, 1.081191274460e+03, 1.081608052700e+03, 1.082023785770e+03, 1.082438479570e+03, 1.082852139940e+03, 1.083264772670e+03, 1.083676383530e+03, 1.084086978210e+03, 1.084496562350e+03, 1.084905141550e+03, 1.085312721370e+03, 1.085719307320e+03, 1.086124904840e+03, 1.086529519360e+03, 1.086933156230e+03, 1.087335820780e+03, 1.087737518280e+03, 1.088138253960e+03, 1.088538033020e+03, 1.088936860600e+03, 1.089334741790e+03, 1.089731681660e+03, 1.090127685220e+03, 1.090522757450e+03, 1.090916903290e+03, 1.091310127630e+03, 1.091702435330e+03, 1.092093831190e+03, 1.092484320010e+03, 1.092873906500e+03, 1.093262595390e+03, 1.093650391310e+03, 1.094037298910e+03, 1.094423322770e+03, 1.094808467430e+03, 1.095192737410e+03, 1.095576137190e+03, 1.095958671220e+03, 1.096340343900e+03, 1.096721159600e+03, 1.097101122660e+03, 1.097480237390e+03, 1.097858508060e+03, 1.098235938910e+03, 1.098612534150e+03, 1.098988297930e+03, 1.099363234420e+03, 1.099737347710e+03, 1.100110641880e+03, 1.100483120980e+03, 1.100854789020e+03, 1.101225649990e+03, 1.101595707840e+03, 1.101964966500e+03, 1.102333429860e+03, 1.102701101790e+03, 1.103067986120e+03, 1.103434086660e+03, 1.103799407190e+03, 1.104163951460e+03, 1.104527723190e+03, 1.104890726080e+03, 1.105252963810e+03, 1.105614440000e+03, 1.105975158270e+03, 1.106335122220e+03, 1.106694335400e+03, 1.107052801340e+03, 1.107410523570e+03, 1.107767505560e+03, 1.108123750780e+03, 1.108479262650e+03, 1.108834044590e+03, 1.109188099980e+03, 1.109541432180e+03, 1.109894044530e+03, 1.110245940350e+03, 1.110597122920e+03, 1.110947595500e+03, 1.111297361350e+03, 1.111646423680e+03, 1.111994785690e+03, 1.112342450560e+03, 1.112689421430e+03, 1.113035701440e+03, 1.113381293700e+03, 1.113726201300e+03, 1.114070427290e+03, 1.114413974730e+03, 1.114756846640e+03, 1.115099046030e+03, 1.115440575870e+03, 1.115781439120e+03, 1.116121638740e+03, 1.116461177630e+03, 1.116800058710e+03, 1.117138284850e+03, 1.117475858910e+03, 1.117812783750e+03, 1.118149062180e+03, 1.118484697010e+03, 1.118819691030e+03, 1.119154047000e+03, 1.119487767670e+03, 1.119820855780e+03, 1.120153314030e+03, 1.120485145130e+03, 1.120816351740e+03, 1.121146936530e+03, 1.121476902140e+03, 1.121806251200e+03, 1.122134986310e+03, 1.122463110060e+03, 1.122790625020e+03, 1.123117533760e+03, 1.123443838810e+03, 1.123769542700e+03, 1.124094647930e+03, 1.124419157000e+03, 1.124743072370e+03, 1.125066396520e+03, 1.125389131880e+03, 1.125711280880e+03, 1.126032845940e+03, 1.126353829450e+03, 1.126674233790e+03, 1.126994061330e+03, 1.127313314430e+03, 1.127631995410e+03, 1.127950106610e+03, 1.128267650330e+03, 1.128584628870e+03, 1.128901044490e+03, 1.129216899480e+03, 1.129532196070e+03, 1.129846936520e+03, 1.130161123030e+03, 1.130474757820e+03, 1.130787843090e+03, 1.131100381010e+03, 1.131412373770e+03, 1.131723823500e+03, 1.132034732360e+03, 1.132345102470e+03, 1.132654935960e+03, 1.132964234920e+03, 1.133273001440e+03, 1.133581237610e+03, 1.133888945490e+03, 1.134196127130e+03, 1.134502784580e+03, 1.134808919860e+03, 1.135114534990e+03, 1.135419631970e+03, 1.135724212800e+03, 1.136028279460e+03, 1.136331833920e+03, 1.136634878130e+03, 1.136937414040e+03, 1.137239443590e+03, 1.137540968700e+03},
    {1.790357755490e+00, 5.467499002350e+00, 9.223771178420e+00, 1.306364023460e+01, 1.699199112670e+01, 2.101418340710e+01, 2.513611656430e+01, 2.936430727550e+01, 3.370598132830e+01, 3.816918375030e+01, 4.276291174130e+01, 4.749727643250e+01, 5.238370146770e+01, 5.743516915660e+01, 6.266652885440e+01, 6.809488785410e+01, 7.374011336840e+01, 7.962548661180e+01, 8.577856911480e+01, 9.223237156420e+01, 9.902696448640e+01, 1.062117525240e+02, 1.138487782180e+02, 1.220176851660e+02, 1.308234808720e+02, 1.404092940220e+02, 1.509786849580e+02, 1.628379531890e+02, 1.764857371280e+02, 1.928361464560e+02, 2.139464761770e+02, 7.302618888780e+02, 7.410914186400e+02, 7.501806099560e+02, 7.580917243300e+02, 7.651397761480e+02, 7.715223334300e+02, 7.773728437730e+02, 7.827863473030e+02, 7.878332886230e+02, 7.925675414700e+02, 7.970313590240e+02, 8.012585763000e+02, 8.052767635850e+02, 8.091087219480e+02, 8.127735505610e+02, 8.162874265030e+02, 8.196641862060e+02, 8.229157668200e+02, 8.260525465630e+02, 8.290836109030e+02, 8.320169633620e+02, 8.348596943870e+02, 8.376181180070e+02, 8.402978834780e+02, 8.429040672730e+02, 8.454412494640e+02, 8.479135776180e+02, 8.503248205940e+02, 8.526784141250e+02, 8.549774996580e+02, 8.572249576220e+02, 8.594234360740e+02, 8.615753754750e+02, 8.636830302160e+02, 8.657484873850e+02, 8.677736832180e+02, 8.697604175390e+02, 8.717103665070e+02, 8.736250938890e+02, 8.755060610660e+02, 8.773546359400e+02, 8.791721008790e+02, 8.809596598390e+02, 8.827184447380e+02, 8.844495212050e+02, 8.861538937480e+02, 8.878325104330e+02, 8.894862671160e+02, 8.911160112820e+02, 8.927225455330e+02, 8.943066307710e+02, 8.958689890900e+02, 8.974103064320e+02, 8.989312350080e+02, 9.004323955190e+02, 9.019143792030e+02, 9.033777497070e+02, 9.048230448170e+02, 9.062507780530e+02, 9.076614401380e+02, 9.090555003620e+02, 9.104334078380e+02, 9.117955926740e+02, 9.131424670520e+02, 9.144744262360e+02, 9.157918495120e+02, 9.170951010510e+02, 9.183845307300e+02, 9.196604748850e+02, 9.209232570180e+02, 9.221731884630e+02, 9.234105690000e+02, 9.246356874380e+02, 9.258488221550e+02, 9.270502416120e+02, 9.282402048280e+02, 9.294189618300e+02, 9.305867540780e+02, 9.317438148610e+02, 9.328903696710e+02, 9.340266365600e+02, 9.351528264680e+02, 9.362691435420e+02, 9.373757854310e+02, 9.384729435680e+02, 9.395608034320e+02, 9.406395448050e+02, 9.417093420060e+02, 9.427703641160e+02, 9.438227751960e+02, 9.448667344840e+02, 9.459023965950e+02, 9.469299116950e+02, 9.479494256840e+02, 9.489610803520e+02, 9.499650135460e+02, 9.509613593100e+02, 9.519502480360e+02, 9.529318065930e+02, 9.539061584610e+02, 9.548734238510e+02, 9.558337198270e+02, 9.567871604140e+02, 9.577338567070e+02, 9.586739169770e+02, 9.596074467620e+02, 9.605345489660e+02, 9.614553239500e+02, 9.623698696110e+02, 9.632782814710e+02, 9.641806527540e+02, 9.650770744580e+02, 9.659676354320e+02, 9.668524224430e+02, 9.677315202420e+02, 9.686050116290e+02, 9.694729775160e+02, 9.703354969800e+02, 9.711926473250e+02, 9.720445041380e+02, 9.728911413320e+02, 9.737326312070e+02, 9.745690444910e+02, 9.754004503900e+02, 9.762269166310e+02, 9.770485095060e+02, 9.778652939150e+02, 9.786773334010e+02, 9.794846901940e+02, 9.802874252460e+02, 9.810855982670e+02, 9.818792677570e+02, 9.826684910460e+02, 9.834533243170e+02, 9.842338226450e+02, 9.850100400230e+02, 9.857820293920e+02, 9.865498426680e+02, 9.873135307720e+02, 9.880731436510e+02, 9.888287303090e+02, 9.895803388260e+02, 9.903280163880e+02, 9.910718093020e+02, 9.918117630260e+02, 9.925479221840e+02, 9.932803305900e+02, 9.940090312700e+02, 9.947340664750e+02, 9.954554777080e+02, 9.961733057350e+02, 9.968875906070e+02, 9.975983716760e+02, 9.983056876120e+02, 9.990095764180e+02, 9.997100754470e+02, 1.000407221420e+03, 1.001101050420e+03, 1.001791597960e+03, 1.002478898920e+03, 1.003162987620e+03, 1.003843897820e+03, 1.004521662720e+03, 1.005196314980e+03, 1.005867886720e+03, 1.006536409570e+03, 1.007201914620e+03, 1.007864432490e+03, 1.008523993320e+03, 1.009180626750e+03, 1.009834361970e+03, 1.010485227740e+03, 1.011133252330e+03, 1.011778463610e+03, 1.012420889010e+03, 1.013060555560e+03, 1.013697489860e+03, 1.014331718110e+03, 1.014963266150e+03, 1.015592159400e+03, 1.016218422920e+03, 1.016842081410e+03, 1.017463159190e+03, 1.018081680240e+03, 1.018697668200e+03, 1.019311146340e+03, 1.019922137630e+03, 1.020530664700e+03, 1.021136749860e+03, 1.021740415100e+03, 1.022341682110e+03, 1.022940572280e+03, 1.023537106700e+03, 1.024131306160e+03, 1.024723191170e+03, 1.025312781980e+03, 1.025900098530e+03, 1.026485160510e+03, 1.027067987350e+03, 1.027648598210e+03, 1.028227012000e+03, 1.028803247370e+03, 1.029377322730e+03, 1.029949256260e+03, 1.030519065880e+03, 1.031086769290e+03, 1.031652383970e+03, 1.032215927150e+03, 1.032777415860e+03, 1.033336866910e+03, 1.033894296890e+03, 1.034449722170e+03, 1.035003158950e+03, 1.035554623180e+03, 1.036104130650e+03, 1.036651696930e+03, 1.037197337420e+03, 1.037741067300e+03, 1.038282901600e+03, 1.038822855130e+03, 1.039360942540e+03, 1.039897178310e+03, 1.040431576740e+03, 1.040964151940e+03, 1.041494917880e+03, 1.042023888350e+03, 1.042551076980e+03, 1.043076497230e+03, 1.043600162410e+03, 1.044122085680e+03, 1.044642280040e+03, 1.045160758350e+03, 1.045677533300e+03, 1.046192617460e+03, 1.046706023230e+03, 1.047217762890e+03, 1.047727848580e+03, 1.048236292280e+03, 1.048743105870e+03, 1.049248301070e+03, 1.049751889480e+03, 1.050253882570e+03, 1.050754291670e+03, 1.051253128020e+03, 1.051750402700e+03, 1.052246126680e+03, 1.052740310830e+03, 1.053232965870e+03, 1.053724102420e+03, 1.054213731000e+03, 1.054701861990e+03, 1.055188505680e+03, 1.055673672240e+03, 1.056157371740e+03, 1.056639614130e+03, 1.057120409270e+03, 1.057599766900e+03, 1.058077696690e+03, 1.058554208170e+03, 1.059029310810e+03, 1.059503013940e+03, 1.059975326830e+03, 1.060446258640e+03, 1.060915818430e+03, 1.061384015190e+03, 1.061850857800e+03, 1.062316355050e+03, 1.062780515660e+03, 1.063243348230e+03, 1.063704861310e+03, 1.064165063340e+03, 1.064623962690e+03, 1.065081567640e+03, 1.065537886400e+03, 1.065992927080e+03, 1.066446697720e+03, 1.066899206290e+03, 1.067350460680e+03, 1.067800468690e+03, 1.068249238060e+03, 1.068696776440e+03, 1.069143091430e+03, 1.069588190540e+03, 1.070032081220e+03, 1.070474770820e+03, 1.070916266670e+03, 1.071356575990e+03, 1.071795705950e+03, 1.072233663640e+03, 1.072670456110e+03, 1.073106090320e+03, 1.073540573170e+03, 1.073973911510e+03, 1.074406112110e+03, 1.074837181680e+03, 1.075267126890e+03, 1.075695954320e+03, 1.076123670500e+03, 1.076550281920e+03, 1.076975794990e+03, 1.077400216070e+03, 1.077823551450e+03, 1.078245807390e+03, 1.078666990070e+03, 1.079087105630e+03, 1.079506160150e+03, 1.079924159660e+03, 1.080341110130e+03, 1.080757017490e+03, 1.081171887590e+03, 1.081585726270e+03, 1.081998539300e+03, 1.082410332380e+03, 1.082821111190e+03, 1.083230881360e+03, 1.083639648450e+03, 1.084047417990e+03, 1.084454195460e+03, 1.084859986280e+03, 1.085264795850e+03, 1.085668629510e+03, 1.086071492550e+03, 1.086473390220e+03, 1.086874327720e+03, 1.087274310230e+03, 1.087673342870e+03, 1.088071430700e+03, 1.088468578770e+03, 1.088864792070e+03, 1.089260075560e+03, 1.089654434150e+03, 1.090047872720e+03, 1.090440396080e+03, 1.090832009050e+03, 1.091222716380e+03, 1.091612522780e+03, 1.092001432930e+03, 1.092389451470e+03, 1.092776583010e+03, 1.093162832120e+03, 1.093548203320e+03, 1.093932701120e+03, 1.094316329970e+03, 1.094699094290e+03, 1.095080998490e+03, 1.095462046900e+03, 1.095842243860e+03, 1.096221593650e+03, 1.096600100520e+03, 1.096977768700e+03, 1.097354602380e+03, 1.097730605710e+03, 1.098105782810e+03, 1.098480137790e+03, 1.098853674690e+03, 1.099226397550e+03, 1.099598310380e+03, 1.099969417130e+03, 1.100339721750e+03, 1.100709228150e+03, 1.101077940210e+03, 1.101445861780e+03, 1.101812996680e+03, 1.102179348700e+03, 1.102544921620e+03, 1.102909719160e+03, 1.103273745050e+03, 1.103637002950e+03, 1.103999496530e+03, 1.104361229410e+03, 1.104722205190e+03, 1.105082427460e+03, 1.105441899760e+03, 1.105800625610e+03, 1.106158608510e+03, 1.106515851940e+03, 1.106872359340e+03, 1.107228134130e+03, 1.107583179720e+03, 1.107937499470e+03, 1.108291096730e+03, 1.108643974840e+03, 1.108996137100e+03, 1.109347586770e+03, 1.109698327120e+03, 1.110048361390e+03, 1.110397692770e+03, 1.110746324470e+03, 1.111094259630e+03, 1.111441501420e+03, 1.111788052940e+03, 1.112133917290e+03, 1.112479097560e+03, 1.112823596800e+03, 1.113167418050e+03, 1.113510564310e+03, 1.113853038600e+03, 1.114194843860e+03, 1.114535983070e+03, 1.114876459160e+03, 1.115216275030e+03, 1.115555433580e+03, 1.115893937680e+03, 1.116231790200e+03, 1.116568993950e+03, 1.116905551770e+03, 1.117241466440e+03, 1.117576740740e+03, 1.117911377440e+03, 1.118245379270e+03, 1.118578748970e+03, 1.118911489220e+03, 1.119243602730e+03, 1.119575092160e+03, 1.119905960170e+03, 1.120236209380e+03, 1.120565842420e+03, 1.120894861890e+03, 1.121223270360e+03, 1.121551070410e+03, 1.121878264590e+03, 1.122204855420e+03, 1.122530845440e+03, 1.122856237130e+03, 1.123181032980e+03, 1.123505235470e+03, 1.123828847040e+03, 1.124151870130e+03, 1.124474307170e+03, 1.124796160550e+03, 1.125117432680e+03, 1.125438125930e+03, 1.125758242660e+03, 1.126077785210e+03, 1.126396755920e+03, 1.126715157100e+03, 1.127032991060e+03, 1.127350260090e+03, 1.127666966450e+03, 1.127983112410e+03, 1.128298700220e+03, 1.128613732100e+03, 1.128928210280e+03, 1.129242136950e+03, 1.129555514310e+03, 1.129868344540e+03, 1.130180629800e+03, 1.130492372240e+03, 1.130803574000e+03, 1.131114237210e+03, 1.131424363970e+03, 1.131733956380e+03, 1.132043016530e+03, 1.132351546500e+03, 1.132659548350e+03, 1.132967024120e+03, 1.133273975850e+03, 1.133580405570e+03, 1.133886315290e+03, 1.134191707000e+03, 1.134496582710e+03, 1.134800944370e+03, 1.135104793970e+03, 1.135408133450e+03},
    {1.784128538010e+00, 5.447818847520e+00, 9.189397050850e+00, 1.301319369780e+01, 1.692393899160e+01, 2.092681485410e+01, 2.502751594460e+01, 2.923232157770e+01, 3.354818100900e+01, 3.798281524090e+01, 4.254483942050e+01, 4.724391113320e+01, 5.209091159260e+01, 5.709816906500e+01, 6.227973716500e+01, 6.765174536510e+01, 7.323284591440e+01, 7.904479152530e+01, 8.511319360020e+01, 9.146853473350e+01, 9.814754749590e+01, 1.051951346260e+02, 1.126671136200e+02, 1.206342609830e+02, 1.291884913860e+02, 1.384527219460e+02, 1.485974978280e+02, 1.598710200260e+02, 1.726585797160e+02, 1.876162822250e+02, 2.060370226360e+02, 2.312795514010e+02, 7.200889588030e+02, 7.319416942110e+02, 7.417308586730e+02, 7.501643419380e+02, 7.576240795050e+02, 7.643435367900e+02, 7.704774364150e+02, 7.761343561090e+02, 7.813938318830e+02, 7.863161090000e+02, 7.909480638130e+02, 7.953269842050e+02, 7.994830829670e+02, 8.034412261100e+02, 8.072221557480e+02, 8.108433767730e+02, 8.143198135860e+02, 8.176643056300e+02, 8.208879874960e+02, 8.240005847860e+02, 8.270106474370e+02, 8.299257359020e+02, 8.327525713180e+02, 8.354971577850e+02, 8.381648828450e+02, 8.407606006950e+02, 8.432887016320e+02, 8.457531703980e+02, 8.481576355130e+02, 8.505054112380e+02, 8.527995334570e+02, 8.550427905260e+02, 8.572377499090e+02, 8.593867812950e+02, 8.614920767300e+02, 8.635556682250e+02, 8.655794432200e+02, 8.675651582000e+02, 8.695144507320e+02, 8.714288501410e+02, 8.733097869950e+02, 8.751586015780e+02, 8.769765514480e+02, 8.787648182330e+02, 8.805245137210e+02, 8.822566853570e+02, 8.839623212000e+02, 8.856423544070e+02, 8.872976673060e+02, 8.889290950850e+02, 8.905374291560e+02, 8.921234202190e+02, 8.936877810630e+02, 8.952311891250e+02, 8.967542888370e+02, 8.982576937790e+02, 8.997419886580e+02, 9.012077311300e+02, 9.026554534770e+02, 9.040856641600e+02, 9.054988492460e+02, 9.068954737370e+02, 9.082759827950e+02, 9.096408028810e+02, 9.109903428150e+02, 9.123249947520e+02, 9.136451351050e+02, 9.149511253900e+02, 9.162433130250e+02, 9.175220320710e+02, 9.187876039230e+02, 9.200403379600e+02, 9.212805321520e+02, 9.225084736280e+02, 9.237244392090e+02, 9.249286959090e+02, 9.261215014050e+02, 9.273031044790e+02, 9.284737454320e+02, 9.296336564790e+02, 9.307830621130e+02, 9.319221794570e+02, 9.330512185920e+02, 9.341703828620e+02, 9.352798691740e+02, 9.363798682690e+02, 9.374705649850e+02, 9.385521385090e+02, 9.396247626050e+02, 9.406886058440e+02, 9.417438318080e+02, 9.427905992980e+02, 9.438290625150e+02, 9.448593712520e+02, 9.458816710540e+02, 9.468961033910e+02, 9.479028058080e+02, 9.489019120760e+02, 9.498935523300e+02, 9.508778532060e+02, 9.518549379690e+02, 9.528249266320e+02, 9.537879360790e+02, 9.547440801690e+02, 9.556934698470e+02, 9.566362132460e+02, 9.575724157810e+02, 9.585021802440e+02, 9.594256068930e+02, 9.603427935370e+02, 9.612538356180e+02, 9.621588262870e+02, 9.630578564800e+02, 9.639510149930e+02, 9.648383885460e+02, 9.657200618510e+02, 9.665961176760e+02, 9.674666369050e+02, 9.683316985960e+02, 9.691913800400e+02, 9.700457568110e+02, 9.708949028220e+02, 9.717388903700e+02, 9.725777901880e+02, 9.734116714910e+02, 9.742406020180e+02, 9.750646480780e+02, 9.758838745880e+02, 9.766983451160e+02, 9.775081219150e+02, 9.783132659670e+02, 9.791138370100e+02, 9.799098935790e+02, 9.807014930360e+02, 9.814886916050e+02, 9.822715443980e+02, 9.830501054490e+02, 9.838244277410e+02, 9.845945632360e+02, 9.853605628970e+02, 9.861224767210e+02, 9.868803537590e+02, 9.876342421400e+02, 9.883841890990e+02, 9.891302409960e+02, 9.898724433400e+02, 9.906108408070e+02, 9.913454772680e+02, 9.920763957990e+02, 9.928036387090e+02, 9.935272475540e+02, 9.942472631570e+02, 9.949637256250e+02, 9.956766743680e+02, 9.963861481110e+02, 9.970921849160e+02, 9.977948221930e+02, 9.984940967170e+02, 9.991900446430e+02, 9.998827015190e+02, 1.000572102300e+03, 1.001258281370e+03, 1.001941272530e+03, 1.002621109050e+03, 1.003297823630e+03, 1.003971448480e+03, 1.004642015260e+03, 1.005309555140e+03, 1.005974098790e+03, 1.006635676390e+03, 1.007294317660e+03, 1.007950051850e+03, 1.008602907750e+03, 1.009252913720e+03, 1.009900097670e+03, 1.010544487090e+03, 1.011186109070e+03, 1.011824990260e+03, 1.012461156920e+03, 1.013094634950e+03, 1.013725449820e+03, 1.014353626640e+03, 1.014979190180e+03, 1.015602164810e+03, 1.016222574560e+03, 1.016840443110e+03, 1.017455793820e+03, 1.018068649690e+03, 1.018679033400e+03, 1.019286967330e+03, 1.019892473510e+03, 1.020495573690e+03, 1.021096289300e+03, 1.021694641490e+03, 1.022290651100e+03, 1.022884338710e+03, 1.023475724580e+03, 1.024064828730e+03, 1.024651670910e+03, 1.025236270570e+03, 1.025818646930e+03, 1.026398818950e+03, 1.026976805340e+03, 1.027552624550e+03, 1.028126294790e+03, 1.028697834060e+03, 1.029267260090e+03, 1.029834590400e+03, 1.030399842290e+03, 1.030963032810e+03, 1.031524178830e+03, 1.032083296980e+03, 1.032640403680e+03, 1.033195515170e+03, 1.033748647450e+03, 1.034299816360e+03, 1.034849037500e+03, 1.035396326310e+03, 1.035941698030e+03, 1.036485167720e+03, 1.037026750240e+03, 1.037566460290e+03, 1.038104312370e+03, 1.038640320830e+03, 1.039174499830e+03, 1.039706863370e+03, 1.040237425290e+03, 1.040766199250e+03, 1.041293198750e+03, 1.041818437160e+03, 1.042341927660e+03, 1.042863683290e+03, 1.043383716940e+03, 1.043902041360e+03, 1.044418669150e+03, 1.044933612740e+03, 1.045446884460e+03, 1.045958496460e+03, 1.046468460800e+03, 1.046976789360e+03, 1.047483493910e+03, 1.047988586080e+03, 1.048492077380e+03, 1.048993979190e+03, 1.049494302750e+03, 1.049993059210e+03, 1.050490259560e+03, 1.050985914690e+03, 1.051480035370e+03, 1.051972632260e+03, 1.052463715900e+03, 1.052953296710e+03, 1.053441385000e+03, 1.053927990990e+03, 1.054413124770e+03, 1.054896796330e+03, 1.055379015550e+03, 1.055859792220e+03, 1.056339136030e+03, 1.056817056540e+03, 1.057293563240e+03, 1.057768665520e+03, 1.058242372650e+03, 1.058714693840e+03, 1.059185638180e+03, 1.059655214670e+03, 1.060123432230e+03, 1.060590299680e+03, 1.061055825770e+03, 1.061520019130e+03, 1.061982888330e+03, 1.062444441850e+03, 1.062904688080e+03, 1.063363635340e+03, 1.063821291840e+03, 1.064277665730e+03, 1.064732765090e+03, 1.065186597910e+03, 1.065639172090e+03, 1.066090495480e+03, 1.066540575830e+03, 1.066989420830e+03, 1.067437038100e+03, 1.067883435160e+03, 1.068328619500e+03, 1.068772598500e+03, 1.069215379500e+03, 1.069656969760e+03, 1.070097376460e+03, 1.070536606730e+03, 1.070974667620e+03, 1.071411566140e+03, 1.071847309210e+03, 1.072281903680e+03, 1.072715356370e+03, 1.073147674000e+03, 1.073578863270e+03, 1.074008930770e+03, 1.074437883080e+03, 1.074865726680e+03, 1.075292468010e+03, 1.075718113460e+03, 1.076142669350e+03, 1.076566141940e+03, 1.076988537450e+03, 1.077409862030e+03, 1.077830121790e+03, 1.078249322780e+03, 1.078667470980e+03, 1.079084572340e+03, 1.079500632760e+03, 1.079915658070e+03, 1.080329654060e+03, 1.080742626480e+03, 1.081154581000e+03, 1.081565523270e+03, 1.081975458890e+03, 1.082384393400e+03, 1.082792332290e+03, 1.083199281020e+03, 1.083605244990e+03, 1.084010229560e+03, 1.084414240040e+03, 1.084817281710e+03, 1.085219359790e+03, 1.085620479460e+03, 1.086020645860e+03, 1.086419864090e+03, 1.086818139200e+03, 1.087215476210e+03, 1.087611880080e+03, 1.088007355750e+03, 1.088401908100e+03, 1.088795541990e+03, 1.089188262240e+03, 1.089580073600e+03, 1.089970980820e+03, 1.090360988590e+03, 1.090750101570e+03, 1.091138324390e+03, 1.091525661610e+03, 1.091912117800e+03, 1.092297697470e+03, 1.092682405080e+03, 1.093066245090e+03, 1.093449221890e+03, 1.093831339860e+03, 1.094212603330e+03, 1.094593016610e+03, 1.094972583970e+03, 1.095351309650e+03, 1.095729197840e+03, 1.096106252720e+03, 1.096482478430e+03, 1.096857879070e+03, 1.097232458730e+03, 1.097606221440e+03, 1.097979171230e+03, 1.098351312070e+03, 1.098722647920e+03, 1.099093182700e+03, 1.099462920310e+03, 1.099831864610e+03, 1.100200019440e+03, 1.100567388610e+03, 1.100933975890e+03, 1.101299785030e+03, 1.101664819770e+03, 1.102029083790e+03, 1.102392580760e+03, 1.102755314330e+03, 1.103117288110e+03, 1.103478505680e+03, 1.103838970620e+03, 1.104198686450e+03, 1.104557656700e+03, 1.104915884830e+03, 1.105273374320e+03, 1.105630128590e+03, 1.105986151070e+03, 1.106341445130e+03, 1.106696014140e+03, 1.107049861430e+03, 1.107402990330e+03, 1.107755404110e+03, 1.108107106050e+03, 1.108458099400e+03, 1.108808387370e+03, 1.109157973160e+03, 1.109506859950e+03, 1.109855050900e+03, 1.110202549130e+03, 1.110549357770e+03, 1.110895479880e+03, 1.111240918560e+03, 1.111585676830e+03, 1.111929757730e+03, 1.112273164270e+03, 1.112615899420e+03, 1.112957966150e+03, 1.113299367400e+03, 1.113640106100e+03, 1.113980185160e+03, 1.114319607450e+03, 1.114658375840e+03, 1.114996493180e+03, 1.115333962290e+03, 1.115670785980e+03, 1.116006967030e+03, 1.116342508230e+03, 1.116677412320e+03, 1.117011682030e+03, 1.117345320090e+03, 1.117678329170e+03, 1.118010711980e+03, 1.118342471160e+03, 1.118673609370e+03, 1.119004129220e+03, 1.119334033340e+03, 1.119663324310e+03, 1.119992004710e+03, 1.120320077090e+03, 1.120647544010e+03, 1.120974407990e+03, 1.121300671530e+03, 1.121626337140e+03, 1.121951407290e+03, 1.122275884440e+03, 1.122599771040e+03, 1.122923069520e+03, 1.123245782290e+03, 1.123567911760e+03, 1.123889460320e+03, 1.124210430320e+03, 1.124530824130e+03, 1.124850644090e+03, 1.125169892510e+03, 1.125488571720e+03, 1.125806684010e+03, 1.126124231660e+03, 1.126441216940e+03, 1.126757642110e+03, 1.127073509400e+03, 1.127388821040e+03, 1.127703579240e+03, 1.128017786200e+03, 1.128331444120e+03, 1.128644555150e+03, 1.128957121450e+03, 1.129269145190e+03, 1.129580628480e+03, 1.129891573440e+03, 1.130201982190e+03, 1.130511856820e+03, 1.130821199410e+03, 1.131130012020e+03, 1.131438296720e+03, 1.131746055550e+03, 1.132053290530e+03, 1.132360003700e+03, 1.132666197060e+03, 1.132971872600e+03, 1.133277032310e+03},
    {1.777943281070e+00, 5.428287348870e+00, 9.155300613170e+00, 1.296318275870e+01, 1.685651472670e+01, 2.084030748790e+01, 2.492005903970e+01, 2.910182137930e+01, 3.339227966290e+01, 3.779884636150e+01, 4.232977402180e+01, 4.699429130980e+01, 5.180276846670e+01, 5.676692030210e+01, 6.190005763510e+01, 6.721740203390e+01, 7.273648438200e+01, 7.847765612440e+01, 8.446475451830e+01, 9.072598231270e+01, 9.729509231450e+01, 1.042130158930e+02, 1.115301557850e+02, 1.193097048650e+02, 1.276326094210e+02, 1.366052875970e+02, 1.463722187760e+02, 1.571377399320e+02, 1.692068040960e+02, 1.830695472470e+02, 1.996050964550e+02, 2.207064958240e+02, 2.524281320240e+02, 7.096717947330e+02, 7.226493613890e+02, 7.331883058160e+02, 7.421723519880e+02, 7.500612486030e+02, 7.571291673990e+02, 7.635545085810e+02, 7.694606943170e+02, 7.749371090220e+02, 7.800508300470e+02, 7.848536485060e+02, 7.893864960480e+02, 7.936823566090e+02, 7.977682500970e+02, 8.016666245070e+02, 8.053963580630e+02, 8.089734967030e+02, 8.124118073960e+02, 8.157232004160e+02, 8.189180565290e+02, 8.220054839660e+02, 8.249935227320e+02, 8.278893088340e+02, 8.306992076290e+02, 8.334289230770e+02, 8.360835880210e+02, 8.386678393430e+02, 8.411858809870e+02, 8.436415371420e+02, 8.460382974040e+02, 8.483793553350e+02, 8.506676415520e+02, 8.529058522770e+02, 8.550964740640e+02, 8.572418053210e+02, 8.593439751100e+02, 8.614049596360e+02, 8.634265967510e+02, 8.654105987750e+02, 8.673585638400e+02, 8.692719859780e+02, 8.711522641080e+02, 8.730007100660e+02, 8.748185557960e+02, 8.766069598130e+02, 8.783670130180e+02, 8.800997439410e+02, 8.818061234900e+02, 8.834870692450e+02, 8.851434493570e+02, 8.867760861060e+02, 8.883857591230e+02, 8.899732083480e+02, 8.915391367290e+02, 8.930842126900e+02, 8.946090723980e+02, 8.961143218500e+02, 8.976005387820e+02, 8.990682744410e+02, 9.005180552050e+02, 9.019503840950e+02, 9.033657421610e+02, 9.047645897720e+02, 9.061473678100e+02, 9.075144987780e+02, 9.088663878290e+02, 9.102034237300e+02, 9.115259797460e+02, 9.128344144800e+02, 9.141290726430e+02, 9.154102857830e+02, 9.166783729620e+02, 9.179336413890e+02, 9.191763870170e+02, 9.204068950980e+02, 9.216254407060e+02, 9.228322892270e+02, 9.240276968230e+02, 9.252119108630e+02, 9.263851703320e+02, 9.275477062150e+02, 9.286997418640e+02, 9.298414933310e+02, 9.309731697000e+02, 9.320949733880e+02, 9.332071004300e+02, 9.343097407580e+02, 9.354030784550e+02, 9.364872919990e+02, 9.375625544980e+02, 9.386290339040e+02, 9.396868932290e+02, 9.407362907330e+02, 9.417773801210e+02, 9.428103107170e+02, 9.438352276320e+02, 9.448522719300e+02, 9.458615807780e+02, 9.468632875930e+02, 9.478575221850e+02, 9.488444108830e+02, 9.498240766670e+02, 9.507966392900e+02, 9.517622153870e+02, 9.527209185930e+02, 9.536728596400e+02, 9.546181464670e+02, 9.555568843090e+02, 9.564891757910e+02, 9.574151210190e+02, 9.583348176590e+02, 9.592483610230e+02, 9.601558441450e+02, 9.610573578520e+02, 9.619529908400e+02, 9.628428297400e+02, 9.637269591820e+02, 9.646054618610e+02, 9.654784185940e+02, 9.663459083830e+02, 9.672080084640e+02, 9.680647943680e+02, 9.689163399650e+02, 9.697627175210e+02, 9.706039977390e+02, 9.714402498100e+02, 9.722715414550e+02, 9.730979389660e+02, 9.739195072520e+02, 9.747363098710e+02, 9.755484090780e+02, 9.763558658510e+02, 9.771587399340e+02, 9.779570898690e+02, 9.787509730280e+02, 9.795404456460e+02, 9.803255628510e+02, 9.811063786940e+02, 9.818829461790e+02, 9.826553172890e+02, 9.834235430120e+02, 9.841876733690e+02, 9.849477574380e+02, 9.857038433800e+02, 9.864559784590e+02, 9.872042090680e+02, 9.879485807480e+02, 9.886891382110e+02, 9.894259253630e+02, 9.901589853170e+02, 9.908883604210e+02, 9.916140922700e+02, 9.923362217260e+02, 9.930547889380e+02, 9.937698333550e+02, 9.944813937460e+02, 9.951895082160e+02, 9.958942142160e+02, 9.965955485680e+02, 9.972935474690e+02, 9.979882465130e+02, 9.986796807020e+02, 9.993678844590e+02, 1.000052891640e+03, 1.000734735550e+03, 1.001413448960e+03, 1.002089064090e+03, 1.002761612670e+03, 1.003431125920e+03, 1.004097634540e+03, 1.004761168770e+03, 1.005421758380e+03, 1.006079432660e+03, 1.006734220450e+03, 1.007386150160e+03, 1.008035249750e+03, 1.008681546750e+03, 1.009325068290e+03, 1.009965841090e+03, 1.010603891440e+03, 1.011239245290e+03, 1.011871928170e+03, 1.012501965230e+03, 1.013129381270e+03, 1.013754200720e+03, 1.014376447670e+03, 1.014996145840e+03, 1.015613318620e+03, 1.016227989070e+03, 1.016840179920e+03, 1.017449913580e+03, 1.018057212140e+03, 1.018662097390e+03, 1.019264590810e+03, 1.019864713600e+03, 1.020462486630e+03, 1.021057930520e+03, 1.021651065610e+03, 1.022241911930e+03, 1.022830489280e+03, 1.023416817160e+03, 1.024000914840e+03, 1.024582801320e+03, 1.025162495340e+03, 1.025740015400e+03, 1.026315379770e+03, 1.026888606460e+03, 1.027459713270e+03, 1.028028717750e+03, 1.028595637230e+03, 1.029160488830e+03, 1.029723289430e+03, 1.030284055730e+03, 1.030842804180e+03, 1.031399551040e+03, 1.031954312380e+03, 1.032507104060e+03, 1.033057941730e+03, 1.033606840870e+03, 1.034153816750e+03, 1.034698884470e+03, 1.035242058930e+03, 1.035783354870e+03, 1.036322786840e+03, 1.036860369200e+03, 1.037396116170e+03, 1.037930041780e+03, 1.038462159910e+03, 1.038992484240e+03, 1.039521028340e+03, 1.040047805570e+03, 1.040572829190e+03, 1.041096112250e+03, 1.041617667690e+03, 1.042137508280e+03, 1.042655646650e+03, 1.043172095300e+03, 1.043686866550e+03, 1.044199972620e+03, 1.044711425580e+03, 1.045221237350e+03, 1.045729419740e+03, 1.046235984410e+03, 1.046740942890e+03, 1.047244306610e+03, 1.047746086840e+03, 1.048246294740e+03, 1.048744941360e+03, 1.049242037610e+03, 1.049737594300e+03, 1.050231622100e+03, 1.050724131590e+03, 1.051215133230e+03, 1.051704637360e+03, 1.052192654220e+03, 1.052679193930e+03, 1.053164266520e+03, 1.053647881890e+03, 1.054130049870e+03, 1.054610780160e+03, 1.055090082360e+03, 1.055567966000e+03, 1.056044440480e+03, 1.056519515100e+03, 1.056993199110e+03, 1.057465501610e+03, 1.057936431650e+03, 1.058405998160e+03, 1.058874210000e+03, 1.059341075920e+03, 1.059806604610e+03, 1.060270804650e+03, 1.060733684550e+03, 1.061195252720e+03, 1.061655517500e+03, 1.062114487150e+03, 1.062572169830e+03, 1.063028573640e+03, 1.063483706600e+03, 1.063937576640e+03, 1.064390191630e+03, 1.064841559340e+03, 1.065291687500e+03, 1.065740583730e+03, 1.066188255600e+03, 1.066634710590e+03, 1.067079956150e+03, 1.067523999600e+03, 1.067966848240e+03, 1.068408509280e+03, 1.068848989870e+03, 1.069288297080e+03, 1.069726437940e+03, 1.070163419390e+03, 1.070599248310e+03, 1.071033931530e+03, 1.071467475810e+03, 1.071899887850e+03, 1.072331174290e+03, 1.072761341700e+03, 1.073190396590e+03, 1.073618345440e+03, 1.074045194650e+03, 1.074470950550e+03, 1.074895619430e+03, 1.075319207530e+03, 1.075741721020e+03, 1.076163166030e+03, 1.076583548620e+03, 1.077002874800e+03, 1.077421150550e+03, 1.077838381770e+03, 1.078254574320e+03, 1.078669734010e+03, 1.079083866590e+03, 1.079496977780e+03, 1.079909073230e+03, 1.080320158560e+03, 1.080730239320e+03, 1.081139321040e+03, 1.081547409190e+03, 1.081954509180e+03, 1.082360626390e+03, 1.082765766160e+03, 1.083169933770e+03, 1.083573134470e+03, 1.083975373460e+03, 1.084376655880e+03, 1.084776986860e+03, 1.085176371470e+03, 1.085574814730e+03, 1.085972321640e+03, 1.086368897140e+03, 1.086764546140e+03, 1.087159273510e+03, 1.087553084070e+03, 1.087945982610e+03, 1.088337973890e+03, 1.088729062610e+03, 1.089119253450e+03, 1.089508551050e+03, 1.089896960000e+03, 1.090284484870e+03, 1.090671130190e+03, 1.091056900440e+03, 1.091441800090e+03, 1.091825833550e+03, 1.092209005210e+03, 1.092591319420e+03, 1.092972780510e+03, 1.093353392740e+03, 1.093733160380e+03, 1.094112087650e+03, 1.094490178720e+03, 1.094867437760e+03, 1.095243868880e+03, 1.095619476180e+03, 1.095994263710e+03, 1.096368235500e+03, 1.096741395550e+03, 1.097113747830e+03, 1.097485296270e+03, 1.097856044790e+03, 1.098225997250e+03, 1.098595157520e+03, 1.098963529410e+03, 1.099331116710e+03, 1.099697923180e+03, 1.100063952570e+03, 1.100429208580e+03, 1.100793694900e+03, 1.101157415170e+03, 1.101520373030e+03, 1.101882572080e+03, 1.102244015880e+03, 1.102604708000e+03, 1.102964651950e+03, 1.103323851230e+03, 1.103682309310e+03, 1.104040029640e+03, 1.104397015650e+03, 1.104753270720e+03, 1.105108798240e+03, 1.105463601550e+03, 1.105817683990e+03, 1.106171048840e+03, 1.106523699400e+03, 1.106875638920e+03, 1.107226870620e+03, 1.107577397730e+03, 1.107927223430e+03, 1.108276350880e+03, 1.108624783220e+03, 1.108972523590e+03, 1.109319575080e+03, 1.109665940770e+03, 1.110011623710e+03, 1.110356626940e+03, 1.110700953480e+03, 1.111044606330e+03, 1.111387588450e+03, 1.111729902800e+03, 1.112071552320e+03, 1.112412539920e+03, 1.112752868490e+03, 1.113092540910e+03, 1.113431560040e+03, 1.113769928710e+03, 1.114107649730e+03, 1.114444725910e+03, 1.114781160030e+03, 1.115116954850e+03, 1.115452113110e+03, 1.115786637540e+03, 1.116120530840e+03, 1.116453795710e+03, 1.116786434800e+03, 1.117118450790e+03, 1.117449846300e+03, 1.117780623950e+03, 1.118110786360e+03, 1.118440336090e+03, 1.118769275720e+03, 1.119097607810e+03, 1.119425334880e+03, 1.119752459460e+03, 1.120078984050e+03, 1.120404911130e+03, 1.120730243190e+03, 1.121054982670e+03, 1.121379132010e+03, 1.121702693630e+03, 1.122025669960e+03, 1.122348063370e+03, 1.122669876250e+03, 1.122991110960e+03, 1.123311769850e+03, 1.123631855260e+03, 1.123951369500e+03, 1.124270314870e+03, 1.124588693680e+03, 1.124906508190e+03, 1.125223760660e+03, 1.125540453350e+03, 1.125856588490e+03, 1.126172168300e+03, 1.126487194990e+03, 1.126801670740e+03, 1.127115597750e+03, 1.127428978170e+03, 1.127741814160e+03, 1.128054107870e+03, 1.128365861410e+03, 1.128677076900e+03, 1.128987756440e+03, 1.129297902130e+03, 1.129607516040e+03, 1.129916600230e+03, 1.130225156750e+03, 1.130533187650e+03, 1.130840694950e+03, 1.131147680660e+03},
    {1.771801509090e+00, 5.408902710550e+00, 9.121478155050e+00, 1.291360102540e+01, 1.678970826760e+01, 2.075464628820e+01, 2.481372420810e+01, 2.897277624960e+01, 3.323823523650e+01, 3.761721971350e+01, 4.211763788630e+01, 4.674831250810e+01, 5.151913208610e+01, 5.644123548490e+01, 6.152723935820e+01, 6.679152114620e+01, 7.225057508830e+01, 7.792346553830e+01, 8.383241198490e+01, 9.000355546780e+01, 9.646797973820e+01, 1.032630981420e+02, 1.104345788860e+02, 1.180390860910e+02, 1.261482992190e+02, 1.348550168790e+02, 1.442828248320e+02, 1.546022246230e+02, 1.660593635400e+02, 1.790318035010e+02, 1.941503872580e+02, 2.126169046510e+02, 2.373244452450e+02, 6.790993276690e+02, 6.990441004350e+02, 7.132409966610e+02, 7.245741851440e+02, 7.341331956890e+02, 7.424659506820e+02, 7.498917876990e+02, 7.566149732920e+02, 7.627749479400e+02, 7.684716158780e+02, 7.737792898690e+02, 7.787549276780e+02, 7.834432671240e+02, 7.878801708640e+02, 7.920948856510e+02, 7.961116159000e+02, 7.999506488560e+02, 8.036291776790e+02, 8.071619156790e+02, 8.105615628310e+02, 8.138391656970e+02, 8.170043990060e+02, 8.200657887440e+02, 8.230308909210e+02, 8.259064363000e+02, 8.286984486850e+02, 8.314123424160e+02, 8.340530033750e+02, 8.366248567780e+02, 8.391319242760e+02, 8.415778723700e+02, 8.439660536810e+02, 8.462995423240e+02, 8.485811643810e+02, 8.508135242690e+02, 8.529990276600e+02, 8.551399014880e+02, 8.572382114690e+02, 8.592958775140e+02, 8.613146873240e+02, 8.632963084220e+02, 8.652422988390e+02, 8.671541166280e+02, 8.690331283580e+02, 8.708806167180e+02, 8.726977873460e+02, 8.744857749650e+02, 8.762456489220e+02, 8.779784181900e+02, 8.796850358980e+02, 8.813664034430e+02, 8.830233742230e+02, 8.846567570450e+02, 8.862673192280e+02, 8.878557894370e+02, 8.894228602860e+02, 8.909691907160e+02, 8.924954081820e+02, 8.940021106650e+02, 8.954898685220e+02, 8.969592261970e+02, 8.984107037940e+02, 8.998447985350e+02, 9.012619861120e+02, 9.026627219370e+02, 9.040474422990e+02, 9.054165654480e+02, 9.067704925910e+02, 9.081096088290e+02, 9.094342840280e+02, 9.107448736250e+02, 9.120417193910e+02, 9.133251501370e+02, 9.145954823730e+02, 9.158530209330e+02, 9.170980595550e+02, 9.183308814250e+02, 9.195517596860e+02, 9.207609579270e+02, 9.219587306250e+02, 9.231453235770e+02, 9.243209742990e+02, 9.254859124030e+02, 9.266403599510e+02, 9.277845317960e+02, 9.289186358920e+02, 9.300428736010e+02, 9.311574399720e+02, 9.322625240090e+02, 9.333583089270e+02, 9.344449723940e+02, 9.355226867520e+02, 9.365916192420e+02, 9.376519322030e+02, 9.387037832710e+02, 9.397473255590e+02, 9.407827078390e+02, 9.418100747060e+02, 9.428295667390e+02, 9.438413206480e+02, 9.448454694270e+02, 9.458421424830e+02, 9.468314657730e+02, 9.478135619260e+02, 9.487885503660e+02, 9.497565474220e+02, 9.507176664400e+02, 9.516720178840e+02, 9.526197094420e+02, 9.535608461120e+02, 9.544955303030e+02, 9.554238619130e+02, 9.563459384190e+02, 9.572618549560e+02, 9.581717043920e+02, 9.590755773990e+02, 9.599735625320e+02, 9.608657462880e+02, 9.617522131760e+02, 9.626330457760e+02, 9.635083248020e+02, 9.643781291590e+02, 9.652425359940e+02, 9.661016207570e+02, 9.669554572440e+02, 9.678041176530e+02, 9.686476726270e+02, 9.694861913000e+02, 9.703197413440e+02, 9.711483890060e+02, 9.719721991550e+02, 9.727912353150e+02, 9.736055597100e+02, 9.744152332910e+02, 9.752203157820e+02, 9.760208657040e+02, 9.768169404150e+02, 9.776085961390e+02, 9.783958879950e+02, 9.791788700300e+02, 9.799575952430e+02, 9.807321156190e+02, 9.815024821480e+02, 9.822687448570e+02, 9.830309528310e+02, 9.837891542400e+02, 9.845433963600e+02, 9.852937255970e+02, 9.860401875080e+02, 9.867828268230e+02, 9.875216874640e+02, 9.882568125670e+02, 9.889882444980e+02, 9.897160248770e+02, 9.904401945910e+02, 9.911607938120e+02, 9.918778620180e+02, 9.925914380030e+02, 9.933015599010e+02, 9.940082651940e+02, 9.947115907320e+02, 9.954115727440e+02, 9.961082468560e+02, 9.968016481020e+02, 9.974918109380e+02, 9.981787692550e+02, 9.988625563910e+02, 9.995432051460e+02, 1.000220747790e+03, 1.000895216080e+03, 1.001566641260e+03, 1.002235054090e+03, 1.002900484840e+03, 1.003562963300e+03, 1.004222518820e+03, 1.004879180260e+03, 1.005532976080e+03, 1.006183934270e+03, 1.006832082400e+03, 1.007477447650e+03, 1.008120056760e+03, 1.008759936080e+03, 1.009397111580e+03, 1.010031608830e+03, 1.010663453030e+03, 1.011292669010e+03, 1.011919281260e+03, 1.012543313870e+03, 1.013164790630e+03, 1.013783734960e+03, 1.014400169960e+03, 1.015014118400e+03, 1.015625602730e+03, 1.016234645080e+03, 1.016841267270e+03, 1.017445490830e+03, 1.018047336970e+03, 1.018646826650e+03, 1.019243980490e+03, 1.019838818870e+03, 1.020431361880e+03, 1.021021629340e+03, 1.021609640800e+03, 1.022195415550e+03, 1.022778972640e+03, 1.023360330850e+03, 1.023939508710e+03, 1.024516524540e+03, 1.025091396380e+03, 1.025664142060e+03, 1.026234779170e+03, 1.026803325090e+03, 1.027369796960e+03, 1.027934211700e+03, 1.028496586050e+03, 1.029056936490e+03, 1.029615279330e+03, 1.030171630660e+03, 1.030726006380e+03, 1.031278422180e+03, 1.031828893570e+03, 1.032377435860e+03, 1.032924064170e+03, 1.033468793460e+03, 1.034011638480e+03, 1.034552613820e+03, 1.035091733890e+03, 1.035629012930e+03, 1.036164465000e+03, 1.036698104010e+03, 1.037229943690e+03, 1.037759997620e+03, 1.038288279230e+03, 1.038814801770e+03, 1.039339578340e+03, 1.039862621920e+03, 1.040383945300e+03, 1.040903561160e+03, 1.041421481990e+03, 1.041937720190e+03, 1.042452287980e+03, 1.042965197470e+03, 1.043476460620e+03, 1.043986089250e+03, 1.044494095060e+03, 1.045000489620e+03, 1.045505284370e+03, 1.046008490630e+03, 1.046510119580e+03, 1.047010182290e+03, 1.047508689710e+03, 1.048005652670e+03, 1.048501081890e+03, 1.048994987950e+03, 1.049487381360e+03, 1.049978272470e+03, 1.050467671560e+03, 1.050955588770e+03, 1.051442034160e+03, 1.051927017670e+03, 1.052410549140e+03, 1.052892638290e+03, 1.053373294780e+03, 1.053852528130e+03, 1.054330347780e+03, 1.054806763070e+03, 1.055281783250e+03, 1.055755417470e+03, 1.056227674780e+03, 1.056698564160e+03, 1.057168094470e+03, 1.057636274520e+03, 1.058103112990e+03, 1.058568618500e+03, 1.059032799580e+03, 1.059495664670e+03, 1.059957222140e+03, 1.060417480250e+03, 1.060876447210e+03, 1.061334131130e+03, 1.061790540050e+03, 1.062245681930e+03, 1.062699564660e+03, 1.063152196040e+03, 1.063603583800e+03, 1.064053735610e+03, 1.064502659050e+03, 1.064950361640e+03, 1.065396850810e+03, 1.065842133950e+03, 1.066286218350e+03, 1.066729111250e+03, 1.067170819820e+03, 1.067611351160e+03, 1.068050712300e+03, 1.068488910220e+03, 1.068925951830e+03, 1.069361843960e+03, 1.069796593390e+03, 1.070230206850e+03, 1.070662691000e+03, 1.071094052420e+03, 1.071524297670e+03, 1.071953433210e+03, 1.072381465470e+03, 1.072808400820e+03, 1.073234245550e+03, 1.073659005930e+03, 1.074082688140e+03, 1.074505298330e+03, 1.074926842580e+03, 1.075347326930e+03, 1.075766757370e+03, 1.076185139800e+03, 1.076602480130e+03, 1.077018784170e+03, 1.077434057690e+03, 1.077848306430e+03, 1.078261536060e+03, 1.078673752210e+03, 1.079084960460e+03, 1.079495166340e+03, 1.079904375350e+03, 1.080312592910e+03, 1.080719824430e+03, 1.081126075250e+03, 1.081531350680e+03, 1.081935655980e+03, 1.082338996360e+03, 1.082741376990e+03, 1.083142803010e+03, 1.083543279510e+03, 1.083942811520e+03, 1.084341404060e+03, 1.084739062080e+03, 1.085135790510e+03, 1.085531594240e+03, 1.085926478090e+03, 1.086320446890e+03, 1.086713505400e+03, 1.087105658330e+03, 1.087496910390e+03, 1.087887266220e+03, 1.088276730440e+03, 1.088665307620e+03, 1.089053002310e+03, 1.089439819020e+03, 1.089825762210e+03, 1.090210836310e+03, 1.090595045740e+03, 1.090978394850e+03, 1.091360887980e+03, 1.091742529420e+03, 1.092123323450e+03, 1.092503274280e+03, 1.092882386130e+03, 1.093260663160e+03, 1.093638109500e+03, 1.094014729260e+03, 1.094390526510e+03, 1.094765505300e+03, 1.095139669620e+03, 1.095513023470e+03, 1.095885570800e+03, 1.096257315510e+03, 1.096628261520e+03, 1.096998412670e+03, 1.097367772800e+03, 1.097736345720e+03, 1.098104135200e+03, 1.098471144990e+03, 1.098837378810e+03, 1.099202840360e+03, 1.099567533310e+03, 1.099931461280e+03, 1.100294627900e+03, 1.100657036750e+03, 1.101018691390e+03, 1.101379595360e+03, 1.101739752160e+03, 1.102099165290e+03, 1.102457838190e+03, 1.102815774310e+03, 1.103172977050e+03, 1.103529449790e+03, 1.103885195910e+03, 1.104240218730e+03, 1.104594521570e+03, 1.104948107720e+03, 1.105300980440e+03, 1.105653142980e+03, 1.106004598570e+03, 1.106355350390e+03, 1.106705401630e+03, 1.107054755450e+03, 1.107403414960e+03, 1.107751383280e+03, 1.108098663510e+03, 1.108445258710e+03, 1.108791171920e+03, 1.109136406180e+03, 1.109480964480e+03, 1.109824849820e+03, 1.110168065150e+03, 1.110510613420e+03, 1.110852497550e+03, 1.111193720460e+03, 1.111534285010e+03, 1.111874194080e+03, 1.112213450520e+03, 1.112552057150e+03, 1.112890016780e+03, 1.113227332200e+03, 1.113564006180e+03, 1.113900041470e+03, 1.114235440810e+03, 1.114570206910e+03, 1.114904342470e+03, 1.115237850170e+03, 1.115570732680e+03, 1.115902992640e+03, 1.116234632670e+03, 1.116565655400e+03, 1.116896063400e+03, 1.117225859280e+03, 1.117555045570e+03, 1.117883624830e+03, 1.118211599580e+03, 1.118538972330e+03, 1.118865745590e+03, 1.119191921830e+03, 1.119517503510e+03, 1.119842493090e+03, 1.120166892990e+03, 1.120490705630e+03, 1.120813933420e+03, 1.121136578730e+03, 1.121458643950e+03, 1.121780131420e+03, 1.122101043500e+03, 1.122421382500e+03, 1.122741150750e+03, 1.123060350530e+03, 1.123378984130e+03, 1.123697053830e+03, 1.124014561880e+03, 1.124331510510e+03, 1.124647901970e+03, 1.124963738460e+03, 1.125279022180e+03, 1.125593755330e+03, 1.125907940070e+03, 1.126221578560e+03, 1.126534672960e+03, 1.126847225400e+03, 1.127159238010e+03, 1.127470712880e+03, 1.127781652110e+03, 1.128092057800e+03, 1.128401932000e+03, 1.128711276790e+03, 1.129020094200e+03},
    {1.765702753560e+00, 5.389663167880e+00, 9.087926039030e+00, 1.286444224590e+01, 1.672350979220e+01, 2.066981663020e+01, 2.470849042880e+01, 2.884515670440e+01, 3.308600709160e+01, 3.743787998640e+01, 4.190835640590e+01, 4.650587470430e+01, 5.123986886050e+01, 5.612093650830e+01, 6.116104487030e+01, 6.637378553820e+01, 7.177469295720e+01, 7.738164710370e+01, 8.321538907210e+01, 8.930019056050e+01, 9.566473696560e+01, 1.023433130710e+02, 1.093774274070e+02, 1.168180895770e+02, 1.247290896720e+02, 1.331918714000e+02, 1.423130495700e+02, 1.522365454490e+02, 1.631643124120e+02, 1.753943781470e+02, 1.893977029860e+02, 2.059959599080e+02, 2.268685368450e+02, 2.566880666490e+02, 6.657350582540e+02, 6.882594837870e+02, 7.037520935650e+02, 7.159143620180e+02, 7.260669895580e+02, 7.348545459120e+02, 7.426451243660e+02, 7.496706062320e+02, 7.560873838980e+02, 7.620064132580e+02, 7.675095601900e+02, 7.726591458420e+02, 7.775038394770e+02, 7.820824640400e+02, 7.864265472160e+02, 7.905620862580e+02, 7.945108022830e+02, 7.982910528300e+02, 8.019185095340e+02, 8.054066705430e+02, 8.087672542650e+02, 8.120105063080e+02, 8.151454418610e+02, 8.181800393370e+02, 8.211213967400e+02, 8.239758591330e+02, 8.267491234840e+02, 8.294463255980e+02, 8.320721127260e+02, 8.346307046380e+02, 8.371259453100e+02, 8.395613469340e+02, 8.419401275800e+02, 8.442652436130e+02, 8.465394177010e+02, 8.487651631460e+02, 8.509448050940e+02, 8.530804990970e+02, 8.551742474290e+02, 8.572279134600e+02, 8.592432343790e+02, 8.612218324760e+02, 8.631652251850e+02, 8.650748340440e+02, 8.669519927110e+02, 8.687979541590e+02, 8.706138971340e+02, 8.724009319900e+02, 8.741601059480e+02, 8.758924078590e+02, 8.775987725240e+02, 8.792800846210e+02, 8.809371822740e+02, 8.825708603180e+02, 8.841818732700e+02, 8.857709380560e+02, 8.873387365050e+02, 8.888859176450e+02, 8.904130998040e+02, 8.919208725510e+02, 8.934097984840e+02, 8.948804148800e+02, 8.963332352180e+02, 8.977687505900e+02, 8.991874310120e+02, 9.005897266300e+02, 9.019760688520e+02, 9.033468713930e+02, 9.047025312460e+02, 9.060434295940e+02, 9.073699326550e+02, 9.086823924700e+02, 9.099811476430e+02, 9.112665240310e+02, 9.125388353920e+02, 9.137983839860e+02, 9.150454611460e+02, 9.162803478100e+02, 9.175033150220e+02, 9.187146244010e+02, 9.199145285850e+02, 9.211032716440e+02, 9.222810894780e+02, 9.234482101820e+02, 9.246048543990e+02, 9.257512356450e+02, 9.268875606250e+02, 9.280140295240e+02, 9.291308362870e+02, 9.302381688810e+02, 9.313362095480e+02, 9.324251350360e+02, 9.335051168300e+02, 9.345763213570e+02, 9.356389101960e+02, 9.366930402660e+02, 9.377388640050e+02, 9.387765295510e+02, 9.398061809020e+02, 9.408279580730e+02, 9.418419972470e+02, 9.428484309180e+02, 9.438473880250e+02, 9.448389940840e+02, 9.458233713100e+02, 9.468006387340e+02, 9.477709123170e+02, 9.487343050600e+02, 9.496909271020e+02, 9.506408858230e+02, 9.515842859360e+02, 9.525212295750e+02, 9.534518163860e+02, 9.543761436090e+02, 9.552943061500e+02, 9.562063966690e+02, 9.571125056410e+02, 9.580127214340e+02, 9.589071303710e+02, 9.597958167990e+02, 9.606788631470e+02, 9.615563499840e+02, 9.624283560830e+02, 9.632949584680e+02, 9.641562324720e+02, 9.650122517830e+02, 9.658630885000e+02, 9.667088131710e+02, 9.675494948450e+02, 9.683852011120e+02, 9.692159981490e+02, 9.700419507530e+02, 9.708631223890e+02, 9.716795752200e+02, 9.724913701470e+02, 9.732985668450e+02, 9.741012237930e+02, 9.748993983090e+02, 9.756931465810e+02, 9.764825236980e+02, 9.772675836790e+02, 9.780483795000e+02, 9.788249631220e+02, 9.795973855190e+02, 9.803656967020e+02, 9.811299457450e+02, 9.818901808070e+02, 9.826464491580e+02, 9.833987971980e+02, 9.841472704820e+02, 9.848919137400e+02, 9.856327708940e+02, 9.863698850860e+02, 9.871032986860e+02, 9.878330533200e+02, 9.885591898850e+02, 9.892817485630e+02, 9.900007688420e+02, 9.907162895330e+02, 9.914283487800e+02, 9.921369840830e+02, 9.928422323070e+02, 9.935441297010e+02, 9.942427119090e+02, 9.949380139840e+02, 9.956300704030e+02, 9.963189150790e+02, 9.970045813730e+02, 9.976871021080e+02, 9.983665095790e+02, 9.990428355630e+02, 9.997161113360e+02, 1.000386367680e+03, 1.001053634890e+03, 1.001717942790e+03, 1.002379320740e+03, 1.003037797660e+03, 1.003693402010e+03, 1.004346161810e+03, 1.004996104690e+03, 1.005643257810e+03, 1.006287647980e+03, 1.006929301570e+03, 1.007568244570e+03, 1.008204502590e+03, 1.008838100860e+03, 1.009469064250e+03, 1.010097417250e+03, 1.010723184020e+03, 1.011346388360e+03, 1.011967053730e+03, 1.012585203260e+03, 1.013200859740e+03, 1.013814045660e+03, 1.014424783190e+03, 1.015033094180e+03, 1.015639000190e+03, 1.016242522460e+03, 1.016843681980e+03, 1.017442499410e+03, 1.018038995160e+03, 1.018633189350e+03, 1.019225101830e+03, 1.019814752190e+03, 1.020402159750e+03, 1.020987343580e+03, 1.021570322510e+03, 1.022151115110e+03, 1.022729739690e+03, 1.023306214360e+03, 1.023880556970e+03, 1.024452785150e+03, 1.025022916290e+03, 1.025590967590e+03, 1.026156955990e+03, 1.026720898250e+03, 1.027282810890e+03, 1.027842710260e+03, 1.028400612470e+03, 1.028956533460e+03, 1.029510488940e+03, 1.030062494450e+03, 1.030612565340e+03, 1.031160716770e+03, 1.031706963710e+03, 1.032251320950e+03, 1.032793803100e+03, 1.033334424620e+03, 1.033873199760e+03, 1.034410142620e+03, 1.034945267140e+03, 1.035478587080e+03, 1.036010116050e+03, 1.036539867500e+03, 1.037067854710e+03, 1.037594090830e+03, 1.038118588840e+03, 1.038641361570e+03, 1.039162421730e+03, 1.039681781840e+03, 1.040199454330e+03, 1.040715451440e+03, 1.041229785310e+03, 1.041742467920e+03, 1.042253511140e+03, 1.042762926680e+03, 1.043270726140e+03, 1.043776920990e+03, 1.044281522570e+03, 1.044784542090e+03, 1.045285990660e+03, 1.045785879230e+03, 1.046284218680e+03, 1.046781019740e+03, 1.047276293030e+03, 1.047770049060e+03, 1.048262298240e+03, 1.048753050850e+03, 1.049242317070e+03, 1.049730106980e+03, 1.050216430530e+03, 1.050701297600e+03, 1.051184717950e+03, 1.051666701230e+03, 1.052147257000e+03, 1.052626394740e+03, 1.053104123790e+03, 1.053580453430e+03, 1.054055392840e+03, 1.054528951090e+03, 1.055001137180e+03, 1.055471960010e+03, 1.055941428390e+03, 1.056409551030e+03, 1.056876336580e+03, 1.057341793580e+03, 1.057805930510e+03, 1.058268755740e+03, 1.058730277570e+03, 1.059190504220e+03, 1.059649443830e+03, 1.060107104470e+03, 1.060563494100e+03, 1.061018620650e+03, 1.061472491930e+03, 1.061925115690e+03, 1.062376499630e+03, 1.062826651350e+03, 1.063275578380e+03, 1.063723288180e+03, 1.064169788150e+03, 1.064615085620e+03, 1.065059187830e+03, 1.065502101980e+03, 1.065943835180e+03, 1.066384394500e+03, 1.066823786930e+03, 1.067262019380e+03, 1.067699098720e+03, 1.068135031750e+03, 1.068569825220e+03, 1.069003485790e+03, 1.069436020080e+03, 1.069867434650e+03, 1.070297735990e+03, 1.070726930550e+03, 1.071155024710e+03, 1.071582024800e+03, 1.072007937070e+03, 1.072432767750e+03, 1.072856522990e+03, 1.073279208910e+03, 1.073700831540e+03, 1.074121396890e+03, 1.074540910910e+03, 1.074959379480e+03, 1.075376808460e+03, 1.075793203630e+03, 1.076208570740e+03, 1.076622915480e+03, 1.077036243510e+03, 1.077448560400e+03, 1.077859871720e+03, 1.078270182970e+03, 1.078679499600e+03, 1.079087827020e+03, 1.079495170600e+03, 1.079901535660e+03, 1.080306927480e+03, 1.080711351270e+03, 1.081114812250e+03, 1.081517315530e+03, 1.081918866240e+03, 1.082319469440e+03, 1.082719130130e+03, 1.083117853300e+03, 1.083515643900e+03, 1.083912506800e+03, 1.084308446880e+03, 1.084703468950e+03, 1.085097577800e+03, 1.085490778150e+03, 1.085883074730e+03, 1.086274472180e+03, 1.086664975150e+03, 1.087054588220e+03, 1.087443315950e+03, 1.087831162860e+03, 1.088218133430e+03, 1.088604232110e+03, 1.088989463320e+03, 1.089373831430e+03, 1.089757340800e+03, 1.090139995720e+03, 1.090521800490e+03, 1.090902759340e+03, 1.091282876490e+03, 1.091662156120e+03, 1.092040602370e+03, 1.092418219370e+03, 1.092795011190e+03, 1.093170981900e+03, 1.093546135510e+03, 1.093920476020e+03, 1.094294007390e+03, 1.094666733550e+03, 1.095038658410e+03, 1.095409785840e+03, 1.095780119690e+03, 1.096149663770e+03, 1.096518421880e+03, 1.096886397760e+03, 1.097253595170e+03, 1.097620017790e+03, 1.097985669310e+03, 1.098350553380e+03, 1.098714673630e+03, 1.099078033640e+03, 1.099440637010e+03, 1.099802487260e+03, 1.100163587920e+03, 1.100523942490e+03, 1.100883554430e+03, 1.101242427190e+03, 1.101600564200e+03, 1.101957968840e+03, 1.102314644480e+03, 1.102670594490e+03, 1.103025822170e+03, 1.103380330840e+03, 1.103734123760e+03, 1.104087204200e+03, 1.104439575380e+03, 1.104791240510e+03, 1.105142202780e+03, 1.105492465350e+03, 1.105842031370e+03, 1.106190903960e+03, 1.106539086210e+03, 1.106886581210e+03, 1.107233392000e+03, 1.107579521630e+03, 1.107924973110e+03, 1.108269749440e+03, 1.108613853580e+03, 1.108957288490e+03, 1.109300057110e+03, 1.109642162340e+03, 1.109983607090e+03, 1.110324394230e+03, 1.110664526610e+03, 1.111004007070e+03, 1.111342838420e+03, 1.111681023470e+03, 1.112018564990e+03, 1.112355465750e+03, 1.112691728490e+03, 1.113027355940e+03, 1.113362350790e+03, 1.113696715740e+03, 1.114030453470e+03, 1.114363566620e+03, 1.114696057840e+03, 1.115027929740e+03, 1.115359184930e+03, 1.115689825980e+03, 1.116019855480e+03, 1.116349275970e+03, 1.116678089990e+03, 1.117006300060e+03, 1.117333908690e+03, 1.117660918350e+03, 1.117987331520e+03, 1.118313150670e+03, 1.118638378220e+03, 1.118963016610e+03, 1.119287068240e+03, 1.119610535510e+03, 1.119933420790e+03, 1.120255726460e+03, 1.120577454860e+03, 1.120898608330e+03, 1.121219189180e+03, 1.121539199720e+03, 1.121858642250e+03, 1.122177519050e+03, 1.122495832360e+03, 1.122813584460e+03, 1.123130777560e+03, 1.123447413900e+03, 1.123763495670e+03, 1.124079025080e+03, 1.124394004310e+03, 1.124708435520e+03, 1.125022320870e+03, 1.125335662500e+03, 1.125648462540e+03, 1.125960723110e+03, 1.126272446300e+03, 1.126583634220e+03, 1.126894288930e+03},
    {1.759646552910e+00, 5.370566986530e+00, 9.054640698550e+00, 1.281570030370e+01, 1.665790971290e+01, 2.058580427100e+01, 2.460433727710e+01, 2.871893416570e+01, 3.293555594040e+01, 3.726077385430e+01, 4.170185786070e+01, 4.626688205050e+01, 5.096485121590e+01, 5.580585392480e+01, 6.080124917980e+01, 6.596389607120e+01, 7.130843910050e+01, 7.685166649080e+01, 8.261296551470e+01, 8.861490875160e+01, 9.488402005590e+01, 1.014517919020e+02, 1.083560619210e+02, 1.156429152890e+02, 1.233693785980e+02, 1.316073440720e+02, 1.404494800220e+02, 1.500184959020e+02, 1.604823927590e+02, 1.720811489650e+02, 1.851772618580e+02, 2.003622942770e+02, 2.187191145300e+02, 2.426510519290e+02, 2.803015012850e+02, 6.521434686270e+02, 6.773957513890e+02, 6.942283047700e+02, 7.072396955750e+02, 7.179965662050e+02, 7.272450013810e+02, 7.354039348830e+02, 7.427339001560e+02, 7.494088083590e+02, 7.555509963300e+02, 7.612500823120e+02, 7.665738747380e+02, 7.715750527270e+02, 7.762954496430e+02, 7.807689062730e+02, 7.850232332890e+02, 7.890815987260e+02, 7.929635326200e+02, 7.966856698090e+02, 8.002623093810e+02, 8.037058429980e+02, 8.070270876820e+02, 8.102355478150e+02, 8.133396238940e+02, 8.163467806590e+02, 8.192636838530e+02, 8.220963124710e+02, 8.248500516510e+02, 8.275297701350e+02, 8.301398853130e+02, 8.326844181970e+02, 8.351670401610e+02, 8.375911129000e+02, 8.399597227790e+02, 8.422757104840e+02, 8.445416967540e+02, 8.467601047910e+02, 8.489331798600e+02, 8.510630065020e+02, 8.531515236880e+02, 8.552005382220e+02, 8.572117366250e+02, 8.591866956960e+02, 8.611268919390e+02, 8.630337099880e+02, 8.649084501670e+02, 8.667523352710e+02, 8.685665166910e+02, 8.703520799290e+02, 8.721100495950e+02, 8.738413939370e+02, 8.755470289510e+02, 8.772278221260e+02, 8.788845958560e+02, 8.805181305530e+02, 8.821291675020e+02, 8.837184114650e+02, 8.852865330830e+02, 8.868341710760e+02, 8.883619342660e+02, 8.898704034480e+02, 8.913601331080e+02, 8.928316530210e+02, 8.942854697180e+02, 8.957220678520e+02, 8.971419114600e+02, 8.985454451440e+02, 8.999330951530e+02, 9.013052704060e+02, 9.026623634300e+02, 9.040047512500e+02, 9.053327962040e+02, 9.066468467150e+02, 9.079472380100e+02, 9.092342927900e+02, 9.105083218590e+02, 9.117696247210e+02, 9.130184901230e+02, 9.142551965840e+02, 9.154800128810e+02, 9.166931985070e+02, 9.178950041050e+02, 9.190856718750e+02, 9.202654359580e+02, 9.214345228000e+02, 9.225931514900e+02, 9.237415340880e+02, 9.248798759250e+02, 9.260083758950e+02, 9.271272267280e+02, 9.282366152470e+02, 9.293367226160e+02, 9.304277245670e+02, 9.315097916270e+02, 9.325830893220e+02, 9.336477783800e+02, 9.347040149160e+02, 9.357519506130e+02, 9.367917328920e+02, 9.378235050770e+02, 9.388474065470e+02, 9.398635728850e+02, 9.408721360160e+02, 9.418732243430e+02, 9.428669628770e+02, 9.438534733530e+02, 9.448328743510e+02, 9.458052814080e+02, 9.467708071190e+02, 9.477295612440e+02, 9.486816508030e+02, 9.496271801690e+02, 9.505662511560e+02, 9.514989631080e+02, 9.524254129770e+02, 9.533456954010e+02, 9.542599027840e+02, 9.551681253610e+02, 9.560704512720e+02, 9.569669666280e+02, 9.578577555730e+02, 9.587429003460e+02, 9.596224813370e+02, 9.604965771490e+02, 9.613652646490e+02, 9.622286190180e+02, 9.630867138060e+02, 9.639396209780e+02, 9.647874109580e+02, 9.656301526800e+02, 9.664679136240e+02, 9.673007598620e+02, 9.681287561000e+02, 9.689519657090e+02, 9.697704507710e+02, 9.705842721080e+02, 9.713934893230e+02, 9.721981608260e+02, 9.729983438730e+02, 9.737940945950e+02, 9.745854680250e+02, 9.753725181340e+02, 9.761552978520e+02, 9.769338590980e+02, 9.777082528080e+02, 9.784785289580e+02, 9.792447365900e+02, 9.800069238360e+02, 9.807651379380e+02, 9.815194252760e+02, 9.822698313830e+02, 9.830164009710e+02, 9.837591779500e+02, 9.844982054460e+02, 9.852335258200e+02, 9.859651806900e+02, 9.866932109450e+02, 9.874176567630e+02, 9.881385576300e+02, 9.888559523530e+02, 9.895698790800e+02, 9.902803753110e+02, 9.909874779160e+02, 9.916912231480e+02, 9.923916466560e+02, 9.930887835030e+02, 9.937826681730e+02, 9.944733345890e+02, 9.951608161240e+02, 9.958451456120e+02, 9.965263553600e+02, 9.972044771610e+02, 9.978795423040e+02, 9.985515815860e+02, 9.992206253200e+02, 9.998867033490e+02, 1.000549845050e+03, 1.001210079350e+03, 1.001867434750e+03, 1.002521939270e+03, 1.003173620570e+03, 1.003822505830e+03, 1.004468621880e+03, 1.005111995100e+03, 1.005752651530e+03, 1.006390616780e+03, 1.007025916110e+03, 1.007658574400e+03, 1.008288616190e+03, 1.008916065650e+03, 1.009540946600e+03, 1.010163282530e+03, 1.010783096590e+03, 1.011400411610e+03, 1.012015250090e+03, 1.012627634220e+03, 1.013237585890e+03, 1.013845126680e+03, 1.014450277870e+03, 1.015053060450e+03, 1.015653495130e+03, 1.016251602330e+03, 1.016847402200e+03, 1.017440914620e+03, 1.018032159210e+03, 1.018621155310e+03, 1.019207922030e+03, 1.019792478200e+03, 1.020374842430e+03, 1.020955033080e+03, 1.021533068260e+03, 1.022108965850e+03, 1.022682743510e+03, 1.023254418670e+03, 1.023824008530e+03, 1.024391530080e+03, 1.024957000090e+03, 1.025520435130e+03, 1.026081851560e+03, 1.026641265520e+03, 1.027198692960e+03, 1.027754149650e+03, 1.028307651140e+03, 1.028859212800e+03, 1.029408849830e+03, 1.029956577210e+03, 1.030502409780e+03, 1.031046362160e+03, 1.031588448830e+03, 1.032128684080e+03, 1.032667082030e+03, 1.033203656650e+03, 1.033738421720e+03, 1.034271390880e+03, 1.034802577600e+03, 1.035331995200e+03, 1.035859656840e+03, 1.036385575540e+03, 1.036909764150e+03, 1.037432235380e+03, 1.037953001820e+03, 1.038472075890e+03, 1.038989469870e+03, 1.039505195930e+03, 1.040019266060e+03, 1.040531692160e+03, 1.041042485970e+03, 1.041551659110e+03, 1.042059223070e+03, 1.042565189220e+03, 1.043069568800e+03, 1.043572372920e+03, 1.044073612590e+03, 1.044573298680e+03, 1.045071441960e+03, 1.045568053070e+03, 1.046063142550e+03, 1.046556720820e+03, 1.047048798190e+03, 1.047539384870e+03, 1.048028490950e+03, 1.048516126410e+03, 1.049002301150e+03, 1.049487024950e+03, 1.049970307490e+03, 1.050452158350e+03, 1.050932587020e+03, 1.051411602880e+03, 1.051889215220e+03, 1.052365433240e+03, 1.052840266040e+03, 1.053313722640e+03, 1.053785811950e+03, 1.054256542810e+03, 1.054725923960e+03, 1.055193964050e+03, 1.055660671660e+03, 1.056126055280e+03, 1.056590123300e+03, 1.057052884050e+03, 1.057514345760e+03, 1.057974516600e+03, 1.058433404640e+03, 1.058891017880e+03, 1.059347364260e+03, 1.059802451610e+03, 1.060256287720e+03, 1.060708880280e+03, 1.061160236920e+03, 1.061610365180e+03, 1.062059272570e+03, 1.062506966470e+03, 1.062953454250e+03, 1.063398743170e+03, 1.063842840440e+03, 1.064285753200e+03, 1.064727488520e+03, 1.065168053420e+03, 1.065607454830e+03, 1.066045699630e+03, 1.066482794650e+03, 1.066918746640e+03, 1.067353562290e+03, 1.067787248230e+03, 1.068219811050e+03, 1.068651257260e+03, 1.069081593300e+03, 1.069510825590e+03, 1.069938960460e+03, 1.070366004200e+03, 1.070791963040e+03, 1.071216843150e+03, 1.071640650660e+03, 1.072063391640e+03, 1.072485072090e+03, 1.072905697980e+03, 1.073325275220e+03, 1.073743809680e+03, 1.074161307150e+03, 1.074577773400e+03, 1.074993214140e+03, 1.075407635030e+03, 1.075821041670e+03, 1.076233439650e+03, 1.076644834460e+03, 1.077055231600e+03, 1.077464636470e+03, 1.077873054460e+03, 1.078280490910e+03, 1.078686951100e+03, 1.079092440290e+03, 1.079496963670e+03, 1.079900526400e+03, 1.080303133600e+03, 1.080704790350e+03, 1.081105501680e+03, 1.081505272580e+03, 1.081904108010e+03, 1.082302012870e+03, 1.082698992040e+03, 1.083095050350e+03, 1.083490192590e+03, 1.083884423520e+03, 1.084277747860e+03, 1.084670170280e+03, 1.085061695420e+03, 1.085452327900e+03, 1.085842072290e+03, 1.086230933100e+03, 1.086618914850e+03, 1.087006021980e+03, 1.087392258940e+03, 1.087777630110e+03, 1.088162139850e+03, 1.088545792480e+03, 1.088928592300e+03, 1.089310543560e+03, 1.089691650480e+03, 1.090071917270e+03, 1.090451348070e+03, 1.090829947030e+03, 1.091207718230e+03, 1.091584665740e+03, 1.091960793590e+03, 1.092336105800e+03, 1.092710606330e+03, 1.093084299140e+03, 1.093457188130e+03, 1.093829277190e+03, 1.094200570180e+03, 1.094571070920e+03, 1.094940783220e+03, 1.095309710850e+03, 1.095677857550e+03, 1.096045227040e+03, 1.096411823010e+03, 1.096777649110e+03, 1.097142708990e+03, 1.097507006250e+03, 1.097870544480e+03, 1.098233327230e+03, 1.098595358040e+03, 1.098956640400e+03, 1.099317177800e+03, 1.099676973700e+03, 1.100036031530e+03, 1.100394354680e+03, 1.100751946550e+03, 1.101108810490e+03, 1.101464949840e+03, 1.101820367900e+03, 1.102175067970e+03, 1.102529053310e+03, 1.102882327150e+03, 1.103234892730e+03, 1.103586753220e+03, 1.103937911820e+03, 1.104288371670e+03, 1.104638135900e+03, 1.104987207610e+03, 1.105335589900e+03, 1.105683285840e+03, 1.106030298460e+03, 1.106376630790e+03, 1.106722285840e+03, 1.107067266580e+03, 1.107411575990e+03, 1.107755216990e+03, 1.108098192520e+03, 1.108440505490e+03, 1.108782158760e+03, 1.109123155210e+03, 1.109463497680e+03, 1.109803188990e+03, 1.110142231960e+03, 1.110480629380e+03, 1.110818384000e+03, 1.111155498600e+03, 1.111491975890e+03, 1.111827818590e+03, 1.112163029400e+03, 1.112497611010e+03, 1.112831566070e+03, 1.113164897230e+03, 1.113497607120e+03, 1.113829698340e+03, 1.114161173500e+03, 1.114492035160e+03, 1.114822285890e+03, 1.115151928240e+03, 1.115480964730e+03, 1.115809397870e+03, 1.116137230160e+03, 1.116464464080e+03, 1.116791102090e+03, 1.117117146640e+03, 1.117442600160e+03, 1.117767465070e+03, 1.118091743770e+03, 1.118415438660e+03, 1.118738552090e+03, 1.119061086430e+03, 1.119383044020e+03, 1.119704427190e+03, 1.120025238250e+03, 1.120345479500e+03, 1.120665153220e+03, 1.120984261690e+03, 1.121302807160e+03, 1.121620791880e+03, 1.121938218070e+03, 1.122255087950e+03, 1.122571403710e+03, 1.122887167560e+03, 1.123202381660e+03, 1.123517048170e+03, 1.123831169250e+03, 1.124144747030e+03, 1.124457783630e+03, 1.124770281160e+03},
    {1.753632452350e+00, 5.351612461880e+00, 9.021618636070e+00, 1.276736921390e+01, 1.659289866890e+01, 2.050259533520e+01, 2.450124490150e+01, 2.859408092390e+01, 3.278684378500e+01, 3.708584987920e+01, 4.149807326680e+01, 4.603124263820e+01, 5.069395722760e+01, 5.549582637620e+01, 6.044763888430e+01, 6.556157025420e+01, 7.085143865410e+01, 7.633302426990e+01, 8.202447219630e+01, 8.794680700460e+01, 9.412459897400e+01, 1.005868399360e+02, 1.073681145860e+02, 1.145101976570e+02, 1.220642806070e+02, 1.300941564400e+02, 1.386809130470e+02, 1.479300986380e+02, 1.579831374290e+02, 1.690364951170e+02, 1.813760582400e+02, 1.954443922090e+02, 2.119890075910e+02, 2.324525983100e+02, 2.603612313600e+02, 3.116778258860e+02, 6.386033786450e+02, 6.665565573750e+02, 6.847258636820e+02, 6.985861518450e+02, 7.099474579550e+02, 7.196568108000e+02, 7.281838963900e+02, 7.358179416220e+02, 7.427504422560e+02, 7.491151751390e+02, 7.550095559630e+02, 7.605069120470e+02, 7.656639531000e+02, 7.705255355150e+02, 7.751278200060e+02, 7.795004335230e+02, 7.836679912340e+02, 7.876511942040e+02, 7.914676380060e+02, 7.951324196560e+02, 7.986586008180e+02, 8.020575666540e+02, 8.053393075700e+02, 8.085126431480e+02, 8.115854020900e+02, 8.145645682840e+02, 8.174564004560e+02, 8.202665310190e+02, 8.230000483670e+02, 8.256615658770e+02, 8.282552801520e+02, 8.307850204750e+02, 8.332542910450e+02, 8.356663072420e+02, 8.380240269090e+02, 8.403301774640e+02, 8.425872795070e+02, 8.447976674410e+02, 8.469635075690e+02, 8.490868140280e+02, 8.511694628580e+02, 8.532132044740e+02, 8.552196747560e+02, 8.571904049240e+02, 8.591268303740e+02, 8.610302985920e+02, 8.629020762610e+02, 8.647433556670e+02, 8.665552604750e+02, 8.683388509510e+02, 8.700951287000e+02, 8.718250409610e+02, 8.735294845230e+02, 8.752093092820e+02, 8.768653215020e+02, 8.784982867810e+02, 8.801089327860e+02, 8.816979517420e+02, 8.832660027370e+02, 8.848137138260e+02, 8.863416839860e+02, 8.878504849010e+02, 8.893406626310e+02, 8.908127391360e+02, 8.922672137010e+02, 8.937045642520e+02, 8.951252485780e+02, 8.965297054640e+02, 8.979183557490e+02, 8.992916033090e+02, 9.006498359730e+02, 9.019934263770e+02, 9.033227327650e+02, 9.046380997340e+02, 9.059398589310e+02, 9.072283297090e+02, 9.085038197370e+02, 9.097666255800e+02, 9.110170332330e+02, 9.122553186320e+02, 9.134817481260e+02, 9.146965789340e+02, 9.159000595580e+02, 9.170924301860e+02, 9.182739230700e+02, 9.194447628730e+02, 9.206051670100e+02, 9.217553459600e+02, 9.228955035690e+02, 9.240258373280e+02, 9.251465386450e+02, 9.262577930960e+02, 9.273597806660e+02, 9.284526759790e+02, 9.295366485100e+02, 9.306118627960e+02, 9.316784786260e+02, 9.327366512300e+02, 9.337865314540e+02, 9.348282659290e+02, 9.358619972280e+02, 9.368878640220e+02, 9.379060012230e+02, 9.389165401220e+02, 9.399196085220e+02, 9.409153308630e+02, 9.419038283430e+02, 9.428852190320e+02, 9.438596179800e+02, 9.448271373280e+02, 9.457878864010e+02, 9.467419718060e+02, 9.476894975290e+02, 9.486305650140e+02, 9.495652732550e+02, 9.504937188720e+02, 9.514159961900e+02, 9.523321973140e+02, 9.532424121990e+02, 9.541467287160e+02, 9.550452327240e+02, 9.559380081270e+02, 9.568251369350e+02, 9.577066993250e+02, 9.585827736970e+02, 9.594534367230e+02, 9.603187634030e+02, 9.611788271150e+02, 9.620336996590e+02, 9.628834513060e+02, 9.637281508430e+02, 9.645678656130e+02, 9.654026615580e+02, 9.662326032600e+02, 9.670577539760e+02, 9.678781756770e+02, 9.686939290820e+02, 9.695050736970e+02, 9.703116678420e+02, 9.711137686860e+02, 9.719114322800e+02, 9.727047135810e+02, 9.734936664890e+02, 9.742783438670e+02, 9.750587975720e+02, 9.758350784830e+02, 9.766072365210e+02, 9.773753206780e+02, 9.781393790380e+02, 9.788994588020e+02, 9.796556063060e+02, 9.804078670490e+02, 9.811562857090e+02, 9.819009061620e+02, 9.826417715060e+02, 9.833789240770e+02, 9.841124054690e+02, 9.848422565480e+02, 9.855685174750e+02, 9.862912277190e+02, 9.870104260710e+02, 9.877261506660e+02, 9.884384389930e+02, 9.891473279120e+02, 9.898528536680e+02, 9.905550519060e+02, 9.912539576820e+02, 9.919496054780e+02, 9.926420292160e+02, 9.933312622670e+02, 9.940173374660e+02, 9.947002871220e+02, 9.953801430330e+02, 9.960569364900e+02, 9.967306982970e+02, 9.974014587720e+02, 9.980692477650e+02, 9.987340946640e+02, 9.993960284060e+02, 1.000055077490e+03, 1.000711269960e+03, 1.001364633480e+03, 1.002015195250e+03, 1.002662982110e+03, 1.003308020450e+03, 1.003950336320e+03, 1.004589955350e+03, 1.005226902810e+03, 1.005861203620e+03, 1.006492882310e+03, 1.007121963060e+03, 1.007748469730e+03, 1.008372425810e+03, 1.008993854490e+03, 1.009612778590e+03, 1.010229220650e+03, 1.010843202870e+03, 1.011454747170e+03, 1.012063875130e+03, 1.012670608060e+03, 1.013274966970e+03, 1.013876972580e+03, 1.014476645360e+03, 1.015074005450e+03, 1.015669072770e+03, 1.016261866940e+03, 1.016852407350e+03, 1.017440713100e+03, 1.018026803080e+03, 1.018610695880e+03, 1.019192409910e+03, 1.019771963290e+03, 1.020349373930e+03, 1.020924659500e+03, 1.021497837460e+03, 1.022068925040e+03, 1.022637939240e+03, 1.023204896860e+03, 1.023769814490e+03, 1.024332708510e+03, 1.024893595080e+03, 1.025452490180e+03, 1.026009409600e+03, 1.026564368910e+03, 1.027117383510e+03, 1.027668468610e+03, 1.028217639230e+03, 1.028764910210e+03, 1.029310296220e+03, 1.029853811740e+03, 1.030395471090e+03, 1.030935288420e+03, 1.031473277700e+03, 1.032009452760e+03, 1.032543827250e+03, 1.033076414670e+03, 1.033607228340e+03, 1.034136281470e+03, 1.034663587070e+03, 1.035189158050e+03, 1.035713007130e+03, 1.036235146900e+03, 1.036755589820e+03, 1.037274348200e+03, 1.037791434220e+03, 1.038306859890e+03, 1.038820637140e+03, 1.039332777720e+03, 1.039843293290e+03, 1.040352195340e+03, 1.040859495280e+03, 1.041365204350e+03, 1.041869333710e+03, 1.042371894370e+03, 1.042872897230e+03, 1.043372353070e+03, 1.043870272570e+03, 1.044366666270e+03, 1.044861544630e+03, 1.045354917960e+03, 1.045846796510e+03, 1.046337190370e+03, 1.046826109570e+03, 1.047313564000e+03, 1.047799563480e+03, 1.048284117700e+03, 1.048767236260e+03, 1.049248928670e+03, 1.049729204340e+03, 1.050208072570e+03, 1.050685542590e+03, 1.051161623500e+03, 1.051636324350e+03, 1.052109654080e+03, 1.052581621530e+03, 1.053052235480e+03, 1.053521504580e+03, 1.053989437440e+03, 1.054456042560e+03, 1.054921328350e+03, 1.055385303170e+03, 1.055847975260e+03, 1.056309352800e+03, 1.056769443900e+03, 1.057228256570e+03, 1.057685798750e+03, 1.058142078310e+03, 1.058597103040e+03, 1.059050880670e+03, 1.059503418820e+03, 1.059954725080e+03, 1.060404806940e+03, 1.060853671840e+03, 1.061301327130e+03, 1.061747780110e+03, 1.062193037990e+03, 1.062637107950e+03, 1.063079997050e+03, 1.063521712340e+03, 1.063962260770e+03, 1.064401649240e+03, 1.064839884580e+03, 1.065276973560e+03, 1.065712922900e+03, 1.066147739250e+03, 1.066581429190e+03, 1.067013999260e+03, 1.067445455930e+03, 1.067875805610e+03, 1.068305054660e+03, 1.068733209380e+03, 1.069160276020e+03, 1.069586260780e+03, 1.070011169770e+03, 1.070435009100e+03, 1.070857784790e+03, 1.071279502810e+03, 1.071700169090e+03, 1.072119789510e+03, 1.072538369880e+03, 1.072955915990e+03, 1.073372433560e+03, 1.073787928260e+03, 1.074202405720e+03, 1.074615871510e+03, 1.075028331180e+03, 1.075439790210e+03, 1.075850254030e+03, 1.076259728040e+03, 1.076668217590e+03, 1.077075727980e+03, 1.077482264470e+03, 1.077887832280e+03, 1.078292436580e+03, 1.078696082510e+03, 1.079098775140e+03, 1.079500519540e+03, 1.079901320700e+03, 1.080301183590e+03, 1.080700113130e+03, 1.081098114210e+03, 1.081495191670e+03, 1.081891350330e+03, 1.082286594940e+03, 1.082680930240e+03, 1.083074360920e+03, 1.083466891630e+03, 1.083858527000e+03, 1.084249271600e+03, 1.084639129980e+03, 1.085028106640e+03, 1.085416206070e+03, 1.085803432700e+03, 1.086189790930e+03, 1.086575285140e+03, 1.086959919660e+03, 1.087343698790e+03, 1.087726626800e+03, 1.088108707940e+03, 1.088489946400e+03, 1.088870346350e+03, 1.089249911940e+03, 1.089628647270e+03, 1.090006556420e+03, 1.090383643440e+03, 1.090759912340e+03, 1.091135367120e+03, 1.091510011710e+03, 1.091883850060e+03, 1.092256886050e+03, 1.092629123560e+03, 1.093000566430e+03, 1.093371218450e+03, 1.093741083430e+03, 1.094110165110e+03, 1.094478467210e+03, 1.094845993440e+03, 1.095212747470e+03, 1.095578732950e+03, 1.095943953500e+03, 1.096308412700e+03, 1.096672114120e+03, 1.097035061310e+03, 1.097397257790e+03, 1.097758707030e+03, 1.098119412510e+03, 1.098479377670e+03, 1.098838605920e+03, 1.099197100660e+03, 1.099554865250e+03, 1.099911903040e+03, 1.100268217340e+03, 1.100623811460e+03, 1.100978688670e+03, 1.101332852210e+03, 1.101686305320e+03, 1.102039051200e+03, 1.102391093040e+03, 1.102742433990e+03, 1.103093077200e+03, 1.103443025780e+03, 1.103792282830e+03, 1.104140851420e+03, 1.104488734620e+03, 1.104835935440e+03, 1.105182456900e+03, 1.105528302000e+03, 1.105873473710e+03, 1.106217974970e+03, 1.106561808720e+03, 1.106904977880e+03, 1.107247485320e+03, 1.107589333930e+03, 1.107930526560e+03, 1.108271066040e+03, 1.108610955180e+03, 1.108950196800e+03, 1.109288793650e+03, 1.109626748510e+03, 1.109964064120e+03, 1.110300743190e+03, 1.110636788440e+03, 1.110972202550e+03, 1.111306988190e+03, 1.111641148010e+03, 1.111974684660e+03, 1.112307600750e+03, 1.112639898880e+03, 1.112971581640e+03, 1.113302651590e+03, 1.113633111280e+03, 1.113962963260e+03, 1.114292210040e+03, 1.114620854130e+03, 1.114948898000e+03, 1.115276344140e+03, 1.115603194990e+03, 1.115929453010e+03, 1.116255120600e+03, 1.116580200190e+03, 1.116904694160e+03, 1.117228604890e+03, 1.117551934760e+03, 1.117874686100e+03, 1.118196861260e+03, 1.118518462550e+03, 1.118839492280e+03, 1.119159952740e+03, 1.119479846210e+03, 1.119799174950e+03, 1.120117941210e+03, 1.120436147230e+03, 1.120753795230e+03, 1.121070887420e+03, 1.121387425980e+03, 1.121703413120e+03, 1.122018850980e+03, 1.122333741730e+03, 1.122648087510e+03},
    {1.747660003790e+00, 5.332797918330e+00, 8.988856421230e+00, 1.271944311950e+01, 1.652846751910e+01, 2.042017630260e+01, 2.439919400160e+01, 2.847057010170e+01, 3.263983385900e+01, 3.691305842010e+01, 4.129693623490e+01, 4.579886827920e+01, 5.042707028230e+01, 5.519070007220e+01, 6.010001136680e+01, 6.516654099810e+01, 7.040333883710e+01, 7.582525286070e+01, 8.144928629940e+01, 8.729505027010e+01, 9.338534477370e+01, 9.974691509660e+01, 1.064114522970e+02, 1.134169403630e+02, 1.208095072840e+02, 1.286460281930e+02, 1.369978857240e+02, 1.459565754900e+02, 1.556423802920e+02, 1.662184144690e+02, 1.779146737880e+02, 1.910722693110e+02, 2.062328640530e+02, 2.243451314200e+02, 2.473480733390e+02, 2.805382084380e+02, 3.634450728570e+02, 6.254816501430e+02, 6.558681838960e+02, 6.753105553640e+02, 6.899944499360e+02, 7.019477487740e+02, 7.121109012630e+02, 7.210015206710e+02, 7.289363225270e+02, 7.361238289790e+02, 7.427089810840e+02, 7.487968473900e+02, 7.544661933420e+02, 7.597777107120e+02, 7.647792472960e+02, 7.695092613240e+02, 7.739991788990e+02, 7.782750484510e+02, 7.823587303640e+02, 7.862687708050e+02, 7.900210557940e+02, 7.936293090950e+02, 7.971054769750e+02, 8.004600295990e+02, 8.037022000590e+02, 8.068401760590e+02, 8.098812552210e+02, 8.128319720890e+02, 8.156982028760e+02, 8.184852525540e+02, 8.211979277610e+02, 8.238405982790e+02, 8.264172491620e+02, 8.289315252290e+02, 8.313867692020e+02, 8.337860546010e+02, 8.361322142170e+02, 8.384278648880e+02, 8.406754291280e+02, 8.428771540970e+02, 8.450351282890e+02, 8.471512962650e+02, 8.492274717050e+02, 8.512653490040e+02, 8.532665135980e+02, 8.552324512010e+02, 8.571645560640e+02, 8.590641384090e+02, 8.609324311050e+02, 8.627705956960e+02, 8.645797278580e+02, 8.663608623270e+02, 8.681149773910e+02, 8.698429989560e+02, 8.715458042660e+02, 8.732242252880e+02, 8.748790518140e+02, 8.765110342920e+02, 8.781208864350e+02, 8.797092876010e+02, 8.812768850000e+02, 8.828242957110e+02, 8.843521085530e+02, 8.858608858050e+02, 8.873511648030e+02, 8.888234594160e+02, 8.902782614110e+02, 8.917160417240e+02, 8.931372516360e+02, 8.945423238760e+02, 8.959316736350e+02, 8.973056995180e+02, 8.986647844350e+02, 9.000092964260e+02, 9.013395894350e+02, 9.026560040370e+02, 9.039588681110e+02, 9.052484974820e+02, 9.065251965090e+02, 9.077892586520e+02, 9.090409669900e+02, 9.102805947210e+02, 9.115084056210e+02, 9.127246544860e+02, 9.139295875420e+02, 9.151234428330e+02, 9.163064505900e+02, 9.174788335720e+02, 9.186408074010e+02, 9.197925808620e+02, 9.209343562030e+02, 9.220663294080e+02, 9.231886904590e+02, 9.243016235870e+02, 9.254053075060e+02, 9.264999156340e+02, 9.275856163100e+02, 9.286625729930e+02, 9.297309444540e+02, 9.307908849580e+02, 9.318425444360e+02, 9.328860686520e+02, 9.339215993620e+02, 9.349492744560e+02, 9.359692281090e+02, 9.369815909120e+02, 9.379864900060e+02, 9.389840492000e+02, 9.399743890950e+02, 9.409576271930e+02, 9.419338780080e+02, 9.429032531660e+02, 9.438658615090e+02, 9.448218091840e+02, 9.457711997360e+02, 9.467141341940e+02, 9.476507111580e+02, 9.485810268710e+02, 9.495051753020e+02, 9.504232482150e+02, 9.513353352440e+02, 9.522415239550e+02, 9.531418999140e+02, 9.540365467470e+02, 9.549255462050e+02, 9.558089782120e+02, 9.566869209310e+02, 9.575594508090e+02, 9.584266426320e+02, 9.592885695730e+02, 9.601453032390e+02, 9.609969137190e+02, 9.618434696230e+02, 9.626850381290e+02, 9.635216850230e+02, 9.643534747360e+02, 9.651804703850e+02, 9.660027338050e+02, 9.668203255890e+02, 9.676333051220e+02, 9.684417306090e+02, 9.692456591120e+02, 9.700451465770e+02, 9.708402478660e+02, 9.716310167870e+02, 9.724175061140e+02, 9.731997676260e+02, 9.739778521210e+02, 9.747518094490e+02, 9.755216885320e+02, 9.762875373890e+02, 9.770494031610e+02, 9.778073321260e+02, 9.785613697300e+02, 9.793115605980e+02, 9.800579485610e+02, 9.808005766730e+02, 9.815394872270e+02, 9.822747217790e+02, 9.830063211600e+02, 9.837343254960e+02, 9.844587742250e+02, 9.851797061120e+02, 9.858971592630e+02, 9.866111711440e+02, 9.873217785930e+02, 9.880290178340e+02, 9.887329244950e+02, 9.894335336140e+02, 9.901308796590e+02, 9.908249965390e+02, 9.915159176120e+02, 9.922036757050e+02, 9.928883031180e+02, 9.935698316400e+02, 9.942482925590e+02, 9.949237166700e+02, 9.955961342920e+02, 9.962655752700e+02, 9.969320689920e+02, 9.975956443940e+02, 9.982563299720e+02, 9.989141537880e+02, 9.995691434840e+02, 1.000221326290e+03, 1.000870729020e+03, 1.001517378090e+03, 1.002161299550e+03, 1.002802519050e+03, 1.003441061860e+03, 1.004076952890e+03, 1.004710216690e+03, 1.005340877460e+03, 1.005968959060e+03, 1.006594485000e+03, 1.007217478460e+03, 1.007837962300e+03, 1.008455959050e+03, 1.009071490930e+03, 1.009684579860e+03, 1.010295247460e+03, 1.010903515030e+03, 1.011509403610e+03, 1.012112933930e+03, 1.012714126460e+03, 1.013313001380e+03, 1.013909578610e+03, 1.014503877790e+03, 1.015095918310e+03, 1.015685719320e+03, 1.016273299690e+03, 1.016858678060e+03, 1.017441872820e+03, 1.018022902140e+03, 1.018601783930e+03, 1.019178535880e+03, 1.019753175470e+03, 1.020325719950e+03, 1.020896186330e+03, 1.021464591440e+03, 1.022030951870e+03, 1.022595284030e+03, 1.023157604100e+03, 1.023717928080e+03, 1.024276271770e+03, 1.024832650760e+03, 1.025387080480e+03, 1.025939576140e+03, 1.026490152790e+03, 1.027038825290e+03, 1.027585608320e+03, 1.028130516390e+03, 1.028673563840e+03, 1.029214764830e+03, 1.029754133360e+03, 1.030291683260e+03, 1.030827428210e+03, 1.031361381730e+03, 1.031893557170e+03, 1.032423967740e+03, 1.032952626480e+03, 1.033479546310e+03, 1.034004739970e+03, 1.034528220090e+03, 1.035049999130e+03, 1.035570089420e+03, 1.036088503150e+03, 1.036605252370e+03, 1.037120349020e+03, 1.037633804870e+03, 1.038145631590e+03, 1.038655840700e+03, 1.039164443620e+03, 1.039671451620e+03, 1.040176875870e+03, 1.040680727400e+03, 1.041183017120e+03, 1.041683755850e+03, 1.042182954270e+03, 1.042680622960e+03, 1.043176772360e+03, 1.043671412840e+03, 1.044164554640e+03, 1.044656207880e+03, 1.045146382600e+03, 1.045635088730e+03, 1.046122336080e+03, 1.046608134370e+03, 1.047092493220e+03, 1.047575422160e+03, 1.048056930610e+03, 1.048537027900e+03, 1.049015723250e+03, 1.049493025820e+03, 1.049968944640e+03, 1.050443488680e+03, 1.050916666810e+03, 1.051388487800e+03, 1.051858960350e+03, 1.052328093060e+03, 1.052795894450e+03, 1.053262372970e+03, 1.053727536970e+03, 1.054191394730e+03, 1.054653954430e+03, 1.055115224190e+03, 1.055575212060e+03, 1.056033925980e+03, 1.056491373840e+03, 1.056947563450e+03, 1.057402502550e+03, 1.057856198780e+03, 1.058308659750e+03, 1.058759892960e+03, 1.059209905860e+03, 1.059658705830e+03, 1.060106300170e+03, 1.060552696130e+03, 1.060997900870e+03, 1.061441921500e+03, 1.061884765070e+03, 1.062326438540e+03, 1.062766948830e+03, 1.063206302800e+03, 1.063644507220e+03, 1.064081568820e+03, 1.064517494270e+03, 1.064952290170e+03, 1.065385963070e+03, 1.065818519450e+03, 1.066249965750e+03, 1.066680308330e+03, 1.067109553520e+03, 1.067537707570e+03, 1.067964776690e+03, 1.068390767020e+03, 1.068815684670e+03, 1.069239535680e+03, 1.069662326050e+03, 1.070084061700e+03, 1.070504748530e+03, 1.070924392370e+03, 1.071342999020e+03, 1.071760574210e+03, 1.072177123630e+03, 1.072592652910e+03, 1.073007167660e+03, 1.073420673420e+03, 1.073833175690e+03, 1.074244679910e+03, 1.074655191510e+03, 1.075064715820e+03, 1.075473258190e+03, 1.075880823870e+03, 1.076287418100e+03, 1.076693046060e+03, 1.077097712890e+03, 1.077501423710e+03, 1.077904183560e+03, 1.078305997460e+03, 1.078706870400e+03, 1.079106807310e+03, 1.079505813080e+03, 1.079903892580e+03, 1.080301050620e+03, 1.080697291980e+03, 1.081092621410e+03, 1.081487043600e+03, 1.081880563230e+03, 1.082273184910e+03, 1.082664913260e+03, 1.083055752820e+03, 1.083445708110e+03, 1.083834783620e+03, 1.084222983800e+03, 1.084610313070e+03, 1.084996775800e+03, 1.085382376350e+03, 1.085767119020e+03, 1.086151008100e+03, 1.086534047830e+03, 1.086916242440e+03, 1.087297596100e+03, 1.087678112960e+03, 1.088057797140e+03, 1.088436652730e+03, 1.088814683800e+03, 1.089191894350e+03, 1.089568288400e+03, 1.089943869900e+03, 1.090318642790e+03, 1.090692610990e+03, 1.091065778360e+03, 1.091438148760e+03, 1.091809726010e+03, 1.092180513890e+03, 1.092550516180e+03, 1.092919736620e+03, 1.093288178900e+03, 1.093655846720e+03, 1.094022743720e+03, 1.094388873540e+03, 1.094754239780e+03, 1.095118846020e+03, 1.095482695810e+03, 1.095845792670e+03, 1.096208140090e+03, 1.096569741570e+03, 1.096930600540e+03, 1.097290720440e+03, 1.097650104650e+03, 1.098008756570e+03, 1.098366679540e+03, 1.098723876890e+03, 1.099080351920e+03, 1.099436107930e+03, 1.099791148170e+03, 1.100145475880e+03, 1.100499094260e+03, 1.100852006520e+03, 1.101204215820e+03, 1.101555725320e+03, 1.101906538130e+03, 1.102256657360e+03, 1.102606086100e+03, 1.102954827410e+03, 1.103302884320e+03, 1.103650259870e+03, 1.103996957040e+03, 1.104342978820e+03, 1.104688328180e+03, 1.105033008040e+03, 1.105377021330e+03, 1.105720370940e+03, 1.106063059770e+03, 1.106405090670e+03, 1.106746466480e+03, 1.107087190020e+03, 1.107427264110e+03, 1.107766691530e+03, 1.108105475030e+03, 1.108443617390e+03, 1.108781121320e+03, 1.109117989530e+03, 1.109454224730e+03, 1.109789829600e+03, 1.110124806790e+03, 1.110459158940e+03, 1.110792888680e+03, 1.111125998630e+03, 1.111458491370e+03, 1.111790369480e+03, 1.112121635520e+03, 1.112452292030e+03, 1.112782341540e+03, 1.113111786550e+03, 1.113440629570e+03, 1.113768873060e+03, 1.114096519500e+03, 1.114423571330e+03, 1.114750030980e+03, 1.115075900860e+03, 1.115401183390e+03, 1.115725880930e+03, 1.116049995880e+03, 1.116373530570e+03, 1.116696487360e+03, 1.117018868570e+03, 1.117340676510e+03, 1.117661913480e+03, 1.117982581770e+03, 1.118302683640e+03, 1.118622221350e+03, 1.118941197140e+03, 1.119259613250e+03, 1.119577471880e+03, 1.119894775240e+03, 1.120211525520e+03, 1.120527724890e+03},
    {1.741728765670e+00, 5.314121708590e+00, 8.956350689170e+00, 1.267191628720e+01, 1.646460733450e+01, 2.033853399530e+01, 2.429816580740e+01, 2.834837562000e+01, 3.249449057380e+01, 3.674235154740e+01, 4.109838283800e+01, 4.556967430120e+01, 5.016407876500e+01, 5.489032831170e+01, 5.975817406110e+01, 6.477855548700e+01, 6.996380720600e+01, 7.532791381290e+01, 8.088682702710e+01, 8.665886466610e+01, 9.266521856060e+01, 9.893060972700e+01, 1.054841459750e+02, 1.123604630850e+02, 1.196012717220e+02, 1.272574989790e+02, 1.353923256890e+02, 1.440856167180e+02, 1.534405996890e+02, 1.635943372930e+02, 1.747349533050e+02, 1.871317097310e+02, 2.011917070020e+02, 2.175783313730e+02, 2.374975567970e+02, 2.635669458060e+02, 3.038629940340e+02, 4.806354535790e+02, 6.131727545860e+02, 6.454703305900e+02, 6.660548842150e+02, 6.815090767170e+02, 6.940277159570e+02, 7.046294925960e+02, 7.138740838080e+02, 7.221030868400e+02, 7.295407804630e+02, 7.363426081070e+02, 7.426209262340e+02, 7.484597265900e+02, 7.539235537630e+02, 7.590631639990e+02, 7.639192570250e+02, 7.685250183450e+02, 7.729079001000e+02, 7.770908991170e+02, 7.810934938270e+02, 7.849323441830e+02, 7.886218233910e+02, 7.921744280330e+02, 7.956010987340e+02, 7.989114740120e+02, 8.021140935150e+02, 8.052165624030e+02, 8.082256855870e+02, 8.111475782760e+02, 8.139877577680e+02, 8.167512202060e+02, 8.194425052200e+02, 8.220657506880e+02, 8.246247394230e+02, 8.271229391650e+02, 8.295635370400e+02, 8.319494693710e+02, 8.342834475920e+02, 8.365679808660e+02, 8.388053959040e+02, 8.409978543920e+02, 8.431473683690e+02, 8.452558138480e+02, 8.473249429030e+02, 8.493563944390e+02, 8.513517038120e+02, 8.533123114410e+02, 8.552395705380e+02, 8.571347540660e+02, 8.589990610130e+02, 8.608336220630e+02, 8.626395047300e+02, 8.644177180200e+02, 8.661692166690e+02, 8.678949050000e+02, 8.695956404460e+02, 8.712722367690e+02, 8.729254670050e+02, 8.745560661700e+02, 8.761647337330e+02, 8.777521359040e+02, 8.793189077310e+02, 8.808656550390e+02, 8.823929562150e+02, 8.839013638680e+02, 8.853914063500e+02, 8.868635891830e+02, 8.883183963690e+02, 8.897562916140e+02, 8.911777194630e+02, 8.925831063600e+02, 8.939728616310e+02, 8.953473784010e+02, 8.967070344590e+02, 8.980521930490e+02, 8.993832036290e+02, 9.007004025670e+02, 9.020041138030e+02, 9.032946494590e+02, 9.045723104240e+02, 9.058373868920e+02, 9.070901588780e+02, 9.083308966930e+02, 9.095598614000e+02, 9.107773052370e+02, 9.119834720220e+02, 9.131785975280e+02, 9.143629098430e+02, 9.155366297080e+02, 9.166999708350e+02, 9.178531402090e+02, 9.189963383730e+02, 9.201297597040e+02, 9.212535926600e+02, 9.223680200320e+02, 9.234732191690e+02, 9.245693622010e+02, 9.256566162420e+02, 9.267351435940e+02, 9.278051019270e+02, 9.288666444670e+02, 9.299199201550e+02, 9.309650738200e+02, 9.320022463250e+02, 9.330315747180e+02, 9.340531923710e+02, 9.350672291130e+02, 9.360738113600e+02, 9.370730622340e+02, 9.380651016810e+02, 9.390500465810e+02, 9.400280108560e+02, 9.409991055720e+02, 9.419634390320e+02, 9.429211168750e+02, 9.438722421620e+02, 9.448169154600e+02, 9.457552349280e+02, 9.466872963920e+02, 9.476131934230e+02, 9.485330174060e+02, 9.494468576110e+02, 9.503548012610e+02, 9.512569335940e+02, 9.521533379250e+02, 9.530440957040e+02, 9.539292865750e+02, 9.548089884280e+02, 9.556832774540e+02, 9.565522281920e+02, 9.574159135820e+02, 9.582744050090e+02, 9.591277723480e+02, 9.599760840070e+02, 9.608194069730e+02, 9.616578068470e+02, 9.624913478850e+02, 9.633200930370e+02, 9.641441039800e+02, 9.649634411570e+02, 9.657781638070e+02, 9.665883299990e+02, 9.673939966650e+02, 9.681952196270e+02, 9.689920536290e+02, 9.697845523650e+02, 9.705727685060e+02, 9.713567537270e+02, 9.721365587300e+02, 9.729122332740e+02, 9.736838261950e+02, 9.744513854280e+02, 9.752149580360e+02, 9.759745902240e+02, 9.767303273660e+02, 9.774822140230e+02, 9.782302939610e+02, 9.789746101740e+02, 9.797152049020e+02, 9.804521196440e+02, 9.811853951830e+02, 9.819150715980e+02, 9.826411882820e+02, 9.833637839570e+02, 9.840828966910e+02, 9.847985639130e+02, 9.855108224270e+02, 9.862197084230e+02, 9.869252575000e+02, 9.876275046670e+02, 9.883264843670e+02, 9.890222304840e+02, 9.897147763540e+02, 9.904041547820e+02, 9.910903980500e+02, 9.917735379290e+02, 9.924536056910e+02, 9.931306321180e+02, 9.938046475130e+02, 9.944756817120e+02, 9.951437640910e+02, 9.958089235770e+02, 9.964711886570e+02, 9.971305873890e+02, 9.977871474070e+02, 9.984408959330e+02, 9.990918597830e+02, 9.997400653770e+02, 1.000385538750e+03, 1.001028305540e+03, 1.001668391050e+03, 1.002305820160e+03, 1.002940617440e+03, 1.003572807100e+03, 1.004202412970e+03, 1.004829458590e+03, 1.005453967130e+03, 1.006075961470e+03, 1.006695464130e+03, 1.007312497350e+03, 1.007927083050e+03, 1.008539242850e+03, 1.009148998080e+03, 1.009756369760e+03, 1.010361378640e+03, 1.010964045210e+03, 1.011564389640e+03, 1.012162431870e+03, 1.012758191550e+03, 1.013351688090e+03, 1.013942940630e+03, 1.014531968060e+03, 1.015118789040e+03, 1.015703421970e+03, 1.016285885010e+03, 1.016866196120e+03, 1.017444372980e+03, 1.018020433080e+03, 1.018594393690e+03, 1.019166271830e+03, 1.019736084340e+03, 1.020303847840e+03, 1.020869578740e+03, 1.021433293240e+03, 1.021995007350e+03, 1.022554736890e+03, 1.023112497460e+03, 1.023668304500e+03, 1.024222173240e+03, 1.024774118750e+03, 1.025324155900e+03, 1.025872299390e+03, 1.026418563730e+03, 1.026962963290e+03, 1.027505512250e+03, 1.028046224610e+03, 1.028585114230e+03, 1.029122194800e+03, 1.029657479840e+03, 1.030190982740e+03, 1.030722716710e+03, 1.031252694830e+03, 1.031780930000e+03, 1.032307435010e+03, 1.032832222490e+03, 1.033355304900e+03, 1.033876694620e+03, 1.034396403830e+03, 1.034914444610e+03, 1.035430828900e+03, 1.035945568510e+03, 1.036458675110e+03, 1.036970160240e+03, 1.037480035340e+03, 1.037988311690e+03, 1.038495000470e+03, 1.039000112740e+03, 1.039503659430e+03, 1.040005651360e+03, 1.040506099240e+03, 1.041005013640e+03, 1.041502405050e+03, 1.041998283840e+03, 1.042492660260e+03, 1.042985544460e+03, 1.043476946490e+03, 1.043966876290e+03, 1.044455343690e+03, 1.044942358440e+03, 1.045427930160e+03, 1.045912068390e+03, 1.046394782570e+03, 1.046876082050e+03, 1.047355976080e+03, 1.047834473810e+03, 1.048311584300e+03, 1.048787316530e+03, 1.049261679390e+03, 1.049734681660e+03, 1.050206332060e+03, 1.050676639200e+03, 1.051145611630e+03, 1.051613257800e+03, 1.052079586080e+03, 1.052544604760e+03, 1.053008322040e+03, 1.053470746060e+03, 1.053931884870e+03, 1.054391746440e+03, 1.054850338680e+03, 1.055307669390e+03, 1.055763746340e+03, 1.056218577190e+03, 1.056672169540e+03, 1.057124530940e+03, 1.057575668830e+03, 1.058025590610e+03, 1.058474303600e+03, 1.058921815060e+03, 1.059368132160e+03, 1.059813262030e+03, 1.060257211730e+03, 1.060699988240e+03, 1.061141598490e+03, 1.061582049350e+03, 1.062021347600e+03, 1.062459500010e+03, 1.062896513230e+03, 1.063332393890e+03, 1.063767148560e+03, 1.064200783720e+03, 1.064633305830e+03, 1.065064721270e+03, 1.065495036370e+03, 1.065924257400e+03, 1.066352390570e+03, 1.066779442070e+03, 1.067205417980e+03, 1.067630324380e+03, 1.068054167260e+03, 1.068476952580e+03, 1.068898686230e+03, 1.069319374080e+03, 1.069739021910e+03, 1.070157635480e+03, 1.070575220490e+03, 1.070991782590e+03, 1.071407327390e+03, 1.071821860450e+03, 1.072235387270e+03, 1.072647913320e+03, 1.073059444020e+03, 1.073469984750e+03, 1.073879540830e+03, 1.074288117550e+03, 1.074695720140e+03, 1.075102353810e+03, 1.075508023720e+03, 1.075912734960e+03, 1.076316492630e+03, 1.076719301730e+03, 1.077121167280e+03, 1.077522094200e+03, 1.077922087410e+03, 1.078321151790e+03, 1.078719292150e+03, 1.079116513300e+03, 1.079512819970e+03, 1.079908216900e+03, 1.080302708750e+03, 1.080696300160e+03, 1.081088995750e+03, 1.081480800070e+03, 1.081871717660e+03, 1.082261753020e+03, 1.082650910600e+03, 1.083039194830e+03, 1.083426610110e+03, 1.083813160790e+03, 1.084198851190e+03, 1.084583685610e+03, 1.084967668300e+03, 1.085350803490e+03, 1.085733095380e+03, 1.086114548120e+03, 1.086495165840e+03, 1.086874952640e+03, 1.087253912590e+03, 1.087632049720e+03, 1.088009368040e+03, 1.088385871530e+03, 1.088761564120e+03, 1.089136449750e+03, 1.089510532300e+03, 1.089883815620e+03, 1.090256303540e+03, 1.090627999870e+03, 1.090998908380e+03, 1.091369032820e+03, 1.091738376910e+03, 1.092106944330e+03, 1.092474738760e+03, 1.092841763820e+03, 1.093208023140e+03, 1.093573520300e+03, 1.093938258850e+03, 1.094302242330e+03, 1.094665474260e+03, 1.095027958100e+03, 1.095389697330e+03, 1.095750695380e+03, 1.096110955650e+03, 1.096470481540e+03, 1.096829276400e+03, 1.097187343570e+03, 1.097544686360e+03, 1.097901308070e+03, 1.098257211970e+03, 1.098612401300e+03, 1.098966879280e+03, 1.099320649110e+03, 1.099673713980e+03, 1.100026077030e+03, 1.100377741400e+03, 1.100728710210e+03, 1.101078986550e+03, 1.101428573480e+03, 1.101777474060e+03, 1.102125691320e+03, 1.102473228250e+03, 1.102820087860e+03, 1.103166273100e+03, 1.103511786930e+03, 1.103856632270e+03, 1.104200812040e+03, 1.104544329110e+03, 1.104887186350e+03, 1.105229386630e+03, 1.105570932770e+03, 1.105911827570e+03, 1.106252073840e+03, 1.106591674350e+03, 1.106930631860e+03, 1.107268949100e+03, 1.107606628790e+03, 1.107943673650e+03, 1.108280086350e+03, 1.108615869560e+03, 1.108951025930e+03, 1.109285558100e+03, 1.109619468680e+03, 1.109952760260e+03, 1.110285435450e+03, 1.110617496790e+03, 1.110948946840e+03, 1.111279788130e+03, 1.111610023180e+03, 1.111939654490e+03, 1.112268684550e+03, 1.112597115820e+03, 1.112924950760e+03, 1.113252191800e+03, 1.113578841380e+03, 1.113904901890e+03, 1.114230375720e+03, 1.114555265270e+03, 1.114879572890e+03, 1.115203300920e+03, 1.115526451700e+03, 1.115849027560e+03, 1.116171030790e+03, 1.116492463680e+03, 1.116813328520e+03, 1.117133627550e+03, 1.117453363040e+03, 1.117772537220e+03, 1.118091152300e+03, 1.118409210500e+03},
    {1.735838302830e+00, 5.295582213100e+00, 8.924098138740e+00, 1.262478310430e+01, 1.640130939220e+01, 2.025765556630e+01, 2.419814205960e+01, 2.822747216600e+01, 3.235077946700e+01, 3.657368296380e+01, 4.090235148770e+01, 4.534357935890e+01, 4.990487577010e+01, 5.459457104190e+01, 5.942194378090e+01, 6.439737415300e+01, 6.953253008220e+01, 7.484059537630e+01, 8.033655181460e+01, 8.603753150610e+01, 9.196326194230e+01, 9.813663505460e+01, 1.045844447810e+02, 1.113383576180e+02, 1.184362116360e+02, 1.259237887540e+02, 1.338572861270e+02, 1.423068500240e+02, 1.513617787200e+02, 1.611384508600e+02, 1.717929153550e+02, 1.835419162440e+02, 1.967002900020e+02, 2.117531397680e+02, 2.295113474420e+02, 2.515040550680e+02, 2.812508407430e+02, 3.309220286770e+02, 5.100161273960e+02, 6.020086433340e+02, 6.355019795130e+02, 6.570336559470e+02, 6.731766610030e+02, 6.862192152780e+02, 6.972358709120e+02, 7.068195462130e+02, 7.153326997440e+02, 7.230133568080e+02, 7.300263887470e+02, 7.364908342440e+02, 7.424955538820e+02, 7.481087249990e+02, 7.533838711810e+02, 7.583638394770e+02, 7.630835093950e+02, 7.675716896070e+02, 7.718524780940e+02, 7.759462582910e+02, 7.798704424050e+02, 7.836400354320e+02, 7.872680696140e+02, 7.907659436670e+02, 7.941436909800e+02, 7.974101940340e+02, 8.005733576190e+02, 8.036402500950e+02, 8.066172195970e+02, 8.095099904010e+02, 8.123237434270e+02, 8.150631839650e+02, 8.177325990000e+02, 8.203359060310e+02, 8.228766948720e+02, 8.253582636280e+02, 8.277836498060e+02, 8.301556573400e+02, 8.324768801610e+02, 8.347497228470e+02, 8.369764187630e+02, 8.391590460720e+02, 8.412995418930e+02, 8.433997148820e+02, 8.454612564200e+02, 8.474857506130e+02, 8.494746832360e+02, 8.514294497630e+02, 8.533513625900e+02, 8.552416575470e+02, 8.571014997850e+02, 8.589319890980e+02, 8.607341647670e+02, 8.625090099440e+02, 8.642574556600e+02, 8.659803844760e+02, 8.676786338140e+02, 8.693529990150e+02, 8.710042361380e+02, 8.726330645250e+02, 8.742401691700e+02, 8.758262028890e+02, 8.773917883310e+02, 8.789375198230e+02, 8.804639650880e+02, 8.819716668220e+02, 8.834611441680e+02, 8.849328940720e+02, 8.863873925520e+02, 8.878250958660e+02, 8.892464416130e+02, 8.906518497490e+02, 8.920417235340e+02, 8.934164504240e+02, 8.947764028950e+02, 8.961219392190e+02, 8.974534041920e+02, 8.987711298070e+02, 9.000754358970e+02, 9.013666307280e+02, 9.026450115620e+02, 9.039108651840e+02, 9.051644683990e+02, 9.064060884960e+02, 9.076359836900e+02, 9.088544035350e+02, 9.100615893160e+02, 9.112577744170e+02, 9.124431846690e+02, 9.136180386810e+02, 9.147825481500e+02, 9.159369181550e+02, 9.170813474380e+02, 9.182160286670e+02, 9.193411486870e+02, 9.204568887580e+02, 9.215634247780e+02, 9.226609275020e+02, 9.237495627420e+02, 9.248294915590e+02, 9.259008704530e+02, 9.269638515310e+02, 9.280185826790e+02, 9.290652077180e+02, 9.301038665540e+02, 9.311346953280e+02, 9.321578265460e+02, 9.331733892160e+02, 9.341815089710e+02, 9.351823081890e+02, 9.361759061060e+02, 9.371624189290e+02, 9.381419599360e+02, 9.391146395800e+02, 9.400805655820e+02, 9.410398430240e+02, 9.419925744370e+02, 9.429388598820e+02, 9.438787970360e+02, 9.448124812660e+02, 9.457400057030e+02, 9.466614613140e+02, 9.475769369720e+02, 9.484865195160e+02, 9.493902938210e+02, 9.502883428550e+02, 9.511807477370e+02, 9.520675877940e+02, 9.529489406140e+02, 9.538248820990e+02, 9.546954865130e+02, 9.555608265330e+02, 9.564209732910e+02, 9.572759964230e+02, 9.581259641090e+02, 9.589709431150e+02, 9.598109988320e+02, 9.606461953190e+02, 9.614765953320e+02, 9.623022603690e+02, 9.631232506950e+02, 9.639396253860e+02, 9.647514423510e+02, 9.655587583690e+02, 9.663616291190e+02, 9.671601092060e+02, 9.679542521920e+02, 9.687441106230e+02, 9.695297360510e+02, 9.703111790660e+02, 9.710884893170e+02, 9.718617155340e+02, 9.726309055550e+02, 9.733961063460e+02, 9.741573640230e+02, 9.749147238730e+02, 9.756682303740e+02, 9.764179272160e+02, 9.771638573180e+02, 9.779060628480e+02, 9.786445852390e+02, 9.793794652090e+02, 9.801107427760e+02, 9.808384572760e+02, 9.815626473760e+02, 9.822833510930e+02, 9.830006058040e+02, 9.837144482680e+02, 9.844249146320e+02, 9.851320404500e+02, 9.858358606950e+02, 9.865364097700e+02, 9.872337215240e+02, 9.879278292610e+02, 9.886187657540e+02, 9.893065632570e+02, 9.899912535120e+02, 9.906728677650e+02, 9.913514367740e+02, 9.920269908210e+02, 9.926995597200e+02, 9.933691728270e+02, 9.940358590510e+02, 9.946996468630e+02, 9.953605643050e+02, 9.960186389960e+02, 9.966738981460e+02, 9.973263685600e+02, 9.979760766470e+02, 9.986230484280e+02, 9.992673095470e+02, 9.999088852710e+02, 1.000547800510e+03, 1.001184079800e+03, 1.001817747350e+03, 1.002448827000e+03, 1.003077342270e+03, 1.003703316350e+03, 1.004326772090e+03, 1.004947732030e+03, 1.005566218400e+03, 1.006182253130e+03, 1.006795857830e+03, 1.007407053830e+03, 1.008015862170e+03, 1.008622303590e+03, 1.009226398570e+03, 1.009828167310e+03, 1.010427629740e+03, 1.011024805520e+03, 1.011619714060e+03, 1.012212374510e+03, 1.012802805770e+03, 1.013391026500e+03, 1.013977055100e+03, 1.014560909760e+03, 1.015142608410e+03, 1.015722168780e+03, 1.016299608350e+03, 1.016874944390e+03, 1.017448193940e+03, 1.018019373850e+03, 1.018588500750e+03, 1.019155591040e+03, 1.019720660960e+03, 1.020283726510e+03, 1.020844803520e+03, 1.021403907610e+03, 1.021961054230e+03, 1.022516258630e+03, 1.023069535870e+03, 1.023620900840e+03, 1.024170368250e+03, 1.024717952640e+03, 1.025263668360e+03, 1.025807529620e+03, 1.026349550440e+03, 1.026889744680e+03, 1.027428126050e+03, 1.027964708080e+03, 1.028499504170e+03, 1.029032527540e+03, 1.029563791280e+03, 1.030093308320e+03, 1.030621091450e+03, 1.031147153310e+03, 1.031671506390e+03, 1.032194163060e+03, 1.032715135540e+03, 1.033234435900e+03, 1.033752076110e+03, 1.034268067970e+03, 1.034782423190e+03, 1.035295153320e+03, 1.035806269790e+03, 1.036315783920e+03, 1.036823706900e+03, 1.037330049790e+03, 1.037834823540e+03, 1.038338038990e+03, 1.038839706840e+03, 1.039339837710e+03, 1.039838442090e+03, 1.040335530350e+03, 1.040831112770e+03, 1.041325199510e+03, 1.041817800620e+03, 1.042308926070e+03, 1.042798585710e+03, 1.043286789270e+03, 1.043773546420e+03, 1.044258866700e+03, 1.044742759570e+03, 1.045225234390e+03, 1.045706300410e+03, 1.046185966800e+03, 1.046664242650e+03, 1.047141136950e+03, 1.047616658580e+03, 1.048090816350e+03, 1.048563619000e+03, 1.049035075150e+03, 1.049505193360e+03, 1.049973982090e+03, 1.050441449720e+03, 1.050907604560e+03, 1.051372454840e+03, 1.051836008690e+03, 1.052298274180e+03, 1.052759259290e+03, 1.053218971940e+03, 1.053677419970e+03, 1.054134611130e+03, 1.054590553110e+03, 1.055045253530e+03, 1.055498719930e+03, 1.055950959780e+03, 1.056401980490e+03, 1.056851789390e+03, 1.057300393750e+03, 1.057747800760e+03, 1.058194017560e+03, 1.058639051220e+03, 1.059082908740e+03, 1.059525597060e+03, 1.059967123050e+03, 1.060407493530e+03, 1.060846715250e+03, 1.061284794900e+03, 1.061721739110e+03, 1.062157554460e+03, 1.062592247440e+03, 1.063025824530e+03, 1.063458292120e+03, 1.063889656550e+03, 1.064319924100e+03, 1.064749101010e+03, 1.065177193450e+03, 1.065604207550e+03, 1.066030149360e+03, 1.066455024920e+03, 1.066878840170e+03, 1.067301601040e+03, 1.067723313390e+03, 1.068143983020e+03, 1.068563615710e+03, 1.068982217160e+03, 1.069399793030e+03, 1.069816348950e+03, 1.070231890480e+03, 1.070646423150e+03, 1.071059952420e+03, 1.071472483740e+03, 1.071884022490e+03, 1.072294574000e+03, 1.072704143570e+03, 1.073112736450e+03, 1.073520357860e+03, 1.073927012960e+03, 1.074332706860e+03, 1.074737444670e+03, 1.075141231400e+03, 1.075544072070e+03, 1.075945971640e+03, 1.076346935010e+03, 1.076746967080e+03, 1.077146072680e+03, 1.077544256610e+03, 1.077941523640e+03, 1.078337878480e+03, 1.078733325840e+03, 1.079127870360e+03, 1.079521516640e+03, 1.079914269280e+03, 1.080306132820e+03, 1.080697111750e+03, 1.081087210550e+03, 1.081476433650e+03, 1.081864785470e+03, 1.082252270350e+03, 1.082638892650e+03, 1.083024656650e+03, 1.083409566640e+03, 1.083793626830e+03, 1.084176841430e+03, 1.084559214630e+03, 1.084940750540e+03, 1.085321453280e+03, 1.085701326930e+03, 1.086080375530e+03, 1.086458603100e+03, 1.086836013630e+03, 1.087212611060e+03, 1.087588399320e+03, 1.087963382320e+03, 1.088337563910e+03, 1.088710947940e+03, 1.089083538220e+03, 1.089455338530e+03, 1.089826352630e+03, 1.090196584240e+03, 1.090566037060e+03, 1.090934714770e+03, 1.091302621020e+03, 1.091669759420e+03, 1.092036133570e+03, 1.092401747030e+03, 1.092766603360e+03, 1.093130706060e+03, 1.093494058630e+03, 1.093856664530e+03, 1.094218527220e+03, 1.094579650100e+03, 1.094940036560e+03, 1.095299689990e+03, 1.095658613730e+03, 1.096016811090e+03, 1.096374285370e+03, 1.096731039870e+03, 1.097087077810e+03, 1.097442402450e+03, 1.097797016980e+03, 1.098150924590e+03, 1.098504128450e+03, 1.098856631690e+03, 1.099208437440e+03, 1.099559548800e+03, 1.099909968850e+03, 1.100259700630e+03, 1.100608747200e+03, 1.100957111560e+03, 1.101304796700e+03, 1.101651805610e+03, 1.101998141240e+03, 1.102343806530e+03, 1.102688804380e+03, 1.103033137690e+03, 1.103376809340e+03, 1.103719822190e+03, 1.104062179080e+03, 1.104403882820e+03, 1.104744936220e+03, 1.105085342050e+03, 1.105425103090e+03, 1.105764222070e+03, 1.106102701730e+03, 1.106440544770e+03, 1.106777753890e+03, 1.107114331760e+03, 1.107450281040e+03, 1.107785604370e+03, 1.108120304370e+03, 1.108454383640e+03, 1.108787844780e+03, 1.109120690360e+03, 1.109452922940e+03, 1.109784545050e+03, 1.110115559220e+03, 1.110445967950e+03, 1.110775773750e+03, 1.111104979070e+03, 1.111433586390e+03, 1.111761598150e+03, 1.112089016770e+03, 1.112415844680e+03, 1.112742084260e+03, 1.113067737910e+03, 1.113392807990e+03, 1.113717296860e+03, 1.114041206850e+03, 1.114364540300e+03, 1.114687299510e+03, 1.115009486770e+03, 1.115331104380e+03, 1.115652154590e+03, 1.115972639670e+03, 1.116292561860e+03},
    {1.729988186450e+00, 5.277177839360e+00, 8.892095530940e+00, 1.257803807530e+01, 1.633856516850e+01, 2.017752848820e+01, 2.409910499050e+01, 2.810783516250e+01, 3.220866715450e+01, 3.640700792800e+01, 4.070878281840e+01, 4.512050525690e+01, 4.964935883260e+01, 5.430329445210e+01, 5.909114610470e+01, 6.402276974320e+01, 6.910921113090e+01, 7.436291032440e+01, 7.979795296720e+01, 8.543038205070e+01, 9.127858873210e+01, 9.736380787450e+01, 1.037107543350e+02, 1.103484514690e+02, 1.173113268570e+02, 1.246406870090e+02, 1.323867417670e+02, 1.406114466650e+02, 1.493925987300e+02, 1.588299196650e+02, 1.690544193160e+02, 1.802434399060e+02, 1.926461321410e+02, 2.066295432460e+02, 2.227694580210e+02, 2.420514902840e+02, 2.663969521900e+02, 3.004480002200e+02, 3.612317374900e+02, 5.162708188360e+02, 5.921729354410e+02, 6.260851015700e+02, 6.483187158870e+02, 6.650438846190e+02, 6.785548208140e+02, 6.899540477670e+02, 6.998564322620e+02, 7.086400155760e+02, 7.165538637340e+02, 7.237707958080e+02, 7.304156813250e+02, 7.365817390740e+02, 7.423404612460e+02, 7.477479338550e+02, 7.528490146510e+02, 7.576801830120e+02, 7.622715371350e+02, 7.666482269450e+02, 7.708315036410e+02, 7.748395027320e+02, 7.786878379310e+02, 7.823900583520e+02, 7.859580052360e+02, 7.894020937340e+02, 7.927315379960e+02, 7.959545328280e+02, 7.990784017030e+02, 8.021097183970e+02, 8.050544077640e+02, 8.079178298640e+02, 8.107048506510e+02, 8.134199017810e+02, 8.160670314960e+02, 8.186499481600e+02, 8.211720577120e+02, 8.236364960250e+02, 8.260461570040e+02, 8.284037170810e+02, 8.307116566590e+02, 8.329722789450e+02, 8.351877265650e+02, 8.373599962530e+02, 8.394909518910e+02, 8.415823361140e+02, 8.436357806690e+02, 8.456528156830e+02, 8.476348779780e+02, 8.495833185540e+02, 8.514994093310e+02, 8.533843492360e+02, 8.552392697230e+02, 8.570652397680e+02, 8.588632704230e+02, 8.606343189490e+02, 8.623792925950e+02, 8.640990520450e+02, 8.657944145760e+02, 8.674661569520e+02, 8.691150180810e+02, 8.707417014530e+02, 8.723468773910e+02, 8.739311851240e+02, 8.754952346970e+02, 8.770396087390e+02, 8.785648641010e+02, 8.800715333690e+02, 8.815601262710e+02, 8.830311309820e+02, 8.844850153360e+02, 8.859222279550e+02, 8.873431993020e+02, 8.887483426590e+02, 8.901380550460e+02, 8.915127180690e+02, 8.928726987250e+02, 8.942183501490e+02, 8.955500123100e+02, 8.968680126750e+02, 8.981726668170e+02, 8.994642790000e+02, 9.007431427180e+02, 9.020095412100e+02, 9.032637479370e+02, 9.045060270390e+02, 9.057366337590e+02, 9.069558148450e+02, 9.081638089360e+02, 9.093608469110e+02, 9.105471522390e+02, 9.117229412900e+02, 9.128884236460e+02, 9.140438023820e+02, 9.151892743430e+02, 9.163250303990e+02, 9.174512556910e+02, 9.185681298630e+02, 9.196758272800e+02, 9.207745172420e+02, 9.218643641790e+02, 9.229455278420e+02, 9.240181634820e+02, 9.250824220220e+02, 9.261384502210e+02, 9.271863908270e+02, 9.282263827300e+02, 9.292585610970e+02, 9.302830575120e+02, 9.313000001030e+02, 9.323095136640e+02, 9.333117197740e+02, 9.343067369090e+02, 9.352946805490e+02, 9.362756632800e+02, 9.372497948950e+02, 9.382171824850e+02, 9.391779305300e+02, 9.401321409880e+02, 9.410799133720e+02, 9.420213448360e+02, 9.429565302470e+02, 9.438855622580e+02, 9.448085313810e+02, 9.457255260510e+02, 9.466366326930e+02, 9.475419357820e+02, 9.484415179060e+02, 9.493354598200e+02, 9.502238405010e+02, 9.511067372050e+02, 9.519842255160e+02, 9.528563793930e+02, 9.537232712200e+02, 9.545849718520e+02, 9.554415506560e+02, 9.562930755570e+02, 9.571396130760e+02, 9.579812283720e+02, 9.588179852770e+02, 9.596499463340e+02, 9.604771728350e+02, 9.612997248500e+02, 9.621176612640e+02, 9.629310398080e+02, 9.637399170880e+02, 9.645443486170e+02, 9.653443888430e+02, 9.661400911770e+02, 9.669315080180e+02, 9.677186907830e+02, 9.685016899260e+02, 9.692805549700e+02, 9.700553345250e+02, 9.708260763100e+02, 9.715928271830e+02, 9.723556331510e+02, 9.731145394010e+02, 9.738695903120e+02, 9.746208294830e+02, 9.753682997420e+02, 9.761120431730e+02, 9.768521011280e+02, 9.775885142470e+02, 9.783213224750e+02, 9.790505650750e+02, 9.797762806490e+02, 9.804985071470e+02, 9.812172818870e+02, 9.819326415670e+02, 9.826446222800e+02, 9.833532595270e+02, 9.840585882280e+02, 9.847606427400e+02, 9.854594568660e+02, 9.861550638650e+02, 9.868474964700e+02, 9.875367868940e+02, 9.882229668430e+02, 9.889060675270e+02, 9.895861196710e+02, 9.902631535240e+02, 9.909371988700e+02, 9.916082850370e+02, 9.922764409090e+02, 9.929416949300e+02, 9.936040751170e+02, 9.942636090690e+02, 9.949203239740e+02, 9.955742466170e+02, 9.962254033890e+02, 9.968738202940e+02, 9.975195229590e+02, 9.981625366380e+02, 9.988028862210e+02, 9.994405962440e+02, 1.000075690890e+03, 1.000708194000e+03, 1.001338129080e+03, 1.001965519300e+03, 1.002590387520e+03, 1.003212756260e+03, 1.003832647760e+03, 1.004450083920e+03, 1.005065086360e+03, 1.005677676410e+03, 1.006287875090e+03, 1.006895703150e+03, 1.007501181070e+03, 1.008104329050e+03, 1.008705167000e+03, 1.009303714600e+03, 1.009899991240e+03, 1.010494016100e+03, 1.011085808050e+03, 1.011675385770e+03, 1.012262767660e+03, 1.012847971910e+03, 1.013431016450e+03, 1.014011919020e+03, 1.014590697100e+03, 1.015167367960e+03, 1.015741948660e+03, 1.016314456040e+03, 1.016884906730e+03, 1.017453317160e+03, 1.018019703550e+03, 1.018584081930e+03, 1.019146468110e+03, 1.019706877750e+03, 1.020265326270e+03, 1.020821828950e+03, 1.021376400850e+03, 1.021929056880e+03, 1.022479811740e+03, 1.023028679990e+03, 1.023575675990e+03, 1.024120813950e+03, 1.024664107900e+03, 1.025205571720e+03, 1.025745219110e+03, 1.026283063620e+03, 1.026819118670e+03, 1.027353397470e+03, 1.027885913140e+03, 1.028416678600e+03, 1.028945706670e+03, 1.029473009970e+03, 1.029998601040e+03, 1.030522492230e+03, 1.031044695780e+03, 1.031565223770e+03, 1.032084088190e+03, 1.032601300840e+03, 1.033116873440e+03, 1.033630817560e+03, 1.034143144630e+03, 1.034653865990e+03, 1.035162992830e+03, 1.035670536240e+03, 1.036176507170e+03, 1.036680916460e+03, 1.037183774850e+03, 1.037685092950e+03, 1.038184881260e+03, 1.038683150180e+03, 1.039179909970e+03, 1.039675170840e+03, 1.040168942830e+03, 1.040661235920e+03, 1.041152059960e+03, 1.041641424730e+03, 1.042129339880e+03, 1.042615814980e+03, 1.043100859480e+03, 1.043584482770e+03, 1.044066694100e+03, 1.044547502680e+03, 1.045026917570e+03, 1.045504947790e+03, 1.045981602250e+03, 1.046456889750e+03, 1.046930819040e+03, 1.047403398770e+03, 1.047874637480e+03, 1.048344543680e+03, 1.048813125740e+03, 1.049280391990e+03, 1.049746350660e+03, 1.050211009910e+03, 1.050674377800e+03, 1.051136462350e+03, 1.051597271460e+03, 1.052056813000e+03, 1.052515094730e+03, 1.052972124350e+03, 1.053427909480e+03, 1.053882457690e+03, 1.054335776460e+03, 1.054787873200e+03, 1.055238755260e+03, 1.055688429920e+03, 1.056136904380e+03, 1.056584185790e+03, 1.057030281240e+03, 1.057475197730e+03, 1.057918942210e+03, 1.058361521570e+03, 1.058802942640e+03, 1.059243212180e+03, 1.059682336880e+03, 1.060120323410e+03, 1.060557178320e+03, 1.060992908160e+03, 1.061427519380e+03, 1.061861018400e+03, 1.062293411560e+03, 1.062724705170e+03, 1.063154905460e+03, 1.063584018630e+03, 1.064012050790e+03, 1.064439008050e+03, 1.064864896410e+03, 1.065289721850e+03, 1.065713490310e+03, 1.066136207640e+03, 1.066557879680e+03, 1.066978512200e+03, 1.067398110910e+03, 1.067816681490e+03, 1.068234229580e+03, 1.068650760740e+03, 1.069066280520e+03, 1.069480794400e+03, 1.069894307810e+03, 1.070306826160e+03, 1.070718354800e+03, 1.071128899020e+03, 1.071538464100e+03, 1.071947055240e+03, 1.072354677640e+03, 1.072761336410e+03, 1.073167036660e+03, 1.073571783430e+03, 1.073975581730e+03, 1.074378436540e+03, 1.074780352770e+03, 1.075181335320e+03, 1.075581389040e+03, 1.075980518740e+03, 1.076378729190e+03, 1.076776025130e+03, 1.077172411250e+03, 1.077567892220e+03, 1.077962472650e+03, 1.078356157140e+03, 1.078748950230e+03, 1.079140856450e+03, 1.079531880260e+03, 1.079922026120e+03, 1.080311298440e+03, 1.080699701590e+03, 1.081087239920e+03, 1.081473917730e+03, 1.081859739300e+03, 1.082244708880e+03, 1.082628830680e+03, 1.083012108870e+03, 1.083394547610e+03, 1.083776151010e+03, 1.084156923150e+03, 1.084536868100e+03, 1.084915989860e+03, 1.085294292450e+03, 1.085671779810e+03, 1.086048455890e+03, 1.086424324590e+03, 1.086799389790e+03, 1.087173655330e+03, 1.087547125040e+03, 1.087919802700e+03, 1.088291692080e+03, 1.088662796910e+03, 1.089033120910e+03, 1.089402667740e+03, 1.089771441080e+03, 1.090139444550e+03, 1.090506681740e+03, 1.090873156240e+03, 1.091238871590e+03, 1.091603831330e+03, 1.091968038950e+03, 1.092331497920e+03, 1.092694211700e+03, 1.093056183710e+03, 1.093417417350e+03, 1.093777916000e+03, 1.094137683020e+03, 1.094496721730e+03, 1.094855035440e+03, 1.095212627440e+03, 1.095569500980e+03, 1.095925659310e+03, 1.096281105640e+03, 1.096635843150e+03, 1.096989875040e+03, 1.097343204430e+03, 1.097695834470e+03, 1.098047768250e+03, 1.098399008870e+03, 1.098749559380e+03, 1.099099422830e+03, 1.099448602250e+03, 1.099797100630e+03, 1.100144920960e+03, 1.100492066190e+03, 1.100838539280e+03, 1.101184343130e+03, 1.101529480670e+03, 1.101873954760e+03, 1.102217768270e+03, 1.102560924050e+03, 1.102903424920e+03, 1.103245273690e+03, 1.103586473150e+03, 1.103927026070e+03, 1.104266935200e+03, 1.104606203280e+03, 1.104944833020e+03, 1.105282827120e+03, 1.105620188260e+03, 1.105956919110e+03, 1.106293022310e+03, 1.106628500490e+03, 1.106963356260e+03, 1.107297592230e+03, 1.107631210960e+03, 1.107964215020e+03, 1.108296606960e+03, 1.108628389310e+03, 1.108959564570e+03, 1.109290135260e+03, 1.109620103840e+03, 1.109949472780e+03, 1.110278244550e+03, 1.110606421560e+03, 1.110934006250e+03, 1.111261001010e+03, 1.111587408240e+03, 1.111913230310e+03, 1.112238469580e+03, 1.112563128390e+03, 1.112887209080e+03, 1.113210713970e+03, 1.113533645350e+03, 1.113856005510e+03, 1.114177796740e+03},
    {1.724177993840e+00, 5.258907021360e+00, 8.860339687320e+00, 1.253167581800e+01, 1.627636633270e+01, 2.009814054200e+01, 2.400103730620e+01, 2.798944073890e+01, 3.206812128460e+01, 3.624228318480e+01, 4.051761957920e+01, 4.490037678530e+01, 4.939742968040e+01, 5.401637059990e+01, 5.876561481190e+01, 6.365452647000e+01, 6.869357007580e+01, 7.389449399880e+01, 7.927055466550e+01, 8.483679287840e+01, 9.061037771030e+01, 9.661103907200e+01, 1.028616182140e+02, 1.093887774140e+02, 1.162239280660e+02, 1.234044639440e+02, 1.309754296280e+02, 1.389918239530e+02, 1.475218546590e+02, 1.566516614580e+02, 1.664923866920e+02, 1.771911572860e+02, 1.889489132920e+02, 2.020509659050e+02, 2.169230717480e+02, 2.342438352610e+02, 2.551984262620e+02, 2.821551926120e+02, 3.209679210840e+02, 3.919230978900e+02, 5.184281089970e+02, 5.836662523240e+02, 6.173094431390e+02, 6.399736992750e+02, 6.571552088230e+02, 6.710667821160e+02, 6.828082940190e+02, 6.930036397380e+02, 7.020402150300e+02, 7.101748441270e+02, 7.175864523370e+02, 7.244046571500e+02, 7.307263739250e+02, 7.366259918190e+02, 7.421618871910e+02, 7.473807461750e+02, 7.523205223670e+02, 7.570125144720e+02, 7.614828595580e+02, 7.657536281650e+02, 7.698436419950e+02, 7.737690943650e+02, 7.775440279340e+02, 7.811807074110e+02, 7.846899138910e+02, 7.880811798650e+02, 7.913629788020e+02, 7.945428795070e+02, 7.976276729070e+02, 8.006234770200e+02, 8.035358245140e+02, 8.063697362510e+02, 8.091297834460e+02, 8.118201405350e+02, 8.144446303760e+02, 8.170067631090e+02, 8.195097697170e+02, 8.219566311520e+02, 8.243501037160e+02, 8.266927412610e+02, 8.289869146960e+02, 8.312348291730e+02, 8.334385392850e+02, 8.355999625560e+02, 8.377208914310e+02, 8.398030039880e+02, 8.418478735130e+02, 8.438569770940e+02, 8.458317033460e+02, 8.477733593730e+02, 8.496831770580e+02, 8.515623187570e+02, 8.534118824570e+02, 8.552329064740e+02, 8.570263737210e+02, 8.587932156010e+02, 8.605343155730e+02, 8.622505123990e+02, 8.639426031370e+02, 8.656113458740e+02, 8.672574622490e+02, 8.688816397650e+02, 8.704845339300e+02, 8.720667702300e+02, 8.736289459440e+02, 8.751716318380e+02, 8.766953737210e+02, 8.782006938970e+02, 8.796880925090e+02, 8.811580487870e+02, 8.826110222120e+02, 8.840474536020e+02, 8.854677661170e+02, 8.868723662050e+02, 8.882616444800e+02, 8.896359765460e+02, 8.909957237670e+02, 8.923412339880e+02, 8.936728422100e+02, 8.949908712270e+02, 8.962956322170e+02, 8.975874253070e+02, 8.988665400930e+02, 9.001332561400e+02, 9.013878434440e+02, 9.026305628760e+02, 9.038616665900e+02, 9.050813984200e+02, 9.062899942460e+02, 9.074876823430e+02, 9.086746837100e+02, 9.098512123830e+02, 9.110174757330e+02, 9.121736747390e+02, 9.133200042600e+02, 9.144566532800e+02, 9.155838051540e+02, 9.167016378280e+02, 9.178103240570e+02, 9.189100316090e+02, 9.200009234590e+02, 9.210831579740e+02, 9.221568890900e+02, 9.232222664800e+02, 9.242794357090e+02, 9.253285383920e+02, 9.263697123380e+02, 9.274030916850e+02, 9.284288070360e+02, 9.294469855840e+02, 9.304577512310e+02, 9.314612247050e+02, 9.324575236710e+02, 9.334467628340e+02, 9.344290540400e+02, 9.354045063760e+02, 9.363732262590e+02, 9.373353175230e+02, 9.382908815100e+02, 9.392400171460e+02, 9.401828210210e+02, 9.411193874640e+02, 9.420498086130e+02, 9.429741744870e+02, 9.438925730500e+02, 9.448050902760e+02, 9.457118102090e+02, 9.466128150230e+02, 9.475081850760e+02, 9.483979989690e+02, 9.492823335930e+02, 9.501612641830e+02, 9.510348643680e+02, 9.519032062100e+02, 9.527663602600e+02, 9.536243955900e+02, 9.544773798460e+02, 9.553253792780e+02, 9.561684587870e+02, 9.570066819570e+02, 9.578401110950e+02, 9.586688072630e+02, 9.594928303150e+02, 9.603122389250e+02, 9.611270906230e+02, 9.619374418230e+02, 9.627433478530e+02, 9.635448629830e+02, 9.643420404550e+02, 9.651349325040e+02, 9.659235903910e+02, 9.667080644200e+02, 9.674884039690e+02, 9.682646575100e+02, 9.690368726320e+02, 9.698050960620e+02, 9.705693736870e+02, 9.713297505770e+02, 9.720862710010e+02, 9.728389784460e+02, 9.735879156410e+02, 9.743331245680e+02, 9.750746464860e+02, 9.758125219450e+02, 9.765467908000e+02, 9.772774922340e+02, 9.780046647660e+02, 9.787283462720e+02, 9.794485739950e+02, 9.801653845640e+02, 9.808788140040e+02, 9.815888977500e+02, 9.822956706640e+02, 9.829991670420e+02, 9.836994206300e+02, 9.843964646350e+02, 9.850903317370e+02, 9.857810541000e+02, 9.864686633830e+02, 9.871531907530e+02, 9.878346668890e+02, 9.885131220020e+02, 9.891885858360e+02, 9.898610876820e+02, 9.905306563870e+02, 9.911973203640e+02, 9.918611075970e+02, 9.925220456540e+02, 9.931801616950e+02, 9.938354824760e+02, 9.944880343640e+02, 9.951378433380e+02, 9.957849350000e+02, 9.964293345840e+02, 9.970710669590e+02, 9.977101566400e+02, 9.983466277920e+02, 9.989805042390e+02, 9.996118094700e+02, 1.000240566640e+03, 1.000866798600e+03, 1.001490527850e+03, 1.002111776610e+03, 1.002730566780e+03, 1.003346919970e+03, 1.003960857500e+03, 1.004572400370e+03, 1.005181569340e+03, 1.005788384870e+03, 1.006392867140e+03, 1.006995036080e+03, 1.007594911340e+03, 1.008192512340e+03, 1.008787858210e+03, 1.009380967860e+03, 1.009971859940e+03, 1.010560552850e+03, 1.011147064780e+03, 1.011731413680e+03, 1.012313617250e+03, 1.012893692990e+03, 1.013471658170e+03, 1.014047529850e+03, 1.014621324870e+03, 1.015193059870e+03, 1.015762751260e+03, 1.016330415280e+03, 1.016896067950e+03, 1.017459725100e+03, 1.018021402370e+03, 1.018581115200e+03, 1.019138878870e+03, 1.019694708450e+03, 1.020248618830e+03, 1.020800624750e+03, 1.021350740750e+03, 1.021898981200e+03, 1.022445360320e+03, 1.022989892140e+03, 1.023532590540e+03, 1.024073469240e+03, 1.024612541800e+03, 1.025149821620e+03, 1.025685321950e+03, 1.026219055890e+03, 1.026751036390e+03, 1.027281276240e+03, 1.027809788110e+03, 1.028336584520e+03, 1.028861677840e+03, 1.029385080310e+03, 1.029906804040e+03, 1.030426860990e+03, 1.030945263000e+03, 1.031462021780e+03, 1.031977148910e+03, 1.032490655850e+03, 1.033002553920e+03, 1.033512854340e+03, 1.034021568190e+03, 1.034528706440e+03, 1.035034279940e+03, 1.035538299440e+03, 1.036040775550e+03, 1.036541718790e+03, 1.037041139560e+03, 1.037539048140e+03, 1.038035454730e+03, 1.038530369410e+03, 1.039023802140e+03, 1.039515762800e+03, 1.040006261170e+03, 1.040495306910e+03, 1.040982909600e+03, 1.041469078710e+03, 1.041953823610e+03, 1.042437153610e+03, 1.042919077870e+03, 1.043399605510e+03, 1.043878745540e+03, 1.044356506870e+03, 1.044832898330e+03, 1.045307928670e+03, 1.045781606540e+03, 1.046253940520e+03, 1.046724939090e+03, 1.047194610660e+03, 1.047662963570e+03, 1.048130006040e+03, 1.048595746250e+03, 1.049060192280e+03, 1.049523352140e+03, 1.049985233760e+03, 1.050445845000e+03, 1.050905193650e+03, 1.051363287420e+03, 1.051820133930e+03, 1.052275740760e+03, 1.052730115400e+03, 1.053183265270e+03, 1.053635197740e+03, 1.054085920080e+03, 1.054535439520e+03, 1.054983763220e+03, 1.055430898260e+03, 1.055876851670e+03, 1.056321630410e+03, 1.056765241370e+03, 1.057207691390e+03, 1.057648987240e+03, 1.058089135630e+03, 1.058528143220e+03, 1.058966016600e+03, 1.059402762300e+03, 1.059838386800e+03, 1.060272896510e+03, 1.060706297810e+03, 1.061138596990e+03, 1.061569800310e+03, 1.061999913960e+03, 1.062428944090e+03, 1.062856896780e+03, 1.063283778070e+03, 1.063709593950e+03, 1.064134350340e+03, 1.064558053140e+03, 1.064980708170e+03, 1.065402321210e+03, 1.065822898000e+03, 1.066242444220e+03, 1.066660965510e+03, 1.067078467450e+03, 1.067494955590e+03, 1.067910435430e+03, 1.068324912410e+03, 1.068738391940e+03, 1.069150879370e+03, 1.069562380030e+03, 1.069972899180e+03, 1.070382442060e+03, 1.070791013840e+03, 1.071198619670e+03, 1.071605264650e+03, 1.072010953840e+03, 1.072415692260e+03, 1.072819484890e+03, 1.073222336660e+03, 1.073624252470e+03, 1.074025237180e+03, 1.074425295610e+03, 1.074824432550e+03, 1.075222652730e+03, 1.075619960860e+03, 1.076016361600e+03, 1.076411859610e+03, 1.076806459450e+03, 1.077200165710e+03, 1.077592982890e+03, 1.077984915500e+03, 1.078375967980e+03, 1.078766144750e+03, 1.079155450190e+03, 1.079543888670e+03, 1.079931464490e+03, 1.080318181940e+03, 1.080704045280e+03, 1.081089058720e+03, 1.081473226450e+03, 1.081856552620e+03, 1.082239041370e+03, 1.082620696780e+03, 1.083001522910e+03, 1.083381523800e+03, 1.083760703450e+03, 1.084139065830e+03, 1.084516614890e+03, 1.084893354530e+03, 1.085269288640e+03, 1.085644421080e+03, 1.086018755670e+03, 1.086392296220e+03, 1.086765046480e+03, 1.087137010220e+03, 1.087508191130e+03, 1.087878592920e+03, 1.088248219240e+03, 1.088617073730e+03, 1.088985160000e+03, 1.089352481630e+03, 1.089719042180e+03, 1.090084845190e+03, 1.090449894160e+03, 1.090814192570e+03, 1.091177743880e+03, 1.091540551520e+03, 1.091902618910e+03, 1.092263949420e+03, 1.092624546420e+03, 1.092984413250e+03, 1.093343553210e+03, 1.093701969610e+03, 1.094059665700e+03, 1.094416644730e+03, 1.094772909930e+03, 1.095128464490e+03, 1.095483311600e+03, 1.095837454400e+03, 1.096190896050e+03, 1.096543639630e+03, 1.096895688260e+03, 1.097247045000e+03, 1.097597712900e+03, 1.097947694990e+03, 1.098296994270e+03, 1.098645613750e+03, 1.098993556370e+03, 1.099340825110e+03, 1.099687422870e+03, 1.100033352570e+03, 1.100378617110e+03, 1.100723219350e+03, 1.101067162130e+03, 1.101410448310e+03, 1.101753080680e+03, 1.102095062050e+03, 1.102436395180e+03, 1.102777082850e+03, 1.103117127780e+03, 1.103456532710e+03, 1.103795300330e+03, 1.104133433340e+03, 1.104470934400e+03, 1.104807806170e+03, 1.105144051280e+03, 1.105479672340e+03, 1.105814671970e+03, 1.106149052750e+03, 1.106482817240e+03, 1.106815968000e+03, 1.107148507550e+03, 1.107480438440e+03, 1.107811763140e+03, 1.108142484160e+03, 1.108472603960e+03, 1.108802125000e+03, 1.109131049730e+03, 1.109459380560e+03, 1.109787119910e+03, 1.110114270180e+03, 1.110440833730e+03, 1.110766812950e+03, 1.111092210180e+03, 1.111417027750e+03, 1.111741268000e+03, 1.112064933220e+03},
    {1.718407308420e+00, 5.240768218960e+00, 8.828827488420e+00, 1.248569106090e+01, 1.621470474170e+01, 2.001947980750e+01, 2.390392216960e+01, 2.787226570400e+01, 3.192911049510e+01, 3.607946689840e+01, 4.032880653220e+01, 4.468312156640e+01, 4.914899400230e+01, 5.373367706420e+01, 5.844519136530e+01, 6.329243923490e+01, 6.828534153270e+01, 7.343500254930e+01, 7.875391029190e+01, 8.425618179050e+01, 8.995786628610e+01, 9.587732368760e+01, 1.020357021400e+02, 1.084575478890e+02, 1.151715944060e+02, 1.222117985130e+02, 1.296187231580e+02, 1.374414169420e+02, 1.457400223210e+02, 1.545894815620e+02, 1.640849479460e+02, 1.743499419120e+02, 1.855491150150e+02, 1.979091464360e+02, 2.117548946020e+02, 2.275764055910e+02, 2.461645652130e+02, 2.689192237410e+02, 2.986626973150e+02, 3.421985088650e+02, 4.186786988410e+02, 5.192142147410e+02, 5.763569480140e+02, 6.092225459320e+02, 6.320494556970e+02, 6.495506606810e+02, 6.637858793670e+02, 6.758225539870e+02, 6.862801563860e+02, 6.955486823150e+02, 7.038890365110e+02, 7.114841276820e+02, 7.184670413090e+02, 7.249375903040e+02, 7.309725470880e+02, 7.366322392540e+02, 7.419649518420e+02, 7.470099536280e+02, 7.517996309880e+02, 7.563610261970e+02, 7.607169682660e+02, 7.648869188440e+02, 7.688876149650e+02, 7.727335644120e+02, 7.764374324330e+02, 7.800103472480e+02, 7.834621440230e+02, 7.868015616580e+02, 7.900364029990e+02, 7.931736663950e+02, 7.962196545650e+02, 7.991800653900e+02, 8.020600681160e+02, 8.048643677550e+02, 8.075972598140e+02, 8.102626770760e+02, 8.128642297950e+02, 8.154052404010e+02, 8.178887735970e+02, 8.203176625820e+02, 8.226945319800e+02, 8.250218179780e+02, 8.273017860570e+02, 8.295365466820e+02, 8.317280692050e+02, 8.338781942390e+02, 8.359886446950e+02, 8.380610356540e+02, 8.400968832220e+02, 8.420976124910e+02, 8.440645647090e+02, 8.459990037610e+02, 8.479021220300e+02, 8.497750457120e+02, 8.516188396520e+02, 8.534345117360e+02, 8.552230169020e+02, 8.569852608050e+02, 8.587221031690e+02, 8.604343608540e+02, 8.621228106790e+02, 8.637881920110e+02, 8.654312091510e+02, 8.670525335300e+02, 8.686528057420e+02, 8.702326374100e+02, 8.717926129310e+02, 8.733332910720e+02, 8.748552064670e+02, 8.763588709980e+02, 8.778447750770e+02, 8.793133888490e+02, 8.807651632970e+02, 8.822005312870e+02, 8.836199085350e+02, 8.850236945090e+02, 8.864122732790e+02, 8.877860143070e+02, 8.891452731870e+02, 8.904903923410e+02, 8.918217016710e+02, 8.931395191690e+02, 8.944441514940e+02, 8.957358945120e+02, 8.970150338030e+02, 8.982818451420e+02, 8.995365949470e+02, 9.007795407090e+02, 9.020109313910e+02, 9.032310078050e+02, 9.044400029730e+02, 9.056381424670e+02, 9.068256447230e+02, 9.080027213530e+02, 9.091695774230e+02, 9.103264117340e+02, 9.114734170700e+02, 9.126107804550e+02, 9.137386833740e+02, 9.148573019990e+02, 9.159668074020e+02, 9.170673657470e+02, 9.181591384840e+02, 9.192422825310e+02, 9.203169504410e+02, 9.213832905730e+02, 9.224414472380e+02, 9.234915608590e+02, 9.245337681060e+02, 9.255682020300e+02, 9.265949922010e+02, 9.276142648220e+02, 9.286261428530e+02, 9.296307461220e+02, 9.306281914330e+02, 9.316185926700e+02, 9.326020608940e+02, 9.335787044420e+02, 9.345486290110e+02, 9.355119377530e+02, 9.364687313510e+02, 9.374191081030e+02, 9.383631639980e+02, 9.393009927890e+02, 9.402326860650e+02, 9.411583333160e+02, 9.420780220010e+02, 9.429918376100e+02, 9.438998637230e+02, 9.448021820680e+02, 9.456988725790e+02, 9.465900134460e+02, 9.474756811700e+02, 9.483559506100e+02, 9.492308950320e+02, 9.501005861550e+02, 9.509650941960e+02, 9.518244879110e+02, 9.526788346370e+02, 9.535282003320e+02, 9.543726496160e+02, 9.552122458020e+02, 9.560470509370e+02, 9.568771258350e+02, 9.577025301070e+02, 9.585233222000e+02, 9.593395594210e+02, 9.601512979740e+02, 9.609585929810e+02, 9.617614985170e+02, 9.625600676340e+02, 9.633543523890e+02, 9.641444038650e+02, 9.649302722030e+02, 9.657120066180e+02, 9.664896554290e+02, 9.672632660740e+02, 9.680328851400e+02, 9.687985583780e+02, 9.695603307230e+02, 9.703182463180e+02, 9.710723485300e+02, 9.718226799700e+02, 9.725692825090e+02, 9.733121972970e+02, 9.740514647790e+02, 9.747871247130e+02, 9.755192161840e+02, 9.762477776200e+02, 9.769728468070e+02, 9.776944609060e+02, 9.784126564630e+02, 9.791274694230e+02, 9.798389351490e+02, 9.805470884280e+02, 9.812519634880e+02, 9.819535940080e+02, 9.826520131310e+02, 9.833472534760e+02, 9.840393471490e+02, 9.847283257510e+02, 9.854142203950e+02, 9.860970617110e+02, 9.867768798570e+02, 9.874537045300e+02, 9.881275649770e+02, 9.887984900000e+02, 9.894665079700e+02, 9.901316468310e+02, 9.907939341150e+02, 9.914533969420e+02, 9.921100620360e+02, 9.927639557280e+02, 9.934151039660e+02, 9.940635323230e+02, 9.947092660010e+02, 9.953523298440e+02, 9.959927483400e+02, 9.966305456280e+02, 9.972657455090e+02, 9.978983714480e+02, 9.985284465840e+02, 9.991559937310e+02, 9.997810353910e+02, 1.000403593750e+03, 1.001023690710e+03, 1.001641347840e+03, 1.002256586450e+03, 1.002869427540e+03, 1.003479891840e+03, 1.004087999810e+03, 1.004693771610e+03, 1.005297227160e+03, 1.005898386100e+03, 1.006497267840e+03, 1.007093891500e+03, 1.007688275980e+03, 1.008280439920e+03, 1.008870401730e+03, 1.009458179570e+03, 1.010043791390e+03, 1.010627254900e+03, 1.011208587580e+03, 1.011787806700e+03, 1.012364929320e+03, 1.012939972270e+03, 1.013512952170e+03, 1.014083885470e+03, 1.014652788370e+03, 1.015219676910e+03, 1.015784566910e+03, 1.016347474010e+03, 1.016908413660e+03, 1.017467401120e+03, 1.018024451470e+03, 1.018579579610e+03, 1.019132800270e+03, 1.019684127990e+03, 1.020233577150e+03, 1.020781161960e+03, 1.021326896460e+03, 1.021870794540e+03, 1.022412869910e+03, 1.022953136140e+03, 1.023491606640e+03, 1.024028294640e+03, 1.024563213270e+03, 1.025096375460e+03, 1.025627794030e+03, 1.026157481640e+03, 1.026685450810e+03, 1.027211713930e+03, 1.027736283230e+03, 1.028259170830e+03, 1.028780388700e+03, 1.029299948680e+03, 1.029817862490e+03, 1.030334141720e+03, 1.030848797820e+03, 1.031361842130e+03, 1.031873285870e+03, 1.032383140130e+03, 1.032891415890e+03, 1.033398124010e+03, 1.033903275220e+03, 1.034406880170e+03, 1.034908949360e+03, 1.035409493210e+03, 1.035908522020e+03, 1.036406045980e+03, 1.036902075180e+03, 1.037396619590e+03, 1.037889689100e+03, 1.038381293490e+03, 1.038871442440e+03, 1.039360145520e+03, 1.039847412220e+03, 1.040333251920e+03, 1.040817673920e+03, 1.041300687430e+03, 1.041782301530e+03, 1.042262525260e+03, 1.042741367540e+03, 1.043218837200e+03, 1.043694943010e+03, 1.044169693620e+03, 1.044643097610e+03, 1.045115163490e+03, 1.045585899670e+03, 1.046055314480e+03, 1.046523416170e+03, 1.046990212930e+03, 1.047455712830e+03, 1.047919923900e+03, 1.048382854090e+03, 1.048844511250e+03, 1.049304903180e+03, 1.049764037600e+03, 1.050221922160e+03, 1.050678564420e+03, 1.051133971890e+03, 1.051588152000e+03, 1.052041112120e+03, 1.052492859550e+03, 1.052943401510e+03, 1.053392745170e+03, 1.053840897620e+03, 1.054287865910e+03, 1.054733656990e+03, 1.055178277770e+03, 1.055621735100e+03, 1.056064035760e+03, 1.056505186460e+03, 1.056945193870e+03, 1.057384064600e+03, 1.057821805180e+03, 1.058258422100e+03, 1.058693921780e+03, 1.059128310600e+03, 1.059561594870e+03, 1.059993780850e+03, 1.060424874750e+03, 1.060854882710e+03, 1.061283810840e+03, 1.061711665180e+03, 1.062138451730e+03, 1.062564176410e+03, 1.062988845140e+03, 1.063412463730e+03, 1.063835038000e+03, 1.064256573670e+03, 1.064677076440e+03, 1.065096551960e+03, 1.065515005820e+03, 1.065932443570e+03, 1.066348870720e+03, 1.066764292720e+03, 1.067178714990e+03, 1.067592142900e+03, 1.068004581760e+03, 1.068416036870e+03, 1.068826513440e+03, 1.069236016690e+03, 1.069644551760e+03, 1.070052123750e+03, 1.070458737740e+03, 1.070864398750e+03, 1.071269111770e+03, 1.071672881750e+03, 1.072075713580e+03, 1.072477612150e+03, 1.072878582260e+03, 1.073278628720e+03, 1.073677756280e+03, 1.074075969640e+03, 1.074473273490e+03, 1.074869672470e+03, 1.075265171180e+03, 1.075659774180e+03, 1.076053486020e+03, 1.076446311170e+03, 1.076838254120e+03, 1.077229319280e+03, 1.077619511040e+03, 1.078008833770e+03, 1.078397291790e+03, 1.078784889390e+03, 1.079171630840e+03, 1.079557520350e+03, 1.079942562130e+03, 1.080326760340e+03, 1.080710119110e+03, 1.081092642540e+03, 1.081474334700e+03, 1.081855199630e+03, 1.082235241340e+03, 1.082614463810e+03, 1.082992870980e+03, 1.083370466780e+03, 1.083747255100e+03, 1.084123239810e+03, 1.084498424730e+03, 1.084872813670e+03, 1.085246410410e+03, 1.085619218700e+03, 1.085991242270e+03, 1.086362484810e+03, 1.086732949980e+03, 1.087102641440e+03, 1.087471562800e+03, 1.087839717650e+03, 1.088207109560e+03, 1.088573742050e+03, 1.088939618660e+03, 1.089304742860e+03, 1.089669118130e+03, 1.090032747900e+03, 1.090395635580e+03, 1.090757784570e+03, 1.091119198240e+03, 1.091479879920e+03, 1.091839832950e+03, 1.092199060610e+03, 1.092557566190e+03, 1.092915352920e+03, 1.093272424060e+03, 1.093628782790e+03, 1.093984432300e+03, 1.094339375760e+03, 1.094693616310e+03, 1.095047157060e+03, 1.095400001120e+03, 1.095752151570e+03, 1.096103611450e+03, 1.096454383810e+03, 1.096804471660e+03, 1.097153877990e+03, 1.097502605780e+03, 1.097850657980e+03, 1.098198037530e+03, 1.098544747340e+03, 1.098890790320e+03, 1.099236169330e+03, 1.099580887230e+03, 1.099924946860e+03, 1.100268351040e+03, 1.100611102570e+03, 1.100953204240e+03, 1.101294658810e+03, 1.101635469020e+03, 1.101975637610e+03, 1.102315167280e+03, 1.102654060720e+03, 1.102992320620e+03, 1.103329949630e+03, 1.103666950390e+03, 1.104003325530e+03, 1.104339077650e+03, 1.104674209340e+03, 1.105008723190e+03, 1.105342621730e+03, 1.105675907520e+03, 1.106008583090e+03, 1.106340650930e+03, 1.106672113550e+03, 1.107002973410e+03, 1.107333232990e+03, 1.107662894730e+03, 1.107991961060e+03, 1.108320434390e+03, 1.108648317130e+03, 1.108975611660e+03, 1.109302320360e+03, 1.109628445570e+03, 1.109953989650e+03},
    {1.712675719540e+00, 5.222759917330e+00, 8.797555872360e+00, 1.244007864010e+01, 1.615357243350e+01, 1.994153465280e+01, 2.380774318380e+01, 2.775628751950e+01, 3.179160437220e+01, 3.591851858920e+01, 4.014229035730e+01, 4.446866991070e+01, 4.890396123450e+01, 5.345509662570e+01, 5.812972443390e+01, 6.293631291830e+01, 6.788427394990e+01, 7.298411134600e+01, 7.824760003720e+01, 8.368800417690e+01, 8.932034492690e+01, 9.516173228090e+01, 1.012317804090e+02, 1.075531333090e+02, 1.141521381760e+02, 1.210597196230e+02, 1.283125316180e+02, 1.359545007130e+02, 1.440389322220e+02, 1.526314457270e+02, 1.618141656130e+02, 1.716918703880e+02, 1.824013106170e+02, 1.941258723260e+02, 2.071197192660e+02, 2.217497805120e+02, 2.385738603680e+02, 2.584985531000e+02, 2.831322486360e+02, 3.156616884150e+02, 3.631003229490e+02, 4.387209674090e+02, 5.194004111190e+02, 5.700678672200e+02, 6.018295870400e+02, 6.245807893890e+02, 6.422638468060e+02, 6.567402462890e+02, 6.690197567090e+02, 6.797046721250e+02, 6.891807994180e+02, 6.977092754730e+02, 7.054746967780e+02, 7.126121928510e+02, 7.192235632060e+02, 7.253873655340e+02, 7.311654770660e+02, 7.366075063040e+02, 7.417538444190e+02, 7.466378278340e+02, 7.512873038360e+02, 7.557257855210e+02, 7.599733181860e+02, 7.640471391560e+02, 7.679621872170e+02, 7.717315008820e+02, 7.753665333340e+02, 7.788774041280e+02, 7.822731023100e+02, 7.855616518350e+02, 7.887502474020e+02, 7.918453668840e+02, 7.948528650560e+02, 7.977780522690e+02, 8.006257609050e+02, 8.034004018510e+02, 8.061060127530e+02, 8.087462994540e+02, 8.113246717750e+02, 8.138442745240e+02, 8.163080145140e+02, 8.187185841740e+02, 8.210784822820e+02, 8.233900322160e+02, 8.256553980990e+02, 8.278765991030e+02, 8.300555221750e+02, 8.321939333900e+02, 8.342934880960e+02, 8.363557400230e+02, 8.383821494620e+02, 8.403740906360e+02, 8.423328583550e+02, 8.442596740470e+02, 8.461556912100e+02, 8.480220003820e+02, 8.498596336540e+02, 8.516695687930e+02, 8.534527329990e+02, 8.552100063470e+02, 8.569422249460e+02, 8.586501838310e+02, 8.603346396190e+02, 8.619963129660e+02, 8.636358908160e+02, 8.652540284870e+02, 8.668513515960e+02, 8.684284578370e+02, 8.699859186320e+02, 8.715242806590e+02, 8.730440672710e+02, 8.745457798180e+02, 8.760298988720e+02, 8.774968853680e+02, 8.789471816760e+02, 8.803812125890e+02, 8.817993862530e+02, 8.832020950380e+02, 8.845897163480e+02, 8.859626133790e+02, 8.873211358380e+02, 8.886656206040e+02, 8.899963923650e+02, 8.913137641970e+02, 8.926180381260e+02, 8.939095056490e+02, 8.951884482220e+02, 8.964551377230e+02, 8.977098368930e+02, 8.989527997410e+02, 9.001842719390e+02, 9.014044911840e+02, 9.026136875500e+02, 9.038120838110e+02, 9.049998957560e+02, 9.061773324840e+02, 9.073445966790e+02, 9.085018848770e+02, 9.096493877160e+02, 9.107872901770e+02, 9.119157718040e+02, 9.130350069270e+02, 9.141451648590e+02, 9.152464100970e+02, 9.163389025000e+02, 9.174227974700e+02, 9.184982461200e+02, 9.195653954320e+02, 9.206243884070e+02, 9.216753642160e+02, 9.227184583360e+02, 9.237538026820e+02, 9.247815257350e+02, 9.258017526590e+02, 9.268146054230e+02, 9.278202029080e+02, 9.288186610100e+02, 9.298100927490e+02, 9.307946083560e+02, 9.317723153760e+02, 9.327433187480e+02, 9.337077208960e+02, 9.346656218090e+02, 9.356171191200e+02, 9.365623081790e+02, 9.375012821300e+02, 9.384341319740e+02, 9.393609466410e+02, 9.402818130520e+02, 9.411968161800e+02, 9.421060391110e+02, 9.430095630970e+02, 9.439074676160e+02, 9.447998304230e+02, 9.456867275970e+02, 9.465682335940e+02, 9.474444212950e+02, 9.483153620450e+02, 9.491811257040e+02, 9.500417806860e+02, 9.508973939990e+02, 9.517480312840e+02, 9.525937568550e+02, 9.534346337340e+02, 9.542707236860e+02, 9.551020872530e+02, 9.559287837890e+02, 9.567508714890e+02, 9.575684074210e+02, 9.583814475550e+02, 9.591900467920e+02, 9.599942589930e+02, 9.607941370040e+02, 9.615897326820e+02, 9.623810969210e+02, 9.631682796770e+02, 9.639513299900e+02, 9.647302960060e+02, 9.655052250030e+02, 9.662761634070e+02, 9.670431568180e+02, 9.678062500270e+02, 9.685654870340e+02, 9.693209110720e+02, 9.700725646210e+02, 9.708204894260e+02, 9.715647265170e+02, 9.723053162220e+02, 9.730422981870e+02, 9.737757113900e+02, 9.745055941540e+02, 9.752319841670e+02, 9.759549184920e+02, 9.766744335810e+02, 9.773905652940e+02, 9.781033489040e+02, 9.788128191180e+02, 9.795190100830e+02, 9.802219554020e+02, 9.809216881450e+02, 9.816182408590e+02, 9.823116455820e+02, 9.830019338510e+02, 9.836891367140e+02, 9.843732847400e+02, 9.850544080300e+02, 9.857325362250e+02, 9.864076985170e+02, 9.870799236580e+02, 9.877492399660e+02, 9.884156753410e+02, 9.890792572660e+02, 9.897400128180e+02, 9.903979686780e+02, 9.910531511370e+02, 9.917055861050e+02, 9.923552991150e+02, 9.930023153350e+02, 9.936466595730e+02, 9.942883562850e+02, 9.949274295790e+02, 9.955639032240e+02, 9.961978006570e+02, 9.968291449880e+02, 9.974579590070e+02, 9.980842651890e+02, 9.987080857000e+02, 9.993294424040e+02, 9.999483568680e+02, 1.000564850370e+03, 1.001178943890e+03, 1.001790658140e+03, 1.002400013560e+03, 1.003007030290e+03, 1.003611728250e+03, 1.004214127050e+03, 1.004814246080e+03, 1.005412104470e+03, 1.006007721070e+03, 1.006601114540e+03, 1.007192303260e+03, 1.007781305400e+03, 1.008368138870e+03, 1.008952821380e+03, 1.009535370420e+03, 1.010115803230e+03, 1.010694136860e+03, 1.011270388150e+03, 1.011844573710e+03, 1.012416709970e+03, 1.012986813130e+03, 1.013554899230e+03, 1.014120984080e+03, 1.014685083320e+03, 1.015247212390e+03, 1.015807386550e+03, 1.016365620880e+03, 1.016921930270e+03, 1.017476329440e+03, 1.018028832950e+03, 1.018579455170e+03, 1.019128210300e+03, 1.019675112390e+03, 1.020220175330e+03, 1.020763412820e+03, 1.021304838430e+03, 1.021844465560e+03, 1.022382307480e+03, 1.022918377280e+03, 1.023452687910e+03, 1.023985252200e+03, 1.024516082790e+03, 1.025045192210e+03, 1.025572592860e+03, 1.026098296960e+03, 1.026622316640e+03, 1.027144663880e+03, 1.027665350510e+03, 1.028184388260e+03, 1.028701788710e+03, 1.029217563330e+03, 1.029731723470e+03, 1.030244280330e+03, 1.030755245020e+03, 1.031264628520e+03, 1.031772441700e+03, 1.032278695290e+03, 1.032783399930e+03, 1.033286566160e+03, 1.033788204380e+03, 1.034288324890e+03, 1.034786937900e+03, 1.035284053490e+03, 1.035779681660e+03, 1.036273832290e+03, 1.036766515160e+03, 1.037257739950e+03, 1.037747516270e+03, 1.038235853580e+03, 1.038722761290e+03, 1.039208248700e+03, 1.039692325000e+03, 1.040174999320e+03, 1.040656280680e+03, 1.041136178010e+03, 1.041614700150e+03, 1.042091855860e+03, 1.042567653830e+03, 1.043042102630e+03, 1.043515210760e+03, 1.043986986660e+03, 1.044457438650e+03, 1.044926575010e+03, 1.045394403900e+03, 1.045860933440e+03, 1.046326171640e+03, 1.046790126460e+03, 1.047252805770e+03, 1.047714217360e+03, 1.048174368960e+03, 1.048633268230e+03, 1.049090922750e+03, 1.049547340030e+03, 1.050002527500e+03, 1.050456492540e+03, 1.050909242450e+03, 1.051360784480e+03, 1.051811125790e+03, 1.052260273480e+03, 1.052708234610e+03, 1.053155016130e+03, 1.053600624970e+03, 1.054045067980e+03, 1.054488351940e+03, 1.054930483590e+03, 1.055371469590e+03, 1.055811316550e+03, 1.056250031030e+03, 1.056687619510e+03, 1.057124088430e+03, 1.057559444170e+03, 1.057993693050e+03, 1.058426841330e+03, 1.058858895240e+03, 1.059289860930e+03, 1.059719744490e+03, 1.060148552000e+03, 1.060576289450e+03, 1.061002962780e+03, 1.061428577890e+03, 1.061853140640e+03, 1.062276656820e+03, 1.062699132180e+03, 1.063120572410e+03, 1.063540983180e+03, 1.063960370090e+03, 1.064378738690e+03, 1.064796094500e+03, 1.065212442980e+03, 1.065627789560e+03, 1.066042139600e+03, 1.066455498440e+03, 1.066867871370e+03, 1.067279263620e+03, 1.067689680410e+03, 1.068099126880e+03, 1.068507608160e+03, 1.068915129320e+03, 1.069321695390e+03, 1.069727311380e+03, 1.070131982220e+03, 1.070535712830e+03, 1.070938508090e+03, 1.071340372840e+03, 1.071741311860e+03, 1.072141329930e+03, 1.072540431750e+03, 1.072938622010e+03, 1.073335905370e+03, 1.073732286430e+03, 1.074127769760e+03, 1.074522359910e+03, 1.074916061370e+03, 1.075308878620e+03, 1.075700816090e+03, 1.076091878180e+03, 1.076482069260e+03, 1.076871393660e+03, 1.077259855670e+03, 1.077647459570e+03, 1.078034209590e+03, 1.078420109930e+03, 1.078805164760e+03, 1.079189378210e+03, 1.079572754410e+03, 1.079955297420e+03, 1.080337011290e+03, 1.080717900040e+03, 1.081097967640e+03, 1.081477218070e+03, 1.081855655240e+03, 1.082233283050e+03, 1.082610105370e+03, 1.082986126050e+03, 1.083361348900e+03, 1.083735777690e+03, 1.084109416190e+03, 1.084482268130e+03, 1.084854337220e+03, 1.085225627110e+03, 1.085596141470e+03, 1.085965883920e+03, 1.086334858060e+03, 1.086703067450e+03, 1.087070515650e+03, 1.087437206170e+03, 1.087803142510e+03, 1.088168328130e+03, 1.088532766490e+03, 1.088896461010e+03, 1.089259415090e+03, 1.089621632090e+03, 1.089983115370e+03, 1.090343868250e+03, 1.090703894040e+03, 1.091063196020e+03, 1.091421777450e+03, 1.091779641560e+03, 1.092136791570e+03, 1.092493230660e+03, 1.092848962000e+03, 1.093203988750e+03, 1.093558314020e+03, 1.093911940920e+03, 1.094264872540e+03, 1.094617111930e+03, 1.094968662140e+03, 1.095319526190e+03, 1.095669707070e+03, 1.096019207780e+03, 1.096368031260e+03, 1.096716180470e+03, 1.097063658320e+03, 1.097410467700e+03, 1.097756611510e+03, 1.098102092610e+03, 1.098446913840e+03, 1.098791078030e+03, 1.099134587980e+03, 1.099477446480e+03, 1.099819656300e+03, 1.100161220200e+03, 1.100502140900e+03, 1.100842421120e+03, 1.101182063570e+03, 1.101521070910e+03, 1.101859445830e+03, 1.102197190950e+03, 1.102534308920e+03, 1.102870802340e+03, 1.103206673820e+03, 1.103541925920e+03, 1.103876561210e+03, 1.104210582240e+03, 1.104543991540e+03, 1.104876791620e+03, 1.105208984980e+03, 1.105540574110e+03, 1.105871561470e+03, 1.106201949510e+03, 1.106531740660e+03, 1.106860937360e+03, 1.107189542000e+03, 1.107517556970e+03, 1.107844984660e+03},
    {1.706982822400e+00, 5.204880626420e+00, 8.766521833350e+00, 1.239483349610e+01, 1.609296162250e+01, 1.986429372520e+01, 2.371248437720e+01, 2.764148427570e+01, 3.165557341230e+01, 3.575939907510e+01, 3.995801956250e+01, 4.425695468310e+01, 4.866224435970e+01, 5.318051697010e+01, 5.781906945370e+01, 6.258596172930e+01, 6.749012864310e+01, 7.254151354290e+01, 7.775122875190e+01, 8.313174978040e+01, 8.869715224510e+01, 9.446340339190e+01, 1.004487241770e+02, 1.066740436060e+02, 1.131635752290e+02, 1.199455576770e+02, 1.270532188510e+02, 1.345260502740e+02, 1.424115197080e+02, 1.507674162370e+02, 1.596651298650e+02, 1.691943495340e+02, 1.794699810280e+02, 1.906426642390e+02, 2.029153712950e+02, 2.165707833340e+02, 2.320188750360e+02, 2.498849016420e+02, 2.711839129120e+02, 2.976908118130e+02, 3.327340414720e+02, 3.825496339870e+02, 4.527343944500e+02, 5.192837985630e+02, 5.646317202050e+02, 5.951034214990e+02, 6.175853093870e+02, 6.353203743550e+02, 6.499542289760e+02, 6.624210530910e+02, 6.732950907060e+02, 6.829516467800e+02, 6.916483168780e+02, 6.995690440480e+02, 7.068495017360e+02, 7.135924893810e+02, 7.198776866470e+02, 7.257680658840e+02, 7.313142420680e+02, 7.365575039590e+02, 7.415319759380e+02, 7.462661915150e+02, 7.507842593690e+02, 7.551067414160e+02, 7.592513236660e+02, 7.632333355360e+02, 7.670661567130e+02, 7.707615394500e+02, 7.743298664970e+02, 7.777803594710e+02, 7.811212486730e+02, 7.843599126300e+02, 7.875029936260e+02, 7.905564940410e+02, 7.935258572310e+02, 7.964160358480e+02, 7.992315498900e+02, 8.019765362950e+02, 8.046547915420e+02, 8.072698084030e+02, 8.098248078110e+02, 8.123227666060e+02, 8.147664417840e+02, 8.171583917820e+02, 8.195009952200e+02, 8.217964674610e+02, 8.240468753000e+02, 8.262541500150e+02, 8.284200990100e+02, 8.305464162330e+02, 8.326346915010e+02, 8.346864189020e+02, 8.367030043430e+02, 8.386857723780e+02, 8.406359723880e+02, 8.425547841700e+02, 8.444433230330e+02, 8.463026444220e+02, 8.481337481410e+02, 8.499375822080e+02, 8.517150463820e+02, 8.534669953940e+02, 8.551942419160e+02, 8.568975592850e+02, 8.585776840100e+02, 8.602353180870e+02, 8.618711311330e+02, 8.634857623560e+02, 8.650798223790e+02, 8.666538949290e+02, 8.682085384060e+02, 8.697442873350e+02, 8.712616537200e+02, 8.727611282970e+02, 8.742431817090e+02, 8.757082655970e+02, 8.771568136170e+02, 8.785892423920e+02, 8.800059524000e+02, 8.814073288100e+02, 8.827937422540e+02, 8.841655495640e+02, 8.855230944540e+02, 8.868667081610e+02, 8.881967100530e+02, 8.895134081920e+02, 8.908170998730e+02, 8.921080721240e+02, 8.933866021810e+02, 8.946529579340e+02, 8.959073983500e+02, 8.971501738690e+02, 8.983815267830e+02, 8.996016915870e+02, 9.008108953190e+02, 9.020093578790e+02, 9.031972923260e+02, 9.043749051660e+02, 9.055423966250e+02, 9.066999609010e+02, 9.078477864120e+02, 9.089860560250e+02, 9.101149472780e+02, 9.112346325870e+02, 9.123452794460e+02, 9.134470506180e+02, 9.145401043120e+02, 9.156245943550e+02, 9.167006703580e+02, 9.177684778720e+02, 9.188281585340e+02, 9.198798502090e+02, 9.209236871310e+02, 9.219598000230e+02, 9.229883162320e+02, 9.240093598360e+02, 9.250230517660e+02, 9.260295099090e+02, 9.270288492130e+02, 9.280211817870e+02, 9.290066169950e+02, 9.299852615460e+02, 9.309572195840e+02, 9.319225927670e+02, 9.328814803520e+02, 9.338339792710e+02, 9.347801842020e+02, 9.357201876410e+02, 9.366540799730e+02, 9.375819495330e+02, 9.385038826730e+02, 9.394199638180e+02, 9.403302755300e+02, 9.412348985580e+02, 9.421339118950e+02, 9.430273928300e+02, 9.439154169970e+02, 9.447980584220e+02, 9.456753895730e+02, 9.465474814000e+02, 9.474144033800e+02, 9.482762235620e+02, 9.491330085990e+02, 9.499848237950e+02, 9.508317331370e+02, 9.516737993330e+02, 9.525110838450e+02, 9.533436469260e+02, 9.541715476480e+02, 9.549948439370e+02, 9.558135926020e+02, 9.566278493640e+02, 9.574376688810e+02, 9.582431047840e+02, 9.590442096950e+02, 9.598410352560e+02, 9.606336321550e+02, 9.614220501460e+02, 9.622063380770e+02, 9.629865439100e+02, 9.637627147430e+02, 9.645348968300e+02, 9.653031356060e+02, 9.660674756990e+02, 9.668279609580e+02, 9.675846344660e+02, 9.683375385580e+02, 9.690867148420e+02, 9.698322042150e+02, 9.705740468760e+02, 9.713122823470e+02, 9.720469494860e+02, 9.727780865020e+02, 9.735057309690e+02, 9.742299198450e+02, 9.749506894780e+02, 9.756680756280e+02, 9.763821134720e+02, 9.770928376240e+02, 9.778002821420e+02, 9.785044805430e+02, 9.792054658120e+02, 9.799032704190e+02, 9.805979263210e+02, 9.812894649820e+02, 9.819779173790e+02, 9.826633140110e+02, 9.833456849110e+02, 9.840250596580e+02, 9.847014673800e+02, 9.853749367690e+02, 9.860454960870e+02, 9.867131731770e+02, 9.873779954680e+02, 9.880399899880e+02, 9.886991833660e+02, 9.893556018460e+02, 9.900092712910e+02, 9.906602171920e+02, 9.913084646720e+02, 9.919540385000e+02, 9.925969630890e+02, 9.932372625120e+02, 9.938749605020e+02, 9.945100804580e+02, 9.951426454590e+02, 9.957726782600e+02, 9.964002013060e+02, 9.970252367330e+02, 9.976478063760e+02, 9.982679317750e+02, 9.988856341770e+02, 9.995009345470e+02, 1.000113853570e+03, 1.000724411640e+03, 1.001332628920e+03, 1.001938525260e+03, 1.002542120280e+03, 1.003143433350e+03, 1.003742483550e+03, 1.004339289760e+03, 1.004933870580e+03, 1.005526244400e+03, 1.006116429360e+03, 1.006704443370e+03, 1.007290304120e+03, 1.007874029070e+03, 1.008455635470e+03, 1.009035140350e+03, 1.009612560530e+03, 1.010187912620e+03, 1.010761213030e+03, 1.011332477970e+03, 1.011901723440e+03, 1.012468965270e+03, 1.013034219080e+03, 1.013597500300e+03, 1.014158824180e+03, 1.014718205800e+03, 1.015275660050e+03, 1.015831201640e+03, 1.016384845110e+03, 1.016936604840e+03, 1.017486495030e+03, 1.018034529710e+03, 1.018580722770e+03, 1.019125087920e+03, 1.019667638720e+03, 1.020208388560e+03, 1.020747350710e+03, 1.021284538260e+03, 1.021819964170e+03, 1.022353641250e+03, 1.022885582140e+03, 1.023415799390e+03, 1.023944305370e+03, 1.024471112330e+03, 1.024996232380e+03, 1.025519677490e+03, 1.026041459520e+03, 1.026561590180e+03, 1.027080081070e+03, 1.027596943640e+03, 1.028112189240e+03, 1.028625829100e+03, 1.029137874300e+03, 1.029648335840e+03, 1.030157224570e+03, 1.030664551260e+03, 1.031170326530e+03, 1.031674560910e+03, 1.032177264820e+03, 1.032678448570e+03, 1.033178122360e+03, 1.033676296280e+03, 1.034172980320e+03, 1.034668184380e+03, 1.035161918230e+03, 1.035654191580e+03, 1.036145014000e+03, 1.036634395000e+03, 1.037122343960e+03, 1.037608870200e+03, 1.038093982920e+03, 1.038577691240e+03, 1.039060004180e+03, 1.039540930700e+03, 1.040020479620e+03, 1.040498659730e+03, 1.040975479700e+03, 1.041450948110e+03, 1.041925073480e+03, 1.042397864240e+03, 1.042869328730e+03, 1.043339475210e+03, 1.043808311880e+03, 1.044275846830e+03, 1.044742088110e+03, 1.045207043650e+03, 1.045670721350e+03, 1.046133129010e+03, 1.046594274350e+03, 1.047054165040e+03, 1.047512808660e+03, 1.047970212730e+03, 1.048426384700e+03, 1.048881331950e+03, 1.049335061780e+03, 1.049787581430e+03, 1.050238898090e+03, 1.050689018870e+03, 1.051137950810e+03, 1.051585700890e+03, 1.052032276040e+03, 1.052477683110e+03, 1.052921928900e+03, 1.053365020130e+03, 1.053806963500e+03, 1.054247765600e+03, 1.054687433000e+03, 1.055125972210e+03, 1.055563389640e+03, 1.055999691710e+03, 1.056434884720e+03, 1.056868974960e+03, 1.057301968650e+03, 1.057733871940e+03, 1.058164690960e+03, 1.058594431760e+03, 1.059023100340e+03, 1.059450702680e+03, 1.059877244650e+03, 1.060302732140e+03, 1.060727170930e+03, 1.061150566780e+03, 1.061572925410e+03, 1.061994252460e+03, 1.062414553550e+03, 1.062833834250e+03, 1.063252100070e+03, 1.063669356490e+03, 1.064085608930e+03, 1.064500862770e+03, 1.064915123360e+03, 1.065328395980e+03, 1.065740685880e+03, 1.066151998280e+03, 1.066562338340e+03, 1.066971711180e+03, 1.067380121880e+03, 1.067787575480e+03, 1.068194076990e+03, 1.068599631350e+03, 1.069004243500e+03, 1.069407918320e+03, 1.069810660630e+03, 1.070212475260e+03, 1.070613366950e+03, 1.071013340450e+03, 1.071412400440e+03, 1.071810551570e+03, 1.072207798460e+03, 1.072604145700e+03, 1.072999597820e+03, 1.073394159340e+03, 1.073787834740e+03, 1.074180628440e+03, 1.074572544870e+03, 1.074963588380e+03, 1.075353763330e+03, 1.075743074010e+03, 1.076131524700e+03, 1.076519119640e+03, 1.076905863040e+03, 1.077291759070e+03, 1.077676811880e+03, 1.078061025580e+03, 1.078444404250e+03, 1.078826951960e+03, 1.079208672710e+03, 1.079589570510e+03, 1.079969649310e+03, 1.080348913050e+03, 1.080727365630e+03, 1.081105010920e+03, 1.081481852780e+03, 1.081857895030e+03, 1.082233141440e+03, 1.082607595800e+03, 1.082981261820e+03, 1.083354143230e+03, 1.083726243700e+03, 1.084097566880e+03, 1.084468116420e+03, 1.084837895900e+03, 1.085206908920e+03, 1.085575159010e+03, 1.085942649700e+03, 1.086309384510e+03, 1.086675366890e+03, 1.087040600320e+03, 1.087405088200e+03, 1.087768833950e+03, 1.088131840940e+03, 1.088494112540e+03, 1.088855652070e+03, 1.089216462840e+03, 1.089576548140e+03, 1.089935911230e+03, 1.090294555360e+03, 1.090652483730e+03, 1.091009699560e+03, 1.091366206000e+03, 1.091722006220e+03, 1.092077103350e+03, 1.092431500490e+03, 1.092785200730e+03, 1.093138207150e+03, 1.093490522780e+03, 1.093842150660e+03, 1.094193093790e+03, 1.094543355160e+03, 1.094892937720e+03, 1.095241844440e+03, 1.095590078230e+03, 1.095937642000e+03, 1.096284538630e+03, 1.096630771000e+03, 1.096976341960e+03, 1.097321254330e+03, 1.097665510930e+03, 1.098009114560e+03, 1.098352067970e+03, 1.098694373940e+03, 1.099036035210e+03, 1.099377054480e+03, 1.099717434480e+03, 1.100057177870e+03, 1.100396287340e+03, 1.100734765530e+03, 1.101072615080e+03, 1.101409838600e+03, 1.101746438700e+03, 1.102082417950e+03, 1.102417778940e+03, 1.102752524200e+03, 1.103086656280e+03, 1.103420177680e+03, 1.103753090930e+03, 1.104085398500e+03, 1.104417102860e+03, 1.104748206470e+03, 1.105078711780e+03, 1.105408621210e+03, 1.105737937160e+03},
    {1.701328217930e+00, 5.187128880390e+00, 8.735722420330e+00, 1.234995067080e+01, 1.603286469380e+01, 1.978774594220e+01, 2.361813018790e+01, 2.752783466770e+01, 3.152098898500e+01, 3.560207041550e+01, 3.977594439980e+01, 4.404791117700e+01, 4.842375972100e+01, 5.290983041290e+01, 5.751308822240e+01, 6.224120860860e+01, 6.710267891520e+01, 7.210691877610e+01, 7.726442401370e+01, 8.258693980890e+01, 8.808767065130e+01, 9.378153693690e+01, 9.968549131930e+01, 1.058189124330e+02, 1.122040999820e+02, 1.188669043280e+02, 1.258375370650e+02, 1.331516288500e+02, 1.408516308220e+02, 1.489887024480e+02, 1.576253025150e+02, 1.668388205820e+02, 1.767267891350e+02, 1.874145700160e+02, 1.990670479540e+02, 2.119070701030e+02, 2.262457502600e+02, 2.425346609390e+02, 2.614603461160e+02, 2.841230701840e+02, 3.123748900570e+02, 3.493580129960e+02, 3.997089758020e+02, 4.626197658480e+02, 5.190029110230e+02, 5.599079080860e+02, 5.889989271880e+02, 6.110648483400e+02, 6.287369426580e+02, 6.434473848370e+02, 6.560450306480e+02, 6.670679650580e+02, 6.768756167230e+02, 6.857185834390e+02, 6.937779019590e+02, 7.011882901630e+02, 7.080525294960e+02, 7.144507186060e+02, 7.204464320300e+02, 7.260909406810e+02, 7.314261782440e+02, 7.364868725140e+02, 7.413021066090e+02, 7.458964822880e+02, 7.502910001230e+02, 7.545037346750e+02, 7.585503589720e+02, 7.624445565900e+02, 7.661983488960e+02, 7.698223574440e+02, 7.733260163130e+02, 7.767177453830e+02, 7.800050928640e+02, 7.831948534050e+02, 7.862931666390e+02, 7.893055999570e+02, 7.922372184470e+02, 7.950926443330e+02, 7.978761077700e+02, 8.005914904640e+02, 8.032423633220e+02, 8.058320190820e+02, 8.083635007280e+02, 8.108396263230e+02, 8.132630107980e+02, 8.156360851440e+02, 8.179611133610e+02, 8.202402074890e+02, 8.224753409730e+02, 8.246683605740e+02, 8.268209970220e+02, 8.289348745720e+02, 8.310115195810e+02, 8.330523682470e+02, 8.350587735820e+02, 8.370320117340e+02, 8.389732877130e+02, 8.408837405890e+02, 8.427644482320e+02, 8.446164316240e+02, 8.464406588070e+02, 8.482380484830e+02, 8.500094733260e+02, 8.517557630120e+02, 8.534777070040e+02, 8.551760571240e+02, 8.568515299090e+02, 8.585048088010e+02, 8.601365461580e+02, 8.617473651200e+02, 8.633378613390e+02, 8.649086045770e+02, 8.664601401950e+02, 8.679929905390e+02, 8.695076562180e+02, 8.710046173080e+02, 8.724843344640e+02, 8.739472499650e+02, 8.753937886830e+02, 8.768243589950e+02, 8.782393536330e+02, 8.796391504810e+02, 8.810241133190e+02, 8.823945925260e+02, 8.837509257380e+02, 8.850934384630e+02, 8.864224446630e+02, 8.877382473000e+02, 8.890411388500e+02, 8.903314017890e+02, 8.916093090470e+02, 8.928751244440e+02, 8.941291030890e+02, 8.953714917730e+02, 8.966025293280e+02, 8.978224469710e+02, 8.990314686280e+02, 9.002298112470e+02, 9.014176850850e+02, 9.025952939850e+02, 9.037628356430e+02, 9.049205018530e+02, 9.060684787440e+02, 9.072069470080e+02, 9.083360821080e+02, 9.094560544880e+02, 9.105670297620e+02, 9.116691689000e+02, 9.127626284020e+02, 9.138475604660e+02, 9.149241131480e+02, 9.159924305150e+02, 9.170526527840e+02, 9.181049164700e+02, 9.191493545100e+02, 9.201860963930e+02, 9.212152682810e+02, 9.222369931230e+02, 9.232513907680e+02, 9.242585780680e+02, 9.252586689790e+02, 9.262517746610e+02, 9.272380035680e+02, 9.282174615390e+02, 9.291902518820e+02, 9.301564754560e+02, 9.311162307510e+02, 9.320696139600e+02, 9.330167190570e+02, 9.339576378590e+02, 9.348924601010e+02, 9.358212734940e+02, 9.367441637890e+02, 9.376612148370e+02, 9.385725086440e+02, 9.394781254280e+02, 9.403781436720e+02, 9.412726401720e+02, 9.421616900890e+02, 9.430453669980e+02, 9.439237429260e+02, 9.447968884040e+02, 9.456648725050e+02, 9.465277628870e+02, 9.473856258300e+02, 9.482385262760e+02, 9.490865278650e+02, 9.499296929690e+02, 9.507680827290e+02, 9.516017570870e+02, 9.524307748170e+02, 9.532551935560e+02, 9.540750698340e+02, 9.548904591050e+02, 9.557014157700e+02, 9.565079932090e+02, 9.573102438050e+02, 9.581082189700e+02, 9.589019691660e+02, 9.596915439330e+02, 9.604769919130e+02, 9.612583608650e+02, 9.620356976960e+02, 9.628090484720e+02, 9.635784584470e+02, 9.643439720790e+02, 9.651056330460e+02, 9.658634842690e+02, 9.666175679290e+02, 9.673679254820e+02, 9.681145976790e+02, 9.688576245800e+02, 9.695970455690e+02, 9.703328993750e+02, 9.710652240780e+02, 9.717940571340e+02, 9.725194353780e+02, 9.732413950480e+02, 9.739599717910e+02, 9.746752006800e+02, 9.753871162250e+02, 9.760957523850e+02, 9.768011425810e+02, 9.775033197070e+02, 9.782023161410e+02, 9.788981637560e+02, 9.795908939310e+02, 9.802805375640e+02, 9.809671250750e+02, 9.816506864230e+02, 9.823312511130e+02, 9.830088482040e+02, 9.836835063200e+02, 9.843552536590e+02, 9.850241179980e+02, 9.856901267090e+02, 9.863533067570e+02, 9.870136847170e+02, 9.876712867780e+02, 9.883261387490e+02, 9.889782660700e+02, 9.896276938170e+02, 9.902744467070e+02, 9.909185491120e+02, 9.915600250580e+02, 9.921988982350e+02, 9.928351920030e+02, 9.934689293980e+02, 9.941001331400e+02, 9.947288256350e+02, 9.953550289840e+02, 9.959787649880e+02, 9.966000551540e+02, 9.972189206960e+02, 9.978353825490e+02, 9.984494613650e+02, 9.990611775250e+02, 9.996705511390e+02, 1.000277602050e+03, 1.000882349850e+03, 1.001484813880e+03, 1.002085013200e+03, 1.002682966660e+03, 1.003278692850e+03, 1.003872210130e+03, 1.004463536630e+03, 1.005052690220e+03, 1.005639688600e+03, 1.006224549190e+03, 1.006807289230e+03, 1.007387925740e+03, 1.007966475520e+03, 1.008542955170e+03, 1.009117381080e+03, 1.009689769440e+03, 1.010260136250e+03, 1.010828497320e+03, 1.011394868260e+03, 1.011959264490e+03, 1.012521701260e+03, 1.013082193620e+03, 1.013640756450e+03, 1.014197404470e+03, 1.014752152210e+03, 1.015305014020e+03, 1.015856004110e+03, 1.016405136500e+03, 1.016952425070e+03, 1.017497883520e+03, 1.018041525410e+03, 1.018583364130e+03, 1.019123412940e+03, 1.019661684920e+03, 1.020198193030e+03, 1.020732950070e+03, 1.021265968690e+03, 1.021797261420e+03, 1.022326840630e+03, 1.022854718580e+03, 1.023380907360e+03, 1.023905418960e+03, 1.024428265220e+03, 1.024949457850e+03, 1.025469008440e+03, 1.025986928460e+03, 1.026503229260e+03, 1.027017922040e+03, 1.027531017900e+03, 1.028042527840e+03, 1.028552462710e+03, 1.029060833270e+03, 1.029567650160e+03, 1.030072923890e+03, 1.030576664890e+03, 1.031078883470e+03, 1.031579589830e+03, 1.032078794060e+03, 1.032576506150e+03, 1.033072736010e+03, 1.033567493410e+03, 1.034060788050e+03, 1.034552629520e+03, 1.035043027310e+03, 1.035531990830e+03, 1.036019529370e+03, 1.036505652160e+03, 1.036990368310e+03, 1.037473686850e+03, 1.037955616730e+03, 1.038436166800e+03, 1.038915345820e+03, 1.039393162470e+03, 1.039869625360e+03, 1.040344742990e+03, 1.040818523790e+03, 1.041290976120e+03, 1.041762108240e+03, 1.042231928330e+03, 1.042700444530e+03, 1.043167664850e+03, 1.043633597250e+03, 1.044098249620e+03, 1.044561629770e+03, 1.045023745430e+03, 1.045484604270e+03, 1.045944213870e+03, 1.046402581770e+03, 1.046859715400e+03, 1.047315622150e+03, 1.047770309340e+03, 1.048223784220e+03, 1.048676053960e+03, 1.049127125690e+03, 1.049577006450e+03, 1.050025703230e+03, 1.050473222950e+03, 1.050919572490e+03, 1.051364758630e+03, 1.051808788120e+03, 1.052251667640e+03, 1.052693403810e+03, 1.053134003190e+03, 1.053573472290e+03, 1.054011817540e+03, 1.054449045350e+03, 1.054885162040e+03, 1.055320173900e+03, 1.055754087150e+03, 1.056186907960e+03, 1.056618642450e+03, 1.057049296680e+03, 1.057478876660e+03, 1.057907388360e+03, 1.058334837690e+03, 1.058761230510e+03, 1.059186572620e+03, 1.059610869790e+03, 1.060034127730e+03, 1.060456352100e+03, 1.060877548530e+03, 1.061297722570e+03, 1.061716879770e+03, 1.062135025590e+03, 1.062552165460e+03, 1.062968304780e+03, 1.063383448890e+03, 1.063797603080e+03, 1.064210772620e+03, 1.064622962710e+03, 1.065034178530e+03, 1.065444425200e+03, 1.065853707820e+03, 1.066262031430e+03, 1.066669401020e+03, 1.067075821580e+03, 1.067481298030e+03, 1.067885835240e+03, 1.068289438070e+03, 1.068692111340e+03, 1.069093859800e+03, 1.069494688190e+03, 1.069894601210e+03, 1.070293603520e+03, 1.070691699740e+03, 1.071088894450e+03, 1.071485192210e+03, 1.071880597540e+03, 1.072275114910e+03, 1.072668748770e+03, 1.073061503530e+03, 1.073453383580e+03, 1.073844393240e+03, 1.074234536850e+03, 1.074623818670e+03, 1.075012242950e+03, 1.075399813900e+03, 1.075786535710e+03, 1.076172412530e+03, 1.076557448470e+03, 1.076941647630e+03, 1.077325014060e+03, 1.077707551800e+03, 1.078089264830e+03, 1.078470157130e+03, 1.078850232630e+03, 1.079229495250e+03, 1.079607948860e+03, 1.079985597320e+03, 1.080362444450e+03, 1.080738494060e+03, 1.081113749890e+03, 1.081488215710e+03, 1.081861895220e+03, 1.082234792110e+03, 1.082606910040e+03, 1.082978252650e+03, 1.083348823540e+03, 1.083718626300e+03, 1.084087664480e+03, 1.084455941610e+03, 1.084823461210e+03, 1.085190226750e+03, 1.085556241690e+03, 1.085921509460e+03, 1.086286033470e+03, 1.086649817110e+03, 1.087012863740e+03, 1.087375176700e+03, 1.087736759290e+03, 1.088097614810e+03, 1.088457746540e+03, 1.088817157710e+03, 1.089175851540e+03, 1.089533831250e+03, 1.089891100010e+03, 1.090247660980e+03, 1.090603517290e+03, 1.090958672060e+03, 1.091313128380e+03, 1.091666889320e+03, 1.092019957950e+03, 1.092372337280e+03, 1.092724030330e+03, 1.093075040090e+03, 1.093425369530e+03, 1.093775021600e+03, 1.094123999230e+03, 1.094472305340e+03, 1.094819942810e+03, 1.095166914520e+03, 1.095513223330e+03, 1.095858872060e+03, 1.096203863540e+03, 1.096548200570e+03, 1.096891885910e+03, 1.097234922340e+03, 1.097577312610e+03, 1.097919059420e+03, 1.098260165500e+03, 1.098600633530e+03, 1.098940466190e+03, 1.099279666140e+03, 1.099618236000e+03, 1.099956178420e+03, 1.100293495980e+03, 1.100630191280e+03, 1.100966266890e+03, 1.101301725370e+03, 1.101636569250e+03, 1.101970801060e+03, 1.102304423320e+03, 1.102637438500e+03, 1.102969849090e+03, 1.103301657540e+03, 1.103632866310e+03},
    {1.695711512690e+00, 5.169503237090e+00, 8.705154735650e+00, 1.230542530540e+01, 1.597327419830e+01, 1.971188048280e+01, 2.352466545010e+01, 2.741531797280e+01, 3.138782329870e+01, 3.544649585830e+01, 3.959601678600e+01, 4.384147699690e+01, 4.818842684850e+01, 5.264293364370e+01, 5.721164852500e+01, 6.190188467930e+01, 6.672170925180e+01, 7.168005198190e+01, 7.678683438320e+01, 8.205312435100e+01, 8.749132249540e+01, 9.311538839950e+01, 9.894111759060e+01, 1.049864835690e+02, 1.112720642060e+02, 1.178215788320e+02, 1.246625724090e+02, 1.318272978670e+02, 1.393538694880e+02, 1.472877932770e+02, 1.556840313230e+02, 1.646098381900e+02, 1.741487389640e+02, 1.844062388800e+02, 1.955182344230e+02, 2.076637732890e+02, 2.210850584500e+02, 2.361199537550e+02, 2.532567545640e+02, 2.732290986720e+02, 2.971792185660e+02, 3.269034989860e+02, 3.650326384500e+02, 4.142014739600e+02, 4.698276848270e+02, 5.186313808200e+02, 5.557817482090e+02, 5.834647279450e+02, 6.050089096260e+02, 6.225213435840e+02, 6.372337740060e+02, 6.499069907710e+02, 6.610379055250e+02, 6.709659662540e+02, 6.799318421710e+02, 6.881116262050e+02, 6.956376602620e+02, 7.026117075250e+02, 7.091135733780e+02, 7.152069219220e+02, 7.209433072670e+02, 7.263650342470e+02, 7.315072309140e+02, 7.363993776800e+02, 7.410664539440e+02, 7.455298104630e+02, 7.498078417900e+02, 7.539165107910e+02, 7.578697622820e+02, 7.616798525020e+02, 7.653576140080e+02, 7.689126705050e+02, 7.723536124940e+02, 7.756881419690e+02, 7.789231924720e+02, 7.820650293780e+02, 7.851193341880e+02, 7.880912758090e+02, 7.909855711550e+02, 7.938065369650e+02, 7.965581343080e+02, 7.992440070040e+02, 8.018675149380e+02, 8.044317630570e+02, 8.069396267300e+02, 8.093937739870e+02, 8.117966851110e+02, 8.141506699500e+02, 8.164578832520e+02, 8.187203383120e+02, 8.209399191290e+02, 8.231183912880e+02, 8.252574117000e+02, 8.273585373700e+02, 8.294232332860e+02, 8.314528795410e+02, 8.334487777850e+02, 8.354121570630e+02, 8.373441791310e+02, 8.392459432810e+02, 8.411184907490e+02, 8.429628087430e+02, 8.447798341160e+02, 8.465704567450e+02, 8.483355226250e+02, 8.500758367120e+02, 8.517921655430e+02, 8.534852396480e+02, 8.551557557800e+02, 8.568043789660e+02, 8.584317444180e+02, 8.600384592910e+02, 8.616251043220e+02, 8.631922353450e+02, 8.647403847010e+02, 8.662700625560e+02, 8.677817581160e+02, 8.692759407710e+02, 8.707530611570e+02, 8.722135521490e+02, 8.736578297890e+02, 8.750862941550e+02, 8.764993301710e+02, 8.778973083760e+02, 8.792805856310e+02, 8.806495057970e+02, 8.820044003610e+02, 8.833455890300e+02, 8.846733802880e+02, 8.859880719250e+02, 8.872899515240e+02, 8.885792969350e+02, 8.898563767130e+02, 8.911214505290e+02, 8.923747695690e+02, 8.936165769010e+02, 8.948471078270e+02, 8.960665902150e+02, 8.972752448140e+02, 8.984732855510e+02, 8.996609198150e+02, 9.008383487230e+02, 9.020057673770e+02, 9.031633651030e+02, 9.043113256830e+02, 9.054498275710e+02, 9.065790441010e+02, 9.076991436860e+02, 9.088102900040e+02, 9.099126421780e+02, 9.110063549480e+02, 9.120915788290e+02, 9.131684602740e+02, 9.142371418120e+02, 9.152977621990e+02, 9.163504565470e+02, 9.173953564530e+02, 9.184325901270e+02, 9.194622825050e+02, 9.204845553630e+02, 9.214995274260e+02, 9.225073144700e+02, 9.235080294220e+02, 9.245017824540e+02, 9.254886810740e+02, 9.264688302140e+02, 9.274423323100e+02, 9.284092873890e+02, 9.293697931370e+02, 9.303239449810e+02, 9.312718361540e+02, 9.322135577680e+02, 9.331491988730e+02, 9.340788465280e+02, 9.350025858560e+02, 9.359205001020e+02, 9.368326706930e+02, 9.377391772870e+02, 9.386400978300e+02, 9.395355086030e+02, 9.404254842690e+02, 9.413100979210e+02, 9.421894211300e+02, 9.430635239810e+02, 9.439324751220e+02, 9.447963417960e+02, 9.456551898900e+02, 9.465090839630e+02, 9.473580872870e+02, 9.482022618800e+02, 9.490416685410e+02, 9.498763668820e+02, 9.507064153600e+02, 9.515318713050e+02, 9.523527909530e+02, 9.531692294700e+02, 9.539812409840e+02, 9.547888786110e+02, 9.555921944750e+02, 9.563912397410e+02, 9.571860646340e+02, 9.579767184640e+02, 9.587632496480e+02, 9.595457057330e+02, 9.603241334160e+02, 9.610985785650e+02, 9.618690862400e+02, 9.626357007110e+02, 9.633984654750e+02, 9.641574232810e+02, 9.649126161400e+02, 9.656640853440e+02, 9.664118714880e+02, 9.671560144790e+02, 9.678965535550e+02, 9.686335273010e+02, 9.693669736620e+02, 9.700969299580e+02, 9.708234328980e+02, 9.715465185950e+02, 9.722662225750e+02, 9.729825797960e+02, 9.736956246560e+02, 9.744053910040e+02, 9.751119121560e+02, 9.758152209050e+02, 9.765153495310e+02, 9.772123298110e+02, 9.779061930350e+02, 9.785969700070e+02, 9.792846910650e+02, 9.799693860840e+02, 9.806510844870e+02, 9.813298152570e+02, 9.820056069420e+02, 9.826784876660e+02, 9.833484851390e+02, 9.840156266600e+02, 9.846799391320e+02, 9.853414490660e+02, 9.860001825870e+02, 9.866561654460e+02, 9.873094230250e+02, 9.879599803440e+02, 9.886078620680e+02, 9.892530925140e+02, 9.898956956600e+02, 9.905356951450e+02, 9.911731142840e+02, 9.918079760660e+02, 9.924403031680e+02, 9.930701179530e+02, 9.936974424810e+02, 9.943222985140e+02, 9.949447075180e+02, 9.955646906740e+02, 9.961822688780e+02, 9.967974627480e+02, 9.974102926320e+02, 9.980207786080e+02, 9.986289404910e+02, 9.992347978380e+02, 9.998383699520e+02, 1.000439675890e+03, 1.001038734450e+03, 1.001635564220e+03, 1.002230183510e+03, 1.002822610430e+03, 1.003412862860e+03, 1.004000958420e+03, 1.004586914560e+03, 1.005170748480e+03, 1.005752477170e+03, 1.006332117420e+03, 1.006909685810e+03, 1.007485198720e+03, 1.008058672310e+03, 1.008630122580e+03, 1.009199565300e+03, 1.009767016080e+03, 1.010332490330e+03, 1.010896003270e+03, 1.011457569960e+03, 1.012017205250e+03, 1.012574923840e+03, 1.013130740260e+03, 1.013684668860e+03, 1.014236723820e+03, 1.014786919150e+03, 1.015335268740e+03, 1.015881786260e+03, 1.016426485270e+03, 1.016969379150e+03, 1.017510481150e+03, 1.018049804340e+03, 1.018587361680e+03, 1.019123165960e+03, 1.019657229830e+03, 1.020189565800e+03, 1.020720186250e+03, 1.021249103410e+03, 1.021776329400e+03, 1.022301876170e+03, 1.022825755570e+03, 1.023347979320e+03, 1.023868558990e+03, 1.024387506050e+03, 1.024904831830e+03, 1.025420547560e+03, 1.025934664320e+03, 1.026447193110e+03, 1.026958144770e+03, 1.027467530060e+03, 1.027975359620e+03, 1.028481643970e+03, 1.028986393520e+03, 1.029489618590e+03, 1.029991329380e+03, 1.030491535970e+03, 1.030990248360e+03, 1.031487476450e+03, 1.031983230020e+03, 1.032477518760e+03, 1.032970352270e+03, 1.033461740030e+03, 1.033951691450e+03, 1.034440215840e+03, 1.034927322410e+03, 1.035413020280e+03, 1.035897318480e+03, 1.036380225960e+03, 1.036861751560e+03, 1.037341904070e+03, 1.037820692150e+03, 1.038298124410e+03, 1.038774209360e+03, 1.039248955440e+03, 1.039722370990e+03, 1.040194464290e+03, 1.040665243530e+03, 1.041134716820e+03, 1.041602892200e+03, 1.042069777630e+03, 1.042535381000e+03, 1.042999710110e+03, 1.043462772710e+03, 1.043924576460e+03, 1.044385128960e+03, 1.044844437730e+03, 1.045302510220e+03, 1.045759353830e+03, 1.046214975870e+03, 1.046669383590e+03, 1.047122584170e+03, 1.047574584740e+03, 1.048025392350e+03, 1.048475014000e+03, 1.048923456600e+03, 1.049370727030e+03, 1.049816832090e+03, 1.050261778530e+03, 1.050705573030e+03, 1.051148222200e+03, 1.051589732630e+03, 1.052030110820e+03, 1.052469363210e+03, 1.052907496210e+03, 1.053344516150e+03, 1.053780429320e+03, 1.054215241930e+03, 1.054648960180e+03, 1.055081590180e+03, 1.055513138000e+03, 1.055943609660e+03, 1.056373011120e+03, 1.056801348290e+03, 1.057228627050e+03, 1.057654853210e+03, 1.058080032530e+03, 1.058504170730e+03, 1.058927273480e+03, 1.059349346400e+03, 1.059770395070e+03, 1.060190425010e+03, 1.060609441710e+03, 1.061027450600e+03, 1.061444457080e+03, 1.061860466500e+03, 1.062275484160e+03, 1.062689515310e+03, 1.063102565190e+03, 1.063514638960e+03, 1.063925741760e+03, 1.064335878680e+03, 1.064745054760e+03, 1.065153275030e+03, 1.065560544460e+03, 1.065966867960e+03, 1.066372250440e+03, 1.066776696740e+03, 1.067180211690e+03, 1.067582800050e+03, 1.067984466560e+03, 1.068385215930e+03, 1.068785052820e+03, 1.069183981860e+03, 1.069582007630e+03, 1.069979134710e+03, 1.070375367590e+03, 1.070770710780e+03, 1.071165168720e+03, 1.071558745830e+03, 1.071951446500e+03, 1.072343275070e+03, 1.072734235850e+03, 1.073124333140e+03, 1.073513571180e+03, 1.073901954190e+03, 1.074289486360e+03, 1.074676171850e+03, 1.075062014770e+03, 1.075447019230e+03, 1.075831189290e+03, 1.076214528970e+03, 1.076597042280e+03, 1.076978733200e+03, 1.077359605660e+03, 1.077739663590e+03, 1.078118910870e+03, 1.078497351350e+03, 1.078874988860e+03, 1.079251827210e+03, 1.079627870180e+03, 1.080003121490e+03, 1.080377584890e+03, 1.080751264050e+03, 1.081124162640e+03, 1.081496284300e+03, 1.081867632650e+03, 1.082238211280e+03, 1.082608023740e+03, 1.082977073570e+03, 1.083345364280e+03, 1.083712899360e+03, 1.084079682270e+03, 1.084445716450e+03, 1.084811005320e+03, 1.085175552260e+03, 1.085539360630e+03, 1.085902433790e+03, 1.086264775040e+03, 1.086626387690e+03, 1.086987275010e+03, 1.087347440250e+03, 1.087706886630e+03, 1.088065617370e+03, 1.088423635650e+03, 1.088780944630e+03, 1.089137547450e+03, 1.089493447240e+03, 1.089848647080e+03, 1.090203150050e+03, 1.090556959230e+03, 1.090910077630e+03, 1.091262508270e+03, 1.091614254150e+03, 1.091965318250e+03, 1.092315703520e+03, 1.092665412900e+03, 1.093014449300e+03, 1.093362815610e+03, 1.093710514730e+03, 1.094057549500e+03, 1.094403922760e+03, 1.094749637340e+03, 1.095094696040e+03, 1.095439101640e+03, 1.095782856910e+03, 1.096125964600e+03, 1.096468427440e+03, 1.096810248140e+03, 1.097151429390e+03, 1.097491973880e+03, 1.097831884260e+03, 1.098171163180e+03, 1.098509813260e+03, 1.098847837120e+03, 1.099185237340e+03, 1.099522016510e+03, 1.099858177180e+03, 1.100193721900e+03, 1.100528653190e+03, 1.100862973570e+03, 1.101196685530e+03, 1.101529791560e+03},
    {1.690132318780e+00, 5.152002277570e+00, 8.674815933760e+00, 1.226125263700e+01, 1.591418284780e+01, 1.963668677910e+01, 2.343207538040e+01, 2.730391402960e+01, 3.125604936720e+01, 3.529263979030e+01, 3.941819022790e+01, 4.363759194710e+01, 4.795616829590e+01, 5.237972748860e+01, 5.691462378590e+01, 6.156782874180e+01, 6.634701458400e+01, 7.126065232070e+01, 7.631812782830e+01, 8.152988005550e+01, 8.690756663150e+01, 9.246426370590e+01, 9.821470891360e+01, 1.041755991760e+02, 1.103659589230e+02, 1.168075997150e+02, 1.235256998820e+02, 1.305495437380e+02, 1.379134758390e+02, 1.456581494340e+02, 1.538321838430e+02, 1.624944005290e+02, 1.717168941370e+02, 1.815893343280e+02, 1.922251248310e+02, 2.037704367410e+02, 2.164178095670e+02, 2.304272015140e+02, 2.461594330220e+02, 2.641302495410e+02, 2.850966151440e+02, 3.101802254080e+02, 3.409739771080e+02, 3.793829009260e+02, 4.261106980030e+02, 4.752657602170e+02, 5.182118218320e+02, 5.521596596660e+02, 5.784496158050e+02, 5.993987747240e+02, 6.166733733410e+02, 6.313216911210e+02, 6.440183979520e+02, 6.552170332430e+02, 6.652343555530e+02, 6.742988417280e+02, 6.825799227710e+02, 6.902062948540e+02, 6.972777687360e+02, 7.038731674490e+02, 7.100557338180e+02, 7.158769242230e+02, 7.213791287510e+02, 7.265976596620e+02, 7.315622301470e+02, 7.362980709800e+02, 7.408267853620e+02, 7.451670114800e+02, 7.493349418440e+02, 7.533447345800e+02, 7.572088422600e+02, 7.609382771490e+02, 7.645428269150e+02, 7.680312314230e+02, 7.714113286750e+02, 7.746901761160e+02, 7.778741521030e+02, 7.809690413020e+02, 7.839801069720e+02, 7.869121524830e+02, 7.897695739390e+02, 7.925564054110e+02, 7.952763579980e+02, 7.979328537130e+02, 8.005290549820e+02, 8.030678904490e+02, 8.055520776180e+02, 8.079841427870e+02, 8.103664386750e+02, 8.127011600410e+02, 8.149903575690e+02, 8.172359502530e+02, 8.194397364730e+02, 8.216034039200e+02, 8.237285385170e+02, 8.258166324540e+02, 8.278690914450e+02, 8.298872412910e+02, 8.318723338340e+02, 8.338255523610e+02, 8.357480165260e+02, 8.376407868340e+02, 8.395048687460e+02, 8.413412164180e+02, 8.431507361460e+02, 8.449342895100e+02, 8.466926962800e+02, 8.484267370750e+02, 8.501371558270e+02, 8.518246620480e+02, 8.534899329230e+02, 8.551336152520e+02, 8.567563272490e+02, 8.583586602020e+02, 8.599411800240e+02, 8.615044286880e+02, 8.630489255670e+02, 8.645751686740e+02, 8.660836358320e+02, 8.675747857470e+02, 8.690490590290e+02, 8.705068791320e+02, 8.719486532420e+02, 8.733747731030e+02, 8.747856157980e+02, 8.761815444740e+02, 8.775629090280e+02, 8.789300467500e+02, 8.802832829240e+02, 8.816229314000e+02, 8.829492951290e+02, 8.842626666650e+02, 8.855633286420e+02, 8.868515542220e+02, 8.881276075210e+02, 8.893917440030e+02, 8.906442108660e+02, 8.918852473960e+02, 8.931150853030e+02, 8.943339490450e+02, 8.955420561340e+02, 8.967396174180e+02, 8.979268373580e+02, 8.991039142870e+02, 9.002710406580e+02, 9.014284032730e+02, 9.025761835100e+02, 9.037145575340e+02, 9.048436964940e+02, 9.059637667230e+02, 9.070749299110e+02, 9.081773432850e+02, 9.092711597750e+02, 9.103565281680e+02, 9.114335932620e+02, 9.125024960100e+02, 9.135633736570e+02, 9.146163598700e+02, 9.156615848660e+02, 9.166991755320e+02, 9.177292555370e+02, 9.187519454450e+02, 9.197673628200e+02, 9.207756223240e+02, 9.217768358170e+02, 9.227711124480e+02, 9.237585587450e+02, 9.247392786960e+02, 9.257133738360e+02, 9.266809433210e+02, 9.276420840090e+02, 9.285968905240e+02, 9.295454553320e+02, 9.304878688050e+02, 9.314242192860e+02, 9.323545931510e+02, 9.332790748650e+02, 9.341977470450e+02, 9.351106905100e+02, 9.360179843340e+02, 9.369197059000e+02, 9.378159309490e+02, 9.387067336220e+02, 9.395921865130e+02, 9.404723607070e+02, 9.413473258240e+02, 9.422171500630e+02, 9.430819002370e+02, 9.439416418140e+02, 9.447964389510e+02, 9.456463545330e+02, 9.464914502050e+02, 9.473317864060e+02, 9.481674223990e+02, 9.489984163050e+02, 9.498248251310e+02, 9.506467048000e+02, 9.514641101770e+02, 9.522770950990e+02, 9.530857123980e+02, 9.538900139290e+02, 9.546900505940e+02, 9.554858723640e+02, 9.562775283030e+02, 9.570650665940e+02, 9.578485345530e+02, 9.586279786570e+02, 9.594034445600e+02, 9.601749771150e+02, 9.609426203920e+02, 9.617064176970e+02, 9.624664115900e+02, 9.632226439010e+02, 9.639751557510e+02, 9.647239875610e+02, 9.654691790780e+02, 9.662107693810e+02, 9.669487969010e+02, 9.676832994350e+02, 9.684143141610e+02, 9.691418776490e+02, 9.698660258770e+02, 9.705867942430e+02, 9.713042175760e+02, 9.720183301540e+02, 9.727291657080e+02, 9.734367574410e+02, 9.741411380350e+02, 9.748423396610e+02, 9.755403939960e+02, 9.762353322260e+02, 9.769271850600e+02, 9.776159827420e+02, 9.783017550550e+02, 9.789845313340e+02, 9.796643404770e+02, 9.803412109500e+02, 9.810151707960e+02, 9.816862476460e+02, 9.823544687260e+02, 9.830198608650e+02, 9.836824505040e+02, 9.843422636990e+02, 9.849993261370e+02, 9.856536631330e+02, 9.863052996460e+02, 9.869542602820e+02, 9.876005692990e+02, 9.882442506170e+02, 9.888853278230e+02, 9.895238241780e+02, 9.901597626200e+02, 9.907931657760e+02, 9.914240559610e+02, 9.920524551900e+02, 9.926783851770e+02, 9.933018673470e+02, 9.939229228370e+02, 9.945415725040e+02, 9.951578369260e+02, 9.957717364110e+02, 9.963832910020e+02, 9.969925204770e+02, 9.975994443580e+02, 9.982040819170e+02, 9.988064521730e+02, 9.994065739050e+02, 1.000004465650e+03, 1.000600145710e+03, 1.001193632160e+03, 1.001784942830e+03, 1.002374095360e+03, 1.002961107130e+03, 1.003545995330e+03, 1.004128776940e+03, 1.004709468710e+03, 1.005288087210e+03, 1.005864648780e+03, 1.006439169580e+03, 1.007011665570e+03, 1.007582152530e+03, 1.008150646020e+03, 1.008717161440e+03, 1.009281714010e+03, 1.009844318750e+03, 1.010404990510e+03, 1.010963743970e+03, 1.011520593640e+03, 1.012075553850e+03, 1.012628638790e+03, 1.013179862440e+03, 1.013729238670e+03, 1.014276781160e+03, 1.014822503440e+03, 1.015366418880e+03, 1.015908540730e+03, 1.016448882040e+03, 1.016987455760e+03, 1.017524274670e+03, 1.018059351420e+03, 1.018592698500e+03, 1.019124328290e+03, 1.019654253010e+03, 1.020182484760e+03, 1.020709035500e+03, 1.021233917070e+03, 1.021757141180e+03, 1.022278719390e+03, 1.022798663170e+03, 1.023316983840e+03, 1.023833692630e+03, 1.024348800610e+03, 1.024862318770e+03, 1.025374257960e+03, 1.025884628930e+03, 1.026393442300e+03, 1.026900708610e+03, 1.027406438270e+03, 1.027910641570e+03, 1.028413328710e+03, 1.028914509800e+03, 1.029414194810e+03, 1.029912393630e+03, 1.030409116060e+03, 1.030904371780e+03, 1.031398170390e+03, 1.031890521360e+03, 1.032381434120e+03, 1.032870917950e+03, 1.033358982070e+03, 1.033845635610e+03, 1.034330887590e+03, 1.034814746960e+03, 1.035297222560e+03, 1.035778323180e+03, 1.036258057480e+03, 1.036736434070e+03, 1.037213461470e+03, 1.037689148100e+03, 1.038163502320e+03, 1.038636532400e+03, 1.039108246540e+03, 1.039578652830e+03, 1.040047759340e+03, 1.040515574010e+03, 1.040982104740e+03, 1.041447359330e+03, 1.041911345530e+03, 1.042374071010e+03, 1.042835543360e+03, 1.043295770100e+03, 1.043754758700e+03, 1.044212516540e+03, 1.044669050950e+03, 1.045124369170e+03, 1.045578478390e+03, 1.046031385740e+03, 1.046483098270e+03, 1.046933622980e+03, 1.047382966790e+03, 1.047831136590e+03, 1.048278139160e+03, 1.048723981270e+03, 1.049168669590e+03, 1.049612210760e+03, 1.050054611340e+03, 1.050495877840e+03, 1.050936016730e+03, 1.051375034390e+03, 1.051812937170e+03, 1.052249731350e+03, 1.052685423170e+03, 1.053120018810e+03, 1.053553524390e+03, 1.053985945980e+03, 1.054417289600e+03, 1.054847561230e+03, 1.055276766780e+03, 1.055704912120e+03, 1.056132003070e+03, 1.056558045400e+03, 1.056983044830e+03, 1.057407007020e+03, 1.057829937620e+03, 1.058251842200e+03, 1.058672726280e+03, 1.059092595360e+03, 1.059511454870e+03, 1.059929310220e+03, 1.060346166750e+03, 1.060762029780e+03, 1.061176904570e+03, 1.061590796330e+03, 1.062003710260e+03, 1.062415651480e+03, 1.062826625100e+03, 1.063236636170e+03, 1.063645689710e+03, 1.064053790680e+03, 1.064460944020e+03, 1.064867154640e+03, 1.065272427380e+03, 1.065676767060e+03, 1.066080178470e+03, 1.066482666350e+03, 1.066884235390e+03, 1.067284890280e+03, 1.067684635640e+03, 1.068083476070e+03, 1.068481416130e+03, 1.068878460340e+03, 1.069274613200e+03, 1.069669879160e+03, 1.070064262640e+03, 1.070457768020e+03, 1.070850399670e+03, 1.071242161900e+03, 1.071633059010e+03, 1.072023095240e+03, 1.072412274810e+03, 1.072800601940e+03, 1.073188080760e+03, 1.073574715410e+03, 1.073960509990e+03, 1.074345468560e+03, 1.074729595170e+03, 1.075112893820e+03, 1.075495368480e+03, 1.075877023110e+03, 1.076257861630e+03, 1.076637887910e+03, 1.077017105840e+03, 1.077395519230e+03, 1.077773131900e+03, 1.078149947630e+03, 1.078525970150e+03, 1.078901203210e+03, 1.079275650490e+03, 1.079649315660e+03, 1.080022202370e+03, 1.080394314240e+03, 1.080765654860e+03, 1.081136227780e+03, 1.081506036570e+03, 1.081875084720e+03, 1.082243375740e+03, 1.082610913080e+03, 1.082977700200e+03, 1.083343740500e+03, 1.083709037380e+03, 1.084073594210e+03, 1.084437414350e+03, 1.084800501100e+03, 1.085162857780e+03, 1.085524487660e+03, 1.085885393990e+03, 1.086245580010e+03, 1.086605048930e+03, 1.086963803940e+03, 1.087321848200e+03, 1.087679184850e+03, 1.088035817030e+03, 1.088391747840e+03, 1.088746980350e+03, 1.089101517620e+03, 1.089455362710e+03, 1.089808518620e+03, 1.090160988360e+03, 1.090512774900e+03, 1.090863881210e+03, 1.091214310220e+03, 1.091564064860e+03, 1.091913148020e+03, 1.092261562600e+03, 1.092609311440e+03, 1.092956397400e+03, 1.093302823300e+03, 1.093648591940e+03, 1.093993706130e+03, 1.094338168610e+03, 1.094681982160e+03, 1.095025149500e+03, 1.095367673340e+03, 1.095709556400e+03, 1.096050801340e+03, 1.096391410840e+03, 1.096731387540e+03, 1.097070734080e+03, 1.097409453060e+03, 1.097747547080e+03, 1.098085018730e+03, 1.098421870560e+03, 1.098758105130e+03, 1.099093724970e+03, 1.099428732590e+03},
    {1.684590253720e+00, 5.134624605570e+00, 8.644703219920e+00, 1.221742799670e+01, 1.585558351040e+01, 1.956215450850e+01, 2.334034556470e+01, 2.719360321700e+01, 3.112564097900e+01, 3.514046768960e+01, 3.924241975200e+01, 4.343619792840e+01, 4.772690948910e+01, 5.212011668820e+01, 5.662189274850e+01, 6.123888680730e+01, 6.597839961290e+01, 7.084847219610e+01, 7.585799029610e+01, 8.101680804420e+01, 8.633589535090e+01, 9.182751468690e+01, 9.750543460810e+01, 1.033851896020e+02, 1.094843988630e+02, 1.158231607810e+02, 1.224245457080e+02, 1.293152177880e+02, 1.365262283010e+02, 1.440940399910e+02, 1.520618669250e+02, 1.604814520630e+02, 1.694154622760e+02, 1.789407701770e+02, 1.891530344870e+02, 2.001732206640e+02, 2.121570801720e+02, 2.253092243610e+02, 2.399044122800e+02, 2.563200599390e+02, 2.750850714900e+02, 2.969466823060e+02, 3.229331221810e+02, 3.543138321450e+02, 3.921988358260e+02, 4.357959485030e+02, 4.794765038410e+02, 5.177700942170e+02, 5.489646451230e+02, 5.739048933560e+02, 5.942109878910e+02, 6.111863528150e+02, 6.257138942110e+02, 6.383866004530e+02, 6.496145602010e+02, 6.596904316570e+02, 6.688289508690e+02, 6.771915540050e+02, 6.849022280550e+02, 6.920580056860e+02, 6.987360926570e+02, 7.049988235620e+02, 7.108971833760e+02, 7.164733600660e+02, 7.217626283210e+02, 7.267947622150e+02, 7.315951099510e+02, 7.361854221720e+02, 7.405844978430e+02, 7.448086932580e+02, 7.488723270990e+02, 7.527880056800e+02, 7.565668862960e+02, 7.602188921070e+02, 7.637528887650e+02, 7.671768306040e+02, 7.704978823980e+02, 7.737225214240e+02, 7.768566234910e+02, 7.799055358560e+02, 7.828741393630e+02, 7.857669016420e+02, 7.885879228850e+02, 7.913409753960e+02, 7.940295379230e+02, 7.966568255670e+02, 7.992258159490e+02, 8.017392721770e+02, 8.041997630820e+02, 8.066096811030e+02, 8.089712581450e+02, 8.112865796820e+02, 8.135575973310e+02, 8.157861401020e+02, 8.179739244780e+02, 8.201225634780e+02, 8.222335748230e+02, 8.243083883070e+02, 8.263483524650e+02, 8.283547406210e+02, 8.303287563850e+02, 8.322715386510e+02, 8.341841661580e+02, 8.360676616520e+02, 8.379229957030e+02, 8.397510901910e+02, 8.415528215120e+02, 8.433290235270e+02, 8.450804902690e+02, 8.468079784460e+02, 8.485122097450e+02, 8.501938729630e+02, 8.518536259830e+02, 8.534920975950e+02, 8.551098891950e+02, 8.567075763550e+02, 8.582857102890e+02, 8.598448192090e+02, 8.613854095910e+02, 8.629079673630e+02, 8.644129590000e+02, 8.659008325570e+02, 8.673720186320e+02, 8.688269312630e+02, 8.702659687750e+02, 8.716895145700e+02, 8.730979378660e+02, 8.744915943970e+02, 8.758708270640e+02, 8.772359665500e+02, 8.785873319000e+02, 8.799252310660e+02, 8.812499614190e+02, 8.825618102340e+02, 8.838610551460e+02, 8.851479645820e+02, 8.864227981650e+02, 8.876858071060e+02, 8.889372345610e+02, 8.901773159790e+02, 8.914062794310e+02, 8.926243459120e+02, 8.938317296420e+02, 8.950286383410e+02, 8.962152734900e+02, 8.973918305890e+02, 8.985584993890e+02, 8.997154641190e+02, 9.008629037060e+02, 9.020009919740e+02, 9.031298978450e+02, 9.042497855190e+02, 9.053608146570e+02, 9.064631405430e+02, 9.075569142520e+02, 9.086422827990e+02, 9.097193892890e+02, 9.107883730520e+02, 9.118493697830e+02, 9.129025116650e+02, 9.139479274950e+02, 9.149857427970e+02, 9.160160799380e+02, 9.170390582330e+02, 9.180547940470e+02, 9.190634008950e+02, 9.200649895350e+02, 9.210596680610e+02, 9.220475419830e+02, 9.230287143190e+02, 9.240032856680e+02, 9.249713542900e+02, 9.259330161770e+02, 9.268883651260e+02, 9.278374928060e+02, 9.287804888230e+02, 9.297174407830e+02, 9.306484343530e+02, 9.315735533170e+02, 9.324928796350e+02, 9.334064934930e+02, 9.343144733600e+02, 9.352168960320e+02, 9.361138366840e+02, 9.370053689150e+02, 9.378915647960e+02, 9.387724949050e+02, 9.396482283780e+02, 9.405188329430e+02, 9.413843749610e+02, 9.422449194630e+02, 9.431005301860e+02, 9.439512696090e+02, 9.447971989840e+02, 9.456383783710e+02, 9.464748666690e+02, 9.473067216460e+02, 9.481339999680e+02, 9.489567572280e+02, 9.497750479760e+02, 9.505889257380e+02, 9.513984430530e+02, 9.522036514880e+02, 9.530046016670e+02, 9.538013432930e+02, 9.545939251730e+02, 9.553823952370e+02, 9.561668005600e+02, 9.569471873840e+02, 9.577236011380e+02, 9.584960864540e+02, 9.592646871900e+02, 9.600294464480e+02, 9.607904065880e+02, 9.615476092490e+02, 9.623010953640e+02, 9.630509051770e+02, 9.637970782570e+02, 9.645396535160e+02, 9.652786692210e+02, 9.660141630110e+02, 9.667461719100e+02, 9.674747323400e+02, 9.681998801350e+02, 9.689216505520e+02, 9.696400782880e+02, 9.703551974880e+02, 9.710670417560e+02, 9.717756441730e+02, 9.724810373010e+02, 9.731832531970e+02, 9.738823234240e+02, 9.745782790620e+02, 9.752711507140e+02, 9.759609685200e+02, 9.766477621670e+02, 9.773315608940e+02, 9.780123935050e+02, 9.786902883750e+02, 9.793652734620e+02, 9.800373763120e+02, 9.807066240680e+02, 9.813730434810e+02, 9.820366609120e+02, 9.826975023450e+02, 9.833555933920e+02, 9.840109593000e+02, 9.846636249590e+02, 9.853136149070e+02, 9.859609533400e+02, 9.866056641150e+02, 9.872477707600e+02, 9.878872964740e+02, 9.885242641430e+02, 9.891586963340e+02, 9.897906153120e+02, 9.904200430390e+02, 9.910470011790e+02, 9.916715111080e+02, 9.922935939170e+02, 9.929132704150e+02, 9.935305611380e+02, 9.941454863500e+02, 9.947580660510e+02, 9.953683199800e+02, 9.959762676210e+02, 9.965819282040e+02, 9.971853207150e+02, 9.977864638960e+02, 9.983853762490e+02, 9.989820760420e+02, 9.995765813150e+02, 1.000168909880e+03, 1.000759079320e+03, 1.001347107020e+03, 1.001933010120e+03, 1.002516805560e+03, 1.003098510090e+03, 1.003678140240e+03, 1.004255712320e+03, 1.004831242480e+03, 1.005404746660e+03, 1.005976240580e+03, 1.006545739830e+03, 1.007113259750e+03, 1.007678815550e+03, 1.008242422230e+03, 1.008804094630e+03, 1.009363847410e+03, 1.009921695040e+03, 1.010477651870e+03, 1.011031732030e+03, 1.011583949520e+03, 1.012134318170e+03, 1.012682851650e+03, 1.013229563480e+03, 1.013774467030e+03, 1.014317575510e+03, 1.014858901970e+03, 1.015398459350e+03, 1.015936260410e+03, 1.016472317780e+03, 1.017006643970e+03, 1.017539251320e+03, 1.018070152050e+03, 1.018599358250e+03, 1.019126881870e+03, 1.019652734740e+03, 1.020176928550e+03, 1.020699474870e+03, 1.021220385150e+03, 1.021739670710e+03, 1.022257342770e+03, 1.022773412390e+03, 1.023287890550e+03, 1.023800788100e+03, 1.024312115780e+03, 1.024821884220e+03, 1.025330103920e+03, 1.025836785300e+03, 1.026341938660e+03, 1.026845574190e+03, 1.027347701980e+03, 1.027848332000e+03, 1.028347474160e+03, 1.028845138220e+03, 1.029341333880e+03, 1.029836070710e+03, 1.030329358220e+03, 1.030821205800e+03, 1.031311622750e+03, 1.031800618270e+03, 1.032288201490e+03, 1.032774381440e+03, 1.033259167060e+03, 1.033742567190e+03, 1.034224590610e+03, 1.034705245990e+03, 1.035184541940e+03, 1.035662486960e+03, 1.036139089490e+03, 1.036614357880e+03, 1.037088300400e+03, 1.037560925240e+03, 1.038032240510e+03, 1.038502254260e+03, 1.038970974450e+03, 1.039438408950e+03, 1.039904565590e+03, 1.040369452110e+03, 1.040833076170e+03, 1.041295445370e+03, 1.041756567240e+03, 1.042216449230e+03, 1.042675098740e+03, 1.043132523080e+03, 1.043588729500e+03, 1.044043725210e+03, 1.044497517310e+03, 1.044950112870e+03, 1.045401518890e+03, 1.045851742290e+03, 1.046300789950e+03, 1.046748668680e+03, 1.047195385210e+03, 1.047640946250e+03, 1.048085358420e+03, 1.048528628290e+03, 1.048970762380e+03, 1.049411767140e+03, 1.049851648970e+03, 1.050290414220e+03, 1.050728069170e+03, 1.051164620050e+03, 1.051600073060e+03, 1.052034434320e+03, 1.052467709890e+03, 1.052899905820e+03, 1.053331028060e+03, 1.053761082530e+03, 1.054190075120e+03, 1.054618011640e+03, 1.055044897860e+03, 1.055470739500e+03, 1.055895542250e+03, 1.056319311730e+03, 1.056742053520e+03, 1.057163773160e+03, 1.057584476140e+03, 1.058004167900e+03, 1.058422853840e+03, 1.058840539320e+03, 1.059257229650e+03, 1.059672930090e+03, 1.060087645880e+03, 1.060501382190e+03, 1.060914144170e+03, 1.061325936910e+03, 1.061736765470e+03, 1.062146634880e+03, 1.062555550110e+03, 1.062963516090e+03, 1.063370537730e+03, 1.063776619890e+03, 1.064181767390e+03, 1.064585985010e+03, 1.064989277500e+03, 1.065391649570e+03, 1.065793105890e+03, 1.066193651090e+03, 1.066593289780e+03, 1.066992026520e+03, 1.067389865840e+03, 1.067786812230e+03, 1.068182870150e+03, 1.068578044020e+03, 1.068972338250e+03, 1.069365757170e+03, 1.069758305130e+03, 1.070149986410e+03, 1.070540805270e+03, 1.070930765930e+03, 1.071319872600e+03, 1.071708129440e+03, 1.072095540570e+03, 1.072482110100e+03, 1.072867842110e+03, 1.073252740630e+03, 1.073636809670e+03, 1.074020053220e+03, 1.074402475220e+03, 1.074784079610e+03, 1.075164870270e+03, 1.075544851060e+03, 1.075924025840e+03, 1.076302398400e+03, 1.076679972530e+03, 1.077056751990e+03, 1.077432740490e+03, 1.077807941750e+03, 1.078182359430e+03, 1.078555997180e+03, 1.078928858630e+03, 1.079300947370e+03, 1.079672266960e+03, 1.080042820970e+03, 1.080412612890e+03, 1.080781646240e+03, 1.081149924480e+03, 1.081517451050e+03, 1.081884229380e+03, 1.082250262860e+03, 1.082615554870e+03, 1.082980108760e+03, 1.083343927860e+03, 1.083707015470e+03, 1.084069374880e+03, 1.084431009330e+03, 1.084791922080e+03, 1.085152116320e+03, 1.085511595260e+03, 1.085870362070e+03, 1.086228419890e+03, 1.086585771850e+03, 1.086942421060e+03, 1.087298370610e+03, 1.087653623550e+03, 1.088008182940e+03, 1.088362051790e+03, 1.088715233110e+03, 1.089067729880e+03, 1.089419545070e+03, 1.089770681610e+03, 1.090121142440e+03, 1.090470930460e+03, 1.090820048550e+03, 1.091168499580e+03, 1.091516286400e+03, 1.091863411840e+03, 1.092209878700e+03, 1.092555689790e+03, 1.092900847870e+03, 1.093245355690e+03, 1.093589216010e+03, 1.093932431530e+03, 1.094275004970e+03, 1.094616939000e+03, 1.094958236290e+03, 1.095298899510e+03, 1.095638931270e+03, 1.095978334200e+03, 1.096317110900e+03, 1.096655263950e+03, 1.096992795920e+03, 1.097329709360e+03},
    {1.679084940350e+00, 5.117368847040e+00, 8.614813849050e+00, 1.217394680650e+01, 1.579746920590e+01, 1.948827358590e+01, 2.324946194620e+01, 2.708436643520e+01, 3.099657266680e+01, 3.498994608090e+01, 3.906866183800e+01, 4.323723884030e+01, 4.750057858330e+01, 5.186400969180e+01, 5.633333917570e+01, 6.091491166740e+01, 6.561567818850e+01, 7.044327635800e+01, 7.540612441540e+01, 8.051353202980e+01, 8.577583163570e+01, 9.120453504750e+01, 9.681252143240e+01, 1.026142644910e+02, 1.086261090740e+02, 1.148666106810e+02, 1.213569557220e+02, 1.281214865710e+02, 1.351883640480e+02, 1.425904122990e+02, 1.503662087630e+02, 1.585615079070e+02, 1.672311260320e+02, 1.764414727150e+02, 1.862740035200e+02, 1.968300064180e+02, 2.082373484350e+02, 2.206601381190e+02, 2.343127414620e+02, 2.494801885370e+02, 2.665473101680e+02, 2.860370513730e+02, 3.086478898200e+02, 3.352473160090e+02, 3.667188256950e+02, 4.034305948970e+02, 4.436844665670e+02, 4.828047768620e+02, 5.173221708600e+02, 5.461328103310e+02, 5.697848186550e+02, 5.894196866680e+02, 6.060488091480e+02, 6.204082511820e+02, 6.330148692630e+02, 6.442365633840e+02, 6.543415171640e+02, 6.635298450480e+02, 6.719540584500e+02, 6.797326101800e+02, 6.869590686880e+02, 6.937084676130e+02, 7.000417905110e+02, 7.060091998400e+02, 7.116524038100e+02, 7.170064201450e+02, 7.221009102360e+02, 7.269612020940e+02, 7.316090843110e+02, 7.360634291420e+02, 7.403406864010e+02, 7.444552785790e+02, 7.484199196150e+02, 7.522458741070e+02, 7.559431696270e+02, 7.595207718370e+02, 7.629867298540e+02, 7.663482976550e+02, 7.696120360710e+02, 7.727838989330e+02, 7.758693062290e+02, 7.788732065280e+02, 7.818001305050e+02, 7.846542370480e+02, 7.874393531320e+02, 7.901590084640e+02, 7.928164656820e+02, 7.954147468000e+02, 7.979566564220e+02, 8.004448022150e+02, 8.028816130040e+02, 8.052693548200e+02, 8.076101451840e+02, 8.099059658400e+02, 8.121586741460e+02, 8.143700132940e+02, 8.165416214930e+02, 8.186750402370e+02, 8.207717217860e+02, 8.228330359220e+02, 8.248602760820e+02, 8.268546649290e+02, 8.288173594230e+02, 8.307494554490e+02, 8.326519920360e+02, 8.345259552240e+02, 8.363722816100e+02, 8.381918615970e+02, 8.399855423810e+02, 8.417541307070e+02, 8.434983954000e+02, 8.452190697130e+02, 8.469168534800e+02, 8.485924151280e+02, 8.502463935250e+02, 8.518793997050e+02, 8.534920184600e+02, 8.550848098270e+02, 8.566583104710e+02, 8.582130349670e+02, 8.597494770050e+02, 8.612681105050e+02, 8.627693906660e+02, 8.642537549410e+02, 8.657216239540e+02, 8.671734023550e+02, 8.686094796260e+02, 8.700302308300e+02, 8.714360173220e+02, 8.728271874130e+02, 8.742040769950e+02, 8.755670101270e+02, 8.769162995910e+02, 8.782522474120e+02, 8.795751453530e+02, 8.808852753740e+02, 8.821829100770e+02, 8.834683131140e+02, 8.847417395820e+02, 8.860034363900e+02, 8.872536426150e+02, 8.884925898280e+02, 8.897205024120e+02, 8.909375978600e+02, 8.921440870560e+02, 8.933401745460e+02, 8.945260587890e+02, 8.957019324050e+02, 8.968679824020e+02, 8.980243903920e+02, 8.991713328060e+02, 9.003089810890e+02, 9.014375018880e+02, 9.025570572340e+02, 9.036678047120e+02, 9.047698976260e+02, 9.058634851570e+02, 9.069487125070e+02, 9.080257210480e+02, 9.090946484520e+02, 9.101556288240e+02, 9.112087928290e+02, 9.122542678040e+02, 9.132921778800e+02, 9.143226440840e+02, 9.153457844480e+02, 9.163617141070e+02, 9.173705453950e+02, 9.183723879380e+02, 9.193673487380e+02, 9.203555322660e+02, 9.213370405320e+02, 9.223119731730e+02, 9.232804275220e+02, 9.242424986810e+02, 9.251982795900e+02, 9.261478610950e+02, 9.270913320090e+02, 9.280287791770e+02, 9.289602875300e+02, 9.298859401470e+02, 9.308058183080e+02, 9.317200015440e+02, 9.326285676910e+02, 9.335315929380e+02, 9.344291518740e+02, 9.353213175350e+02, 9.362081614430e+02, 9.370897536540e+02, 9.379661627970e+02, 9.388374561090e+02, 9.397036994820e+02, 9.405649574900e+02, 9.414212934320e+02, 9.422727693610e+02, 9.431194461210e+02, 9.439613833750e+02, 9.447986396410e+02, 9.456312723170e+02, 9.464593377130e+02, 9.472828910780e+02, 9.481019866250e+02, 9.489166775630e+02, 9.497270161150e+02, 9.505330535490e+02, 9.513348401970e+02, 9.521324254810e+02, 9.529258579350e+02, 9.537151852250e+02, 9.545004541710e+02, 9.552817107670e+02, 9.560590002040e+02, 9.568323668820e+02, 9.576018544360e+02, 9.583675057490e+02, 9.591293629720e+02, 9.598874675410e+02, 9.606418601900e+02, 9.613925809730e+02, 9.621396692750e+02, 9.628831638270e+02, 9.636231027240e+02, 9.643595234350e+02, 9.650924628220e+02, 9.658219571470e+02, 9.665480420920e+02, 9.672707527660e+02, 9.679901237210e+02, 9.687061889610e+02, 9.694189819570e+02, 9.701285356570e+02, 9.708348824950e+02, 9.715380544060e+02, 9.722380828320e+02, 9.729349987360e+02, 9.736288326090e+02, 9.743196144820e+02, 9.750073739350e+02, 9.756921401040e+02, 9.763739416920e+02, 9.770528069780e+02, 9.777287638260e+02, 9.784018396890e+02, 9.790720616240e+02, 9.797394562920e+02, 9.804040499730e+02, 9.810658685700e+02, 9.817249376150e+02, 9.823812822790e+02, 9.830349273770e+02, 9.836858973760e+02, 9.843342164020e+02, 9.849799082430e+02, 9.856229963610e+02, 9.862635038930e+02, 9.869014536590e+02, 9.875368681700e+02, 9.881697696290e+02, 9.888001799420e+02, 9.894281207190e+02, 9.900536132810e+02, 9.906766786670e+02, 9.912973376360e+02, 9.919156106730e+02, 9.925315179970e+02, 9.931450795600e+02, 9.937563150570e+02, 9.943652439270e+02, 9.949718853590e+02, 9.955762582970e+02, 9.961783814420e+02, 9.967782732590e+02, 9.973759519790e+02, 9.979714356030e+02, 9.985647419090e+02, 9.991558884490e+02, 9.997448925610e+02, 1.000331771370e+03, 1.000916541780e+03, 1.001499220500e+03, 1.002079824040e+03, 1.002658368680e+03, 1.003234870550e+03, 1.003809345530e+03, 1.004381809370e+03, 1.004952277570e+03, 1.005520765500e+03, 1.006087288320e+03, 1.006651861010e+03, 1.007214498380e+03, 1.007775215080e+03, 1.008334025560e+03, 1.008890944130e+03, 1.009445984930e+03, 1.009999161920e+03, 1.010550488920e+03, 1.011099979570e+03, 1.011647647400e+03, 1.012193505720e+03, 1.012737567760e+03, 1.013279846550e+03, 1.013820354990e+03, 1.014359105850e+03, 1.014896111740e+03, 1.015431385150e+03, 1.015964938410e+03, 1.016496783730e+03, 1.017026933180e+03, 1.017555398700e+03, 1.018082192120e+03, 1.018607325100e+03, 1.019130809220e+03, 1.019652655900e+03, 1.020172876460e+03, 1.020691482100e+03, 1.021208483890e+03, 1.021723892790e+03, 1.022237719630e+03, 1.022749975160e+03, 1.023260669990e+03, 1.023769814630e+03, 1.024277419470e+03, 1.024783494810e+03, 1.025288050840e+03, 1.025791097620e+03, 1.026292645160e+03, 1.026792703320e+03, 1.027291281870e+03, 1.027788390510e+03, 1.028284038800e+03, 1.028778236240e+03, 1.029270992220e+03, 1.029762316030e+03, 1.030252216880e+03, 1.030740703880e+03, 1.031227786070e+03, 1.031713472370e+03, 1.032197771630e+03, 1.032680692620e+03, 1.033162244010e+03, 1.033642434400e+03, 1.034121272300e+03, 1.034598766140e+03, 1.035074924260e+03, 1.035549754940e+03, 1.036023266360e+03, 1.036495466640e+03, 1.036966363820e+03, 1.037435965850e+03, 1.037904280610e+03, 1.038371315930e+03, 1.038837079530e+03, 1.039301579100e+03, 1.039764822210e+03, 1.040226816400e+03, 1.040687569130e+03, 1.041147087780e+03, 1.041605379670e+03, 1.042062452060e+03, 1.042518312140e+03, 1.042972967030e+03, 1.043426423780e+03, 1.043878689390e+03, 1.044329770800e+03, 1.044779674880e+03, 1.045228408430e+03, 1.045675978210e+03, 1.046122390900e+03, 1.046567653130e+03, 1.047011771480e+03, 1.047454752450e+03, 1.047896602510e+03, 1.048337328060e+03, 1.048776935440e+03, 1.049215430940e+03, 1.049652820800e+03, 1.050089111190e+03, 1.050524308260e+03, 1.050958418070e+03, 1.051391446660e+03, 1.051823399980e+03, 1.052254283970e+03, 1.052684104500e+03, 1.053112867400e+03, 1.053540578430e+03, 1.053967243320e+03, 1.054392867750e+03, 1.054817457350e+03, 1.055241017700e+03, 1.055663554350e+03, 1.056085072780e+03, 1.056505578440e+03, 1.056925076730e+03, 1.057343573010e+03, 1.057761072590e+03, 1.058177580750e+03, 1.058593102700e+03, 1.059007643650e+03, 1.059421208720e+03, 1.059833803030e+03, 1.060245431630e+03, 1.060656099550e+03, 1.061065811760e+03, 1.061474573200e+03, 1.061882388790e+03, 1.062289263370e+03, 1.062695201780e+03, 1.063100208800e+03, 1.063504289180e+03, 1.063907447640e+03, 1.064309688840e+03, 1.064711017420e+03, 1.065111438000e+03, 1.065510955130e+03, 1.065909573340e+03, 1.066307297150e+03, 1.066704130990e+03, 1.067100079310e+03, 1.067495146500e+03, 1.067889336920e+03, 1.068282654900e+03, 1.068675104730e+03, 1.069066690670e+03, 1.069457416960e+03, 1.069847287790e+03, 1.070236307330e+03, 1.070624479710e+03, 1.071011809050e+03, 1.071398299410e+03, 1.071783954830e+03, 1.072168779350e+03, 1.072552776930e+03, 1.072935951530e+03, 1.073318307080e+03, 1.073699847480e+03, 1.074080576600e+03, 1.074460498270e+03, 1.074839616310e+03, 1.075217934500e+03, 1.075595456610e+03, 1.075972186370e+03, 1.076348127460e+03, 1.076723283590e+03, 1.077097658390e+03, 1.077471255490e+03, 1.077844078490e+03, 1.078216130960e+03, 1.078587416450e+03, 1.078957938490e+03, 1.079327700570e+03, 1.079696706170e+03, 1.080064958730e+03, 1.080432461680e+03, 1.080799218430e+03, 1.081165232350e+03, 1.081530506780e+03, 1.081895045080e+03, 1.082258850530e+03, 1.082621926430e+03, 1.082984276050e+03, 1.083345902610e+03, 1.083706809340e+03, 1.084066999430e+03, 1.084426476050e+03, 1.084785242370e+03, 1.085143301510e+03, 1.085500656570e+03, 1.085857310660e+03, 1.086213266830e+03, 1.086568528140e+03, 1.086923097610e+03, 1.087276978240e+03, 1.087630173030e+03, 1.087982684940e+03, 1.088334516920e+03, 1.088685671900e+03, 1.089036152780e+03, 1.089385962450e+03, 1.089735103790e+03, 1.090083579640e+03, 1.090431392830e+03, 1.090778546190e+03, 1.091125042500e+03, 1.091470884550e+03, 1.091816075080e+03, 1.092160616860e+03, 1.092504512590e+03, 1.092847764980e+03, 1.093190376730e+03, 1.093532350510e+03, 1.093873688960e+03, 1.094214394730e+03, 1.094554470430e+03, 1.094893918680e+03, 1.095232742060e+03},
    {1.673616006780e+00, 5.100233649670e+00, 8.585145124480e+00, 1.213080457740e+01, 1.573983310140e+01, 1.941503415640e+01, 2.315941081290e+01, 2.697618508690e+01, 3.086881967980e+01, 3.484104249300e+01, 3.889687435560e+01, 4.304066048890e+01, 4.727710632960e+01, 5.161131846350e+01, 5.604885157240e+01, 6.059576249500e+01, 6.525867273780e+01, 7.004484108270e+01, 7.496224831770e+01, 8.001969661320e+01, 8.522692669340e+01, 9.059475677670e+01, 9.613524831920e+01, 1.018619050080e+02, 1.077899133130e+02, 1.139364354080e+02, 1.203209686530e+02, 1.269657904670e+02, 1.338965138030e+02, 1.411427873640e+02, 1.487391873450e+02, 1.567263655280e+02, 1.651525446630e+02, 1.740754902980e+02, 1.835651445950e+02, 1.937071912710e+02, 2.046079443930e+02, 2.164011320360e+02, 2.292573856580e+02, 2.433975032020e+02, 2.591105842150e+02, 2.767770261320e+02, 2.968913442190e+02, 3.200647358010e+02, 3.469583906680e+02, 3.780665414800e+02, 4.131548022250e+02, 4.501593619380e+02, 4.854813170050e+02, 5.168778554850e+02, 5.436107858870e+02, 5.660465741840e+02, 5.849979391390e+02, 6.012459632380e+02, 6.153986252430e+02, 6.279027276920e+02, 6.390859835630e+02, 6.491924485170e+02, 6.584072837480e+02, 6.668735201330e+02, 6.747034948860e+02, 6.819867815750e+02, 6.887957846020e+02, 6.951897539170e+02, 7.012177142970e+02, 7.069206369630e+02, 7.123330738310e+02, 7.174844045390e+02, 7.223998001460e+02, 7.271009765080e+02, 7.316067894180e+02, 7.359337092370e+02, 7.400962027050e+02, 7.441070425690e+02, 7.479775605150e+02, 7.517178552280e+02, 7.553369646430e+02, 7.588430094240e+02, 7.622433131620e+02, 7.655445036260e+02, 7.687525984970e+02, 7.718730783190e+02, 7.749109488760e+02, 7.778707947620e+02, 7.807568255850e+02, 7.835729159940e+02, 7.863226404760e+02, 7.890093037340e+02, 7.916359673000e+02, 7.942054729180e+02, 7.967204631810e+02, 7.991833997750e+02, 8.015965796800e+02, 8.039621495770e+02, 8.062821187140e+02, 8.085583704130e+02, 8.107926723930e+02, 8.129866860500e+02, 8.151419748260e+02, 8.172600117600e+02, 8.193421863310e+02, 8.213898106580e+02, 8.234041251380e+02, 8.253863035760e+02, 8.273374578680e+02, 8.292586422750e+02, 8.311508573380e+02, 8.330150534620e+02, 8.348521342110e+02, 8.366629593350e+02, 8.384483475550e+02, 8.402090791310e+02, 8.419458982400e+02, 8.436595151590e+02, 8.453506082990e+02, 8.470198260830e+02, 8.486677886890e+02, 8.502950896690e+02, 8.519022974540e+02, 8.534899567540e+02, 8.550585898610e+02, 8.566086978690e+02, 8.581407618050e+02, 8.596552436930e+02, 8.611525875430e+02, 8.626332202790e+02, 8.640975526110e+02, 8.655459798480e+02, 8.669788826610e+02, 8.683966278010e+02, 8.697995687770e+02, 8.711880464860e+02, 8.725623898120e+02, 8.739229161850e+02, 8.752699321160e+02, 8.766037336910e+02, 8.779246070430e+02, 8.792328288020e+02, 8.805286665080e+02, 8.818123790140e+02, 8.830842168580e+02, 8.843444226220e+02, 8.855932312690e+02, 8.868308704580e+02, 8.880575608530e+02, 8.892735164050e+02, 8.904789446270e+02, 8.916740468560e+02, 8.928590184910e+02, 8.940340492360e+02, 8.951993233160e+02, 8.963550196910e+02, 8.975013122600e+02, 8.986383700480e+02, 8.997663573920e+02, 9.008854341160e+02, 9.019957556960e+02, 9.030974734180e+02, 9.041907345310e+02, 9.052756823930e+02, 9.063524566060e+02, 9.074211931490e+02, 9.084820245070e+02, 9.095350797880e+02, 9.105804848430e+02, 9.116183623710e+02, 9.126488320330e+02, 9.136720105440e+02, 9.146880117800e+02, 9.156969468620e+02, 9.166989242540e+02, 9.176940498430e+02, 9.186824270250e+02, 9.196641567800e+02, 9.206393377540e+02, 9.216080663270e+02, 9.225704366820e+02, 9.235265408800e+02, 9.244764689170e+02, 9.254203087900e+02, 9.263581465550e+02, 9.272900663890e+02, 9.282161506400e+02, 9.291364798840e+02, 9.300511329770e+02, 9.309601871010e+02, 9.318637178160e+02, 9.327617991050e+02, 9.336545034160e+02, 9.345419017090e+02, 9.354240634940e+02, 9.363010568740e+02, 9.371729485810e+02, 9.380398040140e+02, 9.389016872760e+02, 9.397586612070e+02, 9.406107874200e+02, 9.414581263330e+02, 9.423007371960e+02, 9.431386781290e+02, 9.439720061460e+02, 9.448007771830e+02, 9.456250461300e+02, 9.464448668530e+02, 9.472602922220e+02, 9.480713741370e+02, 9.488781635490e+02, 9.496807104860e+02, 9.504790640750e+02, 9.512732725620e+02, 9.520633833390e+02, 9.528494429580e+02, 9.536314971530e+02, 9.544095908630e+02, 9.551837682470e+02, 9.559540727020e+02, 9.567205468830e+02, 9.574832327210e+02, 9.582421714360e+02, 9.589974035560e+02, 9.597489689320e+02, 9.604969067540e+02, 9.612412555640e+02, 9.619820532730e+02, 9.627193371720e+02, 9.634531439510e+02, 9.641835097040e+02, 9.649104699500e+02, 9.656340596400e+02, 9.663543131740e+02, 9.670712644090e+02, 9.677849466690e+02, 9.684953927640e+02, 9.692026349920e+02, 9.699067051560e+02, 9.706076345710e+02, 9.713054540740e+02, 9.720001940390e+02, 9.726918843770e+02, 9.733805545550e+02, 9.740662335990e+02, 9.747489501050e+02, 9.754287322490e+02, 9.761056077920e+02, 9.767796040910e+02, 9.774507481040e+02, 9.781190664040e+02, 9.787845851790e+02, 9.794473302430e+02, 9.801073270450e+02, 9.807646006740e+02, 9.814191758630e+02, 9.820710770030e+02, 9.827203281420e+02, 9.833669529970e+02, 9.840109749560e+02, 9.846524170890e+02, 9.852913021470e+02, 9.859276525750e+02, 9.865614905140e+02, 9.871928378050e+02, 9.878217160010e+02, 9.884481463630e+02, 9.890721498720e+02, 9.896937472340e+02, 9.903129588790e+02, 9.909298049720e+02, 9.915443054180e+02, 9.921564798590e+02, 9.927663476880e+02, 9.933739280460e+02, 9.939792398330e+02, 9.945823017060e+02, 9.951831320850e+02, 9.957817491620e+02, 9.963781708960e+02, 9.969724150240e+02, 9.975644990630e+02, 9.981544403130e+02, 9.987422558590e+02, 9.993279625780e+02, 9.999115771400e+02, 1.000493116010e+03, 1.001072595470e+03, 1.001650031570e+03, 1.002225440210e+03, 1.002798837060e+03, 1.003370237640e+03, 1.003939657260e+03, 1.004507111060e+03, 1.005072614000e+03, 1.005636180860e+03, 1.006197826270e+03, 1.006757564650e+03, 1.007315410280e+03, 1.007871377290e+03, 1.008425479610e+03, 1.008977731040e+03, 1.009528145210e+03, 1.010076735610e+03, 1.010623515550e+03, 1.011168498220e+03, 1.011711696650e+03, 1.012253123720e+03, 1.012792792170e+03, 1.013330714610e+03, 1.013866903490e+03, 1.014401371150e+03, 1.014934129770e+03, 1.015465191410e+03, 1.015994567990e+03, 1.016522271330e+03, 1.017048313080e+03, 1.017572704800e+03, 1.018095457900e+03, 1.018616583690e+03, 1.019136093360e+03, 1.019653997960e+03, 1.020170308430e+03, 1.020685035630e+03, 1.021198190250e+03, 1.021709782920e+03, 1.022219824130e+03, 1.022728324260e+03, 1.023235293610e+03, 1.023740742350e+03, 1.024244680540e+03, 1.024747118170e+03, 1.025248065100e+03, 1.025747531110e+03, 1.026245525860e+03, 1.026742058920e+03, 1.027237139780e+03, 1.027730777830e+03, 1.028222982350e+03, 1.028713762530e+03, 1.029203127500e+03, 1.029691086270e+03, 1.030177647770e+03, 1.030662820840e+03, 1.031146614240e+03, 1.031629036650e+03, 1.032110096660e+03, 1.032589802760e+03, 1.033068163390e+03, 1.033545186890e+03, 1.034020881520e+03, 1.034495255480e+03, 1.034968316860e+03, 1.035440073710e+03, 1.035910533980e+03, 1.036379705550e+03, 1.036847596240e+03, 1.037314213770e+03, 1.037779565810e+03, 1.038243659960e+03, 1.038706503740e+03, 1.039168104610e+03, 1.039628469950e+03, 1.040087607070e+03, 1.040545523240e+03, 1.041002225640e+03, 1.041457721390e+03, 1.041912017550e+03, 1.042365121110e+03, 1.042817039000e+03, 1.043267778100e+03, 1.043717345210e+03, 1.044165747080e+03, 1.044612990400e+03, 1.045059081800e+03, 1.045504027850e+03, 1.045947835070e+03, 1.046390509910e+03, 1.046832058760e+03, 1.047272487990e+03, 1.047711803870e+03, 1.048150012640e+03, 1.048587120500e+03, 1.049023133550e+03, 1.049458057890e+03, 1.049891899540e+03, 1.050324664470e+03, 1.050756358600e+03, 1.051186987810e+03, 1.051616557930e+03, 1.052045074720e+03, 1.052472543910e+03, 1.052898971190e+03, 1.053324362190e+03, 1.053748722480e+03, 1.054172057620e+03, 1.054594373080e+03, 1.055015674330e+03, 1.055435966760e+03, 1.055855255730e+03, 1.056273546550e+03, 1.056690844510e+03, 1.057107154830e+03, 1.057522482690e+03, 1.057936833240e+03, 1.058350211590e+03, 1.058762622790e+03, 1.059174071880e+03, 1.059584563830e+03, 1.059994103580e+03, 1.060402696050e+03, 1.060810346090e+03, 1.061217058530e+03, 1.061622838160e+03, 1.062027689730e+03, 1.062431617960e+03, 1.062834627510e+03, 1.063236723050e+03, 1.063637909160e+03, 1.064038190410e+03, 1.064437571350e+03, 1.064836056470e+03, 1.065233650230e+03, 1.065630357070e+03, 1.066026181380e+03, 1.066421127520e+03, 1.066815199830e+03, 1.067208402600e+03, 1.067600740110e+03, 1.067992216570e+03, 1.068382836190e+03, 1.068772603140e+03, 1.069161521560e+03, 1.069549595560e+03, 1.069936829220e+03, 1.070323226570e+03, 1.070708791650e+03, 1.071093528420e+03, 1.071477440860e+03, 1.071860532900e+03, 1.072242808420e+03, 1.072624271310e+03, 1.073004925410e+03, 1.073384774530e+03, 1.073763822460e+03, 1.074142072970e+03, 1.074519529780e+03, 1.074896196600e+03, 1.075272077120e+03, 1.075647174980e+03, 1.076021493820e+03, 1.076395037230e+03, 1.076767808790e+03, 1.077139812060e+03, 1.077511050560e+03, 1.077881527800e+03, 1.078251247230e+03, 1.078620212330e+03, 1.078988426520e+03, 1.079355893200e+03, 1.079722615760e+03, 1.080088597540e+03, 1.080453841890e+03, 1.080818352110e+03, 1.081182131490e+03, 1.081545183300e+03, 1.081907510780e+03, 1.082269117160e+03, 1.082630005620e+03, 1.082990179350e+03, 1.083349641510e+03, 1.083708395220e+03, 1.084066443600e+03, 1.084423789740e+03, 1.084780436710e+03, 1.085136387570e+03, 1.085491645340e+03, 1.085846213030e+03, 1.086200093630e+03, 1.086553290120e+03, 1.086905805440e+03, 1.087257642520e+03, 1.087608804270e+03, 1.087959293600e+03, 1.088309113360e+03, 1.088658266410e+03, 1.089006755590e+03, 1.089354583720e+03, 1.089701753590e+03, 1.090048267990e+03, 1.090394129670e+03, 1.090739341380e+03, 1.091083905860e+03, 1.091427825800e+03, 1.091771103900e+03, 1.092113742840e+03, 1.092455745260e+03, 1.092797113820e+03, 1.093137851140e+03},
    {1.668183086220e+00, 5.083217682440e+00, 8.555694396840e+00, 1.208799690670e+01, 1.568266850740e+01, 1.934242658830e+01, 2.307017878680e+01, 2.686904105980e+01, 3.074235795620e+01, 3.469372541840e+01, 3.872701650540e+01, 4.284641049990e+01, 4.705642594950e+01, 5.136195830250e+01, 5.576832292790e+01, 6.028130447590e+01, 6.490721373810e+01, 6.965295341980e+01, 7.452609456060e+01, 7.953496574170e+01, 8.468875773870e+01, 8.999764694000e+01, 9.547294171110e+01, 1.011272570000e+02, 1.069747239540e+02, 1.130312432400e+02, 1.193147933930e+02, 1.258458089550e+02, 1.326476479720e+02, 1.397471748790e+02, 1.471754938620e+02, 1.549688803670e+02, 1.631699762130e+02, 1.718293389920e+02, 1.810074724130e+02, 1.907775154860e+02, 2.012288404170e+02, 2.124719065770e+02, 2.246448372020e+02, 2.379222900530e+02, 2.525271382010e+02, 2.687447994780e+02, 2.869374749750e+02, 3.075483183510e+02, 3.310709472020e+02, 3.579438534970e+02, 3.883124753790e+02, 4.215241864490e+02, 4.555282895980e+02, 4.876654918680e+02, 5.164429914650e+02, 5.413538005420e+02, 5.626502869250e+02, 5.809185057180e+02, 5.967608786020e+02, 6.106758079700e+02, 6.230464831980e+02, 6.341628329390e+02, 6.442455809710e+02, 6.534650066920e+02, 6.619544165080e+02, 6.698196746150e+02, 6.771459842570e+02, 6.840027688590e+02, 6.904472324770e+02, 6.965269930050e+02, 7.022820567060e+02, 7.077463188240e+02, 7.129487186020e+02, 7.179141389000e+02, 7.226641145150e+02, 7.272173954590e+02, 7.315903989400e+02, 7.357975750600e+02, 7.398517049310e+02, 7.437641454130e+02, 7.475450313400e+02, 7.512034436300e+02, 7.547475498420e+02, 7.581847223320e+02, 7.615216380860e+02, 7.647643634870e+02, 7.679184266270e+02, 7.709888792710e+02, 7.739803501760e+02, 7.768970911660e+02, 7.797430171030e+02, 7.825217406860e+02, 7.852366028730e+02, 7.878906995460e+02, 7.904869049830e+02, 7.930278925760e+02, 7.955161531670e+02, 7.979540113340e+02, 8.003436398960e+02, 8.026870728630e+02, 8.049862170190e+02, 8.072428623340e+02, 8.094586913110e+02, 8.116352874280e+02, 8.137741427540e+02, 8.158766648510e+02, 8.179441830380e+02, 8.199779540720e+02, 8.219791673380e+02, 8.239489495700e+02, 8.258883691740e+02, 8.277984401780e+02, 8.296801258590e+02, 8.315343420690e+02, 8.333619602960e+02, 8.351638104780e+02, 8.369406836050e+02, 8.386933341120e+02, 8.404224821010e+02, 8.421288153870e+02, 8.438129914050e+02, 8.454756389690e+02, 8.471173599130e+02, 8.487387306110e+02, 8.503403033990e+02, 8.519226078880e+02, 8.534861522020e+02, 8.550314241250e+02, 8.565588921730e+02, 8.580690066010e+02, 8.595622003420e+02, 8.610388898840e+02, 8.624994761010e+02, 8.639443450230e+02, 8.653738685640e+02, 8.667884052070e+02, 8.681883006440e+02, 8.695738883850e+02, 8.709454903190e+02, 8.723034172630e+02, 8.736479694560e+02, 8.749794370440e+02, 8.762981005300e+02, 8.776042311980e+02, 8.788980915180e+02, 8.801799355240e+02, 8.814500091810e+02, 8.827085507210e+02, 8.839557909670e+02, 8.851919536450e+02, 8.864172556720e+02, 8.876319074290e+02, 8.888361130340e+02, 8.900300705790e+02, 8.912139723770e+02, 8.923880051840e+02, 8.935523504120e+02, 8.947071843380e+02, 8.958526782940e+02, 8.969889988560e+02, 8.981163080200e+02, 8.992347633700e+02, 9.003445182400e+02, 9.014457218680e+02, 9.025385195410e+02, 9.036230527390e+02, 9.046994592640e+02, 9.057678733740e+02, 9.068284258990e+02, 9.078812443660e+02, 9.089264531050e+02, 9.099641733590e+02, 9.109945233890e+02, 9.120176185700e+02, 9.130335714860e+02, 9.140424920230e+02, 9.150444874540e+02, 9.160396625220e+02, 9.170281195250e+02, 9.180099583880e+02, 9.189852767370e+02, 9.199541699760e+02, 9.209167313480e+02, 9.218730520080e+02, 9.228232210790e+02, 9.237673257180e+02, 9.247054511740e+02, 9.256376808440e+02, 9.265640963240e+02, 9.274847774660e+02, 9.283998024290e+02, 9.293092477210e+02, 9.302131882540e+02, 9.311116973830e+02, 9.320048469530e+02, 9.328927073400e+02, 9.337753474920e+02, 9.346528349650e+02, 9.355252359690e+02, 9.363926153940e+02, 9.372550368530e+02, 9.381125627140e+02, 9.389652541320e+02, 9.398131710800e+02, 9.406563723850e+02, 9.414949157490e+02, 9.423288577880e+02, 9.431582540510e+02, 9.439831590530e+02, 9.448036262980e+02, 9.456197083070e+02, 9.464314566390e+02, 9.472389219170e+02, 9.480421538510e+02, 9.488412012600e+02, 9.496361120930e+02, 9.504269334520e+02, 9.512137116080e+02, 9.519964920260e+02, 9.527753193810e+02, 9.535502375770e+02, 9.543212897650e+02, 9.550885183620e+02, 9.558519650640e+02, 9.566116708680e+02, 9.573676760840e+02, 9.581200203510e+02, 9.588687426520e+02, 9.596138813330e+02, 9.603554741100e+02, 9.610935580870e+02, 9.618281697710e+02, 9.625593450820e+02, 9.632871193650e+02, 9.640115274080e+02, 9.647326034460e+02, 9.654503811790e+02, 9.661648937830e+02, 9.668761739160e+02, 9.675842537350e+02, 9.682891649020e+02, 9.689909385970e+02, 9.696896055260e+02, 9.703851959340e+02, 9.710777396100e+02, 9.717672658970e+02, 9.724538037070e+02, 9.731373815210e+02, 9.738180274030e+02, 9.744957690080e+02, 9.751706335860e+02, 9.758426479970e+02, 9.765118387120e+02, 9.771782318240e+02, 9.778418530540e+02, 9.785027277600e+02, 9.791608809410e+02, 9.798163372470e+02, 9.804691209830e+02, 9.811192561170e+02, 9.817667662860e+02, 9.824116748020e+02, 9.830540046590e+02, 9.836937785380e+02, 9.843310188100e+02, 9.849657475500e+02, 9.855979865310e+02, 9.862277572390e+02, 9.868550808740e+02, 9.874799783560e+02, 9.881024703280e+02, 9.887225771640e+02, 9.893403189730e+02, 9.899557156020e+02, 9.905687866410e+02, 9.911795514300e+02, 9.917880290610e+02, 9.923942383820e+02, 9.929981980020e+02, 9.935999262970e+02, 9.941994414100e+02, 9.947967612570e+02, 9.953919035330e+02, 9.959848857110e+02, 9.965757250500e+02, 9.971644385970e+02, 9.977510431890e+02, 9.983355554600e+02, 9.989179918390e+02, 9.994983685610e+02, 1.000076701660e+03, 1.000653006990e+03, 1.001227300190e+03, 1.001799596750e+03, 1.002369911950e+03, 1.002938260900e+03, 1.003504658520e+03, 1.004069119570e+03, 1.004631658650e+03, 1.005192290160e+03, 1.005751028350e+03, 1.006307887320e+03, 1.006862880980e+03, 1.007416023100e+03, 1.007967327290e+03, 1.008516807020e+03, 1.009064475570e+03, 1.009610346120e+03, 1.010154431660e+03, 1.010696745070e+03, 1.011237299060e+03, 1.011776106210e+03, 1.012313178980e+03, 1.012848529670e+03, 1.013382170440e+03, 1.013914113340e+03, 1.014444370290e+03, 1.014972953060e+03, 1.015499873310e+03, 1.016025142560e+03, 1.016548772240e+03, 1.017070773620e+03, 1.017591157870e+03, 1.018109936040e+03, 1.018627119080e+03, 1.019142717780e+03, 1.019656742870e+03, 1.020169204940e+03, 1.020680114470e+03, 1.021189481850e+03, 1.021697317340e+03, 1.022203631110e+03, 1.022708433230e+03, 1.023211733650e+03, 1.023713542240e+03, 1.024213868750e+03, 1.024712722840e+03, 1.025210114090e+03, 1.025706051970e+03, 1.026200545840e+03, 1.026693605000e+03, 1.027185238630e+03, 1.027675455840e+03, 1.028164265630e+03, 1.028651676930e+03, 1.029137698590e+03, 1.029622339340e+03, 1.030105607860e+03, 1.030587512730e+03, 1.031068062440e+03, 1.031547265430e+03, 1.032025130030e+03, 1.032501664490e+03, 1.032976877000e+03, 1.033450775670e+03, 1.033923368520e+03, 1.034394663510e+03, 1.034864668500e+03, 1.035333391320e+03, 1.035800839690e+03, 1.036267021260e+03, 1.036731943640e+03, 1.037195614340e+03, 1.037658040820e+03, 1.038119230450e+03, 1.038579190550e+03, 1.039037928380e+03, 1.039495451110e+03, 1.039951765860e+03, 1.040406879700e+03, 1.040860799610e+03, 1.041313532530e+03, 1.041765085310e+03, 1.042215464770e+03, 1.042664677650e+03, 1.043112730640e+03, 1.043559630370e+03, 1.044005383410e+03, 1.044449996270e+03, 1.044893475410e+03, 1.045335827220e+03, 1.045777058060e+03, 1.046217174200e+03, 1.046656181890e+03, 1.047094087300e+03, 1.047530896580e+03, 1.047966615780e+03, 1.048401250940e+03, 1.048834808030e+03, 1.049267292980e+03, 1.049698711660e+03, 1.050129069880e+03, 1.050558373440e+03, 1.050986628040e+03, 1.051413839380e+03, 1.051840013090e+03, 1.052265154740e+03, 1.052689269880e+03, 1.053112364010e+03, 1.053534442560e+03, 1.053955510950e+03, 1.054375574540e+03, 1.054794638640e+03, 1.055212708520e+03, 1.055629789410e+03, 1.056045886500e+03, 1.056461004950e+03, 1.056875149840e+03, 1.057288326260e+03, 1.057700539210e+03, 1.058111793680e+03, 1.058522094630e+03, 1.058931446950e+03, 1.059339855510e+03, 1.059747325130e+03, 1.060153860610e+03, 1.060559466700e+03, 1.060964148120e+03, 1.061367909540e+03, 1.061770755610e+03, 1.062172690930e+03, 1.062573720070e+03, 1.062973847580e+03, 1.063373077940e+03, 1.063771415630e+03, 1.064168865080e+03, 1.064565430680e+03, 1.064961116810e+03, 1.065355927800e+03, 1.065749867930e+03, 1.066142941490e+03, 1.066535152700e+03, 1.066926505770e+03, 1.067317004870e+03, 1.067706654140e+03, 1.068095457700e+03, 1.068483419600e+03, 1.068870543920e+03, 1.069256834670e+03, 1.069642295830e+03, 1.070026931370e+03, 1.070410745210e+03, 1.070793741270e+03, 1.071175923410e+03, 1.071557295480e+03, 1.071937861300e+03, 1.072317624660e+03, 1.072696589320e+03, 1.073074759020e+03, 1.073452137460e+03, 1.073828728330e+03, 1.074204535290e+03, 1.074579561970e+03, 1.074953811960e+03, 1.075327288860e+03, 1.075699996200e+03, 1.076071937530e+03, 1.076443116340e+03, 1.076813536120e+03, 1.077183200320e+03, 1.077552112360e+03, 1.077920275660e+03, 1.078287693600e+03, 1.078654369540e+03, 1.079020306810e+03, 1.079385508740e+03, 1.079749978600e+03, 1.080113719660e+03, 1.080476735180e+03, 1.080839028380e+03, 1.081200602450e+03, 1.081561460580e+03, 1.081921605920e+03, 1.082281041620e+03, 1.082639770780e+03, 1.082997796500e+03, 1.083355121860e+03, 1.083711749900e+03, 1.084067683670e+03, 1.084422926170e+03, 1.084777480390e+03, 1.085131349310e+03, 1.085484535880e+03, 1.085837043040e+03, 1.086188873690e+03, 1.086540030740e+03, 1.086890517060e+03, 1.087240335500e+03, 1.087589488900e+03, 1.087937980080e+03, 1.088285811850e+03, 1.088632986990e+03, 1.088979508260e+03, 1.089325378410e+03, 1.089670600170e+03, 1.090015176250e+03, 1.090359109350e+03, 1.090702402140e+03, 1.091045057280e+03},
    {1.662785816950e+00, 5.066319635170e+00, 8.526459062950e+00, 1.204551947610e+01, 1.562596887340e+01, 1.927044146620e+01, 2.298175281250e+01, 2.676291670920e+01, 3.061716409750e+01, 3.454796427500e+01, 3.855904876260e+01, 4.265443823740e+01, 4.683847301710e+01, 5.111584767410e+01, 5.549165047390e+01, 5.997140846540e+01, 6.456113922980e+01, 6.926741050130e+01, 7.409740914500e+01, 7.905902130860e+01, 8.416092599070e+01, 8.941270480500e+01, 9.482497141200e+01, 1.004095249760e+02, 1.061795331650e+02, 1.121497517150e+02, 1.183367895750e+02, 1.247594312870e+02, 1.314390317940e+02, 1.384000036120e+02, 1.456704227990e+02, 1.532827889610e+02, 1.612749866840e+02, 1.696915125440e+02, 1.785850549120e+02, 1.880185452980e+02, 1.980678421440e+02, 2.088252611430e+02, 2.204042243580e+02, 2.329453364190e+02, 2.466241228870e+02, 2.616602378470e+02, 2.783265440690e+02, 2.969527936040e+02, 3.179111254110e+02, 3.415610975040e+02, 3.681283136610e+02, 3.974764425120e+02, 4.287203920240e+02, 4.600279029950e+02, 4.894703807610e+02, 5.160208590550e+02, 5.393242015460e+02, 5.595591491100e+02, 5.771543629780e+02, 5.925752998330e+02, 6.062283672130e+02, 6.184398510270e+02, 6.294645580250e+02, 6.395009466200e+02, 6.487047564480e+02, 6.571995616190e+02, 6.650845833720e+02, 6.724404202980e+02, 6.793332660800e+02, 6.858180401420e+02, 6.919407358480e+02, 6.977402019700e+02, 7.032495100170e+02, 7.084970157640e+02, 7.135071924150e+02, 7.183012912300e+02, 7.228978702930e+02, 7.273132214050e+02, 7.315617174530e+02, 7.356560971110e+02, 7.396076997500e+02, 7.434266604380e+02, 7.471220727640e+02, 7.507021255150e+02, 7.541742179910e+02, 7.575450577710e+02, 7.608207439840e+02, 7.640068385290e+02, 7.671084272710e+02, 7.701301728140e+02, 7.730763601940e+02, 7.759509365980e+02, 7.787575459930e+02, 7.814995594500e+02, 7.841801017600e+02, 7.868020748820e+02, 7.893681786660e+02, 7.918809292160e+02, 7.943426752080e+02, 7.967556124310e+02, 7.991217967860e+02, 8.014431559190e+02, 8.037214996750e+02, 8.059585295000e+02, 8.081558469250e+02, 8.103149612310e+02, 8.124372963960e+02, 8.145241973960e+02, 8.165769359340e+02, 8.185967156620e+02, 8.205846769410e+02, 8.225419011960e+02, 8.244694149030e+02, 8.263681932400e+02, 8.282391634460e+02, 8.300832079080e+02, 8.319011669990e+02, 8.336938417040e+02, 8.354619960390e+02, 8.372063592880e+02, 8.389276280750e+02, 8.406264682850e+02, 8.423035168430e+02, 8.439593833720e+02, 8.455946517270e+02, 8.472098814310e+02, 8.488056090070e+02, 8.503823492240e+02, 8.519405962570e+02, 8.534808247750e+02, 8.550034909530e+02, 8.565090334250e+02, 8.579978741740e+02, 8.594704193640e+02, 8.609270601270e+02, 8.623681732970e+02, 8.637941220990e+02, 8.652052568040e+02, 8.666019153370e+02, 8.679844238550e+02, 8.693530972890e+02, 8.707082398580e+02, 8.720501455520e+02, 8.733790985880e+02, 8.746953738410e+02, 8.759992372540e+02, 8.772909462220e+02, 8.785707499590e+02, 8.798388898430e+02, 8.810955997430e+02, 8.823411063340e+02, 8.835756293880e+02, 8.847993820560e+02, 8.860125711350e+02, 8.872153973190e+02, 8.884080554390e+02, 8.895907346950e+02, 8.907636188720e+02, 8.919268865450e+02, 8.930807112780e+02, 8.942252618150e+02, 8.953607022550e+02, 8.964871922240e+02, 8.976048870420e+02, 8.987139378730e+02, 8.998144918790e+02, 9.009066923590e+02, 9.019906788870e+02, 9.030665874390e+02, 9.041345505220e+02, 9.051946972900e+02, 9.062471536570e+02, 9.072920424090e+02, 9.083294833070e+02, 9.093595931850e+02, 9.103824860510e+02, 9.113982731760e+02, 9.124070631830e+02, 9.134089621290e+02, 9.144040735930e+02, 9.153924987480e+02, 9.163743364390e+02, 9.173496832540e+02, 9.183186335940e+02, 9.192812797430e+02, 9.202377119240e+02, 9.211880183680e+02, 9.221322853730e+02, 9.230705973560e+02, 9.240030369130e+02, 9.249296848720e+02, 9.258506203410e+02, 9.267659207590e+02, 9.276756619450e+02, 9.285799181400e+02, 9.294787620560e+02, 9.303722649160e+02, 9.312604964940e+02, 9.321435251560e+02, 9.330214179020e+02, 9.338942403950e+02, 9.347620570040e+02, 9.356249308360e+02, 9.364829237690e+02, 9.373360964850e+02, 9.381845085000e+02, 9.390282181970e+02, 9.398672828520e+02, 9.407017586640e+02, 9.415317007840e+02, 9.423571633380e+02, 9.431781994540e+02, 9.439948612880e+02, 9.448072000490e+02, 9.456152660170e+02, 9.464191085730e+02, 9.472187762160e+02, 9.480143165830e+02, 9.488057764750e+02, 9.495932018750e+02, 9.503766379630e+02, 9.511561291400e+02, 9.519317190450e+02, 9.527034505720e+02, 9.534713658870e+02, 9.542355064450e+02, 9.549959130070e+02, 9.557526256550e+02, 9.565056838070e+02, 9.572551262320e+02, 9.580009910660e+02, 9.587433158230e+02, 9.594821374120e+02, 9.602174921480e+02, 9.609494157640e+02, 9.616779434290e+02, 9.624031097530e+02, 9.631249488030e+02, 9.638434941140e+02, 9.645587787010e+02, 9.652708350670e+02, 9.659796952180e+02, 9.666853906680e+02, 9.673879524550e+02, 9.680874111470e+02, 9.687837968520e+02, 9.694771392290e+02, 9.701674674950e+02, 9.708548104340e+02, 9.715391964100e+02, 9.722206533680e+02, 9.728992088490e+02, 9.735748899940e+02, 9.742477235520e+02, 9.749177358920e+02, 9.755849530020e+02, 9.762494005060e+02, 9.769111036640e+02, 9.775700873820e+02, 9.782263762170e+02, 9.788799943870e+02, 9.795309657710e+02, 9.801793139240e+02, 9.808250620750e+02, 9.814682331370e+02, 9.821088497140e+02, 9.827469341030e+02, 9.833825083020e+02, 9.840155940150e+02, 9.846462126570e+02, 9.852743853600e+02, 9.859001329780e+02, 9.865234760890e+02, 9.871444350060e+02, 9.877630297740e+02, 9.883792801820e+02, 9.889932057630e+02, 9.896048258000e+02, 9.902141593290e+02, 9.908212251450e+02, 9.914260418050e+02, 9.920286276340e+02, 9.926290007250e+02, 9.932271789480e+02, 9.938231799480e+02, 9.944170211550e+02, 9.950087197820e+02, 9.955982928330e+02, 9.961857571040e+02, 9.967711291850e+02, 9.973544254670e+02, 9.979356621460e+02, 9.985148552190e+02, 9.990920204950e+02, 9.996671735940e+02, 1.000240329950e+03, 1.000811504820e+03, 1.001380713280e+03, 1.001947970220e+03, 1.002513290380e+03, 1.003076688290e+03, 1.003638178360e+03, 1.004197774790e+03, 1.004755491650e+03, 1.005311342840e+03, 1.005865342090e+03, 1.006417502990e+03, 1.006967838970e+03, 1.007516363310e+03, 1.008063089130e+03, 1.008608029440e+03, 1.009151197060e+03, 1.009692604700e+03, 1.010232264910e+03, 1.010770190130e+03, 1.011306392620e+03, 1.011840884560e+03, 1.012373677960e+03, 1.012904784710e+03, 1.013434216570e+03, 1.013961985180e+03, 1.014488102060e+03, 1.015012578600e+03, 1.015535426060e+03, 1.016056655600e+03, 1.016576278250e+03, 1.017094304930e+03, 1.017610746450e+03, 1.018125613500e+03, 1.018638916660e+03, 1.019150666400e+03, 1.019660873080e+03, 1.020169546980e+03, 1.020676698230e+03, 1.021182336890e+03, 1.021686472910e+03, 1.022189116140e+03, 1.022690276330e+03, 1.023189963120e+03, 1.023688186080e+03, 1.024184954660e+03, 1.024680278230e+03, 1.025174166070e+03, 1.025666627350e+03, 1.026157671170e+03, 1.026647306530e+03, 1.027135542340e+03, 1.027622387440e+03, 1.028107850570e+03, 1.028591940380e+03, 1.029074665460e+03, 1.029556034290e+03, 1.030036055290e+03, 1.030514736790e+03, 1.030992087050e+03, 1.031468114230e+03, 1.031942826440e+03, 1.032416231700e+03, 1.032888337960e+03, 1.033359153080e+03, 1.033828684870e+03, 1.034296941060e+03, 1.034763929300e+03, 1.035229657170e+03, 1.035694132200e+03, 1.036157361830e+03, 1.036619353430e+03, 1.037080114320e+03, 1.037539651760e+03, 1.037997972900e+03, 1.038455084880e+03, 1.038910994740e+03, 1.039365709470e+03, 1.039819236010e+03, 1.040271581200e+03, 1.040722751860e+03, 1.041172754730e+03, 1.041621596490e+03, 1.042069283780e+03, 1.042515823150e+03, 1.042961221120e+03, 1.043405484140e+03, 1.043848618610e+03, 1.044290630880e+03, 1.044731527220e+03, 1.045171313870e+03, 1.045609997010e+03, 1.046047582780e+03, 1.046484077240e+03, 1.046919486420e+03, 1.047353816290e+03, 1.047787072770e+03, 1.048219261740e+03, 1.048650389020e+03, 1.049080460380e+03, 1.049509481540e+03, 1.049937458200e+03, 1.050364395970e+03, 1.050790300450e+03, 1.051215177170e+03, 1.051639031620e+03, 1.052061869260e+03, 1.052483695480e+03, 1.052904515650e+03, 1.053324335080e+03, 1.053743159050e+03, 1.054160992780e+03, 1.054577841470e+03, 1.054993710250e+03, 1.055408604240e+03, 1.055822528500e+03, 1.056235488060e+03, 1.056647487890e+03, 1.057058532940e+03, 1.057468628120e+03, 1.057877778290e+03, 1.058285988280e+03, 1.058693262890e+03, 1.059099606870e+03, 1.059505024930e+03, 1.059909521750e+03, 1.060313101970e+03, 1.060715770210e+03, 1.061117531040e+03, 1.061518388980e+03, 1.061918348550e+03, 1.062317414220e+03, 1.062715590400e+03, 1.063112881510e+03, 1.063509291910e+03, 1.063904825930e+03, 1.064299487880e+03, 1.064693282020e+03, 1.065086212590e+03, 1.065478283790e+03, 1.065869499790e+03, 1.066259864750e+03, 1.066649382760e+03, 1.067038057910e+03, 1.067425894260e+03, 1.067812895810e+03, 1.068199066560e+03, 1.068584410480e+03, 1.068968931500e+03, 1.069352633510e+03, 1.069735520400e+03, 1.070117596020e+03, 1.070498864180e+03, 1.070879328680e+03, 1.071258993270e+03, 1.071637861710e+03, 1.072015937690e+03, 1.072393224910e+03, 1.072769727020e+03, 1.073145447650e+03, 1.073520390410e+03, 1.073894558890e+03, 1.074267956630e+03, 1.074640587170e+03, 1.075012454020e+03, 1.075383560640e+03, 1.075753910510e+03, 1.076123507050e+03, 1.076492353680e+03, 1.076860453770e+03, 1.077227810690e+03, 1.077594427780e+03, 1.077960308360e+03, 1.078325455700e+03, 1.078689873100e+03, 1.079053563790e+03, 1.079416530990e+03, 1.079778777920e+03, 1.080140307760e+03, 1.080501123660e+03, 1.080861228760e+03, 1.081220626180e+03, 1.081579319020e+03, 1.081937310360e+03, 1.082294603250e+03, 1.082651200720e+03, 1.083007105780e+03, 1.083362321450e+03, 1.083716850680e+03, 1.084070696430e+03, 1.084423861640e+03, 1.084776349230e+03, 1.085128162090e+03, 1.085479303100e+03, 1.085829775120e+03, 1.086179580990e+03, 1.086528723540e+03, 1.086877205560e+03, 1.087225029830e+03, 1.087572199140e+03, 1.087918716230e+03, 1.088264583830e+03, 1.088609804650e+03, 1.088954381390e+03},
    {1.657423842210e+00, 5.049538218080e+00, 8.497436564710e+00, 1.200336804940e+01, 1.556972778420e+01, 1.919906958450e+01, 2.289412014720e+01, 2.665779484220e+01, 3.049321534440e+01, 3.440372936950e+01, 3.839293282310e+01, 4.246469472630e+01, 4.662318534800e+01, 5.087290805140e+01, 5.521873546070e+01, 5.966595067060e+01, 6.422029436700e+01, 6.888801890560e+01, 7.367595061610e+01, 7.859156187870e+01, 8.364305486330e+01, 8.883945926050e+01, 9.419074688920e+01, 9.970796677210e+01, 1.054034051790e+02, 1.112907763180e+02, 1.173854508700e+02, 1.237047315840e+02, 1.302681877510e+02, 1.370980638960e+02, 1.442197826440e+02, 1.516625679350e+02, 1.594602230770e+02, 1.676521092760e+02, 1.762843850960e+02, 1.854115867680e+02, 1.950986538020e+02, 2.054235330470e+02, 2.164805210920e+02, 2.283845107190e+02, 2.412762386780e+02, 2.553283541360e+02, 2.707513235320e+02, 2.877962251370e+02, 3.067475231430e+02, 3.278932995580e+02, 3.514575692880e+02, 3.774808070580e+02, 4.056238740870e+02, 4.349223846950e+02, 4.638379195100e+02, 4.909780991600e+02, 5.156130914880e+02, 5.374902815280e+02, 5.567395496530e+02, 5.736791502870e+02, 5.886702772300e+02, 6.020433553970e+02, 6.140745753840e+02, 6.249864872590e+02, 6.349565278540e+02, 6.441264140370e+02, 6.526101456650e+02, 6.605002755030e+02, 6.678726811620e+02, 6.747901702840e+02, 6.813052093770e+02, 6.874620021340e+02, 6.932980856820e+02, 6.988455682630e+02, 7.041320984480e+02, 7.091816316280e+02, 7.140150419450e+02, 7.186506151630e+02, 7.231044489570e+02, 7.273907804340e+02, 7.315222560140e+02, 7.355101552100e+02, 7.393645772850e+02, 7.430945977980e+02, 7.467084005580e+02, 7.502133893890e+02, 7.536162832280e+02, 7.569231973850e+02, 7.601397132610e+02, 7.632709384200e+02, 7.663215585140e+02, 7.692958823630e+02, 7.721978812040e+02, 7.750312229940e+02, 7.777993024810e+02, 7.805052676550e+02, 7.831520430740e+02, 7.857423505070e+02, 7.882787272510e+02, 7.907635424160e+02, 7.931990114640e+02, 7.955872092000e+02, 7.979300814200e+02, 8.002294553810e+02, 8.024870492250e+02, 8.047044804800e+02, 8.068832737600e+02, 8.090248677290e+02, 8.111306214340e+02, 8.132018200580e+02, 8.152396801690e+02, 8.172453545040e+02, 8.192199363550e+02, 8.211644635670e+02, 8.230799222250e+02, 8.249672500290e+02, 8.268273393990e+02, 8.286610403410e+02, 8.304691630840e+02, 8.322524805220e+02, 8.340117304680e+02, 8.357476177370e+02, 8.374608160910e+02, 8.391519700270e+02, 8.408216964460e+02, 8.424705862120e+02, 8.440992055870e+02, 8.457080975840e+02, 8.472977832200e+02, 8.488687626890e+02, 8.504215164590e+02, 8.519565062930e+02, 8.534741762140e+02, 8.549749533990e+02, 8.564592490260e+02, 8.579274590610e+02, 8.593799650060e+02, 8.608171345940e+02, 8.622393224450e+02, 8.636468706900e+02, 8.650401095460e+02, 8.664193578710e+02, 8.677849236790e+02, 8.691371046290e+02, 8.704761884870e+02, 8.718024535640e+02, 8.731161691260e+02, 8.744175957850e+02, 8.757069858710e+02, 8.769845837770e+02, 8.782506262960e+02, 8.795053429340e+02, 8.807489562080e+02, 8.819816819290e+02, 8.832037294740e+02, 8.844153020380e+02, 8.856165968800e+02, 8.868078055550e+02, 8.879891141320e+02, 8.891607034050e+02, 8.903227490950e+02, 8.914754220340e+02, 8.926188883540e+02, 8.937533096570e+02, 8.948788431800e+02, 8.959956419520e+02, 8.971038549470e+02, 8.982036272280e+02, 8.992951000820e+02, 9.003784111540e+02, 9.014536945720e+02, 9.025210810690e+02, 9.035806980970e+02, 9.046326699360e+02, 9.056771178040e+02, 9.067141599560e+02, 9.077439117790e+02, 9.087664858910e+02, 9.097819922250e+02, 9.107905381190e+02, 9.117922283950e+02, 9.127871654430e+02, 9.137754492910e+02, 9.147571776830e+02, 9.157324461460e+02, 9.167013480620e+02, 9.176639747290e+02, 9.186204154220e+02, 9.195707574620e+02, 9.205150862620e+02, 9.214534853930e+02, 9.223860366330e+02, 9.233128200170e+02, 9.242339138930e+02, 9.251493949650e+02, 9.260593383400e+02, 9.269638175760e+02, 9.278629047230e+02, 9.287566703630e+02, 9.296451836530e+02, 9.305285123640e+02, 9.314067229170e+02, 9.322798804190e+02, 9.331480486990e+02, 9.340112903440e+02, 9.348696667270e+02, 9.357232380420e+02, 9.365720633340e+02, 9.374162005290e+02, 9.382557064610e+02, 9.390906369020e+02, 9.399210465870e+02, 9.407469892420e+02, 9.415685176070e+02, 9.423856834620e+02, 9.431985376490e+02, 9.440071300980e+02, 9.448115098450e+02, 9.456117250570e+02, 9.464078230520e+02, 9.471998503180e+02, 9.479878525340e+02, 9.487718745880e+02, 9.495519605970e+02, 9.503281539230e+02, 9.511004971920e+02, 9.518690323110e+02, 9.526338004820e+02, 9.533948422220e+02, 9.541521973750e+02, 9.549059051280e+02, 9.556560040270e+02, 9.564025319890e+02, 9.571455263190e+02, 9.578850237190e+02, 9.586210603040e+02, 9.593536716170e+02, 9.600828926350e+02, 9.608087577860e+02, 9.615313009610e+02, 9.622505555220e+02, 9.629665543140e+02, 9.636793296790e+02, 9.643889134630e+02, 9.650953370270e+02, 9.657986312590e+02, 9.664988265800e+02, 9.671959529570e+02, 9.678900399100e+02, 9.685811165230e+02, 9.692692114490e+02, 9.699543529250e+02, 9.706365687720e+02, 9.713158864110e+02, 9.719923328660e+02, 9.726659347720e+02, 9.733367183860e+02, 9.740047095910e+02, 9.746699339040e+02, 9.753324164820e+02, 9.759921821330e+02, 9.766492553160e+02, 9.773036601540e+02, 9.779554204370e+02, 9.786045596270e+02, 9.792511008680e+02, 9.798950669880e+02, 9.805364805070e+02, 9.811753636420e+02, 9.818117383120e+02, 9.824456261460e+02, 9.830770484850e+02, 9.837060263860e+02, 9.843325806330e+02, 9.849567317370e+02, 9.855784999420e+02, 9.861979052280e+02, 9.868149673210e+02, 9.874297056910e+02, 9.880421395600e+02, 9.886522879070e+02, 9.892601694680e+02, 9.898658027440e+02, 9.904692060060e+02, 9.910703972940e+02, 9.916693944250e+02, 9.922662149950e+02, 9.928608763820e+02, 9.934533957540e+02, 9.940437900660e+02, 9.946320760690e+02, 9.952182703090e+02, 9.958023891340e+02, 9.963844486960e+02, 9.969644649510e+02, 9.975424536700e+02, 9.981184304310e+02, 9.986924106340e+02, 9.992644094940e+02, 9.998344420490e+02, 1.000402523160e+03, 1.000968667520e+03, 1.001532889650e+03, 1.002095203900e+03, 1.002655624460e+03, 1.003214165350e+03, 1.003770840450e+03, 1.004325663460e+03, 1.004878647940e+03, 1.005429807290e+03, 1.005979154760e+03, 1.006526703460e+03, 1.007072466350e+03, 1.007616456240e+03, 1.008158685800e+03, 1.008699167580e+03, 1.009237913970e+03, 1.009774937240e+03, 1.010310249510e+03, 1.010843862790e+03, 1.011375788940e+03, 1.011906039710e+03, 1.012434626720e+03, 1.012961561450e+03, 1.013486855280e+03, 1.014010519470e+03, 1.014532565140e+03, 1.015053003310e+03, 1.015571844890e+03, 1.016089100650e+03, 1.016604781290e+03, 1.017118897360e+03, 1.017631459310e+03, 1.018142477510e+03, 1.018651962190e+03, 1.019159923490e+03, 1.019666371450e+03, 1.020171316000e+03, 1.020674766990e+03, 1.021176734140e+03, 1.021677227100e+03, 1.022176255400e+03, 1.022673828500e+03, 1.023169955750e+03, 1.023664646410e+03, 1.024157909660e+03, 1.024649754570e+03, 1.025140190140e+03, 1.025629225270e+03, 1.026116868780e+03, 1.026603129410e+03, 1.027088015810e+03, 1.027571536540e+03, 1.028053700080e+03, 1.028534514850e+03, 1.029013989160e+03, 1.029492131260e+03, 1.029968949320e+03, 1.030444451420e+03, 1.030918645590e+03, 1.031391539760e+03, 1.031863141800e+03, 1.032333459490e+03, 1.032802500570e+03, 1.033270272680e+03, 1.033736783410e+03, 1.034202040250e+03, 1.034666050650e+03, 1.035128821990e+03, 1.035590361570e+03, 1.036050676630e+03, 1.036509774350e+03, 1.036967661830e+03, 1.037424346130e+03, 1.037879834230e+03, 1.038334133050e+03, 1.038787249460e+03, 1.039239190240e+03, 1.039689962140e+03, 1.040139571850e+03, 1.040588025980e+03, 1.041035331090e+03, 1.041481493700e+03, 1.041926520260e+03, 1.042370417150e+03, 1.042813190720e+03, 1.043254847260e+03, 1.043695392980e+03, 1.044134834080e+03, 1.044573176680e+03, 1.045010426850e+03, 1.045446590600e+03, 1.045881673920e+03, 1.046315682720e+03, 1.046748622860e+03, 1.047180500180e+03, 1.047611320450e+03, 1.048041089380e+03, 1.048469812660e+03, 1.048897495920e+03, 1.049324144730e+03, 1.049749764640e+03, 1.050174361140e+03, 1.050597939680e+03, 1.051020505650e+03, 1.051442064420e+03, 1.051862621300e+03, 1.052282181560e+03, 1.052700750440e+03, 1.053118333120e+03, 1.053534934740e+03, 1.053950560410e+03, 1.054365215200e+03, 1.054778904130e+03, 1.055191632180e+03, 1.055603404300e+03, 1.056014225380e+03, 1.056424100310e+03, 1.056833033900e+03, 1.057241030950e+03, 1.057648096210e+03, 1.058054234400e+03, 1.058459450200e+03, 1.058863748240e+03, 1.059267133150e+03, 1.059669609480e+03, 1.060071181780e+03, 1.060471854560e+03, 1.060871632260e+03, 1.061270519340e+03, 1.061668520200e+03, 1.062065639190e+03, 1.062461880650e+03, 1.062857248890e+03, 1.063251748160e+03, 1.063645382720e+03, 1.064038156770e+03, 1.064430074470e+03, 1.064821139970e+03, 1.065211357390e+03, 1.065600730800e+03, 1.065989264250e+03, 1.066376961780e+03, 1.066763827370e+03, 1.067149864980e+03, 1.067535078540e+03, 1.067919471970e+03, 1.068303049140e+03, 1.068685813900e+03, 1.069067770060e+03, 1.069448921430e+03, 1.069829271770e+03, 1.070208824810e+03, 1.070587584280e+03, 1.070965553850e+03, 1.071342737190e+03, 1.071719137930e+03, 1.072094759670e+03, 1.072469606000e+03, 1.072843680480e+03, 1.073216986640e+03, 1.073589527990e+03, 1.073961308000e+03, 1.074332330140e+03, 1.074702597840e+03, 1.075072114510e+03, 1.075440883540e+03, 1.075808908290e+03, 1.076176192110e+03, 1.076542738300e+03, 1.076908550170e+03, 1.077273630980e+03, 1.077637983990e+03, 1.078001612410e+03, 1.078364519470e+03, 1.078726708330e+03, 1.079088182170e+03, 1.079448944130e+03, 1.079808997310e+03, 1.080168344830e+03, 1.080526989760e+03, 1.080884935150e+03, 1.081242184050e+03, 1.081598739460e+03, 1.081954604400e+03, 1.082309781830e+03, 1.082664274710e+03, 1.083018085970e+03, 1.083371218540e+03, 1.083723675320e+03, 1.084075459180e+03, 1.084426572990e+03, 1.084777019590e+03, 1.085126801800e+03, 1.085475922430e+03, 1.085824384260e+03, 1.086172190080e+03, 1.086519342610e+03, 1.086865844610e+03},
    {1.652096810120e+00, 5.032872161380e+00, 8.468624388080e+00, 1.196153847010e+01, 1.551393895610e+01, 1.912830194110e+01, 2.280726835030e+01, 2.655365870180e+01, 3.037048955260e+01, 3.426099186300e+01, 3.822863155370e+01, 4.227713257930e+01, 4.641050289520e+01, 5.063306376670e+01, 5.494948294680e+01, 5.936481235400e+01, 6.388453100060e+01, 6.851459407110e+01, 7.326148923820e+01, 7.813230152660e+01, 8.313478832420e+01, 8.827746649140e+01, 9.356971396600e+01, 9.902188882570e+01, 1.046454694850e+02, 1.104532206180e+02, 1.164593905760e+02, 1.226799475660e+02, 1.291328638560e+02, 1.358384597830e+02, 1.428198227020e+02, 1.501033203920e+02, 1.577192340820e+02, 1.657025437400e+02, 1.740939077360e+02, 1.829408909170e+02, 1.922995093190e+02, 2.022361745920e+02, 2.128301319730e+02, 2.241764788450e+02, 2.363897936670e+02, 2.496082203120e+02, 2.639973729030e+02, 2.797523298730e+02, 2.970938268600e+02, 3.162515032910e+02, 3.374246124980e+02, 3.607123980490e+02, 3.860076325710e+02, 4.128464894230e+02, 4.402907029380e+02, 4.670949716870e+02, 4.922494904890e+02, 5.152202850700e+02, 5.358253223650e+02, 5.541611377360e+02, 5.704675715010e+02, 5.850266622370e+02, 5.981068802100e+02, 6.099409897790e+02, 6.207222976430e+02, 6.306085975240e+02, 6.397282188760e+02, 6.481858579000e+02, 6.560674777640e+02, 6.634442105030e+02, 6.703753984570e+02, 6.769109494180e+02, 6.830931616350e+02, 6.889581445810e+02, 6.945369325960e+02, 6.998563648060e+02, 7.049397863340e+02, 7.098076118630e+02, 7.144777823510e+02, 7.189661380310e+02, 7.232867252740e+02, 7.274520507160e+02, 7.314732930230e+02, 7.353604803450e+02, 7.391226398060e+02, 7.427679240460e+02, 7.463037188240e+02, 7.497367349150e+02, 7.530730869080e+02, 7.563183610400e+02, 7.594776737990e+02, 7.625557227420e+02, 7.655568307040e+02, 7.684849843940e+02, 7.713438681940e+02, 7.741368938390e+02, 7.768672265790e+02, 7.795378082840e+02, 7.821513779220e+02, 7.847104897470e+02, 7.872175295050e+02, 7.896747288980e+02, 7.920841785380e+02, 7.944478395630e+02, 7.967675540910e+02, 7.990450546300e+02, 8.012819725870e+02, 8.034798459630e+02, 8.056401263270e+02, 8.077641851580e+02, 8.098533196140e+02, 8.119087577860e+02, 8.139316635060e+02, 8.159231407380e+02, 8.178842376000e+02, 8.198159500580e+02, 8.217192253230e+02, 8.235949649640e+02, 8.254440277940e+02, 8.272672325180e+02, 8.290653601900e+02, 8.308391564760e+02, 8.325893337600e+02, 8.343165730860e+02, 8.360215259710e+02, 8.377048160810e+02, 8.393670407950e+02, 8.410087726590e+02, 8.426305607450e+02, 8.442329319110e+02, 8.458163919880e+02, 8.473814268800e+02, 8.489285036010e+02, 8.504580712370e+02, 8.519705618590e+02, 8.534663913680e+02, 8.549459602950e+02, 8.564096545510e+02, 8.578578461320e+02, 8.592908937780e+02, 8.607091436030e+02, 8.621129296780e+02, 8.635025745890e+02, 8.648783899570e+02, 8.662406769330e+02, 8.675897266630e+02, 8.689258207290e+02, 8.702492315680e+02, 8.715602228630e+02, 8.728590499180e+02, 8.741459600110e+02, 8.754211927300e+02, 8.766849802930e+02, 8.779375478440e+02, 8.791791137450e+02, 8.804098898460e+02, 8.816300817440e+02, 8.828398890300e+02, 8.840395055200e+02, 8.852291194800e+02, 8.864089138400e+02, 8.875790663920e+02, 8.887397499840e+02, 8.898911327060e+02, 8.910333780630e+02, 8.921666451400e+02, 8.932910887670e+02, 8.944068596690e+02, 8.955141046080e+02, 8.966129665310e+02, 8.977035846930e+02, 8.987860947910e+02, 8.998606290860e+02, 9.009273165170e+02, 9.019862828120e+02, 9.030376506000e+02, 9.040815395090e+02, 9.051180662680e+02, 9.061473447990e+02, 9.071694863070e+02, 9.081845993690e+02, 9.091927900190e+02, 9.101941618240e+02, 9.111888159630e+02, 9.121768513020e+02, 9.131583644640e+02, 9.141334498980e+02, 9.151021999450e+02, 9.160647048990e+02, 9.170210530730e+02, 9.179713308530e+02, 9.189156227560e+02, 9.198540114850e+02, 9.207865779820e+02, 9.217134014760e+02, 9.226345595350e+02, 9.235501281110e+02, 9.244601815850e+02, 9.253647928150e+02, 9.262640331700e+02, 9.271579725800e+02, 9.280466795680e+02, 9.289302212930e+02, 9.298086635840e+02, 9.306820709770e+02, 9.315505067470e+02, 9.324140329440e+02, 9.332727104250e+02, 9.341265988830e+02, 9.349757568760e+02, 9.358202418610e+02, 9.366601102170e+02, 9.374954172750e+02, 9.383262173440e+02, 9.391525637370e+02, 9.399745087910e+02, 9.407921039000e+02, 9.416053995290e+02, 9.424144452430e+02, 9.432192897250e+02, 9.440199807980e+02, 9.448165654490e+02, 9.456090898410e+02, 9.463975993420e+02, 9.471821385360e+02, 9.479627512440e+02, 9.487394805440e+02, 9.495123687830e+02, 9.502814576000e+02, 9.510467879340e+02, 9.518084000480e+02, 9.525663335390e+02, 9.533206273550e+02, 9.540713198100e+02, 9.548184485940e+02, 9.555620507910e+02, 9.563021628910e+02, 9.570388208020e+02, 9.577720598630e+02, 9.585019148570e+02, 9.592284200220e+02, 9.599516090610e+02, 9.606715151570e+02, 9.613881709810e+02, 9.621016087030e+02, 9.628118600040e+02, 9.635189560840e+02, 9.642229276720e+02, 9.649238050390e+02, 9.656216180020e+02, 9.663163959380e+02, 9.670081677870e+02, 9.676969620700e+02, 9.683828068850e+02, 9.690657299280e+02, 9.697457584920e+02, 9.704229194770e+02, 9.710972394000e+02, 9.717687444000e+02, 9.724374602470e+02, 9.731034123470e+02, 9.737666257510e+02, 9.744271251590e+02, 9.750849349300e+02, 9.757400790880e+02, 9.763925813230e+02, 9.770424650050e+02, 9.776897531850e+02, 9.783344686020e+02, 9.789766336870e+02, 9.796162705730e+02, 9.802534010960e+02, 9.808880468030e+02, 9.815202289550e+02, 9.821499685360e+02, 9.827772862510e+02, 9.834022025390e+02, 9.840247375730e+02, 9.846449112650e+02, 9.852627432700e+02, 9.858782529940e+02, 9.864914595940e+02, 9.871023819850e+02, 9.877110388440e+02, 9.883174486110e+02, 9.889216294980e+02, 9.895235994890e+02, 9.901233763450e+02, 9.907209776070e+02, 9.913164206030e+02, 9.919097224470e+02, 9.925009000440e+02, 9.930899700960e+02, 9.936769491020e+02, 9.942618533630e+02, 9.948446989850e+02, 9.954255018810e+02, 9.960042777760e+02, 9.965810422090e+02, 9.971558105350e+02, 9.977285979320e+02, 9.982994193960e+02, 9.988682897540e+02, 9.994352236570e+02, 1.000000235590e+03, 1.000563339870e+03, 1.001124550650e+03, 1.001683881920e+03, 1.002241347520e+03, 1.002796961130e+03, 1.003350736260e+03, 1.003902686290e+03, 1.004452824440e+03, 1.005001163790e+03, 1.005547717270e+03, 1.006092497660e+03, 1.006635517620e+03, 1.007176789650e+03, 1.007716326130e+03, 1.008254139290e+03, 1.008790241240e+03, 1.009324643950e+03, 1.009857359270e+03, 1.010388398930e+03, 1.010917774510e+03, 1.011445497490e+03, 1.011971579220e+03, 1.012496030920e+03, 1.013018863720e+03, 1.013540088610e+03, 1.014059716470e+03, 1.014577758060e+03, 1.015094224060e+03, 1.015609125000e+03, 1.016122471320e+03, 1.016634273370e+03, 1.017144541350e+03, 1.017653285410e+03, 1.018160515550e+03, 1.018666241710e+03, 1.019170473690e+03, 1.019673221220e+03, 1.020174493920e+03, 1.020674301330e+03, 1.021172652880e+03, 1.021669557910e+03, 1.022165025670e+03, 1.022659065310e+03, 1.023151685910e+03, 1.023642896450e+03, 1.024132705830e+03, 1.024621122840e+03, 1.025108156210e+03, 1.025593814580e+03, 1.026078106510e+03, 1.026561040470e+03, 1.027042624860e+03, 1.027522867980e+03, 1.028001778080e+03, 1.028479363310e+03, 1.028955631760e+03, 1.029430591430e+03, 1.029904250250e+03, 1.030376616080e+03, 1.030847696710e+03, 1.031317499840e+03, 1.031786033130e+03, 1.032253304130e+03, 1.032719320360e+03, 1.033184089250e+03, 1.033647618170e+03, 1.034109914410e+03, 1.034570985210e+03, 1.035030837740e+03, 1.035489479100e+03, 1.035946916330e+03, 1.036403156420e+03, 1.036858206280e+03, 1.037312072770e+03, 1.037764762670e+03, 1.038216282730e+03, 1.038666639610e+03, 1.039115839950e+03, 1.039563890290e+03, 1.040010797140e+03, 1.040456566940e+03, 1.040901206090e+03, 1.041344720920e+03, 1.041787117720e+03, 1.042228402700e+03, 1.042668582040e+03, 1.043107661870e+03, 1.043545648260e+03, 1.043982547210e+03, 1.044418364710e+03, 1.044853106660e+03, 1.045286778930e+03, 1.045719387350e+03, 1.046150937670e+03, 1.046581435640e+03, 1.047010886900e+03, 1.047439297110e+03, 1.047866671830e+03, 1.048293016610e+03, 1.048718336920e+03, 1.049142638220e+03, 1.049565925920e+03, 1.049988205350e+03, 1.050409481840e+03, 1.050829760660e+03, 1.051249047040e+03, 1.051667346160e+03, 1.052084663170e+03, 1.052501003160e+03, 1.052916371210e+03, 1.053330772340e+03, 1.053744211520e+03, 1.054156693700e+03, 1.054568223790e+03, 1.054978806660e+03, 1.055388447120e+03, 1.055797149970e+03, 1.056204919960e+03, 1.056611761810e+03, 1.057017680190e+03, 1.057422679760e+03, 1.057826765110e+03, 1.058229940830e+03, 1.058632211430e+03, 1.059033581440e+03, 1.059434055310e+03, 1.059833637480e+03, 1.060232332360e+03, 1.060630144290e+03, 1.061027077630e+03, 1.061423136670e+03, 1.061818325680e+03, 1.062212648900e+03, 1.062606110530e+03, 1.062998714750e+03, 1.063390465700e+03, 1.063781367500e+03, 1.064171424220e+03, 1.064560639910e+03, 1.064949018610e+03, 1.065336564290e+03, 1.065723280930e+03, 1.066109172460e+03, 1.066494242790e+03, 1.066878495790e+03, 1.067261935320e+03, 1.067644565180e+03, 1.068026389190e+03, 1.068407411100e+03, 1.068787634660e+03, 1.069167063580e+03, 1.069545701540e+03, 1.069923552210e+03, 1.070300619220e+03, 1.070676906190e+03, 1.071052416680e+03, 1.071427154260e+03, 1.071801122470e+03, 1.072174324810e+03, 1.072546764770e+03, 1.072918445800e+03, 1.073289371350e+03, 1.073659544810e+03, 1.074028969590e+03, 1.074397649040e+03, 1.074765586510e+03, 1.075132785320e+03, 1.075499248760e+03, 1.075864980100e+03, 1.076229982590e+03, 1.076594259470e+03, 1.076957813940e+03, 1.077320649180e+03, 1.077682768370e+03, 1.078044174630e+03, 1.078404871100e+03, 1.078764860860e+03, 1.079124147010e+03, 1.079482732600e+03, 1.079840620670e+03, 1.080197814230e+03, 1.080554316290e+03, 1.080910129820e+03, 1.081265257790e+03, 1.081619703120e+03, 1.081973468740e+03, 1.082326557560e+03, 1.082678972440e+03, 1.083030716260e+03, 1.083381791870e+03, 1.083732202070e+03, 1.084081949690e+03, 1.084431037510e+03, 1.084779468310e+03},
    {1.646804373560e+00, 5.016320214850e+00, 8.440020062050e+00, 1.192002666000e+01, 1.545859623320e+01, 1.905812973130e+01, 2.272118527370e+01, 2.645049195220e+01, 3.024896517120e+01, 3.411972373770e+01, 3.806610894360e+01, 4.209170592790e+01, 4.620036765060e+01, 5.039624187280e+01, 5.468380160280e+01, 5.906787955810e+01, 6.355370729150e+01, 6.814695975580e+01, 7.285380623790e+01, 7.768096877510e+01, 8.263578940470e+01, 8.772630788100e+01, 9.296135185550e+01, 9.835064196220e+01, 1.039049148140e+02, 1.096360676290e+02, 1.155573291010e+02, 1.216834622880e+02, 1.280310067320e+02, 1.346185689300e+02, 1.414671726270e+02, 1.486006835200e+02, 1.560463267970e+02, 1.638353209140e+02, 1.720036570330e+02, 1.805930607790e+02, 1.896521809340e+02, 1.992380568290e+02, 2.094179188810e+02, 2.202713654910e+02, 2.318929138690e+02, 2.443947969670e+02, 2.579095816580e+02, 2.725915455740e+02, 2.886145339970e+02, 3.061621312710e+02, 3.254041836500e+02, 3.464541886200e+02, 3.693052680260e+02, 3.937430546950e+02, 4.192462292380e+02, 4.449623413660e+02, 4.699038832470e+02, 4.933304442280e+02, 5.148424060530e+02, 5.343068004520e+02, 5.517967864000e+02, 5.674957308130e+02, 5.816255188210e+02, 5.944045628000e+02, 6.060284930730e+02, 6.166644531120e+02, 6.264520791370e+02, 6.355070376170e+02, 6.439250701710e+02, 6.517857024090e+02, 6.591553637120e+02, 6.660899120140e+02, 6.726366423840e+02, 6.788358751220e+02, 6.847222112880e+02, 6.903255287750e+02, 6.956717769840e+02, 7.007836151540e+02, 7.056809289000e+02, 7.103812513160e+02, 7.149001087800e+02, 7.192513069030e+02, 7.234471684820e+02, 7.274987327200e+02, 7.314159229190e+02, 7.352076883600e+02, 7.388821248980e+02, 7.424465779250e+02, 7.459077306250e+02, 7.492716799280e+02, 7.525440021070e+02, 7.557298096290e+02, 7.588338005990e+02, 7.618603018900e+02, 7.648133068890e+02, 7.676965086250e+02, 7.705133289350e+02, 7.732669442060e+02, 7.759603081660e+02, 7.785961721020e+02, 7.811771028590e+02, 7.837054988840e+02, 7.861836045810e+02, 7.886135231690e+02, 7.909972282280e+02, 7.933365741020e+02, 7.956333052710e+02, 7.978890648320e+02, 8.001054021710e+02, 8.022837799300e+02, 8.044255803380e+02, 8.065321109750e+02, 8.086046100320e+02, 8.106442511160e+02, 8.126521476440e+02, 8.146293568770e+02, 8.165768836180e+02, 8.184956836110e+02, 8.203866666730e+02, 8.222506995790e+02, 8.240886087260e+02, 8.259011825890e+02, 8.276891740080e+02, 8.294533022850e+02, 8.311942551520e+02, 8.329126905790e+02, 8.346092384660e+02, 8.362845022120e+02, 8.379390601800e+02, 8.395734670570e+02, 8.411882551320e+02, 8.427839354800e+02, 8.443609990750e+02, 8.459199178290e+02, 8.474611455690e+02, 8.489851189450e+02, 8.504922582890e+02, 8.519829684180e+02, 8.534576393900e+02, 8.549166472120e+02, 8.563603545090e+02, 8.577891111550e+02, 8.592032548620e+02, 8.606031117390e+02, 8.619889968230e+02, 8.633612145730e+02, 8.647200593410e+02, 8.660658158180e+02, 8.673987594500e+02, 8.687191568420e+02, 8.700272661300e+02, 8.713233373390e+02, 8.726076127240e+02, 8.738803270870e+02, 8.751417080830e+02, 8.763919765100e+02, 8.776313465850e+02, 8.788600262020e+02, 8.800782171830e+02, 8.812861155120e+02, 8.824839115620e+02, 8.836717903100e+02, 8.848499315370e+02, 8.860185100270e+02, 8.871776957500e+02, 8.883276540410e+02, 8.894685457660e+02, 8.906005274860e+02, 8.917237516100e+02, 8.928383665420e+02, 8.939445168230e+02, 8.950423432640e+02, 8.961319830760e+02, 8.972135699930e+02, 8.982872343890e+02, 8.993531033940e+02, 9.004113009990e+02, 9.014619481600e+02, 9.025051629030e+02, 9.035410604100e+02, 9.045697531190e+02, 9.055913508100e+02, 9.066059606850e+02, 9.076136874540e+02, 9.086146334130e+02, 9.096088985140e+02, 9.105965804440e+02, 9.115777746870e+02, 9.125525745950e+02, 9.135210714540e+02, 9.144833545390e+02, 9.154395111790e+02, 9.163896268130e+02, 9.173337850430e+02, 9.182720676900e+02, 9.192045548440e+02, 9.201313249110e+02, 9.210524546640e+02, 9.219680192890e+02, 9.228780924250e+02, 9.237827462130e+02, 9.246820513300e+02, 9.255760770370e+02, 9.264648912090e+02, 9.273485603800e+02, 9.282271497730e+02, 9.291007233370e+02, 9.299693437830e+02, 9.308330726100e+02, 9.316919701450e+02, 9.325460955650e+02, 9.333955069320e+02, 9.342402612200e+02, 9.350804143410e+02, 9.359160211740e+02, 9.367471355880e+02, 9.375738104710e+02, 9.383960977500e+02, 9.392140484170e+02, 9.400277125510e+02, 9.408371393410e+02, 9.416423771060e+02, 9.424434733150e+02, 9.432404746120e+02, 9.440334268280e+02, 9.448223750060e+02, 9.456073634190e+02, 9.463884355830e+02, 9.471656342790e+02, 9.479390015690e+02, 9.487085788100e+02, 9.494744066750e+02, 9.502365251610e+02, 9.509949736110e+02, 9.517497907250e+02, 9.525010145730e+02, 9.532486826120e+02, 9.539928317000e+02, 9.547334981030e+02, 9.554707175150e+02, 9.562045250660e+02, 9.569349553330e+02, 9.576620423570e+02, 9.583858196500e+02, 9.591063202060e+02, 9.598235765130e+02, 9.605376205650e+02, 9.612484838690e+02, 9.619561974570e+02, 9.626607918970e+02, 9.633622972970e+02, 9.640607433220e+02, 9.647561591950e+02, 9.654485737140e+02, 9.661380152510e+02, 9.668245117700e+02, 9.675080908290e+02, 9.681887795870e+02, 9.688666048180e+02, 9.695415929130e+02, 9.702137698890e+02, 9.708831613960e+02, 9.715497927250e+02, 9.722136888120e+02, 9.728748742500e+02, 9.735333732880e+02, 9.741892098460e+02, 9.748424075140e+02, 9.754929895600e+02, 9.761409789390e+02, 9.767863982950e+02, 9.774292699680e+02, 9.780696160000e+02, 9.787074581400e+02, 9.793428178490e+02, 9.799757163060e+02, 9.806061744110e+02, 9.812342127930e+02, 9.818598518120e+02, 9.824831115650e+02, 9.831040118910e+02, 9.837225723740e+02, 9.843388123490e+02, 9.849527509040e+02, 9.855644068900e+02, 9.861737989150e+02, 9.867809453590e+02, 9.873858643690e+02, 9.879885738710e+02, 9.885890915660e+02, 9.891874349370e+02, 9.897836212570e+02, 9.903776675830e+02, 9.909695907690e+02, 9.915594074630e+02, 9.921471341130e+02, 9.927327869700e+02, 9.933163820920e+02, 9.938979353440e+02, 9.944774624050e+02, 9.950549787680e+02, 9.956304997460e+02, 9.962040404700e+02, 9.967756158980e+02, 9.973452408120e+02, 9.979129298250e+02, 9.984786973830e+02, 9.990425577620e+02, 9.996045250810e+02, 1.000164613290e+03, 1.000722836200e+03, 1.001279207440e+03, 1.001833740510e+03, 1.002386448740e+03, 1.002937345320e+03, 1.003486443300e+03, 1.004033755580e+03, 1.004579294910e+03, 1.005123073920e+03, 1.005665105080e+03, 1.006205400730e+03, 1.006743973100e+03, 1.007280834250e+03, 1.007815996130e+03, 1.008349470580e+03, 1.008881269270e+03, 1.009411403790e+03, 1.009939885570e+03, 1.010466725960e+03, 1.010991936140e+03, 1.011515527230e+03, 1.012037510180e+03, 1.012557895870e+03, 1.013076695040e+03, 1.013593918320e+03, 1.014109576250e+03, 1.014623679240e+03, 1.015136237620e+03, 1.015647261580e+03, 1.016156761230e+03, 1.016664746590e+03, 1.017171227540e+03, 1.017676213890e+03, 1.018179715350e+03, 1.018681741530e+03, 1.019182301940e+03, 1.019681406000e+03, 1.020179063040e+03, 1.020675282290e+03, 1.021170072890e+03, 1.021663443900e+03, 1.022155404290e+03, 1.022645962940e+03, 1.023135128640e+03, 1.023622910110e+03, 1.024109315970e+03, 1.024594354760e+03, 1.025078034950e+03, 1.025560364920e+03, 1.026041352970e+03, 1.026521007340e+03, 1.026999336150e+03, 1.027476347500e+03, 1.027952049370e+03, 1.028426449680e+03, 1.028899556290e+03, 1.029371376960e+03, 1.029841919410e+03, 1.030311191270e+03, 1.030779200090e+03, 1.031245953380e+03, 1.031711458560e+03, 1.032175722980e+03, 1.032638753950e+03, 1.033100558690e+03, 1.033561144350e+03, 1.034020518050e+03, 1.034478686800e+03, 1.034935657590e+03, 1.035391437320e+03, 1.035846032840e+03, 1.036299450940e+03, 1.036751698350e+03, 1.037202781740e+03, 1.037652707720e+03, 1.038101482850e+03, 1.038549113610e+03, 1.038995606460e+03, 1.039440967770e+03, 1.039885203890e+03, 1.040328321070e+03, 1.040770325560e+03, 1.041211223510e+03, 1.041651021040e+03, 1.042089724220e+03, 1.042527339070e+03, 1.042963871540e+03, 1.043399327550e+03, 1.043833712960e+03, 1.044267033590e+03, 1.044699295200e+03, 1.045130503510e+03, 1.045560664200e+03, 1.045989782880e+03, 1.046417865140e+03, 1.046844916510e+03, 1.047270942470e+03, 1.047695948470e+03, 1.048119939910e+03, 1.048542922130e+03, 1.048964900460e+03, 1.049385880150e+03, 1.049805866440e+03, 1.050224864510e+03, 1.050642879490e+03, 1.051059916500e+03, 1.051475980600e+03, 1.051891076790e+03, 1.052305210070e+03, 1.052718385370e+03, 1.053130607610e+03, 1.053541881640e+03, 1.053952212280e+03, 1.054361604340e+03, 1.054770062550e+03, 1.055177591640e+03, 1.055584196280e+03, 1.055989881110e+03, 1.056394650740e+03, 1.056798509740e+03, 1.057201462660e+03, 1.057603513970e+03, 1.058004668170e+03, 1.058404929670e+03, 1.058804302880e+03, 1.059202792160e+03, 1.059600401850e+03, 1.059997136250e+03, 1.060392999630e+03, 1.060787996210e+03, 1.061182130220e+03, 1.061575405820e+03, 1.061967827150e+03, 1.062359398330e+03, 1.062750123440e+03, 1.063140006540e+03, 1.063529051630e+03, 1.063917262720e+03, 1.064304643770e+03, 1.064691198710e+03, 1.065076931460e+03, 1.065461845880e+03, 1.065845945830e+03, 1.066229235130e+03, 1.066611717570e+03, 1.066993396930e+03, 1.067374276930e+03, 1.067754361300e+03, 1.068133653730e+03, 1.068512157860e+03, 1.068889877350e+03, 1.069266815790e+03, 1.069642976770e+03, 1.070018363860e+03, 1.070392980570e+03, 1.070766830430e+03, 1.071139916920e+03, 1.071512243500e+03, 1.071883813600e+03, 1.072254630630e+03, 1.072624697990e+03, 1.072994019050e+03, 1.073362597130e+03, 1.073730435560e+03, 1.074097537650e+03, 1.074463906650e+03, 1.074829545830e+03, 1.075194458420e+03, 1.075558647610e+03, 1.075922116610e+03, 1.076284868560e+03, 1.076646906620e+03, 1.077008233910e+03, 1.077368853520e+03, 1.077728768550e+03, 1.078087982040e+03, 1.078446497040e+03, 1.078804316570e+03, 1.079161443630e+03, 1.079517881190e+03, 1.079873632220e+03, 1.080228699650e+03, 1.080583086410e+03, 1.080936795400e+03, 1.081289829500e+03, 1.081642191580e+03, 1.081993884480e+03, 1.082344911030e+03, 1.082695274040e+03},
    {1.641546190130e+00, 4.999881147440e+00, 8.411621157630e+00, 1.187882861640e+01, 1.540369358400e+01, 1.898854434190e+01, 2.263585905280e+01, 2.634827866460e+01, 3.012862122090e+01, 3.397989776560e+01, 3.790533005890e+01, 4.190837035720e+01, 4.599272355210e+01, 5.016237201130e+01, 5.442160352730e+01, 5.877504284810e+01, 6.322768735240e+01, 6.778494753660e+01, 7.245269310680e+01, 7.723730562430e+01, 8.214573884330e+01, 8.718558811170e+01, 9.236517049470e+01, 9.769361763600e+01, 1.031809838150e+02, 1.088383722110e+02, 1.146780830770e+02, 1.207137883900e+02, 1.269607386400e+02, 1.334360087800e+02, 1.401587920830e+02, 1.471507527890e+02, 1.544364510920e+02, 1.620438572270e+02, 1.700049752720e+02, 1.783566015090e+02, 1.871412464520e+02, 1.964082525420e+02, 2.062151382040e+02, 2.166291871280e+02, 2.277292659240e+02, 2.396077667370e+02, 2.523723817940e+02, 2.661470325990e+02, 2.810705689210e+02, 2.972907461260e+02, 3.149497794780e+02, 3.341575886610e+02, 3.549509318500e+02, 3.772392233770e+02, 4.007396803250e+02, 4.249241863400e+02, 4.490511853660e+02, 4.723455882740e+02, 4.942561248480e+02, 5.144790595630e+02, 5.329157184550e+02, 5.496224605120e+02, 5.647413781210e+02, 5.784484639060e+02, 5.909219082530e+02, 6.023259406370e+02, 6.128045887390e+02, 6.224808912580e+02, 6.314586480700e+02, 6.398250545430e+02, 6.476534030450e+02, 6.550055115960e+02, 6.619337794070e+02, 6.684828753440e+02, 6.746911049160e+02, 6.805915105730e+02, 6.862127567760e+02, 6.915798438300e+02, 6.967146862810e+02, 7.016365843060e+02, 7.063626103740e+02, 7.109079284670e+02, 7.152860593470e+02, 7.195091023310e+02, 7.235879217750e+02, 7.275323047220e+02, 7.313510948390e+02, 7.350523067090e+02, 7.386432238020e+02, 7.421304827540e+02, 7.455201461630e+02, 7.488177656700e+02, 7.520284368070e+02, 7.551568468380e+02, 7.582073166130e+02, 7.611838372950e+02, 7.640901026710e+02, 7.669295376630e+02, 7.697053235500e+02, 7.724204203360e+02, 7.750775866340e+02, 7.776793973970e+02, 7.802282597460e+02, 7.827264271500e+02, 7.851760121470e+02, 7.875789977830e+02, 7.899372479230e+02, 7.922525165590e+02, 7.945264562300e+02, 7.967606256580e+02, 7.989564966750e+02, 8.011154605340e+02, 8.032388336530e+02, 8.053278628580e+02, 8.073837301820e+02, 8.094075572510e+02, 8.114004093140e+02, 8.133632989370e+02, 8.152971894090e+02, 8.172029978670e+02, 8.190815981890e+02, 8.209338236590e+02, 8.227604694310e+02, 8.245622948110e+02, 8.263400253740e+02, 8.280943549190e+02, 8.298259472990e+02, 8.315354381060e+02, 8.332234362560e+02, 8.348905254490e+02, 8.365372655440e+02, 8.381641938340e+02, 8.397718262430e+02, 8.413606584380e+02, 8.429311668800e+02, 8.444838097960e+02, 8.460190281020e+02, 8.475372462630e+02, 8.490388730970e+02, 8.505243025410e+02, 8.519939143610e+02, 8.534480748240e+02, 8.548871373340e+02, 8.563114430270e+02, 8.577213213310e+02, 8.591170905020e+02, 8.604990581200e+02, 8.618675215660e+02, 8.632227684680e+02, 8.645650771250e+02, 8.658947169070e+02, 8.672119486370e+02, 8.685170249490e+02, 8.698101906260e+02, 8.710916829320e+02, 8.723617319110e+02, 8.736205606810e+02, 8.748683857120e+02, 8.761054170920e+02, 8.773318587710e+02, 8.785479088050e+02, 8.797537595800e+02, 8.809495980310e+02, 8.821356058430e+02, 8.833119596530e+02, 8.844788312350e+02, 8.856363876760e+02, 8.867847915510e+02, 8.879242010820e+02, 8.890547702970e+02, 8.901766491770e+02, 8.912899837960e+02, 8.923949164610e+02, 8.934915858390e+02, 8.945801270850e+02, 8.956606719550e+02, 8.967333489280e+02, 8.977982833080e+02, 8.988555973360e+02, 8.999054102840e+02, 9.009478385550e+02, 9.019829957780e+02, 9.030109928890e+02, 9.040319382250e+02, 9.050459376010e+02, 9.060530943900e+02, 9.070535095990e+02, 9.080472819390e+02, 9.090345079010e+02, 9.100152818180e+02, 9.109896959300e+02, 9.119578404500e+02, 9.129198036200e+02, 9.138756717730e+02, 9.148255293830e+02, 9.157694591250e+02, 9.167075419210e+02, 9.176398569950e+02, 9.185664819170e+02, 9.194874926520e+02, 9.204029636030e+02, 9.213129676550e+02, 9.222175762180e+02, 9.231168592640e+02, 9.240108853710e+02, 9.248997217530e+02, 9.257834343050e+02, 9.266620876330e+02, 9.275357450860e+02, 9.284044687960e+02, 9.292683197030e+02, 9.301273575870e+02, 9.309816411020e+02, 9.318312278000e+02, 9.326761741590e+02, 9.335165356150e+02, 9.343523665810e+02, 9.351837204770e+02, 9.360106497550e+02, 9.368332059200e+02, 9.376514395520e+02, 9.384654003340e+02, 9.392751370680e+02, 9.400806976990e+02, 9.408821293320e+02, 9.416794782590e+02, 9.424727899680e+02, 9.432621091700e+02, 9.440474798150e+02, 9.448289451050e+02, 9.456065475170e+02, 9.463803288170e+02, 9.471503300750e+02, 9.479165916840e+02, 9.486791533700e+02, 9.494380542110e+02, 9.501933326520e+02, 9.509450265140e+02, 9.516931730110e+02, 9.524378087650e+02, 9.531789698130e+02, 9.539166916260e+02, 9.546510091170e+02, 9.553819566530e+02, 9.561095680690e+02, 9.568338766760e+02, 9.575549152750e+02, 9.582727161650e+02, 9.589873111560e+02, 9.596987315740e+02, 9.604070082790e+02, 9.611121716660e+02, 9.618142516800e+02, 9.625132778230e+02, 9.632092791620e+02, 9.639022843420e+02, 9.645923215860e+02, 9.652794187130e+02, 9.659636031390e+02, 9.666449018870e+02, 9.673233415940e+02, 9.679989485220e+02, 9.686717485600e+02, 9.693417672330e+02, 9.700090297120e+02, 9.706735608140e+02, 9.713353850170e+02, 9.719945264590e+02, 9.726510089490e+02, 9.733048559690e+02, 9.739560906850e+02, 9.746047359500e+02, 9.752508143080e+02, 9.758943480040e+02, 9.765353589840e+02, 9.771738689060e+02, 9.778098991400e+02, 9.784434707790e+02, 9.790746046370e+02, 9.797033212590e+02, 9.803296409240e+02, 9.809535836490e+02, 9.815751691960e+02, 9.821944170740e+02, 9.828113465430e+02, 9.834259766200e+02, 9.840383260830e+02, 9.846484134750e+02, 9.852562571060e+02, 9.858618750600e+02, 9.864652851960e+02, 9.870665051550e+02, 9.876655523580e+02, 9.882624440170e+02, 9.888571971340e+02, 9.894498285040e+02, 9.900403547190e+02, 9.906287921740e+02, 9.912151570680e+02, 9.917994654040e+02, 9.923817330000e+02, 9.929619754830e+02, 9.935402083000e+02, 9.941164467140e+02, 9.946907058130e+02, 9.952630005080e+02, 9.958333455380e+02, 9.964017554730e+02, 9.969682447160e+02, 9.975328275040e+02, 9.980955179140e+02, 9.986563298620e+02, 9.992152771080e+02, 9.997723732580e+02, 1.000327631760e+03, 1.000881065930e+03, 1.001432688910e+03, 1.001982513710e+03, 1.002530553200e+03, 1.003076820110e+03, 1.003621327020e+03, 1.004164086390e+03, 1.004705110510e+03, 1.005244411580e+03, 1.005782001650e+03, 1.006317892630e+03, 1.006852096330e+03, 1.007384624390e+03, 1.007915488380e+03, 1.008444699720e+03, 1.008972269690e+03, 1.009498209500e+03, 1.010022530200e+03, 1.010545242740e+03, 1.011066357970e+03, 1.011585886610e+03, 1.012103839270e+03, 1.012620226460e+03, 1.013135058580e+03, 1.013648345930e+03, 1.014160098690e+03, 1.014670326940e+03, 1.015179040670e+03, 1.015686249780e+03, 1.016191964030e+03, 1.016696193110e+03, 1.017198946630e+03, 1.017700234070e+03, 1.018200064840e+03, 1.018698448250e+03, 1.019195393510e+03, 1.019690909760e+03, 1.020185006040e+03, 1.020677691290e+03, 1.021168974380e+03, 1.021658864100e+03, 1.022147369140e+03, 1.022634498120e+03, 1.023120259550e+03, 1.023604661910e+03, 1.024087713540e+03, 1.024569422750e+03, 1.025049797750e+03, 1.025528846680e+03, 1.026006577580e+03, 1.026482998460e+03, 1.026958117210e+03, 1.027431941690e+03, 1.027904479640e+03, 1.028375738770e+03, 1.028845726690e+03, 1.029314450970e+03, 1.029781919090e+03, 1.030248138450e+03, 1.030713116430e+03, 1.031176860290e+03, 1.031639377250e+03, 1.032100674470e+03, 1.032560759040e+03, 1.033019637990e+03, 1.033477318270e+03, 1.033933806790e+03, 1.034389110400e+03, 1.034843235870e+03, 1.035296189920e+03, 1.035747979210e+03, 1.036198610360e+03, 1.036648089910e+03, 1.037096424350e+03, 1.037543620120e+03, 1.037989683580e+03, 1.038434621080e+03, 1.038878438880e+03, 1.039321143190e+03, 1.039762740180e+03, 1.040203235960e+03, 1.040642636600e+03, 1.041080948100e+03, 1.041518176410e+03, 1.041954327450e+03, 1.042389407080e+03, 1.042823421110e+03, 1.043256375290e+03, 1.043688275350e+03, 1.044119126950e+03, 1.044548935710e+03, 1.044977707200e+03, 1.045405446960e+03, 1.045832160460e+03, 1.046257853150e+03, 1.046682530420e+03, 1.047106197620e+03, 1.047528860060e+03, 1.047950523000e+03, 1.048371191670e+03, 1.048790871250e+03, 1.049209566870e+03, 1.049627283640e+03, 1.050044026620e+03, 1.050459800810e+03, 1.050874611200e+03, 1.051288462740e+03, 1.051701360310e+03, 1.052113308790e+03, 1.052524312990e+03, 1.052934377710e+03, 1.053343507680e+03, 1.053751707640e+03, 1.054158982240e+03, 1.054565336130e+03, 1.054970773920e+03, 1.055375300180e+03, 1.055778919440e+03, 1.056181636190e+03, 1.056583454910e+03, 1.056984380020e+03, 1.057384415940e+03, 1.057783567010e+03, 1.058181837570e+03, 1.058579231920e+03, 1.058975754340e+03, 1.059371409050e+03, 1.059766200260e+03, 1.060160132140e+03, 1.060553208830e+03, 1.060945434450e+03, 1.061336813080e+03, 1.061727348770e+03, 1.062117045530e+03, 1.062505907370e+03, 1.062893938240e+03, 1.063281142080e+03, 1.063667522800e+03, 1.064053084270e+03, 1.064437830340e+03, 1.064821764830e+03, 1.065204891550e+03, 1.065587214240e+03, 1.065968736660e+03, 1.066349462520e+03, 1.066729395510e+03, 1.067108539280e+03, 1.067486897470e+03, 1.067864473690e+03, 1.068241271530e+03, 1.068617294540e+03, 1.068992546260e+03, 1.069367030200e+03, 1.069740749830e+03, 1.070113708630e+03, 1.070485910030e+03, 1.070857357440e+03, 1.071228054240e+03, 1.071598003820e+03, 1.071967209500e+03, 1.072335674610e+03, 1.072703402450e+03, 1.073070396280e+03, 1.073436659370e+03, 1.073802194940e+03, 1.074167006200e+03, 1.074531096340e+03, 1.074894468520e+03, 1.075257125890e+03, 1.075619071560e+03, 1.075980308650e+03, 1.076340840230e+03, 1.076700669360e+03, 1.077059799080e+03, 1.077418232420e+03, 1.077775972370e+03, 1.078133021910e+03, 1.078489384010e+03, 1.078845061600e+03, 1.079200057610e+03, 1.079554374940e+03, 1.079908016470e+03, 1.080260985070e+03, 1.080613283580e+03},
    {1.636321922070e+00, 4.983553746890e+00, 8.383425286930e+00, 1.183794041120e+01, 1.534922509800e+01, 1.891953734580e+01, 2.255127809720e+01, 2.624700330330e+01, 3.000943727350e+01, 3.384148747870e+01, 3.774626099940e+01, 4.172708284420e+01, 4.578751639610e+01, 4.993138629020e+01, 5.416280407520e+01, 5.848619707180e+01, 6.290634091400e+01, 6.742839634710e+01, 7.205795096040e+01, 7.680106666110e+01, 8.166433384920e+01, 8.665493344440e+01, 9.178070814210e+01, 9.705024456510e+01, 1.024729683170e+02, 1.080592543800e+02, 1.138205558230e+02, 1.197695544420e+02, 1.259203377670e+02, 1.322886078590e+02, 1.388919285470e+02, 1.457500192620e+02, 1.528851053840e+02, 1.603223371580e+02, 1.680902915470e+02, 1.762215738450e+02, 1.847535378790e+02, 1.937291441930e+02, 2.031979726300e+02, 2.132173946880e+02, 2.238538834830e+02, 2.351843783490e+02, 2.472974969070e+02, 2.602941484850e+02, 2.742866784140e+02, 2.893950120080e+02, 3.057374847550e+02, 3.234136860740e+02, 3.424776374250e+02, 3.629019470570e+02, 3.845353309240e+02, 4.070599357970e+02, 4.299742499630e+02, 4.526507314720e+02, 4.744829913210e+02, 4.950538678760e+02, 5.141296641860e+02, 5.316360403980e+02, 5.476170118050e+02, 5.621840544530e+02, 5.754779355060e+02, 5.876446042770e+02, 5.988219598770e+02, 6.091338320560e+02, 6.186882541030e+02, 6.275780116460e+02, 6.358822102480e+02, 6.436681531430e+02, 6.509931735020e+02, 6.579062701020e+02, 6.644495025290e+02, 6.706591525890e+02, 6.765666789730e+02, 6.821994976640e+02, 6.875816192730e+02, 6.927341705540e+02, 6.976758228100e+02, 7.024231455640e+02, 7.069909001740e+02, 7.113922850010e+02, 7.156391413030e+02, 7.197421270800e+02, 7.237108646440e+02, 7.275540664660e+02, 7.312796429900e+02, 7.348947953770e+02, 7.384060955930e+02, 7.418195558110e+02, 7.451406887600e+02, 7.483745603650e+02, 7.515258358050e+02, 7.545988199160e+02, 7.575974927540e+02, 7.605255409490e+02, 7.633863854530e+02, 7.661832061270e+02, 7.689189636040e+02, 7.715964187590e+02, 7.742181500960e+02, 7.767865693060e+02, 7.793039352170e+02, 7.817723663370e+02, 7.841938521360e+02, 7.865702632400e+02, 7.889033606360e+02, 7.911948040120e+02, 7.934461593250e+02, 7.956589056810e+02, 7.978344415920e+02, 7.999740906920e+02, 8.020791069510e+02, 8.041506794500e+02, 8.061899367510e+02, 8.081979509140e+02, 8.101757411790e+02, 8.121242773630e+02, 8.140444829840e+02, 8.159372381480e+02, 8.178033822080e+02, 8.196437162330e+02, 8.214590052890e+02, 8.232499805500e+02, 8.250173412660e+02, 8.267617565850e+02, 8.284838672460e+02, 8.301842871640e+02, 8.318636048950e+02, 8.335223850120e+02, 8.351611693870e+02, 8.367804783860e+02, 8.383808119920e+02, 8.399626508490e+02, 8.415264572500e+02, 8.430726760550e+02, 8.446017355550e+02, 8.461140482850e+02, 8.476100117870e+02, 8.490900093260e+02, 8.505544105680e+02, 8.520035722160e+02, 8.534378386070e+02, 8.548575422790e+02, 8.562630045060e+02, 8.576545358010e+02, 8.590324363930e+02, 8.603969966750e+02, 8.617484976340e+02, 8.630872112500e+02, 8.644134008850e+02, 8.657273216350e+02, 8.670292206830e+02, 8.683193376210e+02, 8.695979047560e+02, 8.708651474110e+02, 8.721212841970e+02, 8.733665272840e+02, 8.746010826490e+02, 8.758251503190e+02, 8.770389245990e+02, 8.782425942890e+02, 8.794363428940e+02, 8.806203488190e+02, 8.817947855590e+02, 8.829598218800e+02, 8.841156219890e+02, 8.852623457000e+02, 8.864001485920e+02, 8.875291821530e+02, 8.886495939320e+02, 8.897615276690e+02, 8.908651234300e+02, 8.919605177320e+02, 8.930478436620e+02, 8.941272309940e+02, 8.951988062970e+02, 8.962626930450e+02, 8.973190117130e+02, 8.983678798800e+02, 8.994094123180e+02, 9.004437210830e+02, 9.014709156050e+02, 9.024911027630e+02, 9.035043869710e+02, 9.045108702530e+02, 9.055106523140e+02, 9.065038306130e+02, 9.074905004310e+02, 9.084707549350e+02, 9.094446852420e+02, 9.104123804790e+02, 9.113739278420e+02, 9.123294126520e+02, 9.132789184080e+02, 9.142225268430e+02, 9.151603179700e+02, 9.160923701310e+02, 9.170187600480e+02, 9.179395628650e+02, 9.188548521900e+02, 9.197647001410e+02, 9.206691773860e+02, 9.215683531780e+02, 9.224622953990e+02, 9.233510705930e+02, 9.242347440030e+02, 9.251133796030e+02, 9.259870401350e+02, 9.268557871380e+02, 9.277196809800e+02, 9.285787808880e+02, 9.294331449770e+02, 9.302828302780e+02, 9.311278927660e+02, 9.319683873860e+02, 9.328043680780e+02, 9.336358878020e+02, 9.344629985620e+02, 9.352857514310e+02, 9.361041965700e+02, 9.369183832530e+02, 9.377283598870e+02, 9.385341740320e+02, 9.393358724220e+02, 9.401335009850e+02, 9.409271048600e+02, 9.417167284150e+02, 9.425024152670e+02, 9.432842082980e+02, 9.440621496710e+02, 9.448362808470e+02, 9.456066426020e+02, 9.463732750400e+02, 9.471362176090e+02, 9.478955091150e+02, 9.486511877370e+02, 9.494032910400e+02, 9.501518559900e+02, 9.508969189630e+02, 9.516385157610e+02, 9.523766816240e+02, 9.531114512420e+02, 9.538428587630e+02, 9.545709378110e+02, 9.552957214900e+02, 9.560172423990e+02, 9.567355326430e+02, 9.574506238390e+02, 9.581625471300e+02, 9.588713331930e+02, 9.595770122480e+02, 9.602796140690e+02, 9.609791679900e+02, 9.616757029160e+02, 9.623692473310e+02, 9.630598293060e+02, 9.637474765070e+02, 9.644322162050e+02, 9.651140752780e+02, 9.657930802260e+02, 9.664692571720e+02, 9.671426318740e+02, 9.678132297280e+02, 9.684810757760e+02, 9.691461947160e+02, 9.698086109020e+02, 9.704683483580e+02, 9.711254307760e+02, 9.717798815290e+02, 9.724317236740e+02, 9.730809799560e+02, 9.737276728170e+02, 9.743718243990e+02, 9.750134565520e+02, 9.756525908370e+02, 9.762892485300e+02, 9.769234506310e+02, 9.775552178660e+02, 9.781845706910e+02, 9.788115293020e+02, 9.794361136310e+02, 9.800583433600e+02, 9.806782379160e+02, 9.812958164830e+02, 9.819110980030e+02, 9.825241011800e+02, 9.831348444820e+02, 9.837433461520e+02, 9.843496242020e+02, 9.849536964260e+02, 9.855555803960e+02, 9.861552934730e+02, 9.867528528040e+02, 9.873482753280e+02, 9.879415777820e+02, 9.885327767000e+02, 9.891218884180e+02, 9.897089290790e+02, 9.902939146330e+02, 9.908768608420e+02, 9.914577832830e+02, 9.920366973510e+02, 9.926136182590e+02, 9.931885610460e+02, 9.937615405760e+02, 9.943325715410e+02, 9.949016684650e+02, 9.954688457060e+02, 9.960341174590e+02, 9.965974977570e+02, 9.971590004750e+02, 9.977186393320e+02, 9.982764278930e+02, 9.988323795730e+02, 9.993865076370e+02, 9.999388252010e+02, 1.000489345240e+03, 1.001038080580e+03, 1.001585043920e+03, 1.002130247810e+03, 1.002673704660e+03, 1.003215426750e+03, 1.003755426240e+03, 1.004293715140e+03, 1.004830305350e+03, 1.005365208610e+03, 1.005898436590e+03, 1.006430000780e+03, 1.006959912580e+03, 1.007488183280e+03, 1.008014824020e+03, 1.008539845840e+03, 1.009063259670e+03, 1.009585076330e+03, 1.010105306510e+03, 1.010623960810e+03, 1.011141049710e+03, 1.011656583590e+03, 1.012170572710e+03, 1.012683027240e+03, 1.013193957250e+03, 1.013703372700e+03, 1.014211283460e+03, 1.014717699280e+03, 1.015222629840e+03, 1.015726084710e+03, 1.016228073360e+03, 1.016728605180e+03, 1.017227689470e+03, 1.017725335420e+03, 1.018221552150e+03, 1.018716348670e+03, 1.019209733930e+03, 1.019701716770e+03, 1.020192305960e+03, 1.020681510180e+03, 1.021169338030e+03, 1.021655798030e+03, 1.022140898600e+03, 1.022624648110e+03, 1.023107054830e+03, 1.023588126970e+03, 1.024067872640e+03, 1.024546299890e+03, 1.025023416690e+03, 1.025499230950e+03, 1.025973750490e+03, 1.026446983060e+03, 1.026918936340e+03, 1.027389617950e+03, 1.027859035420e+03, 1.028327196240e+03, 1.028794107810e+03, 1.029259777460e+03, 1.029724212480e+03, 1.030187420070e+03, 1.030649407370e+03, 1.031110181470e+03, 1.031569749370e+03, 1.032028118050e+03, 1.032485294380e+03, 1.032941285200e+03, 1.033396097290e+03, 1.033849737360e+03, 1.034302212060e+03, 1.034753528000e+03, 1.035203691700e+03, 1.035652709660e+03, 1.036100588300e+03, 1.036547334000e+03, 1.036992953060e+03, 1.037437451760e+03, 1.037880836310e+03, 1.038323112860e+03, 1.038764287520e+03, 1.039204366350e+03, 1.039643355330e+03, 1.040081260440e+03, 1.040518087570e+03, 1.040953842580e+03, 1.041388531260e+03, 1.041822159380e+03, 1.042254732650e+03, 1.042686256730e+03, 1.043116737230e+03, 1.043546179720e+03, 1.043974589730e+03, 1.044401972740e+03, 1.044828334180e+03, 1.045253679440e+03, 1.045678013860e+03, 1.046101342760e+03, 1.046523671390e+03, 1.046945004980e+03, 1.047365348690e+03, 1.047784707670e+03, 1.048203087000e+03, 1.048620491760e+03, 1.049036926940e+03, 1.049452397530e+03, 1.049866908460e+03, 1.050280464620e+03, 1.050693070890e+03, 1.051104732080e+03, 1.051515452970e+03, 1.051925238310e+03, 1.052334092800e+03, 1.052742021140e+03, 1.053149027940e+03, 1.053555117820e+03, 1.053960295340e+03, 1.054364565040e+03, 1.054767931400e+03, 1.055170398900e+03, 1.055571971970e+03, 1.055972654990e+03, 1.056372452340e+03, 1.056771368350e+03, 1.057169407300e+03, 1.057566573480e+03, 1.057962871100e+03, 1.058358304390e+03, 1.058752877500e+03, 1.059146594580e+03, 1.059539459730e+03, 1.059931477050e+03, 1.060322650570e+03, 1.060712984320e+03, 1.061102482290e+03, 1.061491148440e+03, 1.061878986710e+03, 1.062266001000e+03, 1.062652195180e+03, 1.063037573110e+03, 1.063422138600e+03, 1.063805895450e+03, 1.064188847430e+03, 1.064570998270e+03, 1.064952351690e+03, 1.065332911380e+03, 1.065712680990e+03, 1.066091664160e+03, 1.066469864490e+03, 1.066847285590e+03, 1.067223930990e+03, 1.067599804230e+03, 1.067974908830e+03, 1.068349248270e+03, 1.068722826000e+03, 1.069095645470e+03, 1.069467710090e+03, 1.069839023240e+03, 1.070209588300e+03, 1.070579408600e+03, 1.070948487460e+03, 1.071316828180e+03, 1.071684434040e+03, 1.072051308290e+03, 1.072417454160e+03, 1.072782874860e+03, 1.073147573570e+03, 1.073511553450e+03, 1.073874817670e+03, 1.074237369330e+03, 1.074599211530e+03, 1.074960347370e+03, 1.075320779900e+03, 1.075680512160e+03, 1.076039547180e+03, 1.076397887940e+03, 1.076755537440e+03, 1.077112498640e+03, 1.077468774470e+03, 1.077824367860e+03, 1.078179281720e+03, 1.078533518920e+03},
    {1.631131236130e+00, 4.967336819330e+00, 8.355430102200e+00, 1.179735818790e+01, 1.529518498220e+01, 1.885110049640e+01, 2.246743108270e+01, 2.614665071280e+01, 2.989139343260e+01, 3.370446713990e+01, 3.758886885760e+01, 4.154780169890e+01, 4.558469375510e+01, 4.970321916770e+01, 5.390732169580e+01, 5.820124113600e+01, 6.258954301500e+01, 6.707715204880e+01, 7.166938994580e+01, 7.637201824160e+01, 8.119128697280e+01, 8.613399015430e+01, 9.120752920760e+01, 9.641998570950e+01, 1.017802051080e+02, 1.072978933740e+02, 1.129837289800e+02, 1.188494930600e+02, 1.249082212340e+02, 1.311743812800e+02, 1.376640817130e+02, 1.443953173620e+02, 1.513882592220e+02, 1.586655971880e+02, 1.662529456990e+02, 1.741793235480e+02, 1.824777198700e+02, 1.911857576700e+02, 2.003464626740e+02, 2.100091356820e+02, 2.202303053750e+02, 2.310746952840e+02, 2.426160552880e+02, 2.549375554020e+02, 2.681311777700e+02, 2.822951402020e+02, 2.975278858440e+02, 3.139168383260e+02, 3.315205023870e+02, 3.503439745380e+02, 3.703097539120e+02, 3.912272130560e+02, 4.127694638240e+02, 4.344802342810e+02, 4.558373920200e+02, 4.763659375310e+02, 4.957452030170e+02, 5.137935621320e+02, 5.304542135120e+02, 5.457619283860e+02, 5.598051439540e+02, 5.726973862130e+02, 5.845587563330e+02, 5.955052009580e+02, 6.056430632160e+02, 6.150669485800e+02, 6.238595171750e+02, 6.320922803870e+02, 6.398268288890e+02, 6.471161645840e+02, 6.540059681910e+02, 6.605357293440e+02, 6.667397183050e+02, 6.726478044830e+02, 6.782861382380e+02, 6.836777158530e+02, 6.888428471300e+02, 6.937995429610e+02, 6.985638376010e+02, 7.031500578090e+02, 7.075710487100e+02, 7.118383643150e+02, 7.159624290530e+02, 7.199526753910e+02, 7.238176616440e+02, 7.275651732610e+02, 7.312023102490e+02, 7.347355629350e+02, 7.381708778350e+02, 7.415137151200e+02, 7.447690988940e+02, 7.479416613240e+02, 7.510356814660e+02, 7.540551195240e+02, 7.570036471470e+02, 7.598846743000e+02, 7.627013731350e+02, 7.654566992590e+02, 7.681534107210e+02, 7.707940849880e+02, 7.733811341810e+02, 7.759168187430e+02, 7.784032597530e+02, 7.808424500250e+02, 7.832362641430e+02, 7.855864675330e+02, 7.878947247010e+02, 7.901626067080e+02, 7.923915979730e+02, 7.945831024770e+02, 7.967384494200e+02, 7.988588983970e+02, 8.009456441370e+02, 8.029998208420e+02, 8.050225061850e+02, 8.070147249700e+02, 8.089774525140e+02, 8.109116177560e+02, 8.128181061390e+02, 8.146977622580e+02, 8.165513923220e+02, 8.183797664310e+02, 8.201836206870e+02, 8.219636591560e+02, 8.237205556880e+02, 8.254549556130e+02, 8.271674773220e+02, 8.288587137380e+02, 8.305292336860e+02, 8.321795831820e+02, 8.338102866250e+02, 8.354218479240e+02, 8.370147515480e+02, 8.385894635080e+02, 8.401464322810e+02, 8.416860896790e+02, 8.432088516600e+02, 8.447151190960e+02, 8.462052784900e+02, 8.476797026590e+02, 8.491387513650e+02, 8.505827719260e+02, 8.520120997760e+02, 8.534270590060e+02, 8.548279628680e+02, 8.562151142550e+02, 8.575888061520e+02, 8.589493220670e+02, 8.602969364330e+02, 8.616319149960e+02, 8.629545151750e+02, 8.642649864100e+02, 8.655635704900e+02, 8.668505018640e+02, 8.681260079350e+02, 8.693903093400e+02, 8.706436202230e+02, 8.718861484810e+02, 8.731180960120e+02, 8.743396589420e+02, 8.755510278460e+02, 8.767523879570e+02, 8.779439193630e+02, 8.791257972000e+02, 8.802981918330e+02, 8.814612690260e+02, 8.826151901120e+02, 8.837601121500e+02, 8.848961880730e+02, 8.860235668350e+02, 8.871423935510e+02, 8.882528096230e+02, 8.893549528760e+02, 8.904489576700e+02, 8.915349550210e+02, 8.926130727140e+02, 8.936834354040e+02, 8.947461647240e+02, 8.958013793790e+02, 8.968491952440e+02, 8.978897254510e+02, 8.989230804780e+02, 8.999493682340e+02, 9.009686941320e+02, 9.019811611770e+02, 9.029868700310e+02, 9.039859190880e+02, 9.049784045420e+02, 9.059644204540e+02, 9.069440588150e+02, 9.079174096050e+02, 9.088845608560e+02, 9.098455987040e+02, 9.108006074500e+02, 9.117496696060e+02, 9.126928659490e+02, 9.136302755710e+02, 9.145619759250e+02, 9.154880428710e+02, 9.164085507180e+02, 9.173235722720e+02, 9.182331788710e+02, 9.191374404280e+02, 9.200364254680e+02, 9.209302011660e+02, 9.218188333840e+02, 9.227023867010e+02, 9.235809244530e+02, 9.244545087590e+02, 9.253232005560e+02, 9.261870596310e+02, 9.270461446450e+02, 9.279005131660e+02, 9.287502216940e+02, 9.295953256890e+02, 9.304358795980e+02, 9.312719368750e+02, 9.321035500110e+02, 9.329307705550e+02, 9.337536491360e+02, 9.345722354880e+02, 9.353865784660e+02, 9.361967260730e+02, 9.370027254770e+02, 9.378046230300e+02, 9.386024642880e+02, 9.393962940310e+02, 9.401861562770e+02, 9.409720943040e+02, 9.417541506620e+02, 9.425323671930e+02, 9.433067850480e+02, 9.440774446960e+02, 9.448443859480e+02, 9.456076479640e+02, 9.463672692720e+02, 9.471232877780e+02, 9.478757407840e+02, 9.486246649970e+02, 9.493700965450e+02, 9.501120709850e+02, 9.508506233210e+02, 9.515857880110e+02, 9.523175989800e+02, 9.530460896320e+02, 9.537712928580e+02, 9.544932410510e+02, 9.552119661120e+02, 9.559274994620e+02, 9.566398720530e+02, 9.573491143740e+02, 9.580552564650e+02, 9.587583279200e+02, 9.594583579010e+02, 9.601553751460e+02, 9.608494079730e+02, 9.615404842930e+02, 9.622286316180e+02, 9.629138770640e+02, 9.635962473620e+02, 9.642757688670e+02, 9.649524675610e+02, 9.656263690640e+02, 9.662974986380e+02, 9.669658811940e+02, 9.676315413010e+02, 9.682945031900e+02, 9.689547907610e+02, 9.696124275900e+02, 9.702674369320e+02, 9.709198417310e+02, 9.715696646220e+02, 9.722169279400e+02, 9.728616537210e+02, 9.735038637120e+02, 9.741435793720e+02, 9.747808218820e+02, 9.754156121450e+02, 9.760479707940e+02, 9.766779181930e+02, 9.773054744490e+02, 9.779306594080e+02, 9.785534926640e+02, 9.791739935640e+02, 9.797921812100e+02, 9.804080744640e+02, 9.810216919530e+02, 9.816330520700e+02, 9.822421729810e+02, 9.828490726300e+02, 9.834537687370e+02, 9.840562788070e+02, 9.846566201310e+02, 9.852548097920e+02, 9.858508646650e+02, 9.864448014220e+02, 9.870366365360e+02, 9.876263862840e+02, 9.882140667480e+02, 9.887996938220e+02, 9.893832832110e+02, 9.899648504370e+02, 9.905444108400e+02, 9.911219795820e+02, 9.916975716510e+02, 9.922712018580e+02, 9.928428848470e+02, 9.934126350940e+02, 9.939804669110e+02, 9.945463944440e+02, 9.951104316830e+02, 9.956725924580e+02, 9.962328904460e+02, 9.967913391690e+02, 9.973479520010e+02, 9.979027421640e+02, 9.984557227380e+02, 9.990069066570e+02, 9.995563067150e+02, 1.000103935560e+03, 1.000649805720e+03, 1.001193929560e+03, 1.001736319340e+03, 1.002276987170e+03, 1.002815945030e+03, 1.003353204780e+03, 1.003888778160e+03, 1.004422676760e+03, 1.004954912090e+03, 1.005485495490e+03, 1.006014438220e+03, 1.006541751400e+03, 1.007067446040e+03, 1.007591533060e+03, 1.008114023220e+03, 1.008634927200e+03, 1.009154255580e+03, 1.009672018800e+03, 1.010188227230e+03, 1.010702891100e+03, 1.011216020560e+03, 1.011727625650e+03, 1.012237716320e+03, 1.012746302390e+03, 1.013253393630e+03, 1.013758999660e+03, 1.014263130050e+03, 1.014765794250e+03, 1.015267001630e+03, 1.015766761450e+03, 1.016265082910e+03, 1.016761975080e+03, 1.017257446990e+03, 1.017751507540e+03, 1.018244165580e+03, 1.018735429830e+03, 1.019225308980e+03, 1.019713811610e+03, 1.020200946200e+03, 1.020686721190e+03, 1.021171144910e+03, 1.021654225630e+03, 1.022135971520e+03, 1.022616390710e+03, 1.023095491210e+03, 1.023573280990e+03, 1.024049767940e+03, 1.024524959870e+03, 1.024998864510e+03, 1.025471489540e+03, 1.025942842550e+03, 1.026412931080e+03, 1.026881762600e+03, 1.027349344480e+03, 1.027815684070e+03, 1.028280788620e+03, 1.028744665330e+03, 1.029207321340e+03, 1.029668763710e+03, 1.030128999450e+03, 1.030588035510e+03, 1.031045878770e+03, 1.031502536050e+03, 1.031958014120e+03, 1.032412319690e+03, 1.032865459390e+03, 1.033317439810e+03, 1.033768267500e+03, 1.034217948910e+03, 1.034666490480e+03, 1.035113898570e+03, 1.035560179480e+03, 1.036005339470e+03, 1.036449384750e+03, 1.036892321460e+03, 1.037334155710e+03, 1.037774893540e+03, 1.038214540950e+03, 1.038653103880e+03, 1.039090588230e+03, 1.039526999850e+03, 1.039962344530e+03, 1.040396628030e+03, 1.040829856050e+03, 1.041262034250e+03, 1.041693168240e+03, 1.042123263570e+03, 1.042552325770e+03, 1.042980360320e+03, 1.043407372640e+03, 1.043833368110e+03, 1.044258352090e+03, 1.044682329860e+03, 1.045105306690e+03, 1.045527287790e+03, 1.045948278330e+03, 1.046368283450e+03, 1.046787308230e+03, 1.047205357730e+03, 1.047622436950e+03, 1.048038550880e+03, 1.048453704430e+03, 1.048867902510e+03, 1.049281149970e+03, 1.049693451630e+03, 1.050104812260e+03, 1.050515236620e+03, 1.050924729410e+03, 1.051333295300e+03, 1.051740938920e+03, 1.052147664880e+03, 1.052553477730e+03, 1.052958382010e+03, 1.053362382220e+03, 1.053765482810e+03, 1.054167688210e+03, 1.054569002810e+03, 1.054969430980e+03, 1.055368977050e+03, 1.055767645300e+03, 1.056165440010e+03, 1.056562365410e+03, 1.056958425690e+03, 1.057353625030e+03, 1.057747967560e+03, 1.058141457390e+03, 1.058534098610e+03, 1.058925895260e+03, 1.059316851360e+03, 1.059706970890e+03, 1.060096257820e+03, 1.060484716090e+03, 1.060872349580e+03, 1.061259162190e+03, 1.061645157750e+03, 1.062030340090e+03, 1.062414712990e+03, 1.062798280230e+03, 1.063181045550e+03, 1.063563012640e+03, 1.063944185210e+03, 1.064324566900e+03, 1.064704161360e+03, 1.065082972190e+03, 1.065461002970e+03, 1.065838257260e+03, 1.066214738600e+03, 1.066590450490e+03, 1.066965396420e+03, 1.067339579840e+03, 1.067713004190e+03, 1.068085672890e+03, 1.068457589320e+03, 1.068828756850e+03, 1.069199178820e+03, 1.069568858540e+03, 1.069937799330e+03, 1.070306004440e+03, 1.070673477140e+03, 1.071040220650e+03, 1.071406238180e+03, 1.071771532920e+03, 1.072136108030e+03, 1.072499966660e+03, 1.072863111930e+03, 1.073225546940e+03, 1.073587274770e+03, 1.073948298490e+03, 1.074308621140e+03, 1.074668245730e+03, 1.075027175260e+03, 1.075385412730e+03, 1.075742961080e+03, 1.076099823260e+03, 1.076456002200e+03},
    {1.625973803540e+00, 4.951229188930e+00, 8.327633294890e+00, 1.175707816090e+01, 1.524156755800e+01, 1.878322572220e+01, 2.238430694220e+01, 2.604720610520e+01, 2.977447031490e+01, 3.356881171670e+01, 3.743312167920e+01, 4.137048650900e+01, 4.538420489920e+01, 4.947780734310e+01, 5.365507778130e+01, 5.792007779610e+01, 6.227717371270e+01, 6.673106703290e+01, 7.128682869480e+01, 7.594993773990e+01, 8.072632507130e+01, 8.562242310770e+01, 9.064522228760e+01, 9.580233555360e+01, 1.011020721590e+02, 1.065535223940e+02, 1.121666551400e+02, 1.179524305220e+02, 1.239229303730e+02, 1.300915097400e+02, 1.364729732790e+02, 1.430837810670e+02, 1.499422891640e+02, 1.570690310350e+02, 1.644870467750e+02, 1.722222675660e+02, 1.803039628340e+02, 1.887652563360e+02, 1.976437138100e+02, 2.069819967550e+02, 2.168285605200e+02, 2.272383437400e+02, 2.382733391230e+02, 2.500028356880e+02, 2.625029568530e+02, 2.758548682380e+02, 2.901407111030e+02, 3.054360572830e+02, 3.217977829510e+02, 3.392470766810e+02, 3.577487118330e+02, 3.771889617600e+02, 3.973561099120e+02, 4.179326070310e+02, 4.385151900120e+02, 4.586736312050e+02, 4.780343498370e+02, 4.963472922590e+02, 5.134700858410e+02, 5.293587639450e+02, 5.440410633880e+02, 5.575878489350e+02, 5.700914045930e+02, 5.816510639000e+02, 5.923645318500e+02, 6.023231203380e+02, 6.116095266380e+02, 6.202971876880e+02, 6.284505455840e+02, 6.361257821050e+02, 6.433717436700e+02, 6.502308940080e+02, 6.567402088840e+02, 6.629319748600e+02, 6.688344814260e+02, 6.744726101960e+02, 6.798683314770e+02, 6.850411207240e+02, 6.900083073220e+02, 6.947853670220e+02, 6.993861677730e+02, 7.038231771590e+02, 7.081076381900e+02, 7.122497189400e+02, 7.162586405310e+02, 7.201427870580e+02, 7.239098004160e+02, 7.275666624300e+02, 7.311197662430e+02, 7.345749785880e+02, 7.379376942790e+02, 7.412128840330e+02, 7.444051365720e+02, 7.475186957610e+02, 7.505574934790e+02, 7.535251787620e+02, 7.564251437070e+02, 7.592605465560e+02, 7.620343322930e+02, 7.647492510820e+02, 7.674078747860e+02, 7.700126118120e+02, 7.725657204620e+02, 7.750693209780e+02, 7.775254064180e+02, 7.799358524930e+02, 7.823024264930e+02, 7.846267953840e+02, 7.869105331700e+02, 7.891551276110e+02, 7.913619863360e+02, 7.935324424390e+02, 7.956677596020e+02, 7.977691367770e+02, 7.998377124980e+02, 8.018745688330e+02, 8.038807350240e+02, 8.058571908430e+02, 8.078048696830e+02, 8.097246614160e+02, 8.116174150380e+02, 8.134839411150e+02, 8.153250140480e+02, 8.171413741810e+02, 8.189337297540e+02, 8.207027587200e+02, 8.224491104350e+02, 8.241734072400e+02, 8.258762459240e+02, 8.275581990990e+02, 8.292198164830e+02, 8.308616260950e+02, 8.324841353810e+02, 8.340878322610e+02, 8.356731861180e+02, 8.372406487170e+02, 8.387906550790e+02, 8.403236242890e+02, 8.418399602690e+02, 8.433400524970e+02, 8.448242766860e+02, 8.462929954250e+02, 8.477465587800e+02, 8.491853048670e+02, 8.506095603900e+02, 8.520196411460e+02, 8.534158525050e+02, 8.547984898680e+02, 8.561678390950e+02, 8.575241769120e+02, 8.588677712940e+02, 8.601988818380e+02, 8.615177601010e+02, 8.628246499390e+02, 8.641197878090e+02, 8.654034030740e+02, 8.666757182840e+02, 8.679369494400e+02, 8.691873062560e+02, 8.704269923970e+02, 8.716562057130e+02, 8.728751384580e+02, 8.740839775020e+02, 8.752829045320e+02, 8.764720962400e+02, 8.776517245090e+02, 8.788219565870e+02, 8.799829552500e+02, 8.811348789660e+02, 8.822778820440e+02, 8.834121147800e+02, 8.845377235970e+02, 8.856548511780e+02, 8.867636365910e+02, 8.878642154170e+02, 8.889567198610e+02, 8.900412788670e+02, 8.911180182240e+02, 8.921870606720e+02, 8.932485259980e+02, 8.943025311330e+02, 8.953491902410e+02, 8.963886148090e+02, 8.974209137290e+02, 8.984461933810e+02, 8.994645577080e+02, 9.004761082920e+02, 9.014809444290e+02, 9.024791631920e+02, 9.034708595010e+02, 9.044561261900e+02, 9.054350540600e+02, 9.064077319490e+02, 9.073742467820e+02, 9.083346836280e+02, 9.092891257540e+02, 9.102376546750e+02, 9.111803502060e+02, 9.121172905070e+02, 9.130485521290e+02, 9.139742100610e+02, 9.148943377700e+02, 9.158090072460e+02, 9.167182890380e+02, 9.176222522950e+02, 9.185209648030e+02, 9.194144930210e+02, 9.203029021170e+02, 9.211862559990e+02, 9.220646173510e+02, 9.229380476630e+02, 9.238066072620e+02, 9.246703553390e+02, 9.255293499830e+02, 9.263836482050e+02, 9.272333059640e+02, 9.280783781980e+02, 9.289189188420e+02, 9.297549808610e+02, 9.305866162640e+02, 9.314138761370e+02, 9.322368106580e+02, 9.330554691220e+02, 9.338698999590e+02, 9.346801507600e+02, 9.354862682900e+02, 9.362882985130e+02, 9.370862866050e+02, 9.378802769790e+02, 9.386703132940e+02, 9.394564384790e+02, 9.402386947490e+02, 9.410171236160e+02, 9.417917659100e+02, 9.425626617920e+02, 9.433298507670e+02, 9.440933717030e+02, 9.448532628400e+02, 9.456095618070e+02, 9.463623056340e+02, 9.471115307640e+02, 9.478572730690e+02, 9.485995678550e+02, 9.493384498830e+02, 9.500739533740e+02, 9.508061120210e+02, 9.515349590030e+02, 9.522605269940e+02, 9.529828481710e+02, 9.537019542280e+02, 9.544178763840e+02, 9.551306453930e+02, 9.558402915510e+02, 9.565468447100e+02, 9.572503342820e+02, 9.579507892520e+02, 9.586482381800e+02, 9.593427092190e+02, 9.600342301120e+02, 9.607228282100e+02, 9.614085304720e+02, 9.620913634770e+02, 9.627713534280e+02, 9.634485261620e+02, 9.641229071560e+02, 9.647945215330e+02, 9.654633940690e+02, 9.661295492010e+02, 9.667930110290e+02, 9.674538033280e+02, 9.681119495510e+02, 9.687674728320e+02, 9.694203959970e+02, 9.700707415680e+02, 9.707185317660e+02, 9.713637885200e+02, 9.720065334690e+02, 9.726467879680e+02, 9.732845730950e+02, 9.739199096540e+02, 9.745528181810e+02, 9.751833189460e+02, 9.758114319610e+02, 9.764371769820e+02, 9.770605735160e+02, 9.776816408220e+02, 9.783003979180e+02, 9.789168635820e+02, 9.795310563600e+02, 9.801429945670e+02, 9.807526962920e+02, 9.813601794020e+02, 9.819654615450e+02, 9.825685601520e+02, 9.831694924450e+02, 9.837682754360e+02, 9.843649259330e+02, 9.849594605440e+02, 9.855518956740e+02, 9.861422475390e+02, 9.867305321600e+02, 9.873167653680e+02, 9.879009628110e+02, 9.884831399520e+02, 9.890633120750e+02, 9.896414942870e+02, 9.902177015190e+02, 9.907919485320e+02, 9.913642499190e+02, 9.919346201020e+02, 9.925030733450e+02, 9.930696237460e+02, 9.936342852460e+02, 9.941970716300e+02, 9.947579965280e+02, 9.953170734180e+02, 9.958743156300e+02, 9.964297363450e+02, 9.969833485990e+02, 9.975351652860e+02, 9.980851991600e+02, 9.986334628340e+02, 9.991799687850e+02, 9.997247293560e+02, 1.000267756760e+03, 1.000809063070e+03, 1.001348660230e+03, 1.001886560090e+03, 1.002422774320e+03, 1.002957314500e+03, 1.003490192090e+03, 1.004021418420e+03, 1.004551004700e+03, 1.005078962040e+03, 1.005605301400e+03, 1.006130033670e+03, 1.006653169600e+03, 1.007174719830e+03, 1.007694694910e+03, 1.008213105270e+03, 1.008729961230e+03, 1.009245273010e+03, 1.009759050720e+03, 1.010271304400e+03, 1.010782043950e+03, 1.011291279190e+03, 1.011799019840e+03, 1.012305275520e+03, 1.012810055770e+03, 1.013313370010e+03, 1.013815227600e+03, 1.014315637780e+03, 1.014814609710e+03, 1.015312152480e+03, 1.015808275050e+03, 1.016302986340e+03, 1.016796295160e+03, 1.017288210220e+03, 1.017778740190e+03, 1.018267893630e+03, 1.018755679010e+03, 1.019242104750e+03, 1.019727179150e+03, 1.020210910480e+03, 1.020693306900e+03, 1.021174376500e+03, 1.021654127290e+03, 1.022132567230e+03, 1.022609704190e+03, 1.023085545950e+03, 1.023560100250e+03, 1.024033374740e+03, 1.024505377010e+03, 1.024976114570e+03, 1.025445594890e+03, 1.025913825320e+03, 1.026380813200e+03, 1.026846565780e+03, 1.027311090230e+03, 1.027774393680e+03, 1.028236483190e+03, 1.028697365760e+03, 1.029157048310e+03, 1.029615537720e+03, 1.030072840810e+03, 1.030528964320e+03, 1.030983914960e+03, 1.031437699350e+03, 1.031890324080e+03, 1.032341795670e+03, 1.032792120580e+03, 1.033241305220e+03, 1.033689355960e+03, 1.034136279080e+03, 1.034582080850e+03, 1.035026767450e+03, 1.035470345020e+03, 1.035912819660e+03, 1.036354197410e+03, 1.036794484240e+03, 1.037233686120e+03, 1.037671808910e+03, 1.038108858460e+03, 1.038544840570e+03, 1.038979760970e+03, 1.039413625370e+03, 1.039846439420e+03, 1.040278208710e+03, 1.040708938800e+03, 1.041138635220e+03, 1.041567303410e+03, 1.041994948820e+03, 1.042421576820e+03, 1.042847192740e+03, 1.043271801880e+03, 1.043695409490e+03, 1.044118020780e+03, 1.044539640920e+03, 1.044960275030e+03, 1.045379928200e+03, 1.045798605480e+03, 1.046216311870e+03, 1.046633052330e+03, 1.047048831810e+03, 1.047463655180e+03, 1.047877527300e+03, 1.048290452980e+03, 1.048702437000e+03, 1.049113484100e+03, 1.049523598980e+03, 1.049932786310e+03, 1.050341050720e+03, 1.050748396800e+03, 1.051154829120e+03, 1.051560352200e+03, 1.051964970530e+03, 1.052368688580e+03, 1.052771510760e+03, 1.053173441470e+03, 1.053574485060e+03, 1.053974645860e+03, 1.054373928170e+03, 1.054772336240e+03, 1.055169874300e+03, 1.055566546560e+03, 1.055962357170e+03, 1.056357310280e+03, 1.056751409990e+03, 1.057144660380e+03, 1.057537065490e+03, 1.057928629340e+03, 1.058319355920e+03, 1.058709249180e+03, 1.059098313060e+03, 1.059486551460e+03, 1.059873968250e+03, 1.060260567270e+03, 1.060646352350e+03, 1.061031327270e+03, 1.061415495800e+03, 1.061798861670e+03, 1.062181428600e+03, 1.062563200270e+03, 1.062944180340e+03, 1.063324372430e+03, 1.063703780170e+03, 1.064082407120e+03, 1.064460256850e+03, 1.064837332880e+03, 1.065213638730e+03, 1.065589177870e+03, 1.065963953770e+03, 1.066337969860e+03, 1.066711229550e+03, 1.067083736230e+03, 1.067455493260e+03, 1.067826503990e+03, 1.068196771730e+03, 1.068566299780e+03, 1.068935091410e+03, 1.069303149880e+03, 1.069670478410e+03, 1.070037080220e+03, 1.070402958480e+03, 1.070768116360e+03, 1.071132557010e+03, 1.071496283550e+03, 1.071859299080e+03, 1.072221606680e+03, 1.072583209410e+03, 1.072944110320e+03, 1.073304312420e+03, 1.073663818710e+03, 1.074022632170e+03, 1.074380755770e+03},
    {1.620849299910e+00, 4.935229697520e+00, 8.300032594830e+00, 1.171709661290e+01, 1.518836725870e+01, 1.871590512220e+01, 2.230189485880e+01, 2.594865504810e+01, 2.965864903240e+01, 3.343449685440e+01, 3.727898842630e+01, 4.119509808700e+01, 4.518600072270e+01, 4.925508965410e+01, 5.340599652480e+01, 5.764261346070e+01, 6.196911781360e+01, 6.638999985140e+01, 7.091009381810e+01, 7.553461285550e+01, 8.026918836210e+01, 8.511991446310e+01, 9.009339838360e+01, 9.519681765400e+01, 1.004379852440e+02, 1.058254239170e+02, 1.113684513260e+02, 1.170772776580e+02, 1.229631179250e+02, 1.290383214110e+02, 1.353165211640e+02, 1.418128069180e+02, 1.485439252940e+02, 1.555285116280e+02, 1.627873581220e+02, 1.703437231550e+02, 1.782236861630e+02, 1.864565510650e+02, 1.950752978820e+02, 2.041170755010e+02, 2.136237159120e+02, 2.236422275300e+02, 2.342251854760e+02, 2.454308699090e+02, 2.573228961250e+02, 2.699689210330e+02, 2.834378064560e+02, 2.977944325350e+02, 3.130913473210e+02, 3.293568534540e+02, 3.465800612160e+02, 3.646945196470e+02, 3.835629060370e+02, 4.029668574710e+02, 4.226097003550e+02, 4.421419023060e+02, 4.612106660120e+02, 4.795205083890e+02, 4.968739680590e+02, 5.131585955280e+02, 5.283399557490e+02, 5.424403617480e+02, 5.555171082170e+02, 5.676457725190e+02, 5.789089425140e+02, 5.893891848870e+02, 5.991649575650e+02, 6.083084766640e+02, 6.168848484250e+02, 6.249519880960e+02, 6.325609938960e+02, 6.397567513720e+02, 6.465786237570e+02, 6.530611421520e+02, 6.592346491270e+02, 6.651258747920e+02, 6.707584395940e+02, 6.761532863950e+02, 6.813290484420e+02, 6.863023612910e+02, 6.910881268440e+02, 6.956997370060e+02, 7.001492635820e+02, 7.044476200000e+02, 7.086046995900e+02, 7.126294942740e+02, 7.165301968770e+02, 7.203142896610e+02, 7.239886212430e+02, 7.275594736520e+02, 7.310326210030e+02, 7.344133809810e+02, 7.377066601650e+02, 7.409169940270e+02, 7.440485823350e+02, 7.471053205520e+02, 7.500908277560e+02, 7.530084715240e+02, 7.558613901480e+02, 7.586525125170e+02, 7.613845759440e+02, 7.640601421740e+02, 7.666816118030e+02, 7.692512372640e+02, 7.717711345640e+02, 7.742432939060e+02, 7.766695893060e+02, 7.790517873280e+02, 7.813915550330e+02, 7.836904672010e+02, 7.859500129360e+02, 7.881716016860e+02, 7.903565687550e+02, 7.925061803530e+02, 7.946216382260e+02, 7.967040839090e+02, 7.987546026370e+02, 8.007742269540e+02, 8.027639400280e+02, 8.047246787220e+02, 8.066573364270e+02, 8.085627656880e+02, 8.104417806320e+02, 8.122951592200e+02, 8.141236453460e+02, 8.159279507730e+02, 8.177087569510e+02, 8.194667166930e+02, 8.212024557510e+02, 8.229165742810e+02, 8.246096482090e+02, 8.262822305080e+02, 8.279348524000e+02, 8.295680244690e+02, 8.311822377180e+02, 8.327779645460e+02, 8.343556596740e+02, 8.359157610140e+02, 8.374586904850e+02, 8.389848547740e+02, 8.404946460650e+02, 8.419884427150e+02, 8.434666098940e+02, 8.449295001930e+02, 8.463774541910e+02, 8.478108009960e+02, 8.492298587510e+02, 8.506349351200e+02, 8.520263277390e+02, 8.534043246500e+02, 8.547692047070e+02, 8.561212379650e+02, 8.574606860450e+02, 8.587878024840e+02, 8.601028330650e+02, 8.614060161290e+02, 8.626975828750e+02, 8.639777576430e+02, 8.652467581840e+02, 8.665047959160e+02, 8.677520761690e+02, 8.689887984160e+02, 8.702151564970e+02, 8.714313388270e+02, 8.726375286030e+02, 8.738339039890e+02, 8.750206383070e+02, 8.761979002070e+02, 8.773658538360e+02, 8.785246590000e+02, 8.796744713120e+02, 8.808154423450e+02, 8.819477197670e+02, 8.830714474770e+02, 8.841867657320e+02, 8.852938112730e+02, 8.863927174370e+02, 8.874836142770e+02, 8.885666286650e+02, 8.896418843980e+02, 8.907095022970e+02, 8.917696003020e+02, 8.928222935650e+02, 8.938676945400e+02, 8.949059130610e+02, 8.959370564320e+02, 8.969612294980e+02, 8.979785347260e+02, 8.989890722720e+02, 8.999929400550e+02, 9.009902338230e+02, 9.019810472150e+02, 9.029654718280e+02, 9.039435972710e+02, 9.049155112260e+02, 9.058812995060e+02, 9.068410461040e+02, 9.077948332450e+02, 9.087427414410e+02, 9.096848495320e+02, 9.106212347380e+02, 9.115519727020e+02, 9.124771375320e+02, 9.133968018440e+02, 9.143110368020e+02, 9.152199121590e+02, 9.161234962890e+02, 9.170218562320e+02, 9.179150577210e+02, 9.188031652220e+02, 9.196862419620e+02, 9.205643499660e+02, 9.214375500840e+02, 9.223059020230e+02, 9.231694643750e+02, 9.240282946430e+02, 9.248824492740e+02, 9.257319836790e+02, 9.265769522620e+02, 9.274174084430e+02, 9.282534046830e+02, 9.290849925070e+02, 9.299122225260e+02, 9.307351444600e+02, 9.315538071580e+02, 9.323682586170e+02, 9.331785460050e+02, 9.339847156810e+02, 9.347868132090e+02, 9.355848833810e+02, 9.363789702320e+02, 9.371691170570e+02, 9.379553664310e+02, 9.387377602220e+02, 9.395163396060e+02, 9.402911450880e+02, 9.410622165090e+02, 9.418295930660e+02, 9.425933133250e+02, 9.433534152340e+02, 9.441099361370e+02, 9.448629127860e+02, 9.456123813530e+02, 9.463583774460e+02, 9.471009361170e+02, 9.478400918760e+02, 9.485758787000e+02, 9.493083300470e+02, 9.500374788630e+02, 9.507633575980e+02, 9.514859982110e+02, 9.522054321800e+02, 9.529216905170e+02, 9.536348037730e+02, 9.543448020460e+02, 9.550517149950e+02, 9.557555718430e+02, 9.564564013910e+02, 9.571542320230e+02, 9.578490917140e+02, 9.585410080390e+02, 9.592300081810e+02, 9.599161189380e+02, 9.605993667300e+02, 9.612797776070e+02, 9.619573772530e+02, 9.626321910000e+02, 9.633042438270e+02, 9.639735603690e+02, 9.646401649260e+02, 9.653040814660e+02, 9.659653336320e+02, 9.666239447500e+02, 9.672799378300e+02, 9.679333355760e+02, 9.685841603910e+02, 9.692324343810e+02, 9.698781793610e+02, 9.705214168590e+02, 9.711621681230e+02, 9.718004541240e+02, 9.724362955640e+02, 9.730697128770e+02, 9.737007262360e+02, 9.743293555540e+02, 9.749556204960e+02, 9.755795404740e+02, 9.762011346580e+02, 9.768204219770e+02, 9.774374211250e+02, 9.780521505630e+02, 9.786646285230e+02, 9.792748730150e+02, 9.798829018250e+02, 9.804887325250e+02, 9.810923824720e+02, 9.816938688130e+02, 9.822932084880e+02, 9.828904182360e+02, 9.834855145930e+02, 9.840785139000e+02, 9.846694323040e+02, 9.852582857630e+02, 9.858450900450e+02, 9.864298607350e+02, 9.870126132370e+02, 9.875933627740e+02, 9.881721243970e+02, 9.887489129790e+02, 9.893237432260e+02, 9.898966296750e+02, 9.904675866980e+02, 9.910366285040e+02, 9.916037691410e+02, 9.921690225000e+02, 9.927324023180e+02, 9.932939221760e+02, 9.938535955060e+02, 9.944114355900e+02, 9.949674555670e+02, 9.955216684270e+02, 9.960740870230e+02, 9.966247240640e+02, 9.971735921230e+02, 9.977207036370e+02, 9.982660709090e+02, 9.988097061100e+02, 9.993516212810e+02, 9.998918283350e+02, 1.000430339060e+03, 1.000967165110e+03, 1.001502318040e+03, 1.002035809250e+03, 1.002567650050e+03, 1.003097851620e+03, 1.003626425030e+03, 1.004153381220e+03, 1.004678731030e+03, 1.005202485190e+03, 1.005724654320e+03, 1.006245248930e+03, 1.006764279410e+03, 1.007281756070e+03, 1.007797689110e+03, 1.008312088600e+03, 1.008824964540e+03, 1.009336326830e+03, 1.009846185250e+03, 1.010354549500e+03, 1.010861429180e+03, 1.011366833790e+03, 1.011870772750e+03, 1.012373255370e+03, 1.012874290890e+03, 1.013373888450e+03, 1.013872057090e+03, 1.014368805780e+03, 1.014864143400e+03, 1.015358078750e+03, 1.015850620520e+03, 1.016341777360e+03, 1.016831557800e+03, 1.017319970300e+03, 1.017807023260e+03, 1.018292724980e+03, 1.018777083680e+03, 1.019260107520e+03, 1.019741804570e+03, 1.020222182840e+03, 1.020701250250e+03, 1.021179014650e+03, 1.021655483840e+03, 1.022130665510e+03, 1.022604567310e+03, 1.023077196810e+03, 1.023548561520e+03, 1.024018668860e+03, 1.024487526210e+03, 1.024955140880e+03, 1.025421520080e+03, 1.025886671010e+03, 1.026350600770e+03, 1.026813316400e+03, 1.027274824880e+03, 1.027735133140e+03, 1.028194248050e+03, 1.028652176390e+03, 1.029108924920e+03, 1.029564500320e+03, 1.030018909210e+03, 1.030472158150e+03, 1.030924253670e+03, 1.031375202220e+03, 1.031825010190e+03, 1.032273683940e+03, 1.032721229750e+03, 1.033167653870e+03, 1.033612962470e+03, 1.034057161700e+03, 1.034500257630e+03, 1.034942256290e+03, 1.035383163670e+03, 1.035822985690e+03, 1.036261728230e+03, 1.036699397140e+03, 1.037135998180e+03, 1.037571537110e+03, 1.038006019600e+03, 1.038439451300e+03, 1.038871837800e+03, 1.039303184670e+03, 1.039733497390e+03, 1.040162781440e+03, 1.040591042230e+03, 1.041018285130e+03, 1.041444515470e+03, 1.041869738550e+03, 1.042293959600e+03, 1.042717183840e+03, 1.043139416410e+03, 1.043560662450e+03, 1.043980927030e+03, 1.044400215190e+03, 1.044818531940e+03, 1.045235882230e+03, 1.045652271000e+03, 1.046067703120e+03, 1.046482183440e+03, 1.046895716760e+03, 1.047308307870e+03, 1.047719961500e+03, 1.048130682340e+03, 1.048540475060e+03, 1.048949344290e+03, 1.049357294610e+03, 1.049764330590e+03, 1.050170456740e+03, 1.050575677550e+03, 1.050979997480e+03, 1.051383420950e+03, 1.051785952350e+03, 1.052187596020e+03, 1.052588356300e+03, 1.052988237470e+03, 1.053387243780e+03, 1.053785379470e+03, 1.054182648740e+03, 1.054579055730e+03, 1.054974604600e+03, 1.055369299430e+03, 1.055763144310e+03, 1.056156143280e+03, 1.056548300350e+03, 1.056939619510e+03, 1.057330104700e+03, 1.057719759870e+03, 1.058108588910e+03, 1.058496595680e+03, 1.058883784040e+03, 1.059270157800e+03, 1.059655720740e+03, 1.060040476640e+03, 1.060424429220e+03, 1.060807582190e+03, 1.061189939230e+03, 1.061571504000e+03, 1.061952280140e+03, 1.062332271240e+03, 1.062711480890e+03, 1.063089912640e+03, 1.063467570020e+03, 1.063844456530e+03, 1.064220575670e+03, 1.064595930880e+03, 1.064970525590e+03, 1.065344363230e+03, 1.065717447170e+03, 1.066089780780e+03, 1.066461367400e+03, 1.066832210340e+03, 1.067202312900e+03, 1.067571678360e+03, 1.067940309960e+03, 1.068308210930e+03, 1.068675384480e+03, 1.069041833800e+03, 1.069407562040e+03, 1.069772572350e+03, 1.070136867860e+03, 1.070500451660e+03, 1.070863326830e+03, 1.071225496440e+03, 1.071586963520e+03, 1.071947731090e+03, 1.072307802150e+03},
    {1.615757405150e+00, 4.919337204240e+00, 8.272625769360e+00, 1.167740989380e+01, 1.513557862540e+01, 1.864913096050e+01, 2.222018425730e+01, 2.585098345310e+01, 2.954391117510e+01, 3.330149885170e+01, 3.712643894190e+01, 4.102159842070e+01, 4.499003367360e+01, 4.903500697970e+01, 5.316000478600e+01, 5.736875800740e+01, 6.166526462210e+01, 6.605381487320e+01, 7.053901943720e+01, 7.512584097520e+01, 7.981962955210e+01, 8.462616248550e+01, 8.955168928210e+01, 9.460298242370e+01, 9.978739489760e+01, 1.051129255130e+02, 1.105882932160e+02, 1.162230218300e+02, 1.220275368580e+02, 1.280132762590e+02, 1.341928173920e+02, 1.405800226120e+02, 1.471902062990e+02, 1.540403263540e+02, 1.611492032900e+02, 1.685377699460e+02, 1.762293542030e+02, 1.842499956620e+02, 1.926287942250e+02, 2.013982831040e+02, 2.105948089550e+02, 2.202588851350e+02, 2.304354559480e+02, 2.411739641970e+02, 2.525280433910e+02, 2.645545528160e+02, 2.773115411990e+02, 2.908545944020e+02, 3.052309821350e+02, 3.204712169380e+02, 3.365781963380e+02, 3.535149120870e+02, 3.711924414400e+02, 3.894605902270e+02, 4.081049052080e+02, 4.268557164150e+02, 4.454139297220e+02, 4.634906870920e+02, 4.808507465700e+02, 4.973364932690e+02, 5.128584978030e+02, 5.273895040340e+02, 5.409475982360e+02, 5.535794782640e+02, 5.653474700680e+02, 5.763205976550e+02, 5.865688609860e+02, 5.961597629470e+02, 6.051563496930e+02, 6.136162584330e+02, 6.215914248660e+02, 6.291282043320e+02, 6.362677317970e+02, 6.430463997330e+02, 6.494963746910e+02, 6.556461043270e+02, 6.615207884610e+02, 6.671428020510e+02, 6.725320668590e+02, 6.777063735670e+02, 6.826816586690e+02, 6.874722414710e+02, 6.920910266270e+02, 6.965496773060e+02, 7.008587635400e+02, 7.050278896410e+02, 7.090658040220e+02, 7.129804941890e+02, 7.167792692090e+02, 7.204688315590e+02, 7.240553399600e+02, 7.275444644920e+02, 7.309414351010e+02, 7.342510844090e+02, 7.374778855990e+02, 7.406259860360e+02, 7.436992371600e+02, 7.467012211370e+02, 7.496352746690e+02, 7.525045103020e+02, 7.553118355390e+02, 7.580599700170e+02, 7.607514609650e+02, 7.633886971480e+02, 7.659739214610e+02, 7.685092423290e+02, 7.709966440310e+02, 7.734379960790e+02, 7.758350617440e+02, 7.781895058180e+02, 7.805029016910e+02, 7.827767378200e+02, 7.850124236340e+02, 7.872112949550e+02, 7.893746189640e+02, 7.915035987520e+02, 7.935993775220e+02, 7.956630424430e+02, 7.976956282090e+02, 7.996981203200e+02, 8.016714581190e+02, 8.036165375950e+02, 8.055342139800e+02, 8.074253041620e+02, 8.092905889150e+02, 8.111308149800e+02, 8.129466969920e+02, 8.147389192790e+02, 8.165081375380e+02, 8.182549803940e+02, 8.199800508590e+02, 8.216839276950e+02, 8.233671666850e+02, 8.250303018260e+02, 8.266738464450e+02, 8.282982942460e+02, 8.299041202910e+02, 8.314917819210e+02, 8.330617196230e+02, 8.346143578440e+02, 8.361501057530e+02, 8.376693579700e+02, 8.391724952390e+02, 8.406598850680e+02, 8.421318823410e+02, 8.435888298790e+02, 8.450310589860e+02, 8.464588899550e+02, 8.478726325500e+02, 8.492725864640e+02, 8.506590417480e+02, 8.520322792210e+02, 8.533925708600e+02, 8.547401801620e+02, 8.560753625000e+02, 8.573983654480e+02, 8.587094291000e+02, 8.600087863660e+02, 8.612966632600e+02, 8.625732791640e+02, 8.638388470940e+02, 8.650935739380e+02, 8.663376606930e+02, 8.675713026850e+02, 8.687946897860e+02, 8.700080066100e+02, 8.712114327110e+02, 8.724051427660e+02, 8.735893067490e+02, 8.747640901020e+02, 8.759296538950e+02, 8.770861549780e+02, 8.782337461320e+02, 8.793725762040e+02, 8.805027902480e+02, 8.816245296490e+02, 8.827379322480e+02, 8.838431324640e+02, 8.849402614000e+02, 8.860294469600e+02, 8.871108139480e+02, 8.881844841710e+02, 8.892505765310e+02, 8.903092071250e+02, 8.913604893260e+02, 8.924045338720e+02, 8.934414489480e+02, 8.944713402620e+02, 8.954943111270e+02, 8.965104625250e+02, 8.975198931850e+02, 8.985226996450e+02, 8.995189763210e+02, 9.005088155680e+02, 9.014923077380e+02, 9.024695412410e+02, 9.034406026030e+02, 9.044055765120e+02, 9.053645458800e+02, 9.063175918860e+02, 9.072647940270e+02, 9.082062301660e+02, 9.091419765740e+02, 9.100721079760e+02, 9.109966975930e+02, 9.119158171820e+02, 9.128295370760e+02, 9.137379262190e+02, 9.146410522090e+02, 9.155389813250e+02, 9.164317785710e+02, 9.173195077010e+02, 9.182022312540e+02, 9.190800105880e+02, 9.199529059050e+02, 9.208209762840e+02, 9.216842797090e+02, 9.225428730930e+02, 9.233968123070e+02, 9.242461522080e+02, 9.250909466580e+02, 9.259312485540e+02, 9.267671098480e+02, 9.275985815700e+02, 9.284257138510e+02, 9.292485559440e+02, 9.300671562440e+02, 9.308815623090e+02, 9.316918208790e+02, 9.324979778950e+02, 9.333000785180e+02, 9.340981671460e+02, 9.348922874310e+02, 9.356824822970e+02, 9.364687939570e+02, 9.372512639260e+02, 9.380299330390e+02, 9.388048414650e+02, 9.395760287230e+02, 9.403435336940e+02, 9.411073946360e+02, 9.418676491980e+02, 9.426243344320e+02, 9.433774868060e+02, 9.441271422170e+02, 9.448733360030e+02, 9.456161029540e+02, 9.463554773240e+02, 9.470914928410e+02, 9.478241827210e+02, 9.485535796730e+02, 9.492797159170e+02, 9.500026231860e+02, 9.507223327410e+02, 9.514388753790e+02, 9.521522814430e+02, 9.528625808280e+02, 9.535698029940e+02, 9.542739769720e+02, 9.549751313740e+02, 9.556732944000e+02, 9.563684938460e+02, 9.570607571110e+02, 9.577501112080e+02, 9.584365827670e+02, 9.591201980460e+02, 9.598009829360e+02, 9.604789629670e+02, 9.611541633170e+02, 9.618266088190e+02, 9.624963239630e+02, 9.631633329080e+02, 9.638276594840e+02, 9.644893272000e+02, 9.651483592490e+02, 9.658047785140e+02, 9.664586075740e+02, 9.671098687090e+02, 9.677585839040e+02, 9.684047748570e+02, 9.690484629830e+02, 9.696896694180e+02, 9.703284150230e+02, 9.709647203940e+02, 9.715986058590e+02, 9.722300914900e+02, 9.728591971000e+02, 9.734859422560e+02, 9.741103462760e+02, 9.747324282350e+02, 9.753522069710e+02, 9.759697010900e+02, 9.765849289630e+02, 9.771979087400e+02, 9.778086583450e+02, 9.784171954830e+02, 9.790235376460e+02, 9.796277021120e+02, 9.802297059520e+02, 9.808295660330e+02, 9.814272990160e+02, 9.820229213700e+02, 9.826164493630e+02, 9.832078990760e+02, 9.837972863960e+02, 9.843846270270e+02, 9.849699364900e+02, 9.855532301250e+02, 9.861345230950e+02, 9.867138303870e+02, 9.872911668190e+02, 9.878665470360e+02, 9.884399855190e+02, 9.890114965840e+02, 9.895810943860e+02, 9.901487929210e+02, 9.907146060260e+02, 9.912785473870e+02, 9.918406305350e+02, 9.924008688540e+02, 9.929592755770e+02, 9.935158637960e+02, 9.940706464550e+02, 9.946236363610e+02, 9.951748461800e+02, 9.957242884410e+02, 9.962719755380e+02, 9.968179197330e+02, 9.973621331570e+02, 9.979046278090e+02, 9.984454155650e+02, 9.989845081710e+02, 9.995219172520e+02, 1.000057654310e+03, 1.000591730730e+03, 1.001124157770e+03, 1.001654946580e+03, 1.002184108190e+03, 1.002711653520e+03, 1.003237593380e+03, 1.003761938450e+03, 1.004284699330e+03, 1.004805886490e+03, 1.005325510310e+03, 1.005843581050e+03, 1.006360108870e+03, 1.006875103850e+03, 1.007388575940e+03, 1.007900535010e+03, 1.008410990820e+03, 1.008919953030e+03, 1.009427431240e+03, 1.009933434900e+03, 1.010437973420e+03, 1.010941056080e+03, 1.011442692100e+03, 1.011942890590e+03, 1.012441660570e+03, 1.012939010990e+03, 1.013434950710e+03, 1.013929488500e+03, 1.014422633040e+03, 1.014914392950e+03, 1.015404776740e+03, 1.015893792860e+03, 1.016381449670e+03, 1.016867755470e+03, 1.017352718460e+03, 1.017836346770e+03, 1.018318648470e+03, 1.018799631530e+03, 1.019279303860e+03, 1.019757673310e+03, 1.020234747630e+03, 1.020710534530e+03, 1.021185041620e+03, 1.021658276470e+03, 1.022130246550e+03, 1.022600959290e+03, 1.023070422050e+03, 1.023538642110e+03, 1.024005626690e+03, 1.024471382950e+03, 1.024935917990e+03, 1.025399238840e+03, 1.025861352470e+03, 1.026322265780e+03, 1.026781985640e+03, 1.027240518810e+03, 1.027697872040e+03, 1.028154052000e+03, 1.028609065300e+03, 1.029062918490e+03, 1.029515618090e+03, 1.029967170520e+03, 1.030417582180e+03, 1.030866859410e+03, 1.031315008480e+03, 1.031762035630e+03, 1.032207947030e+03, 1.032652748800e+03, 1.033096447020e+03, 1.033539047700e+03, 1.033980556820e+03, 1.034420980300e+03, 1.034860324010e+03, 1.035298593780e+03, 1.035735795370e+03, 1.036171934520e+03, 1.036607016900e+03, 1.037041048160e+03, 1.037474033880e+03, 1.037905979610e+03, 1.038336890830e+03, 1.038766773010e+03, 1.039195631560e+03, 1.039623471840e+03, 1.040050299170e+03, 1.040476118830e+03, 1.040900936070e+03, 1.041324756080e+03, 1.041747584010e+03, 1.042169424990e+03, 1.042590284070e+03, 1.043010166310e+03, 1.043429076690e+03, 1.043847020170e+03, 1.044264001670e+03, 1.044680026060e+03, 1.045095098190e+03, 1.045509222860e+03, 1.045922404840e+03, 1.046334648860e+03, 1.046745959610e+03, 1.047156341740e+03, 1.047565799890e+03, 1.047974338630e+03, 1.048381962530e+03, 1.048788676090e+03, 1.049194483800e+03, 1.049599390110e+03, 1.050003399440e+03, 1.050406516170e+03, 1.050808744650e+03, 1.051210089200e+03, 1.051610554100e+03, 1.052010143610e+03, 1.052408861950e+03, 1.052806713310e+03, 1.053203701860e+03, 1.053599831710e+03, 1.053995106980e+03, 1.054389531740e+03, 1.054783110010e+03, 1.055175845820e+03, 1.055567743140e+03, 1.055958805930e+03, 1.056349038110e+03, 1.056738443580e+03, 1.057127026210e+03, 1.057514789830e+03, 1.057901738270e+03, 1.058287875300e+03, 1.058673204680e+03, 1.059057730150e+03, 1.059441455420e+03, 1.059824384160e+03, 1.060206520020e+03, 1.060587866650e+03, 1.060968427630e+03, 1.061348206540e+03, 1.061727206950e+03, 1.062105432370e+03, 1.062482886310e+03, 1.062859572250e+03, 1.063235493650e+03, 1.063610653940e+03, 1.063985056520e+03, 1.064358704780e+03, 1.064731602080e+03, 1.065103751760e+03, 1.065475157140e+03, 1.065845821500e+03, 1.066215748130e+03, 1.066584940270e+03, 1.066953401140e+03, 1.067321133950e+03, 1.067688141890e+03, 1.068054428120e+03, 1.068419995780e+03, 1.068784847990e+03, 1.069148987850e+03, 1.069512418440e+03, 1.069875142810e+03, 1.070237164020e+03},
    {1.610697803430e+00, 4.903550585220e+00, 8.245410622430e+00, 1.163801441870e+01, 1.508319630540e+01, 1.858289566200e+01, 2.213916479720e+01, 2.575417756460e+01, 2.943023879480e+01, 3.316979463700e+01, 3.697544391630e+01, 4.084995062540e+01, 4.479625768810e+01, 4.881750214880e+01, 5.291703196580e+01, 5.709842461040e+01, 6.136550770480e+01, 6.572238196420e+01, 7.017344674940e+01, 7.472342858360e+01, 7.937741303870e+01, 8.414088046110e+01, 8.901974607990e+01, 9.402040512440e+01, 9.914978367660e+01, 1.044153960970e+02, 1.098254099950e+02, 1.153887198480e+02, 1.211150305610e+02, 1.270149524140e+02, 1.331001090280e+02, 1.393832601910e+02, 1.458784415120e+02, 1.526011229630e+02, 1.595683883710e+02, 1.667991375860e+02, 1.743143124310e+02, 1.821371460750e+02, 1.902934329900e+02, 1.988118121000e+02, 2.077240482220e+02, 2.170652843840e+02, 2.268742174740e+02, 2.371931180500e+02, 2.480675673290e+02, 2.595457162840e+02, 2.716767847060e+02, 2.845084286470e+02, 2.980825606260e+02, 3.124292964310e+02, 3.275590216840e+02, 3.434531232610e+02, 3.600545355610e+02, 3.772597130310e+02, 3.949141314050e+02, 4.128143015280e+02, 4.307198153840e+02, 4.483768123240e+02, 4.655486428020e+02, 4.820467273000e+02, 4.977441245090e+02, 5.125692523410e+02, 5.265003344450e+02, 5.395521347570e+02, 5.517629936440e+02, 5.631846408070e+02, 5.738750580130e+02, 5.838937974500e+02, 5.932990425140e+02, 6.021458526490e+02, 6.104852102510e+02, 6.183636112710e+02, 6.258230171290e+02, 6.329010346150e+02, 6.396312265030e+02, 6.460434843510e+02, 6.521644178350e+02, 6.580177323840e+02, 6.636245794190e+02, 6.690038719160e+02, 6.741725634220e+02, 6.791458917400e+02, 6.839375901780e+02, 6.885600699360e+02, 6.930245773300e+02, 6.973413293640e+02, 7.015196308090e+02, 7.055679755540e+02, 7.094941346050e+02, 7.133052327280e+02, 7.170078154350e+02, 7.206079077110e+02, 7.241110656750e+02, 7.275224221600e+02, 7.308467270400e+02, 7.340883830160e+02, 7.372514774380e+02, 7.403398106860e+02, 7.433569215190e+02, 7.463061097840e+02, 7.491904567850e+02, 7.520128435930e+02, 7.547759675390e+02, 7.574823570860e+02, 7.601343852770e+02, 7.627342818970e+02, 7.652841445060e+02, 7.677859484550e+02, 7.702415559910e+02, 7.726527245490e+02, 7.750211143190e+02, 7.773482951490e+02, 7.796357528650e+02, 7.818848950540e+02, 7.840970563670e+02, 7.862735033910e+02, 7.884154391250e+02, 7.905240070990e+02, 7.926002951760e+02, 7.946453390480e+02, 7.966601254800e+02, 7.986455952980e+02, 8.006026461590e+02, 8.025321351220e+02, 8.044348810260e+02, 8.063116667050e+02, 8.081632410410e+02, 8.099903208820e+02, 8.117935928170e+02, 8.135737148440e+02, 8.153313179170e+02, 8.170670073940e+02, 8.187813643900e+02, 8.204749470440e+02, 8.221482917030e+02, 8.238019140320e+02, 8.254363100590e+02, 8.270519571490e+02, 8.286493149220e+02, 8.302288261210e+02, 8.317909174150e+02, 8.333360001710e+02, 8.348644711700e+02, 8.363767132820e+02, 8.378730961130e+02, 8.393539766010e+02, 8.408196995920e+02, 8.422705983720e+02, 8.437069951840e+02, 8.451292017020e+02, 8.465375194920e+02, 8.479322404420e+02, 8.493136471710e+02, 8.506820134180e+02, 8.520376044070e+02, 8.533806772020e+02, 8.547114810320e+02, 8.560302576110e+02, 8.573372414340e+02, 8.586326600670e+02, 8.599167344110e+02, 8.611896789660e+02, 8.624517020740e+02, 8.637030061530e+02, 8.649437879220e+02, 8.661742386100e+02, 8.673945441630e+02, 8.686048854360e+02, 8.698054383790e+02, 8.709963742120e+02, 8.721778595930e+02, 8.733500567840e+02, 8.745131238010e+02, 8.756672145630e+02, 8.768124790330e+02, 8.779490633550e+02, 8.790771099820e+02, 8.801967577980e+02, 8.813081422420e+02, 8.824113954180e+02, 8.835066462070e+02, 8.845940203700e+02, 8.856736406490e+02, 8.867456268640e+02, 8.878100960070e+02, 8.888671623300e+02, 8.899169374290e+02, 8.909595303310e+02, 8.919950475690e+02, 8.930235932580e+02, 8.940452691730e+02, 8.950601748130e+02, 8.960684074760e+02, 8.970700623180e+02, 8.980652324200e+02, 8.990540088450e+02, 9.000364807010e+02, 9.010127351950e+02, 9.019828576860e+02, 9.029469317390e+02, 9.039050391770e+02, 9.048572601260e+02, 9.058036730650e+02, 9.067443548710e+02, 9.076793808620e+02, 9.086088248400e+02, 9.095327591310e+02, 9.104512546270e+02, 9.113643808210e+02, 9.122722058460e+02, 9.131747965120e+02, 9.140722183360e+02, 9.149645355810e+02, 9.158518112840e+02, 9.167341072900e+02, 9.176114842810e+02, 9.184840018050e+02, 9.193517183080e+02, 9.202146911560e+02, 9.210729766640e+02, 9.219266301250e+02, 9.227757058310e+02, 9.236202570970e+02, 9.244603362890e+02, 9.252959948430e+02, 9.261272832880e+02, 9.269542512670e+02, 9.277769475600e+02, 9.285954201010e+02, 9.294097160030e+02, 9.302198815700e+02, 9.310259623200e+02, 9.318280030040e+02, 9.326260476180e+02, 9.334201394270e+02, 9.342103209770e+02, 9.349966341100e+02, 9.357791199850e+02, 9.365578190880e+02, 9.373327712490e+02, 9.381040156540e+02, 9.388715908650e+02, 9.396355348260e+02, 9.403958848800e+02, 9.411526777820e+02, 9.419059497110e+02, 9.426557362820e+02, 9.434020725580e+02, 9.441449930610e+02, 9.448845317840e+02, 9.456207222010e+02, 9.463535972790e+02, 9.470831894870e+02, 9.478095308080e+02, 9.485326527450e+02, 9.492525863350e+02, 9.499693621580e+02, 9.506830103400e+02, 9.513935605720e+02, 9.521010421090e+02, 9.528054837850e+02, 9.535069140180e+02, 9.542053608190e+02, 9.549008518000e+02, 9.555934141820e+02, 9.562830748010e+02, 9.569698601160e+02, 9.576537962180e+02, 9.583349088350e+02, 9.590132233370e+02, 9.596887647470e+02, 9.603615577460e+02, 9.610316266740e+02, 9.616989955470e+02, 9.623636880520e+02, 9.630257275600e+02, 9.636851371280e+02, 9.643419395070e+02, 9.649961571470e+02, 9.656478122020e+02, 9.662969265330e+02, 9.669435217170e+02, 9.675876190520e+02, 9.682292395570e+02, 9.688684039830e+02, 9.695051328130e+02, 9.701394462690e+02, 9.707713643150e+02, 9.714009066650e+02, 9.720280927820e+02, 9.726529418850e+02, 9.732754729540e+02, 9.738957047330e+02, 9.745136557330e+02, 9.751293442390e+02, 9.757427883080e+02, 9.763540057790e+02, 9.769630142740e+02, 9.775698312000e+02, 9.781744737560e+02, 9.787769589310e+02, 9.793773035150e+02, 9.799755240960e+02, 9.805716370640e+02, 9.811656586170e+02, 9.817576047630e+02, 9.823474913210e+02, 9.829353339250e+02, 9.835211480300e+02, 9.841049489090e+02, 9.846867516620e+02, 9.852665712120e+02, 9.858444223160e+02, 9.864203195600e+02, 9.869942773630e+02, 9.875663099860e+02, 9.881364315260e+02, 9.887046559220e+02, 9.892709969590e+02, 9.898354682670e+02, 9.903980833270e+02, 9.909588554710e+02, 9.915177978820e+02, 9.920749236010e+02, 9.926302455270e+02, 9.931837764180e+02, 9.937355288930e+02, 9.942855154370e+02, 9.948337484000e+02, 9.953802399980e+02, 9.959250023210e+02, 9.964680473260e+02, 9.970093868470e+02, 9.975490325920e+02, 9.980869961440e+02, 9.986232889690e+02, 9.991579224100e+02, 9.996909076930e+02, 1.000222255930e+03, 1.000751978110e+03, 1.001280085130e+03, 1.001806587740e+03, 1.002331496620e+03, 1.002854822310e+03, 1.003376575270e+03, 1.003896765830e+03, 1.004415404230e+03, 1.004932500600e+03, 1.005448064970e+03, 1.005962107280e+03, 1.006474637370e+03, 1.006985664970e+03, 1.007495199710e+03, 1.008003251160e+03, 1.008509828750e+03, 1.009014941860e+03, 1.009518599750e+03, 1.010020811610e+03, 1.010521586510e+03, 1.011020933480e+03, 1.011518861420e+03, 1.012015379170e+03, 1.012510495490e+03, 1.013004219020e+03, 1.013496558360e+03, 1.013987522020e+03, 1.014477118410e+03, 1.014965355870e+03, 1.015452242680e+03, 1.015937787030e+03, 1.016421997020e+03, 1.016904880690e+03, 1.017386446020e+03, 1.017866700880e+03, 1.018345653110e+03, 1.018823310440e+03, 1.019299680560e+03, 1.019774771070e+03, 1.020248589510e+03, 1.020721143350e+03, 1.021192440000e+03, 1.021662486790e+03, 1.022131290990e+03, 1.022598859820e+03, 1.023065200410e+03, 1.023530319840e+03, 1.023994225140e+03, 1.024456923260e+03, 1.024918421090e+03, 1.025378725470e+03, 1.025837843170e+03, 1.026295780910e+03, 1.026752545350e+03, 1.027208143100e+03, 1.027662580680e+03, 1.028115864600e+03, 1.028568001280e+03, 1.029018997100e+03, 1.029468858390e+03, 1.029917591420e+03, 1.030365202390e+03, 1.030811697480e+03, 1.031257082800e+03, 1.031701364410e+03, 1.032144548330e+03, 1.032586640500e+03, 1.033027646850e+03, 1.033467573230e+03, 1.033906425460e+03, 1.034344209310e+03, 1.034780930480e+03, 1.035216594670e+03, 1.035651207490e+03, 1.036084774510e+03, 1.036517301290e+03, 1.036948793300e+03, 1.037379255980e+03, 1.037808694760e+03, 1.038237114970e+03, 1.038664521930e+03, 1.039090920930e+03, 1.039516317180e+03, 1.039940715890e+03, 1.040364122190e+03, 1.040786541190e+03, 1.041207977970e+03, 1.041628437550e+03, 1.042047924920e+03, 1.042466445020e+03, 1.042884002780e+03, 1.043300603050e+03, 1.043716250680e+03, 1.044130950470e+03, 1.044544707170e+03, 1.044957525510e+03, 1.045369410180e+03, 1.045780365830e+03, 1.046190397070e+03, 1.046599508490e+03, 1.047007704640e+03, 1.047414990020e+03, 1.047821369120e+03, 1.048226846370e+03, 1.048631426200e+03, 1.049035112980e+03, 1.049437911060e+03, 1.049839824740e+03, 1.050240858320e+03, 1.050641016030e+03, 1.051040302110e+03, 1.051438720730e+03, 1.051836276050e+03, 1.052232972200e+03, 1.052628813280e+03, 1.053023803340e+03, 1.053417946430e+03, 1.053811246560e+03, 1.054203707700e+03, 1.054595333800e+03, 1.054986128780e+03, 1.055376096540e+03, 1.055765240930e+03, 1.056153565810e+03, 1.056541074970e+03, 1.056927772200e+03, 1.057313661260e+03, 1.057698745870e+03, 1.058083029740e+03, 1.058466516550e+03, 1.058849209950e+03, 1.059231113550e+03, 1.059612230970e+03, 1.059992565780e+03, 1.060372121530e+03, 1.060750901740e+03, 1.061128909920e+03, 1.061506149540e+03, 1.061882624050e+03, 1.062258336890e+03, 1.062633291460e+03, 1.063007491140e+03, 1.063380939300e+03, 1.063753639270e+03, 1.064125594360e+03, 1.064496807860e+03, 1.064867283050e+03, 1.065237023170e+03, 1.065606031450e+03, 1.065974311080e+03, 1.066341865260e+03, 1.066708697150e+03, 1.067074809870e+03, 1.067440206560e+03, 1.067804890310e+03, 1.068168864200e+03},
    {1.605670183060e+00, 4.887868733200e+00, 8.218384993870e+00, 1.159890666650e+01, 1.503121504840e+01, 1.851719180780e+01, 2.205882636580e+01, 2.565822394940e+01, 2.931761438930e+01, 3.303936174590e+01, 3.682597485460e+01, 4.068011889900e+01, 4.460462812780e+01, 4.860251985390e+01, 5.267700988750e+01, 5.683152957910e+01, 6.106974467170e+01, 6.539557618970e+01, 6.981322362510e+01, 7.432719071830e+01, 7.894231417360e+01, 8.366379570240e+01, 8.849723783910e+01, 9.344868404510e+01, 9.852466368850e+01, 1.037322425720e+02, 1.090790797760e+02, 1.145734916910e+02, 1.202245242210e+02, 1.260420342600e+02, 1.320367816650e+02, 1.382205330270e+02, 1.446061786460e+02, 1.512078640580e+02, 1.580411373910e+02, 1.651231134350e+02, 1.724726546380e+02, 1.801105679840e+02, 1.880598144800e+02, 1.963457244110e+02, 2.049962055480e+02, 2.140419221980e+02, 2.235164082790e+02, 2.334560553830e+02, 2.438998839810e+02, 2.548889602150e+02, 2.664652627090e+02, 2.786697426700e+02, 2.915392827870e+02, 3.051022975830e+02, 3.193728963290e+02, 3.343438825400e+02, 3.499793203550e+02, 3.662077960220e+02, 3.829177761660e+02, 3.999567803010e+02, 4.171364712270e+02, 4.342454415870e+02, 4.510692557710e+02, 4.674136564700e+02, 4.831264127030e+02, 4.981045347630e+02, 5.122903713660e+02, 5.256663820700e+02, 5.382447011030e+02, 5.500570195640e+02, 5.611465297060e+02, 5.715621764580e+02, 5.813548053220e+02, 5.905746762360e+02, 5.992699144260e+02, 6.074856028170e+02, 6.152633191400e+02, 6.226409808440e+02, 6.296528969900e+02, 6.363299509440e+02, 6.426998570710e+02, 6.487874508520e+02, 6.546149849840e+02, 6.602024143400e+02, 6.655676600990e+02, 6.707268485820e+02, 6.756945236190e+02, 6.804838333510e+02, 6.851066933940e+02, 6.895739288390e+02, 6.938953976190e+02, 6.980800977250e+02, 7.021362604690e+02, 7.060714318130e+02, 7.098925434480e+02, 7.136059751120e+02, 7.172176093750e+02, 7.207328799630e+02, 7.241568144920e+02, 7.274940723780e+02, 7.307489785400e+02, 7.339255534660e+02, 7.370275400590e+02, 7.400584277030e+02, 7.430214738420e+02, 7.459197233960e+02, 7.487560262520e+02, 7.515330530410e+02, 7.542533094110e+02, 7.569191489440e+02, 7.595327848720e+02, 7.620963007200e+02, 7.646116599830e+02, 7.670807149450e+02, 7.695052147220e+02, 7.718868126040e+02, 7.742270727820e+02, 7.765274764940e+02, 7.787894276730e+02, 7.810142581270e+02, 7.832032323010e+02, 7.853575516610e+02, 7.874783587350e+02, 7.895667408380e+02, 7.916237335140e+02, 7.936503237200e+02, 7.956474527670e+02, 7.976160190540e+02, 7.995568805960e+02, 8.014708573780e+02, 8.033587335320e+02, 8.052212593780e+02, 8.070591533080e+02, 8.088731035540e+02, 8.106637698300e+02, 8.124317848720e+02, 8.141777558670e+02, 8.159022658000e+02, 8.176058747090e+02, 8.192891208610e+02, 8.209525218540e+02, 8.225965756560e+02, 8.242217615740e+02, 8.258285411680e+02, 8.274173591080e+02, 8.289886439870e+02, 8.305428090730e+02, 8.320802530340e+02, 8.336013606110e+02, 8.351065032540e+02, 8.365960397240e+02, 8.380703166650e+02, 8.395296691370e+02, 8.409744211270e+02, 8.424048860280e+02, 8.438213670960e+02, 8.452241578810e+02, 8.466135426340e+02, 8.479897966970e+02, 8.493531868710e+02, 8.507039717640e+02, 8.520424021210e+02, 8.533687211430e+02, 8.546831647860e+02, 8.559859620440e+02, 8.572773352200e+02, 8.585575001890e+02, 8.598266666380e+02, 8.610850383060e+02, 8.623328132010e+02, 8.635701838210e+02, 8.647973373510e+02, 8.660144558610e+02, 8.672217164890e+02, 8.684192916200e+02, 8.696073490530e+02, 8.707860521670e+02, 8.719555600710e+02, 8.731160277540e+02, 8.742676062300e+02, 8.754104426690e+02, 8.765446805270e+02, 8.776704596750e+02, 8.787879165160e+02, 8.798971840970e+02, 8.809983922250e+02, 8.820916675630e+02, 8.831771337430e+02, 8.842549114530e+02, 8.853251185340e+02, 8.863878700710e+02, 8.874432784780e+02, 8.884914535790e+02, 8.895325026910e+02, 8.905665306970e+02, 8.915936401220e+02, 8.926139312030e+02, 8.936275019580e+02, 8.946344482510e+02, 8.956348638550e+02, 8.966288405140e+02, 8.976164680010e+02, 8.985978341740e+02, 8.995730250330e+02, 9.005421247710e+02, 9.015052158230e+02, 9.024623789200e+02, 9.034136931320e+02, 9.043592359140e+02, 9.052990831530e+02, 9.062333092090e+02, 9.071619869560e+02, 9.080851878190e+02, 9.090029818210e+02, 9.099154376100e+02, 9.108226225010e+02, 9.117246025100e+02, 9.126214423860e+02, 9.135132056470e+02, 9.143999546060e+02, 9.152817504070e+02, 9.161586530510e+02, 9.170307214270e+02, 9.178980133380e+02, 9.187605855280e+02, 9.196184937110e+02, 9.204717925910e+02, 9.213205358890e+02, 9.221647763710e+02, 9.230045658620e+02, 9.238399552760e+02, 9.246709946350e+02, 9.254977330900e+02, 9.263202189400e+02, 9.271384996540e+02, 9.279526218900e+02, 9.287626315120e+02, 9.295685736090e+02, 9.303704925150e+02, 9.311684318220e+02, 9.319624343970e+02, 9.327525424030e+02, 9.335387973090e+02, 9.343212399090e+02, 9.350999103340e+02, 9.358748480700e+02, 9.366460919670e+02, 9.374136802580e+02, 9.381776505690e+02, 9.389380399330e+02, 9.396948848020e+02, 9.404482210610e+02, 9.411980840360e+02, 9.419445085100e+02, 9.426875287330e+02, 9.434271784320e+02, 9.441634908210e+02, 9.448964986140e+02, 9.456262340330e+02, 9.463527288200e+02, 9.470760142450e+02, 9.477961211160e+02, 9.485130797870e+02, 9.492269201690e+02, 9.499376717380e+02, 9.506453635430e+02, 9.513500242150e+02, 9.520516819740e+02, 9.527503646380e+02, 9.534460996320e+02, 9.541389139910e+02, 9.548288343720e+02, 9.555158870600e+02, 9.562000979730e+02, 9.568814926720e+02, 9.575600963640e+02, 9.582359339120e+02, 9.589090298400e+02, 9.595794083390e+02, 9.602470932710e+02, 9.609121081820e+02, 9.615744762980e+02, 9.622342205380e+02, 9.628913635170e+02, 9.635459275520e+02, 9.641979346670e+02, 9.648474065970e+02, 9.654943647940e+02, 9.661388304340e+02, 9.667808244190e+02, 9.674203673820e+02, 9.680574796930e+02, 9.686921814630e+02, 9.693244925490e+02, 9.699544325560e+02, 9.705820208440e+02, 9.712072765300e+02, 9.718302184960e+02, 9.724508653870e+02, 9.730692356200e+02, 9.736853473840e+02, 9.742992186470e+02, 9.749108671590e+02, 9.755203104530e+02, 9.761275658500e+02, 9.767326504660e+02, 9.773355812090e+02, 9.779363747870e+02, 9.785350477100e+02, 9.791316162900e+02, 9.797260966510e+02, 9.803185047250e+02, 9.809088562600e+02, 9.814971668200e+02, 9.820834517890e+02, 9.826677263720e+02, 9.832500056030e+02, 9.838303043410e+02, 9.844086372760e+02, 9.849850189320e+02, 9.855594636700e+02, 9.861319856860e+02, 9.867025990210e+02, 9.872713175560e+02, 9.878381550180e+02, 9.884031249840e+02, 9.889662408780e+02, 9.895275159790e+02, 9.900869634200e+02, 9.906445961890e+02, 9.912004271340e+02, 9.917544689650e+02, 9.923067342540e+02, 9.928572354360e+02, 9.934059848150e+02, 9.939529945650e+02, 9.944982767270e+02, 9.950418432170e+02, 9.955837058250e+02, 9.961238762160e+02, 9.966623659350e+02, 9.971991864040e+02, 9.977343489290e+02, 9.982678646950e+02, 9.987997447760e+02, 9.993300001290e+02, 9.998586416010e+02, 1.000385679930e+03, 1.000911125730e+03, 1.001434989530e+03, 1.001957281740e+03, 1.002478012670e+03, 1.002997192520e+03, 1.003514831390e+03, 1.004030939290e+03, 1.004545526110e+03, 1.005058601660e+03, 1.005570175650e+03, 1.006080257690e+03, 1.006588857290e+03, 1.007095983890e+03, 1.007601646810e+03, 1.008105855300e+03, 1.008608618520e+03, 1.009109945520e+03, 1.009609845300e+03, 1.010108326730e+03, 1.010605398630e+03, 1.011101069720e+03, 1.011595348650e+03, 1.012088243970e+03, 1.012579764180e+03, 1.013069917650e+03, 1.013558712730e+03, 1.014046157650e+03, 1.014532260580e+03, 1.015017029610e+03, 1.015500472770e+03, 1.015982597990e+03, 1.016463413150e+03, 1.016942926040e+03, 1.017421144400e+03, 1.017898075890e+03, 1.018373728090e+03, 1.018848108520e+03, 1.019321224640e+03, 1.019793083840e+03, 1.020263693430e+03, 1.020733060670e+03, 1.021201192740e+03, 1.021668096790e+03, 1.022133779870e+03, 1.022598248980e+03, 1.023061511070e+03, 1.023523573010e+03, 1.023984441620e+03, 1.024444123680e+03, 1.024902625870e+03, 1.025359954840e+03, 1.025816117180e+03, 1.026271119430e+03, 1.026724968050e+03, 1.027177669460e+03, 1.027629230040e+03, 1.028079656080e+03, 1.028528953860e+03, 1.028977129570e+03, 1.029424189360e+03, 1.029870139350e+03, 1.030314985560e+03, 1.030758734020e+03, 1.031201390660e+03, 1.031642961400e+03, 1.032083452070e+03, 1.032522868480e+03, 1.032961216390e+03, 1.033398501510e+03, 1.033834729490e+03, 1.034269905970e+03, 1.034704036490e+03, 1.035137126600e+03, 1.035569181770e+03, 1.036000207440e+03, 1.036430209000e+03, 1.036859191800e+03, 1.037287161140e+03, 1.037714122300e+03, 1.038140080490e+03, 1.038565040900e+03, 1.038989008670e+03, 1.039411988890e+03, 1.039833986630e+03, 1.040255006900e+03, 1.040675054700e+03, 1.041094134950e+03, 1.041512252570e+03, 1.041929412410e+03, 1.042345619320e+03, 1.042760878080e+03, 1.043175193430e+03, 1.043588570110e+03, 1.044001012800e+03, 1.044412526130e+03, 1.044823114720e+03, 1.045232783150e+03, 1.045641535960e+03, 1.046049377640e+03, 1.046456312690e+03, 1.046862345530e+03, 1.047267480570e+03, 1.047671722190e+03, 1.048075074720e+03, 1.048477542480e+03, 1.048879129740e+03, 1.049279840740e+03, 1.049679679700e+03, 1.050078650800e+03, 1.050476758200e+03, 1.050874006010e+03, 1.051270398320e+03, 1.051665939210e+03, 1.052060632690e+03, 1.052454482780e+03, 1.052847493450e+03, 1.053239668640e+03, 1.053631012260e+03, 1.054021528220e+03, 1.054411220360e+03, 1.054800092530e+03, 1.055188148530e+03, 1.055575392140e+03, 1.055961827110e+03, 1.056347457170e+03, 1.056732286020e+03, 1.057116317320e+03, 1.057499554740e+03, 1.057882001890e+03, 1.058263662370e+03, 1.058644539750e+03, 1.059024637570e+03, 1.059403959370e+03, 1.059782508640e+03, 1.060160288860e+03, 1.060537303470e+03, 1.060913555900e+03, 1.061289049560e+03, 1.061663787820e+03, 1.062037774040e+03, 1.062411011560e+03, 1.062783503690e+03, 1.063155253720e+03, 1.063526264910e+03, 1.063896540500e+03, 1.064266083740e+03, 1.064634897800e+03, 1.065002985880e+03, 1.065370351140e+03, 1.065736996700e+03, 1.066102925700e+03},
    {1.600674236470e+00, 4.872290557250e+00, 8.191546758590e+00, 1.156008317850e+01, 1.497962970450e+01, 1.845201213090e+01, 2.197915907140e+01, 2.556310948630e+01, 2.920602088800e+01, 3.291017829940e+01, 3.667800404670e+01, 4.051206847900e+01, 4.441510172000e+01, 4.839000656920e+01, 5.243987268560e+01, 5.656799220650e+01, 6.077787696970e+01, 6.507327753640e+01, 6.945820423330e+01, 7.393695046520e+01, 7.851411858350e+01, 8.319464863470e+01, 8.798385035700e+01, 9.288743884480e+01, 9.791157435520e+01, 1.030629067950e+02, 1.083486254920e+02, 1.137765149300e+02, 1.193550172030e+02, 1.250933020070e+02, 1.310013450740e+02, 1.370900159760e+02, 1.433711762270e+02, 1.498577885880e+02, 1.565640382350e+02, 1.635054661410e+02, 1.706991143230e+02, 1.781636814910e+02, 1.859196857950e+02, 1.939896283950e+02, 2.023981470010e+02, 2.111721413970e+02, 2.203408422250e+02, 2.299357784100e+02, 2.399905757430e+02, 2.505404879890e+02, 2.616215227070e+02, 2.732689820370e+02, 2.855152092090e+02, 2.983863445760e+02, 3.118979936290e+02, 3.260499278760e+02, 3.408202601450e+02, 3.561598651140e+02, 3.719880470050e+02, 3.881905938450e+02, 4.046214928100e+02, 4.211095817870e+02, 4.374707148190e+02, 4.535242038560e+02, 4.691101535580e+02, 4.841048073830e+02, 4.984241339230e+02, 5.120214151680e+02, 5.248824238120e+02, 5.370172006210e+02, 5.484521055720e+02, 5.592234042120e+02, 5.693726069450e+02, 5.789432822890e+02, 5.879789510540e+02, 5.965217300080e+02, 6.046114926540e+02, 6.122853932750e+02, 6.195776494760e+02, 6.265195067180e+02, 6.331393260480e+02, 6.394627494040e+02, 6.455129079170e+02, 6.513106482670e+02, 6.568747600620e+02, 6.622221934770e+02, 6.673682610460e+02, 6.723268207580e+02, 6.771104397340e+02, 6.817305391180e+02, 6.861975215310e+02, 6.905208827840e+02, 6.947093096410e+02, 6.987707653790e+02, 7.027125647300e+02, 7.065414396580e+02, 7.102635972080e+02, 7.138847705240e+02, 7.174102639580e+02, 7.208449930670e+02, 7.241935201720e+02, 7.274600860500e+02, 7.306486382570e+02, 7.337628564850e+02, 7.368061753280e+02, 7.397818047570e+02, 7.426927485690e+02, 7.455418210390e+02, 7.483316619900e+02, 7.510647504300e+02, 7.537434169380e+02, 7.563698549070e+02, 7.589461307810e+02, 7.614741933850e+02, 7.639558824300e+02, 7.663929362960e+02, 7.687869991430e+02, 7.711396274280e+02, 7.734522958870e+02, 7.757264030190e+02, 7.779632761380e+02, 7.801641760170e+02, 7.823303011680e+02, 7.844627917930e+02, 7.865627334330e+02, 7.886311603360e+02, 7.906690585860e+02, 7.926773689870e+02, 7.946569897500e+02, 7.966087789820e+02, 7.985335570010e+02, 8.004321084830e+02, 8.023051844720e+02, 8.041535042420e+02, 8.059777570430e+02, 8.077786037210e+02, 8.095566782410e+02, 8.113125891040e+02, 8.130469206780e+02, 8.147602344400e+02, 8.164530701440e+02, 8.181259469170e+02, 8.197793642800e+02, 8.214138031190e+02, 8.230297265880e+02, 8.246275809630e+02, 8.262077964470e+02, 8.277707879220e+02, 8.293169556670e+02, 8.308466860280e+02, 8.323603520510e+02, 8.338583140850e+02, 8.353409203450e+02, 8.368085074520e+02, 8.382614009340e+02, 8.396999157090e+02, 8.411243565370e+02, 8.425350184550e+02, 8.439321871780e+02, 8.453161394920e+02, 8.466871436190e+02, 8.480454595650e+02, 8.493913394540e+02, 8.507250278400e+02, 8.520467620090e+02, 8.533567722630e+02, 8.546552821900e+02, 8.559425089260e+02, 8.572186633980e+02, 8.584839505580e+02, 8.597385696110e+02, 8.609827142230e+02, 8.622165727290e+02, 8.634403283250e+02, 8.646541592540e+02, 8.658582389850e+02, 8.670527363810e+02, 8.682378158610e+02, 8.694136375560e+02, 8.705803574570e+02, 8.717381275600e+02, 8.728870959950e+02, 8.740274071650e+02, 8.751592018630e+02, 8.762826173990e+02, 8.773977877090e+02, 8.785048434690e+02, 8.796039121990e+02, 8.806951183640e+02, 8.817785834730e+02, 8.828544261710e+02, 8.839227623330e+02, 8.849837051420e+02, 8.860373651810e+02, 8.870838505090e+02, 8.881232667360e+02, 8.891557171010e+02, 8.901813025410e+02, 8.912001217600e+02, 8.922122712930e+02, 8.932178455730e+02, 8.942169369910e+02, 8.952096359530e+02, 8.961960309410e+02, 8.971762085630e+02, 8.981502536110e+02, 8.991182491080e+02, 9.000802763600e+02, 9.010364150000e+02, 9.019867430400e+02, 9.029313369090e+02, 9.038702715020e+02, 9.048036202130e+02, 9.057314549860e+02, 9.066538463440e+02, 9.075708634310e+02, 9.084825740490e+02, 9.093890446900e+02, 9.102903405720e+02, 9.111865256700e+02, 9.120776627510e+02, 9.129638134010e+02, 9.138450380560e+02, 9.147213960330e+02, 9.155929455540e+02, 9.164597437760e+02, 9.173218468170e+02, 9.181793097780e+02, 9.190321867730e+02, 9.198805309490e+02, 9.207243945110e+02, 9.215638287430e+02, 9.223988840300e+02, 9.232296098810e+02, 9.240560549470e+02, 9.248782670450e+02, 9.256962931700e+02, 9.265101795210e+02, 9.273199715170e+02, 9.281257138120e+02, 9.289274503180e+02, 9.297252242140e+02, 9.305190779680e+02, 9.313090533540e+02, 9.320951914610e+02, 9.328775327130e+02, 9.336561168820e+02, 9.344309831020e+02, 9.352021698840e+02, 9.359697151280e+02, 9.367336561350e+02, 9.374940296240e+02, 9.382508717370e+02, 9.390042180610e+02, 9.397541036290e+02, 9.405005629400e+02, 9.412436299650e+02, 9.419833381600e+02, 9.427197204750e+02, 9.434528093660e+02, 9.441826368060e+02, 9.449092342890e+02, 9.456326328460e+02, 9.463528630520e+02, 9.470699550340e+02, 9.477839384790e+02, 9.484948426480e+02, 9.492026963770e+02, 9.499075280900e+02, 9.506093658060e+02, 9.513082371450e+02, 9.520041693390e+02, 9.526971892360e+02, 9.533873233090e+02, 9.540745976640e+02, 9.547590380430e+02, 9.554406698350e+02, 9.561195180820e+02, 9.567956074820e+02, 9.574689624000e+02, 9.581396068700e+02, 9.588075646020e+02, 9.594728589920e+02, 9.601355131220e+02, 9.607955497670e+02, 9.614529914030e+02, 9.621078602100e+02, 9.627601780800e+02, 9.634099666160e+02, 9.640572471450e+02, 9.647020407160e+02, 9.653443681100e+02, 9.659842498400e+02, 9.666217061610e+02, 9.672567570680e+02, 9.678894223070e+02, 9.685197213740e+02, 9.691476735210e+02, 9.697732977620e+02, 9.703966128740e+02, 9.710176374030e+02, 9.716363896680e+02, 9.722528877620e+02, 9.728671495590e+02, 9.734791927160e+02, 9.740890346750e+02, 9.746966926720e+02, 9.753021837330e+02, 9.759055246830e+02, 9.765067321460e+02, 9.771058225510e+02, 9.777028121320e+02, 9.782977169330e+02, 9.788905528110e+02, 9.794813354380e+02, 9.800700803060e+02, 9.806568027260e+02, 9.812415178330e+02, 9.818242405920e+02, 9.824049857950e+02, 9.829837680650e+02, 9.835606018620e+02, 9.841355014810e+02, 9.847084810600e+02, 9.852795545750e+02, 9.858487358480e+02, 9.864160385480e+02, 9.869814761950e+02, 9.875450621550e+02, 9.881068096530e+02, 9.886667317660e+02, 9.892248414310e+02, 9.897811514440e+02, 9.903356744610e+02, 9.908884230050e+02, 9.914394094640e+02, 9.919886460930e+02, 9.925361450150e+02, 9.930819182300e+02, 9.936259776050e+02, 9.941683348860e+02, 9.947090016950e+02, 9.952479895330e+02, 9.957853097810e+02, 9.963209737020e+02, 9.968549924440e+02, 9.973873770370e+02, 9.979181384010e+02, 9.984472873450e+02, 9.989748345640e+02, 9.995007906490e+02, 1.000025166080e+03, 1.000547971240e+03, 1.001069216390e+03, 1.001588911710e+03, 1.002107067260e+03, 1.002623693020e+03, 1.003138798840e+03, 1.003652394520e+03, 1.004164489710e+03, 1.004675094000e+03, 1.005184216890e+03, 1.005691867760e+03, 1.006198055920e+03, 1.006702790590e+03, 1.007206080890e+03, 1.007707935860e+03, 1.008208364450e+03, 1.008707375520e+03, 1.009204977870e+03, 1.009701180190e+03, 1.010195991100e+03, 1.010689419130e+03, 1.011181472730e+03, 1.011672160300e+03, 1.012161490110e+03, 1.012649470410e+03, 1.013136109320e+03, 1.013621414930e+03, 1.014105395230e+03, 1.014588058140e+03, 1.015069411520e+03, 1.015549463140e+03, 1.016028220700e+03, 1.016505691860e+03, 1.016981884180e+03, 1.017456805150e+03, 1.017930462230e+03, 1.018402862760e+03, 1.018874014070e+03, 1.019343923380e+03, 1.019812597860e+03, 1.020280044630e+03, 1.020746270740e+03, 1.021211283170e+03, 1.021675088840e+03, 1.022137694620e+03, 1.022599107320e+03, 1.023059333680e+03, 1.023518380380e+03, 1.023976254060e+03, 1.024432961300e+03, 1.024888508590e+03, 1.025342902420e+03, 1.025796149170e+03, 1.026248255210e+03, 1.026699226840e+03, 1.027149070290e+03, 1.027597791760e+03, 1.028045397380e+03, 1.028491893260e+03, 1.028937285410e+03, 1.029381579850e+03, 1.029824782490e+03, 1.030266899230e+03, 1.030707935920e+03, 1.031147898330e+03, 1.031586792230e+03, 1.032024623300e+03, 1.032461397200e+03, 1.032897119530e+03, 1.033331795850e+03, 1.033765431690e+03, 1.034198032500e+03, 1.034629603720e+03, 1.035060150730e+03, 1.035489678860e+03, 1.035918193430e+03, 1.036345699680e+03, 1.036772202820e+03, 1.037197708040e+03, 1.037622220460e+03, 1.038045745170e+03, 1.038468287230e+03, 1.038889851650e+03, 1.039310443400e+03, 1.039730067410e+03, 1.040148728590e+03, 1.040566431780e+03, 1.040983181820e+03, 1.041398983480e+03, 1.041813841520e+03, 1.042227760630e+03, 1.042640745510e+03, 1.043052800780e+03, 1.043463931060e+03, 1.043874140910e+03, 1.044283434860e+03, 1.044691817420e+03, 1.045099293050e+03, 1.045505866180e+03, 1.045911541230e+03, 1.046316322550e+03, 1.046720214470e+03, 1.047123221310e+03, 1.047525347330e+03, 1.047926596770e+03, 1.048326973840e+03, 1.048726482730e+03, 1.049125127560e+03, 1.049522912470e+03, 1.049919841530e+03, 1.050315918810e+03, 1.050711148330e+03, 1.051105534090e+03, 1.051499080060e+03, 1.051891790180e+03, 1.052283668360e+03, 1.052674718500e+03, 1.053064944430e+03, 1.053454350010e+03, 1.053842939020e+03, 1.054230715240e+03, 1.054617682420e+03, 1.055003844290e+03, 1.055389204530e+03, 1.055773766830e+03, 1.056157534820e+03, 1.056540512120e+03, 1.056922702330e+03, 1.057304109020e+03, 1.057684735720e+03, 1.058064585970e+03, 1.058443663240e+03, 1.058821971030e+03, 1.059199512770e+03, 1.059576291880e+03, 1.059952311770e+03, 1.060327575820e+03, 1.060702087380e+03, 1.061075849770e+03, 1.061448866320e+03, 1.061821140300e+03, 1.062192674980e+03, 1.062563473600e+03, 1.062933539390e+03, 1.063302875540e+03, 1.063671485230e+03, 1.064039371620e+03},
    {1.595709660090e+00, 4.856814982420e+00, 8.164893825760e+00, 1.152154055640e+01, 1.492843522130e+01, 1.838734951160e+01, 2.190015323630e+01, 2.546882135620e+01, 2.909544163650e+01, 3.278222298380e+01, 3.653150453800e+01, 4.034576560250e+01, 4.422763650220e+01, 4.817991047360e+01, 5.220555670040e+01, 5.630773462620e+01, 6.048980969060e+01, 6.475537065320e+01, 6.910824869270e+01, 7.355253849150e+01, 7.809262154430e+01, 8.273319195710e+01, 8.747928504160e+01, 9.233630904330e+01, 9.731008038570e+01, 1.024068628580e+02, 1.076334112220e+02, 1.129970197550e+02, 1.185055763130e+02, 1.241676225110e+02, 1.299924206720e+02, 1.359900281920e+02, 1.421713799200e+02, 1.485483790850e+02, 1.551339970650e+02, 1.619423819350e+02, 1.689889751430e+02, 1.762906346410e+02, 1.838657612650e+02, 1.917344227740e+02, 1.999184663250e+02, 2.084416047330e+02, 2.173294539060e+02, 2.266094873300e+02, 2.363108573260e+02, 2.464640112890e+02, 2.571000042330e+02, 2.682493801700e+02, 2.799404727070e+02, 2.921969778560e+02, 3.050347064450e+02, 3.184575567270e+02, 3.324529633320e+02, 3.469873319510e+02, 3.620021776010e+02, 3.774117930290e+02, 3.931033071310e+02, 4.089399749210e+02, 4.247683138620e+02, 4.404289538910e+02, 4.557697696200e+02, 4.706587287220e+02, 4.849945329940e+02, 4.987083145240e+02, 5.117619857490e+02, 5.241439390340e+02, 5.358625404450e+02, 5.469398464390e+02, 5.574064669120e+02, 5.672977648220e+02, 5.766512068940e+02, 5.855045757530e+02, 5.938947871920e+02, 6.018571278690e+02, 6.094247907250e+02, 6.166286259910e+02, 6.234970490240e+02, 6.300560597860e+02, 6.363293380100e+02, 6.423383855770e+02, 6.481026943900e+02, 6.536399238720e+02, 6.589660772350e+02, 6.640956695900e+02, 6.690418839290e+02, 6.738167131330e+02, 6.784310875520e+02, 6.828949886100e+02, 6.872175493700e+02, 6.914071432690e+02, 6.954714622890e+02, 6.994175858070e+02, 7.032520412950e+02, 7.069808579040e+02, 7.106096138780e+02, 7.141434785880e+02, 7.175872498990e+02, 7.209453874670e+02, 7.242220424820e+02, 7.274210843020e+02, 7.305461243520e+02, 7.336005376210e+02, 7.365874820360e+02, 7.395099159530e+02, 7.423706139840e+02, 7.451721813310e+02, 7.479170668060e+02, 7.506075746540e+02, 7.532458753290e+02, 7.558340153070e+02, 7.583739260510e+02, 7.608674322020e+02, 7.633162590780e+02, 7.657220395440e+02, 7.680863203200e+02, 7.704105677670e+02, 7.726961732170e+02, 7.749444578750e+02, 7.771566773380e+02, 7.793340257640e+02, 7.814776397290e+02, 7.835886017810e+02, 7.856679437440e+02, 7.877166497700e+02, 7.897356591690e+02, 7.917258690430e+02, 7.936881367270e+02, 7.956232820640e+02, 7.975320895200e+02, 7.994153101540e+02, 8.012736634590e+02, 8.031078390760e+02, 8.049184983990e+02, 8.067062760700e+02, 8.084717813870e+02, 8.102155996100e+02, 8.119382931990e+02, 8.136404029610e+02, 8.153224491390e+02, 8.169849324230e+02, 8.186283349120e+02, 8.202531210090e+02, 8.218597382700e+02, 8.234486181970e+02, 8.250201769950e+02, 8.265748162740e+02, 8.281129237210e+02, 8.296348737290e+02, 8.311410279950e+02, 8.326317360830e+02, 8.341073359550e+02, 8.355681544770e+02, 8.370145078960e+02, 8.384467022930e+02, 8.398650340110e+02, 8.412697900620e+02, 8.426612485160e+02, 8.440396788590e+02, 8.454053423520e+02, 8.467584923530e+02, 8.480993746360e+02, 8.494282276890e+02, 8.507452829990e+02, 8.520507653200e+02, 8.533448929360e+02, 8.546278779050e+02, 8.558999262910e+02, 8.571612383920e+02, 8.584120089500e+02, 8.596524273570e+02, 8.608826778450e+02, 8.621029396810e+02, 8.633133873330e+02, 8.645141906490e+02, 8.657055150150e+02, 8.668875215110e+02, 8.680603670600e+02, 8.692242045700e+02, 8.703791830710e+02, 8.715254478450e+02, 8.726631405510e+02, 8.737923993470e+02, 8.749133590000e+02, 8.760261510040e+02, 8.771309036790e+02, 8.782277422780e+02, 8.793167890790e+02, 8.803981634860e+02, 8.814719821140e+02, 8.825383588770e+02, 8.835974050710e+02, 8.846492294550e+02, 8.856939383290e+02, 8.867316356070e+02, 8.877624228870e+02, 8.887863995220e+02, 8.898036626850e+02, 8.908143074360e+02, 8.918184267790e+02, 8.928161117220e+02, 8.938074513370e+02, 8.947925328150e+02, 8.957714415140e+02, 8.967442610180e+02, 8.977110731800e+02, 8.986719581730e+02, 8.996269945350e+02, 9.005762592170e+02, 9.015198276190e+02, 9.024577736400e+02, 9.033901697100e+02, 9.043170868370e+02, 9.052385946380e+02, 9.061547613790e+02, 9.070656540080e+02, 9.079713381910e+02, 9.088718783460e+02, 9.097673376690e+02, 9.106577781730e+02, 9.115432607110e+02, 9.124238450080e+02, 9.132995896920e+02, 9.141705523130e+02, 9.150367893790e+02, 9.158983563730e+02, 9.167553077850e+02, 9.176076971310e+02, 9.184555769780e+02, 9.192989989670e+02, 9.201380138330e+02, 9.209726714310e+02, 9.218030207500e+02, 9.226291099390e+02, 9.234509863220e+02, 9.242686964200e+02, 9.250822859670e+02, 9.258917999310e+02, 9.266972825270e+02, 9.274987772370e+02, 9.282963268280e+02, 9.290899733610e+02, 9.298797582140e+02, 9.306657220930e+02, 9.314479050480e+02, 9.322263464860e+02, 9.330010851860e+02, 9.337721593140e+02, 9.345396064300e+02, 9.353034635090e+02, 9.360637669470e+02, 9.368205525770e+02, 9.375738556790e+02, 9.383237109900e+02, 9.390701527200e+02, 9.398132145560e+02, 9.405529296800e+02, 9.412893307740e+02, 9.420224500310e+02, 9.427523191670e+02, 9.434789694280e+02, 9.442024316020e+02, 9.449227360250e+02, 9.456399125920e+02, 9.463539907660e+02, 9.470649995830e+02, 9.477729676650e+02, 9.484779232260e+02, 9.491798940770e+02, 9.498789076400e+02, 9.505749909480e+02, 9.512681706570e+02, 9.519584730550e+02, 9.526459240620e+02, 9.533305492430e+02, 9.540123738110e+02, 9.546914226380e+02, 9.553677202540e+02, 9.560412908610e+02, 9.567121583330e+02, 9.573803462260e+02, 9.580458777810e+02, 9.587087759310e+02, 9.593690633060e+02, 9.600267622390e+02, 9.606818947710e+02, 9.613344826560e+02, 9.619845473650e+02, 9.626321100940e+02, 9.632771917630e+02, 9.639198130300e+02, 9.645599942840e+02, 9.651977556610e+02, 9.658331170380e+02, 9.664660980470e+02, 9.670967180720e+02, 9.677249962550e+02, 9.683509515020e+02, 9.689746024850e+02, 9.695959676470e+02, 9.702150652060e+02, 9.708319131560e+02, 9.714465292740e+02, 9.720589311220e+02, 9.726691360510e+02, 9.732771612030e+02, 9.738830235180e+02, 9.744867397320e+02, 9.750883263850e+02, 9.756877998200e+02, 9.762851761910e+02, 9.768804714610e+02, 9.774737014070e+02, 9.780648816270e+02, 9.786540275330e+02, 9.792411543650e+02, 9.798262771860e+02, 9.804094108880e+02, 9.809905701920e+02, 9.815697696560e+02, 9.821470236700e+02, 9.827223464660e+02, 9.832957521150e+02, 9.838672545310e+02, 9.844368674750e+02, 9.850046045540e+02, 9.855704792270e+02, 9.861345048050e+02, 9.866966944530e+02, 9.872570611940e+02, 9.878156179080e+02, 9.883723773370e+02, 9.889273520860e+02, 9.894805546240e+02, 9.900319972890e+02, 9.905816922850e+02, 9.911296516880e+02, 9.916758874470e+02, 9.922204113840e+02, 9.927632351980e+02, 9.933043704660e+02, 9.938438286430e+02, 9.943816210670e+02, 9.949177589590e+02, 9.954522534220e+02, 9.959851154480e+02, 9.965163559150e+02, 9.970459855910e+02, 9.975740151340e+02, 9.981004550960e+02, 9.986253159190e+02, 9.991486079440e+02, 9.996703414070e+02, 1.000190526440e+03, 1.000709173080e+03, 1.001226291260e+03, 1.001741890810e+03, 1.002255981480e+03, 1.002768572910e+03, 1.003279674640e+03, 1.003789296140e+03, 1.004297446770e+03, 1.004804135800e+03, 1.005309372420e+03, 1.005813165720e+03, 1.006315524720e+03, 1.006816458340e+03, 1.007315975410e+03, 1.007814084700e+03, 1.008310794880e+03, 1.008806114530e+03, 1.009300052170e+03, 1.009792616220e+03, 1.010283815050e+03, 1.010773656920e+03, 1.011262150040e+03, 1.011749302510e+03, 1.012235122410e+03, 1.012719617680e+03, 1.013202796250e+03, 1.013684665920e+03, 1.014165234480e+03, 1.014644509590e+03, 1.015122498880e+03, 1.015599209900e+03, 1.016074650130e+03, 1.016548826990e+03, 1.017021747820e+03, 1.017493419900e+03, 1.017963850460e+03, 1.018433046660e+03, 1.018901015570e+03, 1.019367764240e+03, 1.019833299640e+03, 1.020297628660e+03, 1.020760758150e+03, 1.021222694910e+03, 1.021683445670e+03, 1.022143017090e+03, 1.022601415790e+03, 1.023058648330e+03, 1.023514721210e+03, 1.023969640870e+03, 1.024423413700e+03, 1.024876046050e+03, 1.025327544200e+03, 1.025777914370e+03, 1.026227162750e+03, 1.026675295460e+03, 1.027122318570e+03, 1.027568238120e+03, 1.028013060080e+03, 1.028456790370e+03, 1.028899434870e+03, 1.029340999410e+03, 1.029781489760e+03, 1.030220911670e+03, 1.030659270820e+03, 1.031096572850e+03, 1.031532823350e+03, 1.031968027880e+03, 1.032402191930e+03, 1.032835320970e+03, 1.033267420420e+03, 1.033698495640e+03, 1.034128551970e+03, 1.034557594700e+03, 1.034985629070e+03, 1.035412660280e+03, 1.035838693510e+03, 1.036263733870e+03, 1.036687786440e+03, 1.037110856280e+03, 1.037532948380e+03, 1.037954067700e+03, 1.038374219180e+03, 1.038793407700e+03, 1.039211638110e+03, 1.039628915230e+03, 1.040045243820e+03, 1.040460628640e+03, 1.040875074380e+03, 1.041288585710e+03, 1.041701167270e+03, 1.042112823650e+03, 1.042523559400e+03, 1.042933379070e+03, 1.043342287150e+03, 1.043750288080e+03, 1.044157386310e+03, 1.044563586220e+03, 1.044968892180e+03, 1.045373308520e+03, 1.045776839520e+03, 1.046179489460e+03, 1.046581262570e+03, 1.046982163060e+03, 1.047382195080e+03, 1.047781362800e+03, 1.048179670300e+03, 1.048577121690e+03, 1.048973721010e+03, 1.049369472270e+03, 1.049764379490e+03, 1.050158446610e+03, 1.050551677580e+03, 1.050944076300e+03, 1.051335646650e+03, 1.051726392490e+03, 1.052116317640e+03, 1.052505425900e+03, 1.052893721030e+03, 1.053281206790e+03, 1.053667886900e+03, 1.054053765030e+03, 1.054438844870e+03, 1.054823130040e+03, 1.055206624170e+03, 1.055589330850e+03, 1.055971253640e+03, 1.056352396070e+03, 1.056732761670e+03, 1.057112353940e+03, 1.057491176320e+03, 1.057869232280e+03, 1.058246525220e+03, 1.058623058560e+03, 1.058998835650e+03, 1.059373859860e+03, 1.059748134510e+03, 1.060121662900e+03, 1.060494448320e+03, 1.060866494040e+03, 1.061237803280e+03, 1.061608379280e+03, 1.061978225220e+03},
    {1.590776154330e+00, 4.841440949440e+00, 8.138424138110e+00, 1.148327546150e+01, 1.487762664130e+01, 1.832319697390e+01, 2.182179939110e+01, 2.537534703290e+01, 2.898586038400e+01, 3.265547503030e+01, 3.638645010130e+01, 4.018117746650e+01, 4.404219176810e+01, 4.797218137810e+01, 5.197400037860e+01, 5.605068167840e+01, 6.020545139000e+01, 6.444174460850e+01, 6.876322274760e+01, 7.317379261200e+01, 7.767762740160e+01, 8.227918986920e+01, 8.698325787970e+01, 9.179495264390e+01, 9.671976993460e+01, 1.017636146200e+02, 1.069328388820e+02, 1.122342845280e+02, 1.176753298040e+02, 1.232639411810e+02, 1.290087305350e+02, 1.349190181800e+02, 1.410049020960e+02, 1.472773336290e+02, 1.537481996890e+02, 1.604304111450e+02, 1.673379965590e+02, 1.744861995460e+02, 1.818915767430e+02, 1.895720914480e+02, 1.975471951140e+02, 2.058378847280e+02, 2.144667181160e+02, 2.234577608110e+02, 2.328364265820e+02, 2.426291585580e+02, 2.528628793780e+02, 2.635641187000e+02, 2.747577102810e+02, 2.864649493000e+02, 2.987011306750e+02, 3.114724701180e+02, 3.247725509960e+02, 3.385786242680e+02, 3.528482651470e+02, 3.675170003640e+02, 3.824975425330e+02, 3.976812240500e+02, 4.129421017660e+02, 4.281438891050e+02, 4.431492352590e+02, 4.578300243300e+02, 4.720768465040e+02, 4.858062760680e+02, 4.989616420590e+02, 5.115117199280e+02, 5.234469940120e+02, 5.347744848790e+02, 5.455127537900e+02, 5.556877660710e+02, 5.653297769020e+02, 5.744711192640e+02, 5.831446819490e+02, 5.913828798240e+02, 5.992169688750e+02, 6.066766066520e+02, 6.137895922600e+02, 6.205817399060e+02, 6.270768510770e+02, 6.332967573150e+02, 6.392614106560e+02, 6.449890034460e+02, 6.504961034260e+02, 6.557977937910e+02, 6.609078110940e+02, 6.658386764470e+02, 6.706018173970e+02, 6.752076792360e+02, 6.796658254710e+02, 6.839850277690e+02, 6.881733460500e+02, 6.922381995930e+02, 6.961864300420e+02, 7.000243572570e+02, 7.037578288190e+02, 7.073922639900e+02, 7.109326928170e+02, 7.143837909730e+02, 7.177499108890e+02, 7.210351096170e+02, 7.242431738350e+02, 7.273776423340e+02, 7.304418262750e+02, 7.334388274870e+02, 7.363715550210e+02, 7.392427401460e+02, 7.420549499700e+02, 7.448105998250e+02, 7.475119645410e+02, 7.501611887330e+02, 7.527602961950e+02, 7.553111984870e+02, 7.578157028030e+02, 7.602755191820e+02, 7.626922671220e+02, 7.650674816700e+02, 7.674026190120e+02, 7.696990616260e+02, 7.719581230380e+02, 7.741810522040e+02, 7.763690375600e+02, 7.785232107770e+02, 7.806446502210e+02, 7.827343841720e+02, 7.847933938100e+02, 7.868226159750e+02, 7.888229457460e+02, 7.907952388340e+02, 7.927403138050e+02, 7.946589541610e+02, 7.965519102740e+02, 7.984199011920e+02, 8.002636163270e+02, 8.020837170330e+02, 8.038808380830e+02, 8.056555890490e+02, 8.074085556010e+02, 8.091403007160e+02, 8.108513658230e+02, 8.125422718690e+02, 8.142135203280e+02, 8.158655941450e+02, 8.174989586260e+02, 8.191140622770e+02, 8.207113375930e+02, 8.222912018020e+02, 8.238540575680e+02, 8.254002936550e+02, 8.269302855510e+02, 8.284443960630e+02, 8.299429758740e+02, 8.314263640720e+02, 8.328948886550e+02, 8.343488670050e+02, 8.357886063330e+02, 8.372144041150e+02, 8.386265484880e+02, 8.400253186390e+02, 8.414109851720e+02, 8.427838104470e+02, 8.441440489200e+02, 8.454919474470e+02, 8.468277455900e+02, 8.481516758940e+02, 8.494639641640e+02, 8.507648297180e+02, 8.520544856340e+02, 8.533331389850e+02, 8.546009910600e+02, 8.558582375820e+02, 8.571050689060e+02, 8.583416702150e+02, 8.595682217090e+02, 8.607848987780e+02, 8.619918721740e+02, 8.631893081730e+02, 8.643773687330e+02, 8.655562116360e+02, 8.667259906380e+02, 8.678868556010e+02, 8.690389526240e+02, 8.701824241700e+02, 8.713174091860e+02, 8.724440432180e+02, 8.735624585180e+02, 8.746727841570e+02, 8.757751461210e+02, 8.768696674140e+02, 8.779564681470e+02, 8.790356656320e+02, 8.801073744650e+02, 8.811717066170e+02, 8.822287715060e+02, 8.832786760780e+02, 8.843215248840e+02, 8.853574201460e+02, 8.863864618320e+02, 8.874087477160e+02, 8.884243734490e+02, 8.894334326150e+02, 8.904360167940e+02, 8.914322156180e+02, 8.924221168260e+02, 8.934058063180e+02, 8.943833682070e+02, 8.953548848670e+02, 8.963204369830e+02, 8.972801035970e+02, 8.982339621500e+02, 8.991820885290e+02, 9.001245571080e+02, 9.010614407870e+02, 9.019928110300e+02, 9.029187379080e+02, 9.038392901300e+02, 9.047545350790e+02, 9.056645388510e+02, 9.065693662820e+02, 9.074690809840e+02, 9.083637453740e+02, 9.092534207060e+02, 9.101381670970e+02, 9.110180435600e+02, 9.118931080250e+02, 9.127634173720e+02, 9.136290274510e+02, 9.144899931110e+02, 9.153463682240e+02, 9.161982057030e+02, 9.170455575320e+02, 9.178884747850e+02, 9.187270076440e+02, 9.195612054270e+02, 9.203911166010e+02, 9.212167888060e+02, 9.220382688720e+02, 9.228556028390e+02, 9.236688359730e+02, 9.244780127860e+02, 9.252831770490e+02, 9.260843718140e+02, 9.268816394240e+02, 9.276750215320e+02, 9.284645591180e+02, 9.292502924990e+02, 9.300322613440e+02, 9.308105046940e+02, 9.315850609680e+02, 9.323559679790e+02, 9.331232629480e+02, 9.338869825160e+02, 9.346471627550e+02, 9.354038391810e+02, 9.361570467640e+02, 9.369068199430e+02, 9.376531926310e+02, 9.383961982310e+02, 9.391358696430e+02, 9.398722392780e+02, 9.406053390630e+02, 9.413352004540e+02, 9.420618544430e+02, 9.427853315710e+02, 9.435056619310e+02, 9.442228751840e+02, 9.449370005620e+02, 9.456480668750e+02, 9.463561025270e+02, 9.470611355160e+02, 9.477631934450e+02, 9.484623035290e+02, 9.491584926020e+02, 9.498517871260e+02, 9.505422131960e+02, 9.512297965450e+02, 9.519145625570e+02, 9.525965362670e+02, 9.532757423710e+02, 9.539522052290e+02, 9.546259488770e+02, 9.552969970250e+02, 9.559653730710e+02, 9.566311000990e+02, 9.572942008890e+02, 9.579546979250e+02, 9.586126133910e+02, 9.592679691860e+02, 9.599207869260e+02, 9.605710879430e+02, 9.612188933010e+02, 9.618642237890e+02, 9.625070999360e+02, 9.631475420080e+02, 9.637855700160e+02, 9.644212037190e+02, 9.650544626290e+02, 9.656853660170e+02, 9.663139329120e+02, 9.669401821090e+02, 9.675641321730e+02, 9.681858014420e+02, 9.688052080270e+02, 9.694223698240e+02, 9.700373045100e+02, 9.706500295480e+02, 9.712605621960e+02, 9.718689195020e+02, 9.724751183130e+02, 9.730791752760e+02, 9.736811068440e+02, 9.742809292740e+02, 9.748786586340e+02, 9.754743108060e+02, 9.760679014860e+02, 9.766594461910e+02, 9.772489602570e+02, 9.778364588460e+02, 9.784219569480e+02, 9.790054693800e+02, 9.795870107920e+02, 9.801665956720e+02, 9.807442383400e+02, 9.813199529610e+02, 9.818937535380e+02, 9.824656539210e+02, 9.830356678070e+02, 9.836038087410e+02, 9.841700901200e+02, 9.847345251940e+02, 9.852971270690e+02, 9.858579087110e+02, 9.864168829410e+02, 9.869740624480e+02, 9.875294597800e+02, 9.880830873540e+02, 9.886349574530e+02, 9.891850822310e+02, 9.897334737130e+02, 9.902801437980e+02, 9.908251042580e+02, 9.913683667460e+02, 9.919099427900e+02, 9.924498438000e+02, 9.929880810690e+02, 9.935246657710e+02, 9.940596089680e+02, 9.945929216070e+02, 9.951246145260e+02, 9.956546984500e+02, 9.961831839970e+02, 9.967100816780e+02, 9.972354018990e+02, 9.977591549590e+02, 9.982813510580e+02, 9.988020002920e+02, 9.993211126590e+02, 9.998386980560e+02, 1.000354766280e+03, 1.000869327050e+03, 1.001382389960e+03, 1.001893964530e+03, 1.002404060190e+03, 1.002912686270e+03, 1.003419852010e+03, 1.003925566570e+03, 1.004429839010e+03, 1.004932678310e+03, 1.005434093370e+03, 1.005934092990e+03, 1.006432685910e+03, 1.006929880770e+03, 1.007425686130e+03, 1.007920110480e+03, 1.008413162220e+03, 1.008904849670e+03, 1.009395181090e+03, 1.009884164650e+03, 1.010371808450e+03, 1.010858120500e+03, 1.011343108770e+03, 1.011826781120e+03, 1.012309145370e+03, 1.012790209240e+03, 1.013269980410e+03, 1.013748466470e+03, 1.014225674940e+03, 1.014701613300e+03, 1.015176288920e+03, 1.015649709150e+03, 1.016121881250e+03, 1.016592812400e+03, 1.017062509760e+03, 1.017530980390e+03, 1.017998231310e+03, 1.018464269460e+03, 1.018929101740e+03, 1.019392734970e+03, 1.019855175920e+03, 1.020316431310e+03, 1.020776507790e+03, 1.021235411970e+03, 1.021693150370e+03, 1.022149729490e+03, 1.022605155750e+03, 1.023059435540e+03, 1.023512575170e+03, 1.023964580910e+03, 1.024415458980e+03, 1.024865215550e+03, 1.025313856720e+03, 1.025761388550e+03, 1.026207817070e+03, 1.026653148220e+03, 1.027097387930e+03, 1.027540542050e+03, 1.027982616400e+03, 1.028423616750e+03, 1.028863548830e+03, 1.029302418290e+03, 1.029740230780e+03, 1.030176991880e+03, 1.030612707120e+03, 1.031047382000e+03, 1.031481021960e+03, 1.031913632420e+03, 1.032345218730e+03, 1.032775786210e+03, 1.033205340150e+03, 1.033633885770e+03, 1.034061428280e+03, 1.034487972820e+03, 1.034913524500e+03, 1.035338088410e+03, 1.035761669580e+03, 1.036184272990e+03, 1.036605903610e+03, 1.037026566360e+03, 1.037446266100e+03, 1.037865007690e+03, 1.038282795920e+03, 1.038699635570e+03, 1.039115531370e+03, 1.039530488010e+03, 1.039944510150e+03, 1.040357602420e+03, 1.040769769400e+03, 1.041181015650e+03, 1.041591345690e+03, 1.042000764010e+03, 1.042409275060e+03, 1.042816883250e+03, 1.043223592990e+03, 1.043629408610e+03, 1.044034334440e+03, 1.044438374780e+03, 1.044841533880e+03, 1.045243815970e+03, 1.045645225240e+03, 1.046045765860e+03, 1.046445441960e+03, 1.046844257660e+03, 1.047242217020e+03, 1.047639324100e+03, 1.048035582900e+03, 1.048430997410e+03, 1.048825571610e+03, 1.049219309400e+03, 1.049612214710e+03, 1.050004291400e+03, 1.050395543310e+03, 1.050785974280e+03, 1.051175588090e+03, 1.051564388520e+03, 1.051952379280e+03, 1.052339564120e+03, 1.052725946700e+03, 1.053111530690e+03, 1.053496319730e+03, 1.053880317420e+03, 1.054263527360e+03, 1.054645953110e+03, 1.055027598190e+03, 1.055408466130e+03, 1.055788560410e+03, 1.056167884490e+03, 1.056546441810e+03, 1.056924235800e+03, 1.057301269830e+03, 1.057677547300e+03, 1.058053071530e+03, 1.058427845860e+03, 1.058801873590e+03, 1.059175158000e+03, 1.059547702350e+03, 1.059919509870e+03},
    {1.585873423460e+00, 4.826167414420e+00, 8.112135671230e+00, 1.144528461290e+01, 1.482719910000e+01, 1.825954768150e+01, 2.174408826840e+01, 2.528267427420e+01, 2.887726126910e+01, 3.252991419690e+01, 3.624281521090e+01, 4.001827219210e+01, 4.385872801740e+01, 4.776677065620e+01, 5.174514418050e+01, 5.579676078410e+01, 5.992471391870e+01, 6.413229266400e+01, 6.842299746540e+01, 7.280055738580e+01, 7.726894903500e+01, 8.183241735930e+01, 8.649549849020e+01, 9.126304487440e+01, 9.614025292700e+01, 1.011326934710e+02, 1.062463452500e+02, 1.114876318040e+02, 1.168634620430e+02, 1.223812748090e+02, 1.280490877460e+02, 1.338755507420e+02, 1.398700041360e+02, 1.460425417540e+02, 1.524040786430e+02, 1.589664230460e+02, 1.657423516480e+02, 1.727456864320e+02, 1.799913703570e+02, 1.874955375260e+02, 1.952755712300e+02, 2.033501400470e+02, 2.117391976380e+02, 2.204639256740e+02, 2.295465909800e+02, 2.390102772220e+02, 2.488784384610e+02, 2.591742078950e+02, 2.699193834060e+02, 2.811330087440e+02, 2.928294859420e+02, 3.050162040090e+02, 3.176907606140e+02, 3.308379826490e+02, 3.444270910770e+02, 3.584094615110e+02, 3.727174694900e+02, 3.872648743360e+02, 4.019491014770e+02, 4.166556132520e+02, 4.312642483470e+02, 4.456569331740e+02, 4.597256567110e+02, 4.733794188520e+02, 4.865491398270e+02, 4.991880038700e+02, 5.112702827370e+02, 5.227881463360e+02, 5.337475298820e+02, 5.441641402670e+02, 5.540601084360e+02, 5.634614263220e+02, 5.723960928840e+02, 5.808928149700e+02, 5.889801109310e+02, 5.966856991270e+02, 6.040360901010e+02, 6.110563286410e+02, 6.177698489020e+02, 6.241984152660e+02, 6.303621271000e+02, 6.362794692140e+02, 6.419673929830e+02, 6.474414160230e+02, 6.527157310790e+02, 6.578033172590e+02, 6.627160488820e+02, 6.674647988530e+02, 6.720595348280e+02, 6.765094073100e+02, 6.808228295170e+02, 6.850075492310e+02, 6.890707131360e+02, 6.930189242460e+02, 6.968582930910e+02, 7.005944833240e+02, 7.042327523860e+02, 7.077779877910e+02, 7.112347395610e+02, 7.146072492820e+02, 7.178994761620e+02, 7.211151204730e+02, 7.242576446700e+02, 7.273302924530e+02, 7.303361060210e+02, 7.332779417000e+02, 7.361584841370e+02, 7.389802592130e+02, 7.417456457940e+02, 7.444568864660e+02, 7.471160973290e+02, 7.497252769690e+02, 7.522863146660e+02, 7.548009979320e+02, 7.572710194250e+02, 7.596979833100e+02, 7.620834111120e+02, 7.644287471120e+02, 7.667353633160e+02, 7.690045640540e+02, 7.712375902180e+02, 7.734356231970e+02, 7.755997885090e+02, 7.777311591720e+02, 7.798307588340e+02, 7.818995646750e+02, 7.839385101080e+02, 7.859484872910e+02, 7.879303494640e+02, 7.898849131260e+02, 7.918129600730e+02, 7.937152392880e+02, 7.955924687180e+02, 7.974453369270e+02, 7.992745046510e+02, 8.010806062410e+02, 8.028642510320e+02, 8.046260246120e+02, 8.063664900190e+02, 8.080861888680e+02, 8.097856424040e+02, 8.114653524930e+02, 8.131258025630e+02, 8.147674584740e+02, 8.163907693550e+02, 8.179961683820e+02, 8.195840735120e+02, 8.211548881830e+02, 8.227090019700e+02, 8.242467912050e+02, 8.257686195610e+02, 8.272748386140e+02, 8.287657883630e+02, 8.302417977300e+02, 8.317031850330e+02, 8.331502584300e+02, 8.345833163460e+02, 8.360026478750e+02, 8.374085331600e+02, 8.388012437620e+02, 8.401810429970e+02, 8.415481862730e+02, 8.429029213940e+02, 8.442454888640e+02, 8.455761221650e+02, 8.468950480290e+02, 8.482024866960e+02, 8.494986521560e+02, 8.507837523820e+02, 8.520579895590e+02, 8.533215602880e+02, 8.545746557980e+02, 8.558174621320e+02, 8.570501603370e+02, 8.582729266430e+02, 8.594859326270e+02, 8.606893453800e+02, 8.618833276600e+02, 8.630680380410e+02, 8.642436310580e+02, 8.654102573380e+02, 8.665680637350e+02, 8.677171934560e+02, 8.688577861780e+02, 8.699899781640e+02, 8.711139023750e+02, 8.722296885780e+02, 8.733374634440e+02, 8.744373506480e+02, 8.755294709620e+02, 8.766139423500e+02, 8.776908800480e+02, 8.787603966530e+02, 8.798226022030e+02, 8.808776042510e+02, 8.819255079450e+02, 8.829664160930e+02, 8.840004292390e+02, 8.850276457260e+02, 8.860481617600e+02, 8.870620714730e+02, 8.880694669830e+02, 8.890704384510e+02, 8.900650741340e+02, 8.910534604430e+02, 8.920356819940e+02, 8.930118216530e+02, 8.939819605890e+02, 8.949461783210e+02, 8.959045527580e+02, 8.968571602470e+02, 8.978040756150e+02, 8.987453722040e+02, 8.996811219180e+02, 9.006113952550e+02, 9.015362613460e+02, 9.024557879910e+02, 9.033700416920e+02, 9.042790876860e+02, 9.051829899810e+02, 9.060818113790e+02, 9.069756135170e+02, 9.078644568870e+02, 9.087484008680e+02, 9.096275037550e+02, 9.105018227830e+02, 9.113714141520e+02, 9.122363330580e+02, 9.130966337080e+02, 9.139523693520e+02, 9.148035923010e+02, 9.156503539490e+02, 9.164927048000e+02, 9.173306944800e+02, 9.181643717660e+02, 9.189937845990e+02, 9.198189801060e+02, 9.206400046210e+02, 9.214569036970e+02, 9.222697221290e+02, 9.230785039690e+02, 9.238832925400e+02, 9.246841304580e+02, 9.254810596420e+02, 9.262741213330e+02, 9.270633561040e+02, 9.278488038820e+02, 9.286305039540e+02, 9.294084949860e+02, 9.301828150330e+02, 9.309535015570e+02, 9.317205914310e+02, 9.324841209590e+02, 9.332441258850e+02, 9.340006414050e+02, 9.347537021750e+02, 9.355033423300e+02, 9.362495954850e+02, 9.369924947530e+02, 9.377320727520e+02, 9.384683616140e+02, 9.392013929970e+02, 9.399311980930e+02, 9.406578076380e+02, 9.413812519200e+02, 9.421015607900e+02, 9.428187636660e+02, 9.435328895460e+02, 9.442439670150e+02, 9.449520242520e+02, 9.456570890350e+02, 9.463591887560e+02, 9.470583504210e+02, 9.477546006610e+02, 9.484479657390e+02, 9.491384715540e+02, 9.498261436520e+02, 9.505110072290e+02, 9.511930871390e+02, 9.518724079000e+02, 9.525489937000e+02, 9.532228684020e+02, 9.538940555540e+02, 9.545625783880e+02, 9.552284598310e+02, 9.558917225070e+02, 9.565523887470e+02, 9.572104805890e+02, 9.578660197840e+02, 9.585190278040e+02, 9.591695258460e+02, 9.598175348330e+02, 9.604630754240e+02, 9.611061680150e+02, 9.617468327450e+02, 9.623850894980e+02, 9.630209579130e+02, 9.636544573810e+02, 9.642856070530e+02, 9.649144258460e+02, 9.655409324400e+02, 9.661651452910e+02, 9.667870826260e+02, 9.674067624540e+02, 9.680242025640e+02, 9.686394205320e+02, 9.692524337220e+02, 9.698632592930e+02, 9.704719141980e+02, 9.710784151910e+02, 9.716827788260e+02, 9.722850214660e+02, 9.728851592790e+02, 9.734832082480e+02, 9.740791841690e+02, 9.746731026560e+02, 9.752649791430e+02, 9.758548288870e+02, 9.764426669710e+02, 9.770285083080e+02, 9.776123676400e+02, 9.781942595440e+02, 9.787741984330e+02, 9.793521985590e+02, 9.799282740140e+02, 9.805024387360e+02, 9.810747065060e+02, 9.816450909550e+02, 9.822136055630e+02, 9.827802636650e+02, 9.833450784490e+02, 9.839080629600e+02, 9.844692301020e+02, 9.850285926410e+02, 9.855861632040e+02, 9.861419542860e+02, 9.866959782470e+02, 9.872482473160e+02, 9.877987735930e+02, 9.883475690520e+02, 9.888946455380e+02, 9.894400147750e+02, 9.899836883650e+02, 9.905256777880e+02, 9.910659944050e+02, 9.916046494630e+02, 9.921416540910e+02, 9.926770193040e+02, 9.932107560060e+02, 9.937428749890e+02, 9.942733869380e+02, 9.948023024270e+02, 9.953296319270e+02, 9.958553858020e+02, 9.963795743130e+02, 9.969022076190e+02, 9.974232957790e+02, 9.979428487510e+02, 9.984608763980e+02, 9.989773884840e+02, 9.994923946770e+02, 1.000005904550e+03, 1.000517927590e+03, 1.001028473190e+03, 1.001537550630e+03, 1.002045169140e+03, 1.002551337840e+03, 1.003056065760e+03, 1.003559361840e+03, 1.004061234960e+03, 1.004561693900e+03, 1.005060747340e+03, 1.005558403910e+03, 1.006054672150e+03, 1.006549560500e+03, 1.007043077350e+03, 1.007535230990e+03, 1.008026029640e+03, 1.008515481460e+03, 1.009003594510e+03, 1.009490376800e+03, 1.009975836250e+03, 1.010459980710e+03, 1.010942817960e+03, 1.011424355720e+03, 1.011904601620e+03, 1.012383563240e+03, 1.012861248100e+03, 1.013337663620e+03, 1.013812817170e+03, 1.014286716070e+03, 1.014759367560e+03, 1.015230778820e+03, 1.015700956960e+03, 1.016169909030e+03, 1.016637642030e+03, 1.017104162890e+03, 1.017569478480e+03, 1.018033595620e+03, 1.018496521040e+03, 1.018958261450e+03, 1.019418823480e+03, 1.019878213720e+03, 1.020336438690e+03, 1.020793504850e+03, 1.021249418630e+03, 1.021704186380e+03, 1.022157814400e+03, 1.022610308950e+03, 1.023061676230e+03, 1.023511922390e+03, 1.023961053530e+03, 1.024409075690e+03, 1.024855994880e+03, 1.025301817030e+03, 1.025746548040e+03, 1.026190193770e+03, 1.026632760020e+03, 1.027074252550e+03, 1.027514677050e+03, 1.027954039200e+03, 1.028392344600e+03, 1.028829598840e+03, 1.029265807430e+03, 1.029700975850e+03, 1.030135109550e+03, 1.030568213910e+03, 1.031000294290e+03, 1.031431356000e+03, 1.031861404300e+03, 1.032290444410e+03, 1.032718481530e+03, 1.033145520790e+03, 1.033571567300e+03, 1.033996626120e+03, 1.034420702270e+03, 1.034843800750e+03, 1.035265926480e+03, 1.035687084390e+03, 1.036107279330e+03, 1.036526516160e+03, 1.036944799650e+03, 1.037362134570e+03, 1.037778525630e+03, 1.038193977540e+03, 1.038608494940e+03, 1.039022082430e+03, 1.039434744620e+03, 1.039846486030e+03, 1.040257311190e+03, 1.040667224570e+03, 1.041076230620e+03, 1.041484333750e+03, 1.041891538340e+03, 1.042297848730e+03, 1.042703269250e+03, 1.043107804170e+03, 1.043511457740e+03, 1.043914234190e+03, 1.044316137710e+03, 1.044717172450e+03, 1.045117342540e+03, 1.045516652090e+03, 1.045915105160e+03, 1.046312705780e+03, 1.046709457980e+03, 1.047105365730e+03, 1.047500432990e+03, 1.047894663670e+03, 1.048288061690e+03, 1.048680630900e+03, 1.049072375140e+03, 1.049463298250e+03, 1.049853403990e+03, 1.050242696130e+03, 1.050631178410e+03, 1.051018854530e+03, 1.051405728180e+03, 1.051791803010e+03, 1.052177082660e+03, 1.052561570730e+03, 1.052945270790e+03, 1.053328186420e+03, 1.053710321130e+03, 1.054091678440e+03, 1.054472261830e+03, 1.054852074760e+03, 1.055231120670e+03, 1.055609402960e+03, 1.055986925030e+03, 1.056363690240e+03, 1.056739701940e+03, 1.057114963450e+03, 1.057489478060e+03, 1.057863249060e+03},
    {1.581001175600e+00, 4.810993348550e+00, 8.086026432810e+00, 1.140756478610e+01, 1.477714782280e+01, 1.819639493380e+01, 2.166701079670e+01, 2.519079111260e+01, 2.876962880810e+01, 3.240552074980e+01, 3.610057501680e+01, 3.985701878890e+01, 4.367720690760e+01, 4.756363117900e+01, 5.151893049080e+01, 5.554590182590e+01, 5.964751226280e+01, 6.382691206260e+01, 6.808744895480e+01, 7.243268374240e+01, 7.686640736180e+01, 8.139265954640e+01, 8.601574925360e+01, 9.074027703390e+01, 9.557115953070e+01, 1.005136563150e+02, 1.055733992720e+02, 1.107564247550e+02, 1.160692087440e+02, 1.215187052070e+02, 1.271123878710e+02, 1.328582955390e+02, 1.387650810240e+02, 1.448420636390e+02, 1.510992849880e+02, 1.575475675070e+02, 1.641985747780e+02, 1.710648720070e+02, 1.781599841500e+02, 1.854984479010e+02, 1.930958519540e+02, 2.009688574230e+02, 2.091351868990e+02, 2.176135659430e+02, 2.264235947420e+02, 2.355855198950e+02, 2.451198671300e+02, 2.550468858220e+02, 2.653857478130e+02, 2.761534401380e+02, 2.873633007400e+02, 2.990231767530e+02, 3.111332442690e+02, 3.236836172490e+02, 3.366519782440e+02, 3.500015567520e+02, 3.636798303460e+02, 3.776183114710e+02, 3.917337129310e+02, 4.059306676310e+02, 4.201060024320e+02, 4.341543072830e+02, 4.479742149880e+02, 4.614745195460e+02, 4.745792466770e+02, 4.872309227210e+02, 4.993907267370e+02, 5.110373615330e+02, 5.221643659910e+02, 5.327767963880e+02, 5.428880167440e+02, 5.525169771390e+02, 5.616860959970e+02, 5.704197010890e+02, 5.787429178470e+02, 5.866808885830e+02, 5.942582285750e+02, 6.014986523960e+02, 6.084247259230e+02, 6.150577139230e+02, 6.214175014620e+02, 6.275225719800e+02, 6.333900277110e+02, 6.390356402860e+02, 6.444739213790e+02, 6.497182052150e+02, 6.547807366180e+02, 6.596727599570e+02, 6.644046057580e+02, 6.689857728900e+02, 6.734250051230e+02, 6.777303614790e+02, 6.819092802610e+02, 6.859686369320e+02, 6.899147961930e+02, 6.937536587160e+02, 6.974907030090e+02, 7.011310229220e+02, 7.046793612360e+02, 7.081401398010e+02, 7.115174865950e+02, 7.148152600610e+02, 7.180370710470e+02, 7.211863026100e+02, 7.242661279270e+02, 7.272795265350e+02, 7.302292990700e+02, 7.331180806750e+02, 7.359483532090e+02, 7.387224564000e+02, 7.414425980200e+02, 7.441108632150e+02, 7.467292230350e+02, 7.492995422810e+02, 7.518235866950e+02, 7.543030295890e+02, 7.567394579350e+02, 7.591343779920e+02, 7.614892204880e+02, 7.638053454100e+02, 7.660840464350e+02, 7.683265550300e+02, 7.705340442480e+02, 7.727076322500e+02, 7.748483855760e+02, 7.769573221740e+02, 7.790354142300e+02, 7.810835907910e+02, 7.831027402100e+02, 7.850937124280e+02, 7.870573210990e+02, 7.889943455760e+02, 7.909055327670e+02, 7.927915988660e+02, 7.946532309800e+02, 7.964910886450e+02, 7.983058052530e+02, 8.000979893820e+02, 8.018682260580e+02, 8.036170779230e+02, 8.053450863440e+02, 8.070527724540e+02, 8.087406381300e+02, 8.104091669120e+02, 8.120588248720e+02, 8.136900614310e+02, 8.153033101340e+02, 8.168989893740e+02, 8.184775030820e+02, 8.200392413780e+02, 8.215845811830e+02, 8.231138868000e+02, 8.246275104670e+02, 8.261257928750e+02, 8.276090636660e+02, 8.290776418940e+02, 8.305318364760e+02, 8.319719466080e+02, 8.333982621670e+02, 8.348110640910e+02, 8.362106247410e+02, 8.375972082400e+02, 8.389710708040e+02, 8.403324610500e+02, 8.416816202930e+02, 8.430187828270e+02, 8.443441761920e+02, 8.456580214320e+02, 8.469605333410e+02, 8.482519206880e+02, 8.495323864480e+02, 8.508021280090e+02, 8.520613373750e+02, 8.533102013620e+02, 8.545489017790e+02, 8.557776156100e+02, 8.569965151790e+02, 8.582057683130e+02, 8.594055385000e+02, 8.605959850340e+02, 8.617772631560e+02, 8.629495241980e+02, 8.641129157020e+02, 8.652675815570e+02, 8.664136621110e+02, 8.675512942900e+02, 8.686806117080e+02, 8.698017447720e+02, 8.709148207860e+02, 8.720199640490e+02, 8.731172959480e+02, 8.742069350480e+02, 8.752889971820e+02, 8.763635955310e+02, 8.774308407090e+02, 8.784908408360e+02, 8.795437016160e+02, 8.805895264080e+02, 8.816284162950e+02, 8.826604701510e+02, 8.836857847030e+02, 8.847044545980e+02, 8.857165724590e+02, 8.867222289430e+02, 8.877215127970e+02, 8.887145109140e+02, 8.897013083800e+02, 8.906819885280e+02, 8.916566329850e+02, 8.926253217180e+02, 8.935881330800e+02, 8.945451438520e+02, 8.954964292890e+02, 8.964420631550e+02, 8.973821177670e+02, 8.983166640300e+02, 8.992457714760e+02, 9.001695082990e+02, 9.010879413880e+02, 9.020011363620e+02, 9.029091576020e+02, 9.038120682810e+02, 9.047099303970e+02, 9.056028047990e+02, 9.064907512180e+02, 9.073738282940e+02, 9.082520936020e+02, 9.091256036770e+02, 9.099944140410e+02, 9.108585792280e+02, 9.117181528060e+02, 9.125731873970e+02, 9.134237347070e+02, 9.142698455410e+02, 9.151115698240e+02, 9.159489566280e+02, 9.167820541820e+02, 9.176109099010e+02, 9.184355703950e+02, 9.192560814970e+02, 9.200724882720e+02, 9.208848350380e+02, 9.216931653850e+02, 9.224975221860e+02, 9.232979476150e+02, 9.240944831660e+02, 9.248871696590e+02, 9.256760472640e+02, 9.264611555100e+02, 9.272425332990e+02, 9.280202189180e+02, 9.287942500590e+02, 9.295646638210e+02, 9.303314967320e+02, 9.310947847540e+02, 9.318545632980e+02, 9.326108672360e+02, 9.333637309110e+02, 9.341131881450e+02, 9.348592722540e+02, 9.356020160580e+02, 9.363414518860e+02, 9.370776115920e+02, 9.378105265590e+02, 9.385402277130e+02, 9.392667455300e+02, 9.399901100430e+02, 9.407103508520e+02, 9.414274971350e+02, 9.421415776520e+02, 9.428526207560e+02, 9.435606543980e+02, 9.442657061370e+02, 9.449678031480e+02, 9.456669722240e+02, 9.463632397920e+02, 9.470566319100e+02, 9.477471742820e+02, 9.484348922590e+02, 9.491198108490e+02, 9.498019547220e+02, 9.504813482140e+02, 9.511580153370e+02, 9.518319797820e+02, 9.525032649260e+02, 9.531718938370e+02, 9.538378892790e+02, 9.545012737210e+02, 9.551620693350e+02, 9.558202980090e+02, 9.564759813460e+02, 9.571291406730e+02, 9.577797970430e+02, 9.584279712420e+02, 9.590736837910e+02, 9.597169549540e+02, 9.603578047380e+02, 9.609962529010e+02, 9.616323189550e+02, 9.622660221680e+02, 9.628973815730e+02, 9.635264159670e+02, 9.641531439160e+02, 9.647775837620e+02, 9.653997536230e+02, 9.660196713980e+02, 9.666373547710e+02, 9.672528212120e+02, 9.678660879860e+02, 9.684771721490e+02, 9.690860905570e+02, 9.696928598680e+02, 9.702974965430e+02, 9.709000168510e+02, 9.715004368710e+02, 9.720987724970e+02, 9.726950394360e+02, 9.732892532190e+02, 9.738814291950e+02, 9.744715825400e+02, 9.750597282570e+02, 9.756458811790e+02, 9.762300559710e+02, 9.768122671340e+02, 9.773925290070e+02, 9.779708557680e+02, 9.785472614390e+02, 9.791217598860e+02, 9.796943648220e+02, 9.802650898110e+02, 9.808339482660e+02, 9.814009534560e+02, 9.819661185060e+02, 9.825294563980e+02, 9.830909799750e+02, 9.836507019420e+02, 9.842086348680e+02, 9.847647911870e+02, 9.853191832050e+02, 9.858718230920e+02, 9.864227228960e+02, 9.869718945340e+02, 9.875193498000e+02, 9.880651003660e+02, 9.886091577810e+02, 9.891515334760e+02, 9.896922387640e+02, 9.902312848410e+02, 9.907686827900e+02, 9.913044435790e+02, 9.918385780650e+02, 9.923710969980e+02, 9.929020110150e+02, 9.934313306490e+02, 9.939590663270e+02, 9.944852283720e+02, 9.950098270030e+02, 9.955328723400e+02, 9.960543744000e+02, 9.965743431050e+02, 9.970927882770e+02, 9.976097196420e+02, 9.981251468330e+02, 9.986390793880e+02, 9.991515267530e+02, 9.996624982830e+02, 1.000172003240e+03, 1.000680050810e+03, 1.001186650070e+03, 1.001691810030e+03, 1.002195539600e+03, 1.002697847610e+03, 1.003198742820e+03, 1.003698233900e+03, 1.004196329420e+03, 1.004693037890e+03, 1.005188367740e+03, 1.005682327320e+03, 1.006174924900e+03, 1.006666168670e+03, 1.007156066760e+03, 1.007644627200e+03, 1.008131857980e+03, 1.008617766990e+03, 1.009102362050e+03, 1.009585650930e+03, 1.010067641310e+03, 1.010548340820e+03, 1.011027756990e+03, 1.011505897320e+03, 1.011982769210e+03, 1.012458380020e+03, 1.012932737040e+03, 1.013405847480e+03, 1.013877718500e+03, 1.014348357200e+03, 1.014817770610e+03, 1.015285965690e+03, 1.015752949370e+03, 1.016218728490e+03, 1.016683309850e+03, 1.017146700170e+03, 1.017608906140e+03, 1.018069934360e+03, 1.018529791410e+03, 1.018988483790e+03, 1.019446017960e+03, 1.019902400290e+03, 1.020357637150e+03, 1.020811734820e+03, 1.021264699530e+03, 1.021716537470e+03, 1.022167254780e+03, 1.022616857530e+03, 1.023065351760e+03, 1.023512743440e+03, 1.023959038520e+03, 1.024404242870e+03, 1.024848362330e+03, 1.025291402690e+03, 1.025733369680e+03, 1.026174269010e+03, 1.026614106320e+03, 1.027052887220e+03, 1.027490617250e+03, 1.027927301940e+03, 1.028362946740e+03, 1.028797557100e+03, 1.029231138380e+03, 1.029663695930e+03, 1.030095235040e+03, 1.030525760970e+03, 1.030955278940e+03, 1.031383794100e+03, 1.031811311610e+03, 1.032237836540e+03, 1.032663373950e+03, 1.033087928850e+03, 1.033511506220e+03, 1.033934110990e+03, 1.034355748060e+03, 1.034776422300e+03, 1.035196138510e+03, 1.035614901500e+03, 1.036032716000e+03, 1.036449586730e+03, 1.036865518370e+03, 1.037280515560e+03, 1.037694582900e+03, 1.038107724980e+03, 1.038519946320e+03, 1.038931251430e+03, 1.039341644790e+03, 1.039751130820e+03, 1.040159713930e+03, 1.040567398500e+03, 1.040974188870e+03, 1.041380089330e+03, 1.041785104160e+03, 1.042189237620e+03, 1.042592493910e+03, 1.042994877210e+03, 1.043396391690e+03, 1.043797041440e+03, 1.044196830580e+03, 1.044595763160e+03, 1.044993843220e+03, 1.045391074760e+03, 1.045787461740e+03, 1.046183008140e+03, 1.046577717850e+03, 1.046971594770e+03, 1.047364642760e+03, 1.047756865670e+03, 1.048148267290e+03, 1.048538851410e+03, 1.048928621790e+03, 1.049317582150e+03, 1.049705736200e+03, 1.050093087610e+03, 1.050479640040e+03, 1.050865397100e+03, 1.051250362410e+03, 1.051634539540e+03, 1.052017932030e+03, 1.052400543430e+03, 1.052782377220e+03, 1.053163436880e+03, 1.053543725880e+03, 1.053923247640e+03, 1.054302005570e+03, 1.054680003060e+03, 1.055057243470e+03, 1.055433730130e+03, 1.055809466370e+03},
    {1.576159122640e+00, 4.795917737820e+00, 8.060094461970e+00, 1.137011281210e+01, 1.472746812340e+01, 1.813373216240e+01, 2.159055809560e+01, 2.509968584750e+01, 2.866294788210e+01, 3.228227544650e+01, 3.595970532110e+01, 3.969738712190e+01, 4.349759120840e+01, 4.736271725280e+01, 5.129530353570e+01, 5.529803703660e+01, 5.937376439520e+01, 6.352550382960e+01, 6.775645810190e+01, 7.207002863290e+01, 7.646983087640e+01, 8.095971107260e+01, 8.554376451010e+01, 9.022635543610e+01, 9.501213876240e+01, 9.990608372170e+01, 1.049134996350e+02, 1.100400639500e+02, 1.152918527020e+02, 1.206753735400e+02, 1.261976014200e+02, 1.318660170260e+02, 1.376886478880e+02, 1.436741120650e+02, 1.498316640390e+02, 1.561712422550e+02, 1.627035173230e+02, 1.694399393810e+02, 1.763927823590e+02, 1.835751818600e+02, 1.910011619050e+02, 1.986856438520e+02, 2.066444281480e+02, 2.148941360840e+02, 2.234520942110e+02, 2.323361384640e+02, 2.415643084710e+02, 2.511543954600e+02, 2.611233011790e+02, 2.714861627070e+02, 2.822552034860e+02, 2.934382899980e+02, 3.050372120540e+02, 3.170457650970e+02, 3.294477893930e+02, 3.422153972390e+02, 3.553076714820e+02, 3.686701263700e+02, 3.822351786950e+02, 3.959237912580e+02, 4.096483312090e+02, 4.233165306190e+02, 4.368362428020e+02, 4.501204813520e+02, 4.630920812280e+02, 4.756873737440e+02, 4.878583406050e+02, 4.995726706560e+02, 5.108126610300e+02, 5.215729703010e+02, 5.318579401100e+02, 5.416790022900e+02, 5.510524563320e+02, 5.599977133250e+02, 5.685359811460e+02, 5.766893110280e+02, 5.844799167960e+02, 5.919296920150e+02, 5.990598703470e+02, 6.058907918760e+02, 6.124417503890e+02, 6.187309038850e+02, 6.247752347070e+02, 6.305905479870e+02, 6.361914986770e+02, 6.415916388250e+02, 6.468034781280e+02, 6.518385521280e+02, 6.567074937050e+02, 6.614201046670e+02, 6.659854252100e+02, 6.704117997950e+02, 6.747069386050e+02, 6.788779741860e+02, 6.829315131910e+02, 6.868736833520e+02, 6.907101759470e+02, 6.944462840850e+02, 6.980869371810e+02, 7.016367319670e+02, 7.050999604260e+02, 7.084806349380e+02, 7.117825109770e+02, 7.150091076030e+02, 7.181637260090e+02, 7.212494663300e+02, 7.242692429000e+02, 7.272257981330e+02, 7.301217151680e+02, 7.329594294070e+02, 7.357412390610e+02, 7.384693148000e+02, 7.411457086020e+02, 7.437723618740e+02, 7.463511129120e+02, 7.488837037700e+02, 7.513717865800e+02, 7.538169293930e+02, 7.562206215550e+02, 7.585842786880e+02, 7.609092472960e+02, 7.631968090180e+02, 7.654481845870e+02, 7.676645374920e+02, 7.698469773780e+02, 7.719965632080e+02, 7.741143062060e+02, 7.762011725930e+02, 7.782580861340e+02, 7.802859305200e+02, 7.822855515820e+02, 7.842577593640e+02, 7.862033300590e+02, 7.881230078190e+02, 7.900175064470e+02, 7.918875109870e+02, 7.937336792090e+02, 7.955566430020e+02, 7.973570096880e+02, 7.991353632430e+02, 8.008922654620e+02, 8.026282570360e+02, 8.043438585800e+02, 8.060395715940e+02, 8.077158793690e+02, 8.093732478400e+02, 8.110121263970e+02, 8.126329486430e+02, 8.142361331100e+02, 8.158220839440e+02, 8.173911915420e+02, 8.189438331620e+02, 8.204803734960e+02, 8.220011652170e+02, 8.235065494910e+02, 8.249968564680e+02, 8.264724057440e+02, 8.279335068040e+02, 8.293804594370e+02, 8.308135541310e+02, 8.322330724550e+02, 8.336392874130e+02, 8.350324637840e+02, 8.364128584510e+02, 8.377807207060e+02, 8.391362925440e+02, 8.404798089440e+02, 8.418114981360e+02, 8.431315818540e+02, 8.444402755810e+02, 8.457377887800e+02, 8.470243251130e+02, 8.483000826550e+02, 8.495652540950e+02, 8.508200269290e+02, 8.520645836420e+02, 8.532991018900e+02, 8.545237546630e+02, 8.557387104480e+02, 8.569441333870e+02, 8.581401834220e+02, 8.593270164350e+02, 8.605047843920e+02, 8.616736354620e+02, 8.628337141540e+02, 8.639851614280e+02, 8.651281148130e+02, 8.662627085210e+02, 8.673890735480e+02, 8.685073377800e+02, 8.696176260870e+02, 8.707200604230e+02, 8.718147599090e+02, 8.729018409280e+02, 8.739814172040e+02, 8.750535998820e+02, 8.761184976110e+02, 8.771762166100e+02, 8.782268607510e+02, 8.792705316170e+02, 8.803073285770e+02, 8.813373488490e+02, 8.823606875570e+02, 8.833774377970e+02, 8.843876906910e+02, 8.853915354460e+02, 8.863890594040e+02, 8.873803480950e+02, 8.883654852900e+02, 8.893445530440e+02, 8.903176317500e+02, 8.912848001760e+02, 8.922461355140e+02, 8.932017134220e+02, 8.941516080610e+02, 8.950958921390e+02, 8.960346369470e+02, 8.969679123950e+02, 8.978957870490e+02, 8.988183281660e+02, 8.997356017250e+02, 9.006476724630e+02, 9.015546039040e+02, 9.024564583880e+02, 9.033532971040e+02, 9.042451801170e+02, 9.051321663940e+02, 9.060143138340e+02, 9.068916792900e+02, 9.077643185980e+02, 9.086322865990e+02, 9.094956371650e+02, 9.103544232180e+02, 9.112086967560e+02, 9.120585088750e+02, 9.129039097860e+02, 9.137449488390e+02, 9.145816745430e+02, 9.154141345820e+02, 9.162423758370e+02, 9.170664444030e+02, 9.178863856060e+02, 9.187022440200e+02, 9.195140634870e+02, 9.203218871280e+02, 9.211257573640e+02, 9.219257159260e+02, 9.227218038750e+02, 9.235140616140e+02, 9.243025289010e+02, 9.250872448670e+02, 9.258682480220e+02, 9.266455762780e+02, 9.274192669520e+02, 9.281893567840e+02, 9.289558819500e+02, 9.297188780670e+02, 9.304783802120e+02, 9.312344229280e+02, 9.319870402400e+02, 9.327362656580e+02, 9.334821321940e+02, 9.342246723700e+02, 9.349639182260e+02, 9.356999013310e+02, 9.364326527940e+02, 9.371622032700e+02, 9.378885829690e+02, 9.386118216690e+02, 9.393319487170e+02, 9.400489930460e+02, 9.407629831750e+02, 9.414739472210e+02, 9.421819129060e+02, 9.428869075660e+02, 9.435889581550e+02, 9.442880912530e+02, 9.449843330750e+02, 9.456777094770e+02, 9.463682459600e+02, 9.470559676810e+02, 9.477408994550e+02, 9.484230657640e+02, 9.491024907620e+02, 9.497791982810e+02, 9.504532118360e+02, 9.511245546330e+02, 9.517932495700e+02, 9.524593192500e+02, 9.531227859770e+02, 9.537836717670e+02, 9.544419983550e+02, 9.550977871910e+02, 9.557510594550e+02, 9.564018360560e+02, 9.570501376380e+02, 9.576959845820e+02, 9.583393970170e+02, 9.589803948170e+02, 9.596189976090e+02, 9.602552247770e+02, 9.608890954660e+02, 9.615206285840e+02, 9.621498428090e+02, 9.627767565900e+02, 9.634013881530e+02, 9.640237555030e+02, 9.646438764290e+02, 9.652617685040e+02, 9.658774490960e+02, 9.664909353610e+02, 9.671022442570e+02, 9.677113925380e+02, 9.683183967640e+02, 9.689232732990e+02, 9.695260383190e+02, 9.701267078090e+02, 9.707252975730e+02, 9.713218232300e+02, 9.719163002220e+02, 9.725087438140e+02, 9.730991690970e+02, 9.736875909920e+02, 9.742740242500e+02, 9.748584834590e+02, 9.754409830400e+02, 9.760215372580e+02, 9.766001602150e+02, 9.771768658590e+02, 9.777516679850e+02, 9.783245802350e+02, 9.788956161020e+02, 9.794647889340e+02, 9.800321119310e+02, 9.805975981530e+02, 9.811612605170e+02, 9.817231118040e+02, 9.822831646550e+02, 9.828414315800e+02, 9.833979249540e+02, 9.839526570210e+02, 9.845056398960e+02, 9.850568855700e+02, 9.856064059030e+02, 9.861542126360e+02, 9.867003173870e+02, 9.872447316520e+02, 9.877874668110e+02, 9.883285341250e+02, 9.888679447410e+02, 9.894057096930e+02, 9.899418399000e+02, 9.904763461740e+02, 9.910092392160e+02, 9.915405296190e+02, 9.920702278700e+02, 9.925983443540e+02, 9.931248893480e+02, 9.936498730300e+02, 9.941733054790e+02, 9.946951966700e+02, 9.952155564840e+02, 9.957343947040e+02, 9.962517210180e+02, 9.967675450200e+02, 9.972818762090e+02, 9.977947239960e+02, 9.983060976990e+02, 9.988160065470e+02, 9.993244596810e+02, 9.998314661560e+02, 1.000337034940e+03, 1.000841174910e+03, 1.001343894880e+03, 1.001845203550e+03, 1.002345109570e+03, 1.002843621490e+03, 1.003340747770e+03, 1.003836496830e+03, 1.004330876980e+03, 1.004823896450e+03, 1.005315563420e+03, 1.005805885980e+03, 1.006294872150e+03, 1.006782529860e+03, 1.007268867000e+03, 1.007753891360e+03, 1.008237610680e+03, 1.008720032620e+03, 1.009201164760e+03, 1.009681014640e+03, 1.010159589720e+03, 1.010636897380e+03, 1.011112944950e+03, 1.011587739690e+03, 1.012061288810e+03, 1.012533599440e+03, 1.013004678640e+03, 1.013474533440e+03, 1.013943170780e+03, 1.014410597550e+03, 1.014876820580e+03, 1.015341846640e+03, 1.015805682450e+03, 1.016268334660e+03, 1.016729809880e+03, 1.017190114640e+03, 1.017649255430e+03, 1.018107238680e+03, 1.018564070780e+03, 1.019019758050e+03, 1.019474306760e+03, 1.019927723130e+03, 1.020380013320e+03, 1.020831183460e+03, 1.021281239610e+03, 1.021730187780e+03, 1.022178033940e+03, 1.022624784000e+03, 1.023070443850e+03, 1.023515019280e+03, 1.023958516090e+03, 1.024400939990e+03, 1.024842296670e+03, 1.025282591750e+03, 1.025721830830e+03, 1.026160019450e+03, 1.026597163110e+03, 1.027033267260e+03, 1.027468337320e+03, 1.027902378650e+03, 1.028335396590e+03, 1.028767396410e+03, 1.029198383350e+03, 1.029628362630e+03, 1.030057339390e+03, 1.030485318770e+03, 1.030912305830e+03, 1.031338305630e+03, 1.031763323150e+03, 1.032187363370e+03, 1.032610431210e+03, 1.033032531560e+03, 1.033453669260e+03, 1.033873849140e+03, 1.034293075950e+03, 1.034711354450e+03, 1.035128689330e+03, 1.035545085270e+03, 1.035960546900e+03, 1.036375078800e+03, 1.036788685560e+03, 1.037201371690e+03, 1.037613141690e+03, 1.038024000020e+03, 1.038433951110e+03, 1.038842999360e+03, 1.039251149130e+03, 1.039658404740e+03, 1.040064770510e+03, 1.040470250690e+03, 1.040874849520e+03, 1.041278571210e+03, 1.041681419930e+03, 1.042083399820e+03, 1.042484515010e+03, 1.042884769570e+03, 1.043284167560e+03, 1.043682713000e+03, 1.044080409900e+03, 1.044477262220e+03, 1.044873273900e+03, 1.045268448850e+03, 1.045662790950e+03, 1.046056304080e+03, 1.046448992040e+03, 1.046840858650e+03, 1.047231907680e+03, 1.047622142870e+03, 1.048011567970e+03, 1.048400186650e+03, 1.048788002590e+03, 1.049175019440e+03, 1.049561240810e+03, 1.049946670310e+03, 1.050331311500e+03, 1.050715167920e+03, 1.051098243110e+03, 1.051480540550e+03, 1.051862063730e+03, 1.052242816090e+03, 1.052622801050e+03, 1.053002022020e+03, 1.053380482380e+03, 1.053758185490e+03},
    {1.571346980140e+00, 4.780939582700e+00, 8.034337828640e+00, 1.133292557560e+01, 1.467815540130e+01, 1.807155292750e+01, 2.151472146920e+01, 2.500934703720e+01, 2.855720372590e+01, 3.216015951940e+01, 3.582018255440e+01, 3.953934787970e+01, 4.331984475840e+01, 4.716398456090e+01, 5.107420930390e+01, 5.505310089300e+01, 5.910339113470e+01, 6.322797258670e+01, 6.742991032440e+01, 7.171245470550e+01, 7.607905522290e+01, 8.053337554130e+01, 8.507930982130e+01, 8.972100043880e+01, 9.446285721190e+01, 9.930957825260e+01, 1.042661725580e+02, 1.093379844490e+02, 1.145307199620e+02, 1.198504752680e+02, 1.253037671540e+02, 1.308975655830e+02, 1.366393282140e+02, 1.425370366990e+02, 1.485992343800e+02, 1.548350647820e+02, 1.612543099640e+02, 1.678674273590e+02, 1.746855830740e+02, 1.817206787890e+02, 1.889853682480e+02, 1.964930577770e+02, 2.042578832260e+02, 2.122946530960e+02, 2.206187442450e+02, 2.292459324510e+02, 2.381921353580e+02, 2.474730403050e+02, 2.571035851640e+02, 2.670972583250e+02, 2.774651871440e+02, 2.882149963110e+02, 2.993494429360e+02, 3.108648761130e+02, 3.227496234620e+02, 3.349824665630e+02, 3.475314154570e+02, 3.603530121340e+02, 3.733923729200e+02, 3.865841208810e+02, 3.998542716070e+02, 4.131230284810e+02, 4.263083204430e+02, 4.393297815620e+02, 4.521127500080e+02, 4.645917984170e+02, 4.767133728380e+02, 4.884372052660e+02, 4.997363042910e+02, 5.105958993260e+02, 5.210115703840e+02, 5.309870755980e+02, 5.405322460820e+02, 5.496611632840e+02, 5.583906978940e+02, 5.667393980290e+02, 5.747266698490e+02, 5.823721833430e+02, 5.896954440600e+02, 5.967154859620e+02, 6.034506541640e+02, 6.099184564660e+02, 6.161354690050e+02, 6.221172849990e+02, 6.278784976100e+02, 6.334327091650e+02, 6.387925599790e+02, 6.439697709270e+02, 6.489751948980e+02, 6.538188731980e+02, 6.585100938630e+02, 6.630574496610e+02, 6.674688941940e+02, 6.717517951120e+02, 6.759129838290e+02, 6.799588014720e+02, 6.838951410000e+02, 6.877274856000e+02, 6.914609435430e+02, 6.951002797570e+02, 6.986499443720e+02, 7.021140985310e+02, 7.054966377180e+02, 7.088012128720e+02, 7.120312495040e+02, 7.151899650300e+02, 7.182803845230e+02, 7.213053550350e+02, 7.242675586460e+02, 7.271695243710e+02, 7.300136390500e+02, 7.328021573070e+02, 7.355372106850e+02, 7.382208160300e+02, 7.408548832010e+02, 7.434412221600e+02, 7.459815495110e+02, 7.484774945310e+02, 7.509306047370e+02, 7.533423510380e+02, 7.557141324990e+02, 7.580472807540e+02, 7.603430641000e+02, 7.626026912980e+02, 7.648273150960e+02, 7.670180355140e+02, 7.691759028930e+02, 7.713019207410e+02, 7.733970483760e+02, 7.754622033990e+02, 7.774982640040e+02, 7.795060711220e+02, 7.814864304450e+02, 7.834401143000e+02, 7.853678634210e+02, 7.872703885920e+02, 7.891483722020e+02, 7.910024696910e+02, 7.928333109200e+02, 7.946415014470e+02, 7.964276237320e+02, 7.981922382680e+02, 7.999358846460e+02, 8.016590825620e+02, 8.033623327550e+02, 8.050461179070e+02, 8.067109034730e+02, 8.083571384850e+02, 8.099852562930e+02, 8.115956752780e+02, 8.131887995190e+02, 8.147650194270e+02, 8.163247123450e+02, 8.178682431140e+02, 8.193959646100e+02, 8.209082182560e+02, 8.224053345070e+02, 8.238876333010e+02, 8.253554245020e+02, 8.268090083110e+02, 8.282486756580e+02, 8.296747085760e+02, 8.310873805550e+02, 8.324869568830e+02, 8.338736949650e+02, 8.352478446290e+02, 8.366096484210e+02, 8.379593418780e+02, 8.392971537980e+02, 8.406233064910e+02, 8.419380160200e+02, 8.432414924310e+02, 8.445339399740e+02, 8.458155573140e+02, 8.470865377300e+02, 8.483470693060e+02, 8.495973351190e+02, 8.508375134100e+02, 8.520677777550e+02, 8.532882972270e+02, 8.544992365450e+02, 8.557007562260e+02, 8.568930127280e+02, 8.580761585800e+02, 8.592503425170e+02, 8.604157096040e+02, 8.615724013530e+02, 8.627205558420e+02, 8.638603078230e+02, 8.649917888260e+02, 8.661151272670e+02, 8.672304485400e+02, 8.683378751140e+02, 8.694375266230e+02, 8.705295199520e+02, 8.716139693230e+02, 8.726909863740e+02, 8.737606802370e+02, 8.748231576100e+02, 8.758785228360e+02, 8.769268779650e+02, 8.779683228250e+02, 8.790029550830e+02, 8.800308703130e+02, 8.810521620490e+02, 8.820669218480e+02, 8.830752393410e+02, 8.840772022930e+02, 8.850728966490e+02, 8.860624065880e+02, 8.870458145680e+02, 8.880232013780e+02, 8.889946461770e+02, 8.899602265420e+02, 8.909200185100e+02, 8.918740966160e+02, 8.928225339360e+02, 8.937654021230e+02, 8.947027714430e+02, 8.956347108140e+02, 8.965612878400e+02, 8.974825688400e+02, 8.983986188870e+02, 8.993095018360e+02, 9.002152803520e+02, 9.011160159470e+02, 9.020117689990e+02, 9.029025987880e+02, 9.037885635180e+02, 9.046697203450e+02, 9.055461254020e+02, 9.064178338240e+02, 9.072848997690e+02, 9.081473764450e+02, 9.090053161290e+02, 9.098587701920e+02, 9.107077891190e+02, 9.115524225250e+02, 9.123927191830e+02, 9.132287270380e+02, 9.140604932250e+02, 9.148880640910e+02, 9.157114852110e+02, 9.165308014050e+02, 9.173460567530e+02, 9.181572946150e+02, 9.189645576450e+02, 9.197678878070e+02, 9.205673263860e+02, 9.213629140110e+02, 9.221546906610e+02, 9.229426956820e+02, 9.237269678010e+02, 9.245075451390e+02, 9.252844652220e+02, 9.260577649970e+02, 9.268274808390e+02, 9.275936485670e+02, 9.283563034540e+02, 9.291154802380e+02, 9.298712131320e+02, 9.306235358370e+02, 9.313724815500e+02, 9.321180829760e+02, 9.328603723350e+02, 9.335993813760e+02, 9.343351413830e+02, 9.350676831820e+02, 9.357970371580e+02, 9.365232332530e+02, 9.372463009840e+02, 9.379662694450e+02, 9.386831673180e+02, 9.393970228800e+02, 9.401078640110e+02, 9.408157182000e+02, 9.415206125560e+02, 9.422225738090e+02, 9.429216283240e+02, 9.436178021030e+02, 9.443111207940e+02, 9.450016096950e+02, 9.456892937640e+02, 9.463741976220e+02, 9.470563455610e+02, 9.477357615480e+02, 9.484124692330e+02, 9.490864919540e+02, 9.497578527410e+02, 9.504265743230e+02, 9.510926791350e+02, 9.517561893170e+02, 9.524171267260e+02, 9.530755129390e+02, 9.537313692540e+02, 9.543847167000e+02, 9.550355760380e+02, 9.556839677690e+02, 9.563299121330e+02, 9.569734291190e+02, 9.576145384670e+02, 9.582532596720e+02, 9.588896119870e+02, 9.595236144290e+02, 9.601552857820e+02, 9.607846446020e+02, 9.614117092170e+02, 9.620364977370e+02, 9.626590280510e+02, 9.632793178350e+02, 9.638973845530e+02, 9.645132454620e+02, 9.651269176140e+02, 9.657384178610e+02, 9.663477628570e+02, 9.669549690590e+02, 9.675600527350e+02, 9.681630299640e+02, 9.687639166360e+02, 9.693627284630e+02, 9.699594809720e+02, 9.705541895180e+02, 9.711468692760e+02, 9.717375352520e+02, 9.723262022840e+02, 9.729128850390e+02, 9.734975980250e+02, 9.740803555830e+02, 9.746611718990e+02, 9.752400610000e+02, 9.758170367590e+02, 9.763921128950e+02, 9.769653029800e+02, 9.775366204350e+02, 9.781060785370e+02, 9.786736904200e+02, 9.792394690750e+02, 9.798034273540e+02, 9.803655779720e+02, 9.809259335080e+02, 9.814845064070e+02, 9.820413089850e+02, 9.825963534250e+02, 9.831496517840e+02, 9.837012159920e+02, 9.842510578550e+02, 9.847991890560e+02, 9.853456211590e+02, 9.858903656050e+02, 9.864334337220e+02, 9.869748367180e+02, 9.875145856890e+02, 9.880526916180e+02, 9.885891653760e+02, 9.891240177260e+02, 9.896572593200e+02, 9.901889007040e+02, 9.907189523220e+02, 9.912474245090e+02, 9.917743275020e+02, 9.922996714330e+02, 9.928234663360e+02, 9.933457221470e+02, 9.938664487050e+02, 9.943856557510e+02, 9.949033529340e+02, 9.954195498060e+02, 9.959342558310e+02, 9.964474803790e+02, 9.969592327320e+02, 9.974695220810e+02, 9.979783575320e+02, 9.984857481020e+02, 9.989917027260e+02, 9.994962302520e+02, 9.999993394460e+02, 1.000501038990e+03, 1.001001337490e+03, 1.001500243460e+03, 1.001997765360e+03, 1.002493911540e+03, 1.002988690290e+03, 1.003482109820e+03, 1.003974178280e+03, 1.004464903720e+03, 1.004954294130e+03, 1.005442357440e+03, 1.005929101470e+03, 1.006414534020e+03, 1.006898662770e+03, 1.007381495380e+03, 1.007863039400e+03, 1.008343302330e+03, 1.008822291620e+03, 1.009300014610e+03, 1.009776478630e+03, 1.010251690900e+03, 1.010725658600e+03, 1.011198388840e+03, 1.011669888670e+03, 1.012140165080e+03, 1.012609225010e+03, 1.013077075300e+03, 1.013543722790e+03, 1.014009174200e+03, 1.014473436250e+03, 1.014936515560e+03, 1.015398418720e+03, 1.015859152240e+03, 1.016318722590e+03, 1.016777136200e+03, 1.017234399410e+03, 1.017690518540e+03, 1.018145499840e+03, 1.018599349500e+03, 1.019052073680e+03, 1.019503678490e+03, 1.019954169950e+03, 1.020403554090e+03, 1.020851836830e+03, 1.021299024100e+03, 1.021745121730e+03, 1.022190135530e+03, 1.022634071260e+03, 1.023076934630e+03, 1.023518731300e+03, 1.023959466900e+03, 1.024399146990e+03, 1.024837777110e+03, 1.025275362740e+03, 1.025711909330e+03, 1.026147422260e+03, 1.026581906900e+03, 1.027015368560e+03, 1.027447812510e+03, 1.027879243970e+03, 1.028309668150e+03, 1.028739090180e+03, 1.029167515170e+03, 1.029594948190e+03, 1.030021394270e+03, 1.030446858410e+03, 1.030871345540e+03, 1.031294860590e+03, 1.031717408430e+03, 1.032138993900e+03, 1.032559621800e+03, 1.032979296890e+03, 1.033398023910e+03, 1.033815807540e+03, 1.034232652440e+03, 1.034648563250e+03, 1.035063544530e+03, 1.035477600850e+03, 1.035890736730e+03, 1.036302956650e+03, 1.036714265060e+03, 1.037124666390e+03, 1.037534165010e+03, 1.037942765280e+03, 1.038350471540e+03, 1.038757288050e+03, 1.039163219090e+03, 1.039568268880e+03, 1.039972441620e+03, 1.040375741480e+03, 1.040778172580e+03, 1.041179739040e+03, 1.041580444940e+03, 1.041980294320e+03, 1.042379291200e+03, 1.042777439570e+03, 1.043174743380e+03, 1.043571206580e+03, 1.043966833070e+03, 1.044361626730e+03, 1.044755591390e+03, 1.045148730900e+03, 1.045541049040e+03, 1.045932549580e+03, 1.046323236270e+03, 1.046713112810e+03, 1.047102182910e+03, 1.047490450230e+03, 1.047877918400e+03, 1.048264591040e+03, 1.048650471740e+03, 1.049035564070e+03, 1.049419871560e+03, 1.049803397730e+03, 1.050186146080e+03, 1.050568120070e+03, 1.050949323140e+03, 1.051329758710e+03, 1.051709430190e+03},
    {1.566564467340e+00, 4.766057897930e+00, 8.008754632850e+00, 1.129600001410e+01, 1.462920513950e+01, 1.800985091470e+01, 2.143949240190e+01, 2.491976349060e+01, 2.845238191650e+01, 3.203915465980e+01, 3.568198375440e+01, 3.938287254430e+01, 4.314393242400e+01, 4.696739010780e+01, 5.085559547150e+01, 5.481103001700e+01, 5.883631601390e+01, 6.293422637690e+01, 6.710769534110e+01, 7.135983000250e+01, 7.569392279740e+01, 8.011346499560e+01, 8.462216128770e+01, 8.922394555050e+01, 9.392299787370e+01, 9.872376293310e+01, 1.036309697890e+02, 1.086496531800e+02, 1.137851763780e+02, 1.190432556170e+02, 1.244299861220e+02, 1.299518696540e+02, 1.356158434560e+02, 1.414293103600e+02, 1.474001696570e+02, 1.535368481400e+02, 1.598483304470e+02, 1.663441874190e+02, 1.730346007150e+02, 1.799303811700e+02, 1.870429775150e+02, 1.943844708080e+02, 2.019675483790e+02, 2.098054490380e+02, 2.179118688030e+02, 2.263008133390e+02, 2.349863798460e+02, 2.439824474990e+02, 2.533022524080e+02, 2.629578214970e+02, 2.729592416480e+02, 2.833137483260e+02, 2.940246347330e+02, 3.050900104580e+02, 3.165014773320e+02, 3.282428350760e+02, 3.402889705150e+02, 3.526051084760e+02, 3.651465988060e+02, 3.778593779340e+02, 3.906811801290e+02, 4.035434934770e+02, 4.163741678830e+02, 4.291004925450e+02, 4.416524747860e+02, 4.539659864690e+02, 4.659854238400e+02, 4.776655786020e+02, 4.889725688970e+02, 4.998837662300e+02, 5.103868048470e+02, 5.204780270880e+02, 5.301607124690e+02, 5.394433601520e+02, 5.483381880490e+02, 5.568599132100e+02, 5.650248093360e+02, 5.728500013000e+02, 5.803529458870e+02, 5.875510520350e+02, 5.944614039560e+02, 6.011005609230e+02, 6.074844158010e+02, 6.136280999370e+02, 6.195459253450e+02, 6.252513569470e+02, 6.307570086720e+02, 6.360746579690e+02, 6.412152739070e+02, 6.461890547230e+02, 6.510054713570e+02, 6.556733141850e+02, 6.602007408160e+02, 6.645953233700e+02, 6.688640941090e+02, 6.730135887270e+02, 6.770498868560e+02, 6.809786496040e+02, 6.848051540740e+02, 6.885343249550e+02, 6.921707633120e+02, 6.957187727710e+02, 6.991823832980e+02, 7.025653727860e+02, 7.058712866490e+02, 7.091034556190e+02, 7.122650119290e+02, 7.153589040410e+02, 7.183879100720e+02, 7.213546500470e+02, 7.242615971000e+02, 7.271110877290e+02, 7.299053311940e+02, 7.326464181530e+02, 7.353363285900e+02, 7.379769391280e+02, 7.405700297520e+02, 7.431172900310e+02, 7.456203248510e+02, 7.480806597270e+02, 7.504997457200e+02, 7.528789639850e+02, 7.552196300030e+02, 7.575229975010e+02, 7.597902620980e+02, 7.620225646950e+02, 7.642209946330e+02, 7.663865926310e+02, 7.685203535310e+02, 7.706232288510e+02, 7.726961291800e+02, 7.747399264040e+02, 7.767554557970e+02, 7.787435179710e+02, 7.807048807080e+02, 7.826402806740e+02, 7.845504250220e+02, 7.864359929030e+02, 7.882976368840e+02, 7.901359842730e+02, 7.919516383740e+02, 7.937451796590e+02, 7.955171668800e+02, 7.972681381100e+02, 7.989986117230e+02, 8.007090873250e+02, 8.024000466250e+02, 8.040719542620e+02, 8.057252585830e+02, 8.073603923830e+02, 8.089777735950e+02, 8.105778059570e+02, 8.121608796290e+02, 8.137273717850e+02, 8.152776471760e+02, 8.168120586560e+02, 8.183309476900e+02, 8.198346448230e+02, 8.213234701430e+02, 8.227977337040e+02, 8.242577359390e+02, 8.257037680460e+02, 8.271361123580e+02, 8.285550426980e+02, 8.299608247080e+02, 8.313537161740e+02, 8.327339673220e+02, 8.341018211170e+02, 8.354575135290e+02, 8.368012738040e+02, 8.381333247090e+02, 8.394538827780e+02, 8.407631585340e+02, 8.420613567120e+02, 8.433486764680e+02, 8.446253115760e+02, 8.458914506210e+02, 8.471472771800e+02, 8.483929700000e+02, 8.496287031610e+02, 8.508546462400e+02, 8.520709644630e+02, 8.532778188510e+02, 8.544753663640e+02, 8.556637600310e+02, 8.568431490870e+02, 8.580136790910e+02, 8.591754920490e+02, 8.603287265260e+02, 8.614735177600e+02, 8.626099977640e+02, 8.637382954290e+02, 8.648585366230e+02, 8.659708442830e+02, 8.670753385060e+02, 8.681721366370e+02, 8.692613533500e+02, 8.703431007330e+02, 8.714174883610e+02, 8.724846233710e+02, 8.735446105390e+02, 8.745975523430e+02, 8.756435490320e+02, 8.766826986940e+02, 8.777150973120e+02, 8.787408388280e+02, 8.797600151990e+02, 8.807727164540e+02, 8.817790307470e+02, 8.827790444100e+02, 8.837728420010e+02, 8.847605063550e+02, 8.857421186270e+02, 8.867177583450e+02, 8.876875034440e+02, 8.886514303150e+02, 8.896096138430e+02, 8.905621274490e+02, 8.915090431240e+02, 8.924504314720e+02, 8.933863617400e+02, 8.943169018550e+02, 8.952421184600e+02, 8.961620769420e+02, 8.970768414680e+02, 8.979864750110e+02, 8.988910393840e+02, 8.997905952640e+02, 9.006852022260e+02, 9.015749187620e+02, 9.024598023140e+02, 9.033399092970e+02, 9.042152951220e+02, 9.050860142220e+02, 9.059521200720e+02, 9.068136652170e+02, 9.076707012870e+02, 9.085232790260e+02, 9.093714483040e+02, 9.102152581430e+02, 9.110547567350e+02, 9.118899914590e+02, 9.127210089010e+02, 9.135478548730e+02, 9.143705744270e+02, 9.151892118730e+02, 9.160038107990e+02, 9.168144140800e+02, 9.176210639000e+02, 9.184238017640e+02, 9.192226685100e+02, 9.200177043300e+02, 9.208089487790e+02, 9.215964407870e+02, 9.223802186780e+02, 9.231603201780e+02, 9.239367824290e+02, 9.247096419990e+02, 9.254789349010e+02, 9.262446965930e+02, 9.270069620000e+02, 9.277657655190e+02, 9.285211410300e+02, 9.292731219100e+02, 9.300217410380e+02, 9.307670308100e+02, 9.315090231440e+02, 9.322477494910e+02, 9.329832408480e+02, 9.337155277590e+02, 9.344446403320e+02, 9.351706082420e+02, 9.358934607410e+02, 9.366132266670e+02, 9.373299344490e+02, 9.380436121200e+02, 9.387542873170e+02, 9.394619872970e+02, 9.401667389370e+02, 9.408685687440e+02, 9.415675028630e+02, 9.422635670800e+02, 9.429567868330e+02, 9.436471872150e+02, 9.443347929830e+02, 9.450196285600e+02, 9.457017180450e+02, 9.463810852190e+02, 9.470577535450e+02, 9.477317461800e+02, 9.484030859800e+02, 9.490717954990e+02, 9.497378970010e+02, 9.504014124640e+02, 9.510623635800e+02, 9.517207717670e+02, 9.523766581680e+02, 9.530300436580e+02, 9.536809488510e+02, 9.543293940990e+02, 9.549753995010e+02, 9.556189849040e+02, 9.562601699110e+02, 9.568989738810e+02, 9.575354159360e+02, 9.581695149630e+02, 9.588012896200e+02, 9.594307583370e+02, 9.600579393220e+02, 9.606828505640e+02, 9.613055098350e+02, 9.619259346960e+02, 9.625441424990e+02, 9.631601503910e+02, 9.637739753150e+02, 9.643856340180e+02, 9.649951430470e+02, 9.656025187600e+02, 9.662077773230e+02, 9.668109347170e+02, 9.674120067370e+02, 9.680110089980e+02, 9.686079569380e+02, 9.692028658160e+02, 9.697957507220e+02, 9.703866265740e+02, 9.709755081210e+02, 9.715624099490e+02, 9.721473464810e+02, 9.727303319790e+02, 9.733113805470e+02, 9.738905061340e+02, 9.744677225370e+02, 9.750430434000e+02, 9.756164822200e+02, 9.761880523460e+02, 9.767577669840e+02, 9.773256391970e+02, 9.778916819070e+02, 9.784559078990e+02, 9.790183298220e+02, 9.795789601900e+02, 9.801378113840e+02, 9.806948956550e+02, 9.812502251280e+02, 9.818038117960e+02, 9.823556675310e+02, 9.829058040810e+02, 9.834542330720e+02, 9.840009660090e+02, 9.845460142800e+02, 9.850893891570e+02, 9.856311017950e+02, 9.861711632390e+02, 9.867095844180e+02, 9.872463761530e+02, 9.877815491560e+02, 9.883151140320e+02, 9.888470812800e+02, 9.893774612920e+02, 9.899062643600e+02, 9.904335006730e+02, 9.909591803200e+02, 9.914833132910e+02, 9.920059094770e+02, 9.925269786750e+02, 9.930465305830e+02, 9.935645748090e+02, 9.940811208670e+02, 9.945961781780e+02, 9.951097560730e+02, 9.956218637960e+02, 9.961325105010e+02, 9.966417052560e+02, 9.971494570420e+02, 9.976557747550e+02, 9.981606672110e+02, 9.986641431390e+02, 9.991662111880e+02, 9.996668799270e+02, 1.000166157850e+03, 1.000664053350e+03, 1.001160574790e+03, 1.001655730400e+03, 1.002149528370e+03, 1.002641976810e+03, 1.003133083750e+03, 1.003622857140e+03, 1.004111304880e+03, 1.004598434780e+03, 1.005084254590e+03, 1.005568771990e+03, 1.006051994580e+03, 1.006533929890e+03, 1.007014585420e+03, 1.007493968560e+03, 1.007972086650e+03, 1.008448946970e+03, 1.008924556740e+03, 1.009398923110e+03, 1.009872053150e+03, 1.010343953910e+03, 1.010814632340e+03, 1.011284095350e+03, 1.011752349780e+03, 1.012219402430e+03, 1.012685260020e+03, 1.013149929220e+03, 1.013613416640e+03, 1.014075728860e+03, 1.014536872350e+03, 1.014996853590e+03, 1.015455678950e+03, 1.015913354790e+03, 1.016369887380e+03, 1.016825282960e+03, 1.017279547710e+03, 1.017732687770e+03, 1.018184709220e+03, 1.018635618080e+03, 1.019085420340e+03, 1.019534121930e+03, 1.019981728730e+03, 1.020428246580e+03, 1.020873681270e+03, 1.021318038540e+03, 1.021761324080e+03, 1.022203543540e+03, 1.022644702520e+03, 1.023084806590e+03, 1.023523861260e+03, 1.023961872000e+03, 1.024398844230e+03, 1.024834783330e+03, 1.025269694660e+03, 1.025703583510e+03, 1.026136455130e+03, 1.026568314740e+03, 1.026999167510e+03, 1.027429018590e+03, 1.027857873070e+03, 1.028285736000e+03, 1.028712612390e+03, 1.029138507230e+03, 1.029563425460e+03, 1.029987371970e+03, 1.030410351630e+03, 1.030832369270e+03, 1.031253429680e+03, 1.031673537600e+03, 1.032092697770e+03, 1.032510914850e+03, 1.032928193500e+03, 1.033344538320e+03, 1.033759953910e+03, 1.034174444790e+03, 1.034588015470e+03, 1.035000670440e+03, 1.035412414140e+03, 1.035823250960e+03, 1.036233185300e+03, 1.036642221500e+03, 1.037050363860e+03, 1.037457616670e+03, 1.037863984180e+03, 1.038269470610e+03, 1.038674080150e+03, 1.039077816950e+03, 1.039480685140e+03, 1.039882688820e+03, 1.040283832060e+03, 1.040684118890e+03, 1.041083553340e+03, 1.041482139370e+03, 1.041879880950e+03, 1.042276781990e+03, 1.042672846400e+03, 1.043068078040e+03, 1.043462480770e+03, 1.043856058380e+03, 1.044248814680e+03, 1.044640753420e+03, 1.045031878330e+03, 1.045422193140e+03, 1.045811701520e+03, 1.046200407130e+03, 1.046588313610e+03, 1.046975424560e+03, 1.047361743560e+03, 1.047747274180e+03, 1.048132019940e+03, 1.048515984370e+03, 1.048899170930e+03, 1.049281583100e+03, 1.049663224320e+03},
    {1.561811307010e+00, 4.751271712220e+00, 7.983343004140e+00, 1.125933311670e+01, 1.458061290290e+01, 1.794861993130e+01, 2.136486255310e+01, 2.483092426060e+01, 2.834846836280e+01, 3.191924300280e+01, 3.554508654460e+01, 3.922793336200e+01, 4.296982005990e+01, 4.677289216630e+01, 5.063941133200e+01, 5.457176308100e+01, 5.857246515470e+01, 6.264417650100e+01, 6.678970695640e+01, 7.101202767740e+01, 7.531428237850e+01, 7.969979943590e+01, 8.417210491840e+01, 8.873493660890e+01, 9.339225907550e+01, 9.814827985300e+01, 1.030074667840e+02, 1.079745665610e+02, 1.130546244980e+02, 1.182530055200e+02, 1.235754163480e+02, 1.290279287760e+02, 1.346170038910e+02, 1.403495169750e+02, 1.462327826910e+02, 1.522745799830e+02, 1.584831758730e+02, 1.648673470050e+02, 1.714363973510e+02, 1.782001699440e+02, 1.851690497180e+02, 1.923539536060e+02, 1.997663027770e+02, 2.074179703640e+02, 2.153211961120e+02, 2.234884571100e+02, 2.319322812270e+02, 2.406649872410e+02, 2.496983333630e+02, 2.590430547100e+02, 2.687082714540e+02, 2.787007546530e+02, 2.890240480640e+02, 2.996774634320e+02, 3.106549940550e+02, 3.219442247380e+02, 3.335253495180e+02, 3.453704328810e+02, 3.574430559900e+02, 3.696984706930e+02, 3.820843410870e+02, 3.945420923660e+02, 4.070088193380e+02, 4.194196405270e+02, 4.317103228300e+02, 4.438199511940e+02, 4.556933869440e+02, 4.672832610550e+02, 4.785512780380e+02, 4.894688418070e+02, 5.000169151240e+02, 5.101851141360e+02, 5.199704147490e+02, 5.293757019490e+02, 5.384083617890e+02, 5.470790404360e+02, 5.554006229110e+02, 5.633874322670e+02, 5.710546211410e+02, 5.784177174860e+02, 5.854922877940e+02, 5.922936879820e+02, 5.988368799890e+02, 6.051362987700e+02, 6.112057591520e+02, 6.170583949000e+02, 6.227066241120e+02, 6.281621359290e+02, 6.334358941820e+02, 6.385381540340e+02, 6.434784881540e+02, 6.482658194170e+02, 6.529084576800e+02, 6.574141386090e+02, 6.617900630470e+02, 6.660429357730e+02, 6.701790028690e+02, 6.742040871600e+02, 6.781236214460e+02, 6.819426793630e+02, 6.856660038610e+02, 6.892980333450e+02, 6.928429256000e+02, 6.963045796280e+02, 6.996866555590e+02, 7.029925927970e+02, 7.062256265540e+02, 7.093888029210e+02, 7.124849926270e+02, 7.155169035990e+02, 7.184870924560e+02, 7.213979750270e+02, 7.242518359990e+02, 7.270508377770e+02, 7.297970286290e+02, 7.324923501830e+02, 7.351386443450e+02, 7.377376596730e+02, 7.402910572790e+02, 7.428004162800e+02, 7.452672388560e+02, 7.476929549260e+02, 7.500789264980e+02, 7.524264517090e+02, 7.547367685740e+02, 7.570110584780e+02, 7.592504494340e+02, 7.614560191090e+02, 7.636287976540e+02, 7.657697703450e+02, 7.678798800440e+02, 7.699600295100e+02, 7.720110835520e+02, 7.740338710480e+02, 7.760291868370e+02, 7.779977934930e+02, 7.799404229850e+02, 7.818577782400e+02, 7.837505346100e+02, 7.856193412460e+02, 7.874648223970e+02, 7.892875786310e+02, 7.910881879740e+02, 7.928672070030e+02, 7.946251718570e+02, 7.963625991980e+02, 7.980799871250e+02, 7.997778160230e+02, 8.014565493750e+02, 8.031166345270e+02, 8.047585034110e+02, 8.063825732250e+02, 8.079892470880e+02, 8.095789146480e+02, 8.111519526620e+02, 8.127087255520e+02, 8.142495859230e+02, 8.157748750610e+02, 8.172849234040e+02, 8.187800509860e+02, 8.202605678650e+02, 8.217267745260e+02, 8.231789622630e+02, 8.246174135490e+02, 8.260424023780e+02, 8.274541945990e+02, 8.288530482330e+02, 8.302392137700e+02, 8.316129344600e+02, 8.329744465800e+02, 8.343239797030e+02, 8.356617569390e+02, 8.369879951800e+02, 8.383029053200e+02, 8.396066924760e+02, 8.408995561980e+02, 8.421816906590e+02, 8.434532848540e+02, 8.447145227750e+02, 8.459655835880e+02, 8.472066417980e+02, 8.484378674090e+02, 8.496594260780e+02, 8.508714792570e+02, 8.520741843380e+02, 8.532676947880e+02, 8.544521602710e+02, 8.556277267820e+02, 8.567945367580e+02, 8.579527291970e+02, 8.591024397640e+02, 8.602438009010e+02, 8.613769419240e+02, 8.625019891210e+02, 8.636190658510e+02, 8.647282926270e+02, 8.658297872060e+02, 8.669236646740e+02, 8.680100375220e+02, 8.690890157280e+02, 8.701607068290e+02, 8.712252159950e+02, 8.722826460930e+02, 8.733330977610e+02, 8.743766694660e+02, 8.754134575710e+02, 8.764435563910e+02, 8.774670582520e+02, 8.784840535490e+02, 8.794946307960e+02, 8.804988766800e+02, 8.814968761110e+02, 8.824887122720e+02, 8.834744666630e+02, 8.844542191470e+02, 8.854280479950e+02, 8.863960299270e+02, 8.873582401560e+02, 8.883147524220e+02, 8.892656390340e+02, 8.902109709090e+02, 8.911508176020e+02, 8.920852473450e+02, 8.930143270810e+02, 8.939381224950e+02, 8.948566980430e+02, 8.957701169900e+02, 8.966784414300e+02, 8.975817323240e+02, 8.984800495200e+02, 8.993734517850e+02, 9.002619968300e+02, 9.011457413340e+02, 9.020247409710e+02, 9.028990504300e+02, 9.037687234430e+02, 9.046338128040e+02, 9.054943703930e+02, 9.063504471940e+02, 9.072020933200e+02, 9.080493580310e+02, 9.088922897500e+02, 9.097309360900e+02, 9.105653438640e+02, 9.113955591060e+02, 9.122216270910e+02, 9.130435923480e+02, 9.138614986790e+02, 9.146753891720e+02, 9.154853062190e+02, 9.162912915320e+02, 9.170933861540e+02, 9.178916304770e+02, 9.186860642530e+02, 9.194767266100e+02, 9.202636560620e+02, 9.210468905260e+02, 9.218264673320e+02, 9.226024232330e+02, 9.233747944230e+02, 9.241436165420e+02, 9.249089246920e+02, 9.256707534440e+02, 9.264291368540e+02, 9.271841084660e+02, 9.279357013290e+02, 9.286839480040e+02, 9.294288805740e+02, 9.301705306510e+02, 9.309089293920e+02, 9.316441074980e+02, 9.323760952330e+02, 9.331049224250e+02, 9.338306184760e+02, 9.345532123750e+02, 9.352727326980e+02, 9.359892076230e+02, 9.367026649320e+02, 9.374131320220e+02, 9.381206359100e+02, 9.388252032430e+02, 9.395268603000e+02, 9.402256330030e+02, 9.409215469230e+02, 9.416146272830e+02, 9.423048989710e+02, 9.429923865370e+02, 9.436771142070e+02, 9.443591058850e+02, 9.450383851590e+02, 9.457149753080e+02, 9.463888993060e+02, 9.470601798280e+02, 9.477288392540e+02, 9.483948996760e+02, 9.490583829040e+02, 9.497193104650e+02, 9.503777036160e+02, 9.510335833430e+02, 9.516869703660e+02, 9.523378851450e+02, 9.529863478870e+02, 9.536323785420e+02, 9.542759968180e+02, 9.549172221760e+02, 9.555560738390e+02, 9.561925707950e+02, 9.568267317990e+02, 9.574585753800e+02, 9.580881198410e+02, 9.587153832680e+02, 9.593403835270e+02, 9.599631382710e+02, 9.605836649450e+02, 9.612019807870e+02, 9.618181028290e+02, 9.624320479070e+02, 9.630438326570e+02, 9.636534735230e+02, 9.642609867580e+02, 9.648663884250e+02, 9.654696944060e+02, 9.660709203980e+02, 9.666700819200e+02, 9.672671943130e+02, 9.678622727470e+02, 9.684553322190e+02, 9.690463875570e+02, 9.696354534240e+02, 9.702225443200e+02, 9.708076745820e+02, 9.713908583900e+02, 9.719721097670e+02, 9.725514425830e+02, 9.731288705550e+02, 9.737044072510e+02, 9.742780660910e+02, 9.748498603510e+02, 9.754198031640e+02, 9.759879075220e+02, 9.765541862760e+02, 9.771186521420e+02, 9.776813177010e+02, 9.782421954010e+02, 9.788012975570e+02, 9.793586363570e+02, 9.799142238610e+02, 9.804680720020e+02, 9.810201925900e+02, 9.815705973130e+02, 9.821192977380e+02, 9.826663053130e+02, 9.832116313690e+02, 9.837552871220e+02, 9.842972836730e+02, 9.848376320120e+02, 9.853763430150e+02, 9.859134274520e+02, 9.864488959840e+02, 9.869827591640e+02, 9.875150274420e+02, 9.880457111640e+02, 9.885748205710e+02, 9.891023658070e+02, 9.896283569140e+02, 9.901528038360e+02, 9.906757164210e+02, 9.911971044200e+02, 9.917169774910e+02, 9.922353451970e+02, 9.927522170110e+02, 9.932676023130e+02, 9.937815103940e+02, 9.942939504580e+02, 9.948049316200e+02, 9.953144629080e+02, 9.958225532680e+02, 9.963292115590e+02, 9.968344465580e+02, 9.973382669590e+02, 9.978406813760e+02, 9.983416983440e+02, 9.988413263180e+02, 9.993395736740e+02, 9.998364487120e+02, 1.000331959660e+03, 1.000826114660e+03, 1.001318921780e+03, 1.001810389040e+03, 1.002300524360e+03, 1.002789335600e+03, 1.003276830540e+03, 1.003763016900e+03, 1.004247902330e+03, 1.004731494410e+03, 1.005213800650e+03, 1.005694828490e+03, 1.006174585330e+03, 1.006653078460e+03, 1.007130315150e+03, 1.007606302580e+03, 1.008081047880e+03, 1.008554558100e+03, 1.009026840250e+03, 1.009497901280e+03, 1.009967748060e+03, 1.010436387420e+03, 1.010903826110e+03, 1.011370070860e+03, 1.011835128290e+03, 1.012299005020e+03, 1.012761707570e+03, 1.013223242420e+03, 1.013683616010e+03, 1.014142834700e+03, 1.014600904820e+03, 1.015057832630e+03, 1.015513624350e+03, 1.015968286150e+03, 1.016421824120e+03, 1.016874244340e+03, 1.017325552820e+03, 1.017775755520e+03, 1.018224858360e+03, 1.018672867190e+03, 1.019119787840e+03, 1.019565626090e+03, 1.020010387650e+03, 1.020454078190e+03, 1.020896703370e+03, 1.021338268760e+03, 1.021778779900e+03, 1.022218242290e+03, 1.022656661390e+03, 1.023094042620e+03, 1.023530391320e+03, 1.023965712850e+03, 1.024400012470e+03, 1.024833295440e+03, 1.025265566940e+03, 1.025696832160e+03, 1.026127096200e+03, 1.026556364140e+03, 1.026984641040e+03, 1.027411931890e+03, 1.027838241660e+03, 1.028263575270e+03, 1.028687937610e+03, 1.029111333540e+03, 1.029533767870e+03, 1.029955245380e+03, 1.030375770800e+03, 1.030795348850e+03, 1.031213984200e+03, 1.031631681470e+03, 1.032048445280e+03, 1.032464280190e+03, 1.032879190720e+03, 1.033293181380e+03, 1.033706256640e+03, 1.034118420920e+03, 1.034529678620e+03, 1.034940034120e+03, 1.035349491740e+03, 1.035758055790e+03, 1.036165730540e+03, 1.036572520240e+03, 1.036978429080e+03, 1.037383461250e+03, 1.037787620900e+03, 1.038190912150e+03, 1.038593339080e+03, 1.038994905760e+03, 1.039395616210e+03, 1.039795474450e+03, 1.040194484430e+03, 1.040592650110e+03, 1.040989975410e+03, 1.041386464210e+03, 1.041782120380e+03, 1.042176947740e+03, 1.042570950120e+03, 1.042964131280e+03, 1.043356494990e+03, 1.043748044970e+03, 1.044138784920e+03, 1.044528718530e+03, 1.044917849440e+03, 1.045306181280e+03, 1.045693717650e+03, 1.046080462130e+03, 1.046466418270e+03, 1.046851589590e+03, 1.047235979600e+03, 1.047619591790e+03}
    };
};

struct TabulatedEnthalpyTraits
{
    using Scalar = double;
    static constexpr std::string_view name = "enthalpy";
    static constexpr int numTempSteps = 50;
    static constexpr Scalar minTemp = 290.0;
    static constexpr Scalar maxTemp = 340.0;
    static constexpr int numPressSteps = 495;
    static constexpr Scalar minPress = 100000.0;
    static constexpr Scalar maxPress = 100000000.0;
    static constexpr Scalar vals[numTempSteps][numPressSteps] = {
        {1.408170317687e+04, 1.206094560787e+04, 9.999074027871e+03, 7.893570186871e+03, 5.741655988871e+03, 3.540254335871e+03, 1.285942053871e+03, -1.025107150129e+03, -3.397192399129e+03, -5.835171637130e+03, -8.344568777129e+03, -1.093170876013e+04, -1.360388987113e+04, -1.636960673613e+04, -1.923884338713e+04, -2.222346527813e+04, -2.533775429913e+04, -2.859915597713e+04, -3.202935143913e+04, -3.565584469213e+04, -3.951440377013e+04, -4.365299298513e+04, -4.813848548513e+04, -5.306901658913e+04, -5.859915902013e+04, -6.499942342413e+04, -2.394171458381e+05, -2.402571060261e+05, -2.410293327171e+05, -2.417443818501e+05, -2.424103577441e+05, -2.430336455861e+05, -2.436193824671e+05, -2.441717736191e+05, -2.446943122191e+05, -2.451899366221e+05, -2.456611455771e+05, -2.461100844201e+05, -2.465386107231e+05, -2.469483450921e+05, -2.473407110201e+05, -2.477169665451e+05, -2.480782296861e+05, -2.484254990821e+05, -2.487596708981e+05, -2.490815527861e+05, -2.493918755031e+05, -2.496913026591e+05, -2.499804389301e+05, -2.502598370381e+05, -2.505300037071e+05, -2.507914047761e+05, -2.510444696091e+05, -2.512895949131e+05, -2.515271480651e+05, -2.517574700181e+05, -2.519808778451e+05, -2.521976669851e+05, -2.524081132251e+05, -2.526124744531e+05, -2.528109922221e+05, -2.530038931391e+05, -2.531913901121e+05, -2.533736834611e+05, -2.535509619231e+05, -2.537234035541e+05, -2.538911765451e+05, -2.540544399561e+05, -2.542133443891e+05, -2.543680325911e+05, -2.545186400121e+05, -2.546652953051e+05, -2.548081207891e+05, -2.549472328721e+05, -2.550827424321e+05, -2.552147551811e+05, -2.553433719831e+05, -2.554686891601e+05, -2.555907987671e+05, -2.557097888471e+05, -2.558257436721e+05, -2.559387439591e+05, -2.560488670741e+05, -2.561561872191e+05, -2.562607756121e+05, -2.563627006451e+05, -2.564620280421e+05, -2.565588209921e+05, -2.566531402911e+05, -2.567450444591e+05, -2.568345898571e+05, -2.569218307981e+05, -2.570068196431e+05, -2.570896069021e+05, -2.571702413191e+05, -2.572487699591e+05, -2.573252382831e+05, -2.573996902251e+05, -2.574721682631e+05, -2.575427134801e+05, -2.576113656311e+05, -2.576781631991e+05, -2.577431434511e+05, -2.578063424921e+05, -2.578677953091e+05, -2.579275358251e+05, -2.579855969381e+05, -2.580420105671e+05, -2.580968076871e+05, -2.581500183711e+05, -2.582016718251e+05, -2.582517964211e+05, -2.583004197291e+05, -2.583475685501e+05, -2.583932689431e+05, -2.584375462551e+05, -2.584804251451e+05, -2.585219296131e+05, -2.585620830201e+05, -2.586009081141e+05, -2.586384270491e+05, -2.586746614121e+05, -2.587096322341e+05, -2.587433600181e+05, -2.587758647521e+05, -2.588071659271e+05, -2.588372825571e+05, -2.588662331911e+05, -2.588940359301e+05, -2.589207084431e+05, -2.589462679801e+05, -2.589707313831e+05, -2.589941151051e+05, -2.590164352171e+05, -2.590377074221e+05, -2.590579470661e+05, -2.590771691501e+05, -2.590953883391e+05, -2.591126189751e+05, -2.591288750831e+05, -2.591441703831e+05, -2.591585183001e+05, -2.591719319671e+05, -2.591844242421e+05, -2.591960077101e+05, -2.592066946901e+05, -2.592164972491e+05, -2.592254272021e+05, -2.592334961231e+05, -2.592407153521e+05, -2.592470959991e+05, -2.592526489521e+05, -2.592573848831e+05, -2.592613142521e+05, -2.592644473161e+05, -2.592667941321e+05, -2.592683645621e+05, -2.592691682801e+05, -2.592692147761e+05, -2.592685133591e+05, -2.592670731661e+05, -2.592649031601e+05, -2.592620121411e+05, -2.592584087461e+05, -2.592541014551e+05, -2.592490985931e+05, -2.592434083341e+05, -2.592370387091e+05, -2.592299976041e+05, -2.592222927641e+05, -2.592139318011e+05, -2.592049221921e+05, -2.591952712851e+05, -2.591849863021e+05, -2.591740743411e+05, -2.591625423781e+05, -2.591503972731e+05, -2.591376457691e+05, -2.591242944961e+05, -2.591103499771e+05, -2.590958186231e+05, -2.590807067431e+05, -2.590650205411e+05, -2.590487661231e+05, -2.590319494941e+05, -2.590145765631e+05, -2.589966531461e+05, -2.589781849671e+05, -2.589591776581e+05, -2.589396367651e+05, -2.589195677461e+05, -2.588989759741e+05, -2.588778667421e+05, -2.588562452581e+05, -2.588341166551e+05, -2.588114859851e+05, -2.587883582241e+05, -2.587647382751e+05, -2.587406309681e+05, -2.587160410591e+05, -2.586909732361e+05, -2.586654321191e+05, -2.586394222561e+05, -2.586129481351e+05, -2.585860141761e+05, -2.585586247341e+05, -2.585307841061e+05, -2.585024965231e+05, -2.584737661611e+05, -2.584445971331e+05, -2.584149934961e+05, -2.583849592521e+05, -2.583544983461e+05, -2.583236146671e+05, -2.582923120541e+05, -2.582605942911e+05, -2.582284651121e+05, -2.581959281981e+05, -2.581629871851e+05, -2.581296456551e+05, -2.580959071461e+05, -2.580617751471e+05, -2.580272531021e+05, -2.579923444101e+05, -2.579570524231e+05, -2.579213804531e+05, -2.578853317661e+05, -2.578489095871e+05, -2.578121171001e+05, -2.577749574481e+05, -2.577374337331e+05, -2.576995490191e+05, -2.576613063301e+05, -2.576227086541e+05, -2.575837589401e+05, -2.575444601001e+05, -2.575048150111e+05, -2.574648265141e+05, -2.574244974151e+05, -2.573838304861e+05, -2.573428284641e+05, -2.573014940551e+05, -2.572598299311e+05, -2.572178387311e+05, -2.571755230641e+05, -2.571328855071e+05, -2.570899286071e+05, -2.570466548801e+05, -2.570030668121e+05, -2.569591668621e+05, -2.569149574581e+05, -2.568704410011e+05, -2.568256198641e+05, -2.567804963921e+05, -2.567350729041e+05, -2.566893516911e+05, -2.566433350201e+05, -2.565970251311e+05, -2.565504242371e+05, -2.565035345301e+05, -2.564563581731e+05, -2.564088973091e+05, -2.563611540541e+05, -2.563131305031e+05, -2.562648287241e+05, -2.562162507671e+05, -2.561673986571e+05, -2.561182743971e+05, -2.560688799671e+05, -2.560192173291e+05, -2.559692884201e+05, -2.559190951601e+05, -2.558686394441e+05, -2.558179231511e+05, -2.557669481361e+05, -2.557157162391e+05, -2.556642292761e+05, -2.556124890471e+05, -2.555604973321e+05, -2.555082558921e+05, -2.554557664701e+05, -2.554030307921e+05, -2.553500505641e+05, -2.552968274771e+05, -2.552433632031e+05, -2.551896593961e+05, -2.551357176961e+05, -2.550815397241e+05, -2.550271270851e+05, -2.549724813691e+05, -2.549176041501e+05, -2.548624969831e+05, -2.548071614121e+05, -2.547515989641e+05, -2.546958111491e+05, -2.546397994641e+05, -2.545835653911e+05, -2.545271103981e+05, -2.544704359381e+05, -2.544135434491e+05, -2.543564343561e+05, -2.542991100721e+05, -2.542415719921e+05, -2.541838215021e+05, -2.541258599711e+05, -2.540676887591e+05, -2.540093092101e+05, -2.539507226561e+05, -2.538919304161e+05, -2.538329337981e+05, -2.537737340971e+05, -2.537143325951e+05, -2.536547305631e+05, -2.535949292611e+05, -2.535349299351e+05, -2.534747338211e+05, -2.534143421431e+05, -2.533537561161e+05, -2.532929769401e+05, -2.532320058081e+05, -2.531708438981e+05, -2.531094923821e+05, -2.530479524171e+05, -2.529862251531e+05, -2.529243117281e+05, -2.528622132691e+05, -2.527999308961e+05, -2.527374657151e+05, -2.526748188251e+05, -2.526119913151e+05, -2.525489842641e+05, -2.524857987411e+05, -2.524224358061e+05, -2.523588965101e+05, -2.522951818941e+05, -2.522312929921e+05, -2.521672308271e+05, -2.521029964131e+05, -2.520385907581e+05, -2.519740148581e+05, -2.519092697021e+05, -2.518443562711e+05, -2.517792755381e+05, -2.517140284651e+05, -2.516486160091e+05, -2.515830391181e+05, -2.515172987311e+05, -2.514513957801e+05, -2.513853311891e+05, -2.513191058751e+05, -2.512527207461e+05, -2.511861767041e+05, -2.511194746421e+05, -2.510526154471e+05, -2.509855999981e+05, -2.509184291671e+05, -2.508511038191e+05, -2.507836248111e+05, -2.507159929951e+05, -2.506482092131e+05, -2.505802743041e+05, -2.505121890981e+05, -2.504439544181e+05, -2.503755710811e+05, -2.503070398981e+05, -2.502383616731e+05, -2.501695372031e+05, -2.501005672811e+05, -2.500314526901e+05, -2.499621942111e+05, -2.498927926151e+05, -2.498232486701e+05, -2.497535631361e+05, -2.496837367681e+05, -2.496137703141e+05, -2.495436645191e+05, -2.494734201191e+05, -2.494030378471e+05, -2.493325184271e+05, -2.492618625811e+05, -2.491910710231e+05, -2.491201444631e+05, -2.490490836051e+05, -2.489778891481e+05, -2.489065617851e+05, -2.488351022041e+05, -2.487635110891e+05, -2.486917891171e+05, -2.486199369611e+05, -2.485479552891e+05, -2.484758447631e+05, -2.484036060411e+05, -2.483312397771e+05, -2.482587466181e+05, -2.481861272071e+05, -2.481133821841e+05, -2.480405121821e+05, -2.479675178301e+05, -2.478943997531e+05, -2.478211585711e+05, -2.477477949001e+05, -2.476743093511e+05, -2.476007025291e+05, -2.475269750391e+05, -2.474531274761e+05, -2.473791604361e+05, -2.473050745081e+05, -2.472308702761e+05, -2.471565483221e+05, -2.470821092221e+05, -2.470075535501e+05, -2.469328818741e+05, -2.468580947581e+05, -2.467831927631e+05, -2.467081764471e+05, -2.466330463621e+05, -2.465578030561e+05, -2.464824470751e+05, -2.464069789611e+05, -2.463313992511e+05, -2.462557084781e+05, -2.461799071731e+05, -2.461039958621e+05, -2.460279750691e+05, -2.459518453121e+05, -2.458756071071e+05, -2.457992609671e+05, -2.457228073991e+05, -2.456462469101e+05, -2.455695800011e+05, -2.454928071701e+05, -2.454159289121e+05, -2.453389457201e+05, -2.452618580811e+05, -2.451846664801e+05, -2.451073714001e+05, -2.450299733181e+05, -2.449524727111e+05, -2.448748700511e+05, -2.447971658061e+05, -2.447193604431e+05, -2.446414544251e+05, -2.445634482111e+05, -2.444853422591e+05, -2.444071370221e+05, -2.443288329511e+05, -2.442504304941e+05, -2.441719300961e+05, -2.440933321991e+05, -2.440146372421e+05, -2.439358456621e+05, -2.438569578921e+05, -2.437779743641e+05, -2.436988955041e+05, -2.436197217391e+05, -2.435404534901e+05, -2.434610911781e+05, -2.433816352201e+05, -2.433020860301e+05, -2.432224440201e+05, -2.431427095991e+05, -2.430628831741e+05, -2.429829651501e+05, -2.429029559261e+05, -2.428228559031e+05, -2.427426654771e+05, -2.426623850421e+05, -2.425820149891e+05, -2.425015557091e+05, -2.424210075861e+05, -2.423403710061e+05, -2.422596463501e+05, -2.421788339981e+05, -2.420979343281e+05, -2.420169477131e+05, -2.419358745271e+05, -2.418547151401e+05, -2.417734699201e+05, -2.416921392331e+05, -2.416107234421e+05, -2.415292229081e+05, -2.414476379911e+05, -2.413659690471e+05, -2.412842164321e+05, -2.412023804971e+05, -2.411204615951e+05, -2.410384600721e+05, -2.409563762751e+05, -2.408742105501e+05, -2.407919632361e+05, -2.407096346761e+05, -2.406272252071e+05, -2.405447351641e+05, -2.404621648831e+05, -2.403795146961e+05},
    {1.494233352987e+04, 1.293773704887e+04, 1.089296418787e+04, 8.805599875871e+03, 6.672984322870e+03, 4.492176886871e+03, 2.259912771871e+03, -2.744924112945e+01, -2.373990394129e+03, -4.784310055130e+03, -7.263622560130e+03, -9.817878573129e+03, -1.245391900913e+04, -1.517967283313e+04, -1.800441501013e+04, -2.093910854613e+04, -2.399686661113e+04, -2.719359052313e+04, -3.054887269613e+04, -3.408731238213e+04, -3.784050041813e+04, -4.185014098113e+04, -4.617322218813e+04, -5.089116210213e+04, -5.612745532713e+04, -6.208611520013e+04, -6.915242163013e+04, -2.365590129721e+05, -2.374606629301e+05, -2.382857249881e+05, -2.390467891921e+05, -2.397533775571e+05, -2.404129016941e+05, -2.410312647801e+05, -2.416132582691e+05, -2.421628333951e+05, -2.426832928401e+05, -2.431774296121e+05, -2.436476299051e+05, -2.440959507791e+05, -2.445241797871e+05, -2.449338814431e+05, -2.453264339121e+05, -2.457030583101e+05, -2.460648423681e+05, -2.464127597221e+05, -2.467476857751e+05, -2.470704108501e+05, -2.473816511721e+05, -2.476820581091e+05, -2.479722259891e+05, -2.482526987481e+05, -2.485239756371e+05, -2.487865161081e+05, -2.490407440631e+05, -2.492870515201e+05, -2.495258018271e+05, -2.497573324631e+05, -2.499819575071e+05, -2.501999698131e+05, -2.504116429331e+05, -2.506172328191e+05, -2.508169793521e+05, -2.510111076901e+05, -2.511998294851e+05, -2.513833439741e+05, -2.515618389571e+05, -2.517354916801e+05, -2.519044696381e+05, -2.520689312921e+05, -2.522290267301e+05, -2.523848982631e+05, -2.525366809681e+05, -2.526845031861e+05, -2.528284869711e+05, -2.529687485141e+05, -2.531053985151e+05, -2.532385425421e+05, -2.533682813461e+05, -2.534947111631e+05, -2.536179239851e+05, -2.537380078161e+05, -2.538550469031e+05, -2.539691219581e+05, -2.540803103521e+05, -2.541886863111e+05, -2.542943210831e+05, -2.543972831011e+05, -2.544976381371e+05, -2.545954494391e+05, -2.546907778641e+05, -2.547836820031e+05, -2.548742182891e+05, -2.549624411131e+05, -2.550484029171e+05, -2.551321542971e+05, -2.552137440821e+05, -2.552932194251e+05, -2.553706258791e+05, -2.554460074691e+05, -2.555194067661e+05, -2.555908649471e+05, -2.556604218601e+05, -2.557281160841e+05, -2.557939849811e+05, -2.558580647481e+05, -2.559203904681e+05, -2.559809961591e+05, -2.560399148121e+05, -2.560971784391e+05, -2.561528181071e+05, -2.562068639831e+05, -2.562593453621e+05, -2.563102907071e+05, -2.563597276771e+05, -2.564076831631e+05, -2.564541833111e+05, -2.564992535541e+05, -2.565429186381e+05, -2.565852026451e+05, -2.566261290221e+05, -2.566657205981e+05, -2.567039996101e+05, -2.567409877221e+05, -2.567767060481e+05, -2.568111751681e+05, -2.568444151461e+05, -2.568764455501e+05, -2.569072854681e+05, -2.569369535231e+05, -2.569654678891e+05, -2.569928463071e+05, -2.570191060961e+05, -2.570442641701e+05, -2.570683370491e+05, -2.570913408701e+05, -2.571132914051e+05, -2.571342040641e+05, -2.571540939121e+05, -2.571729756801e+05, -2.571908637691e+05, -2.572077722681e+05, -2.572237149571e+05, -2.572387053191e+05, -2.572527565481e+05, -2.572658815591e+05, -2.572780929921e+05, -2.572894032261e+05, -2.572998243791e+05, -2.573093683231e+05, -2.573180466861e+05, -2.573258708581e+05, -2.573328520031e+05, -2.573390010581e+05, -2.573443287471e+05, -2.573488455801e+05, -2.573525618611e+05, -2.573554876961e+05, -2.573576329951e+05, -2.573590074771e+05, -2.573596206781e+05, -2.573594819541e+05, -2.573586004841e+05, -2.573569852761e+05, -2.573546451741e+05, -2.573515888571e+05, -2.573478248451e+05, -2.573433615061e+05, -2.573382070561e+05, -2.573323695641e+05, -2.573258569551e+05, -2.573186770141e+05, -2.573108373911e+05, -2.573023456021e+05, -2.572932090301e+05, -2.572834349351e+05, -2.572730304501e+05, -2.572620025881e+05, -2.572503582431e+05, -2.572381041931e+05, -2.572252471031e+05, -2.572117935271e+05, -2.571977499111e+05, -2.571831225971e+05, -2.571679178211e+05, -2.571521417191e+05, -2.571358003291e+05, -2.571188995911e+05, -2.571014453501e+05, -2.570834433611e+05, -2.570648992861e+05, -2.570458186981e+05, -2.570262070861e+05, -2.570060698511e+05, -2.569854123131e+05, -2.569642397091e+05, -2.569425571981e+05, -2.569203698581e+05, -2.568976826941e+05, -2.568745006341e+05, -2.568508285331e+05, -2.568266711741e+05, -2.568020332691e+05, -2.567769194611e+05, -2.567513343271e+05, -2.567252823761e+05, -2.566987680511e+05, -2.566717957331e+05, -2.566443697401e+05, -2.566164943291e+05, -2.565881736941e+05, -2.565594119731e+05, -2.565302132471e+05, -2.565005815361e+05, -2.564705208071e+05, -2.564400349731e+05, -2.564091278911e+05, -2.563778033661e+05, -2.563460651541e+05, -2.563139169551e+05, -2.562813624251e+05, -2.562484051661e+05, -2.562150487341e+05, -2.561812966401e+05, -2.561471523441e+05, -2.561126192641e+05, -2.560777007731e+05, -2.560424001981e+05, -2.560067208261e+05, -2.559706658971e+05, -2.559342386151e+05, -2.558974421381e+05, -2.558602795871e+05, -2.558227540421e+05, -2.557848685441e+05, -2.557466260981e+05, -2.557080296681e+05, -2.556690821841e+05, -2.556297865371e+05, -2.555901455861e+05, -2.555501621501e+05, -2.555098390191e+05, -2.554691789441e+05, -2.554281846461e+05, -2.553868588111e+05, -2.553452040951e+05, -2.553032231191e+05, -2.552609184771e+05, -2.552182927271e+05, -2.551753484021e+05, -2.551320880011e+05, -2.550885139951e+05, -2.550446288291e+05, -2.550004349141e+05, -2.549559346381e+05, -2.549111303601e+05, -2.548660244101e+05, -2.548206190931e+05, -2.547749166881e+05, -2.547289194471e+05, -2.546826295991e+05, -2.546360493431e+05, -2.545891808591e+05, -2.545420262991e+05, -2.544945877921e+05, -2.544468674441e+05, -2.543988673361e+05, -2.543505895281e+05, -2.543020360571e+05, -2.542532089371e+05, -2.542041101611e+05, -2.541547416991e+05, -2.541051055021e+05, -2.540552034981e+05, -2.540050375941e+05, -2.539546096791e+05, -2.539039216211e+05, -2.538529752671e+05, -2.538017724441e+05, -2.537503149641e+05, -2.536986046151e+05, -2.536466431681e+05, -2.535944323781e+05, -2.535419739781e+05, -2.534892696861e+05, -2.534363212011e+05, -2.533831302041e+05, -2.533296983611e+05, -2.532760273181e+05, -2.532221187071e+05, -2.531679741421e+05, -2.531135952221e+05, -2.530589835291e+05, -2.530041406281e+05, -2.529490680711e+05, -2.528937673921e+05, -2.528382401131e+05, -2.527824877381e+05, -2.527265117571e+05, -2.526703136461e+05, -2.526138948661e+05, -2.525572568641e+05, -2.525004010741e+05, -2.524433289141e+05, -2.523860417901e+05, -2.523285410941e+05, -2.522708282051e+05, -2.522129044871e+05, -2.521547712941e+05, -2.520964299651e+05, -2.520378818271e+05, -2.519791281951e+05, -2.519201703711e+05, -2.518610096451e+05, -2.518016472951e+05, -2.517420845871e+05, -2.516823227751e+05, -2.516223631021e+05, -2.515622068001e+05, -2.515018550891e+05, -2.514413091761e+05, -2.513805702621e+05, -2.513196395311e+05, -2.512585181611e+05, -2.511972073171e+05, -2.511357081531e+05, -2.510740218161e+05, -2.510121494391e+05, -2.509500921471e+05, -2.508878510531e+05, -2.508254272631e+05, -2.507628218711e+05, -2.507000359621e+05, -2.506370706121e+05, -2.505739268871e+05, -2.505106058431e+05, -2.504471085281e+05, -2.503834359801e+05, -2.503195892301e+05, -2.502555692971e+05, -2.501913771941e+05, -2.501270139231e+05, -2.500624804791e+05, -2.499977778481e+05, -2.499329070081e+05, -2.498678689281e+05, -2.498026645701e+05, -2.497372948851e+05, -2.496717608211e+05, -2.496060633131e+05, -2.495402032911e+05, -2.494741816761e+05, -2.494079993831e+05, -2.493416573181e+05, -2.492751563791e+05, -2.492084974581e+05, -2.491416814391e+05, -2.490747092001e+05, -2.490075816081e+05, -2.489402995291e+05, -2.488728638161e+05, -2.488052753181e+05, -2.487375348781e+05, -2.486696433301e+05, -2.486016015031e+05, -2.485334102181e+05, -2.484650702901e+05, -2.483965825291e+05, -2.483279477361e+05, -2.482591667061e+05, -2.481902402311e+05, -2.481211690921e+05, -2.480519540671e+05, -2.479825959271e+05, -2.479130954371e+05, -2.478434533561e+05, -2.477736704381e+05, -2.477037474281e+05, -2.476336850701e+05, -2.475634840991e+05, -2.474931452441e+05, -2.474226692311e+05, -2.473520567781e+05, -2.472813085991e+05, -2.472104254021e+05, -2.471394078901e+05, -2.470682567591e+05, -2.469969727021e+05, -2.469255564061e+05, -2.468540085531e+05, -2.467823298181e+05, -2.467105208741e+05, -2.466385823861e+05, -2.465665150171e+05, -2.464943194241e+05, -2.464219962571e+05, -2.463495461631e+05, -2.462769697861e+05, -2.462042677621e+05, -2.461314407241e+05, -2.460584893001e+05, -2.459854141141e+05, -2.459122157841e+05, -2.458388949261e+05, -2.457654521501e+05, -2.456918880601e+05, -2.456182032581e+05, -2.455443983421e+05, -2.454704739031e+05, -2.453964305311e+05, -2.453222688101e+05, -2.452479893191e+05, -2.451735926351e+05, -2.450990793291e+05, -2.450244499691e+05, -2.449497051201e+05, -2.448748453411e+05, -2.447998711871e+05, -2.447247832121e+05, -2.446495819621e+05, -2.445742679831e+05, -2.444988418161e+05, -2.444233039961e+05, -2.443476550571e+05, -2.442718955281e+05, -2.441960259351e+05, -2.441200468011e+05, -2.440439586441e+05, -2.439677619791e+05, -2.438914573171e+05, -2.438150451661e+05, -2.437385260321e+05, -2.436619004151e+05, -2.435851688131e+05, -2.435083317201e+05, -2.434313896281e+05, -2.433543430241e+05, -2.432771923931e+05, -2.431999382151e+05, -2.431225809711e+05, -2.430451211331e+05, -2.429675591731e+05, -2.428898955611e+05, -2.428121307621e+05, -2.427342652391e+05, -2.426562994501e+05, -2.425782338511e+05, -2.425000688981e+05, -2.424218050391e+05, -2.423434427221e+05, -2.422649823931e+05, -2.421864244921e+05, -2.421077694581e+05, -2.420290177271e+05, -2.419501697331e+05, -2.418712259061e+05, -2.417921866731e+05, -2.417130524591e+05, -2.416338236871e+05, -2.415545007761e+05, -2.414750841421e+05, -2.413955742001e+05, -2.413159713621e+05, -2.412362760351e+05, -2.411564886271e+05, -2.410766095411e+05, -2.409966391791e+05, -2.409165779381e+05, -2.408364262161e+05, -2.407561844061e+05, -2.406758528991e+05, -2.405954320841e+05, -2.405149223471e+05, -2.404343240731e+05, -2.403536376431e+05, -2.402728634371e+05, -2.401920018301e+05, -2.401110531991e+05, -2.400300179151e+05, -2.399488963481e+05, -2.398676888661e+05, -2.397863958351e+05, -2.397050176181e+05, -2.396235545751e+05, -2.395420070671e+05, -2.394603754491e+05, -2.393786600771e+05, -2.392968613021e+05, -2.392149794761e+05, -2.391330149461e+05, -2.390509680581e+05, -2.389688391571e+05, -2.388866285841e+05, -2.388043366801e+05, -2.387219637831e+05},
    {1.580396596187e+04, 1.381531535187e+04, 1.178740076287e+04, 9.717905613870e+03, 7.604282812871e+03, 5.443721383871e+03, 3.233106628871e+03, 9.689722158705e+02, -1.352557883130e+03, -3.735840385130e+03, -6.185800271129e+03, -8.708039972129e+03, -1.130897700213e+04, -1.399601966913e+04, -1.677779451013e+04, -1.966444534713e+04, -2.266803348613e+04, -2.580308417513e+04, -2.908735019113e+04, -3.254290794213e+04, -3.619778148213e+04, -4.008844168713e+04, -4.426383444213e+04, -4.879226049813e+04, -5.377404009413e+04, -5.936731378013e+04, -6.584891029313e+04, -7.379645216813e+04, -2.336701397921e+05, -2.346387338491e+05, -2.355205786011e+05, -2.363307235251e+05, -2.370803687811e+05, -2.377781203821e+05, -2.384307605121e+05, -2.390437453921e+05, -2.396215405461e+05, -2.401678542571e+05, -2.406858047381e+05, -2.411780426981e+05, -2.416468430621e+05, -2.420941748311e+05, -2.425217551361e+05, -2.429310916461e+05, -2.433235162481e+05, -2.437002121031e+05, -2.440622356021e+05, -2.444105343391e+05, -2.447459619571e+05, -2.450692905041e+05, -2.453812207871e+05, -2.456823911051e+05, -2.459733846681e+05, -2.462547359221e+05, -2.465269359841e+05, -2.467904373211e+05, -2.470456578131e+05, -2.472929842781e+05, -2.475327755601e+05, -2.477653652321e+05, -2.479910639801e+05, -2.482101616991e+05, -2.484229293631e+05, -2.486296206761e+05, -2.488304735511e+05, -2.490257114251e+05, -2.492155444471e+05, -2.494001705361e+05, -2.495797763361e+05, -2.497545380841e+05, -2.499246223901e+05, -2.500901869411e+05, -2.502513811491e+05, -2.504083467351e+05, -2.505612182611e+05, -2.507101236201e+05, -2.508551844791e+05, -2.509965166931e+05, -2.511342306741e+05, -2.512684317431e+05, -2.513992204391e+05, -2.515266928211e+05, -2.516509407341e+05, -2.517720520561e+05, -2.518901109371e+05, -2.520051980071e+05, -2.521173905791e+05, -2.522267628331e+05, -2.523333859851e+05, -2.524373284521e+05, -2.525386560001e+05, -2.526374318821e+05, -2.527337169671e+05, -2.528275698671e+05, -2.529190470451e+05, -2.530082029241e+05, -2.530950899901e+05, -2.531797588801e+05, -2.532622584751e+05, -2.533426359811e+05, -2.534209370071e+05, -2.534972056381e+05, -2.535714845051e+05, -2.536438148511e+05, -2.537142365901e+05, -2.537827883681e+05, -2.538495076131e+05, -2.539144305951e+05, -2.539775924681e+05, -2.540390273181e+05, -2.540987682111e+05, -2.541568472281e+05, -2.542132955111e+05, -2.542681432981e+05, -2.543214199571e+05, -2.543731540231e+05, -2.544233732291e+05, -2.544721045341e+05, -2.545193741581e+05, -2.545652076061e+05, -2.546096296921e+05, -2.546526645721e+05, -2.546943357581e+05, -2.547346661521e+05, -2.547736780571e+05, -2.548113932071e+05, -2.548478327821e+05, -2.548830174281e+05, -2.549169672751e+05, -2.549497019571e+05, -2.549812406261e+05, -2.550116019691e+05, -2.550408042221e+05, -2.550688651881e+05, -2.550958022481e+05, -2.551216323761e+05, -2.551463721531e+05, -2.551700377741e+05, -2.551926450701e+05, -2.552142095081e+05, -2.552347462111e+05, -2.552542699641e+05, -2.552727952251e+05, -2.552903361371e+05, -2.553069065341e+05, -2.553225199511e+05, -2.553371896371e+05, -2.553509285551e+05, -2.553637494001e+05, -2.553756645971e+05, -2.553866863171e+05, -2.553968264791e+05, -2.554060967591e+05, -2.554145085951e+05, -2.554220731981e+05, -2.554288015521e+05, -2.554347044251e+05, -2.554397923721e+05, -2.554440757421e+05, -2.554475646841e+05, -2.554502691501e+05, -2.554521989041e+05, -2.554533635221e+05, -2.554537724011e+05, -2.554534347621e+05, -2.554523596521e+05, -2.554505559551e+05, -2.554480323871e+05, -2.554447975101e+05, -2.554408597271e+05, -2.554362272921e+05, -2.554309083111e+05, -2.554249107451e+05, -2.554182424161e+05, -2.554109110081e+05, -2.554029240711e+05, -2.553942890251e+05, -2.553850131621e+05, -2.553751036491e+05, -2.553645675311e+05, -2.553534117351e+05, -2.553416430711e+05, -2.553292682351e+05, -2.553162938141e+05, -2.553027262841e+05, -2.552885720161e+05, -2.552738372771e+05, -2.552585282331e+05, -2.552426509491e+05, -2.552262113961e+05, -2.552092154471e+05, -2.551916688841e+05, -2.551735773961e+05, -2.551549465861e+05, -2.551357819671e+05, -2.551160889681e+05, -2.550958729351e+05, -2.550751391301e+05, -2.550538927381e+05, -2.550321388631e+05, -2.550098825341e+05, -2.549871287041e+05, -2.549638822501e+05, -2.549401479821e+05, -2.549159306331e+05, -2.548912348721e+05, -2.548660652971e+05, -2.548404264391e+05, -2.548143227651e+05, -2.547877586781e+05, -2.547607385171e+05, -2.547332665601e+05, -2.547053470251e+05, -2.546769840691e+05, -2.546481817921e+05, -2.546189442381e+05, -2.545892753931e+05, -2.545591791891e+05, -2.545286595041e+05, -2.544977201631e+05, -2.544663649381e+05, -2.544345975521e+05, -2.544024216771e+05, -2.543698409351e+05, -2.543368589001e+05, -2.543034791011e+05, -2.542697050161e+05, -2.542355400811e+05, -2.542009876861e+05, -2.541660511761e+05, -2.541307338531e+05, -2.540950389781e+05, -2.540589697671e+05, -2.540225293971e+05, -2.539857210051e+05, -2.539485476871e+05, -2.539110125011e+05, -2.538731184651e+05, -2.538348685611e+05, -2.537962657321e+05, -2.537573128881e+05, -2.537180128991e+05, -2.536783686021e+05, -2.536383827981e+05, -2.535980582571e+05, -2.535573977101e+05, -2.535164038611e+05, -2.534750793771e+05, -2.534334268961e+05, -2.533914490211e+05, -2.533491483291e+05, -2.533065273621e+05, -2.532635886351e+05, -2.532203346321e+05, -2.531767678081e+05, -2.531328905911e+05, -2.530887053791e+05, -2.530442145431e+05, -2.529994204281e+05, -2.529543253501e+05, -2.529089315991e+05, -2.528632414411e+05, -2.528172571151e+05, -2.527709808341e+05, -2.527244147881e+05, -2.526775611401e+05, -2.526304220321e+05, -2.525829995801e+05, -2.525352958771e+05, -2.524873129931e+05, -2.524390529771e+05, -2.523905178541e+05, -2.523417096261e+05, -2.522926302761e+05, -2.522432817631e+05, -2.521936660261e+05, -2.521437849851e+05, -2.520936405361e+05, -2.520432345571e+05, -2.519925689061e+05, -2.519416454211e+05, -2.518904659211e+05, -2.518390322051e+05, -2.517873460551e+05, -2.517354092341e+05, -2.516832234841e+05, -2.516307905321e+05, -2.515781120881e+05, -2.515251898411e+05, -2.514720254651e+05, -2.514186206171e+05, -2.513649769361e+05, -2.513110960461e+05, -2.512569795541e+05, -2.512026290491e+05, -2.511480461081e+05, -2.510932322891e+05, -2.510381891351e+05, -2.509829181761e+05, -2.509274209231e+05, -2.508716988761e+05, -2.508157535191e+05, -2.507595863191e+05, -2.507031987331e+05, -2.506465922021e+05, -2.505897681511e+05, -2.505327279941e+05, -2.504754731301e+05, -2.504180049461e+05, -2.503603248151e+05, -2.503024340961e+05, -2.502443341371e+05, -2.501860262711e+05, -2.501275118201e+05, -2.500687920941e+05, -2.500098683891e+05, -2.499507419911e+05, -2.498914141721e+05, -2.498318861941e+05, -2.497721593061e+05, -2.497122347461e+05, -2.496521137411e+05, -2.495917975061e+05, -2.495312872461e+05, -2.494705841541e+05, -2.494096894131e+05, -2.493486041941e+05, -2.492873296601e+05, -2.492258669601e+05, -2.491642172361e+05, -2.491023816181e+05, -2.490403612261e+05, -2.489781571711e+05, -2.489157705541e+05, -2.488532024651e+05, -2.487904539871e+05, -2.487275261891e+05, -2.486644201361e+05, -2.486011368811e+05, -2.485376774671e+05, -2.484740429301e+05, -2.484102342961e+05, -2.483462525821e+05, -2.482820987971e+05, -2.482177739401e+05, -2.481532790041e+05, -2.480886149711e+05, -2.480237828151e+05, -2.479587835041e+05, -2.478936179961e+05, -2.478282872401e+05, -2.477627921791e+05, -2.476971337471e+05, -2.476313128711e+05, -2.475653304711e+05, -2.474991874561e+05, -2.474328847311e+05, -2.473664231921e+05, -2.472998037281e+05, -2.472330272211e+05, -2.471660945451e+05, -2.470990065671e+05, -2.470317641481e+05, -2.469643681401e+05, -2.468968193911e+05, -2.468291187401e+05, -2.467612670201e+05, -2.466932650561e+05, -2.466251136691e+05, -2.465568136711e+05, -2.464883658701e+05, -2.464197710641e+05, -2.463510300481e+05, -2.462821436101e+05, -2.462131125301e+05, -2.461439375851e+05, -2.460746195421e+05, -2.460051591661e+05, -2.459355572141e+05, -2.458658144371e+05, -2.457959315811e+05, -2.457259093841e+05, -2.456557485821e+05, -2.455854499031e+05, -2.455150140701e+05, -2.454444418001e+05, -2.453737338051e+05, -2.453028907921e+05, -2.452319134611e+05, -2.451608025081e+05, -2.450895586251e+05, -2.450181824951e+05, -2.449466748011e+05, -2.448750362161e+05, -2.448032674111e+05, -2.447313690511e+05, -2.446593417961e+05, -2.445871863011e+05, -2.445149032181e+05, -2.444424931911e+05, -2.443699568611e+05, -2.442972948651e+05, -2.442245078341e+05, -2.441515963951e+05, -2.440785611711e+05, -2.440054027791e+05, -2.439321218321e+05, -2.438587189411e+05, -2.437851947081e+05, -2.437115497351e+05, -2.436377846171e+05, -2.435638999461e+05, -2.434898963091e+05, -2.434157742901e+05, -2.433415344671e+05, -2.432671774161e+05, -2.431927037081e+05, -2.431181139081e+05, -2.430434085821e+05, -2.429685882861e+05, -2.428936535771e+05, -2.428186050051e+05, -2.427434431171e+05, -2.426681684581e+05, -2.425927815671e+05, -2.425172829801e+05, -2.424416732291e+05, -2.423659528431e+05, -2.422901223461e+05, -2.422141822611e+05, -2.421381331051e+05, -2.420619753921e+05, -2.419857096341e+05, -2.419093363371e+05, -2.418328560051e+05, -2.417562691391e+05, -2.416795762351e+05, -2.416027777891e+05, -2.415258742891e+05, -2.414488662231e+05, -2.413717540761e+05, -2.412945383271e+05, -2.412172194541e+05, -2.411397979321e+05, -2.410622742311e+05, -2.409846488201e+05, -2.409069221641e+05, -2.408290947231e+05, -2.407511669581e+05, -2.406731393241e+05, -2.405950122741e+05, -2.405167862581e+05, -2.404384617221e+05, -2.403600391111e+05, -2.402815188661e+05, -2.402029014261e+05, -2.401241872251e+05, -2.400453766961e+05, -2.399664702701e+05, -2.398874683741e+05, -2.398083714311e+05, -2.397291798651e+05, -2.396498940921e+05, -2.395705145311e+05, -2.394910415951e+05, -2.394114756941e+05, -2.393318172381e+05, -2.392520666321e+05, -2.391722242791e+05, -2.390922905811e+05, -2.390122659361e+05, -2.389321507391e+05, -2.388519453841e+05, -2.387716502631e+05, -2.386912657621e+05, -2.386107922691e+05, -2.385302301671e+05, -2.384495798371e+05, -2.383688416591e+05, -2.382880160091e+05, -2.382071032611e+05, -2.381261037881e+05, -2.380450179581e+05, -2.379638461411e+05, -2.378825887001e+05, -2.378012459991e+05, -2.377198183991e+05, -2.376383062591e+05, -2.375567099341e+05, -2.374750297801e+05, -2.373932661481e+05, -2.373114193891e+05, -2.372294898511e+05, -2.371474778791e+05, -2.370653838181e+05},
    {1.666660014787e+04, 1.469368432587e+04, 1.268239252887e+04, 1.063050223087e+04, 8.535573720871e+03, 6.394919290870e+03, 4.205566557871e+03, 1.964214534871e+03, -3.328194121295e+02, -2.689664120129e+03, -5.110973905129e+03, -7.602026870130e+03, -1.016884812213e+04, -1.281836606813e+04, -1.555861328113e+04, -1.839898848713e+04, -2.135060398913e+04, -2.442675511313e+04, -2.764356833813e+04, -3.102091863013e+04, -3.458376609513e+04, -3.836417196813e+04, -4.240446851813e+04, -4.676250623013e+04, -5.152092505713e+04, -5.680500664713e+04, -6.282141542413e+04, -6.995892546113e+04, -2.295848188161e+05, -2.307501693971e+05, -2.317913171801e+05, -2.327340763111e+05, -2.335964621931e+05, -2.343916543871e+05, -2.351296429601e+05, -2.358182139941e+05, -2.364635738331e+05, -2.370707624701e+05, -2.376439373721e+05, -2.381865742661e+05, -2.387016128191e+05, -2.391915646501e+05, -2.396585949541e+05, -2.401045851871e+05, -2.405311819471e+05, -2.409398355621e+05, -2.413318309341e+05, -2.417083124501e+05, -2.420703042851e+05, -2.424187271131e+05, -2.427544119581e+05, -2.430781117651e+05, -2.433905111351e+05, -2.436922345601e+05, -2.439838534421e+05, -2.442658920901e+05, -2.445388328911e+05, -2.448031207711e+05, -2.450591670751e+05, -2.453073529461e+05, -2.455480322841e+05, -2.457815343521e+05, -2.460081660611e+05, -2.462282140041e+05, -2.464419462531e+05, -2.466496139641e+05, -2.468514528061e+05, -2.470476842381e+05, -2.472385166661e+05, -2.474241464661e+05, -2.476047589241e+05, -2.477805290731e+05, -2.479516224561e+05, -2.481181958211e+05, -2.482803977461e+05, -2.484383692141e+05, -2.485922441361e+05, -2.487421498281e+05, -2.488882074491e+05, -2.490305324051e+05, -2.491692347141e+05, -2.493044193461e+05, -2.494361865391e+05, -2.495646320831e+05, -2.496898475901e+05, -2.498119207391e+05, -2.499309355021e+05, -2.500469723621e+05, -2.501601085011e+05, -2.502704179931e+05, -2.503779719631e+05, -2.504828387541e+05, -2.505850840721e+05, -2.506847711221e+05, -2.507819607391e+05, -2.508767115081e+05, -2.509690798781e+05, -2.510591202651e+05, -2.511468851531e+05, -2.512324251881e+05, -2.513157892641e+05, -2.513970246041e+05, -2.514761768411e+05, -2.515532900891e+05, -2.516284070101e+05, -2.517015688801e+05, -2.517728156541e+05, -2.518421860161e+05, -2.519097174401e+05, -2.519754462401e+05, -2.520394076171e+05, -2.521016357061e+05, -2.521621636231e+05, -2.522210235021e+05, -2.522782465361e+05, -2.523338630151e+05, -2.523879023641e+05, -2.524403931711e+05, -2.524913632231e+05, -2.525408395381e+05, -2.525888483891e+05, -2.526354153371e+05, -2.526805652541e+05, -2.527243223491e+05, -2.527667101931e+05, -2.528077517421e+05, -2.528474693571e+05, -2.528858848261e+05, -2.529230193841e+05, -2.529588937321e+05, -2.529935280571e+05, -2.530269420451e+05, -2.530591549031e+05, -2.530901853711e+05, -2.531200517391e+05, -2.531487718631e+05, -2.531763631761e+05, -2.532028427041e+05, -2.532282270781e+05, -2.532525325461e+05, -2.532757749881e+05, -2.532979699211e+05, -2.533191325171e+05, -2.533392776111e+05, -2.533584197091e+05, -2.533765730001e+05, -2.533937513671e+05, -2.534099683911e+05, -2.534252373671e+05, -2.534395713041e+05, -2.534529829411e+05, -2.534654847491e+05, -2.534770889421e+05, -2.534878074821e+05, -2.534976520881e+05, -2.535066342421e+05, -2.535147651951e+05, -2.535220559731e+05, -2.535285173841e+05, -2.535341600241e+05, -2.535389942811e+05, -2.535430303451e+05, -2.535462782061e+05, -2.535487476651e+05, -2.535504483371e+05, -2.535513896551e+05, -2.535515808781e+05, -2.535510310901e+05, -2.535497492081e+05, -2.535477439871e+05, -2.535450240221e+05, -2.535415977501e+05, -2.535374734601e+05, -2.535326592911e+05, -2.535271632381e+05, -2.535209931551e+05, -2.535141567591e+05, -2.535066616311e+05, -2.534985152221e+05, -2.534897248571e+05, -2.534802977321e+05, -2.534702409231e+05, -2.534595613871e+05, -2.534482659631e+05, -2.534363613781e+05, -2.534238542441e+05, -2.534107510691e+05, -2.533970582501e+05, -2.533827820821e+05, -2.533679287581e+05, -2.533525043721e+05, -2.533365149191e+05, -2.533199662991e+05, -2.533028643211e+05, -2.532852146991e+05, -2.532670230611e+05, -2.532482949451e+05, -2.532290358061e+05, -2.532092510121e+05, -2.531889458511e+05, -2.531681255301e+05, -2.531467951791e+05, -2.531249598471e+05, -2.531026245121e+05, -2.530797940741e+05, -2.530564733631e+05, -2.530326671371e+05, -2.530083800841e+05, -2.529836168241e+05, -2.529583819111e+05, -2.529326798321e+05, -2.529065150111e+05, -2.528798918061e+05, -2.528528145171e+05, -2.528252873811e+05, -2.527973145751e+05, -2.527689002191e+05, -2.527400483751e+05, -2.527107630501e+05, -2.526810481931e+05, -2.526509077001e+05, -2.526203454171e+05, -2.525893651331e+05, -2.525579705881e+05, -2.525261654721e+05, -2.524939534261e+05, -2.524613380401e+05, -2.524283228601e+05, -2.523949113821e+05, -2.523611070571e+05, -2.523269132931e+05, -2.522923334511e+05, -2.522573708491e+05, -2.522220287631e+05, -2.521863104261e+05, -2.521502190311e+05, -2.521137577301e+05, -2.520769296341e+05, -2.520397378151e+05, -2.520021853081e+05, -2.519642751081e+05, -2.519260101741e+05, -2.518873934291e+05, -2.518484277581e+05, -2.518091160131e+05, -2.517694610081e+05, -2.517294655251e+05, -2.516891323121e+05, -2.516484640841e+05, -2.516074635231e+05, -2.515661332781e+05, -2.515244759691e+05, -2.514824941811e+05, -2.514401904721e+05, -2.513975673681e+05, -2.513546273671e+05, -2.513113729361e+05, -2.512678065141e+05, -2.512239305121e+05, -2.511797473131e+05, -2.511352592731e+05, -2.510904687211e+05, -2.510453779581e+05, -2.509999892621e+05, -2.509543048821e+05, -2.509083270431e+05, -2.508620579461e+05, -2.508154997661e+05, -2.507686546541e+05, -2.507215247381e+05, -2.506741121221e+05, -2.506264188871e+05, -2.505784470901e+05, -2.505301987681e+05, -2.504816759331e+05, -2.504328805781e+05, -2.503838146721e+05, -2.503344801651e+05, -2.502848789841e+05, -2.502350130381e+05, -2.501848842131e+05, -2.501344943771e+05, -2.500838453771e+05, -2.500329390411e+05, -2.499817771791e+05, -2.499303615811e+05, -2.498786940171e+05, -2.498267762421e+05, -2.497746099891e+05, -2.497221969771e+05, -2.496695389051e+05, -2.496166374561e+05, -2.495634942931e+05, -2.495101110661e+05, -2.494564894061e+05, -2.494026309291e+05, -2.493485372321e+05, -2.492942099001e+05, -2.492396504981e+05, -2.491848605801e+05, -2.491298416801e+05, -2.490745953211e+05, -2.490191230081e+05, -2.489634262321e+05, -2.489075064711e+05, -2.488513651871e+05, -2.487950038271e+05, -2.487384238271e+05, -2.486816266071e+05, -2.486246135731e+05, -2.485673861191e+05, -2.485099456251e+05, -2.484522934581e+05, -2.483944309711e+05, -2.483363595071e+05, -2.482780803931e+05, -2.482195949461e+05, -2.481609044691e+05, -2.481020102541e+05, -2.480429135801e+05, -2.479836157161e+05, -2.479241179181e+05, -2.478644214291e+05, -2.478045274841e+05, -2.477444373031e+05, -2.476841520971e+05, -2.476236730671e+05, -2.475630014001e+05, -2.475021382761e+05, -2.474410848611e+05, -2.473798423121e+05, -2.473184117761e+05, -2.472567943891e+05, -2.471949912781e+05, -2.471330035591e+05, -2.470708323381e+05, -2.470084787111e+05, -2.469459437671e+05, -2.468832285821e+05, -2.468203342251e+05, -2.467572617541e+05, -2.466940122181e+05, -2.466305866591e+05, -2.465669861081e+05, -2.465032115881e+05, -2.464392641121e+05, -2.463751446851e+05, -2.463108543051e+05, -2.462463939591e+05, -2.461817646281e+05, -2.461169672821e+05, -2.460520028851e+05, -2.459868723921e+05, -2.459215767501e+05, -2.458561168991e+05, -2.457904937701e+05, -2.457247082871e+05, -2.456587613661e+05, -2.455926539151e+05, -2.455263868351e+05, -2.454599610211e+05, -2.453933773571e+05, -2.453266367241e+05, -2.452597399921e+05, -2.451926880271e+05, -2.451254816871e+05, -2.450581218231e+05, -2.449906092771e+05, -2.449229448891e+05, -2.448551294871e+05, -2.447871638961e+05, -2.447190489341e+05, -2.446507854101e+05, -2.445823741301e+05, -2.445138158911e+05, -2.444451114851e+05, -2.443762616981e+05, -2.443072673081e+05, -2.442381290901e+05, -2.441688478101e+05, -2.440994242291e+05, -2.440298591031e+05, -2.439601531811e+05, -2.438903072071e+05, -2.438203219191e+05, -2.437501980491e+05, -2.436799363241e+05, -2.436095374651e+05, -2.435390021871e+05, -2.434683312011e+05, -2.433975252121e+05, -2.433265849191e+05, -2.432555110151e+05, -2.431843041921e+05, -2.431129651311e+05, -2.430414945121e+05, -2.429698930081e+05, -2.428981612891e+05, -2.428263000171e+05, -2.427543098511e+05, -2.426821914451e+05, -2.426099454491e+05, -2.425375725071e+05, -2.424650732571e+05, -2.423924483351e+05, -2.423196983721e+05, -2.422468239921e+05, -2.421738258171e+05, -2.421007044641e+05, -2.420274605451e+05, -2.419540946661e+05, -2.418806074331e+05, -2.418069994431e+05, -2.417332712921e+05, -2.416594235701e+05, -2.415854568631e+05, -2.415113717541e+05, -2.414371688201e+05, -2.413628486351e+05, -2.412884117701e+05, -2.412138587891e+05, -2.411391902561e+05, -2.410644067271e+05, -2.409895087581e+05, -2.409144968971e+05, -2.408393716921e+05, -2.407641336851e+05, -2.406887834151e+05, -2.406133214171e+05, -2.405377482221e+05, -2.404620643581e+05, -2.403862703491e+05, -2.403103667161e+05, -2.402343539761e+05, -2.401582326421e+05, -2.400820032251e+05, -2.400056662301e+05, -2.399292221601e+05, -2.398526715171e+05, -2.397760147951e+05, -2.396992524881e+05, -2.396223850851e+05, -2.395454130741e+05, -2.394683369361e+05, -2.393911571531e+05, -2.393138742001e+05, -2.392364885521e+05, -2.391590006791e+05, -2.390814110481e+05, -2.390037201241e+05, -2.389259283681e+05, -2.388480362371e+05, -2.387700441881e+05, -2.386919526731e+05, -2.386137621401e+05, -2.385354730361e+05, -2.384570858051e+05, -2.383786008881e+05, -2.383000187221e+05, -2.382213397411e+05, -2.381425643791e+05, -2.380636930651e+05, -2.379847262251e+05, -2.379056642831e+05, -2.378265076601e+05, -2.377472567751e+05, -2.376679120441e+05, -2.375884738801e+05, -2.375089426941e+05, -2.374293188941e+05, -2.373496028851e+05, -2.372697950691e+05, -2.371898958491e+05, -2.371099056211e+05, -2.370298247811e+05, -2.369496537211e+05, -2.368693928331e+05, -2.367890425041e+05, -2.367086031211e+05, -2.366280750661e+05, -2.365474587201e+05, -2.364667544631e+05, -2.363859626691e+05, -2.363050837141e+05, -2.362241179691e+05, -2.361430658031e+05, -2.360619275841e+05, -2.359807036771e+05, -2.358993944431e+05, -2.358180002441e+05, -2.357365214391e+05, -2.356549583831e+05, -2.355733114301e+05, -2.354915809321e+05, -2.354097672401e+05},
    {1.753023570987e+04, 1.557284761687e+04, 1.357794796187e+04, 1.154340405887e+04, 9.466878557871e+03, 7.345800970870e+03, 5.177333914871e+03, 2.958332802870e+03, 6.852973668705e+02, -1.645687081129e+03, -4.039021464130e+03, -6.499681539129e+03, -9.033328319129e+03, -1.164644728913e+04, -1.434652607313e+04, -1.714228439713e+04, -2.004397637613e+04, -2.306379494013e+04, -2.621642385813e+04, -2.951980875613e+04, -3.299626319813e+04, -3.667410615913e+04, -4.059017912713e+04, -4.479389693013e+04, -4.935415097213e+04, -5.437197477013e+04, -6.000621076913e+04, -6.653350945913e+04, -7.452456132113e+04, -2.265360997741e+05, -2.277990846291e+05, -2.289187203221e+05, -2.299266714851e+05, -2.308445164511e+05, -2.316877588441e+05, -2.324679844321e+05, -2.331941196651e+05, -2.338732125801e+05, -2.345109411851e+05, -2.351119575181e+05, -2.356801280551e+05, -2.362187062971e+05, -2.367304595571e+05, -2.372177639671e+05, -2.376826769501e+05, -2.381269933351e+05, -2.385522894291e+05, -2.389599580541e+05, -2.393512366991e+05, -2.397272303921e+05, -2.400889304321e+05, -2.404372298701e+05, -2.407729363971e+05, -2.410967831601e+05, -2.414094378791e+05, -2.417115105941e+05, -2.420035602891e+05, -2.422861005641e+05, -2.425596045351e+05, -2.428245090861e+05, -2.430812185591e+05, -2.433301079851e+05, -2.435715259201e+05, -2.438057969331e+05, -2.440332238151e+05, -2.442540895301e+05, -2.444686589471e+05, -2.446771803931e+05, -2.448798870291e+05, -2.450769980971e+05, -2.452687200291e+05, -2.454552474521e+05, -2.456367640931e+05, -2.458134435961e+05, -2.459854502711e+05, -2.461529397601e+05, -2.463160596601e+05, -2.464749500751e+05, -2.466297441311e+05, -2.467805684461e+05, -2.469275435541e+05, -2.470707843001e+05, -2.472104002061e+05, -2.473464957981e+05, -2.474791709171e+05, -2.476085210011e+05, -2.477346373471e+05, -2.478576073551e+05, -2.479775147521e+05, -2.480944398011e+05, -2.482084594921e+05, -2.483196477281e+05, -2.484280754861e+05, -2.485338109811e+05, -2.486369198041e+05, -2.487374650641e+05, -2.488355075131e+05, -2.489311056671e+05, -2.490243159141e+05, -2.491151926231e+05, -2.492037882401e+05, -2.492901533801e+05, -2.493743369151e+05, -2.494563860521e+05, -2.495363464151e+05, -2.496142621151e+05, -2.496901758171e+05, -2.497641288031e+05, -2.498361610381e+05, -2.499063112221e+05, -2.499746168481e+05, -2.500411142491e+05, -2.501058386531e+05, -2.501688242211e+05, -2.502301040981e+05, -2.502897104481e+05, -2.503476744981e+05, -2.504040265751e+05, -2.504587961361e+05, -2.505120118091e+05, -2.505637014191e+05, -2.506138920221e+05, -2.506626099341e+05, -2.507098807541e+05, -2.507557293981e+05, -2.508001801171e+05, -2.508432565251e+05, -2.508849816191e+05, -2.509253778051e+05, -2.509644669141e+05, -2.510022702251e+05, -2.510388084841e+05, -2.510741019211e+05, -2.511081702651e+05, -2.511410327681e+05, -2.511727082141e+05, -2.512032149371e+05, -2.512325708351e+05, -2.512607933851e+05, -2.512878996571e+05, -2.513139063251e+05, -2.513388296801e+05, -2.513626856421e+05, -2.513854897741e+05, -2.514072572881e+05, -2.514280030601e+05, -2.514477416381e+05, -2.514664872521e+05, -2.514842538261e+05, -2.515010549821e+05, -2.515169040521e+05, -2.515318140871e+05, -2.515457978651e+05, -2.515588678951e+05, -2.515710364311e+05, -2.515823154711e+05, -2.515927167731e+05, -2.516022518561e+05, -2.516109320081e+05, -2.516187682901e+05, -2.516257715491e+05, -2.516319524141e+05, -2.516373213111e+05, -2.516418884621e+05, -2.516456638931e+05, -2.516486574401e+05, -2.516508787511e+05, -2.516523372921e+05, -2.516530423551e+05, -2.516530030571e+05, -2.516522283491e+05, -2.516507270151e+05, -2.516485076811e+05, -2.516455788191e+05, -2.516419487461e+05, -2.516376256331e+05, -2.516326175031e+05, -2.516269322411e+05, -2.516205775931e+05, -2.516135611701e+05, -2.516058904511e+05, -2.515975727881e+05, -2.515886154071e+05, -2.515790254111e+05, -2.515688097841e+05, -2.515579753921e+05, -2.515465289891e+05, -2.515344772141e+05, -2.515218265991e+05, -2.515085835691e+05, -2.514947544431e+05, -2.514803454401e+05, -2.514653626781e+05, -2.514498121771e+05, -2.514336998621e+05, -2.514170315641e+05, -2.513998130231e+05, -2.513820498881e+05, -2.513637477221e+05, -2.513449120011e+05, -2.513255481171e+05, -2.513056613811e+05, -2.512852570211e+05, -2.512643401871e+05, -2.512429159531e+05, -2.512209893151e+05, -2.511985651961e+05, -2.511756484471e+05, -2.511522438451e+05, -2.511283560991e+05, -2.511039898501e+05, -2.510791496711e+05, -2.510538400681e+05, -2.510280654831e+05, -2.510018302981e+05, -2.509751388271e+05, -2.509479953271e+05, -2.509204039961e+05, -2.508923689691e+05, -2.508638943281e+05, -2.508349840961e+05, -2.508056422411e+05, -2.507758726781e+05, -2.507456792661e+05, -2.507150658141e+05, -2.506840360801e+05, -2.506525937681e+05, -2.506207425361e+05, -2.505884859911e+05, -2.505558276941e+05, -2.505227711591e+05, -2.504893198521e+05, -2.504554771941e+05, -2.504212465651e+05, -2.503866312961e+05, -2.503516346791e+05, -2.503162599621e+05, -2.502805103521e+05, -2.502443890151e+05, -2.502078990781e+05, -2.501710436271e+05, -2.501338257101e+05, -2.500962483371e+05, -2.500583144811e+05, -2.500200270761e+05, -2.499813890241e+05, -2.499424031881e+05, -2.499030723961e+05, -2.498633994441e+05, -2.498233870921e+05, -2.497830380671e+05, -2.497423550651e+05, -2.497013407471e+05, -2.496599977441e+05, -2.496183286551e+05, -2.495763360511e+05, -2.495340224681e+05, -2.494913904161e+05, -2.494484423741e+05, -2.494051807931e+05, -2.493616080961e+05, -2.493177266771e+05, -2.492735389031e+05, -2.492290471131e+05, -2.491842536221e+05, -2.491391607151e+05, -2.490937706551e+05, -2.490480856771e+05, -2.490021079921e+05, -2.489558397861e+05, -2.489092832201e+05, -2.488624404321e+05, -2.488153135361e+05, -2.487679046241e+05, -2.487202157621e+05, -2.486722489981e+05, -2.486240063531e+05, -2.485754898301e+05, -2.485267014081e+05, -2.484776430451e+05, -2.484283166791e+05, -2.483787242271e+05, -2.483288675841e+05, -2.482787486291e+05, -2.482283692171e+05, -2.481777311851e+05, -2.481268363511e+05, -2.480756865141e+05, -2.480242834541e+05, -2.479726289331e+05, -2.479207246941e+05, -2.478685724631e+05, -2.478161739491e+05, -2.477635308411e+05, -2.477106448121e+05, -2.476575175201e+05, -2.476041506031e+05, -2.475505456861e+05, -2.474967043741e+05, -2.474426282581e+05, -2.473883189131e+05, -2.473337778991e+05, -2.472790067591e+05, -2.472240070231e+05, -2.471687802021e+05, -2.471133277971e+05, -2.470576512911e+05, -2.470017521551e+05, -2.469456318431e+05, -2.468892917961e+05, -2.468327334431e+05, -2.467759581981e+05, -2.467189674591e+05, -2.466617626151e+05, -2.466043450391e+05, -2.465467160911e+05, -2.464888771211e+05, -2.464308294621e+05, -2.463725744371e+05, -2.463141133571e+05, -2.462554475201e+05, -2.461965782111e+05, -2.461375067061e+05, -2.460782342651e+05, -2.460187621411e+05, -2.459590915721e+05, -2.458992237851e+05, -2.458391599991e+05, -2.457789014181e+05, -2.457184492371e+05, -2.456578046401e+05, -2.455969688011e+05, -2.455359428811e+05, -2.454747280351e+05, -2.454133254031e+05, -2.453517361171e+05, -2.452899613001e+05, -2.452280020631e+05, -2.451658595081e+05, -2.451035347291e+05, -2.450410288081e+05, -2.449783428171e+05, -2.449154778231e+05, -2.448524348791e+05, -2.447892150311e+05, -2.447258193151e+05, -2.446622487611e+05, -2.445985043861e+05, -2.445345872011e+05, -2.444704982071e+05, -2.444062383981e+05, -2.443418087581e+05, -2.442772102651e+05, -2.442124438851e+05, -2.441475105791e+05, -2.440824112991e+05, -2.440171469891e+05, -2.439517185851e+05, -2.438861270161e+05, -2.438203732021e+05, -2.437544580561e+05, -2.436883824831e+05, -2.436221473831e+05, -2.435557536451e+05, -2.434892021541e+05, -2.434224937841e+05, -2.433556294061e+05, -2.432886098821e+05, -2.432214360671e+05, -2.431541088091e+05, -2.430866289491e+05, -2.430189973221e+05, -2.429512147571e+05, -2.428832820741e+05, -2.428152000881e+05, -2.427469696091e+05, -2.426785914371e+05, -2.426100663691e+05, -2.425413951951e+05, -2.424725786961e+05, -2.424036176521e+05, -2.423345128311e+05, -2.422652650011e+05, -2.421958749191e+05, -2.421263433401e+05, -2.420566710101e+05, -2.419868586721e+05, -2.419169070601e+05, -2.418468169071e+05, -2.417765889351e+05, -2.417062238661e+05, -2.416357224121e+05, -2.415650852821e+05, -2.414943131781e+05, -2.414234068001e+05, -2.413523668381e+05, -2.412811939811e+05, -2.412098889111e+05, -2.411384523041e+05, -2.410668848321e+05, -2.409951871641e+05, -2.409233599601e+05, -2.408514038781e+05, -2.407793195701e+05, -2.407071076841e+05, -2.406347688631e+05, -2.405623037441e+05, -2.404897129621e+05, -2.404169971441e+05, -2.403441569161e+05, -2.402711928961e+05, -2.401981057011e+05, -2.401248959411e+05, -2.400515642221e+05, -2.399781111471e+05, -2.399045373131e+05, -2.398308433141e+05, -2.397570297391e+05, -2.396830971731e+05, -2.396090461981e+05, -2.395348773891e+05, -2.394605913201e+05, -2.393861885591e+05, -2.393116696721e+05, -2.392370352181e+05, -2.391622857561e+05, -2.390874218371e+05, -2.390124440111e+05, -2.389373528231e+05, -2.388621488151e+05, -2.387868325251e+05, -2.387114044871e+05, -2.386358652301e+05, -2.385602152831e+05, -2.384844551681e+05, -2.384085854051e+05, -2.383326065091e+05, -2.382565189941e+05, -2.381803233691e+05, -2.381040201391e+05, -2.380276098061e+05, -2.379510928691e+05, -2.378744698251e+05, -2.377977411641e+05, -2.377209073761e+05, -2.376439689471e+05, -2.375669263591e+05, -2.374897800911e+05, -2.374125306191e+05, -2.373351784161e+05, -2.372577239521e+05, -2.371801676931e+05, -2.371025101031e+05, -2.370247516431e+05, -2.369468927701e+05, -2.368689339391e+05, -2.367908756011e+05, -2.367127182041e+05, -2.366344621961e+05, -2.365561080181e+05, -2.364776561101e+05, -2.363991069101e+05, -2.363204608531e+05, -2.362417183691e+05, -2.361628798871e+05, -2.360839458341e+05, -2.360049166321e+05, -2.359257927041e+05, -2.358465744661e+05, -2.357672623331e+05, -2.356878567201e+05, -2.356083580351e+05, -2.355287666861e+05, -2.354490830791e+05, -2.353693076151e+05, -2.352894406951e+05, -2.352094827161e+05, -2.351294340731e+05, -2.350492951591e+05, -2.349690663631e+05, -2.348887480731e+05, -2.348083406751e+05, -2.347278445521e+05, -2.346472600831e+05, -2.345665876471e+05, -2.344858276211e+05, -2.344049803771e+05, -2.343240462861e+05, -2.342430257191e+05, -2.341619190421e+05, -2.340807266191e+05, -2.339994488121e+05, -2.339180859831e+05, -2.338366384881e+05, -2.337551066841e+05},
    {1.839487221387e+04, 1.645280871087e+04, 1.447407523787e+04, 1.245662495287e+04, 1.039821810187e+04, 8.296395721871e+03, 6.148448537870e+03, 3.951379974870e+03, 1.701861835871e+03, -6.038191911295e+02, -2.969826622129e+03, -5.400854094129e+03, -7.902224437130e+03, -1.048001381113e+04, -1.314120910513e+04, -1.589391014113e+04, -1.874759326513e+04, -2.171346068013e+04, -2.480491213413e+04, -2.803819554513e+04, -3.143332685413e+04, -3.501542846013e+04, -3.881674431313e+04, -4.287980100213e+04, -4.726262401213e+04, -5.204791426113e+04, -5.736060592713e+04, -6.340559069213e+04, -7.056410529813e+04, -7.972575409713e+04, -2.234472439941e+05, -2.248172683351e+05, -2.260216032851e+05, -2.270991211351e+05, -2.280756577491e+05, -2.289694299431e+05, -2.297938522571e+05, -2.305591386061e+05, -2.312732756051e+05, -2.319426454231e+05, -2.325724412261e+05, -2.331669539081e+05, -2.337297757291e+05, -2.342639485251e+05, -2.347720738771e+05, -2.352563965221e+05, -2.357188685491e+05, -2.361611995151e+05, -2.365848960831e+05, -2.369912937331e+05, -2.373815824121e+05, -2.377568274591e+05, -2.381179868481e+05, -2.384659254871e+05, -2.388014271861e+05, -2.391252047271e+05, -2.394379083871e+05, -2.397401332111e+05, -2.400324252371e+05, -2.403152868531e+05, -2.405891814341e+05, -2.408545373691e+05, -2.411117515771e+05, -2.413611925831e+05, -2.416032032251e+05, -2.418381030371e+05, -2.420661903661e+05, -2.422877442431e+05, -2.425030260561e+05, -2.427122810361e+05, -2.429157395971e+05, -2.431136185301e+05, -2.433061220831e+05, -2.434934429301e+05, -2.436757630541e+05, -2.438532545381e+05, -2.440260802901e+05, -2.441943946981e+05, -2.443583442301e+05, -2.445180679781e+05, -2.446736981591e+05, -2.448253605691e+05, -2.449731750061e+05, -2.451172556541e+05, -2.452577114341e+05, -2.453946463371e+05, -2.455281597171e+05, -2.456583465791e+05, -2.457852978261e+05, -2.459091005061e+05, -2.460298380281e+05, -2.461475903701e+05, -2.462624342661e+05, -2.463744433901e+05, -2.464836885151e+05, -2.465902376691e+05, -2.466941562831e+05, -2.467955073191e+05, -2.468943514011e+05, -2.469907469291e+05, -2.470847501921e+05, -2.471764154701e+05, -2.472657951341e+05, -2.473529397301e+05, -2.474378980741e+05, -2.475207173251e+05, -2.476014430681e+05, -2.476801193781e+05, -2.477567888931e+05, -2.478314928771e+05, -2.479042712771e+05, -2.479751627841e+05, -2.480442048831e+05, -2.481114339081e+05, -2.481768850871e+05, -2.482405925871e+05, -2.483025895611e+05, -2.483629081861e+05, -2.484215797031e+05, -2.484786344521e+05, -2.485341019121e+05, -2.485880107291e+05, -2.486403887531e+05, -2.486912630621e+05, -2.487406599961e+05, -2.487886051841e+05, -2.488351235671e+05, -2.488802394241e+05, -2.489239764011e+05, -2.489663575251e+05, -2.490074052321e+05, -2.490471413851e+05, -2.490855872971e+05, -2.491227637451e+05, -2.491586909931e+05, -2.491933888041e+05, -2.492268764641e+05, -2.492591727911e+05, -2.492902961521e+05, -2.493202644831e+05, -2.493490952931e+05, -2.493768056881e+05, -2.494034123751e+05, -2.494289316821e+05, -2.494533795631e+05, -2.494767716151e+05, -2.494991230861e+05, -2.495204488861e+05, -2.495407635981e+05, -2.495600814871e+05, -2.495784165081e+05, -2.495957823201e+05, -2.496121922881e+05, -2.496276594971e+05, -2.496421967581e+05, -2.496558166141e+05, -2.496685313501e+05, -2.496803530001e+05, -2.496912933531e+05, -2.497013639591e+05, -2.497105761391e+05, -2.497189409871e+05, -2.497264693801e+05, -2.497331719801e+05, -2.497390592421e+05, -2.497441414191e+05, -2.497484285691e+05, -2.497519305561e+05, -2.497546570591e+05, -2.497566175751e+05, -2.497578214231e+05, -2.497582777501e+05, -2.497579955341e+05, -2.497569835911e+05, -2.497552505731e+05, -2.497528049801e+05, -2.497496551561e+05, -2.497458093001e+05, -2.497412754631e+05, -2.497360615551e+05, -2.497301753491e+05, -2.497236244821e+05, -2.497164164601e+05, -2.497085586601e+05, -2.497000583331e+05, -2.496909226081e+05, -2.496811584921e+05, -2.496707728771e+05, -2.496597725411e+05, -2.496481641471e+05, -2.496359542501e+05, -2.496231492991e+05, -2.496097556371e+05, -2.495957795051e+05, -2.495812270441e+05, -2.495661042971e+05, -2.495504172101e+05, -2.495341716371e+05, -2.495173733391e+05, -2.495000279871e+05, -2.494821411651e+05, -2.494637183711e+05, -2.494447650161e+05, -2.494252864331e+05, -2.494052878691e+05, -2.493847744961e+05, -2.493637514061e+05, -2.493422236151e+05, -2.493201960671e+05, -2.492976736301e+05, -2.492746611021e+05, -2.492511632091e+05, -2.492271846121e+05, -2.492027299011e+05, -2.491778036021e+05, -2.491524101751e+05, -2.491265540161e+05, -2.491002394621e+05, -2.490734707841e+05, -2.490462521971e+05, -2.490185878551e+05, -2.489904818551e+05, -2.489619382381e+05, -2.489329609891e+05, -2.489035540361e+05, -2.488737212581e+05, -2.488434664791e+05, -2.488127934701e+05, -2.487817059531e+05, -2.487502076021e+05, -2.487183020381e+05, -2.486859928381e+05, -2.486532835291e+05, -2.486201775941e+05, -2.485866784681e+05, -2.485527895441e+05, -2.485185141701e+05, -2.484838556491e+05, -2.484488172461e+05, -2.484134021801e+05, -2.483776136311e+05, -2.483414547401e+05, -2.483049286061e+05, -2.482680382901e+05, -2.482307868161e+05, -2.481931771701e+05, -2.481552122991e+05, -2.481168951161e+05, -2.480782284971e+05, -2.480392152851e+05, -2.479998582861e+05, -2.479601602721e+05, -2.479201239841e+05, -2.478797521271e+05, -2.478390473771e+05, -2.477980123751e+05, -2.477566497331e+05, -2.477149620311e+05, -2.476729518181e+05, -2.476306216151e+05, -2.475879739131e+05, -2.475450111731e+05, -2.475017358291e+05, -2.474581502851e+05, -2.474142569201e+05, -2.473700580841e+05, -2.473255561011e+05, -2.472807532671e+05, -2.472356518551e+05, -2.471902541111e+05, -2.471445622551e+05, -2.470985784831e+05, -2.470523049671e+05, -2.470057438541e+05, -2.469588972681e+05, -2.469117673101e+05, -2.468643560571e+05, -2.468166655641e+05, -2.467686978631e+05, -2.467204549651e+05, -2.466719388591e+05, -2.466231515131e+05, -2.465740948721e+05, -2.465247708631e+05, -2.464751813901e+05, -2.464253283391e+05, -2.463752135751e+05, -2.463248389441e+05, -2.462742062721e+05, -2.462233173651e+05, -2.461721740141e+05, -2.461207779871e+05, -2.460691310361e+05, -2.460172348951e+05, -2.459650912801e+05, -2.459127018891e+05, -2.458600684031e+05, -2.458071924871e+05, -2.457540757881e+05, -2.457007199361e+05, -2.456471265461e+05, -2.455932972161e+05, -2.455392335291e+05, -2.454849370501e+05, -2.454304093321e+05, -2.453756519091e+05, -2.453206663031e+05, -2.452654540191e+05, -2.452100165491e+05, -2.451543553701e+05, -2.450984719421e+05, -2.450423677161e+05, -2.449860441251e+05, -2.449295025891e+05, -2.448727445151e+05, -2.448157712981e+05, -2.447585843161e+05, -2.447011849381e+05, -2.446435745181e+05, -2.445857543971e+05, -2.445277259041e+05, -2.444694903551e+05, -2.444110490561e+05, -2.443524032971e+05, -2.442935543601e+05, -2.442345035131e+05, -2.441752520111e+05, -2.441158011021e+05, -2.440561520171e+05, -2.439963059811e+05, -2.439362642031e+05, -2.438760278861e+05, -2.438155982181e+05, -2.437549763781e+05, -2.436941635351e+05, -2.436331608461e+05, -2.435719694601e+05, -2.435105905131e+05, -2.434490251321e+05, -2.433872744361e+05, -2.433253395321e+05, -2.432632215161e+05, -2.432009214791e+05, -2.431384404971e+05, -2.430757796401e+05, -2.430129399691e+05, -2.429499225341e+05, -2.428867283771e+05, -2.428233585311e+05, -2.427598140191e+05, -2.426960958571e+05, -2.426322050511e+05, -2.425681425991e+05, -2.425039094911e+05, -2.424395067071e+05, -2.423749352221e+05, -2.423101959981e+05, -2.422452899931e+05, -2.421802181561e+05, -2.421149814271e+05, -2.420495807391e+05, -2.419840170181e+05, -2.419182911801e+05, -2.418524041351e+05, -2.417863567871e+05, -2.417201500301e+05, -2.416537847521e+05, -2.415872618341e+05, -2.415205821491e+05, -2.414537465621e+05, -2.413867559351e+05, -2.413196111181e+05, -2.412523129581e+05, -2.411848622931e+05, -2.411172599561e+05, -2.410495067711e+05, -2.409816035581e+05, -2.409135511281e+05, -2.408453502891e+05, -2.407770018391e+05, -2.407085065731e+05, -2.406398652761e+05, -2.405710787301e+05, -2.405021477101e+05, -2.404330729841e+05, -2.403638553161e+05, -2.402944954621e+05, -2.402249941741e+05, -2.401553521961e+05, -2.400855702691e+05, -2.400156491271e+05, -2.399455894971e+05, -2.398753921031e+05, -2.398050576611e+05, -2.397345868841e+05, -2.396639804781e+05, -2.395932391451e+05, -2.395223635801e+05, -2.394513544731e+05, -2.393802125111e+05, -2.393089383741e+05, -2.392375327371e+05, -2.391659962701e+05, -2.390943296381e+05, -2.390225335031e+05, -2.389506085191e+05, -2.388785553381e+05, -2.388063746051e+05, -2.387340669611e+05, -2.386616330441e+05, -2.385890734851e+05, -2.385163889101e+05, -2.384435799441e+05, -2.383706472051e+05, -2.382975913051e+05, -2.382244128551e+05, -2.381511124601e+05, -2.380776907201e+05, -2.380041482321e+05, -2.379304855881e+05, -2.378567033761e+05, -2.377828021801e+05, -2.377087825791e+05, -2.376346451491e+05, -2.375603904611e+05, -2.374860190841e+05, -2.374115315801e+05, -2.373369285091e+05, -2.372622104271e+05, -2.371873778861e+05, -2.371124314341e+05, -2.370373716151e+05, -2.369621989691e+05, -2.368869140331e+05, -2.368115173411e+05, -2.367360094211e+05, -2.366603908001e+05, -2.365846619991e+05, -2.365088235371e+05, -2.364328759301e+05, -2.363568196891e+05, -2.362806553211e+05, -2.362043833331e+05, -2.361280042251e+05, -2.360515184941e+05, -2.359749266371e+05, -2.358982291431e+05, -2.358214265021e+05, -2.357445191971e+05, -2.356675077101e+05, -2.355903925211e+05, -2.355131741031e+05, -2.354358529291e+05, -2.353584294691e+05, -2.352809041871e+05, -2.352032775461e+05, -2.351255500071e+05, -2.350477220271e+05, -2.349697940581e+05, -2.348917665531e+05, -2.348136399581e+05, -2.347354147201e+05, -2.346570912791e+05, -2.345786700761e+05, -2.345001515471e+05, -2.344215361261e+05, -2.343428242431e+05, -2.342640163271e+05, -2.341851128031e+05, -2.341061140941e+05, -2.340270206211e+05, -2.339478327991e+05, -2.338685510451e+05, -2.337891757701e+05, -2.337097073841e+05, -2.336301462941e+05, -2.335504929051e+05, -2.334707476171e+05, -2.333909108311e+05, -2.333109829441e+05, -2.332309643501e+05, -2.331508554401e+05, -2.330706566051e+05, -2.329903682321e+05, -2.329099907061e+05, -2.328295244081e+05, -2.327489697191e+05, -2.326683270171e+05, -2.325875966771e+05, -2.325067790721e+05, -2.324258745731e+05, -2.323448835491e+05, -2.322638063661e+05, -2.321826433891e+05, -2.321013949781e+05},
    {1.926050917687e+04, 1.733357094187e+04, 1.537078224787e+04, 1.337017829687e+04, 1.132961242987e+04, 9.246731816871e+03, 7.118948808871e+03, 4.943406973871e+03, 2.716940553871e+03, 4.360257348705e+02, -1.903278401129e+03, -4.305402017129e+03, -6.775353488129e+03, -9.318830235129e+03, -1.194235836213e+04, -1.465347075113e+04, -1.746093739113e+04, -2.037506787013e+04, -2.340811564113e+04, -2.657482910313e+04, -2.989322011413e+04, -3.338566436413e+04, -3.708052653613e+04, -4.101465100013e+04, -4.523735454913e+04, -4.981719543913e+04, -5.485430365713e+04, -6.050510626413e+04, -6.703916410513e+04, -7.500106158213e+04, -2.185585250131e+05, -2.203184524271e+05, -2.218055568761e+05, -2.231010112921e+05, -2.242525075751e+05, -2.252909336511e+05, -2.262376516971e+05, -2.271081576131e+05, -2.279141088761e+05, -2.286645316211e+05, -2.293665799591e+05, -2.300260352141e+05, -2.306476462471e+05, -2.312353686221e+05, -2.317925370571e+05, -2.323219925801e+05, -2.328261781181e+05, -2.333072115961e+05, -2.337669426861e+05, -2.342069974601e+05, -2.346288139601e+05, -2.350336708391e+05, -2.354227106521e+05, -2.357969589741e+05, -2.361573402141e+05, -2.365046908051e+05, -2.368397702791e+05, -2.371632706201e+05, -2.374758242181e+05, -2.377780106751e+05, -2.380703626411e+05, -2.383533708631e+05, -2.386274885601e+05, -2.388931352361e+05, -2.391507000121e+05, -2.394005445501e+05, -2.396430056371e+05, -2.398783974591e+05, -2.401070136211e+05, -2.403291289461e+05, -2.405450010761e+05, -2.407548719081e+05, -2.409589688751e+05, -2.411575061061e+05, -2.413506854611e+05, -2.415386974761e+05, -2.417217222111e+05, -2.418999300231e+05, -2.420734822631e+05, -2.422425319181e+05, -2.424072241921e+05, -2.425676970351e+05, -2.427240816301e+05, -2.428765028431e+05, -2.430250796261e+05, -2.431699253981e+05, -2.433111483891e+05, -2.434488519601e+05, -2.435831348981e+05, -2.437140916901e+05, -2.438418127721e+05, -2.439663847651e+05, -2.440878906961e+05, -2.442064101901e+05, -2.443220196671e+05, -2.444347925121e+05, -2.445447992391e+05, -2.446521076401e+05, -2.447567829311e+05, -2.448588878841e+05, -2.449584829481e+05, -2.450556263651e+05, -2.451503742851e+05, -2.452427808601e+05, -2.453328983461e+05, -2.454207771901e+05, -2.455064661151e+05, -2.455900122011e+05, -2.456714609601e+05, -2.457508564051e+05, -2.458282411201e+05, -2.459036563191e+05, -2.459771419101e+05, -2.460487365471e+05, -2.461184776871e+05, -2.461864016381e+05, -2.462525436081e+05, -2.463169377511e+05, -2.463796172051e+05, -2.464406141411e+05, -2.464999597941e+05, -2.465576845051e+05, -2.466138177531e+05, -2.466683881881e+05, -2.467214236651e+05, -2.467729512711e+05, -2.468229973581e+05, -2.468715875661e+05, -2.469187468491e+05, -2.469644995041e+05, -2.470088691921e+05, -2.470518789581e+05, -2.470935512581e+05, -2.471339079751e+05, -2.471729704441e+05, -2.472107594631e+05, -2.472472953191e+05, -2.472825978001e+05, -2.473166862151e+05, -2.473495794061e+05, -2.473812957681e+05, -2.474118532601e+05, -2.474412694201e+05, -2.474695613771e+05, -2.474967458681e+05, -2.475228392461e+05, -2.475478574941e+05, -2.475718162341e+05, -2.475947307441e+05, -2.476166159611e+05, -2.476374864951e+05, -2.476573566391e+05, -2.476762403781e+05, -2.476941513961e+05, -2.477111030881e+05, -2.477271085671e+05, -2.477421806701e+05, -2.477563319681e+05, -2.477695747761e+05, -2.477819211531e+05, -2.477933829171e+05, -2.478039716461e+05, -2.478136986871e+05, -2.478225751621e+05, -2.478306119741e+05, -2.478378198121e+05, -2.478442091591e+05, -2.478497902941e+05, -2.478545733011e+05, -2.478585680701e+05, -2.478617843071e+05, -2.478642315331e+05, -2.478659190951e+05, -2.478668561641e+05, -2.478670517441e+05, -2.478665146751e+05, -2.478652536351e+05, -2.478632771471e+05, -2.478605935821e+05, -2.478572111601e+05, -2.478531379591e+05, -2.478483819111e+05, -2.478429508141e+05, -2.478368523271e+05, -2.478300939791e+05, -2.478226831711e+05, -2.478146271771e+05, -2.478059331461e+05, -2.477966081111e+05, -2.477866589851e+05, -2.477760925651e+05, -2.477649155391e+05, -2.477531344821e+05, -2.477407558651e+05, -2.477277860511e+05, -2.477142313031e+05, -2.477000977821e+05, -2.476853915511e+05, -2.476701185771e+05, -2.476542847341e+05, -2.476378958021e+05, -2.476209574731e+05, -2.476034753491e+05, -2.475854549471e+05, -2.475669017001e+05, -2.475478209551e+05, -2.475282179821e+05, -2.475080979691e+05, -2.474874660271e+05, -2.474663271911e+05, -2.474446864221e+05, -2.474225486041e+05, -2.473999185551e+05, -2.473768010181e+05, -2.473532006701e+05, -2.473291221191e+05, -2.473045699071e+05, -2.472795485111e+05, -2.472540623451e+05, -2.472281157591e+05, -2.472017130431e+05, -2.471748584261e+05, -2.471475560801e+05, -2.471198101161e+05, -2.470916245901e+05, -2.470630035031e+05, -2.470339507991e+05, -2.470044703711e+05, -2.469745660571e+05, -2.469442416451e+05, -2.469135008711e+05, -2.468823474211e+05, -2.468507849331e+05, -2.468188169961e+05, -2.467864471531e+05, -2.467536788991e+05, -2.467205156841e+05, -2.466869609141e+05, -2.466530179491e+05, -2.466186901091e+05, -2.465839806681e+05, -2.465488928601e+05, -2.465134298781e+05, -2.464775948741e+05, -2.464413909621e+05, -2.464048212151e+05, -2.463678886671e+05, -2.463305963181e+05, -2.462929471281e+05, -2.462549440211e+05, -2.462165898851e+05, -2.461778875731e+05, -2.461388399051e+05, -2.460994496641e+05, -2.460597196011e+05, -2.460196524351e+05, -2.459792508501e+05, -2.459385175011e+05, -2.458974550081e+05, -2.458560659641e+05, -2.458143529281e+05, -2.457723184321e+05, -2.457299649771e+05, -2.456872950351e+05, -2.456443110491e+05, -2.456010154351e+05, -2.455574105801e+05, -2.455134988461e+05, -2.454692825661e+05, -2.454247640471e+05, -2.453799455701e+05, -2.453348293911e+05, -2.452894177401e+05, -2.452437128231e+05, -2.451977168211e+05, -2.451514318891e+05, -2.451048601621e+05, -2.450580037491e+05, -2.450108647361e+05, -2.449634451871e+05, -2.449157471441e+05, -2.448677726261e+05, -2.448195236301e+05, -2.447710021331e+05, -2.447222100891e+05, -2.446731494321e+05, -2.446238220771e+05, -2.445742299161e+05, -2.445243748231e+05, -2.444742586511e+05, -2.444238832361e+05, -2.443732503921e+05, -2.443223619151e+05, -2.442712195841e+05, -2.442198251571e+05, -2.441681803771e+05, -2.441162869661e+05, -2.440641466311e+05, -2.440117610601e+05, -2.439591319251e+05, -2.439062608811e+05, -2.438531495651e+05, -2.437997995991e+05, -2.437462125881e+05, -2.436923901221e+05, -2.436383337741e+05, -2.435840451031e+05, -2.435295256511e+05, -2.434747769461e+05, -2.434198005001e+05, -2.433645978111e+05, -2.433091703631e+05, -2.432535196241e+05, -2.431976470501e+05, -2.431415540801e+05, -2.430852421421e+05, -2.430287126491e+05, -2.429719670011e+05, -2.429150065851e+05, -2.428578327741e+05, -2.428004469271e+05, -2.427428503941e+05, -2.426850445091e+05, -2.426270305941e+05, -2.425688099601e+05, -2.425103839041e+05, -2.424517537131e+05, -2.423929206611e+05, -2.423338860111e+05, -2.422746510121e+05, -2.422152169061e+05, -2.421555849191e+05, -2.420957562691e+05, -2.420357321621e+05, -2.419755137921e+05, -2.419151023451e+05, -2.418544989941e+05, -2.417937049021e+05, -2.417327212221e+05, -2.416715490971e+05, -2.416101896591e+05, -2.415486440301e+05, -2.414869133241e+05, -2.414249986421e+05, -2.413629010791e+05, -2.413006217181e+05, -2.412381616321e+05, -2.411755218881e+05, -2.411127035401e+05, -2.410497076351e+05, -2.409865352101e+05, -2.409231872961e+05, -2.408596649101e+05, -2.407959690651e+05, -2.407321007621e+05, -2.406680609971e+05, -2.406038507541e+05, -2.405394710121e+05, -2.404749227391e+05, -2.404102068961e+05, -2.403453244371e+05, -2.402802763061e+05, -2.402150634411e+05, -2.401496867701e+05, -2.400841472171e+05, -2.400184456951e+05, -2.399525831111e+05, -2.398865603641e+05, -2.398203783461e+05, -2.397540379421e+05, -2.396875400291e+05, -2.396208854781e+05, -2.395540751511e+05, -2.394871099061e+05, -2.394199905911e+05, -2.393527180501e+05, -2.392852931181e+05, -2.392177166261e+05, -2.391499893941e+05, -2.390821122411e+05, -2.390140859741e+05, -2.389459113991e+05, -2.388775893111e+05, -2.388091205021e+05, -2.387405057561e+05, -2.386717458521e+05, -2.386028415621e+05, -2.385337936531e+05, -2.384646028851e+05, -2.383952700131e+05, -2.383257957851e+05, -2.382561809461e+05, -2.381864262311e+05, -2.381165323731e+05, -2.380465000991e+05, -2.379763301281e+05, -2.379060231761e+05, -2.378355799541e+05, -2.377650011641e+05, -2.376942875071e+05, -2.376234396771e+05, -2.375524583621e+05, -2.374813442451e+05, -2.374100980061e+05, -2.373387203171e+05, -2.372672118481e+05, -2.371955732621e+05, -2.371238052171e+05, -2.370519083671e+05, -2.369798833621e+05, -2.369077308451e+05, -2.368354514561e+05, -2.367630458301e+05, -2.366905145981e+05, -2.366178583841e+05, -2.365450778111e+05, -2.364721734951e+05, -2.363991460481e+05, -2.363259960781e+05, -2.362527241901e+05, -2.361793309811e+05, -2.361058170481e+05, -2.360321829801e+05, -2.359584293641e+05, -2.358845567841e+05, -2.358105658171e+05, -2.357364570371e+05, -2.356622310141e+05, -2.355878883161e+05, -2.355134295041e+05, -2.354388551361e+05, -2.353641657671e+05, -2.352893619481e+05, -2.352144442251e+05, -2.351394131411e+05, -2.350642692371e+05, -2.349890130461e+05, -2.349136451021e+05, -2.348381659331e+05, -2.347625760631e+05, -2.346868760131e+05, -2.346110663001e+05, -2.345351474401e+05, -2.344591199421e+05, -2.343829843141e+05, -2.343067410601e+05, -2.342303906791e+05, -2.341539336681e+05, -2.340773705221e+05, -2.340007017311e+05, -2.339239277811e+05, -2.338470491571e+05, -2.337700663391e+05, -2.336929798051e+05, -2.336157900291e+05, -2.335384974831e+05, -2.334611026331e+05, -2.333836059461e+05, -2.333060078841e+05, -2.332283089051e+05, -2.331505094661e+05, -2.330726100181e+05, -2.329946110141e+05, -2.329165128991e+05, -2.328383161181e+05, -2.327600211121e+05, -2.326816283211e+05, -2.326031381791e+05, -2.325245511201e+05, -2.324458675751e+05, -2.323670879691e+05, -2.322882127291e+05, -2.322092422771e+05, -2.321301770311e+05, -2.320510174081e+05, -2.319717638241e+05, -2.318924166881e+05, -2.318129764101e+05, -2.317334433971e+05, -2.316538180531e+05, -2.315741007771e+05, -2.314942919711e+05, -2.314143920281e+05, -2.313344013451e+05, -2.312543203111e+05, -2.311741493171e+05, -2.310938887481e+05, -2.310135389901e+05, -2.309331004231e+05, -2.308525734271e+05, -2.307719583801e+05, -2.306912556571e+05, -2.306104656291e+05, -2.305295886691e+05, -2.304486251431e+05},
    {2.012714606187e+04, 1.821513749087e+04, 1.626807659887e+04, 1.428407702487e+04, 1.226108093387e+04, 1.019683653987e+04, 8.088871710871e+03, 5.934462769871e+03, 3.730597393871e+03, 1.473930193871e+03, -8.392708771295e+02, -3.213189706129e+03, -5.652541982129e+03, -8.162674273129e+03, -1.074968805613e+04, -1.342059685313e+04, -1.618352787013e+04, -1.904798455413e+04, -2.202519405913e+04, -2.512857564413e+04, -2.837438549713e+04, -3.178262624013e+04, -3.537836649413e+04, -3.919372046313e+04, -4.327093932513e+04, -4.766748259913e+04, -5.246487006413e+04, -5.778543370613e+04, -6.382772777913e+04, -7.095460632413e+04, -7.998083723013e+04, -2.152109397531e+05, -2.171505472651e+05, -2.187652894021e+05, -2.201583958121e+05, -2.213882470701e+05, -2.224916714721e+05, -2.234936463551e+05, -2.244120172981e+05, -2.252600479161e+05, -2.260479071561e+05, -2.267835893581e+05, -2.274735106871e+05, -2.281229108471e+05, -2.287361324581e+05, -2.293168206851e+05, -2.298680692641e+05, -2.303925295141e+05, -2.308924931801e+05, -2.313699563881e+05, -2.318266697321e+05, -2.322641779831e+05, -2.326838519451e+05, -2.330869142591e+05, -2.334744605181e+05, -2.338474766861e+05, -2.342068535781e+05, -2.345533989911e+05, -2.348878479301e+05, -2.352108712801e+05, -2.355230832131e+05, -2.358250475301e+05, -2.361172831381e+05, -2.364002687851e+05, -2.366744471851e+05, -2.369402286131e+05, -2.371979940651e+05, -2.374480980281e+05, -2.376908709401e+05, -2.379266213511e+05, -2.381556378591e+05, -2.383781908241e+05, -2.385945339051e+05, -2.388049054361e+05, -2.390095296591e+05, -2.392086178381e+05, -2.394023692611e+05, -2.395909721471e+05, -2.397746044681e+05, -2.399534346961e+05, -2.401276224801e+05, -2.402973192671e+05, -2.404626688641e+05, -2.406238079571e+05, -2.407808665831e+05, -2.409339685651e+05, -2.410832319071e+05, -2.412287691661e+05, -2.413706877881e+05, -2.415090904201e+05, -2.416440751971e+05, -2.417757360121e+05, -2.419041627601e+05, -2.420294415691e+05, -2.421516550111e+05, -2.422708823031e+05, -2.423871994881e+05, -2.425006796061e+05, -2.426113928581e+05, -2.427194067511e+05, -2.428247862401e+05, -2.429275938561e+05, -2.430278898311e+05, -2.431257322101e+05, -2.432211769611e+05, -2.433142780721e+05, -2.434050876491e+05, -2.434936560021e+05, -2.435800317311e+05, -2.436642618051e+05, -2.437463916341e+05, -2.438264651391e+05, -2.439045248221e+05, -2.439806118221e+05, -2.440547659811e+05, -2.441270258951e+05, -2.441974289661e+05, -2.442660114571e+05, -2.443328085341e+05, -2.443978543141e+05, -2.444611819061e+05, -2.445228234521e+05, -2.445828101661e+05, -2.446411723701e+05, -2.446979395251e+05, -2.447531402711e+05, -2.448068024531e+05, -2.448589531531e+05, -2.449096187171e+05, -2.449588247831e+05, -2.450065963081e+05, -2.450529575891e+05, -2.450979322911e+05, -2.451415434681e+05, -2.451838135811e+05, -2.452247645241e+05, -2.452644176391e+05, -2.453027937401e+05, -2.453399131241e+05, -2.453757955931e+05, -2.454104604711e+05, -2.454439266171e+05, -2.454762124411e+05, -2.455073359191e+05, -2.455373146061e+05, -2.455661656521e+05, -2.455939058111e+05, -2.456205514551e+05, -2.456461185881e+05, -2.456706228541e+05, -2.456940795481e+05, -2.457165036291e+05, -2.457379097301e+05, -2.457583121641e+05, -2.457777249381e+05, -2.457961617591e+05, -2.458136360431e+05, -2.458301609251e+05, -2.458457492641e+05, -2.458604136561e+05, -2.458741664361e+05, -2.458870196861e+05, -2.458989852471e+05, -2.459100747201e+05, -2.459202994741e+05, -2.459296706541e+05, -2.459381991851e+05, -2.459458957791e+05, -2.459527709421e+05, -2.459588349741e+05, -2.459640979831e+05, -2.459685698811e+05, -2.459722603951e+05, -2.459751790711e+05, -2.459773352741e+05, -2.459787382001e+05, -2.459793968741e+05, -2.459793201571e+05, -2.459785167491e+05, -2.459769951961e+05, -2.459747638881e+05, -2.459718310671e+05, -2.459682048321e+05, -2.459638931371e+05, -2.459589037991e+05, -2.459532444991e+05, -2.459469227871e+05, -2.459399460821e+05, -2.459323216791e+05, -2.459240567491e+05, -2.459151583431e+05, -2.459056333931e+05, -2.458954887171e+05, -2.458847310221e+05, -2.458733669031e+05, -2.458614028491e+05, -2.458488452421e+05, -2.458357003631e+05, -2.458219743931e+05, -2.458076734131e+05, -2.457928034081e+05, -2.457773702701e+05, -2.457613797981e+05, -2.457448377001e+05, -2.457277495971e+05, -2.457101210231e+05, -2.456919574271e+05, -2.456732641761e+05, -2.456540465541e+05, -2.456343097681e+05, -2.456140589451e+05, -2.455932991361e+05, -2.455720353181e+05, -2.455502723931e+05, -2.455280151931e+05, -2.455052684791e+05, -2.454820369421e+05, -2.454583252061e+05, -2.454341378281e+05, -2.454094793011e+05, -2.453843540541e+05, -2.453587664531e+05, -2.453327208011e+05, -2.453062213431e+05, -2.452792722651e+05, -2.452518776921e+05, -2.452240416961e+05, -2.451957682911e+05, -2.451670614351e+05, -2.451379250341e+05, -2.451083629421e+05, -2.450783789591e+05, -2.450479768341e+05, -2.450171602681e+05, -2.449859329121e+05, -2.449542983671e+05, -2.449222601891e+05, -2.448898218871e+05, -2.448569869231e+05, -2.448237587151e+05, -2.447901406381e+05, -2.447561360201e+05, -2.447217481511e+05, -2.446869802761e+05, -2.446518356001e+05, -2.446163172871e+05, -2.445804284621e+05, -2.445441722101e+05, -2.445075515781e+05, -2.444705695761e+05, -2.444332291741e+05, -2.443955333101e+05, -2.443574848821e+05, -2.443190867541e+05, -2.442803417561e+05, -2.442412526831e+05, -2.442018222971e+05, -2.441620533261e+05, -2.441219484661e+05, -2.440815103811e+05, -2.440407417031e+05, -2.439996450331e+05, -2.439582229431e+05, -2.439164779721e+05, -2.438744126331e+05, -2.438320294061e+05, -2.437893307461e+05, -2.437463190781e+05, -2.437029967981e+05, -2.436593662771e+05, -2.436154298571e+05, -2.435711898561e+05, -2.435266485641e+05, -2.434818082441e+05, -2.434366711381e+05, -2.433912394581e+05, -2.433455153961e+05, -2.432995011151e+05, -2.432531987591e+05, -2.432066104451e+05, -2.431597382681e+05, -2.431125843011e+05, -2.430651505931e+05, -2.430174391711e+05, -2.429694520431e+05, -2.429211911911e+05, -2.428726585801e+05, -2.428238561501e+05, -2.427747858241e+05, -2.427254495031e+05, -2.426758490681e+05, -2.426259863811e+05, -2.425758632821e+05, -2.425254815961e+05, -2.424748431271e+05, -2.424239496581e+05, -2.423728029581e+05, -2.423214047751e+05, -2.422697568401e+05, -2.422178608651e+05, -2.421657185471e+05, -2.421133315641e+05, -2.420607015781e+05, -2.420078302341e+05, -2.419547191601e+05, -2.419013699691e+05, -2.418477842561e+05, -2.417939636011e+05, -2.417399095711e+05, -2.416856237131e+05, -2.416311075631e+05, -2.415763626381e+05, -2.415213904441e+05, -2.414661924701e+05, -2.414107701911e+05, -2.413551250681e+05, -2.412992585481e+05, -2.412431720651e+05, -2.411868670371e+05, -2.411303448711e+05, -2.410736069591e+05, -2.410166546801e+05, -2.409594894011e+05, -2.409021124751e+05, -2.408445252431e+05, -2.407867290331e+05, -2.407287251611e+05, -2.406705149321e+05, -2.406120996351e+05, -2.405534805521e+05, -2.404946589491e+05, -2.404356360851e+05, -2.403764132021e+05, -2.403169915361e+05, -2.402573723071e+05, -2.401975567281e+05, -2.401375459991e+05, -2.400773413091e+05, -2.400169438371e+05, -2.399563547521e+05, -2.398955752121e+05, -2.398346063631e+05, -2.397734493441e+05, -2.397121052821e+05, -2.396505752941e+05, -2.395888604891e+05, -2.395269619631e+05, -2.394648808071e+05, -2.394026180981e+05, -2.393401749061e+05, -2.392775522921e+05, -2.392147513061e+05, -2.391517729911e+05, -2.390886183811e+05, -2.390252884991e+05, -2.389617843611e+05, -2.388981069741e+05, -2.388342573381e+05, -2.387702364411e+05, -2.387060452651e+05, -2.386416847851e+05, -2.385771559651e+05, -2.385124597631e+05, -2.384475971281e+05, -2.383825690021e+05, -2.383173763181e+05, -2.382520200011e+05, -2.381865009711e+05, -2.381208201381e+05, -2.380549784061e+05, -2.379889766691e+05, -2.379228158181e+05, -2.378564967331e+05, -2.377900202891e+05, -2.377233873531e+05, -2.376565987851e+05, -2.375896554381e+05, -2.375225581591e+05, -2.374553077881e+05, -2.373879051581e+05, -2.373203510951e+05, -2.372526464191e+05, -2.371847919441e+05, -2.371167884761e+05, -2.370486368161e+05, -2.369803377581e+05, -2.369118920911e+05, -2.368433005971e+05, -2.367745640521e+05, -2.367056832251e+05, -2.366366588811e+05, -2.365674917771e+05, -2.364981826671e+05, -2.364287322951e+05, -2.363591414041e+05, -2.362894107281e+05, -2.362195409961e+05, -2.361495329331e+05, -2.360793872581e+05, -2.360091046821e+05, -2.359386859151e+05, -2.358681316571e+05, -2.357974426071e+05, -2.357266194571e+05, -2.356556628921e+05, -2.355845735941e+05, -2.355133522411e+05, -2.354419995031e+05, -2.353705160481e+05, -2.352989025361e+05, -2.352271596251e+05, -2.351552879671e+05, -2.350832882081e+05, -2.350111609921e+05, -2.349389069561e+05, -2.348665267331e+05, -2.347940209521e+05, -2.347213902371e+05, -2.346486352081e+05, -2.345757564801e+05, -2.345027546631e+05, -2.344296303631e+05, -2.343563841841e+05, -2.342830167221e+05, -2.342095285701e+05, -2.341359203191e+05, -2.340621925541e+05, -2.339883458551e+05, -2.339143807981e+05, -2.338402979581e+05, -2.337660979031e+05, -2.336917811981e+05, -2.336173484031e+05, -2.335428000761e+05, -2.334681367701e+05, -2.333933590341e+05, -2.333184674131e+05, -2.332434624511e+05, -2.331683446831e+05, -2.330931146461e+05, -2.330177728691e+05, -2.329423198811e+05, -2.328667562031e+05, -2.327910823571e+05, -2.327152988581e+05, -2.326394062201e+05, -2.325634049511e+05, -2.324872955591e+05, -2.324110785461e+05, -2.323347544101e+05, -2.322583236491e+05, -2.321817867541e+05, -2.321051442151e+05, -2.320283965181e+05, -2.319515441461e+05, -2.318745875781e+05, -2.317975272921e+05, -2.317203637601e+05, -2.316430974531e+05, -2.315657288371e+05, -2.314882583781e+05, -2.314106865361e+05, -2.313330137691e+05, -2.312552405331e+05, -2.311773672791e+05, -2.310993944571e+05, -2.310213225141e+05, -2.309431518921e+05, -2.308648830321e+05, -2.307865163721e+05, -2.307080523481e+05, -2.306294913911e+05, -2.305508339311e+05, -2.304720803941e+05, -2.303932312051e+05, -2.303142867841e+05, -2.302352475521e+05, -2.301561139231e+05, -2.300768863111e+05, -2.299975651271e+05, -2.299181507791e+05, -2.298386436731e+05, -2.297590442121e+05, -2.296793527971e+05, -2.295995698261e+05, -2.295196956941e+05, -2.294397307951e+05, -2.293596755201e+05, -2.292795302571e+05, -2.291992953931e+05, -2.291189713091e+05, -2.290385583901e+05, -2.289580570121e+05, -2.288774675531e+05, -2.287967903871e+05},
    {2.099478228487e+04, 1.909751139487e+04, 1.716596562787e+04, 1.519833363287e+04, 1.319264234987e+04, 1.114673622787e+04, 9.058252890871e+03, 6.924594475871e+03, 4.742893667870e+03, 2.509973184870e+03, 2.222971068705e+02, -2.124088057129e+03, -4.533625310129e+03, -7.011335817129e+03, -9.562929230129e+03, -1.219494252913e+04, -1.491491688213e+04, -1.773162603713e+04, -2.065537578013e+04, -2.369840334213e+04, -2.687542062213e+04, -3.020436963813e+04, -3.370750062013e+04, -3.741295781313e+04, -4.135719758313e+04, -4.558884001313e+04, -5.017514427113e+04, -5.521367158813e+04, -6.085530157713e+04, -6.735586554913e+04, -7.521713925113e+04, -8.573778739213e+04, -2.118051597951e+05, -2.139451225061e+05, -2.156983604161e+05, -2.171956271421e+05, -2.185080866641e+05, -2.196794698351e+05, -2.207388652271e+05, -2.217067456851e+05, -2.225981462541e+05, -2.234244821101e+05, -2.241946552221e+05, -2.249157620591e+05, -2.255935647811e+05, -2.262328156241e+05, -2.268374866901e+05, -2.274109367391e+05, -2.279560348541e+05, -2.284752538591e+05, -2.289707420581e+05, -2.294443791371e+05, -2.298978203021e+05, -2.303325315411e+05, -2.307498181041e+05, -2.311508477171e+05, -2.315366696851e+05, -2.319082307301e+05, -2.322663882281e+05, -2.326119213411e+05, -2.329455404401e+05, -2.332678951361e+05, -2.335795811501e+05, -2.338811462241e+05, -2.341730952431e+05, -2.344558946681e+05, -2.347299764151e+05, -2.349957412451e+05, -2.352535617451e+05, -2.355037849561e+05, -2.357467346991e+05, -2.359827136391e+05, -2.362120051241e+05, -2.364348748211e+05, -2.366515721881e+05, -2.368623317881e+05, -2.370673744781e+05, -2.372669084701e+05, -2.374611303061e+05, -2.376502257201e+05, -2.378343704421e+05, -2.380137309121e+05, -2.381884649361e+05, -2.383587222911e+05, -2.385246452651e+05, -2.386863691611e+05, -2.388440227601e+05, -2.389977287351e+05, -2.391476040481e+05, -2.392937602991e+05, -2.394363040621e+05, -2.395753371841e+05, -2.397109570731e+05, -2.398432569521e+05, -2.399723261041e+05, -2.400982500941e+05, -2.402211109821e+05, -2.403409875111e+05, -2.404579552911e+05, -2.405720869641e+05, -2.406834523651e+05, -2.407921186671e+05, -2.408981505151e+05, -2.410016101591e+05, -2.411025575691e+05, -2.412010505521e+05, -2.412971448561e+05, -2.413908942691e+05, -2.414823507101e+05, -2.415715643211e+05, -2.416585835461e+05, -2.417434552101e+05, -2.418262245941e+05, -2.419069355001e+05, -2.419856303181e+05, -2.420623500901e+05, -2.421371345651e+05, -2.422100222571e+05, -2.422810504921e+05, -2.423502554651e+05, -2.424176722801e+05, -2.424833349981e+05, -2.425472766771e+05, -2.426095294151e+05, -2.426701243841e+05, -2.427290918701e+05, -2.427864613031e+05, -2.428422612951e+05, -2.428965196661e+05, -2.429492634761e+05, -2.430005190531e+05, -2.430503120211e+05, -2.430986673221e+05, -2.431456092461e+05, -2.431911614491e+05, -2.432353469771e+05, -2.432781882891e+05, -2.433197072771e+05, -2.433599252841e+05, -2.433988631221e+05, -2.434365410941e+05, -2.434729790041e+05, -2.435081961831e+05, -2.435422114951e+05, -2.435750433591e+05, -2.436067097581e+05, -2.436372282591e+05, -2.436666160211e+05, -2.436948898091e+05, -2.437220660081e+05, -2.437481606331e+05, -2.437731893421e+05, -2.437971674451e+05, -2.438201099151e+05, -2.438420313991e+05, -2.438629462251e+05, -2.438828684171e+05, -2.439018116961e+05, -2.439197894971e+05, -2.439368149701e+05, -2.439529009921e+05, -2.439680601751e+05, -2.439823048711e+05, -2.439956471821e+05, -2.440080989651e+05, -2.440196718391e+05, -2.440303771911e+05, -2.440402261851e+05, -2.440492297641e+05, -2.440573986581e+05, -2.440647433911e+05, -2.440712742831e+05, -2.440770014581e+05, -2.440819348491e+05, -2.440860842001e+05, -2.440894590751e+05, -2.440920688601e+05, -2.440939227681e+05, -2.440950298421e+05, -2.440953989611e+05, -2.440950388461e+05, -2.440939580581e+05, -2.440921650081e+05, -2.440896679551e+05, -2.440864750161e+05, -2.440825941631e+05, -2.440780332321e+05, -2.440727999211e+05, -2.440669017971e+05, -2.440603463001e+05, -2.440531407391e+05, -2.440452923041e+05, -2.440368080631e+05, -2.440276949671e+05, -2.440179598501e+05, -2.440076094351e+05, -2.439966503361e+05, -2.439850890571e+05, -2.439729319981e+05, -2.439601854581e+05, -2.439468556321e+05, -2.439329486181e+05, -2.439184704181e+05, -2.439034269401e+05, -2.438878239991e+05, -2.438716673191e+05, -2.438549625371e+05, -2.438377152021e+05, -2.438199307801e+05, -2.438016146511e+05, -2.437827721181e+05, -2.437634084001e+05, -2.437435286391e+05, -2.437231379041e+05, -2.437022411831e+05, -2.436808433961e+05, -2.436589493881e+05, -2.436365639351e+05, -2.436136917411e+05, -2.435903374461e+05, -2.435665056211e+05, -2.435422007721e+05, -2.435174273411e+05, -2.434921897091e+05, -2.434664921941e+05, -2.434403390531e+05, -2.434137344841e+05, -2.433866826281e+05, -2.433591875681e+05, -2.433312533311e+05, -2.433028838901e+05, -2.432740831631e+05, -2.432448550161e+05, -2.432152032611e+05, -2.431851316611e+05, -2.431546439281e+05, -2.431237437251e+05, -2.430924346671e+05, -2.430607203211e+05, -2.430286042081e+05, -2.429960898011e+05, -2.429631805311e+05, -2.429298797831e+05, -2.428961908981e+05, -2.428621171771e+05, -2.428276618761e+05, -2.427928282101e+05, -2.427576193571e+05, -2.427220384501e+05, -2.426860885861e+05, -2.426497728241e+05, -2.426130941821e+05, -2.425760556441e+05, -2.425386601551e+05, -2.425009106251e+05, -2.424628099291e+05, -2.424243609061e+05, -2.423855663621e+05, -2.423464290691e+05, -2.423069517641e+05, -2.422671371531e+05, -2.422269879101e+05, -2.421865066771e+05, -2.421456960641e+05, -2.421045586531e+05, -2.420630969931e+05, -2.420213136041e+05, -2.419792109781e+05, -2.419367915771e+05, -2.418940578351e+05, -2.418510121591e+05, -2.418076569281e+05, -2.417639944931e+05, -2.417200271801e+05, -2.416757572871e+05, -2.416311870881e+05, -2.415863188311e+05, -2.415411547381e+05, -2.414956970081e+05, -2.414499478141e+05, -2.414039093071e+05, -2.413575836131e+05, -2.413109728351e+05, -2.412640790521e+05, -2.412169043241e+05, -2.411694506851e+05, -2.411217201481e+05, -2.410737147071e+05, -2.410254363301e+05, -2.409768869691e+05, -2.409280685521e+05, -2.408789829881e+05, -2.408296321661e+05, -2.407800179541e+05, -2.407301422011e+05, -2.406800067381e+05, -2.406296133751e+05, -2.405789639051e+05, -2.405280601021e+05, -2.404769037211e+05, -2.404254965001e+05, -2.403738401591e+05, -2.403219364001e+05, -2.402697869091e+05, -2.402173933551e+05, -2.401647573881e+05, -2.401118806431e+05, -2.400587647401e+05, -2.400054112811e+05, -2.399518218531e+05, -2.398979980261e+05, -2.398439413561e+05, -2.397896533831e+05, -2.397351356341e+05, -2.396803896181e+05, -2.396254168301e+05, -2.395702187531e+05, -2.395147968521e+05, -2.394591525821e+05, -2.394032873811e+05, -2.393472026731e+05, -2.392908998721e+05, -2.392343803741e+05, -2.391776455651e+05, -2.391206968171e+05, -2.390635354891e+05, -2.390061629281e+05, -2.389485804661e+05, -2.388907894261e+05, -2.388327911161e+05, -2.387745868341e+05, -2.387161778651e+05, -2.386575654811e+05, -2.385987509441e+05, -2.385397355051e+05, -2.384805204021e+05, -2.384211068621e+05, -2.383614961021e+05, -2.383016893271e+05, -2.382416877321e+05, -2.381814925001e+05, -2.381211048051e+05, -2.380605258101e+05, -2.379997566661e+05, -2.379387985171e+05, -2.378776524941e+05, -2.378163197201e+05, -2.377548013071e+05, -2.376930983581e+05, -2.376312119651e+05, -2.375691432121e+05, -2.375068931741e+05, -2.374444629141e+05, -2.373818534901e+05, -2.373190659471e+05, -2.372561013231e+05, -2.371929606471e+05, -2.371296449391e+05, -2.370661552101e+05, -2.370024924631e+05, -2.369386576921e+05, -2.368746518831e+05, -2.368104760141e+05, -2.367461310541e+05, -2.366816179651e+05, -2.366169377001e+05, -2.365520912051e+05, -2.364870794171e+05, -2.364219032651e+05, -2.363565636741e+05, -2.362910615561e+05, -2.362253978191e+05, -2.361595733631e+05, -2.360935890811e+05, -2.360274458571e+05, -2.359611445701e+05, -2.358946860911e+05, -2.358280712841e+05, -2.357613010051e+05, -2.356943761041e+05, -2.356272974271e+05, -2.355600658071e+05, -2.354926820771e+05, -2.354251470591e+05, -2.353574615711e+05, -2.352896264221e+05, -2.352216424171e+05, -2.351535103531e+05, -2.350852310231e+05, -2.350168052121e+05, -2.349482336981e+05, -2.348795172561e+05, -2.348106566531e+05, -2.347416526501e+05, -2.346725060021e+05, -2.346032174591e+05, -2.345337877661e+05, -2.344642176601e+05, -2.343945078751e+05, -2.343246591371e+05, -2.342546721681e+05, -2.341845476841e+05, -2.341142863971e+05, -2.340438890101e+05, -2.339733562261e+05, -2.339026887371e+05, -2.338318872351e+05, -2.337609524041e+05, -2.336898849231e+05, -2.336186854671e+05, -2.335473547061e+05, -2.334758933031e+05, -2.334043019191e+05, -2.333325812081e+05, -2.332607318211e+05, -2.331887544021e+05, -2.331166495931e+05, -2.330444180301e+05, -2.329720603441e+05, -2.328995771611e+05, -2.328269691041e+05, -2.327542367911e+05, -2.326813808351e+05, -2.326084018451e+05, -2.325353004271e+05, -2.324620771791e+05, -2.323887326991e+05, -2.323152675791e+05, -2.322416824051e+05, -2.321679777621e+05, -2.320941542281e+05, -2.320202123801e+05, -2.319461527891e+05, -2.318719760221e+05, -2.317976826431e+05, -2.317232732101e+05, -2.316487482801e+05, -2.315741084041e+05, -2.314993541311e+05, -2.314244860051e+05, -2.313495045651e+05, -2.312744103491e+05, -2.311992038901e+05, -2.311238857181e+05, -2.310484563571e+05, -2.309729163311e+05, -2.308972661581e+05, -2.308215063531e+05, -2.307456374291e+05, -2.306696598931e+05, -2.305935742501e+05, -2.305173810021e+05, -2.304410806461e+05, -2.303646736791e+05, -2.302881605911e+05, -2.302115418701e+05, -2.301348180021e+05, -2.300579894691e+05, -2.299810567491e+05, -2.299040203171e+05, -2.298268806461e+05, -2.297496382061e+05, -2.296722934611e+05, -2.295948468771e+05, -2.295172989131e+05, -2.294396500251e+05, -2.293619006681e+05, -2.292840512951e+05, -2.292061023511e+05, -2.291280542851e+05, -2.290499075371e+05, -2.289716625471e+05, -2.288933197541e+05, -2.288148795901e+05, -2.287363424871e+05, -2.286577088741e+05, -2.285789791771e+05, -2.285001538191e+05, -2.284212332201e+05, -2.283422177991e+05, -2.282631079711e+05, -2.281839041491e+05, -2.281046067411e+05, -2.280252161571e+05, -2.279457328011e+05, -2.278661570761e+05, -2.277864893811e+05, -2.277067301141e+05, -2.276268796701e+05, -2.275469384421e+05, -2.274669068201e+05, -2.273867851911e+05, -2.273065739411e+05, -2.272262734541e+05, -2.271458841081e+05},
    {2.186341721287e+04, 1.998069554887e+04, 1.806445640787e+04, 1.611296019387e+04, 1.412431478187e+04, 1.209645630487e+04, 1.002712671187e+04, 7.913847430871e+03, 5.753888259871e+03, 3.544230397871e+03, 1.281522213871e+03, -1.037974069130e+03, -3.418447157129e+03, -5.864616087129e+03, -8.381828479129e+03, -1.097618339013e+04, -1.365468671513e+04, -1.642545027713e+04, -1.929795055313e+04, -2.228337031013e+04, -2.539505792513e+04, -2.864915777413e+04, -3.206549605713e+04, -3.566886030313e+04, -3.949090831113e+04, -4.357312883013e+04, -4.797165613813e+04, -5.276557755013e+04, -5.807240992313e+04, -6.408007956513e+04, -7.112366954413e+04, -7.991989446413e+04, -9.274810933113e+04, -2.083430127391e+05, -2.107047555371e+05, -2.126072866651e+05, -2.142150052181e+05, -2.156140935451e+05, -2.168561810161e+05, -2.179749701271e+05, -2.189938377701e+05, -2.199297530571e+05, -2.207954779571e+05, -2.216008868741e+05, -2.223537998371e+05, -2.230605311881e+05, -2.237262637681e+05, -2.243553117811e+05, -2.249513101971e+05, -2.255173542861e+05, -2.260561044061e+05, -2.265698660611e+05, -2.270606520071e+05, -2.275302310721e+05, -2.279801670271e+05, -2.284118498621e+05, -2.288265212141e+05, -2.292252952251e+05, -2.296091757951e+05, -2.299790709641e+05, -2.303358049921e+05, -2.306801285531e+05, -2.310127274121e+05, -2.313342298381e+05, -2.316452129711e+05, -2.319462083241e+05, -2.322377065521e+05, -2.325201616101e+05, -2.327939943841e+05, -2.330595958841e+05, -2.333173300501e+05, -2.335675362411e+05, -2.338105314341e+05, -2.340466121881e+05, -2.342760563811e+05, -2.344991247851e+05, -2.347160624571e+05, -2.349271000031e+05, -2.351324547121e+05, -2.353323315781e+05, -2.355269242291e+05, -2.357164157661e+05, -2.359009795281e+05, -2.360807797851e+05, -2.362559723711e+05, -2.364267052671e+05, -2.365931191251e+05, -2.367553477571e+05, -2.369135185791e+05, -2.370677530191e+05, -2.372181668981e+05, -2.373648707741e+05, -2.375079702611e+05, -2.376475663311e+05, -2.377837555851e+05, -2.379166305051e+05, -2.380462796931e+05, -2.381727880901e+05, -2.382962371781e+05, -2.384167051701e+05, -2.385342671861e+05, -2.386489954201e+05, -2.387609592891e+05, -2.388702255841e+05, -2.389768585951e+05, -2.390809202461e+05, -2.391824702061e+05, -2.392815660051e+05, -2.393782631321e+05, -2.394726151381e+05, -2.395646737231e+05, -2.396544888281e+05, -2.397421087091e+05, -2.398275800201e+05, -2.399109478791e+05, -2.399922559441e+05, -2.400715464701e+05, -2.401488603731e+05, -2.402242372871e+05, -2.402977156191e+05, -2.403693326001e+05, -2.404391243331e+05, -2.405071258401e+05, -2.405733711071e+05, -2.406378931231e+05, -2.407007239211e+05, -2.407618946171e+05, -2.408214354441e+05, -2.408793757831e+05, -2.409357442031e+05, -2.409905684841e+05, -2.410438756521e+05, -2.410956920011e+05, -2.411460431281e+05, -2.411949539481e+05, -2.412424487291e+05, -2.412885511071e+05, -2.413332841111e+05, -2.413766701851e+05, -2.414187312081e+05, -2.414594885121e+05, -2.414989629011e+05, -2.415371746691e+05, -2.415741436191e+05, -2.416098890731e+05, -2.416444298961e+05, -2.416777845051e+05, -2.417099708851e+05, -2.417410066021e+05, -2.417709088191e+05, -2.417996943051e+05, -2.418273794491e+05, -2.418539802731e+05, -2.418795124391e+05, -2.419039912661e+05, -2.419274317341e+05, -2.419498484981e+05, -2.419712558971e+05, -2.419916679631e+05, -2.420110984291e+05, -2.420295607391e+05, -2.420470680551e+05, -2.420636332661e+05, -2.420792689961e+05, -2.420939876091e+05, -2.421078012201e+05, -2.421207216981e+05, -2.421327606771e+05, -2.421439295571e+05, -2.421542395141e+05, -2.421637015081e+05, -2.421723262811e+05, -2.421801243731e+05, -2.421871061171e+05, -2.421932816531e+05, -2.421986609271e+05, -2.422032537001e+05, -2.422070695491e+05, -2.422101178761e+05, -2.422124079091e+05, -2.422139487061e+05, -2.422147491631e+05, -2.422148180131e+05, -2.422141638351e+05, -2.422127950531e+05, -2.422107199441e+05, -2.422079466371e+05, -2.422044831231e+05, -2.422003372501e+05, -2.421955167331e+05, -2.421900291541e+05, -2.421838819671e+05, -2.421770824981e+05, -2.421696379521e+05, -2.421615554111e+05, -2.421528418401e+05, -2.421435040901e+05, -2.421335488991e+05, -2.421229828941e+05, -2.421118125961e+05, -2.421000444181e+05, -2.420876846741e+05, -2.420747395731e+05, -2.420612152291e+05, -2.420471176581e+05, -2.420324527821e+05, -2.420172264291e+05, -2.420014443391e+05, -2.419851121621e+05, -2.419682354621e+05, -2.419508197171e+05, -2.419328703241e+05, -2.419143925961e+05, -2.418953917681e+05, -2.418758729961e+05, -2.418558413591e+05, -2.418353018631e+05, -2.418142594371e+05, -2.417927189421e+05, -2.417706851661e+05, -2.417481628261e+05, -2.417251565751e+05, -2.417016709961e+05, -2.416777106101e+05, -2.416532798701e+05, -2.416283831681e+05, -2.416030248361e+05, -2.415772091431e+05, -2.415509403001e+05, -2.415242224601e+05, -2.414970597161e+05, -2.414694561101e+05, -2.414414156241e+05, -2.414129421891e+05, -2.413840396811e+05, -2.413547119271e+05, -2.413249626981e+05, -2.412947957201e+05, -2.412642146661e+05, -2.412332231611e+05, -2.412018247851e+05, -2.411700230681e+05, -2.411378214961e+05, -2.411052235101e+05, -2.410722325051e+05, -2.410388518341e+05, -2.410050848061e+05, -2.409709346901e+05, -2.409364047111e+05, -2.409014980551e+05, -2.408662178681e+05, -2.408305672561e+05, -2.407945492871e+05, -2.407581669901e+05, -2.407214233581e+05, -2.406843213481e+05, -2.406468638771e+05, -2.406090538301e+05, -2.405708940571e+05, -2.405323873731e+05, -2.404935365571e+05, -2.404543443571e+05, -2.404148134901e+05, -2.403749466361e+05, -2.403347464481e+05, -2.402942155451e+05, -2.402533565161e+05, -2.402121719191e+05, -2.401706642851e+05, -2.401288361131e+05, -2.400866898741e+05, -2.400442280111e+05, -2.400014529381e+05, -2.399583670421e+05, -2.399149726841e+05, -2.398712721981e+05, -2.398272678891e+05, -2.397829620411e+05, -2.397383569071e+05, -2.396934547201e+05, -2.396482576861e+05, -2.396027679851e+05, -2.395569877751e+05, -2.395109191911e+05, -2.394645643431e+05, -2.394179253201e+05, -2.393710041851e+05, -2.393238029821e+05, -2.392763237321e+05, -2.392285684351e+05, -2.391805390671e+05, -2.391322375861e+05, -2.390836659281e+05, -2.390348260091e+05, -2.389857197241e+05, -2.389363489491e+05, -2.388867155411e+05, -2.388368213351e+05, -2.387866681501e+05, -2.387362577851e+05, -2.386855920201e+05, -2.386346726171e+05, -2.385835013211e+05, -2.385320798581e+05, -2.384804099371e+05, -2.384284932491e+05, -2.383763314701e+05, -2.383239262571e+05, -2.382712792511e+05, -2.382183920781e+05, -2.381652663451e+05, -2.381119036461e+05, -2.380583055571e+05, -2.380044736411e+05, -2.379504094431e+05, -2.378961144941e+05, -2.378415903111e+05, -2.377868383961e+05, -2.377318602341e+05, -2.376766572991e+05, -2.376212310501e+05, -2.375655829311e+05, -2.375097143721e+05, -2.374536267901e+05, -2.373973215911e+05, -2.373408001621e+05, -2.372840638831e+05, -2.372271141181e+05, -2.371699522171e+05, -2.371125795211e+05, -2.370549973551e+05, -2.369972070341e+05, -2.369392098601e+05, -2.368810071231e+05, -2.368226001021e+05, -2.367639900621e+05, -2.367051782601e+05, -2.366461659371e+05, -2.365869543281e+05, -2.365275446521e+05, -2.364679381211e+05, -2.364081359331e+05, -2.363481392771e+05, -2.362879493311e+05, -2.362275672621e+05, -2.361669942281e+05, -2.361062313751e+05, -2.360452798401e+05, -2.359841407491e+05, -2.359228152201e+05, -2.358613043601e+05, -2.357996092651e+05, -2.357377310251e+05, -2.356756707171e+05, -2.356134294111e+05, -2.355510081661e+05, -2.354884080331e+05, -2.354256300551e+05, -2.353626752641e+05, -2.352995446841e+05, -2.352362393311e+05, -2.351727602111e+05, -2.351091083241e+05, -2.350452846581e+05, -2.349812901961e+05, -2.349171259101e+05, -2.348527927681e+05, -2.347882917241e+05, -2.347236237301e+05, -2.346587897261e+05, -2.345937906471e+05, -2.345286274171e+05, -2.344633009561e+05, -2.343978121751e+05, -2.343321619771e+05, -2.342663512581e+05, -2.342003809071e+05, -2.341342518061e+05, -2.340679648291e+05, -2.340015208441e+05, -2.339349207121e+05, -2.338681652851e+05, -2.338012554121e+05, -2.337341919321e+05, -2.336669756781e+05, -2.335996074781e+05, -2.335320881521e+05, -2.334644185131e+05, -2.333965993691e+05, -2.333286315221e+05, -2.332605157661e+05, -2.331922528901e+05, -2.331238436761e+05, -2.330552889021e+05, -2.329865893371e+05, -2.329177457461e+05, -2.328487588881e+05, -2.327796295151e+05, -2.327103583751e+05, -2.326409462101e+05, -2.325713937541e+05, -2.325017017391e+05, -2.324318708881e+05, -2.323619019221e+05, -2.322917955531e+05, -2.322215524901e+05, -2.321511734371e+05, -2.320806590911e+05, -2.320100101451e+05, -2.319392272851e+05, -2.318683111961e+05, -2.317972625541e+05, -2.317260820311e+05, -2.316547702941e+05, -2.315833280071e+05, -2.315117558271e+05, -2.314400544061e+05, -2.313682243931e+05, -2.312962664321e+05, -2.312241811601e+05, -2.311519692121e+05, -2.310796312171e+05, -2.310071678001e+05, -2.309345795831e+05, -2.308618671801e+05, -2.307890312031e+05, -2.307160722601e+05, -2.306429909541e+05, -2.305697878831e+05, -2.304964636421e+05, -2.304230188201e+05, -2.303494540041e+05, -2.302757697761e+05, -2.302019667141e+05, -2.301280453911e+05, -2.300540063781e+05, -2.299798502391e+05, -2.299055775381e+05, -2.298311888311e+05, -2.297566846741e+05, -2.296820656161e+05, -2.296073322051e+05, -2.295324849811e+05, -2.294575244861e+05, -2.293824512541e+05, -2.293072658171e+05, -2.292319687031e+05, -2.291565604361e+05, -2.290810415381e+05, -2.290054125261e+05, -2.289296739141e+05, -2.288538262121e+05, -2.287778699271e+05, -2.287018055641e+05, -2.286256336221e+05, -2.285493545981e+05, -2.284729689861e+05, -2.283964772771e+05, -2.283198799581e+05, -2.282431775111e+05, -2.281663704191e+05, -2.280894591591e+05, -2.280124442051e+05, -2.279353260281e+05, -2.278581050971e+05, -2.277807818771e+05, -2.277033568301e+05, -2.276258304151e+05, -2.275482030891e+05, -2.274704753041e+05, -2.273926475111e+05, -2.273147201581e+05, -2.272366936881e+05, -2.271585685441e+05, -2.270803451641e+05, -2.270020239841e+05, -2.269236054381e+05, -2.268450899571e+05, -2.267664779671e+05, -2.266877698951e+05, -2.266089661621e+05, -2.265300671881e+05, -2.264510733911e+05, -2.263719851841e+05, -2.262928029801e+05, -2.262135271881e+05, -2.261341582151e+05, -2.260546964651e+05, -2.259751423391e+05, -2.258954962381e+05, -2.258157585571e+05, -2.257359296911e+05, -2.256560100321e+05, -2.255759999691e+05, -2.254958998901e+05},
    {2.273305016787e+04, 2.086469270987e+04, 1.896355575887e+04, 1.702796837487e+04, 1.505611572587e+04, 1.304602132187e+04, 1.099552631387e+04, 8.902265284871e+03, 6.763637745871e+03, 4.576774387871e+03, 2.338496865870e+03, 4.526952787052e+01, -2.306858968129e+03, -4.722326839129e+03, -7.206146782129e+03, -9.764014831129e+03, -1.240244712813e+04, -1.512895381413e+04, -1.795226309913e+04, -2.088261431113e+04, -2.393214764413e+04, -2.711543241113e+04, -3.045019874313e+04, -3.395837673813e+04, -3.766761605313e+04, -4.161358635813e+04, -4.584360844413e+04, -5.042268811113e+04, -5.544421708713e+04, -6.105064271213e+04, -6.747838920113e+04, -7.517427137713e+04, -8.520382077713e+04, -2.011481972061e+05, -2.048288796211e+05, -2.074332920111e+05, -2.094952982351e+05, -2.112192770871e+05, -2.127086437371e+05, -2.140238893001e+05, -2.152038093471e+05, -2.162749461911e+05, -2.172563555541e+05, -2.181622399821e+05, -2.190035064011e+05, -2.197887382311e+05, -2.205248291761e+05, -2.212174116941e+05, -2.218711556851e+05, -2.224899822571e+05, -2.230772202421e+05, -2.236357231101e+05, -2.241679578571e+05, -2.246760736541e+05, -2.251619556021e+05, -2.256272673841e+05, -2.260734854621e+05, -2.265019268081e+05, -2.269137715871e+05, -2.273100818751e+05, -2.276918172311e+05, -2.280598477371e+05, -2.284149650031e+05, -2.287578914991e+05, -2.290892885241e+05, -2.294097630421e+05, -2.297198735761e+05, -2.300201353161e+05, -2.303110245591e+05, -2.305929825931e+05, -2.308664190961e+05, -2.311317151391e+05, -2.313892258241e+05, -2.316392826301e+05, -2.318821954951e+05, -2.321182546761e+05, -2.323477323991e+05, -2.325708843521e+05, -2.327879510191e+05, -2.329991588761e+05, -2.332047214821e+05, -2.334048404561e+05, -2.335997063661e+05, -2.337894995361e+05, -2.339743907771e+05, -2.341545420631e+05, -2.343301071331e+05, -2.345012320561e+05, -2.346680557371e+05, -2.348307103911e+05, -2.349893219721e+05, -2.351440105691e+05, -2.352948907711e+05, -2.354420720081e+05, -2.355856588571e+05, -2.357257513351e+05, -2.358624451641e+05, -2.359958320161e+05, -2.361259997501e+05, -2.362530326151e+05, -2.363770114591e+05, -2.364980139071e+05, -2.366161145391e+05, -2.367313850441e+05, -2.368438943771e+05, -2.369537088981e+05, -2.370608925021e+05, -2.371655067411e+05, -2.372676109451e+05, -2.373672623261e+05, -2.374645160801e+05, -2.375594254871e+05, -2.376520419931e+05, -2.377424153051e+05, -2.378305934631e+05, -2.379166229171e+05, -2.380005486001e+05, -2.380824139961e+05, -2.381622611981e+05, -2.382401309731e+05, -2.383160628181e+05, -2.383900950091e+05, -2.384622646591e+05, -2.385326077601e+05, -2.386011592331e+05, -2.386679529681e+05, -2.387330218671e+05, -2.387963978831e+05, -2.388581120551e+05, -2.389181945471e+05, -2.389766746781e+05, -2.390335809571e+05, -2.390889411101e+05, -2.391427821131e+05, -2.391951302171e+05, -2.392460109731e+05, -2.392954492621e+05, -2.393434693141e+05, -2.393900947351e+05, -2.394353485251e+05, -2.394792531021e+05, -2.395218303211e+05, -2.395631014941e+05, -2.396030874071e+05, -2.396418083361e+05, -2.396792840681e+05, -2.397155339171e+05, -2.397505767331e+05, -2.397844309251e+05, -2.398171144701e+05, -2.398486449281e+05, -2.398790394571e+05, -2.399083148231e+05, -2.399364874111e+05, -2.399635732421e+05, -2.399895879791e+05, -2.400145469391e+05, -2.400384651061e+05, -2.400613571371e+05, -2.400832373741e+05, -2.401041198511e+05, -2.401240183081e+05, -2.401429461931e+05, -2.401609166751e+05, -2.401779426491e+05, -2.401940367441e+05, -2.402092113351e+05, -2.402234785431e+05, -2.402368502461e+05, -2.402493380861e+05, -2.402609534721e+05, -2.402717075911e+05, -2.402816114111e+05, -2.402906756861e+05, -2.402989109641e+05, -2.403063275911e+05, -2.403129357171e+05, -2.403187452981e+05, -2.403237661081e+05, -2.403280077361e+05, -2.403314795931e+05, -2.403341909201e+05, -2.403361507881e+05, -2.403373681031e+05, -2.403378516131e+05, -2.403376099061e+05, -2.403366514201e+05, -2.403349844451e+05, -2.403326171221e+05, -2.403295574531e+05, -2.403258133011e+05, -2.403213923931e+05, -2.403163023231e+05, -2.403105505571e+05, -2.403041444361e+05, -2.402970911751e+05, -2.402893978701e+05, -2.402810715001e+05, -2.402721189271e+05, -2.402625469021e+05, -2.402523620661e+05, -2.402415709501e+05, -2.402301799821e+05, -2.402181954881e+05, -2.402056236901e+05, -2.401924707131e+05, -2.401787425871e+05, -2.401644452461e+05, -2.401495845301e+05, -2.401341661921e+05, -2.401181958951e+05, -2.401016792141e+05, -2.400846216401e+05, -2.400670285811e+05, -2.400489053641e+05, -2.400302572341e+05, -2.400110893601e+05, -2.399914068341e+05, -2.399712146721e+05, -2.399505178171e+05, -2.399293211401e+05, -2.399076294391e+05, -2.398854474471e+05, -2.398627798251e+05, -2.398396311691e+05, -2.398160060111e+05, -2.397919088151e+05, -2.397673439851e+05, -2.397423158641e+05, -2.397168287311e+05, -2.396908868091e+05, -2.396644942611e+05, -2.396376551941e+05, -2.396103736561e+05, -2.395826536431e+05, -2.395544990961e+05, -2.395259139021e+05, -2.394969018971e+05, -2.394674668641e+05, -2.394376125381e+05, -2.394073426021e+05, -2.393766606941e+05, -2.393455704011e+05, -2.393140752651e+05, -2.392821787811e+05, -2.392498843991e+05, -2.392171955251e+05, -2.391841155231e+05, -2.391506477101e+05, -2.391167953641e+05, -2.390825617211e+05, -2.390479499751e+05, -2.390129632831e+05, -2.389776047601e+05, -2.389418774821e+05, -2.389057844881e+05, -2.388693287801e+05, -2.388325133231e+05, -2.387953410451e+05, -2.387578148401e+05, -2.387199375641e+05, -2.386817120411e+05, -2.386431410621e+05, -2.386042273811e+05, -2.385649737241e+05, -2.385253827801e+05, -2.384854572091e+05, -2.384451996391e+05, -2.384046126681e+05, -2.383636988621e+05, -2.383224607601e+05, -2.382809008681e+05, -2.382390216661e+05, -2.381968256041e+05, -2.381543151051e+05, -2.381114925641e+05, -2.380683603491e+05, -2.380249208011e+05, -2.379811762341e+05, -2.379371289381e+05, -2.378927811761e+05, -2.378481351861e+05, -2.378031931821e+05, -2.377579573511e+05, -2.377124298601e+05, -2.376666128501e+05, -2.376205084371e+05, -2.375741187181e+05, -2.375274457641e+05, -2.374804916251e+05, -2.374332583291e+05, -2.373857478821e+05, -2.373379622691e+05, -2.372899034531e+05, -2.372415733771e+05, -2.371929739631e+05, -2.371441071141e+05, -2.370949747121e+05, -2.370455786201e+05, -2.369959206791e+05, -2.369460027161e+05, -2.368958265351e+05, -2.368453939221e+05, -2.367947066471e+05, -2.367437664601e+05, -2.366925750921e+05, -2.366411342601e+05, -2.365894456611e+05, -2.365375109751e+05, -2.364853318671e+05, -2.364329099821e+05, -2.363802469531e+05, -2.363273443931e+05, -2.362742039011e+05, -2.362208270601e+05, -2.361672154371e+05, -2.361133705831e+05, -2.360592940361e+05, -2.360049873171e+05, -2.359504519331e+05, -2.358956893761e+05, -2.358407011251e+05, -2.357854886431e+05, -2.357300533811e+05, -2.356743967731e+05, -2.356185202431e+05, -2.355624251991e+05, -2.355061130381e+05, -2.354495851401e+05, -2.353928428771e+05, -2.353358876031e+05, -2.352787206641e+05, -2.352213433911e+05, -2.351637571031e+05, -2.351059631071e+05, -2.350479626981e+05, -2.349897571591e+05, -2.349313477621e+05, -2.348727357651e+05, -2.348139224181e+05, -2.347549089561e+05, -2.346956966071e+05, -2.346362865851e+05, -2.345766800931e+05, -2.345168783251e+05, -2.344568824631e+05, -2.343966936791e+05, -2.343363131351e+05, -2.342757419811e+05, -2.342149813581e+05, -2.341540323991e+05, -2.340928962241e+05, -2.340315739431e+05, -2.339700666601e+05, -2.339083754651e+05, -2.338465014421e+05, -2.337844456641e+05, -2.337222091951e+05, -2.336597930891e+05, -2.335971983941e+05, -2.335344261441e+05, -2.334714773701e+05, -2.334083530891e+05, -2.333450543141e+05, -2.332815820451e+05, -2.332179372781e+05, -2.331541209971e+05, -2.330901341791e+05, -2.330259777951e+05, -2.329616528041e+05, -2.328971601601e+05, -2.328325008081e+05, -2.327676756851e+05, -2.327026857221e+05, -2.326375318391e+05, -2.325722149521e+05, -2.325067359681e+05, -2.324410957871e+05, -2.323752953001e+05, -2.323093353931e+05, -2.322432169441e+05, -2.321769408241e+05, -2.321105078971e+05, -2.320439190191e+05, -2.319771750421e+05, -2.319102768091e+05, -2.318432251561e+05, -2.317760209141e+05, -2.317086649061e+05, -2.316411579501e+05, -2.315735008551e+05, -2.315056944271e+05, -2.314377394641e+05, -2.313696367561e+05, -2.313013870911e+05, -2.312329912481e+05, -2.311644500001e+05, -2.310957641151e+05, -2.310269343541e+05, -2.309579614741e+05, -2.308888462251e+05, -2.308195893511e+05, -2.307501915911e+05, -2.306806536781e+05, -2.306109763391e+05, -2.305411602971e+05, -2.304712062691e+05, -2.304011149651e+05, -2.303308870911e+05, -2.302605233491e+05, -2.301900244341e+05, -2.301193910351e+05, -2.300486238391e+05, -2.299777235251e+05, -2.299066907681e+05, -2.298355262381e+05, -2.297642306021e+05, -2.296928045181e+05, -2.296212486421e+05, -2.295495636251e+05, -2.294777501141e+05, -2.294058087491e+05, -2.293337401671e+05, -2.292615450001e+05, -2.291892238761e+05, -2.291167774171e+05, -2.290442062421e+05, -2.289715109661e+05, -2.288986921981e+05, -2.288257505431e+05, -2.287526866021e+05, -2.286795009731e+05, -2.286061942481e+05, -2.285327670161e+05, -2.284592198601e+05, -2.283855533621e+05, -2.283117680961e+05, -2.282378646361e+05, -2.281638435491e+05, -2.280897054001e+05, -2.280154507501e+05, -2.279410801531e+05, -2.278665941641e+05, -2.277919933301e+05, -2.277172781981e+05, -2.276424493071e+05, -2.275675071961e+05, -2.274924523991e+05, -2.274172854461e+05, -2.273420068641e+05, -2.272666171751e+05, -2.271911169001e+05, -2.271155065541e+05, -2.270397866501e+05, -2.269639576981e+05, -2.268880202021e+05, -2.268119746661e+05, -2.267358215891e+05, -2.266595614651e+05, -2.265831947881e+05, -2.265067220471e+05, -2.264301437271e+05, -2.263534603121e+05, -2.262766722811e+05, -2.261997801101e+05, -2.261227842731e+05, -2.260456852401e+05, -2.259684834771e+05, -2.258911794501e+05, -2.258137736191e+05, -2.257362664421e+05, -2.256586583741e+05, -2.255809498681e+05, -2.255031413731e+05, -2.254252333351e+05, -2.253472261971e+05, -2.252691204011e+05, -2.251909163851e+05, -2.251126145831e+05, -2.250342154271e+05, -2.249557193481e+05, -2.248771267721e+05, -2.247984381241e+05, -2.247196538241e+05, -2.246407742921e+05, -2.245617999441e+05, -2.244827311941e+05, -2.244035684521e+05, -2.243243121281e+05, -2.242449626261e+05, -2.241655203511e+05, -2.240859857031e+05, -2.240063590811e+05, -2.239266408811e+05, -2.238468314961e+05},
    {2.360368042687e+04, 2.174950550387e+04, 1.986327025687e+04, 1.794336945187e+04, 1.598806209687e+04, 1.399545499687e+04, 1.196348367187e+04, 9.889890082870e+03, 7.772196513870e+03, 5.607674750871e+03, 3.393309494871e+03, 1.125754627871e+03, -1.198719434129e+03, -3.584289627130e+03, -6.035658428130e+03, -8.558150446129e+03, -1.115783297613e+04, -1.384166816813e+04, -1.661770751413e+04, -1.949534390813e+04, -2.248564342213e+04, -2.560178974213e+04, -2.885969148613e+04, -3.227883126013e+04, -3.588348448713e+04, -3.970452422113e+04, -4.378219440113e+04, -4.817056690513e+04, -5.294511583513e+04, -5.821654224813e+04, -6.415852820213e+04, -7.107147761513e+04, -7.956286452713e+04, -9.130070961713e+04, -1.970762351641e+05, -2.012707484491e+05, -2.041361875511e+05, -2.063664550901e+05, -2.082116676181e+05, -2.097944174931e+05, -2.111848919751e+05, -2.124273936261e+05, -2.135518565141e+05, -2.145795556571e+05, -2.155262160931e+05, -2.164038299201e+05, -2.172217788851e+05, -2.179875598511e+05, -2.187072714741e+05, -2.193859511321e+05, -2.200278145841e+05, -2.206364304801e+05, -2.212148500691e+05, -2.217657053671e+05, -2.222912846641e+05, -2.227935914451e+05, -2.232743909551e+05, -2.237352474561e+05, -2.241775543221e+05, -2.246025586171e+05, -2.250113813171e+05, -2.254050341041e+05, -2.257844333991e+05, -2.261504121841e+05, -2.265037300171e+05, -2.268450815571e+05, -2.271751038841e+05, -2.274943827891e+05, -2.278034582251e+05, -2.281028290391e+05, -2.283929571061e+05, -2.286742709481e+05, -2.289471689141e+05, -2.292120219911e+05, -2.294691762921e+05, -2.297189552571e+05, -2.299616616231e+05, -2.301975791821e+05, -2.304269743441e+05, -2.306500975581e+05, -2.308671845731e+05, -2.310784575891e+05, -2.312841262861e+05, -2.314843887681e+05, -2.316794324051e+05, -2.318694346141e+05, -2.320545635581e+05, -2.322349787911e+05, -2.324108318451e+05, -2.325822667681e+05, -2.327494206171e+05, -2.329124239111e+05, -2.330714010471e+05, -2.332264706851e+05, -2.333777461011e+05, -2.335253355121e+05, -2.336693423841e+05, -2.338098657061e+05, -2.339470002501e+05, -2.340808368151e+05, -2.342114624481e+05, -2.343389606531e+05, -2.344634115831e+05, -2.345848922241e+05, -2.347034765581e+05, -2.348192357271e+05, -2.349322381731e+05, -2.350425497821e+05, -2.351502340081e+05, -2.352553519951e+05, -2.353579626911e+05, -2.354581229561e+05, -2.355558876561e+05, -2.356513097641e+05, -2.357444404441e+05, -2.358353291351e+05, -2.359240236311e+05, -2.360105701521e+05, -2.360950134181e+05, -2.361773967111e+05, -2.362577619391e+05, -2.363361496971e+05, -2.364125993171e+05, -2.364871489291e+05, -2.365598355021e+05, -2.366306949001e+05, -2.366997619201e+05, -2.367670703401e+05, -2.368326529551e+05, -2.368965416211e+05, -2.369587672851e+05, -2.370193600261e+05, -2.370783490831e+05, -2.371357628931e+05, -2.371916291131e+05, -2.372459746541e+05, -2.372988257091e+05, -2.373502077761e+05, -2.374001456841e+05, -2.374486636161e+05, -2.374957851351e+05, -2.375415332021e+05, -2.375859301991e+05, -2.376289979451e+05, -2.376707577231e+05, -2.377112302881e+05, -2.377504358921e+05, -2.377883943001e+05, -2.378251248011e+05, -2.378606462281e+05, -2.378949769721e+05, -2.379281349931e+05, -2.379601378391e+05, -2.379910026541e+05, -2.380207461931e+05, -2.380493848321e+05, -2.380769345841e+05, -2.381034111051e+05, -2.381288297061e+05, -2.381532053661e+05, -2.381765527381e+05, -2.381988861611e+05, -2.382202196701e+05, -2.382405670011e+05, -2.382599416031e+05, -2.382783566451e+05, -2.382958250241e+05, -2.383123593721e+05, -2.383279720651e+05, -2.383426752271e+05, -2.383564807411e+05, -2.383694002531e+05, -2.383814451781e+05, -2.383926267061e+05, -2.384029558121e+05, -2.384124432561e+05, -2.384210995921e+05, -2.384289351741e+05, -2.384359601571e+05, -2.384421845081e+05, -2.384476180051e+05, -2.384522702471e+05, -2.384561506531e+05, -2.384592684731e+05, -2.384616327861e+05, -2.384632525071e+05, -2.384641363931e+05, -2.384642930421e+05, -2.384637309011e+05, -2.384624582691e+05, -2.384604832981e+05, -2.384578140001e+05, -2.384544582451e+05, -2.384504237731e+05, -2.384457181881e+05, -2.384403489661e+05, -2.384343234561e+05, -2.384276488861e+05, -2.384203323631e+05, -2.384123808731e+05, -2.384038012921e+05, -2.383946003801e+05, -2.383847847881e+05, -2.383743610591e+05, -2.383633356311e+05, -2.383517148401e+05, -2.383395049201e+05, -2.383267120061e+05, -2.383133421391e+05, -2.382994012621e+05, -2.382848952291e+05, -2.382698298001e+05, -2.382542106511e+05, -2.382380433661e+05, -2.382213334481e+05, -2.382040863151e+05, -2.381863073041e+05, -2.381680016721e+05, -2.381491745981e+05, -2.381298311831e+05, -2.381099764551e+05, -2.380896153661e+05, -2.380687527981e+05, -2.380473935601e+05, -2.380255423951e+05, -2.380032039731e+05, -2.379803829021e+05, -2.379570837221e+05, -2.379333109091e+05, -2.379090688771e+05, -2.378843619761e+05, -2.378591944991e+05, -2.378335706771e+05, -2.378074946821e+05, -2.377809706311e+05, -2.377540025841e+05, -2.377265945441e+05, -2.376987504621e+05, -2.376704742351e+05, -2.376417697081e+05, -2.376126406751e+05, -2.375830908791e+05, -2.375531240131e+05, -2.375227437231e+05, -2.374919536061e+05, -2.374607572131e+05, -2.374291580481e+05, -2.373971595711e+05, -2.373647651961e+05, -2.373319782941e+05, -2.372988021951e+05, -2.372652401821e+05, -2.372312955021e+05, -2.371969713581e+05, -2.371622709131e+05, -2.371271972921e+05, -2.370917535801e+05, -2.370559428241e+05, -2.370197680341e+05, -2.369832321841e+05, -2.369463382091e+05, -2.369090890111e+05, -2.368714874561e+05, -2.368335363751e+05, -2.367952385671e+05, -2.367565967941e+05, -2.367176137891e+05, -2.366782922501e+05, -2.366386348451e+05, -2.365986442091e+05, -2.365583229471e+05, -2.365176736341e+05, -2.364766988151e+05, -2.364354010051e+05, -2.363937826901e+05, -2.363518463291e+05, -2.363095943511e+05, -2.362670291591e+05, -2.362241531271e+05, -2.361809686041e+05, -2.361374779121e+05, -2.360936833471e+05, -2.360495871781e+05, -2.360051916521e+05, -2.359604989871e+05, -2.359155113811e+05, -2.358702310041e+05, -2.358246600051e+05, -2.357788005081e+05, -2.357326546151e+05, -2.356862244041e+05, -2.356395119311e+05, -2.355925192321e+05, -2.355452483181e+05, -2.354977011801e+05, -2.354498797881e+05, -2.354017860921e+05, -2.353534220201e+05, -2.353047894811e+05, -2.352558903621e+05, -2.352067265321e+05, -2.351572998411e+05, -2.351076121191e+05, -2.350576651761e+05, -2.350074608071e+05, -2.349570007841e+05, -2.349062868641e+05, -2.348553207861e+05, -2.348041042701e+05, -2.347526390201e+05, -2.347009267211e+05, -2.346489690441e+05, -2.345967676401e+05, -2.345443241461e+05, -2.344916401811e+05, -2.344387173511e+05, -2.343855572421e+05, -2.343321614281e+05, -2.342785314651e+05, -2.342246688961e+05, -2.341705752471e+05, -2.341162520301e+05, -2.340617007441e+05, -2.340069228711e+05, -2.339519198801e+05, -2.338966932251e+05, -2.338412443491e+05, -2.337855746781e+05, -2.337296856261e+05, -2.336735785931e+05, -2.336172549671e+05, -2.335607161221e+05, -2.335039634191e+05, -2.334469982081e+05, -2.333898218231e+05, -2.333324355901e+05, -2.332748408191e+05, -2.332170388101e+05, -2.331590308511e+05, -2.331008182181e+05, -2.330424021741e+05, -2.329837839741e+05, -2.329249648571e+05, -2.328659460541e+05, -2.328067287841e+05, -2.327473142561e+05, -2.326877036661e+05, -2.326278982021e+05, -2.325678990391e+05, -2.325077073441e+05, -2.324473242711e+05, -2.323867509661e+05, -2.323259885631e+05, -2.322650381891e+05, -2.322039009591e+05, -2.321425779771e+05, -2.320810703411e+05, -2.320193791371e+05, -2.319575054421e+05, -2.318954503231e+05, -2.318332148411e+05, -2.317708000441e+05, -2.317082069731e+05, -2.316454366611e+05, -2.315824901291e+05, -2.315193683941e+05, -2.314560724611e+05, -2.313926033261e+05, -2.313289619801e+05, -2.312651494041e+05, -2.312011665701e+05, -2.311370144421e+05, -2.310726939771e+05, -2.310082061251e+05, -2.309435518251e+05, -2.308787320121e+05, -2.308137476101e+05, -2.307485995371e+05, -2.306832887051e+05, -2.306178160151e+05, -2.305521823641e+05, -2.304863886411e+05, -2.304204357261e+05, -2.303543244941e+05, -2.302880558121e+05, -2.302216305411e+05, -2.301550495341e+05, -2.300883136371e+05, -2.300214236911e+05, -2.299543805291e+05, -2.298871849771e+05, -2.298198378571e+05, -2.297523399811e+05, -2.296846921571e+05, -2.296168951861e+05, -2.295489498641e+05, -2.294808569781e+05, -2.294126173121e+05, -2.293442316421e+05, -2.292757007381e+05, -2.292070253661e+05, -2.291382062841e+05, -2.290692442451e+05, -2.290001399971e+05, -2.289308942801e+05, -2.288615078321e+05, -2.287919813821e+05, -2.287223156551e+05, -2.286525113721e+05, -2.285825692441e+05, -2.285124899831e+05, -2.284422742901e+05, -2.283719228641e+05, -2.283014363981e+05, -2.282308155801e+05, -2.281600610921e+05, -2.280891736131e+05, -2.280181538141e+05, -2.279470023651e+05, -2.278757199271e+05, -2.278043071591e+05, -2.277327647131e+05, -2.276610932401e+05, -2.275892933811e+05, -2.275173657771e+05, -2.274453110621e+05, -2.273731298661e+05, -2.273008228141e+05, -2.272283905281e+05, -2.271558336241e+05, -2.270831527131e+05, -2.270103484051e+05, -2.269374213021e+05, -2.268643720041e+05, -2.267912011051e+05, -2.267179091961e+05, -2.266444968651e+05, -2.265709646931e+05, -2.264973132591e+05, -2.264235431371e+05, -2.263496548981e+05, -2.262756491091e+05, -2.262015263311e+05, -2.261272871241e+05, -2.260529320421e+05, -2.259784616361e+05, -2.259038764531e+05, -2.258291770381e+05, -2.257543639291e+05, -2.256794376621e+05, -2.256043987711e+05, -2.255292477841e+05, -2.254539852251e+05, -2.253786116181e+05, -2.253031274791e+05, -2.252275333241e+05, -2.251518296641e+05, -2.250760170061e+05, -2.250000958551e+05, -2.249240667121e+05, -2.248479300751e+05, -2.247716864381e+05, -2.246953362921e+05, -2.246188801261e+05, -2.245423184231e+05, -2.244656516661e+05, -2.243888803321e+05, -2.243120048971e+05, -2.242350258331e+05, -2.241579436091e+05, -2.240807586901e+05, -2.240034715411e+05, -2.239260826201e+05, -2.238485923851e+05, -2.237710012901e+05, -2.236933097861e+05, -2.236155183211e+05, -2.235376273401e+05, -2.234596372861e+05, -2.233815485991e+05, -2.233033617151e+05, -2.232250770681e+05, -2.231466950901e+05, -2.230682162101e+05, -2.229896408531e+05, -2.229109694431e+05, -2.228322023991e+05, -2.227533401411e+05, -2.226743830831e+05, -2.225953316381e+05, -2.225161862161e+05, -2.224369472251e+05, -2.223576150701e+05, -2.222781901541e+05, -2.221986728751e+05},
    {2.447530722587e+04, 2.263513643187e+04, 2.076360624387e+04, 1.885917433087e+04, 1.692017025287e+04, 1.494478025387e+04, 1.293102964987e+04, 1.087676234487e+04, 8.779616881870e+03, 6.636998288870e+03, 4.446044773870e+03, 2.203588081871e+03, -9.389402012946e+01, -2.450335121129e+03, -4.870150031129e+03, -7.358320588129e+03, -9.920502076129e+03, -1.256315662213e+04, -1.529372236413e+04, -1.812083079713e+04, -2.105459006213e+04, -2.410696024913e+04, -2.729225974913e+04, -3.062786284013e+04, -3.413518406413e+04, -3.784110658313e+04, -4.178012413913e+04, -4.599768230013e+04, -5.055564795313e+04, -5.554182056913e+04, -6.108781941413e+04, -6.740648963013e+04, -7.488312465713e+04, -8.436084163313e+04, -9.864293539113e+04, -1.929319788751e+05, -1.976812424781e+05, -2.008208433491e+05, -2.032257740751e+05, -2.051959240461e+05, -2.068744066971e+05, -2.083416891741e+05, -2.096478778911e+05, -2.108264657721e+05, -2.119010478791e+05, -2.128889355031e+05, -2.138032477251e+05, -2.146541927821e+05, -2.154498900841e+05, -2.161969180511e+05, -2.169006911791e+05, -2.175657268121e+05, -2.181958384631e+05, -2.187942788671e+05, -2.193638478131e+05, -2.199069747781e+05, -2.204257831851e+05, -2.209221410451e+05, -2.213977013321e+05, -2.218539345481e+05, -2.222921552211e+05, -2.227135436811e+05, -2.231191640991e+05, -2.235099795351e+05, -2.238868645951e+05, -2.242506161291e+05, -2.246019623391e+05, -2.249415705721e+05, -2.252700540231e+05, -2.255879775251e+05, -2.258958625831e+05, -2.261941917551e+05, -2.264834124891e+05, -2.267639404951e+05, -2.270361627221e+05, -2.273004399781e+05, -2.275571092721e+05, -2.278064858841e+05, -2.280488652201e+05, -2.282845244691e+05, -2.285137240941e+05, -2.287367091631e+05, -2.289537105611e+05, -2.291649460741e+05, -2.293706213771e+05, -2.295709309291e+05, -2.297660587851e+05, -2.299561793341e+05, -2.301414579761e+05, -2.303220517401e+05, -2.304981098431e+05, -2.306697742141e+05, -2.308371799641e+05, -2.310004558241e+05, -2.311597245481e+05, -2.313151032831e+05, -2.314667039101e+05, -2.316146333621e+05, -2.317589939161e+05, -2.318998834641e+05, -2.320373957671e+05, -2.321716206821e+05, -2.323026443891e+05, -2.324305495851e+05, -2.325554156781e+05, -2.326773189581e+05, -2.327963327651e+05, -2.329125276411e+05, -2.330259714741e+05, -2.331367296301e+05, -2.332448650821e+05, -2.333504385261e+05, -2.334535084921e+05, -2.335541314491e+05, -2.336523619011e+05, -2.337482524821e+05, -2.338418540381e+05, -2.339332157151e+05, -2.340223850291e+05, -2.341094079441e+05, -2.341943289381e+05, -2.342771910691e+05, -2.343580360341e+05, -2.344369042311e+05, -2.345138348101e+05, -2.345888657251e+05, -2.346620337891e+05, -2.347333747111e+05, -2.348029231501e+05, -2.348707127511e+05, -2.349367761871e+05, -2.350011451971e+05, -2.350638506221e+05, -2.351249224391e+05, -2.351843897931e+05, -2.352422810321e+05, -2.352986237321e+05, -2.353534447261e+05, -2.354067701361e+05, -2.354586253921e+05, -2.355090352601e+05, -2.355580238661e+05, -2.356056147181e+05, -2.356518307261e+05, -2.356966942241e+05, -2.357402269891e+05, -2.357824502601e+05, -2.358233847581e+05, -2.358630506981e+05, -2.359014678111e+05, -2.359386553581e+05, -2.359746321441e+05, -2.360094165331e+05, -2.360430264631e+05, -2.360754794591e+05, -2.361067926461e+05, -2.361369827601e+05, -2.361660661631e+05, -2.361940588501e+05, -2.362209764641e+05, -2.362468343061e+05, -2.362716473421e+05, -2.362954302161e+05, -2.363181972591e+05, -2.363399624981e+05, -2.363607396631e+05, -2.363805421971e+05, -2.363993832651e+05, -2.364172757601e+05, -2.364342323121e+05, -2.364502652931e+05, -2.364653868281e+05, -2.364796087981e+05, -2.364929428481e+05, -2.365054003951e+05, -2.365169926301e+05, -2.365277305301e+05, -2.365376248561e+05, -2.365466861661e+05, -2.365549248151e+05, -2.365623509641e+05, -2.365689745821e+05, -2.365748054531e+05, -2.365798531791e+05, -2.365841271861e+05, -2.365876367271e+05, -2.365903908881e+05, -2.365923985901e+05, -2.365936685951e+05, -2.365942095081e+05, -2.365940297831e+05, -2.365931377251e+05, -2.365915414941e+05, -2.365892491071e+05, -2.365862684461e+05, -2.365826072531e+05, -2.365782731421e+05, -2.365732735981e+05, -2.365676159771e+05, -2.365613075141e+05, -2.365543553251e+05, -2.365467664061e+05, -2.365385476381e+05, -2.365297057911e+05, -2.365202475251e+05, -2.365101793921e+05, -2.364995078371e+05, -2.364882392061e+05, -2.364763797411e+05, -2.364639355871e+05, -2.364509127911e+05, -2.364373173081e+05, -2.364231549991e+05, -2.364084316351e+05, -2.363931528981e+05, -2.363773243841e+05, -2.363609516021e+05, -2.363440399821e+05, -2.363265948671e+05, -2.363086215261e+05, -2.362901251441e+05, -2.362711108331e+05, -2.362515836291e+05, -2.362315484941e+05, -2.362110103181e+05, -2.361899739201e+05, -2.361684440501e+05, -2.361464253891e+05, -2.361239225521e+05, -2.361009400901e+05, -2.360774824861e+05, -2.360535541631e+05, -2.360291594821e+05, -2.360043027411e+05, -2.359789881831e+05, -2.359532199871e+05, -2.359270022791e+05, -2.359003391271e+05, -2.358732345441e+05, -2.358456924891e+05, -2.358177168671e+05, -2.357893115321e+05, -2.357604802861e+05, -2.357312268801e+05, -2.357015550171e+05, -2.356714683491e+05, -2.356409704831e+05, -2.356100649781e+05, -2.355787553461e+05, -2.355470450541e+05, -2.355149375261e+05, -2.354824361401e+05, -2.354495442331e+05, -2.354162650981e+05, -2.353826019891e+05, -2.353485581161e+05, -2.353141366511e+05, -2.352793407261e+05, -2.352441734341e+05, -2.352086378301e+05, -2.351727369321e+05, -2.351364737191e+05, -2.350998511371e+05, -2.350628720931e+05, -2.350255394621e+05, -2.349878560811e+05, -2.349498247571e+05, -2.349114482601e+05, -2.348727293281e+05, -2.348336706691e+05, -2.347942749561e+05, -2.347545448321e+05, -2.347144829091e+05, -2.346740917691e+05, -2.346333739631e+05, -2.345923320141e+05, -2.345509684151e+05, -2.345092856311e+05, -2.344672860991e+05, -2.344249722271e+05, -2.343823463971e+05, -2.343394109641e+05, -2.342961682561e+05, -2.342526205761e+05, -2.342087702011e+05, -2.341646193821e+05, -2.341201703451e+05, -2.340754252941e+05, -2.340303864051e+05, -2.339850558331e+05, -2.339394357091e+05, -2.338935281401e+05, -2.338473352111e+05, -2.338008589851e+05, -2.337541015021e+05, -2.337070647801e+05, -2.336597508161e+05, -2.336121615861e+05, -2.335642990461e+05, -2.335161651281e+05, -2.334677617491e+05, -2.334190908011e+05, -2.333701541591e+05, -2.333209536781e+05, -2.332714911941e+05, -2.332217685231e+05, -2.331717874641e+05, -2.331215497971e+05, -2.330710572821e+05, -2.330203116641e+05, -2.329693146691e+05, -2.329180680061e+05, -2.328665733661e+05, -2.328148324231e+05, -2.327628468361e+05, -2.327106182461e+05, -2.326581482781e+05, -2.326054385411e+05, -2.325524906281e+05, -2.324993061171e+05, -2.324458865691e+05, -2.323922335321e+05, -2.323383485381e+05, -2.322842331041e+05, -2.322298887311e+05, -2.321753169091e+05, -2.321205191101e+05, -2.320654967941e+05, -2.320102514071e+05, -2.319547843811e+05, -2.318990971351e+05, -2.318431910731e+05, -2.317870675871e+05, -2.317307280561e+05, -2.316741738461e+05, -2.316174063091e+05, -2.315604267881e+05, -2.315032366101e+05, -2.314458370901e+05, -2.313882295341e+05, -2.313304152321e+05, -2.312723954651e+05, -2.312141715011e+05, -2.311557445981e+05, -2.310971160001e+05, -2.310382869431e+05, -2.309792586491e+05, -2.309200323311e+05, -2.308606091901e+05, -2.308009904171e+05, -2.307411771921e+05, -2.306811706851e+05, -2.306209720541e+05, -2.305605824501e+05, -2.305000030111e+05, -2.304392348671e+05, -2.303782791361e+05, -2.303171369281e+05, -2.302558093441e+05, -2.301942974721e+05, -2.301326023951e+05, -2.300707251841e+05, -2.300086669011e+05, -2.299464286001e+05, -2.298840113251e+05, -2.298214161121e+05, -2.297586439871e+05, -2.296956959691e+05, -2.296325730681e+05, -2.295692762831e+05, -2.295058066081e+05, -2.294421650271e+05, -2.293783525161e+05, -2.293143700441e+05, -2.292502185701e+05, -2.291858990461e+05, -2.291214124171e+05, -2.290567596201e+05, -2.289919415821e+05, -2.289269592251e+05, -2.288618134631e+05, -2.287965052031e+05, -2.287310353431e+05, -2.286654047741e+05, -2.285996143821e+05, -2.285336650441e+05, -2.284675576311e+05, -2.284012930041e+05, -2.283348720231e+05, -2.282682955351e+05, -2.282015643841e+05, -2.281346794061e+05, -2.280676414321e+05, -2.280004512831e+05, -2.279331097781e+05, -2.278656177261e+05, -2.277979759321e+05, -2.277301851931e+05, -2.276622463011e+05, -2.275941600421e+05, -2.275259271941e+05, -2.274575485321e+05, -2.273890248221e+05, -2.273203568271e+05, -2.272515453021e+05, -2.271825909971e+05, -2.271134946561e+05, -2.270442570191e+05, -2.269748788191e+05, -2.269053607821e+05, -2.268357036311e+05, -2.267659080831e+05, -2.266959748491e+05, -2.266259046341e+05, -2.265556981411e+05, -2.264853560641e+05, -2.264148790941e+05, -2.263442679161e+05, -2.262735232101e+05, -2.262026456521e+05, -2.261316359111e+05, -2.260604946541e+05, -2.259892225401e+05, -2.259178202261e+05, -2.258462883621e+05, -2.257746275951e+05, -2.257028385661e+05, -2.256309219121e+05, -2.255588782651e+05, -2.254867082541e+05, -2.254144125021e+05, -2.253419916271e+05, -2.252694462431e+05, -2.251967769621e+05, -2.251239843891e+05, -2.250510691251e+05, -2.249780317681e+05, -2.249048729101e+05, -2.248315931411e+05, -2.247581930441e+05, -2.246846732021e+05, -2.246110341901e+05, -2.245372765811e+05, -2.244634009441e+05, -2.243894078441e+05, -2.243152978401e+05, -2.242410714901e+05, -2.241667293481e+05, -2.240922719621e+05, -2.240176998781e+05, -2.239430136381e+05, -2.238682137801e+05, -2.237933008381e+05, -2.237182753431e+05, -2.236431378231e+05, -2.235678888001e+05, -2.234925287961e+05, -2.234170583271e+05, -2.233414779061e+05, -2.232657880431e+05, -2.231899892441e+05, -2.231140820121e+05, -2.230380668471e+05, -2.229619442461e+05, -2.228857147001e+05, -2.228093787001e+05, -2.227329367331e+05, -2.226563892821e+05, -2.225797368261e+05, -2.225029798441e+05, -2.224261188091e+05, -2.223491541911e+05, -2.222720864591e+05, -2.221949160781e+05, -2.221176435091e+05, -2.220402692101e+05, -2.219627936381e+05, -2.218852172461e+05, -2.218075404831e+05, -2.217297637971e+05, -2.216518876311e+05, -2.215739124281e+05, -2.214958386251e+05, -2.214176666591e+05, -2.213393969621e+05, -2.212610299651e+05, -2.211825660951e+05, -2.211040057781e+05, -2.210253494351e+05, -2.209465974861e+05, -2.208677503481e+05, -2.207888084351e+05, -2.207097721601e+05, -2.206306419301e+05, -2.205514181541e+05},
    {2.534792975787e+04, 2.352158787187e+04, 2.166456983887e+04, 1.977539355587e+04, 1.785245602187e+04, 1.589401925787e+04, 1.389819405887e+04, 1.186292114587e+04, 9.785949208871e+03, 7.664809166871e+03, 5.496783842871e+03, 3.278872011870e+03, 1.007745483871e+03, -1.320302474129e+03, -3.709419619129e+03, -6.164271047129e+03, -8.690133274129e+03, -1.129301136113e+04, -1.397978635513e+04, -1.675840314213e+04, -1.963811303413e+04, -2.262979177213e+04, -2.574636346613e+04, -2.900337667913e+04, -3.241980446913e+04, -3.601918380713e+04, -3.983128665313e+04, -4.389465757613e+04, -4.826063227913e+04, -5.300003847113e+04, -5.821512385613e+04, -6.406268781413e+04, -7.080456203913e+04, -7.893891325913e+04, -8.966061420713e+04, -1.082242701141e+05, -1.887641428401e+05, -1.940782381601e+05, -1.974968252721e+05, -2.000793326341e+05, -2.021763660241e+05, -2.039519350001e+05, -2.054969863891e+05, -2.068675534071e+05, -2.081007687451e+05, -2.092226026011e+05, -2.102519907761e+05, -2.112032062351e+05, -2.120873027961e+05, -2.129130360481e+05, -2.136874739681e+05, -2.144164152251e+05, -2.151046838501e+05, -2.157563419171e+05, -2.163748463261e+05, -2.169631665881e+05, -2.175238747741e+05, -2.180592152741e+05, -2.185711595951e+05, -2.190614499651e+05, -2.195316343891e+05, -2.199830951331e+05, -2.204170720761e+05, -2.208346820101e+05, -2.212369347311e+05, -2.216247465301e+05, -2.219989515951e+05, -2.223603116971e+05, -2.227095244731e+05, -2.230472305301e+05, -2.233740195951e+05, -2.236904358331e+05, -2.239969824891e+05, -2.242941259441e+05, -2.245822992751e+05, -2.248619053911e+05, -2.251333198081e+05, -2.253968931051e+05, -2.256529531121e+05, -2.259018068611e+05, -2.261437423261e+05, -2.263790299941e+05, -2.266079242621e+05, -2.268306647091e+05, -2.270474772371e+05, -2.272585751081e+05, -2.274641598771e+05, -2.276644222501e+05, -2.278595428541e+05, -2.280496929481e+05, -2.282350350671e+05, -2.284157236121e+05, -2.285919053941e+05, -2.287637201281e+05, -2.289313008911e+05, -2.290947745431e+05, -2.292542621131e+05, -2.294098791561e+05, -2.295617360841e+05, -2.297099384741e+05, -2.298545873461e+05, -2.299957794321e+05, -2.301336074131e+05, -2.302681601521e+05, -2.303995229001e+05, -2.305277774951e+05, -2.306530025441e+05, -2.307752735961e+05, -2.308946633001e+05, -2.310112415521e+05, -2.311250756411e+05, -2.312362303741e+05, -2.313447682021e+05, -2.314507493361e+05, -2.315542318491e+05, -2.316552717881e+05, -2.317539232591e+05, -2.318502385251e+05, -2.319442680861e+05, -2.320360607621e+05, -2.321256637671e+05, -2.322131227801e+05, -2.322984820111e+05, -2.323817842691e+05, -2.324630710191e+05, -2.325423824361e+05, -2.326197574661e+05, -2.326952338711e+05, -2.327688482801e+05, -2.328406362351e+05, -2.329106322351e+05, -2.329788697751e+05, -2.330453813871e+05, -2.331101986781e+05, -2.331733523661e+05, -2.332348723111e+05, -2.332947875501e+05, -2.333531263281e+05, -2.334099161241e+05, -2.334651836831e+05, -2.335189550391e+05, -2.335712555461e+05, -2.336221098931e+05, -2.336715421381e+05, -2.337195757221e+05, -2.337662334931e+05, -2.338115377291e+05, -2.338555101521e+05, -2.338981719511e+05, -2.339395437991e+05, -2.339796458681e+05, -2.340184978481e+05, -2.340561189591e+05, -2.340925279711e+05, -2.341277432151e+05, -2.341617825991e+05, -2.341946636161e+05, -2.342264033661e+05, -2.342570185591e+05, -2.342865255331e+05, -2.343149402641e+05, -2.343422783721e+05, -2.343685551421e+05, -2.343937855211e+05, -2.344179841391e+05, -2.344411653131e+05, -2.344633430561e+05, -2.344845310871e+05, -2.345047428381e+05, -2.345239914641e+05, -2.345422898501e+05, -2.345596506181e+05, -2.345760861331e+05, -2.345916085151e+05, -2.346062296391e+05, -2.346199611471e+05, -2.346328144501e+05, -2.346448007391e+05, -2.346559309851e+05, -2.346662159501e+05, -2.346756661901e+05, -2.346842920611e+05, -2.346921037221e+05, -2.346991111421e+05, -2.347053241081e+05, -2.347107522221e+05, -2.347154049111e+05, -2.347192914321e+05, -2.347224208721e+05, -2.347248021551e+05, -2.347264440471e+05, -2.347273551561e+05, -2.347275439401e+05, -2.347270187081e+05, -2.347257876241e+05, -2.347238587101e+05, -2.347212398521e+05, -2.347179387981e+05, -2.347139631671e+05, -2.347093204481e+05, -2.347040180051e+05, -2.346980630781e+05, -2.346914627871e+05, -2.346842241351e+05, -2.346763540111e+05, -2.346678591911e+05, -2.346587463391e+05, -2.346490220151e+05, -2.346386926721e+05, -2.346277646601e+05, -2.346162442301e+05, -2.346041375331e+05, -2.345914506251e+05, -2.345781894651e+05, -2.345643599231e+05, -2.345499677771e+05, -2.345350187161e+05, -2.345195183441e+05, -2.345034721771e+05, -2.344868856511e+05, -2.344697641191e+05, -2.344521128551e+05, -2.344339370541e+05, -2.344152418341e+05, -2.343960322391e+05, -2.343763132381e+05, -2.343560897291e+05, -2.343353665391e+05, -2.343141484251e+05, -2.342924400761e+05, -2.342702461151e+05, -2.342475710981e+05, -2.342244195191e+05, -2.342007958081e+05, -2.341767043321e+05, -2.341521493981e+05, -2.341271352551e+05, -2.341016660921e+05, -2.340757460411e+05, -2.340493791781e+05, -2.340225695221e+05, -2.339953210421e+05, -2.339676376491e+05, -2.339395232041e+05, -2.339109815181e+05, -2.338820163481e+05, -2.338526314051e+05, -2.338228303491e+05, -2.337926167941e+05, -2.337619943051e+05, -2.337309664031e+05, -2.336995365611e+05, -2.336677082111e+05, -2.336354847381e+05, -2.336028694861e+05, -2.335698657571e+05, -2.335364768081e+05, -2.335027058601e+05, -2.334685560891e+05, -2.334340306371e+05, -2.333991326011e+05, -2.333638650451e+05, -2.333282309931e+05, -2.332922334311e+05, -2.332558753121e+05, -2.332191595501e+05, -2.331820890261e+05, -2.331446665851e+05, -2.331068950391e+05, -2.330687771651e+05, -2.330303157091e+05, -2.329915133841e+05, -2.329523728701e+05, -2.329128968171e+05, -2.328730878421e+05, -2.328329485351e+05, -2.327924814541e+05, -2.327516891261e+05, -2.327105740511e+05, -2.326691387011e+05, -2.326273855181e+05, -2.325853169181e+05, -2.325429352881e+05, -2.325002429881e+05, -2.324572423551e+05, -2.324139356961e+05, -2.323703252931e+05, -2.323264134051e+05, -2.322822022631e+05, -2.322376940771e+05, -2.321928910291e+05, -2.321477952791e+05, -2.321024089641e+05, -2.320567341981e+05, -2.320107730711e+05, -2.319645276501e+05, -2.319179999821e+05, -2.318711920911e+05, -2.318241059791e+05, -2.317767436271e+05, -2.317291069971e+05, -2.316811980281e+05, -2.316330186391e+05, -2.315845707291e+05, -2.315358561791e+05, -2.314868768491e+05, -2.314376345791e+05, -2.313881311921e+05, -2.313383684901e+05, -2.312883482591e+05, -2.312380722661e+05, -2.311875422591e+05, -2.311367599701e+05, -2.310857271121e+05, -2.310344453811e+05, -2.309829164571e+05, -2.309311420031e+05, -2.308791236661e+05, -2.308268630741e+05, -2.307743618411e+05, -2.307216215671e+05, -2.306686438321e+05, -2.306154302041e+05, -2.305619822351e+05, -2.305083014601e+05, -2.304543894021e+05, -2.304002475671e+05, -2.303458774481e+05, -2.302912805221e+05, -2.302364582551e+05, -2.301814120971e+05, -2.301261434821e+05, -2.300706538351e+05, -2.300149445651e+05, -2.299590170681e+05, -2.299028727271e+05, -2.298465129121e+05, -2.297899389811e+05, -2.297331522781e+05, -2.296761541351e+05, -2.296189458741e+05, -2.295615288021e+05, -2.295039042141e+05, -2.294460733961e+05, -2.293880376201e+05, -2.293297981461e+05, -2.292713562241e+05, -2.292127130931e+05, -2.291538699791e+05, -2.290948280991e+05, -2.290355886581e+05, -2.289761528511e+05, -2.289165218611e+05, -2.288566968621e+05, -2.287966790181e+05, -2.287364694811e+05, -2.286760693931e+05, -2.286154798881e+05, -2.285547020881e+05, -2.284937371081e+05, -2.284325860491e+05, -2.283712500061e+05, -2.283097300641e+05, -2.282480272971e+05, -2.281861427731e+05, -2.281240775471e+05, -2.280618326671e+05, -2.279994091731e+05, -2.279368080951e+05, -2.278740304541e+05, -2.278110772631e+05, -2.277479495261e+05, -2.276846482401e+05, -2.276211743921e+05, -2.275575289611e+05, -2.274937129191e+05, -2.274297272291e+05, -2.273655728471e+05, -2.273012507191e+05, -2.272367617851e+05, -2.271721069791e+05, -2.271072872231e+05, -2.270423034341e+05, -2.269771565231e+05, -2.269118473911e+05, -2.268463769321e+05, -2.267807460351e+05, -2.267149555801e+05, -2.266490064401e+05, -2.265828994821e+05, -2.265166355651e+05, -2.264502155421e+05, -2.263836402581e+05, -2.263169105541e+05, -2.262500272611e+05, -2.261829912051e+05, -2.261158032071e+05, -2.260484640801e+05, -2.259809746301e+05, -2.259133356581e+05, -2.258455479581e+05, -2.257776123191e+05, -2.257095295221e+05, -2.256413003451e+05, -2.255729255561e+05, -2.255044059211e+05, -2.254357421981e+05, -2.253669351391e+05, -2.252979854911e+05, -2.252288939971e+05, -2.251596613911e+05, -2.250902884031e+05, -2.250207757591e+05, -2.249511241771e+05, -2.248813343711e+05, -2.248114070511e+05, -2.247413429181e+05, -2.246711426721e+05, -2.246008070051e+05, -2.245303366051e+05, -2.244597321541e+05, -2.243889943301e+05, -2.243181238061e+05, -2.242471212491e+05, -2.241759873231e+05, -2.241047226851e+05, -2.240333279901e+05, -2.239618038851e+05, -2.238901510141e+05, -2.238183700161e+05, -2.237464615271e+05, -2.236744261761e+05, -2.236022645891e+05, -2.235299773861e+05, -2.234575651861e+05, -2.233850285991e+05, -2.233123682341e+05, -2.232395846951e+05, -2.231666785811e+05, -2.230936504871e+05, -2.230205010051e+05, -2.229472307201e+05, -2.228738402171e+05, -2.228003300731e+05, -2.227267008631e+05, -2.226529531581e+05, -2.225790875251e+05, -2.225051045271e+05, -2.224310047231e+05, -2.223567886671e+05, -2.222824569111e+05, -2.222080100021e+05, -2.221334484851e+05, -2.220587729001e+05, -2.219839837821e+05, -2.219090816651e+05, -2.218340670781e+05, -2.217589405471e+05, -2.216837025931e+05, -2.216083537351e+05, -2.215328944891e+05, -2.214573253651e+05, -2.213816468721e+05, -2.213058595161e+05, -2.212299637971e+05, -2.211539602131e+05, -2.210778492601e+05, -2.210016314291e+05, -2.209253072091e+05, -2.208488770841e+05, -2.207723415371e+05, -2.206957010471e+05, -2.206189560881e+05, -2.205421071341e+05, -2.204651546551e+05, -2.203880991161e+05, -2.203109409821e+05, -2.202336807121e+05, -2.201563187651e+05, -2.200788555941e+05, -2.200012916521e+05, -2.199236273871e+05, -2.198458632451e+05, -2.197679996691e+05, -2.196900371001e+05, -2.196119759741e+05, -2.195338167271e+05, -2.194555597901e+05, -2.193772055921e+05, -2.192987545611e+05, -2.192202071201e+05, -2.191415636891e+05, -2.190628246891e+05, -2.189839905331e+05, -2.189050616361e+05},
    {2.622154717987e+04, 2.440886208787e+04, 2.256616694487e+04, 2.069203733487e+04, 1.878493472787e+04, 1.684319345487e+04, 1.486500570687e+04, 1.284840419487e+04, 1.079124200487e+04, 8.691169064871e+03, 6.545604521870e+03, 4.351704110871e+03, 2.106321374871e+03, -1.940387361295e+02, -2.553275794129e+03, -4.975761849130e+03, -7.466424700129e+03, -1.003085089913e+04, -1.267541449213e+04, -1.540743974913e+04, -1.823540944013e+04, -2.116923515813e+04, -2.422061363813e+04, -2.740350474013e+04, -3.073478547113e+04, -3.423516546413e+04, -3.793050255713e+04, -4.185375251813e+04, -4.604796651513e+04, -5.057110875913e+04, -5.550423816713e+04, -6.096641538213e+04, -6.714451939313e+04, -7.436121454713e+04, -8.326407155413e+04, -9.557765697713e+04, -1.234781771651e+05, -1.846413601241e+05, -1.904846285501e+05, -1.941758397021e+05, -1.969343037031e+05, -1.991579243471e+05, -2.010306854971e+05, -2.026537095081e+05, -2.040888523281e+05, -2.053768548141e+05, -2.065460576911e+05, -2.076170261381e+05, -2.086051944211e+05, -2.095224692301e+05, -2.103782485781e+05, -2.111800950101e+05, -2.119341952341e+05, -2.126456828711e+05, -2.133188706831e+05, -2.139574212641e+05, -2.145644749131e+05, -2.151427470241e+05, -2.156946034181e+05, -2.162221193891e+05, -2.167271265631e+05, -2.172112505051e+05, -2.176759412081e+05, -2.181224980391e+05, -2.185520903321e+05, -2.189657745111e+05, -2.193645084441e+05, -2.197491635421e+05, -2.201205350331e+05, -2.204793507291e+05, -2.208262785511e+05, -2.211619330171e+05, -2.214868808631e+05, -2.218016459391e+05, -2.221067134811e+05, -2.224025338621e+05, -2.226895258911e+05, -2.229680797341e+05, -2.232385594921e+05, -2.235013054991e+05, -2.237566363711e+05, -2.240048508351e+05, -2.242462293671e+05, -2.244810356691e+05, -2.247095179891e+05, -2.249319103261e+05, -2.251484335081e+05, -2.253592961721e+05, -2.255646956621e+05, -2.257648188321e+05, -2.259598427881e+05, -2.261499355671e+05, -2.263352567451e+05, -2.265159580111e+05, -2.266921836801e+05, -2.268640711731e+05, -2.270317514511e+05, -2.271953494251e+05, -2.273549843231e+05, -2.275107700381e+05, -2.276628154441e+05, -2.278112246931e+05, -2.279560974881e+05, -2.280975293351e+05, -2.282356117811e+05, -2.283704326351e+05, -2.285020761671e+05, -2.286306233031e+05, -2.287561518031e+05, -2.288787364251e+05, -2.289984490791e+05, -2.291153589771e+05, -2.292295327651e+05, -2.293410346521e+05, -2.294499265281e+05, -2.295562680811e+05, -2.296601168961e+05, -2.297615285601e+05, -2.298605567531e+05, -2.299572533341e+05, -2.300516684271e+05, -2.301438505011e+05, -2.302338464361e+05, -2.303217016011e+05, -2.304074599181e+05, -2.304911639181e+05, -2.305728548101e+05, -2.306525725291e+05, -2.307303557931e+05, -2.308062421501e+05, -2.308802680291e+05, -2.309524687841e+05, -2.310228787331e+05, -2.310915312071e+05, -2.311584585801e+05, -2.312236923121e+05, -2.312872629791e+05, -2.313492003111e+05, -2.314095332221e+05, -2.314682898401e+05, -2.315254975351e+05, -2.315811829481e+05, -2.316353720171e+05, -2.316880900021e+05, -2.317393615091e+05, -2.317892105111e+05, -2.318376603751e+05, -2.318847338781e+05, -2.319304532271e+05, -2.319748400841e+05, -2.320179155751e+05, -2.320597003171e+05, -2.321002144301e+05, -2.321394775531e+05, -2.321775088601e+05, -2.322143270761e+05, -2.322499504911e+05, -2.322843969731e+05, -2.323176839821e+05, -2.323498285791e+05, -2.323808474461e+05, -2.324107568891e+05, -2.324395728551e+05, -2.324673109411e+05, -2.324939864031e+05, -2.325196141701e+05, -2.325442088481e+05, -2.325677847341e+05, -2.325903558231e+05, -2.326119358161e+05, -2.326325381321e+05, -2.326521759091e+05, -2.326708620191e+05, -2.326886090721e+05, -2.327054294221e+05, -2.327213351771e+05, -2.327363382041e+05, -2.327504501351e+05, -2.327636823741e+05, -2.327760461051e+05, -2.327875522941e+05, -2.327982116951e+05, -2.328080348611e+05, -2.328170321431e+05, -2.328252136961e+05, -2.328325894881e+05, -2.328391693011e+05, -2.328449627361e+05, -2.328499792191e+05, -2.328542280051e+05, -2.328577181811e+05, -2.328604586721e+05, -2.328624582421e+05, -2.328637255011e+05, -2.328642689091e+05, -2.328640967741e+05, -2.328632172621e+05, -2.328616383991e+05, -2.328593680711e+05, -2.328564140301e+05, -2.328527838981e+05, -2.328484851661e+05, -2.328435252011e+05, -2.328379112481e+05, -2.328316504291e+05, -2.328247497531e+05, -2.328172161121e+05, -2.328090562841e+05, -2.328002769421e+05, -2.327908846471e+05, -2.327808858581e+05, -2.327702869301e+05, -2.327590941191e+05, -2.327473135801e+05, -2.327349513761e+05, -2.327220134701e+05, -2.327085057391e+05, -2.326944339641e+05, -2.326798038421e+05, -2.326646209811e+05, -2.326488909051e+05, -2.326326190541e+05, -2.326158107871e+05, -2.325984713851e+05, -2.325806060481e+05, -2.325622199011e+05, -2.325433179941e+05, -2.325239053021e+05, -2.325039867291e+05, -2.324835671081e+05, -2.324626512031e+05, -2.324412437101e+05, -2.324193492561e+05, -2.323969724071e+05, -2.323741176601e+05, -2.323507894521e+05, -2.323269921581e+05, -2.323027300911e+05, -2.322780075061e+05, -2.322528285981e+05, -2.322271975071e+05, -2.322011183131e+05, -2.321745950451e+05, -2.321476316741e+05, -2.321202321221e+05, -2.320924002541e+05, -2.320641398871e+05, -2.320354547871e+05, -2.320063486691e+05, -2.319768252001e+05, -2.319468880011e+05, -2.319165406431e+05, -2.318857866531e+05, -2.318546295121e+05, -2.318230726561e+05, -2.317911194791e+05, -2.317587733301e+05, -2.317260375161e+05, -2.316929153031e+05, -2.316594099161e+05, -2.316255245401e+05, -2.315912623191e+05, -2.315566263611e+05, -2.315216197321e+05, -2.314862454651e+05, -2.314505065511e+05, -2.314144059491e+05, -2.313779465801e+05, -2.313411313291e+05, -2.313039630501e+05, -2.312664445581e+05, -2.312285786391e+05, -2.311903680431e+05, -2.311518154891e+05, -2.311129236641e+05, -2.310736952221e+05, -2.310341327891e+05, -2.309942389571e+05, -2.309540162911e+05, -2.309134673261e+05, -2.308725945661e+05, -2.308314004881e+05, -2.307898875411e+05, -2.307480581451e+05, -2.307059146941e+05, -2.306634595551e+05, -2.306206950671e+05, -2.305776235441e+05, -2.305342472761e+05, -2.304905685231e+05, -2.304465895251e+05, -2.304023124951e+05, -2.303577396221e+05, -2.303128730701e+05, -2.302677149831e+05, -2.302222674791e+05, -2.301765326521e+05, -2.301305125781e+05, -2.300842093051e+05, -2.300376248651e+05, -2.299907612641e+05, -2.299436204891e+05, -2.298962045051e+05, -2.298485152571e+05, -2.298005546691e+05, -2.297523246471e+05, -2.297038270741e+05, -2.296550638151e+05, -2.296060367171e+05, -2.295567476061e+05, -2.295071982901e+05, -2.294573905601e+05, -2.294073261851e+05, -2.293570069211e+05, -2.293064345031e+05, -2.292556106481e+05, -2.292045370591e+05, -2.291532154191e+05, -2.291016473961e+05, -2.290498346401e+05, -2.289977787861e+05, -2.289454814511e+05, -2.288929442401e+05, -2.288401687371e+05, -2.287871565141e+05, -2.287339091281e+05, -2.286804281181e+05, -2.286267150121e+05, -2.285727713191e+05, -2.285185985381e+05, -2.284641981501e+05, -2.284095716231e+05, -2.283547204121e+05, -2.282996459571e+05, -2.282443496861e+05, -2.281888330111e+05, -2.281330973331e+05, -2.280771440401e+05, -2.280209745041e+05, -2.279645900881e+05, -2.279079921411e+05, -2.278511819981e+05, -2.277941609831e+05, -2.277369304101e+05, -2.276794915771e+05, -2.276218457721e+05, -2.275639942721e+05, -2.275059383421e+05, -2.274476792341e+05, -2.273892181921e+05, -2.273305564451e+05, -2.272716952131e+05, -2.272126357061e+05, -2.271533791221e+05, -2.270939266481e+05, -2.270342794611e+05, -2.269744387281e+05, -2.269144056051e+05, -2.268541812391e+05, -2.267937667661e+05, -2.267331633121e+05, -2.266723719951e+05, -2.266113939191e+05, -2.265502301841e+05, -2.264888818771e+05, -2.264273500771e+05, -2.263656358521e+05, -2.263037402641e+05, -2.262416643621e+05, -2.261794091901e+05, -2.261169757821e+05, -2.260543651601e+05, -2.259915783431e+05, -2.259286163361e+05, -2.258654801401e+05, -2.258021707461e+05, -2.257386891341e+05, -2.256750362821e+05, -2.256112131531e+05, -2.255472207081e+05, -2.254830598961e+05, -2.254187316591e+05, -2.253542369341e+05, -2.252895766481e+05, -2.252247517191e+05, -2.251597630611e+05, -2.250946115791e+05, -2.250292981701e+05, -2.249638237241e+05, -2.248981891261e+05, -2.248323952501e+05, -2.247664429671e+05, -2.247003331391e+05, -2.246340666211e+05, -2.245676442621e+05, -2.245010669031e+05, -2.244343353811e+05, -2.243674505251e+05, -2.243004131551e+05, -2.242332240891e+05, -2.241658841361e+05, -2.240983940991e+05, -2.240307547751e+05, -2.239629669551e+05, -2.238950314241e+05, -2.238269489601e+05, -2.237587203371e+05, -2.236903463211e+05, -2.236218276731e+05, -2.235531651481e+05, -2.234843594971e+05, -2.234154114611e+05, -2.233463217811e+05, -2.232770911881e+05, -2.232077204101e+05, -2.231382101671e+05, -2.230685611781e+05, -2.229987741521e+05, -2.229288497941e+05, -2.228587888071e+05, -2.227885918841e+05, -2.227182597161e+05, -2.226477929871e+05, -2.225771923791e+05, -2.225064585651e+05, -2.224355922161e+05, -2.223645939971e+05, -2.222934645691e+05, -2.222222045861e+05, -2.221508147001e+05, -2.220792955571e+05, -2.220076477981e+05, -2.219358720601e+05, -2.218639689761e+05, -2.217919391731e+05, -2.217197832741e+05, -2.216475018981e+05, -2.215750956611e+05, -2.215025651701e+05, -2.214299110341e+05, -2.213571338521e+05, -2.212842342231e+05, -2.212112127391e+05, -2.211380699901e+05, -2.210648065591e+05, -2.209914230291e+05, -2.209179199751e+05, -2.208442979711e+05, -2.207705575851e+05, -2.206966993811e+05, -2.206227239221e+05, -2.205486317631e+05, -2.204744234591e+05, -2.204000995581e+05, -2.203256606071e+05, -2.202511071471e+05, -2.201764397171e+05, -2.201016588511e+05, -2.200267650811e+05, -2.199517589341e+05, -2.198766409341e+05, -2.198014116011e+05, -2.197260714521e+05, -2.196506210011e+05, -2.195750607581e+05, -2.194993912301e+05, -2.194236129191e+05, -2.193477263261e+05, -2.192717319471e+05, -2.191956302761e+05, -2.191194218031e+05, -2.190431070141e+05, -2.189666863941e+05, -2.188901604231e+05, -2.188135295781e+05, -2.187367943341e+05, -2.186599551621e+05, -2.185830125311e+05, -2.185059669041e+05, -2.184288187451e+05, -2.183515685121e+05, -2.182742166621e+05, -2.181967636481e+05, -2.181192099211e+05, -2.180415559281e+05, -2.179638021131e+05, -2.178859489181e+05, -2.178079967831e+05, -2.177299461441e+05, -2.176517974341e+05, -2.175735510841e+05, -2.174952075221e+05, -2.174167671731e+05, -2.173382304601e+05, -2.172595978041e+05},
    {2.709615860887e+04, 2.529696123087e+04, 2.346840326187e+04, 2.160911554687e+04, 1.971762121287e+04, 1.779232360687e+04, 1.583149245487e+04, 1.383324790387e+04, 1.179554202987e+04, 9.716137322871e+03, 7.592581506871e+03, 5.422177916871e+03, 3.201950431871e+03, 9.286017018705e+02, -1.401536964129e+03, -3.792566160129e+03, -6.249092190129e+03, -8.776317778129e+03, -1.138015468513e+04, -1.406736495613e+04, -1.684574019713e+04, -1.972433204513e+04, -2.271375273113e+04, -2.582657340613e+04, -2.907786164313e+04, -3.248592183713e+04, -3.607333943113e+04, -3.986849454713e+04, -4.390782783513e+04, -4.823936545913e+04, -5.292846608213e+04, -5.806775483913e+04, -6.379563674513e+04, -7.033446107013e+04, -7.808132167513e+04, -8.787819564313e+04, -1.022330898581e+05, -1.529861816691e+05, -1.806424110361e+05, -1.869269657451e+05, -1.908713833531e+05, -1.937988814911e+05, -1.961461474481e+05, -1.981147253961e+05, -1.998150275421e+05, -2.013143630491e+05, -2.026569155861e+05, -2.038733196351e+05, -2.049857336731e+05, -2.060107364931e+05, -2.069610801671e+05, -2.078468020841e+05, -2.086759583631e+05, -2.094551236311e+05, -2.101897412891e+05, -2.108843750171e+05, -2.115428932721e+05, -2.121686072001e+05, -2.127643754791e+05, -2.133326852391e+05, -2.138757153621e+05, -2.143953866331e+05, -2.148934019041e+05, -2.153712786021e+05, -2.158303752891e+05, -2.162719135401e+05, -2.166969961221e+05, -2.171066222001e+05, -2.175017001451e+05, -2.178830583931e+05, -2.182514547021e+05, -2.186075840901e+05, -2.189520856671e+05, -2.192855485591e+05, -2.196085170521e+05, -2.199214950831e+05, -2.202249501831e+05, -2.205193169341e+05, -2.208050000351e+05, -2.210823770031e+05, -2.213518005821e+05, -2.216136008871e+05, -2.218680873161e+05, -2.221155502691e+05, -2.223562626821e+05, -2.225904814211e+05, -2.228184485251e+05, -2.230403923431e+05, -2.232565285541e+05, -2.234670610991e+05, -2.236721830241e+05, -2.238720772561e+05, -2.240669173001e+05, -2.242568678871e+05, -2.244420855591e+05, -2.246227192131e+05, -2.247989105931e+05, -2.249707947491e+05, -2.251385004581e+05, -2.253021506071e+05, -2.254618625591e+05, -2.256177484761e+05, -2.257699156311e+05, -2.259184666921e+05, -2.260634999821e+05, -2.262051097281e+05, -2.263433862881e+05, -2.264784163611e+05, -2.266102831891e+05, -2.267390667381e+05, -2.268648438721e+05, -2.269876885121e+05, -2.271076717911e+05, -2.272248621921e+05, -2.273393256791e+05, -2.274511258251e+05, -2.275603239251e+05, -2.276669791071e+05, -2.277711484351e+05, -2.278728870051e+05, -2.279722480361e+05, -2.280692829581e+05, -2.281640414911e+05, -2.282565717201e+05, -2.283469201721e+05, -2.284351318781e+05, -2.285212504431e+05, -2.286053181051e+05, -2.286873757901e+05, -2.287674631721e+05, -2.288456187191e+05, -2.289218797481e+05, -2.289962824681e+05, -2.290688620241e+05, -2.291396525421e+05, -2.292086871651e+05, -2.292759980951e+05, -2.293416166281e+05, -2.294055731851e+05, -2.294678973521e+05, -2.295286179031e+05, -2.295877628371e+05, -2.296453594001e+05, -2.297014341181e+05, -2.297560128201e+05, -2.298091206621e+05, -2.298607821511e+05, -2.299110211711e+05, -2.299598609991e+05, -2.300073243291e+05, -2.300534332941e+05, -2.300982094781e+05, -2.301416739421e+05, -2.301838472341e+05, -2.302247494141e+05, -2.302644000611e+05, -2.303028182961e+05, -2.303400227911e+05, -2.303760317861e+05, -2.304108631041e+05, -2.304445341591e+05, -2.304770619741e+05, -2.305084631891e+05, -2.305387540761e+05, -2.305679505461e+05, -2.305960681641e+05, -2.306231221561e+05, -2.306491274201e+05, -2.306740985381e+05, -2.306980497801e+05, -2.307209951171e+05, -2.307429482291e+05, -2.307639225121e+05, -2.307839310871e+05, -2.308029868061e+05, -2.308211022621e+05, -2.308382897931e+05, -2.308545614931e+05, -2.308699292151e+05, -2.308844045771e+05, -2.308979989731e+05, -2.309107235751e+05, -2.309225893381e+05, -2.309336070091e+05, -2.309437871301e+05, -2.309531400451e+05, -2.309616759031e+05, -2.309694046641e+05, -2.309763361041e+05, -2.309824798211e+05, -2.309878452341e+05, -2.309924415961e+05, -2.309962779891e+05, -2.309993633341e+05, -2.310017063941e+05, -2.310033157781e+05, -2.310041999401e+05, -2.310043671911e+05, -2.310038256941e+05, -2.310025834751e+05, -2.310006484191e+05, -2.309980282791e+05, -2.309947306771e+05, -2.309907631041e+05, -2.309861329301e+05, -2.309808473981e+05, -2.309749136361e+05, -2.309683386511e+05, -2.309611293371e+05, -2.309532924771e+05, -2.309448347441e+05, -2.309357627031e+05, -2.309260828151e+05, -2.309158014391e+05, -2.309049248331e+05, -2.308934591561e+05, -2.308814104741e+05, -2.308687847541e+05, -2.308555878761e+05, -2.308418256271e+05, -2.308275037061e+05, -2.308126277251e+05, -2.307972032131e+05, -2.307812356151e+05, -2.307647302941e+05, -2.307476925351e+05, -2.307301275421e+05, -2.307120404461e+05, -2.306934363001e+05, -2.306743200861e+05, -2.306546967131e+05, -2.306345710171e+05, -2.306139477681e+05, -2.305928316671e+05, -2.305712273471e+05, -2.305491393771e+05, -2.305265722611e+05, -2.305035304421e+05, -2.304800182981e+05, -2.304560401491e+05, -2.304316002541e+05, -2.304067028141e+05, -2.303813519751e+05, -2.303555518221e+05, -2.303293063891e+05, -2.303026196541e+05, -2.302754955431e+05, -2.302479379271e+05, -2.302199506281e+05, -2.301915374181e+05, -2.301627020171e+05, -2.301334480991e+05, -2.301037792881e+05, -2.300736991631e+05, -2.300432112561e+05, -2.300123190531e+05, -2.299810259961e+05, -2.299493354831e+05, -2.299172508701e+05, -2.298847754701e+05, -2.298519125531e+05, -2.298186653511e+05, -2.297850370531e+05, -2.297510308101e+05, -2.297166497341e+05, -2.296818968981e+05, -2.296467753381e+05, -2.296112880541e+05, -2.295754380081e+05, -2.295392281271e+05, -2.295026613021e+05, -2.294657403901e+05, -2.294284682151e+05, -2.293908475651e+05, -2.293528811971e+05, -2.293145718361e+05, -2.292759221721e+05, -2.292369348671e+05, -2.291976125501e+05, -2.291579578201e+05, -2.291179732471e+05, -2.290776613691e+05, -2.290370246971e+05, -2.289960657131e+05, -2.289547868711e+05, -2.289131905971e+05, -2.288712792881e+05, -2.288290553171e+05, -2.287865210291e+05, -2.287436787431e+05, -2.287005307511e+05, -2.286570793221e+05, -2.286133266991e+05, -2.285692751001e+05, -2.285249267181e+05, -2.284802837251e+05, -2.284353482661e+05, -2.283901224651e+05, -2.283446084231e+05, -2.282988082171e+05, -2.282527239041e+05, -2.282063575171e+05, -2.281597110681e+05, -2.281127865501e+05, -2.280655859301e+05, -2.280181111601e+05, -2.279703641691e+05, -2.279223468641e+05, -2.278740611351e+05, -2.278255088531e+05, -2.277766918661e+05, -2.277276120071e+05, -2.276782710881e+05, -2.276286709031e+05, -2.275788132281e+05, -2.275286998211e+05, -2.274783324221e+05, -2.274277127531e+05, -2.273768425201e+05, -2.273257234121e+05, -2.272743570991e+05, -2.272227452371e+05, -2.271708894651e+05, -2.271187914041e+05, -2.270664526621e+05, -2.270138748281e+05, -2.269610594791e+05, -2.269080081751e+05, -2.268547224601e+05, -2.268012038631e+05, -2.267474539021e+05, -2.266934740751e+05, -2.266392658701e+05, -2.265848307581e+05, -2.265301701981e+05, -2.264752856341e+05, -2.264201784971e+05, -2.263648502041e+05, -2.263093021601e+05, -2.262535357541e+05, -2.261975523651e+05, -2.261413533591e+05, -2.260849400871e+05, -2.260283138901e+05, -2.259714760951e+05, -2.259144280181e+05, -2.258571709621e+05, -2.257997062191e+05, -2.257420350691e+05, -2.256841587801e+05, -2.256260786091e+05, -2.255677958011e+05, -2.255093115911e+05, -2.254506272021e+05, -2.253917438461e+05, -2.253326627251e+05, -2.252733850311e+05, -2.252139119431e+05, -2.251542446311e+05, -2.250943842551e+05, -2.250343319661e+05, -2.249740889011e+05, -2.249136561921e+05, -2.248530349571e+05, -2.247922263081e+05, -2.247312313431e+05, -2.246700511561e+05, -2.246086868271e+05, -2.245471394281e+05, -2.244854100241e+05, -2.244234996681e+05, -2.243614094061e+05, -2.242991402741e+05, -2.242366933001e+05, -2.241740695031e+05, -2.241112698941e+05, -2.240482954741e+05, -2.239851472381e+05, -2.239218261721e+05, -2.238583332521e+05, -2.237946694471e+05, -2.237308357201e+05, -2.236668330231e+05, -2.236026623021e+05, -2.235383244951e+05, -2.234738205311e+05, -2.234091513351e+05, -2.233443178201e+05, -2.232793208941e+05, -2.232141614591e+05, -2.231488404071e+05, -2.230833586241e+05, -2.230177169891e+05, -2.229519163741e+05, -2.228859576441e+05, -2.228198416571e+05, -2.227535692651e+05, -2.226871413131e+05, -2.226205586371e+05, -2.225538220701e+05, -2.224869324361e+05, -2.224198905551e+05, -2.223526972371e+05, -2.222853532901e+05, -2.222178595111e+05, -2.221502166961e+05, -2.220824256311e+05, -2.220144870971e+05, -2.219464018691e+05, -2.218781707171e+05, -2.218097944051e+05, -2.217412736891e+05, -2.216726093221e+05, -2.216038020501e+05, -2.215348526131e+05, -2.214657617471e+05, -2.213965301801e+05, -2.213271586381e+05, -2.212576478381e+05, -2.211879984941e+05, -2.211182113131e+05, -2.210482869991e+05, -2.209782262491e+05, -2.209080297551e+05, -2.208376982041e+05, -2.207672322791e+05, -2.206966326571e+05, -2.206259000091e+05, -2.205550350051e+05, -2.204840383051e+05, -2.204129105681e+05, -2.203416524461e+05, -2.202702645881e+05, -2.201987476381e+05, -2.201271022341e+05, -2.200553290101e+05, -2.199834285981e+05, -2.199114016211e+05, -2.198392487001e+05, -2.197669704531e+05, -2.196945674921e+05, -2.196220404231e+05, -2.195493898521e+05, -2.194766163761e+05, -2.194037205911e+05, -2.193307030881e+05, -2.192575644541e+05, -2.191843052711e+05, -2.191109261181e+05, -2.190374275701e+05, -2.189638101981e+05, -2.188900745671e+05, -2.188162212421e+05, -2.187422507801e+05, -2.186681637381e+05, -2.185939606651e+05, -2.185196421111e+05, -2.184452086181e+05, -2.183706607281e+05, -2.182959989761e+05, -2.182212238951e+05, -2.181463360151e+05, -2.180713358611e+05, -2.179962239561e+05, -2.179210008181e+05, -2.178456669631e+05, -2.177702229011e+05, -2.176946691431e+05, -2.176190061921e+05, -2.175432345501e+05, -2.174673547171e+05, -2.173913671861e+05, -2.173152724491e+05, -2.172390709961e+05, -2.171627633111e+05, -2.170863498771e+05, -2.170098311731e+05, -2.169332076751e+05, -2.168564798551e+05, -2.167796481841e+05, -2.167027131281e+05, -2.166256751501e+05, -2.165485347131e+05, -2.164712922731e+05, -2.163939482851e+05, -2.163165032021e+05, -2.162389574721e+05, -2.161613115421e+05, -2.160835658541e+05, -2.160057208501e+05, -2.159277769681e+05, -2.158497346421e+05, -2.157715943041e+05, -2.156933563841e+05, -2.156150213101e+05},
    {2.797176312987e+04, 2.618588734987e+04, 2.437128429687e+04, 2.252663776687e+04, 2.065052986187e+04, 1.874142982487e+04, 1.679768126487e+04, 1.481748746087e+04, 1.279889439687e+04, 1.073977107587e+04, 8.637786563870e+03, 6.490383080871e+03, 4.294744277871e+03, 2.047757603871e+03, -2.540306291294e+02, -2.614469278129e+03, -5.037867846130e+03, -7.529076499129e+03, -1.009358471213e+04, -1.273764413213e+04, -1.546842323713e+04, -1.829420437913e+04, -2.122463816513e+04, -2.427107673713e+04, -2.744701768613e+04, -3.076870646513e+04, -3.425597141013e+04, -3.793340979813e+04, -4.183212103913e+04, -4.599232566813e+04, -5.046748501613e+04, -5.533110786913e+04, -6.068871148013e+04, -6.670058570013e+04, -7.363004559713e+04, -8.196279066913e+04, -9.278345204713e+04, -1.096560792921e+05, -1.577092850901e+05, -1.768389155981e+05, -1.834330786561e+05, -1.875980791111e+05, -1.906820813881e+05, -1.931471574321e+05, -1.952085140951e+05, -1.969843741861e+05, -1.985468510591e+05, -1.999432604231e+05, -2.012063729441e+05, -2.023598572131e+05, -2.034213913071e+05, -2.044045474611e+05, -2.053199880661e+05, -2.061762544151e+05, -2.069803040011e+05, -2.077378868261e+05, -2.084538153821e+05, -2.091321624871e+05, -2.097764089921e+05, -2.103895559221e+05, -2.109742109031e+05, -2.115326556851e+05, -2.120668995421e+05, -2.125787219951e+05, -2.130697073321e+05, -2.135412727651e+05, -2.139946916091e+05, -2.144311125031e+05, -2.148515754711e+05, -2.152570254401e+05, -2.156483236771e+05, -2.160262575431e+05, -2.163915488291e+05, -2.167448609501e+05, -2.170868051491e+05, -2.174179459041e+05, -2.177388056281e+05, -2.180498688011e+05, -2.183515855931e+05, -2.186443750611e+05, -2.189286279831e+05, -2.192047093701e+05, -2.194729607031e+05, -2.197337019361e+05, -2.199872332791e+05, -2.202338368091e+05, -2.204737779151e+05, -2.207073065981e+05, -2.209346586501e+05, -2.211560567231e+05, -2.213717112941e+05, -2.215818215471e+05, -2.217865761731e+05, -2.219861541031e+05, -2.221807251751e+05, -2.223704507501e+05, -2.225554842661e+05, -2.227359717611e+05, -2.229120523421e+05, -2.230838586241e+05, -2.232515171331e+05, -2.234151486731e+05, -2.235748686791e+05, -2.237307875231e+05, -2.238830108211e+05, -2.240316396981e+05, -2.241767710451e+05, -2.243184977581e+05, -2.244569089531e+05, -2.245920901751e+05, -2.247241235871e+05, -2.248530881501e+05, -2.249790597891e+05, -2.251021115501e+05, -2.252223137441e+05, -2.253397340871e+05, -2.254544378261e+05, -2.255664878601e+05, -2.256759448541e+05, -2.257828673451e+05, -2.258873118431e+05, -2.259893329221e+05, -2.260889833141e+05, -2.261863139881e+05, -2.262813742311e+05, -2.263742117231e+05, -2.264648726081e+05, -2.265534015591e+05, -2.266398418391e+05, -2.267242353691e+05, -2.268066227731e+05, -2.268870434401e+05, -2.269655355731e+05, -2.270421362351e+05, -2.271168813971e+05, -2.271898059781e+05, -2.272609438921e+05, -2.273303280811e+05, -2.273979905591e+05, -2.274639624411e+05, -2.275282739791e+05, -2.275909545991e+05, -2.276520329231e+05, -2.277115368051e+05, -2.277694933591e+05, -2.278259289801e+05, -2.278808693761e+05, -2.279343395871e+05, -2.279863640141e+05, -2.280369664341e+05, -2.280861700301e+05, -2.281339974021e+05, -2.281804705941e+05, -2.282256111091e+05, -2.282694399281e+05, -2.283119775271e+05, -2.283532438921e+05, -2.283932585381e+05, -2.284320405211e+05, -2.284696084551e+05, -2.285059805221e+05, -2.285411744911e+05, -2.285752077261e+05, -2.286080972011e+05, -2.286398595111e+05, -2.286705108851e+05, -2.287000671931e+05, -2.287285439611e+05, -2.287559563791e+05, -2.287823193111e+05, -2.288076473041e+05, -2.288319546001e+05, -2.288552551401e+05, -2.288775625761e+05, -2.288988902791e+05, -2.289192513441e+05, -2.289386586011e+05, -2.289571246211e+05, -2.289746617221e+05, -2.289912819781e+05, -2.290069972241e+05, -2.290218190631e+05, -2.290357588711e+05, -2.290488278041e+05, -2.290610368061e+05, -2.290723966091e+05, -2.290829177441e+05, -2.290926105431e+05, -2.291014851431e+05, -2.291095514971e+05, -2.291168193701e+05, -2.291232983511e+05, -2.291289978541e+05, -2.291339271211e+05, -2.291380952301e+05, -2.291415110971e+05, -2.291441834771e+05, -2.291461209741e+05, -2.291473320401e+05, -2.291478249801e+05, -2.291476079541e+05, -2.291466889841e+05, -2.291450759541e+05, -2.291427766141e+05, -2.291397985831e+05, -2.291361493521e+05, -2.291318362881e+05, -2.291268666361e+05, -2.291212475191e+05, -2.291149859471e+05, -2.291080888131e+05, -2.291005629011e+05, -2.290924148821e+05, -2.290836513241e+05, -2.290742786881e+05, -2.290643033341e+05, -2.290537315211e+05, -2.290425694121e+05, -2.290308230711e+05, -2.290184984701e+05, -2.290056014891e+05, -2.289921379181e+05, -2.289781134591e+05, -2.289635337261e+05, -2.289484042511e+05, -2.289327304801e+05, -2.289165177821e+05, -2.288997714421e+05, -2.288824966701e+05, -2.288646985981e+05, -2.288463822851e+05, -2.288275527141e+05, -2.288082147971e+05, -2.287883733761e+05, -2.287680332251e+05, -2.287471990461e+05, -2.287258754781e+05, -2.287040670931e+05, -2.286817784011e+05, -2.286590138461e+05, -2.286357778121e+05, -2.286120746231e+05, -2.285879085431e+05, -2.285632837761e+05, -2.285382044721e+05, -2.285126747231e+05, -2.284866985641e+05, -2.284602799801e+05, -2.284334229001e+05, -2.284061312001e+05, -2.283784087081e+05, -2.283502591971e+05, -2.283216863961e+05, -2.282926939801e+05, -2.282632855801e+05, -2.282334647781e+05, -2.282032351121e+05, -2.281726000721e+05, -2.281415631041e+05, -2.281101276131e+05, -2.280782969581e+05, -2.280460744561e+05, -2.280134633841e+05, -2.279804669761e+05, -2.279470884281e+05, -2.279133308951e+05, -2.278791974931e+05, -2.278446913011e+05, -2.278098153601e+05, -2.277745726731e+05, -2.277389662081e+05, -2.277029988961e+05, -2.276666736331e+05, -2.276299932821e+05, -2.275929606701e+05, -2.275555785911e+05, -2.275178498061e+05, -2.274797770431e+05, -2.274413630001e+05, -2.274026103421e+05, -2.273635217021e+05, -2.273240996851e+05, -2.272843468641e+05, -2.272442657841e+05, -2.272038589591e+05, -2.271631288761e+05, -2.271220779941e+05, -2.270807087421e+05, -2.270390235231e+05, -2.269970247151e+05, -2.269547146651e+05, -2.269120956991e+05, -2.268691701131e+05, -2.268259401801e+05, -2.267824081481e+05, -2.267385762381e+05, -2.266944466501e+05, -2.266500215581e+05, -2.266053031121e+05, -2.265602934411e+05, -2.265149946501e+05, -2.264694088211e+05, -2.264235380141e+05, -2.263773842671e+05, -2.263309495971e+05, -2.262842359991e+05, -2.262372454471e+05, -2.261899798961e+05, -2.261424412781e+05, -2.260946315061e+05, -2.260465524741e+05, -2.259982060551e+05, -2.259495941051e+05, -2.259007184571e+05, -2.258515809301e+05, -2.258021833201e+05, -2.257525274091e+05, -2.257026149571e+05, -2.256524477091e+05, -2.256020273921e+05, -2.255513557141e+05, -2.255004343691e+05, -2.254492650311e+05, -2.253978493601e+05, -2.253461889971e+05, -2.252942855701e+05, -2.252421406881e+05, -2.251897559471e+05, -2.251371329241e+05, -2.250842731841e+05, -2.250311782761e+05, -2.249778497331e+05, -2.249242890741e+05, -2.248704978041e+05, -2.248164774121e+05, -2.247622293741e+05, -2.247077551521e+05, -2.246530561941e+05, -2.245981339351e+05, -2.245429897951e+05, -2.244876251821e+05, -2.244320414901e+05, -2.243762401001e+05, -2.243202223821e+05, -2.242639896911e+05, -2.242075433701e+05, -2.241508847511e+05, -2.240940151531e+05, -2.240369358821e+05, -2.239796482341e+05, -2.239221534911e+05, -2.238644529261e+05, -2.238065477981e+05, -2.237484393561e+05, -2.236901288381e+05, -2.236316174701e+05, -2.235729064691e+05, -2.235139970381e+05, -2.234548903721e+05, -2.233955876541e+05, -2.233360900591e+05, -2.232763987471e+05, -2.232165148731e+05, -2.231564395801e+05, -2.230961739981e+05, -2.230357192521e+05, -2.229750764551e+05, -2.229142467111e+05, -2.228532311121e+05, -2.227920307451e+05, -2.227306466841e+05, -2.226690799971e+05, -2.226073317391e+05, -2.225454029591e+05, -2.224832946981e+05, -2.224210079841e+05, -2.223585438411e+05, -2.222959032821e+05, -2.222330873111e+05, -2.221700969261e+05, -2.221069331141e+05, -2.220435968561e+05, -2.219800891241e+05, -2.219164108821e+05, -2.218525630861e+05, -2.217885466841e+05, -2.217243626171e+05, -2.216600118171e+05, -2.215954952101e+05, -2.215308137141e+05, -2.214659682401e+05, -2.214009596891e+05, -2.213357889581e+05, -2.212704569361e+05, -2.212049645051e+05, -2.211393125371e+05, -2.210735019021e+05, -2.210075334601e+05, -2.209414080651e+05, -2.208751265631e+05, -2.208086897961e+05, -2.207420985971e+05, -2.206753537931e+05, -2.206084562061e+05, -2.205414066501e+05, -2.204742059341e+05, -2.204068548581e+05, -2.203393542201e+05, -2.202717048081e+05, -2.202039074061e+05, -2.201359627921e+05, -2.200678717371e+05, -2.199996350071e+05, -2.199312533621e+05, -2.198627275561e+05, -2.197940583381e+05, -2.197252464511e+05, -2.196562926311e+05, -2.195871976111e+05, -2.195179621161e+05, -2.194485868671e+05, -2.193790725811e+05, -2.193094199661e+05, -2.192396297281e+05, -2.191697025671e+05, -2.190996391761e+05, -2.190294402461e+05, -2.189591064601e+05, -2.188886384991e+05, -2.188180370351e+05, -2.187473027401e+05, -2.186764362771e+05, -2.186054383061e+05, -2.185343094821e+05, -2.184630504551e+05, -2.183916618711e+05, -2.183201443711e+05, -2.182484985921e+05, -2.181767251641e+05, -2.181048247161e+05, -2.180327978701e+05, -2.179606452451e+05, -2.178883674551e+05, -2.178159651091e+05, -2.177434388131e+05, -2.176707891681e+05, -2.175980167721e+05, -2.175251222161e+05, -2.174521060901e+05, -2.173789689781e+05, -2.173057114601e+05, -2.172323341141e+05, -2.171588375111e+05, -2.170852222211e+05, -2.170114888071e+05, -2.169376378311e+05, -2.168636698501e+05, -2.167895854161e+05, -2.167153850801e+05, -2.166410693861e+05, -2.165666388761e+05, -2.164920940891e+05, -2.164174355601e+05, -2.163426638191e+05, -2.162677793951e+05, -2.161927828091e+05, -2.161176745851e+05, -2.160424552371e+05, -2.159671252801e+05, -2.158916852231e+05, -2.158161355741e+05, -2.157404768351e+05, -2.156647095071e+05, -2.155888340871e+05, -2.155128510671e+05, -2.154367609381e+05, -2.153605641871e+05, -2.152842612981e+05, -2.152078527511e+05, -2.151313390251e+05, -2.150547205921e+05, -2.149779979261e+05, -2.149011714931e+05, -2.148242417601e+05, -2.147472091891e+05, -2.146700742391e+05, -2.145928373651e+05, -2.145154990231e+05, -2.144380596621e+05, -2.143605197311e+05, -2.142828796731e+05, -2.142051399311e+05, -2.141273009451e+05, -2.140493631501e+05, -2.139713269811e+05},
    {2.884835978987e+04, 2.707564238887e+04, 2.527481536787e+04, 2.344461327087e+04, 2.158367462487e+04, 1.969053160587e+04, 1.776359824487e+04, 1.580115689387e+04, 1.380134265987e+04, 1.176212538087e+04, 9.681288697871e+03, 7.556405601871e+03, 5.384809712871e+03, 3.163561360871e+03, 8.894072658705e+02, -1.441267714130e+03, -3.832498742129e+03, -6.288811653130e+03, -8.815309524129e+03, -1.141777971313e+04, -1.410282758513e+04, -1.687804540713e+04, -1.975222831013e+04, -2.273565418813e+04, -2.584045197613e+04, -2.908109449213e+04, -3.247507069513e+04, -3.604382289313e+04, -3.981408656313e+04, -4.381986261113e+04, -4.810542287213e+04, -5.273008476113e+04, -5.777619382113e+04, -6.336335484913e+04, -6.967600883113e+04, -7.702324576913e+04, -8.599118516213e+04, -9.794072402913e+04, -1.174947432001e+05, -1.573552108491e+05, -1.732760922541e+05, -1.800290296271e+05, -1.843707921831e+05, -1.875934262541e+05, -1.901675453341e+05, -1.923168811811e+05, -1.941654568621e+05, -1.957892769421e+05, -1.972383341991e+05, -1.985472941851e+05, -1.997412017551e+05, -2.008387573761e+05, -2.018543079631e+05, -2.027991132851e+05, -2.036821825841e+05, -2.045108451771e+05, -2.052911503721e+05, -2.060281544801e+05, -2.067261311321e+05, -2.073887282561e+05, -2.080190871941e+05, -2.086199344431e+05, -2.091936532671e+05, -2.097423402831e+05, -2.102678506761e+05, -2.107718347011e+05, -2.112557674281e+05, -2.117209731881e+05, -2.121686458391e+05, -2.125998656701e+05, -2.130156136161e+05, -2.134167832831e+05, -2.138041911721e+05, -2.141785854381e+05, -2.145406534131e+05, -2.148910281131e+05, -2.152302938841e+05, -2.155589913261e+05, -2.158776215991e+05, -2.161866502131e+05, -2.164865103641e+05, -2.167776058821e+05, -2.170603138581e+05, -2.173349869681e+05, -2.176019555631e+05, -2.178615295211e+05, -2.181139999281e+05, -2.183596405731e+05, -2.185987093081e+05, -2.188314492691e+05, -2.190580899881e+05, -2.192788483911e+05, -2.194939297211e+05, -2.197035283641e+05, -2.199078286081e+05, -2.201070053411e+05, -2.203012246821e+05, -2.204906445641e+05, -2.206754152691e+05, -2.208556799191e+05, -2.210315749291e+05, -2.212032304221e+05, -2.213707706161e+05, -2.215343141811e+05, -2.216939745661e+05, -2.218498603041e+05, -2.220020752991e+05, -2.221507190861e+05, -2.222958870751e+05, -2.224376707801e+05, -2.225761580331e+05, -2.227114331781e+05, -2.228435772581e+05, -2.229726681861e+05, -2.230987809101e+05, -2.232219875571e+05, -2.233423575831e+05, -2.234599578991e+05, -2.235748530001e+05, -2.236871050791e+05, -2.237967741371e+05, -2.239039180901e+05, -2.240085928611e+05, -2.241108524771e+05, -2.242107491501e+05, -2.243083333651e+05, -2.244036539501e+05, -2.244967581551e+05, -2.245876917161e+05, -2.246764989231e+05, -2.247632226791e+05, -2.248479045621e+05, -2.249305848771e+05, -2.250113027071e+05, -2.250900959701e+05, -2.251670014551e+05, -2.252420548791e+05, -2.253152909191e+05, -2.253867432581e+05, -2.254564446241e+05, -2.255244268221e+05, -2.255907207751e+05, -2.256553565521e+05, -2.257183634021e+05, -2.257797697851e+05, -2.258396033971e+05, -2.258978912041e+05, -2.259546594601e+05, -2.260099337401e+05, -2.260637389581e+05, -2.261160993951e+05, -2.261670387151e+05, -2.262165799901e+05, -2.262647457211e+05, -2.263115578541e+05, -2.263570377991e+05, -2.264012064501e+05, -2.264440842001e+05, -2.264856909551e+05, -2.265260461571e+05, -2.265651687891e+05, -2.266030773961e+05, -2.266397901001e+05, -2.266753246061e+05, -2.267096982211e+05, -2.267429278641e+05, -2.267750300791e+05, -2.268060210431e+05, -2.268359165801e+05, -2.268647321721e+05, -2.268924829661e+05, -2.269191837861e+05, -2.269448491411e+05, -2.269694932361e+05, -2.269931299781e+05, -2.270157729871e+05, -2.270374356021e+05, -2.270581308901e+05, -2.270778716531e+05, -2.270966704351e+05, -2.271145395301e+05, -2.271314909871e+05, -2.271475366201e+05, -2.271626880101e+05, -2.271769565141e+05, -2.271903532701e+05, -2.272028892021e+05, -2.272145750261e+05, -2.272254212571e+05, -2.272354382131e+05, -2.272446360171e+05, -2.272530246081e+05, -2.272606137391e+05, -2.272674129881e+05, -2.272734317561e+05, -2.272786792781e+05, -2.272831646201e+05, -2.272868966881e+05, -2.272898842331e+05, -2.272921358481e+05, -2.272936599781e+05, -2.272944649221e+05, -2.272945588341e+05, -2.272939497311e+05, -2.272926454901e+05, -2.272906538581e+05, -2.272879824481e+05, -2.272846387491e+05, -2.272806301241e+05, -2.272759638131e+05, -2.272706469391e+05, -2.272646865071e+05, -2.272580894091e+05, -2.272508624271e+05, -2.272430122321e+05, -2.272345453881e+05, -2.272254683571e+05, -2.272157874981e+05, -2.272055090691e+05, -2.271946392321e+05, -2.271831840531e+05, -2.271711495031e+05, -2.271585414621e+05, -2.271453657221e+05, -2.271316279841e+05, -2.271173338641e+05, -2.271024888951e+05, -2.270870985261e+05, -2.270711681241e+05, -2.270547029781e+05, -2.270377082991e+05, -2.270201892221e+05, -2.270021508051e+05, -2.269835980371e+05, -2.269645358301e+05, -2.269449690291e+05, -2.269249024091e+05, -2.269043406781e+05, -2.268832884741e+05, -2.268617503751e+05, -2.268397308901e+05, -2.268172344691e+05, -2.267942654981e+05, -2.267708283051e+05, -2.267469271551e+05, -2.267225662581e+05, -2.266977497651e+05, -2.266724817711e+05, -2.266467663181e+05, -2.266206073911e+05, -2.265940089231e+05, -2.265669747941e+05, -2.265395088361e+05, -2.265116148261e+05, -2.264832964951e+05, -2.264545575241e+05, -2.264254015461e+05, -2.263958321481e+05, -2.263658528691e+05, -2.263354672061e+05, -2.263046786091e+05, -2.262734904851e+05, -2.262419061971e+05, -2.262099290671e+05, -2.261775623751e+05, -2.261448093611e+05, -2.261116732221e+05, -2.260781571191e+05, -2.260442641731e+05, -2.260099974641e+05, -2.259753600391e+05, -2.259403549051e+05, -2.259049850331e+05, -2.258692533591e+05, -2.258331627841e+05, -2.257967161721e+05, -2.257599163571e+05, -2.257227661361e+05, -2.256852682741e+05, -2.256474255051e+05, -2.256092405281e+05, -2.255707160131e+05, -2.255318545981e+05, -2.254926588921e+05, -2.254531314721e+05, -2.254132748851e+05, -2.253730916521e+05, -2.253325842631e+05, -2.252917551801e+05, -2.252506068381e+05, -2.252091416431e+05, -2.251673619771e+05, -2.251252701911e+05, -2.250828686151e+05, -2.250401595491e+05, -2.249971452711e+05, -2.249538280301e+05, -2.249102100551e+05, -2.248662935471e+05, -2.248220806851e+05, -2.247775736241e+05, -2.247327744951e+05, -2.246876854081e+05, -2.246423084481e+05, -2.245966456801e+05, -2.245506991461e+05, -2.245044708651e+05, -2.244579628381e+05, -2.244111770421e+05, -2.243641154361e+05, -2.243167799551e+05, -2.242691725171e+05, -2.242212950181e+05, -2.241731493371e+05, -2.241247373321e+05, -2.240760608411e+05, -2.240271216861e+05, -2.239779216671e+05, -2.239284625691e+05, -2.238787461561e+05, -2.238287741781e+05, -2.237785483631e+05, -2.237280704251e+05, -2.236773420601e+05, -2.236263649471e+05, -2.235751407481e+05, -2.235236711101e+05, -2.234719576611e+05, -2.234200020161e+05, -2.233678057741e+05, -2.233153705161e+05, -2.232626978091e+05, -2.232097892071e+05, -2.231566462451e+05, -2.231032704471e+05, -2.230496633201e+05, -2.229958263571e+05, -2.229417610381e+05, -2.228874688271e+05, -2.228329511771e+05, -2.227782095241e+05, -2.227232452941e+05, -2.226680598961e+05, -2.226126547281e+05, -2.225570311751e+05, -2.225011906091e+05, -2.224451343901e+05, -2.223888638621e+05, -2.223323803621e+05, -2.222756852111e+05, -2.222187797191e+05, -2.221616651841e+05, -2.221043428931e+05, -2.220468141201e+05, -2.219890801281e+05, -2.219311421691e+05, -2.218730014851e+05, -2.218146593031e+05, -2.217561168441e+05, -2.216973753151e+05, -2.216384359131e+05, -2.215792998241e+05, -2.215199682261e+05, -2.214604422821e+05, -2.214007231501e+05, -2.213408119751e+05, -2.212807098921e+05, -2.212204180271e+05, -2.211599374961e+05, -2.210992694061e+05, -2.210384148531e+05, -2.209773749261e+05, -2.209161507011e+05, -2.208547432491e+05, -2.207931536281e+05, -2.207313828911e+05, -2.206694320801e+05, -2.206073022271e+05, -2.205449943581e+05, -2.204825094881e+05, -2.204198486251e+05, -2.203570127691e+05, -2.202940029101e+05, -2.202308200311e+05, -2.201674651061e+05, -2.201039391041e+05, -2.200402429811e+05, -2.199763776891e+05, -2.199123441711e+05, -2.198481433621e+05, -2.197837761911e+05, -2.197192435771e+05, -2.196545464341e+05, -2.195896856671e+05, -2.195246621731e+05, -2.194594768451e+05, -2.193941305671e+05, -2.193286242141e+05, -2.192629586581e+05, -2.191971347611e+05, -2.191311533791e+05, -2.190650153621e+05, -2.189987215531e+05, -2.189322727881e+05, -2.188656698971e+05, -2.187989137021e+05, -2.187320050221e+05, -2.186649446651e+05, -2.185977334371e+05, -2.185303721361e+05, -2.184628615531e+05, -2.183952024741e+05, -2.183273956791e+05, -2.182594419421e+05, -2.181913420311e+05, -2.181230967071e+05, -2.180547067281e+05, -2.179861728441e+05, -2.179174957991e+05, -2.178486763341e+05, -2.177797151811e+05, -2.177106130701e+05, -2.176413707231e+05, -2.175719888581e+05, -2.175024681871e+05, -2.174328094171e+05, -2.173630132501e+05, -2.172930803811e+05, -2.172230115031e+05, -2.171528073021e+05, -2.170824684601e+05, -2.170119956511e+05, -2.169413895491e+05, -2.168706508201e+05, -2.167997801241e+05, -2.167287781201e+05, -2.166576454591e+05, -2.165863827901e+05, -2.165149907541e+05, -2.164434699901e+05, -2.163718211321e+05, -2.163000448091e+05, -2.162281416461e+05, -2.161561122631e+05, -2.160839572761e+05, -2.160116772971e+05, -2.159392729331e+05, -2.158667447861e+05, -2.157940934561e+05, -2.157213195381e+05, -2.156484236211e+05, -2.155754062931e+05, -2.155022681351e+05, -2.154290097251e+05, -2.153556316391e+05, -2.152821344451e+05, -2.152085187121e+05, -2.151347850001e+05, -2.150609338701e+05, -2.149869658751e+05, -2.149128815661e+05, -2.148386814921e+05, -2.147643661951e+05, -2.146899362161e+05, -2.146153920911e+05, -2.145407343521e+05, -2.144659635291e+05, -2.143910801471e+05, -2.143160847281e+05, -2.142409777901e+05, -2.141657598501e+05, -2.140904314181e+05, -2.140149930021e+05, -2.139394451081e+05, -2.138637882381e+05, -2.137880228891e+05, -2.137121495561e+05, -2.136361687311e+05, -2.135600809041e+05, -2.134838865581e+05, -2.134075861771e+05, -2.133311802391e+05, -2.132546692201e+05, -2.131780535931e+05, -2.131013338291e+05, -2.130245103931e+05, -2.129475837501e+05, -2.128705543591e+05, -2.127934226801e+05, -2.127161891671e+05, -2.126388542731e+05, -2.125614184451e+05, -2.124838821311e+05, -2.124062457741e+05, -2.123285098151e+05},
    {2.972594760787e+04, 2.796622819987e+04, 2.617900161787e+04, 2.436305105587e+04, 2.251706903787e+04, 2.063964785987e+04, 1.872926869487e+04, 1.678428913587e+04, 1.480292890087e+04, 1.278325333687e+04, 1.072315432687e+04, 8.620328060870e+03, 6.472249027870e+03, 4.276139421871e+03, 2.028932934871e+03, -2.727683651295e+02, -2.632745741129e+03, -5.055226257129e+03, -7.544958834129e+03, -1.010730768713e+04, -1.274836816413e+04, -1.547511187213e+04, -1.829557054513e+04, -2.121907192313e+04, -2.425654655413e+04, -2.742093303113e+04, -3.072772255113e+04, -3.419570513213e+04, -3.784801516713e+04, -4.171363450613e+04, -4.582961896313e+04, -5.024451552913e+04, -5.502383574313e+04, -6.025929411813e+04, -6.608546304713e+04, -7.271246553313e+04, -8.049787467613e+04, -9.013176373813e+04, -1.032303981031e+05, -1.247777925451e+05, -1.561572830071e+05, -1.699628819521e+05, -1.767359077811e+05, -1.812036335401e+05, -1.845425464511e+05, -1.872142033771e+05, -1.894449633661e+05, -1.913622415201e+05, -1.930448016581e+05, -1.945447296671e+05, -1.958982650971e+05, -1.971316445221e+05, -1.982644808741e+05, -1.993118283471e+05, -2.002855016801e+05, -2.011949507441e+05, -2.020478586451e+05, -2.028505618001e+05, -2.036083518711e+05, -2.043256972561e+05, -2.050064085181e+05, -2.056537639501e+05, -2.062706062981e+05, -2.068594182371e+05, -2.074223819921e+05, -2.079614269561e+05, -2.084782681051e+05, -2.089744372781e+05, -2.094513088691e+05, -2.099101211021e+05, -2.103519937681e+05, -2.107779431261e+05, -2.111888944971e+05, -2.115856929641e+05, -2.119691125311e+05, -2.123398639811e+05, -2.126986016771e+05, -2.130459294451e+05, -2.133824057131e+05, -2.137085480021e+05, -2.140248368621e+05, -2.143317193531e+05, -2.146296121111e+05, -2.149189040751e+05, -2.151999589101e+05, -2.154731171661e+05, -2.157386982161e+05, -2.159970019891e+05, -2.162483105291e+05, -2.164928894001e+05, -2.167309889561e+05, -2.169628454901e+05, -2.171886822751e+05, -2.174087105111e+05, -2.176231301891e+05, -2.178321308721e+05, -2.180358924191e+05, -2.182345856381e+05, -2.184283728891e+05, -2.186174086391e+05, -2.188018399691e+05, -2.189818070391e+05, -2.191574435271e+05, -2.193288770161e+05, -2.194962293731e+05, -2.196596170821e+05, -2.198191515621e+05, -2.199749394631e+05, -2.201270829291e+05, -2.202756798621e+05, -2.204208241461e+05, -2.205626058751e+05, -2.207011115521e+05, -2.208364242801e+05, -2.209686239421e+05, -2.210977873681e+05, -2.212239884881e+05, -2.213472984791e+05, -2.214677859041e+05, -2.215855168371e+05, -2.217005549881e+05, -2.218129618121e+05, -2.219227966171e+05, -2.220301166681e+05, -2.221349772741e+05, -2.222374318851e+05, -2.223375321711e+05, -2.224353281041e+05, -2.225308680291e+05, -2.226241987421e+05, -2.227153655501e+05, -2.228044123371e+05, -2.228913816241e+05, -2.229763146271e+05, -2.230592513091e+05, -2.231402304321e+05, -2.232192896071e+05, -2.232964653351e+05, -2.233717930581e+05, -2.234453071961e+05, -2.235170411861e+05, -2.235870275231e+05, -2.236552977941e+05, -2.237218827111e+05, -2.237868121451e+05, -2.238501151591e+05, -2.239118200331e+05, -2.239719542971e+05, -2.240305447521e+05, -2.240876175041e+05, -2.241431979811e+05, -2.241973109611e+05, -2.242499805921e+05, -2.243012304161e+05, -2.243510833871e+05, -2.243995618931e+05, -2.244466877741e+05, -2.244924823391e+05, -2.245369663841e+05, -2.245801602111e+05, -2.246220836411e+05, -2.246627560291e+05, -2.247021962821e+05, -2.247404228701e+05, -2.247774538411e+05, -2.248133068341e+05, -2.248479990911e+05, -2.248815474681e+05, -2.249139684521e+05, -2.249452781631e+05, -2.249754923731e+05, -2.250046265121e+05, -2.250326956791e+05, -2.250597146531e+05, -2.250856978991e+05, -2.251106595801e+05, -2.251346135641e+05, -2.251575734341e+05, -2.251795524911e+05, -2.252005637701e+05, -2.252206200401e+05, -2.252397338141e+05, -2.252579173561e+05, -2.252751826901e+05, -2.252915416001e+05, -2.253070056441e+05, -2.253215861551e+05, -2.253352942471e+05, -2.253481408231e+05, -2.253601365801e+05, -2.253712920121e+05, -2.253816174191e+05, -2.253911229071e+05, -2.253998183981e+05, -2.254077136301e+05, -2.254148181651e+05, -2.254211413911e+05, -2.254266925291e+05, -2.254314806341e+05, -2.254355146001e+05, -2.254388031651e+05, -2.254413549131e+05, -2.254431782801e+05, -2.254442815531e+05, -2.254446728811e+05, -2.254443602691e+05, -2.254433515901e+05, -2.254416545811e+05, -2.254392768501e+05, -2.254362258791e+05, -2.254325090251e+05, -2.254281335241e+05, -2.254231064931e+05, -2.254174349331e+05, -2.254111257321e+05, -2.254041856671e+05, -2.253966214061e+05, -2.253884395111e+05, -2.253796464391e+05, -2.253702485481e+05, -2.253602520941e+05, -2.253496632361e+05, -2.253384880381e+05, -2.253267324711e+05, -2.253144024141e+05, -2.253015036551e+05, -2.252880418961e+05, -2.252740227541e+05, -2.252594517591e+05, -2.252443343601e+05, -2.252286759261e+05, -2.252124817451e+05, -2.251957570281e+05, -2.251785069111e+05, -2.251607364531e+05, -2.251424506421e+05, -2.251236543931e+05, -2.251043525521e+05, -2.250845498941e+05, -2.250642511281e+05, -2.250434608961e+05, -2.250221837741e+05, -2.250004242761e+05, -2.249781868521e+05, -2.249554758911e+05, -2.249322957201e+05, -2.249086506101e+05, -2.248845447711e+05, -2.248599823571e+05, -2.248349674651e+05, -2.248095041391e+05, -2.247835963661e+05, -2.247572480831e+05, -2.247304631731e+05, -2.247032454691e+05, -2.246755987511e+05, -2.246475267531e+05, -2.246190331591e+05, -2.245901216051e+05, -2.245607956801e+05, -2.245310589281e+05, -2.245009148471e+05, -2.244703668901e+05, -2.244394184681e+05, -2.244080729471e+05, -2.243763336531e+05, -2.243442038671e+05, -2.243116868341e+05, -2.242787857541e+05, -2.242455037911e+05, -2.242118440681e+05, -2.241778096711e+05, -2.241434036481e+05, -2.241086290111e+05, -2.240734887341e+05, -2.240379857561e+05, -2.240021229801e+05, -2.239659032771e+05, -2.239293294811e+05, -2.238924043951e+05, -2.238551307851e+05, -2.238175113901e+05, -2.237795489131e+05, -2.237412460261e+05, -2.237026053721e+05, -2.236636295621e+05, -2.236243211771e+05, -2.235846827691e+05, -2.235447168611e+05, -2.235044259461e+05, -2.234638124911e+05, -2.234228789331e+05, -2.233816276831e+05, -2.233400611251e+05, -2.232981816161e+05, -2.232559914871e+05, -2.232134930431e+05, -2.231706885641e+05, -2.231275803051e+05, -2.230841704961e+05, -2.230404613431e+05, -2.229964550281e+05, -2.229521537091e+05, -2.229075595211e+05, -2.228626745771e+05, -2.228175009671e+05, -2.227720407581e+05, -2.227262959951e+05, -2.226802687031e+05, -2.226339608841e+05, -2.225873745191e+05, -2.225405115701e+05, -2.224933739761e+05, -2.224459636591e+05, -2.223982825191e+05, -2.223503324371e+05, -2.223021152731e+05, -2.222536328711e+05, -2.222048870541e+05, -2.221558796281e+05, -2.221066123801e+05, -2.220570870771e+05, -2.220073054721e+05, -2.219572692981e+05, -2.219069802721e+05, -2.218564400911e+05, -2.218056504391e+05, -2.217546129811e+05, -2.217033293671e+05, -2.216518012291e+05, -2.216000301851e+05, -2.215480178361e+05, -2.214957657671e+05, -2.214432755491e+05, -2.213905487381e+05, -2.213375868731e+05, -2.212843914811e+05, -2.212309640701e+05, -2.211773061401e+05, -2.211234191711e+05, -2.210693046321e+05, -2.210149639771e+05, -2.209603986461e+05, -2.209056100681e+05, -2.208505996571e+05, -2.207953688111e+05, -2.207399189201e+05, -2.206842513581e+05, -2.206283674881e+05, -2.205722686581e+05, -2.205159562061e+05, -2.204594314581e+05, -2.204026957251e+05, -2.203457503101e+05, -2.202885965011e+05, -2.202312355751e+05, -2.201736688001e+05, -2.201158974291e+05, -2.200579227061e+05, -2.199997458631e+05, -2.199413681231e+05, -2.198827906951e+05, -2.198240147791e+05, -2.197650415651e+05, -2.197058722321e+05, -2.196465079481e+05, -2.195869498711e+05, -2.195271991501e+05, -2.194672569231e+05, -2.194071243181e+05, -2.193468024551e+05, -2.192862924421e+05, -2.192255953791e+05, -2.191647123551e+05, -2.191036444521e+05, -2.190423927421e+05, -2.189809582871e+05, -2.189193421401e+05, -2.188575453471e+05, -2.187955689421e+05, -2.187334139541e+05, -2.186710814011e+05, -2.186085722931e+05, -2.185458876321e+05, -2.184830284111e+05, -2.184199956161e+05, -2.183567902231e+05, -2.182934132021e+05, -2.182298655131e+05, -2.181661481111e+05, -2.181022619401e+05, -2.180382079381e+05, -2.179739870361e+05, -2.179096001561e+05, -2.178450482131e+05, -2.177803321151e+05, -2.177154527641e+05, -2.176504110511e+05, -2.175852078641e+05, -2.175198440821e+05, -2.174543205771e+05, -2.173886382141e+05, -2.173227978521e+05, -2.172568003421e+05, -2.171906465291e+05, -2.171243372531e+05, -2.170578733451e+05, -2.169912556301e+05, -2.169244849271e+05, -2.168575620491e+05, -2.167904878021e+05, -2.167232629871e+05, -2.166558883981e+05, -2.165883648221e+05, -2.165206930411e+05, -2.164528738311e+05, -2.163849079631e+05, -2.163167962011e+05, -2.162485393021e+05, -2.161801380201e+05, -2.161115931011e+05, -2.160429052881e+05, -2.159740753151e+05, -2.159051039131e+05, -2.158359918081e+05, -2.157667397181e+05, -2.156973483581e+05, -2.156278184371e+05, -2.155581506581e+05, -2.154883457191e+05, -2.154184043151e+05, -2.153483271331e+05, -2.152781148571e+05, -2.152077681651e+05, -2.151372877301e+05, -2.150666742211e+05, -2.149959283001e+05, -2.149250506281e+05, -2.148540418591e+05, -2.147829026401e+05, -2.147116336181e+05, -2.146402354321e+05, -2.145687087181e+05, -2.144970541071e+05, -2.144252722251e+05, -2.143533636951e+05, -2.142813291341e+05, -2.142091691551e+05, -2.141368843681e+05, -2.140644753771e+05, -2.139919427831e+05, -2.139192871831e+05, -2.138465091681e+05, -2.137736093261e+05, -2.137005882411e+05, -2.136274464941e+05, -2.135541846611e+05, -2.134808033121e+05, -2.134073030171e+05, -2.133336843401e+05, -2.132599478401e+05, -2.131860940751e+05, -2.131121235961e+05, -2.130380369541e+05, -2.129638346921e+05, -2.128895173541e+05, -2.128150854751e+05, -2.127405395921e+05, -2.126658802341e+05, -2.125911079281e+05, -2.125162231991e+05, -2.124412265661e+05, -2.123661185461e+05, -2.122908996521e+05, -2.122155703941e+05, -2.121401312781e+05, -2.120645828071e+05, -2.119889254821e+05, -2.119131597981e+05, -2.118372862491e+05, -2.117613053251e+05, -2.116852175131e+05, -2.116090232961e+05, -2.115327231561e+05, -2.114563175681e+05, -2.113798070081e+05, -2.113031919471e+05, -2.112264728531e+05, -2.111496501911e+05, -2.110727244241e+05, -2.109956960101e+05, -2.109185654061e+05, -2.108413330661e+05, -2.107639994401e+05, -2.106865649751e+05},
    {3.060452556987e+04, 2.885764653787e+04, 2.708384802087e+04, 2.528195985087e+04, 2.345072624287e+04, 2.158879693787e+04, 1.969471714687e+04, 1.776691607787e+04, 1.580369380987e+04, 1.380320619287e+04, 1.176344742087e+04, 9.682229820871e+03, 7.557160289871e+03, 5.385612685871e+03, 3.164695292871e+03, 8.912122558705e+02, -1.438382434129e+03, -3.828040233129e+03, -6.282184963129e+03, -8.805794486129e+03, -1.140450124813e+04, -1.408471725413e+04, -1.685379103113e+04, -1.972020703813e+04, -2.269384223613e+04, -2.578630083213e+04, -2.901135788213e+04, -3.238555752913e+04, -3.592903589413e+04, -3.966667880613e+04, -4.362979358913e+04, -4.785859755513e+04, -5.240605771313e+04, -5.734407668413e+04, -6.277399856413e+04, -6.884566440613e+04, -7.579498294313e+04, -8.402637108113e+04, -9.431942280713e+04, -1.084168293431e+05, -1.304180882031e+05, -1.546881313901e+05, -1.668808530221e+05, -1.735673433851e+05, -1.781089420741e+05, -1.815387181221e+05, -1.842940968261e+05, -1.865980933451e+05, -1.885789033301e+05, -1.903167694561e+05, -1.918651862481e+05, -1.932615782501e+05, -1.945331426771e+05, -1.957002629431e+05, -1.967786108581e+05, -1.977804981551e+05, -1.987157770211e+05, -1.995924585091e+05, -2.004171483471e+05, -2.011953610351e+05, -2.019317507411e+05, -2.026302840501e+05, -2.032943712331e+05, -2.039269674531e+05, -2.045306517761e+05, -2.051076896061e+05, -2.056600825481e+05, -2.061896086231e+05, -2.066978550161e+05, -2.071862449571e+05, -2.076560599631e+05, -2.081084583841e+05, -2.085444909661e+05, -2.089651139901e+05, -2.093712004511e+05, -2.097635495901e+05, -2.101428950941e+05, -2.105099121691e+05, -2.108652236691e+05, -2.112094054361e+05, -2.115429909681e+05, -2.118664755111e+05, -2.121803196721e+05, -2.124849526011e+05, -2.127807748201e+05, -2.130681607341e+05, -2.133474608721e+05, -2.136190038931e+05, -2.138830983791e+05, -2.141400344531e+05, -2.143900852291e+05, -2.146335081321e+05, -2.148705460781e+05, -2.151014285601e+05, -2.153263726201e+05, -2.155455837441e+05, -2.157592566701e+05, -2.159675761371e+05, -2.161707175551e+05, -2.163688476341e+05, -2.165621249511e+05, -2.167507004791e+05, -2.169347180621e+05, -2.171143148691e+05, -2.172896218001e+05, -2.174607638641e+05, -2.176278605361e+05, -2.177910260781e+05, -2.179503698411e+05, -2.181059965481e+05, -2.182580065491e+05, -2.184064960721e+05, -2.185515574401e+05, -2.186932792861e+05, -2.188317467491e+05, -2.189670416561e+05, -2.190992426961e+05, -2.192284255781e+05, -2.193546631831e+05, -2.194780257061e+05, -2.195985807841e+05, -2.197163936251e+05, -2.198315271231e+05, -2.199440419661e+05, -2.200539967431e+05, -2.201614480371e+05, -2.202664505201e+05, -2.203690570401e+05, -2.204693186971e+05, -2.205672849281e+05, -2.206630035751e+05, -2.207565209531e+05, -2.208478819201e+05, -2.209371299331e+05, -2.210243071121e+05, -2.211094542911e+05, -2.211926110731e+05, -2.212738158781e+05, -2.213531059941e+05, -2.214305176181e+05, -2.215060859011e+05, -2.215798449861e+05, -2.216518280531e+05, -2.217220673471e+05, -2.217905942211e+05, -2.218574391661e+05, -2.219226318401e+05, -2.219862011061e+05, -2.220481750531e+05, -2.221085810291e+05, -2.221674456641e+05, -2.222247949001e+05, -2.222806540081e+05, -2.223350476201e+05, -2.223879997411e+05, -2.224395337791e+05, -2.224896725621e+05, -2.225384383561e+05, -2.225858528831e+05, -2.226319373441e+05, -2.226767124291e+05, -2.227201983411e+05, -2.227624148041e+05, -2.228033810841e+05, -2.228431160011e+05, -2.228816379411e+05, -2.229189648741e+05, -2.229551143651e+05, -2.229901035831e+05, -2.230239493181e+05, -2.230566679881e+05, -2.230882756541e+05, -2.231187880291e+05, -2.231482204841e+05, -2.231765880671e+05, -2.232039055041e+05, -2.232301872111e+05, -2.232554473041e+05, -2.232796996071e+05, -2.233029576601e+05, -2.233252347241e+05, -2.233465437951e+05, -2.233668976041e+05, -2.233863086291e+05, -2.234047891031e+05, -2.234223510141e+05, -2.234390061181e+05, -2.234547659441e+05, -2.234696417941e+05, -2.234836447591e+05, -2.234967857161e+05, -2.235090753371e+05, -2.235205240941e+05, -2.235311422641e+05, -2.235409399341e+05, -2.235499270041e+05, -2.235581131951e+05, -2.235655080501e+05, -2.235721209421e+05, -2.235779610731e+05, -2.235830374831e+05, -2.235873590531e+05, -2.235909345061e+05, -2.235937724141e+05, -2.235958811981e+05, -2.235972691371e+05, -2.235979443651e+05, -2.235979148781e+05, -2.235971885391e+05, -2.235957730761e+05, -2.235936760871e+05, -2.235909050461e+05, -2.235874673031e+05, -2.235833700841e+05, -2.235786205021e+05, -2.235732255491e+05, -2.235671921081e+05, -2.235605269491e+05, -2.235532367361e+05, -2.235453280251e+05, -2.235368072701e+05, -2.235276808221e+05, -2.235179549341e+05, -2.235076357621e+05, -2.234967293661e+05, -2.234852417131e+05, -2.234731786801e+05, -2.234605460531e+05, -2.234473495311e+05, -2.234335947271e+05, -2.234192871721e+05, -2.234044323121e+05, -2.233890355151e+05, -2.233731020661e+05, -2.233566371781e+05, -2.233396459831e+05, -2.233221335411e+05, -2.233041048391e+05, -2.232855647931e+05, -2.232665182461e+05, -2.232469699751e+05, -2.232269246891e+05, -2.232063870291e+05, -2.231853615731e+05, -2.231638528351e+05, -2.231418652651e+05, -2.231194032531e+05, -2.230964711281e+05, -2.230730731601e+05, -2.230492135621e+05, -2.230248964871e+05, -2.230001260361e+05, -2.229749062521e+05, -2.229492411251e+05, -2.229231345931e+05, -2.228965905401e+05, -2.228696128011e+05, -2.228422051601e+05, -2.228143713491e+05, -2.227861150571e+05, -2.227574399201e+05, -2.227283495311e+05, -2.226988474341e+05, -2.226689371311e+05, -2.226386220761e+05, -2.226079056831e+05, -2.225767913201e+05, -2.225452823151e+05, -2.225133819531e+05, -2.224810934791e+05, -2.224484200981e+05, -2.224153649741e+05, -2.223819312351e+05, -2.223481219681e+05, -2.223139402251e+05, -2.222793890191e+05, -2.222444713271e+05, -2.222091900921e+05, -2.221735482201e+05, -2.221375485821e+05, -2.221011940181e+05, -2.220644873311e+05, -2.220274312931e+05, -2.219900286431e+05, -2.219522820881e+05, -2.219141943041e+05, -2.218757679351e+05, -2.218370055961e+05, -2.217979098711e+05, -2.217584833151e+05, -2.217187284531e+05, -2.216786477831e+05, -2.216382437721e+05, -2.215975188631e+05, -2.215564754681e+05, -2.215151159751e+05, -2.214734427431e+05, -2.214314581071e+05, -2.213891643751e+05, -2.213465638301e+05, -2.213036587301e+05, -2.212604513071e+05, -2.212169437711e+05, -2.211731383081e+05, -2.211290370771e+05, -2.210846422181e+05, -2.210399558461e+05, -2.209949800541e+05, -2.209497169121e+05, -2.209041684691e+05, -2.208583367511e+05, -2.208122237651e+05, -2.207658314951e+05, -2.207191619051e+05, -2.206722169381e+05, -2.206249985181e+05, -2.205775085481e+05, -2.205297489121e+05, -2.204817214751e+05, -2.204334280821e+05, -2.203848705591e+05, -2.203360507161e+05, -2.202869703421e+05, -2.202376312091e+05, -2.201880350701e+05, -2.201381836631e+05, -2.200880787061e+05, -2.200377219021e+05, -2.199871149361e+05, -2.199362594761e+05, -2.198851571751e+05, -2.198338096681e+05, -2.197822185761e+05, -2.197303855021e+05, -2.196783120361e+05, -2.196259997511e+05, -2.195734502041e+05, -2.195206649401e+05, -2.194676454851e+05, -2.194143933541e+05, -2.193609100471e+05, -2.193071970481e+05, -2.192532558281e+05, -2.191990878441e+05, -2.191446945411e+05, -2.190900773471e+05, -2.190352376811e+05, -2.189801769441e+05, -2.189248965281e+05, -2.188693978091e+05, -2.188136821541e+05, -2.187577509141e+05, -2.187016054291e+05, -2.186452470271e+05, -2.185886770231e+05, -2.185318967211e+05, -2.184749074131e+05, -2.184177103801e+05, -2.183603068891e+05, -2.183026981981e+05, -2.182448855541e+05, -2.181868701911e+05, -2.181286533341e+05, -2.180702361951e+05, -2.180116199761e+05, -2.179528058711e+05, -2.178937950601e+05, -2.178345887151e+05, -2.177751879951e+05, -2.177155940531e+05, -2.176558080281e+05, -2.175958310521e+05, -2.175356642461e+05, -2.174753087221e+05, -2.174147655811e+05, -2.173540359151e+05, -2.172931208091e+05, -2.172320213371e+05, -2.171707385621e+05, -2.171092735421e+05, -2.170476273231e+05, -2.169858009441e+05, -2.169237954341e+05, -2.168616118141e+05, -2.167992510971e+05, -2.167367142861e+05, -2.166740023781e+05, -2.166111163601e+05, -2.165480572111e+05, -2.164848259031e+05, -2.164214233991e+05, -2.163578506541e+05, -2.162941086171e+05, -2.162301982271e+05, -2.161661204161e+05, -2.161018761111e+05, -2.160374662271e+05, -2.159728916761e+05, -2.159081533591e+05, -2.158432521731e+05, -2.157781890061e+05, -2.157129647391e+05, -2.156475802471e+05, -2.155820363961e+05, -2.155163340491e+05, -2.154504740581e+05, -2.153844572701e+05, -2.153182845271e+05, -2.152519566621e+05, -2.151854745021e+05, -2.151188388701e+05, -2.150520505781e+05, -2.149851104371e+05, -2.149180192471e+05, -2.148507778061e+05, -2.147833869021e+05, -2.147158473201e+05, -2.146481598381e+05, -2.145803252281e+05, -2.145123442561e+05, -2.144442176821e+05, -2.143759462601e+05, -2.143075307411e+05, -2.142389718671e+05, -2.141702703751e+05, -2.141014269991e+05, -2.140324424651e+05, -2.139633174941e+05, -2.138940528031e+05, -2.138246491011e+05, -2.137551070951e+05, -2.136854274851e+05, -2.136156109671e+05, -2.135456582291e+05, -2.134755699581e+05, -2.134053468331e+05, -2.133349895301e+05, -2.132644987181e+05, -2.131938750631e+05, -2.131231192261e+05, -2.130522318621e+05, -2.129812136231e+05, -2.129100651551e+05, -2.128387871001e+05, -2.127673800951e+05, -2.126958447721e+05, -2.126241817611e+05, -2.125523916851e+05, -2.124804751631e+05, -2.124084328101e+05, -2.123362652371e+05, -2.122639730511e+05, -2.121915568531e+05, -2.121190172421e+05, -2.120463548111e+05, -2.119735701501e+05, -2.119006638451e+05, -2.118276364781e+05, -2.117544886251e+05, -2.116812208601e+05, -2.116078337531e+05, -2.115343278711e+05, -2.114607037731e+05, -2.113869620201e+05, -2.113131031641e+05, -2.112391277571e+05, -2.111650363451e+05, -2.110908294711e+05, -2.110165076751e+05, -2.109420714921e+05, -2.108675214541e+05, -2.107928580911e+05, -2.107180819271e+05, -2.106431934841e+05, -2.105681932791e+05, -2.104930818291e+05, -2.104178596431e+05, -2.103425272291e+05, -2.102670850941e+05, -2.101915337361e+05, -2.101158736551e+05, -2.100401053451e+05, -2.099642292981e+05, -2.098882460011e+05, -2.098121559401e+05, -2.097359595971e+05, -2.096596574501e+05, -2.095832499761e+05, -2.095067376461e+05, -2.094301209311e+05, -2.093534002961e+05, -2.092765762071e+05, -2.091996491221e+05, -2.091226195001e+05, -2.090454877951e+05},
    {3.148409263287e+04, 2.974989907487e+04, 2.798935938587e+04, 2.620134812887e+04, 2.438465900487e+04, 2.253799666087e+04, 2.065996740087e+04, 1.874906862287e+04, 1.680367675787e+04, 1.482203344987e+04, 1.280222964887e+04, 1.074218722287e+04, 8.639637603871e+03, 6.492096861871e+03, 4.296836452871e+03, 2.050848704870e+03, -2.491941711294e+02, -2.606989065129e+03, -5.026660918129e+03, -7.512834226129e+03, -1.007072045513e+04, -1.270622588113e+04, -1.542608591213e+04, -1.823803417413e+04, -2.115101781813e+04, -2.417547520613e+04, -2.732369913413e+04, -3.061031943113e+04, -3.405295557913e+04, -3.767311694113e+04, -4.149747303013e+04, -4.555969330213e+04, -4.990319400013e+04, -5.458538875513e+04, -5.968455322013e+04, -6.531149819413e+04, -7.163070447513e+04, -7.890159773813e+04, -8.756639148113e+04, -9.845050966213e+04, -1.131807028721e+05, -1.339747649421e+05, -1.531085662881e+05, -1.640016229841e+05, -1.705289025411e+05, -1.750963901831e+05, -1.785903650711e+05, -1.814139818481e+05, -1.837816386241e+05, -1.858197371941e+05, -1.876086610891e+05, -1.892025680961e+05, -1.906396289921e+05, -1.919477324931e+05, -1.931478621381e+05, -1.942561967721e+05, -1.952854717951e+05, -1.962458921441e+05, -1.971457627141e+05, -1.979919346611e+05, -1.987901283011e+05, -1.995451712341e+05, -2.002611769521e+05, -2.009416808491e+05, -2.015897452471e+05, -2.022080415031e+05, -2.027989149311e+05, -2.033644367011e+05, -2.039064457051e+05, -2.044265826531e+05, -2.049263180611e+05, -2.054069754141e+05, -2.058697504701e+05, -2.063157274461e+05, -2.067458926991e+05, -2.071611463341e+05, -2.075623121251e+05, -2.079501460211e+05, -2.083253434971e+05, -2.086885459111e+05, -2.090403460461e+05, -2.093812929461e+05, -2.097118961601e+05, -2.100326294811e+05, -2.103439342431e+05, -2.106462222501e+05, -2.109398783711e+05, -2.112252628571e+05, -2.115027134201e+05, -2.117725470791e+05, -2.120350618361e+05, -2.122905381751e+05, -2.125392404191e+05, -2.127814179551e+05, -2.130173063481e+05, -2.132471283491e+05, -2.134710948161e+05, -2.136894055481e+05, -2.139022500561e+05, -2.141098082571e+05, -2.143122511161e+05, -2.145097412391e+05, -2.147024334081e+05, -2.148904750801e+05, -2.150740068491e+05, -2.152531628631e+05, -2.154280712211e+05, -2.155988543291e+05, -2.157656292421e+05, -2.159285079661e+05, -2.160875977551e+05, -2.162430013711e+05, -2.163948173421e+05, -2.165431401851e+05, -2.166880606301e+05, -2.168296658191e+05, -2.169680394951e+05, -2.171032621771e+05, -2.172354113281e+05, -2.173645615081e+05, -2.174907845201e+05, -2.176141495431e+05, -2.177347232651e+05, -2.178525699971e+05, -2.179677517911e+05, -2.180803285441e+05, -2.181903580951e+05, -2.182978963251e+05, -2.184029972431e+05, -2.185057130701e+05, -2.186060943171e+05, -2.187041898611e+05, -2.188000470181e+05, -2.188937116051e+05, -2.189852280081e+05, -2.190746392411e+05, -2.191619870001e+05, -2.192473117201e+05, -2.193306526251e+05, -2.194120477801e+05, -2.194915341301e+05, -2.195691475511e+05, -2.196449228881e+05, -2.197188939951e+05, -2.197910937751e+05, -2.198615542141e+05, -2.199303064151e+05, -2.199973806321e+05, -2.200628063011e+05, -2.201266120711e+05, -2.201888258271e+05, -2.202494747251e+05, -2.203085852111e+05, -2.203661830521e+05, -2.204222933541e+05, -2.204769405881e+05, -2.205301486101e+05, -2.205819406841e+05, -2.206323394991e+05, -2.206813671911e+05, -2.207290453591e+05, -2.207753950831e+05, -2.208204369411e+05, -2.208641910271e+05, -2.209066769621e+05, -2.209479139141e+05, -2.209879206071e+05, -2.210267153401e+05, -2.210643159961e+05, -2.211007400571e+05, -2.211360046161e+05, -2.211701263861e+05, -2.212031217161e+05, -2.212350065981e+05, -2.212657966771e+05, -2.212955072671e+05, -2.213241533531e+05, -2.213517496041e+05, -2.213783103831e+05, -2.214038497541e+05, -2.214283814911e+05, -2.214519190861e+05, -2.214744757561e+05, -2.214960644511e+05, -2.215166978631e+05, -2.215363884301e+05, -2.215551483461e+05, -2.215729895641e+05, -2.215899238051e+05, -2.216059625631e+05, -2.216211171111e+05, -2.216353985091e+05, -2.216488176041e+05, -2.216613850421e+05, -2.216731112671e+05, -2.216840065311e+05, -2.216940808981e+05, -2.217033442441e+05, -2.217118062681e+05, -2.217194764931e+05, -2.217263642701e+05, -2.217324787831e+05, -2.217378290551e+05, -2.217424239481e+05, -2.217462721681e+05, -2.217493822721e+05, -2.217517626661e+05, -2.217534216141e+05, -2.217543672361e+05, -2.217546075171e+05, -2.217541503051e+05, -2.217530033171e+05, -2.217511741411e+05, -2.217486702391e+05, -2.217454989501e+05, -2.217416674931e+05, -2.217371829681e+05, -2.217320523621e+05, -2.217262825481e+05, -2.217198802901e+05, -2.217128522411e+05, -2.217052049541e+05, -2.216969448731e+05, -2.216880783451e+05, -2.216786116171e+05, -2.216685508391e+05, -2.216579020661e+05, -2.216466712601e+05, -2.216348642931e+05, -2.216224869471e+05, -2.216095449181e+05, -2.215960438151e+05, -2.215819891641e+05, -2.215673864101e+05, -2.215522409161e+05, -2.215365579671e+05, -2.215203427701e+05, -2.215036004581e+05, -2.214863360871e+05, -2.214685546441e+05, -2.214502610401e+05, -2.214314601201e+05, -2.214121566581e+05, -2.213923553621e+05, -2.213720608731e+05, -2.213512777671e+05, -2.213300105571e+05, -2.213082636941e+05, -2.212860415671e+05, -2.212633485051e+05, -2.212401887771e+05, -2.212165665961e+05, -2.211924861171e+05, -2.211679514391e+05, -2.211429666071e+05, -2.211175356111e+05, -2.210916623891e+05, -2.210653508271e+05, -2.210386047591e+05, -2.210114279711e+05, -2.209838241961e+05, -2.209557971231e+05, -2.209273503911e+05, -2.208984875921e+05, -2.208692122731e+05, -2.208395279361e+05, -2.208094380381e+05, -2.207789459931e+05, -2.207480551721e+05, -2.207167689021e+05, -2.206850904731e+05, -2.206530231291e+05, -2.206205700781e+05, -2.205877344861e+05, -2.205545194821e+05, -2.205209281551e+05, -2.204869635601e+05, -2.204526287111e+05, -2.204179265871e+05, -2.203828601341e+05, -2.203474322591e+05, -2.203116458371e+05, -2.202755037071e+05, -2.202390086771e+05, -2.202021635201e+05, -2.201649709781e+05, -2.201274337591e+05, -2.200895545421e+05, -2.200513359751e+05, -2.200127806731e+05, -2.199738912231e+05, -2.199346701821e+05, -2.198951200791e+05, -2.198552434121e+05, -2.198150426541e+05, -2.197745202471e+05, -2.197336786081e+05, -2.196925201251e+05, -2.196510471631e+05, -2.196092620561e+05, -2.195671671171e+05, -2.195247646301e+05, -2.194820568551e+05, -2.194390460301e+05, -2.193957343651e+05, -2.193521240471e+05, -2.193082172421e+05, -2.192640160891e+05, -2.192195227071e+05, -2.191747391911e+05, -2.191296676141e+05, -2.190843100271e+05, -2.190386684611e+05, -2.189927449231e+05, -2.189465414011e+05, -2.189000598601e+05, -2.188533022481e+05, -2.188062704901e+05, -2.187589664931e+05, -2.187113921421e+05, -2.186635493051e+05, -2.186154398311e+05, -2.185670655481e+05, -2.185184282681e+05, -2.184695297821e+05, -2.184203718661e+05, -2.183709562761e+05, -2.183212847511e+05, -2.182713590141e+05, -2.182211807681e+05, -2.181707517011e+05, -2.181200734861e+05, -2.180691477761e+05, -2.180179762101e+05, -2.179665604111e+05, -2.179149019851e+05, -2.178630025241e+05, -2.178108636041e+05, -2.177584867851e+05, -2.177058736131e+05, -2.176530256191e+05, -2.175999443191e+05, -2.175466312151e+05, -2.174930877951e+05, -2.174393155321e+05, -2.173853158861e+05, -2.173310903031e+05, -2.172766402151e+05, -2.172219670421e+05, -2.171670721901e+05, -2.171119570511e+05, -2.170566230061e+05, -2.170010714211e+05, -2.169453036521e+05, -2.168893210411e+05, -2.168331249181e+05, -2.167767166021e+05, -2.167200973981e+05, -2.166632686001e+05, -2.166062314921e+05, -2.165489873441e+05, -2.164915374171e+05, -2.164338829581e+05, -2.163760252051e+05, -2.163179653841e+05, -2.162597047121e+05, -2.162012443921e+05, -2.161425856191e+05, -2.160837295771e+05, -2.160246774401e+05, -2.159654303701e+05, -2.159059895201e+05, -2.158463560341e+05, -2.157865310441e+05, -2.157265156751e+05, -2.156663110401e+05, -2.156059182441e+05, -2.155453383801e+05, -2.154845725351e+05, -2.154236217861e+05, -2.153624871981e+05, -2.153011698311e+05, -2.152396707341e+05, -2.151779909461e+05, -2.151161315011e+05, -2.150540934211e+05, -2.149918777201e+05, -2.149294854051e+05, -2.148669174751e+05, -2.148041749171e+05, -2.147412587151e+05, -2.146781698411e+05, -2.146149092621e+05, -2.145514779331e+05, -2.144878768071e+05, -2.144241068231e+05, -2.143601689181e+05, -2.142960640181e+05, -2.142317930431e+05, -2.141673569041e+05, -2.141027565061e+05, -2.140379927481e+05, -2.139730665181e+05, -2.139079787021e+05, -2.138427301751e+05, -2.137773218071e+05, -2.137117544591e+05, -2.136460289891e+05, -2.135801462451e+05, -2.135141070701e+05, -2.134479122991e+05, -2.133815627621e+05, -2.133150592831e+05, -2.132484026761e+05, -2.131815937541e+05, -2.131146333201e+05, -2.130475221711e+05, -2.129802611011e+05, -2.129128508931e+05, -2.128452923291e+05, -2.127775861831e+05, -2.127097332211e+05, -2.126417342061e+05, -2.125735898961e+05, -2.125053010401e+05, -2.124368683831e+05, -2.123682926661e+05, -2.122995746221e+05, -2.122307149801e+05, -2.121617144621e+05, -2.120925737881e+05, -2.120232936691e+05, -2.119538748121e+05, -2.118843179211e+05, -2.118146236901e+05, -2.117447928131e+05, -2.116748259761e+05, -2.116047238611e+05, -2.115344871451e+05, -2.114641165001e+05, -2.113936125921e+05, -2.113229760841e+05, -2.112522076331e+05, -2.111813078931e+05, -2.111102775111e+05, -2.110391171301e+05, -2.109678273911e+05, -2.108964089261e+05, -2.108248623661e+05, -2.107531883371e+05, -2.106813874591e+05, -2.106094603491e+05, -2.105374076191e+05, -2.104652298771e+05, -2.103929277271e+05, -2.103205017681e+05, -2.102479525971e+05, -2.101752808031e+05, -2.101024869751e+05, -2.100295716951e+05, -2.099565355431e+05, -2.098833790931e+05, -2.098101029171e+05, -2.097367075811e+05, -2.096631936501e+05, -2.095895616831e+05, -2.095158122351e+05, -2.094419458581e+05, -2.093679631021e+05, -2.092938645091e+05, -2.092196506221e+05, -2.091453219771e+05, -2.090708791071e+05, -2.089963225441e+05, -2.089216528131e+05, -2.088468704371e+05, -2.087719759361e+05, -2.086969698271e+05, -2.086218526211e+05, -2.085466248281e+05, -2.084712869531e+05, -2.083958395001e+05, -2.083202829681e+05, -2.082446178531e+05, -2.081688446471e+05, -2.080929638411e+05, -2.080169759201e+05, -2.079408813671e+05, -2.078646806641e+05, -2.077883742861e+05, -2.077119627091e+05, -2.076354464021e+05, -2.075588258331e+05, -2.074821014681e+05, -2.074052737681e+05},
    {3.236464772587e+04, 3.064298739487e+04, 2.889554036987e+04, 2.712122411487e+04, 2.531887972587e+04, 2.348726433987e+04, 2.162504256087e+04, 1.973077673287e+04, 1.780291586087e+04, 1.583978295387e+04, 1.383956050387e+04, 1.180027375787e+04, 9.719771351870e+03, 7.595702799871e+03, 5.425492182871e+03, 3.206307233870e+03, 9.350228068705e+02, -1.391822567129e+03, -3.778078667130e+03, -6.228046259129e+03, -8.746553222130e+03, -1.133904778113e+04, -1.401171371213e+04, -1.677161409913e+04, -1.962687259913e+04, -2.258690469613e+04, -2.566271648713e+04, -2.886729621713e+04, -3.221613543113e+04, -3.572793495013e+04, -3.942558027913e+04, -4.333751993013e+04, -4.749976392813e+04, -5.195886928913e+04, -5.677655780713e+04, -6.203715631813e+04, -6.786017150013e+04, -7.442273622013e+04, -8.200202417013e+04, -9.105844318613e+04, -1.023865109481e+05, -1.172422792831e+05, -1.358694734111e+05, -1.514820099471e+05, -1.612979119841e+05, -1.676196415631e+05, -1.721724451681e+05, -1.757045770891e+05, -1.785800860971e+05, -1.810007959261e+05, -1.830890279381e+05, -1.849240141091e+05, -1.865598169631e+05, -1.880348889741e+05, -1.893775155991e+05, -1.906090881951e+05, -1.917461643431e+05, -1.928018158951e+05, -1.937865402691e+05, -1.947088939041e+05, -1.955759432631e+05, -1.963935927201e+05, -1.971668273531e+05, -1.978998956881e+05, -1.985964492821e+05, -1.992596507431e+05, -1.998922583591e+05, -2.004966931281e+05, -2.010750923821e+05, -2.016293531121e+05, -2.021611672621e+05, -2.026720507201e+05, -2.031633673151e+05, -2.036363488131e+05, -2.040921117021e+05, -2.045316713571e+05, -2.049559540711e+05, -2.053658073221e+05, -2.057620085941e+05, -2.061452729711e+05, -2.065162597261e+05, -2.068755780461e+05, -2.072237920391e+05, -2.075614251161e+05, -2.078889638551e+05, -2.082068614091e+05, -2.085155405261e+05, -2.088153962321e+05, -2.091067982211e+05, -2.093900929931e+05, -2.096656057701e+05, -2.099336422121e+05, -2.101944899681e+05, -2.104484200731e+05, -2.106956882091e+05, -2.109365358551e+05, -2.111711913231e+05, -2.113998707031e+05, -2.116227787301e+05, -2.118401095671e+05, -2.120520475261e+05, -2.122587677291e+05, -2.124604367121e+05, -2.126572129811e+05, -2.128492475211e+05, -2.130366842751e+05, -2.132196605691e+05, -2.133983075221e+05, -2.135727504131e+05, -2.137431090271e+05, -2.139094979751e+05, -2.140720269891e+05, -2.142308011951e+05, -2.143859213771e+05, -2.145374842061e+05, -2.146855824691e+05, -2.148303052731e+05, -2.149717382431e+05, -2.151099636981e+05, -2.152450608231e+05, -2.153771058291e+05, -2.155061721001e+05, -2.156323303331e+05, -2.157556486711e+05, -2.158761928211e+05, -2.159940261771e+05, -2.161092099231e+05, -2.162218031391e+05, -2.163318628971e+05, -2.164394443501e+05, -2.165446008201e+05, -2.166473838821e+05, -2.167478434361e+05, -2.168460277801e+05, -2.169419836821e+05, -2.170357564441e+05, -2.171273899611e+05, -2.172169267821e+05, -2.173044081661e+05, -2.173898741301e+05, -2.174733635051e+05, -2.175549139781e+05, -2.176345621421e+05, -2.177123435341e+05, -2.177882926781e+05, -2.178624431261e+05, -2.179348274891e+05, -2.180054774801e+05, -2.180744239401e+05, -2.181416968741e+05, -2.182073254811e+05, -2.182713381851e+05, -2.183337626581e+05, -2.183946258491e+05, -2.184539540121e+05, -2.185117727271e+05, -2.185681069231e+05, -2.186229809031e+05, -2.186764183631e+05, -2.187284424131e+05, -2.187790755961e+05, -2.188283399071e+05, -2.188762568131e+05, -2.189228472671e+05, -2.189681317251e+05, -2.190121301651e+05, -2.190548620961e+05, -2.190963465811e+05, -2.191366022441e+05, -2.191756472851e+05, -2.192134994951e+05, -2.192501762681e+05, -2.192856946111e+05, -2.193200711591e+05, -2.193533221801e+05, -2.193854635931e+05, -2.194165109741e+05, -2.194464795671e+05, -2.194753842921e+05, -2.195032397571e+05, -2.195300602651e+05, -2.195558598221e+05, -2.195806521491e+05, -2.196044506851e+05, -2.196272685961e+05, -2.196491187861e+05, -2.196700138991e+05, -2.196899663311e+05, -2.197089882321e+05, -2.197270915161e+05, -2.197442878651e+05, -2.197605887351e+05, -2.197760053661e+05, -2.197905487811e+05, -2.198042297961e+05, -2.198170590251e+05, -2.198290468831e+05, -2.198402035941e+05, -2.198505391931e+05, -2.198600635301e+05, -2.198687862801e+05, -2.198767169411e+05, -2.198838648411e+05, -2.198902391431e+05, -2.198958488471e+05, -2.199007027951e+05, -2.199048096751e+05, -2.199081780251e+05, -2.199108162331e+05, -2.199127325461e+05, -2.199139350691e+05, -2.199144317691e+05, -2.199142304811e+05, -2.199133389081e+05, -2.199117646231e+05, -2.199095150761e+05, -2.199065975931e+05, -2.199030193831e+05, -2.198987875331e+05, -2.198939090201e+05, -2.198883907071e+05, -2.198822393461e+05, -2.198754615851e+05, -2.198680639641e+05, -2.198600529211e+05, -2.198514347941e+05, -2.198422158221e+05, -2.198324021491e+05, -2.198219998221e+05, -2.198110147981e+05, -2.197994529411e+05, -2.197873200281e+05, -2.197746217501e+05, -2.197613637101e+05, -2.197475514301e+05, -2.197331903491e+05, -2.197182858261e+05, -2.197028431421e+05, -2.196868675011e+05, -2.196703640321e+05, -2.196533377871e+05, -2.196357937491e+05, -2.196177368281e+05, -2.195991718661e+05, -2.195801036331e+05, -2.195605368361e+05, -2.195404761131e+05, -2.195199260381e+05, -2.194988911241e+05, -2.194773758181e+05, -2.194553845081e+05, -2.194329215231e+05, -2.194099911301e+05, -2.193865975401e+05, -2.193627449091e+05, -2.193384373341e+05, -2.193136788601e+05, -2.192884734761e+05, -2.192628251191e+05, -2.192367376761e+05, -2.192102149811e+05, -2.191832608181e+05, -2.191558789231e+05, -2.191280729821e+05, -2.190998466361e+05, -2.190712034781e+05, -2.190421470541e+05, -2.190126808671e+05, -2.189828083741e+05, -2.189525329911e+05, -2.189218580881e+05, -2.188907869951e+05, -2.188593230011e+05, -2.188274693521e+05, -2.187952292561e+05, -2.187626058821e+05, -2.187296023581e+05, -2.186962217761e+05, -2.186624671901e+05, -2.186283416171e+05, -2.185938480381e+05, -2.185589893991e+05, -2.185237686091e+05, -2.184881885431e+05, -2.184522520451e+05, -2.184159619211e+05, -2.183793209481e+05, -2.183423318681e+05, -2.183049973921e+05, -2.182673202001e+05, -2.182293029411e+05, -2.181909482341e+05, -2.181522586661e+05, -2.181132367981e+05, -2.180738851581e+05, -2.180342062471e+05, -2.179942025411e+05, -2.179538764831e+05, -2.179132304911e+05, -2.178722669581e+05, -2.178309882481e+05, -2.177893966991e+05, -2.177474946241e+05, -2.177052843111e+05, -2.176627680231e+05, -2.176199479961e+05, -2.175768264461e+05, -2.175334055611e+05, -2.174896875081e+05, -2.174456744311e+05, -2.174013684481e+05, -2.173567716581e+05, -2.173118861361e+05, -2.172667139351e+05, -2.172212570881e+05, -2.171755176041e+05, -2.171294974741e+05, -2.170831986661e+05, -2.170366231281e+05, -2.169897727891e+05, -2.169426495571e+05, -2.168952553211e+05, -2.168475919501e+05, -2.167996612961e+05, -2.167514651881e+05, -2.167030054411e+05, -2.166542838501e+05, -2.166053021901e+05, -2.165560622221e+05, -2.165065656861e+05, -2.164568143081e+05, -2.164068097921e+05, -2.163565538311e+05, -2.163060480971e+05, -2.162552942471e+05, -2.162042939231e+05, -2.161530487481e+05, -2.161015603331e+05, -2.160498302711e+05, -2.159978601401e+05, -2.159456515031e+05, -2.158932059081e+05, -2.158405248871e+05, -2.157876099601e+05, -2.157344626311e+05, -2.156810843901e+05, -2.156274767121e+05, -2.155736410601e+05, -2.155195788821e+05, -2.154652916121e+05, -2.154107806721e+05, -2.153560474711e+05, -2.153010934021e+05, -2.152459198501e+05, -2.151905281821e+05, -2.151349197571e+05, -2.150790959191e+05, -2.150230579991e+05, -2.149668073201e+05, -2.149103451881e+05, -2.148536729001e+05, -2.147967917421e+05, -2.147397029861e+05, -2.146824078951e+05, -2.146249077181e+05, -2.145672036961e+05, -2.145092970581e+05, -2.144511890201e+05, -2.143928807911e+05, -2.143343735651e+05, -2.142756685301e+05, -2.142167668601e+05, -2.141576697221e+05, -2.140983782701e+05, -2.140388936491e+05, -2.139792169951e+05, -2.139193494341e+05, -2.138592920811e+05, -2.137990460421e+05, -2.137386124151e+05, -2.136779922871e+05, -2.136171867371e+05, -2.135561968331e+05, -2.134950236361e+05, -2.134336681981e+05, -2.133721315601e+05, -2.133104147571e+05, -2.132485188141e+05, -2.131864447471e+05, -2.131241935651e+05, -2.130617662671e+05, -2.129991638451e+05, -2.129363872841e+05, -2.128734375571e+05, -2.128103156331e+05, -2.127470224711e+05, -2.126835590231e+05, -2.126199262331e+05, -2.125561250371e+05, -2.124921563651e+05, -2.124280211381e+05, -2.123637202691e+05, -2.122992546651e+05, -2.122346252271e+05, -2.121698328451e+05, -2.121048784071e+05, -2.120397627881e+05, -2.119744868621e+05, -2.119090514931e+05, -2.118434575371e+05, -2.117777058471e+05, -2.117117972661e+05, -2.116457326321e+05, -2.115795127761e+05, -2.115131385241e+05, -2.114466106931e+05, -2.113799300951e+05, -2.113130975381e+05, -2.112461138191e+05, -2.111789797331e+05, -2.111116960681e+05, -2.110442636041e+05, -2.109766831181e+05, -2.109089553791e+05, -2.108410811511e+05, -2.107730611921e+05, -2.107048962551e+05, -2.106365870871e+05, -2.105681344281e+05, -2.104995390151e+05, -2.104308015771e+05, -2.103619228401e+05, -2.102929035241e+05, -2.102237443411e+05, -2.101544460021e+05, -2.100850092101e+05, -2.100154346631e+05, -2.099457230541e+05, -2.098758750731e+05, -2.098058914031e+05, -2.097357727221e+05, -2.096655197031e+05, -2.095951330161e+05, -2.095246133231e+05, -2.094539612851e+05, -2.093831775551e+05, -2.093122627831e+05, -2.092412176141e+05, -2.091700426881e+05, -2.090987386431e+05, -2.090273061081e+05, -2.089557457101e+05, -2.088840580731e+05, -2.088122438151e+05, -2.087403035491e+05, -2.086682378841e+05, -2.085960474261e+05, -2.085237327761e+05, -2.084512945311e+05, -2.083787332841e+05, -2.083060496221e+05, -2.082332441321e+05, -2.081603173921e+05, -2.080872699801e+05, -2.080141024691e+05, -2.079408154261e+05, -2.078674094171e+05, -2.077938850031e+05, -2.077202427411e+05, -2.076464831831e+05, -2.075726068811e+05, -2.074986143791e+05, -2.074245062201e+05, -2.073502829431e+05, -2.072759450821e+05, -2.072014931691e+05, -2.071269277321e+05, -2.070522492941e+05, -2.069774583781e+05, -2.069025555001e+05, -2.068275411751e+05, -2.067524159121e+05, -2.066771802191e+05, -2.066018346011e+05, -2.065263795571e+05, -2.064508155851e+05, -2.063751431801e+05, -2.062993628311e+05, -2.062234750271e+05, -2.061474802521e+05, -2.060713789871e+05, -2.059951717111e+05, -2.059188588991e+05, -2.058424410231e+05, -2.057659185531e+05},
    {3.324618975187e+04, 3.153691299987e+04, 2.980239547587e+04, 2.804159580087e+04, 2.625340046487e+04, 2.443661679987e+04, 2.258996506287e+04, 2.071206947287e+04, 1.880144804087e+04, 1.685650097287e+04, 1.487549741487e+04, 1.285656022387e+04, 1.079764840787e+04, 8.696536786871e+03, 6.550792311871e+03, 4.357746360870e+03, 2.114462155871e+03, -1.823037971295e+02, -2.536147603130e+03, -4.951072989129e+03, -7.431557688129e+03, -9.982634189129e+03, -1.260998875013e+04, -1.532008392213e+04, -1.812031160813e+04, -2.101918632513e+04, -2.402659204113e+04, -2.715410145313e+04, -3.041539476713e+04, -3.382681755213e+04, -3.740813679213e+04, -4.118358559713e+04, -4.518333874113e+04, -4.944564923713e+04, -5.402003132913e+04, -5.897215881813e+04, -6.439168433713e+04, -7.040522649713e+04, -7.719876595913e+04, -8.505692552713e+04, -9.442698684413e+04, -1.059800216851e+05, -1.204521475211e+05, -1.366991822601e+05, -1.498390135911e+05, -1.587465872571e+05, -1.648346657841e+05, -1.693403654671e+05, -1.728867356691e+05, -1.757977880741e+05, -1.782603578991e+05, -1.803908878721e+05, -1.822663127851e+05, -1.839398794511e+05, -1.854498593151e+05, -1.868246297121e+05, -1.880857843911e+05, -1.892501186561e+05, -1.903309424891e+05, -1.913389762781e+05, -1.922829782411e+05, -1.931701940861e+05, -1.940066857951e+05, -1.947975762971e+05, -1.955472344151e+05, -1.962594166541e+05, -1.969373772771e+05, -1.975839547801e+05, -1.982016405201e+05, -1.987926337451e+05, -1.993588861001e+05, -1.999021379361e+05, -2.004239481691e+05, -2.009257190001e+05, -2.014087165371e+05, -2.018740881061e+05, -2.023228768631e+05, -2.027560342071e+05, -2.031744303791e+05, -2.035788635541e+05, -2.039700676841e+05, -2.043487192881e+05, -2.047154433631e+05, -2.050708185391e+05, -2.054153816011e+05, -2.057496314701e+05, -2.060740327141e+05, -2.063890186591e+05, -2.066949941481e+05, -2.069923380101e+05, -2.072814052541e+05, -2.075625290381e+05, -2.078360224441e+05, -2.081021800631e+05, -2.083612794381e+05, -2.086135823591e+05, -2.088593360421e+05, -2.090987741971e+05, -2.093321180001e+05, -2.095595769791e+05, -2.097813498201e+05, -2.099976251131e+05, -2.102085820221e+05, -2.104143909081e+05, -2.106152139051e+05, -2.108112054381e+05, -2.110025127101e+05, -2.111892761491e+05, -2.113716298211e+05, -2.115497018061e+05, -2.117236145601e+05, -2.118934852331e+05, -2.120594259801e+05, -2.122215442421e+05, -2.123799430051e+05, -2.125347210491e+05, -2.126859731751e+05, -2.128337904151e+05, -2.129782602351e+05, -2.131194667181e+05, -2.132574907391e+05, -2.133924101271e+05, -2.135242998171e+05, -2.136532319951e+05, -2.137792762321e+05, -2.139024996091e+05, -2.140229668341e+05, -2.141407403601e+05, -2.142558804821e+05, -2.143684454441e+05, -2.144784915261e+05, -2.145860731351e+05, -2.146912428901e+05, -2.147940516951e+05, -2.148945488211e+05, -2.149927819711e+05, -2.150887973461e+05, -2.151826397141e+05, -2.152743524631e+05, -2.153639776611e+05, -2.154515561111e+05, -2.155371273981e+05, -2.156207299431e+05, -2.157024010431e+05, -2.157821769201e+05, -2.158600927611e+05, -2.159361827551e+05, -2.160104801361e+05, -2.160830172131e+05, -2.161538254101e+05, -2.162229352961e+05, -2.162903766141e+05, -2.163561783161e+05, -2.164203685881e+05, -2.164829748761e+05, -2.165440239161e+05, -2.166035417551e+05, -2.166615537781e+05, -2.167180847281e+05, -2.167731587291e+05, -2.168267993081e+05, -2.168790294111e+05, -2.169298714271e+05, -2.169793472031e+05, -2.170274780641e+05, -2.170742848281e+05, -2.171197878211e+05, -2.171640068971e+05, -2.172069614481e+05, -2.172486704231e+05, -2.172891523361e+05, -2.173284252861e+05, -2.173665069641e+05, -2.174034146691e+05, -2.174391653171e+05, -2.174737754541e+05, -2.175072612671e+05, -2.175396385951e+05, -2.175709229361e+05, -2.176011294611e+05, -2.176302730201e+05, -2.176583681531e+05, -2.176854291001e+05, -2.177114698031e+05, -2.177365039251e+05, -2.177605448461e+05, -2.177836056801e+05, -2.178056992771e+05, -2.178268382311e+05, -2.178470348901e+05, -2.178663013571e+05, -2.178846495031e+05, -2.179020909651e+05, -2.179186371611e+05, -2.179342992901e+05, -2.179490883371e+05, -2.179630150831e+05, -2.179760901071e+05, -2.179883237911e+05, -2.179997263261e+05, -2.180103077171e+05, -2.180200777851e+05, -2.180290461761e+05, -2.180372223611e+05, -2.180446156421e+05, -2.180512351571e+05, -2.180570898831e+05, -2.180621886391e+05, -2.180665400901e+05, -2.180701527521e+05, -2.180730349941e+05, -2.180751950431e+05, -2.180766409861e+05, -2.180773807721e+05, -2.180774222181e+05, -2.180767730111e+05, -2.180754407081e+05, -2.180734327451e+05, -2.180707564321e+05, -2.180674189641e+05, -2.180634274171e+05, -2.180587887521e+05, -2.180535098211e+05, -2.180475973661e+05, -2.180410580221e+05, -2.180338983181e+05, -2.180261246831e+05, -2.180177434451e+05, -2.180087608331e+05, -2.179991829821e+05, -2.179890159311e+05, -2.179782656281e+05, -2.179669379311e+05, -2.179550386081e+05, -2.179425733431e+05, -2.179295477321e+05, -2.179159672921e+05, -2.179018374551e+05, -2.178871635741e+05, -2.178719509251e+05, -2.178562047061e+05, -2.178399300401e+05, -2.178231319781e+05, -2.178058154951e+05, -2.177879854991e+05, -2.177696468251e+05, -2.177508042431e+05, -2.177314624541e+05, -2.177116260931e+05, -2.176912997331e+05, -2.176704878801e+05, -2.176491949811e+05, -2.176274254211e+05, -2.176051835261e+05, -2.175824735611e+05, -2.175592997361e+05, -2.175356662041e+05, -2.175115770601e+05, -2.174870363471e+05, -2.174620480551e+05, -2.174366161171e+05, -2.174107444201e+05, -2.173844367961e+05, -2.173576970291e+05, -2.173305288541e+05, -2.173029359561e+05, -2.172749219761e+05, -2.172464905051e+05, -2.172176450901e+05, -2.171883892331e+05, -2.171587263911e+05, -2.171286599801e+05, -2.170981933701e+05, -2.170673298901e+05, -2.170360728301e+05, -2.170044254371e+05, -2.169723909181e+05, -2.169399724431e+05, -2.169071731401e+05, -2.168739961021e+05, -2.168404443831e+05, -2.168065210011e+05, -2.167722289381e+05, -2.167375711401e+05, -2.167025505161e+05, -2.166671699451e+05, -2.166314322691e+05, -2.165953402961e+05, -2.165588968041e+05, -2.165221045351e+05, -2.164849662041e+05, -2.164474844891e+05, -2.164096620421e+05, -2.163715014821e+05, -2.163330053981e+05, -2.162941763521e+05, -2.162550168731e+05, -2.162155294661e+05, -2.161757166051e+05, -2.161355807351e+05, -2.160951242781e+05, -2.160543496261e+05, -2.160132591451e+05, -2.159718551761e+05, -2.159301400311e+05, -2.158881160021e+05, -2.158457853521e+05, -2.158031503201e+05, -2.157602131231e+05, -2.157169759511e+05, -2.156734409721e+05, -2.156296103311e+05, -2.155854861511e+05, -2.155410705301e+05, -2.154963655451e+05, -2.154513732521e+05, -2.154060956851e+05, -2.153605348551e+05, -2.153146927541e+05, -2.152685713531e+05, -2.152221726031e+05, -2.151754984321e+05, -2.151285507521e+05, -2.150813314521e+05, -2.150338424061e+05, -2.149860854641e+05, -2.149380624611e+05, -2.148897752111e+05, -2.148412255111e+05, -2.147924151411e+05, -2.147433458601e+05, -2.146940194131e+05, -2.146444375251e+05, -2.145946019051e+05, -2.145445142461e+05, -2.144941762231e+05, -2.144435894951e+05, -2.143927557051e+05, -2.143416764801e+05, -2.142903534311e+05, -2.142387881541e+05, -2.141869822281e+05, -2.141349372181e+05, -2.140826546751e+05, -2.140301361341e+05, -2.139773831151e+05, -2.139243971251e+05, -2.138711796551e+05, -2.138177321831e+05, -2.137640561731e+05, -2.137101530761e+05, -2.136560243281e+05, -2.136016713531e+05, -2.135470955611e+05, -2.134922983491e+05, -2.134372811011e+05, -2.133820451891e+05, -2.133265919721e+05, -2.132709227971e+05, -2.132150389981e+05, -2.131589418961e+05, -2.131026328041e+05, -2.130461130181e+05, -2.129893838261e+05, -2.129324465031e+05, -2.128753023131e+05, -2.128179525081e+05, -2.127603983311e+05, -2.127026410121e+05, -2.126446817691e+05, -2.125865218131e+05, -2.125281623421e+05, -2.124696045431e+05, -2.124108495941e+05, -2.123518986621e+05, -2.122927529051e+05, -2.122334134691e+05, -2.121738814911e+05, -2.121141581001e+05, -2.120542444131e+05, -2.119941415381e+05, -2.119338505741e+05, -2.118733726101e+05, -2.118127087281e+05, -2.117518599971e+05, -2.116908274801e+05, -2.116296122301e+05, -2.115682152921e+05, -2.115066377011e+05, -2.114448804841e+05, -2.113829446611e+05, -2.113208312391e+05, -2.112585412231e+05, -2.111960756041e+05, -2.111334353691e+05, -2.110706214941e+05, -2.110076349491e+05, -2.109444766961e+05, -2.108811476871e+05, -2.108176488691e+05, -2.107539811791e+05, -2.106901455491e+05, -2.106261429021e+05, -2.105619741531e+05, -2.104976402111e+05, -2.104331419781e+05, -2.103684803461e+05, -2.103036562041e+05, -2.102386704311e+05, -2.101735239001e+05, -2.101082174771e+05, -2.100427520221e+05, -2.099771283881e+05, -2.099113474201e+05, -2.098454099571e+05, -2.097793168341e+05, -2.097130688761e+05, -2.096466669031e+05, -2.095801117291e+05, -2.095134041611e+05, -2.094465450021e+05, -2.093795350461e+05, -2.093123750821e+05, -2.092450658931e+05, -2.091776082571e+05, -2.091100029451e+05, -2.090422507231e+05, -2.089743523501e+05, -2.089063085811e+05, -2.088381201631e+05, -2.087697878401e+05, -2.087013123491e+05, -2.086326944221e+05, -2.085639347851e+05, -2.084950341591e+05, -2.084259932611e+05, -2.083568127991e+05, -2.082874934811e+05, -2.082180360051e+05, -2.081484410661e+05, -2.080787093561e+05, -2.080088415571e+05, -2.079388383521e+05, -2.078687004131e+05, -2.077984284131e+05, -2.077280230151e+05, -2.076574848801e+05, -2.075868146651e+05, -2.075160130211e+05, -2.074450805931e+05, -2.073740180241e+05, -2.073028259511e+05, -2.072315050071e+05, -2.071600558201e+05, -2.070884790141e+05, -2.070167752101e+05, -2.069449450211e+05, -2.068729890591e+05, -2.068009079311e+05, -2.067287022391e+05, -2.066563725811e+05, -2.065839195511e+05, -2.065113437401e+05, -2.064386457331e+05, -2.063658261121e+05, -2.062928854551e+05, -2.062198243361e+05, -2.061466433251e+05, -2.060733429881e+05, -2.059999238871e+05, -2.059263865811e+05, -2.058527316251e+05, -2.057789595691e+05, -2.057050709611e+05, -2.056310663441e+05, -2.055569462581e+05, -2.054827112391e+05, -2.054083618211e+05, -2.053338985311e+05, -2.052593218971e+05, -2.051846324391e+05, -2.051098306781e+05, -2.050349171271e+05, -2.049598922981e+05, -2.048847567011e+05, -2.048095108401e+05, -2.047341552181e+05, -2.046586903321e+05, -2.045831166791e+05, -2.045074347491e+05, -2.044316450331e+05, -2.043557480151e+05, -2.042797441791e+05, -2.042036340031e+05, -2.041274179651e+05},
    {3.412871758587e+04, 3.243167731587e+04, 3.070992906387e+04, 2.896247094687e+04, 2.718823294387e+04, 2.538607039687e+04, 2.355475670087e+04, 2.169297504687e+04, 1.979930908087e+04, 1.787223228187e+04, 1.591009585387e+04, 1.391111487087e+04, 1.187335233987e+04, 9.794700820871e+03, 7.672861105871e+03, 5.505317380871e+03, 3.289308138871e+03, 1.021791935871e+03, -1.300593167129e+03, -3.681577937129e+03, -6.125319943129e+03, -8.636473596130e+03, -1.122027541613e+04, -1.388264868513e+04, -1.663033304713e+04, -1.947104654313e+04, -2.241369031013e+04, -2.546861016213e+04, -2.864793499413e+04, -3.196602061513e+04, -3.544004069013e+04, -3.909078677513e+04, -4.294377179313e+04, -4.703078399613e+04, -5.139212673113e+04, -5.607993100113e+04, -6.116319372413e+04, -6.673566267313e+04, -7.292847400513e+04, -7.993046732113e+04, -8.801856923413e+04, -9.758790207813e+04, -1.091137189591e+05, -1.228004554941e+05, -1.368701266241e+05, -1.481959048681e+05, -1.563282474581e+05, -1.621672464011e+05, -1.666006761571e+05, -1.701403383471e+05, -1.730713493541e+05, -1.755644814681e+05, -1.777290782271e+05, -1.796388567111e+05, -1.813456130961e+05, -1.828870050371e+05, -1.842912038201e+05, -1.855797975541e+05, -1.867696720201e+05, -1.878742698561e+05, -1.889044580651e+05, -1.898691408071e+05, -1.907757018221e+05, -1.916303299871e+05, -1.924382629771e+05, -1.932039723581e+05, -1.939313061191e+05, -1.946235997441e+05, -1.952837637591e+05, -1.959143534191e+05, -1.965176247081e+05, -1.970955797441e+05, -1.976500038701e+05, -1.981824962031e+05, -1.986944949651e+05, -1.991872986241e+05, -1.996620836571e+05, -2.001199195561e+05, -2.005617815771e+05, -2.009885616341e+05, -2.014010776401e+05, -2.018000815721e+05, -2.021862664481e+05, -2.025602724051e+05, -2.029226919981e+05, -2.032740748471e+05, -2.036149317301e+05, -2.039457381871e+05, -2.042669377201e+05, -2.045789446271e+05, -2.048821465341e+05, -2.051769066511e+05, -2.054635657961e+05, -2.057424442091e+05, -2.060138431881e+05, -2.062780465631e+05, -2.065353220281e+05, -2.067859223501e+05, -2.070300864621e+05, -2.072680404641e+05, -2.074999985261e+05, -2.077261637231e+05, -2.079467287871e+05, -2.081618768061e+05, -2.083717818571e+05, -2.085766095941e+05, -2.087765177831e+05, -2.089716568001e+05, -2.091621700881e+05, -2.093481945781e+05, -2.095298610811e+05, -2.097072946491e+05, -2.098806149091e+05, -2.100499363801e+05, -2.102153687531e+05, -2.103770171691e+05, -2.105349824641e+05, -2.106893614031e+05, -2.108402469011e+05, -2.109877282231e+05, -2.111318911751e+05, -2.112728182851e+05, -2.114105889641e+05, -2.115452796671e+05, -2.116769640361e+05, -2.118057130411e+05, -2.119315951071e+05, -2.120546762331e+05, -2.121750201141e+05, -2.122926882381e+05, -2.124077399981e+05, -2.125202327781e+05, -2.126302220531e+05, -2.127377614641e+05, -2.128429029061e+05, -2.129456966011e+05, -2.130461911721e+05, -2.131444337071e+05, -2.132404698281e+05, -2.133343437491e+05, -2.134260983381e+05, -2.135157751661e+05, -2.136034145651e+05, -2.136890556741e+05, -2.137727364891e+05, -2.138544939031e+05, -2.139343637541e+05, -2.140123808651e+05, -2.140885790771e+05, -2.141629912941e+05, -2.142356495101e+05, -2.143065848501e+05, -2.143758275971e+05, -2.144434072221e+05, -2.145093524171e+05, -2.145736911211e+05, -2.146364505451e+05, -2.146976571981e+05, -2.147573369141e+05, -2.148155148721e+05, -2.148722156181e+05, -2.149274630891e+05, -2.149812806341e+05, -2.150336910271e+05, -2.150847164941e+05, -2.151343787261e+05, -2.151826988981e+05, -2.152296976831e+05, -2.152753952741e+05, -2.153198113901e+05, -2.153629653011e+05, -2.154048758321e+05, -2.154455613861e+05, -2.154850399481e+05, -2.155233291061e+05, -2.155604460561e+05, -2.155964076181e+05, -2.156312302441e+05, -2.156649300331e+05, -2.156975227361e+05, -2.157290237721e+05, -2.157594482311e+05, -2.157888108891e+05, -2.158171262131e+05, -2.158444083711e+05, -2.158706712431e+05, -2.158959284231e+05, -2.159201932331e+05, -2.159434787251e+05, -2.159657976941e+05, -2.159871626801e+05, -2.160075859761e+05, -2.160270796381e+05, -2.160456554851e+05, -2.160633251111e+05, -2.160800998871e+05, -2.160959909691e+05, -2.161110093031e+05, -2.161251656281e+05, -2.161384704861e+05, -2.161509342221e+05, -2.161625669921e+05, -2.161733787671e+05, -2.161833793361e+05, -2.161925783131e+05, -2.162009851391e+05, -2.162086090871e+05, -2.162154592681e+05, -2.162215446311e+05, -2.162268739701e+05, -2.162314559251e+05, -2.162352989891e+05, -2.162384115071e+05, -2.162408016851e+05, -2.162424775881e+05, -2.162434471461e+05, -2.162437181551e+05, -2.162432982841e+05, -2.162421950721e+05, -2.162404159371e+05, -2.162379681741e+05, -2.162348589591e+05, -2.162310953551e+05, -2.162266843091e+05, -2.162216326571e+05, -2.162159471281e+05, -2.162096343421e+05, -2.162027008191e+05, -2.161951529741e+05, -2.161869971231e+05, -2.161782394861e+05, -2.161688861841e+05, -2.161589432481e+05, -2.161484166171e+05, -2.161373121371e+05, -2.161256355691e+05, -2.161133925871e+05, -2.161005887811e+05, -2.160872296571e+05, -2.160733206411e+05, -2.160588670801e+05, -2.160438742411e+05, -2.160283473161e+05, -2.160122914231e+05, -2.159957116041e+05, -2.159786128311e+05, -2.159610000051e+05, -2.159428779571e+05, -2.159242514511e+05, -2.159051251831e+05, -2.158855037851e+05, -2.158653918231e+05, -2.158447938021e+05, -2.158237141631e+05, -2.158021572881e+05, -2.157801274991e+05, -2.157576290591e+05, -2.157346661741e+05, -2.157112429931e+05, -2.156873636101e+05, -2.156630320651e+05, -2.156382523441e+05, -2.156130283801e+05, -2.155873640571e+05, -2.155612632041e+05, -2.155347296041e+05, -2.155077669891e+05, -2.154803790451e+05, -2.154525694091e+05, -2.154243416721e+05, -2.153956993801e+05, -2.153666460321e+05, -2.153371850871e+05, -2.153073199561e+05, -2.152770540121e+05, -2.152463905821e+05, -2.152153329541e+05, -2.151838843761e+05, -2.151520480541e+05, -2.151198271571e+05, -2.150872248141e+05, -2.150542441181e+05, -2.150208881211e+05, -2.149871598431e+05, -2.149530622641e+05, -2.149185983321e+05, -2.148837709551e+05, -2.148485830131e+05, -2.148130373471e+05, -2.147771367671e+05, -2.147408840501e+05, -2.147042819401e+05, -2.146673331491e+05, -2.146300403611e+05, -2.145924062241e+05, -2.145544333601e+05, -2.145161243591e+05, -2.144774817821e+05, -2.144385081601e+05, -2.143992059981e+05, -2.143595777711e+05, -2.143196259281e+05, -2.142793528871e+05, -2.142387610441e+05, -2.141978527651e+05, -2.141566303921e+05, -2.141150962401e+05, -2.140732526001e+05, -2.140311017361e+05, -2.139886458901e+05, -2.139458872791e+05, -2.139028280941e+05, -2.138594705061e+05, -2.138158166601e+05, -2.137718686801e+05, -2.137276286651e+05, -2.136830986961e+05, -2.136382808271e+05, -2.135931770951e+05, -2.135477895131e+05, -2.135021200751e+05, -2.134561707511e+05, -2.134099434941e+05, -2.133634402361e+05, -2.133166628881e+05, -2.132696133421e+05, -2.132222934721e+05, -2.131747051311e+05, -2.131268501551e+05, -2.130787303601e+05, -2.130303475451e+05, -2.129817034891e+05, -2.129327999561e+05, -2.128836386901e+05, -2.128342214201e+05, -2.127845498551e+05, -2.127346256911e+05, -2.126844506021e+05, -2.126340262521e+05, -2.125833542841e+05, -2.125324363261e+05, -2.124812739921e+05, -2.124298688781e+05, -2.123782225671e+05, -2.123263366261e+05, -2.122742126051e+05, -2.122218520421e+05, -2.121692564591e+05, -2.121164273651e+05, -2.120633662521e+05, -2.120100746001e+05, -2.119565538751e+05, -2.119028055301e+05, -2.118488310021e+05, -2.117946317171e+05, -2.117402090861e+05, -2.116855645101e+05, -2.116306993721e+05, -2.115756150481e+05, -2.115203128981e+05, -2.114647942691e+05, -2.114090604991e+05, -2.113531129121e+05, -2.112969528181e+05, -2.112405815191e+05, -2.111840003031e+05, -2.111272104481e+05, -2.110702132181e+05, -2.110130098691e+05, -2.109556016431e+05, -2.108979897731e+05, -2.108401754801e+05, -2.107821599741e+05, -2.107239444571e+05, -2.106655301181e+05, -2.106069181351e+05, -2.105481096771e+05, -2.104891059051e+05, -2.104299079661e+05, -2.103705169991e+05, -2.103109341341e+05, -2.102511604901e+05, -2.101911971781e+05, -2.101310452971e+05, -2.100707059391e+05, -2.100101801851e+05, -2.099494691101e+05, -2.098885737761e+05, -2.098274952381e+05, -2.097662345421e+05, -2.097047927271e+05, -2.096431708201e+05, -2.095813698411e+05, -2.095193908031e+05, -2.094572347091e+05, -2.093949025531e+05, -2.093323953241e+05, -2.092697139991e+05, -2.092068595501e+05, -2.091438329401e+05, -2.090806351241e+05, -2.090172670491e+05, -2.089537296561e+05, -2.088900238761e+05, -2.088261506351e+05, -2.087621108501e+05, -2.086979054301e+05, -2.086335352791e+05, -2.085690012931e+05, -2.085043043601e+05, -2.084394453611e+05, -2.083744251711e+05, -2.083092446581e+05, -2.082439046841e+05, -2.081784061011e+05, -2.081127497581e+05, -2.080469364961e+05, -2.079809671491e+05, -2.079148425451e+05, -2.078485635061e+05, -2.077821308471e+05, -2.077155453771e+05, -2.076488078991e+05, -2.075819192111e+05, -2.075148801011e+05, -2.074476913561e+05, -2.073803537541e+05, -2.073128680681e+05, -2.072452350641e+05, -2.071774555041e+05, -2.071095301451e+05, -2.070414597341e+05, -2.069732450181e+05, -2.069048867341e+05, -2.068363856161e+05, -2.067677423911e+05, -2.066989577841e+05, -2.066300325091e+05, -2.065609672801e+05, -2.064917628031e+05, -2.064224197791e+05, -2.063529389061e+05, -2.062833208731e+05, -2.062135663681e+05, -2.061436760721e+05, -2.060736506621e+05, -2.060034908081e+05, -2.059331971781e+05, -2.058627704331e+05, -2.057922112311e+05, -2.057215202241e+05, -2.056506980611e+05, -2.055797453841e+05, -2.055086628311e+05, -2.054374510381e+05, -2.053661106341e+05, -2.052946422441e+05, -2.052230464901e+05, -2.051513239861e+05, -2.050794753471e+05, -2.050075011801e+05, -2.049354020881e+05, -2.048631786721e+05, -2.047908315261e+05, -2.047183612411e+05, -2.046457684061e+05, -2.045730536031e+05, -2.045002174121e+05, -2.044272604071e+05, -2.043541831601e+05, -2.042809862391e+05, -2.042076702061e+05, -2.041342356221e+05, -2.040606830421e+05, -2.039870130201e+05, -2.039132261021e+05, -2.038393228341e+05, -2.037653037571e+05, -2.036911694091e+05, -2.036169203231e+05, -2.035425570301e+05, -2.034680800571e+05, -2.033934899271e+05, -2.033187871601e+05, -2.032439722721e+05, -2.031690457761e+05, -2.030940081821e+05, -2.030188599971e+05, -2.029436017231e+05, -2.028682338611e+05, -2.027927569051e+05, -2.027171713511e+05, -2.026414776881e+05, -2.025656764021e+05, -2.024897679781e+05},
    {3.501223007987e+04, 3.332728168787e+04, 3.161814535387e+04, 2.988385709387e+04, 2.812338856287e+04, 2.633564103487e+04, 2.451943865487e+04, 2.267352083587e+04, 2.079653367387e+04, 1.888702022087e+04, 1.694340942887e+04, 1.496400352987e+04, 1.294696358387e+04, 1.089029285287e+04, 8.791817603871e+03, 6.649164826871e+03, 4.459736261871e+03, 2.220677953871e+03, -7.115561012947e+01, -2.419244011129e+03, -4.827451589129e+03, -7.300088281129e+03, -9.841983175129e+03, -1.245857409213e+04, -1.515601758213e+04, -1.794132517613e+04, -2.082253374913e+04, -2.380892071913e+04, -2.691127889713e+04, -3.014227177813e+04, -3.351689886313e+04, -3.705311394213e+04, -4.077265964313e+04, -4.470221358213e+04, -4.887499245513e+04, -5.333304321313e+04, -5.813058552813e+04, -6.333898642213e+04, -6.905426542813e+04, -7.540833648013e+04, -8.258466470513e+04, -9.083378636813e+04, -1.004624083601e+05, -1.117235846661e+05, -1.243852124841e+05, -1.366227095401e+05, -1.465615888071e+05, -1.540262315331e+05, -1.596099363611e+05, -1.639518567031e+05, -1.674670526571e+05, -1.704037533721e+05, -1.729164949851e+05, -1.751068383261e+05, -1.770446232071e+05, -1.787796805981e+05, -1.803486761471e+05, -1.817793006441e+05, -1.830929369241e+05, -1.843064150821e+05, -1.854332026201e+05, -1.864842330891e+05, -1.874684967261e+05, -1.883934702151e+05, -1.892654351491e+05, -1.900897178431e+05, -1.908708725221e+05, -1.916128230801e+05, -1.923189740491e+05, -1.929922983871e+05, -1.936354076091e+05, -1.942506083201e+05, -1.948399481531e+05, -1.954052533961e+05, -1.959481600361e+05, -1.964701395421e+05, -1.969725204191e+05, -1.974565063311e+05, -1.979231914291e+05, -1.983735733921e+05, -1.988085645571e+05, -1.992290014931e+05, -1.996356532481e+05, -2.000292285041e+05, -2.004103817921e+05, -2.007797189281e+05, -2.011378017751e+05, -2.014851524331e+05, -2.018222569341e+05, -2.021495685191e+05, -2.024675105431e+05, -2.027764790701e+05, -2.030768451811e+05, -2.033689570491e+05, -2.036531417981e+05, -2.039297071781e+05, -2.041989430701e+05, -2.044611228561e+05, -2.047165046531e+05, -2.049653324331e+05, -2.052078370471e+05, -2.054442371511e+05, -2.056747400551e+05, -2.058995425021e+05, -2.061188313721e+05, -2.063327843391e+05, -2.065415704671e+05, -2.067453507591e+05, -2.069442786661e+05, -2.071385005551e+05, -2.073281561391e+05, -2.075133788781e+05, -2.076942963471e+05, -2.078710305811e+05, -2.080436983931e+05, -2.082124116691e+05, -2.083772776431e+05, -2.085383991511e+05, -2.086958748761e+05, -2.088497995641e+05, -2.090002642341e+05, -2.091473563751e+05, -2.092911601231e+05, -2.094317564371e+05, -2.095692232531e+05, -2.097036356381e+05, -2.098350659281e+05, -2.099635838611e+05, -2.100892566991e+05, -2.102121493501e+05, -2.103323244731e+05, -2.104498425821e+05, -2.105647621461e+05, -2.106771396811e+05, -2.107870298341e+05, -2.108944854681e+05, -2.109995577401e+05, -2.111022961731e+05, -2.112027487261e+05, -2.113009618601e+05, -2.113969806001e+05, -2.114908485961e+05, -2.115826081781e+05, -2.116723004081e+05, -2.117599651341e+05, -2.118456410341e+05, -2.119293656651e+05, -2.120111755061e+05, -2.120911059991e+05, -2.121691915871e+05, -2.122454657551e+05, -2.123199610631e+05, -2.123927091801e+05, -2.124637409191e+05, -2.125330862671e+05, -2.126007744141e+05, -2.126668337811e+05, -2.127312920491e+05, -2.127941761831e+05, -2.128555124581e+05, -2.129153264831e+05, -2.129736432231e+05, -2.130304870201e+05, -2.130858816151e+05, -2.131398501671e+05, -2.131924152741e+05, -2.132435989891e+05, -2.132934228381e+05, -2.133419078401e+05, -2.133890745171e+05, -2.134349429181e+05, -2.134795326241e+05, -2.135228627721e+05, -2.135649520611e+05, -2.136058187711e+05, -2.136454807721e+05, -2.136839555381e+05, -2.137212601591e+05, -2.137574113521e+05, -2.137924254711e+05, -2.138263185191e+05, -2.138591061571e+05, -2.138908037151e+05, -2.139214262001e+05, -2.139509883061e+05, -2.139795044251e+05, -2.140069886491e+05, -2.140334547861e+05, -2.140589163631e+05, -2.140833866341e+05, -2.141068785901e+05, -2.141294049621e+05, -2.141509782321e+05, -2.141716106391e+05, -2.141913141811e+05, -2.142101006281e+05, -2.142279815211e+05, -2.142449681851e+05, -2.142610717281e+05, -2.142763030511e+05, -2.142906728521e+05, -2.143041916291e+05, -2.143168696891e+05, -2.143287171481e+05, -2.143397439401e+05, -2.143499598201e+05, -2.143593743661e+05, -2.143679969871e+05, -2.143758369241e+05, -2.143829032581e+05, -2.143892049071e+05, -2.143947506371e+05, -2.143995490621e+05, -2.144036086471e+05, -2.144069377141e+05, -2.144095444411e+05, -2.144114368731e+05, -2.144126229141e+05, -2.144131103411e+05, -2.144129068001e+05, -2.144120198121e+05, -2.144104567741e+05, -2.144082249611e+05, -2.144053315351e+05, -2.144017835371e+05, -2.143975878991e+05, -2.143927514421e+05, -2.143872808771e+05, -2.143811828121e+05, -2.143744637511e+05, -2.143671300951e+05, -2.143591881471e+05, -2.143506441131e+05, -2.143415041041e+05, -2.143317741381e+05, -2.143214601411e+05, -2.143105679511e+05, -2.142991033161e+05, -2.142870719021e+05, -2.142744792871e+05, -2.142613309691e+05, -2.142476323641e+05, -2.142333888111e+05, -2.142186055681e+05, -2.142032878211e+05, -2.141874406771e+05, -2.141710691741e+05, -2.141541782761e+05, -2.141367728751e+05, -2.141188577981e+05, -2.141004378021e+05, -2.140815175771e+05, -2.140621017491e+05, -2.140421948781e+05, -2.140218014651e+05, -2.140009259471e+05, -2.139795726991e+05, -2.139577460381e+05, -2.139354502251e+05, -2.139126894601e+05, -2.138894678891e+05, -2.138657896021e+05, -2.138416586351e+05, -2.138170789711e+05, -2.137920545401e+05, -2.137665892221e+05, -2.137406868441e+05, -2.137143511851e+05, -2.136875859771e+05, -2.136603949001e+05, -2.136327815911e+05, -2.136047496371e+05, -2.135763025841e+05, -2.135474439281e+05, -2.135181771261e+05, -2.134885055881e+05, -2.134584326841e+05, -2.134279617411e+05, -2.133970960451e+05, -2.133658388421e+05, -2.133341933391e+05, -2.133021627011e+05, -2.132697500591e+05, -2.132369585021e+05, -2.132037910841e+05, -2.131702508221e+05, -2.131363406981e+05, -2.131020636551e+05, -2.130674226061e+05, -2.130324204271e+05, -2.129970599601e+05, -2.129613440131e+05, -2.129252753651e+05, -2.128888567591e+05, -2.128520909081e+05, -2.128149804941e+05, -2.127775281661e+05, -2.127397365461e+05, -2.127016082241e+05, -2.126631457611e+05, -2.126243516901e+05, -2.125852285151e+05, -2.125457787111e+05, -2.125060047261e+05, -2.124659089831e+05, -2.124254938741e+05, -2.123847617691e+05, -2.123437150071e+05, -2.123023559071e+05, -2.122606867591e+05, -2.122187098281e+05, -2.121764273571e+05, -2.121338415621e+05, -2.120909546381e+05, -2.120477687531e+05, -2.120042860571e+05, -2.119605086711e+05, -2.119164386991e+05, -2.118720782191e+05, -2.118274292891e+05, -2.117824939451e+05, -2.117372742021e+05, -2.116917720541e+05, -2.116459894741e+05, -2.115999284151e+05, -2.115535908091e+05, -2.115069785711e+05, -2.114600935921e+05, -2.114129377471e+05, -2.113655128921e+05, -2.113178208621e+05, -2.112698634761e+05, -2.112216425331e+05, -2.111731598151e+05, -2.111244170851e+05, -2.110754160911e+05, -2.110261585601e+05, -2.109766462061e+05, -2.109268807221e+05, -2.108768637891e+05, -2.108265970671e+05, -2.107760822031e+05, -2.107253208261e+05, -2.106743145521e+05, -2.106230649771e+05, -2.105715736871e+05, -2.105198422491e+05, -2.104678722151e+05, -2.104156651261e+05, -2.103632225031e+05, -2.103105458561e+05, -2.102576366821e+05, -2.102044964601e+05, -2.101511266581e+05, -2.100975287291e+05, -2.100437041141e+05, -2.099896542391e+05, -2.099353805181e+05, -2.098808843501e+05, -2.098261671241e+05, -2.097712302151e+05, -2.097160749841e+05, -2.096607027821e+05, -2.096051149461e+05, -2.095493128011e+05, -2.094932976621e+05, -2.094370708301e+05, -2.093806335961e+05, -2.093239872371e+05, -2.092671330211e+05, -2.092100722041e+05, -2.091528060301e+05, -2.090953357351e+05, -2.090376625391e+05, -2.089797876571e+05, -2.089217122891e+05, -2.088634376261e+05, -2.088049648491e+05, -2.087462951301e+05, -2.086874296271e+05, -2.086283694921e+05, -2.085691158651e+05, -2.085096698771e+05, -2.084500326481e+05, -2.083902052921e+05, -2.083301889081e+05, -2.082699845911e+05, -2.082095934231e+05, -2.081490164801e+05, -2.080882548251e+05, -2.080273095171e+05, -2.079661816021e+05, -2.079048721191e+05, -2.078433820991e+05, -2.077817125621e+05, -2.077198645241e+05, -2.076578389871e+05, -2.075956369501e+05, -2.075332594001e+05, -2.074707073181e+05, -2.074079816761e+05, -2.073450834391e+05, -2.072820135641e+05, -2.072187729991e+05, -2.071553626861e+05, -2.070917835591e+05, -2.070280365441e+05, -2.069641225591e+05, -2.069000425171e+05, -2.068357973221e+05, -2.067713878701e+05, -2.067068150521e+05, -2.066420797521e+05, -2.065771828441e+05, -2.065121251981e+05, -2.064469076771e+05, -2.063815311371e+05, -2.063159964251e+05, -2.062503043861e+05, -2.061844558541e+05, -2.061184516591e+05, -2.060522926241e+05, -2.059859795661e+05, -2.059195132961e+05, -2.058528946171e+05, -2.057861243281e+05, -2.057192032211e+05, -2.056521320821e+05, -2.055849116911e+05, -2.055175428221e+05, -2.054500262441e+05, -2.053823627201e+05, -2.053145530061e+05, -2.052465978531e+05, -2.051784980081e+05, -2.051102542111e+05, -2.050418671951e+05, -2.049733376911e+05, -2.049046664221e+05, -2.048358541071e+05, -2.047669014591e+05, -2.046978091861e+05, -2.046285779901e+05, -2.045592085701e+05, -2.044897016191e+05, -2.044200578231e+05, -2.043502778651e+05, -2.042803624241e+05, -2.042103121721e+05, -2.041401277771e+05, -2.040698099021e+05, -2.039993592061e+05, -2.039287763431e+05, -2.038580619611e+05, -2.037872167061e+05, -2.037162412171e+05, -2.036451361291e+05, -2.035739020741e+05, -2.035025396791e+05, -2.034310495651e+05, -2.033594323501e+05, -2.032876886471e+05, -2.032158190671e+05, -2.031438242131e+05, -2.030717046871e+05, -2.029994610851e+05, -2.029270940001e+05, -2.028546040201e+05, -2.027819917291e+05, -2.027092577091e+05, -2.026364025351e+05, -2.025634267791e+05, -2.024903310121e+05, -2.024171157961e+05, -2.023437816931e+05, -2.022703292611e+05, -2.021967590521e+05, -2.021230716171e+05, -2.020492675001e+05, -2.019753472451e+05, -2.019013113901e+05, -2.018271604701e+05, -2.017528950171e+05, -2.016785155581e+05, -2.016040226191e+05, -2.015294167191e+05, -2.014546983781e+05, -2.013798681081e+05, -2.013049264211e+05, -2.012298738231e+05, -2.011547108201e+05, -2.010794379131e+05, -2.010040555971e+05, -2.009285643691e+05, -2.008529647191e+05},
    {3.589672605887e+04, 3.422372739087e+04, 3.252704842687e+04, 3.080576156687e+04, 2.905887840787e+04, 2.728534417887e+04, 2.548403151087e+04, 2.365373342687e+04, 2.179315546487e+04, 1.990090676387e+04, 1.797548996787e+04, 1.601528973087e+04, 1.401855960487e+04, 1.198340700787e+04, 9.907775919871e+03, 7.789426895870e+03, 5.625913855870e+03, 3.414557007871e+03, 1.152411105871e+03, -1.163771970130e+03, -3.537587580130e+03, -5.973031250129e+03, -8.474562211129e+03, -1.104718022413e+04, -1.369651915113e+04, -1.642896182713e+04, -1.925178229713e+04, -2.217332353113e+04, -2.520322167813e+04, -2.835269202813e+04, -3.163489787513e+04, -3.506543218613e+04, -3.866295503613e+04, -4.245004930613e+04, -4.645438705513e+04, -5.071034470813e+04, -5.526127490013e+04, -6.016274434113e+04, -6.548717461113e+04, -7.133039435413e+04, -7.782022487113e+04, -8.512482487313e+04, -9.344960479213e+04, -1.029914905411e+05, -1.137977835361e+05, -1.253516007311e+05, -1.360952562611e+05, -1.449406113511e+05, -1.518258889791e+05, -1.571549528481e+05, -1.613909683121e+05, -1.648669481071e+05, -1.677966783681e+05, -1.703187778781e+05, -1.725267486681e+05, -1.744861422041e+05, -1.762444450381e+05, -1.778370239791e+05, -1.792908518961e+05, -1.806269252171e+05, -1.818618805581e+05, -1.830091053301e+05, -1.840795194781e+05, -1.850821378851e+05, -1.860244829281e+05, -1.869128924001e+05, -1.877527529031e+05, -1.885486791931e+05, -1.893046537001e+05, -1.900241362731e+05, -1.907101513951e+05, -1.913653581101e+05, -1.919921065971e+05, -1.925924842861e+05, -1.931683537311e+05, -1.937213839311e+05, -1.942530763981e+05, -1.947647869901e+05, -1.952577442971e+05, -1.957330652111e+05, -1.961917681841e+05, -1.966347845681e+05, -1.970629683591e+05, -1.974771046091e+05, -1.978779167211e+05, -1.982660727961e+05, -1.986421911781e+05, -1.990068453201e+05, -1.993605680571e+05, -1.997038553871e+05, -2.000371698191e+05, -2.003609433371e+05, -2.006755800521e+05, -2.009814585631e+05, -2.012789340751e+05, -2.015683403011e+05, -2.018499911671e+05, -2.021241823601e+05, -2.023911927191e+05, -2.026512854991e+05, -2.029047095161e+05, -2.031517001891e+05, -2.033924804921e+05, -2.036272618171e+05, -2.038562447681e+05, -2.040796198871e+05, -2.042975683161e+05, -2.045102624131e+05, -2.047178663091e+05, -2.049205364271e+05, -2.051184219631e+05, -2.053116653211e+05, -2.055004025271e+05, -2.056847636011e+05, -2.058648729121e+05, -2.060408494991e+05, -2.062128073741e+05, -2.063808558051e+05, -2.065450995741e+05, -2.067056392221e+05, -2.068625712781e+05, -2.070159884661e+05, -2.071659799121e+05, -2.073126313201e+05, -2.074560251531e+05, -2.075962407931e+05, -2.077333546931e+05, -2.078674405211e+05, -2.079985692961e+05, -2.081268095151e+05, -2.082522272681e+05, -2.083748863551e+05, -2.084948483891e+05, -2.086121728971e+05, -2.087269174131e+05, -2.088391375681e+05, -2.089488871721e+05, -2.090562182971e+05, -2.091611813451e+05, -2.092638251301e+05, -2.093641969321e+05, -2.094623425741e+05, -2.095583064711e+05, -2.096521316951e+05, -2.097438600261e+05, -2.098335320051e+05, -2.099211869831e+05, -2.100068631661e+05, -2.100905976611e+05, -2.101724265211e+05, -2.102523847781e+05, -2.103305064881e+05, -2.104068247651e+05, -2.104813718171e+05, -2.105541789771e+05, -2.106252767371e+05, -2.106946947761e+05, -2.107624619911e+05, -2.108286065271e+05, -2.108931557951e+05, -2.109561365061e+05, -2.110175746921e+05, -2.110774957291e+05, -2.111359243571e+05, -2.111928847071e+05, -2.112484003141e+05, -2.113024941431e+05, -2.113551886051e+05, -2.114065055731e+05, -2.114564664021e+05, -2.115050919461e+05, -2.115524025711e+05, -2.115984181721e+05, -2.116431581881e+05, -2.116866416161e+05, -2.117288870211e+05, -2.117699125551e+05, -2.118097359671e+05, -2.118483746121e+05, -2.118858454671e+05, -2.119221651401e+05, -2.119573498811e+05, -2.119914155931e+05, -2.120243778401e+05, -2.120562518591e+05, -2.120870525701e+05, -2.121167945801e+05, -2.121454921991e+05, -2.121731594401e+05, -2.121998100351e+05, -2.122254574371e+05, -2.122501148301e+05, -2.122737951381e+05, -2.122965110261e+05, -2.123182749141e+05, -2.123390989791e+05, -2.123589951631e+05, -2.123779751781e+05, -2.123960505121e+05, -2.124132324381e+05, -2.124295320141e+05, -2.124449600921e+05, -2.124595273251e+05, -2.124732441641e+05, -2.124861208751e+05, -2.124981675321e+05, -2.125093940281e+05, -2.125198100801e+05, -2.125294252291e+05, -2.125382488481e+05, -2.125462901431e+05, -2.125535581611e+05, -2.125600617891e+05, -2.125658097621e+05, -2.125708106641e+05, -2.125750729301e+05, -2.125786048551e+05, -2.125814145911e+05, -2.125835101551e+05, -2.125848994281e+05, -2.125855901631e+05, -2.125855899811e+05, -2.125849063811e+05, -2.125835467391e+05, -2.125815183101e+05, -2.125788282341e+05, -2.125754835341e+05, -2.125714911231e+05, -2.125668578031e+05, -2.125615902691e+05, -2.125556951121e+05, -2.125491788181e+05, -2.125420477741e+05, -2.125343082681e+05, -2.125259664911e+05, -2.125170285411e+05, -2.125075004201e+05, -2.124973880431e+05, -2.124866972351e+05, -2.124754337321e+05, -2.124636031871e+05, -2.124512111691e+05, -2.124382631641e+05, -2.124247645771e+05, -2.124107207381e+05, -2.123961368951e+05, -2.123810182241e+05, -2.123653698241e+05, -2.123491967231e+05, -2.123325038771e+05, -2.123152961721e+05, -2.122975784251e+05, -2.122793553861e+05, -2.122606317381e+05, -2.122414121001e+05, -2.122217010271e+05, -2.122015030111e+05, -2.121808224831e+05, -2.121596638151e+05, -2.121380313171e+05, -2.121159292431e+05, -2.120933617891e+05, -2.120703330961e+05, -2.120468472501e+05, -2.120229082811e+05, -2.119985201681e+05, -2.119736868371e+05, -2.119484121641e+05, -2.119226999721e+05, -2.118965540381e+05, -2.118699780871e+05, -2.118429757991e+05, -2.118155508051e+05, -2.117877066921e+05, -2.117594470001e+05, -2.117307752251e+05, -2.117016948191e+05, -2.116722091911e+05, -2.116423217081e+05, -2.116120356941e+05, -2.115813544351e+05, -2.115502811741e+05, -2.115188191141e+05, -2.114869714221e+05, -2.114547412241e+05, -2.114221316091e+05, -2.113891456301e+05, -2.113557863011e+05, -2.113220566031e+05, -2.112879594801e+05, -2.112534978411e+05, -2.112186745621e+05, -2.111834924851e+05, -2.111479544161e+05, -2.111120631341e+05, -2.110758213801e+05, -2.110392318681e+05, -2.110022972771e+05, -2.109650202581e+05, -2.109274034321e+05, -2.108894493871e+05, -2.108511606861e+05, -2.108125398601e+05, -2.107735894141e+05, -2.107343118221e+05, -2.106947095321e+05, -2.106547849671e+05, -2.106145405201e+05, -2.105739785591e+05, -2.105331014251e+05, -2.104919114361e+05, -2.104504108831e+05, -2.104086020311e+05, -2.103664871221e+05, -2.103240683751e+05, -2.102813479831e+05, -2.102383281171e+05, -2.101950109231e+05, -2.101513985271e+05, -2.101074930311e+05, -2.100632965141e+05, -2.100188110361e+05, -2.099740386321e+05, -2.099289813181e+05, -2.098836410881e+05, -2.098380199171e+05, -2.097921197581e+05, -2.097459425451e+05, -2.096994901911e+05, -2.096527645911e+05, -2.096057676191e+05, -2.095585011311e+05, -2.095109669661e+05, -2.094631669411e+05, -2.094151028561e+05, -2.093667764961e+05, -2.093181896241e+05, -2.092693439881e+05, -2.092202413171e+05, -2.091708833261e+05, -2.091212717091e+05, -2.090714081471e+05, -2.090212943021e+05, -2.089709318221e+05, -2.089203223381e+05, -2.088694674651e+05, -2.088183688031e+05, -2.087670279361e+05, -2.087154464341e+05, -2.086636258501e+05, -2.086115677251e+05, -2.085592735831e+05, -2.085067449351e+05, -2.084539832761e+05, -2.084009900901e+05, -2.083477668461e+05, -2.082943149961e+05, -2.082406359841e+05, -2.081867312361e+05, -2.081326021671e+05, -2.080782501801e+05, -2.080236766631e+05, -2.079688829911e+05, -2.079138705301e+05, -2.078586406291e+05, -2.078031946281e+05, -2.077475338531e+05, -2.076916596201e+05, -2.076355732311e+05, -2.075792759791e+05, -2.075227691421e+05, -2.074660539901e+05, -2.074091317791e+05, -2.073520037561e+05, -2.072946711561e+05, -2.072371352031e+05, -2.071793971111e+05, -2.071214580821e+05, -2.070633193101e+05, -2.070049819761e+05, -2.069464472521e+05, -2.068877163011e+05, -2.068287902721e+05, -2.067696703101e+05, -2.067103575451e+05, -2.066508531001e+05, -2.065911580881e+05, -2.065312736121e+05, -2.064712007671e+05, -2.064109406371e+05, -2.063504942981e+05, -2.062898628161e+05, -2.062290472511e+05, -2.061680486491e+05, -2.061068680521e+05, -2.060455064921e+05, -2.059839649901e+05, -2.059222445631e+05, -2.058603462151e+05, -2.057982709461e+05, -2.057360197451e+05, -2.056735935941e+05, -2.056109934661e+05, -2.055482203271e+05, -2.054852751361e+05, -2.054221588431e+05, -2.053588723901e+05, -2.052954167131e+05, -2.052317927391e+05, -2.051680013891e+05, -2.051040435751e+05, -2.050399202031e+05, -2.049756321711e+05, -2.049111803711e+05, -2.048465656871e+05, -2.047817889971e+05, -2.047168511711e+05, -2.046517530731e+05, -2.045864955601e+05, -2.045210794821e+05, -2.044555056831e+05, -2.043897750001e+05, -2.043238882641e+05, -2.042578463001e+05, -2.041916499241e+05, -2.041252999501e+05, -2.040587971821e+05, -2.039921424211e+05, -2.039253364591e+05, -2.038583800831e+05, -2.037912740771e+05, -2.037240192131e+05, -2.036566162641e+05, -2.035890659921e+05, -2.035213691551e+05, -2.034535265071e+05, -2.033855387941e+05, -2.033174067571e+05, -2.032491311331e+05, -2.031807126511e+05, -2.031121520381e+05, -2.030434500131e+05, -2.029746072891e+05, -2.029056245771e+05, -2.028365025811e+05, -2.027672420001e+05, -2.026978435261e+05, -2.026283078501e+05, -2.025586356551e+05, -2.024888276211e+05, -2.024188844201e+05, -2.023488067231e+05, -2.022785951931e+05, -2.022082504911e+05, -2.021377732721e+05, -2.020671641851e+05, -2.019964238771e+05, -2.019255529891e+05, -2.018545521571e+05, -2.017834220131e+05, -2.017121631861e+05, -2.016407762981e+05, -2.015692619691e+05, -2.014976208131e+05, -2.014258534411e+05, -2.013539604581e+05, -2.012819424661e+05, -2.012098000641e+05, -2.011375338451e+05, -2.010651443991e+05, -2.009926323101e+05, -2.009199981621e+05, -2.008472425301e+05, -2.007743659891e+05, -2.007013691091e+05, -2.006282524551e+05, -2.005550165891e+05, -2.004816620701e+05, -2.004081894521e+05, -2.003345992851e+05, -2.002608921181e+05, -2.001870684921e+05, -2.001131289491e+05, -2.000390740231e+05, -1.999649042481e+05, -1.998906201531e+05, -1.998162222631e+05, -1.997417111011e+05, -1.996670871841e+05, -1.995923510281e+05, -1.995175031461e+05, -1.994425440451e+05, -1.993674742311e+05, -1.992922942051e+05, -1.992170044661e+05},
    {3.678220432687e+04, 3.512101562287e+04, 3.343664223187e+04, 3.172819148087e+04, 2.999471326087e+04, 2.823519487387e+04, 2.644855527687e+04, 2.463363863687e+04, 2.278920708787e+04, 2.091393256787e+04, 1.900638759587e+04, 1.706503480587e+04, 1.508821504887e+04, 1.307413379087e+04, 1.102084552387e+04, 8.926235821871e+03, 6.788000594871e+03, 4.603621994871e+03, 2.370340301871e+03, 8.512095987052e+01, -2.255384300129e+03, -4.654883540129e+03, -7.117499596129e+03, -9.647836063129e+03, -1.225105705913e+04, -1.493298430113e+04, -1.770021619813e+04, -2.056027511613e+04, -2.352179109013e+04, -2.659473312813e+04, -2.979070332413e+04, -3.312331479913e+04, -3.660868279713e+04, -4.026607035613e+04, -4.411874753313e+04, -4.819514865213e+04, -5.253044809913e+04, -5.716872240813e+04, -6.216591443413e+04, -6.759381046413e+04, -7.354497886513e+04, -8.013742302113e+04, -8.751374333613e+04, -9.582035012213e+04, -1.051430773351e+05, -1.153648110731e+05, -1.258404442211e+05, -1.353726436311e+05, -1.433348266611e+05, -1.497141683581e+05, -1.547942814251e+05, -1.589141001731e+05, -1.623388090781e+05, -1.652505931691e+05, -1.677727304241e+05, -1.699906491741e+05, -1.719654017641e+05, -1.737418801781e+05, -1.753539231531e+05, -1.768275937211e+05, -1.781833469901e+05, -1.794375029551e+05, -1.806032716881e+05, -1.816914829321e+05, -1.827111159241e+05, -1.836696912421e+05, -1.845735654161e+05, -1.854281557191e+05, -1.862381139411e+05, -1.870074623471e+05, -1.877397011581e+05, -1.884378943651e+05, -1.891047388481e+05, -1.897426204881e+05, -1.903536600821e+05, -1.909397511671e+05, -1.915025913871e+05, -1.920437086651e+05, -1.925644831691e+05, -1.930661658511e+05, -1.935498941711e+05, -1.940167055001e+05, -1.944675486101e+05, -1.949032935481e+05, -1.953247401731e+05, -1.957326255691e+05, -1.961276304881e+05, -1.965103849921e+05, -1.968814734041e+05, -1.972414386581e+05, -1.975907861481e+05, -1.979299871331e+05, -1.982594817641e+05, -1.985796817811e+05, -1.988909729221e+05, -1.991937170781e+05, -1.994882542331e+05, -1.997749042051e+05, -2.000539682211e+05, -2.003257303331e+05, -2.005904587111e+05, -2.008484068071e+05, -2.010998144201e+05, -2.013449086641e+05, -2.015839048471e+05, -2.018170072871e+05, -2.020444100421e+05, -2.022662975941e+05, -2.024828454691e+05, -2.026942208131e+05, -2.029005829151e+05, -2.031020836981e+05, -2.032988681661e+05, -2.034910748221e+05, -2.036788360501e+05, -2.038622784731e+05, -2.040415232861e+05, -2.042166865591e+05, -2.043878795291e+05, -2.045552088621e+05, -2.047187769041e+05, -2.048786819091e+05, -2.050350182601e+05, -2.051878766691e+05, -2.053373443651e+05, -2.054835052721e+05, -2.056264401761e+05, -2.057662268791e+05, -2.059029403461e+05, -2.060366528441e+05, -2.061674340681e+05, -2.062953512631e+05, -2.064204693401e+05, -2.065428509841e+05, -2.066625567521e+05, -2.067796451731e+05, -2.068941728371e+05, -2.070061944801e+05, -2.071157630631e+05, -2.072229298551e+05, -2.073277444971e+05, -2.074302550771e+05, -2.075305081911e+05, -2.076285490081e+05, -2.077244213231e+05, -2.078181676201e+05, -2.079098291191e+05, -2.079994458281e+05, -2.080870565891e+05, -2.081726991271e+05, -2.082564100901e+05, -2.083382250891e+05, -2.084181787411e+05, -2.084963047041e+05, -2.085726357111e+05, -2.086472036071e+05, -2.087200393801e+05, -2.087911731911e+05, -2.088606344041e+05, -2.089284516141e+05, -2.089946526771e+05, -2.090592647301e+05, -2.091223142201e+05, -2.091838269261e+05, -2.092438279831e+05, -2.093023419021e+05, -2.093593925901e+05, -2.094150033721e+05, -2.094691970101e+05, -2.095219957211e+05, -2.095734211901e+05, -2.096234945961e+05, -2.096722366201e+05, -2.097196674641e+05, -2.097658068651e+05, -2.098106741121e+05, -2.098542880541e+05, -2.098966671211e+05, -2.099378293291e+05, -2.099777922971e+05, -2.100165732581e+05, -2.100541890701e+05, -2.100906562271e+05, -2.101259908701e+05, -2.101602087951e+05, -2.101933254661e+05, -2.102253560221e+05, -2.102563152891e+05, -2.102862177851e+05, -2.103150777301e+05, -2.103429090571e+05, -2.103697254161e+05, -2.103955401821e+05, -2.104203664661e+05, -2.104442171181e+05, -2.104671047351e+05, -2.104890416701e+05, -2.105100400371e+05, -2.105301117131e+05, -2.105492683531e+05, -2.105675213871e+05, -2.105848820321e+05, -2.106013612921e+05, -2.106169699701e+05, -2.106317186651e+05, -2.106456177841e+05, -2.106586775441e+05, -2.106709079761e+05, -2.106823189301e+05, -2.106929200811e+05, -2.107027209291e+05, -2.107117308091e+05, -2.107199588901e+05, -2.107274141821e+05, -2.107341055391e+05, -2.107400416601e+05, -2.107452310981e+05, -2.107496822571e+05, -2.107534034001e+05, -2.107564026521e+05, -2.107586880001e+05, -2.107602673001e+05, -2.107611482751e+05, -2.107613385241e+05, -2.107608455201e+05, -2.107596766151e+05, -2.107578390431e+05, -2.107553399191e+05, -2.107521862461e+05, -2.107483849171e+05, -2.107439427131e+05, -2.107388663111e+05, -2.107331622821e+05, -2.107268370941e+05, -2.107198971181e+05, -2.107123486251e+05, -2.107041977881e+05, -2.106954506911e+05, -2.106861133201e+05, -2.106761915771e+05, -2.106656912701e+05, -2.106546181241e+05, -2.106429777781e+05, -2.106307757881e+05, -2.106180176271e+05, -2.106047086911e+05, -2.105908542951e+05, -2.105764596781e+05, -2.105615300041e+05, -2.105460703631e+05, -2.105300857721e+05, -2.105135811781e+05, -2.104965614581e+05, -2.104790314201e+05, -2.104609958041e+05, -2.104424592861e+05, -2.104234264771e+05, -2.104039019231e+05, -2.103838901111e+05, -2.103633954621e+05, -2.103424223421e+05, -2.103209750551e+05, -2.102990578481e+05, -2.102766749121e+05, -2.102538303801e+05, -2.102305283331e+05, -2.102067727961e+05, -2.101825677421e+05, -2.101579170921e+05, -2.101328247171e+05, -2.101072944361e+05, -2.100813300201e+05, -2.100549351911e+05, -2.100281136241e+05, -2.100008689461e+05, -2.099732047411e+05, -2.099451245441e+05, -2.099166318471e+05, -2.098877301011e+05, -2.098584227101e+05, -2.098287130371e+05, -2.097986044061e+05, -2.097681000971e+05, -2.097372033511e+05, -2.097059173701e+05, -2.096742453161e+05, -2.096421903141e+05, -2.096097554521e+05, -2.095769437781e+05, -2.095437583071e+05, -2.095102020151e+05, -2.094762778461e+05, -2.094419887081e+05, -2.094073374731e+05, -2.093723269821e+05, -2.093369600411e+05, -2.093012394261e+05, -2.092651678781e+05, -2.092287481091e+05, -2.091919827971e+05, -2.091548745931e+05, -2.091174261141e+05, -2.090796399511e+05, -2.090415186631e+05, -2.090030647811e+05, -2.089642808091e+05, -2.089251692221e+05, -2.088857324671e+05, -2.088459729651e+05, -2.088058931091e+05, -2.087654952671e+05, -2.087247817801e+05, -2.086837549651e+05, -2.086424171121e+05, -2.086007704881e+05, -2.085588173341e+05, -2.085165598671e+05, -2.084740002821e+05, -2.084311407481e+05, -2.083879834141e+05, -2.083445304041e+05, -2.083007838201e+05, -2.082567457431e+05, -2.082124182301e+05, -2.081678033191e+05, -2.081229030251e+05, -2.080777193431e+05, -2.080322542471e+05, -2.079865096921e+05, -2.079404876111e+05, -2.078941899181e+05, -2.078476185081e+05, -2.078007752571e+05, -2.077536620201e+05, -2.077062806371e+05, -2.076586329251e+05, -2.076107206861e+05, -2.075625457041e+05, -2.075141097431e+05, -2.074654145521e+05, -2.074164618621e+05, -2.073672533861e+05, -2.073177908201e+05, -2.072680758461e+05, -2.072181101271e+05, -2.071678953111e+05, -2.071174330301e+05, -2.070667249001e+05, -2.070157725221e+05, -2.069645774801e+05, -2.069131413451e+05, -2.068614656731e+05, -2.068095520031e+05, -2.067574018611e+05, -2.067050167591e+05, -2.066523981941e+05, -2.065995476491e+05, -2.065464665941e+05, -2.064931564851e+05, -2.064396187631e+05, -2.063858548581e+05, -2.063318661851e+05, -2.062776541461e+05, -2.062232201331e+05, -2.061685655221e+05, -2.061136916771e+05, -2.060585999511e+05, -2.060032916841e+05, -2.059477682031e+05, -2.058920308251e+05, -2.058360808541e+05, -2.057799195821e+05, -2.057235482911e+05, -2.056669682501e+05, -2.056101807161e+05, -2.055531869391e+05, -2.054959881531e+05, -2.054385855851e+05, -2.053809804481e+05, -2.053231739481e+05, -2.052651672771e+05, -2.052069616201e+05, -2.051485581481e+05, -2.050899580251e+05, -2.050311624041e+05, -2.049721724281e+05, -2.049129892291e+05, -2.048536139331e+05, -2.047940476521e+05, -2.047342914911e+05, -2.046743465461e+05, -2.046142139031e+05, -2.045538946381e+05, -2.044933898191e+05, -2.044327005061e+05, -2.043718277491e+05, -2.043107725881e+05, -2.042495360571e+05, -2.041881191791e+05, -2.041265229721e+05, -2.040647484421e+05, -2.040027965881e+05, -2.039406684011e+05, -2.038783648651e+05, -2.038158869551e+05, -2.037532356361e+05, -2.036904118701e+05, -2.036274166061e+05, -2.035642507901e+05, -2.035009153571e+05, -2.034374112371e+05, -2.033737393501e+05, -2.033099006111e+05, -2.032458959271e+05, -2.031817261961e+05, -2.031173923131e+05, -2.030528951621e+05, -2.029882356221e+05, -2.029234145651e+05, -2.028584328551e+05, -2.027932913511e+05, -2.027279909051e+05, -2.026625323611e+05, -2.025969165571e+05, -2.025311443261e+05, -2.024652164931e+05, -2.023991338781e+05, -2.023328972931e+05, -2.022665075451e+05, -2.021999654351e+05, -2.021332717571e+05, -2.020664273001e+05, -2.019994328471e+05, -2.019322891731e+05, -2.018649970501e+05, -2.017975572441e+05, -2.017299705131e+05, -2.016622376111e+05, -2.015943592861e+05, -2.015263362811e+05, -2.014581693321e+05, -2.013898591731e+05, -2.013214065271e+05, -2.012528121181e+05, -2.011840766601e+05, -2.011152008631e+05, -2.010461854331e+05, -2.009770310701e+05, -2.009077384691e+05, -2.008383083201e+05, -2.007687413081e+05, -2.006990381131e+05, -2.006291994101e+05, -2.005592258691e+05, -2.004891181561e+05, -2.004188769331e+05, -2.003485028541e+05, -2.002779965711e+05, -2.002073587321e+05, -2.001365899791e+05, -2.000656909481e+05, -1.999946622741e+05, -1.999235045851e+05, -1.998522185061e+05, -1.997808046561e+05, -1.997092636521e+05, -1.996375961051e+05, -1.995658026221e+05, -1.994938838051e+05, -1.994218402551e+05, -1.993496725651e+05, -1.992773813271e+05, -1.992049671261e+05, -1.991324305461e+05, -1.990597721651e+05, -1.989869925581e+05, -1.989140922941e+05, -1.988410719431e+05, -1.987679320651e+05, -1.986946732211e+05, -1.986212959661e+05, -1.985478008521e+05, -1.984741884271e+05, -1.984004592351e+05, -1.983266138181e+05, -1.982526527111e+05, -1.981785764491e+05, -1.981043855631e+05, -1.980300805771e+05, -1.979556620161e+05, -1.978811303991e+05, -1.978064862431e+05, -1.977317300601e+05, -1.976568623591e+05, -1.975818836481e+05},
    {3.766866366387e+04, 3.601914751387e+04, 3.434693058587e+04, 3.265115374887e+04, 3.093090360887e+04, 2.918520774587e+04, 2.741302940287e+04, 2.561326154387e+04, 2.378472020787e+04, 2.192613702587e+04, 2.003615080487e+04, 1.811329799087e+04, 1.615600186987e+04, 1.416256026487e+04, 1.213113148787e+04, 1.005971822087e+04, 7.946148967871e+03, 5.788056607871e+03, 3.582853476871e+03, 1.327702309871e+03, -9.805178511295e+02, -3.345251828129e+03, -5.770315930129e+03, -8.259954727129e+03, -1.081890914813e+04, -1.345249871113e+04, -1.616672148013e+04, -1.896837647813e+04, -2.186521473313e+04, -2.486612718313e+04, -2.798138042213e+04, -3.122291511713e+04, -3.460472724813e+04, -3.814335981413e+04, -4.185854295413e+04, -4.577403462313e+04, -4.991873240113e+04, -5.432814812313e+04, -5.904635154313e+04, -6.412846330713e+04, -6.964360294613e+04, -7.567755961713e+04, -8.233256281113e+04, -8.971714417413e+04, -9.791347943913e+04, -1.069107565621e+05, -1.164786501411e+05, -1.259664334091e+05, -1.345097328761e+05, -1.417444346571e+05, -1.476794152581e+05, -1.525197740241e+05, -1.565166594221e+05, -1.598804440021e+05, -1.627649346311e+05, -1.652788292691e+05, -1.674996224491e+05, -1.694837972811e+05, -1.712735080141e+05, -1.729009093211e+05, -1.743910102871e+05, -1.757636002871e+05, -1.770345786451e+05, -1.782168924861e+05, -1.793212115581e+05, -1.803564228581e+05, -1.813299994251e+05, -1.822482796131e+05, -1.831166815821e+05, -1.839398701211e+05, -1.847218878901e+05, -1.854662597291e+05, -1.861760763391e+05, -1.868540619871e+05, -1.875026297041e+05, -1.881239266181e+05, -1.887198714361e+05, -1.892921856151e+05, -1.898424194581e+05, -1.903719740611e+05, -1.908821198791e+05, -1.913740125171e+05, -1.918487062001e+05, -1.923071653441e+05, -1.927502745201e+05, -1.931788470721e+05, -1.935936326041e+05, -1.939953235101e+05, -1.943845606831e+05, -1.947619385291e+05, -1.951280093841e+05, -1.954832874131e+05, -1.958282520721e+05, -1.961633511791e+05, -1.964890036531e+05, -1.968056019621e+05, -1.971135143151e+05, -1.974130866331e+05, -1.977046443181e+05, -1.979884938551e+05, -1.982649242551e+05, -1.985342083661e+05, -1.987966040591e+05, -1.990523553121e+05, -1.993016931941e+05, -1.995448367611e+05, -1.997819938801e+05, -2.000133619821e+05, -2.002391287491e+05, -2.004594727511e+05, -2.006745640261e+05, -2.008845646191e+05, -2.010896290781e+05, -2.012899049081e+05, -2.014855329971e+05, -2.016766480081e+05, -2.018633787391e+05, -2.020458484641e+05, -2.022241752421e+05, -2.023984722121e+05, -2.025688478621e+05, -2.027354062811e+05, -2.028982473981e+05, -2.030574671971e+05, -2.032131579281e+05, -2.033654082971e+05, -2.035143036461e+05, -2.036599261241e+05, -2.038023548411e+05, -2.039416660231e+05, -2.040779331471e+05, -2.042112270721e+05, -2.043416161681e+05, -2.044691664261e+05, -2.045939415721e+05, -2.047160031691e+05, -2.048354107121e+05, -2.049522217271e+05, -2.050664918491e+05, -2.051782749131e+05, -2.052876230241e+05, -2.053945866361e+05, -2.054992146231e+05, -2.056015543371e+05, -2.057016516791e+05, -2.057995511571e+05, -2.058952959381e+05, -2.059889279061e+05, -2.060804877121e+05, -2.061700148211e+05, -2.062575475601e+05, -2.063431231601e+05, -2.064267778011e+05, -2.065085466461e+05, -2.065884638861e+05, -2.066665627701e+05, -2.067428756441e+05, -2.068174339811e+05, -2.068902684141e+05, -2.069614087651e+05, -2.070308840751e+05, -2.070987226291e+05, -2.071649519861e+05, -2.072295989991e+05, -2.072926898451e+05, -2.073542500431e+05, -2.074143044791e+05, -2.074728774251e+05, -2.075299925611e+05, -2.075856729921e+05, -2.076399412711e+05, -2.076928194131e+05, -2.077443289121e+05, -2.077944907591e+05, -2.078433254591e+05, -2.078908530431e+05, -2.079370930861e+05, -2.079820647171e+05, -2.080257866361e+05, -2.080682771261e+05, -2.081095540641e+05, -2.081496349371e+05, -2.081885368471e+05, -2.082262765291e+05, -2.082628703571e+05, -2.082983343581e+05, -2.083326842171e+05, -2.083659352921e+05, -2.083981026211e+05, -2.084292009291e+05, -2.084592446411e+05, -2.084882478871e+05, -2.085162245091e+05, -2.085431880741e+05, -2.085691518761e+05, -2.085941289451e+05, -2.086181320581e+05, -2.086411737371e+05, -2.086632662661e+05, -2.086844216881e+05, -2.087046518191e+05, -2.087239682471e+05, -2.087423823431e+05, -2.087599052641e+05, -2.087765479581e+05, -2.087923211721e+05, -2.088072354541e+05, -2.088213011591e+05, -2.088345284551e+05, -2.088469273231e+05, -2.088585075701e+05, -2.088692788251e+05, -2.088792505461e+05, -2.088884320261e+05, -2.088968323941e+05, -2.089044606221e+05, -2.089113255251e+05, -2.089174357681e+05, -2.089227998671e+05, -2.089274261941e+05, -2.089313229791e+05, -2.089344983141e+05, -2.089369601571e+05, -2.089387163341e+05, -2.089397745401e+05, -2.089401423461e+05, -2.089398271981e+05, -2.089388364241e+05, -2.089371772301e+05, -2.089348567101e+05, -2.089318818421e+05, -2.089282594971e+05, -2.089239964341e+05, -2.089190993081e+05, -2.089135746711e+05, -2.089074289721e+05, -2.089006685601e+05, -2.088932996901e+05, -2.088853285171e+05, -2.088767611051e+05, -2.088676034281e+05, -2.088578613671e+05, -2.088475407181e+05, -2.088366471891e+05, -2.088251864041e+05, -2.088131639061e+05, -2.088005851541e+05, -2.087874555301e+05, -2.087737803371e+05, -2.087595648021e+05, -2.087448140761e+05, -2.087295332381e+05, -2.087137272941e+05, -2.086974011791e+05, -2.086805597601e+05, -2.086632078351e+05, -2.086453501341e+05, -2.086269913241e+05, -2.086081360071e+05, -2.085887887191e+05, -2.085689539391e+05, -2.085486360801e+05, -2.085278395001e+05, -2.085065684951e+05, -2.084848273041e+05, -2.084626201111e+05, -2.084399510441e+05, -2.084168241741e+05, -2.083932435211e+05, -2.083692130531e+05, -2.083447366831e+05, -2.083198182771e+05, -2.082944616481e+05, -2.082686705621e+05, -2.082424487361e+05, -2.082157998391e+05, -2.081887274941e+05, -2.081612352781e+05, -2.081333267251e+05, -2.081050053211e+05, -2.080762745111e+05, -2.080471376971e+05, -2.080175982391e+05, -2.079876594551e+05, -2.079573246231e+05, -2.079265969791e+05, -2.078954797231e+05, -2.078639760141e+05, -2.078320889731e+05, -2.077998216841e+05, -2.077671771951e+05, -2.077341585151e+05, -2.077007686211e+05, -2.076670104511e+05, -2.076328869121e+05, -2.075984008751e+05, -2.075635551781e+05, -2.075283526241e+05, -2.074927959881e+05, -2.074568880091e+05, -2.074206313961e+05, -2.073840288271e+05, -2.073470829491e+05, -2.073097963811e+05, -2.072721717091e+05, -2.072342114931e+05, -2.071959182631e+05, -2.071572945191e+05, -2.071183427371e+05, -2.070790653621e+05, -2.070394648141e+05, -2.069995434861e+05, -2.069593037441e+05, -2.069187479291e+05, -2.068778783561e+05, -2.068366973151e+05, -2.067952070731e+05, -2.067534098691e+05, -2.067113079211e+05, -2.066689034221e+05, -2.066261985421e+05, -2.065831954281e+05, -2.065398962051e+05, -2.064963029741e+05, -2.064524178151e+05, -2.064082427861e+05, -2.063637799251e+05, -2.063190312461e+05, -2.062739987441e+05, -2.062286843941e+05, -2.061830901491e+05, -2.061372179431e+05, -2.060910696911e+05, -2.060446472871e+05, -2.059979526071e+05, -2.059509875071e+05, -2.059037538261e+05, -2.058562533821e+05, -2.058084879781e+05, -2.057604593961e+05, -2.057121694031e+05, -2.056636197471e+05, -2.056148121581e+05, -2.055657483521e+05, -2.055164300241e+05, -2.054668588561e+05, -2.054170365131e+05, -2.053669646421e+05, -2.053166448751e+05, -2.052660788311e+05, -2.052152681091e+05, -2.051642142951e+05, -2.051129189611e+05, -2.050613836611e+05, -2.050096099371e+05, -2.049575993161e+05, -2.049053533091e+05, -2.048528734151e+05, -2.048001611171e+05, -2.047472178851e+05, -2.046940451771e+05, -2.046406444341e+05, -2.045870170871e+05, -2.045331645511e+05, -2.044790882311e+05, -2.044247895171e+05, -2.043702697871e+05, -2.043155304071e+05, -2.042605727291e+05, -2.042053980941e+05, -2.041500078311e+05, -2.040944032571e+05, -2.040385856781e+05, -2.039825563851e+05, -2.039263166611e+05, -2.038698677781e+05, -2.038132109931e+05, -2.037563475551e+05, -2.036992787021e+05, -2.036420056591e+05, -2.035845296431e+05, -2.035268518571e+05, -2.034689734981e+05, -2.034108957491e+05, -2.033526197841e+05, -2.032941467671e+05, -2.032354778521e+05, -2.031766141821e+05, -2.031175568931e+05, -2.030583071081e+05, -2.029988659431e+05, -2.029392345031e+05, -2.028794138851e+05, -2.028194051761e+05, -2.027592094531e+05, -2.026988277851e+05, -2.026382612331e+05, -2.025775108481e+05, -2.025165776711e+05, -2.024554627371e+05, -2.023941670721e+05, -2.023326916911e+05, -2.022710376041e+05, -2.022092058101e+05, -2.021471973021e+05, -2.020850130641e+05, -2.020226540721e+05, -2.019601212941e+05, -2.018974156891e+05, -2.018345382121e+05, -2.017714898071e+05, -2.017082714101e+05, -2.016448839531e+05, -2.015813283571e+05, -2.015176055371e+05, -2.014537164021e+05, -2.013896618531e+05, -2.013254427821e+05, -2.012610600771e+05, -2.011965146161e+05, -2.011318072741e+05, -2.010669389161e+05, -2.010019104011e+05, -2.009367225811e+05, -2.008713763041e+05, -2.008058724071e+05, -2.007402117251e+05, -2.006743950841e+05, -2.006084233041e+05, -2.005422971981e+05, -2.004760175761e+05, -2.004095852391e+05, -2.003430009811e+05, -2.002762655941e+05, -2.002093798591e+05, -2.001423445571e+05, -2.000751604571e+05, -2.000078283261e+05, -1.999403489251e+05, -1.998727230091e+05, -1.998049513261e+05, -1.997370346201e+05, -1.996689736301e+05, -1.996007690881e+05, -1.995324217201e+05, -1.994639322501e+05, -1.993953013941e+05, -1.993265298631e+05, -1.992576183641e+05, -1.991885675971e+05, -1.991193782601e+05, -1.990500510411e+05, -1.989805866291e+05, -1.989109857031e+05, -1.988412489411e+05, -1.987713770121e+05, -1.987013705851e+05, -1.986312303211e+05, -1.985609568761e+05, -1.984905509041e+05, -1.984200130521e+05, -1.983493439631e+05, -1.982785442771e+05, -1.982076146271e+05, -1.981365556441e+05, -1.980653679521e+05, -1.979940521731e+05, -1.979226089231e+05, -1.978510388151e+05, -1.977793424571e+05, -1.977075204541e+05, -1.976355734041e+05, -1.975635019031e+05, -1.974913065441e+05, -1.974189879141e+05, -1.973465465961e+05, -1.972739831701e+05, -1.972012982111e+05, -1.971284922921e+05, -1.970555659801e+05, -1.969825198391e+05, -1.969093544291e+05, -1.968360703071e+05, -1.967626680251e+05, -1.966891481331e+05, -1.966155111761e+05, -1.965417576951e+05, -1.964678882291e+05, -1.963939033121e+05, -1.963198034751e+05, -1.962455892461e+05, -1.961712611481e+05, -1.960968197031e+05, -1.960222654271e+05, -1.959475988341e+05},
    {3.855610282687e+04, 3.691812412187e+04, 3.525791717987e+04, 3.357465508087e+04, 3.186745964487e+04, 3.013539702387e+04, 2.837747279787e+04, 2.659262650587e+04, 2.477972554387e+04, 2.293755831087e+04, 2.106482651587e+04, 1.916013650587e+04, 1.722198945087e+04, 1.524877022187e+04, 1.323873471887e+04, 1.118999540487e+04, 9.100504706871e+03, 6.968035916871e+03, 4.790161104871e+03, 2.564225478871e+03, 2.873174728705e+02, -2.043766307129e+03, -4.432562379130e+03, -6.882989291129e+03, -9.399405865129e+03, -1.198668097013e+04, -1.465027761713e+04, -1.739635498513e+04, -2.023189301013e+04, -2.316484563013e+04, -2.620433061013e+04, -2.936086647913e+04, -3.264667045713e+04, -3.607603591613e+04, -3.966581390913e+04, -4.343603111113e+04, -4.741068559013e+04, -5.161877036513e+04, -5.609557542513e+04, -6.088429079913e+04, -6.603781716313e+04, -7.162033137213e+04, -7.770719178713e+04, -8.437960838913e+04, -9.170728256613e+04, -9.971192877313e+04, -1.083082580231e+05, -1.172039243481e+05, -1.258162185831e+05, -1.335431468771e+05, -1.401686720421e+05, -1.457112757391e+05, -1.503233121421e+05, -1.541935818721e+05, -1.574889498491e+05, -1.603383181971e+05, -1.628367462011e+05, -1.650540380791e+05, -1.670421289301e+05, -1.688403710961e+05, -1.704791404761e+05, -1.719822924991e+05, -1.733688573941e+05, -1.746542309191e+05, -1.758510257971e+05, -1.769696912311e+05, -1.780189712371e+05, -1.790062491061e+05, -1.799378100891e+05, -1.808190444291e+05, -1.816546061961e+05, -1.824485389391e+05, -1.832043760631e+05, -1.839252217561e+05, -1.846138167571e+05, -1.852725922151e+05, -1.859037140991e+05, -1.865091200431e+05, -1.870905501041e+05, -1.876495725871e+05, -1.881876058241e+05, -1.887059366571e+05, -1.892057361771e+05, -1.896880731951e+05, -1.901539258281e+05, -1.906041914891e+05, -1.910396955511e+05, -1.914611988781e+05, -1.918694043971e+05, -1.922649628611e+05, -1.926484779041e+05, -1.930205105041e+05, -1.933815829271e+05, -1.937321822231e+05, -1.940727633381e+05, -1.944037518871e+05, -1.947255466321e+05, -1.950385217081e+05, -1.953430286111e+05, -1.956393980051e+05, -1.959279413351e+05, -1.962089523001e+05, -1.964827081761e+05, -1.967494710281e+05, -1.970094888011e+05, -1.972629963241e+05, -1.975102162221e+05, -1.977513597471e+05, -1.979866275481e+05, -1.982162103641e+05, -1.984402896751e+05, -1.986590382911e+05, -1.988726208951e+05, -1.990811945511e+05, -1.992849091661e+05, -1.994839079191e+05, -1.996783276631e+05, -1.998682992881e+05, -2.000539480701e+05, -2.002353939831e+05, -2.004127519991e+05, -2.005861323621e+05, -2.007556408471e+05, -2.009213789951e+05, -2.010834443401e+05, -2.012419306201e+05, -2.013969279681e+05, -2.015485230981e+05, -2.016967994761e+05, -2.018418374841e+05, -2.019837145651e+05, -2.021225053701e+05, -2.022582818921e+05, -2.023911135871e+05, -2.025210674961e+05, -2.026482083581e+05, -2.027725987101e+05, -2.028942989901e+05, -2.030133676291e+05, -2.031298611411e+05, -2.032438342071e+05, -2.033553397501e+05, -2.034644290151e+05, -2.035711516381e+05, -2.036755557111e+05, -2.037776878511e+05, -2.038775932531e+05, -2.039753157571e+05, -2.040708978931e+05, -2.041643809381e+05, -2.042558049671e+05, -2.043452088941e+05, -2.044326305211e+05, -2.045181065801e+05, -2.046016727721e+05, -2.046833638061e+05, -2.047632134381e+05, -2.048412545021e+05, -2.049175189471e+05, -2.049920378671e+05, -2.050648415331e+05, -2.051359594191e+05, -2.052054202361e+05, -2.052732519511e+05, -2.053394818201e+05, -2.054041364051e+05, -2.054672416041e+05, -2.055288226711e+05, -2.055889042331e+05, -2.056475103201e+05, -2.057046643761e+05, -2.057603892811e+05, -2.058147073721e+05, -2.058676404541e+05, -2.059192098251e+05, -2.059694362831e+05, -2.060183401501e+05, -2.060659412811e+05, -2.061122590801e+05, -2.061573125151e+05, -2.062011201281e+05, -2.062437000521e+05, -2.062850700201e+05, -2.063252473781e+05, -2.063642490961e+05, -2.064020917781e+05, -2.064387916751e+05, -2.064743646941e+05, -2.065088264061e+05, -2.065421920581e+05, -2.065744765801e+05, -2.066056945961e+05, -2.066358604311e+05, -2.066649881181e+05, -2.066930914101e+05, -2.067201837831e+05, -2.067462784461e+05, -2.067713883481e+05, -2.067955261831e+05, -2.068187044001e+05, -2.068409352051e+05, -2.068622305741e+05, -2.068826022491e+05, -2.069020617551e+05, -2.069206203971e+05, -2.069382892711e+05, -2.069550792641e+05, -2.069710010661e+05, -2.069860651671e+05, -2.070002818711e+05, -2.070136612901e+05, -2.070262133581e+05, -2.070379478321e+05, -2.070488742921e+05, -2.070590021531e+05, -2.070683406621e+05, -2.070768989071e+05, -2.070846858181e+05, -2.070917101701e+05, -2.070979805891e+05, -2.071035055541e+05, -2.071082934011e+05, -2.071123523261e+05, -2.071156903861e+05, -2.071183155071e+05, -2.071202354821e+05, -2.071214579771e+05, -2.071219905331e+05, -2.071218405671e+05, -2.071210153791e+05, -2.071195221491e+05, -2.071173679431e+05, -2.071145597171e+05, -2.071111043151e+05, -2.071070084731e+05, -2.071022788251e+05, -2.070969218971e+05, -2.070909441201e+05, -2.070843518211e+05, -2.070771512341e+05, -2.070693484961e+05, -2.070609496511e+05, -2.070519606561e+05, -2.070423873731e+05, -2.070322355811e+05, -2.070215109721e+05, -2.070102191541e+05, -2.069983656541e+05, -2.069859559161e+05, -2.069729953071e+05, -2.069594891161e+05, -2.069454425561e+05, -2.069308607661e+05, -2.069157488111e+05, -2.069001116841e+05, -2.068839543101e+05, -2.068672815421e+05, -2.068500981671e+05, -2.068324089061e+05, -2.068142184141e+05, -2.067955312821e+05, -2.067763520371e+05, -2.067566851481e+05, -2.067365350191e+05, -2.067159059981e+05, -2.066948023731e+05, -2.066732283751e+05, -2.066511881791e+05, -2.066286859041e+05, -2.066057256151e+05, -2.065823113251e+05, -2.065584469931e+05, -2.065341365271e+05, -2.065093837831e+05, -2.064841925711e+05, -2.064585666491e+05, -2.064325097271e+05, -2.064060254701e+05, -2.063791174951e+05, -2.063517893721e+05, -2.063240446311e+05, -2.062958867521e+05, -2.062673191761e+05, -2.062383453001e+05, -2.062089684781e+05, -2.061791920251e+05, -2.061490192131e+05, -2.061184532761e+05, -2.060874974091e+05, -2.060561547661e+05, -2.060244284651e+05, -2.059923215881e+05, -2.059598371761e+05, -2.059269782381e+05, -2.058937477461e+05, -2.058601486361e+05, -2.058261838111e+05, -2.057918561391e+05, -2.057571684551e+05, -2.057221235611e+05, -2.056867242271e+05, -2.056509731921e+05, -2.056148731611e+05, -2.055784268101e+05, -2.055416367851e+05, -2.055045057001e+05, -2.054670361431e+05, -2.054292306681e+05, -2.053910918061e+05, -2.053526220551e+05, -2.053138238871e+05, -2.052746997481e+05, -2.052352520561e+05, -2.051954832011e+05, -2.051553955491e+05, -2.051149914401e+05, -2.050742731871e+05, -2.050332430791e+05, -2.049919033801e+05, -2.049502563311e+05, -2.049083041481e+05, -2.048660490231e+05, -2.048234931241e+05, -2.047806385991e+05, -2.047374875701e+05, -2.046940421401e+05, -2.046503043861e+05, -2.046062763671e+05, -2.045619601181e+05, -2.045173576561e+05, -2.044724709731e+05, -2.044273020441e+05, -2.043818528221e+05, -2.043361252411e+05, -2.042901212141e+05, -2.042438426371e+05, -2.041972913841e+05, -2.041504693111e+05, -2.041033782581e+05, -2.040560200421e+05, -2.040083964661e+05, -2.039605093121e+05, -2.039123603471e+05, -2.038639513181e+05, -2.038152839581e+05, -2.037663599791e+05, -2.037171810791e+05, -2.036677489401e+05, -2.036180652251e+05, -2.035681315831e+05, -2.035179496471e+05, -2.034675210341e+05, -2.034168473441e+05, -2.033659301641e+05, -2.033147710661e+05, -2.032633716041e+05, -2.032117333211e+05, -2.031598577431e+05, -2.031077463831e+05, -2.030554007391e+05, -2.030028222951e+05, -2.029500125221e+05, -2.028969728781e+05, -2.028437048041e+05, -2.027902097321e+05, -2.027364890791e+05, -2.026825442471e+05, -2.026283766301e+05, -2.025739876041e+05, -2.025193785371e+05, -2.024645507821e+05, -2.024095056791e+05, -2.023542445591e+05, -2.022987687401e+05, -2.022430795261e+05, -2.021871782111e+05, -2.021310660791e+05, -2.020747444011e+05, -2.020182144351e+05, -2.019614774331e+05, -2.019045346301e+05, -2.018473872551e+05, -2.017900365231e+05, -2.017324836401e+05, -2.016747298031e+05, -2.016167761951e+05, -2.015586239911e+05, -2.015002743571e+05, -2.014417284471e+05, -2.013829874051e+05, -2.013240523661e+05, -2.012649244571e+05, -2.012056047931e+05, -2.011460944791e+05, -2.010863946151e+05, -2.010265062861e+05, -2.009664305721e+05, -2.009061685441e+05, -2.008457212611e+05, -2.007850897751e+05, -2.007242751301e+05, -2.006632783611e+05, -2.006021004941e+05, -2.005407425461e+05, -2.004792055271e+05, -2.004174904381e+05, -2.003555982721e+05, -2.002935300131e+05, -2.002312866401e+05, -2.001688691201e+05, -2.001062784151e+05, -2.000435154791e+05, -1.999805812571e+05, -1.999174766881e+05, -1.998542027021e+05, -1.997907602231e+05, -1.997271501661e+05, -1.996633734421e+05, -1.995994309501e+05, -1.995353235871e+05, -1.994710522391e+05, -1.994066177871e+05, -1.993420211051e+05, -1.992772630601e+05, -1.992123445111e+05, -1.991472663131e+05, -1.990820293121e+05, -1.990166343491e+05, -1.989510822571e+05, -1.988853738651e+05, -1.988195099931e+05, -1.987534914561e+05, -1.986873190641e+05, -1.986209936181e+05, -1.985545159161e+05, -1.984878867471e+05, -1.984211068971e+05, -1.983541771441e+05, -1.982870982611e+05, -1.982198710151e+05, -1.981524961671e+05, -1.980849744741e+05, -1.980173066851e+05, -1.979494935441e+05, -1.978815357921e+05, -1.978134341601e+05, -1.977451893791e+05, -1.976768021701e+05, -1.976082732511e+05, -1.975396033341e+05, -1.974707931281e+05, -1.974018433331e+05, -1.973327546461e+05, -1.972635277611e+05, -1.971941633631e+05, -1.971246621351e+05, -1.970550247531e+05, -1.969852518911e+05, -1.969153442151e+05, -1.968453023891e+05, -1.967751270701e+05, -1.967048189111e+05, -1.966343785631e+05, -1.965638066681e+05, -1.964931038661e+05, -1.964222707931e+05, -1.963513080801e+05, -1.962802163521e+05, -1.962089962311e+05, -1.961376483361e+05, -1.960661732791e+05, -1.959945716701e+05, -1.959228441141e+05, -1.958509912101e+05, -1.957790135571e+05, -1.957069117461e+05, -1.956346863661e+05, -1.955623380021e+05, -1.954898672331e+05, -1.954172746361e+05, -1.953445607851e+05, -1.952717262471e+05, -1.951987715881e+05, -1.951256973681e+05, -1.950525041461e+05, -1.949791924741e+05, -1.949057629021e+05, -1.948322159781e+05, -1.947585522421e+05, -1.946847722351e+05, -1.946108764921e+05, -1.945368655441e+05, -1.944627399211e+05, -1.943885001461e+05, -1.943141467411e+05},
    {3.944452055287e+04, 3.781794643787e+04, 3.616960557587e+04, 3.449870199087e+04, 3.280439128087e+04, 3.108577654087e+04, 2.934190383787e+04, 2.757175717987e+04, 2.577425290587e+04, 2.394823341487e+04, 2.209246013987e+04, 2.020560564387e+04, 1.828624472187e+04, 1.633284432987e+04, 1.434375216487e+04, 1.231718366687e+04, 1.025120717387e+04, 8.143726904871e+03, 5.992463365871e+03, 3.794930726870e+03, 1.548410552871e+03, -7.500788031295e+02, -3.103818049130e+03, -5.516429128130e+03, -7.991925103129e+03, -1.053476945813e+04, -1.314994699513e+04, -1.584304905713e+04, -1.862037660313e+04, -2.148906559313e+04, -2.445724046813e+04, -2.753420317113e+04, -3.073066731913e+04, -3.405904996713e+04, -3.753383685613e+04, -4.117204117813e+04, -4.499378005313e+04, -4.902299534413e+04, -5.328834105213e+04, -5.782423508513e+04, -6.267199530913e+04, -6.788076876113e+04, -7.350744529313e+04, -7.961363608613e+04, -8.625601654813e+04, -9.346540123113e+04, -1.012130599261e+05, -1.093630134921e+05, -1.176045995651e+05, -1.254535326701e+05, -1.324980339691e+05, -1.386062802921e+05, -1.438006369641e+05, -1.481970073761e+05, -1.519395203961e+05, -1.551609287781e+05, -1.579687446601e+05, -1.604455002801e+05, -1.626536428871e+05, -1.646406428871e+05, -1.664430408771e+05, -1.680893857001e+05, -1.696023165171e+05, -1.710000392471e+05, -1.722973840381e+05, -1.735065727881e+05, -1.746377842141e+05, -1.756995759371e+05, -1.766992042231e+05, -1.776428695081e+05, -1.785359073091e+05, -1.793829384331e+05, -1.801879884081e+05, -1.809545833811e+05, -1.816858277921e+05, -1.823844677841e+05, -1.830529433621e+05, -1.836934315641e+05, -1.843078824241e+05, -1.848980491041e+05, -1.854655132651e+05, -1.860117065441e+05, -1.865379288171e+05, -1.870453638091e+05, -1.875350924681e+05, -1.880081045051e+05, -1.884653083611e+05, -1.889075398621e+05, -1.893355697581e+05, -1.897501103141e+05, -1.901518210891e+05, -1.905413140191e+05, -1.909191579071e+05, -1.912858823911e+05, -1.916419814701e+05, -1.919879166411e+05, -1.923241196951e+05, -1.926509952251e+05, -1.929689228651e+05, -1.932782593141e+05, -1.935793401451e+05, -1.938724814551e+05, -1.941579813401e+05, -1.944361212461e+05, -1.947071671871e+05, -1.949713708571e+05, -1.952289706441e+05, -1.954801925501e+05, -1.957252510441e+05, -1.959643498271e+05, -1.961976825521e+05, -1.964254334691e+05, -1.966477780281e+05, -1.968648834351e+05, -1.970769091571e+05, -1.972840073981e+05, -1.974863235331e+05, -1.976839965111e+05, -1.978771592301e+05, -1.980659388861e+05, -1.982504572941e+05, -1.984308311881e+05, -1.986071725031e+05, -1.987795886331e+05, -1.989481826751e+05, -1.991130536561e+05, -1.992742967481e+05, -1.994320034621e+05, -1.995862618371e+05, -1.997371566141e+05, -1.998847693981e+05, -2.000291788141e+05, -2.001704606501e+05, -2.003086879911e+05, -2.004439313501e+05, -2.005762587851e+05, -2.007057360131e+05, -2.008324265191e+05, -2.009563916531e+05, -2.010776907291e+05, -2.011963811121e+05, -2.013125183031e+05, -2.014261560221e+05, -2.015373462821e+05, -2.016461394601e+05, -2.017525843691e+05, -2.018567283191e+05, -2.019586171811e+05, -2.020582954421e+05, -2.021558062661e+05, -2.022511915401e+05, -2.023444919311e+05, -2.024357469271e+05, -2.025249948871e+05, -2.026122730821e+05, -2.026976177351e+05, -2.027810640651e+05, -2.028626463181e+05, -2.029423978081e+05, -2.030203509481e+05, -2.030965372831e+05, -2.031709875201e+05, -2.032437315621e+05, -2.033147985301e+05, -2.033842167961e+05, -2.034520140031e+05, -2.035182170971e+05, -2.035828523441e+05, -2.036459453561e+05, -2.037075211131e+05, -2.037676039821e+05, -2.038262177401e+05, -2.038833855911e+05, -2.039391301831e+05, -2.039934736311e+05, -2.040464375281e+05, -2.040980429651e+05, -2.041483105451e+05, -2.041972603991e+05, -2.042449122021e+05, -2.042912851821e+05, -2.043363981381e+05, -2.043802694531e+05, -2.044229171031e+05, -2.044643586701e+05, -2.045046113581e+05, -2.045436919961e+05, -2.045816170571e+05, -2.046184026611e+05, -2.046540645921e+05, -2.046886183021e+05, -2.047220789221e+05, -2.047544612721e+05, -2.047857798681e+05, -2.048160489311e+05, -2.048452823981e+05, -2.048734939221e+05, -2.049006968891e+05, -2.049269044181e+05, -2.049521293711e+05, -2.049763843591e+05, -2.049996817521e+05, -2.050220336771e+05, -2.050434520351e+05, -2.050639484971e+05, -2.050835345171e+05, -2.051022213331e+05, -2.051200199751e+05, -2.051369412671e+05, -2.051529958381e+05, -2.051681941211e+05, -2.051825463591e+05, -2.051960626131e+05, -2.052087527621e+05, -2.052206265111e+05, -2.052316933911e+05, -2.052419627681e+05, -2.052514438431e+05, -2.052601456581e+05, -2.052680771001e+05, -2.052752469021e+05, -2.052816636481e+05, -2.052873357791e+05, -2.052922715911e+05, -2.052964792421e+05, -2.052999667561e+05, -2.053027420231e+05, -2.053048128011e+05, -2.053061867251e+05, -2.053068713021e+05, -2.053068739221e+05, -2.053062018511e+05, -2.053048622441e+05, -2.053028621371e+05, -2.053002084591e+05, -2.052969080281e+05, -2.052929675561e+05, -2.052883936501e+05, -2.052831928141e+05, -2.052773714551e+05, -2.052709358781e+05, -2.052638922951e+05, -2.052562468221e+05, -2.052480054831e+05, -2.052391742141e+05, -2.052297588591e+05, -2.052197651781e+05, -2.052091988451e+05, -2.051980654511e+05, -2.051863705041e+05, -2.051741194331e+05, -2.051613175901e+05, -2.051479702471e+05, -2.051340826031e+05, -2.051196597811e+05, -2.051047068331e+05, -2.050892287381e+05, -2.050732304081e+05, -2.050567166831e+05, -2.050396923371e+05, -2.050221620801e+05, -2.050041305531e+05, -2.049856023381e+05, -2.049665819511e+05, -2.049470738471e+05, -2.049270824241e+05, -2.049066120171e+05, -2.048856669041e+05, -2.048642513091e+05, -2.048423693951e+05, -2.048200252741e+05, -2.047972230021e+05, -2.047739665831e+05, -2.047502599681e+05, -2.047261070571e+05, -2.047015117001e+05, -2.046764776971e+05, -2.046510088001e+05, -2.046251087131e+05, -2.045987810931e+05, -2.045720295501e+05, -2.045448576501e+05, -2.045172689141e+05, -2.044892668191e+05, -2.044608547971e+05, -2.044320362421e+05, -2.044028145011e+05, -2.043731928841e+05, -2.043431746601e+05, -2.043127630561e+05, -2.042819612611e+05, -2.042507724281e+05, -2.042191996681e+05, -2.041872460591e+05, -2.041549146391e+05, -2.041222084131e+05, -2.040891303471e+05, -2.040556833751e+05, -2.040218703951e+05, -2.039876942731e+05, -2.039531578401e+05, -2.039182638961e+05, -2.038830152061e+05, -2.038474145051e+05, -2.038114644981e+05, -2.037751678571e+05, -2.037385272241e+05, -2.037015452121e+05, -2.036642244061e+05, -2.036265673581e+05, -2.035885765941e+05, -2.035502546131e+05, -2.035116038841e+05, -2.034726268501e+05, -2.034333259271e+05, -2.033937035051e+05, -2.033537619461e+05, -2.033135035881e+05, -2.032729307441e+05, -2.032320457001e+05, -2.031908507211e+05, -2.031493480431e+05, -2.031075398811e+05, -2.030654284271e+05, -2.030230158491e+05, -2.029803042911e+05, -2.029372958751e+05, -2.028939927021e+05, -2.028503968501e+05, -2.028065103751e+05, -2.027623353121e+05, -2.027178736761e+05, -2.026731274601e+05, -2.026280986361e+05, -2.025827891591e+05, -2.025372009591e+05, -2.024913359511e+05, -2.024451960291e+05, -2.023987830661e+05, -2.023520989201e+05, -2.023051454271e+05, -2.022579244061e+05, -2.022104376581e+05, -2.021626869671e+05, -2.021146740971e+05, -2.020664007961e+05, -2.020178687951e+05, -2.019690798091e+05, -2.019200355341e+05, -2.018707376501e+05, -2.018211878231e+05, -2.017713877011e+05, -2.017213389161e+05, -2.016710430851e+05, -2.016205018091e+05, -2.015697166741e+05, -2.015186892521e+05, -2.014674210981e+05, -2.014159137541e+05, -2.013641687471e+05, -2.013121875891e+05, -2.012599717791e+05, -2.012075228021e+05, -2.011548421281e+05, -2.011019312141e+05, -2.010487915041e+05, -2.009954244281e+05, -2.009418314041e+05, -2.008880138361e+05, -2.008339731161e+05, -2.007797106211e+05, -2.007252277191e+05, -2.006705257641e+05, -2.006156060971e+05, -2.005604700481e+05, -2.005051189351e+05, -2.004495540641e+05, -2.003937767301e+05, -2.003377882151e+05, -2.002815897921e+05, -2.002251827201e+05, -2.001685682491e+05, -2.001117476181e+05, -2.000547220541e+05, -1.999974927741e+05, -1.999400609841e+05, -1.998824278801e+05, -1.998245946481e+05, -1.997665624631e+05, -1.997083324901e+05, -1.996499058841e+05, -1.995912837921e+05, -1.995324673481e+05, -1.994734576791e+05, -1.994142559011e+05, -1.993548631211e+05, -1.992952804371e+05, -1.992355089371e+05, -1.991755497021e+05, -1.991154038011e+05, -1.990550722951e+05, -1.989945562381e+05, -1.989338566741e+05, -1.988729746371e+05, -1.988119111561e+05, -1.987506672471e+05, -1.986892439221e+05, -1.986276421821e+05, -1.985658630211e+05, -1.985039074251e+05, -1.984417763711e+05, -1.983794708281e+05, -1.983169917601e+05, -1.982543401201e+05, -1.981915168551e+05, -1.981285229041e+05, -1.980653591981e+05, -1.980020266611e+05, -1.979385262121e+05, -1.978748587581e+05, -1.978110252031e+05, -1.977470264411e+05, -1.976828633621e+05, -1.976185368471e+05, -1.975540477691e+05, -1.974893969981e+05, -1.974245853931e+05, -1.973596138091e+05, -1.972944830951e+05, -1.972291940901e+05, -1.971637476311e+05, -1.970981445441e+05, -1.970323856531e+05, -1.969664717731e+05, -1.969004037141e+05, -1.968341822791e+05, -1.967678082651e+05, -1.967012824641e+05, -1.966346056621e+05, -1.965677786371e+05, -1.965008021641e+05, -1.964336770101e+05, -1.963664039381e+05, -1.962989837041e+05, -1.962314170591e+05, -1.961637047481e+05, -1.960958475121e+05, -1.960278460851e+05, -1.959597011951e+05, -1.958914135671e+05, -1.958229839201e+05, -1.957544129651e+05, -1.956857014111e+05, -1.956168499621e+05, -1.955478593151e+05, -1.954787301631e+05, -1.954094631931e+05, -1.953400590881e+05, -1.952705185271e+05, -1.952008421831e+05, -1.951310307241e+05, -1.950610848131e+05, -1.949910051091e+05, -1.949207922671e+05, -1.948504469361e+05, -1.947799697611e+05, -1.947093613841e+05, -1.946386224381e+05, -1.945677535581e+05, -1.944967553681e+05, -1.944256284931e+05, -1.943543735511e+05, -1.942829911551e+05, -1.942114819161e+05, -1.941398464401e+05, -1.940680853281e+05, -1.939961991781e+05, -1.939241885821e+05, -1.938520541311e+05, -1.937797964091e+05, -1.937074159981e+05, -1.936349134751e+05, -1.935622894131e+05, -1.934895443831e+05, -1.934166789491e+05, -1.933436936751e+05, -1.932705891171e+05, -1.931973658301e+05, -1.931240243651e+05, -1.930505652701e+05, -1.929769890871e+05, -1.929032963561e+05, -1.928294876131e+05, -1.927555633921e+05, -1.926815242211e+05},
    {4.033391555487e+04, 3.871861538387e+04, 3.708199921487e+04, 3.542330080187e+04, 3.374170814987e+04, 3.203635974587e+04, 3.030634038587e+04, 2.855067654287e+04, 2.676833121787e+04, 2.495819819487e+04, 2.311909562587e+04, 2.124975883887e+04, 1.934883225487e+04, 1.741486027887e+04, 1.544627700487e+04, 1.344139454087e+04, 1.139838972187e+04, 9.315288946871e+03, 7.189950799870e+03, 5.020046068871e+03, 2.803034673871e+03, 5.361389148705e+02, -1.783687661130e+03, -4.159796676130e+03, -6.595887687129e+03, -9.096058668129e+03, -1.166486587513e+04, -1.430739519113e+04, -1.702934760013e+04, -1.983714208513e+04, -2.273804015513e+04, -2.574029726813e+04, -2.885334780113e+04, -3.208803191413e+04, -3.545687454413e+04, -3.897442887713e+04, -4.265769813813e+04, -4.652664924313e+04, -5.060482600413e+04, -5.492004989513e+04, -5.950514364313e+04, -6.439848424213e+04, -6.964390046413e+04, -7.528883355913e+04, -8.137869807413e+04, -8.794455370813e+04, -9.498242218813e+04, -1.024257808221e+05, -1.101100547421e+05, -1.177377662141e+05, -1.249251035191e+05, -1.313921214741e+05, -1.370558172371e+05, -1.419395686401e+05, -1.461333920961e+05, -1.497490291231e+05, -1.528926693781e+05, -1.556537841891e+05, -1.581036164851e+05, -1.602976770131e+05, -1.622791049741e+05, -1.640816573341e+05, -1.657320406231e+05, -1.672516435471e+05, -1.686578059841e+05, -1.699647490581e+05, -1.711842617981e+05, -1.723262133571e+05, -1.733989395551e+05, -1.744095381551e+05, -1.753640971531e+05, -1.762678733651e+05, -1.771254336711e+05, -1.779407678861e+05, -1.787173798121e+05, -1.794583613051e+05, -1.801664530241e+05, -1.808440945871e+05, -1.814934662641e+05, -1.821165238401e+05, -1.827150279221e+05, -1.832905687131e+05, -1.838445870461e+05, -1.843783923201e+05, -1.848931778721e+05, -1.853900341981e+05, -1.858699603631e+05, -1.863338738951e+05, -1.867826193901e+05, -1.872169760171e+05, -1.876376640921e+05, -1.880453508531e+05, -1.884406555481e+05, -1.888241539251e+05, -1.891963822181e+05, -1.895578406801e+05, -1.899089967421e+05, -1.902502878201e+05, -1.905821238381e+05, -1.909048894861e+05, -1.912189462551e+05, -1.915246342701e+05, -1.918222739401e+05, -1.921121674611e+05, -1.923946001691e+05, -1.926698417761e+05, -1.929381474941e+05, -1.931997590541e+05, -1.934549056451e+05, -1.937038047651e+05, -1.939466630071e+05, -1.941836767771e+05, -1.944150329551e+05, -1.946409095001e+05, -1.948614760121e+05, -1.950768942491e+05, -1.952873186031e+05, -1.954928965431e+05, -1.956937690221e+05, -1.958900708571e+05, -1.960819310821e+05, -1.962694732741e+05, -1.964528158581e+05, -1.966320723891e+05, -1.968073518161e+05, -1.969787587321e+05, -1.971463936001e+05, -1.973103529711e+05, -1.974707296821e+05, -1.976276130491e+05, -1.977810890411e+05, -1.979312404461e+05, -1.980781470271e+05, -1.982218856691e+05, -1.983625305151e+05, -1.985001530971e+05, -1.986348224571e+05, -1.987666052601e+05, -1.988955659071e+05, -1.990217666331e+05, -1.991452676051e+05, -1.992661270121e+05, -1.993844011551e+05, -1.995001445241e+05, -1.996134098751e+05, -1.997242483101e+05, -1.998327093371e+05, -1.999388409421e+05, -2.000426896491e+05, -2.001443005811e+05, -2.002437175131e+05, -2.003409829281e+05, -2.004361380671e+05, -2.005292229791e+05, -2.006202765631e+05, -2.007093366151e+05, -2.007964398711e+05, -2.008816220411e+05, -2.009649178531e+05, -2.010463610871e+05, -2.011259846081e+05, -2.012038204021e+05, -2.012798996031e+05, -2.013542525291e+05, -2.014269087061e+05, -2.014978968961e+05, -2.015672451271e+05, -2.016349807131e+05, -2.017011302821e+05, -2.017657197991e+05, -2.018287745851e+05, -2.018903193401e+05, -2.019503781651e+05, -2.020089745791e+05, -2.020661315401e+05, -2.021218714601e+05, -2.021762162241e+05, -2.022291872081e+05, -2.022808052931e+05, -2.023310908801e+05, -2.023800639061e+05, -2.024277438581e+05, -2.024741497851e+05, -2.025193003141e+05, -2.025632136591e+05, -2.026059076371e+05, -2.026473996761e+05, -2.026877068311e+05, -2.027268457891e+05, -2.027648328821e+05, -2.028016841011e+05, -2.028374150991e+05, -2.028720412041e+05, -2.029055774301e+05, -2.029380384801e+05, -2.029694387601e+05, -2.029997923861e+05, -2.030291131871e+05, -2.030574147211e+05, -2.030847102751e+05, -2.031110128761e+05, -2.031363352961e+05, -2.031606900601e+05, -2.031840894521e+05, -2.032065455221e+05, -2.032280700901e+05, -2.032486747521e+05, -2.032683708891e+05, -2.032871696691e+05, -2.033050820521e+05, -2.033221187991e+05, -2.033382904721e+05, -2.033536074431e+05, -2.033680798971e+05, -2.033817178361e+05, -2.033945310831e+05, -2.034065292871e+05, -2.034177219301e+05, -2.034281183261e+05, -2.034377276261e+05, -2.034465588251e+05, -2.034546207641e+05, -2.034619221311e+05, -2.034684714671e+05, -2.034742771711e+05, -2.034793474981e+05, -2.034836905691e+05, -2.034873143671e+05, -2.034902267461e+05, -2.034924354291e+05, -2.034939480151e+05, -2.034947719791e+05, -2.034949146771e+05, -2.034943833441e+05, -2.034931851041e+05, -2.034913269641e+05, -2.034888158241e+05, -2.034856584731e+05, -2.034818615971e+05, -2.034774317761e+05, -2.034723754901e+05, -2.034666991191e+05, -2.034604089451e+05, -2.034535111571e+05, -2.034460118481e+05, -2.034379170211e+05, -2.034292325891e+05, -2.034199643771e+05, -2.034101181241e+05, -2.033996994841e+05, -2.033887140301e+05, -2.033771672521e+05, -2.033650645611e+05, -2.033524112911e+05, -2.033392126981e+05, -2.033254739641e+05, -2.033112001961e+05, -2.032963964321e+05, -2.032810676351e+05, -2.032652187031e+05, -2.032488544611e+05, -2.032319796721e+05, -2.032145990291e+05, -2.031967171631e+05, -2.031783386411e+05, -2.031594679691e+05, -2.031401095911e+05, -2.031202678901e+05, -2.030999471931e+05, -2.030791517681e+05, -2.030578858241e+05, -2.030361535191e+05, -2.030139589511e+05, -2.029913061691e+05, -2.029681991671e+05, -2.029446418861e+05, -2.029206382191e+05, -2.028961920061e+05, -2.028713070401e+05, -2.028459870641e+05, -2.028202357741e+05, -2.027940568201e+05, -2.027674538051e+05, -2.027404302871e+05, -2.027129897811e+05, -2.026851357551e+05, -2.026568716391e+05, -2.026282008161e+05, -2.025991266301e+05, -2.025696523841e+05, -2.025397813401e+05, -2.025095167211e+05, -2.024788617111e+05, -2.024478194561e+05, -2.024163930631e+05, -2.023845856041e+05, -2.023524001141e+05, -2.023198395911e+05, -2.022869069991e+05, -2.022536052661e+05, -2.022199372871e+05, -2.021859059221e+05, -2.021515140001e+05, -2.021167643151e+05, -2.020816596301e+05, -2.020462026761e+05, -2.020103961541e+05, -2.019742427331e+05, -2.019377450531e+05, -2.019009057221e+05, -2.018637273221e+05, -2.018262124021e+05, -2.017883634871e+05, -2.017501830711e+05, -2.017116736221e+05, -2.016728375791e+05, -2.016336773571e+05, -2.015941953411e+05, -2.015543938941e+05, -2.015142753491e+05, -2.014738420191e+05, -2.014330961871e+05, -2.013920401141e+05, -2.013506760371e+05, -2.013090061691e+05, -2.012670326991e+05, -2.012247577921e+05, -2.011821835931e+05, -2.011393122211e+05, -2.010961457761e+05, -2.010526863341e+05, -2.010089359501e+05, -2.009648966581e+05, -2.009205704711e+05, -2.008759593811e+05, -2.008310653591e+05, -2.007858903571e+05, -2.007404363081e+05, -2.006947051221e+05, -2.006486986941e+05, -2.006024188961e+05, -2.005558675841e+05, -2.005090465941e+05, -2.004619577451e+05, -2.004146028361e+05, -2.003669836491e+05, -2.003191019511e+05, -2.002709594861e+05, -2.002225579871e+05, -2.001738991661e+05, -2.001249847201e+05, -2.000758163291e+05, -2.000263956571e+05, -1.999767243521e+05, -1.999268040471e+05, -1.998766363561e+05, -1.998262228821e+05, -1.997755652111e+05, -1.997246649121e+05, -1.996735235431e+05, -1.996221426431e+05, -1.995705237411e+05, -1.995186683481e+05, -1.994665779631e+05, -1.994142540701e+05, -1.993616981401e+05, -1.993089116301e+05, -1.992558959841e+05, -1.992026526321e+05, -1.991491829911e+05, -1.990954884661e+05, -1.990415704471e+05, -1.989874303151e+05, -1.989330694361e+05, -1.988784891631e+05, -1.988236908391e+05, -1.987686757941e+05, -1.987134453451e+05, -1.986580007991e+05, -1.986023434511e+05, -1.985464745841e+05, -1.984903954701e+05, -1.984341073701e+05, -1.983776115321e+05, -1.983209091971e+05, -1.982640015921e+05, -1.982068899331e+05, -1.981495754291e+05, -1.980920592741e+05, -1.980343426561e+05, -1.979764267491e+05, -1.979183127191e+05, -1.978600017221e+05, -1.978014949051e+05, -1.977427934021e+05, -1.976838983401e+05, -1.976248108371e+05, -1.975655320001e+05, -1.975060629271e+05, -1.974464047071e+05, -1.973865584201e+05, -1.973265251381e+05, -1.972663059211e+05, -1.972059018241e+05, -1.971453138901e+05, -1.970845431561e+05, -1.970235906501e+05, -1.969624573891e+05, -1.969011443861e+05, -1.968396526411e+05, -1.967779831511e+05, -1.967161369001e+05, -1.966541148671e+05, -1.965919180231e+05, -1.965295473301e+05, -1.964670037431e+05, -1.964042882081e+05, -1.963414016671e+05, -1.962783450511e+05, -1.962151192851e+05, -1.961517252861e+05, -1.960881639651e+05, -1.960244362241e+05, -1.959605429601e+05, -1.958964850621e+05, -1.958322634121e+05, -1.957678788851e+05, -1.957033323491e+05, -1.956386246661e+05, -1.955737566911e+05, -1.955087292731e+05, -1.954435432541e+05, -1.953781994681e+05, -1.953126987451e+05, -1.952470419081e+05, -1.951812297731e+05, -1.951152631511e+05, -1.950491428451e+05, -1.949828696541e+05, -1.949164443701e+05, -1.948498677791e+05, -1.947831406611e+05, -1.947162637901e+05, -1.946492379361e+05, -1.945820638611e+05, -1.945147423231e+05, -1.944472740731e+05, -1.943796598571e+05, -1.943119004161e+05, -1.942439964851e+05, -1.941759487931e+05, -1.941077580671e+05, -1.940394250231e+05, -1.939709503771e+05, -1.939023348381e+05, -1.938335791081e+05, -1.937646838871e+05, -1.936956498671e+05, -1.936264777391e+05, -1.935571681841e+05, -1.934877218821e+05, -1.934181395071e+05, -1.933484217281e+05, -1.932785692091e+05, -1.932085826101e+05, -1.931384625861e+05, -1.930682097871e+05, -1.929978248591e+05, -1.929273084431e+05, -1.928566611751e+05, -1.927858836901e+05, -1.927149766131e+05, -1.926439405691e+05, -1.925727761771e+05, -1.925014840521e+05, -1.924300648041e+05, -1.923585190401e+05, -1.922868473631e+05, -1.922150503701e+05, -1.921431286561e+05, -1.920710828111e+05, -1.919989134201e+05, -1.919266210661e+05, -1.918542063271e+05, -1.917816697761e+05, -1.917090119851e+05, -1.916362335201e+05, -1.915633349431e+05, -1.914903168141e+05, -1.914171796871e+05, -1.913439241131e+05, -1.912705506421e+05, -1.911970598171e+05, -1.911234521771e+05, -1.910497282611e+05},
    {4.122428652587e+04, 3.962013181687e+04, 3.799510141687e+04, 3.634845764687e+04, 3.467941960987e+04, 3.298715971087e+04, 3.127079979787e+04, 2.952940690987e+04, 2.776198854387e+04, 2.596748739887e+04, 2.414477551787e+04, 2.229264773887e+04, 2.040981436187e+04, 1.849489290687e+04, 1.654639883287e+04, 1.456273504587e+04, 1.254218001487e+04, 1.048287425587e+04, 8.382804909871e+03, 6.239788095871e+03, 4.051448629871e+03, 1.815196642871e+03, -4.717994851295e+02, -2.812644587129e+03, -5.210753417130e+03, -7.669893669129e+03, -1.019423660413e+04, -1.278841687913e+04, -1.545760357613e+04, -1.820758486313e+04, -2.104486931713e+04, -2.397680766813e+04, -2.701173951913e+04, -3.015917061313e+04, -3.342998721313e+04, -3.683671502913e+04, -4.039383035413e+04, -4.411812955913e+04, -4.802915734213e+04, -5.214967887913e+04, -5.650614479013e+04, -6.112901700713e+04, -6.605265279113e+04, -7.131411097613e+04, -7.694969521913e+04, -8.298747489013e+04, -8.943431567013e+04, -9.625811107713e+04, -1.033671914361e+05, -1.105869448481e+05, -1.176515088191e+05, -1.242654779471e+05, -1.302382263701e+05, -1.355158556941e+05, -1.401212506791e+05, -1.441255675551e+05, -1.476167362751e+05, -1.506803032531e+05, -1.533907325081e+05, -1.558092729111e+05, -1.579849967681e+05, -1.599568923421e+05, -1.617559908991e+05, -1.634071649691e+05, -1.649305391981e+05, -1.663425637001e+05, -1.676568224441e+05, -1.688846421641e+05, -1.700355534771e+05, -1.711176429311e+05, -1.721378243781e+05, -1.731020503251e+05, -1.740154782661e+05, -1.748826029191e+05, -1.757073624041e+05, -1.764932242541e+05, -1.772432556941e+05, -1.779601814891e+05, -1.786464319171e+05, -1.793041827841e+05, -1.799353890131e+05, -1.805418129831e+05, -1.811250485601e+05, -1.816865415731e+05, -1.822276073271e+05, -1.827494456541e+05, -1.832531538931e+05, -1.837397381241e+05, -1.842101229271e+05, -1.846651598921e+05, -1.851056350561e+05, -1.855322754291e+05, -1.859457547321e+05, -1.863466984701e+05, -1.867356884041e+05, -1.871132665361e+05, -1.874799386421e+05, -1.878361774271e+05, -1.881824253451e+05, -1.885190971241e+05, -1.888465820371e+05, -1.891652459411e+05, -1.894754331201e+05, -1.897774679501e+05, -1.900716564041e+05, -1.903582874191e+05, -1.906376341411e+05, -1.909099550541e+05, -1.911754950111e+05, -1.914344861791e+05, -1.916871489021e+05, -1.919336924881e+05, -1.921743159361e+05, -1.924092086051e+05, -1.926385508251e+05, -1.928625144631e+05, -1.930812634471e+05, -1.932949542461e+05, -1.935037363191e+05, -1.937077525261e+05, -1.939071395111e+05, -1.941020280631e+05, -1.942925434391e+05, -1.944788056801e+05, -1.946609298901e+05, -1.948390265101e+05, -1.950132015641e+05, -1.951835568901e+05, -1.953501903651e+05, -1.955131961001e+05, -1.956726646361e+05, -1.958286831251e+05, -1.959813354931e+05, -1.961307025981e+05, -1.962768623831e+05, -1.964198900101e+05, -1.965598579941e+05, -1.966968363271e+05, -1.968308925901e+05, -1.969620920691e+05, -1.970904978531e+05, -1.972161709341e+05, -1.973391703021e+05, -1.974595530271e+05, -1.975773743461e+05, -1.976926877391e+05, -1.978055450041e+05, -1.979159963281e+05, -1.980240903501e+05, -1.981298742311e+05, -1.982333937051e+05, -1.983346931461e+05, -1.984338156121e+05, -1.985308029061e+05, -1.986256956171e+05, -1.987185331731e+05, -1.988093538801e+05, -1.988981949701e+05, -1.989850926341e+05, -1.990700820691e+05, -1.991531975071e+05, -1.992344722551e+05, -1.993139387271e+05, -1.993916284751e+05, -1.994675722201e+05, -1.995417998811e+05, -1.996143406061e+05, -1.996852227911e+05, -1.997544741161e+05, -1.998221215601e+05, -1.998881914301e+05, -1.999527093821e+05, -2.000157004421e+05, -2.000771890261e+05, -2.001371989621e+05, -2.001957535081e+05, -2.002528753691e+05, -2.003085867171e+05, -2.003629092041e+05, -2.004158639821e+05, -2.004674717181e+05, -2.005177526051e+05, -2.005667263831e+05, -2.006144123451e+05, -2.006608293591e+05, -2.007059958711e+05, -2.007499299271e+05, -2.007926491781e+05, -2.008341708951e+05, -2.008745119771e+05, -2.009136889651e+05, -2.009517180511e+05, -2.009886150871e+05, -2.010243955931e+05, -2.010590747721e+05, -2.010926675131e+05, -2.011251884021e+05, -2.011566517291e+05, -2.011870714991e+05, -2.012164614361e+05, -2.012448349931e+05, -2.012722053581e+05, -2.012985854621e+05, -2.013239879841e+05, -2.013484253601e+05, -2.013719097851e+05, -2.013944532261e+05, -2.014160674211e+05, -2.014367638891e+05, -2.014565539341e+05, -2.014754486491e+05, -2.014934589261e+05, -2.015105954561e+05, -2.015268687331e+05, -2.015422890671e+05, -2.015568665781e+05, -2.015706112081e+05, -2.015835327231e+05, -2.015956407151e+05, -2.016069446101e+05, -2.016174536691e+05, -2.016271769931e+05, -2.016361235261e+05, -2.016443020601e+05, -2.016517212371e+05, -2.016583895531e+05, -2.016643153611e+05, -2.016695068761e+05, -2.016739721761e+05, -2.016777192041e+05, -2.016807557741e+05, -2.016830895741e+05, -2.016847281641e+05, -2.016856789851e+05, -2.016859493571e+05, -2.016855464831e+05, -2.016844774541e+05, -2.016827492451e+05, -2.016803687261e+05, -2.016773426571e+05, -2.016736776941e+05, -2.016693803891e+05, -2.016644571961e+05, -2.016589144681e+05, -2.016527584621e+05, -2.016459953411e+05, -2.016386311751e+05, -2.016306719431e+05, -2.016221235361e+05, -2.016129917551e+05, -2.016032823201e+05, -2.015930008631e+05, -2.015821529371e+05, -2.015707440111e+05, -2.015587794791e+05, -2.015462646551e+05, -2.015332047781e+05, -2.015196050121e+05, -2.015054704471e+05, -2.014908061031e+05, -2.014756169301e+05, -2.014599078081e+05, -2.014436835471e+05, -2.014269488961e+05, -2.014097085331e+05, -2.013919670761e+05, -2.013737290791e+05, -2.013549990321e+05, -2.013357813681e+05, -2.013160804591e+05, -2.012959006171e+05, -2.012752461001e+05, -2.012541211051e+05, -2.012325297791e+05, -2.012104762101e+05, -2.011879644361e+05, -2.011649984391e+05, -2.011415821531e+05, -2.011177194601e+05, -2.010934141901e+05, -2.010686701281e+05, -2.010434910071e+05, -2.010178805161e+05, -2.009918422951e+05, -2.009653799401e+05, -2.009384970011e+05, -2.009111969851e+05, -2.008834833531e+05, -2.008553595261e+05, -2.008268288821e+05, -2.007978947571e+05, -2.007685604481e+05, -2.007388292111e+05, -2.007087042611e+05, -2.006781887771e+05, -2.006472858991e+05, -2.006159987291e+05, -2.005843303321e+05, -2.005522837381e+05, -2.005198619401e+05, -2.004870678961e+05, -2.004539045311e+05, -2.004203747331e+05, -2.003864813591e+05, -2.003522272321e+05, -2.003176151431e+05, -2.002826478491e+05, -2.002473280791e+05, -2.002116585281e+05, -2.001756418621e+05, -2.001392807171e+05, -2.001025776971e+05, -2.000655353801e+05, -2.000281563141e+05, -1.999904430181e+05, -1.999523979841e+05, -1.999140236761e+05, -1.998753225311e+05, -1.998362969591e+05, -1.997969493451e+05, -1.997572820461e+05, -1.997172973961e+05, -1.996769977021e+05, -1.996363852461e+05, -1.995954622881e+05, -1.995542310601e+05, -1.995126937741e+05, -1.994708526161e+05, -1.994287097501e+05, -1.993862673181e+05, -1.993435274381e+05, -1.993004922071e+05, -1.992571636991e+05, -1.992135439691e+05, -1.991696350471e+05, -1.991254389461e+05, -1.990809576551e+05, -1.990361931451e+05, -1.989911473651e+05, -1.989458222471e+05, -1.989002197011e+05, -1.988543416191e+05, -1.988081898731e+05, -1.987617663161e+05, -1.987150727851e+05, -1.986681110951e+05, -1.986208830471e+05, -1.985733904211e+05, -1.985256349811e+05, -1.984776184741e+05, -1.984293426281e+05, -1.983808091571e+05, -1.983320197551e+05, -1.982829761031e+05, -1.982336798641e+05, -1.981841326841e+05, -1.981343361961e+05, -1.980842920151e+05, -1.980340017421e+05, -1.979834669611e+05, -1.979326892431e+05, -1.978816701431e+05, -1.978304112031e+05, -1.977789139481e+05, -1.977271798911e+05, -1.976752105291e+05, -1.976230073471e+05, -1.975705718151e+05, -1.975179053901e+05, -1.974650095151e+05, -1.974118856201e+05, -1.973585351231e+05, -1.973049594271e+05, -1.972511599251e+05, -1.971971379941e+05, -1.971428950011e+05, -1.970884323001e+05, -1.970337512331e+05, -1.969788531311e+05, -1.969237393101e+05, -1.968684110771e+05, -1.968128697281e+05, -1.967571165451e+05, -1.967011528001e+05, -1.966449797541e+05, -1.965885986571e+05, -1.965320107471e+05, -1.964752172531e+05, -1.964182193921e+05, -1.963610183711e+05, -1.963036153871e+05, -1.962460116241e+05, -1.961882082601e+05, -1.961302064601e+05, -1.960720073811e+05, -1.960136121671e+05, -1.959550219561e+05, -1.958962378741e+05, -1.958372610391e+05, -1.957780925581e+05, -1.957187335301e+05, -1.956591850441e+05, -1.955994481811e+05, -1.955395240121e+05, -1.954794135981e+05, -1.954191179951e+05, -1.953586382461e+05, -1.952979753881e+05, -1.952371304491e+05, -1.951761044491e+05, -1.951148983971e+05, -1.950535132981e+05, -1.949919501461e+05, -1.949302099271e+05, -1.948682936211e+05, -1.948062021981e+05, -1.947439366211e+05, -1.946814978461e+05, -1.946188868201e+05, -1.945561044841e+05, -1.944931517691e+05, -1.944300296021e+05, -1.943667389011e+05, -1.943032805761e+05, -1.942396555311e+05, -1.941758646621e+05, -1.941119088591e+05, -1.940477890051e+05, -1.939835059751e+05, -1.939190606391e+05, -1.938544538581e+05, -1.937896864881e+05, -1.937247593781e+05, -1.936596733711e+05, -1.935944293021e+05, -1.935290280021e+05, -1.934634702931e+05, -1.933977569931e+05, -1.933318889131e+05, -1.932658668571e+05, -1.931996916231e+05, -1.931333640061e+05, -1.930668847911e+05, -1.930002547591e+05, -1.929334746861e+05, -1.928665453401e+05, -1.927994674861e+05, -1.927322418801e+05, -1.926648692761e+05, -1.925973504201e+05, -1.925296860531e+05, -1.924618769121e+05, -1.923939237261e+05, -1.923258272211e+05, -1.922575881161e+05, -1.921892071261e+05, -1.921206849621e+05, -1.920520223251e+05, -1.919832199171e+05, -1.919142784321e+05, -1.918451985571e+05, -1.917759809791e+05, -1.917066263751e+05, -1.916371354221e+05, -1.915675087871e+05, -1.914977471371e+05, -1.914278511321e+05, -1.913578214271e+05, -1.912876586741e+05, -1.912173635181e+05, -1.911469366021e+05, -1.910763785631e+05, -1.910056900341e+05, -1.909348716441e+05, -1.908639240171e+05, -1.907928477731e+05, -1.907216435271e+05, -1.906503118911e+05, -1.905788534721e+05, -1.905072688731e+05, -1.904355586921e+05, -1.903637235251e+05, -1.902917639631e+05, -1.902196805911e+05, -1.901474739931e+05, -1.900751447471e+05, -1.900026934291e+05, -1.899301206091e+05, -1.898574268541e+05, -1.897846127281e+05, -1.897116787901e+05, -1.896386255971e+05, -1.895654537001e+05, -1.894921636471e+05, -1.894187559841e+05},
    {4.211563214087e+04, 4.052249652787e+04, 3.890891537887e+04, 3.727417846987e+04, 3.561753475487e+04, 3.393818914087e+04, 3.223529893987e+04, 3.050796994987e+04, 2.875525211287e+04, 2.697613470687e+04, 2.516954099587e+04, 2.333432226987e+04, 2.146925118787e+04, 1.957301432787e+04, 1.764420381587e+04, 1.568130790387e+04, 1.368270033287e+04, 1.164662828687e+04, 9.571198707871e+03, 7.454362704871e+03, 5.293897724871e+03, 3.087387101870e+03, 8.321965198705e+02, -1.474553189129e+03, -3.836017578129e+03, -6.255665277130e+03, -8.737320863130e+03, -1.128521510613e+04, -1.390404409413e+04, -1.659903903113e+04, -1.937604889413e+04, -2.224163859313e+04, -2.520320575613e+04, -2.826911981713e+04, -3.144888752313e+04, -3.475334922313e+04, -3.819490985313e+04, -4.178780664313e+04, -4.554841049913e+04, -4.949554636513e+04, -5.365079266613e+04, -5.803866785713e+04, -6.268650845413e+04, -6.762365008013e+04, -7.287920872313e+04, -7.847739228913e+04, -8.442924821313e+04, -9.072081151213e+04, -9.729939436313e+04, -1.040594185751e+05, -1.108301981281e+05, -1.173851026881e+05, -1.235005966621e+05, -1.290458515401e+05, -1.339851008201e+05, -1.383398849331e+05, -1.421672967681e+05, -1.455374913031e+05, -1.485199404541e+05, -1.511767406781e+05, -1.535604276541e+05, -1.557141901631e+05, -1.576730895481e+05, -1.594655160991e+05, -1.611145349931e+05, -1.626390081121e+05, -1.640544852271e+05, -1.653738967481e+05, -1.666080879121e+05, -1.677662305341e+05, -1.688561416271e+05, -1.698845315661e+05, -1.708571989371e+05, -1.717791848801e+05, -1.726548964521e+05, -1.734882061351e+05, -1.742825327811e+05, -1.750409080041e+05, -1.757660310351e+05, -1.764603143731e+05, -1.771259220011e+05, -1.777648015781e+05, -1.783787116921e+05, -1.789692450421e+05, -1.795378482581e+05, -1.800858389081e+05, -1.806144201491e+05, -1.811246934051e+05, -1.816176693631e+05, -1.820942775541e+05, -1.825553747231e+05, -1.830017521591e+05, -1.834341421441e+05, -1.838532236351e+05, -1.842596272821e+05, -1.846539398841e+05, -1.850367083401e+05, -1.854084431701e+05, -1.857696216651e+05, -1.861206906951e+05, -1.864620692391e+05, -1.867941506491e+05, -1.871173046991e+05, -1.874318794261e+05, -1.877382028001e+05, -1.880365842371e+05, -1.883273159711e+05, -1.886106743001e+05, -1.888869207271e+05, -1.891563029941e+05, -1.894190560321e+05, -1.896754028321e+05, -1.899255552341e+05, -1.901697146651e+05, -1.904080728061e+05, -1.906408122151e+05, -1.908681068931e+05, -1.910901228141e+05, -1.913070184081e+05, -1.915189450161e+05, -1.917260473041e+05, -1.919284636531e+05, -1.921263265131e+05, -1.923197627441e+05, -1.925088939221e+05, -1.926938366311e+05, -1.928747027331e+05, -1.930515996181e+05, -1.932246304431e+05, -1.933938943491e+05, -1.935594866701e+05, -1.937214991211e+05, -1.938800199881e+05, -1.940351342871e+05, -1.941869239331e+05, -1.943354678821e+05, -1.944808422791e+05, -1.946231205841e+05, -1.947623737021e+05, -1.948986701001e+05, -1.950320759141e+05, -1.951626550591e+05, -1.952904693241e+05, -1.954155784701e+05, -1.955380403111e+05, -1.956579108081e+05, -1.957752441371e+05, -1.958900927731e+05, -1.960025075581e+05, -1.961125377651e+05, -1.962202311691e+05, -1.963256341011e+05, -1.964287915101e+05, -1.965297470171e+05, -1.966285429651e+05, -1.967252204721e+05, -1.968198194751e+05, -1.969123787791e+05, -1.970029360941e+05, -1.970915280831e+05, -1.971781903941e+05, -1.972629577021e+05, -1.973458637421e+05, -1.974269413451e+05, -1.975062224691e+05, -1.975837382281e+05, -1.976595189251e+05, -1.977335940801e+05, -1.978059924531e+05, -1.978767420751e+05, -1.979458702701e+05, -1.980134036781e+05, -1.980793682821e+05, -1.981437894241e+05, -1.982066918301e+05, -1.982680996281e+05, -1.983280363691e+05, -1.983865250451e+05, -1.984435881051e+05, -1.984992474741e+05, -1.985535245681e+05, -1.986064403121e+05, -1.986580151511e+05, -1.987082690711e+05, -1.987572216061e+05, -1.988048918551e+05, -1.988512984961e+05, -1.988964597951e+05, -1.989403936221e+05, -1.989831174601e+05, -1.990246484171e+05, -1.990650032371e+05, -1.991041983081e+05, -1.991422496761e+05, -1.991791730531e+05, -1.992149838241e+05, -1.992496970591e+05, -1.992833275211e+05, -1.993158896751e+05, -1.993473976931e+05, -1.993778654641e+05, -1.994073066041e+05, -1.994357344581e+05, -1.994631621131e+05, -1.994896023981e+05, -1.995150678981e+05, -1.995395709541e+05, -1.995631236741e+05, -1.995857379351e+05, -1.996074253931e+05, -1.996281974841e+05, -1.996480654341e+05, -1.996670402611e+05, -1.996851327811e+05, -1.997023536131e+05, -1.997187131851e+05, -1.997342217361e+05, -1.997488893251e+05, -1.997627258291e+05, -1.997757409521e+05, -1.997879442301e+05, -1.997993450291e+05, -1.998099525551e+05, -1.998197758561e+05, -1.998288238241e+05, -1.998371051991e+05, -1.998446285761e+05, -1.998514024021e+05, -1.998574349861e+05, -1.998627344951e+05, -1.998673089651e+05, -1.998711662981e+05, -1.998743142671e+05, -1.998767605181e+05, -1.998785125751e+05, -1.998795778411e+05, -1.998799636001e+05, -1.998796770191e+05, -1.998787251541e+05, -1.998771149501e+05, -1.998748532421e+05, -1.998719467601e+05, -1.998684021281e+05, -1.998642258711e+05, -1.998594244121e+05, -1.998540040771e+05, -1.998479710961e+05, -1.998413316061e+05, -1.998340916511e+05, -1.998262571861e+05, -1.998178340761e+05, -1.998088281011e+05, -1.997992449561e+05, -1.997890902521e+05, -1.997783695201e+05, -1.997670882091e+05, -1.997552516911e+05, -1.997428652611e+05, -1.997299341391e+05, -1.997164634691e+05, -1.997024583261e+05, -1.996879237101e+05, -1.996728645531e+05, -1.996572857181e+05, -1.996411920031e+05, -1.996245881351e+05, -1.996074787821e+05, -1.995898685441e+05, -1.995717619601e+05, -1.995531635081e+05, -1.995340776061e+05, -1.995145086121e+05, -1.994944608271e+05, -1.994739384931e+05, -1.994529457991e+05, -1.994314868761e+05, -1.994095658021e+05, -1.993871866031e+05, -1.993643532511e+05, -1.993410696691e+05, -1.993173397271e+05, -1.992931672471e+05, -1.992685560021e+05, -1.992435097171e+05, -1.992180320711e+05, -1.991921266951e+05, -1.991657971751e+05, -1.991390470531e+05, -1.991118798281e+05, -1.990842989531e+05, -1.990563078411e+05, -1.990279098611e+05, -1.989991083431e+05, -1.989699065751e+05, -1.989403078061e+05, -1.989103152471e+05, -1.988799320671e+05, -1.988491614001e+05, -1.988180063431e+05, -1.987864699531e+05, -1.987545552551e+05, -1.987222652361e+05, -1.986896028491e+05, -1.986565710111e+05, -1.986231726071e+05, -1.985894104871e+05, -1.985552874701e+05, -1.985208063421e+05, -1.984859698551e+05, -1.984507807331e+05, -1.984152416671e+05, -1.983793553181e+05, -1.983431243171e+05, -1.983065512661e+05, -1.982696387371e+05, -1.982323892741e+05, -1.981948053921e+05, -1.981568895811e+05, -1.981186443001e+05, -1.980800719821e+05, -1.980411750351e+05, -1.980019558401e+05, -1.979624167511e+05, -1.979225600981e+05, -1.978823881861e+05, -1.978419032931e+05, -1.978011076771e+05, -1.977600035661e+05, -1.977185931711e+05, -1.976768786741e+05, -1.976348622381e+05, -1.975925460011e+05, -1.975499320791e+05, -1.975070225661e+05, -1.974638195361e+05, -1.974203250391e+05, -1.973765411051e+05, -1.973324697441e+05, -1.972881129431e+05, -1.972434726721e+05, -1.971985508781e+05, -1.971533494891e+05, -1.971078704161e+05, -1.970621155481e+05, -1.970160867551e+05, -1.969697858901e+05, -1.969232147871e+05, -1.968763752611e+05, -1.968292691101e+05, -1.967818981141e+05, -1.967342640351e+05, -1.966863686181e+05, -1.966382135921e+05, -1.965898006671e+05, -1.965411315391e+05, -1.964922078851e+05, -1.964430313681e+05, -1.963936036351e+05, -1.963439263151e+05, -1.962940010231e+05, -1.962438293601e+05, -1.961934129091e+05, -1.961427532401e+05, -1.960918519071e+05, -1.960407104511e+05, -1.959893303971e+05, -1.959377132571e+05, -1.958858605281e+05, -1.958337736931e+05, -1.957814542231e+05, -1.957289035731e+05, -1.956761231861e+05, -1.956231144931e+05, -1.955698789101e+05, -1.955164178401e+05, -1.954627326741e+05, -1.954088247911e+05, -1.953546955581e+05, -1.953003463271e+05, -1.952457784401e+05, -1.951909932271e+05, -1.951359920061e+05, -1.950807760821e+05, -1.950253467511e+05, -1.949697052961e+05, -1.949138529871e+05, -1.948577910871e+05, -1.948015208441e+05, -1.947450434971e+05, -1.946883602751e+05, -1.946314723941e+05, -1.945743810611e+05, -1.945170874741e+05, -1.944595928171e+05, -1.944018982671e+05, -1.943440049891e+05, -1.942859141401e+05, -1.942276268671e+05, -1.941691443041e+05, -1.941104675791e+05, -1.940515978101e+05, -1.939925361051e+05, -1.939332835611e+05, -1.938738412691e+05, -1.938142103101e+05, -1.937543917531e+05, -1.936943866631e+05, -1.936341960931e+05, -1.935738210871e+05, -1.935132626841e+05, -1.934525219101e+05, -1.933915997851e+05, -1.933304973211e+05, -1.932692155211e+05, -1.932077553791e+05, -1.931461178841e+05, -1.930843040131e+05, -1.930223147391e+05, -1.929601510231e+05, -1.928978138231e+05, -1.928353040861e+05, -1.927726227521e+05, -1.927097707551e+05, -1.926467490211e+05, -1.925835584671e+05, -1.925202000061e+05, -1.924566745401e+05, -1.923929829671e+05, -1.923291261771e+05, -1.922651050531e+05, -1.922009204711e+05, -1.921365733001e+05, -1.920720644041e+05, -1.920073946371e+05, -1.919425648501e+05, -1.918775758861e+05, -1.918124285801e+05, -1.917471237641e+05, -1.916816622601e+05, -1.916160448871e+05, -1.915502724561e+05, -1.914843457721e+05, -1.914182656341e+05, -1.913520328351e+05, -1.912856481631e+05, -1.912191124001e+05, -1.911524263201e+05, -1.910855906931e+05, -1.910186062831e+05, -1.909514738501e+05, -1.908841941451e+05, -1.908167679161e+05, -1.907491959041e+05, -1.906814788471e+05, -1.906136174751e+05, -1.905456125141e+05, -1.904774646831e+05, -1.904091746991e+05, -1.903407432711e+05, -1.902721711041e+05, -1.902034588981e+05, -1.901346073481e+05, -1.900656171431e+05, -1.899964889671e+05, -1.899272235031e+05, -1.898578214231e+05, -1.897882833991e+05, -1.897186100961e+05, -1.896488021751e+05, -1.895788602921e+05, -1.895087850991e+05, -1.894385772421e+05, -1.893682373651e+05, -1.892977661061e+05, -1.892271640971e+05, -1.891564319691e+05, -1.890855703451e+05, -1.890145798461e+05, -1.889434610901e+05, -1.888722146861e+05, -1.888008412441e+05, -1.887293413671e+05, -1.886577156541e+05, -1.885859647001e+05, -1.885140890981e+05, -1.884420894341e+05, -1.883699662911e+05, -1.882977202491e+05, -1.882253518841e+05, -1.881528617661e+05, -1.880802504651e+05, -1.880075185431e+05, -1.879346665601e+05, -1.878616950741e+05, -1.877886046371e+05},
    {4.300795105287e+04, 4.142571024587e+04, 3.982344418387e+04, 3.820046903787e+04, 3.655606241387e+04, 3.488946038087e+04, 3.319985419687e+04, 3.148638670287e+04, 2.974814834287e+04, 2.798417276387e+04, 2.619343192787e+04, 2.437483069687e+04, 2.252720078887e+04, 2.064929403787e+04, 1.873977485287e+04, 1.679721175387e+04, 1.482006785087e+04, 1.280669011187e+04, 1.075529721887e+04, 8.663965783870e+03, 6.530614684871e+03, 4.352987164871e+03, 2.128630358871e+03, -1.451282341295e+02, -2.471207855129e+03, -4.852805809129e+03, -7.293433797129e+03, -9.796960163129e+03, -1.236765917813e+04, -1.501026868013e+04, -1.773005764013e+04, -2.053290549113e+04, -2.342539534413e+04, -2.641492346613e+04, -2.950982754213e+04, -3.271953610613e+04, -3.605474083613e+04, -3.952759148313e+04, -4.315190898513e+04, -4.694340340613e+04, -5.091986571313e+04, -5.510126795713e+04, -5.950964178113e+04, -6.416848983113e+04, -6.910130035513e+04, -7.432850701113e+04, -7.986213943513e+04, -8.569786053913e+04, -9.180530273913e+04, -9.811842445013e+04, -1.045269379671e+05, -1.108732104251e+05, -1.169701823851e+05, -1.226501794061e+05, -1.278222215601e+05, -1.324624501791e+05, -1.365905947731e+05, -1.402530440651e+05, -1.435064758351e+05, -1.464077821771e+05, -1.490089210821e+05, -1.513549229601e+05, -1.534836770011e+05, -1.554265786091e+05, -1.572094870561e+05, -1.588537029111e+05, -1.603768381401e+05, -1.617935411911e+05, -1.631160810911e+05, -1.643548101851e+05, -1.655185281991e+05, -1.666147684461e+05, -1.676500234501e+05, -1.686299237681e+05, -1.695593806751e+05, -1.704427008781e+05, -1.712836794861e+05, -1.720856759571e+05, -1.728516766091e+05, -1.735843464601e+05, -1.742860724961e+05, -1.749590000321e+05, -1.756050634171e+05, -1.762260121241e+05, -1.768234330111e+05, -1.773987694081e+05, -1.779533375481e+05, -1.784883407591e+05, -1.790048817821e+05, -1.795039734831e+05, -1.799865482081e+05, -1.804534659761e+05, -1.809055216731e+05, -1.813434513921e+05, -1.817679380261e+05, -1.821796162311e+05, -1.825790768191e+05, -1.829668706801e+05, -1.833435122711e+05, -1.837094827431e+05, -1.840652327351e+05, -1.844111848931e+05, -1.847477361261e+05, -1.850752596501e+05, -1.853941068281e+05, -1.857046088381e+05, -1.860070781861e+05, -1.863018100811e+05, -1.865890836841e+05, -1.868691632481e+05, -1.871422991601e+05, -1.874087288921e+05, -1.876686778741e+05, -1.879223602921e+05, -1.881699798231e+05, -1.884117303111e+05, -1.886477963901e+05, -1.888783540571e+05, -1.891035712021e+05, -1.893236080991e+05, -1.895386178591e+05, -1.897487468511e+05, -1.899541350911e+05, -1.901549166061e+05, -1.903512197691e+05, -1.905431676141e+05, -1.907308781301e+05, -1.909144645291e+05, -1.910940355031e+05, -1.912696954621e+05, -1.914415447571e+05, -1.916096798861e+05, -1.917741936881e+05, -1.919351755321e+05, -1.920927114801e+05, -1.922468844561e+05, -1.923977743911e+05, -1.925454583691e+05, -1.926900107611e+05, -1.928315033501e+05, -1.929700054501e+05, -1.931055840201e+05, -1.932383037661e+05, -1.933682272481e+05, -1.934954149651e+05, -1.936199254561e+05, -1.937418153721e+05, -1.938611395681e+05, -1.939779511691e+05, -1.940923016501e+05, -1.942042408981e+05, -1.943138172811e+05, -1.944210777061e+05, -1.945260676821e+05, -1.946288313711e+05, -1.947294116421e+05, -1.948278501241e+05, -1.949241872491e+05, -1.950184623021e+05, -1.951107134621e+05, -1.952009778441e+05, -1.952892915391e+05, -1.953756896481e+05, -1.954602063261e+05, -1.955428748051e+05, -1.956237274391e+05, -1.957027957241e+05, -1.957801103371e+05, -1.958557011591e+05, -1.959295973031e+05, -1.960018271431e+05, -1.960724183391e+05, -1.961413978551e+05, -1.962087919911e+05, -1.962746263991e+05, -1.963389261041e+05, -1.964017155301e+05, -1.964630185111e+05, -1.965228583191e+05, -1.965812576741e+05, -1.966382387661e+05, -1.966938232691e+05, -1.967480323611e+05, -1.968008867331e+05, -1.968524066091e+05, -1.969026117571e+05, -1.969515215071e+05, -1.969991547581e+05, -1.970455299961e+05, -1.970906653011e+05, -1.971345783651e+05, -1.971772864991e+05, -1.972188066451e+05, -1.972591553851e+05, -1.972983489541e+05, -1.973364032491e+05, -1.973733338371e+05, -1.974091559641e+05, -1.974438845661e+05, -1.974775342761e+05, -1.975101194321e+05, -1.975416540871e+05, -1.975721520131e+05, -1.976016267111e+05, -1.976300914181e+05, -1.976575591131e+05, -1.976840425261e+05, -1.977095541391e+05, -1.977341062011e+05, -1.977577107241e+05, -1.977803794981e+05, -1.978021240891e+05, -1.978229558531e+05, -1.978428859311e+05, -1.978619252631e+05, -1.978800845891e+05, -1.978973744541e+05, -1.979138052141e+05, -1.979293870411e+05, -1.979441299221e+05, -1.979580436721e+05, -1.979711379311e+05, -1.979834221731e+05, -1.979949057061e+05, -1.980055976781e+05, -1.980155070801e+05, -1.980246427501e+05, -1.980330133771e+05, -1.980406275021e+05, -1.980474935251e+05, -1.980536197051e+05, -1.980590141651e+05, -1.980636848931e+05, -1.980676397481e+05, -1.980708864601e+05, -1.980734326351e+05, -1.980752857561e+05, -1.980764531851e+05, -1.980769421701e+05, -1.980767598421e+05, -1.980759132191e+05, -1.980744092121e+05, -1.980722546221e+05, -1.980694561461e+05, -1.980660203781e+05, -1.980619538091e+05, -1.980572628331e+05, -1.980519537481e+05, -1.980460327541e+05, -1.980395059601e+05, -1.980323793841e+05, -1.980246589541e+05, -1.980163505111e+05, -1.980074598081e+05, -1.979979925181e+05, -1.979879542271e+05, -1.979773504441e+05, -1.979661865951e+05, -1.979544680321e+05, -1.979422000281e+05, -1.979293877811e+05, -1.979160364181e+05, -1.979021509931e+05, -1.978877364871e+05, -1.978727978151e+05, -1.978573398211e+05, -1.978413672861e+05, -1.978248849201e+05, -1.978078973741e+05, -1.977904092341e+05, -1.977724250211e+05, -1.977539491991e+05, -1.977349861721e+05, -1.977155402821e+05, -1.976956158161e+05, -1.976752170041e+05, -1.976543480201e+05, -1.976330129831e+05, -1.976112159591e+05, -1.975889609611e+05, -1.975662519501e+05, -1.975430928351e+05, -1.975194874761e+05, -1.974954396851e+05, -1.974709532231e+05, -1.974460318061e+05, -1.974206791001e+05, -1.973948987291e+05, -1.973686942681e+05, -1.973420692501e+05, -1.973150271641e+05, -1.972875714561e+05, -1.972597055271e+05, -1.972314327411e+05, -1.972027564191e+05, -1.971736798391e+05, -1.971442062451e+05, -1.971143388371e+05, -1.970840807801e+05, -1.970534351991e+05, -1.970224051831e+05, -1.969909937851e+05, -1.969592040211e+05, -1.969270388721e+05, -1.968945012841e+05, -1.968615941701e+05, -1.968283204071e+05, -1.967946828411e+05, -1.967606842831e+05, -1.967263275151e+05, -1.966916152841e+05, -1.966565503071e+05, -1.966211352711e+05, -1.965853728321e+05, -1.965492656161e+05, -1.965128162201e+05, -1.964760272121e+05, -1.964389011321e+05, -1.964014404901e+05, -1.963636477711e+05, -1.963255254301e+05, -1.962870758981e+05, -1.962483015771e+05, -1.962092048451e+05, -1.961697880521e+05, -1.961300535251e+05, -1.960900035641e+05, -1.960496404471e+05, -1.960089664241e+05, -1.959679837241e+05, -1.959266945521e+05, -1.958851010891e+05, -1.958432054931e+05, -1.958010099011e+05, -1.957585164251e+05, -1.957157271581e+05, -1.956726441691e+05, -1.956292695061e+05, -1.955856051991e+05, -1.955416532521e+05, -1.954974156521e+05, -1.954528943661e+05, -1.954080913391e+05, -1.953630084981e+05, -1.953176477491e+05, -1.952720109811e+05, -1.952261000621e+05, -1.951799168431e+05, -1.951334631561e+05, -1.950867408151e+05, -1.950397516141e+05, -1.949924973341e+05, -1.949449797341e+05, -1.948972005571e+05, -1.948491615321e+05, -1.948008643661e+05, -1.947523107551e+05, -1.947035023741e+05, -1.946544408851e+05, -1.946051279321e+05, -1.945555651451e+05, -1.945057541381e+05, -1.944556965081e+05, -1.944053938401e+05, -1.943548477011e+05, -1.943040596451e+05, -1.942530312121e+05, -1.942017639251e+05, -1.941502592961e+05, -1.940985188201e+05, -1.940465439821e+05, -1.939943362491e+05, -1.939418970771e+05, -1.938892279081e+05, -1.938363301711e+05, -1.937832052831e+05, -1.937298546451e+05, -1.936762796501e+05, -1.936224816731e+05, -1.935684620821e+05, -1.935142222291e+05, -1.934597634541e+05, -1.934050870891e+05, -1.933501944491e+05, -1.932950868401e+05, -1.932397655571e+05, -1.931842318821e+05, -1.931284870871e+05, -1.930725324311e+05, -1.930163691661e+05, -1.929599985271e+05, -1.929034217451e+05, -1.928466400351e+05, -1.927896546041e+05, -1.927324666481e+05, -1.926750773541e+05, -1.926174878961e+05, -1.925596994421e+05, -1.925017131461e+05, -1.924435301541e+05, -1.923851516041e+05, -1.923265786221e+05, -1.922678123241e+05, -1.922088538201e+05, -1.921497042071e+05, -1.920903645761e+05, -1.920308360061e+05, -1.919711195691e+05, -1.919112163281e+05, -1.918511273361e+05, -1.917908536391e+05, -1.917303962741e+05, -1.916697562681e+05, -1.916089346411e+05, -1.915479324061e+05, -1.914867505641e+05, -1.914253901111e+05, -1.913638520351e+05, -1.913021373141e+05, -1.912402469201e+05, -1.911781818171e+05, -1.911159429591e+05, -1.910535312961e+05, -1.909909477691e+05, -1.909281933101e+05, -1.908652688461e+05, -1.908021752941e+05, -1.907389135681e+05, -1.906754845701e+05, -1.906118891981e+05, -1.905481283421e+05, -1.904842028861e+05, -1.904201137061e+05, -1.903558616721e+05, -1.902914476461e+05, -1.902268724861e+05, -1.901621370401e+05, -1.900972421521e+05, -1.900321886591e+05, -1.899669773921e+05, -1.899016091741e+05, -1.898360848231e+05, -1.897704051521e+05, -1.897045709661e+05, -1.896385830641e+05, -1.895724422391e+05, -1.895061492811e+05, -1.894397049701e+05, -1.893731100821e+05, -1.893063653871e+05, -1.892394716501e+05, -1.891724296311e+05, -1.891052400811e+05, -1.890379037491e+05, -1.889704213761e+05, -1.889027937011e+05, -1.888350214541e+05, -1.887671053601e+05, -1.886990461421e+05, -1.886308445151e+05, -1.885625011891e+05, -1.884940168691e+05, -1.884253922551e+05, -1.883566280441e+05, -1.882877249241e+05, -1.882186835821e+05, -1.881495046981e+05, -1.880801889471e+05, -1.880107370001e+05, -1.879411495241e+05, -1.878714271791e+05, -1.878015706231e+05, -1.877315805071e+05, -1.876614574791e+05, -1.875912021811e+05, -1.875208152541e+05, -1.874502973291e+05, -1.873796490381e+05, -1.873088710061e+05, -1.872379638531e+05, -1.871669281961e+05, -1.870957646481e+05, -1.870244738161e+05, -1.869530563061e+05, -1.868815127171e+05, -1.868098436441e+05, -1.867380496811e+05, -1.866661314141e+05, -1.865940894271e+05, -1.865219243001e+05, -1.864496366101e+05, -1.863772269281e+05, -1.863046958231e+05, -1.862320438581e+05, -1.861592715961e+05},
    {4.390124189487e+04, 4.232977363287e+04, 4.073869079787e+04, 3.912733493287e+04, 3.749501115887e+04, 3.584098542287e+04, 3.416448148287e+04, 3.246467759587e+04, 3.074070286287e+04, 2.899163320487e+04, 2.721648690787e+04, 2.541421968887e+04, 2.358371921987e+04, 2.172379903287e+04, 1.983319172187e+04, 1.791054134287e+04, 1.595439489787e+04, 1.396319276987e+04, 1.193525794187e+04, 9.868783825870e+03, 7.761820481870e+03, 5.612258964871e+03, 3.417813498871e+03, 1.176001127871e+03, -1.115881591129e+03, -3.460785349130e+03, -5.861938886129e+03, -8.322884554130e+03, -1.084751943513e+04, -1.344014300213e+04, -1.610551243113e+04, -1.884890682213e+04, -2.167620176313e+04, -2.459395570113e+04, -2.760950960613e+04, -3.073110106613e+04, -3.396799319813e+04, -3.733061698113e+04, -4.083072218813e+04, -4.448152532613e+04, -4.829783045813e+04, -5.229607562713e+04, -5.649421615113e+04, -6.091128515913e+04, -6.556636094513e+04, -7.047652947413e+04, -7.565333856213e+04, -8.109741245613e+04, -8.679157586413e+04, -9.269372715013e+04, -9.873073690513e+04, -1.047945322031e+05, -1.107453761121e+05, -1.164321304091e+05, -1.217293979601e+05, -1.265729654861e+05, -1.309470147631e+05, -1.348693181901e+05, -1.383779708421e+05, -1.415192751391e+05, -1.443402080081e+05, -1.468844314881e+05, -1.491905675241e+05, -1.512917903621e+05, -1.532161166561e+05, -1.549870073311e+05, -1.566240562011e+05, -1.581436481751e+05, -1.595595368291e+05, -1.608833282871e+05, -1.621248763181e+05, -1.632926005671e+05, -1.643937413171e+05, -1.654345631951e+05, -1.664205183931e+05, -1.673563780261e+05, -1.682463384461e+05, -1.690941078521e+05, -1.699029773611e+05, -1.706758797131e+05, -1.714154381251e+05, -1.721240071861e+05, -1.728037073281e+05, -1.734564540211e+05, -1.740839826521e+05, -1.746878698141e+05, -1.752695516141e+05, -1.758303394651e+05, -1.763714337911e+05, -1.768939359241e+05, -1.773988584971e+05, -1.778871345391e+05, -1.783596254551e+05, -1.788171280521e+05, -1.792603807481e+05, -1.796900690551e+05, -1.801068304551e+05, -1.805112587291e+05, -1.809039078221e+05, -1.812852952911e+05, -1.816559053981e+05, -1.820161918791e+05, -1.823665804411e+05, -1.827074710041e+05, -1.830392397331e+05, -1.833622408711e+05, -1.836768084001e+05, -1.839832575481e+05, -1.842818861671e+05, -1.845729759731e+05, -1.848567936931e+05, -1.851335921041e+05, -1.854036109831e+05, -1.856670779881e+05, -1.859242094491e+05, -1.861752111141e+05, -1.864202788201e+05, -1.866595991231e+05, -1.868933498721e+05, -1.871217007391e+05, -1.873448137181e+05, -1.875628435731e+05, -1.877759382701e+05, -1.879842393601e+05, -1.881878823511e+05, -1.883869970441e+05, -1.885817078501e+05, -1.887721340811e+05, -1.889583902301e+05, -1.891405862221e+05, -1.893188276581e+05, -1.894932160361e+05, -1.896638489631e+05, -1.898308203481e+05, -1.899942205931e+05, -1.901541367581e+05, -1.903106527301e+05, -1.904638493711e+05, -1.906138046681e+05, -1.907605938591e+05, -1.909042895711e+05, -1.910449619301e+05, -1.911826786841e+05, -1.913175053011e+05, -1.914495050781e+05, -1.915787392291e+05, -1.917052669841e+05, -1.918291456651e+05, -1.919504307761e+05, -1.920691760701e+05, -1.921854336331e+05, -1.922992539431e+05, -1.924106859421e+05, -1.925197770931e+05, -1.926265734441e+05, -1.927311196811e+05, -1.928334591821e+05, -1.929336340691e+05, -1.930316852541e+05, -1.931276524881e+05, -1.932215744021e+05, -1.933134885501e+05, -1.934034314501e+05, -1.934914386201e+05, -1.935775446171e+05, -1.936617830671e+05, -1.937441867051e+05, -1.938247874001e+05, -1.939036161891e+05, -1.939807033061e+05, -1.940560782071e+05, -1.941297696001e+05, -1.942018054671e+05, -1.942722130921e+05, -1.943410190821e+05, -1.944082493881e+05, -1.944739293301e+05, -1.945380836151e+05, -1.946007363581e+05, -1.946619111001e+05, -1.947216308261e+05, -1.947799179841e+05, -1.948367945001e+05, -1.948922817971e+05, -1.949464008051e+05, -1.949991719841e+05, -1.950506153291e+05, -1.951007503921e+05, -1.951495962901e+05, -1.951971717211e+05, -1.952434949751e+05, -1.952885839451e+05, -1.953324561401e+05, -1.953751286951e+05, -1.954166183821e+05, -1.954569416221e+05, -1.954961144911e+05, -1.955341527331e+05, -1.955710717671e+05, -1.956068866991e+05, -1.956416123261e+05, -1.956752631491e+05, -1.957078533781e+05, -1.957393969401e+05, -1.957699074881e+05, -1.957993984071e+05, -1.958278828221e+05, -1.958553736021e+05, -1.958818833731e+05, -1.959074245161e+05, -1.959320091791e+05, -1.959556492821e+05, -1.959783565191e+05, -1.960001423701e+05, -1.960210181011e+05, -1.960409947711e+05, -1.960600832391e+05, -1.960782941661e+05, -1.960956380211e+05, -1.961121250841e+05, -1.961277654561e+05, -1.961425690571e+05, -1.961565456311e+05, -1.961697047551e+05, -1.961820558381e+05, -1.961936081281e+05, -1.962043707111e+05, -1.962143525221e+05, -1.962235623421e+05, -1.962320088051e+05, -1.962397004001e+05, -1.962466454751e+05, -1.962528522381e+05, -1.962583287651e+05, -1.962630829971e+05, -1.962671227471e+05, -1.962704557001e+05, -1.962730894181e+05, -1.962750313431e+05, -1.962762887981e+05, -1.962768689881e+05, -1.962767790071e+05, -1.962760258371e+05, -1.962746163501e+05, -1.962725573141e+05, -1.962698553911e+05, -1.962665171391e+05, -1.962625490211e+05, -1.962579573961e+05, -1.962527485321e+05, -1.962469286001e+05, -1.962405036791e+05, -1.962334797591e+05, -1.962258627411e+05, -1.962176584371e+05, -1.962088725771e+05, -1.961995108061e+05, -1.961895786881e+05, -1.961790817051e+05, -1.961680252621e+05, -1.961564146871e+05, -1.961442552311e+05, -1.961315520711e+05, -1.961183103121e+05, -1.961045349881e+05, -1.960902310611e+05, -1.960754034251e+05, -1.960600569071e+05, -1.960441962661e+05, -1.960278261991e+05, -1.960109513371e+05, -1.959935762481e+05, -1.959757054381e+05, -1.959573433551e+05, -1.959384943851e+05, -1.959191628581e+05, -1.958993530451e+05, -1.958790691601e+05, -1.958583153641e+05, -1.958370957621e+05, -1.958154144071e+05, -1.957932752981e+05, -1.957706823831e+05, -1.957476395601e+05, -1.957241506771e+05, -1.957002195331e+05, -1.956758498781e+05, -1.956510454151e+05, -1.956258098021e+05, -1.956001466511e+05, -1.955740595271e+05, -1.955475519521e+05, -1.955206274061e+05, -1.954932893241e+05, -1.954655411001e+05, -1.954373860871e+05, -1.954088275971e+05, -1.953798689011e+05, -1.953505132321e+05, -1.953207637841e+05, -1.952906237131e+05, -1.952600961361e+05, -1.952291841351e+05, -1.951978907551e+05, -1.951662190051e+05, -1.951341718591e+05, -1.951017522561e+05, -1.950689631021e+05, -1.950358072681e+05, -1.950022875921e+05, -1.949684068821e+05, -1.949341679101e+05, -1.948995734191e+05, -1.948646261211e+05, -1.948293286971e+05, -1.947936837961e+05, -1.947576940401e+05, -1.947213620211e+05, -1.946846903021e+05, -1.946476814161e+05, -1.946103378711e+05, -1.945726621451e+05, -1.945346566901e+05, -1.944963239321e+05, -1.944576662681e+05, -1.944186860731e+05, -1.943793856921e+05, -1.943397674481e+05, -1.942998336381e+05, -1.942595865341e+05, -1.942190283851e+05, -1.941781614151e+05, -1.941369878261e+05, -1.940955097951e+05, -1.940537294771e+05, -1.940116490041e+05, -1.939692704881e+05, -1.939265960171e+05, -1.938836276581e+05, -1.938403674561e+05, -1.937968174351e+05, -1.937529796011e+05, -1.937088559361e+05, -1.936644484041e+05, -1.936197589481e+05, -1.935747894921e+05, -1.935295419411e+05, -1.934840181811e+05, -1.934382200771e+05, -1.933921494781e+05, -1.933458082151e+05, -1.932991980981e+05, -1.932523209211e+05, -1.932051784611e+05, -1.931577724761e+05, -1.931101047091e+05, -1.930621768841e+05, -1.930139907091e+05, -1.929655478761e+05, -1.929168500601e+05, -1.928678989201e+05, -1.928186961001e+05, -1.927692432281e+05, -1.927195419151e+05, -1.926695937591e+05, -1.926194003411e+05, -1.925689632301e+05, -1.925182839761e+05, -1.924673641181e+05, -1.924162051801e+05, -1.923648086701e+05, -1.923131760841e+05, -1.922613089051e+05, -1.922092085981e+05, -1.921568766201e+05, -1.921043144111e+05, -1.920515234001e+05, -1.919985050001e+05, -1.919452606161e+05, -1.918917916351e+05, -1.918380994351e+05, -1.917841853821e+05, -1.917300508261e+05, -1.916756971091e+05, -1.916211255601e+05, -1.915663374941e+05, -1.915113342171e+05, -1.914561170221e+05, -1.914006871921e+05, -1.913450459971e+05, -1.912891946971e+05, -1.912331345421e+05, -1.911768667681e+05, -1.911203926041e+05, -1.910637132651e+05, -1.910068299591e+05, -1.909497438801e+05, -1.908924562151e+05, -1.908349681381e+05, -1.907772808161e+05, -1.907193954041e+05, -1.906613130471e+05, -1.906030348821e+05, -1.905445620351e+05, -1.904858956231e+05, -1.904270367541e+05, -1.903679865271e+05, -1.903087460301e+05, -1.902493163451e+05, -1.901896985421e+05, -1.901298936831e+05, -1.900699028231e+05, -1.900097270071e+05, -1.899493672721e+05, -1.898888246441e+05, -1.898281001451e+05, -1.897671947851e+05, -1.897061095681e+05, -1.896448454881e+05, -1.895834035341e+05, -1.895217846841e+05, -1.894599899101e+05, -1.893980201751e+05, -1.893358764351e+05, -1.892735596381e+05, -1.892110707261e+05, -1.891484106311e+05, -1.890855802801e+05, -1.890225805921e+05, -1.889594124771e+05, -1.888960768411e+05, -1.888325745811e+05, -1.887689065861e+05, -1.887050737421e+05, -1.886410769231e+05, -1.885769170011e+05, -1.885125948381e+05, -1.884481112901e+05, -1.883834672091e+05, -1.883186634371e+05, -1.882537008111e+05, -1.881885801621e+05, -1.881233023151e+05, -1.880578680871e+05, -1.879922782911e+05, -1.879265337321e+05, -1.878606352101e+05, -1.877945835201e+05, -1.877283794491e+05, -1.876620237781e+05, -1.875955172861e+05, -1.875288607411e+05, -1.874620549081e+05, -1.873951005481e+05, -1.873279984121e+05, -1.872607492511e+05, -1.871933538051e+05, -1.871258128121e+05, -1.870581270031e+05, -1.869902971061e+05, -1.869223238421e+05, -1.868542079261e+05, -1.867859500691e+05, -1.867175509771e+05, -1.866490113501e+05, -1.865803318841e+05, -1.865115132701e+05, -1.864425561931e+05, -1.863734613351e+05, -1.863042293701e+05, -1.862348609711e+05, -1.861653568041e+05, -1.860957175301e+05, -1.860259438071e+05, -1.859560362871e+05, -1.858859956181e+05, -1.858158224431e+05, -1.857455174021e+05, -1.856750811291e+05, -1.856045142541e+05, -1.855338174031e+05, -1.854629911971e+05, -1.853920362531e+05, -1.853209531851e+05, -1.852497426001e+05, -1.851784051041e+05, -1.851069412971e+05, -1.850353517751e+05, -1.849636371301e+05, -1.848917979511e+05, -1.848198348231e+05, -1.847477483241e+05, -1.846755390331e+05, -1.846032075211e+05, -1.845307543571e+05},
    {4.479550328287e+04, 4.323468729387e+04, 4.165465807387e+04, 4.005478156587e+04, 3.843438931187e+04, 3.679277591387e+04, 3.512919625687e+04, 3.344286246087e+04, 3.173294053587e+04, 2.999854669587e+04, 2.823874330187e+04, 2.645253436887e+04, 2.463886060687e+04, 2.279659390587e+04, 2.092453121287e+04, 1.902138771287e+04, 1.708578920487e+04, 1.511626358687e+04, 1.311123128387e+04, 1.106899449887e+04, 8.987725098871e+03, 6.865450911871e+03, 4.700040222871e+03, 2.489184159871e+03, 2.303766528705e+02, -2.079108433129e+03, -4.442243433130e+03, -6.862276817129e+03, -9.342767532129e+03, -1.188762443913e+04, -1.450115163613e+04, -1.718810051813e+04, -1.995372948913e+04, -2.280387221613e+04, -2.574501519213e+04, -2.878438499113e+04, -3.193004483613e+04, -3.519099858013e+04, -3.857729738513e+04, -4.210013930413e+04, -4.577194293013e+04, -4.960636043513e+04, -5.361816815513e+04, -5.782292819313e+04, -6.223624629613e+04, -6.687236336013e+04, -7.174174780813e+04, -7.684741257513e+04, -8.218002983713e+04, -8.771255596813e+04, -9.339549331213e+04, -9.915366733213e+04, -1.048859102961e+05, -1.104719685901e+05, -1.157913566071e+05, -1.207502245191e+05, -1.253025724401e+05, -1.294381145681e+05, -1.331726999441e+05, -1.365378994601e+05, -1.395719128891e+05, -1.423138366451e+05, -1.448005383931e+05, -1.470651990301e+05, -1.491368395321e+05, -1.510403984771e+05, -1.527970894651e+05, -1.544248714481e+05, -1.559389345051e+05, -1.573521501961e+05, -1.586754651011e+05, -1.599182329411e+05, -1.610884891631e+05, -1.621931753391e+05, -1.632383215441e+05, -1.642291943711e+05, -1.651704172421e+05, -1.660660685491e+05, -1.669197620961e+05, -1.677347134241e+05, -1.685137948271e+05, -1.692595812751e+05, -1.699743889881e+05, -1.706603080221e+05, -1.713192299661e+05, -1.719528715821e+05, -1.725627951041e+05, -1.731504257301e+05, -1.737170667631e+05, -1.742639127661e+05, -1.747920610301e+05, -1.753025216061e+05, -1.757962261081e+05, -1.762740354571e+05, -1.767367467171e+05, -1.771850991381e+05, -1.776197795241e+05, -1.780414269951e+05, -1.784506372381e+05, -1.788479662991e+05, -1.792339339801e+05, -1.796090268841e+05, -1.799737011521e+05, -1.803283849281e+05, -1.806734805811e+05, -1.810093667151e+05, -1.813363999901e+05, -1.816549167731e+05, -1.819652346341e+05, -1.822676537181e+05, -1.825624579811e+05, -1.828499163351e+05, -1.831302836791e+05, -1.834038018561e+05, -1.836707005241e+05, -1.839311979561e+05, -1.841855017781e+05, -1.844338096491e+05, -1.846763098821e+05, -1.849131820281e+05, -1.851445974071e+05, -1.853707195981e+05, -1.855917049051e+05, -1.858077027731e+05, -1.860188561891e+05, -1.862253020441e+05, -1.864271714751e+05, -1.866245901841e+05, -1.868176787291e+05, -1.870065528091e+05, -1.871913235131e+05, -1.873720975641e+05, -1.875489775481e+05, -1.877220621211e+05, -1.878914462081e+05, -1.880572211911e+05, -1.882194750781e+05, -1.883782926751e+05, -1.885337557321e+05, -1.886859430921e+05, -1.888349308271e+05, -1.889807923651e+05, -1.891235986121e+05, -1.892634180691e+05, -1.894003169361e+05, -1.895343592151e+05, -1.896656068061e+05, -1.897941196021e+05, -1.899199555701e+05, -1.900431708361e+05, -1.901638197601e+05, -1.902819550141e+05, -1.903976276451e+05, -1.905108871461e+05, -1.906217815191e+05, -1.907303573311e+05, -1.908366597731e+05, -1.909407327141e+05, -1.910426187531e+05, -1.911423592641e+05, -1.912399944491e+05, -1.913355633751e+05, -1.914291040221e+05, -1.915206533211e+05, -1.916102471931e+05, -1.916979205831e+05, -1.917837075021e+05, -1.918676410511e+05, -1.919497534611e+05, -1.920300761191e+05, -1.921086396011e+05, -1.921854736951e+05, -1.922606074321e+05, -1.923340691111e+05, -1.924058863221e+05, -1.924760859711e+05, -1.925446943021e+05, -1.926117369191e+05, -1.926772388061e+05, -1.927412243491e+05, -1.928037173511e+05, -1.928647410561e+05, -1.929243181621e+05, -1.929824708411e+05, -1.930392207531e+05, -1.930945890631e+05, -1.931485964571e+05, -1.932012631551e+05, -1.932526089231e+05, -1.933026530921e+05, -1.933514145671e+05, -1.933989118391e+05, -1.934451629991e+05, -1.934901857501e+05, -1.935339974151e+05, -1.935766149521e+05, -1.936180549591e+05, -1.936583336921e+05, -1.936974670651e+05, -1.937354706681e+05, -1.937723597701e+05, -1.938081493311e+05, -1.938428540081e+05, -1.938764881661e+05, -1.939090658851e+05, -1.939406009641e+05, -1.939711069341e+05, -1.940005970611e+05, -1.940290843551e+05, -1.940565815751e+05, -1.940831012381e+05, -1.941086556201e+05, -1.941332567701e+05, -1.941569165081e+05, -1.941796464361e+05, -1.942014579391e+05, -1.942223621971e+05, -1.942423701811e+05, -1.942614926671e+05, -1.942797402351e+05, -1.942971232741e+05, -1.943136519911e+05, -1.943293364101e+05, -1.943441863801e+05, -1.943582115771e+05, -1.943714215091e+05, -1.943838255201e+05, -1.943954327931e+05, -1.944062523551e+05, -1.944162930791e+05, -1.944255636891e+05, -1.944340727621e+05, -1.944418287331e+05, -1.944488398951e+05, -1.944551144071e+05, -1.944606602911e+05, -1.944654854421e+05, -1.944695976241e+05, -1.944730044771e+05, -1.944757135181e+05, -1.944777321461e+05, -1.944790676391e+05, -1.944797271631e+05, -1.944797177721e+05, -1.944790464091e+05, -1.944777199081e+05, -1.944757449991e+05, -1.944731283081e+05, -1.944698763611e+05, -1.944659955841e+05, -1.944614923061e+05, -1.944563727601e+05, -1.944506430871e+05, -1.944443093361e+05, -1.944373774671e+05, -1.944298533511e+05, -1.944217427751e+05, -1.944130514391e+05, -1.944037849621e+05, -1.943939488811e+05, -1.943835486561e+05, -1.943725896651e+05, -1.943610772121e+05, -1.943490165261e+05, -1.943364127611e+05, -1.943232710001e+05, -1.943095962541e+05, -1.942953934661e+05, -1.942806675081e+05, -1.942654231881e+05, -1.942496652461e+05, -1.942333983591e+05, -1.942166271401e+05, -1.941993561381e+05, -1.941815898451e+05, -1.941633326881e+05, -1.941445890391e+05, -1.941253632091e+05, -1.941056594561e+05, -1.940854819781e+05, -1.940648349201e+05, -1.940437223731e+05, -1.940221483761e+05, -1.940001169131e+05, -1.939776319201e+05, -1.939546972811e+05, -1.939313168301e+05, -1.939074943541e+05, -1.938832335921e+05, -1.938585382351e+05, -1.938334119281e+05, -1.938078582721e+05, -1.937818808211e+05, -1.937554830861e+05, -1.937286685371e+05, -1.937014405981e+05, -1.936738026541e+05, -1.936457580471e+05, -1.936173100791e+05, -1.935884620141e+05, -1.935592170741e+05, -1.935295784441e+05, -1.934995492721e+05, -1.934691326671e+05, -1.934383317031e+05, -1.934071494151e+05, -1.933755888051e+05, -1.933436528411e+05, -1.933113444521e+05, -1.932786665381e+05, -1.932456219631e+05, -1.932122135591e+05, -1.931784441251e+05, -1.931443164281e+05, -1.931098332041e+05, -1.930749971581e+05, -1.930398109661e+05, -1.930042772711e+05, -1.929683986891e+05, -1.929321778051e+05, -1.928956171781e+05, -1.928587193361e+05, -1.928214867801e+05, -1.927839219841e+05, -1.927460273951e+05, -1.927078054331e+05, -1.926692584921e+05, -1.926303889401e+05, -1.925911991201e+05, -1.925516913491e+05, -1.925118679191e+05, -1.924717311001e+05, -1.924312831351e+05, -1.923905262451e+05, -1.923494626271e+05, -1.923080944551e+05, -1.922664238801e+05, -1.922244530321e+05, -1.921821840171e+05, -1.921396189201e+05, -1.920967598061e+05, -1.920536087151e+05, -1.920101676701e+05, -1.919664386711e+05, -1.919224237001e+05, -1.918781247161e+05, -1.918335436591e+05, -1.917886824521e+05, -1.917435429961e+05, -1.916981271741e+05, -1.916524368491e+05, -1.916064738681e+05, -1.915602400581e+05, -1.915137372271e+05, -1.914669671691e+05, -1.914199316561e+05, -1.913726324451e+05, -1.913250712761e+05, -1.912772498711e+05, -1.912291699371e+05, -1.911808331641e+05, -1.911322412241e+05, -1.910833957741e+05, -1.910342984581e+05, -1.909849509001e+05, -1.909353547111e+05, -1.908855114861e+05, -1.908354228061e+05, -1.907850902361e+05, -1.907345153271e+05, -1.906836996151e+05, -1.906326446231e+05, -1.905813518581e+05, -1.905298228141e+05, -1.904780589711e+05, -1.904260617971e+05, -1.903738327451e+05, -1.903213732531e+05, -1.902686847501e+05, -1.902157686491e+05, -1.901626263521e+05, -1.901092592461e+05, -1.900556687071e+05, -1.900018561011e+05, -1.899478227771e+05, -1.898935700761e+05, -1.898390993241e+05, -1.897844118391e+05, -1.897295089231e+05, -1.896743918701e+05, -1.896190619611e+05, -1.895635204661e+05, -1.895077686451e+05, -1.894518077451e+05, -1.893956390041e+05, -1.893392636481e+05, -1.892826828931e+05, -1.892258979451e+05, -1.891689099991e+05, -1.891117202411e+05, -1.890543298441e+05, -1.889967399751e+05, -1.889389517881e+05, -1.888809664271e+05, -1.888227850301e+05, -1.887644087221e+05, -1.887058386201e+05, -1.886470758311e+05, -1.885881214531e+05, -1.885289765751e+05, -1.884696422781e+05, -1.884101196321e+05, -1.883504096991e+05, -1.882905135331e+05, -1.882304321791e+05, -1.881701666721e+05, -1.881097180421e+05, -1.880490873081e+05, -1.879882754801e+05, -1.879272835621e+05, -1.878661125481e+05, -1.878047634271e+05, -1.877432371761e+05, -1.876815347681e+05, -1.876196571651e+05, -1.875576053241e+05, -1.874953801931e+05, -1.874329827121e+05, -1.873704138151e+05, -1.873076744281e+05, -1.872447654691e+05, -1.871816878511e+05, -1.871184424771e+05, -1.870550302451e+05, -1.869914520461e+05, -1.869277087621e+05, -1.868638012711e+05, -1.867997304431e+05, -1.867354971401e+05, -1.866711022191e+05, -1.866065465311e+05, -1.865418309191e+05, -1.864769562201e+05, -1.864119232651e+05, -1.863467328781e+05, -1.862813858781e+05, -1.862158830781e+05, -1.861502252821e+05, -1.860844132911e+05, -1.860184478991e+05, -1.859523298941e+05, -1.858860600591e+05, -1.858196391691e+05, -1.857530679951e+05, -1.856863473031e+05, -1.856194778521e+05, -1.855524603961e+05, -1.854852956821e+05, -1.854179844531e+05, -1.853505274481e+05, -1.852829253981e+05, -1.852151790291e+05, -1.851472890641e+05, -1.850792562171e+05, -1.850110812011e+05, -1.849427647221e+05, -1.848743074791e+05, -1.848057101701e+05, -1.847369734841e+05, -1.846680981091e+05, -1.845990847251e+05, -1.845299340081e+05, -1.844606466291e+05, -1.843912232571e+05, -1.843216645521e+05, -1.842519711721e+05, -1.841821437711e+05, -1.841121829951e+05, -1.840420894901e+05, -1.839718638941e+05, -1.839015068421e+05, -1.838310189641e+05, -1.837604008871e+05, -1.836896532331e+05, -1.836187766181e+05, -1.835477716561e+05, -1.834766389571e+05, -1.834053791251e+05, -1.833339927611e+05, -1.832624804611e+05, -1.831908428191e+05, -1.831190804221e+05, -1.830471938561e+05, -1.829751837021e+05, -1.829030505351e+05},
    {4.569073381287e+04, 4.414045177087e+04, 4.257134875387e+04, 4.098281417187e+04, 3.937420494587e+04, 3.774484316487e+04, 3.609401352887e+04, 3.442096054887e+04, 3.272488547987e+04, 3.100494295687e+04, 2.926023728687e+04, 2.748981837687e+04, 2.569267722487e+04, 2.386774094787e+04, 2.201386726387e+04, 2.012983836987e+04, 1.821435412887e+04, 1.626602448087e+04, 1.428336095487e+04, 1.226476717887e+04, 1.020852821987e+04, 8.112798611871e+03, 5.975588853871e+03, 3.794750173871e+03, 1.567957293871e+03, -7.073111131295e+02, -3.033794592130e+03, -5.414476126129e+03, -7.852610809129e+03, -1.035175850413e+04, -1.291582099513e+04, -1.554908420813e+04, -1.825626606413e+04, -2.104257044513e+04, -2.391374758813e+04, -2.687616081513e+04, -2.993685884213e+04, -3.310365154813e+04, -3.638518491513e+04, -3.979100692813e+04, -4.333160968913e+04, -4.701842196913e+04, -5.086370830213e+04, -5.488030184513e+04, -5.908105541413e+04, -6.347783968013e+04, -6.807986814413e+04, -7.289114051413e+04, -7.790696857713e+04, -8.310994235313e+04, -8.846612619413e+04, -9.392232796613e+04, -9.940510072213e+04, -1.048229067601e+05, -1.100744290611e+05, -1.150643629971e+05, -1.197222557221e+05, -1.240146985741e+05, -1.279352587391e+05, -1.314979872461e+05, -1.347292566511e+05, -1.376608556811e+05, -1.403255614461e+05, -1.427546609371e+05, -1.449767292391e+05, -1.470171561111e+05, -1.488981038041e+05, -1.506387021981e+05, -1.522553594741e+05, -1.537621119811e+05, -1.551709681071e+05, -1.564922224451e+05, -1.577347304731e+05, -1.589061421831e+05, -1.600130974281e+05, -1.610613876311e+05, -1.620560889621e+05, -1.630016718571e+05, -1.639020911781e+05, -1.647608606471e+05, -1.655811145681e+05, -1.663656592491e+05, -1.671170160951e+05, -1.678374579081e+05, -1.685290396411e+05, -1.691936246031e+05, -1.698329068801e+05, -1.704484306381e+05, -1.710416067891e+05, -1.716137274541e+05, -1.721659785651e+05, -1.726994508671e+05, -1.732151495761e+05, -1.737140028631e+05, -1.741968693441e+05, -1.746645447021e+05, -1.751177675541e+05, -1.755572246711e+05, -1.759835556261e+05, -1.763973569431e+05, -1.767991858111e+05, -1.771895634101e+05, -1.775689779051e+05, -1.779378871351e+05, -1.782967210451e+05, -1.786458838731e+05, -1.789857561421e+05, -1.793166964521e+05, -1.796390431241e+05, -1.799531156761e+05, -1.802592161871e+05, -1.805576305281e+05, -1.808486294911e+05, -1.811324698281e+05, -1.814093951921e+05, -1.816796370111e+05, -1.819434152871e+05, -1.822009393291e+05, -1.824524084331e+05, -1.826980125071e+05, -1.829379326461e+05, -1.831723416721e+05, -1.834014046211e+05, -1.836252792071e+05, -1.838441162441e+05, -1.840580600431e+05, -1.842672487791e+05, -1.844718148321e+05, -1.846718851071e+05, -1.848675813271e+05, -1.850590203181e+05, -1.852463142601e+05, -1.854295709331e+05, -1.856088939451e+05, -1.857843829431e+05, -1.859561338091e+05, -1.861242388521e+05, -1.862887869811e+05, -1.864498638681e+05, -1.866075521081e+05, -1.867619313591e+05, -1.869130784831e+05, -1.870610676741e+05, -1.872059705811e+05, -1.873478564221e+05, -1.874867920961e+05, -1.876228422791e+05, -1.877560695291e+05, -1.878865343711e+05, -1.880142953911e+05, -1.881394093121e+05, -1.882619310761e+05, -1.883819139151e+05, -1.884994094261e+05, -1.886144676311e+05, -1.887271370481e+05, -1.888374647451e+05, -1.889454963981e+05, -1.890512763511e+05, -1.891548476591e+05, -1.892562521451e+05, -1.893555304441e+05, -1.894527220451e+05, -1.895478653391e+05, -1.896409976561e+05, -1.897321553031e+05, -1.898213736071e+05, -1.899086869431e+05, -1.899941287721e+05, -1.900777316721e+05, -1.901595273721e+05, -1.902395467751e+05, -1.903178199931e+05, -1.903943763711e+05, -1.904692445121e+05, -1.905424523061e+05, -1.906140269471e+05, -1.906839949631e+05, -1.907523822341e+05, -1.908192140131e+05, -1.908845149481e+05, -1.909483090991e+05, -1.910106199601e+05, -1.910714704721e+05, -1.911308830451e+05, -1.911888795711e+05, -1.912454814431e+05, -1.913007095671e+05, -1.913545843791e+05, -1.914071258571e+05, -1.914583535391e+05, -1.915082865301e+05, -1.915569435191e+05, -1.916043427891e+05, -1.916505022301e+05, -1.916954393511e+05, -1.917391712891e+05, -1.917817148181e+05, -1.918230863641e+05, -1.918633020111e+05, -1.919023775131e+05, -1.919403283001e+05, -1.919771694891e+05, -1.920129158921e+05, -1.920475820241e+05, -1.920811821121e+05, -1.921137301011e+05, -1.921452396631e+05, -1.921757242021e+05, -1.922051968631e+05, -1.922336705391e+05, -1.922611578761e+05, -1.922876712791e+05, -1.923132229211e+05, -1.923378247441e+05, -1.923614884701e+05, -1.923842256031e+05, -1.924060474351e+05, -1.924269650541e+05, -1.924469893441e+05, -1.924661309931e+05, -1.924844004991e+05, -1.925018081721e+05, -1.925183641381e+05, -1.925340783471e+05, -1.925489605741e+05, -1.925630204221e+05, -1.925762673321e+05, -1.925887105791e+05, -1.926003592811e+05, -1.926112224011e+05, -1.926213087521e+05, -1.926306269961e+05, -1.926391856521e+05, -1.926469931001e+05, -1.926540575761e+05, -1.926603871871e+05, -1.926659899031e+05, -1.926708735681e+05, -1.926750458971e+05, -1.926785144821e+05, -1.926812867951e+05, -1.926833701871e+05, -1.926847718951e+05, -1.926854990421e+05, -1.926855586401e+05, -1.926849575901e+05, -1.926837026891e+05, -1.926818006271e+05, -1.926792579951e+05, -1.926760812821e+05, -1.926722768771e+05, -1.926678510761e+05, -1.926628100781e+05, -1.926571599931e+05, -1.926509068371e+05, -1.926440565391e+05, -1.926366149411e+05, -1.926285878001e+05, -1.926199807881e+05, -1.926107994951e+05, -1.926010494341e+05, -1.925907360351e+05, -1.925798646531e+05, -1.925684405661e+05, -1.925564689781e+05, -1.925439550201e+05, -1.925309037521e+05, -1.925173201621e+05, -1.925032091711e+05, -1.924885756311e+05, -1.924734243271e+05, -1.924577599801e+05, -1.924415872461e+05, -1.924249107201e+05, -1.924077349321e+05, -1.923900643541e+05, -1.923719033971e+05, -1.923532564141e+05, -1.923341277011e+05, -1.923145214961e+05, -1.922944419841e+05, -1.922738932921e+05, -1.922528794981e+05, -1.922314046231e+05, -1.922094726381e+05, -1.921870874641e+05, -1.921642529711e+05, -1.921409729811e+05, -1.921172512661e+05, -1.920930915521e+05, -1.920684975171e+05, -1.920434727951e+05, -1.920180209731e+05, -1.919921455951e+05, -1.919658501611e+05, -1.919391381271e+05, -1.919120129081e+05, -1.918844778781e+05, -1.918565363681e+05, -1.918281916721e+05, -1.917994470401e+05, -1.917703056881e+05, -1.917407707911e+05, -1.917108454851e+05, -1.916805328731e+05, -1.916498360181e+05, -1.916187579481e+05, -1.915873016561e+05, -1.915554701011e+05, -1.915232662051e+05, -1.914906928611e+05, -1.914577529231e+05, -1.914244492171e+05, -1.913907845341e+05, -1.913567616341e+05, -1.913223832481e+05, -1.912876520721e+05, -1.912525707761e+05, -1.912171419961e+05, -1.911813683431e+05, -1.911452523951e+05, -1.911087967041e+05, -1.910720037941e+05, -1.910348761591e+05, -1.909974162681e+05, -1.909596265621e+05, -1.909215094571e+05, -1.908830673401e+05, -1.908443025751e+05, -1.908052174991e+05, -1.907658144251e+05, -1.907260956411e+05, -1.906860634111e+05, -1.906457199741e+05, -1.906050675461e+05, -1.905641083211e+05, -1.905228444671e+05, -1.904812781331e+05, -1.904394114441e+05, -1.903972465021e+05, -1.903547853881e+05, -1.903120301631e+05, -1.902689828641e+05, -1.902256455111e+05, -1.901820201001e+05, -1.901381086081e+05, -1.900939129931e+05, -1.900494351921e+05, -1.900046771231e+05, -1.899596406851e+05, -1.899143277571e+05, -1.898687402011e+05, -1.898228798601e+05, -1.897767485581e+05, -1.897303481011e+05, -1.896836802791e+05, -1.896367468631e+05, -1.895895496071e+05, -1.895420902481e+05, -1.894943705071e+05, -1.894463920871e+05, -1.893981566761e+05, -1.893496659451e+05, -1.893009215481e+05, -1.892519251271e+05, -1.892026783041e+05, -1.891531826881e+05, -1.891034398721e+05, -1.890534514361e+05, -1.890032189421e+05, -1.889527439391e+05, -1.889020279631e+05, -1.888510725331e+05, -1.887998791551e+05, -1.887484493231e+05, -1.886967845151e+05, -1.886448861961e+05, -1.885927558181e+05, -1.885403948191e+05, -1.884878046251e+05, -1.884349866481e+05, -1.883819422881e+05, -1.883286729331e+05, -1.882751799571e+05, -1.882214647221e+05, -1.881675285791e+05, -1.881133728671e+05, -1.880589989111e+05, -1.880044080261e+05, -1.879496015171e+05, -1.878945806731e+05, -1.878393467761e+05, -1.877839010951e+05, -1.877282448881e+05, -1.876723794031e+05, -1.876163058751e+05, -1.875600255311e+05, -1.875035395851e+05, -1.874468492431e+05, -1.873899556991e+05, -1.873328601371e+05, -1.872755637321e+05, -1.872180676471e+05, -1.871603730381e+05, -1.871024810481e+05, -1.870443928141e+05, -1.869861094601e+05, -1.869276321021e+05, -1.868689618481e+05, -1.868100997941e+05, -1.867510470301e+05, -1.866918046351e+05, -1.866323736791e+05, -1.865727552251e+05, -1.865129503251e+05, -1.864529600241e+05, -1.863927853571e+05, -1.863324273531e+05, -1.862718870301e+05, -1.862111654001e+05, -1.861502634651e+05, -1.860891822191e+05, -1.860279226501e+05, -1.859664857371e+05, -1.859048724501e+05, -1.858430837521e+05, -1.857811206001e+05, -1.857189839411e+05, -1.856566747161e+05, -1.855941938591e+05, -1.855315422941e+05, -1.854687209411e+05, -1.854057307101e+05, -1.853425725071e+05, -1.852792472291e+05, -1.852157557651e+05, -1.851520989991e+05, -1.850882778091e+05, -1.850242930631e+05, -1.849601456251e+05, -1.848958363511e+05, -1.848313660921e+05, -1.847667356911e+05, -1.847019459861e+05, -1.846369978061e+05, -1.845718919781e+05, -1.845066293181e+05, -1.844412106391e+05, -1.843756367481e+05, -1.843099084431e+05, -1.842440265211e+05, -1.841779917671e+05, -1.841118049661e+05, -1.840454668921e+05, -1.839789783181e+05, -1.839123400091e+05, -1.838455527221e+05, -1.837786172141e+05, -1.837115342311e+05, -1.836443045181e+05, -1.835769288101e+05, -1.835094078421e+05, -1.834417423381e+05, -1.833739330221e+05, -1.833059806091e+05, -1.832378858111e+05, -1.831696493331e+05, -1.831012718781e+05, -1.830327541401e+05, -1.829640968111e+05, -1.828953005771e+05, -1.828263661201e+05, -1.827572941161e+05, -1.826880852371e+05, -1.826187401491e+05, -1.825492595161e+05, -1.824796439941e+05, -1.824098942371e+05, -1.823400108931e+05, -1.822699946071e+05, -1.821998460171e+05, -1.821295657601e+05, -1.820591544651e+05, -1.819886127591e+05, -1.819179412641e+05, -1.818471405981e+05, -1.817762113751e+05, -1.817051542031e+05, -1.816339696881e+05, -1.815626584311e+05, -1.814912210291e+05, -1.814196580751e+05, -1.813479701581e+05, -1.812761578621e+05},
    {4.658693206387e+04, 4.504706754687e+04, 4.348876547187e+04, 4.191143782087e+04, 4.031446589687e+04, 3.869719815987e+04, 3.705894787687e+04, 3.539899054587e+04, 3.371656109387e+04, 3.201085079387e+04, 3.028100389587e+04, 2.852611391687e+04, 2.674521956587e+04, 2.493730024487e+04, 2.310127108487e+04, 2.123597745487e+04, 1.934018887387e+04, 1.741259224587e+04, 1.545178434587e+04, 1.345626343087e+04, 1.142441987987e+04, 9.354525715870e+03, 7.244722873871e+03, 5.093010008871e+03, 2.897227683871e+03, 6.550416718705e+02, -1.636075852129e+03, -3.978867543129e+03, -6.376314842129e+03, -8.831665001129e+03, -1.134846154313e+04, -1.393057850313e+04, -1.658225876413e+04, -1.930815671013e+04, -2.211338521513e+04, -2.500356661013e+04, -2.798488665413e+04, -3.106414944813e+04, -3.424882944913e+04, -3.754711377413e+04, -4.096792320913e+04, -4.452089252513e+04, -4.821627841913e+04, -5.206474438913e+04, -5.607694435313e+04, -6.026279152413e+04, -6.463026531813e+04, -6.918360593813e+04, -7.392082978313e+04, -7.883072551313e+04, -8.388981727713e+04, -8.905997901013e+04, -9.428727358413e+04, -9.950256566113e+04, -1.046251198331e+05, -1.095708259701e+05, -1.142645881131e+05, -1.186532881671e+05, -1.227123756451e+05, -1.264381174251e+05, -1.298429322251e+05, -1.329490058941e+05, -1.357829950301e+05, -1.383725642101e+05, -1.407443973011e+05, -1.429231739701e+05, -1.449311256711e+05, -1.467879307771e+05, -1.485108053651e+05, -1.501147004461e+05, -1.516125478101e+05, -1.530155172741e+05, -1.543332629391e+05, -1.555741466641e+05, -1.567454339741e+05, -1.578534619811e+05, -1.589037812261e+05, -1.599012744421e+05, -1.608502555221e+05, -1.617545518601e+05, -1.626175729191e+05, -1.634423674571e+05, -1.642316714641e+05, -1.649879484991e+05, -1.657134237881e+05, -1.664101132011e+05, -1.670798479931e+05, -1.677242960391e+05, -1.683449801431e+05, -1.689432938841e+05, -1.695205154071e+05, -1.700778194471e+05, -1.706162878681e+05, -1.711369189241e+05, -1.716406354131e+05, -1.721282918991e+05, -1.726006810981e+05, -1.730585395631e+05, -1.735025527401e+05, -1.739333594821e+05, -1.743515560941e+05, -1.747576999471e+05, -1.751523127351e+05, -1.755358833991e+05, -1.759088707681e+05, -1.762717059401e+05, -1.766247944381e+05, -1.769685181681e+05, -1.773032371891e+05, -1.776292913311e+05, -1.779470016581e+05, -1.782566718161e+05, -1.785585892501e+05, -1.788530263281e+05, -1.791402413681e+05, -1.794204795751e+05, -1.796939739081e+05, -1.799609458761e+05, -1.802216062701e+05, -1.804761558341e+05, -1.807247858971e+05, -1.809676789411e+05, -1.812050091361e+05, -1.814369428401e+05, -1.816636390471e+05, -1.818852498211e+05, -1.821019206881e+05, -1.823137910001e+05, -1.825209942841e+05, -1.827236585571e+05, -1.829219066241e+05, -1.831158563561e+05, -1.833056209511e+05, -1.834913091761e+05, -1.836730255951e+05, -1.838508707811e+05, -1.840249415191e+05, -1.841953309891e+05, -1.843621289461e+05, -1.845254218841e+05, -1.846852931921e+05, -1.848418233021e+05, -1.849950898221e+05, -1.851451676751e+05, -1.852921292141e+05, -1.854360443401e+05, -1.855769806151e+05, -1.857150033601e+05, -1.858501757561e+05, -1.859825589371e+05, -1.861122120751e+05, -1.862391924661e+05, -1.863635556061e+05, -1.864853552691e+05, -1.866046435721e+05, -1.867214710501e+05, -1.868358867121e+05, -1.869479381081e+05, -1.870576713811e+05, -1.871651313291e+05, -1.872703614491e+05, -1.873734039931e+05, -1.874743000131e+05, -1.875730894061e+05, -1.876698109571e+05, -1.877645023801e+05, -1.878572003591e+05, -1.879479405831e+05, -1.880367577831e+05, -1.881236857661e+05, -1.882087574471e+05, -1.882920048811e+05, -1.883734592911e+05, -1.884531511021e+05, -1.885311099601e+05, -1.886073647661e+05, -1.886819436961e+05, -1.887548742271e+05, -1.888261831601e+05, -1.888958966421e+05, -1.889640401881e+05, -1.890306387001e+05, -1.890957164861e+05, -1.891592972811e+05, -1.892214042661e+05, -1.892820600791e+05, -1.893412868381e+05, -1.893991061571e+05, -1.894555391561e+05, -1.895106064821e+05, -1.895643283181e+05, -1.896167244011e+05, -1.896678140331e+05, -1.897176160951e+05, -1.897661490561e+05, -1.898134309911e+05, -1.898594795851e+05, -1.899043121511e+05, -1.899479456351e+05, -1.899903966301e+05, -1.900316813841e+05, -1.900718158101e+05, -1.901108154961e+05, -1.901486957111e+05, -1.901854714181e+05, -1.902211572791e+05, -1.902557676641e+05, -1.902893166601e+05, -1.903218180751e+05, -1.903532854511e+05, -1.903837320631e+05, -1.904131709341e+05, -1.904416148381e+05, -1.904690763031e+05, -1.904955676241e+05, -1.905211008641e+05, -1.905456878611e+05, -1.905693402331e+05, -1.905920693861e+05, -1.906138865171e+05, -1.906348026191e+05, -1.906548284861e+05, -1.906739747191e+05, -1.906922517311e+05, -1.907096697481e+05, -1.907262388181e+05, -1.907419688121e+05, -1.907568694291e+05, -1.907709502011e+05, -1.907842204941e+05, -1.907966895181e+05, -1.908083663211e+05, -1.908192598021e+05, -1.908293787091e+05, -1.908387316441e+05, -1.908473270671e+05, -1.908551732961e+05, -1.908622785151e+05, -1.908686507721e+05, -1.908742979871e+05, -1.908792279491e+05, -1.908834483251e+05, -1.908869666561e+05, -1.908897903661e+05, -1.908919267611e+05, -1.908933830321e+05, -1.908941662581e+05, -1.908942834061e+05, -1.908937413381e+05, -1.908925468101e+05, -1.908907064721e+05, -1.908882268761e+05, -1.908851144721e+05, -1.908813756171e+05, -1.908770165671e+05, -1.908720434901e+05, -1.908664624591e+05, -1.908602794581e+05, -1.908535003861e+05, -1.908461310521e+05, -1.908381771821e+05, -1.908296444191e+05, -1.908205383271e+05, -1.908108643871e+05, -1.908006280021e+05, -1.907898345021e+05, -1.907784891391e+05, -1.907665970901e+05, -1.907541634621e+05, -1.907411932911e+05, -1.907276915421e+05, -1.907136631121e+05, -1.906991128311e+05, -1.906840454621e+05, -1.906684657061e+05, -1.906523781971e+05, -1.906357875101e+05, -1.906186981551e+05, -1.906011145851e+05, -1.905830411921e+05, -1.905644823111e+05, -1.905454422201e+05, -1.905259251391e+05, -1.905059352371e+05, -1.904854766241e+05, -1.904645533601e+05, -1.904431694531e+05, -1.904213288571e+05, -1.903990354781e+05, -1.903762931711e+05, -1.903531057441e+05, -1.903294769551e+05, -1.903054105171e+05, -1.902809100931e+05, -1.902559793051e+05, -1.902306217271e+05, -1.902048408911e+05, -1.901786402831e+05, -1.901520233491e+05, -1.901249934931e+05, -1.900975540751e+05, -1.900697084171e+05, -1.900414598011e+05, -1.900128114691e+05, -1.899837666231e+05, -1.899543284291e+05, -1.899245000161e+05, -1.898942844741e+05, -1.898636848571e+05, -1.898327041861e+05, -1.898013454431e+05, -1.897696115791e+05, -1.897375055081e+05, -1.897050301121e+05, -1.896721882411e+05, -1.896389827101e+05, -1.896054163031e+05, -1.895714917731e+05, -1.895372118421e+05, -1.895025792011e+05, -1.894675965101e+05, -1.894322664011e+05, -1.893965914761e+05, -1.893605743081e+05, -1.893242174421e+05, -1.892875233951e+05, -1.892504946551e+05, -1.892131336861e+05, -1.891754429221e+05, -1.891374247731e+05, -1.890990816221e+05, -1.890604158251e+05, -1.890214297161e+05, -1.889821256021e+05, -1.889425057661e+05, -1.889025724661e+05, -1.888623279371e+05, -1.888217743901e+05, -1.887809140131e+05, -1.887397489731e+05, -1.886982814111e+05, -1.886565134491e+05, -1.886144471841e+05, -1.885720846941e+05, -1.885294280351e+05, -1.884864792421e+05, -1.884432403281e+05, -1.883997132871e+05, -1.883559000921e+05, -1.883118026971e+05, -1.882674230371e+05, -1.882227630241e+05, -1.881778245551e+05, -1.881326095071e+05, -1.880871197371e+05, -1.880413570851e+05, -1.879953233721e+05, -1.879490204021e+05, -1.879024499611e+05, -1.878556138181e+05, -1.878085137231e+05, -1.877611514121e+05, -1.877135286011e+05, -1.876656469931e+05, -1.876175082721e+05, -1.875691141061e+05, -1.875204661481e+05, -1.874715660371e+05, -1.874224153921e+05, -1.873730158211e+05, -1.873233689151e+05, -1.872734762491e+05, -1.872233393861e+05, -1.871729598731e+05, -1.871223392411e+05, -1.870714790111e+05, -1.870203806851e+05, -1.869690457551e+05, -1.869174756971e+05, -1.868656719761e+05, -1.868136360401e+05, -1.867613693271e+05, -1.867088732601e+05, -1.866561492521e+05, -1.866031986991e+05, -1.865500229871e+05, -1.864966234911e+05, -1.864430015711e+05, -1.863891585751e+05, -1.863350958421e+05, -1.862808146951e+05, -1.862263164491e+05, -1.861716024061e+05, -1.861166738551e+05, -1.860615320771e+05, -1.860061783391e+05, -1.859506138981e+05, -1.858948400011e+05, -1.858388578831e+05, -1.857826687681e+05, -1.857262738711e+05, -1.856696743961e+05, -1.856128715361e+05, -1.855558664751e+05, -1.854986603851e+05, -1.854412544321e+05, -1.853836497671e+05, -1.853258475351e+05, -1.852678488701e+05, -1.852096548971e+05, -1.851512667321e+05, -1.850926854801e+05, -1.850339122381e+05, -1.849749480951e+05, -1.849157941281e+05, -1.848564514091e+05, -1.847969209971e+05, -1.847372039471e+05, -1.846773013011e+05, -1.846172140951e+05, -1.845569433561e+05, -1.844964901031e+05, -1.844358553471e+05, -1.843750400881e+05, -1.843140453231e+05, -1.842528720361e+05, -1.841915212071e+05, -1.841299938071e+05, -1.840682907971e+05, -1.840064131341e+05, -1.839443617651e+05, -1.838821376301e+05, -1.838197416621e+05, -1.837571747871e+05, -1.836944379231e+05, -1.836315319821e+05, -1.835684578661e+05, -1.835052164741e+05, -1.834418086961e+05, -1.833782354141e+05, -1.833144975051e+05, -1.832505958391e+05, -1.831865312781e+05, -1.831223046801e+05, -1.830579168941e+05, -1.829933687641e+05, -1.829286611261e+05, -1.828637948111e+05, -1.827987706451e+05, -1.827335894441e+05, -1.826682520211e+05, -1.826027591831e+05, -1.825371117281e+05, -1.824713104521e+05, -1.824053561421e+05, -1.823392495811e+05, -1.822729915451e+05, -1.822065828041e+05, -1.821400241251e+05, -1.820733162661e+05, -1.820064599811e+05, -1.819394560191e+05, -1.818723051231e+05, -1.818050080301e+05, -1.817375654731e+05, -1.816699781781e+05, -1.816022468671e+05, -1.815343722571e+05, -1.814663550581e+05, -1.813981959781e+05, -1.813298957171e+05, -1.812614549711e+05, -1.811928744321e+05, -1.811241547861e+05, -1.810552967151e+05, -1.809863008941e+05, -1.809171679961e+05, -1.808478986891e+05, -1.807784936331e+05, -1.807089534891e+05, -1.806392789081e+05, -1.805694705391e+05, -1.804995290261e+05, -1.804294550101e+05, -1.803592491251e+05, -1.802889120021e+05, -1.802184442691e+05, -1.801478465461e+05, -1.800771194521e+05, -1.800062636021e+05, -1.799352796031e+05, -1.798641680631e+05, -1.797929295821e+05, -1.797215647571e+05, -1.796500741811e+05},
    {4.748409659787e+04, 4.595453504787e+04, 4.440691075487e+04, 4.284065741687e+04, 4.125517976387e+04, 3.964985156287e+04, 3.802401345387e+04, 3.637697059187e+04, 3.470799007487e+04, 3.301629813187e+04, 3.130107705187e+04, 2.956146181787e+04, 2.779653641387e+04, 2.600532976687e+04, 2.418681127687e+04, 2.233988589687e+04, 2.046338869187e+04, 1.855607881987e+04, 1.661663287287e+04, 1.464363747287e+04, 1.263558105287e+04, 1.059084469887e+04, 8.507691945871e+03, 6.384257376871e+03, 4.218533889871e+03, 2.008358442870e+03, -2.486039011295e+02, -2.554877835129e+03, -4.913197813129e+03, -7.326530401129e+03, -9.798099239129e+03, -1.233141277513e+04, -1.493029495713e+04, -1.759891889013e+04, -2.034184331313e+04, -2.316405139313e+04, -2.607099079213e+04, -2.906861306713e+04, -3.216340904413e+04, -3.536243455513e+04, -3.867331735113e+04, -4.210423049513e+04, -4.566380898813e+04, -4.936097369613e+04, -5.320460866413e+04, -5.720301506513e+04, -6.136304233813e+04, -6.568879003813e+04, -7.017981398513e+04, -7.482889715013e+04, -7.961966291313e+04, -8.452451500013e+04, -8.950341518513e+04, -9.450387870313e+04, -9.946264146013e+04, -1.043098429551e+05, -1.089763494911e+05, -1.134030609601e+05, -1.175497306801e+05, -1.213981540121e+05, -1.249464903361e+05, -1.282057032341e+05, -1.311945759241e+05, -1.339356143581e+05, -1.364523117291e+05, -1.387675361831e+05, -1.409026703161e+05, -1.428772074561e+05, -1.447086177091e+05, -1.464123736741e+05, -1.480020691101e+05, -1.494895872201e+05, -1.508852892541e+05, -1.521982041571e+05, -1.534362074511e+05, -1.546061831261e+05, -1.557141660971e+05, -1.567654652001e+05, -1.577647680531e+05, -1.587162297451e+05, -1.596235475331e+05, -1.604900236431e+05, -1.613186181041e+05, -1.621119932811e+05, -1.628725515301e+05, -1.636024671601e+05, -1.643037136771e+05, -1.649780871181e+05, -1.656272261171e+05, -1.662526292481e+05, -1.668556700731e+05, -1.674376102441e+05, -1.679996109791e+05, -1.685427431101e+05, -1.690679959551e+05, -1.695762851311e+05, -1.700684594941e+05, -1.705453072851e+05, -1.710075616141e+05, -1.714559053471e+05, -1.718909754821e+05, -1.723133670621e+05, -1.727236367021e+05, -1.731223057481e+05, -1.735098631401e+05, -1.738867679821e+05, -1.742534518791e+05, -1.746103210531e+05, -1.749577582521e+05, -1.752961245031e+05, -1.756257606971e+05, -1.759469890351e+05, -1.762601143511e+05, -1.765654253221e+05, -1.768631955761e+05, -1.771536847051e+05, -1.774371392001e+05, -1.777137933051e+05, -1.779838698101e+05, -1.782475807751e+05, -1.785051282051e+05, -1.787567046651e+05, -1.790024938591e+05, -1.792426711571e+05, -1.794774040881e+05, -1.797068528011e+05, -1.799311704821e+05, -1.801505037561e+05, -1.803649930491e+05, -1.805747729311e+05, -1.807799724401e+05, -1.809807153721e+05, -1.811771205641e+05, -1.813693021531e+05, -1.815573698221e+05, -1.817414290221e+05, -1.819215811911e+05, -1.820979239521e+05, -1.822705513031e+05, -1.824395537891e+05, -1.826050186771e+05, -1.827670301031e+05, -1.829256692241e+05, -1.830810143581e+05, -1.832331411101e+05, -1.833821225001e+05, -1.835280290791e+05, -1.836709290341e+05, -1.838108882981e+05, -1.839479706471e+05, -1.840822377901e+05, -1.842137494601e+05, -1.843425634981e+05, -1.844687359301e+05, -1.845923210411e+05, -1.847133714531e+05, -1.848319381841e+05, -1.849480707171e+05, -1.850618170621e+05, -1.851732238111e+05, -1.852823361951e+05, -1.853891981361e+05, -1.854938522981e+05, -1.855963401331e+05, -1.856967019291e+05, -1.857949768511e+05, -1.858912029841e+05, -1.859854173711e+05, -1.860776560531e+05, -1.861679541021e+05, -1.862563456581e+05, -1.863428639611e+05, -1.864275413811e+05, -1.865104094521e+05, -1.865914988971e+05, -1.866708396571e+05, -1.867484609161e+05, -1.868243911291e+05, -1.868986580471e+05, -1.869712887341e+05, -1.870423095971e+05, -1.871117464031e+05, -1.871796242981e+05, -1.872459678341e+05, -1.873108009791e+05, -1.873741471421e+05, -1.874360291861e+05, -1.874964694481e+05, -1.875554897551e+05, -1.876131114351e+05, -1.876693553381e+05, -1.877242418481e+05, -1.877777908961e+05, -1.878300219741e+05, -1.878809541481e+05, -1.879306060701e+05, -1.879789959921e+05, -1.880261417741e+05, -1.880720608971e+05, -1.881167704761e+05, -1.881602872641e+05, -1.882026276701e+05, -1.882438077611e+05, -1.882838432791e+05, -1.883227496421e+05, -1.883605419591e+05, -1.883972350351e+05, -1.884328433811e+05, -1.884673812191e+05, -1.885008624931e+05, -1.885333008751e+05, -1.885647097711e+05, -1.885951023291e+05, -1.886244914471e+05, -1.886528897741e+05, -1.886803097251e+05, -1.887067634771e+05, -1.887322629841e+05, -1.887568199761e+05, -1.887804459691e+05, -1.888031522661e+05, -1.888249499661e+05, -1.888458499681e+05, -1.888658629741e+05, -1.888849994961e+05, -1.889032698581e+05, -1.889206842041e+05, -1.889372524981e+05, -1.889529845331e+05, -1.889678899301e+05, -1.889819781461e+05, -1.889952584751e+05, -1.890077400551e+05, -1.890194318661e+05, -1.890303427411e+05, -1.890404813611e+05, -1.890498562661e+05, -1.890584758541e+05, -1.890663483841e+05, -1.890734819821e+05, -1.890798846391e+05, -1.890855642191e+05, -1.890905284591e+05, -1.890947849731e+05, -1.890983412531e+05, -1.891012046721e+05, -1.891033824901e+05, -1.891048818491e+05, -1.891057097831e+05, -1.891058732171e+05, -1.891053789681e+05, -1.891042337501e+05, -1.891024441731e+05, -1.891000167491e+05, -1.890969578901e+05, -1.890932739131e+05, -1.890889710401e+05, -1.890840554011e+05, -1.890785330361e+05, -1.890724098941e+05, -1.890656918411e+05, -1.890583846531e+05, -1.890504940251e+05, -1.890420255711e+05, -1.890329848221e+05, -1.890233772321e+05, -1.890132081761e+05, -1.890024829541e+05, -1.889912067921e+05, -1.889793848421e+05, -1.889670221841e+05, -1.889541238291e+05, -1.889406947171e+05, -1.889267397221e+05, -1.889122636511e+05, -1.888972712441e+05, -1.888817671781e+05, -1.888657560691e+05, -1.888492424671e+05, -1.888322308641e+05, -1.888147256921e+05, -1.887967313241e+05, -1.887782520751e+05, -1.887592922041e+05, -1.887398559151e+05, -1.887199473551e+05, -1.886995706211e+05, -1.886787297541e+05, -1.886574287441e+05, -1.886356715321e+05, -1.886134620061e+05, -1.885908040071e+05, -1.885677013261e+05, -1.885441577071e+05, -1.885201768491e+05, -1.884957624011e+05, -1.884709179711e+05, -1.884456471191e+05, -1.884199533651e+05, -1.883938401821e+05, -1.883673110031e+05, -1.883403692191e+05, -1.883130181801e+05, -1.882852611951e+05, -1.882571015351e+05, -1.882285424301e+05, -1.881995870731e+05, -1.881702386181e+05, -1.881405001831e+05, -1.881103748481e+05, -1.880798656601e+05, -1.880489756261e+05, -1.880177077221e+05, -1.879860648871e+05, -1.879540500271e+05, -1.879216660171e+05, -1.878889156941e+05, -1.878558018691e+05, -1.878223273151e+05, -1.877884947781e+05, -1.877543069721e+05, -1.877197665791e+05, -1.876848762541e+05, -1.876496386191e+05, -1.876140562711e+05, -1.875781317741e+05, -1.875418676671e+05, -1.875052664591e+05, -1.874683306341e+05, -1.874310626471e+05, -1.873934649281e+05, -1.873555398791e+05, -1.873172898771e+05, -1.872787172731e+05, -1.872398243951e+05, -1.872006135441e+05, -1.871610869971e+05, -1.871212470071e+05, -1.870810958051e+05, -1.870406355961e+05, -1.869998685631e+05, -1.869587968661e+05, -1.869174226441e+05, -1.868757480131e+05, -1.868337750661e+05, -1.867915058771e+05, -1.867489424951e+05, -1.867060869531e+05, -1.866629412591e+05, -1.866195074041e+05, -1.865757873561e+05, -1.865317830651e+05, -1.864874964611e+05, -1.864429294541e+05, -1.863980839381e+05, -1.863529617841e+05, -1.863075648471e+05, -1.862618949631e+05, -1.862159539501e+05, -1.861697436091e+05, -1.861232657221e+05, -1.860765220541e+05, -1.860295143551e+05, -1.859822443551e+05, -1.859347137701e+05, -1.858869242971e+05, -1.858388776181e+05, -1.857905754001e+05, -1.857420192921e+05, -1.856932109291e+05, -1.856441519311e+05, -1.855948439011e+05, -1.855452884271e+05, -1.854954870841e+05, -1.854454414311e+05, -1.853951530121e+05, -1.853446233591e+05, -1.852938539861e+05, -1.852428463981e+05, -1.851916020811e+05, -1.851401225121e+05, -1.850884091511e+05, -1.850364634461e+05, -1.849842868331e+05, -1.849318807331e+05, -1.848792465561e+05, -1.848263856991e+05, -1.847732995441e+05, -1.847199894651e+05, -1.846664568201e+05, -1.846127029571e+05, -1.845587292111e+05, -1.845045369061e+05, -1.844501273531e+05, -1.843955018541e+05, -1.843406616981e+05, -1.842856081611e+05, -1.842303425121e+05, -1.841748660061e+05, -1.841191798871e+05, -1.840632853901e+05, -1.840071837391e+05, -1.839508761461e+05, -1.838943638151e+05, -1.838376479371e+05, -1.837807296961e+05, -1.837236102621e+05, -1.836662908001e+05, -1.836087724611e+05, -1.835510563881e+05, -1.834931437141e+05, -1.834350355641e+05, -1.833767330521e+05, -1.833182372841e+05, -1.832595493541e+05, -1.832006703511e+05, -1.831416013531e+05, -1.830823434281e+05, -1.830228976371e+05, -1.829632650321e+05, -1.829034466571e+05, -1.828434435451e+05, -1.827832567231e+05, -1.827228872091e+05, -1.826623360141e+05, -1.826016041391e+05, -1.825406925771e+05, -1.824796023141e+05, -1.824183343291e+05, -1.823568895901e+05, -1.822952690621e+05, -1.822334736981e+05, -1.821715044461e+05, -1.821093622461e+05, -1.820470480291e+05, -1.819845627221e+05, -1.819219072421e+05, -1.818590825001e+05, -1.817960893991e+05, -1.817329288361e+05, -1.816696017021e+05, -1.816061088781e+05, -1.815424512421e+05, -1.814786296621e+05, -1.814146450011e+05, -1.813504981161e+05, -1.812861898551e+05, -1.812217210631e+05, -1.811570925751e+05, -1.810923052241e+05, -1.810273598321e+05, -1.809622572171e+05, -1.808969981931e+05, -1.808315835641e+05, -1.807660141311e+05, -1.807002906861e+05, -1.806344140191e+05, -1.805683849121e+05, -1.805022041401e+05, -1.804358724751e+05, -1.803693906821e+05, -1.803027595191e+05, -1.802359797421e+05, -1.801690520971e+05, -1.801019773281e+05, -1.800347561731e+05, -1.799673893641e+05, -1.798998776271e+05, -1.798322216841e+05, -1.797644222521e+05, -1.796964800421e+05, -1.796283957611e+05, -1.795601701081e+05, -1.794918037821e+05, -1.794232974721e+05, -1.793546518651e+05, -1.792858676431e+05, -1.792169454821e+05, -1.791478860551e+05, -1.790786900281e+05, -1.790093580641e+05, -1.789398908211e+05, -1.788702889521e+05, -1.788005531071e+05, -1.787306839281e+05, -1.786606820571e+05, -1.785905481281e+05, -1.785202827721e+05, -1.784498866171e+05, -1.783793602841e+05, -1.783087043921e+05, -1.782379195541e+05, -1.781670063811e+05, -1.780959654771e+05, -1.780247974441e+05},
    {4.838222595887e+04, 4.686285464287e+04, 4.532578702687e+04, 4.377047770387e+04, 4.219635391687e+04, 4.060281372387e+04, 3.898922400387e+04, 3.735491829587e+04, 3.569919444587e+04, 3.402131204087e+04, 3.232048961087e+04, 3.059590157687e+04, 2.884667490787e+04, 2.707188545787e+04, 2.527055394987e+04, 2.344164156487e+04, 2.158404508287e+04, 1.969659153587e+04, 1.777803230487e+04, 1.582703659287e+04, 1.384218420887e+04, 1.182195757387e+04, 9.764732847871e+03, 7.668770085871e+03, 5.532202299871e+03, 3.353023297871e+03, 1.129074163871e+03, -1.141971761130e+03, -3.462625556129e+03, -5.835601202130e+03, -8.263835992129e+03, -1.075051297413e+04, -1.329908546213e+04, -1.591330352413e+04, -1.859724219313e+04, -2.135533082513e+04, -2.419238256913e+04, -2.711362219613e+04, -3.012470935813e+04, -3.323175268113e+04, -3.644130737113e+04, -3.976034507713e+04, -4.319617875613e+04, -4.675631664313e+04, -5.044820749313e+04, -5.427882429113e+04, -5.825401838413e+04, -6.237756909613e+04, -6.664987298813e+04, -7.106628736013e+04, -7.561527860413e+04, -8.027669074913e+04, -8.502053972113e+04, -8.980667920013e+04, -9.458558860713e+04, -9.930061996813e+04, -1.038921691701e+05, -1.083037743081e+05, -1.124889024671e+05, -1.164169038601e+05, -1.200742017351e+05, -1.234602753541e+05, -1.265848061041e+05, -1.294637904451e+05, -1.321163474111e+05, -1.345625399291e+05, -1.368220563901e+05, -1.389134834881e+05, -1.408539445431e+05, -1.426589554641e+05, -1.443424112061e+05, -1.459166512411e+05, -1.473925712931e+05, -1.487797588561e+05, -1.500866367391e+05, -1.513206040391e+05, -1.524881679891e+05, -1.535950632291e+05, -1.546463571871e+05, -1.556465416991e+05, -1.565996117651e+05, -1.575091327771e+05, -1.583782976801e+05, -1.592099754871e+05, -1.600067524771e+05, -1.607709672281e+05, -1.615047405001e+05, -1.622100007951e+05, -1.628885063231e+05, -1.635418639361e+05, -1.641715455301e+05, -1.647789023081e+05, -1.653651772251e+05, -1.659315158961e+05, -1.664789761851e+05, -1.670085366651e+05, -1.675211041001e+05, -1.680175200921e+05, -1.684985669851e+05, -1.689649731421e+05, -1.694174176501e+05, -1.698565345501e+05, -1.702829166231e+05, -1.706971188041e+05, -1.710996612541e+05, -1.714910321401e+05, -1.718716901401e+05, -1.722420667261e+05, -1.726025682221e+05, -1.729535776881e+05, -1.732954566231e+05, -1.736285465251e+05, -1.739531703181e+05, -1.742696336451e+05, -1.745782260651e+05, -1.748792221421e+05, -1.751728824491e+05, -1.754594544871e+05, -1.757391735361e+05, -1.760122634341e+05, -1.762789372971e+05, -1.765393981871e+05, -1.767938397281e+05, -1.770424466711e+05, -1.772853954271e+05, -1.775228545551e+05, -1.777549852141e+05, -1.779819415901e+05, -1.782038712881e+05, -1.784209156941e+05, -1.786332103221e+05, -1.788408851311e+05, -1.790440648181e+05, -1.792428691031e+05, -1.794374129821e+05, -1.796278069771e+05, -1.798141573591e+05, -1.799965663621e+05, -1.801751323881e+05, -1.803499501911e+05, -1.805211110561e+05, -1.806887029651e+05, -1.808528107551e+05, -1.810135162641e+05, -1.811708984701e+05, -1.813250336241e+05, -1.814759953721e+05, -1.816238548731e+05, -1.817686809101e+05, -1.819105399921e+05, -1.820494964551e+05, -1.821856125561e+05, -1.823189485591e+05, -1.824495628211e+05, -1.825775118711e+05, -1.827028504851e+05, -1.828256317591e+05, -1.829459071761e+05, -1.830637266701e+05, -1.831791386891e+05, -1.832921902531e+05, -1.834029270101e+05, -1.835113932851e+05, -1.836176321381e+05, -1.837216854051e+05, -1.838235937481e+05, -1.839233966971e+05, -1.840211326901e+05, -1.841168391191e+05, -1.842105523601e+05, -1.843023078141e+05, -1.843921399421e+05, -1.844800822951e+05, -1.845661675491e+05, -1.846504275331e+05, -1.847328932601e+05, -1.848135949531e+05, -1.848925620731e+05, -1.849698233441e+05, -1.850454067761e+05, -1.851193396911e+05, -1.851916487461e+05, -1.852623599501e+05, -1.853314986901e+05, -1.853990897471e+05, -1.854651573181e+05, -1.855297250331e+05, -1.855928159721e+05, -1.856544526831e+05, -1.857146571991e+05, -1.857734510521e+05, -1.858308552881e+05, -1.858868904841e+05, -1.859415767591e+05, -1.859949337901e+05, -1.860469808241e+05, -1.860977366881e+05, -1.861472198071e+05, -1.861954482101e+05, -1.862424395431e+05, -1.862882110811e+05, -1.863327797371e+05, -1.863761620731e+05, -1.864183743101e+05, -1.864594323351e+05, -1.864993517121e+05, -1.865381476931e+05, -1.865758352221e+05, -1.866124289441e+05, -1.866479432171e+05, -1.866823921151e+05, -1.867157894381e+05, -1.867481487161e+05, -1.867794832231e+05, -1.868098059741e+05, -1.868391297381e+05, -1.868674670451e+05, -1.868948301871e+05, -1.869212312281e+05, -1.869466820071e+05, -1.869711941461e+05, -1.869947790551e+05, -1.870174479361e+05, -1.870392117871e+05, -1.870600814091e+05, -1.870800674131e+05, -1.870991802171e+05, -1.871174300591e+05, -1.871348269951e+05, -1.871513809071e+05, -1.871671015061e+05, -1.871819983351e+05, -1.871960807741e+05, -1.872093580431e+05, -1.872218392071e+05, -1.872335331761e+05, -1.872444487131e+05, -1.872545944361e+05, -1.872639788181e+05, -1.872726101941e+05, -1.872804967631e+05, -1.872876465901e+05, -1.872940676101e+05, -1.872997676301e+05, -1.873047543311e+05, -1.873090352751e+05, -1.873126179011e+05, -1.873155095331e+05, -1.873177173801e+05, -1.873192485371e+05, -1.873201099921e+05, -1.873203086231e+05, -1.873198512041e+05, -1.873187444051e+05, -1.873169947951e+05, -1.873146088441e+05, -1.873115929251e+05, -1.873079533161e+05, -1.873036962001e+05, -1.872988276721e+05, -1.872933537331e+05, -1.872872803011e+05, -1.872806132041e+05, -1.872733581881e+05, -1.872655209141e+05, -1.872571069631e+05, -1.872481218371e+05, -1.872385709581e+05, -1.872284596741e+05, -1.872177932551e+05, -1.872065768981e+05, -1.871948157301e+05, -1.871825148031e+05, -1.871696791011e+05, -1.871563135411e+05, -1.871424229701e+05, -1.871280121721e+05, -1.871130858631e+05, -1.870976486991e+05, -1.870817052701e+05, -1.870652601061e+05, -1.870483176781e+05, -1.870308823961e+05, -1.870129586131e+05, -1.869945506251e+05, -1.869756626701e+05, -1.869562989341e+05, -1.869364635451e+05, -1.869161605821e+05, -1.868953940681e+05, -1.868741679771e+05, -1.868524862301e+05, -1.868303527021e+05, -1.868077712151e+05, -1.867847455471e+05, -1.867612794251e+05, -1.867373765321e+05, -1.867130405041e+05, -1.866882749341e+05, -1.866630833691e+05, -1.866374693121e+05, -1.866114362271e+05, -1.865849875311e+05, -1.865581266021e+05, -1.865308567791e+05, -1.865031813591e+05, -1.864751035981e+05, -1.864466267181e+05, -1.864177538981e+05, -1.863884882821e+05, -1.863588329771e+05, -1.863287910521e+05, -1.862983655431e+05, -1.862675594481e+05, -1.862363757321e+05, -1.862048173241e+05, -1.861728871231e+05, -1.861405879901e+05, -1.861079227581e+05, -1.860748942241e+05, -1.860415051561e+05, -1.860077582901e+05, -1.859736563321e+05, -1.859392019561e+05, -1.859043978071e+05, -1.858692465031e+05, -1.858337506291e+05, -1.857979127441e+05, -1.857617353791e+05, -1.857252210371e+05, -1.856883721941e+05, -1.856511912971e+05, -1.856136807711e+05, -1.855758430101e+05, -1.855376803861e+05, -1.854991952441e+05, -1.854603899041e+05, -1.854212666621e+05, -1.853818277891e+05, -1.853420755321e+05, -1.853020121171e+05, -1.852616397421e+05, -1.852209605861e+05, -1.851799768051e+05, -1.851386905301e+05, -1.850971038721e+05, -1.850552189221e+05, -1.850130377451e+05, -1.849705623901e+05, -1.849277948811e+05, -1.848847372251e+05, -1.848413914051e+05, -1.847977593881e+05, -1.847538431181e+05, -1.847096445211e+05, -1.846651655041e+05, -1.846204079551e+05, -1.845753737421e+05, -1.845300647171e+05, -1.844844827111e+05, -1.844386295401e+05, -1.843925070001e+05, -1.843461168701e+05, -1.842994609121e+05, -1.842525408721e+05, -1.842053584771e+05, -1.841579154381e+05, -1.841102134521e+05, -1.840622541971e+05, -1.840140393351e+05, -1.839655705151e+05, -1.839168493681e+05, -1.838678775101e+05, -1.838186565421e+05, -1.837691880501e+05, -1.837194736061e+05, -1.836695147661e+05, -1.836193130721e+05, -1.835688700531e+05, -1.835181872221e+05, -1.834672660801e+05, -1.834161081121e+05, -1.833647147911e+05, -1.833130875761e+05, -1.832612279131e+05, -1.832091372361e+05, -1.831568169641e+05, -1.831042685051e+05, -1.830514932521e+05, -1.829984925891e+05, -1.829452678841e+05, -1.828918204961e+05, -1.828381517701e+05, -1.827842630391e+05, -1.827301556271e+05, -1.826758308421e+05, -1.826212899851e+05, -1.825665343421e+05, -1.825115651901e+05, -1.824563837941e+05, -1.824009914081e+05, -1.823453892761e+05, -1.822895786311e+05, -1.822335606941e+05, -1.821773366781e+05, -1.821209077841e+05, -1.820642752031e+05, -1.820074401171e+05, -1.819504036951e+05, -1.818931671011e+05, -1.818357314841e+05, -1.817780979871e+05, -1.817202677421e+05, -1.816622418711e+05, -1.816040214901e+05, -1.815456077001e+05, -1.814870015991e+05, -1.814282042711e+05, -1.813692167941e+05, -1.813100402361e+05, -1.812506756571e+05, -1.811911241071e+05, -1.811313866301e+05, -1.810714642581e+05, -1.810113580181e+05, -1.809510689271e+05, -1.808905979941e+05, -1.808299462201e+05, -1.807691145981e+05, -1.807081041131e+05, -1.806469157421e+05, -1.805855504541e+05, -1.805240092121e+05, -1.804622929701e+05, -1.804004026751e+05, -1.803383392641e+05, -1.802761036721e+05, -1.802136968211e+05, -1.801511196301e+05, -1.800883730091e+05, -1.800254578601e+05, -1.799623750811e+05, -1.798991255601e+05, -1.798357101801e+05, -1.797721298171e+05, -1.797083853401e+05, -1.796444776101e+05, -1.795804074841e+05, -1.795161758121e+05, -1.794517834351e+05, -1.793872311901e+05, -1.793225199091e+05, -1.792576504131e+05, -1.791926235221e+05, -1.791274400471e+05, -1.790621007931e+05, -1.789966065601e+05, -1.789309581421e+05, -1.788651563271e+05, -1.787992018951e+05, -1.787330956251e+05, -1.786668382851e+05, -1.786004306411e+05, -1.785338734521e+05, -1.784671674721e+05, -1.784003134491e+05, -1.783333121261e+05, -1.782661642391e+05, -1.781988705221e+05, -1.781314317001e+05, -1.780638484961e+05, -1.779961216261e+05, -1.779282518001e+05, -1.778602397261e+05, -1.777920861031e+05, -1.777237916291e+05, -1.776553569951e+05, -1.775867828861e+05, -1.775180699851e+05, -1.774492189671e+05, -1.773802305051e+05, -1.773111052661e+05, -1.772418439131e+05, -1.771724471021e+05, -1.771029154891e+05, -1.770332497211e+05, -1.769634504431e+05, -1.768935182941e+05, -1.768234539101e+05, -1.767532579231e+05, -1.766829309581e+05, -1.766124736381e+05, -1.765418865811e+05, -1.764711704021e+05, -1.764003257101e+05},
    {4.928131867787e+04, 4.777202664587e+04, 4.624539660987e+04, 4.470090326887e+04, 4.313799550587e+04, 4.155609469487e+04, 3.995459287187e+04, 3.833285075087e+04, 3.669019557287e+04, 3.502591876587e+04, 3.333927339987e+04, 3.162947141887e+04, 2.989568061287e+04, 2.813702132387e+04, 2.635256283287e+04, 2.454131940787e+04, 2.270224597487e+04, 2.083423336187e+04, 1.893610307487e+04, 1.700660155087e+04, 1.504439382587e+04, 1.304805655987e+04, 1.101607033387e+04, 8.946811156871e+03, 6.838541075871e+03, 4.689397816871e+03, 2.497383340871e+03, 2.603512328705e+02, -2.024007187129e+03, -4.358178104130e+03, -6.744841878130e+03, -9.186891150129e+03, -1.168745035713e+04, -1.424989648613e+04, -1.687788076513e+04, -1.957535068313e+04, -2.234657137713e+04, -2.519614478613e+04, -2.812902410213e+04, -3.115051968613e+04, -3.426629065113e+04, -3.748231337813e+04, -4.080481400813e+04, -4.424014599613e+04, -4.779458575213e+04, -5.147400942713e+04, -5.528340370813e+04, -5.922615780613e+04, -6.330309289113e+04, -6.751122456013e+04, -7.184233679813e+04, -7.628156273513e+04, -8.080626557813e+04, -8.538552057613e+04, -8.998041454113e+04, -9.454532082713e+04, -9.903035586613e+04, -1.033851828251e+05, -1.075638643601e+05, -1.115297093071e+05, -1.152592607391e+05, -1.187423743291e+05, -1.219794393531e+05, -1.249790157331e+05, -1.277548023781e+05, -1.303231331531e+05, -1.327012300721e+05, -1.349061177261e+05, -1.369540056961e+05, -1.388599666751e+05, -1.406377926631e+05, -1.422999586691e+05, -1.438576528741e+05, -1.453208479591e+05, -1.466983963941e+05, -1.479981372841e+05, -1.492270057861e+05, -1.503911390001e+05, -1.514959745261e+05, -1.525463396861e+05, -1.535465306901e+05, -1.545003819071e+05, -1.554113258591e+05, -1.562824448681e+05, -1.571165153391e+05, -1.579160456741e+05, -1.586833087441e+05, -1.594203697291e+05, -1.601291100391e+05, -1.608112479461e+05, -1.614683564241e+05, -1.621018786331e+05, -1.627131414231e+05, -1.633033671281e+05, -1.638736839371e+05, -1.644251350091e+05, -1.649586865461e+05, -1.654752349341e+05, -1.659756130961e+05, -1.664605961601e+05, -1.669309065111e+05, -1.673872183301e+05, -1.678301616541e+05, -1.682603260371e+05, -1.686782638491e+05, -1.690844932501e+05, -1.694795008821e+05, -1.698637443121e+05, -1.702376542401e+05, -1.706016365171e+05, -1.709560739751e+05, -1.713013280941e+05, -1.716377405311e+05, -1.719656345131e+05, -1.722853161101e+05, -1.725970754111e+05, -1.729011875911e+05, -1.731979139031e+05, -1.734875025821e+05, -1.737701896871e+05, -1.740461998671e+05, -1.743157470761e+05, -1.745790352311e+05, -1.748362588231e+05, -1.750876034791e+05, -1.753332464881e+05, -1.755733572841e+05, -1.758080979011e+05, -1.760376233921e+05, -1.762620822171e+05, -1.764816166131e+05, -1.766963629281e+05, -1.769064519431e+05, -1.771120091651e+05, -1.773131551061e+05, -1.775100055411e+05, -1.777026717551e+05, -1.778912607631e+05, -1.780758755321e+05, -1.782566151761e+05, -1.784335751461e+05, -1.786068474081e+05, -1.787765206101e+05, -1.789426802351e+05, -1.791054087541e+05, -1.792647857631e+05, -1.794208881131e+05, -1.795737900371e+05, -1.797235632651e+05, -1.798702771351e+05, -1.800139987011e+05, -1.801547928261e+05, -1.802927222821e+05, -1.804278478341e+05, -1.805602283281e+05, -1.806899207681e+05, -1.808169803961e+05, -1.809414607571e+05, -1.810634137761e+05, -1.811828898141e+05, -1.812999377371e+05, -1.814146049721e+05, -1.815269375631e+05, -1.816369802221e+05, -1.817447763851e+05, -1.818503682551e+05, -1.819537968511e+05, -1.820551020521e+05, -1.821543226371e+05, -1.822514963281e+05, -1.823466598231e+05, -1.824398488411e+05, -1.825310981471e+05, -1.826204415941e+05, -1.827079121491e+05, -1.827935419271e+05, -1.828773622191e+05, -1.829594035201e+05, -1.830396955541e+05, -1.831182673051e+05, -1.831951470371e+05, -1.832703623171e+05, -1.833439400421e+05, -1.834159064581e+05, -1.834862871791e+05, -1.835551072131e+05, -1.836223909751e+05, -1.836881623081e+05, -1.837524445021e+05, -1.838152603101e+05, -1.838766319631e+05, -1.839365811901e+05, -1.839951292271e+05, -1.840522968371e+05, -1.841081043241e+05, -1.841625715411e+05, -1.842157179101e+05, -1.842675624301e+05, -1.843181236931e+05, -1.843674198891e+05, -1.844154688271e+05, -1.844622879371e+05, -1.845078942851e+05, -1.845523045841e+05, -1.845955352011e+05, -1.846376021681e+05, -1.846785211891e+05, -1.847183076531e+05, -1.847569766401e+05, -1.847945429281e+05, -1.848310210031e+05, -1.848664250671e+05, -1.849007690441e+05, -1.849340665881e+05, -1.849663310891e+05, -1.849975756811e+05, -1.850278132491e+05, -1.850570564331e+05, -1.850853176371e+05, -1.851126090321e+05, -1.851389425661e+05, -1.851643299631e+05, -1.851887827351e+05, -1.852123121841e+05, -1.852349294071e+05, -1.852566453031e+05, -1.852774705751e+05, -1.852974157351e+05, -1.853164911121e+05, -1.853347068521e+05, -1.853520729251e+05, -1.853685991271e+05, -1.853842950871e+05, -1.853991702671e+05, -1.854132339691e+05, -1.854264953391e+05, -1.854389633651e+05, -1.854506468891e+05, -1.854615546021e+05, -1.854716950541e+05, -1.854810766531e+05, -1.854897076691e+05, -1.854975962391e+05, -1.855047503671e+05, -1.855111779291e+05, -1.855168866741e+05, -1.855218842281e+05, -1.855261780971e+05, -1.855297756681e+05, -1.855326842121e+05, -1.855349108881e+05, -1.855364627431e+05, -1.855373467161e+05, -1.855375696391e+05, -1.855371382401e+05, -1.855360591451e+05, -1.855343388801e+05, -1.855319838721e+05, -1.855290004551e+05, -1.855253948651e+05, -1.855211732481e+05, -1.855163416601e+05, -1.855109060651e+05, -1.855048723451e+05, -1.854982462931e+05, -1.854910336201e+05, -1.854832399541e+05, -1.854748708441e+05, -1.854659317591e+05, -1.854564280911e+05, -1.854463651561e+05, -1.854357481961e+05, -1.854245823791e+05, -1.854128728021e+05, -1.854006244921e+05, -1.853878424051e+05, -1.853745314321e+05, -1.853606963941e+05, -1.853463420501e+05, -1.853314730921e+05, -1.853160941511e+05, -1.853002097951e+05, -1.852838245321e+05, -1.852669428101e+05, -1.852495690161e+05, -1.852317074841e+05, -1.852133624871e+05, -1.851945382451e+05, -1.851752389221e+05, -1.851554686301e+05, -1.851352314261e+05, -1.851145313161e+05, -1.850933722551e+05, -1.850717581491e+05, -1.850496928521e+05, -1.850271801731e+05, -1.850042238701e+05, -1.849808276571e+05, -1.849569952001e+05, -1.849327301191e+05, -1.849080359931e+05, -1.848829163541e+05, -1.848573746911e+05, -1.848314144531e+05, -1.848050390441e+05, -1.847782518301e+05, -1.847510561341e+05, -1.847234552421e+05, -1.846954523981e+05, -1.846670508101e+05, -1.846382536461e+05, -1.846090640381e+05, -1.845794850821e+05, -1.845495198351e+05, -1.845191713231e+05, -1.844884425321e+05, -1.844573364181e+05, -1.844258558991e+05, -1.843940038631e+05, -1.843617831631e+05, -1.843291966201e+05, -1.842962470241e+05, -1.842629371331e+05, -1.842292696741e+05, -1.841952473431e+05, -1.841608728061e+05, -1.841261487011e+05, -1.840910776351e+05, -1.840556621871e+05, -1.840199049091e+05, -1.839838083211e+05, -1.839473749211e+05, -1.839106071761e+05, -1.838735075281e+05, -1.838360783921e+05, -1.837983221571e+05, -1.837602411871e+05, -1.837218378211e+05, -1.836831143731e+05, -1.836440731311e+05, -1.836047163611e+05, -1.835650463041e+05, -1.835250651781e+05, -1.834847751781e+05, -1.834441784761e+05, -1.834032772211e+05, -1.833620735411e+05, -1.833205695411e+05, -1.832787673041e+05, -1.832366688941e+05, -1.831942763511e+05, -1.831515916971e+05, -1.831086169321e+05, -1.830653540361e+05, -1.830218049701e+05, -1.829779716751e+05, -1.829338560711e+05, -1.828894600611e+05, -1.828447855291e+05, -1.827998343401e+05, -1.827546083391e+05, -1.827091093561e+05, -1.826633392011e+05, -1.826172996661e+05, -1.825709925281e+05, -1.825244195451e+05, -1.824775824571e+05, -1.824304829901e+05, -1.823831228521e+05, -1.823355037341e+05, -1.822876273121e+05, -1.822394952471e+05, -1.821911091801e+05, -1.821424707431e+05, -1.820935815471e+05, -1.820444431901e+05, -1.819950572571e+05, -1.819454253151e+05, -1.818955489181e+05, -1.818454296061e+05, -1.817950689041e+05, -1.817444683221e+05, -1.816936293601e+05, -1.816425535001e+05, -1.815912422121e+05, -1.815396969541e+05, -1.814879191681e+05, -1.814359102871e+05, -1.813836717261e+05, -1.813312048921e+05, -1.812785111761e+05, -1.812255919601e+05, -1.811724486091e+05, -1.811190824811e+05, -1.810654949191e+05, -1.810116872531e+05, -1.809576608061e+05, -1.809034168841e+05, -1.808489567861e+05, -1.807942817961e+05, -1.807393931901e+05, -1.806842922311e+05, -1.806289801711e+05, -1.805734582531e+05, -1.805177277081e+05, -1.804617897561e+05, -1.804056456081e+05, -1.803492964631e+05, -1.802927435111e+05, -1.802359879331e+05, -1.801790308971e+05, -1.801218735641e+05, -1.800645170831e+05, -1.800069625961e+05, -1.799492112331e+05, -1.798912641151e+05, -1.798331223561e+05, -1.797747870581e+05, -1.797162593151e+05, -1.796575402121e+05, -1.795986308251e+05, -1.795395322221e+05, -1.794802454611e+05, -1.794207715911e+05, -1.793611116551e+05, -1.793012666861e+05, -1.792412377071e+05, -1.791810257351e+05, -1.791206317791e+05, -1.790600568381e+05, -1.789993019051e+05, -1.789383679651e+05, -1.788772559921e+05, -1.788159669571e+05, -1.787545018201e+05, -1.786928615351e+05, -1.786310470471e+05, -1.785690592961e+05, -1.785068992121e+05, -1.784445677191e+05, -1.783820657341e+05, -1.783193941671e+05, -1.782565539201e+05, -1.781935458891e+05, -1.781303709631e+05, -1.780670300231e+05, -1.780035239461e+05, -1.779398535981e+05, -1.778760198421e+05, -1.778120235341e+05, -1.777478655221e+05, -1.776835466491e+05, -1.776190677501e+05, -1.775544296551e+05, -1.774896331891e+05, -1.774246791671e+05, -1.773595684021e+05, -1.772943016991e+05, -1.772288798561e+05, -1.771633036681e+05, -1.770975739211e+05, -1.770316913981e+05, -1.769656568731e+05, -1.768994711181e+05, -1.768331348971e+05, -1.767666489681e+05, -1.767000140851e+05, -1.766332309971e+05, -1.765663004451e+05, -1.764992231661e+05, -1.764319998931e+05, -1.763646313531e+05, -1.762971182651e+05, -1.762294613471e+05, -1.761616613101e+05, -1.760937188591e+05, -1.760256346961e+05, -1.759574095161e+05, -1.758890440111e+05, -1.758205388661e+05, -1.757518947631e+05, -1.756831123791e+05, -1.756141923851e+05, -1.755451354481e+05, -1.754759422311e+05, -1.754066133921e+05, -1.753371495841e+05, -1.752675514551e+05, -1.751978196501e+05, -1.751279548081e+05, -1.750579575661e+05, -1.749878285531e+05, -1.749175683971e+05, -1.748471777191e+05, -1.747766571391e+05},
    {5.018137326587e+04, 4.868205131987e+04, 4.716574172987e+04, 4.563193854787e+04, 4.408011146087e+04, 4.250970423087e+04, 4.092013301487e+04, 3.931078455187e+04, 3.768101418987e+04, 3.603014375887e+04, 3.435745925587e+04, 3.266220833587e+04, 3.094359757687e+04, 2.920078950987e+04, 2.743289937687e+04, 2.563899159587e+04, 2.381807590187e+04, 2.196910312287e+04, 2.009096056487e+04, 1.818246694787e+04, 1.624236686387e+04, 1.426932469287e+04, 1.226191792887e+04, 1.021862984987e+04, 8.137841484871e+03, 6.017822787871e+03, 3.856722974871e+03, 1.652559951871e+03, -5.967912231295e+02, -2.893610889129e+03, -5.240348338130e+03, -7.639636383130e+03, -1.009430681613e+04, -1.260740657313e+04, -1.518221426613e+04, -1.782225649213e+04, -2.053132302613e+04, -2.331347948913e+04, -2.617307537613e+04, -2.911474430313e+04, -3.214339179213e+04, -3.526416382713e+04, -3.848238631813e+04, -4.180346148213e+04, -4.523270162613e+04, -4.877507413313e+04, -5.243482452313e+04, -5.621494023413e+04, -6.011642198513e+04, -6.413735184013e+04, -6.827179722113e+04, -7.250866834313e+04, -7.683072901013e+04, -8.121399988613e+04, -8.562775627713e+04, -9.003524376913e+04, -9.439519648813e+04, -9.866424475113e+04, -1.028001840261e+05, -1.067657142251e+05, -1.105318503691e+05, -1.140805517621e+05, -1.174042649561e+05, -1.205039924771e+05, -1.233873178141e+05, -1.260660345811e+05, -1.285541708361e+05, -1.308665807931e+05, -1.330180461531e+05, -1.350227493381e+05, -1.368939877931e+05, -1.386440357091e+05, -1.402840953141e+05, -1.418243040461e+05, -1.432737774841e+05, -1.446406747131e+05, -1.459322765001e+05, -1.471550689921e+05, -1.483148276221e+05, -1.494166975231e+05, -1.504652681671e+05, -1.514646410481e+05, -1.524184899811e+05, -1.533301141771e+05, -1.542024845351e+05, -1.550382837881e+05, -1.558399412051e+05, -1.566096625301e+05, -1.573494558221e+05, -1.580611537771e+05, -1.587464330551e+05, -1.594068310451e+05, -1.600437604711e+05, -1.606585221471e+05, -1.612523161491e+05, -1.618262516481e+05, -1.623813555811e+05, -1.629185803241e+05, -1.634388105121e+05, -1.639428690971e+05, -1.644315227731e+05, -1.649054868171e+05, -1.653654294391e+05, -1.658119756881e+05, -1.662457109811e+05, -1.666671842731e+05, -1.670769109401e+05, -1.674753753761e+05, -1.678630333641e+05, -1.682403142181e+05, -1.686076227441e+05, -1.689653410191e+05, -1.693138300191e+05, -1.696534311111e+05, -1.699844674101e+05, -1.703072450331e+05, -1.706220542431e+05, -1.709291705001e+05, -1.712288554341e+05, -1.715213577381e+05, -1.718069139871e+05, -1.720857494031e+05, -1.723580785551e+05, -1.726241060081e+05, -1.728840269301e+05, -1.731380276431e+05, -1.733862861491e+05, -1.736289726051e+05, -1.738662497731e+05, -1.740982734401e+05, -1.743251927971e+05, -1.745471508121e+05, -1.747642845611e+05, -1.749767255421e+05, -1.751845999761e+05, -1.753880290801e+05, -1.755871293221e+05, -1.757820126681e+05, -1.759727868091e+05, -1.761595553701e+05, -1.763424181121e+05, -1.765214711221e+05, -1.766968069891e+05, -1.768685149681e+05, -1.770366811411e+05, -1.772013885601e+05, -1.773627173921e+05, -1.775207450451e+05, -1.776755462971e+05, -1.778271934091e+05, -1.779757562381e+05, -1.781213023451e+05, -1.782638970871e+05, -1.784036037191e+05, -1.785404834771e+05, -1.786745956661e+05, -1.788059977391e+05, -1.789347453741e+05, -1.790608925421e+05, -1.791844915851e+05, -1.793055932691e+05, -1.794242468561e+05, -1.795405001581e+05, -1.796543995961e+05, -1.797659902491e+05, -1.798753159111e+05, -1.799824191341e+05, -1.800873412781e+05, -1.801901225531e+05, -1.802908020651e+05, -1.803894178481e+05, -1.804860069141e+05, -1.805806052791e+05, -1.806732480041e+05, -1.807639692271e+05, -1.808528021981e+05, -1.809397793031e+05, -1.810249321011e+05, -1.811082913461e+05, -1.811898870191e+05, -1.812697483511e+05, -1.813479038471e+05, -1.814243813141e+05, -1.814992078791e+05, -1.815724100141e+05, -1.816440135551e+05, -1.817140437251e+05, -1.817825251511e+05, -1.818494818811e+05, -1.819149374081e+05, -1.819789146821e+05, -1.820414361261e+05, -1.821025236571e+05, -1.821621986971e+05, -1.822204821891e+05, -1.822773946121e+05, -1.823329559931e+05, -1.823871859221e+05, -1.824401035631e+05, -1.824917276691e+05, -1.825420765911e+05, -1.825911682891e+05, -1.826390203471e+05, -1.826856499811e+05, -1.827310740461e+05, -1.827753090521e+05, -1.828183711701e+05, -1.828602762431e+05, -1.829010397921e+05, -1.829406770281e+05, -1.829792028571e+05, -1.830166318921e+05, -1.830529784571e+05, -1.830882565981e+05, -1.831224800851e+05, -1.831556624271e+05, -1.831878168711e+05, -1.832189564121e+05, -1.832490938011e+05, -1.832782415491e+05, -1.833064119331e+05, -1.833336170021e+05, -1.833598685831e+05, -1.833851782881e+05, -1.834095575141e+05, -1.834330174571e+05, -1.834555691061e+05, -1.834772232581e+05, -1.834979905171e+05, -1.835178812991e+05, -1.835369058371e+05, -1.835550741871e+05, -1.835723962301e+05, -1.835888816761e+05, -1.836045400691e+05, -1.836193807901e+05, -1.836334130631e+05, -1.836466459541e+05, -1.836590883791e+05, -1.836707491041e+05, -1.836816367511e+05, -1.836917598011e+05, -1.837011265931e+05, -1.837097453341e+05, -1.837176240961e+05, -1.837247708211e+05, -1.837311933241e+05, -1.837368992951e+05, -1.837418963051e+05, -1.837461918021e+05, -1.837497931201e+05, -1.837527074771e+05, -1.837549419791e+05, -1.837565036241e+05, -1.837573993021e+05, -1.837576357971e+05, -1.837572197911e+05, -1.837561578641e+05, -1.837544564991e+05, -1.837521220791e+05, -1.837491608961e+05, -1.837455791451e+05, -1.837413829331e+05, -1.837365782751e+05, -1.837311711001e+05, -1.837251672501e+05, -1.837185724831e+05, -1.837113924761e+05, -1.837036328211e+05, -1.836952990341e+05, -1.836863965511e+05, -1.836769307331e+05, -1.836669068641e+05, -1.836563301551e+05, -1.836452057451e+05, -1.836335387021e+05, -1.836213340241e+05, -1.836085966401e+05, -1.835953314131e+05, -1.835815431391e+05, -1.835672365501e+05, -1.835524163141e+05, -1.835370870371e+05, -1.835212532631e+05, -1.835049194751e+05, -1.834880900991e+05, -1.834707695011e+05, -1.834529619921e+05, -1.834346718231e+05, -1.834159031951e+05, -1.833966602501e+05, -1.833769470791e+05, -1.833567677211e+05, -1.833361261641e+05, -1.833150263421e+05, -1.832934721441e+05, -1.832714674071e+05, -1.832490159201e+05, -1.832261214271e+05, -1.832027876231e+05, -1.831790181581e+05, -1.831548166381e+05, -1.831301866251e+05, -1.831051316351e+05, -1.830796551431e+05, -1.830537605831e+05, -1.830274513461e+05, -1.830007307831e+05, -1.829736022031e+05, -1.829460688791e+05, -1.829181340421e+05, -1.828898008871e+05, -1.828610725701e+05, -1.828319522111e+05, -1.828024428931e+05, -1.827725476631e+05, -1.827422695321e+05, -1.827116114791e+05, -1.826805764461e+05, -1.826491673421e+05, -1.826173870431e+05, -1.825852383931e+05, -1.825527242031e+05, -1.825198472531e+05, -1.824866102901e+05, -1.824530160321e+05, -1.824190671661e+05, -1.823847663501e+05, -1.823501162121e+05, -1.823151193511e+05, -1.822797783361e+05, -1.822440957121e+05, -1.822080739911e+05, -1.821717156621e+05, -1.821350231851e+05, -1.820979989941e+05, -1.820606454971e+05, -1.820229650741e+05, -1.819849600851e+05, -1.819466328581e+05, -1.819079857021e+05, -1.818690209001e+05, -1.818297407081e+05, -1.817901473641e+05, -1.817502430771e+05, -1.817100300371e+05, -1.816695104091e+05, -1.816286863361e+05, -1.815875599411e+05, -1.815461333231e+05, -1.815044085611e+05, -1.814623877101e+05, -1.814200728081e+05, -1.813774658691e+05, -1.813345688901e+05, -1.812913838451e+05, -1.812479126901e+05, -1.812041573601e+05, -1.811601197731e+05, -1.811158018251e+05, -1.810712053971e+05, -1.810263323471e+05, -1.809811845191e+05, -1.809357637371e+05, -1.808900718061e+05, -1.808441105171e+05, -1.807978816391e+05, -1.807513869271e+05, -1.807046281201e+05, -1.806576069371e+05, -1.806103250841e+05, -1.805627842471e+05, -1.805149861001e+05, -1.804669322991e+05, -1.804186244841e+05, -1.803700642801e+05, -1.803212532971e+05, -1.802721931311e+05, -1.802228853611e+05, -1.801733315521e+05, -1.801235332561e+05, -1.800734920091e+05, -1.800232093331e+05, -1.799726867371e+05, -1.799219257151e+05, -1.798709277491e+05, -1.798196943051e+05, -1.797682268391e+05, -1.797165267901e+05, -1.796645955881e+05, -1.796124346481e+05, -1.795600453711e+05, -1.795074291491e+05, -1.794545873591e+05, -1.794015213661e+05, -1.793482325231e+05, -1.792947221731e+05, -1.792409916451e+05, -1.791870422571e+05, -1.791328753151e+05, -1.790784921151e+05, -1.790238939391e+05, -1.789690820621e+05, -1.789140577451e+05, -1.788588222371e+05, -1.788033767811e+05, -1.787477226041e+05, -1.786918609261e+05, -1.786357929561e+05, -1.785795198911e+05, -1.785230429211e+05, -1.784663632221e+05, -1.784094819641e+05, -1.783524003041e+05, -1.782951193921e+05, -1.782376403661e+05, -1.781799643561e+05, -1.781220924821e+05, -1.780640258551e+05, -1.780057655771e+05, -1.779473127411e+05, -1.778886684301e+05, -1.778298337191e+05, -1.777708096741e+05, -1.777115973531e+05, -1.776521978041e+05, -1.775926120681e+05, -1.775328411761e+05, -1.774728861531e+05, -1.774127480121e+05, -1.773524277631e+05, -1.772919264031e+05, -1.772312449241e+05, -1.771703843091e+05, -1.771093455341e+05, -1.770481295661e+05, -1.769867373661e+05, -1.769251698861e+05, -1.768634280721e+05, -1.768015128611e+05, -1.767394251831e+05, -1.766771659621e+05, -1.766147361141e+05, -1.765521365471e+05, -1.764893681641e+05, -1.764264318591e+05, -1.763633285211e+05, -1.763000590301e+05, -1.762366242621e+05, -1.761730250831e+05, -1.761092623561e+05, -1.760453369351e+05, -1.759812496681e+05, -1.759170013971e+05, -1.758525929581e+05, -1.757880251791e+05, -1.757232988841e+05, -1.756584148901e+05, -1.755933740071e+05, -1.755281770401e+05, -1.754628247871e+05, -1.753973180431e+05, -1.753316575931e+05, -1.752658442191e+05, -1.751998786961e+05, -1.751337617941e+05, -1.750674942771e+05, -1.750010769051e+05, -1.749345104291e+05, -1.748677955971e+05, -1.748009331521e+05, -1.747339238301e+05, -1.746667683631e+05, -1.745994674781e+05, -1.745320218941e+05, -1.744644323291e+05, -1.743966994921e+05, -1.743288240901e+05, -1.742608068231e+05, -1.741926483871e+05, -1.741243494731e+05, -1.740559107671e+05, -1.739873329491e+05, -1.739186166971e+05, -1.738497626811e+05, -1.737807715691e+05, -1.737116440231e+05, -1.736423807001e+05, -1.735729822541e+05, -1.735034493331e+05, -1.734337825811e+05, -1.733639826391e+05, -1.732940501401e+05, -1.732239857171e+05, -1.731537899951e+05},
    {5.108238822187e+04, 4.959292887287e+04, 4.808682451487e+04, 4.656358782887e+04, 4.502270850487e+04, 4.346365180387e+04, 4.188585701787e+04, 4.028873581187e+04, 3.867167041687e+04, 3.703401170087e+04, 3.537507705887e+04, 3.369414814487e+04, 3.199046840087e+04, 3.026324038587e+04, 2.851162286387e+04, 2.673472765087e+04, 2.493161617187e+04, 2.310129572187e+04, 2.124271538187e+04, 1.935476157487e+04, 1.743625321187e+04, 1.548593639887e+04, 1.350247865287e+04, 1.148446260287e+04, 9.430379108871e+03, 7.338619771870e+03, 5.207468787870e+03, 3.035094104871e+03, 8.195378608705e+02, -1.441293897129e+03, -3.749642219129e+03, -6.107906892129e+03, -8.518658615130e+03, -1.098465148513e+04, -1.350883543513e+04, -1.609436809513e+04, -1.874462524213e+04, -2.146320860813e+04, -2.425394926213e+04, -2.712090394813e+04, -3.006834064313e+04, -3.310070798813e+04, -3.622258106213e+04, -3.943857301013e+04, -4.275319822913e+04, -4.617066825713e+04, -4.969459682813e+04, -5.332758745413e+04, -5.707067886713e+04, -6.092263646713e+04, -6.487910836313e+04, -6.893171552613e+04, -7.306720751513e+04, -7.726686109813e+04, -8.150629724113e+04, -8.575583852513e+04, -8.998146484613e+04, -9.414639349813e+04, -9.821328036613e+04, -1.021469143381e+05, -1.059170315951e+05, -1.095006896351e+05, -1.128839501521e+05, -1.160612416511e+05, -1.190339665141e+05, -1.218088602041e+05, -1.243961280171e+05, -1.268078777391e+05, -1.290569789091e+05, -1.311563157911e+05, -1.331183365721e+05, -1.349548000301e+05, -1.366766453241e+05, -1.382939372271e+05, -1.398158586481e+05, -1.412507338871e+05, -1.426060720851e+05, -1.438886233901e+05, -1.451044420701e+05, -1.462589521361e+05, -1.473570121661e+05, -1.484029770381e+05, -1.494007551571e+05, -1.503538604381e+05, -1.512654588161e+05, -1.521384094051e+05, -1.529753006261e+05, -1.537784817591e+05, -1.545500904111e+05, -1.552920763791e+05, -1.560062224051e+05, -1.566941622211e+05, -1.573573962711e+05, -1.579973054541e+05, -1.586151631521e+05, -1.592121457971e+05, -1.597893421851e+05, -1.603477617061e+05, -1.608883416381e+05, -1.614119536401e+05, -1.619194095311e+05, -1.624114664661e+05, -1.628888315721e+05, -1.633521661081e+05, -1.638020892231e+05, -1.642391813301e+05, -1.646639871751e+05, -1.650770186001e+05, -1.654787570611e+05, -1.658696559101e+05, -1.662501424781e+05, -1.666206199701e+05, -1.669814691941e+05, -1.673330501471e+05, -1.676757034621e+05, -1.680097517361e+05, -1.683355007501e+05, -1.686532405901e+05, -1.689632466771e+05, -1.692657807161e+05, -1.695610915741e+05, -1.698494160841e+05, -1.701309797991e+05, -1.704059976771e+05, -1.706746747241e+05, -1.709372065881e+05, -1.711937801081e+05, -1.714445738271e+05, -1.716897584701e+05, -1.719294973811e+05, -1.721639469401e+05, -1.723932569471e+05, -1.726175709771e+05, -1.728370267201e+05, -1.730517562921e+05, -1.732618865281e+05, -1.734675392531e+05, -1.736688315471e+05, -1.738658759761e+05, -1.740587808251e+05, -1.742476503091e+05, -1.744325847691e+05, -1.746136808631e+05, -1.747910317411e+05, -1.749647272121e+05, -1.751348538991e+05, -1.753014953891e+05, -1.754647323701e+05, -1.756246427611e+05, -1.757813018421e+05, -1.759347823651e+05, -1.760851546681e+05, -1.762324867801e+05, -1.763768445211e+05, -1.765182915921e+05, -1.766568896711e+05, -1.767926984921e+05, -1.769257759261e+05, -1.770561780621e+05, -1.771839592731e+05, -1.773091722881e+05, -1.774318682591e+05, -1.775520968201e+05, -1.776699061471e+05, -1.777853430161e+05, -1.778984528541e+05, -1.780092797941e+05, -1.781178667191e+05, -1.782242553131e+05, -1.783284861021e+05, -1.784305984981e+05, -1.785306308391e+05, -1.786286204291e+05, -1.787246035721e+05, -1.788186156091e+05, -1.789106909531e+05, -1.790008631201e+05, -1.790891647581e+05, -1.791756276811e+05, -1.792602828921e+05, -1.793431606171e+05, -1.794242903231e+05, -1.795037007501e+05, -1.795814199311e+05, -1.796574752151e+05, -1.797318932921e+05, -1.798047002101e+05, -1.798759213991e+05, -1.799455816891e+05, -1.800137053291e+05, -1.800803160031e+05, -1.801454368511e+05, -1.802090904831e+05, -1.802712989971e+05, -1.803320839931e+05, -1.803914665881e+05, -1.804494674291e+05, -1.805061067131e+05, -1.805614041901e+05, -1.806153791851e+05, -1.806680506071e+05, -1.807194369591e+05, -1.807695563521e+05, -1.808184265151e+05, -1.808660648071e+05, -1.809124882271e+05, -1.809577134191e+05, -1.810017566921e+05, -1.810446340181e+05, -1.810863610491e+05, -1.811269531231e+05, -1.811664252711e+05, -1.812047922271e+05, -1.812420684341e+05, -1.812782680551e+05, -1.813134049771e+05, -1.813474928201e+05, -1.813805449401e+05, -1.814125744431e+05, -1.814435941851e+05, -1.814736167811e+05, -1.815026546101e+05, -1.815307198221e+05, -1.815578243421e+05, -1.815839798771e+05, -1.816091979221e+05, -1.816334897631e+05, -1.816568664821e+05, -1.816793389661e+05, -1.817009179041e+05, -1.817216138011e+05, -1.817414369751e+05, -1.817603975641e+05, -1.817785055301e+05, -1.817957706651e+05, -1.818122025921e+05, -1.818278107691e+05, -1.818426044951e+05, -1.818565929121e+05, -1.818697850081e+05, -1.818821896241e+05, -1.818938154511e+05, -1.819046710401e+05, -1.819147648001e+05, -1.819241050021e+05, -1.819326997871e+05, -1.819405571621e+05, -1.819476850051e+05, -1.819540910701e+05, -1.819597829891e+05, -1.819647682721e+05, -1.819690543111e+05, -1.819726483841e+05, -1.819755576571e+05, -1.819777891821e+05, -1.819793499061e+05, -1.819802466681e+05, -1.819804862051e+05, -1.819800751511e+05, -1.819790200391e+05, -1.819773273071e+05, -1.819750032961e+05, -1.819720542511e+05, -1.819684863291e+05, -1.819643055941e+05, -1.819595180231e+05, -1.819541295031e+05, -1.819481458421e+05, -1.819415727581e+05, -1.819344158921e+05, -1.819266808021e+05, -1.819183729691e+05, -1.819094977961e+05, -1.819000606091e+05, -1.818900666631e+05, -1.818795211361e+05, -1.818684291361e+05, -1.818567957031e+05, -1.818446258041e+05, -1.818319243401e+05, -1.818186961471e+05, -1.818049459941e+05, -1.817906785851e+05, -1.817758985641e+05, -1.817606105091e+05, -1.817448189411e+05, -1.817285283201e+05, -1.817117430461e+05, -1.816944674631e+05, -1.816767058591e+05, -1.816584624641e+05, -1.816397414561e+05, -1.816205469571e+05, -1.816008830371e+05, -1.815807537151e+05, -1.815601629581e+05, -1.815391146831e+05, -1.815176127591e+05, -1.814956610031e+05, -1.814732631891e+05, -1.814504230411e+05, -1.814271442381e+05, -1.814034304131e+05, -1.813792851551e+05, -1.813547120091e+05, -1.813297144781e+05, -1.813042960191e+05, -1.812784600521e+05, -1.812522099521e+05, -1.812255490571e+05, -1.811984806611e+05, -1.811710080231e+05, -1.811431343621e+05, -1.811148628571e+05, -1.810861966541e+05, -1.810571388571e+05, -1.810276925391e+05, -1.809978607341e+05, -1.809676464421e+05, -1.809370526281e+05, -1.809060822251e+05, -1.808747381291e+05, -1.808430232061e+05, -1.808109402901e+05, -1.807784921791e+05, -1.807456816441e+05, -1.807125114231e+05, -1.806789842231e+05, -1.806451027221e+05, -1.806108695681e+05, -1.805762873801e+05, -1.805413587471e+05, -1.805060862311e+05, -1.804704723651e+05, -1.804345196561e+05, -1.803982305831e+05, -1.803616075981e+05, -1.803246531261e+05, -1.802873695681e+05, -1.802497592981e+05, -1.802118246651e+05, -1.801735679921e+05, -1.801349915811e+05, -1.800960977051e+05, -1.800568886171e+05, -1.800173665451e+05, -1.799775336931e+05, -1.799373922431e+05, -1.798969443561e+05, -1.798561921681e+05, -1.798151377941e+05, -1.797737833271e+05, -1.797321308421e+05, -1.796901823871e+05, -1.796479399951e+05, -1.796054056751e+05, -1.795625814161e+05, -1.795194691891e+05, -1.794760709441e+05, -1.794323886121e+05, -1.793884241031e+05, -1.793441793111e+05, -1.792996561111e+05, -1.792548563571e+05, -1.792097818871e+05, -1.791644345201e+05, -1.791188160591e+05, -1.790729282881e+05, -1.790267729741e+05, -1.789803518681e+05, -1.789336667011e+05, -1.788867191931e+05, -1.788395110421e+05, -1.787920439331e+05, -1.787443195341e+05, -1.786963394981e+05, -1.786481054621e+05, -1.785996190471e+05, -1.785508818611e+05, -1.785018954931e+05, -1.784526615211e+05, -1.784031815071e+05, -1.783534569991e+05, -1.783034895291e+05, -1.782532806181e+05, -1.782028317701e+05, -1.781521444781e+05, -1.781012202181e+05, -1.780500604561e+05, -1.779986666441e+05, -1.779470402191e+05, -1.778951826071e+05, -1.778430952211e+05, -1.777907794601e+05, -1.777382367121e+05, -1.776854683531e+05, -1.776324757451e+05, -1.775792602391e+05, -1.775258231741e+05, -1.774721658791e+05, -1.774182896681e+05, -1.773641958451e+05, -1.773098857051e+05, -1.772553605271e+05, -1.772006215841e+05, -1.771456701341e+05, -1.770905074271e+05, -1.770351347011e+05, -1.769795531831e+05, -1.769237640891e+05, -1.768677686281e+05, -1.768115679961e+05, -1.767551633781e+05, -1.766985559511e+05, -1.766417468821e+05, -1.765847373271e+05, -1.765275284341e+05, -1.764701213401e+05, -1.764125171731e+05, -1.763547170521e+05, -1.762967220861e+05, -1.762385333751e+05, -1.761801520111e+05, -1.761215790771e+05, -1.760628156441e+05, -1.760038627791e+05, -1.759447215371e+05, -1.758853929661e+05, -1.758258781051e+05, -1.757661779841e+05, -1.757062936271e+05, -1.756462260461e+05, -1.755859762481e+05, -1.755255452311e+05, -1.754649339861e+05, -1.754041434941e+05, -1.753431747291e+05, -1.752820286591e+05, -1.752207062431e+05, -1.751592084331e+05, -1.750975361721e+05, -1.750356903971e+05, -1.749736720381e+05, -1.749114820171e+05, -1.748491212491e+05, -1.747865906431e+05, -1.747238910981e+05, -1.746610235111e+05, -1.745979887661e+05, -1.745347877461e+05, -1.744714213241e+05, -1.744078903671e+05, -1.743441957351e+05, -1.742803382831e+05, -1.742163188581e+05, -1.741521383011e+05, -1.740877974471e+05, -1.740232971241e+05, -1.739586381561e+05, -1.738938213571e+05, -1.738288475391e+05, -1.737637175061e+05, -1.736984320551e+05, -1.736329919791e+05, -1.735673980651e+05, -1.735016510921e+05, -1.734357518371e+05, -1.733697010681e+05, -1.733034995491e+05, -1.732371480391e+05, -1.731706472891e+05, -1.731039980471e+05, -1.730372010541e+05, -1.729702570471e+05, -1.729031667581e+05, -1.728359309121e+05, -1.727685502291e+05, -1.727010254251e+05, -1.726333572111e+05, -1.725655462921e+05, -1.724975933681e+05, -1.724294991351e+05, -1.723612642841e+05, -1.722928894991e+05, -1.722243754611e+05, -1.721557228481e+05, -1.720869323291e+05, -1.720180045721e+05, -1.719489402391e+05, -1.718797399871e+05, -1.718104044691e+05, -1.717409343331e+05, -1.716713302231e+05, -1.716015927791e+05, -1.715317226351e+05},
    {5.198436202987e+04, 5.050465946487e+04, 4.900864700087e+04, 4.749585525687e+04, 4.596579315487e+04, 4.441794660787e+04, 4.285177710187e+04, 4.126672017587e+04, 3.966218378687e+04, 3.803754653687e+04, 3.639215577287e+04, 3.472532552787e+04, 3.303633429287e+04, 3.132442261687e+04, 2.958879049787e+04, 2.782859456587e+04, 2.604294502287e+04, 2.423090233187e+04, 2.239147361187e+04, 2.052360873987e+04, 1.862619610687e+04, 1.669805802087e+04, 1.473794570687e+04, 1.274453389587e+04, 1.071641495187e+04, 8.652092527870e+03, 6.549974712871e+03, 4.408366665871e+03, 2.225462719871e+03, -6.620151295210e-01, -2.272060525129e+03, -4.590923140129e+03, -6.959587039130e+03, -9.380545770129e+03, -1.185645846313e+04, -1.439015820313e+04, -1.698465885813e+04, -1.964315925913e+04, -2.236904323213e+04, -2.516587331813e+04, -2.803737515413e+04, -3.098740829213e+04, -3.401991764113e+04, -3.713885759213e+04, -4.034807822413e+04, -4.365115986313e+04, -4.705117905513e+04, -5.055038681413e+04, -5.414978091913e+04, -5.784856157113e+04, -6.164347842313e+04, -6.552810979413e+04, -6.949215884613e+04, -7.352089231413e+04, -7.759486301613e+04, -8.169003259013e+04, -8.577835706113e+04, -8.982884756513e+04, -9.380908871213e+04, -9.768715557513e+04, -1.014337675571e+05, -1.050243688581e+05, -1.084407636041e+05, -1.116721488741e+05, -1.147144760171e+05, -1.175693975401e+05, -1.202429131291e+05, -1.227438976001e+05, -1.250828510721e+05, -1.272709711361e+05, -1.293195298441e+05, -1.312394870111e+05, -1.330412656801e+05, -1.347346310301e+05, -1.363286334241e+05, -1.378315918271e+05, -1.392511036131e+05, -1.405940721791e+05, -1.418667464541e+05, -1.430747678031e+05, -1.442232207161e+05, -1.453166844551e+05, -1.463592835591e+05, -1.473547357241e+05, -1.483063961791e+05, -1.492172980711e+05, -1.500901887441e+05, -1.509275620111e+05, -1.517316866451e+05, -1.525046314211e+05, -1.532482870541e+05, -1.539643854041e+05, -1.546545162601e+05, -1.553201420491e+05, -1.559626107271e+05, -1.565831670991e+05, -1.571829628011e+05, -1.577630651011e+05, -1.583244647091e+05, -1.588680826961e+05, -1.593947766691e+05, -1.599053462781e+05, -1.604005381481e+05, -1.608810503091e+05, -1.613475361771e+05, -1.618006081471e+05, -1.622408408361e+05, -1.626687740261e+05, -1.630849153241e+05, -1.634897425861e+05, -1.638837061231e+05, -1.642672307071e+05, -1.646407174071e+05, -1.650045452631e+05, -1.653590728291e+05, -1.657046395721e+05, -1.660415671691e+05, -1.663701606931e+05, -1.666907097061e+05, -1.670034892631e+05, -1.673087608471e+05, -1.676067732161e+05, -1.678977632061e+05, -1.681819564531e+05, -1.684595680811e+05, -1.687308033251e+05, -1.689958581191e+05, -1.692549196341e+05, -1.695081667871e+05, -1.697557707071e+05, -1.699978951721e+05, -1.702346970191e+05, -1.704663265211e+05, -1.706929277421e+05, -1.709146388741e+05, -1.711315925401e+05, -1.713439160871e+05, -1.715517318601e+05, -1.717551574561e+05, -1.719543059601e+05, -1.721492861751e+05, -1.723402028301e+05, -1.725271567761e+05, -1.727102451771e+05, -1.728895616851e+05, -1.730651966011e+05, -1.732372370371e+05, -1.734057670601e+05, -1.735708678301e+05, -1.737326177341e+05, -1.738910925071e+05, -1.740463653501e+05, -1.741985070431e+05, -1.743475860481e+05, -1.744936686061e+05, -1.746368188371e+05, -1.747770988241e+05, -1.749145686991e+05, -1.750492867261e+05, -1.751813093741e+05, -1.753106913901e+05, -1.754374858711e+05, -1.755617443241e+05, -1.756835167331e+05, -1.758028516191e+05, -1.759197960901e+05, -1.760343959011e+05, -1.761466955051e+05, -1.762567380981e+05, -1.763645656691e+05, -1.764702190431e+05, -1.765737379241e+05, -1.766751609361e+05, -1.767745256611e+05, -1.768718686761e+05, -1.769672255901e+05, -1.770606310741e+05, -1.771521189001e+05, -1.772417219661e+05, -1.773294723281e+05, -1.774154012281e+05, -1.774995391231e+05, -1.775819157101e+05, -1.776625599501e+05, -1.777415000931e+05, -1.778187637021e+05, -1.778943776731e+05, -1.779683682601e+05, -1.780407610921e+05, -1.781115811911e+05, -1.781808529981e+05, -1.782486003841e+05, -1.783148466701e+05, -1.783796146471e+05, -1.784429265851e+05, -1.785048042561e+05, -1.785652689431e+05, -1.786243414601e+05, -1.786820421601e+05, -1.787383909531e+05, -1.787934073181e+05, -1.788471103121e+05, -1.788995185871e+05, -1.789506503991e+05, -1.790005236171e+05, -1.790491557401e+05, -1.790965639011e+05, -1.791427648791e+05, -1.791877751111e+05, -1.792316106991e+05, -1.792742874201e+05, -1.793158207331e+05, -1.793562257901e+05, -1.793955174431e+05, -1.794337102521e+05, -1.794708184921e+05, -1.795068561611e+05, -1.795418369881e+05, -1.795757744371e+05, -1.796086817191e+05, -1.796405717921e+05, -1.796714573721e+05, -1.797013509371e+05, -1.797302647351e+05, -1.797582107871e+05, -1.797852008921e+05, -1.798112466381e+05, -1.798363594011e+05, -1.798605503521e+05, -1.798838304641e+05, -1.799062105131e+05, -1.799277010861e+05, -1.799483125841e+05, -1.799680552271e+05, -1.799869390561e+05, -1.800049739401e+05, -1.800221695791e+05, -1.800385355081e+05, -1.800540810981e+05, -1.800688155651e+05, -1.800827479691e+05, -1.800958872201e+05, -1.801082420791e+05, -1.801198211631e+05, -1.801306329501e+05, -1.801406857771e+05, -1.801499878471e+05, -1.801585472311e+05, -1.801663718711e+05, -1.801734695831e+05, -1.801798480561e+05, -1.801855148621e+05, -1.801904774511e+05, -1.801947431581e+05, -1.801983192051e+05, -1.802012127011e+05, -1.802034306471e+05, -1.802049799361e+05, -1.802058673571e+05, -1.802060995971e+05, -1.802056832401e+05, -1.802046247761e+05, -1.802029305921e+05, -1.802006069871e+05, -1.801976601631e+05, -1.801940962311e+05, -1.801899212161e+05, -1.801851410521e+05, -1.801797615891e+05, -1.801737885931e+05, -1.801672277481e+05, -1.801600846541e+05, -1.801523648361e+05, -1.801440737391e+05, -1.801352167301e+05, -1.801257991031e+05, -1.801158260791e+05, -1.801053028061e+05, -1.800942343591e+05, -1.800826257471e+05, -1.800704819081e+05, -1.800578077151e+05, -1.800446079731e+05, -1.800308874241e+05, -1.800166507461e+05, -1.800019025551e+05, -1.799866474041e+05, -1.799708897871e+05, -1.799546341401e+05, -1.799378848401e+05, -1.799206462061e+05, -1.799029225021e+05, -1.798847179361e+05, -1.798660366641e+05, -1.798468827871e+05, -1.798272603521e+05, -1.798071733591e+05, -1.797866257531e+05, -1.797656214331e+05, -1.797441642461e+05, -1.797222579941e+05, -1.796999064281e+05, -1.796771132561e+05, -1.796538821391e+05, -1.796302166921e+05, -1.796061204881e+05, -1.795815970551e+05, -1.795566498781e+05, -1.795312824011e+05, -1.795054980251e+05, -1.794793001131e+05, -1.794526919851e+05, -1.794256769231e+05, -1.793982581701e+05, -1.793704389311e+05, -1.793422223721e+05, -1.793136116241e+05, -1.792846097811e+05, -1.792552198991e+05, -1.792254450011e+05, -1.791952880761e+05, -1.791647520751e+05, -1.791338399201e+05, -1.791025544951e+05, -1.790708986561e+05, -1.790388752221e+05, -1.790064869851e+05, -1.789737367021e+05, -1.789406271021e+05, -1.789071608811e+05, -1.788733407061e+05, -1.788391692171e+05, -1.788046490221e+05, -1.787697827011e+05, -1.787345728061e+05, -1.786990218631e+05, -1.786631323691e+05, -1.786269067931e+05, -1.785903475791e+05, -1.785534571451e+05, -1.785162378821e+05, -1.784786921561e+05, -1.784408223101e+05, -1.784026306581e+05, -1.783641194941e+05, -1.783252910841e+05, -1.782861476741e+05, -1.782466914841e+05, -1.782069247121e+05, -1.781668495331e+05, -1.781264681001e+05, -1.780857825431e+05, -1.780447949721e+05, -1.780035074731e+05, -1.779619221121e+05, -1.779200409361e+05, -1.778778659671e+05, -1.778353992111e+05, -1.777926426511e+05, -1.777495982521e+05, -1.777062679581e+05, -1.776626536941e+05, -1.776187573681e+05, -1.775745808651e+05, -1.775301260561e+05, -1.774853947911e+05, -1.774403889021e+05, -1.773951102051e+05, -1.773495604961e+05, -1.773037415551e+05, -1.772576551451e+05, -1.772113030121e+05, -1.771646868841e+05, -1.771178084741e+05, -1.770706694781e+05, -1.770232715771e+05, -1.769756164341e+05, -1.769277056981e+05, -1.768795410021e+05, -1.768311239651e+05, -1.767824561881e+05, -1.767335392591e+05, -1.766843747521e+05, -1.766349642251e+05, -1.765853092221e+05, -1.765354112731e+05, -1.764852718941e+05, -1.764348925881e+05, -1.763842748421e+05, -1.763334201311e+05, -1.762823299171e+05, -1.762310056491e+05, -1.761794487621e+05, -1.761276606781e+05, -1.760756428071e+05, -1.760233965461e+05, -1.759709232801e+05, -1.759182243821e+05, -1.758653012121e+05, -1.758121551181e+05, -1.757587874381e+05, -1.757051994951e+05, -1.756513926031e+05, -1.755973680641e+05, -1.755431271691e+05, -1.754886711971e+05, -1.754340014161e+05, -1.753791190841e+05, -1.753240254481e+05, -1.752687217431e+05, -1.752132091951e+05, -1.751574890191e+05, -1.751015624201e+05, -1.750454305921e+05, -1.749890947191e+05, -1.749325559771e+05, -1.748758155301e+05, -1.748188745331e+05, -1.747617341301e+05, -1.747043954591e+05, -1.746468596441e+05, -1.745891278041e+05, -1.745312010461e+05, -1.744730804691e+05, -1.744147671621e+05, -1.743562622061e+05, -1.742975666741e+05, -1.742386816281e+05, -1.741796081221e+05, -1.741203472041e+05, -1.740608999111e+05, -1.740012672721e+05, -1.739414503071e+05, -1.738814500311e+05, -1.738212674471e+05, -1.737609035531e+05, -1.737003593371e+05, -1.736396357811e+05, -1.735787338571e+05, -1.735176545311e+05, -1.734563987611e+05, -1.733949674981e+05, -1.733333616861e+05, -1.732715822581e+05, -1.732096301451e+05, -1.731475062671e+05, -1.730852115381e+05, -1.730227468661e+05, -1.729601131511e+05, -1.728973112851e+05, -1.728343421551e+05, -1.727712066421e+05, -1.727079056161e+05, -1.726444399461e+05, -1.725808104901e+05, -1.725170181021e+05, -1.724530636281e+05, -1.723889479091e+05, -1.723246717791e+05, -1.722602360661e+05, -1.721956415911e+05, -1.721308891701e+05, -1.720659796121e+05, -1.720009137211e+05, -1.719356922951e+05, -1.718703161251e+05, -1.718047859961e+05, -1.717391026901e+05, -1.716732669801e+05, -1.716072796351e+05, -1.715411414181e+05, -1.714748530861e+05, -1.714084153931e+05, -1.713418290841e+05, -1.712750949001e+05, -1.712082135781e+05, -1.711411858471e+05, -1.710740124351e+05, -1.710066940601e+05, -1.709392314381e+05, -1.708716252781e+05, -1.708038762861e+05, -1.707359851621e+05, -1.706679526001e+05, -1.705997792911e+05, -1.705314659201e+05, -1.704630131681e+05, -1.703944217101e+05, -1.703256922161e+05, -1.702568253541e+05, -1.701878217861e+05, -1.701186821671e+05, -1.700494071511e+05, -1.699799973861e+05, -1.699104535151e+05},
    {5.288729315887e+04, 5.141724320487e+04, 4.993121113387e+04, 4.842874483587e+04, 4.690937173287e+04, 4.537259757287e+04, 4.381790513587e+04, 4.224475284287e+04, 4.065257325987e+04, 3.904077149687e+04, 3.740872347687e+04, 3.575577407787e+04, 3.408123512787e+04, 3.238438324287e+04, 3.066445750287e+04, 2.892065692987e+04, 2.715213777687e+04, 2.535801059487e+04, 2.353733706487e+04, 2.168912657187e+04, 1.981233251587e+04, 1.790584831287e+04, 1.596850308987e+04, 1.399905704587e+04, 1.199619645587e+04, 9.958528306871e+03, 7.884574557871e+03, 5.772766023870e+03, 3.621435891871e+03, 1.428812896871e+03, -8.069857771295e+02, -3.087961788129e+03, -5.416242958129e+03, -7.794090419130e+03, -1.022390526113e+04, -1.270823419413e+04, -1.524977360413e+04, -1.785137103813e+04, -2.051602285613e+04, -2.324686624813e+04, -2.604716316413e+04, -2.892027280113e+04, -3.186960811413e+04, -3.489857028913e+04, -3.801045323113e+04, -4.120830795513e+04, -4.449475454713e+04, -4.787172784413e+04, -5.134014337013e+04, -5.489947466413e+04, -5.854724474613e+04, -6.227845567413e+04, -6.608501034513e+04, -6.995521311613e+04, -7.387345666113e+04, -7.782019724013e+04, -8.177228658313e+04, -8.570368118313e+04, -8.958651084113e+04, -9.339246091413e+04, -9.709438325213e+04, -1.006679786081e+05, -1.040933124561e+05, -1.073559269681e+05, -1.104474370881e+05, -1.133649663521e+05, -1.161103127031e+05, -1.186888373381e+05, -1.211082954841e+05, -1.233778348481e+05, -1.255072381131e+05, -1.275064019821e+05, -1.293850049381e+05, -1.311523083381e+05, -1.328170446871e+05, -1.343873608621e+05, -1.358707960761e+05, -1.372742826211e+05, -1.386041621661e+05, -1.398662128281e+05, -1.410656834811e+05, -1.422073324311e+05, -1.432954681191e+05, -1.443339899981e+05, -1.453264282101e+05, -1.462759810911e+05, -1.471855499291e+05, -1.480577706821e+05, -1.488950425641e+05, -1.496995535981e+05, -1.504733033061e+05, -1.512181227591e+05, -1.519356922551e+05, -1.526275568831e+05, -1.532951402181e+05, -1.539397563831e+05, -1.545626206851e+05, -1.551648590081e+05, -1.557475161371e+05, -1.563115631341e+05, -1.568579039111e+05, -1.573873810861e+05, -1.579007812291e+05, -1.583988395521e+05, -1.588822441391e+05, -1.593516397481e+05, -1.598076312461e+05, -1.602507867201e+05, -1.606816402891e+05, -1.611006946671e+05, -1.615084234951e+05, -1.619052734551e+05, -1.622916662151e+05, -1.626680001951e+05, -1.630346521921e+05, -1.633919788631e+05, -1.637403180941e+05, -1.640799902491e+05, -1.644112993271e+05, -1.647345340251e+05, -1.650499687171e+05, -1.653578643611e+05, -1.656584693361e+05, -1.659520202131e+05, -1.662387424781e+05, -1.665188511921e+05, -1.667925516091e+05, -1.670600397511e+05, -1.673215029391e+05, -1.675771202901e+05, -1.678270631771e+05, -1.680714956621e+05, -1.683105748971e+05, -1.685444514971e+05, -1.687732698931e+05, -1.689971686581e+05, -1.692162808161e+05, -1.694307341251e+05, -1.696406513511e+05, -1.698461505191e+05, -1.700473451501e+05, -1.702443444821e+05, -1.704372536831e+05, -1.706261740441e+05, -1.708112031691e+05, -1.709924351451e+05, -1.711699607091e+05, -1.713438674051e+05, -1.715142397251e+05, -1.716811592531e+05, -1.718447047931e+05, -1.720049524921e+05, -1.721619759591e+05, -1.723158463711e+05, -1.724666325851e+05, -1.726144012311e+05, -1.727592168081e+05, -1.729011417721e+05, -1.730402366251e+05, -1.731765599891e+05, -1.733101686881e+05, -1.734411178151e+05, -1.735694608061e+05, -1.736952495031e+05, -1.738185342151e+05, -1.739393637821e+05, -1.740577856261e+05, -1.741738458091e+05, -1.742875890831e+05, -1.743990589391e+05, -1.745082976521e+05, -1.746153463291e+05, -1.747202449501e+05, -1.748230324081e+05, -1.749237465491e+05, -1.750224242071e+05, -1.751191012451e+05, -1.752138125801e+05, -1.753065922261e+05, -1.753974733171e+05, -1.754864881411e+05, -1.755736681671e+05, -1.756590440721e+05, -1.757426457701e+05, -1.758245024351e+05, -1.759046425231e+05, -1.759830938011e+05, -1.760598833651e+05, -1.761350376621e+05, -1.762085825101e+05, -1.762805431201e+05, -1.763509441141e+05, -1.764198095411e+05, -1.764871628991e+05, -1.765530271481e+05, -1.766174247261e+05, -1.766803775691e+05, -1.767419071221e+05, -1.768020343541e+05, -1.768607797751e+05, -1.769181634431e+05, -1.769742049861e+05, -1.770289236071e+05, -1.770823380991e+05, -1.771344668561e+05, -1.771853278861e+05, -1.772349388181e+05, -1.772833169181e+05, -1.773304790941e+05, -1.773764419071e+05, -1.774212215831e+05, -1.774648340201e+05, -1.775072947951e+05, -1.775486191761e+05, -1.775888221311e+05, -1.776279183291e+05, -1.776659221561e+05, -1.777028477181e+05, -1.777387088481e+05, -1.777735191161e+05, -1.778072918321e+05, -1.778400400541e+05, -1.778717765971e+05, -1.779025140351e+05, -1.779322647081e+05, -1.779610407321e+05, -1.779888539961e+05, -1.780157161771e+05, -1.780416387381e+05, -1.780666329361e+05, -1.780907098291e+05, -1.781138802761e+05, -1.781361549461e+05, -1.781575443181e+05, -1.781780586911e+05, -1.781977081851e+05, -1.782165027441e+05, -1.782344521411e+05, -1.782515659851e+05, -1.782678537201e+05, -1.782833246321e+05, -1.782979878511e+05, -1.783118523541e+05, -1.783249269701e+05, -1.783372203821e+05, -1.783487411331e+05, -1.783594976241e+05, -1.783694981211e+05, -1.783787507561e+05, -1.783872635331e+05, -1.783950443251e+05, -1.784021008831e+05, -1.784084408351e+05, -1.784140716881e+05, -1.784190008341e+05, -1.784232355491e+05, -1.784267829961e+05, -1.784296502291e+05, -1.784318441951e+05, -1.784333717341e+05, -1.784342395841e+05, -1.784344543791e+05, -1.784340226561e+05, -1.784329508551e+05, -1.784312453191e+05, -1.784289122971e+05, -1.784259579501e+05, -1.784223883441e+05, -1.784182094611e+05, -1.784134271941e+05, -1.784080473541e+05, -1.784020756651e+05, -1.783955177721e+05, -1.783883792401e+05, -1.783806655551e+05, -1.783723821261e+05, -1.783635342861e+05, -1.783541272941e+05, -1.783441663381e+05, -1.783336565321e+05, -1.783226029201e+05, -1.783110104791e+05, -1.782988841181e+05, -1.782862286781e+05, -1.782730489361e+05, -1.782593496051e+05, -1.782451353351e+05, -1.782304107151e+05, -1.782151802711e+05, -1.781994484731e+05, -1.781832197281e+05, -1.781664983901e+05, -1.781492887541e+05, -1.781315950591e+05, -1.781134214921e+05, -1.780947721821e+05, -1.780756512111e+05, -1.780560626041e+05, -1.780360103371e+05, -1.780154983381e+05, -1.779945304831e+05, -1.779731105991e+05, -1.779512424691e+05, -1.779289298251e+05, -1.779061763571e+05, -1.778829857051e+05, -1.778593614691e+05, -1.778353072031e+05, -1.778108264181e+05, -1.777859225821e+05, -1.777605991231e+05, -1.777348594251e+05, -1.777087068361e+05, -1.776821446601e+05, -1.776551761641e+05, -1.776278045771e+05, -1.776000330871e+05, -1.775718648491e+05, -1.775433029781e+05, -1.775143505541e+05, -1.774850106211e+05, -1.774552861891e+05, -1.774251802321e+05, -1.773946956921e+05, -1.773638354751e+05, -1.773326024551e+05, -1.773009994761e+05, -1.772690293451e+05, -1.772366948431e+05, -1.772039987151e+05, -1.771709436801e+05, -1.771375324221e+05, -1.771037676011e+05, -1.770696518431e+05, -1.770351877471e+05, -1.770003778841e+05, -1.769652247961e+05, -1.769297309991e+05, -1.768938989801e+05, -1.768577312011e+05, -1.768212300961e+05, -1.767843980741e+05, -1.767472375181e+05, -1.767097507861e+05, -1.766719402101e+05, -1.766338080991e+05, -1.765953567381e+05, -1.765565883861e+05, -1.765175052791e+05, -1.764781096331e+05, -1.764384036361e+05, -1.763983894561e+05, -1.763580692411e+05, -1.763174451121e+05, -1.762765191731e+05, -1.762352935031e+05, -1.761937701631e+05, -1.761519511911e+05, -1.761098386051e+05, -1.760674344041e+05, -1.760247405651e+05, -1.759817590461e+05, -1.759384917871e+05, -1.758949407071e+05, -1.758511077071e+05, -1.758069946691e+05, -1.757626034561e+05, -1.757179359141e+05, -1.756729938701e+05, -1.756277791341e+05, -1.755822934981e+05, -1.755365387381e+05, -1.754905166101e+05, -1.754442288551e+05, -1.753976771991e+05, -1.753508633491e+05, -1.753037889971e+05, -1.752564558191e+05, -1.752088654741e+05, -1.751610196081e+05, -1.751129198481e+05, -1.750645678091e+05, -1.750159650891e+05, -1.749671132731e+05, -1.749180139301e+05, -1.748686686141e+05, -1.748190788651e+05, -1.747692462111e+05, -1.747191721641e+05, -1.746688582221e+05, -1.746183058701e+05, -1.745675165801e+05, -1.745164918101e+05, -1.744652330051e+05, -1.744137415991e+05, -1.743620190091e+05, -1.743100666421e+05, -1.742578858941e+05, -1.742054781451e+05, -1.741528447661e+05, -1.740999871141e+05, -1.740469065341e+05, -1.739936043611e+05, -1.739400819171e+05, -1.738863405131e+05, -1.738323814471e+05, -1.737782060081e+05, -1.737238154721e+05, -1.736692111071e+05, -1.736143941661e+05, -1.735593658951e+05, -1.735041275261e+05, -1.734486802841e+05, -1.733930253801e+05, -1.733371640181e+05, -1.732810973901e+05, -1.732248266781e+05, -1.731683530561e+05, -1.731116776841e+05, -1.730548017171e+05, -1.729977262981e+05, -1.729404525611e+05, -1.728829816301e+05, -1.728253146201e+05, -1.727674526371e+05, -1.727093967791e+05, -1.726511481341e+05, -1.725927077801e+05, -1.725340767871e+05, -1.724752562181e+05, -1.724162471261e+05, -1.723570505541e+05, -1.722976675401e+05, -1.722380991111e+05, -1.721783462861e+05, -1.721184100771e+05, -1.720582914871e+05, -1.719979915121e+05, -1.719375111391e+05, -1.718768513481e+05, -1.718160131111e+05, -1.717549973921e+05, -1.716938051481e+05, -1.716324373281e+05, -1.715708948741e+05, -1.715091787221e+05, -1.714472897971e+05, -1.713852290201e+05, -1.713229973051e+05, -1.712605955561e+05, -1.711980246741e+05, -1.711352855511e+05, -1.710723790711e+05, -1.710093061131e+05, -1.709460675501e+05, -1.708826642461e+05, -1.708190970601e+05, -1.707553668451e+05, -1.706914744461e+05, -1.706274207031e+05, -1.705632064501e+05, -1.704988325121e+05, -1.704342997111e+05, -1.703696088621e+05, -1.703047607731e+05, -1.702397562471e+05, -1.701745960811e+05, -1.701092810661e+05, -1.700438119861e+05, -1.699781896221e+05, -1.699124147471e+05, -1.698464881281e+05, -1.697804105291e+05, -1.697141827061e+05, -1.696478054111e+05, -1.695812793891e+05, -1.695146053821e+05, -1.694477841241e+05, -1.693808163461e+05, -1.693137027731e+05, -1.692464441241e+05, -1.691790411131e+05, -1.691114944511e+05, -1.690438048411e+05, -1.689759729831e+05, -1.689079995721e+05, -1.688398852971e+05, -1.687716308431e+05, -1.687032368901e+05, -1.686347041131e+05, -1.685660331831e+05, -1.684972247651e+05, -1.684282795221e+05, -1.683591981091e+05, -1.682899811791e+05},
    {5.379118006787e+04, 5.233068015687e+04, 5.085451877387e+04, 4.936226043687e+04, 4.785345036587e+04, 4.632761336587e+04, 4.478425265187e+04, 4.322284857187e+04, 4.164285724887e+04, 4.004370912787e+04, 3.842480740287e+04, 3.678552634687e+04, 3.512520950187e+04, 3.344316774687e+04, 3.173867721087e+04, 3.001097703887e+04, 2.825926697987e+04, 2.648270479987e+04, 2.468040349887e+04, 2.285142831387e+04, 2.099479350387e+04, 1.910945889687e+04, 1.719432618187e+04, 1.524823494487e+04, 1.326995843487e+04, 1.125819904187e+04, 9.211583514870e+03, 7.128657907871e+03, 5.007882278871e+03, 2.847625195871e+03, 6.461580998705e+02, -1.598350364129e+03, -3.887839906129e+03, -6.224363826129e+03, -8.610093624129e+03, -1.104732256313e+04, -1.353846760613e+04, -1.608606891713e+04, -1.869278585213e+04, -2.136138793113e+04, -2.409473881213e+04, -2.689577056713e+04, -2.976744472313e+04, -3.271269537013e+04, -3.573434835913e+04, -3.883500902313e+04, -4.201690938113e+04, -4.528170468913e+04, -4.863020940013e+04, -5.206206546413e+04, -5.557534325813e+04, -5.916608921113e+04, -6.282785472013e+04, -6.655126513413e+04, -7.032370753013e+04, -7.412922098313e+04, -7.794865605713e+04, -8.176013533113e+04, -8.553980798113e+04, -8.926286170513e+04, -9.290473247713e+04, -9.644242210613e+04, -9.985578888313e+04, -1.031286378931e+05, -1.062494638041e+05, -1.092117615981e+05, -1.120135571301e+05, -1.146567208211e+05, -1.171460592041e+05, -1.194883812491e+05, -1.216916919181e+05, -1.237645715431e+05, -1.257157392201e+05, -1.275537670681e+05, -1.292869041441e+05, -1.309229739531e+05, -1.324693192391e+05, -1.339327769191e+05, -1.353196728051e+05, -1.366358298941e+05, -1.378865862531e+05, -1.390768196701e+05, -1.402109767871e+05, -1.412931048381e+05, -1.423268844101e+05, -1.433156620061e+05, -1.442624814661e+05, -1.451701136161e+05, -1.460410837621e+05, -1.468776968291e+05, -1.476820601071e+05, -1.484561036711e+05, -1.492015986021e+05, -1.499201731781e+05, -1.506133272291e+05, -1.512824448481e+05, -1.519288056341e+05, -1.525535946571e+05, -1.531579112811e+05, -1.537427770061e+05, -1.543091424431e+05, -1.548578935331e+05, -1.553898571001e+05, -1.559058058351e+05, -1.564064627621e+05, -1.568925052621e+05, -1.573645687061e+05, -1.578232497311e+05, -1.582691092161e+05, -1.587026749811e+05, -1.591244442381e+05, -1.595348858291e+05, -1.599344422651e+05, -1.603235315921e+05, -1.607025490961e+05, -1.610718688711e+05, -1.614318452501e+05, -1.617828141301e+05, -1.621250941871e+05, -1.624589879891e+05, -1.627847830351e+05, -1.631027527081e+05, -1.634131571491e+05, -1.637162440831e+05, -1.640122495651e+05, -1.643013986881e+05, -1.645839062261e+05, -1.648599772431e+05, -1.651298076541e+05, -1.653935847451e+05, -1.656514876591e+05, -1.659036878551e+05, -1.661503495261e+05, -1.663916299971e+05, -1.666276800911e+05, -1.668586444811e+05, -1.670846620051e+05, -1.673058659761e+05, -1.675223844601e+05, -1.677343405471e+05, -1.679418525981e+05, -1.681450344811e+05, -1.683439957901e+05, -1.685388420551e+05, -1.687296749351e+05, -1.689165924041e+05, -1.690996889221e+05, -1.692790555991e+05, -1.694547803521e+05, -1.696269480471e+05, -1.697956406391e+05, -1.699609372991e+05, -1.701229145411e+05, -1.702816463351e+05, -1.704372042191e+05, -1.705896574021e+05, -1.707390728641e+05, -1.708855154501e+05, -1.710290479561e+05, -1.711697312181e+05, -1.713076241891e+05, -1.714427840141e+05, -1.715752661091e+05, -1.717051242211e+05, -1.718324105001e+05, -1.719571755571e+05, -1.720794685281e+05, -1.721993371271e+05, -1.723168277021e+05, -1.724319852831e+05, -1.725448536351e+05, -1.726554753041e+05, -1.727638916611e+05, -1.728701429421e+05, -1.729742682951e+05, -1.730763058131e+05, -1.731762925751e+05, -1.732742646791e+05, -1.733702572781e+05, -1.734643046131e+05, -1.735564400421e+05, -1.736466960711e+05, -1.737351043841e+05, -1.738216958701e+05, -1.739065006461e+05, -1.739895480891e+05, -1.740708668521e+05, -1.741504848961e+05, -1.742284295051e+05, -1.743047273111e+05, -1.743794043151e+05, -1.744524859061e+05, -1.745239968781e+05, -1.745939614531e+05, -1.746624032941e+05, -1.747293455251e+05, -1.747948107471e+05, -1.748588210501e+05, -1.749213980331e+05, -1.749825628181e+05, -1.750423360611e+05, -1.751007379661e+05, -1.751577883021e+05, -1.752135064111e+05, -1.752679112231e+05, -1.753210212661e+05, -1.753728546771e+05, -1.754234292171e+05, -1.754727622741e+05, -1.755208708821e+05, -1.755677717231e+05, -1.756134811411e+05, -1.756580151491e+05, -1.757013894411e+05, -1.757436193971e+05, -1.757847200921e+05, -1.758247063061e+05, -1.758635925291e+05, -1.759013929711e+05, -1.759381215681e+05, -1.759737919881e+05, -1.760084176411e+05, -1.760420116801e+05, -1.760745870161e+05, -1.761061563141e+05, -1.761367320081e+05, -1.761663263001e+05, -1.761949511691e+05, -1.762226183771e+05, -1.762493394731e+05, -1.762751257981e+05, -1.762999884891e+05, -1.763239384871e+05, -1.763469865391e+05, -1.763691432031e+05, -1.763904188541e+05, -1.764108236851e+05, -1.764303677151e+05, -1.764490607901e+05, -1.764669125881e+05, -1.764839326251e+05, -1.765001302541e+05, -1.765155146721e+05, -1.765300949241e+05, -1.765438799041e+05, -1.765568783601e+05, -1.765690988961e+05, -1.765805499751e+05, -1.765912399261e+05, -1.766011769401e+05, -1.766103690801e+05, -1.766188242771e+05, -1.766265503401e+05, -1.766335549521e+05, -1.766398456761e+05, -1.766454299581e+05, -1.766503151271e+05, -1.766545084011e+05, -1.766580168851e+05, -1.766608475761e+05, -1.766630073651e+05, -1.766645030391e+05, -1.766653412811e+05, -1.766655286771e+05, -1.766650717131e+05, -1.766639767781e+05, -1.766622501701e+05, -1.766598980911e+05, -1.766569266541e+05, -1.766533418841e+05, -1.766491497181e+05, -1.766443560071e+05, -1.766389665191e+05, -1.766329869401e+05, -1.766264228761e+05, -1.766192798521e+05, -1.766115633171e+05, -1.766032786421e+05, -1.765944311261e+05, -1.765850259931e+05, -1.765750683951e+05, -1.765645634131e+05, -1.765535160601e+05, -1.765419312801e+05, -1.765298139491e+05, -1.765171688801e+05, -1.765040008191e+05, -1.764903144511e+05, -1.764761143961e+05, -1.764614052141e+05, -1.764461914071e+05, -1.764304774151e+05, -1.764142676221e+05, -1.763975663541e+05, -1.763803778821e+05, -1.763627064211e+05, -1.763445561331e+05, -1.763259311251e+05, -1.763068354541e+05, -1.762872731251e+05, -1.762672480911e+05, -1.762467642581e+05, -1.762258254821e+05, -1.762044355691e+05, -1.761825982821e+05, -1.761603173331e+05, -1.761375963931e+05, -1.761144390841e+05, -1.760908489861e+05, -1.760668296361e+05, -1.760423845261e+05, -1.760175171081e+05, -1.759922307921e+05, -1.759665289481e+05, -1.759404149041e+05, -1.759138919501e+05, -1.758869633371e+05, -1.758596322781e+05, -1.758319019491e+05, -1.758037754871e+05, -1.757752559931e+05, -1.757463465351e+05, -1.757170501411e+05, -1.756873698091e+05, -1.756573084991e+05, -1.756268691401e+05, -1.755960546251e+05, -1.755648678171e+05, -1.755333115451e+05, -1.755013886081e+05, -1.754691017711e+05, -1.754364537701e+05, -1.754034473121e+05, -1.753700850711e+05, -1.753363696941e+05, -1.753023037991e+05, -1.752678899721e+05, -1.752331307751e+05, -1.751980287411e+05, -1.751625863751e+05, -1.751268061541e+05, -1.750906905311e+05, -1.750542419311e+05, -1.750174627541e+05, -1.749803553741e+05, -1.749429221391e+05, -1.749051653751e+05, -1.748670873811e+05, -1.748286904331e+05, -1.747899767831e+05, -1.747509486601e+05, -1.747116082701e+05, -1.746719577951e+05, -1.746319993951e+05, -1.745917352101e+05, -1.745511673561e+05, -1.745102979271e+05, -1.744691289981e+05, -1.744276626201e+05, -1.743859008271e+05, -1.743438456291e+05, -1.743014990191e+05, -1.742588629681e+05, -1.742159394281e+05, -1.741727303321e+05, -1.741292375941e+05, -1.740854631091e+05, -1.740414087521e+05, -1.739970763821e+05, -1.739524678391e+05, -1.739075849441e+05, -1.738624295021e+05, -1.738170033011e+05, -1.737713081091e+05, -1.737253456791e+05, -1.736791177491e+05, -1.736326260361e+05, -1.735858722451e+05, -1.735388580621e+05, -1.734915851591e+05, -1.734440551921e+05, -1.733962697991e+05, -1.733482306061e+05, -1.732999392221e+05, -1.732513972421e+05, -1.732026062461e+05, -1.731535677991e+05, -1.731042834511e+05, -1.730547547391e+05, -1.730049831871e+05, -1.729549703021e+05, -1.729047175791e+05, -1.728542265001e+05, -1.728034985331e+05, -1.727525351331e+05, -1.727013377421e+05, -1.726499077881e+05, -1.725982466881e+05, -1.725463558451e+05, -1.724942366491e+05, -1.724418904811e+05, -1.723893187071e+05, -1.723365226801e+05, -1.722835037451e+05, -1.722302632311e+05, -1.721768024581e+05, -1.721231227351e+05, -1.720692253571e+05, -1.720151116111e+05, -1.719607827701e+05, -1.719062400971e+05, -1.718514848471e+05, -1.717965182591e+05, -1.717413415661e+05, -1.716859559891e+05, -1.716303627371e+05, -1.715745630111e+05, -1.715185580021e+05, -1.714623488891e+05, -1.714059368431e+05, -1.713493230241e+05, -1.712925085831e+05, -1.712354946621e+05, -1.711782823921e+05, -1.711208728951e+05, -1.710632672861e+05, -1.710054666681e+05, -1.709474721361e+05, -1.708892847771e+05, -1.708309056671e+05, -1.707723358751e+05, -1.707135764621e+05, -1.706546284781e+05, -1.705954929661e+05, -1.705361709611e+05, -1.704766634891e+05, -1.704169715681e+05, -1.703570962081e+05, -1.702970384101e+05, -1.702367991691e+05, -1.701763794711e+05, -1.701157802931e+05, -1.700550026071e+05, -1.699940473751e+05, -1.699329155531e+05, -1.698716080881e+05, -1.698101259221e+05, -1.697484699881e+05, -1.696866412111e+05, -1.696246405101e+05, -1.695624687981e+05, -1.695001269791e+05, -1.694376159511e+05, -1.693749366051e+05, -1.693120898261e+05, -1.692490764901e+05, -1.691858974681e+05, -1.691225536261e+05, -1.690590458191e+05, -1.689953749011e+05, -1.689315417151e+05, -1.688675471011e+05, -1.688033918891e+05, -1.687390769071e+05, -1.686746029751e+05, -1.686099709061e+05, -1.685451815081e+05, -1.684802355821e+05, -1.684151339261e+05, -1.683498773281e+05, -1.682844665731e+05, -1.682189024401e+05, -1.681531857021e+05, -1.680873171261e+05, -1.680212974731e+05, -1.679551275001e+05, -1.678888079571e+05, -1.678223395901e+05, -1.677557231391e+05, -1.676889593381e+05, -1.676220489171e+05, -1.675549926001e+05, -1.674877911061e+05, -1.674204451491e+05, -1.673529554381e+05, -1.672853226761e+05, -1.672175475621e+05, -1.671496307911e+05, -1.670815730521e+05, -1.670133750281e+05, -1.669450374001e+05, -1.668765608411e+05, -1.668079460231e+05, -1.667391936101e+05, -1.666703042631e+05},
    {5.469602119787e+04, 5.324497033887e+04, 5.177857169287e+04, 5.029640579987e+04, 4.879803499687e+04, 4.728300240687e+04, 4.575083085287e+04, 4.420102170387e+04, 4.263305363687e+04, 4.104638131287e+04, 3.944043396387e+04, 3.781461388087e+04, 3.616829478987e+04, 3.450082012087e+04, 3.281150114887e+04, 3.109961500387e+04, 2.936440253887e+04, 2.760506605187e+04, 2.582076683887e+04, 2.401062258487e+04, 2.217370456987e+04, 2.030903469087e+04, 1.841558227987e+04, 1.649226073287e+04, 1.453792392387e+04, 1.255136242787e+04, 1.053129954187e+04, 8.476387138871e+03, 6.385201376871e+03, 4.256238301871e+03, 2.087909440871e+03, -1.214625471295e+02, -2.373647991129e+03, -4.670513905129e+03, -7.014026993129e+03, -9.406255579129e+03, -1.184936994013e+04, -1.434564036913e+04, -1.689743202513e+04, -1.950719534613e+04, -2.217745038613e+04, -2.491076291313e+04, -2.770970947813e+04, -3.057682780613e+04, -3.351454793713e+04, -3.652509844413e+04, -3.961038099813e+04, -4.277180583213e+04, -4.601008071813e+04, -4.932494796413e+04, -5.271486866913e+04, -5.617666255313e+04, -5.970512554413e+04, -6.329266481713e+04, -6.692900769413e+04, -7.060104968413e+04, -7.429290127613e+04, -7.798617116213e+04, -8.166049212013e+04, -8.529426557213e+04, -8.886557897313e+04, -9.235323340513e+04, -9.573779869313e+04, -9.900258900413e+04, -1.021344371611e+05, -1.051241773831e+05, -1.079667771901e+05, -1.106609560641e+05, -1.132086063241e+05, -1.156140517961e+05, -1.178832982571e+05, -1.200233810351e+05, -1.220418548011e+05, -1.239464269211e+05, -1.257447115291e+05, -1.274440737801e+05, -1.290515362641e+05, -1.305737261791e+05, -1.320168488321e+05, -1.333866784421e+05, -1.346885608281e+05, -1.359274245671e+05, -1.371077983101e+05, -1.382338324381e+05, -1.393093235431e+05, -1.403377404461e+05, -1.413222506761e+05, -1.422657465731e+05, -1.431708703691e+05, -1.440400378321e+05, -1.448754602041e+05, -1.456791643021e+05, -1.464530107811e+05, -1.471987105791e+05, -1.479178396771e+05, -1.486118522681e+05, -1.492820925031e+05, -1.499298049351e+05, -1.505561438181e+05, -1.511621813791e+05, -1.517489151871e+05, -1.523172747331e+05, -1.528681273021e+05, -1.534022832381e+05, -1.539205006681e+05, -1.544234897481e+05, -1.549119164911e+05, -1.553864062301e+05, -1.558475467401e+05, -1.562958910871e+05, -1.567319602021e+05, -1.571562452351e+05, -1.575692096951e+05, -1.579712914121e+05, -1.583629043251e+05, -1.587444401291e+05, -1.591162697791e+05, -1.594787448771e+05, -1.598321989451e+05, -1.601769486041e+05, -1.605132946511e+05, -1.608415230651e+05, -1.611619059311e+05, -1.614747022941e+05, -1.617801589541e+05, -1.620785112011e+05, -1.623699834961e+05, -1.626547901101e+05, -1.629331357071e+05, -1.632052159001e+05, -1.634712177541e+05, -1.637313202741e+05, -1.639856948371e+05, -1.642345056171e+05, -1.644779099701e+05, -1.647160587951e+05, -1.649490968781e+05, -1.651771632061e+05, -1.654003912681e+05, -1.656189093351e+05, -1.658328407211e+05, -1.660423040311e+05, -1.662474133911e+05, -1.664482786701e+05, -1.666450056811e+05, -1.668376963771e+05, -1.670264490321e+05, -1.672113584141e+05, -1.673925159441e+05, -1.675700098551e+05, -1.677439253291e+05, -1.679143446381e+05, -1.680813472751e+05, -1.682450100691e+05, -1.684054073101e+05, -1.685626108501e+05, -1.687166902141e+05, -1.688677126921e+05, -1.690157434371e+05, -1.691608455511e+05, -1.693030801721e+05, -1.694425065511e+05, -1.695791821291e+05, -1.697131626091e+05, -1.698445020261e+05, -1.699732528111e+05, -1.700994658541e+05, -1.702231905621e+05, -1.703444749181e+05, -1.704633655331e+05, -1.705799076971e+05, -1.706941454311e+05, -1.708061215291e+05, -1.709158776081e+05, -1.710234541461e+05, -1.711288905261e+05, -1.712322250741e+05, -1.713334950961e+05, -1.714327369151e+05, -1.715299859021e+05, -1.716252765121e+05, -1.717186423141e+05, -1.718101160211e+05, -1.718997295211e+05, -1.719875138981e+05, -1.720734994661e+05, -1.721577157921e+05, -1.722401917161e+05, -1.723209553821e+05, -1.724000342531e+05, -1.724774551391e+05, -1.725532442121e+05, -1.726274270291e+05, -1.727000285531e+05, -1.727710731661e+05, -1.728405846921e+05, -1.729085864091e+05, -1.729751010711e+05, -1.730401509211e+05, -1.731037577031e+05, -1.731659426831e+05, -1.732267266581e+05, -1.732861299721e+05, -1.733441725291e+05, -1.734008738051e+05, -1.734562528601e+05, -1.735103283511e+05, -1.735631185401e+05, -1.736146413101e+05, -1.736649141731e+05, -1.737139542791e+05, -1.737617784281e+05, -1.738084030771e+05, -1.738538443521e+05, -1.738981180561e+05, -1.739412396781e+05, -1.739832243971e+05, -1.740240870991e+05, -1.740638423751e+05, -1.741025045341e+05, -1.741400876121e+05, -1.741766053731e+05, -1.742120713201e+05, -1.742464987031e+05, -1.742799005191e+05, -1.743122895271e+05, -1.743436782461e+05, -1.743740789661e+05, -1.744035037511e+05, -1.744319644461e+05, -1.744594726821e+05, -1.744860398791e+05, -1.745116772571e+05, -1.745363958311e+05, -1.745602064261e+05, -1.745831196751e+05, -1.746051460251e+05, -1.746262957441e+05, -1.746465789211e+05, -1.746660054721e+05, -1.746845851451e+05, -1.747023275221e+05, -1.747192420231e+05, -1.747353379121e+05, -1.747506242971e+05, -1.747651101351e+05, -1.747788042371e+05, -1.747917152671e+05, -1.748038517511e+05, -1.748152220741e+05, -1.748258344871e+05, -1.748356971101e+05, -1.748448179311e+05, -1.748532048141e+05, -1.748608654961e+05, -1.748678075951e+05, -1.748740386101e+05, -1.748795659211e+05, -1.748843967981e+05, -1.748885383961e+05, -1.748919977621e+05, -1.748947818361e+05, -1.748968974521e+05, -1.748983513431e+05, -1.748991501401e+05, -1.748993003741e+05, -1.748988084821e+05, -1.748976808041e+05, -1.748959235871e+05, -1.748935429881e+05, -1.748905450741e+05, -1.748869358231e+05, -1.748827211301e+05, -1.748779068011e+05, -1.748724985641e+05, -1.748665020641e+05, -1.748599228641e+05, -1.748527664531e+05, -1.748450382391e+05, -1.748367435591e+05, -1.748278876741e+05, -1.748184757711e+05, -1.748085129681e+05, -1.747980043121e+05, -1.747869547831e+05, -1.747753692911e+05, -1.747632526831e+05, -1.747506097381e+05, -1.747374451731e+05, -1.747237636421e+05, -1.747095697371e+05, -1.746948679901e+05, -1.746796628731e+05, -1.746639588011e+05, -1.746477601301e+05, -1.746310711601e+05, -1.746138961371e+05, -1.745962392501e+05, -1.745781046371e+05, -1.745594963821e+05, -1.745404185181e+05, -1.745208750271e+05, -1.745008698401e+05, -1.744804068401e+05, -1.744594898611e+05, -1.744381226901e+05, -1.744163090681e+05, -1.743940526891e+05, -1.743713572001e+05, -1.743482262071e+05, -1.743246632711e+05, -1.743006719081e+05, -1.742762555951e+05, -1.742514177651e+05, -1.742261618091e+05, -1.742004910811e+05, -1.741744088931e+05, -1.741479185191e+05, -1.741210231921e+05, -1.740937261111e+05, -1.740660304341e+05, -1.740379392861e+05, -1.740094557521e+05, -1.739805828851e+05, -1.739513237001e+05, -1.739216811801e+05, -1.738916582721e+05, -1.738612578901e+05, -1.738304829161e+05, -1.737993361991e+05, -1.737678205561e+05, -1.737359387721e+05, -1.737036936021e+05, -1.736710877701e+05, -1.736381239701e+05, -1.736048048651e+05, -1.735711330921e+05, -1.735371112551e+05, -1.735027419341e+05, -1.734680276771e+05, -1.734329710061e+05, -1.733975744181e+05, -1.733618403791e+05, -1.733257713331e+05, -1.732893696951e+05, -1.732526378541e+05, -1.732155781771e+05, -1.731781930031e+05, -1.731404846481e+05, -1.731024554021e+05, -1.730641075341e+05, -1.730254432861e+05, -1.729864648801e+05, -1.729471745121e+05, -1.729075743581e+05, -1.728676665711e+05, -1.728274532811e+05, -1.727869365981e+05, -1.727461186081e+05, -1.727050013781e+05, -1.726635869551e+05, -1.726218773631e+05, -1.725798746071e+05, -1.725375806721e+05, -1.724949975231e+05, -1.724521271051e+05, -1.724089713461e+05, -1.723655321521e+05, -1.723218114131e+05, -1.722778109991e+05, -1.722335327621e+05, -1.721889785351e+05, -1.721441501361e+05, -1.720990493631e+05, -1.720536779981e+05, -1.720080378041e+05, -1.719621305311e+05, -1.719159579071e+05, -1.718695216491e+05, -1.718228234541e+05, -1.717758650041e+05, -1.717286479671e+05, -1.716811739921e+05, -1.716334447141e+05, -1.715854617551e+05, -1.715372267191e+05, -1.714887411961e+05, -1.714400067621e+05, -1.713910249781e+05, -1.713417973901e+05, -1.712923255311e+05, -1.712426109201e+05, -1.711926550611e+05, -1.711424594461e+05, -1.710920255511e+05, -1.710413548421e+05, -1.709904487701e+05, -1.709393087721e+05, -1.708879362751e+05, -1.708363326901e+05, -1.707844994191e+05, -1.707324378471e+05, -1.706801493521e+05, -1.706276352971e+05, -1.705748970331e+05, -1.705219358991e+05, -1.704687532231e+05, -1.704153503231e+05, -1.703617285021e+05, -1.703078890561e+05, -1.702538332651e+05, -1.701995624021e+05, -1.701450777271e+05, -1.700903804911e+05, -1.700354719321e+05, -1.699803532791e+05, -1.699250257511e+05, -1.698694905541e+05, -1.698137488891e+05, -1.697578019411e+05, -1.697016508891e+05, -1.696452969001e+05, -1.695887411341e+05, -1.695319847381e+05, -1.694750288521e+05, -1.694178746051e+05, -1.693605231181e+05, -1.693029755011e+05, -1.692452328581e+05, -1.691872962811e+05, -1.691291668541e+05, -1.690708456531e+05, -1.690123337441e+05, -1.689536321851e+05, -1.688947420251e+05, -1.688356643071e+05, -1.687764000621e+05, -1.687169503151e+05, -1.686573160821e+05, -1.685974983711e+05, -1.685374981831e+05, -1.684773165101e+05, -1.684169543361e+05, -1.683564126381e+05, -1.682956923851e+05, -1.682347945381e+05, -1.681737200521e+05, -1.681124698731e+05, -1.680510449401e+05, -1.679894461841e+05, -1.679276745321e+05, -1.678657308991e+05, -1.678036161981e+05, -1.677413313301e+05, -1.676788771931e+05, -1.676162546771e+05, -1.675534646641e+05, -1.674905080311e+05, -1.674273856481e+05, -1.673640983771e+05, -1.673006470751e+05, -1.672370325931e+05, -1.671732557731e+05, -1.671093174541e+05, -1.670452184671e+05, -1.669809596361e+05, -1.669165417811e+05, -1.668519657141e+05, -1.667872322431e+05, -1.667223421671e+05, -1.666572962831e+05, -1.665920953791e+05, -1.665267402391e+05, -1.664612316401e+05, -1.663955703551e+05, -1.663297571501e+05, -1.662637927851e+05, -1.661976780161e+05, -1.661314135931e+05, -1.660650002611e+05, -1.659984387581e+05, -1.659317298191e+05, -1.658648741721e+05, -1.657978725411e+05, -1.657307256431e+05, -1.656634341931e+05, -1.655959988991e+05, -1.655284204621e+05, -1.654606995831e+05, -1.653928369531e+05, -1.653248332631e+05, -1.652566891941e+05, -1.651884054271e+05, -1.651199826341e+05, -1.650514214871e+05},
    {5.560181498387e+04, 5.416011372187e+04, 5.270337158487e+04, 5.123118453787e+04, 4.974313139087e+04, 4.823877286987e+04, 4.671765062387e+04, 4.517928617587e+04, 4.362317979487e+04, 4.204880930187e+04, 4.045562879087e+04, 3.884306726687e+04, 3.721052719087e+04, 3.555738293487e+04, 3.388297911887e+04, 3.218662885287e+04, 3.046761185287e+04, 2.872517243387e+04, 2.695851737387e+04, 2.516681363687e+04, 2.334918596487e+04, 2.150471430387e+04, 1.963243109887e+04, 1.773131842387e+04, 1.580030497487e+04, 1.383826291687e+04, 1.184400460687e+04, 9.816279217870e+03, 7.753769289871e+03, 5.655087270871e+03, 3.518772109871e+03, 1.343286007871e+03, -8.729885361295e+02, -3.131751195130e+03, -5.434785552129e+03, -7.783959761129e+03, -1.018122568813e+04, -1.262861593513e+04, -1.512823795113e+04, -1.768226422613e+04, -2.029291719113e+04, -2.296244711913e+04, -2.569310076613e+04, -2.848707794213e+04, -3.134647245713e+04, -3.427319315913e+04, -3.726886002013e+04, -4.033466972613e+04, -4.347122529313e+04, -4.667832546513e+04, -4.995471279713e+04, -5.329778537713e+04, -5.670328653913e+04, -6.016499940413e+04, -6.367448615113e+04, -6.722092137013e+04, -7.079106932013e+04, -7.436944316613e+04, -7.793866195213e+04, -8.147999481613e+04, -8.497405954013e+04, -8.840162750513e+04, -9.174447640413e+04, -9.498622100413e+04, -9.811304118013e+04, -1.011142240311e+05, -1.039824638541e+05, -1.067138886021e+05, -1.093077495151e+05, -1.117659260491e+05, -1.140923209621e+05, -1.162922553711e+05, -1.183719385431e+05, -1.203380470361e+05, -1.221974162151e+05, -1.239568284731e+05, -1.256228756191e+05, -1.272018737321e+05, -1.286998131961e+05, -1.301223318121e+05, -1.314747031421e+05, -1.327618353091e+05, -1.339882772831e+05, -1.351582306701e+05, -1.362755655701e+05, -1.373438392521e+05, -1.383663166271e+05, -1.393459916131e+05, -1.402856086301e+05, -1.411876836641e+05, -1.420545244411e+05, -1.428882494281e+05, -1.436908054841e+05, -1.444639840731e+05, -1.452094360301e+05, -1.459286849241e+05, -1.466231390901e+05, -1.472941024281e+05, -1.479427840651e+05, -1.485703070041e+05, -1.491777158521e+05, -1.497659837311e+05, -1.503360184621e+05, -1.508886681121e+05, -1.514247259641e+05, -1.519449349981e+05, -1.524499919151e+05, -1.529405507791e+05, -1.534172263081e+05, -1.538805968601e+05, -1.543312071351e+05, -1.547695706501e+05, -1.551961719781e+05, -1.556114688051e+05, -1.560158938081e+05, -1.564098563711e+05, -1.567937441671e+05, -1.571679246051e+05, -1.575327461661e+05, -1.578885396341e+05, -1.582356192241e+05, -1.585742836371e+05, -1.589048170191e+05, -1.592274898671e+05, -1.595425598521e+05, -1.598502725901e+05, -1.601508623591e+05, -1.604445527621e+05, -1.607315573451e+05, -1.610120801711e+05, -1.612863163551e+05, -1.615544525661e+05, -1.618166674931e+05, -1.620731322771e+05, -1.623240109201e+05, -1.625694606671e+05, -1.628096323591e+05, -1.630446707691e+05, -1.632747149141e+05, -1.634998983491e+05, -1.637203494421e+05, -1.639361916331e+05, -1.641475436761e+05, -1.643545198691e+05, -1.645572302691e+05, -1.647557808961e+05, -1.649502739221e+05, -1.651408078501e+05, -1.653274776891e+05, -1.655103751091e+05, -1.656895885971e+05, -1.658652035951e+05, -1.660373026431e+05, -1.662059654981e+05, -1.663712692621e+05, -1.665332884951e+05, -1.666920953231e+05, -1.668477595421e+05, -1.670003487151e+05, -1.671499282671e+05, -1.672965615701e+05, -1.674403100301e+05, -1.675812331631e+05, -1.677193886741e+05, -1.678548325281e+05, -1.679876190181e+05, -1.681178008301e+05, -1.682454291051e+05, -1.683705535011e+05, -1.684932222451e+05, -1.686134821891e+05, -1.687313788631e+05, -1.688469565201e+05, -1.689602581871e+05, -1.690713257061e+05, -1.691801997801e+05, -1.692869200121e+05, -1.693915249421e+05, -1.694940520891e+05, -1.695945379831e+05, -1.696930182011e+05, -1.697895274001e+05, -1.698840993441e+05, -1.699767669421e+05, -1.700675622691e+05, -1.701565165981e+05, -1.702436604271e+05, -1.703290234991e+05, -1.704126348341e+05, -1.704945227481e+05, -1.705747148731e+05, -1.706532381861e+05, -1.707301190231e+05, -1.708053831011e+05, -1.708790555391e+05, -1.709511608741e+05, -1.710217230811e+05, -1.710907655891e+05, -1.711583112951e+05, -1.712243825841e+05, -1.712890013431e+05, -1.713521889731e+05, -1.714139664071e+05, -1.714743541221e+05, -1.715333721511e+05, -1.715910400971e+05, -1.716473771451e+05, -1.717024020741e+05, -1.717561332691e+05, -1.718085887291e+05, -1.718597860821e+05, -1.719097425921e+05, -1.719584751701e+05, -1.720060003831e+05, -1.720523344651e+05, -1.720974933241e+05, -1.721414925521e+05, -1.721843474311e+05, -1.722260729461e+05, -1.722666837861e+05, -1.723061943581e+05, -1.723446187901e+05, -1.723819709401e+05, -1.724182644021e+05, -1.724535125161e+05, -1.724877283671e+05, -1.725209247991e+05, -1.725531144171e+05, -1.725843095951e+05, -1.726145224791e+05, -1.726437649931e+05, -1.726720488471e+05, -1.726993855421e+05, -1.727257863691e+05, -1.727512624231e+05, -1.727758246001e+05, -1.727994836071e+05, -1.728222499631e+05, -1.728441340031e+05, -1.728651458881e+05, -1.728852956001e+05, -1.729045929541e+05, -1.729230475981e+05, -1.729406690171e+05, -1.729574665371e+05, -1.729734493291e+05, -1.729886264121e+05, -1.730030066551e+05, -1.730165987851e+05, -1.730294113831e+05, -1.730414528941e+05, -1.730527316251e+05, -1.730632557501e+05, -1.730730333131e+05, -1.730820722321e+05, -1.730903802971e+05, -1.730979651781e+05, -1.731048344241e+05, -1.731109954691e+05, -1.731164556291e+05, -1.731212221111e+05, -1.731253020091e+05, -1.731287023101e+05, -1.731314298971e+05, -1.731334915471e+05, -1.731348939361e+05, -1.731356436421e+05, -1.731357471451e+05, -1.731352108271e+05, -1.731340409801e+05, -1.731322438021e+05, -1.731298254011e+05, -1.731267917981e+05, -1.731231489241e+05, -1.731189026301e+05, -1.731140586801e+05, -1.731086227581e+05, -1.731026004661e+05, -1.730959973281e+05, -1.730888187931e+05, -1.730810702301e+05, -1.730727569381e+05, -1.730638841391e+05, -1.730544569871e+05, -1.730444805631e+05, -1.730339598801e+05, -1.730228998821e+05, -1.730113054481e+05, -1.729991813911e+05, -1.729865324591e+05, -1.729733633391e+05, -1.729596786521e+05, -1.729454829631e+05, -1.729307807731e+05, -1.729155765271e+05, -1.728998746111e+05, -1.728836793551e+05, -1.728669950311e+05, -1.728498258601e+05, -1.728321760051e+05, -1.728140495791e+05, -1.727954506431e+05, -1.727763832031e+05, -1.727568512201e+05, -1.727368586011e+05, -1.727164092081e+05, -1.726955068521e+05, -1.726741552981e+05, -1.726523582671e+05, -1.726301194301e+05, -1.726074424181e+05, -1.725843308131e+05, -1.725607881591e+05, -1.725368179521e+05, -1.725124236511e+05, -1.724876086691e+05, -1.724623763831e+05, -1.724367301261e+05, -1.724106731941e+05, -1.723842088431e+05, -1.723573402921e+05, -1.723300707221e+05, -1.723024032761e+05, -1.722743410621e+05, -1.722458871531e+05, -1.722170445831e+05, -1.721878163561e+05, -1.721582054381e+05, -1.721282147631e+05, -1.720978472321e+05, -1.720671057131e+05, -1.720359930411e+05, -1.720045120211e+05, -1.719726654251e+05, -1.719404559961e+05, -1.719078864441e+05, -1.718749594521e+05, -1.718416776731e+05, -1.718080437301e+05, -1.717740602171e+05, -1.717397297011e+05, -1.717050547211e+05, -1.716700377891e+05, -1.716346813901e+05, -1.715989879801e+05, -1.715629599931e+05, -1.715265998331e+05, -1.714899098831e+05, -1.714528924961e+05, -1.714155500041e+05, -1.713778847121e+05, -1.713398989041e+05, -1.713015948381e+05, -1.712629747481e+05, -1.712240408461e+05, -1.711847953231e+05, -1.711452403441e+05, -1.711053780541e+05, -1.710652105771e+05, -1.710247400131e+05, -1.709839684421e+05, -1.709428979231e+05, -1.709015304951e+05, -1.708598681751e+05, -1.708179129621e+05, -1.707756668341e+05, -1.707331317481e+05, -1.706903096431e+05, -1.706472024411e+05, -1.706038120401e+05, -1.705601403251e+05, -1.705161891591e+05, -1.704719603881e+05, -1.704274558401e+05, -1.703826773261e+05, -1.703376266381e+05, -1.702923055531e+05, -1.702467158291e+05, -1.702008592091e+05, -1.701547374181e+05, -1.701083521641e+05, -1.700617051421e+05, -1.700147980281e+05, -1.699676324841e+05, -1.699202101561e+05, -1.698725326741e+05, -1.698246016521e+05, -1.697764186931e+05, -1.697279853811e+05, -1.696793032881e+05, -1.696303739701e+05, -1.695811989681e+05, -1.695317798131e+05, -1.694821180171e+05, -1.694322150831e+05, -1.693820724951e+05, -1.693316917301e+05, -1.692810742461e+05, -1.692302214911e+05, -1.691791349001e+05, -1.691278158941e+05, -1.690762658821e+05, -1.690244862611e+05, -1.689724784151e+05, -1.689202437161e+05, -1.688677835241e+05, -1.688150991881e+05, -1.687621920431e+05, -1.687090634141e+05, -1.686557146151e+05, -1.686021469471e+05, -1.685483617021e+05, -1.684943601581e+05, -1.684401435841e+05, -1.683857132381e+05, -1.683310703671e+05, -1.682762162081e+05, -1.682211519851e+05, -1.681658789161e+05, -1.681103982051e+05, -1.680547110481e+05, -1.679988186291e+05, -1.679427221251e+05, -1.678864227001e+05, -1.678299215101e+05, -1.677732197011e+05, -1.677163184111e+05, -1.676592187671e+05, -1.676019218861e+05, -1.675444288781e+05, -1.674867408431e+05, -1.674288588721e+05, -1.673707840461e+05, -1.673125174381e+05, -1.672540601151e+05, -1.671954131311e+05, -1.671365775341e+05, -1.670775543631e+05, -1.670183446491e+05, -1.669589494141e+05, -1.668993696741e+05, -1.668396064341e+05, -1.667796606931e+05, -1.667195334411e+05, -1.666592256611e+05, -1.665987383281e+05, -1.665380724101e+05, -1.664772288661e+05, -1.664162086491e+05, -1.663550127041e+05, -1.662936419681e+05, -1.662320973711e+05, -1.661703798371e+05, -1.661084902821e+05, -1.660464296161e+05, -1.659841987391e+05, -1.659217985481e+05, -1.658592299301e+05, -1.657964937681e+05, -1.657335909361e+05, -1.656705223031e+05, -1.656072887301e+05, -1.655438910741e+05, -1.654803301821e+05, -1.654166068981e+05, -1.653527220571e+05, -1.652886764901e+05, -1.652244710211e+05, -1.651601064681e+05, -1.650955836411e+05, -1.650309033471e+05, -1.649660663871e+05, -1.649010735531e+05, -1.648359256341e+05, -1.647706234121e+05, -1.647051676641e+05, -1.646395591621e+05, -1.645737986701e+05, -1.645078869481e+05, -1.644418247521e+05, -1.643756128291e+05, -1.643092519241e+05, -1.642427427751e+05, -1.641760861161e+05, -1.641092826731e+05, -1.640423331701e+05, -1.639752383231e+05, -1.639079988461e+05, -1.638406154471e+05, -1.637730888261e+05, -1.637054196821e+05, -1.636376087081e+05, -1.635696565