#!/bin/bash
######################################################################
#                                                                    #
#  Copyright (c) 2012, 2025 NoMachine, http://www.nomachine.com.     #
#                                                                    #
#  All rights reserved.                                              #
#                                                                    #
######################################################################


RunDir=${0%nxreportsystemload.sh}
RestrictedDir=$(cd "${RunDir}/../restricted" && pwd)

PARAMS_COUNT=1

. "${RestrictedDir}"/nxfunct.sh

#
# Getting system load sample based on values from /proc/stat.
#

setLastSample()
{
  procsRunning=-1
  cpuCount=-1
  procsBlocked=-1

  while IFS='' read -r line || [[ -n "$line" ]]; do
    if [[ $line =~ (cpu)([0-9]+) ]]; then cpuCount="${BASH_REMATCH[2]}"; fi
    if [[ $line =~ (procs_running )([0-9]+) ]]; then procsRunning="${BASH_REMATCH[2]}"; fi
    if [[ $line =~ (procs_blocked )([0-9]+) ]]; then procsBlocked="${BASH_REMATCH[2]}"; fi
  done < "/proc/stat"

  #
  # As tests showed /proc/stat might provide corrupted data for some checks,
  # we need to ignore sample when that happens.
  #

  if [ $procsRunning -eq -1 ] || [ $cpuCount -eq -1 ] || [ $procsBlocked -eq -1 ]; then
    lastSample=-1
    return
  fi

  ((cpuCount+=1))
  ((procsRunning-=1))

  #
  # Counting system load sample with formula:
  # load = (procs_running / cpus) + procs_blocked
  # result is being normalized, so that load is integer with lowest value being 0.
  #

  procsPerCpu=$((100 * $procsRunning/$cpuCount))

  sample=$(($procsPerCpu+$procsBlocked * 100))

  lastSample=$sample
}

#
# Setting average system load based on system load samples
# for period defined by period variable.
#

setAverageSystemLoad()
{
  setLastSample

  if [ $lastSample -eq -1 ]; then
    return
  fi

  if [ $numberOfSamples -le $maxNumberOfSamples ]; then
    ((numberOfSamples+=1))
  fi

  #
  # Add new sample and remove oldest sample from samples array.
  #

  samplesArray=("${samplesArray[@]:1}")

  samplesArray+=($lastSample)

  allSamples=0

  for i in "${samplesArray[@]}"
  do
     : 
     ((allSamples+=$i))
  done

  systemLoad=$((allSamples/numberOfSamples))
}

die()
{
  ${COMMAND_ECHO} "ERROR! Exiting nxreportsystemload.sh '$1'" 1>&2
  exit 1
}

#
# Retrieve current system load and print it to nx system load file.
#

reportLoad()
{
  setAverageSystemLoad

  ${COMMAND_ECHO} "$systemLoad" > "$nxSysLoadFile"

  if [ $? -ne 0 ]; then
    die "Cannot write to system load file $nxSysLoadFile"
  fi
}

#
# Path to nx system load file, this is the file we report system load to.
#

nxSysLoadFile=$1

systemLoad=0
lastSample=0

#
# Period of time in seconds.
#

period=60

#
# Delay between checks in seconds.
#

delay=1

maxNumberOfSamples=$(($period / $delay))

numberOfSamples=0

#
# Initializing samples array.
#

samplesArray=()

for (( i = 0; i < $maxNumberOfSamples; i++ )) 
do 
  samplesArray[$i]=0
done

#
# Counter that is needed to check if parent is alive
# when iteration reaches threshold.
#

counter=1
parentCheckThreshold=100

#
# Main loop.
# Reports system load and sleeps for seconds defined by delay variable.
#

while true; do
  #
  # Terminate script if parent died without terminating us.
  #

  if [ $counter -ge $parentCheckThreshold ]; then
    statsArray=($(</proc/$$/stat))
    currentPpid=${statsArray[3]}

    if [ $PPID -ne $currentPpid ]; then
      die "Parent process with pid '$PPID' has been terminated"
    else
      counter=1
    fi
  else
    ((counter+=1))
  fi

  reportLoad

  sleep $delay
done

exit 0
