/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2011-2024 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::XiModel

Description
    Base-class for all Xi models used by the b-Xi combustion model

    References:
    \verbatim
        Weller, H. G. (1993).
        The development of a new flame area combustion model
        using conditional averaging.
        Thermo-fluids section report TF 9307.

        Weller, H. G., Tabor, G., Gosman, A. D., & Fureby, C. (1998, January).
        Application of a flame-wrinkling LES combustion model
        to a turbulent mixing layer.
        In Symposium (International) on combustion
        (Vol. 27, No. 1, pp. 899-907). Elsevier.
    \endverbatim

    Xi is given through an algebraic expression (Foam::XiModels::algebraic), by
    solving a transport equation (Foam::XiModels::transport.H) or a fixed value
    (Foam::XiModels::fixed).

    In the algebraic and transport methods \f$\Xi_{eq}\f$ is calculated in
    similar way. In the algebraic approach, \f$\Xi_{eq}\f$ is the value used in
    the \f$ b \f$ transport equation.

    \f$\Xi_{eq}\f$ is calculated as follows:

    \f$\Xi_{eq} = 1 + (1 + 2\Xi_{coeff}(0.5 - \dwea{b}))(\Xi^* - 1)\f$

    where:

        \f$ \dwea{b} \f$ is the regress variable.

        \f$ \Xi_{coeff} \f$ is a model constant.

        \f$ \Xi^* \f$ is the total equilibrium wrinkling combining the effects
        of the flame instability and turbulence interaction and is given by

        \f[
            \Xi^* = \frac {R}{R - G_\eta - G_{in}}
        \f]

    where:

        \f$ G_\eta \f$ is the generation rate of wrinkling due to turbulence
        interaction.

        \f$ G_{in} = \kappa \rho_{u}/\rho_{b} \f$ is the generation
         rate due to the flame instability.

    By adding the removal rates of the two effects:

        \f[
            R = G_\eta \frac{\Xi_{\eta_{eq}}}{\Xi_{\eta_{eq}} - 1}
              + G_{in} \frac{\Xi_{{in}_{eq}}}{\Xi_{{in}_{eq}} - 1}
        \f]

    where:

        \f$ R \f$ is the total removal.

        \f$ G_\eta \f$ is a model constant.

        \f$ \Xi_{\eta_{eq}} \f$ is the flame wrinkling due to turbulence.

        \f$ \Xi_{{in}_{eq}} \f$ is the equilibrium level of the flame wrinkling
        generated by instability. It is a constant (default 2.5).


SourceFiles
    XiModel.C

\*---------------------------------------------------------------------------*/

#ifndef XiModel_H
#define XiModel_H

#include "psiuMulticomponentThermo.H"
#include "fluidThermoThermophysicalTransportModels.H"
#include "runTimeSelectionTables.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                           Class XiModel Declaration
\*---------------------------------------------------------------------------*/

class XiModel
{

protected:

    // Protected data

        //- Thermo
        const psiuMulticomponentThermo& thermo_;

        //- Thermo-physical transport
        const fluidThermoThermophysicalTransportModel& thermoTransport_;

        //- Turbulence
        const compressibleMomentumTransportModel& turbulence_;

        const volScalarField& Su_;

        const volScalarField& rho_;

        const volScalarField& b_;

        //- Flame wrinkling field
        volScalarField Xi_;


    // Protected member functions

        //- Update coefficients from given dictionary
        virtual bool readCoeffs(const dictionary& dict) = 0;


public:

    //- Runtime type information
    TypeName("XiModel");


    // Declare run-time constructor selection table

        declareRunTimeSelectionTable
        (
            autoPtr,
            XiModel,
            dictionary,
            (
                const dictionary& dict,
                const psiuMulticomponentThermo& thermo,
                const fluidThermoThermophysicalTransportModel& turbulence,
                const volScalarField& Su
            ),
            (
                dict,
                thermo,
                turbulence,
                Su
            )
        );


    // Constructors

        //- Construct from components
        XiModel
        (
            const psiuMulticomponentThermo& thermo,
            const fluidThermoThermophysicalTransportModel& turbulence,
            const volScalarField& Su
        );

        //- Disallow default bitwise copy construction
        XiModel(const XiModel&) = delete;


    // Selectors

        //- Return a reference to the selected Xi model
        static autoPtr<XiModel> New
        (
            const dictionary& combustionProperties,
            const psiuMulticomponentThermo& thermo,
            const fluidThermoThermophysicalTransportModel& turbulence,
            const volScalarField& Su
        );


    //- Destructor
    virtual ~XiModel();


    // Member Functions

        //- Return the flame-wrinkling Xi
        virtual const volScalarField& Xi() const
        {
            return Xi_;
        }

        //- Return the flame diffusivity
        virtual tmp<volScalarField> Db() const
        {
            return volScalarField::New
            (
                "Db",
                turbulence_.rho()*turbulence_.nuEff()
            );
        }

        //- Add Xi to the multivariateSurfaceInterpolationScheme table
        //  if required
        virtual void addXi
        (
            multivariateSurfaceInterpolationScheme<scalar>::fieldTable&
        )
        {}

        //- Reset Xi to the unburnt state (1)
        virtual void reset();

        //- Correct the flame-wrinkling Xi
        virtual void correct() = 0;

        //- Update properties from the given combustionProperties dictionary
        bool read(const dictionary& combustionProperties);


    // Member Operators

        //- Disallow default bitwise assignment
        void operator=(const XiModel&) = delete;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
