/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2011-2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::fvCellZone

Description
    cellZone selection or generation class with caching of zone volume

    for fvModels, fvConstraints, functionObjects etc.

    The cellZone is either looked-up from the mesh or generated by a
    zoneGenerator and stored locally.

Usage
    Examples:
    \verbatim
        // Select all cells
        cellZone all;

        // Select the cells within the given cellZone
        cellZone rotor;

        // Select the cells within the given box
        cellZone
        {
            type        box;
            box         (3.48 -0.1 -0.1) (3.5 0.1 0.1);
        }

        // Select the cell containing the given point
        // and update the cells with respect to the fixed points
        // if the mesh moves
        cellZone
        {
            type            containsPoints;
            moveUpdate      true;
            points          ((0.075 0.2 0.05));
        }
    \endverbatim

See also
    Foam::generatedCellZone
    Foam::generatedZoneSet
    Foam::zoneGenerator
    Foam::fvModel
    Foam::fvConstraint

SourceFiles
    fvCellZone.C
    fvCellZoneI.H

\*---------------------------------------------------------------------------*/

#ifndef fvCellZone_H
#define fvCellZone_H

#include "generatedCellZone.H"
#include "writeFile.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class fvMesh;

/*---------------------------------------------------------------------------*\
                        Class fvCellZone Declaration
\*---------------------------------------------------------------------------*/

class fvCellZone
:
    public generatedCellZone
{
    // Private data

        //- Reference to the mesh
        const fvMesh& mesh_;

        //- The global number of cells
        mutable label nGlobalCells_;

        //- Sum of cell volumes
        mutable scalar V_;


    // Private functions

        //- Update the global number of cells and the sum of cell volumes
        void update();


public:

    // Constructors

        //- Construct from mesh selecting all cells
        fvCellZone(const fvMesh& mesh);

        //- Construct from mesh and dictionary
        fvCellZone(const fvMesh& mesh, const dictionary& dict);


    //- Destructor
    ~fvCellZone();


    // Member Functions

        //- Return const access to the global number of cells
        inline label nGlobalCells() const;

        //- Return const access to the total cell volume
        inline scalar V() const;

        //- Output file header information
        void writeFileHeader
        (
            const functionObjects::writeFile& wf,
            Ostream& file
        );


        // Mesh changes

            //- Update for mesh motion
            void movePoints();

            //- Update topology using the given map
            void topoChange(const polyTopoChangeMap&);

            //- Update from another mesh using the given map
            void mapMesh(const polyMeshMap&);

            //- Redistribute or update using the given distribution map
            void distribute(const polyDistributionMap&);


        // IO

            //- Read coefficients dictionary
            bool read(const dictionary& dict);
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "fvCellZoneI.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
