/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::zoneGenerators::truncatedCone

Description
    A zoneGenerator which selects points, cells or faces with centres either
    inside or outside a truncatedCone.

    By default all the points, cells or faces of the mesh are tested for being
    inside or outside the truncatedCone but an optional single zone or list of
    zones may be provided and those points, cells or faces are tested instead.
    This provides an efficient method of hierarchical sub-division of space
    where an initial selection of points cells or faces is refined by selecting
    the sub-set inside or outside the given truncatedCone rather than having to
    generate another zone to intersect with.

Usage
    \table
        Property     | Description             | Required  | Default value
        type         | Type: truncatedCone     | yes       |
        name         | Name of the zone        | no        | zoneGenerator name
        zoneType     | Type of zone            | yes       |
        select       | Select either the inside or outside | no  | inside
        moveUpdate   | Switch to update after mesh motion  | no  | false
        point1       | First point on cone axis            | yes |
        radius1      | Radius of the cone at point1        | yes |
        point2       | Second point on cone axis           | yes |
        radius2      | Radius of the cone at point2        | yes |
        zone         | Optional zone to sub-set            | no  |
        zones        | Optional list of zones to sub-set   | no  |
    \endtable

    A cellZone named \c cone1 containing the cells with centres inside
    a cone can be generated by
    \verbatim
        cone1
        {
            type        truncatedCone;
            zoneType    cell;

            point1      (-0.0075 0 -1);
            point2      (-0.003 0.0025 1);
            radius1     0.001;
            radius2     0.001;
        }
    \endverbatim

See also
    zoneGenerators::volume

SourceFiles
    truncatedCone.C

\*---------------------------------------------------------------------------*/

#ifndef truncatedCone_zoneGenerator_H
#define truncatedCone_zoneGenerator_H

#include "volume.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace zoneGenerators
{

/*---------------------------------------------------------------------------*\
                              Class truncatedCone Declaration
\*---------------------------------------------------------------------------*/

class truncatedCone
:
    public volume
{
    // Private Data

        //- First point on cone axis
        vector point1_;

        //- Second point on cone axis
        vector point2_;

        //- Radius 1
        scalar radius1_;

        //- Radius 2
        scalar radius2_;

        //- Axis vector calculated from point1_ and point2_
        vector axis_;

        //- Magnitude of the axis vector squared calculated from axis_
        scalar magAxis2_;


    // Private Member Functions

        friend class volume;

        //- Return true if the truncatedConees contain the given point
        inline bool contains(const point& p) const;


public:

    //- Runtime type information
    TypeName("truncatedCone");


    // Constructors

        //- Construct from dictionary
        truncatedCone
        (
            const word& name,
            const polyMesh& mesh,
            const dictionary& dict
        );


    //- Destructor
    virtual ~truncatedCone();


    // Member Functions

        virtual zoneSet generate() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace zoneGenerators
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
