/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2022-2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::generatedFaceZone

Description
    faceZone selection or generation class

    The faceZone is either looked-up from the mesh or generated by a
    zoneGenerator and stored locally.

Usage
    Examples:
    \verbatim
        // Select the faces within the faceZone rotor
        faceZone rotor;

        // Select the faces within the given box
        faceZone
        {
            type        box;
            box         (3.48 -0.1 -0.1) (3.5 0.1 0.1);
        }

        // Select the faces representing the given plane
        faceZone
        {
            type        plane;
            point       (0.41 0 0);
            normal      (1 0 0);
        }
    \endverbatim

See also
    Foam::generatedZoneSet
    Foam::zoneGenerator

SourceFiles
    generatedFaceZone.C
    generatedFaceZoneI.H

\*---------------------------------------------------------------------------*/

#ifndef generatedFaceZone_H
#define generatedFaceZone_H

#include "generatedZoneSet.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class polyMesh;
class polyMeshMap;
class polyDistributionMap;

/*---------------------------------------------------------------------------*\
                        Class generatedFaceZone Declaration
\*---------------------------------------------------------------------------*/

class generatedFaceZone
{
    // Private data

        //- Reference to the mesh
        const polyMesh& mesh_;

        generatedZoneSet faceZone_;


public:

    // Constructors

        //- Construct from mesh and dictionary
        generatedFaceZone(const polyMesh& mesh, const dictionary& dict);

        //- Disallow default bitwise copy construction
        generatedFaceZone(const generatedFaceZone&) = delete;


    //- Destructor
    ~generatedFaceZone();


    // Member Functions

        // Access

            //- Return const access to the name of faceZone
            inline const word& name() const;

            //- Return const access to the face set
            inline const faceZone& zone() const;

            //- Return the number of faces in the set
            inline label nFaces() const;

            //- Return the face index corresponding to the face set index
            inline label facei(const label i) const;


        // Mesh changes

            //- Update for mesh motion
            void movePoints();

            //- Update topology using the given map
            void topoChange(const polyTopoChangeMap&);

            //- Update from another mesh using the given map
            void mapMesh(const polyMeshMap&);

            //- Redistribute or update using the given distribution map
            void distribute(const polyDistributionMap&);


        // IO

            //- Read coefficients dictionary
            bool read(const dictionary& dict);


    // Member Operators

        //- Disallow default bitwise assignment
        void operator=(const generatedFaceZone&) = delete;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "generatedFaceZoneI.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
