/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::CloudDerivedField

Description
    A field derived from other state fields of the cloud. Stores and
    virtualises a function or a method which generates the field. Caches the
    result to prevent unnecessary re-evaluation during a calculation step.
    Provides convenient access for a variety of contexts; i.e., full mesh,
    sub-mesh and model.

SourceFiles
    CloudDerivedField.C

\*---------------------------------------------------------------------------*/

#ifndef CloudDerivedField_H
#define CloudDerivedField_H

#include "LagrangianMesh.H"
#include "LagrangianModelRef.H"
#include "LagrangianFieldsFwd.H"
#include "LagrangianSubFieldsFwd.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                       Class CloudDerivedField Declaration
\*---------------------------------------------------------------------------*/

template<class Type>
class CloudDerivedField
{
protected:

    // Protected Classes

        //- Class to virtualise an evaluation procedure
        class Functor;

        //- Class to store an evaluation function
        template<class F>
        class Function;

        //- Class to store an evaluation method
        template<class C>
        class Method;

        //- Class to convert the stored sub-all-field reference into a
        //  temporary Lagrangian field
        class AllFieldToField;


private:

    // Private Data

        //- The name. May be null, in which case the name is automatically
        //  generated by the field algebra.
        const word& name_;

        //- The function
        const autoPtr<Functor> functorPtr_;

        //- The cached Lagrangian sub-all-field
        mutable autoPtr<LagrangianSubField<Type>> psiAllPtr_;

        //- The cached Lagrangian sub-field
        mutable autoPtr<LagrangianSubField<Type>> psiSubPtr_;

        //- Flag to indicate whether the sub-field is up to date
        mutable bool psiSubUpToDate_;

        //- The index of the sub-mesh
        mutable label psiSubMeshIndex_;

        //- The cached Lagrangian sub-sub-field
        mutable autoPtr<LagrangianSubSubField<Type>> psiSubSubPtr_;

        //- Flag to indicate whether the sub-field is up to date
        mutable bool psiSubSubUpToDate_;

        //- The index of the sub-sub-mesh
        mutable label psiSubSubMeshIndex_;


public:

    // Constructors

        //- Construct from a name and a function
        template<class F>
        CloudDerivedField(const word& name, const F& f);

        //- Construct from a function
        template<class F>
        CloudDerivedField(const F& f);

        //- Construct from a name, class and a method
        template<class C>
        CloudDerivedField
        (
            const word& name,
            const C& c,
            tmp<LagrangianSubField<Type>> (C::*m)
            (
                const LagrangianModelRef&,
                const LagrangianSubMesh&
            ) const
        );

        //- Construct from a class and a method
        template<class C>
        CloudDerivedField
        (
            const C& c,
            tmp<LagrangianSubField<Type>> (C::*m)
            (
                const LagrangianModelRef&,
                const LagrangianSubMesh&
            ) const
        );

        //- Disallow default bitwise copy construction
        CloudDerivedField(const CloudDerivedField&) = delete;


    // Member Functions

        //- Compute and return an independent copy of the entire field
        tmp<LagrangianInternalField<Type>> field
        (
            const LagrangianMesh& mesh
        ) const;

        //- Access a part of the field
        LagrangianSubSubField<Type>& ref(const LagrangianSubMesh&) const;

        //- Clear
        void clear(const bool final);


    // Member Operators

        //- Compute and return the entire field. This will be a slice of the
        //  cached sub-all-field, for which the lifetime is not guaranteed. If
        //  the result is being stored (e.g., by a function object) then this
        //  might not be appropriate, and the 'field' method should be used
        //  instead.
        tmp<LagrangianInternalField<Type>> operator()
        (
            const LagrangianMesh& mesh
        ) const;

        //- Compute and access a part of the field
        const LagrangianSubField<Type>& operator()
        (
            const LagrangianSubMesh& subMesh
        ) const;

        //- Compute and access a part of the field associated with a model
        const LagrangianSubField<Type>& operator()
        (
            const LagrangianModelRef& model,
            const LagrangianSubMesh& subMesh
        ) const;

        //- Disallow default bitwise assignment
        void operator=(const CloudDerivedField&) = delete;
};

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
    #include "CloudDerivedField.C"
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
