/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2021-2023 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::nonConformalFvPatch

Description
    Non-conformal FV patch. Provides the necessary interface for a FV patch
    which does not conform to the underlying poly mesh. The non-conformal
    geometry and topology are held in the fvMesh, and are generated by the
    fvMeshStitcher. This class just provides access to this data.

See also
    Foam::fvMesh
    Foam::fvMeshStitcher

SourceFiles
    nonConformalFvPatch.C

\*---------------------------------------------------------------------------*/

#ifndef nonConformalFvPatch_H
#define nonConformalFvPatch_H

#include "fvPatch.H"
#include "nonConformalPolyPatch.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

/*---------------------------------------------------------------------------*\
                  Class nonConformalFvPatch Declaration
\*---------------------------------------------------------------------------*/

class nonConformalFvPatch
{
    // Private Data

        //- Reference to the fvPatch
        const fvPatch& patch_;

        //- Reference to the polyPatch
        const nonConformalPolyPatch& nonConformalPolyPatch_;

        //- Face-cells
        mutable labelList faceCells_;


public:

    //- Runtime type information
    TypeName(nonConformalPolyPatch::typeName_());


    // Constructors

        //- Construct from a patch
        nonConformalFvPatch(const fvPatch& patch);


    //- Destructor
    virtual ~nonConformalFvPatch();


    // Member Functions

        // Access

            //- Reference to the fvPatch
            const fvPatch& patch() const;

            //- Reference to the polyPatch
            const nonConformalPolyPatch& nonConformalPatch() const;

            //- Original patch name
            const word& origPatchName() const;

            //- Original patch ID
            label origPatchIndex() const;

            //- Original patch
            const fvPatch& origPatch() const;

            //- Return face face-poly-faces
            const labelList& polyFaces() const;

            //- Return the face-poly-faces patch field type
            virtual word polyFacesType() const = 0;

            //- Return the start label of this patch in the polyMesh face list.
            //  Raises an error unless the patch is empty.
            virtual label start() const;

            //- Return the size
            virtual label size() const;

            //- Return the face-cells
            virtual const labelUList& faceCells() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
