/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::zoneGenerators::intersection

Description
    A zoneGenerator which selects the elements which are in all the zones
    generated by the given list of zoneGenerators

    An intersection of faceZones is oriented if any of the faceZones combined
    are oriented, otherwise the returned faceZone is unoriented.

Usage
    \table
        Property     | Description             | Required | Default value
        type         | Type: intersection             | yes      |
        name         | Name of the zone        | no       | zoneGenerator name
        zoneType     | Type of zone            | no       | all the zone types
        moveUpdate   | Switch to update after mesh motion | no | false
    \endtable
    These options are followed by a list of any number of zoneGenerators
    which can either generate the zones locally or return zones from the lists
    of zones held by the mesh using the zoneGenerators::lookup zoneGenerator.

    Each zoneGenerator returns a zoneSet containing a pointZone and/or a
    cellZone and/or a faceZone, the zoneGenerators::intersection operation
    combines each of the zone types unless the \c zoneType is specified in which
    case only the zones of that type are combined.

    A cellZone named \c coneAnnulus containing the cells with centres inside the
    intersection of the predefined \c cone1 cellZone which is looked-up from the
    cellZoneList and are also inside the locally defined annulus is be
    generated by
    \verbatim
        coneAnnulus
        {
            type        intersection;

            cone1;

            annulus1
            {
                type        annulus;
                zoneType    cell;

                point1      (-0.0075 0 -1);
                point2      (-0.003 0.0025 1);
                outerRadius 0.002;
                innerRadius 0.001;
            }
        }
    \endverbatim

See also
    zoneGenerators::Union
    zoneGenerators::difference
    zoneGeneratorList

SourceFiles
    intersection.C

\*---------------------------------------------------------------------------*/

#ifndef intersection_zoneGenerator_H
#define intersection_zoneGenerator_H

#include "zoneGeneratorList.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace zoneGenerators
{

/*---------------------------------------------------------------------------*\
                              Class intersection Declaration
\*---------------------------------------------------------------------------*/

class intersection
:
    public zoneGenerator
{
    // Private Data

        //- The type of zone to lookup (optional)
        zoneTypesAll zoneType_;

        zoneGeneratorList zoneGenerators_;


    // Private Member Functions

        //- Count the zone intersections
        void countIntersections
        (
            labelList& nIntersections,
            const labelList& indices
        ) const;

        //- Count the faceZone intersections and set the corresponding flipMap
        void countIntersections
        (
            labelList& nIntersections,
            boolList& flipMap,
            const labelList& indices,
            const boolList& zoneFlipMap
        ) const;

        //- Return the list of indices
        //  for which the number of intersection == the number of zones
        labelList indices(const labelList& nIntersections) const;


public:

    //- Runtime type information
    TypeName("intersection");


    // Constructors

        //- Construct from dictionary
        intersection
        (
            const word& name,
            const polyMesh& mesh,
            const dictionary& dict
        );


    //- Destructor
    virtual ~intersection();


    // Member Functions

        virtual zoneSet generate() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace zoneGenerators
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
