/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::zoneGenerators::annulus

Description
    A zoneGenerator which selects points, cells or faces with centres either
    inside or outside a annulus.

    By default all the points, cells or faces of the mesh are tested for being
    inside or outside the annulus but an optional single zone or list of zones
    may be provided and those points, cells or faces are tested instead.  This
    provides an efficient method of hierarchical sub-division of space where an
    initial selection of points cells or faces is refined by selecting the
    sub-set inside or outside the given annulus rather than having to generate
    another zone to intersect with.

Usage
    \table
        Property     | Description             | Required  | Default value
        type         | Type: annulus           | yes       |
        name         | Name of the zone        | no        | zoneGenerator name
        zoneType     | Type of zone            | yes       |
        select       | Select either the inside or outside | no  | inside
        point1       | First point on annulus axis         | yes |
        point2       | Second point on annulus axis        | yes |
        outerRadius  | Outer radius of the annulus         | yes |
        innerRadius  | Inner radius of the annulus         | yes |
        zone         | Optional zone to sub-set            | no  |
        zones        | Optional list of zones to sub-set   | no  |
    \endtable

    A cellZone named \c annulus1 containing the cells with centres inside
    a annulus is generated by
    \verbatim
        annulus1
        {
            type        annulus;
            zoneType    cell;

            point1      (-0.0075 0 -1);
            point2      (-0.003 0.0025 1);
            outerRadius 0.002;
            innerRadius 0.001;
        }
    \endverbatim

SourceFiles
    annulus.C

\*---------------------------------------------------------------------------*/

#ifndef annulus_zoneGenerator_H
#define annulus_zoneGenerator_H

#include "volume.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace zoneGenerators
{

/*---------------------------------------------------------------------------*\
                            Class annulus Declaration
\*---------------------------------------------------------------------------*/

class annulus
:
    public volume
{
    // Private Data

        //- First point on annulus axis
        vector point1_;

        //- Second point on annulus axis
        vector point2_;

        //- Outer Radius
        scalar outerRadius_;

        //- Inner Radius
        scalar innerRadius_;

        //- Axis vector calculated from point1_ and point2_
        vector axis_;

        //- Magnitude of the axis vector squared calculated from axis_
        scalar magAxis2_;

        //- Outer radius squared
        scalar outerRadius2_;

        //- Inner radius squared
        scalar innerRadius2_;


    // Private Member Functions

        friend class volume;

        //- Return true if the annuluses contain the given point
        inline bool contains(const point& p) const;


public:

    //- Runtime type information
    TypeName("annulus");


    // Constructors

        //- Construct from dictionary
        annulus
        (
            const word& name,
            const polyMesh& mesh,
            const dictionary& dict
        );


    //- Destructor
    virtual ~annulus();


    // Member Functions

        virtual zoneSet generate() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace zoneGenerators
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
