#ifndef QISKIT_H
#define QISKIT_H

/* Generated with cbindgen:0.29.2 */

#include <stdarg.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#ifdef QISKIT_C_PYTHON_INTERFACE
    #include <Python.h>
#endif

#include "qiskit/version.h"
#include "qiskit/attributes.h"
#include "qiskit/complex.h"

// Always expose [cfg(feature = "cbinding")] -- workaround for
// https://github.com/mozilla/cbindgen/issues/995
#define QISKIT_WITH_CBINDINGS

typedef struct QkQuantumRegister QkQuantumRegister;
typedef struct QkClassicalRegister QkClassicalRegister;


/**
 * Integer exit codes returned to C.
 */
enum QkExitCode
#ifdef __cplusplus
  : uint32_t
#endif // __cplusplus
 {
  /**
   * Success.
   */
  QkExitCode_Success = 0,
  /**
   * Error related to data input.
   */
  QkExitCode_CInputError = 100,
  /**
   * Unexpected null pointer.
   */
  QkExitCode_NullPointerError = 101,
  /**
   * Pointer is not aligned to expected data.
   */
  QkExitCode_AlignmentError = 102,
  /**
   * Index out of bounds.
   */
  QkExitCode_IndexError = 103,
  /**
   * Duplicate index.
   */
  QkExitCode_DuplicateIndexError = 104,
  /**
   * Error related to arithmetic operations or similar.
   */
  QkExitCode_ArithmeticError = 200,
  /**
   * Mismatching number of qubits.
   */
  QkExitCode_MismatchedQubits = 201,
  /**
   * Matrix is not unitary.
   */
  QkExitCode_ExpectedUnitary = 202,
  /**
   * Target related error
   */
  QkExitCode_TargetError = 300,
  /**
   * Instruction already exists in the Target
   */
  QkExitCode_TargetInstAlreadyExists = 301,
  /**
   * Properties with incorrect qargs was added
   */
  QkExitCode_TargetQargMismatch = 302,
  /**
   * Trying to query into the target with non-existent qargs.
   */
  QkExitCode_TargetInvalidQargsKey = 303,
  /**
   * Querying an operation that doesn't exist in the Target.
   */
  QkExitCode_TargetInvalidInstKey = 304,
  /**
   * Transpilation failed
   */
  QkExitCode_TranspilerError = 400,
  /**
   * QkDag operation error
   */
  QkExitCode_DagError = 500,
  /**
   * The DAGs have mismatching qubit/clbit amounts during compose.
   */
  QkExitCode_DagComposeMismatch = 501,
  /**
   * One or more bit indices were not found during compose.
   */
  QkExitCode_DagComposeMissingBit = 502,
};
#ifndef __cplusplus
typedef uint32_t QkExitCode;
#endif // __cplusplus

enum QkGate
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  QkGate_GlobalPhase = 0,
  QkGate_H = 1,
  QkGate_I = 2,
  QkGate_X = 3,
  QkGate_Y = 4,
  QkGate_Z = 5,
  QkGate_Phase = 6,
  QkGate_R = 7,
  QkGate_RX = 8,
  QkGate_RY = 9,
  QkGate_RZ = 10,
  QkGate_S = 11,
  QkGate_Sdg = 12,
  QkGate_SX = 13,
  QkGate_SXdg = 14,
  QkGate_T = 15,
  QkGate_Tdg = 16,
  QkGate_U = 17,
  QkGate_U1 = 18,
  QkGate_U2 = 19,
  QkGate_U3 = 20,
  QkGate_CH = 21,
  QkGate_CX = 22,
  QkGate_CY = 23,
  QkGate_CZ = 24,
  QkGate_DCX = 25,
  QkGate_ECR = 26,
  QkGate_Swap = 27,
  QkGate_ISwap = 28,
  QkGate_CPhase = 29,
  QkGate_CRX = 30,
  QkGate_CRY = 31,
  QkGate_CRZ = 32,
  QkGate_CS = 33,
  QkGate_CSdg = 34,
  QkGate_CSX = 35,
  QkGate_CU = 36,
  QkGate_CU1 = 37,
  QkGate_CU3 = 38,
  QkGate_RXX = 39,
  QkGate_RYY = 40,
  QkGate_RZZ = 41,
  QkGate_RZX = 42,
  QkGate_XXMinusYY = 43,
  QkGate_XXPlusYY = 44,
  QkGate_CCX = 45,
  QkGate_CCZ = 46,
  QkGate_CSwap = 47,
  QkGate_RCCX = 48,
  QkGate_C3X = 49,
  QkGate_C3SX = 50,
  QkGate_RC3X = 51,
};
#ifndef __cplusplus
typedef uint8_t QkGate;
#endif // __cplusplus

/**
 * @ingroup QkCircuit
 *
 * Units for circuit delays.
 */
enum QkDelayUnit
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  /**
   * Seconds.
   */
  QkDelayUnit_S = 0,
  /**
   * Milliseconds.
   */
  QkDelayUnit_MS = 1,
  /**
   * Microseconds.
   */
  QkDelayUnit_US = 2,
  /**
   * Nanoseconds.
   */
  QkDelayUnit_NS = 3,
  /**
   * Picoseconds.
   */
  QkDelayUnit_PS = 4,
};
#ifndef __cplusplus
typedef uint8_t QkDelayUnit;
#endif // __cplusplus

/**
 * @ingroup QkCircuit
 *
 * The mode to copy the classical variables, for operations that create a new
 * circuit based on an existing one.
 */
enum QkVarsMode
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  /**
   * Each variable has the same type it had in the input.
   */
  QkVarsMode_Alike = 0,
  /**
   * Each variable becomes a "capture".
   */
  QkVarsMode_Captures = 1,
  /**
   * Do not copy the variable data.
   */
  QkVarsMode_Drop = 2,
};
#ifndef __cplusplus
typedef uint8_t QkVarsMode;
#endif // __cplusplus

/**
 * @ingroup QkCircuit
 *
 * The mode to use to copy blocks in control-flow instructions, for operations that
 * create a new circuit based on an existing one.
 */
enum QkBlocksMode
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  /**
   * Drop the blocks.
   */
  QkBlocksMode_Drop = 0,
  /**
   * Keep the blocks.
   */
  QkBlocksMode_Keep = 1,
};
#ifndef __cplusplus
typedef uint8_t QkBlocksMode;
#endif // __cplusplus

/**
 * The type of node in a ``QkDag``.
 *
 * Operation nodes represent an applied instruction. The rest of the nodes are
 * considered "wire" nodes and represent the endpoints of the DAG's data dependency
 * chains.
 */
enum QkDagNodeType
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  /**
   * Operation node.
   */
  QkDagNodeType_Operation = 0,
  /**
   * Qubit wire start node.
   */
  QkDagNodeType_QubitIn = 1,
  /**
   * Qubit wire end node.
   */
  QkDagNodeType_QubitOut = 2,
  /**
   * Clbit wire start node.
   */
  QkDagNodeType_ClbitIn = 3,
  /**
   * Clbit wire end node.
   */
  QkDagNodeType_ClbitOut = 4,
  /**
   * Classical variable wire start node.
   */
  QkDagNodeType_VarIn = 5,
  /**
   * Classical variable wire end node.
   */
  QkDagNodeType_VarOut = 6,
};
#ifndef __cplusplus
typedef uint8_t QkDagNodeType;
#endif // __cplusplus

/**
 * The operation's kind.
 *
 * This is returned when querying a particular node in the graph with ``qk_dag_op_node_kind``,
 * and is intended to allow the caller to dispatch (e.g. via a "switch") calls specific to
 * the contained operation's kind.
 */
enum QkOperationKind
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  QkOperationKind_Gate = 0,
  QkOperationKind_Barrier = 1,
  QkOperationKind_Delay = 2,
  QkOperationKind_Measure = 3,
  QkOperationKind_Reset = 4,
  QkOperationKind_Unitary = 5,
  QkOperationKind_PauliProductMeasurement = 6,
  QkOperationKind_ControlFlow = 7,
  /**
   * This variant is used as an opaque type for operations not yet
   * implemented in the native data model.
   */
  QkOperationKind_Unknown = 8,
};
#ifndef __cplusplus
typedef uint8_t QkOperationKind;
#endif // __cplusplus

/**
 * Named handle to the alphabet of single-qubit terms.
 *
 * This is just the Rust-space representation.  We make a separate Python-space `enum.IntEnum` to
 * represent the same information, since we enforce strongly typed interactions in Rust, including
 * not allowing the stored values to be outside the valid `BitTerm`s, but doing so in Python would
 * make it very difficult to use the class efficiently with Numpy array views.  We attach this
 * sister class of `BitTerm` to `SparseObservable` as a scoped class.
 *
 * # Representation
 *
 * The `u8` representation and the exact numerical values of these are part of the public API.  The
 * low two bits are the symplectic Pauli representation of the required measurement basis with Z in
 * the Lsb0 and X in the Lsb1 (e.g. X and its eigenstate projectors all have their two low bits as
 * `0b10`).  The high two bits are `00` for the operator, `10` for the projector to the positive
 * eigenstate, and `01` for the projector to the negative eigenstate.
 *
 * The `0b00_00` representation thus ends up being the natural representation of the `I` operator,
 * but this is never stored, and is not named in the enumeration.
 *
 * This operator does not store phase terms of $-i$.  `BitTerm::Y` has `(1, 1)` as its `(z, x)`
 * representation, and represents exactly the Pauli Y operator; any additional phase is stored only
 * in a corresponding coefficient.
 *
 * # Dev notes
 *
 * This type is required to be `u8`, but it's a subtype of `u8` because not all `u8` are valid
 * `BitTerm`s.  For interop with Python space, we accept Numpy arrays of `u8` to represent this,
 * which we transmute into slices of `BitTerm`, after checking that all the values are correct (or
 * skipping the check if Python space promises that it upheld the checks).
 *
 * We deliberately _don't_ impl `numpy::Element` for `BitTerm` (which would let us accept and
 * return `PyArray1<BitTerm>` at Python-space boundaries) so that it's clear when we're doing
 * the transmute, and we have to be explicit about the safety of that.
 */
enum QkBitTerm
#ifdef __cplusplus
  : uint8_t
#endif // __cplusplus
 {
  /**
   * Pauli X operator.
   */
  QkBitTerm_X = 2,
  /**
   * Projector to the positive eigenstate of Pauli X.
   */
  QkBitTerm_Plus = 10,
  /**
   * Projector to the negative eigenstate of Pauli X.
   */
  QkBitTerm_Minus = 6,
  /**
   * Pauli Y operator.
   */
  QkBitTerm_Y = 3,
  /**
   * Projector to the positive eigenstate of Pauli Y.
   */
  QkBitTerm_Right = 11,
  /**
   * Projector to the negative eigenstate of Pauli Y.
   */
  QkBitTerm_Left = 7,
  /**
   * Pauli Z operator.
   */
  QkBitTerm_Z = 1,
  /**
   * Projector to the positive eigenstate of Pauli Z.
   */
  QkBitTerm_Zero = 9,
  /**
   * Projector to the negative eigenstate of Pauli Z.
   */
  QkBitTerm_One = 5,
};
#ifndef __cplusplus
typedef uint8_t QkBitTerm;
#endif // __cplusplus

/**
 * A container for :class:`.QuantumCircuit` instruction listings that stores
 * :class:`.CircuitInstruction` instances in a packed form by interning
 * their :attr:`~.CircuitInstruction.qubits` and
 * :attr:`~.CircuitInstruction.clbits` to native vectors of indices.
 *
 * Before adding a :class:`.CircuitInstruction` to this container, its
 * :class:`.Qubit` and :class:`.Clbit` instances MUST be registered via the
 * constructor or via :meth:`.CircuitData.add_qubit` and
 * :meth:`.CircuitData.add_clbit`. This is because the order in which
 * bits of the same type are added to the container determines their
 * associated indices used for storage and retrieval.
 *
 * Once constructed, this container behaves like a Python list of
 * :class:`.CircuitInstruction` instances. However, these instances are
 * created and destroyed on the fly, and thus should be treated as ephemeral.
 *
 * For example,
 *
 * .. plot::
 *    :include-source:
 *    :no-figs:
 *
 *     qubits = [Qubit()]
 *     data = CircuitData(qubits)
 *     data.append(CircuitInstruction(XGate(), (qubits[0],), ()))
 *     assert(data[0] == data[0]) # => Ok.
 *     assert(data[0] is data[0]) # => PANICS!
 *
 * .. warning::
 *
 *     This is an internal interface and no part of it should be relied upon
 *     outside of Qiskit.
 *
 * Args:
 *     qubits (Iterable[:class:`.Qubit`] | None): The initial sequence of
 *         qubits, used to map :class:`.Qubit` instances to and from its
 *         indices.
 *     clbits (Iterable[:class:`.Clbit`] | None): The initial sequence of
 *         clbits, used to map :class:`.Clbit` instances to and from its
 *         indices.
 *     data (Iterable[:class:`.CircuitInstruction`]): An initial instruction
 *         listing to add to this container. All bits appearing in the
 *         instructions in this iterable must also exist in ``qubits`` and
 *         ``clbits``.
 *     reserve (int): The container's initial capacity. This is reserved
 *         before copying instructions into the container when ``data``
 *         is provided, so the initialized container's unused capacity will
 *         be ``max(0, reserve - len(data))``.
 *
 * Raises:
 *     KeyError: if ``data`` contains a reference to a bit that is not present
 *         in ``qubits`` or ``clbits``.
 */
typedef struct QkCircuit QkCircuit;

/**
 * Quantum circuit as a directed acyclic graph.
 *
 * There are 3 types of nodes in the graph: inputs, outputs, and operations.
 * The nodes are connected by directed edges that correspond to qubits and
 * bits.
 */
typedef struct QkDag QkDag;

typedef struct QkParam QkParam;

/**
 * An observable over Pauli bases that stores its data in a qubit-sparse format.
 *
 * See [PySparseObservable] for detailed docs.
 */
typedef struct QkObs QkObs;

/**
 * The base class for a Python ``Target`` object. Contains data representing the constraints of a particular backend.
 *
 *The intent of this struct is to contain data that can be representable and accessible through both Rust and Python, so it can be used for rust-based transpiler processes.
 *
 *This structure contains duplicates of every element in the Python counterpart of `gate_map`. Which improves access for Python while sacrificing a small amount of memory.
 */
typedef struct QkTarget QkTarget;

/**
 * Represents the mapping between qargs and ``InstructionProperties``
 */
typedef struct QkTargetEntry QkTargetEntry;

/**
 * The "layout" caused by transpilation
 *
 * In general Qiskit's transpiler is unitary-preserving up to the initial layout
 * and routing permutations. The initial layout permutation is caused by
 * setting and applying the initial layout (the mapping from virtual circuit
 * qubits to physical qubits on the target) and the routing permtuations are
 * caused by swap gate insertion or permutation ellision prior to the initial
 * layout. This struct tracks these details and provide an interface to reason
 * about these permutations.
 */
typedef struct QkTranspileLayout QkTranspileLayout;

/**
 * A set of configurations for the VF2 layout passes.
 *
 * See the setter methods associated with this `struct` for the available configurations.
 */
typedef struct QkVF2LayoutConfiguration QkVF2LayoutConfiguration;

/**
 * The result from ``qk_transpiler_pass_standalone_vf2_layout_average()`` and
 * ``qk_transpile_pass_standalone_vf2_layout_exact()``.
 */
typedef struct QkVF2LayoutResult QkVF2LayoutResult;

/**
 * An individual operation count represented by the operation name
 * and the number of instances in the circuit.
 */
typedef struct {
  /**
   * A nul terminated string representing the operation name
   */
  const char *name;
  /**
   * The number of instances of this operation in the circuit
   */
  size_t count;
} QkOpCount;

/**
 * An array of ``OpCount`` objects representing the total counts of all
 * the operation types in a circuit.
 */
typedef struct {
  /**
   * A array of size ``len`` containing ``OpCount`` objects for each
   * type of operation in the circuit
   */
  QkOpCount *data;
  /**
   * The number of elements in ``data``
   */
  size_t len;
} QkOpCounts;

/**
 * A circuit instruction representation.
 *
 * This struct represents the data contained in an individual instruction in a ``QkCircuit``.
 * It is not a pointer to the underlying object, but contains a copy of the properties of the
 * instruction for inspection.
 */
typedef struct {
  /**
   * The instruction name
   */
  char *name;
  /**
   * A pointer to an array of qubit indices this instruction operates on.
   */
  uint32_t *qubits;
  /**
   * A pointer to an array of clbit indices this instruction operates on.
   */
  uint32_t *clbits;
  /**
   * A pointer to an array of parameter values for this instruction.
   */
  double *params;
  /**
   * The number of qubits for this instruction.
   */
  uint32_t num_qubits;
  /**
   * The number of clbits for this instruction.
   */
  uint32_t num_clbits;
  /**
   * The number of parameters for this instruction.
   */
  uint32_t num_params;
} QkCircuitInstruction;

/**
 * A struct for storing successors and predecessors information
 * retrieved from `qk_dag_successors` and `qk_dag_predecessors`, respectively.
 *
 * This object is read-only from C. To satisfy the safety guarantees of `qk_dag_neighbors_clear`,
 * you must not overwrite any data initialized by `qk_dag_successors` or `qk_dag_predecessors`,
 * including any pointed-to data.
 */
typedef struct {
  /**
   * Array of size `num_neighbors` of node indices.
   */
  const uint32_t *neighbors;
  /**
   * The length of the `neighbors` array.
   */
  size_t num_neighbors;
} QkDagNeighbors;

/**
 * A term in a ``QkObs``.
 *
 * This contains the coefficient (``coeff``), the number of qubits of the observable
 * (``num_qubits``) and pointers to the ``bit_terms`` and ``indices`` arrays, which have
 * length ``len``. It's the responsibility of the user that the data is coherent,
 * see also the below section on safety.
 *
 * # Safety
 *
 * * ``bit_terms`` must be a non-null, aligned pointer to ``len`` elements of type ``QkBitTerm``.
 * * ``indices`` must be a non-null, aligned pointer to ``len`` elements of type ``uint32_t``.
 */
typedef struct {
  /**
   * The coefficient of the observable term.
   */
  QkComplex64 coeff;
  /**
   * Length of the ``bit_terms`` and ``indices`` arrays.
   */
  size_t len;
  /**
   * A non-null, aligned pointer to ``len`` elements of type ``QkBitTerm``.
   */
  QkBitTerm *bit_terms;
  /**
   * A non-null, aligned pointer to ``len`` elements of type ``uint32_t``.
   */
  uint32_t *indices;
  /**
   * The number of qubits the observable term is defined on.
   */
  uint32_t num_qubits;
} QkObsTerm;

/**
 * An adjacency-list representation of a coupling graph.
 *
 * This is initialized by `qk_neighbors_from_target`.
 *
 * This object is read-only from C.  To satisfy the safety guarantees of `qk_neighbors_clear`, you
 * must not overwrite any data initialized by `qk_neighbors_from_target`, including any pointed-to
 * data.
 *
 * # Representation
 *
 * After initialization by `qk_neighbors_from_target`, the structure will be in one of two modes:
 *
 * * all-to-all connectivity
 * * limited two-qubit connectivity
 *
 * In the all-to-all case, the `neighbors` and `partition` pointers will both be the null pointer,
 * and `num_qubits` will be the number of qubits in the `QkTarget`.  These objects do not have
 * backing allocations, and do not need to be given to `qk_neighbors_clear` (though this function
 * is a safe no-op in this case).
 *
 * In the limited two-qubit case (which is by far the more common for real hardware), see the
 * documentation of the structure members for their interpretation.
 */
typedef struct {
  /**
   * A partitioned adjacency-list representation of the neighbors of each qubit.  This pointer is
   * valid for exactly `partition[num_qubits + 1]` reads.
   *
   * For qubit number `i`, its neighbors are the values between offsets `partition[i]`
   * (inclusive) and `partition[i + 1]` (exclusive).  The values between these two offsets are
   * sorted in ascending order and contain no duplicates.
   */
  const uint32_t *neighbors;
  /**
   * How the `neighbors` field is partitioned into slices.  This pointer is valid for exactly
   * `num_qubits + 1` reads.  The first value is always 0, and values increase monotonically.
   */
  const size_t *partition;
  /**
   * The number of qubits.
   */
  uint32_t num_qubits;
} QkNeighbors;

/**
 * The options for running ``qk_transpiler_pass_standalone_sabre_layout``. This struct is used
 * as an input to control the behavior of the layout and routing algorithms.
 */
typedef struct {
  /**
   * The number of forward-backward iterations in the sabre routing algorithm
   */
  size_t max_iterations;
  /**
   * The number of trials to run of the sabre routing algorithm for each iteration. When > 1 the
   * trial that routing trial that results in the output with the fewest swap gates will be
   * selected.
   */
  size_t num_swap_trials;
  /**
   * The number of random layout trials to run. The trial that results in the output with the
   * fewest swap gates will be selected.
   */
  size_t num_random_trials;
  /**
   * A seed value for the pRNG used internally.
   */
  uint64_t seed;
} QkSabreLayoutOptions;

/**
 * A representation of a Target operation's instruction properties.
 */
typedef struct {
  /**
   * The duration, in seconds, of the instruction on the specified set of qubits.
   * Will be set to ``NaN`` if the property is not defined.
   */
  double duration;
  /**
   * The average error rate for the instruction on the specified set of qubits.
   * Will be set to ``NaN`` if the property is not defined.
   */
  double error;
} QkInstructionProperties;

/**
 * Representation of an operation identified within the Target.
 *
 * This struct is created natively by the underlying `Target` API
 * and users should not write to it directly nor try to free its
 * attributes manually as it would lead to undefined behavior. To
 * free this struct, users should call ``qk_target_op_clear`` instead.
 */
typedef struct {
  /**
   * The identifier for the current operation.
   */
  QkOperationKind op_type;
  /**
   * The name of the operation.
   */
  char *name;
  /**
   * The number of qubits this operation supports. Will default to
   * `(uint32_t)-1` in the case of a variadic.
   */
  uint32_t num_qubits;
  /**
   * The parameters tied to this operation if fixed, as an array
   * of `double`. If any of the parameters represented are not fixed angles
   * it will be represented as with the `NaN` value. If there are no parameters
   * then this value will be represented with a `NULL` pointer.
   */
  double *params;
  /**
   * The number of parameters supported by this operation. Will default to
   * `(uint32_t)-1` in the case of a variadic.
   */
  uint32_t num_params;
} QkTargetOp;

/**
 * The options for running the transpiler
 */
typedef struct {
  /**
   * The optimization level to run the transpiler with. Valid values are 0, 1, 2, or 3.
   */
  uint8_t optimization_level;
  /**
   * The seed for the transpiler. If set to a negative number this means no seed will be
   * set and the RNGs used in the transpiler will be seeded from system entropy.
   */
  int64_t seed;
  /**
   * The approximation degree a heurstic dial where 1.0 means no approximation (up to numerical
   * tolerance) and 0.0 means the maximum approximation. A `NAN` value indicates that
   * approximation is allowed up to the reported error rate for an operation in the target.
   */
  double approximation_degree;
} QkTranspileOptions;

/**
 * The container result object from ``qk_transpile``
 *
 * When the transpiler successfully compiles a quantum circuit for a given target it
 * returns the transpiled circuit and the layout. The ``qk_transpile`` function will
 * write pointers to the fields in this struct when it successfully executes, you can
 * initialize this struct with null pointers or leave them unset as the values are never
 * read by ``qk_transpile`` and only written to. After calling ``qk_transpile`` you are
 * responsible for calling ``qk_circuit_free`` and ``qk_transpile_layout_free`` on the
 * members of this struct.
 */
typedef struct {
  /**
   * The compiled circuit.
   */
  QkCircuit *circuit;
  /**
   * Metadata about the initial and final virtual-to-physical layouts.
   */
  QkTranspileLayout *layout;
} QkTranspileResult;

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Construct a new circuit with the given number of qubits and clbits.
 *
 * @param num_qubits The number of qubits the circuit contains.
 * @param num_clbits The number of clbits the circuit contains.
 *
 * @return A pointer to the created circuit.
 *
 * # Example
 *
 *     QkCircuit *empty = qk_circuit_new(100, 100);
 *
 */
QkCircuit *qk_circuit_new(uint32_t num_qubits, uint32_t num_clbits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkQuantumRegister
 * Construct a new owning quantum register with a given number of qubits and name
 *
 * @param num_qubits The number of qubits to create the register for
 * @param name The name string for the created register. The name must be comprised of
 * valid UTF-8 characters.
 *
 * @return A pointer to the created register
 *
 * # Example
 * ```c
 *     QkQuantumRegister *qr = qk_quantum_register_new(5, "five_qubits");
 * ```
 *
 * # Safety
 *
 * The `name` parameter must be a pointer to memory that contains a valid
 * nul terminator at the end of the string. It also must be valid for reads of
 * bytes up to and including the nul terminator.
 */
QkQuantumRegister *qk_quantum_register_new(uint32_t num_qubits, const char *name);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkQuantumRegister
 * Free a quantum register.
 *
 * @param reg A pointer to the register to free.
 *
 * # Example
 * ```c
 *     QkQuantumRegister *qr = qk_quantum_register_new(1024, "qreg");
 *     qk_quantum_register_free(qr);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``reg`` is not either null or a valid pointer to a
 * ``QkQuantumRegister``.
 */
void qk_quantum_register_free(QkQuantumRegister *reg);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkClassicalRegister
 * Free a classical register.
 *
 * @param reg A pointer to the register to free.
 *
 * # Example
 * ```c
 *     QkClassicalRegister *cr = qk_classical_register_new(1024, "creg");
 *     qk_classical_register_free(cr);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``reg`` is not either null or a valid pointer to a
 * ``QkClassicalRegister``.
 */
void qk_classical_register_free(QkClassicalRegister *reg);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkClassicalRegister
 * Construct a new owning classical register with a given number of clbits and name
 *
 * @param num_clbits The number of clbits to create the register for
 * @param name The name string for the created register. The name must be comprised of
 * valid UTF-8 characters.
 *
 * @return A pointer to the created register
 *
 * # Example
 * ```c
 *     QkClassicalRegister *cr = qk_classical_register_new(5, "five_qubits");
 * ```
 *
 * # Safety
 *
 * The `name` parameter must be a pointer to memory that contains a valid
 * nul terminator at the end of the string. It also must be valid for reads of
 * bytes up to and including the nul terminator.
 */
QkClassicalRegister *qk_classical_register_new(uint32_t num_clbits, const char *name);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Add a quantum register to a given quantum circuit
 *
 * @param circuit A pointer to the circuit.
 * @param reg A pointer to the quantum register
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(0, 0);
 *     QkQuantumRegister *qr = qk_quantum_register_new(1024, "my_little_register");
 *     qk_circuit_add_quantum_register(qc, qr);
 *     qk_quantum_register_free(qr);
 *     qk_circuit_free(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit`` and
 * if ``reg`` is not a valid, non-null pointer to a ``QkQuantumRegister``.
 */
void qk_circuit_add_quantum_register(QkCircuit *circuit, const QkQuantumRegister *reg);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Add a classical register to a given quantum circuit
 *
 * @param circuit A pointer to the circuit.
 * @param reg A pointer to the classical register
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(0, 0);
 *     QkClassicalRegister *cr = qk_classical_register_new(24, "my_big_register");
 *     qk_circuit_add_classical_register(qc, cr);
 *     qk_classical_register_free(cr);
 *     qk_circuit_free(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit`` and
 * if ``reg`` is not a valid, non-null pointer to a ``QkClassicalRegister``.
 */
void qk_circuit_add_classical_register(QkCircuit *circuit, const QkClassicalRegister *reg);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Create a copy of a ``QkCircuit``.
 *
 * @param circuit A pointer to the circuit to copy.
 *
 * @return A new pointer to a copy of the input ``circuit``.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 100);
 *     QkCircuit *copy = qk_circuit_copy(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkCircuit *qk_circuit_copy(const QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Get the number of qubits the circuit contains.
 *
 * @param circuit A pointer to the circuit.
 *
 * @return The number of qubits the circuit is defined on.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 100);
 *     uint32_t num_qubits = qk_circuit_num_qubits(qc);  // num_qubits==100
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
uint32_t qk_circuit_num_qubits(const QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Get the number of clbits the circuit contains.
 *
 * @param circuit A pointer to the circuit.
 *
 * @return The number of qubits the circuit is defined on.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 50);
 *     uint32_t num_clbits = qk_circuit_num_clbits(qc);  // num_clbits==50
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
uint32_t qk_circuit_num_clbits(const QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Free the circuit.
 *
 * @param circuit A pointer to the circuit to free.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 100);
 *     qk_circuit_free(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not either null or a valid pointer to a
 * ``QkCircuit``.
 */
void qk_circuit_free(QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Append a ``QkGate`` to the circuit.
 *
 * @param circuit A pointer to the circuit to add the gate to.
 * @param gate The StandardGate to add to the circuit.
 * @param qubits The pointer to the array of ``uint32_t`` qubit indices to add the gate on. This
 *     can be a null pointer if there are no qubits for ``gate`` (e.g. ``QkGate_GlobalPhase``).
 * @param params The pointer to the array of ``double`` values to use for the gate parameters.
 *     This can be a null pointer if there are no parameters for ``gate`` (e.g. ``QkGate_H``).
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 0);
 *     uint32_t qubit[1] = {0};
 *     qk_circuit_gate(qc, QkGate_H, qubit, NULL);
 * ```
 *
 * # Safety
 *
 * The ``qubits`` and ``params`` types are expected to be a pointer to an array of ``uint32_t``
 * and ``double`` respectively where the length is matching the expectations for the standard
 * gate. If the array is insufficiently long the behavior of this function is undefined as this
 * will read outside the bounds of the array. It can be a null pointer if there are no qubits
 * or params for a given gate. You can check ``qk_gate_num_qubits`` and ``qk_gate_num_params`` to
 * determine how many qubits and params are required for a given gate.
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkExitCode qk_circuit_gate(QkCircuit *circuit,
                           QkGate gate,
                           const uint32_t *qubits,
                           const double *params);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Get the number of qubits for a ``QkGate``.
 *
 * @param gate The ``QkGate`` to get the number of qubits for.
 *
 * @return The number of qubits the gate acts on.
 *
 * # Example
 * ```c
 *     uint32_t num_qubits = qk_gate_num_qubits(QkGate_CCX);
 * ```
 *
 */
uint32_t qk_gate_num_qubits(QkGate gate);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Get the number of parameters for a ``QkGate``.
 *
 * @param gate The ``QkGate`` to get the number of qubits for.
 *
 * @return The number of parameters the gate has.
 *
 * # Example
 * ```c
 *     uint32_t num_params = qk_gate_num_params(QkGate_R);
 * ```
 *
 */
uint32_t qk_gate_num_params(QkGate gate);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Append a measurement to the circuit
 *
 * @param circuit A pointer to the circuit to add the measurement to
 * @param qubit The ``uint32_t`` for the qubit to measure
 * @param clbit The ``uint32_t`` for the clbit to store the measurement outcome in
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 1);
 *     qk_circuit_measure(qc, 0, 0);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkExitCode qk_circuit_measure(QkCircuit *circuit, uint32_t qubit, uint32_t clbit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Append a reset to the circuit
 *
 * @param circuit A pointer to the circuit to add the reset to
 * @param qubit The ``uint32_t`` for the qubit to reset
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 0);
 *     qk_circuit_reset(qc, 0);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkExitCode qk_circuit_reset(QkCircuit *circuit, uint32_t qubit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Append a barrier to the circuit.
 *
 * @param circuit A pointer to the circuit to add the barrier to.
 * @param num_qubits The number of qubits wide the barrier is.
 * @param qubits The pointer to the array of ``uint32_t`` qubit indices to add the barrier on.
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 1);
 *     uint32_t qubits[5] = {0, 1, 2, 3, 4};
 *     qk_circuit_barrier(qc, qubits, 5);
 * ```
 *
 * # Safety
 *
 * The length of the array ``qubits`` points to must be ``num_qubits``. If there is
 * a mismatch the behavior is undefined.
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkExitCode qk_circuit_barrier(QkCircuit *circuit, const uint32_t *qubits, uint32_t num_qubits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Append an arbitrary unitary matrix to the circuit.
 *
 * @param circuit A pointer to the circuit to append the unitary to.
 * @param matrix A pointer to the ``QkComplex64`` array representing the unitary matrix.
 *     This must be a row-major, unitary matrix of dimension ``2 ^ num_qubits x 2 ^ num_qubits``.
 *     More explicitly: the ``(i, j)``-th element is given by ``matrix[i * 2^n + j]``.
 *     The contents of ``matrix`` are copied inside this function before being added to the circuit,
 *     so caller keeps ownership of the original memory that ``matrix`` points to and can reuse it
 *     after the call and the caller is responsible for freeing it.
 * @param qubits A pointer to array of qubit indices, of length ``num_qubits``.
 * @param num_qubits The number of qubits the unitary acts on.
 * @param check_input When true, the function verifies that the matrix is unitary.
 *     If set to False the caller is responsible for ensuring the matrix is unitary, if
 *     the matrix is not unitary this is undefined behavior and will result in a corrupt
 *     circuit.
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 * QkComplex64 c0 = {0, 0};  // 0+0i
 * QkComplex64 c1 = {1, 0};  // 1+0i
 *
 * const uint32_t num_qubits = 1;
 * QkComplex64 unitary[2*2] = {c0, c1,  // row 0
 *                             c1, c0}; // row 1
 *
 * QkCircuit *circuit = qk_circuit_new(1, 0);  // 1 qubit circuit
 * uint32_t qubit[1] = {0};  // qubit to apply the unitary on
 * qk_circuit_unitary(circuit, unitary, qubit, num_qubits, true);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if any of the following is violated:
 *
 * * ``circuit`` is a valid, non-null pointer to a ``QkCircuit``
 * * ``matrix`` is an aligned pointer to ``4**num_qubits`` initialized ``QkComplex64`` values
 * * ``qubits`` is an aligned pointer to ``num_qubits`` initialized ``uint32_t`` values
 */
QkExitCode qk_circuit_unitary(QkCircuit *circuit,
                              const QkComplex64 *matrix,
                              const uint32_t *qubits,
                              uint32_t num_qubits,
                              bool check_input);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Return a list of string names for instructions in a circuit and their counts.
 *
 * To properly free the memory allocated by the struct, you should call ``qk_opcounts_clear``.
 * Dropping the ``QkOpCounts`` struct without doing so will leave the stored array of ``QkOpCount``
 * allocated and produce a memory leak.
 *
 * @param circuit A pointer to the circuit to get the counts for.
 *
 * @return An ``QkOpCounts`` struct containing the circuit operation counts.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 0);
 *     uint32_t qubits[1] = {0};
 *     qk_circuit_gate(qc, QkGate_H, qubits, NULL);
 *     QkOpCounts counts = qk_circuit_count_ops(qc);
 *     // .. once done
 *     qk_opcounts_clear(&counts);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkOpCounts qk_circuit_count_ops(const QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Return the total number of instructions in the circuit.
 *
 * @param circuit A pointer to the circuit to get the total number of instructions for.
 *
 * @return The total number of instructions in the circuit.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(100, 0);
 *     uint32_t qubit[1] = {0};
 *     qk_circuit_gate(qc, QkGate_H, qubit, NULL);
 *     size_t num = qk_circuit_num_instructions(qc); // 1
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
size_t qk_circuit_num_instructions(const QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Return the instruction details for an instruction in the circuit.
 *
 * This function is used to get the instruction details for a given instruction in
 * the circuit.
 *
 * This function allocates memory internally for the provided ``QkCircuitInstruction``
 * and thus you are responsible for calling ``qk_circuit_instruction_clear`` to
 * free it.
 *
 * @param circuit A pointer to the circuit to get the instruction details for.
 * @param index The instruction index to get the instruction details of.
 * @param instruction A pointer to where to write out the ``QkCircuitInstruction``
 *
 *
 * # Example
 * ```c
 *     QkCircuitInstruction inst;
 *     QkCircuit *qc = qk_circuit_new(100, 0);
 *     uint32_t qubit[1] = {0};
 *     qk_circuit_gate(qc, QkGate_H, qubit, NULL);
 *     qk_circuit_get_instruction(qc, 0, &inst);
 *     qk_circuit_instruction_clear(&inst);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``. The
 * value for ``index`` must be less than the value returned by ``qk_circuit_num_instructions``
 * otherwise this function will panic. Behavior is undefined if ``instruction`` is not a valid,
 * non-null pointer to a memory allocation with sufficient space for a ``QkCircuitInstruction``.
 */
void qk_circuit_get_instruction(const QkCircuit *circuit,
                                size_t index,
                                QkCircuitInstruction *instruction);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Clear the data in circuit instruction object.
 *
 * This function doesn't free the allocation for the provided ``QkCircuitInstruction`` pointer, it
 * only frees the internal allocations for the data contained in the instruction. You are
 * responsible for allocating and freeing the actual allocation used to store a
 * ``QkCircuitInstruction``.
 *
 * @param inst A pointer to the instruction to free.
 *
 * # Example
 * ```c
 *     QkCircuitInstruction *inst = malloc(sizeof(QkCircuitInstruction));
 *     QkCircuit *qc = qk_circuit_new(100, 0);
 *     uint32_t q0[1] = {0};
 *     qk_circuit_gate(qc, QkGate_H, q0, NULL);
 *     qk_circuit_get_instruction(qc, 0, inst);
 *     qk_circuit_instruction_clear(inst); // clear internal allocations
 *     free(inst); // free struct
 *     qk_circuit_free(qc); // free the circuit
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``inst`` is not a valid, non-null pointer to a ``QkCircuitInstruction``.
 */
void qk_circuit_instruction_clear(QkCircuitInstruction *inst);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Clear the content in a circuit operation count list.
 *
 * @param op_counts The returned op count list from ``qk_circuit_count_ops``.
 *
 * # Safety
 *
 * Behavior is undefined if ``op_counts`` is not the object returned by ``qk_circuit_count_ops``.
 */
void qk_opcounts_clear(QkOpCounts *op_counts);
#endif

#if (defined(QISKIT_C_PYTHON_INTERFACE) && defined(QISKIT_WITH_CBINDINGS))
/**
 * @ingroup QkCircuit
 * Convert to a Python-space ``QuantumCircuit``.
 *
 * This function takes ownership of the pointer and gives it to Python. Using
 * the input ``circuit`` pointer after it's passed to this function is
 * undefined behavior. In particular, ``qk_circuit_free`` should not be called
 * on this pointer anymore.
 *
 * @param circuit The C-space ``QkCircuit`` pointer.
 *
 * @return A Python ``QuantumCircuit`` object.
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to
 * a ``QkCircuit``
 *
 * It is assumed that the thread currently executing this function holds the
 * Python GIL. This is required to create the Python object returned by this
 * function.
 */
PyObject *qk_circuit_to_python(QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Append a delay instruction to the circuit.
 *
 * @param circuit A pointer to the circuit to add the delay to.
 * @param qubit The ``uint32_t`` index of the qubit to apply the delay to.
 * @param duration The duration of the delay.
 * @param unit An enum representing the unit of the duration.
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(1, 0);
 *     qk_circuit_delay(qc, 0, 100.0, QkDelayUnit_NS);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkExitCode qk_circuit_delay(QkCircuit *circuit, uint32_t qubit, double duration, QkDelayUnit unit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Convert a given circuit to a DAG.
 *
 * The new DAG is copied from the circuit; the original ``circuit`` reference is still owned by the
 * caller and still required to be freed with `qk_circuit_free`.  You must free the returned DAG
 * with ``qk_dag_free`` when done with it.
 *
 * @param circuit A pointer to the circuit from which to create the DAG.
 *
 * @return A pointer to the new DAG.
 *
 * # Example
 * ```c
 *     QkCircuit *qc = qk_circuit_new(0, 0);
 *     QkQuantumRegister *qr = qk_quantum_register_new(3, "qr");
 *     qk_circuit_add_quantum_register(qc, qr);
 *     qk_quantum_register_free(qr);
 *
 *     QkDag *dag = qk_circuit_to_dag(qc);
 *
 *     qk_dag_free(dag);
 *     qk_circuit_free(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkDag *qk_circuit_to_dag(const QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuit
 * Return a copy of self with the same structure but empty.
 *
 * That structure includes:
 * * global phase
 * * all the qubits and clbits, including the registers.
 *
 * @param circuit A pointer to the circuit to copy.
 * @param vars_mode The mode for handling classical variables.
 * @param blocks_mode The mode for handling blocks.
 *
 * @return The pointer to the copied circuit.
 *
 * # Example
 * ```c
 * QkCircuit *qc = qk_circuit_new(10, 10);
 * for (int i = 0; i < 10; i++) {
 *     qk_circuit_measure(qc, i, i);
 *     uint32_t qubits[1] = {i};
 *     qk_circuit_gate(qc, QkGate_H, qubits, NULL);
 * }
 *
 * // As the circuit does not contain any control-flow instructions,
 * // vars_mode and blocks_mode do not have any effect.
 * QkCircuit *copy = qk_circuit_copy_empty_like(qc, QkVarsMode_Alike, QkBlocksMode_Drop);
 *
 * size_t num_copy_instructions = qk_circuit_num_instructions(copy); // 0
 *
 * // do something with the copy
 *
 * qk_circuit_free(qc);
 * qk_circuit_free(copy);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkCircuit *qk_circuit_copy_empty_like(const QkCircuit *circuit,
                                      QkVarsMode vars_mode,
                                      QkBlocksMode blocks_mode);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkCircuitLibrary
 * Generate a Quantum Volume model circuit
 *
 * The model circuits are random instances of circuits used to measure the Quantum Volume
 * metric, as introduced in [1]. The model circuits consist of layers of Haar random
 * elements of SU(4) applied between corresponding pairs of qubits in a random bipartition.
 *
 * This function is multithreaded and will launch a thread pool with threads equal to the
 * number of CPUs by default. You can tune the number of threads with the
 * RAYON_NUM_THREADS environment variable. For example, setting RAYON_NUM_THREADS=4 would
 * limit the thread pool to 4 threads.
 *
 * [1] A. Cross et al. Validating quantum computers using randomized model circuits,
 *     Phys. Rev. A 100, 032328 (2019). [arXiv:1811.12926](https://arxiv.org/abs/1811.12926)
 *
 * @param num_qubits The number qubits to use for the generated circuit.
 * @param depth The number of layers for the generated circuit.
 * @param seed An RNG seed used for generating the random SU(4) matrices used
 *   in the output circuit. If the provided number is negative the seed used
 *   will be soured from system entropy.
 *
 * @return A pointer to the quantum volume circuit.
 *
 * # Example
 *
 * ```c
 * QkCircuit *qc = qk_circuit_library_quantum_volume(10, 10, -1)
 * ```
 */
QkCircuit *qk_circuit_library_quantum_volume(uint32_t num_qubits, size_t depth, int64_t seed);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Construct a new empty DAG.
 *
 * You must free the returned DAG with ``qk_dag_free`` when done with it.
 *
 * @return A pointer to the created DAG.
 *
 * # Example
 * ```c
 * QkDag *empty = qk_dag_new();
 * ```
 */
QkDag *qk_dag_new(void);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Add a quantum register to the DAG.
 *
 * @param dag A pointer to the DAG.
 * @param reg A pointer to the quantum register.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1024, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 * qk_quantum_register_free(qr);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag`` and
 * if ``reg`` is not a valid, non-null pointer to a ``QkQuantumRegister``.
 */
void qk_dag_add_quantum_register(QkDag *dag, const QkQuantumRegister *reg);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Add a classical register to the DAG.
 *
 * @param dag A pointer to the DAG.
 * @param reg A pointer to the classical register.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkClassicalRegister *cr = qk_classical_register_new(24, "my_register");
 * qk_dag_add_classical_register(dag, cr);
 * qk_classical_register_free(cr);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag`` and
 * if ``reg`` is not a valid, non-null pointer to a ``QkClassicalRegister``.
 */
void qk_dag_add_classical_register(QkDag *dag, const QkClassicalRegister *reg);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Get the number of qubits the DAG contains.
 *
 * @param dag A pointer to the DAG.
 *
 * @return The number of qubits the DAG is defined on.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(24, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 * uint32_t num_qubits = qk_dag_num_qubits(dag);  // num_qubits==24
 * qk_quantum_register_free(qr);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_num_qubits(const QkDag *dag);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Get the number of clbits the DAG contains.
 *
 * @param dag A pointer to the DAG.
 *
 * @return The number of clbits the DAG is defined on.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkClassicalRegister *cr = qk_classical_register_new(24, "my_register");
 * qk_dag_add_classical_register(dag, cr);
 * uint32_t num_clbits = qk_dag_num_clbits(dag);  // num_clbits==24
 * qk_classical_register_free(cr);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_num_clbits(const QkDag *dag);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Return the total number of operation nodes in the DAG.
 *
 * @param dag A pointer to the DAG.
 *
 * @return The total number of instructions in the DAG.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 *
 * uint32_t qubit[1] = {0};
 * qk_dag_apply_gate(dag, QkGate_H, qubit, NULL, false);
 * size_t num = qk_dag_num_op_nodes(dag); // 1
 *
 * qk_dag_free(dag);
 * qk_quantum_register_free(qr);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
size_t qk_dag_num_op_nodes(const QkDag *dag);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Get the type of the specified node.
 *
 * The result can be used in a switch statement to dispatch proper handling
 * when iterating over nodes of unknown type.
 *
 * @param dag A pointer to the DAG.
 * @param node The node to get the type of.
 *
 * @return The type of the node.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
QkDagNodeType qk_dag_node_type(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the index of the input node of the wire corresponding to the given qubit.
 *
 * @param dag A pointer to the DAG.
 * @param qubit The qubit to get the input node index of.
 *
 * @return The input node of the qubit wire.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_qubit_in_node(const QkDag *dag, uint32_t qubit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the index of the output node of the wire corresponding to the given qubit.
 *
 * @param dag A pointer to the DAG.
 * @param qubit The qubit to get the output node index of.
 *
 * @return The output node of the qubit wire.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_qubit_out_node(const QkDag *dag, uint32_t qubit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the index of the input node of the wire corresponding to the given clbit.
 *
 * @param dag A pointer to the DAG.
 * @param clbit The clbit to get the input node index of.
 *
 * @return The input node of the clbit wire.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_clbit_in_node(const QkDag *dag, uint32_t clbit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the index of the output node of the wire corresponding to the given clbit.
 *
 * @param dag A pointer to the DAG.
 * @param clbit The clbit to get the output node index of.
 *
 * @return The output node of the clbit wire.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_clbit_out_node(const QkDag *dag, uint32_t clbit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the value of a wire endpoint node.
 *
 * @param dag A pointer to the DAG.
 * @param node The endpoint node to get the wire value of.
 *
 * @return The value (e.g. qubit, clbit, or var) within the endpoint node.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_wire_node_value(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Gets the number of qubits of the specified operation node.
 *
 * Panics if the node is not an operation.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the number of qubits of.
 *
 * @return The number of qubits of the operation.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_op_node_num_qubits(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Gets the number of clbits of the specified operation node.
 *
 * Panics if the node is not an operation.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the number of clbits of.
 *
 * @return The number of clbits of the operation.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_op_node_num_clbits(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Gets the number of params of the specified operation node.
 *
 * Panics if the node is not an operation.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the number of params of.
 *
 * @return The number of params of the operation.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_op_node_num_params(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the qubits of the specified operation node.
 *
 * Panics if the node is not an operation.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the qubits of.
 *
 * @return A pointer to the qubits. Use ``qk_dag_op_node_num_qubits`` to determine the number of
 *     elements.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
const uint32_t *qk_dag_op_node_qubits(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the clbits of the specified operation node.
 *
 * Panics if the node is not an operation.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the clbits of.
 *
 * @return A pointer to the clbits. Use ``qk_dag_op_node_num_clbits`` to determine the number of
 *     elements.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
const uint32_t *qk_dag_op_node_clbits(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Apply a ``QkGate`` to the DAG.
 *
 * @param dag A pointer to the DAG to apply the gate to.
 * @param gate The StandardGate to apply.
 * @param qubits The pointer to the array of ``uint32_t`` qubit indices to add the gate on. This
 *     can be a null pointer if there are no qubits for ``gate`` (e.g. ``QkGate_GlobalPhase``).
 * @param params The pointer to the array of ``double`` values to use for the gate parameters.
 *     This can be a null pointer if there are no parameters for ``gate`` (e.g. ``QkGate_H``).
 * @param front If ``true``, the gate is applied as the first operation on the specified qubits,
 *     rather than as the last.
 *
 * @return The index of the newly added operation node.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 *
 * uint32_t qubit[1] = {0};
 * qk_dag_apply_gate(dag, QkGate_H, qubit, NULL, false);
 *
 * qk_dag_free(dag);
 * qk_quantum_register_free(qr);
 * ```
 *
 * # Safety
 *
 * The ``qubits`` and ``params`` types are expected to be a pointer to an array of ``uint32_t``
 * and ``double`` respectively where the length is matching the expectations for the standard
 * gate. If the array is insufficiently long the behavior of this function is undefined as this
 * will read outside the bounds of the array. It can be a null pointer if there are no qubits
 * or params for a given gate. You can check ``qk_gate_num_qubits`` and ``qk_gate_num_params`` to
 * determine how many qubits and params are required for a given gate.
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
uint32_t qk_dag_apply_gate(QkDag *dag,
                           QkGate gate,
                           const uint32_t *qubits,
                           const double *params,
                           bool front);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Apply a measure to a DAG.
 *
 * @param dag The circuit to apply to.
 * @param qubit The qubit index to measure.
 * @param clbit The clbit index to store the result in.
 * @param front Whether to apply the measure at the start of the circuit. Usually `false`.
 *
 * @return The node index of the created instruction.
 *
 * # Example
 *
 * Measure all qubits into the corresponding clbit index at the end of the circuit.
 *
 * ```c
 * uint32_t num_qubits = qk_dag_num_qubits(dag);
 * for (uint32_t i = 0; i < num_qubits; i++) {
 *     qk_dag_apply_measure(dag, i, i, false);
 * }
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if `dag` is not an aligned, non-null pointer to a valid ``QkDag``,
 * or if `qubit` or `clbit` are out of range.
 */
uint32_t qk_dag_apply_measure(QkDag *dag, uint32_t qubit, uint32_t clbit, bool front);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Apply a reset to the DAG.
 *
 * @param dag The circuit to apply to.
 * @param qubit The qubit index to reset.
 * @param front Whether to apply the reset at the start of the circuit. Usually `false`.
 *
 * @return The node index of the created instruction.
 *
 * # Examples
 *
 * Apply initial resets on all qubits.
 *
 * ```c
 * uint32_t num_qubits = qk_dag_num_qubits(dag);
 * for (uint32_t qubit = 0; qubit < num_qubits; qubit++) {
 *     qk_dag_apply_reset(dag, qubit, true);
 * }
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if `dag` is not an aligned, non-null pointer to a valid ``QkDag``,
 * or if `qubit` is out of range.
 */
uint32_t qk_dag_apply_reset(QkDag *dag, uint32_t qubit, bool front);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Apply a barrier to the DAG.
 *
 * @param dag The circuit to apply to.
 * @param qubits The qubit indices to apply the barrier to.  This can be null, in which case
 *     `num_qubits` is not read, and the barrier is applied to all qubits in the DAG.
 * @param num_qubits How many qubits the barrier applies to.
 * @param front Whether to apply the barrier at the start of the circuit. Usually `false`.
 *
 * @return The node index of the created instruction.
 *
 * # Examples
 *
 * Apply a final barrier on all qubits:
 *
 * ```c
 * qk_dag_apply_barrier(dag, NULL, qk_dag_num_qubits(dag), false);
 * ```
 *
 * Apply a barrier at the beginning of a circuit on specified qubit indices:
 *
 * ```c
 * uint32_t qubits[] = {0, 2, 4, 5};
 * uint32_t num_qubits = sizeof(qubits) / sizeof(qubits[0]);
 * qk_dag_apply_barrier(dag, qubits, num_qubits, true);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if:
 *
 * * `dag` is not an aligned, non-null pointer to a valid ``QkDag``,
 * * `qubits` is not aligned or is not valid for `num_qubits` reads of initialized, in-bounds and
 *   unduplicated indices, unless `qubits` is null.
 */
uint32_t qk_dag_apply_barrier(QkDag *dag, const uint32_t *qubits, uint32_t num_qubits, bool front);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Apply a unitary gate to a DAG.
 *
 * The values in `matrix` should form a row-major unitary matrix of the correct size for the number
 * of qubits.  The data is copied out of the pointer, and only needs to be valid for reads until
 * this function returns.
 *
 * See @verbatim embed:rst:inline ::ref:`circuit-conventions` @endverbatim for detail on the
 * bit-labelling and matrix conventions of Qiskit.
 *
 * @param dag The circuit to apply to.
 * @param matrix An initialized row-major unitary matrix of total size ``4**num_qubits``.
 * @param qubits An array of distinct ``uint32_t`` indices of the qubits.
 * @param num_qubits The number of qubits the gate applies to.
 * @param front Whether to apply the gate at the start of the circuit. Usually `false`.
 *
 * @return The node index of the created instruction.
 *
 * # Safety
 *
 * Behavior is undefined if any of:
 * * `dag` is not an aligned, non-null pointer to a valid ``QkDag``,
 * * `matrix` is not an aligned pointer to `4**num_qubits` initialized values,
 * * `qubits` is not an aligned pointer to `num_qubits` initialized values.
 */
uint32_t qk_dag_apply_unitary(QkDag *dag,
                              const QkComplex64 *matrix,
                              const uint32_t *qubits,
                              uint32_t num_qubits,
                              bool front);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the standard gate of the specified node.
 *
 * Panics if the node is not a standard gate operation.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the standard gate of.
 * @param out_params A buffer to be filled with the gate's params or NULL
 *     if they're not wanted.
 *
 * @return The gate value.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 *
 * uint32_t qubit[1] = {0};
 * uint32_t h_gate_idx = qk_dag_apply_gate(dag, QkGate_H, qubit, NULL, false);
 *
 * QkGate gate = qk_dag_op_node_gate_op(dag, h_gate_idx, NULL);
 *
 * qk_dag_free(dag);
 * qk_quantum_register_free(qr);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a
 * ``QkDag``.
 * If ``out_params`` is non-NULL, it must point to a buffer large enough to
 * hold all the gate's params, otherwise behavior is undefined as this
 * function will write beyond its bounds.
 * You can check ``qk_dag_op_node_num_params`` to determine how many params
 * are required for any given operation node.
 */
QkGate qk_dag_op_node_gate_op(const QkDag *dag, uint32_t node, double *out_params);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Copy out the unitary matrix of the corresponding node index.
 *
 * Panics if the node is not a unitary gate.
 *
 * @param dag The circuit to read from.
 * @param node The node index of the unitary matrix instruction.
 * @param out Allocated and aligned memory for `4**num_qubits` complex values in row-major order,
 *     where `num_qubits` is the number of qubits the gate applies to.
 *
 * # Safety
 *
 * Behavior is undefined if `dag` is not a non-null pointer to a valid `QkDag`, if `out` is
 * unaligned, or if `out` is not valid for `4**num_qubits` writes of `QkComplex64`.
 */
void qk_dag_op_node_unitary(const QkDag *dag, uint32_t node, QkComplex64 *out);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Get the "kind" of an operation node.
 *
 * The result can be used in a switch statement to dispatch proper handling
 * when iterating over operation nodes.
 *
 * Panics if ``node`` is not an operation node.
 *
 * @param dag A pointer to the DAG.
 * @param node The operation node to get the "kind" of.
 *
 * @return The "kind" of the node.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
QkOperationKind qk_dag_op_node_kind(const QkDag *dag, uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the successors of the specified node.
 *
 * The successors array and its length are returned as a `QkDagNeighbors` struct, where each element in the
 * array corresponds to a DAG node index.
 * You must call the `qk_dag_neighbors_clear` function when done to free the memory allocated for the struct.
 *
 * @param dag A pointer to the DAG.
 * @param node The node to get the successors of.
 *
 * @return An instance of the `QkDagNeighbors` struct with the successors information.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(2, "qr");
 * qk_dag_add_quantum_register(dag, qr);
 * qk_quantum_register_free(qr);
 *
 * uint32_t node_cx = qk_dag_apply_gate(dag, QkGate_CX, (uint32_t[]){0, 1}, NULL, false);
 *
 * QkDagNeighbors successors = qk_dag_successors(dag, node_cx);
 *
 * qk_dag_neighbors_clear(&successors);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
QkDagNeighbors qk_dag_successors(const QkDag *dag,
                                 uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Retrieve the predecessors of the specified node.
 *
 * The predecessors array and its length are returned as a `QkDagNeighbors` struct, where each element in the
 * array corresponds to a DAG node index.
 * You must call the `qk_dag_neighbors_clear` function when done to free the memory allocated for the struct.
 *
 * @param dag A pointer to the DAG.
 * @param node The node to get the predecessors of.
 *
 * @return An instance of the `QkDagNeighbors` struct with the predecessors information.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(2, "qr");
 * qk_dag_add_quantum_register(dag, qr);
 * qk_quantum_register_free(qr);
 *
 * uint32_t node_cx = qk_dag_apply_gate(dag, QkGate_CX, (uint32_t[]){0, 1}, NULL, false);
 *
 * QkDagNeighbors predecessors = qk_dag_predecessors(dag, node_cx);
 *
 * qk_dag_neighbors_clear(&predecessors);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
QkDagNeighbors qk_dag_predecessors(const QkDag *dag,
                                   uint32_t node);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Clear the fields of the input `QkDagNeighbors` struct.
 *
 * The function deallocates the memory pointed to by the `neighbors` field and sets it to NULL.
 * It also sets the `num_neighbors` field to 0.
 *
 * @param neighbors A pointer to a `QkDagNeighbors` object.
 *
 * # Safety
 *
 * Behavior is undefined if ``neighbors`` is not a valid, non-null pointer to a QkDagNeighbors
 * object populated with either ``qk_dag_successors`` or ``qk_dag_predecessors``.
 */
void qk_dag_neighbors_clear(QkDagNeighbors *neighbors);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Return the details for an instruction in the circuit.
 *
 * This is a mirror of `qk_circuit_get_instruction`.  You can also use individual methods such as
 * `qk_dag_op_node_gate_op` to get individual properties.
 *
 * You must call `qk_circuit_instruction_clear` to reset the `QkCircuitInstruction` before reusing
 * it or dropping it.
 *
 * @param dag The circuit to retrieve the instruction from.
 * @param index The node index.  It is an error to pass an index that is node a valid op node.
 * @param instruction A point to where to write out the `QkCircuitInstruction`.
 *
 * # Examples
 *
 * Iterate through a DAG to find which qubits have measures on them:
 *
 * ```c
 * bool *measured = calloc(qk_dag_num_qubits(dag), sizeof(*measured));
 * uint32_t num_ops = qk_dag_num_op_nodes(dag);
 * uint32_t *ops = malloc(num_ops * sizeof(*ops));
 * qk_dag_topological_op_nodes(dag, ops);
 *
 * // Storage space for the instruction.
 * QkCircuitInstruction inst;
 * for (uint32_t i = 0; i < num_ops; i++) {
 *     qk_dag_get_instruction(dag, ops[i], &inst);
 *     if (!strcmp(inst.name, "measure"))
 *         measured[inst.qubits[0]] = true;
 *     qk_circuit_instruction_clear(&inst);
 * }
 *
 * free(ops);
 * free(measured);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if either `dag` or `instruction` are not valid, aligned, non-null pointers
 * to the relevant data type.  The fields of `instruction` need not be initialized.
 */
void qk_dag_get_instruction(const QkDag *dag, uint32_t index, QkCircuitInstruction *instruction);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Compose the ``other`` DAG onto the ``dag`` instance with the option of a subset
 * of input wires of ``other`` being mapped onto a subset of output wires of ``dag``.
 *
 * ``other`` may include a smaller or equal number of wires for each type.
 *
 * @param dag A pointer to the DAG to be composed on.
 * @param other A pointer to the DAG to compose with ``dag``.
 * @param qubits A list of indices representing the qubit wires to compose
 *     onto.
 * @param clbits A list of indices representing the clbit wires to compose
 *     onto.
 *
 * @return ``QkExitCode_Success`` upon successful decomposition, otherwise a DAG-specific
 *     error code indicating the cause of the failure.
 *
 * # Example
 *
 * ```c
 * // Build the following dag
 * // rqr_0: ──■───────
 * //          │  ┌───┐
 * // rqr_1: ──┼──┤ Y ├
 * //        ┌─┴─┐└───┘
 * // rqr_2: ┤ X ├─────
 * //        └───┘
 * QkDag *dag_right = qk_dag_new();
 * QkQuantumRegister *rqr = qk_quantum_register_new(3, "rqr");
 * qk_dag_add_quantum_register(dag_right, rqr);
 * qk_dag_add_classical_register(dag_right, rcr);
 * qk_dag_apply_gate(dag_right, QkGate_CX, (uint32_t[]){0, 2}, NULL, false);
 * qk_dag_apply_gate(dag_right, QkGate_Y, (uint32_t[]){1}, NULL, false);
 *
 * // Build the following dag
 * //          ┌───┐
 * // lqr_0: ──┤ H ├───
 * //        ┌─┴───┴──┐
 * // lqr_1: ┤ P(0.1) ├
 * //        └────────┘
 * QkDag *dag_left = qk_dag_new();
 * QkQuantumRegister *lqr = qk_quantum_register_new(2, "lqr");
 * qk_dag_add_quantum_register(dag_left, lqr);
 * qk_dag_add_classical_register(dag_left, lcr);
 * qk_dag_apply_gate(dag_left, QkGate_H, (uint32_t[]){0}, NULL, false);
 * qk_dag_apply_gate(dag_left, QkGate_Phase, (uint32_t[]){1}, (double[]){0.1}, false);
 *
 * // Compose left circuit onto right circuit
 * // Should result in circuit
 * //             ┌───┐
 * // rqr_0: ──■──┤ H ├──────────
 * //          │  ├───┤┌────────┐
 * // rqr_1: ──┼──┤ Y ├┤ P(0.1) ├
 * //        ┌─┴─┐└───┘└────────┘
 * // rqr_2: ┤ X ├───────────────
 * //        └───┘
 * qk_dag_compose(dag_right, dag_left, NULL, NULL);
 *
 * // Clean up after you're done
 * qk_dag_free(dag_left);
 * qk_dag_free(dag_right);
 * qk_quantum_register_free(lqr);
 * qk_quantum_register_free(rqr);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` or ``other`` are not valid, non-null pointers to a ``QkDag``.
 * If ``qubit`` nor ``clbit`` are NULL, it must contains a less or equal amount
 * than what the circuit owns.
 */
QkExitCode qk_dag_compose(QkDag *dag,
                          const QkDag *other,
                          const uint32_t *qubits,
                          const uint32_t *clbits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Free the DAG.
 *
 * @param dag A pointer to the DAG to free.
 *
 * # Example
 * ```c
 * QkDag *dag = qk_dag_new();
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not either null or a valid pointer to a
 * ``QkDag``.
 */
void qk_dag_free(QkDag *dag);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Convert a given DAG to a circuit.
 *
 * The new circuit is copied from the DAG; the original ``dag`` reference is still owned by the
 * caller and still required to be freed with `qk_dag_free`.  You must free the returned circuit
 * with ``qk_circuit_free`` when done with it.
 *
 * @param dag A pointer to the DAG from which to create the circuit.
 *
 * @return A pointer to the new circuit.
 *
 * # Example
 *
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(2, "qr");
 * qk_dag_add_quantum_register(dag, qr);
 * qk_quantum_register_free(qr);
 *
 * QkCircuit *qc = qk_dag_to_circuit(dag);
 *
 * qk_circuit_free(qc);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``.
 */
QkCircuit *qk_dag_to_circuit(const QkDag *dag);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Return the operation nodes in the DAG listed in topological order.
 *
 * @param dag A pointer to the DAG.
 * @param out_order A pointer to an array of ``qk_dag_num_op_nodes(dag)`` elements
 * of type ``uint32_t``, where this function will write the output to.
 *
 * # Example
 *
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 *
 * uint32_t qubit[1] = {0};
 * qk_dag_apply_gate(dag, QkGate_H, qubit, NULL, false);
 * qk_dag_apply_gate(dag, QkGate_S, qubit, NULL, false);
 *
 * // get the number of operation nodes
 * uint32_t num_ops = qk_dag_num_op_nodes(dag); // 2
 * uint32_t *out_order = malloc(sizeof(uint32_t) * num_ops);
 *
 * // get operation nodes listed in topological order
 * qk_dag_topological_op_nodes(dag, out_order);
 *
 * // do something with the ordered nodes
 * for (uint32_t i = 0; i < num_ops; i++) {
 *     QkGate gate = qk_dag_op_node_gate_op(dag, out_order[i], NULL);
 *     printf("The gate at location %u is %u.\n", i, gate);
 * }
 *
 * // free the out_order array, register, and dag pointer when done
 * free(out_order);
 * qk_quantum_register_free(qr);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid, non-null pointer to a ``QkDag``
 * or if ``out_order`` is not a valid, non-null pointer to a sequence of ``qk_dag_num_op_nodes(dag)``
 * consecutive elements of ``uint32_t``.
 */
void qk_dag_topological_op_nodes(const QkDag *dag,
                                 uint32_t *out_order);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Replace a node in a `QkDag` with a subcircuit specfied by another `QkDag`
 *
 * @param dag A pointer to the DAG.
 * @param node The node index of the operation to replace with the other `QkDag`. This
 *     must be the node index for an operation node in ``dag`` and the qargs and cargs
 *     count must match the number of qubits and clbits in `replacement`.
 * @param replacement The other `QkDag` to replace `node` with. This dag must have
 *     the same number of qubits as the operation for ``node``. The node
 *     bit ordering will be ordering will be handled in order, so `qargs[0]` for
 *     `node` will be mapped to `qubits[0]` in `replacement`, `qargs[1]` to
 *     `qubits[0]`, etc. The same pattern applies to classical bits too.
 *
 * # Example
 *
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 *
 * uint32_t qubit[1] = {0};
 * uint32_t node_to_replace = qk_dag_apply_gate(dag, QkGate_H, qubit, NULL, false);
 * qk_dag_apply_gate(dag, QkGate_S, qubit, NULL, false);
 *
 * // Build replacement dag for H
 * QkDag *replacement = qk_dag_new();
 * QkQuantumRegister *replacement_qr = qk_quantum_register_new(1, "other");
 * qk_dag_add_quantum_register(replacement, replacement_qr);
 * double pi_param[1] = {3.14159};
 * qk_dag_apply_gate(replacement, QkGate_RZ, qubit, pi_param, false);
 * qk_dag_apply_gate(replacement, QkGate_SX, qubit, NULL, false);
 * qk_dag_apply_gate(replacement, QkGate_RZ, qubit, pi_param, false);
 *
 * qk_dag_substitute_node_with_dag(dag, node_to_replace, replacement);
 *
 * // Free the replacement dag, register, dag, and register
 * qk_quantum_register_free(replacement_qr);
 * qk_dag_free(replacement);
 * qk_quantum_register_free(qr);
 * qk_dag_free(dag);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` and ``replacement`` are not a valid, non-null pointer to a
 * ``QkDag``.
 */
void qk_dag_substitute_node_with_dag(QkDag *dag, uint32_t node, const QkDag *replacement);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkDag
 * Return a copy of self with the same structure but empty.
 *
 * That structure includes:
 * * name and other metadata
 * * global phase
 * * duration
 * * all the qubits and clbits, including the registers.
 *
 * @param dag A pointer to the DAG to copy.
 * @param vars_mode The mode for handling classical variables.
 * @param blocks_mode The mode for handling blocks.
 *
 * @return The pointer to the copied DAG circuit.
 *
 * # Example
 *
 * ```c
 * QkDag *dag = qk_dag_new();
 * QkQuantumRegister *qr = qk_quantum_register_new(1, "my_register");
 * qk_dag_add_quantum_register(dag, qr);
 *
 * uint32_t qubit[1] = {0};
 * qk_dag_apply_gate(dag, QkGate_H, qubit, NULL, false);
 *
 * // As the DAG does not contain any control-flow instructions,
 * // vars_mode and blocks_mode do not have any effect.
 * QkDag *copied_dag = qk_dag_copy_empty_like(dag, QkVarsMode_Alike, QkBlocksMode_Drop);
 * uint32_t num_ops_in_copied_dag = qk_dag_num_op_nodes(copied_dag); // 0
 *
 * // do something with copied_dag
 *
 * qk_quantum_register_free(qr);
 * qk_dag_free(dag);
 * qk_dag_free(copied_dag);
 * ```
 * # Safety
 *
 * Behavior is undefined if ``dag`` is not a valid pointer to a ``QkDag``.
 */
QkDag *qk_dag_copy_empty_like(const QkDag *dag, QkVarsMode vars_mode, QkBlocksMode blocks_mode);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Construct a new ``QkParam`` representing an unbound symbol.
 *
 * @param name The name of symbol. This cannot be empty.
 *
 * @return A pointer to the created ``QkParam``.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * ```
 *
 * # Safety
 *
 * The `name` parameter must be a pointer to memory that contains a valid
 * nul terminator at the end of the string. It also must be valid for reads of
 * bytes up to and including the nul terminator.
 */
QkParam *qk_param_new_symbol(const char *name);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Construct a new ``QkParam`` with a value zero.
 *
 * The ``QkParam`` returned from this function can be used
 * to store the result of binary or unary operations.
 *
 * @return A pointer to the created ``QkParam``.
 *
 * # Example
 *
 * ```c
 * QkParam *t = qk_param_zero();
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_new_symbol("b");
 * qk_param_add(t, a, b);
 * ```
 *
 */
QkParam *qk_param_zero(void);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Free the ``QkParam``.
 *
 * @param param A pointer to the ``QkParam`` to free.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * qk_param_free(a);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``param`` is not either null or a valid pointer to a ``QkParam``.
 */
void qk_param_free(QkParam *param);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Construct a new ``QkParam`` from a ``double``.
 *
 * @param value A ``double`` to initialize the ``QkParam``.
 *
 * @return A pointer to the created ``QkParam``.
 *
 * # Example
 *
 * ```c
 * QkParam *r = qk_param_from_double(2.5);
 * ```
 *
 */
QkParam *qk_param_from_double(double value);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Construct a new ``QkParam`` from a complex number, given as ``QkComplex64``.
 *
 * @param value A ``QkComplex64`` to initialize the ``QkParam``.
 *
 * @return A pointer to the created ``QkParam``.
 *
 * # Example
 *
 * ```c
 * QkComplex64 c = {1.0, 2.0};  // 1 + 2i
 * QkParam *param = qk_param_from_complex(c);
 * ```
 *
 */
QkParam *qk_param_from_complex(QkComplex64 value);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Copy a ``QkParam``.
 *
 * @param param The ``QkParam`` to copy.
 *
 * @return A pointer to the created ``QkParam``.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_copy(a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``param`` is not a valid pointer to a non-null ``QkParam``.
 */
QkParam *qk_param_copy(const QkParam *param);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Get a string representation of the ``QkParam``.
 *
 * @param param A pointer to the ``QkParam``.
 *
 * @return A pointer to a nul-terminated char array of the string representation for ``param``.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * char* str = qk_param_str(a);
 * printf(str);
 * qk_str_free(str);
 * qk_param_free(a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``param`` is not a valid pointer to a non-null ``QkParam``.
 *
 * The string must not be freed with the normal C free, you must use ``qk_str_free`` to
 * free the memory consumed by the String. Not calling ``qk_str_free`` will lead to a
 * memory leak.
 *
 * Do not change the length of the string after it's returned (by writing a nul byte somewhere
 * inside the string or removing the final one), although values can be mutated.
 */
char *qk_param_str(const QkParam *param);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Add two ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result of ``lhs + rhs``.
 * @param lhs A pointer to the left hand side ``QkParam``.
 * @param rhs A pointer to the right hand side ``QkParam``.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_new_symbol("b");
 * QkParam *out = qk_param_zero();
 * qk_param_add(out, a, b);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out``, ``lhs`` or ``rhs`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_add(QkParam *out, const QkParam *lhs, const QkParam *rhs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Subtract two ``QkParam``.
 *
 * @param out A pointer to a ``QkParam`` to store the result of ``lhs - rhs``.
 * @param lhs A pointer to the left hand side ``QkParam``.
 * @param rhs A pointer to the right hand side ``QkParam``.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_new_symbol("b");
 * QkParam *out = qk_param_zero();
 * qk_param_sub(out, a, b);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out``, ``lhs`` or ``rhs`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_sub(QkParam *out, const QkParam *lhs, const QkParam *rhs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Multiply two ``QkParam``.
 *
 * @param out A pointer to a ``QkParam`` to store the result of ``lhs * rhs``.
 * @param lhs A pointer to the left hand side ``QkParam``.
 * @param rhs A pointer to the right hand side ``QkParam``.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_new_symbol("b");
 * QkParam *out = qk_param_zero();
 * qk_param_mul(out, a, b);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out``, ``lhs`` or ``rhs`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_mul(QkParam *out, const QkParam *lhs, const QkParam *rhs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Divide a ``QkParam`` by another.
 *
 * @param out A pointer to a ``QkParam`` to store the result of ``lhs / rhs``.
 * @param num A pointer to the numerator ``QkParam``.
 * @param den A pointer to the denominator ``QkParam``.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_new_symbol("b");
 * QkParam *out = qk_param_zero();
 * qk_param_div(out, a, b);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out``, ``num`` or ``den`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_div(QkParam *out, const QkParam *num, const QkParam *den);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Raise a ``QkParam`` to the power of another.
 *
 * @param out A pointer to a ``QkParam`` to store the result of ``lhs ** rhs``.
 * @param base A pointer to the left hand side ``QkParam``.
 * @param pow A pointer to the right hand side ``QkParam``.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *base = qk_param_new_symbol("a");
 * QkParam *pow = qk_param_new_symbol("b");
 * QkParam *out = qk_param_zero();
 * qk_param_pow(out, base, pow);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out``, ``base`` or ``pow`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_pow(QkParam *out, const QkParam *base, const QkParam *pow);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the sine of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``sin(src)``.
 * @param src A pointer to the ``QkParam`` to apply the sine to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_sin(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_sin(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the cosine of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``cos(src)``.
 * @param src A pointer to the ``QkParam`` to apply the cosine to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_cos(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_cos(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the tangent of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``tan(src)``.
 * @param src A pointer to the ``QkParam`` to apply the tangent to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_tan(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_tan(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the arcsine of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``asin(src)``.
 * @param src A pointer to the ``QkParam`` to apply the arcsine to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_asin(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_asin(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the arccosine of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``acos(src)``.
 * @param src A pointer to the ``QkParam`` to apply the cosine to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_acos(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_acos(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the arctangent of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store a result, ``atan(src)``.
 * @param src A pointer to the ``QkParam`` to apply the arctangent to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_atan(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_atan(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the natural logarithm of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``log(src)``.
 * @param src A pointer to the ``QkParam`` to apply the logarithm to.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_log(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_log(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Apply the exponential function to a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``exp(src)``.
 * @param src A pointer to the ``QkParam`` to compute the exponential of.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_exp(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_exp(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the absolute value of a ``QkParam``.
 *
 * @param out A pointer to the QkParam to store a result.
 * @param src A pointer to the QkParam.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_abs(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_abs(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Get the sign of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``sign(src)``.
 * @param src A pointer to the ``QkParam`` to get the sign of.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_sign(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_sign(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Negate a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``-src``.
 * @param src A pointer to the ``QkParam`` to negate.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_neg(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_neg(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Calculate the complex conjugate of a ``QkParam``.
 *
 * @param out A pointer to the ``QkParam`` to store the result, ``conj(src)``.
 * @param src A pointer to the ``QkParam`` to conjugate.
 *
 * @return An exit code indicating ``QkExitCode_Success`` upon success and an error otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *out = qk_param_zero();
 * qk_param_conjugate(out, a);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``out`` or ``src`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
QkExitCode qk_param_conjugate(QkParam *out, const QkParam *src);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Compare two ``QkParam`` for equality.
 *
 * @param lhs A pointer to the left hand side ``QkParam``.
 * @param rhs A pointer to the right hand side ``QkParam``.
 *
 * @return ``true`` if the ``QkParam`` objects are equal, ``false`` otherwise.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *one = qk_param_from_value(1.0);
 * QkParam *mone = qk_param_from_value(-1.0);
 *
 * QkParam *x = qk_param_add(a, one);
 * QkParam *y = qk_param_sub(a, mone);
 *
 * bool equal = qk_param_equal(x, y);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if any of ``lhs`` or ``rhs`` is not a valid, non-null
 * pointer to a ``QkParam``.
 */
bool qk_param_equal(const QkParam *lhs, const QkParam *rhs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkParam
 * Attempt casting the ``QkParam`` as ``double``.
 *
 * If the parameter could not be cast to a ``double``, because there were unbound parameters,
 * ``NAN`` is returned. Note that for ``QkParam`` representing complex values the real part is
 * returned.
 *
 * @param param A pointer to the ``QkParam`` to evaluate.
 *
 * @return The value, if casting was successful, otherwise ``NAN``.
 *
 * # Example
 *
 * ```c
 * QkParam *a = qk_param_new_symbol("a");
 * QkParam *b = qk_param_new_symbol("b");
 * QkParam *x = qk_param_zero();
 * qk_param_add(x, a, b);
 *
 * const QkParam* keys[2] = {a, b};
 * QkParam *y = qk_param_zero();
 * qk_param_bind(y, x, keys, {1.0, 2.0}, 2);
 *
 * double out = qk_param_as_real(y);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``param`` is not a valid, non-null pointer to a ``QkParam``.
 */
double qk_param_as_real(const QkParam *param);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Construct the zero observable (without any terms).
 *
 * @param num_qubits The number of qubits the observable is defined on.
 *
 * @return A pointer to the created observable.
 *
 * # Example
 * ```c
 *     QkObs *zero = qk_obs_zero(100);
 * ```
 *
 */
QkObs *qk_obs_zero(uint32_t num_qubits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Construct the identity observable.
 *
 * @param num_qubits The number of qubits the observable is defined on.
 *
 * @return A pointer to the created observable.
 *
 * # Example
 * ```c
 *     QkObs *identity = qk_obs_identity(100);
 * ```
 *
 */
QkObs *qk_obs_identity(uint32_t num_qubits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Construct a new observable from raw data.
 *
 * @param num_qubits The number of qubits the observable is defined on.
 * @param num_terms The number of terms.
 * @param num_bits The total number of non-identity bit terms.
 * @param coeffs A pointer to the first element of the coefficients array, which has length
 *     ``num_terms``.
 * @param bit_terms A pointer to the first element of the bit terms array, which has length
 *     ``num_bits``.
 * @param indices A pointer to the first element of the indices array, which has length
 *     ``num_bits``. Note that, per term, these *must* be sorted incrementally.
 * @param boundaries A pointer to the first element of the boundaries array, which has length
 *     ``num_terms + 1``.
 *
 * @return If the input data was coherent and the construction successful, the result is a pointer
 *     to the observable. Otherwise a null pointer is returned.
 *
 * # Example
 * ```c
 *    // define the raw data for the 100-qubit observable |01><01|_{0, 1} - |+-><+-|_{98, 99}
 *    uint32_t num_qubits = 100;
 *    uint64_t num_terms = 2;  // we have 2 terms: |01><01|, -1 * |+-><+-|
 *    uint64_t num_bits = 4; // we have 4 non-identity bits: 0, 1, +, -
 *    QkComplex64 coeffs = {1, -1};
 *    QkBitTerm bits[4] = {QkBitTerm_Zero, QkBitTerm_One, QkBitTerm_Plus, QkBitTerm_Minus};
 *
 *    uint32_t indices[4] = {0, 1, 98, 99};  // <-- e.g. {1, 0, 99, 98} would be invalid
 *    size_t boundaries[3] = {0, 2, 4};
 *    QkObs *obs = qk_obs_new(
 *        num_qubits, num_terms, num_bits, &coeffs, bits, indices, boundaries
 *    );
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if any of the following conditions are violated:
 *
 *   * ``coeffs`` is a pointer to a ``QkComplex64`` array of length ``num_terms``
 *   * ``bit_terms`` is a pointer to an array of valid ``QkBitTerm`` elements of length ``num_bits``
 *   * ``indices`` is a pointer to a ``uint32_t`` array of length ``num_bits``, which is
 *     term-wise sorted in strict ascending order, and every element is smaller than ``num_qubits``
 *   * ``boundaries`` is a pointer to a ``size_t`` array of length ``num_terms + 1``, which is
 *     sorted in ascending order, the first element is 0 and the last element is
 *     smaller than ``num_terms``
 */
QkObs *qk_obs_new(uint32_t num_qubits,
                  uint64_t num_terms,
                  uint64_t num_bits,
                  QkComplex64 *coeffs,
                  QkBitTerm *bit_terms,
                  uint32_t *indices,
                  size_t *boundaries);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Free the observable.
 *
 * @param obs A pointer to the observable to free.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_zero(100);
 *     qk_obs_free(obs);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``obs`` is not either null or a valid pointer to a ``QkObs``.
 */
void qk_obs_free(QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Add a term to the observable.
 *
 * @param obs A pointer to the observable.
 * @param cterm A pointer to the term to add.
 *
 * @return An exit code. This is ``>0`` if the term is incoherent or adding the term fails.
 *
 * # Example
 * ```c
 * uint32_t num_qubits = 100;
 * QkObs *obs = qk_obs_zero(num_qubits);
 *
 * QkComplex64 coeff = {1, 0};
 * QkBitTerm bit_terms[3] = {QkBitTerm_X, QkBitTerm_Y, QkBitTerm_Z};
 * uint32_t indices[3] = {0, 1, 2};
 * QkObsTerm term = {coeff, 3, bit_terms, indices, num_qubits};
 *
 * QkExitCode exit_code = qk_obs_add_term(obs, &term);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if any of the following is violated:
 *
 *   * ``obs`` is a valid, non-null pointer to a ``QkObs``
 *   * ``cterm`` is a valid, non-null pointer to a ``QkObsTerm``
 */
QkExitCode qk_obs_add_term(QkObs *obs, const QkObsTerm *cterm);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get an observable term by reference.
 *
 * A ``QkObsTerm`` contains pointers to the indices and bit terms in the term, which
 * can be used to modify the internal data of the observable. This can leave the observable
 * in an incoherent state and should be avoided, unless great care is taken. It is generally
 * safer to construct a new observable instead of attempting in-place modifications.
 *
 * @param obs A pointer to the observable.
 * @param index The index of the term to get.
 * @param out A pointer to a ``QkObsTerm`` used to return the observable term.
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     QkObsTerm term;
 *     QkExitCode exit_code = qk_obs_term(obs, 0, &term);
 *     // out-of-bounds indices return an error code
 *     // QkExitCode error = qk_obs_term(obs, 12, &term);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if any of the following is violated
 * * ``obs`` is a valid, non-null pointer to a ``QkObs``
 * * ``out`` is a valid, non-null pointer to a ``QkObsTerm``
 */
QkExitCode qk_obs_term(QkObs *obs, uint64_t index, QkObsTerm *out);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get the number of terms in the observable.
 *
 * @param obs A pointer to the observable.
 *
 * @return The number of terms in the observable.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     size_t num_terms = qk_obs_num_terms(obs);  // num_terms==1
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
size_t qk_obs_num_terms(const QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get the number of qubits the observable is defined on.
 *
 * @param obs A pointer to the observable.
 *
 * @return The number of qubits the observable is defined on.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     uint32_t num_qubits = qk_obs_num_qubits(obs);  // num_qubits==100
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
uint32_t qk_obs_num_qubits(const QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get the number of bit terms/indices in the observable.
 *
 * @param obs A pointer to the observable.
 *
 * @return The number of terms in the observable.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     size_t len = qk_obs_len(obs);  // len==0, as there are no non-trivial bit terms
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
size_t qk_obs_len(const QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get a pointer to the coefficients.
 *
 * This can be used to read and modify the observable's coefficients. The resulting
 * pointer is valid to read for ``qk_obs_num_terms(obs)`` elements of ``QkComplex64``.
 *
 * @param obs A pointer to the observable.
 *
 * @return A pointer to the coefficients.
 *
 * # Example
 * ```c
 *    QkObs *obs = qk_obs_identity(100);
 *    size_t num_terms = qk_obs_num_terms(obs);
 *    QkComplex64 *coeffs = qk_obs_coeffs(obs);
 *
 *    for (size_t i = 0; i < num_terms; i++) {
 *        printf("%f + i%f\n", coeffs[i].re, coeffs[i].im);
 *    }
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
QkComplex64 *qk_obs_coeffs(QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get a pointer to the indices.
 *
 * This can be used to read and modify the observable's indices. The resulting pointer is
 * valid to read for ``qk_obs_len(obs)`` elements of size ``uint32_t``.
 *
 * @param obs A pointer to the observable.
 *
 * @return A pointer to the indices.
 *
 * # Example
 * ```c
 * uint32_t num_qubits = 100;
 * QkObs *obs = qk_obs_zero(num_qubits);
 *
 * QkComplex64 coeff = {1, 0};
 * QkBitTerm bit_terms[3] = {QkBitTerm_X, QkBitTerm_Y, QkBitTerm_Z};
 * uint32_t term_indices[3] = {0, 1, 2};
 * QkObsTerm term = {coeff, 3, bit_terms, term_indices, num_qubits};
 * qk_obs_add_term(obs, &term);
 *
 * size_t len = qk_obs_len(obs);
 * uint32_t *indices = qk_obs_indices(obs);
 *
 * for (size_t i = 0; i < len; i++) {
 *     printf("index %i: %i\n", i, indices[i]);
 * }
 *
 * qk_obs_free(obs);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
uint32_t *qk_obs_indices(QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get a pointer to the term boundaries.
 *
 * This can be used to read and modify the observable's term boundaries. The resulting pointer is
 * valid to read for ``qk_obs_num_terms(obs) + 1`` elements of size ``size_t``.
 *
 * @param obs A pointer to the observable.
 *
 * @return A pointer to the boundaries.
 *
 * # Example
 * ```c
 *    uint32_t num_qubits = 100;
 *    QkObs *obs = qk_obs_zero(num_qubits);
 *
 *    QkComplex64 coeff = {1, 0};
 *    QkBitTerm bit_terms[3] = {QkBitTerm_X, QkBitTerm_Y, QkBitTerm_Z};
 *    uint32_t indices[3] = {0, 1, 2};
 *    QkObsTerm term = {coeff, 3, bit_terms, indices, num_qubits};
 *    qk_obs_add_term(obs, &term);
 *
 *    size_t num_terms = qk_obs_num_terms(obs);
 *    size_t *boundaries = qk_obs_boundaries(obs);
 *
 *    for (size_t i = 0; i < num_terms + 1; i++) {
 *        printf("boundary %i: %i\n", i, boundaries[i]);
 *    }
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
size_t *qk_obs_boundaries(QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Get a pointer to the bit terms.
 *
 * This can be used to read and modify the observable's bit terms. The resulting pointer is
 * valid to read for ``qk_obs_len(obs)`` elements of size ``uint8_t``.
 *
 * @param obs A pointer to the observable.
 *
 * @return A pointer to the bit terms.
 *
 * # Example
 * ```c
 *     uint32_t num_qubits = 100;
 *     QkObs *obs = qk_obs_zero(num_qubits);
 *
 *     QkComplex64 coeff = {1, 0};
 *     QkBitTerm bit_terms[3] = {QkBitTerm_X, QkBitTerm_Y, QkBitTerm_Z};
 *     uint32_t indices[3] = {0, 1, 2};
 *     QkObsTerm term = {coeff, 3, bit_terms, indices, num_qubits};
 *     qk_obs_add_term(obs, &term);
 *
 *     size_t len = qk_obs_len(obs);
 *     QkBitTerm *bits = qk_obs_bit_terms(obs);
 *
 *     for (size_t i = 0; i < len; i++) {
 *         printf("bit term %i: %i\n", i, bits[i]);
 *     }
 *
 *     qk_obs_free(obs);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``,
 * or if invalid valus are written into the resulting ``QkBitTerm`` pointer.
 */
QkBitTerm *qk_obs_bit_terms(QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Multiply the observable by a complex coefficient.
 *
 * @param obs A pointer to the observable.
 * @param coeff The coefficient to multiply the observable with.
 *
 * @return A pointer to the result ``obs * coeff``.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     QkComplex64 coeff = {2, 0};
 *     QkObs *result = qk_obs_multiply(obs, &coeff);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if any of the following is violated
 * * ``obs`` is a valid, non-null pointer to a ``QkObs``
 * * ``coeff`` is a valid, non-null pointer to a ``QkComplex64``
 */
QkObs *qk_obs_multiply(const QkObs *obs, const QkComplex64 *coeff);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Add two observables.
 *
 * @param left A pointer to the left observable.
 * @param right A pointer to the right observable.
 *
 * @return A pointer to the result ``left + right``.
 *
 * # Example
 * ```c
 *     QkObs *left = qk_obs_identity(100);
 *     QkObs *right = qk_obs_zero(100);
 *     QkObs *result = qk_obs_add(left, right);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``left`` or ``right`` are not valid, non-null pointers to
 * ``QkObs``\ s.
 */
QkObs *qk_obs_add(const QkObs *left, const QkObs *right);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Compose (multiply) two observables.
 *
 * @param first One observable.
 * @param second The other observable.
 *
 * @return ``first.compose(second)`` which equals the observable ``result = second @ first``,
 *     in terms of the matrix multiplication ``@``.
 *
 * # Example
 * ```c
 *     QkObs *first = qk_obs_zero(100);
 *     QkObs *second = qk_obs_identity(100);
 *     QkObs *result = qk_obs_compose(first, second);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``first`` or ``second`` are not valid, non-null pointers to
 * ``QkObs``\ s.
 */
QkObs *qk_obs_compose(const QkObs *first, const QkObs *second);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Compose (multiply) two observables according to a custom qubit order.
 *
 * Notably, this allows composing two observables of different size.
 *
 * @param first One observable.
 * @param second The other observable. The number of qubits must match the length of ``qargs``.
 * @param qargs The qubit arguments specified which indices in ``first`` to associate with
 *     the ones in ``second``.
 *
 * @return ``first.compose(second)`` which equals the observable ``result = second @ first``,
 *     in terms of the matrix multiplication ``@``.
 *
 * # Example
 * ```c
 *     QkObs *first = qk_obs_zero(100);
 *     QkObs *second = qk_obs_identity(100);
 *     QkObs *result = qk_obs_compose(first, second);
 * ```
 *
 * # Safety
 *
 * To call this function safely
 *
 *   * ``first`` and ``second`` must be valid, non-null pointers to ``QkObs``\ s
 *   * ``qargs`` must point to an array of ``uint32_t``, readable for ``qk_obs_num_qubits(second)``
 *     elements (meaning the number of qubits in ``second``)
 */
QkObs *qk_obs_compose_map(const QkObs *first, const QkObs *second, const uint32_t *qargs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Apply a new qubit layout to the observable.
 *
 * The layout is set by an array ``layout`` of new indices, specifying that qubit at current
 * index ``i`` is relabelled to index ``layout[i]``. The number of qubits the observable
 * acts on can be extended by setting a larger ``num_qubits`` than the current observable has.
 *
 * @param obs A pointer to the observable, this observable will be modified in place upon success.
 * Check the exit code to ensure the layout was correctly applied.
 * @param layout A pointer to the layout. The pointer must point to an array to
 * ``qk_obs_num_qubits(obs)`` elements of type ``uint32_t``. Each element must have values
 * in ``[0, num_qubits)``.
 * @param num_qubits The number of output qubits.
 *
 * @return An exit code.
 * * ``QkExitCode_Success`` upon success
 * * ``QkExitCode_DuplicteIndexError`` if duplicate qubit indices were found
 * * ``QkExitCode_MismatchedQubits`` if ``num_qubits`` is smaller than the number of qubits in
 *   the observable
 * * ``QkExitCode_IndexError`` for any other index errors, such as invalid values in ``layout``.
 *
 * # Example
 *
 * This interface allows to relabel and extend the qubit indices:
 *
 * ```c
 * QkObs *obs = qk_obs_zero(4);
 *
 * // add a term to the observable
 * QkBitTerm bit_terms[3] = {QkBitTerm_X, QkBitTerm_Y, QkBitTerm_Z};
 * uint32_t qubits[3] = {1, 2, 3};
 * complex double coeff = 1;
 * QkObsTerm term = {coeff, 3, bit_terms, qubits, 4};
 * qk_obs_add_term(obs, &term);
 *
 * uint32_t layout[3] = {0, 10, 9};  // qubit mapping is: 0->0, 1->10, 2->9
 * uint32_t num_output_qubits = 11;
 * QkExitCode exit = qk_obs_apply_layout(obs, layout, num_output_qubits);
 * ```
 *
 * In a compiler workflow, this function can conveniently be used to apply a
 * ``QkTranspileLayout*`` obtained from a transpiler pass, called ``transpile_layout``
 * in the following example:
 *
 * ```c
 * // get the number of output qubits
 * uint32_t num_output_qubits = qk_transpile_layout_num_output_qubits(transpile_layout);
 *
 * // get the layout including the ancillas (hence the ``false`` in the function call)
 * uint32_t *layout = malloc(sizeof(uint32_t) * num_output_qubits);
 * qk_transpile_layout_final_layout(transpile_layout, false, layout);
 *
 * // apply the layout
 * QkExitCode exit = qk_obs_apply_layout(obs, layout, num_output_qubits);
 *
 * // free the layout array
 * free(layout);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``obs`` is not a valid, non-null pointer to ``QkObs`` or if ``layout``
 * is not a valid, non-null pointer to a sequence of ``qk_obs_num_qubits(obs)`` consecutive
 * elements of ``uint32_t``.
 */
QkExitCode qk_obs_apply_layout(QkObs *obs, const uint32_t *layout, uint32_t num_qubits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Calculate the canonical representation of the observable.
 *
 * @param obs A pointer to the observable.
 * @param tol The tolerance below which coefficients are considered to be zero.
 *
 * @return The canonical representation of the observable.
 *
 * # Example
 * ```c
 *    QkObs *iden = qk_obs_identity(100);
 *    QkObs *two = qk_obs_add(iden, iden);
 *
 *    double tol = 1e-6;
 *    QkObs *canonical = qk_obs_canonicalize(two, tol);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
QkObs *qk_obs_canonicalize(const QkObs *obs, double tol);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Copy the observable.
 *
 * @param obs A pointer to the observable.
 *
 * @return A pointer to a copy of the observable.
 *
 * # Example
 * ```c
 *     QkObs *original = qk_obs_identity(100);
 *     QkObs *copied = qk_obs_copy(original);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 */
QkObs *qk_obs_copy(const QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Compare two observables for equality.
 *
 * Note that this does not compare mathematical equality, but data equality. This means
 * that two observables might represent the same observable but not compare as equal.
 *
 * @param obs A pointer to one observable.
 * @param other A pointer to another observable.
 *
 * @return ``true`` if the observables are equal, ``false`` otherwise.
 *
 * # Example
 * ```c
 *     QkObs *observable = qk_obs_identity(100);
 *     QkObs *other = qk_obs_identity(100);
 *     bool are_equal = qk_obs_equal(observable, other);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``obs`` or ``other`` are not valid, non-null pointers to
 * ``QkObs``\ s.
 */
bool qk_obs_equal(const QkObs *obs, const QkObs *other);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Return a string representation of a ``QkObs``.
 *
 * @param obs A pointer to the ``QkObs`` to get the string for.
 *
 * @return A pointer to a nul-terminated char array of the string representation for ``obs``
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     char *string = qk_obs_str(obs);
 *     qk_str_free(string);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 *
 * The string must not be freed with the normal C free, you must use ``qk_str_free`` to
 * free the memory consumed by the String. Not calling ``qk_str_free`` will lead to a
 * memory leak.
 *
 * Do not change the length of the string after it's returned (by writing a nul byte somewhere
 * inside the string or removing the final one), although values can be mutated.
 */
char *qk_obs_str(const QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObs
 * Free a string representation.
 *
 * @param string A pointer to the returned string representation from ``qk_obs_str`` or
 *     ``qk_obsterm_str``.
 *
 * # Safety
 *
 * Behavior is undefined if ``str`` is not a pointer returned by ``qk_obs_str`` or
 * ``qk_obsterm_str``.
 */
void qk_str_free(char *string);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkObsTerm
 * Return a string representation of the sparse term.
 *
 * @param term A pointer to the term.
 *
 * @return The function exit code. This is ``>0`` if reading the term failed.
 *
 * # Example
 * ```c
 *     QkObs *obs = qk_obs_identity(100);
 *     QkObsTerm term;
 *     qk_obs_term(obs, 0, &term);
 *     char *string = qk_obsterm_str(&term);
 *     qk_str_free(string);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined ``term`` is not a valid, non-null pointer to a ``QkObsTerm``.
 *
 * The string must not be freed with the normal C free, you must use ``qk_str_free`` to
 * free the memory consumed by the String. Not calling ``qk_str_free`` will lead to a
 * memory leak.
 *
 * Do not change the length of the string after it's returned, although values can be mutated.
 */
char *qk_obsterm_str(const QkObsTerm *term);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkBitTerm
 * Get the label for a bit term.
 *
 * @param bit_term The bit term.
 *
 * @return The label as ``uint8_t``, which can be cast to ``char`` to obtain the character.
 *
 * # Example
 * ```c
 *     QkBitTerm bit_term = QkBitTerm_Y;
 *     // cast the uint8_t to char
 *     char label = qk_bitterm_label(bit_term);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``bit_term`` is not a valid ``uint8_t`` value of a ``QkBitTerm``.
 */
uint8_t qk_bitterm_label(QkBitTerm bit_term);
#endif

#if (defined(QISKIT_C_PYTHON_INTERFACE) && defined(QISKIT_WITH_CBINDINGS))
/**
 * @ingroup QkObs
 * Convert to a Python-space ``SparseObservable``.
 *
 * @param obs The C-space ``QkObs`` pointer.
 *
 * @return A Python object representing the ``SparseObservable``.
 *
 * # Safety
 *
 * Behavior is undefined if ``obs`` is not a valid, non-null pointer to a ``QkObs``.
 *
 * It is assumed that the thread currently executing this function holds the
 * Python GIL this is required to create the Python object returned by this
 * function.
 */
PyObject *qk_obs_to_python(const QkObs *obs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkNeighbors
 * Does this coupling graph represent all-to-all connectivity?
 *
 * This is represented by `neighbors` and `partition` being null pointers, so they are not valid
 * for any reads.
 *
 * @param neighbors The coupling graph.
 * @return Whether the graph represents all to all connectivity.
 *
 * # Safety
 *
 * `neighbors` must point to a valid, initialized `QkNeighbors` object.
 */
bool qk_neighbors_is_all_to_all(const QkNeighbors *neighbors);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkNeighbors
 * Initialize a `QkNeighbors` object from a `QkTarget`.
 *
 * If the target contains multi-qubit gates, they will be ignored and the connectivity will only
 * represent the two-qubit coupling constraints.  If the target represents all-to-all connectivity,
 * the function returns `true`, and the output pointers will be initialized to be null pointers, in
 * keeping with the representation of all-to-all connectivity.
 *
 * @param target The target to read the connectivity from.
 * @param neighbors The `QkNeighbors` object to initialize.
 * @return Whether the `QkTarget` represented all-to-all connectivity (`true`) or has regular
 *     connectivity (`false`).
 *
 * # Examples
 *
 * ```c
 * QkTarget *target = build_target_from_somewhere();
 * QkNeighbors neighbors;
 * if (qk_neighbors_from_target(target, &neighbors)) {
 *     printf("All-to-all connectivity on &lu qubits.\n", neighbors.num_qubits);
 *     return;
 * }
 * printf("Qubit 3 has %zu neighbors.\n", neighbors.partition[4] - neighbors.partition[3]);
 * printf("Those neighbors are: [");
 * for (size_t offset = neighbors.partition[3]; offset < neighbors.partition[4]; offset++) {
 *     printf("%u%s",
 *            neighbors.neighbors[offset],
 *            offset + 1 == neighbors.partition[4] ? "" : ", ");
 * }
 * printf("]\n");
 * qk_neighbors_clear(&neighbors);
 * ```
 *
 * # Safety
 *
 * `target` must point to a valid `QkTarget` object.  `neighbors` must be aligned and safe to write
 * to, but need not be initialized.
 */
bool qk_neighbors_from_target(const QkTarget *target, QkNeighbors *neighbors);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkNeighbors
 * Free all the allocations within the object.
 *
 * After calling this function, the `QkNeighbors` object will contain null pointers in all its
 * allocations and present as if it represents all-to-all connectivity.
 *
 * This should only be called on `QkNeighbors` objects that were initialized by
 * `qk_neighbors_from_target`.
 *
 * @param neighbors A pointer to a ``QkNeighbors`` object.
 *
 * # Safety
 *
 * `neighbors` must point to a valid, initialized `QkNeighbors` object, which must have been
 * initialized by a call to `qk_neighbors_from_target` and unaltered since then.
 */
void qk_neighbors_clear(QkNeighbors *neighbors);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the BasisTranslator transpiler pass on a circuit.
 *
 * The BasisTranslator transpiler pass translates gates to a target basis by
 * searching for a set of translations from the standard EquivalenceLibrary.
 *
 * @param circuit A pointer to the circuit to run BasisTranslator on.
 * The circuit will be mutated in-place, unless the circuit is already
 * in the target basis, in which case the circuit remains unchanged.
 * @param target The target where we will obtain basis gates from.
 * @param min_qubits The minimum number of qubits for operations in the input
 * ciruit to translate.
 *
 * # Example
 *
 * ```c
 *    #include <qiskit.h>
 *
 *    QkCircuit *circuit = qk_circuit_new(3, 0);
 *    qk_circuit_gate(circuit, QkGate_CCX, (uint32_t[3]){0, 1, 2}, NULL);
 *
 *    // Create a Target with global properties.
 *    QkTarget *target = qk_target_new(3);
 *    qk_target_add_instruction(target, qk_target_entry_new(QkGate_H));
 *    qk_target_add_instruction(target, qk_target_entry_new(QkGate_T));
 *    qk_target_add_instruction(target, qk_target_entry_new(QkGate_Tdg));
 *    qk_target_add_instruction(target, qk_target_entry_new(QkGate_CX));
 *
 *    // Run pass
 *    qk_transpiler_pass_standalone_basis_translator(circuit, target, 0);
 *
 *    // Free the circuit and target pointers once you're done
 *    qk_circuit_free(circuit);
 *    qk_target_free(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` and/or ``target`` are not valid, non-null
 * pointers to a ``QkCircuit`` or ``QkTarget``.
 */
void qk_transpiler_pass_standalone_basis_translator(QkCircuit *circuit,
                                                    const QkTarget *target,
                                                    size_t min_qubits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the CommutativeCancellation transpiler pass on a circuit.
 *
 * This pass cancels the redundant (self-adjoint) gates through commutation relations.
 *
 * @param circuit A pointer to the circuit to run CommutativeCancellation on. This circuit
 * pointer to will be updated with the modified circuit if the pass is able to remove any gates.
 * @param target This pass will attempt to accumulate all Z rotations into either
 * an RZ, P or U1 gate, depending on which is already used in the circuit. If none
 * is present in the circuit, this (optional) target argument is used as fallback to
 * decide which gate to use. If none of RZ, P or U1 are in the circuit or the target,
 * single-qubit Z rotations will not be optimized.
 * @param approximation_degree The approximation degree used when
 * analyzing commutations. Must be within ``(0, 1]``.
 * @returns The integer return code where 0 represents no error and 1 is
 * used to indicate an error was encountered during the execution of the pass.
 *
 * # Example
 *
 * ```c
 * QkCircuit *qc = qk_circuit_new(4, 0);
 * uint32_t cx_qargs[2] = {0, 1};
 * qk_circuit_gate(qc, QkGate_CX, cx_qargs, NULL);
 * qk_circuit_gate(qc, QkGate_Z, (uint32_t[]){0}, NULL);
 * qk_circuit_gate(qc, QkGate_CX, cx_qargs, NULL);
 * qk_transpiler_pass_standalone_commutative_cancellation(qc, NULL, 1.0);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` is not a valid, ``QkCircuit`` and ``QkTarget``.
 * ``QkCircuit`` is not expected to be null and behavior is undefined if it is.
 */
QkExitCode qk_transpiler_pass_standalone_commutative_cancellation(QkCircuit *circuit,
                                                                  const QkTarget *target,
                                                                  double approximation_degree);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the ConsolidateBlocks pass on a circuit.
 *
 * ConsolidateBlocks is a transpiler pass that consolidates consecutive blocks of
 * gates operating on the same qubits into a Unitary gate, to later on be
 * resynthesized, which leads to a more optimal subcircuit.
 *
 * @param circuit A pointer to the circuit to run ConsolidateBlocks on.
 * @param target A pointer to the target to run ConsolidateBlocks on.
 * @param approximation_degree A float between `[0.0, 1.0]` or a `NaN` which
 * defaults to `1.0`. Lower approximates more.
 * @param force_consolidate: Force block consolidation.
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit`` and
 * if ``target`` is not a valid pointer to a ``QkTarget``.
 */
void qk_transpiler_pass_standalone_consolidate_blocks(QkCircuit *circuit,
                                                      const QkTarget *target,
                                                      double approximation_degree,
                                                      bool force_consolidate);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the ElidePermutations transpiler pass on a circuit.
 *
 * The ElidePermutations transpiler pass removes any permutation operations from a pre-layout
 * circuit.
 *
 * This pass is intended to be run before a layout (mapping virtual qubits to physical qubits) is
 * set during the transpilation pipeline. This pass iterates over the circuit
 * and when a Swap gate is encountered it permutes the virtual qubits in
 * the circuit and removes the swap gate. This will effectively remove any
 * swap gates in the circuit prior to running layout. This optimization is
 * not valid after a layout has been set and should not be run in this case.
 *
 * @param circuit A pointer to the circuit to run ElidePermutations on. If there are changes made
 *     the object pointed to is changed in place. In case of gates being elided the original circuit's
 *     allocations are freed by this function.
 *
 * @return the layout object containing the output permutation induced by the elided gates in the
 *         circuit. If no elisions are performed this will be a null pointer and the input circuit
 *         is unchanged.
 *
 * # Example
 *
 * ```c
 *     QkCircuit *qc = qk_circuit_new(4, 0);
 *     for (uint32_t i = 0; i < qk_circuit_num_qubits(qc) - 1; i++) {
 *         uint32_t qargs[2] = {i, i + 1};
 *         for (uint32_t j = 0; j<i+1; j++) {
 *             qk_circuit_gate(qc, QkGate_CX, qargs, NULL);
 *         }
 *     }
 *     QkTranspileLayout *elide_result = qk_transpiler_pass_standalone_elide_permutations(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit``  is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkTranspileLayout *qk_transpiler_pass_standalone_elide_permutations(QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the ``CheckGateDirection`` pass on a circuit.
 *
 * The pass checks if the directions of two-qubit gates comply with the gate directions specified in a given target.
 *
 * @param circuit A pointer to the circuit on which to run the CheckGateDirection pass.
 * @param target A pointer to the target used for checking gate directions.
 *
 * @return bool - true iff the directions of all two-qubit gates in the circuit comply with the specified target constraints.
 *
 * # Example
 * ```c
 *    QkTarget *target = qk_target_new(2);
 *    uint32_t qargs[3] = {0,1};
 *
 *    QkTargetEntry *cx_entry = qk_target_entry_new(QkGate_CX);
 *    qk_target_entry_add_property(cx_entry, qargs, 2, 0.0, 0.0);
 *    qk_target_add_instruction(target, cx_entry);
 *
 *    QkCircuit *circuit = qk_circuit_new(2, 0);
 *    qk_circuit_gate(circuit, QkGate_CX, (uint32_t[]){1,0}, NULL);
 *
 *    bool direction_ok = qk_transpiler_pass_standalone_check_gate_direction(circuit, target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` are not valid, non-null pointers to ``QkCircuit`` and ``QkTarget`` objects, respectively.
 */
bool qk_transpiler_pass_standalone_check_gate_direction(const QkCircuit *circuit,
                                                        const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the ``GateDirection`` pass on a circuit.
 *
 * The GateDirection pass modifies asymmetric gates to match the hardware coupling directions.
 * This pass supports replacements for the ``cx``, ``cz``, ``ecr``, ``swap``, ``rzx``, ``rxx``, ``ryy`` and
 * ``rzz`` gates, using predefined identities.
 *
 * @param circuit A pointer to the circuit on which to run the GateDirection pass. The circuit will be modified
 *     in place by the pass.
 * @param target A pointer to the target used for checking gate directions.
 *
 * # Example
 * ```c
 *    QkTarget *target = qk_target_new(3);
 *
 *    uint32_t qargs[2] = {0,1};
 *
 *    QkTargetEntry *cx_entry = qk_target_entry_new(QkGate_CX);
 *    qk_target_entry_add_property(cx_entry, qargs, 2, 0.0, 0.0);
 *    qk_target_add_instruction(target, cx_entry);
 *
 *    QkCircuit *circuit = qk_circuit_new(3, 0);
 *    qk_circuit_gate(circuit, QkGate_CX, (uint32_t[]){1,0}, NULL);
 *
 *    qk_transpiler_pass_standalone_gate_direction(circuit, target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` are not valid, non-null pointers to ``QkCircuit`` and ``QkTarget`` objects, respectively.
 */
void qk_transpiler_pass_standalone_gate_direction(QkCircuit *circuit,
                                                  const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the InverseCancellation transpiler pass on a circuit.
 *
 * Cancels pairs of consecutive gates that are inverses of each other.
 * The cancelled gates consist of pairs of self-inverse gates:
 *    - QkGate_H
 *    - QkGate_X
 *    - QkGate_Y
 *    - QkGate_Z
 *    - QkGate_CH
 *    - QkGate_CX
 *    - QkGate_CY
 *    - QkGate_CZ
 *    - QkGate_ECR
 *    - QkGate_Swap
 *    - QkGate_CCX
 *    - QkGate_CCZ
 *    - QkGate_CSwap
 *    - QkGate_RCCX
 *    - QkGate_C3X
 *
 * and pairs of inverse gates:
 *    - (QkGate_T, QkGate_Tdg)
 *    - (QkGate_S, QkGate_Sdg)
 *    - (QkGate_SX, QkGate_SXdg)
 *    - (QkGate_CS, QkGate_CSdg)
 *
 * @param circuit A pointer to the circuit to run InverseCancellation on. If the pass is able to
 * remove any gates, the original circuit will be replaced by the circuit produced by this pass.
 *
 * # Example
 *
 * ```c
 *     QkCircuit *qc = qk_circuit_new(2, 2);
 *     uint32_t qargs[1] = {0};
 *     qk_circuit_gate(qc, QkGate_X, qargs, NULL);
 *     qk_circuit_gate(qc, QkGate_H, qargs, NULL);
 *     qk_circuit_gate(qc, QkGate_H, qargs, NULL);
 *     qk_circuit_gate(qc, QkGate_Y, qargs, NULL);
 *     qk_transpiler_pass_standalone_inverse_cancellation(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
void qk_transpiler_pass_standalone_inverse_cancellation(QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Runs the Optimize1qGatesDecomposition pass in standalone mode on a circuit.
 *
 * Optimize1qGatesDecomposition optimizes single-qubit gate sequences by re-synthesizing
 * the unitary under the constraints of the target's basis gates and error rates.
 *
 * The decision of whether to replace the original chain depends on:
 * - If the original chain was out of basis.
 * - If the original chain was in basis but the replacement has lower error rates.
 * - If the original chain is an identity (chain gets removed).
 *
 * The error is the combined multiplication of the errors of individual gates on the
 * qubit it operates on.
 *
 * @param circuit A pointer to the ``QkCircuit`` object to transform.
 * @param target A pointer to the ``QkTarget`` object or a null pointer.
 * In the case a null pointer is provided and gate errors are unknown
 * the pass will choose the sequence with the least amount of gates,
 * and will support all basis gates on its Euler basis set.
 *
 * # Example
 *
 *     QkTarget *target = qk_target_new(1);
 *     double u_errors[3] = {0., 1e-4, 1e-4};
 *     for (int idx = 0; idx < 3; idx++) {
 *         QkTargetEntry *u_entry = qk_target_entry_new(QkGate_U);
 *         uint32_t qargs[1] = {
 *             0,
 *         };
 *         qk_target_entry_add_property(u_entry, qargs, 1, NAN, u_errors[idx]);
 *         qk_target_add_instruction(target, u_entry);
 *     }
 *
 *     // Build circuit
 *     QkCircuit *circuit = qk_circuit_new(1, 0);
 *     uint32_t qubits[1] = {0};
 *     for (int iter = 0; iter < 3; iter++) {
 *         qk_circuit_gate(circuit, QkGate_H, qubits, NULL);
 *     }
 *
 *     // Run transpiler pass
 *     qk_transpiler_standalone_optimize_1q_sequences(circuit, target);
 *
 *     // Clean up
 *     qk_target_free(target);
 *     qk_circuit_free(circuit);
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit`` and
 * if ``target`` is not a valid pointer to a ``QkTarget``.
 */
void qk_transpiler_standalone_optimize_1q_sequences(QkCircuit *circuit, const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the ``RemoveDiagonalGatesBeforeMeasure`` pass on a circuit.
 *
 * Transpiler pass to remove diagonal gates (like RZ, T, Z, etc) before
 * a measurement. Including diagonal 2Q gates.
 *
 * @param circuit A pointer to the circuit to run this pass on
 *
 * # Example
 *
 * ```c
 *     QkCircuit *qc = qk_circuit_new(1, 1);
 *     qk_circuit_gate(qc, QkGate_Z, {0}, NULL);
 *     qk_circuit_measure(qc, 0, 0);
 *     qk_transpiler_pass_standalone_remove_diagonal_gates_before_measure(qc);
 *     // ...
 *     qk_circuit_free(qc);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
void qk_transpiler_pass_standalone_remove_diagonal_gates_before_measure(QkCircuit *circuit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the RemoveIdentityEquivalent transpiler pass on a circuit.
 *
 * Removes gates whose effect is close to an identity operation up to a global phase
 * and up to the specified tolerance. Parameterized gates are not considered by this pass.
 *
 * For a cutoff fidelity \f$f\f$, this pass removes gates whose average
 * gate fidelity with respect to the identity is below \f$f\f$. Concretely,
 * a gate \f$G\f$ is removed if \f$\bar F < f\f$ where
 *
 * \f[
 * bar{F} = \frac{1 + d F_{\text{process}}}{1 + d},\
 *
 * F_{\text{process}} = \frac{|\mathrm{Tr}(G)|^2}{d^2}
 * \f]
 *
 * where \f$d = 2^n\f$ is the dimension of the gate for \f$n\f$ qubits.
 *
 * @param circuit A pointer to the circuit to run RemoveIdentityEquivalent on. This circuit
 * pointed to will be updated with the modified circuit if the pass is able to remove any gates.
 * @param target The target for the RemoveIdentityEquivalent pass. If ``approximation_degree`` is set to
 * ``NAN`` the tolerance for determining whether an operation is equivalent to
 * identity will be set to the reported error rate in the target. Otherwise
 * the ``target`` is not used as the tolerance is independent of the target.
 * @param approximation_degree The degree to approximate for the equivalence check. This can be a
 * floating point value between 0 and 1, or ``NAN``. If the value is 1 this does not
 * approximate above the floating point precision. For a value < 1 this is used as a
 * scaling factor for the cutoff fidelity. If the value is ``NAN`` this approximates up
 * to the fidelity for the gate specified in ``target``.
 *
 * # Example
 *
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     uint32_t current_num_qubits = qk_target_num_qubits(target);
 *     QkTargetEntry *cx_entry = qk_target_entry_new(QkGate_CX);
 *     for (uint32_t i = 0; i < current_num_qubits - 1; i++) {
 *         uint32_t qargs[2] = {i, i + 1};
 *         double inst_error = 0.0090393 * (current_num_qubits - i);
 *         double inst_duration = 0.020039;
 *         qk_target_entry_add_property(cx_entry, qargs, 2, inst_duration, inst_error);
 *     }
 *     QkExitCode result_cx = qk_target_add_instruction(target, cx_entry);
 *     QkCircuit *qc = qk_circuit_new(4, 0);
 *     for (uint32_t i = 0; i < qk_circuit_num_qubits(qc) - 1; i++) {
 *         uint32_t qargs[2] = {i, i + 1};
 *         for (uint32_t j = 0; j<i+1; j++) {
 *             qk_circuit_gate(qc, QkGate_CX, qargs, NULL);
 *         }
 *     }
 *     uint32_t rz_qargs[1] = {1,};
 *     double rz_params[1] = {0.,};
 *     qk_circuit_gate(qc, QkGate_RZ, rz_qargs, rz_params);
 *     qk_transpiler_pass_standalone_remove_identity_equivalent(qc, target, 1.0);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` is not a valid, non-null pointer to a ``QkCircuit`` and ``QkTarget``.
 */
void qk_transpiler_pass_standalone_remove_identity_equivalent(QkCircuit *circuit,
                                                              const QkTarget *target,
                                                              double approximation_degree);
#endif

/**
 * @ingroup QkSabreLayoutOptions
 *
 * Build a default sabre layout options object. This builds a sabre layout with ``max_iterations``
 * set to 4, both ``num_swap_trials`` and ``num_random_trials`` set to 20, and the seed selected
 * by a RNG seeded from system entropy.
 *
 * @return A ``QkSabreLayoutOptions`` object with default settings.
 */
QkSabreLayoutOptions qk_sabre_layout_options_default(void);

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the SabreLayout transpiler pass on a circuit.
 *
 * The SabreLayout pass chooses a layout via an iterative bidirectional routing of the input
 * circuit.
 *
 * Starting with a random initial Layout, the algorithm does a full routing of the circuit to end up with a final_layout.
 * This final_layout is then used as the initial_layout for routing the reverse circuit. The algorithm iterates a number
 * of times until it finds an initial_layout that reduces full routing cost.
 *
 * This method exploits the reversibility of quantum circuits, and tries to include global circuit information in the
 * choice of initial_layout.
 *
 * This pass will run both layout and routing and will transform the circuit so that the layout is applied to the input
 * (meaning that the output circuit will have ancilla qubits allocated for unused qubits on the coupling map and the
 * qubits will be reordered to match the mapped physical qubits) and then routing will be applied. This is done because
 * the pass will run parallel seed trials with different random seeds for selecting the random initial layout and then
 * selecting the routed output which results in the least number of swap gates needed. This final
 * swap calculation is the same as performing a final routing, so it's more efficient to apply it
 * after computing it.
 *
 * This function is multithreaded and will launch a thread pool with threads equal to the number
 * of CPUs by default. You can tune the number of threads with the ``RAYON_NUM_THREADS``
 * environment variable. For example, setting ``RAYON_NUM_THREADS=4`` would limit the thread pool
 * to 4 threads.
 *
 * # References
 *
 * [1] Henry Zou and Matthew Treinish and Kevin Hartman and Alexander Ivrii and Jake Lishman.
 * "LightSABRE: A Lightweight and Enhanced SABRE Algorithm"
 * [arXiv:2409.08368](https://doi.org/10.48550/arXiv.2409.08368)
 *
 * [2] Li, Gushu, Yufei Ding, and Yuan Xie. "Tackling the qubit mapping problem
 * for NISQ-era quantum devices." ASPLOS 2019.
 * [arXiv:1809.02573](https://arxiv.org/pdf/1809.02573.pdf)
 *
 * @param circuit A pointer to the circuit to run SabreLayout on. The circuit
 *     is modified in place and the original circuit's allocations are freed by this function.
 * @param target A pointer to the target to run SabreLayout on
 * @param options A pointer to the options for SabreLayout
 *
 * @return The transpile layout that describes the layout and output permutation caused
 *     by the pass
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` is not a valid, non-null pointer to a ``QkCircuit`` and ``QkTarget``.
 */
QkTranspileLayout *qk_transpiler_pass_standalone_sabre_layout(QkCircuit *circuit,
                                                              const QkTarget *target,
                                                              const QkSabreLayoutOptions *options);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the Split2QUnitaries transpiler pass
 *
 * @param circuit A mutable pointer to the circuit to run Split2QUnitaries on. This will be
 *     replaced with the new circuit if any gates are optimized and the original will be freed.
 * @param requested_fidelity Allowed tolerance for splitting two-qubit unitaries and gate decompositions.
 * @param split_swaps Whether to attempt to split swap gates, resulting in a permutation of the qubits.
 *
 * @return If any swap equivalent unitaries are split this function returns a pointer to a ``TranspileLayout``
 *     that contains the permutation induced by this circuit optimization. If no swap equivalent
 *     unitaries are split this will be a null pointer.
 *
 * # Example
 *
 * ```c
 * QkCircuit *qc = qk_circuit_new(4, 0);
 * for (uint32_t i = 0; i < qk_circuit_num_qubits(qc) - 1; i++) {
 *     uint32_t qargs[2] = {i, i + 1};
 *     for (uint32_t j = 0; j<i+1; j++) {
 *         qk_circuit_gate(qc, QkGate_CX, qargs, NULL);
 *     }
 * }
 * QkTranspileLayout *result = qk_transpiler_pass_standalone_split_2q_unitaries(qc, 1e-12, true)
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` is not a valid, non-null pointer to a ``QkCircuit``.
 */
QkTranspileLayout *qk_transpiler_pass_standalone_split_2q_unitaries(QkCircuit *circuit,
                                                                    double requested_fidelity,
                                                                    bool split_swaps);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Run the UnitarySynthesis transpiler pass.
 *
 * The UnitarySynthesis transpiler pass will synthesize any UnitaryGates in the circuit into gates
 * available in the target.
 *
 * @param circuit A pointer to the circuit to run UnitarySynthesis on
 * @param target A pointer to the target to run UnitarySynthesis on
 * @param min_qubits The minimum number of qubits in the unitary to synthesize. If the unitary
 *        is less than the specified number of qubits it will not be synthesized.
 * @param approximation_degree heuristic dial used for circuit approximation
 *        (1.0=no approximation, 0.0=maximal approximation). Approximation can
 *        make the synthesized circuit cheaper at the cost of straying from
 *        the original unitary. If NAN, the target approximation is based on gate fidelities
 *        in the ``target``.
 *
 * # Example
 *
 * ```c
 *     QkTarget *target = qk_target_new(2);
 *     uint32_t current_num_qubits = qk_target_num_qubits(target);
 *     QkTargetEntry *cx_entry = qk_target_entry_new(QkGate_CX);
 *     for (uint32_t i = 0; i < current_num_qubits - 1; i++) {
 *         uint32_t qargs[2] = {i, i + 1};
 *         double inst_error = 0.0090393 * (current_num_qubits - i);
 *         double inst_duration = 0.020039;
 *         qk_target_entry_add_property(cx_entry, qargs, 2, inst_duration, inst_error);
 *     }
 *     QkExitCode result_cx = qk_target_add_instruction(target, cx_entry);
 *     QkCircuit *qc = qk_circuit_new(2, 0);
 *     QkComplex64 c0 = {0., 0.};
 *     QkComplex64 c1 = {1., 0.};
 *     QkComplex64 unitary[16] = {c1, c0, c0, c0,  // row 0
 *                                c0, c1, c0, c0,  // row 1
 *                                c0, c0, c1, c0,  // row 2
 *                                c0, c0, c0, c1}; // row 3
 *     uint32_t qargs[2] = {0, 1};
 *     qk_circuit_unitary(qc, unitary, qargs, 2, false);
 *     qk_transpiler_pass_standalone_unitary_synthesis(qc, target, 0, 1.0);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` is not a valid, non-null pointer to a ``QkCircuit`` and ``QkTarget``.
 */
void qk_transpiler_pass_standalone_unitary_synthesis(QkCircuit *circuit,
                                                     const QkTarget *target,
                                                     size_t min_qubits,
                                                     double approximation_degree);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutResult
 * Check whether a result was found.
 *
 * A ``true`` value includes the situation where the configuration specified to try the "trivial"
 * layout and it was found to be the best (and consequently no qubit relabelling is necessary,
 * other than ancilla expansion if appropriate).  See ``qk_vf2_layout_result_has_improvement`` to
 * distinguish whether an explicit remapping is stored.
 *
 * @param layout a pointer to the layout
 *
 * @returns ``true`` if the VF2-based layout pass found any match.
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkVF2LayoutResult``.
 */
bool qk_vf2_layout_result_has_match(const QkVF2LayoutResult *layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutResult
 * Check whether the result is an improvement to the trivial layout.
 *
 * @param layout a pointer to the layout
 *
 * @returns ``true`` if the VF2-based layout pass found an improved match.
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkVF2LayoutResult``.
 */
bool qk_vf2_layout_result_has_improvement(const QkVF2LayoutResult *layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutResult
 * Get the physical qubit for a given virtual qubit
 *
 * @param layout a pointer to the layout
 * @param qubit the virtual qubit to get the physical qubit of
 *
 * @returns The physical qubit mapped to by the specified virtual qubit
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkVF2LayoutResult`` containing a result, or if the qubit is out of range for the initial
 * circuit.
 */
uint32_t qk_vf2_layout_result_map_virtual_qubit(const QkVF2LayoutResult *layout, uint32_t qubit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutResult
 * Free a ``QkVF2LayoutResult`` object
 *
 * @param layout a pointer to the layout to free
 *
 * # Example
 *
 *     QkCircuit *qc = qk_circuit_new(1, 0);
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a ``QkVF2Layout``.
 */
void qk_vf2_layout_result_free(QkVF2LayoutResult *layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Create a new configuration for the VF2 passes that runs everything completely unbounded.
 *
 * Call ``qk_vf2_layout_configuration_free`` with the return value to free the memory when done.
 *
 * @return A pointer to the configuration.
 */
QkVF2LayoutConfiguration *qk_vf2_layout_configuration_new(void);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Free a `QkVf2LayoutConfiguration` object.
 *
 * @param config A pointer to the configuration.
 *
 * # Safety
 *
 * Behavior is undefined if ``config`` is a non-null pointer, but does not point to a valid,
 * aligned `QkVF2LayoutConfiguration` object.
 */
void qk_vf2_layout_configuration_free(QkVF2LayoutConfiguration *config);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Limit the numbers of times that the VF2 algorithm will attempt to extend its mapping before and
 * after it finds the first match.
 *
 * The VF2 algorithm keeps track of the number of steps it has taken, and terminates when it
 * reaches the limit.  After the first match is found, the limit swaps from the "before" limit to
 * the "after" limit without resetting the number of steps taken.
 *
 * @param config The configuration to update.
 * @param before The number of attempts to allow before the first match is found.  Set to a
 *     negative number to have no bound.
 * @param after The number of attempts to allow after the first match (if any) is found.  Set to a
 *     negative number to have no bound.
 *
 * # Safety
 *
 * Behavior is undefined if `config` is not a valid, aligned, non-null pointer to a
 * `QkVF2LayoutConfiguration`.
 */
void qk_vf2_layout_configuration_set_call_limit(QkVF2LayoutConfiguration *config,
                                                int64_t before,
                                                int64_t after);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Limit the runtime of the VF2 search.
 *
 * This is not a hard limit; it is only checked when an improved layout is encountered.  Using this
 * option also makes the pass non-deterministic. It is generally recommended to use
 * `qk_vf2_layout_configuration_set_call_limit` instead.
 *
 * @param config The configuration to update.
 * @param limit The time in seconds to allow.  Set to a non-positive value to run with no limit.
 *
 * # Safety
 *
 * Behavior is undefined if `config` is not a valid, aligned, non-null pointer to a
 * `QkVF2LayoutConfiguration`.
 */
void qk_vf2_layout_configuration_set_time_limit(QkVF2LayoutConfiguration *config, double limit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Limit the total number of complete improvements found.
 *
 * Since the VF2 search tree is pruned on-the-fly based on scoring in the `QkTarget`, this limit
 * is not especially powerful.  See `qk_vf2_layout_configuration_set_call_limit` for a tighter
 * bound.
 *
 * @param config The configuration to update.
 * @param limit The number of complete layouts to allow before terminating.  Set to 0 to run
 *     unbounded.
 *
 * # Safety
 *
 * Behavior is undefined if `config` is not a valid, aligned, non-null pointer to a
 * `QkVF2LayoutConfiguration`.
 */
void qk_vf2_layout_configuration_set_max_trials(QkVF2LayoutConfiguration *config, uint64_t limit);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Activate node shuffling of the input graphs with a given seed.
 *
 * This effectively drives a modification of the matching order of VF2, which in theory means that
 * the space of a bounded search is not biased based on the node indices.  In practice, Qiskit uses
 * the VF2++ ordering improvements when running in "average" mode (corresponding to initial layout
 * search), and starts from the identity mapping in "exact" made.  Both of these ordering
 * heuristics are typically far more likely to find results for the given problem than
 * randomization.
 *
 * If this function was not called, no node shuffling takes place.
 *
 * @param config The configuration to update.
 * @param seed The seed to use for the activated shuffling.
 *
 * # Safety
 *
 * Behavior is undefined if `config` is not a valid, aligned, non-null pointer to a
 * `QkVF2LayoutConfiguration`.
 */
void qk_vf2_layout_configuration_set_shuffle_seed(QkVF2LayoutConfiguration *config, uint64_t seed);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkVF2LayoutConfiguration
 * Whether to eagerly score the initial "trivial" layout of the interaction graph.
 *
 * You typically want to set this ``true`` if you are using the VF2 passes to improve a circuit
 * that is already lowered to hardware, in order to set a baseline for the score-based pruning.  If
 * not, you can leave this as ``false`` (the default), to avoid a calculation that likely will not
 * have any impact.
 *
 * @param config The configuration to update.
 * @param score_initial Whether to eagerly score the initial trivial layout.
 *
 * # Safety
 *
 * Behavior is undefined if `config` is not a valid, aligned, non-null pointer to a
 * `VF2LayoutConfiguration`.
 */
void qk_vf2_layout_configuration_set_score_initial(QkVF2LayoutConfiguration *config,
                                                   bool score_initial);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Use the VF2 algorithm to choose a layout (if possible) for the input circuit, using a
 * noise-aware scoring heuristic based only on hardware error rates, and not the specific gates in
 * the circuit.
 *
 * This function corresponds to the Python-space ``VF2Layout`` pass.
 *
 * This function is suitable for use on circuits that have not yet been fully lowered to hardware.
 * If your circuit has already been completely lowered to hardware and you are looking to _improve_
 * the layout for an exact interaction graph, use ``qk_transpile_pass_standalone_vf2_layout_exact``
 * instead.
 *
 * If this pass finds a solution that means there is a "perfect layout" and that no
 * further swap mapping or routing is needed. However, there is not always a possible
 * solution, or a solution might exist but it is not found within the limits specified
 * when the pass is called.
 *
 * @param circuit A pointer to the circuit to run VF2Layout on
 * @param target A pointer to the target to run the VF2Layout pass on
 * @param config A pointer to the ``QkVF2LayoutConfiguration`` configuration structure.  If this
 *     pointer is null, the pass defaults are used.
 * @param strict_direction If ``true``, the pass will consider the edge direction in the
 *     connectivity described in the ``target``. Typically, setting this to ``false``
 *     is desireable as the error heuristic is already very approximate, and two-qubit gates can
 *     almost invariably be synthesised to "flip" direction using only local one-qubit gates and
 *     the native-direction two-qubit gate.
 *
 * @return A pointer to a result object that contains the results of the pass. This object is heap
 *     allocated and will need to be freed with the `qk_vf2_layout_result_free` function.
 *
 * # Example
 *
 * ```c
 * QkTarget *target = qk_target_new(5);
 * uint32_t current_num_qubits = qk_target_num_qubits(target);
 * QkTargetEntry *cx_entry = qk_target_entry_new(QkGate_CX);
 * for (uint32_t i = 0; i < current_num_qubits - 1; i++) {
 *     uint32_t qargs[2] = {i, i + 1};
 *     double inst_error = 0.0090393 * (current_num_qubits - i);
 *     double inst_duration = 0.020039;
 *     qk_target_entry_add_property(cx_entry, qargs, 2, inst_duration, inst_error);
 * }
 * QkExitCode result_cx = qk_target_add_instruction(target, cx_entry);
 * QkCircuit *qc = qk_circuit_new(4, 0);
 * for (uint32_t i = 0; i < qk_circuit_num_qubits(qc) - 1; i++) {
 *     uint32_t qargs[2] = {i, i + 1};
 *     for (uint32_t j = 0; j<i+1; j++) {
 *         qk_circuit_gate(qc, QkGate_CX, qargs, NULL);
 *     }
 * }
 * QkVF2LayoutConfiguration *config = qk_vf2_layout_configuration_new();
 * qk_vf2_layout_configuration_set_call_limit(config, 10000, 10000);
 * QkVF2LayoutResult *layout_result = qk_transpiler_pass_standalone_vf2_layout_average(qc, target, config, false);
 * qk_vf2_layout_result_free(layout_result);
 * qk_vf2_layout_configuration_free(config);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` is not a valid, non-null pointer to a
 * `QkCircuit` and `QkTarget`.  Behavior is undefined if ``config`` is a non-null pointer that
 * does not point to a valid `QkVF2LayoutConfiguration` object (but a null pointer is fine).
 */
QkVF2LayoutResult *qk_transpiler_pass_standalone_vf2_layout_average(const QkCircuit *circuit,
                                                                    const QkTarget *target,
                                                                    const QkVF2LayoutConfiguration *config,
                                                                    bool strict_direction);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Use the VF2 algorithm to choose a layout (if possible) for the input circuit, using a
 * noise-aware scoring heuristic that requires the result is already fully compatible with
 * the hardware.
 *
 * This function corresponds to the Python-space ``VF2PostLayout`` pass with
 * ``strict_direction=True``.
 *
 * This function is suitable for use on circuits that have already been fully lowered to hardware,
 * and you are now looking to see if a qubit permutation can lead to better estimated error rates.
 * If your circuit is still in terms of non-hardware-supported operations, use
 * `qk_transpile_pass_standalone_vf2_layout_average` instead.
 *
 * Typically, you call this pass after layout, routing, translation to a native basis set and
 * optimization, such that the input circuit is already executable on hardware with the qubit
 * indices referring to physical qubits.  The pass will return a result indicating one of:
 *
 * * there is a better choice of initial virtual-to-physical qubit mapping than what the circuit is
 *   currently using.
 * * the current choice of physical qubits is the best the pass found within its call limit.
 * * there is no valid choice of virtual-to-physical mapping that results in an executable circuit
 *   (or at least, the pass failed to find one within its specified limits).
 *
 * In both of the first two cases, `qk_vf2_layout_has_match` will return ``true``.  In only the
 * first case, `qk_vf2_layout_has_improvement` will return ``true``.
 *
 * @param circuit A pointer to the circuit to run the layout search on.
 * @param target A pointer to the target representing the QPU.
 * @param config A pointer to the `QkVF2LayoutConfiguration` configuration structure.  If this
 *     pointer is null, the pass defaults are used.
 *
 * @return A pointer to a result object that contains the results of the pass. This object is heap
 *     allocated and will need to be freed with the `qk_vf2_layout_result_free` function.
 *
 * # Example
 *
 * ```c
 * QkTarget *target = qk_target_new(5)
 * QkTargetEntry *cx_entry = qk_target_entry_new(QkGate_CX);
 * for (uint32_t i = 0; i < current_num_qubits - 1; i++) {
 *     uint32_t qargs[2] = {i, i + 1};
 *     double inst_error = 0.0090393 * (current_num_qubits - i);
 *     double inst_duration = 0.020039;
 *     qk_target_entry_add_property(cx_entry, qargs, 2, inst_duration, inst_error);
 * }
 * QkExitCode result_cx = qk_target_add_instruction(target, cx_entry);
 * QkCircuit *qc = qk_circuit_new(4, 0);
 * for (uint32_t i = 0; i < qk_circuit_num_qubits(qc) - 1; i++) {
 *     uint32_t qargs[2] = {i, i + 1};
 *     for (uint32_t j = 0; j<i+1; j++) {
 *         qk_circuit_gate(qc, QkGate_CX, qargs, NULL);
 *     }
 * }
 * QkVF2LayoutConfiguration *config = qk_vf2_layout_configuration_new();
 * qk_vf2_layout_configuration_call_limit(config, 10000, 10000);
 * QkVF2LayoutResult *layout_result = qk_transpiler_pass_standalone_vf2_layout_exact(qc, target, config);
 * qk_vf2_layout_result_free(layout_result);
 * qk_vf2_layout_configuration_free(config);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit`` or ``target`` is not a valid, non-null pointer to a
 * `QkCircuit` and `QkTarget`.  Behavior is undefined if ``config`` is a non-null pointer that
 * does not point to a valid `QkVF2LayoutConfiguration` object (but a null pointer is fine).
 */
QkVF2LayoutResult *qk_transpiler_pass_standalone_vf2_layout_exact(const QkCircuit *circuit,
                                                                  const QkTarget *target,
                                                                  const QkVF2LayoutConfiguration *config);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspilerPasses
 * Deprecated version of `qk_transpiler_pass_standalone_vf2_layout_average`.
 *
 * This legacy interface does not use `QkVf2LayoutConfiguration`, and has a name that is not clear
 * about how it handles the error heuristic (it averages over all gates in the `QkTarget` for a
 * given qubit or link).
 *
 * \qk_deprecated{2.3.0|Replaced by :c:func:`qk_transpiler_pass_standalone_vf2_layout_average`.}
 *
 * @param circuit As in `qk_transpiler_pass_standalone_vf2_layout_average`.
 * @param target As in `qk_transpiler_pass_standalone_vf2_layout_average`.
 * @param strict_direction As in `qk_transpiler_pass_standalone_vf2_layout_average`.
 * @param call_limit As in `qk_vf2_layout_configuration_set_call_limit`, but the same value is used
 *     for both `before` and `after`.
 * @param time_limit As in `qk_vf2_layout_configuration_set_time_limit`.
 * @param max_trials As in `qk_vf2_layout_configuration_set_max_trials`.
 *
 * @return As in `qk_transpiler_pass_standalone_vf2_layout_average`.
 *
 * # Safety
 *
 * The safety requirements of `qk_transpiler_pass_standalone_vf2_layout_average` must be respected
 * for `circuit` and `target`.
 */
Qk_DEPRECATED_FN_NOTE("use `qk_transpiler_pass_standalone_vf2_layout_average` instead")
QkVF2LayoutResult *qk_transpiler_pass_standalone_vf2_layout(const QkCircuit *circuit,
                                                            const QkTarget *target,
                                                            bool strict_direction,
                                                            int64_t call_limit,
                                                            double time_limit,
                                                            int64_t max_trials);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Construct a new ``QkTarget`` with the given number of qubits.
 * The number of qubits is bound to change if an instruction is added with properties
 * that apply to a collection of qargs in which any index is higher than the specified
 * number of qubits
 *
 * @param num_qubits The number of qubits the ``QkTarget`` will explicitly support.
 *
 * @return A pointer to the new ``QkTarget``
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 * ```
 *
 */
QkTarget *qk_target_new(uint32_t num_qubits);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the number of qubits of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The number of qubits this target can use.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     uint32_t num_qubits = qk_target_num_qubits(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
uint32_t qk_target_num_qubits(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the dt value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The dt value of this ``QkTarget`` or ``NAN`` if not assigned.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     qk_target_set_dt(target, 10e-9);
 *     double dt = qk_target_dt(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
double qk_target_dt(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the granularity value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The ``granularity`` value of this ``QkTarget``.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 1
 *     uint32_t granularity = qk_target_granularity(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
uint32_t qk_target_granularity(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the ``min_length`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The ``min_length`` value of this ``QkTarget``.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 1
 *     size_t min_length = qk_target_min_length(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
uint32_t qk_target_min_length(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the ``pulse_alignment`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The ``pulse_alignment`` value of this ``QkTarget``.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 1
 *     uint32_t pulse_alignment = qk_target_pulse_alignment(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
uint32_t qk_target_pulse_alignment(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the ``acquire_alignment`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The ``acquire_alignment`` value of this ``QkTarget``.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 0
 *     uint32_t acquire_alignment = qk_target_pulse_alignment(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
uint32_t qk_target_acquire_alignment(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Sets the dt value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param dt The ``dt`` value for the system time resolution of input.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 *
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     double dt = qk_target_set_dt(target, 10e-9);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
QkExitCode qk_target_set_dt(QkTarget *target, double dt);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Sets the ``granularity`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param granularity The value for the minimum pulse gate resolution in
 *     units of ``dt``.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 1
 *     qk_target_set_granularity(target, 2);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
QkExitCode qk_target_set_granularity(QkTarget *target, uint32_t granularity);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Sets the ``min_length`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param min_length The minimum pulse gate length value in units of ``dt``.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 *
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 1
 *     qk_target_set_min_length(target, 3);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
QkExitCode qk_target_set_min_length(QkTarget *target, uint32_t min_length);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the ``pulse_alignment`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param pulse_alignment value representing a time resolution of gate.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 1
 *     qk_target_set_pulse_alignment(target, 4);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
QkExitCode qk_target_set_pulse_alignment(QkTarget *target, uint32_t pulse_alignment);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Sets the ``acquire_alignment`` value of this ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param acquire_alignment value representing a time resolution of measure instruction
 *     starting time.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     // The value defaults to 0
 *     qk_target_set_acquire_alignment(target, 5);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
QkExitCode qk_target_set_acquire_alignment(QkTarget *target, uint32_t acquire_alignment);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Creates a copy of the ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget`` to copy.
 *
 * @return A pointer to the new copy of the ``QkTarget``.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 *     QkExitCode result = qk_target_add_instruction(target, entry);
 *
 *     QkTarget *copied = qk_target_copy(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
QkTarget *qk_target_copy(QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Free the ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget`` to free.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     qk_target_free(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
void qk_target_free(QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Creates an entry to the ``QkTarget`` based on a ``QkGate`` instance.
 *
 * @param operation The ``QkGate`` whose properties this target entry defines. If the ``QkGate``
 * takes parameters (which can be checked with ``qk_gate_num_params``) it will be added as a
 * an instruction on the target which accepts any parameter value. If the gate only accepts a
 * fixed parameter value you can use ``qk_target_entry_new_fixed`` instead.
 *
 * @return A pointer to the new ``QkTargetEntry``.
 *
 * # Example
 * ```c
 *     QkTargetEntry *had_entry = qk_target_entry_new(QkGate_H);
 * ```
 */
QkTargetEntry *qk_target_entry_new(QkGate operation);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Creates a new entry for adding a measurement instruction to a ``QkTarget``.
 *
 * @return A pointer to the new ``QkTargetEntry`` for a measurement instruction.
 *
 * # Example
 * ```c
 *     QkTargetEntry *entry = qk_target_entry_new_measure();
 *     // Add fixed duration and error rates from qubits at index 0 to 4.
 *     for (uint32_t i = 0; i < 5; i++) {
 *         // Measure is a single qubit instruction
 *         uint32_t qargs[1] = {i};
 *         qk_target_entry_add_property(entry, qargs, 1, 1.928e-10, 7.9829e-11);
 *     }
 *
 *     // Add the entry to a target with 5 qubits
 *     QkTarget *measure_target = qk_target_new(5);
 *     qk_target_add_instruction(measure_target, entry);
 * ```
 */
QkTargetEntry *qk_target_entry_new_measure(void);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Creates a new entry for adding a reset instruction to a ``QkTarget``.
 *
 * @return A pointer to the new ``QkTargetEntry`` for a reset instruction.
 *
 * # Example
 * ```c
 *     QkTargetEntry *entry = qk_target_entry_new_reset();
 *     // Add fixed duration and error rates from qubits at index 0 to 2.
 *     for (uint32_t i = 0; i < 3; i++) {
 *         // Reset is a single qubit instruction
 *         uint32_t qargs[1] = {i};
 *         qk_target_entry_add_property(entry, qargs, 1, 1.2e-11, 5.9e-13);
 *     }
 *
 *     // Add the entry to a target with 3 qubits
 *     QkTarget *reset_target = qk_target_new(3);
 *     qk_target_add_instruction(reset_target, entry);
 * ```
 */
QkTargetEntry *qk_target_entry_new_reset(void);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Creates an entry in the ``QkTarget`` based on a ``QkGate`` instance with
 * no parameters.
 *
 * @note Adding a ``QkGate`` with regular parameters is not currently supported.
 *
 * @param operation The ``QkGate`` whose properties this target entry defines.
 * @param params A pointer to the parameters that the instruction is calibrated for.
 * @param name An optional name for the instruction in the target. If a null pointer is
 * provided, the entry will use the operation's default name.
 *
 * @return A pointer to the new ``QkTargetEntry``.
 *
 * # Example
 * ```c
 *     double crx_params[1] = {3.14};
 *     QkTargetEntry *entry = qk_target_entry_new_fixed(QkGate_CRX, crx_params, "crx_fixed")";
 * ```
 *
 * # Safety
 *
 * The ``params`` type is expected to be a pointer to an array of ``double`` where the length
 * matches the expectations of the ``QkGate``. If the array is insufficiently long the
 * behavior of this function is undefined as this will read outside the bounds of the array.
 * It can be a null pointer if there are no params for a given gate. You can check
 * ``qk_gate_num_params`` to determine how many qubits are required for a given gate.
 *
 * The ``name`` pointer is expected to be either a C string comprising of valid UTF-8 characters
 * or a null pointer.
 */
QkTargetEntry *qk_target_entry_new_fixed(QkGate operation, double *params, const char *name);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Retrieves the number of properties stored in the target entry.
 *
 * @param entry The pointer to the mapping object.
 *
 * @return The number of properties in the ``QkTargetEntry``.
 *
 * # Example
 * ```c
 *     // Create an entry for an H gate
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_H);
 *     size_t props_size = qk_target_entry_num_properties(entry);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``entry`` is not a valid,
 * non-null pointer to a ``QkTargetEntry`` object.
 */
size_t qk_target_entry_num_properties(const QkTargetEntry *entry);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Frees the entry.
 *
 * @note An entry pointer will be freed when added to a ``QkTarget`` via
 * ``qk_target_add_instruction``, this function is only meant to be used
 * alternatively if an entry is never added to a ``QkTarget`` instance.
 *
 * @param entry The pointer to the mapping object to be freed.
 *
 * # Example
 * ```c
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_H);
 *     qk_target_entry_free(entry);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``entry`` is not a valid,
 * non-null pointer to a ``QkTargetEntry`` object.
 */
void qk_target_entry_free(QkTargetEntry *entry);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Adds an instruction property instance based on its assigned qargs.
 *
 * @param entry The pointer to the entry object.
 * @param qargs A pointer to the array of ``uint32_t`` qubit indices to add the
 *     gate on, can be a null pointer to check for global properties.
 * @param num_qubits The length of the qargs array.
 * @param duration The instruction's duration in seconds on the specific set of
 *     qubits.
 * @param error The instruction's average error rate on the specific set of qubits.
 *
 * @return An exit code.
 *
 * # Example
 * ```c
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``entry`` is not a valid, non-null pointer
 * to a ``QkTargetEntry`` object.
 */
QkExitCode qk_target_entry_add_property(QkTargetEntry *entry,
                                        uint32_t *qargs,
                                        uint32_t num_qubits,
                                        double duration,
                                        double error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTargetEntry
 * Sets a custom name to the target entry.
 *
 * @param entry The pointer to the entry object.
 * @param name The name to be set for the target entry.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 * ```c
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     qk_target_entry_set_name(entry, "cx_gate");
 * ```
 *
 * # Safety
 *
 * The behavior is undefined if ``entry`` is not a valid, non-null pointer
 * to a ``QkTargetEntry`` object.
 * The ``name`` pointer is expected to be either a C string comprising
 * of valid UTF-8 characters or a null pointer.
 */
QkExitCode qk_target_entry_set_name(QkTargetEntry *entry, const char *name);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Adds a gate to the ``QkTarget`` through a ``QkTargetEntry``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param target_entry A pointer to the ``QkTargetEntry``. The pointer
 * gets freed when added to the ``QkTarget``.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 *     QkExitCode result = qk_target_add_instruction(target, entry);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 *
 * Behavior is undefined if ``entry`` is not a valid, non-null pointer to a ``QkTargetEntry``.
 */
QkExitCode qk_target_add_instruction(QkTarget *target, QkTargetEntry *target_entry);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Modifies the properties of a gate in the ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param instruction The instruction to modify.
 * @param qargs The pointer to the array of ``uint32_t`` values to use as
 * qargs. Can be ``NULL`` if global.
 * @param num_qubits The number of qubits of the instruction..
 * @param duration The instruction's duration in seconds on the specific set of
 *     qubits.
 * @param error The instruction's average error rate on the specific set of qubits.
 *
 * @return ``QkExitCode`` specifying if the operation was successful.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     double params[1] = {3.1415};
 *     QkTargetEntry *entry = qk_target_entry_new_fixed(QkGate_CRX, params, "crx_pi");
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 *     qk_target_add_instruction(target, entry);
 *
 *     qk_target_update_property(target, QkGate_CRX, qargs, 2, 0.0012, 1.1);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 *
 * The ``qargs`` type is expected to be a pointer to an array of ``uint32_t`` where the length
 * matches is specified by ``num_qubits`` and has to match the expectation of the gate. If the
 * array is insufficiently long the behavior of this function is undefined as this will read
 * outside the bounds of the array. It can be a null pointer if there are no qubits for
 * a given gate. You can check ``qk_gate_num_qubits`` to determine how many qubits are required
 * for a given gate.
 */
QkExitCode qk_target_update_property(QkTarget *target,
                                     QkGate instruction,
                                     uint32_t *qargs,
                                     uint32_t num_qubits,
                                     double duration,
                                     double error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Returns the number of instructions tracked by a ``QkTarget``.
 *
 * @param target A pointer to the ``QkTarget``.
 *
 * @return The length of the target.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *target_entry = qk_target_entry_new(QkGate_H);
 *     qk_target_add_instruction(target, target_entry);
 *
 *     size_t num_instructions = qk_target_num_instructions(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
size_t qk_target_num_instructions(const QkTarget *target);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Checks if the provided instruction and its qargs are supported by this
 * ``Target``.
 *
 * @param target A pointer to the ``Target``.
 * @param operation_name The instruction name to check for.
 * @param qargs The pointer to the array of ``uint32_t`` values to use as
 * qargs. Can be ``NULL`` if global.
 * @param params A pointer to an array of pointers of ``QkParam`` objects as parameters
 * to check. Can be ``NULL`` if no parameters are present.
 *
 * @return Whether the instruction is supported or not.
 *
 * # Example
 * ```c
 *     // Create a mock target with only a global crx entry
 *     // and 3.14 as its rotation parameter.
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *crx_entry = qk_target_entry_new_fixed(QkGate_CRX, (double[]){3.14});
 *     qk_target_entry_add_property(crx_entry, NULL, 0, 0.0, 0.1);
 *     qk_target_add_instruction(target, crx_entry);
 *
 *     // Check if target is compatible with a "crx" gate
 *     // at [0, 1] with 3.14 rotation.
 *     QkParam *params[1] = {qk_param_from_double(3.14)};
 *     qk_target_instruction_supported(target, "crx", (uint32_t []){0, 1}, params);
 *
 *     // Free the pointers
 *     qk_param_free(params[0]);
 *     qk_target_free(target);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``target`` is not a valid, non-null pointer to a ``QkTarget``.
 *
 * The ``qargs`` argument is expected to be a pointer to an array of ``u32int_t`` where the length
 * matches the expectation of the gate. If the array is insufficiently long the behavior of this
 * function is undefined as this will read outside the bounds of the array. It can be a null
 * pointer if there are no qubits for a given gate. You can check `qk_gate_num_qubits` to
 * determine how many qubits are required for a given gate.
 *
 * The ``params`` argument is expected to be an array of ``QkParam`` where the length matches the
 * expectation of the operation in question. If the array is insufficiently long, the behavior will
 * be undefined just as mentioned above for the ``qargs`` argument. You can always check ``qk_gate_num_params``
 * in the case of a ``QkGate``.
 */
bool qk_target_instruction_supported(const QkTarget *target,
                                     const char *operation_name,
                                     const uint32_t *qargs,
                                     QkParam **params);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Return the index at which an operation is located based on its name.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param name The name to get the index of.
 *
 * @return the index in which the operation is the maximum value of ``size_t``
 *     in the case it is not in the Target.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *target_entry = qk_target_entry_new(QkGate_H);
 *     qk_target_add_instruction(target, target_entry);
 *
 *     size_t op_idx = qk_target_op_index(target, "h");
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 * Behavior is undefined if ``name`` is not a pointer to a valid null-terminated string.
 */
size_t qk_target_op_index(const QkTarget *target, const char *name);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Return the name of the operation stored at that index in the ``QkTarget`` instance's
 * gate map.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param index The index at which the gate is stored.
 *
 * @return The name of the operation associated with the provided index.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *target_entry = qk_target_entry_new(QkGate_H);
 *     qk_target_add_instruction(target, target_entry);
 *
 *     char *op_name = qk_target_op_name(target, 0);
 *     // Free after use
 *     qk_str_free(op_name);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
char *qk_target_op_name(const QkTarget *target, size_t index);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Return the number of properties defined for the specified operation in
 * the ``QkTarget`` instance, a.k.a. the length of the property map. Panics
 * if the operation index is not present.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param index The index in which the gate is stored.
 *
 * @return The number of properties specified for the operation associated with
 * that index.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *     QkTargetEntry *target_entry = qk_target_entry_new(QkGate_H);
 *     qk_target_add_instruction(target, target_entry);
 *
 *     size_t num_props = qk_target_op_num_properties(target, 0);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
size_t qk_target_op_num_properties(const QkTarget *target, size_t index);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Retrieve the index at which some qargs are stored. Returns ``SIZE_MAX``
 * if not found.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param op_idx The index at which the operation is stored.
 * @param qargs A pointer to the array of ``uint32_t`` qubit indices to
 *     check for, can be a null pointer to check for global properties.
 *
 * @return The index of the qargs associated with the instruction at the
 * specified `op_idx` index or ``SIZE_MAX`` if the qargs are not present.
 *
 * # Example
 * ```c
 * QkTarget *target = qk_target_new(5);
 *
 * QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 * uint32_t qargs[2] = {0, 1};
 * qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 * qk_target_add_instruction(target, entry);
 *
 * size_t idx_0_1 = qk_target_op_qargs_index(target, 0, qargs);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``QkTarget`` is not a valid, non-null pointer to a ``QkTarget``.
 */
size_t qk_target_op_qargs_index(const QkTarget *target, size_t op_idx, const uint32_t *qargs);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Retrieve the qargs for the operation by index.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param op_idx The index at which the gate is stored.
 * @param qarg_idx The index at which the qargs are stored.
 * @param qargs_out An out pointer to an array qubits. If ``op_idx`` refers to a a global
 *     operation, a null pointer will be written.  The written pointer is borrowed from the
 *     target and must not be freed.  A zero-qargs instruction will write out a non-null pointer,
 *     though one that is invalid for reads.
 * @param qargs_len An out pointer to the length of the qargs in `qargs_out`. If the index is
 *     global, the written length is not defined.
 *
 * Panics if any of the indices are out of range.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 *     qk_target_add_instruction(target, entry);
 *
 *     uint32_t *qargs_retrieved;
 *     uint32_t qargs_length;
 *     qk_target_op_qargs(target, 0, 0, &qargs_retrieved, &qargs_length);
 *     if (qargs_retrieved) {
 *         // We should enter this branch.
 *         printf("Number of qargs: %lu\n", qargs_length);
 *     } else {
 *         printf("Qargs are global\n");
 *     }
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``target`` is not a valid, non-null pointer to a ``QkTarget``.
 * Behavior is undefined if each `qargs_out` or `qargs_len` are not aligned and writeable for a
 * single value of the correct type.
 */
void qk_target_op_qargs(const QkTarget *target,
                        size_t op_idx,
                        size_t qarg_idx,
                        uint32_t **qargs_out,
                        uint32_t *qargs_len);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Retrieve the qargs for the operation stored in its respective indices.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param op_idx The index in which the gate is stored.
 * @param qarg_idx The index in which the qargs are stored.
 * @param inst_props A pointer to write out the ``QkInstructionProperties`` instance.
 *
 * Panics if any of the indices are out of range.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 *     qk_target_add_instruction(target, entry);
 *
 *     QkInstructionProperties inst_props;
 *     qk_target_op_props(target, 0, 0, &inst_props);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``target`` is not a valid, non-null pointer to a ``QkTarget``.
 * Behavior is undefined if ``inst_props`` does not point to an address of the correct size to
 * store ``QkInstructionProperties`` in.
 */
void qk_target_op_props(const QkTarget *target,
                        size_t op_idx,
                        size_t qarg_idx,
                        QkInstructionProperties *inst_props);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Retrieves information about an operation in the Target via index.
 * If the index is not present, this function will panic. You can check
 * the ``QkTarget`` total number of instructions using ``qk_target_num_instructions``.
 *
 * @param target A pointer to the ``QkTarget``.
 * @param index The index in which the gate is stored.
 * @param out_op A pointer to the space where the ``QkTargetOp``
 *     will be stored.
 *
 * # Example
 * ```c
 * QkTarget *target = qk_target_new(5);
 *
 * QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 * uint32_t qargs[2] = {0, 1};
 * qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 * qk_target_add_instruction(target, entry);
 *
 * QkTargetOp op;
 * qk_target_op_get(target, 0, &op);
 *
 * // Clean up after you're done
 * qk_target_op_clear(&op);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if ``target`` is not a valid, non-null pointer to a ``QkTarget``.
 * Behavior is undefined if ``out_op`` does not point to an address of the correct size to
 * store ``QkTargetOp`` in.
 */
void qk_target_op_get(const QkTarget *target, size_t index, QkTargetOp *out_op);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Tries to retrieve a ``QkGate`` based on the operation stored in an index.
 * The user is responsible for checking whether this operation is a gate in the ``QkTarget``
 * via using ``qk_target_op_get``. If not, this function will panic.
 *
 * @param target A pointer to the ``Target`` instance.
 * @param index The index at which the operation is located.
 *
 * @return The ``QkGate`` enum in said index.
 *
 * # Example
 * ```c
 *     QkTarget *target = qk_target_new(5);
 *
 *     QkTargetEntry *entry = qk_target_entry_new(QkGate_CX);
 *     uint32_t qargs[2] = {0, 1};
 *     qk_target_entry_add_property(entry, qargs, 2, 0.0, 0.1);
 *     qk_target_add_instruction(target, entry);
 *
 *     QkTargetOp op;
 *     qk_target_op_get(target, 0, &op);
 *
 *     // Check if the operation is a gate;
 *     if (op.op_type == QkOperationKind_Gate) {
 *         QkGate gate = qk_target_op_gate(target, 0);
 *         // Do something
 *     }
 *
 *     // Clean up after you're done.
 *     qk_target_op_clear(&op);
 * ```
 *
 * # Safety
 *
 * Behavior is undefined if the ``target`` pointer is null or not aligned.
 */
QkGate qk_target_op_gate(const QkTarget *target, size_t index);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTarget
 * Clears the ``QkTargetOp`` object.
 *
 * @param op The pointer to a ``QkTargetOp`` object.
 *
 * # Safety
 *
 * The behavior will be undefined if the pointer is null or not-aligned.
 * The data belonging to a ``QkTargetOp`` originates in Rust and
 * can only be freed using this function.
 */
void qk_target_op_clear(QkTargetOp *op);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 *
 * Generate transpiler options defaults
 *
 * This function generates a QkTranspileOptions with the default settings
 * This currently is ``optimization_level`` 2, no seed, and no approximation.
 *
 * @return A ``QkTranspileOptions`` object with default settings.
 */
QkTranspileOptions qk_transpiler_default_options(void);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 * Run the preset init stage of the transpiler on a circuit
 *
 * The Qiskit transpiler is a quantum circuit compiler that rewrites a given
 * input circuit to match the constraints of a QPU and optimizes the circuit
 * for execution. This function runs the first stage of the transpiler,
 * **init**, which runs abstract-circuit optimizations, and reduces multi-qubit
 * operations into one- and two-qubit operations. You can refer to
 * @verbatim embed:rst:inline :ref:`transpiler-preset-stage-init` @endverbatim for more details.
 *
 * This function should only be used with circuits constructed
 * using Qiskit's C API. It makes assumptions on the circuit only using features exposed via C,
 * if you are in a mixed Python and C environment it is typically better to invoke the transpiler
 * via Python.
 *
 * This function is multithreaded internally and will launch a thread pool
 * with threads equal to the number of CPUs reported by the operating system by default.
 * This will include logical cores on CPUs with simultaneous multithreading. You can tune the
 * number of threads with the ``RAYON_NUM_THREADS`` environment variable. For example, setting
 * ``RAYON_NUM_THREADS=4`` would limit the thread pool to 4 threads.
 *
 * @param dag A pointer to the circuit to run the transpiler on.
 * @param target A pointer to the target to compile the circuit for.
 * @param options A pointer to an options object that defines user options. If this is a null
 *   pointer the default values will be used. See ``qk_transpile_default_options``
 *   for more details on the default values.
 * @param layout A pointer to a pointer to a ``QkTranspileLayout`` object. On a successful
 *   execution (return code 0) a pointer to the layout object created transpiler will be written
 *   to this pointer.
 * @param error A pointer to a pointer with an nul terminated string with an error description.
 *   If the transpiler fails a pointer to the string with the error description will be written
 *   to this pointer. That pointer needs to be freed with ``qk_str_free``. This can be a null
 *   pointer in which case the error will not be written out.
 *
 * @returns The return code for the transpiler, ``QkExitCode_Success`` means success and all
 *   other values indicate an error.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag``, ``target``, or ``layout``, are not valid, non-null
 * pointers to a ``QkDag``, ``QkTarget``, or a ``QkTranspileLayout`` pointer
 * respectively. ``options`` must be a valid pointer a to a ``QkTranspileOptions`` or ``NULL``.
 * ``error`` must be a valid pointer to a ``char`` pointer or ``NULL``. The value of the inner
 * pointer for ``layout`` will be overwritten by this function. If the value pointed to needs to
 * be freed this must be done outside of this function as it will not be freed by this function.
 */
QkExitCode qk_transpile_stage_init(QkDag *dag,
                                   const QkTarget *target,
                                   const QkTranspileOptions *options,
                                   QkTranspileLayout **layout,
                                   char **error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 * Run the preset routing stage of the transpiler on a circuit
 *
 * The Qiskit transpiler is a quantum circuit compiler that rewrites a given
 * input circuit to match the constraints of a QPU and optimizes the circuit
 * for execution. This function runs the third stage of the preset pass manager,
 * **routing**, which translates all the instructions in the circuit into
 * those supported by the target. You can refer to
 * @verbatim embed:rst:inline :ref:`transpiler-preset-stage-routing` @endverbatim for more details.
 *
 * This function should only be used with circuits constructed
 * using Qiskit's C API. It makes assumptions on the circuit only using features exposed via C,
 * if you are in a mixed Python and C environment it is typically better to invoke the transpiler
 * via Python.
 *
 * This function is multithreaded internally and will launch a thread pool
 * with threads equal to the number of CPUs reported by the operating system by default.
 * This will include logical cores on CPUs with simultaneous multithreading. You can tune the
 * number of threads with the ``RAYON_NUM_THREADS`` environment variable. For example, setting
 * ``RAYON_NUM_THREADS=4`` would limit the thread pool to 4 threads.
 *
 * @param dag A pointer to the circuit to run the transpiler on.
 * @param target A pointer to the target to compile the circuit for.
 * @param options A pointer to an options object that defines user options. If this is a null
 *   pointer the default values will be used. See ``qk_transpile_default_options``
 *   for more details on the default values.
 * @param layout A pointer to a pointer to a ``QkTranspileLayout`` object. Typically you will need
 *   to run the `qk_transpile_stage_layout` prior to this function and that will provide a
 *   `QkTranspileLayout` object with the initial layout set you want to take that output layout from
 *   that function and use this as the input for this. If you don't have a layout object (e.g. you ran
 *   your own layout pass). You can run ``qk_transpile_layout_generate_from_mapping`` to generate a trivial
 *   layout (where virtual qubit 0 in the circuit is mapped to physical qubit 0 in the target,
 *   1->1, 2->2, etc) for the dag at it's current state. This will enable you to generate a layout
 *   object for the routing stage if you generate your own layout. Note that while this makes a
 *   valid layout object to track the permutation caused by routing it does not correctly reflect
 *   the initial layout if your custom layout pass is not a trivial layout. You will need to track
 *   the initial layout independently in this case.
 * @param error A pointer to a pointer with an nul terminated string with an error description.
 *   If the transpiler fails a pointer to the string with the error description will be written
 *   to this pointer. That pointer needs to be freed with ``qk_str_free``. This can be a null
 *   pointer in which case the error will not be written out.
 *
 * @returns The return code for the transpiler, ``QkExitCode_Success`` means success and all
 *   other values indicate an error.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag``, ``target``, or ``layout``, are not valid, non-null
 * pointers to a ``QkDag``, ``QkTarget``, or a ``QkTranspileLayout`` pointer
 * respectively. ``options`` must be a valid pointer a to a ``QkTranspileOptions`` or ``NULL``.
 * ``error`` must be a valid pointer to a ``char`` pointer or ``NULL``.
 */
QkExitCode qk_transpile_stage_routing(QkDag *dag,
                                      const QkTarget *target,
                                      const QkTranspileOptions *options,
                                      QkTranspileLayout *layout,
                                      char **error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 * Run the preset optimization stage of the transpiler on a circuit
 *
 * The Qiskit transpiler is a quantum circuit compiler that rewrites a given
 * input circuit to match the constraints of a QPU and optimizes the circuit
 * for execution. This function runs the fourth stage of the preset pass manager,
 * **optimization**, which optimizes the circuit for the given target after the
 * circuit has been transformed into a physical circuit. You can refer to
 * @verbatim embed:rst:inline :ref:`transpiler-preset-stage-optimization` @endverbatim for
 * more details.
 *
 * This function should only be used with circuits constructed
 * using Qiskit's C API. It makes assumptions on the circuit only using features exposed via C,
 * if you are in a mixed Python and C environment it is typically better to invoke the transpiler
 * via Python.
 *
 * This function is multithreaded internally and will launch a thread pool
 * with threads equal to the number of CPUs reported by the operating system by default.
 * This will include logical cores on CPUs with simultaneous multithreading. You can tune the
 * number of threads with the ``RAYON_NUM_THREADS`` environment variable. For example, setting
 * ``RAYON_NUM_THREADS=4`` would limit the thread pool to 4 threads.
 *
 * @param dag A pointer to the circuit to run the transpiler on.
 * @param target A pointer to the target to compile the circuit for.
 * @param options A pointer to an options object that defines user options. If this is a null
 *   pointer the default values will be used. See ``qk_transpile_default_options``
 *   for more details on the default values.
 * @param error A pointer to a pointer with an nul terminated string with an error description.
 *   If the transpiler fails a pointer to the string with the error description will be written
 *   to this pointer. That pointer needs to be freed with ``qk_str_free``. This can be a null
 *   pointer in which case the error will not be written out.
 *
 * @returns The return code for the transpiler, ``QkExitCode_Success`` means success and all
 *   other values indicate an error.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` and ``target`` are not valid, non-null
 * pointers to a ``QkDag``, or a ``QkTarget`` respectively. ``options`` must
 * be a valid pointer a to a ``QkTranspileOptions`` or ``NULL``. ``error`` must
 * be a valid pointer to a ``char`` pointer or ``NULL``.
 */
QkExitCode qk_transpile_stage_optimization(QkDag *dag,
                                           const QkTarget *target,
                                           const QkTranspileOptions *options,
                                           char **error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 * Run the preset translation stage of the transpiler on a circuit
 *
 * The Qiskit transpiler is a quantum circuit compiler that rewrites a given
 * input circuit to match the constraints of a QPU and optimizes the circuit
 * for execution. This function runs the fourth stage of the preset pass manager,
 * **translation**, which translates all the instructions in the circuit into
 * those supported by the target. You can refer to
 * @verbatim embed:rst:inline :ref:`transpiler-preset-stage-translation` @endverbatim for more details.
 *
 * This function should only be used with circuits constructed
 * using Qiskit's C API. It makes assumptions on the circuit only using features exposed via C,
 * if you are in a mixed Python and C environment it is typically better to invoke the transpiler
 * via Python.
 *
 * This function is multithreaded internally and will launch a thread pool
 * with threads equal to the number of CPUs reported by the operating system by default.
 * This will include logical cores on CPUs with simultaneous multithreading. You can tune the
 * number of threads with the ``RAYON_NUM_THREADS`` environment variable. For example, setting
 * ``RAYON_NUM_THREADS=4`` would limit the thread pool to 4 threads.
 *
 * @param dag A pointer to the circuit to run the transpiler on.
 * @param target A pointer to the target to compile the circuit for.
 * @param options A pointer to an options object that defines user options. If this is a null
 *   pointer the default values will be used. See ``qk_transpile_default_options``
 *   for more details on the default values.
 * @param error A pointer to a pointer with an nul terminated string with an error description.
 *   If the transpiler fails a pointer to the string with the error description will be written
 *   to this pointer. That pointer needs to be freed with ``qk_str_free``. This can be a null
 *   pointer in which case the error will not be written out.
 *
 * @returns The return code for the transpiler, ``QkExitCode_Success`` means success and all
 *   other values indicate an error.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` and ``target`` are not valid, non-null
 * pointers to a ``QkDag``, ``QkTarget`` respectively. ``options`` must be a valid pointer a to
 * a ``QkTranspileOptions`` or ``NULL``. ``error`` must be a valid pointer to a ``char`` pointer
 * or ``NULL``.
 */
QkExitCode qk_transpile_stage_translation(QkDag *dag,
                                          const QkTarget *target,
                                          const QkTranspileOptions *options,
                                          char **error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 * Run the preset layout stage of the transpiler on a circuit
 *
 * The Qiskit transpiler is a quantum circuit compiler that rewrites a given
 * input circuit to match the constraints of a QPU and optimizes the circuit
 * for execution. This function runs the second stage of the preset pass manager
 * **layout**, which chooses the initial mapping of virtual qubits to
 * physical qubits, including expansion of the circuit to contain explicit
 * ancillas. You can refer to
 * @verbatim embed:rst:inline :ref:`transpiler-preset-stage-layout` @endverbatim for more details.
 *
 * This function should only be used with circuits constructed
 * using Qiskit's C API. It makes assumptions on the circuit only using features exposed via C,
 * if you are in a mixed Python and C environment it is typically better to invoke the transpiler
 * via Python.
 *
 * This function is multithreaded internally and will launch a thread pool
 * with threads equal to the number of CPUs reported by the operating system by default.
 * This will include logical cores on CPUs with simultaneous multithreading. You can tune the
 * number of threads with the ``RAYON_NUM_THREADS`` environment variable. For example, setting
 * ``RAYON_NUM_THREADS=4`` would limit the thread pool to 4 threads.
 *
 * @param dag A pointer to the circuit to run the transpiler on.
 * @param target A pointer to the target to compile the circuit for.
 * @param options A pointer to an options object that defines user options. If this is a null
 *   pointer the default values will be used. See ``qk_transpile_default_options``
 *   for more details on the default values.
 * @param layout A pointer to a pointer to a ``QkTranspileLayout`` object. On a successful
 *   execution (return code 0) a pointer to the layout object created transpiler will be written
 *   to this pointer. The inner pointer for this can be null if there is no existing layout
 *   object. Typically if you run `qk_transpile_stage_init` you would take the output layout from
 *   that function and use this as the input for this. But if you don't have a layout the inner
 *   pointer can be null and a new `QkTranspileLayout` will be allocated and that pointer will be
 *   set for the inner value of the layout here.
 * @param error A pointer to a pointer with an nul terminated string with an error description.
 *   If the transpiler fails a pointer to the string with the error description will be written
 *   to this pointer. That pointer needs to be freed with ``qk_str_free``. This can be a null
 *   pointer in which case the error will not be written out.
 *
 * @returns The return code for the transpiler, ``QkExitCode_Success`` means success and all
 *   other values indicate an error.
 *
 * # Safety
 *
 * Behavior is undefined if ``dag`` or ``target``, are not valid, non-null
 * pointers to a ``QkDag``, or a ``QkTarget`` respectively. Behavior is also undefined if ``layout``
 * is not a valid, aligned, pointer to a pointer to a ``QkTranspileLayout`` or a pointer to a
 * ``NULL`` pointer. ``options`` must be a valid pointer a to a ``QkTranspileOptions`` or ``NULL``.
 * ``error`` must be a valid pointer to a ``char`` pointer or ``NULL``.
 */
QkExitCode qk_transpile_stage_layout(QkDag *dag,
                                     const QkTarget *target,
                                     const QkTranspileOptions *options,
                                     QkTranspileLayout **layout,
                                     char **error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspiler
 * Transpile a single circuit.
 *
 * The Qiskit transpiler is a quantum circuit compiler that rewrites a given
 * input circuit to match the constraints of a QPU and optimizes the circuit
 * for execution. This function should only be used with circuits constructed
 * using Qiskit's C API. It makes assumptions on the circuit only using features exposed via C,
 * if you are in a mixed Python and C environment it is typically better to invoke the transpiler
 * via Python.
 *
 * This function is multithreaded internally and will launch a thread pool
 * with threads equal to the number of CPUs reported by the operating system by default.
 * This will include logical cores on CPUs with simultaneous multithreading. You can tune the
 * number of threads with the ``RAYON_NUM_THREADS`` environment variable. For example, setting
 * ``RAYON_NUM_THREADS=4`` would limit the thread pool to 4 threads.
 *
 * @param qc A pointer to the circuit to run the transpiler on.
 * @param target A pointer to the target to compile the circuit for.
 * @param options A pointer to an options object that defines user options. If this is a null
 *   pointer the default values will be used. See ``qk_transpile_default_options``
 *   for more details on the default values.
 * @param result A pointer to the memory location of the transpiler result. On a successful
 *   execution (return code 0) the output of the transpiler will be written to the pointer. The
 *   members of the result struct are owned by the caller and you are responsible for freeing
 *   the members using the respective free functions.
 * @param error A pointer to a pointer with an nul terminated string with an error description.
 *   If the transpiler fails a pointer to the string with the error description will be written
 *   to this pointer. That pointer needs to be freed with ``qk_str_free``. This can be a null
 *   pointer in which case the error will not be written out.
 *
 * @returns The return code for the transpiler, ``QkExitCode_Success`` means success and all
 *   other values indicate an error.
 *
 * # Safety
 *
 * Behavior is undefined if ``circuit``, ``target``, or ``result``, are not valid, non-null
 * pointers to a ``QkCircuit``, ``QkTarget``, or ``QkTranspileResult`` respectively.
 * ``options`` must be a valid pointer a to a ``QkTranspileOptions`` or ``NULL``.
 * ``error`` must be a valid pointer to a ``char`` pointer or ``NULL``.
 */
QkExitCode qk_transpile(const QkCircuit *qc,
                        const QkTarget *target,
                        const QkTranspileOptions *options,
                        QkTranspileResult *result,
                        char **error);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Return the number of qubits in the input circuit to the transpiler.
 *
 * @param layout A pointer to the ``QkTranspileLayout``.
 *
 * @return The number of input qubits
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkTranspileLayout``.
 */
uint32_t qk_transpile_layout_num_input_qubits(const QkTranspileLayout *layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Return the number of qubits in the output circuit from the transpiler.
 *
 * @param layout A pointer to the ``QkTranspileLayout``.
 *
 * @return The number of output qubits
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkTranspileLayout``.
 */
uint32_t qk_transpile_layout_num_output_qubits(const QkTranspileLayout *layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Query the initial layout of a ``QkTranspileLayout``.
 *
 * The output array from this function represents the mapping from the virutal qubits in the
 * original input circuit to the physical qubit in the output circuit. The
 * index in the array is the virtual qubit and the value is the physical qubit. For example an
 * output array of:
 *
 * ```
 * [1, 0, 2]
 * ```
 *
 * indicates that the layout maps virtual qubit 0 -> physical qubit 1, virtual qubit 1 -> physical
 * qubit -> 0, and virtual qubit 2 -> physical qubit 2.
 *
 * @param layout A pointer to the ``QkTranspileLayout``.
 * @param filter_ancillas If set to true the output array will not include any indicies for any
 * ancillas added by the transpiler.
 * @param initial_layout A pointer to the array where this function will write the initial layout
 * to. This must have sufficient space for the full array which will either be
 * ``qk_transpile_layout_num_input_qubits()`` or ``qk_transpile_layout_num_output_qubits()`` for
 * ``filter_ancillas`` being true or false respectively.
 *
 * @returns True if there was a initial_layout written to ``initial_layout`` and false if there
 * is no initial layout.
 *
 * # Safety
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkTranspileLayout``. ``initial_layout`` must be a valid, non-null pointer with a large enough
 * allocation to store the size necessary for the initial layout. If ``filter_ancillas`` is true
 * this will be number of input qubits (which can be checked with
 * ``qk_transpile_layout_num_input_qubits()``) or the number of output qubits if ``filter_ancillas``
 * is false (which can be queried with ``qk_transpile_layout_num_output_qubits()``).
 */
bool qk_transpile_layout_initial_layout(const QkTranspileLayout *layout,
                                        bool filter_ancillas,
                                        uint32_t *initial_layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Query the output permutation of a ``QkTranspileLayout``
 *
 * The output array from this function represents the permutation induced by the transpiler where
 * the index indicates the qubit at the start of the circuit and the value is the position of the
 * qubit at the end of the circuit. For example an output array of:
 *
 * ```
 * [1, 2, 0]
 * ```
 *
 * indicates that qubit 0 from the start of the circuit is at qubit 1 at the end of the circuit,
 * 1 -> 2, and 2 -> 0.
 *
 * @param layout A pointer to the ``QkTranspileLayout``.
 * @param output_permutation A pointer to the array where this function will write the output permutation
 * to. This must have sufficient space for the output which will be the number of output qubits in
 * the layout. This can be queried with ``qk_transpile_layout_num_output_qubits``.
 *
 * @returns True if there is an output permutation that was written to ``output_permutation``
 * false if the ``QkTranspileLayout`` does not contain an output permutation.
 *
 * # Safety
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkTranspileLayout``. ``output_permutation`` must be a valid, non-null pointer with a large enough
 * allocation to store the size necessary for the output_permutation. This will always be the number
 * of output qubits in the ``QkTranspileLayout`` which can be queried with
 * ``qk_transpile_layout_num_output_qubits()``.
 */
bool qk_transpile_layout_output_permutation(const QkTranspileLayout *layout,
                                            uint32_t *output_permutation);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Query the final layout of a ``QkTranspileLayout``
 *
 * The output array represents the mapping from the virtual qubit in the original input circuit to
 * the physical qubit at the end of the transpile circuit that has that qubit's state. The array
 * index represents the virtual qubit and the value represents the physical qubit at the end of
 * the transpiled circuit which has that virtual qubit's state. For example, an output array of:
 *
 * ```
 * [2, 0, 1]
 * ```
 *
 * indicates that virtual qubit 0's state in the original circuit is on
 * physical qubit 2 at the end of the transpiled circuit, 1 -> 0, and 2 -> 1.
 *
 * @param layout A pointer to the ``QkTranspileLayout``.
 * @param filter_ancillas If set to true the output array will not include any indicies for any
 * ancillas added by the transpiler.
 * @param final_layout A pointer to the array where this function will write the final layout to.
 * This must have sufficient space for the output which will either be the number of input or
 * output qubits depending on the value of filter_ancillas.
 *
 * # Safety
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a
 * ``QkTranspileLayout``. ``final_layout`` must be a valid, non-null pointer with a large enough
 * allocation to store the size necessary for the final layout. If ``filter_ancillas`` is true
 * this will be number of input qubits (which can be checked with
 * ``qk_transpile_layout_num_input_qubits()``) or the number of output qubits if ``filter_ancillas``
 * is false (which can be queried with ``qk_transpile_layout_num_output_qubits()``).
 */
void qk_transpile_layout_final_layout(const QkTranspileLayout *layout,
                                      bool filter_ancillas,
                                      uint32_t *final_layout);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Generate a ``QkTranspileLayout`` from a initial layout mapping
 *
 * This will generate a ``QkTranspileLayout`` with the initial layout set (and no ouptput
 * permutation) from a provided mapping. The intent of this function is to enable creating
 * a custom layout pass that also creates a ``QkTranspileLayout`` that you can use with
 * subsequent stage functions such as ``qk_transpile_stage_routing``.
 *
 * @param original_dag: A pointer to the original dag prior to running a custom layout pass. This
 *     dag must have fewer than or the same number of qubits as ``target``.
 * @param target: A pointer to the target that layout was run on. This target must have fixed
 *     number of qubits set.
 * @param qubit_mapping: A pointer to the layout mapping array. This array must have the same
 *     number of elements as there are qubits in target and each element is a unique integer and
 *     the all must fall in the range of 0 to ``num_qubits`` where ``num_qubits`` is the
 *     number of qubits indicated in the provided value for ``target``.
 *     The first elements represent the virtual qubits in ``original_dag`` and the value
 *     represents the physical qubit in the target which the virtual qubit is mapped too.
 *     For example an array of ``[1, 0, 2]`` would map virtual qubit 0 -> physical qubit 1,
 *     virtual qubit 1 -> physical qubit 0, and virtual qubit 2 -> physical qubit 2. For elements
 *     that are not in the original dag these are treated as ancilla qubits, but still must be
 *     mapped to a physical qubit. This array will be copied into the output ``QkTranspileLayout``
 *     so you must still free it after calling this function.
 *
 * @returns The QkTranspileLayout object with the initial layout set
 *
 * # Safety
 * Behavior is undefined if ``original_dag`` and target ``target`` are not a valid, aligned,
 * non-null pointer to a ``QkDag`` or a ``QkTarget`` respectively. ``qubit_mapping`` must be a
 * valid pointer to a contiguous array of ``uint32_t`` with enough space for the number of qubits
 * indicated in ``target``.
 */
QkTranspileLayout *qk_transpile_layout_generate_from_mapping(const QkDag *original_dag,
                                                             const QkTarget *target,
                                                             const uint32_t *qubit_mapping);
#endif

#if defined(QISKIT_WITH_CBINDINGS)
/**
 * @ingroup QkTranspileLayout
 * Free a ``QkTranspileLayout`` object
 *
 * @param layout a pointer to the layout to free
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` is not a valid, non-null pointer to a ``QkTranspileLayout``.
 */
void qk_transpile_layout_free(QkTranspileLayout *layout);
#endif

#if (defined(QISKIT_C_PYTHON_INTERFACE) && defined(QISKIT_WITH_CBINDINGS))
/**
 * @ingroup QkTranspileLayout
 * Generate a Python-space ``TranspileLayout`` object from a ``QkTranspileLayout``.
 *
 * The created Python-space object is a copy of the ``QkTranspileLayout`` provided, the data
 * representation is different between C and Python and the data is not moved to Python like
 * for some other ``*_to_python`` functions.
 *
 * @param layout a pointer to a ``QkTranspileLayout``.
 * @param circuit a pointer to the original ``QkCircuit``.
 * @return the PyObject pointer for the Python space TranspileLayout object.
 *
 * # Safety
 *
 * Behavior is undefined if ``layout`` and ``circuit`` are not valid, non-null pointers to a
 * ``QkTranspileLayout`` and ``QkCircuit`` respectively. It is assumed that the thread currently
 * executing this function holds the Python GIL. This is required to create the Python object
 * returned by this function.
 */
PyObject *qk_transpile_layout_to_python(const QkTranspileLayout *layout,
                                          const QkCircuit *circuit);
#endif

#ifdef __cplusplus
}  // extern "C"
#endif  // __cplusplus

#endif  /* QISKIT_H */
